#!/bin/sh
#
# run_examples.sh
#
# Run the Ab Initio examples

program=`basename $0`

# Make a test directory for the new runs
/bin/rm -rf test
mkdir test

# Enter the test directory
cd test
echo "Test directory: `pwd`"

# Scripts used in the examples
# NOTE: Change 'rosetta' to be the executable you are using.
rosetta="../../source/rosetta.gcc"
rosettaAB="../../scripts/bin/rosettaAB.pl"
cluster="../../scripts/bin/cluster.pl"
extract="../../scripts/bin/extract.pl"

# Check if all dependent scripts exist.
found_all="true"
scripts="$rosettaAB $cluster $extract"
for script in $scripts; do
  if [ ! -e $script ]; then
    echo "$program: Didn't find '$script'."
    found_all=""
  fi
done

# If all the scripts haven't been found, quit
if [ -z "$found_all" ]; then
    echo "$program: Missing scripts.  Have you run 'RosettaAbinitio/setup.sh?'"
    echo "$program: Exiting."
    exit
fi

# Make sure executable is valid
if [ ! -e $rosetta ]; then
    echo "$program: Executable missing.  Two possibilities:"
    echo "  (1) You didn't run 'RosettaAbinitio/setup.sh'"
    echo "  (2) You didn't build 'rosetta.gcc'."
    echo "      (If you are on a Mac modify run_examples.sh to use 'rosetta.mac"
    exit;
fi

# Make a test directory for the new runs
# Link input files (fragment files) to test directory.
ln -s ../1d3z/1d3z_.fasta .
ln -s ../1d3z/aa1d3z_03_05.200_v1_3 .
ln -s ../1d3z/aa1d3z_09_05.200_v1_3 .
ln -s ../1d3z/paths.txt .

# Run Rosetta's ab initio protocol

echo -n "Rosetta ab initio decoy generation test ..."
$rosettaAB -binary $rosetta -fasta 1d3z_.fasta -nstruct 1 -outdir ./ -verbose -additional_args "-constant_seed" > 1d3z_rosettaAB.log 2>&1
echo "... done.  Status in '1d3z_rosettaAB.log'"

echo -n "Rosetta ab initio clustering test (clustering 70 decoys) ..."
$cluster -silentfile ../1d3z/aa1d3z_silentmode_70decoys.out > 1d3z_cluster.log 2>&1
echo " done.  Status in '1d3z_cluster.log'"

echo -n "Rosetta ab initio decoy extraction test (extracting decoy #50) ..."
$extract -binary $rosetta ../1d3z/aa1d3z_silentmode_70decoys.out ../1d3z/aa1d3z_silentmode_70decoys.list > 1d3z_extract.log 2>&1
echo " done.  Status in '1d3z_extract.log'"

echo -n "Rosetta ab initio cluster center extraction test ..."
$cluster -binary $rosetta -silentfile ../1d3z/aa1d3z_silentmode_70decoys.out -get_centers 5 > 1d3z_extract_cluster_centers.log 2>&1
echo " done.  Status in '1d3z_extract_cluster_centers.log'"


