// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: johnk $


// Rosetta Headers
#include "AnkyrinRepeat_classes.h"
#include "AnkyrinRepeat_ns.h"
#include "atom_tree_routines.h"
#include "DesignMap.h"
#include "minimize.h"
#include "misc.h"
#include "nblist.h"
#include "orient_rms.h"
#include "pack.h"
#include "PackerTask.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_io.h"
#include "score.h"
#include "score_data.h"
#include "score_name.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>

// C++ Headers
#include <iostream>
#include <string>


//////  STARTING METHODS FOR AnkyrinModule  //////


////////////////////////////////////////////////////////////////////////////////
/// @begin AnkyrinModule::read_module
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void AnkyrinModule::read_module( std::string const & in_pdbfile ){

	std::cout << "Instantiating an AnkyrinModule from PDB file " << in_pdbfile << std::endl;
	name_ = in_pdbfile;

	bool const fullatom(true);
	bool const idealize(false);
	bool const read_all_chains(true);
	pose_from_pdb( pose_, in_pdbfile, fullatom, idealize, read_all_chains );

	assert ( pose_.total_residue() == AnkyrinRepeat_ns::module_length );

	return;

}



////////////////////////////////////////////////////////////////////////////////
/// @begin replace_module_in_pose
///
/// @brief
///    Replace an AR module which is part of a pose
///    with the module in the AnkyrinModule object
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void AnkyrinModule::replace_module_in_pose( pose_ns::Pose & protein_pose, int const start_seqpos ) {

	using namespace pose_ns;

	std::cout << "Replacing an AR module starting at seqpos " << start_seqpos << std::endl;
	AnkyrinRepeat_ns::module_info.clear();

	assert ( pose_.total_residue() == AnkyrinRepeat_ns::module_length );

	//	protein_pose.dump_pdb("jk_pre_swapped.pdb");

	// Build a pose containing just the region to be replaced, save the starting AR sequence
	Pose AR_for_replacement;
	AR_for_replacement = pose_;  // just to get a pose of the right size...
	AR_for_replacement.copy_segment( AnkyrinRepeat_ns::module_length, protein_pose, 1, start_seqpos );
	FArray1D_int start_res( AR_for_replacement.total_residue() );
	start_res = AR_for_replacement.res();

	// Orient "*this" AnkyrinModule to match the corresponding region in AR_for_replacement
	FArray1D_int align_start( 1, 1 );
	FArray1D_int align_end( 1, AR_for_replacement.total_residue() );
	float rms;
	orient_region(pose_.total_residue(),pose_.Eposition(),pose_.full_coord(),AR_for_replacement.Eposition(),
								1,align_start,align_end,1,align_start,align_end,rms);
	std::cout << "Incoming loop has an RMS from starting loop of: " << rms << std::endl;

	// Setup atom tree, so that we can transfer backbone coors
	protein_pose.clear_atom_tree();
	protein_pose.setup_atom_tree();

	// Replace the coors of the existing module with the new one
	// jk We could perhaps use "copy_segment" here, I'm just not sure that the orientation is preserved...
	for ( int seqpos = start_seqpos, i = 1; i <= pose_.total_residue(); ++seqpos, ++i ) {
		// jk Note: the "false" at the end of copy_sidechain leads the backbone coors to be copied too
		protein_pose.copy_sidechain( seqpos, pose_.res(i), pose_.res_variant(i), pose_.full_coord()(1,1,i), false);
	}
	// Trigger a refold - also for safety, before calling pack_rotamers
	protein_pose.copy_to_misc();

	//	protein_pose.dump_pdb("jk_module_swapped.pdb");

	// Bring back the starting sequence on the new backbone (via repacking)
	std::cout << "Fixing module sequence after replacement" << std::endl;
	PackerTask Task(protein_pose);
	FArray1D_bool allow_repack( protein_pose.total_residue(), false );
	Task.set_task("packrot",false,allow_repack,false,false);
	Task.setup_residues_to_vary();
	for ( int seqpos = start_seqpos, i = 1; i <= AnkyrinRepeat_ns::module_length; ++seqpos, ++i ) {
		Task.get_designmap().set(seqpos,start_res(i));
	}
	pack_rotamers( protein_pose, Task);

	//	protein_pose.dump_pdb("jk_repacked.pdb");

	// Relax the new module (fixing the environment)
	std::cout << "Starting minimization after module replacement" << std::endl;
	bool const init_nblist_setting = get_use_nblist();  // jk save this for later
	float const minimization_tolerance(0.001);
	pose_setup_minimizer(minimization_tolerance);
	pose_set_use_nblist(false);
	set_use_nblist(false);
	minimize_set_vary_rb_angle( false );
	minimize_set_vary_rb_trans( false );

	// Go back to fold tree
	//	protein_pose.clear_atom_tree();

	pose_ns::Fold_tree ft;
	//	ft.clear();

	if ( start_seqpos < misc::domain_end(1) ) {
		std::cout << "Case for AR as first chain is not yet implemented..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	} else {
		ft.add_edge(1, misc::domain_end(1), pose_param::PEPTIDE);
		int const chainB_start = misc::domain_end(1)+1;
		int const postAR_start = start_seqpos + AnkyrinRepeat_ns::module_length;
		ft.add_edge(1, chainB_start, 1);
		ft.add_edge(chainB_start, start_seqpos-1, pose_param::PEPTIDE);
		ft.add_edge(chainB_start, start_seqpos, 2);
		ft.add_edge(start_seqpos, postAR_start-1, pose_param::PEPTIDE);
		ft.add_edge(chainB_start, postAR_start, 3);
		ft.add_edge(postAR_start, misc::domain_end(2), pose_param::PEPTIDE);
	}

	ft.reorder(1);
	protein_pose.set_fold_tree(ft);
	protein_pose.set_allow_jump_move( 1, false );
	protein_pose.set_allow_jump_move( 2, true );
	protein_pose.set_allow_jump_move( 3, false );

	// jk Apply backbone/sidechain flexibility
	FArray1D_bool allow_move( protein_pose.total_residue(), false );
	for ( int seqpos = start_seqpos, i = 1; i <= AnkyrinRepeat_ns::module_length; ++seqpos, ++i ) {
		allow_move(seqpos) = true;
	}
	protein_pose.set_allow_bb_move( allow_move );
	protein_pose.set_allow_chi_move( allow_move );

	// jk Use score12
	Score_weight_map weight_map;
	weight_map.clear();
	weight_map.set_weights( score12 );

	// jk Minimize
	protein_pose.main_minimize( weight_map, "linmin" );

	//	protein_pose.dump_pdb("jk_minimized.pdb");
	//	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

	set_use_nblist(init_nblist_setting); // jk restore original setting
	std::cout << "Done minimization after module replacement" << std::endl;

	std::ostringstream info_string;
	info_string << "REMARK    INTRODUCED AR MODULE " << name_ << " WITH RMSD " << rms;
	AnkyrinRepeat_ns::module_info.clear();
	AnkyrinRepeat_ns::module_info.push_back(info_string.str());

	std::cout << "Done replacing AR module." << std::endl;

	return;
}


//////  DONE WITH METHODS FOR AnkyrinModule  //////

