// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: johnk $

// Rosetta Headers
#include "AnkyrinRepeat_functions.h"
#include "after_opts.h"
#include "AnkyrinRepeat_classes.h"
#include "AnkyrinRepeat_ns.h"
#include "DesignMap.h"
#include "HotspotRotamer.h"
#include "misc.h"
#include "pack.h"
#include "PackerTask.h"
#include "param_aa.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_io.h"
#include "pose_param.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <cassert>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>

//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/izstream.hh>


////////////////////////////////////////////////////////////////////////////////
/// @begin do_AR_setup
///
/// @brief
/// jk setup AR functions
///
/// @detailed
///
/// @param  none
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void do_AR_setup() {

	if ( truefalseoption( "AR_module_list" ) ) {

		std::string list_filename;
		stringafteroption( "AR_module_list", "none", list_filename );

		utility::io::izstream module_filelist;
		module_filelist.clear();
		module_filelist.open( list_filename );

		AnkyrinRepeat_ns::AR_module_set.clear();

		if ( !module_filelist ) {
			std::cout << "Open failed for AR list file: " << module_filelist.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		std::string inpline;
		std::string pdbname;
		int num_modules(0);
		while ( getline( module_filelist, inpline ) ) {
			std::istringstream line_stream ( inpline,std::istringstream::in );
			line_stream >> pdbname;
			ObjexxFCL::strip_whitespace( pdbname ); // Trim off any leading or trailing whitespace
			if ( has_suffix( pdbname, ".gz" ) ) pdbname.erase( pdbname.length() - 3 ); // strip terminal .gz
			// jk Expect a ".pdb" at the end of the filename...
			if ( has_suffix( pdbname, ".pdb" ) ) pdbname.erase( pdbname.length() - 4 ); // strip terminal .pdb
			AnkyrinModule new_AR(pdbname+".pdb");
			AnkyrinRepeat_ns::AR_module_set.push_back(new_AR);
			++num_modules;
		}

		std::cout << "Read this many AR modules for loop swapping: " << num_modules << std::endl;

		module_filelist.close();
		module_filelist.clear();

	}

	return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin load_AR_data
///
/// @brief
/// jk read the conserved residues
///
/// @detailed
///
/// @param  none
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void load_AR_data( std::string const & AR_conservation_file ) {

	using namespace AnkyrinRepeat_ns;

	if ( AR_conservation_file == "none" ) return;
	std::cout << "Reading conserved AR residues from " << AR_conservation_file << std::endl;

	using_AR = true;
	AspAsn_seqpos = false;

	utility::io::izstream AR_instream;
	AR_instream.clear();
	AR_instream.open( AR_conservation_file );
	if ( !AR_instream ) {
		std::cout << "Open failed for file: " << AR_instream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	bool eof(false);
	while(1) {
		int seqpos;
		AR_instream >> seqpos >> std::skipws;
		std::cout << "Applying conserved AR residue at seqpos " << seqpos << std::endl;
		AspAsn_seqpos(seqpos) = true;
		if(AR_instream.eof()) {
			eof = true;
			break;
		}
	}

	AR_instream.close();
	AR_instream.clear();

}



////////////////////////////////////////////////////////////////////////////////
/// @begin update_AR_interface_neighbors
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void update_AR_interface_neighbors( pose_ns::Pose const & pose ) {

	using namespace AnkyrinRepeat_ns;

	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		if ( AspAsn_seqpos(seqpos) ) {
			AspAsn_neighbors(seqpos) = count_interface_neighbors( pose, seqpos );
		}
	}

	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert_all_to_Asn
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void convert_all_to_Asn( pose_ns::Pose & pose ) {

	using namespace AnkyrinRepeat_ns;

	bool sequence_change = false;
	PackerTask conv_Task(pose);
	FArray1D_bool allow_repack( pose.total_residue(), false );
	conv_Task.set_task("packrot",false,allow_repack,false,false);
	conv_Task.setup_residues_to_vary();
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		if ( ! AspAsn_seqpos(seqpos) ) continue;
		conv_Task.get_designmap().fix_completely(seqpos);
		if ( pose.res(seqpos) == param_aa::aa_asn ) continue;
		conv_Task.get_designmap().set( seqpos, param_aa::aa_asn );
		sequence_change = true;
	}

	if ( ! sequence_change ) return;

	pack_rotamers( pose, conv_Task);
	pose.copy_to_misc();

	std::cout << "Converted all Asp/Asn to Asn." << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert_all_to_Asp
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void convert_all_to_Asp( pose_ns::Pose & pose ) {

	using namespace AnkyrinRepeat_ns;

	bool sequence_change = false;
	PackerTask conv_Task(pose);
	FArray1D_bool allow_repack( pose.total_residue(), false );
	conv_Task.set_task("packrot",false,allow_repack,false,false);
	conv_Task.setup_residues_to_vary();
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		if ( ! AspAsn_seqpos(seqpos) ) continue;
		conv_Task.get_designmap().fix_completely(seqpos);
		if ( pose.res(seqpos) == param_aa::aa_asp ) continue;
		conv_Task.get_designmap().set( seqpos, param_aa::aa_asp );
		sequence_change = true;
	}

	if ( ! sequence_change ) return;

	pack_rotamers( pose, conv_Task);
	pose.copy_to_misc();

	std::cout << "Converted all Asp/Asn to Asp." << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert_all_to_Ser
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void convert_all_to_Ser( pose_ns::Pose & pose ) {

	using namespace AnkyrinRepeat_ns;

	bool sequence_change = false;
	PackerTask conv_Task(pose);
	FArray1D_bool allow_repack( pose.total_residue(), false );
	conv_Task.set_task("packrot",false,allow_repack,false,false);
	conv_Task.setup_residues_to_vary();
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		if ( ! AspAsn_seqpos(seqpos) ) continue;
		conv_Task.get_designmap().fix_completely(seqpos);
		if ( pose.res(seqpos) == param_aa::aa_ser ) continue;
		conv_Task.get_designmap().set( seqpos, param_aa::aa_ser );
		sequence_change = true;
	}

	if ( ! sequence_change ) return;

	pack_rotamers( pose, conv_Task);
	pose.copy_to_misc();

	std::cout << "Converted all Asp/Asn to Ser." << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert_Asp_to_Asn
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void convert_Asp_to_Asn( pose_ns::Pose & pose, int const seqpos ) {

	if ( pose.res(seqpos) == param_aa::aa_asn ) {
		std::cout << "Position " << seqpos << " is already Asn - nothing to convert." << std::endl;
		return;
	}

	std::cout << "Converting seqpos " << seqpos << " to Asn." << std::endl;

	// Create PackerTask and set up values for a call to pack_rotamers
	PackerTask conv_Task(pose);
	FArray1D_bool allow_repack( pose.total_residue(), false );
	allow_repack( seqpos ) = true;
	conv_Task.set_task("packrot",false,allow_repack,false,false);
	conv_Task.setup_residues_to_vary();
	conv_Task.get_designmap().fix_completely(seqpos);
	conv_Task.get_designmap().set( seqpos, param_aa::aa_asn );
	pack_rotamers( pose, conv_Task);
	pose.copy_to_misc();

	std::cout << "Done converting seqpos " << seqpos << " to Asn." << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert_Asn_to_Asp
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void convert_Asn_to_Asp( pose_ns::Pose & pose, int const seqpos ) {

	if ( pose.res(seqpos) == param_aa::aa_asp ) {
		std::cout << "Position " << seqpos << " is already Asp - nothing to convert." << std::endl;
		return;
	}

	std::cout << "Converting seqpos " << seqpos << " to Asp." << std::endl;

	// Create PackerTask and set up values for a call to pack_rotamers
	PackerTask conv_Task(pose);
	FArray1D_bool allow_repack( pose.total_residue(), false );
	allow_repack( seqpos ) = true;
	conv_Task.set_task("packrot",false,allow_repack,false,false);
	conv_Task.setup_residues_to_vary();
	conv_Task.get_designmap().fix_completely(seqpos);
	conv_Task.get_designmap().set( seqpos, param_aa::aa_asp );
	pack_rotamers( pose, conv_Task);
	pose.copy_to_misc();

	std::cout << "Done converting seqpos " << seqpos << " to Asp." << std::endl;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin repack_AspAsn
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void repack_AspAsn( pose_ns::Pose & pose ) {

	using namespace param_aa;

	std::cout << "Repacking all conserved Asp/Asn." << std::endl;

	// Create PackerTask and setup values for a call to pack_rotamers
	PackerTask conv_Task(pose);
	FArray1D_bool allow_repack( pose.total_residue(), false );
	conv_Task.set_task("packrot",false,allow_repack,false,false);
	conv_Task.setup_residues_to_vary();

	std::cout << "Starting configuration is:";
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		if ( ! AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) continue;
		conv_Task.get_designmap().set( seqpos, aa_asn );
		conv_Task.get_designmap().set( seqpos, aa_asp );
		std::cout << "  " << aa_name3(pose.res(seqpos)) << seqpos;
	}
	std::cout << std::endl;

	pack_rotamers( pose, conv_Task);
	pose.copy_to_misc();

	std::cout << "Final configuration is:";
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		if ( ! AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) continue;
		std::cout << "  " << aa_name3(pose.res(seqpos)) << seqpos;
	}
	std::cout << std::endl;

	std::cout << "Done repacking all conserved Asp/Asn." << std::endl;

}

