// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
#include "CorrelatedSimAnnealer.h"
#include "InteractionGraphBase.h"
#include "PDInteractionGraph.h"
#include "ligand_ns.h"
#include "misc.h"
#include "random_numbers.h"
#include "param.h"
#include "RotamerSet.h"

#include <utility/basic_sys_util.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.o.hh> // F(), I(), A()...

#include <iostream>

#ifdef GL_GRAPHICS
#include "gl_graphics.h"
#endif
#ifdef BOINC_GRAPHICS
#include "boinc/boinc_rosetta_graphics.h"
#include <ObjexxFCL/FArray3D.hh>
#endif

namespace pack{

bool use_CorrelatedSimAnnealer = false;

////////////////////////////////////////////////////////////////////////////////
/// @begin CorrelatedSimAnnealer::CorrelatedSimAnnealer()
///
/// @brief
/// constructor
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified

////////////////////////////////////////////////////////////////////////////////
CorrelatedSimAnnealer::CorrelatedSimAnnealer(
	std::vector<int> & rot_to_pack,
	std::vector< std::vector<int> > & rotamer_correlations,
	FArray1D_int & bestrotamer_at_seqpos,
	float & bestenergy,
	bool start_with_current, // start simulation with current rotamers
	pack::InteractionGraphBase * ig,
	const RotamerSet * p_rotamer_set,
	FArray1DB_int & current_rot_index,
	bool calc_rot_freq,
	FArray1D_float & rot_freq
):
	SimAnnealerBase(
	(int) rot_to_pack.size(),
	bestrotamer_at_seqpos,
	bestenergy,
	start_with_current, // start simulation with current rotamers
	p_rotamer_set,
	current_rot_index,
	calc_rot_freq,
	rot_freq
	),
	ig_(ig),
	rot_to_pack_( rot_to_pack ),
	rotamer_correlations_ ( rotamer_correlations )
{}

CorrelatedSimAnnealer::CorrelatedSimAnnealer(
	std::vector< std::vector<int> > & rotamer_correlations,
	FArray1D_int & bestrotamer_at_seqpos,
	float & bestenergy,
	bool start_with_current, // start simulation with current rotamers
	pack::InteractionGraphBase * ig,
	const RotamerSet * p_rotamer_set,
	FArray1DB_int & current_rot_index,
	bool calc_rot_freq,
	FArray1D_float & rot_freq
):
	SimAnnealerBase(
	(ig->get_num_total_states()),
	bestrotamer_at_seqpos,
	bestenergy,
	start_with_current, // start simulation with current rotamers
	p_rotamer_set,
	current_rot_index,
	calc_rot_freq,
	rot_freq
	),
	ig_(ig),
	rot_to_pack_( ig_->get_num_total_states() ),
	rotamer_correlations_ ( rotamer_correlations )
{
	for (unsigned int ii =1; ii <= rot_to_pack_.size(); ++ii) {
		rot_to_pack_[ ii-1 ] = ii;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin CorrelatedSimAnnealer::~CorrelatedSimAnnealer()
///
/// @brief
/// virtual destructor
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
CorrelatedSimAnnealer::~CorrelatedSimAnnealer()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin CorrelatedSimAnnealer::run
///
/// @brief
/// sim_annealing for fixed backbone design mode
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void CorrelatedSimAnnealer::run()
{
	using namespace param;
	using namespace misc;

	std::cout << "Running CorrelatedSimAnnealer..." << std::endl;

	//--------------------------------------------------------------------
	//internal variables

	int ranrotamer, prevrotamer_state, moltenres_id, rotamer_state_on_moltenres;
	int correlated_rotamer, moltenres2_id, rotamer2_state_on_moltenres;

	float currentenergy, prev_energy, delta_energy, delta_pair;
	int const nmoltenres = ig_->get_num_nodes();

	//FArray1D_int list( p_rotamer_set_->nrotamers() );
	FArray1D_int state_on_node( nmoltenres, 0 );
	FArray1D_int best_state_on_node( nmoltenres, 0 );
	FArray1D_float loopenergy( maxouteriterations, 0.0 );

	//kwk internal rot_index expanded to handled non amino acid molten residues
	//kwk rot_to_moltenres(1,X)=moltenres_id
	//kwk rot_to_moltenres(2,X)=moltenres_state
	FArray2D_int rot_2_moltenres( 2, ig_->get_num_total_states(), 0 );

	//--------------------------------------------------------------------
	//initialize variables

	currentenergy = 0.0;

	ig_->prepare_for_simulated_annealing();
	ig_->blanket_assign_state_0();

	int rot_tmp=1;
	for( int ii =1; ii<=nmoltenres; ++ii){
		for(int jj =1; jj<=ig_->get_num_states_for_node( ii); ++jj){
			rot_2_moltenres(1,rot_tmp)=ii;
			rot_2_moltenres(2,rot_tmp)=jj;
			rot_tmp++;
		}
	}

	//--------------------------------------------------------------------
	if ( num_of_rot_to_pack_ == 0 ) return;

	setup_iterations();

	FArray1D_float previous_nsteps_for_rot( p_rotamer_set_->nrotamers(), 0.0 );

	int outeriterations = get_outeriterations();

	//outer loop
	for (int nn = 1; nn <= outeriterations; ++nn ) {
		setup_temperature(loopenergy,nn);
		if ( quench() ) {
			currentenergy = bestenergy_;
			state_on_node = best_state_on_node;
			ig_->set_network_state( state_on_node );
		}
		//rh std::cout << "Sim Annealer Temperature: " << get_temperature() << std::endl;

		int inneriterations = get_inneriterations();

		float threshold_for_deltaE_inaccuracy = std::sqrt( get_temperature() );
		ig_->set_errorfull_deltaE_threshold( threshold_for_deltaE_inaccuracy );

		//inner loop
		for (int n = 1; n <= inneriterations; ++n ){
			ranrotamer = pick_a_rotamer(n,nn);
			if ( ranrotamer == -1 ) continue;

			moltenres_id = rot_2_moltenres( 1, ranrotamer );
			rotamer_state_on_moltenres = rot_2_moltenres( 2, ranrotamer );
			prevrotamer_state = state_on_node( moltenres_id );

			if ( rotamer_state_on_moltenres == prevrotamer_state ) continue;

			// the meat of this SimAnnealer variety begins here
			correlated_rotamer = pick_correlated_rotamer( ranrotamer );

			if ( correlated_rotamer != -1 ) { // paired substitution
			// the multi-layered implementation of a one-by-one
			// consideration/substitution regime in the InteractionGraph classes
			// requires that the substitutions be considered/committed in series

				moltenres2_id = rot_2_moltenres( 1, correlated_rotamer );
				rotamer2_state_on_moltenres = rot_2_moltenres( 2, correlated_rotamer );

				ig_->consider_substitution( moltenres_id, rotamer_state_on_moltenres,
				 delta_energy, prev_energy );
				delta_pair = delta_energy;

//			std::cout <<
//			 "rotamer1 " << I(5,ranrotamer) << ": " << F(7,2,delta_energy) <<
//			 " rotamer2 " << I(5,correlated_rotamer) << ": ";

				ig_->consider_substitution( moltenres2_id, rotamer2_state_on_moltenres,
				 delta_energy, prev_energy );
				delta_pair += delta_energy;

//			std::cout << F(7,2,delta_energy) << " pair total " << F(7,2,delta_pair);

				if ( prevrotamer_state == 0 ||
					   pass_metropolis( prev_energy, delta_pair ) ) {
//				std::cout << " PASS (prevrotamer_state " << prevrotamer_state << ")";

					// the correlate was the last considered, so commit it first
					ig_->commit_considered_substitution();
					state_on_node(moltenres2_id) = rotamer2_state_on_moltenres;

					// now must re-'consider' and commit the primary rotamer
					ig_->consider_substitution( moltenres_id,
					 rotamer_state_on_moltenres, delta_energy, prev_energy );

					currentenergy = ig_->commit_considered_substitution();
					state_on_node(moltenres_id) = rotamer_state_on_moltenres;

					if ( prevrotamer_state == 0 || currentenergy < bestenergy_ ) {
//					std::cout << "currentenergy " << F(7,2,currentenergy) <<
//					 "bestenergy " << F(7,2,bestenergy_);
						bestenergy_ = currentenergy;
						best_state_on_node = state_on_node; // these are 1D arrays
					}
				}
//				std::cout << std::endl;

			} else { // normal single substitution

				ig_->consider_substitution( moltenres_id, rotamer_state_on_moltenres,
				 delta_energy, prev_energy );

				if ( prevrotamer_state == 0 ||
					   pass_metropolis( prev_energy, delta_energy ) ) {

					currentenergy = ig_->commit_considered_substitution();
					state_on_node(moltenres_id) = rotamer_state_on_moltenres;

					if ( prevrotamer_state == 0 || currentenergy < bestenergy_ ) {
						bestenergy_ = currentenergy;
						best_state_on_node = state_on_node;
					}
				}
			}
			loopenergy(nn) = currentenergy;

			// pKa mode not implemented due to uncertain suitability

		} // end inner
	} //end outer

	if ( ig_->any_vertex_state_unassigned() )
	{
		std::cerr << "Critical error -- In CorrelatedSimAnnealer, one or more vertex states unassigned at annealing's completion." << std::endl;
		std::cerr << "Critical error -- assignment and energy of assignment meaningless" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		assert( ! ig_->any_vertex_state_unassigned() );
	}

	//convert best_state_on_node into best_rotamer_at_seqpos
	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		int iiresid = p_rotamer_set_->moltenres_2_resid(ii);
		if( iiresid > 0 ){
			bestrotamer_at_seqpos_(iiresid) =
			 best_state_on_node(ii) + p_rotamer_set_->rotindex_offsets(iiresid);

//		std::cout << "bestrotindex: " << bestrotamer_at_seqpos_(iiresid) <<
//		 " at resid: " << iiresid << std::endl;

		} else if ( iiresid < 0 &&
		            size_t(-1*iiresid) <= ligand::ligand_ptr_vector.size() ){
			//kwk may need to recover current position.
			//	std::cout << best_state_on_node(ii)-1 << std::endl;
			ligand::ligand_ptr_vector[-1*iiresid-1]->
				change_to_ligand_conformation( size_t( best_state_on_node(ii)-1 ) );
			ligand::ligand_ptr_vector[-1*iiresid-1]->set_best_coordinates();
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin CorrelatedSimAnnealer::pick_a_rotamer()
///
/// @brief pick a rotamer from a list
///
/// @detailed:if no rotamer is available, return a nonsense number, like -1
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int CorrelatedSimAnnealer::pick_a_rotamer
(
 int innercycle,
 int outercycle
 )
{
	using namespace misc;

	bool start_with_current = get_start_with_current();
	int ranrotamer = -1;
	int num = 0;
	//bk if quench cycle, pass through all rotamers before
	//bk repeating a rotamer
	if ( quench() ) {

		num = mod( innercycle - 1, num_of_rot_to_pack_ );
		if ( num == 0 ) {
			std::random_shuffle( rot_to_pack_.begin(), rot_to_pack_.end() );
		}
		ranrotamer = rot_to_pack_.at( num );
		//bk if start of run and start_with_current is true then first nres
		//bk iterations will be used to place the current rotamers
	} else if ( start_with_current && outercycle == 1 &&
	           innercycle <= total_residue ) {

		if (current_rot_index_(innercycle) != -1 ){
			ranrotamer = current_rot_index_(innercycle);
		}
		else return -1;

	} else {
		ranrotamer = rot_to_pack_.at( static_cast<int>( num_of_rot_to_pack_ * ran3() ) );
	}
	return ranrotamer;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin CorrelatedSimAnnealer::pick_correlated_rotamer()
///
/// @brief
/// Pick a second new rotamer from a list of ones that may be correlated to the
/// first.  Return no second rotamer if the input rotamer has no correlates.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
int CorrelatedSimAnnealer::pick_correlated_rotamer
(
 int const first_rotamer
)
{
	int num_corr = (rotamer_correlations_.at(first_rotamer-1)).size();
	if ( num_corr == 0 ) return -1;

	return rotamer_correlations_.at(first_rotamer-1).at( static_cast<int>( num_corr * ran3() ) );
}

}//end of namespace
