// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 16636 $
//  $Date: 2007-08-21 15:47:16 -0700 (Tue, 21 Aug 2007) $
//  $Author: ashworth $


// Rosetta headers
#include "DesignMap.h"
#include "aaproperties_pack.h"
#include "misc.h"
#include "param.h"
#include "random_numbers.h" // ran3

// Utility headers
#include <utility/basic_sys_util.hh>
#include <utility/vector1.hh>

// ObjexxFCL headers
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray1D.hh>

// C++ headers
#include <cstdlib>
#include <cstdio>
#include <iostream>


using namespace param;


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  set_native_aa
///
/// @brief
/// Set the native amino acide from res(), which loop each
/// residue pistions and return the native amino acid
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::DesignMapPerResidue::set_native_aa(int native_aa){
  the_native_aa = native_aa;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin set_residue
///
/// @brief
/// enable an given residues to be available for redesign
///
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::DesignMapPerResidue::set_residue(int the_aa){
  if(!residues[the_aa]){       // if(residues[the_aa] != true)
    residues[the_aa] = true;
    number_of_available_aa++;
  }
  // check condition: has any available aa to be redesigned, means that allow the residue to move.
  // so that repack_residue must be true. design_map => repack_residue
  if ( number_of_available_aa>0 ){
    repack_residue = true;
  }
}

//////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  disable_residue
///
/// @brief
/// disable this position and if repack_residue is true, set the native aa be true
///
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::DesignMapPerResidue::disable_residue(int the_aa){
  if(residues[the_aa]!=false){
    residues[the_aa] = false;
    if(number_of_available_aa>0){
      number_of_available_aa--;
    }
    else{
      std::cerr << "error: number of available aa is negative" << std::endl;
			utility::exit( 2, __FILE__, __LINE__);
    }
  }

  //=====================================================================
  // disable_residue: if disable all the aa (1-28) to be false,
  // and set repack_residue is still be true.
  //=====================================================================
  if( repack_residue && ( number_of_available_aa==0 ) && ( number_of_available_aav==1 ) ){
    repack_residue = false;
  }

}

//////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  enable_variant_type_per_residue
///
/// @brief
/// allow this variant type at this position
///
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::DesignMapPerResidue::enable_variant_type_per_residue(int the_variant_type){

  if(	! allowed_variant_types(the_variant_type) ) {
    allowed_variant_types(the_variant_type) = true;
		++number_of_available_aav;
		// jk The "terminus" variants are mutually exclusive with "base rotamer"
		if ( the_variant_type == aaproperties_pack::aav_base ) {
			disable_variant_type_per_residue(aaproperties_pack::aav_Nterm);
			disable_variant_type_per_residue(aaproperties_pack::aav_Cterm);
		}
		if ( ( the_variant_type == aaproperties_pack::aav_Nterm ) ||
				 ( the_variant_type == aaproperties_pack::aav_Cterm ) ) {
			disable_variant_type_per_residue(aaproperties_pack::aav_base);
		}
  }

  if( number_of_available_aav>1 ){
		repack_residue = true;
	}

}

//////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  disable_variant_type_per_residue
///
/// @brief
/// disable this variant type at this position
///
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::DesignMapPerResidue::disable_variant_type_per_residue(int the_variant_type){

  if(	allowed_variant_types(the_variant_type) ) {
    allowed_variant_types(the_variant_type) = false;
		number_of_available_aav--;

		if ( ( the_variant_type == aaproperties_pack::aav_Nterm) &&
				 ( ! allowed_variant_types(aaproperties_pack::aav_Cterm)) ) {
			enable_variant_type_per_residue(aaproperties_pack::aav_base);
		}
		if ( ( the_variant_type == aaproperties_pack::aav_Cterm) &&
				 ( ! allowed_variant_types(aaproperties_pack::aav_Nterm)) ) {
			enable_variant_type_per_residue(aaproperties_pack::aav_base);
		}

    if(number_of_available_aav<1){
      std::cerr << "error: number of available aav cannot be less than one" << std::endl;
			utility::exit( 2, __FILE__, __LINE__);
    }
  }

  if ( repack_residue && ( number_of_available_aa==0 ) && ( number_of_available_aav==1 ) ){
    repack_residue = false;
  }

}

//////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  variant_type_is_allowed_per_residue
///
/// @brief
/// check whether this variant type is allowed at this position
///
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
bool DesignMap::DesignMapPerResidue::variant_type_is_allowed_per_residue(int the_variant_type) const {
  return allowed_variant_types(the_variant_type);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  fix_sidechain_of_residue
///
/// @brief
/// fix the sidechain of the residue
///
/// @detailed
/// note: does NOT affect state of allowed rotamer variants!
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::DesignMapPerResidue::fix_sidechain_of_residue(){
  //=================================================================
  // fix_sidechain_all_residues: fix all the current positons.
  // If no more change needed, fix the current positions
  //=================================================================
  for(int i=1; i<=MAX_AA()(); i++){
    if(residues[i]){
      residues[i] = false;
    }
  }
  number_of_available_aa = 0;
	//  number_of_available_aav = 1;
	//	allowed_variant_types = false;
	//	allowed_variant_types(aaproperties_pack::aav_base)=true;
  if ( number_of_available_aav == 1 ) repack_residue = false;

}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin require_base_rotamer_per_residue
///
/// @brief
/// disallow all variants for a given residue
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::DesignMapPerResidue::require_base_rotamer_per_residue(){

	// jk Note: preserve the status of the "terminus" variants
	// (mutually exclusive with "base rotamer")

	bool const Nterm_enabled = allowed_variant_types(aaproperties_pack::aav_Nterm);
	bool const Cterm_enabled = allowed_variant_types(aaproperties_pack::aav_Cterm);

  number_of_available_aav = 1;
	allowed_variant_types = false;
	allowed_variant_types(aaproperties_pack::aav_base)=true;

	if ( Nterm_enabled ) {
		enable_variant_type_per_residue(aaproperties_pack::aav_Nterm);
	}
	if ( Cterm_enabled ) {
		enable_variant_type_per_residue(aaproperties_pack::aav_Cterm);
	}

  if ( number_of_available_aa == 0 ) {
    repack_residue = false;
  }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin design_map_per_residue
///
/// @brief
/// return whether a given aa is available for redesign
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
bool DesignMap::DesignMapPerResidue::design_map_per_residue(int the_aa) const {
  return residues[the_aa];
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin  repack_per_residue
///
/// @brief
/// return whether the residue is to be fixed
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
bool DesignMap::DesignMapPerResidue::repack_per_residue() const {
  return repack_residue;
}

//============================================================
//return whether the base rotamer is required at this position
//============================================================
bool DesignMap::DesignMapPerResidue::use_base_rotamer_per_residue() const {
  if ( number_of_available_aav == 1 ) return true;
  return false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin random_allowed_aa
///
/// @brief
///
/// return a random choice from the allowed amino acid types at this residue
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
int
DesignMap::DesignMapPerResidue::random_allowed_aa() const
{
	if ( !repack_residue || ( number_of_available_aa == 0 ) ) {
		// improper use
		utility::exit( 2, __FILE__, __LINE__ );
	}

	int const choice_index( static_cast<int>( ran3() * number_of_available_aa ) );

	int counter(0);
	for ( int aa(1), aae( residues.size() ); aa <= aae; ++aa ) {
		if ( residues[aa] ) { // residues that are repacking
			if ( counter == choice_index ) return aa; // success
			++counter;
		}
	}
	// should not arrive here
	utility::exit( 2, __FILE__, __LINE__ );
	return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin set_from_res
///
/// @brief
/// set the design map for a given residue to match a different residue
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: johnk
///
/// @last_modified 2006-03-19
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::copy_res(const int source_residue, const int destination_residue){
  map[destination_residue]=map[source_residue];
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin set
///
/// @brief
/// set the design map for a given residue and a given amino acid
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::set(const int the_residue, const int the_aa){
  map[the_residue].set_residue(the_aa);
}


//=================================================
//set the design map by giving a design_matrix
//=================================================
void DesignMap::set(const int nResidues, FArray2D<bool>& design_matrix){
  using namespace param;
  int size=map.size();
  if(size<nResidues){
    //extend to reach the length of the input matrix
    map.insert(map.end(),nResidues-map.size(),DesignMapPerResidue());
  }
  else if(size>nResidues){
    int size = map.size()-nResidues;
    for(int i=1; i<=size; i++){
      map.pop_back();   // The pop_back() function removes the last element of the vector.
    }
  }
  for(int i=1; i<=nResidues; i++){
    for(int j=1; j<=MAX_AA(); j++){
      if(design_matrix(i,j)){
	map[i].set_residue(j);
      }
      else{
	map[i].disable_residue(j);
      }
    }
  }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin disable
///
/// @brief
/// disable an amino acid in the residue position
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::disable(const int the_residue, const int the_aa){
  map[the_residue].disable_residue(the_aa);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin disable_all
///
/// @brief
/// disable all amino acid in the residue at once
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::disable_all(){
  //using namespace misc;
  for(unsigned int i=1; i <= map.size(); i++){
    for(int j=1; j<=MAX_AA();j++) {
      map[i].disable_residue(j);
    }
  }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin fix_completely
///
/// @brief
/// fix both the sidechain and the variant
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: johnk
///
/// @last_modified 2006-11-12
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::fix_completely(const int the_residue){
  map[the_residue].fix_sidechain_of_residue();
  map[the_residue].require_base_rotamer_per_residue();
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin fix_completely
///
/// @brief
/// fix both the sidechain and the variant for all residues
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: johnk
///
/// @last_modified 2006-11-12
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::fix_completely(){
  //using namespace misc;
  for(unsigned int i=1; i<=map.size(); i++){
    for(int j=1; j<=MAX_AA();j++) {
			map[j].fix_sidechain_of_residue();
			map[j].require_base_rotamer_per_residue();
    }
  }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin fix_sidechain_allowing_variants
///
/// @brief
/// fix sidechain at the position
///
/// @detailed
/// note: sets the rotamer variant status to the default
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
//====================================
//fix_sidechain the position
//====================================
void DesignMap::fix_sidechain_allowing_variants(const int the_residue){
  map[the_residue].fix_sidechain_of_residue();
	apply_variant_defaults(the_residue);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin require_base_rotamer
///
/// @brief
/// allow only the base rotamer
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::require_base_rotamer(){
  //using namespace misc;
  for(unsigned int i=1; i<=map.size(); i++){
		require_base_rotamer(i);
	}
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin require_base_rotamer
///
/// @brief
/// allow only the base rotamer
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::require_base_rotamer(const int the_residue){
  map[the_residue].require_base_rotamer_per_residue();
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin get
///
/// @brief
/// return design_map
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
bool DesignMap::get(const int the_residue, const int the_aa) const {
  return map[the_residue].design_map_per_residue(the_aa);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin has_rotamers
///
/// @brief
///
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
bool DesignMap::has_rotamers(const int the_residue, const int the_aa) const {
  return ( map[the_residue].design_map_per_residue(the_aa) ||
		(( map[the_residue].num_allowed_aa_per_residue() == 0 ) &&
		 ( map[the_residue].num_allowed_aav_per_residue() > 1 ) &&
		 ( map[the_residue].get_native_aa() == the_aa )) );
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin repack_residue
///
/// @brief
/// return repack_residue
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
bool DesignMap::repack_residue( int const the_residue ) const {
  return map[the_residue].repack_per_residue();
}

//==================================
//return use_base_rotamer
//==================================
bool DesignMap::use_base_rotamer( int const the_residue ) const {
  return map[the_residue].use_base_rotamer_per_residue();
}

//==================================
//print the current design map
//==================================
void DesignMap::print_dm() const {
  int length = map.size();
  for(int i=1; i<= length; i++){
    std::cout << i << " ";
    for(int j=1; j<=MAX_AA(); j++){
      std::cout << map[i].design_map_per_residue(j) << " ";
    }
    std::cout << map[i].repack_per_residue() << " ";
    std::cout << map[i].use_base_rotamer_per_residue() << std::endl;
  }
}

void DesignMap::print_dm(int theRES) const {
  int length = map.size();
  if(theRES>0 && theRES<=length){
    std::cout << theRES << " ";
    for(int j=1; j<=MAX_AA(); j++){
      std::cout << map[theRES].design_map_per_residue(j) << " ";
    }
    std::cout << map[theRES].repack_per_residue() << " ";
    std::cout << map[theRES].use_base_rotamer_per_residue() << std::endl;
  }
  else{
    std::cout << "out of range" << std::endl;
  }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin convert_to_dm
///
/// @brief
/// Covert namespace protamers/sidechain to design_map format format for namespace members
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::convert_to_dm(FArray1D_bool const & protamers, FArray2D_bool const & sidechain){
  //using namespace misc;
  assert( protamers.size() == map.size() );
  for (unsigned int seqpos=1; seqpos<=map.size(); seqpos++) {
    if(!protamers(seqpos)) {
      fix_sidechain_allowing_variants(seqpos);
    }
    else if(protamers(seqpos)){
      for (int aa=1; aa<=MAX_AA(); aa++){
        if(sidechain(aa,seqpos))
         set(seqpos,aa);
        else if(!sidechain(aa,seqpos))
         disable(seqpos,aa);
      }
    }
  }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin convert_to_ns
///
/// @brief
/// Covert design map to protamers format for namespace members
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::convert_to_ns(FArray1D_bool & ns_protamers){
  //using namespace misc;
  for (unsigned int seqpos=1; seqpos<=map.size(); seqpos++){
    if(repack_residue(seqpos))
      ns_protamers(seqpos) = true;
    else if(!repack_residue(seqpos))
      ns_protamers(seqpos) = false;
  }
}



///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin all_true_repack_residue
///
/// @brief
/// set repack residue in special situations
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: yiliu
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::all_true_repack_residue(){
  //using namespace misc;
  for (unsigned int seqpos=1; seqpos<=map.size(); seqpos++){
    set(seqpos, map[seqpos].get_native_aa() );
  }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin allow_variant_type
///
/// @brief
/// allow a variant type at all positions
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::allow_variant_type(const int variant_type){
  //using namespace misc;
  for (unsigned int seqpos=1; seqpos<=map.size(); seqpos++){
		map[seqpos].enable_variant_type_per_residue(variant_type);
  }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin disallow_variant_type
///
/// @brief
/// disallow a variant type at all positions
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::disallow_variant_type(const int variant_type){
  //using namespace misc;
  for (unsigned int seqpos=1; seqpos<=map.size(); seqpos++){
		map[seqpos].disable_variant_type_per_residue(variant_type);
  }
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin allow_variant_type_at_residue
///
/// @brief
/// allow a variant type at a specific residue
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::allow_variant_type_at_residue(const int variant_type, const int seqpos ){
	map[seqpos].enable_variant_type_per_residue(variant_type);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin disallow_variant_type_at_residue
///
/// @brief
/// disallow a variant type at a specific residue
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::disallow_variant_type_at_residue(const int variant_type, const int seqpos){
	map[seqpos].disable_variant_type_per_residue(variant_type);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin variant_type_is_allowed
///
/// @brief
/// return whether a a variant type is allowed at this position
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
bool DesignMap::variant_type_is_allowed(const int variant_type, const int seqpos) const {
	return map[seqpos].variant_type_is_allowed_per_residue(variant_type);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin set_variant_defaults
///
/// @brief
/// set the default allowed variants
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::set_variant_defaults(
	FArray1DB_bool const & in_default_variant_array
)
{
	for (int i=1; i<=aaproperties_pack::number_aav_type; i++){
		default_variant_type(i) = in_default_variant_array(i);
	}
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin apply_variant_defaults
///
/// @brief
/// apply the default allowed variants to a specified residue
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::apply_variant_defaults( int const the_residue ){

	// start by enabling the base variant
	map[the_residue].enable_variant_type_per_residue(1);
	// exclude the base variant from this loop
  for (int i=2; i<=aaproperties_pack::number_aav_type; i++){
		if ( default_variant_type(i) ) {
			map[the_residue].enable_variant_type_per_residue(i);
		} else {
			map[the_residue].disable_variant_type_per_residue(i);
		}
	}

	return;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin apply_variant_defaults
///
/// @brief
/// apply the default allowed variants to all residues
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
void DesignMap::apply_variant_defaults(){
  //using namespace misc;

	for(unsigned int seqpos=1; seqpos<=map.size(); seqpos++){
		apply_variant_defaults(seqpos);
	}

	return;
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin num_allowed_aa
///
/// @brief
/// return the number of allowed aa at this seqpos
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
int DesignMap::num_allowed_aa( int const the_residue ) const {
	return map[the_residue].num_allowed_aa_per_residue();
}


///////////////////////////////////////////////////////////////////////////////////////////////////////
/// @begin num_allowed_aav
///
/// @brief
/// return the number of allowed aav at this seqpos
///
/// @detailed
///
///
/// @param
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: jk
///
/// @last_modified 2006-01-31
///////////////////////////////////////////////////////////////////////////////////////////////////////
int DesignMap::num_allowed_aav( int const the_residue ) const {
	return map[the_residue].num_allowed_aav_per_residue();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin random_allowed_aa
///
/// @brief
/// return a random choice from the allowed amino acid types at seqpos
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
int DesignMap::random_allowed_aa( int const seqpos ) const
{
	return map[seqpos].random_allowed_aa();
}

