// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

/// @authors Xavier Ambroggio

// C++ Headers

#include <iostream>
#include <fstream>
#include <vector>
#include <string>
#include <list>
#include <map>
#include <set>
#include <algorithm>
#include <iterator>
#include <sstream>
#include <ctype.h>

// Rosetta Headers

#include "EquivResfileReader.h"
#include "misc.h"
#include "pdb.h"

//Utility Headers
#include <utility/basic_sys_util.hh>

using namespace std;

EquivResfileReader::EquivResfileReader(){}

EquivResfileReader::~EquivResfileReader(){}

int EquivResfileReader::seq_pos_from_pdb_res_num_chain( int const pdb_res_num_in, char const pdb_chain_in )
{
  // This function is derived from res_num_from_pdb_res_num_chain() function written by Jeff Gray in docking_constraints.cc

  using namespace misc;
  using namespace pdb;

  for ( int i = 1; i <= total_residue; ++i )
    if ( res_chain(i) == pdb_chain_in && pdb_res_num(i) == pdb_res_num_in ) return i;

  cout << "WARNING: pdb residue not found " << pdb_res_num_in << ' ' << pdb_chain_in << " , exiting.\n";
  utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

  return -1;
}

bool EquivResfileReader::valid_word( string query )
{
  for(unsigned int i = 0; i < query.size(); i++)
    if( ! isalpha( char( query.at(i) ) ) ) return false;

  return true;
}

string EquivResfileReader::itos( int i ) // convert int to string
{
  stringstream s;
  s << i;
  return s.str();
}

void EquivResfileReader::expand_range_into_unordered_list(string & value)
{
  int next;
  vector<int> range;

  if( find( value.begin(), value.end(), ':') == value.end() ) return;

  replace( value.begin(), value.end(), ':', ' ');
  istringstream value_stream(value);
  while( value_stream >> next) range.push_back(next);

  if( range.size() != 2 ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  value.clear();

  sort( range.begin(), range.end() );

  for(int i = range.at(0); i <= range.at(1); i++) value += itos(i) + " ";

}

void EquivResfileReader::expand_range_into_ordered_list(string & value){
  int next;
  vector<int> range;

  if( find( value.begin(), value.end(), '-') == value.end() ) return;

  replace( value.begin(), value.end(), '-', ' ');
  istringstream value_stream(value);
  while( value_stream >> next) range.push_back(next);

  if( range.size() != 2 ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  value.clear();

  if( range.at(0) < range.at(1) ){
    for(int i = range.at(0); i <= range.at(1); i++){
      if( i == range.at(1) ) value += itos(i);
      else value += itos(i) + ",";
    }
  }else{
    for(int i = range.at(0); i >= range.at(1); i--){
      if( i == range.at(1) ) value += itos(i);
      else value += itos(i) + ",";
    }
  }

}

string EquivResfileReader::process_string_without_ranges(string & value)
{
  string type_of_range;

  if( find( value.begin(), value.end(), ',') != value.end() ) type_of_range = "ordered";
  else type_of_range = "unordered";

  return type_of_range;
}


string EquivResfileReader::process_string_with_ranges(string & value)
{
  string next;
  string type_of_range;

  if( find( value.begin(), value.end(), ':') != value.end() ){
    type_of_range = "unordered";
  }else if( find( value.begin(), value.end(), '-') != value.end() ){
    type_of_range = "ordered";
  }else{
    type_of_range = "none";
    return type_of_range;
  }

  istringstream value_stream(value);
  value.clear();
  while( value_stream >> next){
    if( type_of_range == "ordered" )expand_range_into_ordered_list(next);
    else if( type_of_range == "unordered" )expand_range_into_unordered_list(next);
    value += next;
  }

  return type_of_range;

}

void EquivResfileReader::check_size_of_ordered_lists_equal( map<string, string> query_map){
  map<string, string>::iterator map_SoS_iter;
  vector<int> size_of_ordered_lists;
  vector< vector<int> > number_of_ordered_lists;
  string next_list;
  int current_size = 0;

  for(map_SoS_iter = query_map.begin(); map_SoS_iter != query_map.end(); map_SoS_iter++) {
    istringstream value_stream(map_SoS_iter->second);
    while(value_stream >> next_list){
      current_size = count( next_list.begin(), next_list.end(), ',' );
      if( current_size == 0 ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
      size_of_ordered_lists.push_back( current_size );
    }
    number_of_ordered_lists.push_back( size_of_ordered_lists );
    size_of_ordered_lists.clear();
  }

  for(unsigned int i = 0; i < number_of_ordered_lists.size(); i++)
    for(unsigned int j = 0; j < number_of_ordered_lists.size(); j++)
      if( i != j && number_of_ordered_lists.at(i) != number_of_ordered_lists.at(j) ) utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

}

void EquivResfileReader::create_vector_of_ordered_lists( map<string, string> query_map, vector< list<string> > & vector_list )
{
  string key, value, current_residue_number, current_residue;
  int index = 0;
  map<string, string>::iterator map_SoS_iter;
  list<string> initialize_lists;

  for(map_SoS_iter = query_map.begin(); map_SoS_iter != query_map.end(); map_SoS_iter++) {
    index = 0;
    key = map_SoS_iter->first;
    value = map_SoS_iter->second;

    if( find( value.begin(), value.end(), ',') == value.end() ) return;

    replace( value.begin(), value.end(), ',', ' ');
    istringstream value_stream(value);

    while( value_stream >> current_residue_number ){
      current_residue = key + ":" + current_residue_number;
      if( map_SoS_iter == query_map.begin() ){
	initialize_lists.push_back( current_residue );
	vector_list.push_back( initialize_lists );
	initialize_lists.clear();
      }else{
	vector_list.at(index).push_back(current_residue);
	index++;
      }
    }
  }
}

void EquivResfileReader::create_vector_of_unordered_lists( map<string, string> query_map, vector< list<string> > & vector_list )
{
  string key, value, current_residue_number, current_residue;
  map<string, string>::iterator map_SoS_iter;
  list<string> initialize_lists;

  for(map_SoS_iter = query_map.begin(); map_SoS_iter != query_map.end(); map_SoS_iter++) {
    key = map_SoS_iter->first;
    value = map_SoS_iter->second;

    istringstream value_stream(value);
    while( value_stream >> current_residue_number ){
      current_residue = key + ":" + current_residue_number;
      initialize_lists.push_back( current_residue );
    }
  }

  vector_list.push_back( initialize_lists );

}

bool EquivResfileReader::merging_lists_of_strings( list<string> & LoS1, list<string> LoS2){
  list<string>::iterator list_oS_iter;

  LoS1.sort();
  LoS2.sort();

  for(list_oS_iter = LoS1.begin(); list_oS_iter != LoS1.end(); list_oS_iter++){
    if( find( LoS2.begin(), LoS2.end(), *list_oS_iter) != LoS2.end() ){
      LoS1.merge(LoS2);
      LoS1.unique();
      return true;
    }
  }

  return false;

}

void EquivResfileReader::merge_vectors_of_lists_of_strings( vector< list<string> > & VoLoS1, vector< list<string> > VoLoS2)
{
  bool merging_lists;
  set<int> merged_lists_in_VoLoS2;

  for(unsigned int i = 0; i < VoLoS1.size(); i++){
    for(unsigned int j = 0; j < VoLoS2.size(); j++){
      merging_lists = merging_lists_of_strings(VoLoS1.at(i), VoLoS2.at(j));
      if( merging_lists ) merged_lists_in_VoLoS2.insert(j);
    }
  }

  for( unsigned int i = 0; i < VoLoS2.size(); i++){
    if( merged_lists_in_VoLoS2.find(i) == merged_lists_in_VoLoS2.end() ) VoLoS1.push_back( VoLoS2.at( i ) );
  }

  //cout << endl;

  for(unsigned int i = 0; i < VoLoS1.size(); i++){
    for(unsigned int j = 0; j < VoLoS1.size(); j++){
      merging_lists_of_strings(VoLoS1.at(i), VoLoS1.at(j));
      if( i != j && VoLoS1.at(i) == VoLoS1.at(j) ){
	VoLoS1.at(j) = VoLoS1.at( VoLoS1.size() - 1 );
	VoLoS1.pop_back();
      }
    }
  }

}

void EquivResfileReader::convert_vector_of_list_of_pdb_residues_to_rosetta_numbering( vector< list<string> > & equivalent_pdb_residues,
										      vector< list<int> > & equivalent_rosetta_residues )
{
  list<string>::iterator list_oS_iter;
  string value, chain_string;
  char chain;
  int pdb_residue_number, rosetta_residue_number;

  equivalent_rosetta_residues.clear();
  equivalent_rosetta_residues.resize( equivalent_pdb_residues.size() );

  for(unsigned int i = 0; i < equivalent_pdb_residues.size(); i++){
    for(list_oS_iter = equivalent_pdb_residues.at(i).begin(); list_oS_iter != equivalent_pdb_residues.at(i).end(); list_oS_iter++){
      value = * list_oS_iter;
      replace( value.begin(), value.end(), ':', ' ');
      istringstream value_stream(value);
      value_stream >> chain >> pdb_residue_number;
      chain_string = chain;
      if( ! valid_word( chain_string ) ) exit(1);
      rosetta_residue_number = seq_pos_from_pdb_res_num_chain( pdb_residue_number, chain );
      equivalent_rosetta_residues.at(i).push_back( rosetta_residue_number );
    }
  }
}

void EquivResfileReader::read_equivalent_residue_file(string equiv_resfile_name, vector< list<int> > & equivalent_residues)
{
  vector< list<string> > current_equivalent_residues, equivalent_pdb_residues;

  ifstream input_equiv_resfile( equiv_resfile_name.c_str() );
  if ( input_equiv_resfile.fail() ){
    cout << "Equivalent resfile file not found. Do not pass GO, do not collect $200.\n";
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }else{
    cout << "Reading " << equiv_resfile_name << endl;
  }

  int line_number = 1;
  string current_line_string, next, residues;
  string chain;
  istringstream current_line_stream;
  map<string, string> current_line_map;
  map<string, string>::iterator map_SoS_iter;
  list<string>::iterator list_oS_iter;
  list<int>::iterator list_oI_iter;
  string type_of_range;

  while( getline( input_equiv_resfile, current_line_string ) ){
    type_of_range.clear();
    current_line_map.clear();
    istringstream current_line_stream( current_line_string );
    while(current_line_stream >> next){

      if( valid_word( next ) ){
	chain = next;
      }else{
	type_of_range = process_string_with_ranges( next );
	if( type_of_range == "none") type_of_range = process_string_without_ranges( next );
	current_line_map[chain] += next + " ";
      }

    }

    if( type_of_range == "ordered" ){
      check_size_of_ordered_lists_equal( current_line_map );
      create_vector_of_ordered_lists( current_line_map, current_equivalent_residues );
    }else if( type_of_range == "unordered" ){
      create_vector_of_unordered_lists( current_line_map, current_equivalent_residues );
    }

    if( line_number == 1) equivalent_pdb_residues = current_equivalent_residues;
    else merge_vectors_of_lists_of_strings( equivalent_pdb_residues, current_equivalent_residues);

    current_equivalent_residues.clear();
    line_number++;

  }

  convert_vector_of_list_of_pdb_residues_to_rosetta_numbering( equivalent_pdb_residues, equivalent_residues );

}
