// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

#include "after_opts.h"
#include "FixbbSimAnnealer.h"
#include "RotamerAssigningAnnealer.h"
#include "InteractionGraphBase.h"
#include "PDInteractionGraph.h"
#include "ligand_ns.h"
#include "misc.h"
#include "random_numbers.h"
#include "param.h"
#include "RotamerSet.h"
#include "pKa_mode.h"

#include <utility/basic_sys_util.hh>

#include <ObjexxFCL/Fmath.hh>

#include <iostream>

#ifdef GL_GRAPHICS
#include "gl_graphics.h"
#include "protein_graphics.h"
#include "trajectory.h"
#endif
#ifdef BOINC_GRAPHICS
#include "boinc/boinc_rosetta_graphics.h"
#include "protein_graphics.h"
#include "trajectory.h"
#include <ObjexxFCL/FArray3D.hh>
#endif

namespace pack{

////////////////////////////////////////////////////////////////////////////////
/// @begin FixbbSimAnnealer::FixbbSimAnnealer()
///
/// @brief
/// constructor
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified

////////////////////////////////////////////////////////////////////////////////
FixbbSimAnnealer::FixbbSimAnnealer(
	std::vector<int> & rot_to_pack,
	FArray1D_int & bestrotamer_at_seqpos,
	float & bestenergy,
	bool start_with_current, // start simulation with current rotamers
	pack::InteractionGraphBase * ig,
	const RotamerSet * p_rotamer_set,
	FArray1DB_int & current_rot_index,
	bool calc_rot_freq,
	FArray1D_float & rot_freq
):
	RotamerAssigningAnnealer(
	rot_to_pack,
	(int) rot_to_pack.size(),
	bestrotamer_at_seqpos,
	bestenergy,
	start_with_current, // start simulation with current rotamers
	p_rotamer_set,
	current_rot_index,
	calc_rot_freq,
	rot_freq
	), ig_(ig)
{
}

FixbbSimAnnealer::FixbbSimAnnealer(
	FArray1D_int & bestrotamer_at_seqpos,
	float & bestenergy,
	bool start_with_current, // start simulation with current rotamers
	pack::InteractionGraphBase * ig,
	const RotamerSet * p_rotamer_set,
	FArray1DB_int & current_rot_index,
	bool calc_rot_freq,
	FArray1D_float & rot_freq
):
	RotamerAssigningAnnealer(
	(ig->get_num_total_states()),
	bestrotamer_at_seqpos,
	bestenergy,
	start_with_current, // start simulation with current rotamers
	p_rotamer_set,
	current_rot_index,
	calc_rot_freq,
	rot_freq
	), ig_(ig)
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin FixbbSimAnnealer::~FixbbSimAnnealer()
///
/// @brief
/// virtual destructor
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
FixbbSimAnnealer::~FixbbSimAnnealer()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin FixbbSimAnnealer::run
///
/// @brief
/// sim_annealing for fixed backbone design mode
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void FixbbSimAnnealer::run()
{
	using namespace param;
	using namespace misc;

	//--------------------------------------------------------------------
	//internal variables

	int ranrotamer,prevrotamer_state;
	int moltenres_id,rotamer_state_on_moltenres;
	float currentenergy,previous_energy_for_node,delta_energy;
	int const nmoltenres = ig_->get_num_nodes();

	//FArray1D_int list( p_rotamer_set_->nrotamers() );
	FArray1D_int state_on_node( nmoltenres,0 );
	FArray1D_int best_state_on_node( nmoltenres,0 );
	FArray1D_float loopenergy(maxouteriterations,0.0);

	//kwk internal rot_index expanded to handled non amino acid molten residues
	//kwk rot_to_moltenres(1,X)=moltenres_id
	//kwk rot_to_moltenres(2,X)=moltenres_state
	FArray2D_int rot_2_moltenres( 2,ig_->get_num_total_states(),0);

	//bk variables for calculating rotamer frequencies during simulation
	int nsteps = 0;
	FArray1D_int nsteps_for_rot( ig_->get_num_total_states(), 0 ); //apl bug fix; this needs to index all rotamers

	//--------------------------------------------------------------------
	//initialize variables

	currentenergy = 0.0;

	ig_->prepare_for_simulated_annealing();
	ig_->blanket_assign_state_0();

	int rot_tmp=1;
	for( int ii =1; ii<=nmoltenres; ++ii){
		for(int jj =1; jj<=ig_->get_num_states_for_node( ii); ++jj){
			rot_2_moltenres(1,rot_tmp)=ii;
			rot_2_moltenres(2,rot_tmp)=jj;
			rot_tmp++;
		}
	}

	//--------------------------------------------------------------------
	if ( num_of_rot_to_pack_ == 0 ) return;

	setup_iterations();

	if ( pKa_mode::get_pKa_flag() ) {
		std::cout << "Nrot " << p_rotamer_set_->nrotamers() << "\tNres " << total_residue << std::endl;
		//rh Reset internal moving variables
		// jk Put off dimensioning and setting these until we know we're going to use them
		pKa_mode::pKa_packer_flags::store_rotstep_E.dimension( p_rotamer_set_->nrotamers(), 0. );
	}
	FArray1D_float previous_nsteps_for_rot(p_rotamer_set_->nrotamers(), 0.0);

	int outeriterations = get_outeriterations();

	std::ofstream annealer_trajectory;
	static bool const record_annealer_trajectory( truefalseoption("record_annealer_trajectory") ); //look up once
	if ( record_annealer_trajectory )
	{
		std::string trajectory_file_name( stringafteroption("record_annealer_trajectory" ) );
		annealer_trajectory.open(trajectory_file_name.c_str() );
	}

	//If user desires, disallow quenching, and don't keep track of a best rotamer -- i.e., run a true Monte Carlo.
	static bool const disallow_quench( truefalseoption("disallow_quench") );

	float prev_beta( 0.0 ), beta( 0.0 ), logZ( 0.0 );

	//outer loop
	for (int nn = 1; nn <= outeriterations; ++nn ){
		setup_temperature(loopenergy,nn);
		//rh std::cout << "Outeriteration: " << nn << "\tTemperature: " << get_temperature() << std::endl;

		if (disallow_quench) set_not_to_quench();

		if (quench()){
			currentenergy = bestenergy_;
			state_on_node = best_state_on_node;
			ig_->set_network_state( state_on_node );
		}

		int inneriterations = get_inneriterations();

		float treshold_for_deltaE_inaccuracy = std::sqrt( get_temperature() );
		ig_->set_errorfull_deltaE_threshold( treshold_for_deltaE_inaccuracy );

		float sumenergy = 0.0;
		int sumcount( 0 );

		//inner loop
		for (int n = 1; n <= inneriterations; ++n ){
			ranrotamer = pick_a_rotamer( n );
			if (ranrotamer != -1){
				moltenres_id = rot_2_moltenres( 1, ranrotamer );
				rotamer_state_on_moltenres = rot_2_moltenres(2, ranrotamer );
				prevrotamer_state = state_on_node(moltenres_id);

				if (rotamer_state_on_moltenres == prevrotamer_state ) continue;//skip iteration

				ig_->consider_substitution( moltenres_id, rotamer_state_on_moltenres,
					delta_energy, previous_energy_for_node);
				//std::cerr << "mres: " << moltenres_id << ", state: ";
				//std::cerr << rotamer_state_on_moltenres << ", deltaE: " << delta_energy << std::endl;

				//std::cout << "Rot: " << ranrotamer << "\t Current Energy: " << previous_energy_for_node + delta_energy << std::endl;

				//bk keep new rotamer if it is lower in energy or accept it at some
				//bk probability if it is higher in energy, if it is the first
				//bk rotamer to be tried at this position automatically accept it.
				if ((prevrotamer_state == 0)||pass_metropolis(previous_energy_for_node,delta_energy))
				{
					//std::cerr << "Accepted" << std::endl;
					currentenergy = ig_->commit_considered_substitution();
					state_on_node(moltenres_id) = rotamer_state_on_moltenres;
					if ((prevrotamer_state == 0)||(currentenergy < bestenergy_)||disallow_quench)
					{
						bestenergy_ = currentenergy;
						best_state_on_node = state_on_node;
					}

					if ( record_annealer_trajectory )
					{
						annealer_trajectory << moltenres_id << " " << rotamer_state_on_moltenres << " A\n";
					}
					if ( save_trajectory() ) {
						save_traj_inx( ranrotamer );
					}

#ifdef GL_GRAPHICS
					using namespace protein_graphics;
					if( false && get_show_repack(n) ) {
						std::cout<<n<<" "<<inneriterations<<std::endl;
						graphics_new_packer_to_misc( *p_rotamer_set_, state_on_node );
						gl_graphics_new_pose(misc::total_residue,true,1);//update the pose frame to show
						gl_graphics_save_scores( currentenergy, false );
						float rms_err (0.0);
						gl_graphics_append_trajectory( currentenergy, rms_err);
					}
#endif
#ifdef BOINC_GRAPHICS
					using namespace protein_graphics;
					if( false && get_show_repack(n) ) {
						graphics_new_packer_to_misc( *p_rotamer_set_, state_on_node );
						float rms_err (0.0);
						append_trajectory( currentenergy, rms_err);
					}
#endif
				}              // end Metropolis criteria
				else if ( record_annealer_trajectory )
				{
					annealer_trajectory << moltenres_id << " " << rotamer_state_on_moltenres << " R\n";
				}
				else if ( save_trajectory() ) {
					save_traj_inx( -1 );
				}

				loopenergy(nn) = currentenergy;
				float temperature = get_temperature();

				if ( calc_rot_freq_ && ( temperature <= calc_freq_temp ) )
				{
					++nsteps;
					for (int ii = 1; ii <= nmoltenres; ++ii )
					{
						int iistate = state_on_node(ii);
						if (iistate != 0)
						{
							++nsteps_for_rot(iistate + p_rotamer_set_->nrotoffset(ii));
						}
						int rot = ranrotamer;
						if ( pKa_mode::get_pKa_flag() && rot != 0 ) {
							float delta_nsteps_for_rot = nsteps_for_rot(rot) - previous_nsteps_for_rot(rot);
							pKa_mode::pKa_packer_flags::store_rotstep_E(rot) += (delta_nsteps_for_rot * currentenergy);
							previous_nsteps_for_rot(rot) = nsteps_for_rot(rot);
						}
					}
				}
			}

			//rhiju keep track of average energy at each temperature.
			if (n > inneriterations*(3/4)){
				sumenergy += currentenergy;
				sumcount++;
			}

		} // end of inneriteration loop
#ifdef GL_GRAPHICS
		using namespace protein_graphics;
		if( get_show_repack() ) {
			graphics_new_packer_to_misc( *p_rotamer_set_, best_state_on_node );
			gl_graphics_new_pose(misc::total_residue,true,1);//update the pose frame to show
			gl_graphics_save_scores( bestenergy_, false );
			float rms_err (0.0);
			gl_graphics_append_trajectory( bestenergy_, rms_err);
		}
#endif
#ifdef BOINC_GRAPHICS
		using namespace protein_graphics;
		if( get_show_repack() ) {
			graphics_new_packer_to_misc( *p_rotamer_set_, best_state_on_node );
			float rms_err (0.0);
			append_trajectory( bestenergy_, rms_err);
		}
#endif

		//rhiju:  integrate  dbeta * <energy> to get partition function
		{
			float const meanenergy = sumenergy / sumcount;
			beta = 1.0 / get_temperature();
			//might be better to use Simpson's rule...
			logZ += (beta - prev_beta) * meanenergy;
			prev_beta = beta;
		}

	} //end of outeriteration loop


	//rhiju: save delG.
	float const delG  = logZ / beta;
	set_delG( delG );

	if ( ig_->any_vertex_state_unassigned() )
	{
		std::cerr << "Critical error -- In FixbbSimAnnealer, one or more vertex states unassigned at annealing's completion." << std::endl;
		std::cerr << "Critical error -- assignment and energy of assignment meaningless" << std::endl;

		FArray1D_int nstates_for_moltenres( p_rotamer_set_->nmoltenres(), 0 );
		for (int ii = 0; ii < num_of_rot_to_pack_; ++ii)
		{
			++nstates_for_moltenres( p_rotamer_set_->report_seqpos( rot_to_pack_[ ii ] ) );
		}

		for (int ii = 1; ii <= p_rotamer_set_->nmoltenres(); ++ii)
		{
			if ( best_state_on_node( ii ) == 0 )
			{
				std::cout << "Molten res " << ii << " (residue " << p_rotamer_set_->moltenres_2_resid( ii );
				std::cout << " ) assigned state 0 despite having " << nstates_for_moltenres( ii ) << " states to choose from" << std::endl;
			}
		}
		assert( ! ig_->any_vertex_state_unassigned() );
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);


	}

	//convert best_state_on_node into best_rotamer_at_seqpos
	for (int ii = 1;ii <= nmoltenres; ++ii){
		int iiresid = p_rotamer_set_->moltenres_2_resid(ii);
		if( iiresid > 0 ){
			bestrotamer_at_seqpos_(iiresid) = best_state_on_node(ii) + p_rotamer_set_->rotindex_offsets(iiresid);
			//std::cout << "bestrotindex: " << bestrotamer_at_seqpos_(iiresid) << "at resid: " << iiresid << std::endl;
		}else if(iiresid < 0 && (size_t(-1*iiresid))<=ligand::ligand_ptr_vector.size() ){
			//kwk may need to recover current position.
			//	std::cout << best_state_on_node(ii)-1 << std::endl;
			ligand::ligand_ptr_vector[-1*iiresid-1]->
    		change_to_ligand_conformation(
													size_t(best_state_on_node(ii)-1));
			ligand::ligand_ptr_vector[-1*iiresid-1]->set_best_coordinates();
		}
	}

	//pKa stuff from Ryan
	for ( int rot = 1; rot <= num_of_rot_to_pack_; ++rot ) {
		if( p_rotamer_set_->moltenres_2_resid(rot_2_moltenres( 1, rot ))> 0){;
			rot_freq_(rot) = static_cast< float > (nsteps_for_rot(rot)) /
				static_cast< float > (nsteps);
		}
		//kwk at the moment rot_freq statistics aren't implemented for non amino acid molten residues.
	}

	if ( pKa_mode::get_pKa_flag() ) {

		if ( pKa_mode::get_output_ensemble_stats_flag() ) {
			// jk Put off dimensioning and setting these until we know we're going to use them
			pKa_mode::pKa_packer_flags::store_rot_delta_energy.dimension( p_rotamer_set_->nrotamers(), 0. );
		}

		if ( calc_rot_freq_ && pKa_mode::pKa_control_flags::output_ensemble_stats_flag ) {
			for ( int rot = 1; rot <= p_rotamer_set_->nrotamers(); ++rot ) {
				pKa_mode::pKa_packer_flags::store_rot_delta_energy(rot) =	( pKa_mode::pKa_packer_flags::store_rotstep_E(rot) / nsteps_for_rot(rot) ) - bestenergy_;
				//std::cout << "rot " << rot << "\t store_rotE " << pKa_mode::pKa_packer_flags::store_rot_delta_energy(rot) << "\t store_rotstep_E " << pKa_mode::pKa_packer_flags::store_rotstep_E(rot) << "\t nsteps_for_rot " << nsteps_for_rot(rot) << std::endl;
			}
		}

		if ( pKa_mode::get_pKa_flag() && calc_rot_freq_ && pKa_mode::get_output_ensemble_stats_flag() ) {
			//rh Warning: if rot_freq is not calculated, there will be errors
			pKa_mode::output_ensemble_stats( rot_freq_, *p_rotamer_set_ );
		}
	}

}

}//end of namespace
