// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: johnk $

// Rosetta Headers
#include "HotspotRotamer.h"
#include "aaproperties_pack.h"
#include "AnkyrinRepeat_functions.h"
#include "AnkyrinRepeat_ns.h"
#include "are_they_neighbors.h"
#include "aromatic_stack.h"
#include "atom_tree_routines.h"
#include "decoystats.h"
#include "decoystats_ns.h"
#include "decoystats_interface.h"
#include "design.h"
#include "design_structure.h"
#include "DesignMap.h"
#include "docking.h"
#include "docking_ns.h"
#include "etable_manager.h"
#include "favor_residue_ns.h"
#include "files_paths.h"
#include "FixbbSimAnnealer.h"
#include "fold_tree.h"
#include "fullatom_sasa.h"
#include "fullatom_setup.h"
#include "fullatom_energy.h"
#include "fullatom_energies.h"
#include "hbonds.h"
#include "hbonds_geom.h"
#include "hbonds_ns.h"
#include "hotspot_ns.h"
#include "input_pdb.h"
#include "InteractionGraphBase.h"
#include "InteractionGraphSupport.h"
#include "job_distributor.h"
#include "LazyInteractionGraph.h"
#include "make_pdb.h"
#include "minimize.h"
#include "misc.h"
#include "monte_carlo.h"
#include "nblist.h"
#include "OnTheFlyInteractionGraph.h"
#include "output_decoy.h"
#include "pack.h"
#include "PackerTask.h"
#include "pack_geom_inline.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_rotamer_trie.h"
#include "pdbstatistics_pack.h"
#include "PDInteractionGraph.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_io.h"
#include "pose_param.h"
#include "PrecomputedInteractionGraph.h"
//#include "prof.h"
#include "random_numbers.h"
#include "refold.h"
#include "RotamerSet.h"
#include "rotamer_functions.h"
#include "rotamer_trie_calc_energies.h"
#include "score.h"
#include "score_data.h"
#include "score_name.h"
#include "SimAnnealerBase.h"
#include "symmetric_design.h"
#include "template_pack.h"
#include "void.h"
#include "water.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>

//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/izstream.hh>




// jk This lightweight class is effectively just a struct containing all the info required for repacking
// jk Note: it's a class just so that handling of pointers is a little safer...

packingPackage::packingPackage() {
	ig = NULL; rotamer_set = NULL; Task = NULL;
}

packingPackage::~packingPackage() {
	delete ig; delete rotamer_set; delete Task;
}

void packingPackage::reset() {
	delete ig; delete rotamer_set; ig = NULL; rotamer_set = NULL;
}

void packingPackage::reset_IG_only() {
	delete ig; ig = NULL;
}


//////  STARTING METHODS FOR rotamerFix  //////


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamerFix::rotamerFix
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
rotamerFix::rotamerFix( int const fix_seqpos, packingPackage * packing_package ) :
	// store input arguments as member data
	starting_design_map( packing_package->Task->get_designmap() ),
	fix_seqpos_(fix_seqpos),
	packing_package_(packing_package)
{

	if ( fix_seqpos_ == 0 ) {
		std::cout << "Seqpos to fix is zero - returning..." << std::endl;
		return;
	}

	if ( ( ! hotspot_ns::preserve_RotamerSet ) || ( packing_package_->rotamer_set == NULL ) ) {
		packing_package_->Task->get_designmap().fix_completely(fix_seqpos_);
		return;
	}

	if ( ! packing_package_->Task->get_designmap().repack_residue(fix_seqpos_) ) return;
	if ( packing_package_->seqpos_with_fixed_rotamers(fix_seqpos_) ) return;

	//	std::cout << "Fixing rotamer at seqpos " << fix_seqpos_ << std::endl;

	// Update allowed_rotamers by excluding rotamers of fix_seqpos_, EXCEPT the current rotamer
	int const rot_to_fix = packing_package_->current_rot_index(fix_seqpos_);
	assert ( packing_package_->rotamer_set->report_seqpos(rot_to_fix) == fix_seqpos_ );
	for ( int i = 1; i <= packing_package_->rotamer_set->nrotamers(); ++i ) {
		if ( i != rot_to_fix ) continue;
		if ( packing_package_->rotamer_set->report_seqpos(i) == fix_seqpos_ ) continue;
		packing_package_->allowed_rotamers(i)=false;
	}

	// Make sure the current rotamer is allowed
	assert ( packing_package_->allowed_rotamers(packing_package_->current_rot_index(fix_seqpos_)) );

	// Maintain a list of fixed rotamers
	packing_package_->seqpos_with_fixed_rotamers(fix_seqpos_)=true;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin rotamerFix::~rotamerFix
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
rotamerFix::~rotamerFix() {

	if ( fix_seqpos_ == 0 ) {
		return;
	}

	if ( ! hotspot_ns::preserve_RotamerSet ) {
		packing_package_->Task->set_designmap(starting_design_map);
		return;
	}

	if ( packing_package_->rotamer_set == NULL ) {
		return;
	}

	packing_package_->seqpos_with_fixed_rotamers(fix_seqpos_)=false;

	// Update allowed_rotamers by allowing rotamers of unfix_seqpos
	for ( int i = 1; i <= packing_package_->rotamer_set->nrotamers(); ++i ) {
		if ( packing_package_->rotamer_set->report_seqpos(i) == fix_seqpos_ ) {
			packing_package_->allowed_rotamers(i)=true;
		}
	}

	return;
}


//////  STARTING METHODS FOR HotspotHbond  //////


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotHbond::apply_atom_tree
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
HotspotHbond::apply_atom_tree( pose_ns::Pose & curr_pose, bool const use_atom_tree ){

	using namespace pose_ns;

	// jk Setup a fold_tree tethering the hotspot residue to the Hbonded residue
	// jk Allow the relative positions across the "jump" (the Hbond) to change

	pose_ns::Fold_tree ft;
	//	ft.clear();

	assert ( misc::domain_end(2) == curr_pose.total_residue() );

	if ( don_seqpos < acc_seqpos ) {
		ft.add_edge(don_seqpos, acc_seqpos, 1); // jump
		ft.add_edge(1,don_seqpos,pose_param::PEPTIDE);
		ft.add_edge(don_seqpos,misc::domain_end(1),pose_param::PEPTIDE);
		ft.add_edge(misc::domain_end(1)+1,acc_seqpos,pose_param::PEPTIDE);
		ft.add_edge(acc_seqpos, misc::domain_end(2),pose_param::PEPTIDE);
		ft.set_jump_atoms( 1, don_Hbond_atm, acc_Hbond_atm, true );  // true for a jump, false for a bond
	} else {
		ft.add_edge(acc_seqpos, don_seqpos, 1); // jump
		ft.add_edge(1,acc_seqpos,pose_param::PEPTIDE);
		ft.add_edge(acc_seqpos,misc::domain_end(1),pose_param::PEPTIDE);
		ft.add_edge(misc::domain_end(1)+1,don_seqpos,pose_param::PEPTIDE);
		ft.add_edge(don_seqpos, misc::domain_end(2),pose_param::PEPTIDE);
		ft.set_jump_atoms( 1, acc_Hbond_atm, don_Hbond_atm, true );  // true for a jump, false for a bond
	}

	ft.reorder(1);
	curr_pose.set_fold_tree(ft);

	if ( use_atom_tree ) {
		curr_pose.setup_atom_tree();
	}

	curr_pose.set_allow_jump_move( true );

	return;

}


//////  DONE WITH METHODS FOR HotspotHbond  //////


//////  STARTING METHODS FOR HotspotRotamer  //////

////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::HotspotRotamer (constructor)
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
HotspotRotamer::HotspotRotamer(
  int const in_aa,
	int const in_aav,
	int const in_seqpos,
	FArray1Da_float in_rchi,
	FArray2Da_float in_rotcoord,
	FArray1Da_float in_rotactcoord
){

	using namespace param;

	initialize( in_seqpos, in_aa, in_aav, 1 );

	in_rchi.dimension( MAX_CHI );
	in_rotcoord.dimension( 3, MAX_ATOM() );
	in_rotactcoord.dimension( 3 );

	rchi_ = in_rchi;
	rotcoord_ = in_rotcoord;
	rotactcoord_ = in_rotactcoord;

	partner_seqpos_ = 0;
	stack_seqpos_ = 0;

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::HotspotRotamer (constructor)
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
HotspotRotamer::HotspotRotamer(
  int const in_aa,
	int const in_aav,
	int const in_seqpos,
	FArray1Da_float in_rchi
){

	using namespace param;

	initialize( in_seqpos, in_aa, in_aav, 1 );

	in_rchi.dimension( MAX_CHI );

	rchi_ = in_rchi;

	fill_coors_from_chi();

	partner_seqpos_ = 0;
	stack_seqpos_ = 0;

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::explode
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::explode( std::list < HotspotRotamer > & exploded_rotamers ) {

	// Use this rotamer to generate many others with similar chi angles,
	// return these as a list

	float const chi1_min_offset(-10.);
	float const chi1_max_offset(10.);
	int const num_steps1(5);

	float const chi2_min_offset(-10.);
	float const chi2_max_offset(10.);
	int const num_steps2(5);

	float const step_size1 = (chi1_max_offset - chi1_min_offset) / num_steps1;
	float const step_size2 = (chi2_max_offset - chi2_min_offset) / num_steps2;

	float curr_chi1 = rchi_(1) + chi1_min_offset;
	for ( int i = 0; i < num_steps1; ++i ) {
		HotspotRotamer first_move(*this);
		first_move.change_chi_angle_update_coors(1,curr_chi1);
		float curr_chi2 = rchi_(2) + chi2_min_offset;
		for ( int j = 0; j < num_steps2; ++j ) {
			HotspotRotamer second_move(first_move);
			second_move.change_chi_angle_update_coors(2,curr_chi2);
			exploded_rotamers.push_back(second_move);
			curr_chi2 += step_size2;
		}
		curr_chi1 += step_size1;
	}

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::hr_bump_check
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::hr_bump_check(
  pose_ns::Pose const & pose,
  bool const self,
	bool const interface,
  float const hr_bump_thres
){

	// jk Ensure that this rotamer does not clash with the "self" backbone
	// jk Note: returns true for "clash", false for "no clash"
	// jk Note: assumes that "self" is chain 1, and "interface" is chain 2
	// jk Note: in interface mode, assumes that "interface_residue" array has been set!

	using namespace design; // interface_residue is here
	using namespace param_aa;
	using namespace param_pack;

	// Set the residues to loop over when checking for clashes
	int startres = 1;
	int endres = misc::domain_end(2);
	if ( !self ) {
		if ( seqpos_ <= misc::domain_end(1) ) {
			startres = misc::domain_end(1)+1;
		} else {
			endres = misc::domain_end(1);
		}
	}
	if ( ! interface ) {
		if ( seqpos_ <= misc::domain_end(1) ) {
			endres = misc::domain_end(1);
		} else {
			startres = misc::domain_end(1)+1;
		}
	}

	float bumpenergy = 0.0;
	for ( int bbres = startres; bbres <= endres; ++bbres ) {
		float dis;
		// for distance check, use CB of the hotspot residue and CA of bbres
		distance_bk(rotcoord_(1,5),pose.full_coord()(1,2,bbres),dis);
		if ( dis > 10 ) continue;
		float solvE, atrE, repE, elecE, h2oE, h2ohbE;
		get_sc_bbE(aa_,aav_,pose.res(bbres),pose.res_variant(bbres),rotcoord_,
							 pose.full_coord()(1,1,bbres),seqpos_,bbres,solvE,atrE,repE,elecE);
		get_sc_bb_h2oE(aa_,aav_,pose.res(bbres),pose.res_variant(bbres),rotcoord_,
									 pose.full_coord()(1,1,bbres),seqpos_,bbres,h2oE,h2ohbE);
		float const new_bumpenergy = pack_wts.Watr()*atrE+pack_wts.Wrep()*repE+pack_wts.Wh2o()*h2oE;
		if ( new_bumpenergy > 0. ) {
			bumpenergy += new_bumpenergy;
			if ( bumpenergy > hr_bump_thres ) {
				return true;
			}
		}
	}

	FArray1D_float dummyactcoord1( 3 );
	FArray1D_float dummyactcoord2( 3 );
	for ( int i = 1; i <= 3; ++i ) {
		dummyactcoord1(i) = 1.0;
		dummyactcoord2(i) = 3.0;
	}

	for ( int scres = startres; scres <= endres; ++scres ) {
		//			if ( interface && ! interface_residue(scres) ) continue;
		if ( seqpos_ == scres ) continue;
		int scaa = pose.res(scres);

		float dis;
		// for distance check, use CB of the hotspot residue and CA of scres
		distance_bk(rotcoord_(1,5),pose.full_coord()(1,2,scres),dis);
		if ( dis > 18 ) continue;
		float solvE, atrE, repE, pairE, plane_totalE, elecE, h2oE, h2ohbE;
		get_sc_scE(aa_,aav_,rotcoord_,scaa,pose.res_variant(scres),
							 pose.full_coord()(1,1,scres),dummyactcoord1,dummyactcoord2,seqpos_,scres,
							 solvE,atrE,repE,pairE,plane_totalE,elecE);
		get_sc_sc_h2oE(aa_,aav_,rotcoord_,scaa,pose.res_variant(scres),
									 pose.full_coord()(1,1,scres),seqpos_,scres,h2oE,h2ohbE);
		float const new_bumpenergy = pack_wts.Watr()*atrE+pack_wts.Wrep()*repE+pack_wts.Wh2o()*h2oE;
		if ( new_bumpenergy > 0. ) {
			bumpenergy += new_bumpenergy;
			if ( bumpenergy > hr_bump_thres ) {
				return true;
			}
		}
	}

	return false;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::find_self_Hbond
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::find_self_Hbond( pose_ns::Pose const & pose ) {

	// jk Check whether this rotamer makes an Hbond to "self"

	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

	// jk ANKYRIN SPECIFIC REQUIREMENT
	// jk Look for the special self-satisfying Hbond

	bool const disallow_backbone_Hbond = true;
	bool const disallow_sidechain_Hbond = false;

	int fixed_seqpos;
	if ( aa_ == param_aa::aa_trp ) {
		fixed_seqpos = seqpos_ + 29;
		if ( fixed_seqpos <= misc::domain_end(2) ) {
			if ( AnkyrinRepeat_ns::AspAsn_seqpos(fixed_seqpos) ) {
				int const fixedaa = pose.res(fixed_seqpos);
				int const fixedaav = pose.res_variant(fixed_seqpos);
				if ( test_hotspot_HB_ene(fixed_seqpos,fixedaa,fixedaav,pose.full_coord()(1,1,fixed_seqpos),
						hotspot_ns::self_satisfying_hb_thres, disallow_backbone_Hbond, disallow_sidechain_Hbond) ) {
					partner_seqpos_ = fixed_seqpos;
					return true;
				}
			}
		}
	}

	fixed_seqpos = seqpos_ - 9;
	if ( fixed_seqpos > misc::domain_end(1) ) {
		if ( AnkyrinRepeat_ns::AspAsn_seqpos(fixed_seqpos) ) {
			int const fixedaa = pose.res(fixed_seqpos);
			int const fixedaav = pose.res_variant(fixed_seqpos);
			if ( test_hotspot_HB_ene(fixed_seqpos,fixedaa,fixedaav,pose.full_coord()(1,1,fixed_seqpos),
					hotspot_ns::self_satisfying_hb_thres, disallow_backbone_Hbond, disallow_sidechain_Hbond) ) {
				partner_seqpos_ = fixed_seqpos;
				return true;
			}
		}
	}

	return false;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::find_interface_Hbond
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::find_interface_Hbond( pose_ns::Pose const & pose, bool const use_strict_criteria ) {

	// jk Check whether this rotamer makes an Hbond to the partner's backbone
	// jk Note: consider only "interface_residues" of the partner

	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

	// jk ANKYRIN SPECIFIC REQUIREMENT
	// jk If hotspot is on second chain (ie. the AR), ignore the hotspot_ns::require_Hbond_to_sidechain flag

	bool disallow_backbone_Hbond = false;
	int startres = 1;
	int endres = misc::domain_end(1);
	if ( seqpos_ <= misc::domain_end(1) ) {
		disallow_backbone_Hbond = hotspot_ns::require_Hbond_to_sidechain;
		startres = misc::domain_end(1)+1;
		endres = misc::domain_end(2);
	}

	//	int const best_AspAsn = find_best_available_AspAsn( pose );

	for ( int fixedres = startres; fixedres <= endres; ++fixedres ) {
		int const fixedaa = pose.res(fixedres);
		int const fixedaav = pose.res_variant(fixedres);

		if ( ! interface_residue(fixedres) ) continue;

		// jk ANKYRIN SPECIFIC REQUIREMENT
		// if we require a sidechain hotspot Hbond, require that it's a conserved AR Asp/Asn
		// if backbone Hbonds are allowed, though, the acceptor can be anything

		if ( disallow_backbone_Hbond && ( ! AnkyrinRepeat_ns::AspAsn_seqpos(fixedres) ) ) continue;
		//		if ( disallow_backbone_Hbond && ( fixedres != best_AspAsn ) ) continue;

		bool const disallow_sidechain_Hbond = false;

		float HB_thres = hotspot_ns::permissive_hb_thres;
		if ( use_strict_criteria ) {
			HB_thres = hotspot_ns::hs_hb_thres;
		}

		if ( test_hotspot_HB_ene(fixedres,fixedaa,fixedaav,pose.full_coord()(1,1,fixedres),
					 HB_thres, disallow_backbone_Hbond, disallow_sidechain_Hbond) ) {
			partner_seqpos_ = fixedres;
			return true;
		}

	}

	return false;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::find_interface_stack
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::find_interface_stack( pose_ns::Pose const & pose, bool const use_strict_criteria, bool const silent_mode ) {

	// jk Check whether this (AR) rotamer is close to an aromatic on the partner protein

	using namespace aaproperties_pack;
	using namespace design;
	using namespace param_aa;

	// jk Note: interaction energy (without repulsive) must be at least this good
	float interaction_thres = -1.0;
	if ( use_strict_criteria ) {
		interaction_thres = -1.5;
	}
	stack_seqpos_ = 0;
	float best_interaction = 999.;
	bool found_stack = false;

	int startres = 1;
	int endres = misc::domain_end(1);
	for ( int fixedres = startres; fixedres <= endres; ++fixedres ) {
		int const fixedaa = pose.res(fixedres);
		int const fixedaav = pose.res_variant(fixedres);
		if ( ! interface_residue(fixedres) ) continue;

		float solvE,atrE,repE,elecE,pairE,plane_totalE;
		FArray1D_float dummyactcoord1( 3, 1.0 );
		FArray1D_float dummyactcoord2( 3, 3.0 );
		get_sc_scE(aa_,aav_,rotcoord_,fixedaa,fixedaav,pose.full_coord()(1,1,fixedres),
			dummyactcoord1,dummyactcoord2,seqpos_,fixedres,solvE,atrE,repE,pairE,plane_totalE,elecE);

		float const curr_interaction = atrE + plane_totalE;
		if ( curr_interaction < interaction_thres ) {
			found_stack = true;
		}
		if ( curr_interaction < best_interaction ) {
			best_interaction = curr_interaction;
			stack_seqpos_ = fixedres;
		}

	}

	if ( found_stack && ( ! silent_mode ) ) {
		std::cout << "Found a stack involving seqpos " << stack_seqpos_ << std::endl;
	}
	return found_stack;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::valid_self_OneBody
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::valid_self_OneBody( pose_ns::Pose const & pose, bool const silent_mode ) {

	// To be safe, verify that the rotcoord_ coors match those in the pose
	// This is important because clash check, Hbond check, etc. use coors from both rotcoord_ and pose
	// Note: this check is expected to fail during the current rotamer-building process, since we don't
	//         actually put the rotamers on before doing this check...
	bool verify_coors = false;
	if ( verify_coors ) {
		for ( int j = 1; j <= aaproperties_pack::natoms(aa_,aav_); ++j ) {
			for ( int k = 1; k <= 3; ++k ){
				if ( std::abs( get_rotcoord(k,j) - pose.full_coord()(k,j,seqpos_) ) > 1E-8 ) {
					std::cout << "ERROR: rotcoord_ does not match pose coors!!" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

  float const hr_bump_thres = 0.5;
	if ( hr_bump_check(pose, true, false, hr_bump_thres) ) {
		if ( ! silent_mode ) std::cout << "valid_self_OneBody fails bump check" << std::endl;
		return false;
	}
	if ( hotspot_ns::hotspot_on_AR ) {
		if ( ! find_self_Hbond( pose ) ) {
			if ( ! silent_mode ) std::cout << "valid_self_OneBody fails self-Hbond test" << std::endl;
			return false;
		}
	}
	return true;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::valid_interface_OneBody
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::valid_interface_OneBody( pose_ns::Pose const & pose, bool const use_strict_criteria, bool const silent_mode ) {

	// To be safe, verify that the rotcoord_ coors match those in the pose
	// This is important because clash check, Hbond check, etc. use coors from both rotcoord_ and pose
	// Note: this check is expected to fail during the current rotamer-building process, since we don't
	//         actually put the rotamers on before doing this check...
	bool verify_coors = false;
	if ( verify_coors ) {
		for ( int j = 1; j <= aaproperties_pack::natoms(aa_,aav_); ++j ) {
			for ( int k = 1; k <= 3; ++k ){
				if ( std::abs( get_rotcoord(k,j) - pose.full_coord()(k,j,seqpos_) ) > 1E-8 ) {
					std::cout << "ERROR: rotcoord_ does not match pose coors!!" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
		}
	}

	if ( use_strict_criteria ) {
		float const hr_bump_thres = 0.5;
		if ( hr_bump_check( pose, false, true, hr_bump_thres ) ) {
			if ( ! silent_mode ) std::cout << "valid_interface_OneBody fails bump check" << std::endl;
			return false;
		}
	}
	if ( ! test_for_hotspot_burial(pose) ) {
		if ( ! silent_mode ) std::cout << "valid_interface_OneBody fails burial test" << std::endl;
		return false;
	}
	if ( ! hotspot_ns::hotspot_on_AR ) {
		if ( ! find_interface_Hbond( pose, use_strict_criteria ) ) {
			if ( ! silent_mode ) std::cout << "valid_interface_OneBody fails Hbond test" << std::endl;
			return false;
		}
	} else {
		if ( ! find_interface_stack( pose, use_strict_criteria, silent_mode ) ) {
			if ( ! silent_mode ) std::cout << "valid_interface_OneBody fails stack test" << std::endl;
			return false;
		}
	}
	return true;

}



////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::test_hotspot_HB_ene
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::test_hotspot_HB_ene(
  int const fixedres,
  int const fixedaa,
  int const fixedaav,
	FArray2Da_float fixedxyz,
	float const HB_thres,
	bool const disallow_backbone,
	bool const disallow_sidechain
) {

	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace param_aa; // aa_pro is here
	using namespace param;

	fixedxyz.dimension( 3, MAX_ATOM() );

	if ( aav_ > 1 ) return false; // for now, don't allow hotspot rotamer variants

	// Check the CA-CA distance first
	float dis;
	distance_bk(rotcoord_(1,2),fixedxyz(1,2),dis);
	if ( dis > 18 ) return false;

	// no Hbonds longer than sqrt of this (the square)
	float const dist_cut2 = 27.;   // 5.2*5.2

	int fixed_startnum = 1;
	if ( disallow_backbone ) fixed_startnum = 2;

	// Hotspot acting as an acceptor
	for ( int anum = 2; anum <= nacceptors(aa_,aav_); ++anum ) {

		if ( fixedaa == aa_pro ) continue;
		int fixed_endnum = nH_polar(fixedaa,fixedaav);
		if ( disallow_sidechain ) fixed_endnum = 1;

		int const aatm = accpt_pos(anum,aa_,aav_);
		int const base_atm = atom_base(aatm,aa_,aav_);
		int const base2_atm = abase2(aatm,aa_,aav_);
		FArray1D_float hotspot_acceptor( 3 );
		FArray1D_float hotspot_base_atm( 3 );
		FArray1D_float hotspot_base2_atm( 3 );
		for ( int i = 1; i <= 3; ++i ) {
			hotspot_acceptor(i) = rotcoord_(i,aatm);
			hotspot_base_atm(i) = rotcoord_(i,base_atm);
			hotspot_base2_atm(i) = rotcoord_(i,base2_atm);
		}

		// jk For now, Tyr OH is the only allowed acceptor
		if ( fixedaa != aa_pro ) {
			for ( int hnum = fixed_startnum; hnum <= fixed_endnum; ++hnum ) {
				int const dhatm = Hpos_polar(hnum,fixedaa,fixedaav);
				int const datm = atom_base(dhatm,fixedaa,fixedaav);

				// if a backbone donor participates in a backbone-backbone Hbond,
				// don't allow an Hbond to it
				if ( ( datm == 1 ) && hbond_set.BBhbToNH_exists(fixedres) ) continue;

				// if the distance is > 5.2 A from the base atom of the donor,
				// there's no Hbond
				float base_dis2;
				distance2_bk(fixedxyz(1,datm),hotspot_acceptor,base_dis2);
				if ( base_dis2 > dist_cut2 ) continue;

				// if distance to donor base atom is less than distance to donor H
				// there's no Hbond
				float hdis2;
				distance2_bk(fixedxyz(1,dhatm),hotspot_acceptor,hdis2);
				if ( hdis2 > base_dis2 ) continue;

				// if distance to acceptor base atom is less than distance to acceptor
				// there's no Hbond
				float bdis2;
				distance2_bk(fixedxyz(1,dhatm),hotspot_base_atm,bdis2);
				if ( bdis2 < hdis2 ) continue;

				float HB_ene;
				HBEvalType const hbe_type = hbond_evaluation_type(datm, fixedaa, fixedaav, aatm, aa_, aav_);
				hb_energy_deriv(hbe_type,fixedxyz(1,datm),fixedxyz(1,dhatm),
												hotspot_acceptor,hotspot_base_atm,hotspot_base2_atm,HB_ene);

				if ( HB_ene < HB_thres ) {
					hsr_Hbond.set_all(fixedres,dhatm,datm,seqpos_,aatm,base_atm,true);
					return true;
				}

			} // for loop over each donor
		}
	}


	// Hotspot acting as a donor
	for ( int dnum = 2; dnum <= nH_polar(aa_,aav_); ++dnum ) {

		int fixed_endnum = nacceptors(fixedaa,fixedaav);
		if ( disallow_sidechain ) fixed_endnum = 1;

		for ( int anum = fixed_startnum; anum <= fixed_endnum; ++anum ) {

			int const dhatm = Hpos_polar(dnum,aa_,aav_);
			int const datm = atom_base(dhatm,aa_,aav_);
			FArray1D_float hotspot_Hdon_atm( 3 );
			FArray1D_float hotspot_base_atm( 3 );
			for ( int i = 1; i <= 3; ++i ) {
				hotspot_Hdon_atm(i) = rotcoord_(i,dhatm);
				hotspot_base_atm(i) = rotcoord_(i,datm);
			}

			int const aatm = accpt_pos(anum,fixedaa,fixedaav);
			int const base_atm = atom_base(aatm,fixedaa,fixedaav);
			int const base2_atm = abase2(aatm,fixedaa,fixedaav);

			// if a backbone acceptor participates in a backbone-backbone Hbond,
			// there's no Hbond
			if ( ( aatm == 4 ) && hbond_set.BBhbToO_exists(fixedres) ) continue;

			// if the distance is > 5.2 A from the base atom of the donor,
			// there's no Hbond
			float base_dis2;
			distance2_bk(fixedxyz(1,aatm),hotspot_base_atm,base_dis2);
			if ( base_dis2 > dist_cut2 ) continue;

			// if distance to donor base atom is less than distance to donor H,
			// there's no Hbond
			float hdis2;
			distance2_bk(fixedxyz(1,aatm),hotspot_Hdon_atm,hdis2);
			if ( hdis2 > base_dis2 ) continue;

			// if distance to acceptor base atom is less than distance to acceptor,
			// there's no Hbond
			float bdis2;
			distance2_bk(fixedxyz(1,base_atm),hotspot_Hdon_atm,bdis2);
			if ( bdis2 < hdis2 ) continue;

			float HB_ene;
			HBEvalType const hbe_type = hbond_evaluation_type(datm, aa_, aav_, aatm, fixedaa, fixedaav);
			hb_energy_deriv(hbe_type,hotspot_base_atm,hotspot_Hdon_atm,
											fixedxyz(1,aatm),fixedxyz(1,base_atm),fixedxyz(1,base2_atm),HB_ene);

			if ( HB_ene < HB_thres ) {
				hsr_Hbond.set_all(seqpos_,dhatm,datm,fixedres,aatm,base_atm,false);
				return true;
			}

		} // for loop over each donor

	}

	return false;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::test_for_hotspot_burial
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool HotspotRotamer::test_for_hotspot_burial( pose_ns::Pose const & pose ) {

	if ( count_interface_neighbors(pose,seqpos_) >= hotspot_ns::burial_neighbor_thres ) return true;
	return false;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin count_interface_neighbors
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int count_interface_neighbors( pose_ns::Pose const & pose, int const seqpos ) {

	// jk Count how many C-betas on the other chain are nearby,
	// use this as a quick way to assess burial

	// Set the residues to loop over
	int startres = 1;
	int endres = pose.total_residue();
	if ( seqpos <= misc::domain_end(1) ) {
		startres = misc::domain_end(1)+1;
	} else {
		endres = misc::domain_end(1);
	}

	float const dis2_cutoff = { 200.0 };
	int num_neighbors(0);
	int self_aa = pose.res(seqpos);
	for ( int res1 = startres; res1 <= endres; ++res1 ) {
		float dis2;
		bool neighbor;
		are_they_neighbors(self_aa,pose.res(res1),pose.full_coord()(1,1,seqpos),
											 pose.full_coord()(1,1,res1),dis2,neighbor);
		if ( dis2 < dis2_cutoff ) ++num_neighbors;
	}

	return num_neighbors;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::find_best_available_AspAsn
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int HotspotRotamer::find_best_available_AspAsn( pose_ns::Pose const & pose ) const {

	using namespace AnkyrinRepeat_ns;

	// Return the seqpos of the most buried AspAsn not currently involved in a hotspot Hbond
	update_AR_interface_neighbors( pose );
	int best_available_seqpos = 0;
	int best_num_neighbors = 0;
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		// Note: partner_seqpos_ is zero before we've decided on the hotspot,
		// then takes a value once we've done this
		if ( seqpos == partner_seqpos_ ) continue;
		if ( ! AspAsn_seqpos(seqpos) ) continue;
		if ( AspAsn_neighbors(seqpos) > best_num_neighbors ) {
			best_num_neighbors = AspAsn_neighbors(seqpos);
			best_available_seqpos = seqpos;
		}
	}

	if ( best_num_neighbors == 0 ) return partner_seqpos_;

	return best_available_seqpos;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::write_outpdb
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::write_outpdb( pose_ns::Pose const & pose, std::string const tag ) const {

	std::string outpdbname = get_outpdb_fname(tag);
	std::cout << "writing output PDBfile " << outpdbname << std::endl;

	pose.copy_to_misc();

	score_set_new_pose();
	mc_global_track::mc_score::score = score12();
	calc_aprox_sasa();
	calc_sasa_pack_score(pose.total_residue(), pose.res(), pose.res_variant(), pose.full_coord());
	calc_sasa_prob();

	decoystats_store_decoy();

	make_pdb( get_cat_pdb_stream( outpdbname ),true);

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::get_outpdb_fname
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
std::string HotspotRotamer::get_outpdb_fname( std::string const tag ) const {

	using namespace files_paths;
	using namespace hotspot_ns;

	std::string fullname;

	int const num_length = { 6 };

	bool exists(true);
	while ( exists ) {

		// Check for valid output_decoy_index_num
		if ( output_decoy_index_num > 999999 ) {
			std::cout << "error - too many output pdbs" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// Create the filename
		fullname = pdb_out_path;
		if ( pdbout == "des" ) {
			fullname += output_file;
		} else {
			fullname += pdbout;
		}
		std::ostringstream seqpos_stream;
		seqpos_stream << seqpos_;
		fullname += "_hs" + seqpos_stream.str() +
			"_index" + lead_zero_string_of( output_decoy_index_num, num_length ) +
			'.' + tag + ".pdb";
		if (output_pdb_gz) fullname += ".gz";

		// Check whether this one exists - if so, increment output_decoy_index_num
		utility::io::izstream pdb_out_checkstream;
		pdb_out_checkstream.open( fullname, std::ios_base::in );
		if ( pdb_out_checkstream ) { // File exists: Increment name
			++output_decoy_index_num;
		} else {
			exists = false;
		}
		pdb_out_checkstream.close();
		pdb_out_checkstream.clear();

	}

	return fullname;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::incorporate_hotspot
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::incorporate_hotspot(
	pose_ns::Pose const & start_pose,
	packingPackage & hsr_surrounding_packing_package
) {

	// jk Build perturbed binding modes around this hotspot rotamer,
	// jk and redesign each of these while keeping the hotspot fixed.
	// jk Write out the final structure IF it contains a hotspot

	// jk Note: when redesigning, we will NOT update the list of interface residues!

	using namespace hotspot_ns;
	using namespace param;
	using namespace param_aa;
	using namespace pose_ns;

	// jk Copy this to a "working" pose
	Pose curr_pose;
	curr_pose = start_pose;

	// jk Put the coors of the current hotspot rotamer into curr_pose
	curr_pose.copy_sidechain(seqpos_,aa_,aav_,rotcoord_);
	curr_pose.copy_to_misc();

	// Check whether we have the beginnings of a stack
	//	score_set_new_pose();
	//	score12();
	//	if ( fullatom_energies::plane_plane(stack_seqpos(curr_pose),seqpos_) > -0.0001 ) {
	//		std::cout << "Jumping out due to insufficient plane term before minimization" << std::endl;
	//		return;
	//	}

	rotamerFix partner_fix( partner_seqpos_, &hsr_surrounding_packing_package );

	rotamerFix stack_fix( stack_seqpos(curr_pose), &hsr_surrounding_packing_package );

	// jk Build a poly-Ala interface here (note: we may have done this already if we're not buiding a stack),
	{ // "build Ala interface" scope
		// Create PackerTask and setup values before pass into pack_rotamers
		//		int const orig_fix_target_seq = design::fix_target_seq;
		//		design::fix_target_seq = 0;
		files_paths::query_defined = false;
		PackerTask ala_interface_Task(curr_pose);
		ala_interface_Task.set_task("design",false,design::interface_residue,true,false);
		design::build_Ala_interface = true;
		ala_interface_Task.setup_residues_to_vary();
		ala_interface_Task.get_designmap().fix_completely(seqpos_);
		ala_interface_Task.get_designmap().fix_completely(partner_seqpos_);
		ala_interface_Task.get_designmap().fix_completely(stack_seqpos(curr_pose));
		std::cout << "Rebuilding poly-Ala interface around stack" << std::endl;
		pack_rotamers( curr_pose, ala_interface_Task);
		curr_pose.copy_to_misc();
		design::build_Ala_interface = false;
		files_paths::query_defined = false;
		//		design::fix_target_seq = orig_fix_target_seq;
	} // "build Ala interface" scope

	// jk Jump out if we have a really big clash, since this could cause problems in the minimization
	if ( hr_bump_check( curr_pose, false, true, 1000. ) ) {
		std::cout << "Jumping out due to bad starting clash" << std::endl;
		return;
	}

	// jk Minimize the hotspot Hbond
	std::cout << "Minimizing poly-Ala interface" << std::endl;
	//	PROF_START( prof::RB_MINIMIZE );

	bool allow_chi_move = false;
	if ( hotspot_ns::hotspot_on_AR ) allow_chi_move = true;
	hsr_minimize( curr_pose, *(hsr_surrounding_packing_package.Task), allow_chi_move, false );
	//	PROF_STOP( prof::RB_MINIMIZE );

	// Check whether we have the beginnings of a stack
	curr_pose.copy_to_misc();
	score_set_new_pose();
	score12();

	//	if ( fullatom_energies::plane_plane(stack_seqpos(curr_pose),seqpos_) > -0.0001 ) {
	//		std::cout << "Jumping out due to insufficient plane term after minimization" << std::endl;
	//		return;
	//	}

	// jk Check for a clash between the backbones (or Ala sidechains)
	float const delta_rep_thres(1.);
	float delta_rep(0.);
	Score_weight_map rep_weight_map;
	rep_weight_map.clear();
	rep_weight_map.set_weight( FA_REP, 1.0 );
	curr_pose.score( rep_weight_map );
	for ( int i = 1; i <= misc::domain_end(1); ++i ) {
		for ( int j = misc::domain_end(1)+1; j <= curr_pose.total_residue(); ++j ) {
			delta_rep += fullatom_energies::rep_pair(i,j);
		}
	}
	if ( delta_rep > delta_rep_thres ) {
		std::cout << "Jumping out due to clash after initial poly-Ala minimization" << std::endl;
		return;
	}

	// Recheck "validity" of hotspot residue (buried, no clash, Hbond)
	if ( ! valid_self_OneBody( curr_pose, false) ) {
		std::cout << "Jumping out due to lack of valid self hotspot after initial poly-Ala minimization" << std::endl;
		return;
	}
	if ( ! valid_interface_OneBody(curr_pose, true, false) ) {
		std::cout << "Jumping out due to lack of valid interface hotspot after initial poly-Ala minimization" << std::endl;
		return;
	}

	// jk Jump out if we have delta_uns
	std::cout << "Testing for UNS in minimized interface containing hotspot rotamer" << std::endl;
	interface_ds hsr_interface(&curr_pose);
	if ( hsr_interface.count_interface_delta_group_uns_list() > binding_mode_group_uns_thres ) {
		std::cout << "Jumping out due to delta_group_uns" << std::endl;
		return;
	}
	std::cout << "Interface containing hotspot rotamer passes delta_uns criterion" << std::endl;

	// jk ANKYRIN SPECIFIC REQUIREMENT
	// jk Require an Hbond from the first chain to an Asp/Asn on the second chain
	int const best_AspAsn = find_best_available_AspAsn( curr_pose );
	if ( ( best_AspAsn != partner_seqpos_ ) &&
			 ( std::abs(best_AspAsn - partner_seqpos_) != AnkyrinRepeat_ns::module_length ) ) {
		std::cout << "Jumping out - best available AspAsn is not exactly one repeat away from hotspot partner" << std::endl;
		return;
	}

	// jk Check that there are no additional buried Asp/Asn
	//	for ( int seqpos = 1; seqpos <= curr_pose.total_residue(); ++seqpos ) {
	//		if ( seqpos == partner_seqpos_ ) continue;
	//		if ( seqpos == best_AspAsn ) continue;
	//		if ( ! AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) continue;
	//		if ( AnkyrinRepeat_ns::AspAsn_neighbors(seqpos) > max_unplanned_AspAsn_neighbors ) {
	//			std::cout << "Jumping out due to unanticipated burial of additional Asp/Asn at seqpos " << seqpos <<
	//				" (counted " << AnkyrinRepeat_ns::AspAsn_neighbors(seqpos) << " neighbors)." << std::endl;
	//			return;
	//		}
	//	}

	transform_rotamers_for_new_binding_mode( curr_pose, hsr_surrounding_packing_package );

	// Do a design without trying a second Hbond
	std::cout << "Starting with a design that does not incorporate a second Hbond" << std::endl;
	float const delta_sasa = loop_over_AR_modules( curr_pose, hsr_surrounding_packing_package );
	if ( delta_sasa > delta_sasa_thres ) {
		std::cout << "delta_sasa_thres exceeded - moving on to next hotspot rotamer" << std::endl;
		return;
	}

	{ // scope do a design with Ser replacing the next Asp/Asn (instead of a second Hbond to the Asp/Asn)
		std::cout << "Starting with a design with Ser instead of a second Hbond" << std::endl;
		PackerTask conv_Task(curr_pose);
		FArray1D_bool allow_repack( curr_pose.total_residue(), false );
		allow_repack( best_AspAsn ) = true;
		conv_Task.set_task("packrot",false,allow_repack,false,false);
		conv_Task.setup_residues_to_vary();
		conv_Task.get_designmap().fix_completely(best_AspAsn);
		conv_Task.get_designmap().set( best_AspAsn, param_aa::aa_ser );
		pack_rotamers( curr_pose, conv_Task);
		curr_pose.copy_to_misc();
		loop_over_AR_modules( curr_pose, hsr_surrounding_packing_package );
	}

	{ // scope do a design with Thr replacing the next Asp/Asn (instead of a second Hbond to the Asp/Asn)
		std::cout << "Starting with a design with Thr instead of a second Hbond" << std::endl;
		PackerTask conv_Task(curr_pose);
		FArray1D_bool allow_repack( curr_pose.total_residue(), false );
		allow_repack( best_AspAsn ) = true;
		conv_Task.set_task("packrot",false,allow_repack,false,false);
		conv_Task.setup_residues_to_vary();
		conv_Task.get_designmap().fix_completely(best_AspAsn);
		conv_Task.get_designmap().set( best_AspAsn, param_aa::aa_thr );
		pack_rotamers( curr_pose, conv_Task);
		curr_pose.copy_to_misc();
		loop_over_AR_modules( curr_pose, hsr_surrounding_packing_package );
	}

	FArray1D_bool valid_partner_positions(curr_pose.total_residue(), false);
	valid_partner_positions(best_AspAsn) = true;

	Pose pre_second_Hbond_pose;
	pre_second_Hbond_pose = curr_pose;
	DesignMap pre_second_Hbond_DesignMap = hsr_surrounding_packing_package.Task->get_designmap();

	// Look for a second Hbond rotamer

	{ // conserved Asp scope
		std::cout << "Finding second Hbond donors to conserved Asp" << std::endl;
		//		PROF_START( prof::FINDING_SECOND_HBOND );
		RotamerSet second_Hbond_rotamer_set;
		convert_Asn_to_Asp( curr_pose, best_AspAsn );

		PackerTask second_Hbond_Task = *(hsr_surrounding_packing_package.Task);
		second_Hbond_Task.get_designmap().fix_completely(best_AspAsn);

		keep_rotamer_from_pose( best_AspAsn, curr_pose, hsr_surrounding_packing_package );

		rotamerFix secondary_AspAsn( best_AspAsn, &hsr_surrounding_packing_package );

		find_second_Hbond( curr_pose, second_Hbond_rotamer_set, valid_partner_positions, second_Hbond_Task );
		std::cout << "Found " << second_Hbond_rotamer_set.nrotamers() <<
			" candidate donating rotamers for the second Hbond" << std::endl;
		//		PROF_STOP( prof::FINDING_SECOND_HBOND );
		for ( int second_Hbond_num = 1; second_Hbond_num <= second_Hbond_rotamer_set.nrotamers(); ++second_Hbond_num ) {
			float const delta_sasa = loop_over_second_Hbond(second_Hbond_num, pre_second_Hbond_pose,
				second_Hbond_rotamer_set, hsr_surrounding_packing_package );
			if ( delta_sasa > delta_sasa_thres ) {
				std::cout << "delta_sasa_thres exceeded - moving on to next hotspot rotamer" << std::endl;
				break;
			}
		}

	} // conserved Asp scope

	hsr_surrounding_packing_package.Task->set_designmap(pre_second_Hbond_DesignMap);

	{ // conserved Asn scope
		std::cout << "Finding second Hbond acceptors to conserved Asp" << std::endl;
		//		PROF_START( prof::FINDING_SECOND_HBOND );
		RotamerSet second_Hbond_rotamer_set;
		convert_Asp_to_Asn( curr_pose, best_AspAsn );

		PackerTask second_Hbond_Task = *(hsr_surrounding_packing_package.Task);
		second_Hbond_Task.get_designmap().fix_completely(best_AspAsn);

		keep_rotamer_from_pose( best_AspAsn, curr_pose, hsr_surrounding_packing_package );
		rotamerFix secondary_AspAsn( best_AspAsn, &hsr_surrounding_packing_package );

		find_second_Hbond( curr_pose, second_Hbond_rotamer_set, valid_partner_positions, second_Hbond_Task );
		std::cout << "Found " << second_Hbond_rotamer_set.nrotamers() <<
			" candidate accepting rotamers for the second Hbond" << std::endl;
		//		PROF_STOP( prof::FINDING_SECOND_HBOND );
		for ( int second_Hbond_num = 1; second_Hbond_num <= second_Hbond_rotamer_set.nrotamers(); ++second_Hbond_num ) {
			float const delta_sasa = loop_over_second_Hbond(second_Hbond_num, pre_second_Hbond_pose,
				second_Hbond_rotamer_set, hsr_surrounding_packing_package );
			if ( delta_sasa > delta_sasa_thres ) {
				std::cout << "delta_sasa_thres exceeded - moving on to next hotspot rotamer" << std::endl;
				break;
			}
		}

	} // conserved Asn scope

	hsr_surrounding_packing_package.Task->set_designmap(pre_second_Hbond_DesignMap);

	std::cout << "Done testing candidate second Hbond rotamers - moving on to next hotspot rotamer" << std::endl;

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::loop_over_second_Hbond
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float HotspotRotamer::loop_over_second_Hbond(
  int const second_Hbond_num,
	pose_ns::Pose & pre_second_Hbond_pose,
	RotamerSet const & second_Hbond_rotamer_set,
	packingPackage & packing_package
) {

	using namespace hotspot_ns;
	using namespace param;
	using namespace param_aa;
	using namespace pose_ns;

	Pose curr_pose;
	curr_pose = pre_second_Hbond_pose;

	int const curr_Hbond_seqpos = second_Hbond_rotamer_set.report_seqpos(second_Hbond_num);
	int const curr_Hbond_aa = second_Hbond_rotamer_set.report_aa(second_Hbond_num);
	int const curr_Hbond_aav = second_Hbond_rotamer_set.report_aav(second_Hbond_num);

	// jk Put the coors of the "second Hbond" rotamer into curr_pose
	curr_pose.copy_sidechain(curr_Hbond_seqpos, curr_Hbond_aa, curr_Hbond_aav,
													 second_Hbond_rotamer_set.get_rotcoord(second_Hbond_num));
	curr_pose.copy_to_misc();

	std::cout << "Placed a " << aa_name3(curr_Hbond_aa) << " at position " << curr_Hbond_seqpos << " to form a second Hbond" << std::endl;

	// Fix Asp/Asn as needed (the "second Hbond" rotamer may be expecting either an Asp or an Asn)
	//	repack_AspAsn(curr_pose);

	// Re-minimize with the second Hbond in place
	std::cout << "Minimizing interface with second Hbond in place" << std::endl;
	//	PROF_START( prof::RB_MINIMIZE );
	PackerTask second_hbond_Task( curr_pose );
	second_hbond_Task.set_designmap( packing_package.Task->get_designmap() );
	second_hbond_Task.get_designmap().fix_completely(curr_Hbond_seqpos);
	hsr_minimize( curr_pose, second_hbond_Task, true, false );
	//	PROF_STOP( prof::RB_MINIMIZE );

	// Recheck "validity" of hotspot residue (buried, no clash, Hbond)
	if ( ( ! valid_self_OneBody( curr_pose, false ) ) || ( ! valid_interface_OneBody(curr_pose, true, false) ) ) {
		std::cout << "Jumping out due to lack of valid hotspot (no Hbond and/or backbone clash)" << std::endl;
		return 0.;
	}

	// jk Check for a clash between the backbones (or fixed sidechains)
	float const delta_rep_thres(1.);
	float delta_rep(0.);
	Score_weight_map rep_weight_map;
	rep_weight_map.clear();
	rep_weight_map.set_weight( FA_REP, 1.0 );
	curr_pose.score( rep_weight_map );
	for ( int i = 1; i <= misc::domain_end(1); ++i ) {
		for ( int j = misc::domain_end(1)+1; j <= misc::domain_end(2); ++j ) {
			delta_rep += fullatom_energies::rep_pair(i,j);
		}
	}
	if ( delta_rep > delta_rep_thres ) {
		std::cout << "Jumping out due to backbone (or fixed residue) clash after minimization" << std::endl;
		return 0.;
	}

	// jk Check for delta_uns
	std::cout << "Instantiating second_hbond_interface" << std::endl;
	interface_ds second_hbond_interface(&curr_pose);
	if ( second_hbond_interface.count_interface_delta_group_uns_list() > second_hbond_group_uns_thres ) {
		return 0.;
	}

	// jk Check that there isn't a more buried Asp/Asn
	int const best_AspAsn = find_best_available_AspAsn( curr_pose );
	for ( int seqpos = 1; seqpos <= curr_pose.total_residue(); ++seqpos ) {
		if ( seqpos == partner_seqpos_ ) continue;
		if ( seqpos == best_AspAsn ) continue;
		if ( ! AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) continue;
		if ( AnkyrinRepeat_ns::AspAsn_neighbors(seqpos) > max_unplanned_AspAsn_neighbors ) {
			std::cout << "Jumping out due to unanticipated burial of additional Asp/Asn at seqpos " << seqpos <<
				" (counted " << AnkyrinRepeat_ns::AspAsn_neighbors(seqpos) << " neighbors)." << std::endl;
			return 0.;
		}
	}

	DesignMap pre_second_Hbond_DesignMap = packing_package.Task->get_designmap();
	transform_rotamers_for_new_binding_mode( curr_pose, packing_package );
	keep_rotamer_from_pose( curr_Hbond_seqpos, curr_pose, packing_package );

	float delta_sasa;

	{ // scope for rotamerFix
		rotamerFix secondary_Hbond( curr_Hbond_seqpos, &packing_package );
		delta_sasa = loop_over_AR_modules(curr_pose, packing_package );
	} // scope for rotamerFix

	packing_package.Task->set_designmap(pre_second_Hbond_DesignMap);
	return delta_sasa;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::loop_over_AR_modules
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float HotspotRotamer::loop_over_AR_modules(
	pose_ns::Pose & in_pose,
	packingPackage & hsr_surrounding_packing_package
) {

	using namespace AnkyrinRepeat_ns;
	using namespace hotspot_ns;
	using namespace param;
	using namespace pose_ns;

	AnkyrinRepeat_ns::module_info.clear();

	// jk Find the position for the incoming module (before the one with the hotspot partner)
	int const module_start_seqpos = partner_seqpos_ -
		( AnkyrinRepeat_ns::module_length + AnkyrinRepeat_ns::module_library_offset );

	bool need_IG_update = true;
	Pose start_pose;
	start_pose = in_pose;
	for ( int design_num = 1; design_num <= output_designs_per_binding_mode; ++design_num ) {
		// Start with the current AR module
		in_pose.copy_to_misc();
		float delta_sasa = design_around_hotspot( in_pose, need_IG_update, hsr_surrounding_packing_package );
		in_pose.copy_to_misc();
		if ( delta_sasa > delta_sasa_thres ) {
			std::cout << "delta_sasa_thres exceeded - moving on to next hotspot rotamer" << std::endl;
			return delta_sasa;
		}
		need_IG_update = false;
		in_pose = start_pose;
	}

	// If we don't have a module library or the module we would be swapping is off the AR,
	// return now
	if ( AR_module_set.empty() || ( module_start_seqpos <= misc::domain_end(1) ) )
		return 0.;

	// Loop over alternate modules
	for ( std::list<AnkyrinModule>::iterator curr_AR_module = AR_module_set.begin();
				curr_AR_module != AR_module_set.end(); ++curr_AR_module ) {

		Pose curr_pose;
		curr_pose = in_pose;
		curr_AR_module->replace_module_in_pose( curr_pose, module_start_seqpos );

		for ( int module_seqpos = module_start_seqpos, i = 1; i <= AnkyrinRepeat_ns::module_length; ++module_seqpos, ++i ) {
			rebuild_rotamers_at_seqpos( module_seqpos, curr_pose, hsr_surrounding_packing_package );
			keep_rotamer_from_pose( module_seqpos, curr_pose, hsr_surrounding_packing_package );
		}

		curr_pose.copy_to_misc();
		transform_rotamers_for_new_binding_mode( curr_pose, hsr_surrounding_packing_package );
		update_all_1body( hsr_surrounding_packing_package );

		bool need_IG_update = true;
		Pose start_pose;
		start_pose = curr_pose;
		for ( int design_num = 1; design_num <= output_designs_per_binding_mode; ++design_num ) {
			// Start with the current AR module
			curr_pose.copy_to_misc();
			design_around_hotspot( curr_pose, need_IG_update, hsr_surrounding_packing_package );
			curr_pose.copy_to_misc();
			need_IG_update = false;
			curr_pose = start_pose;
		}

	}

	// We've altered the RotamerSet to fit onto the modules, now restore it to fit onto in_pose again
	in_pose.copy_to_misc();
	for ( int module_seqpos = module_start_seqpos, i = 1; i <= AnkyrinRepeat_ns::module_length; ++module_seqpos, ++i ) {
		rebuild_rotamers_at_seqpos( module_seqpos, in_pose, hsr_surrounding_packing_package );
		keep_rotamer_from_pose( module_seqpos, in_pose, hsr_surrounding_packing_package );
	}
	transform_rotamers_for_new_binding_mode( in_pose, hsr_surrounding_packing_package );
	update_all_1body( hsr_surrounding_packing_package );

	return 0.;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::design_around_hotspot
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float HotspotRotamer::design_around_hotspot(
	pose_ns::Pose & in_pose,
	bool const update_packingPackage,
	packingPackage & hsr_surrounding_packing_package
) {

	using namespace hotspot_ns;
	using namespace param;
	using namespace pose_ns;

	// Redesign the hotspot environment
	std::cout << "Starting design of hotspot environment" << std::endl;
	//	PROF_START( prof::ONE_BODY_REDESIGN );
	redesign( true, ! do_periphery_last, update_packingPackage, in_pose, hsr_surrounding_packing_package );
	//	PROF_STOP( prof::ONE_BODY_REDESIGN );

	//	Pose premin_pose;
	//	premin_pose = curr_pose;

	pose_ns::Pose curr_pose;
	curr_pose = in_pose;

	if ( hotspot_ns::preserve_IG || hotspot_ns::preserve_RotamerSet ) {
		std::cout << "WARNING: Skipping minimization of hotspot environment" << std::endl;
	} else {
		//		PROF_START( prof::FLEX_MINIMIZE );
		std::cout << "Minimizing hotspot environment" << std::endl;
		hsr_minimize( curr_pose, *hsr_surrounding_packing_package.Task, true, ! do_periphery_last );
		//		PROF_STOP( prof::FLEX_MINIMIZE );
	}

	if ( ! do_periphery_last ) {

		// Check delta sasa
		std::cout << "Instantiating complete_interface" << std::endl;
		interface_ds complete_interface(&curr_pose);

		bool good_design(true);
		//		if ( hotspot_on_AR ) {
			// jk Note: aromatic stacks are detected upon instantiation of complete_interface
			//			aromatic_stack stack;
			//			complete_interface.test_for_stack(seqpos_, stack_seqpos(curr_pose), stack);
		//			if ( complete_interface.is_stack(seqpos_) ) good_design = true;
		//		} else {
		//			hotspot_residue hr;
		//			complete_interface.test_for_hotspot(seqpos_, hr);
		//			std::cout << "Attempt to find hotspot gave: " << hr.describe() << std::endl;
		//			if ( complete_interface.is_hotspot(seqpos_) ) good_design=true;
		//		}

		float const delta_sasa = complete_interface.get_delta_sasa();
		//		int const delta_group_uns_list = complete_interface.count_interface_delta_group_uns_list();
		//		int const num_buried_AspAsn = complete_interface.count_buried_AspAsn();
		//		if ( good_design ) {
			//			std::cout << "Delta_sasa is " << delta_sasa << std::endl;
			//			std::cout << "Delta_group_uns_list is " << delta_group_uns_list << std::endl;
			//			std::cout << "Number of buried AspAsn is " << num_buried_AspAsn << std::endl;
			//			if ( ( delta_sasa > delta_sasa_thres ) || ( delta_group_uns_list > hotspot_env_group_uns_thres ) ) {
			//			if ( delta_sasa > delta_sasa_thres ) {
			//				good_design=false;
			//			}
		//		}

		float const delta_score = complete_interface.get_delta_score();

		if ( ( delta_sasa > min_delta_sasa_output ) && ( delta_score < max_delta_score_output ) ) {
			if ( good_design ) {
				// jk Write out the final minimized design
				std::cout << "Writing an output design" << std::endl;
				//			write_outpdb(premin_pose, "premin");
				write_outpdb(curr_pose, "des_neutral");
				++output_decoy_index_num;
			} else {
				std::cout << "Design did not pass hotspot criteria" << std::endl;
				if ( output_all_designs ) {
					write_outpdb(curr_pose, "fail");
					++output_decoy_index_num;
				}
			}
		}
		return delta_sasa;

	} else {

		// Redesign the periphery only if we meet hotspot requirements
		// and we have at most one group_uns near the hotspot
		std::cout << "Instantiating hsr_surrounding_interface" << std::endl;
		interface_ds hsr_surrounding_interface(&curr_pose);

		bool good_design(false);
		if ( hotspot_on_AR ) {
			// jk Note: aromatic stacks are detected upon instantiation of hsr_surrounding_interface
			//			aromatic_stack stack;
			//			hsr_surrounding_interface.test_for_stack(seqpos_, stack_seqpos(curr_pose), stack);
			if ( hsr_surrounding_interface.is_stack(seqpos_) ) good_design = true;
		} else {
			hotspot_residue hr;
			hsr_surrounding_interface.test_for_hotspot(seqpos_, hr);
			std::cout << "Attempt to find hotspot gave: " << hr.describe() << std::endl;
			if ( hsr_surrounding_interface.is_hotspot(seqpos_) ) good_design=true;
		}

		float const delta_sasa = hsr_surrounding_interface.get_delta_sasa();
		int const delta_group_uns_list = hsr_surrounding_interface.count_interface_delta_group_uns_list();
		//		int const num_buried_AspAsn = hsr_surrounding_interface.count_buried_AspAsn();
		if ( good_design ) {
			std::cout << "Delta_sasa is " << delta_sasa << std::endl;
			std::cout << "Delta_group_uns_list is " << delta_group_uns_list << std::endl;
			//			std::cout << "Number of buried AspAsn is " << num_buried_AspAsn << std::endl;
			if ( ( delta_sasa > delta_sasa_thres ) || ( delta_group_uns_list > hotspot_env_group_uns_thres ) ) {
				good_design=false;
			}
		}

		if ( good_design ) {

			// jk Write out the "hotspot surroundings" to a PDB
			write_outpdb(curr_pose, "onebody");
			std::cout << "Writing a hotspot design" << std::endl;

			// Redesign the periphery
			std::cout << "Starting design of periphery" << std::endl;
			//			PROF_START( prof::FULL_REDESIGN );
			packingPackage periphery_packing_package;
			periphery_packing_package.Task = new PackerTask( curr_pose );
			periphery_packing_package.Task->set_designmap( hsr_surrounding_packing_package.Task->get_designmap() );

			redesign( false, true, true, curr_pose, periphery_packing_package );
			//			PROF_STOP( prof::FULL_REDESIGN );

			// jk Do a flexible minimization
			Pose final_design_pose;
			final_design_pose = curr_pose;
			std::cout << "Minimizing flexible interface" << std::endl;
			//			PROF_START( prof::FLEX_MINIMIZE );
			hsr_minimize( curr_pose, *periphery_packing_package.Task, true, true );
			//			PROF_STOP( prof::FLEX_MINIMIZE );

			std::cout << "Instantiating complete_interface" << std::endl;
			interface_ds complete_interface(&curr_pose);
			//			if ( ( complete_interface.get_delta_sasa() < delta_sasa_thres ) &&
			//					 ( complete_interface.is_hotspot(seqpos_) ) &&
			//					 ( complete_interface.count_buried_AspAsn() <= 2 ) &&
			//					 ( complete_interface.count_interface_delta_group_uns_list() <= hotspot_env_group_uns_thres ) ) {

			// jk Write out the final minimized design
			std::cout << "Writing a complete output design" << std::endl;
			write_outpdb(curr_pose, "des_neutral");
			//			}

			// jk Build a similar design with electrostatic complementarity
			if ( hotspot_ns::do_electrostatic_complementarity ) {
				//				PROF_START( prof::ELEC_COMPLEMENTARITY_REDESIGNS );

				Pose elec_design_pose;

				std::cout << "Building electrostatically complementary interfaces" << std::endl;

				// Apply Eshift in one direction
				float Eshift = hotspot_ns::electrostatic_complementarity_shift;
				apply_electrostatic_complementarity_shift( periphery_packing_package.ig, *periphery_packing_package.rotamer_set, Eshift );
				elec_design_pose = final_design_pose;
				redesign( false, true, false, elec_design_pose, periphery_packing_package );
				hsr_minimize( elec_design_pose, *periphery_packing_package.Task, true, true );

				interface_ds elec1_interface(&elec_design_pose);
				//				if ( ( elec1_interface.get_delta_sasa() < delta_sasa_thres ) &&
				//						 ( elec1_interface.is_hotspot(seqpos_) ) &&
				//						 ( elec1_interface.count_interface_delta_group_uns_list() <= hotspot_env_group_uns_thres ) ) {
				// jk Write out the final minimized design
				std::cout << "Writing elec1" << std::endl;
				write_outpdb(elec_design_pose, "des_elec1");
				//				}

				// Now shift in the other direction
				Eshift *= -2.;
				apply_electrostatic_complementarity_shift( periphery_packing_package.ig, *periphery_packing_package.rotamer_set, Eshift );
				elec_design_pose = final_design_pose;
				redesign( false, true, false, elec_design_pose, periphery_packing_package );
				hsr_minimize( elec_design_pose, *periphery_packing_package.Task, true, true );

				interface_ds elec2_interface(&elec_design_pose);
				//				if ( ( elec2_interface.get_delta_sasa() < delta_sasa_thres ) &&
				//						 ( elec2_interface.is_hotspot(seqpos_) ) &&
				//						 ( elec2_interface.count_interface_delta_group_uns_list() <= hotspot_env_group_uns_thres ) ) {
				// jk Write out the final minimized design
				std::cout << "Writing elec2" << std::endl;
				write_outpdb(elec_design_pose, "des_elec2");
				//				}

				//				PROF_STOP( prof::ELEC_COMPLEMENTARITY_REDESIGNS );
			}

			++output_decoy_index_num;

			periphery_packing_package.reset();
			return complete_interface.get_delta_sasa();

		} else {
			std::cout << "No hotspot present after design of hotspot environment" << std::endl;
			return hsr_surrounding_interface.get_delta_sasa();
		}

	}

	// never used
	return 0.;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::hsr_minimize
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::hsr_minimize(
	pose_ns::Pose & curr_pose,
	PackerTask const & Task,
	bool const flexible_chi,
	bool const flexible_backbone
) {

	using namespace pose_ns;

	std::cout << "Starting minimization" << std::endl;

	bool use_atom_tree = false;
	if ( flexible_chi || flexible_backbone ) use_atom_tree = true;

	// jk Save this for later
	bool const init_nblist_setting = get_use_nblist();

	float const minimization_tolerance(0.001);
	pose_setup_minimizer(minimization_tolerance);
	pose_set_use_nblist(false);
	set_use_nblist(false);

	// Minimize the rigid body orientation only if we're fixing
	// backbone and sidechains
	minimize_set_vary_rb_angle( true );
	minimize_set_vary_rb_trans( true );

	if ( hotspot_ns::hotspot_on_AR ) {

		int const stack_pos = stack_seqpos( curr_pose );
		if ( stack_pos == 0 ) {
			std::cout << "Warning: cannot minimize without a stack around which to setup the fold tree" << std::endl;
			return;
		}

		// jk Setup fold tree around the hotspot stack
		pose_ns::Fold_tree ft;
		ft.add_edge(stack_pos, seqpos_, 1); // jump
		ft.add_edge(1,stack_pos,pose_param::PEPTIDE);
		ft.add_edge(stack_pos,misc::domain_end(1),pose_param::PEPTIDE);
		ft.add_edge(misc::domain_end(1)+1,seqpos_,pose_param::PEPTIDE);
		ft.add_edge(seqpos_, misc::domain_end(2),pose_param::PEPTIDE);
		ft.set_jump_atoms( 1, 5, 5, true );  // true for a jump, false for a bond
		ft.reorder(1);
		curr_pose.set_fold_tree(ft);
		if ( use_atom_tree ) {
			curr_pose.setup_atom_tree();
		}
		curr_pose.set_allow_jump_move( true );

	} else {
		// jk Setup fold tree around the hotspot Hbond
		hsr_Hbond.apply_atom_tree(curr_pose, use_atom_tree);
	}

	// jk Apply backbone/sidechain flexibility
	// (note: hotspot and partner will always be allowed to move)
	FArray1D_bool allow_move( curr_pose.total_residue(), false );
	curr_pose.set_allow_bb_move( allow_move );
	curr_pose.set_allow_chi_move( allow_move );
	if ( use_atom_tree ) {
		for ( int i = 1; i <= curr_pose.total_residue(); ++i ) {
			if ( Task.get_designmap().repack_residue(i) ) allow_move(i) = true;
		}
		allow_move(seqpos_) = true;
		allow_move(hsr_Hbond.partner_seqpos()) = true;
		if ( flexible_backbone ) curr_pose.set_allow_bb_move( allow_move );
		if ( flexible_chi ) curr_pose.set_allow_chi_move( allow_move );
	}

	// jk For rigid-body minimizations, enable ONLY the FA_REP and HB_SC terms
	// jk For flexible minimizations, use score12
	Score_weight_map weight_map;
	weight_map.clear();
	if ( use_atom_tree ) {
		weight_map.set_weights( score12 );
		weight_map.set_weight( KIN_1D_CST, 1.0 );
		weight_map.set_weight( KIN_3D_CST, 1.0 );
		weight_map.set_weight( ATOMPAIR_CST, 1.0 );

		// JK DEBUG
		//		set_all_bond_angle_allow_move( curr_pose );
		//		set_all_bond_distance_allow_move( curr_pose );
		//		atom_tree_set_allow_move( curr_pose, flexible_backbone, flexible_chi, true );

	} else {
		if ( hotspot_ns::hotspot_on_AR ) {
			weight_map.set_weight( FA_ATR, 1.0 );
			weight_map.set_weight( PLANE, 3.0 );
			weight_map.set_weight( FA_REP, 1.0 );
		} else {
			weight_map.set_weight( FA_ATR, 1.0 );
			weight_map.set_weight( PLANE, 1.0 );
			weight_map.set_weight( FA_REP, 1.0 );
			weight_map.set_weight( HB_SC, 1.0 );
		}
	}

	// jk Minimize
	curr_pose.score( weight_map );
	curr_pose.main_minimize( weight_map, "dfpmin" );
	//	curr_pose.main_minimize( weight_map, "linmin" );
	curr_pose.score( weight_map );

	// jk Restore the "use_nblist" setting
	set_use_nblist(init_nblist_setting);

	std::cout << "Done with minimization" << std::endl;

	if (use_atom_tree) curr_pose.clear_atom_tree();

	// jk Rigid-body minimization changes the cartesian coors of the hotspot rotamer - update this
	// jk Note: we may have changed the hotspot residue angles too, so update these too
	for ( int j = 1; j <= aaproperties_pack::natoms(aa_,aav_); ++j ) {
		for ( int k = 1; k <= 3; ++k ){
			set_rotcoord(k,j) = curr_pose.full_coord()(k,j,seqpos_);
		}
	}
	// Update rotactcoord, chi angles, and rperc accordingly (using Rotamer member functions)
	fill_rotactcoord();
	fill_chi_from_coors();
	compute_rperc();

	// jk Transfer the minimized starting pose back to misc
	curr_pose.copy_to_misc();

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::redesign
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::redesign(
  	bool const design_hotspot_environment,
		bool const design_periphery,
		bool const update_packingPackage,
	  pose_ns::Pose & curr_pose,
		packingPackage & packing_package
) {

	using namespace aaproperties_pack;
	using namespace param;

	std::cout << "Redesigning a perturbed interface at position " << seqpos_ << std::endl;

	// Make sure misc is up to date
	curr_pose.copy_to_misc();

	if ( update_packingPackage ) {
		setup_packingPackage( design_hotspot_environment, design_periphery, curr_pose, packing_package );
	}

	// Convert allowed_rotamers to rotamers_to_pack_list
	std::vector<int> rotamers_to_pack_list;
	FArray1D_bool found_rotamer( curr_pose.total_residue(), false );
	rotamers_to_pack_list.reserve(packing_package.rotamer_set->nrotamers());
	for ( int i = 1; i <= packing_package.rotamer_set->nrotamers(); ++i ) {
		if ( packing_package.allowed_rotamers(i) ) {
			rotamers_to_pack_list.push_back(i);
			found_rotamer( packing_package.rotamer_set->report_seqpos(i) ) = true;
		}
	}

	// jk Check that we have at least one allowed rotamer at each repackable seqpos
	for ( int i = 1; i <= curr_pose.total_residue(); ++i ) {
		if ( ! packing_package.Task->get_designmap().repack_residue(i) ) continue;
		assert (found_rotamer(i));
	}

	// jk Redesign
	bool start_with_current(true);
	bool calc_rot_freq(false);
	FArray1D_float rot_freq_junk( packing_package.rotamer_set->nrotamers() );
	FArray1D_int bestrotindex( curr_pose.total_residue() );
	float bestenergy;
	std::cout << "Starting SimAnnealer" << std::endl;
	pack::FixbbSimAnnealer * sa = new pack::FixbbSimAnnealer( rotamers_to_pack_list, bestrotindex,
    bestenergy, start_with_current, packing_package.ig, packing_package.rotamer_set,
    packing_package.current_rot_index, calc_rot_freq, rot_freq_junk);

	// jk Change the number of iterations to get diversity...
	// jk DO NOT scale the number of inner iterations lower than one - this can lead to unassigned states!
	//	sa->scale_inneriterations(0.5);
	//	sa->scale_outeriterations(0.5);
	sa->run();
	delete sa;
	std::cout << "Finished with SimAnnealer" << std::endl;

	// jk Update sequence and coors
	for ( int i = 1; i <= curr_pose.total_residue(); ++i ) {
		// Don't update positions fixed by NATRO
		if ( ! packing_package.Task->get_designmap().repack_residue(i) ) continue;
		assert (bestrotindex(i) > 0);
		assert (i == packing_package.rotamer_set->report_seqpos(bestrotindex(i)));
		curr_pose.copy_sidechain(i, packing_package.rotamer_set->report_aa(bestrotindex(i)),
														 packing_package.rotamer_set->report_aav(bestrotindex(i)),
														 packing_package.rotamer_set->get_rotcoord(bestrotindex(i)));
		packing_package.current_rot_index(i) = bestrotindex(i);
	}
	pack_update_new_rotamer(packing_package.Task->get_designmap(),curr_pose.total_residue());

	// jk Update energy terms too
	pose_ns::Score_weight_map score12_weight_map;
	score12_weight_map.clear();
	score12_weight_map.set_weights( score12 );
	curr_pose.score( score12_weight_map );

	// Make sure misc is up to date
	curr_pose.copy_to_misc();

	std::cout << "Done redesign" << std::endl;

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::setup_packingPackage
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::setup_packingPackage(
	bool const design_hotspot_environment,
	bool const design_periphery,
	pose_ns::Pose & curr_pose,
	packingPackage & interface_packing_package
){

	// jk function sets up rotamers and InteractionGraph by analogy to pack_rotamers...

	using namespace aaproperties_pack;
	using namespace design;
	using namespace param;
	using namespace param_aa;
	using namespace template_pack;

	if ( interface_packing_package.Task == NULL ) {
		std::cout << "error - cannot setup packingPackage with unspecified Task" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( hotspot_ns::preserve_IG && ! hotspot_ns::preserve_RotamerSet ) {
		std::cout << "error - cannot preserve IG without preserving RotamerSet" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( ! hotspot_ns::preserve_IG ) {
		if ( ! hotspot_ns::preserve_RotamerSet ) {
			interface_packing_package.reset();
		} else {
			interface_packing_package.reset_IG_only();
		}
	}

	int const NRES = curr_pose.total_residue();

	// Fill misc arrays
	curr_pose.copy_to_misc();

	enable_packing_etables(NRES, curr_pose.res(),
												 curr_pose.res_variant(), curr_pose.full_coord());

	// jk If the packingPackage has been filled, update it with the current hotspot rotamer and backbone
	if ( ( interface_packing_package.rotamer_set != NULL ) ||
			 ( interface_packing_package.ig != NULL ) ) {
		std::cout << "packingPackage has already been set up - updating hotspot rotamer" << std::endl;

		// Update 1-body energies
		update_all_1body( interface_packing_package );

		// Update rotamer positions and interface energies, since we did a rigid-body minimization
		transform_rotamers_for_new_binding_mode( curr_pose, interface_packing_package );

	} else {

		std::cout << "Setting up packingPackage...." << std::endl;

		interface_packing_package.current_rot_index.dimension( NRES, -1 );
		interface_packing_package.seqpos_with_fixed_rotamers.dimension( NRES, false );
		interface_packing_package.neighbor_indexno.dimension( NRES, NRES );

		if ( ! design_hotspot_environment ) {
			// Fix the hotspot environment in the design map
			std::cout << "Fixing hotspot-contacting residues" << std::endl;
			// JK NOTE: WE HAVE A PROBLEM HERE (IF WE WANT TO DESIGN PERIPHERY SEPARATELY),
			//          BECAUSE WE NEED TO "UNFIX" THIS AT SOME POINT
			std::cout << "error - unsupported code in setup_packingPackage" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

			fix_hotspot_contacts( curr_pose, *interface_packing_package.Task );
		}

		interface_packing_package.rotamer_set = new RotamerSet;

		if ( design_periphery ) {
			// jk We wish to apply a "favor native" bonus, only on the first chain.
			// jk Note: this is NOT applied if design is restricted to the hotspot environment
			// jk To do this, set the "native" residue to id=0 for those we don't want to scale
			// jk (so that the bonus is never applied)
			favor_residue::res_native = 0;
			for ( int seqpos = 1; seqpos <= misc::domain_end(1); ++seqpos ) {
				favor_residue::res_native(seqpos) = hotspot_ns::starting_sequence(seqpos);
			}

		} else {
			std::cout << "The following residues will not be designed because periphery is fixed at this stage: ";
			for ( int seqpos = 1; seqpos <= NRES; ++seqpos ) {
				if ( interface_packing_package.Task->get_designmap().repack_residue(seqpos) ) {
					bool current_is_neighbor;
					float ca_dis, cb_dis;
					are_they_neighbors(aa_,curr_pose.res(seqpos),rotcoord_,
            curr_pose.full_coord()(1,1,seqpos),11.0,11.0,ca_dis,cb_dis,current_is_neighbor);
					if ( ! current_is_neighbor ) {

						// JK FIX THIS TOO
						std::cout << "error - further unsupported code in setup_packingPackage" << std::endl;
						utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

						interface_packing_package.Task->get_designmap().fix_completely(seqpos);
						std::cout << seqpos << ", ";
					}
				}
			}
			std::cout << std::endl;
		}

		set_design_matrix(interface_packing_package.Task->get_designmap(),NRES);  // for safety

		// Fill phi+psi, for backbone-dependent rotamers
		ang_from_pdb_wsc(NRES,curr_pose.res(),curr_pose.res_variant(),
			curr_pose.full_coord(),template_pack::phi,template_pack::psi,chiarray,rotarray);

		// Fill centroids
		for ( int seqpos = 1; seqpos <= NRES; ++seqpos ) {
			put_wcentroid(curr_pose.full_coord()(1,1,seqpos),actcoord(1,seqpos),curr_pose.res(seqpos));
		}

		// Fill neighbor lists
		make_neighbor_info(curr_pose.res(),NRES,curr_pose.full_coord(),
											 neighborlist,neighbors);
		make_neighbor_indexno(interface_packing_package.neighbor_indexno,
													interface_packing_package.Task->get_designmap(),curr_pose);

		// Fill Hbond arrays
		float junk1,junk2,junk3,junk4;
		fill_hbond_arrays(true,curr_pose.res(),curr_pose.res_variant(),curr_pose.full_coord(),
											NRES,neighborlist,neighbors,hbonds::hbderiv_NONE,junk1,junk2,junk3,junk4);

		// Fill the rotamer set
		FArray2D_int extra_rot( MAX_CHI, NRES );
		FArray2D_float extra_chi( MAX_CHI, NRES );

		// Replace the "get_rotamers" call
		interface_packing_package.rotamer_set->reset();
		std::cout << "About to build interface rotamer set" << std::endl;
		RotamerOptions saved_rotamer_options = active_rotamer_options;
		active_rotamer_options = hotspot_ns::hs_rotamer_options;
		active_rotamer_options.do_rotamer_bias = hotspot_ns::do_rotamer_improvement;
		active_rotamer_options.bias_rotamers_by_replacement = false;
		active_rotamer_options.extra_bias_weights.insert(
			std::pair<int, float>(seqpos_, hotspot_ns::hotspot_weight_factor));

		// jk Get chi's from current coors
		FArray2D_float chi_curr( MAX_CHI, NRES );
		FArray2D_int rot_curr( MAX_CHI, NRES );
		get_chi_and_rot_from_coords( NRES, curr_pose.res(),
		  curr_pose.res_variant(), curr_pose.full_coord(), chi_curr, rot_curr );

		// Build with packrot if we're preserving RotamerSet, to accommodate variations on the binding mode
		std::string mode("design");
		if ( hotspot_ns::preserve_RotamerSet ) mode = "packrot";

		for ( int seqpos = 1; seqpos <= NRES; ++seqpos ) {

			if ( interface_packing_package.Task->get_designmap().repack_residue(seqpos) ) {

				// For the hotspot residue, allow only hydroxyl changes when building rotamers
				if ( seqpos == seqpos_) {
					int nrotaa(0);
					interface_packing_package.rotamer_set->get_rotamers_seqpos_aa_aav(NRES,
						curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord(), aa_,
            aav_, seqpos_, nrotaa, false, "optimizeH", *interface_packing_package.Task );
					continue;
				}

				if ( AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) {

					int const aav = 1;
					assert(AnkyrinRepeat_ns::using_AR);
					if ( ( curr_pose.res(seqpos) == param_aa::aa_asp ) ||  ( curr_pose.res(seqpos) == param_aa::aa_asn ) ) {
						// jk Only the "known" rotamer for both Asp and Asn will be built at this position
						if ( interface_packing_package.Task->get_designmap().get(seqpos,aa_asp) ) {
							interface_packing_package.rotamer_set->get_AR_AspAsn_rotamers( NRES,
								curr_pose.full_coord(), chi_curr, rot_curr, param_aa::aa_asp, aav, seqpos );
							if ( curr_pose.res(seqpos) == param_aa::aa_asp ) {
								interface_packing_package.current_rot_index(seqpos) =
									interface_packing_package.rotamer_set->nrotamers();
							}
						}
						if ( interface_packing_package.Task->get_designmap().get(seqpos,aa_asn) ) {
							interface_packing_package.rotamer_set->get_AR_AspAsn_rotamers( NRES,
								curr_pose.full_coord(), chi_curr, rot_curr, param_aa::aa_asn, aav, seqpos );
							if ( curr_pose.res(seqpos) == param_aa::aa_asn ) {
								interface_packing_package.current_rot_index(seqpos) =
									interface_packing_package.rotamer_set->nrotamers();
							}
							// For Asn, also build a rotamer in which chi2 is rotated 180 degrees
							// (ie. amide plane flip)
							chi_curr(2,seqpos) =
								set_chi_to_periodic_range( (chi_curr(2,seqpos)-180.), aa_asn, 2 );
							rotamer_from_chi( chi_curr(1,seqpos), aa_asn, rot_curr(1,seqpos) );
							interface_packing_package.rotamer_set->get_AR_AspAsn_rotamers( NRES,
								curr_pose.full_coord(), chi_curr, rot_curr, param_aa::aa_asn, aav, seqpos );
						}

					} else {
						// We don't have an existing Asp/Asn to build from, so just build these normally
						if ( interface_packing_package.Task->get_designmap().get(seqpos,param_aa::aa_asp) ) {
							int nrotaa = 0;
							interface_packing_package.rotamer_set->get_rotamers_seqpos_aa_aav( curr_pose.total_residue(),
								curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord(), param_aa::aa_asp, aav,
								seqpos, nrotaa, false, "packrot", *(interface_packing_package.Task) );
						}
						if ( interface_packing_package.Task->get_designmap().get(seqpos,param_aa::aa_asn) ) {
							int nrotaa = 0;
							interface_packing_package.rotamer_set->get_rotamers_seqpos_aa_aav( curr_pose.total_residue(),
								curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord(), param_aa::aa_asn, aav,
								seqpos, nrotaa, false, "packrot", *(interface_packing_package.Task) );
						}
					}

					// Build Ser and Thr, if desired
					if ( interface_packing_package.Task->get_designmap().get(seqpos,param_aa::aa_ser) ) {
						int nrotaa = 0;
						interface_packing_package.rotamer_set->get_rotamers_seqpos_aa_aav( curr_pose.total_residue(),
							curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord(), param_aa::aa_ser, aav,
							seqpos, nrotaa, false, "packrot", *(interface_packing_package.Task) );
					}
					if ( interface_packing_package.Task->get_designmap().get(seqpos,param_aa::aa_thr) ) {
						int nrotaa = 0;
						interface_packing_package.rotamer_set->get_rotamers_seqpos_aa_aav( curr_pose.total_residue(),
							curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord(), param_aa::aa_thr, aav,
							seqpos, nrotaa, false, "packrot", *(interface_packing_package.Task) );
					}

					continue;
				}

				int const initial_rotamer_count = interface_packing_package.rotamer_set->nrotamers();

				// jk Note: differs from "get_rotamers" in that we're NOT building rotamers for positions
				// jk       at which "repack_residue" is false
				for ( int aa = 1, aae = MAX_AA(); aa <= aae; ++aa ) {

					if ( ( design_hotspot_environment && design_periphery ) ||
							 ( design_hotspot_environment && valid_hotspot_neighbor_aa(aa) ) ||
							 ( design_periphery && valid_periphery_aa(aa) ) ) {

						int nrotaa = 0;

						// jk add rotamers with current coordinates
						if ( ( curr_pose.res(seqpos) == aa ) &&
								 interface_packing_package.Task->get_designmap().get(seqpos,aa) ) {
							interface_packing_package.rotamer_set->add_extra_rot_from_coord(seqpos,curr_pose.res(),
                curr_pose.res_variant(),nrotaa, interface_packing_package.Task->get_designmap(),
                template_pack::phi,template_pack::psi,rotarray,
							  chiarray,NRES,actcoord,curr_pose.full_coord(),
								interface_packing_package.current_rot_index);
						}

						// jk build rotamers
						int const aave = nvar(aa);
						for ( int aav = 1; aav <= aave; ++aav ) {
							// jk jumpout if this aav corresponds to a variant we're not using
							bool jumpout(true);
							for ( int vtype = 1; vtype <= aaproperties_pack::number_aav_type; ++vtype ) {
								if ( variant_type( vtype, aa, aav ) &&
										 interface_packing_package.Task->get_designmap().variant_type_is_allowed(vtype,seqpos) ) {
									jumpout=false;
								}
							}
							if ( jumpout ) continue;
							if ( ( aa == param_aa::aa_his ) && ( hotspot_ns::starting_sequence(seqpos) != param_aa::aa_his ) )
								continue;  // never build His rotamers (anywhere) unless His is the native
							if ( interface_packing_package.Task->get_designmap().get(seqpos,aa) ) {
								interface_packing_package.rotamer_set->get_rotamers_seqpos_aa_aav(NRES,
                  curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord(), aa, aav,
                  seqpos, nrotaa, true, mode, *interface_packing_package.Task );
							}
						}

					}
				} // for aa

				if ( interface_packing_package.rotamer_set->nrotamers() == initial_rotamer_count ) {
					// either this is not a designable_seqpos, or we didn't find any valid rotamers to add
					// (eg. if conserved His position is close to hotspot, but His not a valid hotspot_contacting residue)
					std::cout << "Error - no valid rotamers found at a designable seqpos " << seqpos << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
					// Note: we used to fix this position, but now instead give an error so that we can fix this more robustly
					//				interface_packing_package.Task->get_designmap().fix_completely(seqpos);
				}

			} // if repack_residue
		} // for seqpos

		active_rotamer_options = saved_rotamer_options;

		std::cout << "Built this many rotamers for hotspot design: " <<
			SS( interface_packing_package.rotamer_set->nrotamers() ) << std::endl;

		interface_packing_package.allowed_rotamers.dimension( interface_packing_package.rotamer_set->nrotamers() );
		interface_packing_package.allowed_rotamers = true;

		// Instantiate the InteractionGraph
		if ( hotspot_ns::preserve_IG ) {
			std::cout << "Setting up Lazy IG" << std::endl;
			interface_packing_package.ig = pack::create_and_initialize_LazyInteractionGraph( *interface_packing_package.rotamer_set );
		} else {
			std::cout << "Setting up IG" << std::endl;
			interface_packing_package.ig = pack::get_interaction_graph( *interface_packing_package.rotamer_set,
				*interface_packing_package.Task, interface_packing_package.neighbor_indexno );
		}

		// Initial filling of the InteractionGraph
		FArray1D_float ligenergy1b_junk( interface_packing_package.rotamer_set->nrotamers() );

		std::cout << "Filling IG" << std::endl;
		precompute_rotamer_energies( interface_packing_package.ig, *interface_packing_package.rotamer_set,
      *interface_packing_package.Task, interface_packing_package.neighbor_indexno, ligenergy1b_junk,
      curr_pose );

		std::cout << "Done filling IG" << std::endl;

		favor_residue::res_native = 0;

	}

	disable_packing_etables(NRES, curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord());

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin HotspotRotamer::fix_hotspot_contacts
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::fix_hotspot_contacts(
	pose_ns::Pose const & pose,
	PackerTask & Task
) {

	using namespace param_aa;

	// note: returns "true" for the hotspot itself as a hotspot-contacting residue

	float const energy_threshold = -0.3;

	int num_fixed(0);
	FArray1D_float actcoord( 3 );
	std::cout << "The following residues will be fixed, so that the periphery may be designed separately: ";
	for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
		if ( Task.get_designmap().repack_residue(seqpos) ) {

			int const aa = pose.res(seqpos);
			int const aav = pose.res_variant(seqpos);
			put_wcentroid(pose.full_coord()(1,1,seqpos),actcoord,aa);

			float const hsr_interaction_energy = get_base_2body_energy( seqpos_, aa_, aav_,
				rotcoord_, rotactcoord_, seqpos, aa, aav, pose.full_coord()(1,1,seqpos), actcoord, Task );

			if ( hsr_interaction_energy <= energy_threshold ) {

				// JK NOTE: THIS IS PROBLEMATIC, BECAUSE IT WON'T BE FIXED FOR THE REPACKING STEP...

				std::cout << "error - unsupported code in fix_hotspot_contacts" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

				Task.get_designmap().fix_completely(seqpos);
				++num_fixed;
				std::cout << seqpos << ", ";
			}
		}
	}
	std::cout << std::endl;

	std::cout << "Fixed " << num_fixed << " hotspot-contacting residues." << std::endl;

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin find_second_Hbond
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void HotspotRotamer::find_second_Hbond(
  pose_ns::Pose const & pose,
	RotamerSet & second_Hbond_rotamer_set,
	const FArray1D_bool & valid_partner_positions,
	const PackerTask & Task
) {

	// jk Look for rotamers which make a second Hbond to a NATRO sidechain

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace design;
	using namespace hotspot_ns;
	using namespace param;
	using namespace param_aa;

	// Loop over all NATRO interface residues on chain B which have a polar sidechain not making an Hbond
	std::cout << "Starting to look for a second Hbond" << std::endl;

	// This is the cutoff for defining an Hbond
	float const hbond_energy_threshold(-0.1);

	// Use the setup in decoystats used for UNS
  pose.copy_to_misc();
	fullatom_energy(pose.res(),pose.res_variant(),pose.full_coord(),pose.total_residue(),false);
	copy_hbenergies();

	for ( int natro_seqpos = misc::domain_end(1)+1; natro_seqpos <= misc::domain_end(2); ++natro_seqpos ) {

		if ( ! valid_partner_positions(natro_seqpos) ) continue;
		if ( ! interface_residue(natro_seqpos) ) continue;

		int const natro_aa = pose.res(natro_seqpos);
		int const natro_aav = pose.res_variant(natro_seqpos);

		std::cout << "Looking for an Hbond to fixed " << aa_name3(natro_aa) << " at position " << natro_seqpos << std::endl;

		// for every potential donor on the NATRO sidechain
		bool donor_found(false);
		for ( int hnum = 2; hnum <= nH_polar(natro_aa,natro_aav); ++hnum ) {
			// check for an existing Hbond
			int const dhatm = Hpos_polar(hnum,natro_aa,natro_aav);
			float hbE = atom_hbondE(dhatm,natro_seqpos);
			if ( hbE > hbond_energy_threshold ) {
				donor_found=true;
			}
		}

		// look for an available acceptor at this natro_seqpos
		bool acceptor_found(false);
		for ( int anum = 2; anum <= nacceptors(natro_aa,natro_aav); ++anum ) {
			// check for an existing Hbond
			int const aatm = accpt_pos(anum,natro_aa,natro_aav);
			float hbE = atom_hbondE(aatm,natro_seqpos);
			if ( hbE > hbond_energy_threshold ) {
				acceptor_found=true;
			}
		}

		if ( ! ( donor_found || acceptor_found ) ) continue;

		// try the TWO special "known" Trps AR self-hotspot position (note: no interface bump check!)
		if ( acceptor_found ) {
			int const trpA_rot_seqpos = natro_seqpos - 29;
			if ( (trpA_rot_seqpos != seqpos_) && ( trpA_rot_seqpos > misc::domain_end(1) ) ) {
				if ( Task.get_designmap().get(trpA_rot_seqpos,aa_trp) ) {
					std::cout << "Testing self-satisfying Trp donor at seqpos " << trpA_rot_seqpos << std::endl;
					explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_trp, trpA_rot_seqpos, natro_seqpos, Task, true, false);
					std::cout << "Reached " << second_Hbond_rotamer_set.nrotamers() <<
						" total candidate rotamers for second Hbond" << std::endl;
				}
			}

			int const trpB_rot_seqpos = natro_seqpos + 9;
			if ( (trpB_rot_seqpos != seqpos_) && ( trpB_rot_seqpos > misc::domain_end(1) ) ) {
				if ( Task.get_designmap().get(trpB_rot_seqpos,aa_trp) ) {
					std::cout << "Testing self-satisfying Trp donor at seqpos " << trpB_rot_seqpos << std::endl;
					explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_trp, trpB_rot_seqpos, natro_seqpos, Task, true, false);
					std::cout << "Reached " << second_Hbond_rotamer_set.nrotamers() <<
						" total candidate rotamers for second Hbond" << std::endl;
				}
			}
		}

		// try the special "known" Tyr AR self-hotspot position (note: no bump check!)
		int const tyr_rot_seqpos = natro_seqpos + 9;
		if ( tyr_rot_seqpos <= misc::domain_end(2) ) {
			if ( (tyr_rot_seqpos != seqpos_) && Task.get_designmap().get(tyr_rot_seqpos,aa_tyr) ) {
				std::cout << "Testing self-satisfying Tyr at seqpos " << tyr_rot_seqpos << std::endl;
				explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_tyr, tyr_rot_seqpos, natro_seqpos, Task, true, false);
				std::cout << "Reached " << second_Hbond_rotamer_set.nrotamers() <<
					" total candidate rotamers for second Hbond" << std::endl;
			}
		}

		// try rotamers of the native (ie. starting) AA at each position of the partner protein (chain A)
		std::cout << "Testing native rotamers for second Hbond" << std::endl;
		for ( int rot_seqpos = 1; rot_seqpos <= misc::domain_end(1); ++rot_seqpos ) {
			if ( (rot_seqpos != seqpos_) && interface_residue(rot_seqpos) && Task.get_designmap().repack_residue(rot_seqpos) ) {
				int native_aa = hotspot_ns::starting_sequence(rot_seqpos);
				explode_for_second_Hbond(pose,second_Hbond_rotamer_set, native_aa, rot_seqpos, natro_seqpos, Task);
			}
		}
		std::cout << "Reached " << second_Hbond_rotamer_set.nrotamers() <<
			" total candidate rotamers for second Hbond" << std::endl;

		if ( fix_target_seq ) return;

		// try hydroxyls for every suitable residue on the partner protein (chain A)
		std::cout << "Testing hydroxyls for second Hbond" << std::endl;
		for ( int rot_seqpos = 1; rot_seqpos <= misc::domain_end(1); ++rot_seqpos ) {
			if ( (rot_seqpos != seqpos_) && interface_residue(rot_seqpos) && Task.get_designmap().repack_residue(rot_seqpos) ) {
				int native_aa = hotspot_ns::starting_sequence(rot_seqpos);
				if ( native_aa != aa_ser )
					explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_ser, rot_seqpos, natro_seqpos, Task);
				if ( native_aa != aa_thr )
					explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_thr, rot_seqpos, natro_seqpos, Task);
				if ( native_aa != aa_tyr )
					explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_tyr, rot_seqpos, natro_seqpos, Task);
			}
		}
		std::cout << "Reached " << second_Hbond_rotamer_set.nrotamers() <<
			" total candidate rotamers for second Hbond" << std::endl;

	}

	bool const disallow_Asn_Gln(true);
	if ( disallow_Asn_Gln ||
			 ( float(second_Hbond_rotamer_set.nrotamers()) >= ( std::max((max_second_Hbond_rot / 5.), 10.) ) ) ) {
		std::cout << "After self-satisfying Tyr/Trp and hydroxyls, done looking for a second Hbond" << std::endl;
		return;
	}

	// Search for Asn/Gln to expand the set

	for ( int natro_seqpos = misc::domain_end(1)+1; natro_seqpos <= misc::domain_end(2); ++natro_seqpos ) {

		if ( ! valid_partner_positions(natro_seqpos) ) continue;
		if ( ! interface_residue(natro_seqpos) ) continue;

		int const natro_aa = pose.res(natro_seqpos);
		int const natro_aav = pose.res_variant(natro_seqpos);

		std::cout << "Trying Asn/Gln for second Hbond" << std::endl;

		// for every potential donor on the NATRO sidechain
		bool donor_found(false);
		for ( int hnum = 2; hnum <= nH_polar(natro_aa,natro_aav); ++hnum ) {
			// check for an existing Hbond
			int const dhatm = Hpos_polar(hnum,natro_aa,natro_aav);
			float hbE = atom_hbondE(dhatm,natro_seqpos);
			if ( hbE > hbond_energy_threshold ) {
				donor_found=true;
			}
		}

		// look for an available acceptor at this natro_seqpos
		bool acceptor_found(false);
		for ( int anum = 2; anum <= nacceptors(natro_aa,natro_aav); ++anum ) {
			// check for an existing Hbond
			int const aatm = accpt_pos(anum,natro_aa,natro_aav);
			float hbE = atom_hbondE(aatm,natro_seqpos);
			if ( hbE > hbond_energy_threshold ) {
				acceptor_found=true;
			}
		}

		if ( ! ( donor_found || acceptor_found ) ) continue;

		// try uncharged polars for every suitable residue on the partner protein (chain A)
		for ( int rot_seqpos = 1; rot_seqpos <= misc::domain_end(1); ++rot_seqpos ) {
			if ( (rot_seqpos != seqpos_) && interface_residue(rot_seqpos) && Task.get_designmap().repack_residue(rot_seqpos) ) {
				std::cout << "Testing Asn/Gln at seqpos " << rot_seqpos << std::endl;
				int native_aa = hotspot_ns::starting_sequence(rot_seqpos);
				if ( native_aa != aa_asn )
					explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_asn, rot_seqpos, natro_seqpos, Task);
				if ( native_aa != aa_gln )
					explode_for_second_Hbond(pose,second_Hbond_rotamer_set, aa_gln, rot_seqpos, natro_seqpos, Task);
				std::cout << "Reached " << second_Hbond_rotamer_set.nrotamers() <<
					" total candidate rotamers for second Hbond" << std::endl;
			}
		}
	}

	return;

}


//////  DONE WITH METHODS FOR HotspotRotamer  //////


//////  Supporting functions  //////


////////////////////////////////////////////////////////////////////////////////
/// @begin transform_rotamers_for_new_binding_mode
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void transform_rotamers_for_new_binding_mode(
	pose_ns::Pose & curr_pose,
	packingPackage & packing_package
) {

	if ( ( ! hotspot_ns::preserve_RotamerSet ) || ( packing_package.rotamer_set == NULL ) ) {
		return;
	}

	// Transform all rotamers onto the current backbone
	int seqpos = 0;
	FArray2D_float Mgl(4,4);
	for ( int i = 1; i <= packing_package.rotamer_set->nrotamers(); ++i ) {

		int const aa = packing_package.rotamer_set->report_aa(i);
		int const aav = packing_package.rotamer_set->report_aav(i);

		if ( packing_package.rotamer_set->report_seqpos(i) != seqpos ) {
			seqpos = packing_package.rotamer_set->report_seqpos(i);
			get_sym_rotamer_transform( packing_package.rotamer_set->get_rotcoord(i),
				curr_pose.full_coord()(1,1,seqpos), Mgl );
		}

		transfer_sym_rotamer(aa, aav, Mgl,
      packing_package.rotamer_set->get_rotcoord(i), packing_package.rotamer_set->get_rotactcoord(i),
      packing_package.rotamer_set->get_rotcoord(i), packing_package.rotamer_set->get_rotactcoord(i) );

	}

	if ( ( ! hotspot_ns::preserve_IG ) || ( packing_package.ig == NULL ) ) {
		return;
	}

	pack::OnTheFlyInteractionGraph * otfig =
		dynamic_cast< pack::OnTheFlyInteractionGraph *> ( packing_package.ig );

	if ( ! otfig ) {
		std::cout << "Error - InteractionGraph was not of type OnTheFly..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// "Quietly" update the coors in the IG (ie. without marking them as changed)
	for ( int rotinx = 1; rotinx <= packing_package.rotamer_set->nrotamers(); ++rotinx ) {
		int const seqpos = packing_package.rotamer_set->report_seqpos(rotinx);
		int const ii = packing_package.rotamer_set->resid_2_moltenres(seqpos);
		int const ii_rotindex_offset = packing_package.rotamer_set->nrotoffset( ii );
		int const jj = rotinx - ii_rotindex_offset;
		FArray2Da_float jj_coords( packing_package.rotamer_set->get_rotcoord(rotinx), 3, param::MAX_ATOM() );
		FArray1Da_float jj_actcoord( packing_package.rotamer_set->get_rotactcoord( rotinx ), 3 );
		otfig->quietly_set_rotamer_coordinates_for_node_state( ii, jj, jj_coords, jj_actcoord );
	}

	pack::LazyInteractionGraph * lazy_ig =
		dynamic_cast< pack::LazyInteractionGraph *> ( packing_package.ig );

	if ( ! lazy_ig ) {
		std::cout << "Error - InteractionGraph was not of type Lazy..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int const nmoltenres = packing_package.ig->get_num_nodes();
	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		int const iiresid = packing_package.rotamer_set->moltenres_2_resid(ii);
		if ( iiresid > misc::domain_end(1) ) break;
		for ( int jj = ii+1; jj <= nmoltenres; ++jj ) {
			int const jjresid = packing_package.rotamer_set->moltenres_2_resid(jj);
			if ( jjresid <= misc::domain_end(1) ) continue;
			lazy_ig->reset_node_pair( ii, jj );
		}
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rebuild_rotamers_at_seqpos
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void rebuild_rotamers_at_seqpos(
  int const seqpos,
	pose_ns::Pose & curr_pose,
	packingPackage & packing_package
) {

	if ( ( ! hotspot_ns::preserve_RotamerSet ) || ( packing_package.rotamer_set == NULL ) ) {
		return;
	}

	if ( ! packing_package.Task->get_designmap().repack_residue(seqpos) ) {
		return;
	}

	curr_pose.copy_to_misc();

	// jk Get chi's from current coors if we'll need them
	FArray2D_float chi_curr( param::MAX_CHI, curr_pose.total_residue() );
	FArray2D_int rot_curr( param::MAX_CHI, curr_pose.total_residue() );
	get_chi_and_rot_from_coords( curr_pose.total_residue(), curr_pose.res(),
		curr_pose.res_variant(), curr_pose.full_coord(), chi_curr, rot_curr );

	// jk ensure get phi,psi and chi angles are up to date
	ang_from_pdb_wsc( curr_pose.total_residue(), curr_pose.res(), curr_pose.res_variant(),
		curr_pose.full_coord(), template_pack::phi,template_pack::psi, template_pack::chiarray,
		template_pack::rotarray );

	// Build with packrot if we're preserving RotamerSet, to accommodate variations on the binding mode
	std::string mode("design");
	if ( hotspot_ns::preserve_RotamerSet ) mode = "packrot";

	// Build a new rotamers for each AA, use these to replace the existing rotamers
	RotamerSet new_rotamers;
	for ( int aa = 1, aae = param::MAX_AA(); aa <= aae; ++aa ) {

		int rot_to_replace = packing_package.rotamer_set->first_rot_for_aa(aa,
			packing_package.rotamer_set->resid_2_moltenres(seqpos));

		// skip this aa if it is not represented in the original RotamerSet
		if ( rot_to_replace == 0 ) continue;

		int nrotaa(0);
		new_rotamers.reset();

		if ( ( AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) &&
  	  	 ( ( curr_pose.res(seqpos) == param_aa::aa_asp ) || ( curr_pose.res(seqpos) == param_aa::aa_asn ) ) ) {
			assert(AnkyrinRepeat_ns::using_AR);
			int const aav = 1;
			// jk Only the "known" rotamer for both Asp and Asn will be built at this position
			if ( aa == param_aa::aa_asp ) {
				// Build Asp
				new_rotamers.get_AR_AspAsn_rotamers( curr_pose.total_residue(), curr_pose.full_coord(),
					chi_curr, rot_curr, param_aa::aa_asp, aav, seqpos );
				if ( curr_pose.res(seqpos) == param_aa::aa_asp )
					packing_package.current_rot_index(seqpos) = new_rotamers.nrotamers();
			} else {
				// Build Asn
				new_rotamers.get_AR_AspAsn_rotamers( curr_pose.total_residue(), curr_pose.full_coord(),
					chi_curr, rot_curr, param_aa::aa_asn, aav, seqpos );
				if ( curr_pose.res(seqpos) == param_aa::aa_asn )
					packing_package.current_rot_index(seqpos) = new_rotamers.nrotamers();
				// For Asn, also build a rotamer in which chi2 is rotated 180 degrees (ie. amide plane flip)
				chi_curr(2,seqpos) = set_chi_to_periodic_range( (chi_curr(2,seqpos)-180.), param_aa::aa_asn, 2 );
				new_rotamers.get_AR_AspAsn_rotamers( curr_pose.total_residue(), curr_pose.full_coord(),
					chi_curr, rot_curr, param_aa::aa_asn, aav, seqpos );
			}

			if ( new_rotamers.nrotamers() < 1 ) {
				std::cout << "Error - failed to build replacement rotamers for conserved Asp/Asn" << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

		} else {
			// All positions other than conserved Asp/Asn

			// jk add rotamers with current chi angles first
			if ( curr_pose.res(seqpos) == aa ) {
				new_rotamers.add_extra_rot_from_chi( seqpos, curr_pose.res(), curr_pose.res_variant(), nrotaa,
					packing_package.Task->get_designmap(), template_pack::phi, template_pack::psi,
					packing_package.Task->get_extra_rot(), packing_package.Task->get_extra_chi(),
					curr_pose.total_residue(), curr_pose.full_coord(), packing_package.current_rot_index );
			}

			// jk build additional rotamers
			int const aave = aaproperties_pack::nvar(aa);
			for ( int aav = 1; aav <= aave; ++aav ) {
				// jk jumpout if this aav corresponds to a variant we're not using
				bool jumpout(true);
				for ( int vtype = 1; vtype <= aaproperties_pack::number_aav_type; ++vtype ) {
					if ( aaproperties_pack::variant_type( vtype, aa, aav ) &&
						packing_package.Task->get_designmap().variant_type_is_allowed(vtype,seqpos) ) {
						jumpout=false;
					}
				}
				if ( jumpout ) continue;
				new_rotamers.get_rotamers_seqpos_aa_aav(curr_pose.total_residue(),
					curr_pose.res(), curr_pose.res_variant(), curr_pose.full_coord(), aa, aav,
					seqpos, nrotaa, false, mode, *packing_package.Task );
			}

		}

		if ( new_rotamers.nrotamers() < 1 ) {
			std::cout << "Error - no replacement rotamers found" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		bool do_replace = true;
		int new_rot_inx=1;
		// Transfer each rotamer from new_rotamers to packing_package, activate them all
		// Jumpout once the packing_package rotamer set can't hold any more
		// If we don't have enough, pad it with copies of the last rotamer built
		while ( do_replace ) {
			do_replace = false;
			if ( ( packing_package.rotamer_set->report_aa(rot_to_replace) == aa ) &&
				( packing_package.rotamer_set->report_seqpos(rot_to_replace) == seqpos ) ) {
				curr_pose.copy_sidechain(seqpos, aa, new_rotamers.report_aav(new_rot_inx),
					new_rotamers.get_rotcoord(new_rot_inx));
				keep_rotamer_from_pose( seqpos, curr_pose, rot_to_replace, packing_package );
				packing_package.allowed_rotamers(rot_to_replace) = true;
				if ( new_rot_inx < new_rotamers.nrotamers() ) {
					++new_rot_inx;
				}
				++rot_to_replace;
				do_replace=true;
			}
		}

	} // for aa

		// Update current_rot_index
	int const curr_rot = packing_package.rotamer_set->first_rot_for_aa(curr_pose.res(seqpos),
		packing_package.rotamer_set->resid_2_moltenres(seqpos));
	packing_package.current_rot_index(seqpos) = curr_rot;
	// Put the current rotamer back into the pose, for safety
	curr_pose.copy_sidechain(seqpos, curr_pose.res(seqpos), packing_package.rotamer_set->report_aav(curr_rot),
		packing_package.rotamer_set->get_rotcoord(curr_rot));

	// Update the IG, if appropriate
	if ( ( ! hotspot_ns::preserve_IG ) || ( packing_package.ig == NULL ) ) {
		return;
	}

	pack::OnTheFlyInteractionGraph * otfig =
		dynamic_cast< pack::OnTheFlyInteractionGraph *> ( packing_package.ig );

	// Update the coors in the IG, marking them as changed
	int const ii = packing_package.rotamer_set->resid_2_moltenres(seqpos);
	int const ii_rotindex_offset = packing_package.rotamer_set->nrotoffset( ii );
	for ( int rotinx = 1; rotinx <= packing_package.rotamer_set->nrotamers(); ++rotinx ) {
		if ( packing_package.rotamer_set->report_seqpos(rotinx) != seqpos ) continue;
		int const jj = rotinx - ii_rotindex_offset;
		FArray2Da_float jj_coords( packing_package.rotamer_set->get_rotcoord(rotinx), 3, param::MAX_ATOM() );
		FArray1Da_float jj_actcoord( packing_package.rotamer_set->get_rotactcoord(rotinx), 3 );
		otfig->set_rotamer_coordinates_for_node_state( ii, jj, jj_coords, jj_actcoord );
	}

	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin keep_rotamer_from_pose
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void keep_rotamer_from_pose(
  int const seqpos,
	pose_ns::Pose & curr_pose,
	packingPackage & packing_package
) {

	if ( ( ! hotspot_ns::preserve_RotamerSet ) || ( packing_package.rotamer_set == NULL ) ) {
		return;
	}

	// Note: overloading "keep_rotamer_from_pose" function below, this version
	// replaces the current rotamer if the aa matches, otherwise replaces the
	// first rotamer of this aa type

	int const aa = curr_pose.res(seqpos);

	int rot_to_replace = packing_package.current_rot_index(seqpos);
	if ( rot_to_replace > 0 ) {
		if ( packing_package.rotamer_set->report_aa(rot_to_replace) == aa ) {
			keep_rotamer_from_pose( seqpos, curr_pose, rot_to_replace, packing_package );
			return;
		}
	}

	if ( packing_package.Task->get_designmap().repack_residue(seqpos) ) {
		if ( ! packing_package.Task->get_designmap().get(seqpos,aa) ) {
			std::cout << "WARNING - keep_rotamer_from_pose was called with a disallowed aa" << std::endl;
			return;
		}
		int const moltenres = packing_package.rotamer_set->resid_2_moltenres(seqpos);
		rot_to_replace = packing_package.rotamer_set->first_rot_for_aa(aa,moltenres);
		if ( rot_to_replace == 0 ) {
			std::cout << "rot_to_replace is zero - attempted to replace an aa for which there are no rotamers" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	// Note: if this is a fixed position, rot_to_replace will remain as -1
	keep_rotamer_from_pose( seqpos, curr_pose, rot_to_replace, packing_package );

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin keep_rotamer_from_pose
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void keep_rotamer_from_pose(
  int const seqpos,
	pose_ns::Pose & curr_pose,
	int const rot_to_replace,
	packingPackage & packing_package
) {

	if ( ( ! hotspot_ns::preserve_RotamerSet ) || ( packing_package.rotamer_set == NULL ) ) {
		return;
	}

	curr_pose.copy_to_misc();

	if ( ! packing_package.Task->get_designmap().repack_residue(seqpos) ) {
		return;
	}

	// Update the coors in the RotamerSet
	packing_package.rotamer_set->replace_rot_with_pose_coors(curr_pose, rot_to_replace);

	if ( ( ! hotspot_ns::preserve_IG ) || ( packing_package.ig == NULL ) ) {
		return;
	}

	pack::OnTheFlyInteractionGraph * otfig =
		dynamic_cast< pack::OnTheFlyInteractionGraph *> ( packing_package.ig );

	// Update the coors in the IG (marking them as changed)
	int const ii = packing_package.rotamer_set->resid_2_moltenres(seqpos);
	int const ii_rotindex_offset = packing_package.rotamer_set->nrotoffset( ii );
	int const jj = rot_to_replace - ii_rotindex_offset;
	FArray2Da_float jj_coords( packing_package.rotamer_set->get_rotcoord(rot_to_replace), 3, param::MAX_ATOM() );
	FArray1Da_float jj_actcoord( packing_package.rotamer_set->get_rotactcoord(rot_to_replace), 3 );
	otfig->set_rotamer_coordinates_for_node_state( ii, jj, jj_coords, jj_actcoord );

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin update_all_1body
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void update_all_1body(
	packingPackage & packing_package
) {

	if ( ( ! hotspot_ns::preserve_IG ) || ( packing_package.ig == NULL ) ) {
		return;
	}

	int const nmoltenres = packing_package.ig->get_num_nodes();
	if (nmoltenres == 0) return;

	FArray1D_float energy1b_base( packing_package.rotamer_set->nrotamers(), 0. );
	pack::OnTheFlyInteractionGraph * otfig =
		dynamic_cast< pack::OnTheFlyInteractionGraph *> ( packing_package.ig );

	for ( int ii = 1; ii <= nmoltenres; ++ii ) {
		packing_package.ig->zero_one_body_energies_for_node(ii);
		int const seqpos = packing_package.rotamer_set->moltenres_2_resid(ii);
		int const ii_num_states = packing_package.rotamer_set->num_states_for_moltenres(ii);
		int const ii_rotindex_offset = packing_package.rotamer_set->nrotoffset( ii );
		for (int jj = 1; jj <= ii_num_states; ++jj) {
			int const rotinx = ii_rotindex_offset + jj;
			int const aa = packing_package.rotamer_set->report_aa(rotinx);
			int const aav = packing_package.rotamer_set->report_aav(rotinx);

			float const energy1b_this_rotamer = get_1body_energy( *packing_package.rotamer_set, rotinx,
				seqpos, aa, aav, packing_package.rotamer_set->get_rotcoord(rotinx),
				energy1b_base, *packing_package.Task );

			otfig->add_to_one_body_energy_for_node_state( ii, jj, energy1b_this_rotamer );

		}
	}

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin explode_for_second_Hbond
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void explode_for_second_Hbond(
  pose_ns::Pose const & pose,
	RotamerSet & second_Hbond_rotamer_set,
	int const rot_aa,
	int const rot_seqpos,
	int const natro_seqpos,
	const PackerTask & Task,
	bool const do_self_bump_check, // default true
	bool const do_interface_bump_check // default true
) {

	using namespace hbonds;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;

	if ( ! Task.get_designmap().get(rot_seqpos,rot_aa) ) return;

	// jk Disallow Lys and Arg (FOR RUNNING ON BLUE GENE)
	//	if ( ( rot_aa == aa_lys ) || (rot_aa == aa_arg ) ) return;

	bool const current_chi_already_included(false);
	int const rot_aav(1);
	int nrotaa(0);

	bool const do_explode(true);
	RotamerOptions saved_rotamer_options = design::active_rotamer_options;
	if ( do_explode ) {
		design::active_rotamer_options.extrachi_flag=exchi_flags::EX_FOUR_HALF_STEP_STDDEVS;
		// NOTE: FOR RUNNING ON BLUE GENE
		if ( ( rot_aa == aa_glu ) || ( rot_aa == aa_lys ) || ( rot_aa == aa_gln ) || ( rot_aa == aa_arg ) ) {
			design::active_rotamer_options.extrachi_flag=exchi_flags::EX_ONE_STDDEV;
		}
		design::active_rotamer_options.rot_explode=true;
		design::active_rotamer_options.rot_explode_accept_all=true;
	}

	RotamerSet explosion_set;

	std::string mode("design");
	if ( ( ! do_self_bump_check ) || ( ! do_interface_bump_check ) ) mode = "packrot";
	explosion_set.get_rotamers_seqpos_aa_aav(pose.total_residue(), pose.res(), pose.res_variant(),
		pose.full_coord(), rot_aa, rot_aav, rot_seqpos, nrotaa, current_chi_already_included, mode, Task);

	if ( do_explode ) {
		design::active_rotamer_options = saved_rotamer_options;
	}

	if ( explosion_set.nrotamers() == 0 ) return;

	int const natro_aa = pose.res(natro_seqpos);
	int const natro_aav = pose.res_variant(natro_seqpos);

	// Vars for Hbond call
	float sc_sc_Hbond(0.), fjunk1(0.), fjunk2(0.), fjunk3(0.);

	// This is the cutoff for defining an Hbond
	float const hbond_energy_threshold(-0.5);

	for ( int rotnum = 1; rotnum <= explosion_set.nrotamers(); ++rotnum ) {

		int const max_h_neighbors = 20;
		get_hbE(true,rot_aa,natro_aa,rot_aav,natro_aav,rot_seqpos,natro_seqpos,
			max_h_neighbors, max_h_neighbors, explosion_set.get_rotcoord(rotnum),
			pose.full_coord()(1,1,natro_seqpos), sc_sc_Hbond,fjunk1,fjunk2,fjunk3);

		if ( sc_sc_Hbond > hbond_energy_threshold ) continue;

		float bumpenergy = 0.0;
		float const bump_thres = 0.5;

		if ( do_self_bump_check || do_interface_bump_check ) {

			// Set the residues to loop over when checking for clashes
			int startres = 1;
			int endres = misc::domain_end(2);
			if ( !do_self_bump_check ) {
				if ( rot_seqpos <= misc::domain_end(1) ) {
					startres = misc::domain_end(1)+1;
				} else {
					endres = misc::domain_end(1);
				}
			}
			if ( ! do_interface_bump_check ) {
				if ( rot_seqpos <= misc::domain_end(1) ) {
					endres = misc::domain_end(1);
				} else {
					startres = misc::domain_end(1)+1;
				}
			}

			for ( int bbres = startres; bbres <= endres; ++bbres ) {
				// loop through backbone positions
				float dis;
				// for distance check, use CB of the new Hbonding rotamer and CA of bbres
				distance_bk(explosion_set.get_rotcoord(rotnum)(1,5),pose.full_coord()(1,2,bbres),dis);
				if ( dis > 10 ) continue;
				float solvE, atrE, repE, elecE, h2oE, h2ohbE;
				get_sc_bbE(rot_aa,rot_aav,pose.res(bbres),pose.res_variant(bbres),explosion_set.get_rotcoord(rotnum),
									 pose.full_coord()(1,1,bbres),rot_seqpos,bbres,solvE,atrE,repE,elecE);
				get_sc_bb_h2oE(rot_aa,rot_aav,pose.res(bbres),pose.res_variant(bbres),explosion_set.get_rotcoord(rotnum),
											 pose.full_coord()(1,1,bbres),rot_seqpos,bbres,h2oE,h2ohbE);

				float const new_bumpenergy = pack_wts.Watr()*atrE+pack_wts.Wrep()*repE+pack_wts.Wh2o()*h2oE;
				if ( new_bumpenergy > 0. ) {
					bumpenergy += new_bumpenergy;
					if ( bumpenergy > bump_thres ) break;
				}
			}
			if ( bumpenergy > bump_thres ) continue;

			// Sidechain bump check
			FArray1D_float dummyactcoord1( 3 );
			FArray1D_float dummyactcoord2( 3 );
			for ( int i = 1; i <= 3; ++i ) {
				dummyactcoord1(i) = 1.0;
				dummyactcoord2(i) = 3.0;
			}

			// NOTE: ANKYRIN-SPECIFIC REQUIREMENT
			// TREAT THE "NOT ACGPS" POSITION AS FIXED WHEN LOOKING FOR CLASHES!
			FArray1D_bool temporarily_fixed_positions( pose.total_residue(), false );
			for ( int seqpos = 1; seqpos <= pose.total_residue(); ++seqpos ) {
				if ( ! AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) continue;
				int const NOTAA_ACGPS_position = seqpos + 4;
				if ( NOTAA_ACGPS_position <= pose.total_residue() ) {
					temporarily_fixed_positions( NOTAA_ACGPS_position ) = true;
				}
			}

			for ( int scres = startres; scres <= endres; ++scres ) {
				// loop through sidechain positions
				if ( Task.get_designmap().repack_residue(scres) &&
						 (scres != natro_seqpos) && ( ! temporarily_fixed_positions(scres) ) ) continue;
				if ( rot_seqpos == scres ) continue;
				//			if ( interface && ! interface_residue(scres) ) continue;
				float dis;
				// for distance check, use CB of the new Hbonding rotamer and CA of scres
				distance_bk(explosion_set.get_rotcoord(rotnum)(1,5),pose.full_coord()(1,2,scres),dis);
				if ( dis > 18 ) continue;
				int scaa = pose.res(scres);
				float solvE, atrE, repE, pairE, plane_totalE, elecE, h2oE, h2ohbE;
				get_sc_scE(rot_aa,rot_aav,explosion_set.get_rotcoord(rotnum),scaa,pose.res_variant(scres),
									 pose.full_coord()(1,1,scres),dummyactcoord1,dummyactcoord2,rot_seqpos,scres,
									 solvE,atrE,repE,pairE,plane_totalE,elecE);
				get_sc_sc_h2oE(rot_aa,rot_aav,explosion_set.get_rotcoord(rotnum),scaa,pose.res_variant(scres),
											 pose.full_coord()(1,1,scres),rot_seqpos,scres,h2oE,h2ohbE);

				float const new_bumpenergy = pack_wts.Watr()*atrE+pack_wts.Wrep()*repE+pack_wts.Wh2o()*h2oE;
				if ( new_bumpenergy > 0. ) {
					bumpenergy += new_bumpenergy;
					if ( bumpenergy > bump_thres ) break;
				}
			}
			if ( bumpenergy > bump_thres ) continue;
		}

		second_Hbond_rotamer_set.append_rotamer(explosion_set, rotnum);

	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin reweight_interface_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void reweight_interface_energies(
	PackerTask & Task,
	const pose_ns::Pose & curr_pose,
  float const interface_weight_factor
) {

	// jk Add a weight to energies across the interface

	for ( int i = 1; i <= misc::domain_end(1); ++i ) {
		for ( int j = misc::domain_end(1)+1; j <= Task.total_residue(); ++j ) {
			// Scale one AND two-body energies, but only in cases where one is repacked
			// and they're neighbors
			if ( Task.get_designmap().repack_residue(i) || Task.get_designmap().repack_residue(j) ) {
				bool current_is_neighbor;
				float ca_dis, cb_dis;
				are_they_neighbors(curr_pose.res(i),curr_pose.res(j),curr_pose.full_coord()(1,1,i),
					curr_pose.full_coord()(1,1,j),11.0,11.0,ca_dis,cb_dis,current_is_neighbor);
				if ( current_is_neighbor ) Task.set_residue_weight(interface_weight_factor, i, j);
			}
		}
	}

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin reweight_hotspot_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void reweight_hotspot_energies(
	PackerTask & Task,
	const pose_ns::Pose & curr_pose,
	int const hsr_seqpos,
  float const hotspot_weight_factor
) {

	// jk Add a weight to hotspot energies

	for ( int i = 1; i <= Task.total_residue(); ++i ) {
		if ( i == hsr_seqpos ) continue;
			// Scale one AND two-body energies, but only in cases where one is repacked
			// and they're neighbors
			if ( Task.get_designmap().repack_residue(i) || Task.get_designmap().repack_residue(hsr_seqpos) ) {
				bool current_is_neighbor;
				float ca_dis, cb_dis;
				are_they_neighbors(curr_pose.res(i),curr_pose.res(hsr_seqpos),curr_pose.full_coord()(1,1,i),
					curr_pose.full_coord()(1,1,hsr_seqpos),11.0,11.0,ca_dis,cb_dis,current_is_neighbor);
				if ( current_is_neighbor ) Task.set_residue_weight(hotspot_weight_factor, i, hsr_seqpos);
			}
	}

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin valid_hotspot_neighbor_aa
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool valid_hotspot_neighbor_aa(
	int const aa
) {

	using namespace param_aa;

	if ( aa == aa_his ) return false;

	return ( ( aa != aa_thr ) &&
					 ( ( aa_is_nonpolar(aa) ) || ( aa == aa_arg ) ) );
}



////////////////////////////////////////////////////////////////////////////////
/// @begin valid_periphery_aa
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool valid_periphery_aa(
	int const aa
) {

	using namespace param_aa;

	if ( aa == aa_met ) return false;
	if ( aa == aa_his ) return false;
	if ( ! hotspot_ns::make_periphery_polar ) return true;
	return ( aa_is_polar(aa) || ( aa == aa_tyr ) );
}


