// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: johnk $


// Rosetta Headers
#include "HotspotRotamerList.h"
#include "HotspotRotamer.h"
#include "aaproperties_pack.h"
#include "AnkyrinRepeat_ns.h"
#include "ConformerLibHandler.h"
#include "DesignMap.h"
#include "files_paths.h"
#include "fullatom_setup.h"
#include "hotspot_ns.h"
#include "misc.h"
#include "pack.h"
#include "PackerTask.h"
#include "param_aa.h"
#include "pose.h"
#include "pose_io.h"
//#include "prof.h"
#include "RotamerSet.h"
#include "template_pack.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/formatted.io.hh>

//Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/file/file_sys_util.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>


//////  STARTING METHODS FOR position_hotspot_rotamer_list  //////


////////////////////////////////////////////////////////////////////////////////
/// @begin position_hotspot_rotamer_list::setup_position_rotamers
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void position_hotspot_rotamer_list::setup_position_rotamers(
  int const aa,
  int const aav,
	DesignMap & design_map
){

	using namespace design;
	using namespace param;
	using namespace param_aa;

	// Set up Tyr/Trp rotamers for this (designed) position
	// Check for backbone clashes, but only to the "self" chain
	// (because the other chain will be moved around)

	int this_aa(aa);
	int this_aav(aav);
	int this_resnum(resnum);

	int nrotaa(0);

	// Build rotamers for this residue at this position using standard technique
	// Note: we've already pulled the subunits apart before getting here

	bool const include_current(false);

	design::active_rotamer_options.ex1 = true;
	design::active_rotamer_options.ex2 = false;
	design::active_rotamer_options.ex3 = false;
	design::active_rotamer_options.ex4 = false;
	design::active_rotamer_options.ex1aro = true;
	design::active_rotamer_options.ex1aro_half = false;
	design::active_rotamer_options.ex2aro_only = false;
	design::active_rotamer_options.ex1aro_exposed = false;
	design::active_rotamer_options.ex2aro_exposed = false;
	design::active_rotamer_options.exOH = true;
	design::active_rotamer_options.excys = false;
	design::active_rotamer_options.ex2aro_only = true;
	design::active_rotamer_options.extrachi_cutoff=0;
	// call setup_exflags so that changes to ex1/ex2 flags take effect...
	design::active_rotamer_options.setup_exflags();

	// Get the rotamers
	RotamerSet hotspot_rotamer_set;

	pose_ns::Pose start_pose;
	pose_from_misc( start_pose, true, false, true ); // coords_init

	std::cout << "About to build " << aa_name3(this_aa) << " hotspot rotamer set" << std::endl;
	if ( hotspot_ns::use_hotspot_conformers ) {
		if ( confLibHandler.isConformerKeyValid(confLibHandler.conformerKeyFromCoord(this_resnum,this_aa))) {
			hotspot_rotamer_set.buildBDAconformers( start_pose.total_residue(), start_pose.res(),
        start_pose.res_variant(), start_pose.full_coord(), design_map, this_resnum, this_aa);
			hotspot_rotamer_set.makeSerThrTyrOHConformer();
		} else {
			std::cout << "Error: confLibHandler code not found" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else {
		PackerTask Task( start_pose );
		Task.set_designmap( design_map );
		hotspot_rotamer_set.get_rotamers_seqpos_aa_aav( start_pose.total_residue(), start_pose.res(),
      start_pose.res_variant(), start_pose.full_coord(), this_aa, this_aav, this_resnum, nrotaa,
      include_current, "packrot", Task );
	}

	// Convert each rotamer to type "HotspotRotamer", explode around it,
	// push any exploded rotamers that don't clash with "self" onto "rotamer_list"
	int num_valid_rot(0);
	int tot_rot(0);
	for ( int rotnum = 1; rotnum <= hotspot_rotamer_set.nrotamers(); ++rotnum ) {

		HotspotRotamer rot(aa, aav, resnum, hotspot_rotamer_set.get_rchi(rotnum),
			hotspot_rotamer_set.get_rotcoord(rotnum), hotspot_rotamer_set.get_rotactcoord(rotnum));

    // use a strict criteria at this stage (eg. for Hbond threshold) ONLY
    // if we're not planning on minimizing
    bool const use_strict_criteria(false);

		// Use this rotamer to "explode"
		std::list<HotspotRotamer> exploded_rotamers;
		rot.explode( exploded_rotamers );
		for ( std::list<HotspotRotamer>::iterator hsr = exploded_rotamers.begin();
				 hsr != exploded_rotamers.end(); ++hsr ) {
			// jk Note: clash check takes place when building rotamers
			if ( hsr->valid_self_OneBody( start_pose ) ) {
				bool interface_one_body_okay = false;
				if ( hotspot_ns::screen_for_hotspot ) {
					interface_one_body_okay = true;
				} else {
					interface_one_body_okay = hsr->valid_interface_OneBody(start_pose, use_strict_criteria);
				}
				if ( interface_one_body_okay ) {
					// No clash with self or across interface, push this rotamer onto "rotamer_list"
					add_rotamer(*hsr);
					++num_valid_rot;
				}
			}
			++tot_rot;
		}
	}

	std::cout << "Done building " << aa_name3(this_aa) << " hotspot rotamer set, found " <<
		num_valid_rot << " valid rotamers after explosion" << std::endl;

	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin position_hotspot_rotamer_list::redesign_around_rotamer_list
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void position_hotspot_rotamer_list::redesign_around_rotamer_list( DesignMap & design_map ) {

	using namespace hotspot_ns;
	using namespace param;
	using namespace pose_ns;

	std::cout << "Starting redesign_around_rotamer_list at position " << resnum << std::endl;

	// jk flag determines whether sc/bb energies are counted as 1-body or 2-body
	design::eval_sc_bbE_for_energy1b = false;

	int last_completed_seqpos, last_completed_rotnum;
	read_checkpoint_file( last_completed_seqpos, last_completed_rotnum );
	if ( last_completed_seqpos > resnum ) return;
	if ( last_completed_seqpos != resnum ) {
		last_completed_rotnum = 0;
	} else {
		std::cout << "Restarting from checkpoint file at " <<
			last_completed_seqpos << ' ' << last_completed_rotnum << std::endl;
	}

	// jk Load the starting structure into a pose object (for storage)
	// jk Note: all the rotamers are built relative to the backbone position in start_pose!
	Pose start_pose;
	pose_from_misc( start_pose, true, false, true ); // coords_init

	std::cout << "Instantiating packingPackage" << std::endl;
	packingPackage hsr_surrounding_packing_package;
	hsr_surrounding_packing_package.Task = new PackerTask( start_pose );
	hsr_surrounding_packing_package.Task->set_designmap( design_map );

	// Fix the hotspot rotamer
	hsr_surrounding_packing_package.Task->get_designmap().fix_completely(resnum);

	int curr_rotnum=0;
	for ( std::list<HotspotRotamer>::iterator hsr = rotamer_list.begin();
			 hsr != rotamer_list.end(); ++hsr ) {

		// jk Check that we haven't already tried this rotamer
		++curr_rotnum;
		if ( curr_rotnum <= last_completed_rotnum ) continue;

		//		PROF_START( prof::INCORPORATE_HOTSPOT );
		// jk Loop over each suitable rotamer, build perturbed binding modes,
		// jk redesign each of these while keeping the hotspot fixed
		std::cout << "Minimizing and redesigning a perturbed interface at position " << resnum << std::endl;
		hsr->incorporate_hotspot( start_pose, hsr_surrounding_packing_package );

		// Restore coors (for the next attempt)
		start_pose.copy_to_misc();

		//		PROF_STOP( prof::INCORPORATE_HOTSPOT );

		write_checkpoint_file( resnum, curr_rotnum );

	}

	hsr_surrounding_packing_package.reset();

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin position_hotspot_rotamer_list::screen_across_rotamer_list
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void position_hotspot_rotamer_list::screen_across_rotamer_list( DesignMap & design_map ) {

	using namespace hotspot_ns;
	using namespace param;
	using namespace pose_ns;

	std::cout << "Starting screen_across_rotamer_list at position " << resnum << std::endl;

	int last_completed_seqpos, last_completed_rotnum;
	read_checkpoint_file( last_completed_seqpos, last_completed_rotnum );
	if ( last_completed_seqpos > resnum ) return;
	if ( last_completed_seqpos != resnum ) {
		last_completed_rotnum = 0;
	} else {
		std::cout << "Restarting from checkpoint file at " <<
			last_completed_seqpos << ' ' << last_completed_rotnum << std::endl;
	}

	// jk Load the starting structure into a pose object (for storage)
	// jk Note: all the rotamers are built relative to the backbone position in start_pose!
	Pose start_pose;
	pose_from_misc( start_pose, true, false, true ); // coords_init

	int curr_rotnum=0;
	for ( std::list<HotspotRotamer>::iterator hsr = rotamer_list.begin();
			 hsr != rotamer_list.end(); ++hsr ) {

		// jk Check that we haven't already tried this rotamer
		++curr_rotnum;
		if ( curr_rotnum <= last_completed_rotnum ) continue;

		// Loop over each Asp/Asn of the second chain
		for ( int AR_seqpos = misc::domain_end(1)+1; AR_seqpos <= start_pose.total_residue(); ++AR_seqpos ) {
			if ( ! AnkyrinRepeat_ns::AspAsn_seqpos(AR_seqpos) ) continue;

			// jk Copy this to a "working" pose
			pose_ns::Pose curr_pose;
			curr_pose = start_pose;

			// jk Put the coors of the current hotspot rotamer into curr_pose
			curr_pose.copy_sidechain(hsr->seqpos(),hsr->aa(),hsr->aav(),hsr->get_rotcoord());
			curr_pose.copy_to_misc();

			// jk Position the AR using an atom tree (ideal Hbond geometry)
			int hs_atom;
			if ( hsr->aa() == param_aa::aa_tyr ) hs_atom = 16;
			int AR_atom = 8; // OD2. OD1 is 7

			Fold_tree f;
			f.clear();
			f.add_edge( hsr->seqpos(), AR_seqpos, 1 ); // JK MAKE THIS A BOND
			f.add_edge(1,hsr->seqpos(), pose_param::PEPTIDE );
			f.add_edge(hsr->seqpos(),misc::domain_end(1), pose_param::PEPTIDE );
			f.add_edge(misc::domain_end(1)+1,AR_seqpos, pose_param::PEPTIDE);
			f.add_edge(AR_seqpos,curr_pose.total_residue(), pose_param::PEPTIDE);
			f.reorder(1);

			// JK FILL THIS IN

			// JK MINIMIZE RB - OR STEP THROUGH NON-HBOND DIHEDRAL ANGLES?
			// JK FILL THIS IN

			// jk Check interface one-body for this pose
			if ( ! hsr->valid_interface_OneBody(curr_pose, true) ) continue;

			// jk Restore the starting sequence with a repack
			// JK FILL THIS IN

			// jk Write an output PDB
			hsr->write_outpdb(curr_pose, "screened");
			++hotspot_ns::output_decoy_index_num;

			// Restore coors (for the next attempt)
			start_pose.copy_to_misc();

		}

		write_checkpoint_file( resnum, curr_rotnum );

	}

	return;

}



//////  DONE WITH METHODS FOR position_hotspot_rotamer_list  //////


//////  SUPPORTING FUNCTIONS  //////


////////////////////////////////////////////////////////////////////////////////
/// @begin get_checkpoint_fname
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
std::string get_checkpoint_fname() {

	using namespace files_paths;

	std::string fullname;
	fullname = pdb_out_path;
	if ( pdbout == "des" ) {
		fullname += output_file;
	} else {
		fullname += pdbout;
	}
	fullname += "_0001.checkpoint";

	return fullname;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin remove_checkpoint_file
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void remove_checkpoint_file() {

	utility::file::file_delete( get_checkpoint_fname() );
	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin read_checkpoint_file
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void read_checkpoint_file( int & last_completed_seqpos, int & last_completed_rotnum ) {

	last_completed_seqpos = 0;
	last_completed_rotnum = 0;

	// If overwrite flag is set, ignore all checkpoint files
	if ( files_paths::overwrite_pdbs ) return;

	// If file exists, read the data
	std::ifstream checkpoint_input( get_checkpoint_fname().c_str() );
  if ( checkpoint_input.fail() ){
		std::cout << "Not reading a checkpoint file" << std::endl;
    return;

  } else {
		std::string inpline;
		while ( getline( checkpoint_input, inpline ) ) {
			std::istringstream line_stream ( inpline,std::istringstream::in );
			line_stream >> last_completed_seqpos >> last_completed_rotnum;
		}
  }

	assert( last_completed_seqpos > 0 );
	assert ( last_completed_rotnum > 0 );

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin write_checkpoint_file
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void write_checkpoint_file( int const last_completed_seqpos, int const last_completed_rotnum ) {

	std::ofstream checkpoint_output;
	checkpoint_output.clear();
	checkpoint_output.open( get_checkpoint_fname().c_str() );
	if ( !checkpoint_output ) {
		std::cout << "Cannot open hotspot checkpoint file for writing " << get_checkpoint_fname() << std::endl;
		std::exit( EXIT_FAILURE );
	}
	checkpoint_output << last_completed_seqpos << ' ' << last_completed_rotnum << std::endl;
	checkpoint_output.close();

	return;

}



