// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.10 $
//  $Date: 2006/10/24 12:20:24 $
//  $Author: plato $

//Rosetta Headers
#include "are_they_neighbors.h"
#include "constraints.h"
#include "design.h"
#include "disulfides.h"
#include "fullatom_sasa.h"
#include "InteractionGraphSupport.h"
#include "InteractionGraphBase.h"
#include "LazyInteractionGraph.h"
#include "ligand.h"
#include "LinearMemoryInteractionGraph.h"
#include "MinimalistInteractionGraph.h"
#include "MultiCoolAnnealer.h"
#include "OnTheFlyInteractionGraph.h"
#include "pack.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "PDInteractionGraph.h"
#include "pKa_mode.h"
#include "PrecomputedInteractionGraph.h"
#include "RotamerDots.h"
#include "RotamerSet.h"
#include "SASAInteractionGraph.h"
#include "SASAGraph.h"
#include "scale_res_energy.h"
#include "symmetric_design.h"
#include "template_pack.h"
#include "void.h"

//ObjexxFCL
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>

//STL
#include <vector>
#include <fstream>
#include <iostream>
#include <sstream>

namespace pack{

bool dynamically_decide_between_lazy_and_precomputed = true;
bool tight_memory_restrictions = false;
bool lazy_ig = false;
bool linmem_ig = false;
bool minimalist_ig = false;
bool use_sasa_pack_score = false;
bool output_dot_kinemage = false;
int megs_for_rpes_limit = 256;

InteractionGraphBase*
get_interaction_graph(
	RotamerSet & rotamer_set,
	PackerTask const & task,
	FArray2DB_short & neighbor_indexno
)
{

	assert( rotamer_set.nmoltenres() == count_moltenres( task.get_designmap() ) );

	InteractionGraphBase* ig;
	bool on_the_fly_compatible = on_the_fly_graph_compatible_with_task();
	bool const use_sasa = ( use_sasa_pack_score && task.get_mode() != "optimizeH" );
	if ( on_the_fly_compatible )
	{
		if ( tight_memory_restrictions )
		{
			if ( predict_precomputed_graph_exceeds_memory_restriction( rotamer_set, neighbor_indexno ) )
			{
				if ( use_sasa )
				{
					ig = create_and_initialize_LinearMemorySASAInteractionGraph( rotamer_set, task );
				}
				else
				{
					ig = create_and_initialize_LinearMemoryInteractionGraph( rotamer_set );
				}
				return ig;
			}
		}

		if ( dynamically_decide_between_lazy_and_precomputed )
		{

			if ( predict_lazy_faster_than_precomputed( rotamer_set, neighbor_indexno ) )
			{
				if ( use_sasa )
				{
					ig = create_and_initialize_LazySASAInteractionGraph( rotamer_set, task );
				}
				else
				{
					ig = create_and_initialize_LazyInteractionGraph( rotamer_set );
				}
			}
			else
			{
				if ( use_sasa )
				{
					ig = 	create_and_initialize_SASAInteractionGraph( rotamer_set, task );
				}
				else
				{
					ig = 	create_and_initialize_PDInteractionGraph( rotamer_set );
				}
			}

		}

		else if ( lazy_ig )
		{
			if ( use_sasa )
			{
				ig = create_and_initialize_LazySASAInteractionGraph( rotamer_set, task );
			}
			else
			{
				ig = create_and_initialize_LazyInteractionGraph( rotamer_set );
			}

		}
		else if ( linmem_ig )
		{
			if ( use_sasa )
			{
				ig = create_and_initialize_LinearMemorySASAInteractionGraph( rotamer_set, task );
			}
			else
			{
				ig = create_and_initialize_LinearMemoryInteractionGraph( rotamer_set );
			}
		}
		else if ( minimalist_ig )
		{
			if ( use_sasa )
			{
				ig = create_and_initialize_MinimalistSASAInteractionGraph( rotamer_set, task );
			}
			else
			{
				ig = create_and_initialize_MinimalistInteractionGraph( rotamer_set );
			}
		}
		else
		{
			if ( use_sasa )
			{
				ig = 	create_and_initialize_SASAInteractionGraph( rotamer_set, task );
			}
			else
			{
				ig = 	create_and_initialize_PDInteractionGraph( rotamer_set );
			}

		}
	}
	else
	{
		ig = create_and_initialize_PDInteractionGraph( rotamer_set );
	}

	assert ( ig->get_num_nodes() == rotamer_set.nmoltenres() );

	return ig;
}

PDInteractionGraph *
create_and_initialize_PDInteractionGraph(
	RotamerSet & rotamer_set
)
{
	std::cout << "Instantiating PDInteractionGraph" << std::endl;
	PDInteractionGraph * ig = new PDInteractionGraph( rotamer_set.nmoltenres() );
	initialize_interaction_graph(
		dynamic_cast< PrecomputedPairEnergiesInteractionGraph* > (ig),
		rotamer_set );
	return ig;
}

SparseSASAInteractionGraph *
create_and_initialize_SASAInteractionGraph(
	RotamerSet & rotamer_set,
	PackerTask const & task
)
{
	std::cout << "Instantiating SASAInteractionGraph" << std::endl;
	SparseSASAInteractionGraph * ig = new SparseSASAInteractionGraph( rotamer_set.nmoltenres() );
	initialize_interaction_graph(
		ig,
		rotamer_set );
	initialize_sasa_graph(
		ig,
		rotamer_set,
		task.get_designmap() );
	return ig;
}

LazyInteractionGraph*
create_and_initialize_LazyInteractionGraph(
	RotamerSet & rotamer_set
)
{
	std::cout << "Instantiating LazyInteractionGraph" << std::endl;
	LazyInteractionGraph * ig = new LazyInteractionGraph( rotamer_set.nmoltenres() );
	initialize_on_the_fly_interaction_graph(
		ig,
		rotamer_set
	);
	return ig;
}

MinimalistInteractionGraph*
create_and_initialize_MinimalistInteractionGraph(
	RotamerSet & rotamer_set
)
{
	std::cout << "Instantiating MinimalistInteractionGraph" << std::endl;
	MinimalistInteractionGraph * ig = new MinimalistInteractionGraph( rotamer_set.nmoltenres() );
	initialize_on_the_fly_interaction_graph(
		ig,
		rotamer_set
	);
	return ig;
}

LinearMemoryInteractionGraph*
create_and_initialize_LinearMemoryInteractionGraph(
	RotamerSet & rotamer_set
)
{
	std::cout << "Instantiating LinearMemoryInteractionGraph" << std::endl;
	LinearMemoryInteractionGraph * ig = new LinearMemoryInteractionGraph( rotamer_set.nmoltenres() );
	initialize_on_the_fly_interaction_graph( ig, rotamer_set );
	return ig;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_interaction_graph
///
/// @brief
/// initializes the interaction graph so that it knows how many states
/// there are for each node, and the amino acid type of each state.
///
/// @detailed
///
/// @param  ig - [in/out] - the interaction graph in which to store rotamer
///    one body and two body energies.
/// @param  design_map
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
/// apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_interaction_graph
(
	PrecomputedPairEnergiesInteractionGraph * ig,
	RotamerSet & rotamer_set
)
{
	using namespace design_sym;
	using namespace misc;
	using namespace param;
	using namespace template_pack;

	int const nmoltenres = ig->get_num_nodes();
	if ( nmoltenres == 0 ) return;

	ig->set_num_aatypes( MAX_AA() );
	for (int ii = 1; ii <= rotamer_set.nmoltenres(); ++ii ) {
		int const ii_num_states = rotamer_set.num_states_for_moltenres(ii);
		int const ii_rotindex_offset = rotamer_set.nrotoffset( ii );
		std::vector<int> aatype_for_state( ii_num_states + 1);
		for (int jj = 1; jj <= ii_num_states; ++jj) {
			if ( rotamer_set.moltenres_2_resid(ii) > 0) {
				int const jj_rot_index = ii_rotindex_offset + jj;
				int const jjaa = rotamer_set.report_aa(jj_rot_index );
				aatype_for_state[ jj ] = jjaa;
			} else if (size_t(-rotamer_set.moltenres_2_resid(ii))<= ligand::ligand_ptr_vector.size()) {
				aatype_for_state[jj ] = 1;
			}
		}
		ig->set_num_states_for_node( ii, ii_num_states);
		ig->set_aatypes_for_node( ii, aatype_for_state );
	}

}


void
initialize_sasa_graph(
	SASAGraph * ig,
	RotamerSet & rotamer_set,
	const DesignMap & design_map
)
{
	using namespace misc;
	using namespace param;
	using namespace template_pack;

	input_sasa_dats();
	initialize_sasa_pack_arrays();

	ig->setNumResiduesInProtein( total_residue );
	int const nmoltenres = ig->get_num_nodes_v();
	int nbackground = total_residue - nmoltenres;
	ig->setNumBackgroundResidues( nbackground );
	for (int ii = 1; ii <= total_residue; ++ii) {
		if (! design_map.repack_residue(ii) ) {
			ig->setResidueAsBackgroundResidue( ii );

			FArray2Da_float bgrescoords( full_coord(1, 1, ii ), 3, MAX_ATOM() );
			RotamerCoords coords( res( ii ), res_variant( ii ), bgrescoords );

			ig->setBackgroundResidueRotamerCoords( ii, coords );
		}
	}

	for (int ii = 1; ii <= rotamer_set.nrotamers(); ++ii) {
		int const iiresid = rotamer_set.report_seqpos(ii);
		int const iinode = rotamer_set.resid_2_moltenres( iiresid );
		int const iistate = ii - rotamer_set.rotindex_offsets( iiresid );
		int const iiaa = rotamer_set.report_aa(ii);
		int const iiaav = rotamer_set.report_aav( ii );
		FArray2Da_float rotcoords( rotamer_set.get_rotcoord(ii), 3, MAX_ATOM() );

		RotamerCoords coords( iiaa, iiaav, rotcoords );

		ig->setRotamerCoordsForNodeState( iinode, iistate, coords );
	}

}

void
initialize_on_the_fly_interaction_graph
(
	OnTheFlyInteractionGraph * ig,
	RotamerSet & rotamer_set
)
{
	using namespace design_sym;
	using namespace misc;
	using namespace param;
	using namespace template_pack;

	int const nmoltenres = ig->get_num_nodes();
	if ( nmoltenres == 0 ) return;

	ig->set_num_aatypes( MAX_AA() );
	for (int ii = 1; ii <= nmoltenres; ++ii ) {
		int const ii_num_states = rotamer_set.num_states_for_moltenres(ii);
		int const ii_rotindex_offset = rotamer_set.nrotoffset( ii );
		std::vector<int> aatype_for_state( ii_num_states + 1);
		std::vector<int> aavtype_for_state( ii_num_states + 1 );
		for (int jj = 1; jj <= ii_num_states; ++jj) {
			if ( rotamer_set.moltenres_2_resid(ii) > 0) {
				int const jj_rot_index = ii_rotindex_offset + jj;
				int const jjaa = rotamer_set.report_aa(jj_rot_index );
				aatype_for_state[ jj ] = jjaa;
				aavtype_for_state[ jj ] = rotamer_set.report_aav( jj_rot_index );
			}else if (size_t(-rotamer_set.moltenres_2_resid(ii))<= ligand::ligand_ptr_vector.size()) {
				aatype_for_state[jj ] = 1;
				//apl lazy IG not ready for on-the-fly
				aavtype_for_state[jj] = 1;
			}
		}
		ig->set_num_states_for_node( ii, ii_num_states );
		ig->set_residue_index_for_node( ii, rotamer_set.moltenres_2_resid(ii) );
		ig->set_amino_acid_types_for_node( ii, aatype_for_state, aavtype_for_state );
	}

	for (int ii = 1; ii <= rotamer_set.nmoltenres(); ++ii )
	{
		int const ii_num_states = rotamer_set.num_states_for_moltenres(ii);
		int const ii_rotindex_offset = rotamer_set.nrotoffset( ii );
		std::vector<int> aatype_for_state( ii_num_states + 1);
		std::vector<int> aavtype_for_state( ii_num_states + 1 );
		for (int jj = 1; jj <= ii_num_states; ++jj) {
			int const jj_rot_index = ii_rotindex_offset + jj;
			FArray2Da_float jj_coords( rotamer_set.get_rotcoord(jj_rot_index), 3, MAX_ATOM() );
			FArray1Da_float jj_actcoord( rotamer_set.get_rotactcoord( jj_rot_index ), 3 );

			ig->set_rotamer_coordinates_for_node_state(
				ii, jj, jj_coords, jj_actcoord );
		}

	}

}

LazySASAInteractionGraph*
create_and_initialize_LazySASAInteractionGraph(
	RotamerSet & rotamer_set,
	PackerTask const & task
)
{
	std::cout << "Instantiating LazySASAInteractionGraph" << std::endl;
	LazySASAInteractionGraph * ig = new LazySASAInteractionGraph( rotamer_set.nmoltenres() );
	initialize_on_the_fly_interaction_graph( ig, rotamer_set );
	initialize_sasa_graph( ig, rotamer_set, task.get_designmap() );

	return ig;
}

MinimalistSASAInteractionGraph*
create_and_initialize_MinimalistSASAInteractionGraph(
	RotamerSet & rotamer_set,
	PackerTask const & task
)
{
	std::cout << "Instantiating MinimalistSASAInteractionGraph" << std::endl;
	MinimalistSASAInteractionGraph * ig = new MinimalistSASAInteractionGraph( rotamer_set.nmoltenres() );
	initialize_on_the_fly_interaction_graph( ig, rotamer_set );
	initialize_sasa_graph( ig, rotamer_set, task.get_designmap() );
	return ig;
}


LinMemSASAInteractionGraph*
create_and_initialize_LinearMemorySASAInteractionGraph(
	RotamerSet & rotamer_set,
	PackerTask const & task
)
{
	std::cout << "Instantiating LinearMemorySASAInteractionGraph" << std::endl;
	LinMemSASAInteractionGraph * ig = new LinMemSASAInteractionGraph( rotamer_set.nmoltenres() );
	initialize_on_the_fly_interaction_graph( ig, rotamer_set );
	initialize_sasa_graph( ig, rotamer_set, task.get_designmap() );

	return ig;
}



bool
on_the_fly_graph_compatible_with_task()
{
	return ( ! get_scale_res_energy_flag()
		&& ! classical_constraints::BOUNDARY::get_constraints_exist()
		&& ! param_pack::gen_born
		&& ! geometric_sol::geometric_sol_flag
		&& ( design_sym::num_clones == 0 )
		&& ! design::explicit_h2o
		&& ! get_ligand_flexible_flag()
		&& ! pKa_mode::get_pKa_flag()
		&& (! disulfides::BOUNDARY::get_disulf_flag() || disulfides::BOUNDARY::get_norepack_disulf() ) ) ;
}

bool
predict_precomputed_graph_exceeds_memory_restriction(
	RotamerSet & rotamer_set,
	FArray2DB_short & neighbor_indexno
)
{

	if ( rotamer_set.nmoltenres() == 0 ) return false;
	unsigned int num_rot_pairs = predict_num_rotamer_pairs( rotamer_set, neighbor_indexno );
	return ( sizeof( float ) * num_rot_pairs > (unsigned int) 1048576 * megs_for_rpes_limit );
}

bool
predict_lazy_faster_than_precomputed(
	RotamerSet & rotamer_set,
	FArray2DB_short & neighbor_indexno
)
{

	if ( rotamer_set.nmoltenres() == 0 ) return false;

	//number of rotamers at which lazy and precomputing break even assuming a fully
	//conntected interaction graph; ie num_rot_pairs == ( num_rots*(num_rots-1) ) / 2
	//inexact, of course
	int const crossover[2] = { 4800, 4300 };

	const int nrots = rotamer_set.nrotamers();
	if ( nrots < crossover[ use_multi_cool_annealer ] || design::pack_in_parallel ) return false;

	unsigned int num_rotamer_pairs = predict_num_rotamer_pairs( rotamer_set, neighbor_indexno );

	float const lazy_params[2] = {0.0081, 0.0110};
	float const sa_precomp_params[2] = {0.0013, 0.0049};
	float const RPEcalc_precomp_param = 2.8516E-6;

	float lazy_runtime_2p8GH_Xeon = nrots *  lazy_params[ use_multi_cool_annealer ];
	float precomp_runtime_2p8GHz_Xeon = nrots * sa_precomp_params[ use_multi_cool_annealer ] +
		num_rotamer_pairs * RPEcalc_precomp_param;

	return lazy_runtime_2p8GH_Xeon < precomp_runtime_2p8GHz_Xeon;
}

unsigned int
predict_num_rotamer_pairs(
	RotamerSet & rotamer_set,
	FArray2DB_short & neighbor_indexno
)
{
	using namespace param;

	if ( rotamer_set.nmoltenres() == 0 ) return 0;

	FArray2D_int num_rots_for_aa_for_moltres( MAX_AA(), rotamer_set.nrotamers(), 0 );
	for (int ii = 1; ii <= rotamer_set.nrotamers(); ++ii)
	{
		int const iiaa = rotamer_set.report_aa(ii);
		int const iimoltres = rotamer_set.resid_2_moltenres( rotamer_set.report_seqpos( ii ));
		++num_rots_for_aa_for_moltres( iiaa, iimoltres );
	}

	unsigned int total_rotamer_pairs=0;
	int const nmoltenres = rotamer_set.nmoltenres();

	for (int ii = 1; ii <= nmoltenres; ++ii)
	{
		int iiresid = rotamer_set.moltenres_2_resid( ii );
		for (int jj = ii+1; jj <= nmoltenres; ++jj)
		{
			int jjresid = rotamer_set.moltenres_2_resid( jj );
			if ( neighbor_indexno( iiresid, jjresid ) == 0 ) continue;

			for (int kk = 1; kk <= MAX_AA(); ++kk)
			{
				if (num_rots_for_aa_for_moltres(kk,ii) == 0 ) continue;
				for (int ll = 1; ll <= MAX_AA(); ++ll)
				{
					if (num_rots_for_aa_for_moltres(ll,jj) == 0 ) continue;
					bool are_neighbors;
					float dis2;
					are_they_neighbors(kk,ll,
						rotamer_set.get_first_rotcoord_for_aa(kk,ii),
						rotamer_set.get_first_rotcoord_for_aa(ll,jj),
						dis2,are_neighbors
					);

					if (are_neighbors)
					{
						total_rotamer_pairs +=
							num_rots_for_aa_for_moltres( kk, ii )
							* num_rots_for_aa_for_moltres( ll, jj );
					}
				}
			}
		}
	}
	return total_rotamer_pairs;
}

/*
void
writeDotsForRotamerAssignment(
	PrecomputedPairEnergiesInteractionGraph * ig,
	DesignMap & designmap,
	RotamerSet const & rotamer_set,
	FArray1DB_int & rotamer_assignment,
	int run
)
{
	using namespace template_pack;
	using namespace param;
	using namespace param_aa;

	if ( dynamic_cast<SASAInteractionGraph *> (ig) != 0)
	{
		write_output_dot_kinemage( dynamic_cast<SASAInteractionGraph * > (ig), run );
		return;
	}

	int const nmoltenres = ig->get_num_nodes();

	SASAInteractionGraph * sig = new SASAInteractionGraph( nmoltenres );
	sig->set_num_aatypes( 1 );

	{ //scope
	std::vector< int > alanine_aa_type( 2, aa_ala);
	for (int ii = 1; ii <= nmoltenres; ++ii)
	{
		sig->set_num_states_for_node(ii, 1);
		sig->set_aatypes_for_node(ii, alanine_aa_type );
	}

	}//end scope

	for (int ii = 1; ii <= nmoltenres; ++ii)
	{
		for (int jj = ii+1; jj <= nmoltenres; ++jj)
		{
			if (ig->get_edge_exists( ii, jj ) )
			{
				sig->add_edge( ii, jj );
				sig->force_all_aa_neighbors_for_edge(ii, jj );
			}
		}
	}

	int count_background_residues = 0;
	for (int ii = 1; ii <= nres; ++ii)
	{
		if ( designmap.repack_residue( ii ) )
		{
			int ii_node = rotamer_set.resid_2_moltenres( ii );
			int ii_rot = rotamer_assignment( ii );
			int ii_aa = rotindex( 2, ii_rot );
			int ii_aav = rot_aa_variant( ii_rot );
			FArray2Da_float ii_coords( rotcoord( 1, 1, ii_rot ), 3, MAX_ATOM() );
			RotamerCoords ii_rotcoords( ii_aa, ii_aav, ii_coords );

			sig->setRotamerCoordsForNodeState( ii_node, 1, ii_rotcoords );
		}
		else
		{
			++count_background_residues;
		}
	}

	sig->setNumResiduesInProtein( nres );
	sig->setNumBackgroundResidues( count_background_residues );
	for (int ii = 1; ii <= nres; ++ii )
	{
		if ( ! designmap.repack_residue( ii ) )
		{
			FArray2Da_float ii_coords( xyz(1, 1, ii ), 3, MAX_ATOM() );
			sig->setResidueAsBackgroundResidue( ii );

			int ii_aa = aan( ii );
			int ii_aav = aa_variant( ii );
			RotamerCoords ii_rotcoords( ii_aa, ii_aav, ii_coords );

			sig->setBackgroundResidueRotamerCoords( ii, ii_rotcoords );
		}
	}
	if ( ligand::ligand_one != 0 && ligand::ligand_one->ligand_flag)
	{
		sig->initialize_overlap_with_ligand( * ligand::ligand_one );
	}


	sig->prepare_for_simulated_annealing();
	FArray1D_int state_one( nmoltenres, 1 );
	sig->set_network_state( state_one );

	//write kin
	write_output_dot_kinemage( sig, run );
	delete sig;
}

void
write_output_dot_kinemage
(
	SASAInteractionGraph * sig,
	int run
)
{
	std::stringstream ss;
	std::string dotfilename = "output_dots_";
	std::string runstring;
	ss << run;
	ss >> runstring;
	dotfilename += runstring;
	dotfilename += ".kin";

	std::ofstream kinfile( dotfilename.c_str(), std::ios::out );
	sig->write_dot_kinemage( kinfile );
	kinfile.close();
}
*/


}//namespace pack
