// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.12 $
//  $Date: 2006/10/20 13:14:14 $
//  $Author: plato $

#include "MinimalistInteractionGraph.h"
#include "rotamer_trie_calc_energies.h"
#include "param_aa.h"

#include <iostream>

namespace pack{

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::MinimalistNode(InteractionGraphBase *, int, int)
///
/// @brief
/// main constructor, no default or copy constructors
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
MinimalistNode::MinimalistNode(
	InteractionGraphBase * owner,
	int node_id,
	int num_states
) :
	OnTheFlyNode( owner, node_id, num_states ),
	current_state_( 0 ),
	curr_state_one_body_energy_( 0.0f ),
	curr_state_total_energy_( 0.0f ),
	alternate_state_( 0 ),
	alternate_state_one_body_energy_( 0 ),
	alternate_state_total_energy_( 0 ),
	alternate_state_is_being_considered_( false )
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::~MinimalistNode()
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
MinimalistNode::~MinimalistNode()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::prepare_for_simulated_annealing
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::prepare_for_simulated_annealing()
{
	if (! get_edge_vector_up_to_date() ) update_internal_vectors();
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		mark_coordinates_current( ii );
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::print
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::print() const
{

	std::cerr << "MinimalistNode " << get_node_index() << " with " << get_num_states() << " states" << std::endl;
	std::cerr << "curr_state " << current_state_ << " ";
	std::cerr << "curr_state_sparse_mat_info_ ";
	std::cerr << curr_state_sparse_mat_info_.get_aa_type() << " ";
	std::cerr << curr_state_sparse_mat_info_.get_state_ind_for_this_aa_type() << " ";
	std::cerr << "Curr One Body Energy: " << curr_state_one_body_energy_ << std::endl;
	std::cerr << "Curr Total Energy: " << curr_state_total_energy_ << std::endl;
	std::cerr << "Curr Two Body Energies:";
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		std::cerr << " " << get_index_of_adjacent_node(ii) << ":" << curr_state_two_body_energies_[ ii ];
	}
	std::cerr << std::endl;

	if ( ! alternate_state_is_being_considered_ ) return;
	std::cerr << "Alt One Body Energy: " << alternate_state_one_body_energy_ << std::endl;
	std::cerr << "Alt Two Body Energies:";
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		std::cerr << " " << get_index_of_adjacent_node(ii) << ":" << alternate_state_two_body_energies_[ ii ];
	}
	std::cerr << std::endl  << "-----------------" << std::endl;


}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::getMemoryUsageInBytes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
unsigned int
MinimalistNode::count_static_memory() const
{
	return sizeof( MinimalistNode );
}

unsigned int
MinimalistNode::count_dynamic_memory() const
{
	unsigned int total_memory = OnTheFlyNode::count_dynamic_memory();

	total_memory += aa_neighbors_for_edges_.size() * sizeof( unsigned char );
	total_memory += neighbors_curr_state_.size() * sizeof( int );
	total_memory += neighbors_curr_state_sparse_info_.size() * sizeof( SparseMatrixIndex );

	total_memory += curr_state_two_body_energies_.size() * sizeof( float );
	total_memory += alternate_state_two_body_energies_.size() * sizeof( float );

	return total_memory;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::assign_zero_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::assign_zero_state()
{
	current_state_ = 0;
	alternate_state_ = 0;
	alternate_state_is_being_considered_ = false;

	curr_state_one_body_energy_ = 0.0f;
	//fills from [1] to end
	std::vector< float >::iterator position1 = curr_state_two_body_energies_.begin();
	++position1;
	std::fill( position1,
		curr_state_two_body_energies_.end(),
		0.0f);
	curr_state_total_energy_ = 0.0f;

	for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_minimalist_edge(ii)->
			acknowledge_state_zeroed( get_node_index() );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::assign_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::assign_state(int new_state)
{
	assert( new_state >= 0 && new_state <= get_num_states());

	if (new_state == 0) assign_zero_state();
	else
	{
		//std::cerr << "assign_state: node -  " << get_node_index() <<
		// " new state " << new_state << "...";
		current_state_ = new_state;
		curr_state_sparse_mat_info_ =
			get_sparse_mat_info_for_state( current_state_ );
		curr_state_one_body_energy_ = get_one_body_energy( current_state_ );
		curr_state_total_energy_ = curr_state_one_body_energy_;
		alternate_state_is_being_considered_ = false;

		for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
		{
			get_incident_minimalist_edge(ii)->acknowledge_state_change(
				get_node_index(),
				current_state_,
				curr_state_sparse_mat_info_,
				curr_state_two_body_energies_[ii]);

			curr_state_total_energy_ += curr_state_two_body_energies_[ ii ];
		}
		//std::cerr<< "..done" << std::endl;
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::partial_assign_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::partial_assign_state( int new_state )
{
	if (new_state == 0 )
	{
		assign_zero_state();
		return;
	}

	current_state_ = new_state;
	curr_state_sparse_mat_info_ =
		get_sparse_mat_info_for_state( current_state_ );
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_minimalist_edge(ii)->acknowledge_partial_state_change(
			get_node_index(),
			current_state_,
			curr_state_sparse_mat_info_);
	}
	alternate_state_is_being_considered_ = false;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::complete_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void MinimalistNode::complete_state_assignment()
{
	if ( current_state_ == 0 ) return;

	curr_state_total_energy_ = curr_state_one_body_energy_ =
		get_one_body_energy( current_state_ );
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		curr_state_two_body_energies_[ ii ] =
			get_incident_minimalist_edge( ii )->
			get_energy_following_partial_state_assignment();
		curr_state_total_energy_ += curr_state_two_body_energies_[ ii ];
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::project_deltaE_for_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistNode::project_deltaE_for_substitution
(
	int alternate_state,
	float & prev_node_energy
)
{
	alternate_state_is_being_considered_ = true;
	//procrastinated_ = false;
	//std::cerr << "proj_deltaE: node -  " << get_node_index()
	// << " alt state " << alternate_state << "...";

	alternate_state_ = alternate_state;
	rotamer_trie const & alternate_rotamer = get_rotamer( alternate_state_ );
	FArray1Da_float alt_rotamer_actcoords(
		get_rotamer_actcoords( alternate_state_ ), 3 );

	alt_state_sparse_mat_info_ = get_sparse_mat_info_for_state( alternate_state );
	alternate_state_one_body_energy_ = get_one_body_energy( alternate_state );
	alternate_state_total_energy_ = alternate_state_one_body_energy_;
	prev_node_energy = curr_state_total_energy_;

	int aa_neighb_linear_index_offset = aa_neighbors_for_edges_.
		index(1, 1, alt_state_sparse_mat_info_.get_aa_type() ) - 1;

	for (int ii = 1; ii <= get_num_incident_edges();
		++ii, aa_neighb_linear_index_offset += get_num_aa_types())
	{

		if ( neighbors_curr_state_[ ii ] != 0 &&
			aa_neighbors_for_edges_[ aa_neighb_linear_index_offset
			+ neighbors_curr_state_sparse_info_[ ii ].get_aa_type() ] )
		{

			FArray1Da_float neighbor_curr_rotamer_actcoords(
				get_adjacent_minimalist_node( ii )->
				get_curr_rotamer_actcoords(), 3 );

			alternate_state_two_body_energies_[ ii ] = compute_rotamer_pair_energy(
				ii,
				alternate_state_,
				neighbors_curr_state_[ ii ],
				alt_state_sparse_mat_info_.get_aa_type(),
				neighbors_curr_state_sparse_info_[ ii ].get_aa_type(),
				alternate_rotamer,
				get_adjacent_minimalist_node( ii )->get_current_rotamer(),
				alt_rotamer_actcoords,
				neighbor_curr_rotamer_actcoords
				);
		}
		else
		{
			alternate_state_two_body_energies_[ ii ] = 0;
		}

		alternate_state_total_energy_ += alternate_state_two_body_energies_[ ii ];
	}


	return alternate_state_total_energy_ - curr_state_total_energy_;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::commit_considered_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::commit_considered_substitution()
{
	assert( alternate_state_is_being_considered_ );

	current_state_ = alternate_state_;
	curr_state_sparse_mat_info_ = alt_state_sparse_mat_info_;
	curr_state_one_body_energy_ = alternate_state_one_body_energy_;
	curr_state_total_energy_ = alternate_state_total_energy_;

	//copies from [1] to end
	std::vector< float >::iterator alt_position1 = alternate_state_two_body_energies_.begin();
	++alt_position1;
	std::vector< float >::iterator curr_position1 = curr_state_two_body_energies_.begin();
	++curr_position1;

	std::copy( alt_position1,
		alternate_state_two_body_energies_.end(),
		 curr_position1 );

	for ( int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_minimalist_edge(ii)->acknowledge_substitution(
			get_node_index(),
			alternate_state_two_body_energies_[ ii ],
			current_state_,
			curr_state_sparse_mat_info_
		);
	}

	alternate_state_is_being_considered_ = false;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::compute_pair_energy_for_current_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistNode::compute_pair_energy_for_current_state(
	int edge_making_energy_request
)
{
	if ( aa_neighbors_for_edges_(
		neighbors_curr_state_sparse_info_[ edge_making_energy_request ].get_aa_type(),
		edge_making_energy_request,
		curr_state_sparse_mat_info_.get_aa_type() ) )
	{

		FArray1Da_float curr_rotamer_actcoords(
			get_rotamer_actcoords( current_state_ ), 3 );

		FArray1Da_float neighbor_curr_rotamer_actcoords(
			get_adjacent_minimalist_node( edge_making_energy_request )->
			get_curr_rotamer_actcoords(), 3 );

		return compute_rotamer_pair_energy(
			edge_making_energy_request,
			current_state_,
			neighbors_curr_state_[ edge_making_energy_request ],
			curr_state_sparse_mat_info_.get_aa_type(),
			neighbors_curr_state_sparse_info_[ edge_making_energy_request ].get_aa_type(),
			get_rotamer( current_state_ ),
			get_adjacent_minimalist_node( edge_making_energy_request )->get_current_rotamer(),
			curr_rotamer_actcoords,
			neighbor_curr_rotamer_actcoords
		);
	}
	else
	{
		return 0;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::acknowledge_neighbors_state_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::acknowledge_neighbors_state_substitution(
	int edge_to_altered_neighbor,
	float new_edge_energy,
	int other_node_new_state,
	SparseMatrixIndex const & other_node_new_state_sparse_info
)
{
	curr_state_total_energy_ +=
		new_edge_energy - curr_state_two_body_energies_[edge_to_altered_neighbor];
	curr_state_two_body_energies_[edge_to_altered_neighbor] = new_edge_energy;
	neighbors_curr_state_[ edge_to_altered_neighbor ] = other_node_new_state;
	neighbors_curr_state_sparse_info_[ edge_to_altered_neighbor ]  =
		other_node_new_state_sparse_info;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::acknowledge_neighbors_partial_state_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::acknowledge_neighbors_partial_state_substitution(
	int edge_to_altered_neighbor,
	int other_node_new_state,
	SparseMatrixIndex const & other_node_new_state_sparse_info
)
{
	curr_state_total_energy_ = 0;
	curr_state_two_body_energies_[ edge_to_altered_neighbor ] = 0;
	neighbors_curr_state_[ edge_to_altered_neighbor ] = other_node_new_state;
	neighbors_curr_state_sparse_info_[ edge_to_altered_neighbor ]  =
		other_node_new_state_sparse_info;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::get_sparse_mat_info_for_curr_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
SparseMatrixIndex const &
MinimalistNode::get_sparse_mat_info_for_curr_state() const
{
	return get_sparse_mat_info_for_state( current_state_ );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::print_internal_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::print_internal_energies()
{
	std::cerr << "curr_state " << current_state_ << " ";
	std::cerr << "curr_state_sparse_mat_info_ ";
	std::cerr << curr_state_sparse_mat_info_.get_aa_type() << " ";
	std::cerr << curr_state_sparse_mat_info_.get_state_ind_for_this_aa_type() << " ";
	std::cerr << "curr_state_one_body_energy_ ";
	std::cerr << curr_state_one_body_energy_ << " ";
	std::cerr << "curr_state_total_energy_" << curr_state_total_energy_ << " ";
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		std::cerr << "(" << get_index_of_adjacent_node(ii) << ":" << curr_state_two_body_energies_[ ii ] << ") ";
	}
	std::cerr << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::update_internal_energy_sums
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistNode::update_internal_energy_sums()
{
	assert( get_edge_vector_up_to_date() );
	curr_state_total_energy_ = 0;
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		curr_state_total_energy_ +=
			get_incident_minimalist_edge(ii)->get_current_two_body_energy();
	}
	curr_state_total_energy_ += curr_state_one_body_energy_;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::get_curr_rotamer_actcoords
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float &
MinimalistNode::get_curr_rotamer_actcoords()
{
	return get_rotamer_actcoords( current_state_ );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::compute_rotamer_pair_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::update_internal_vectors
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void MinimalistNode::update_internal_vectors()
{
	NodeBase::update_edge_vector();
	neighbors_curr_state_.resize( get_num_incident_edges() + 1);
	neighbors_curr_state_sparse_info_.resize( get_num_incident_edges() + 1);

	aa_neighbors_for_edges_.dimension(
		get_num_aa_types(), get_num_incident_edges(), get_num_aa_types());

	//copy sparse aa-neighbor info from edges
	int count_neighbs_with_higher_indices = 0;
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		neighbors_curr_state_sparse_info_[ii].set_aa_type( 1 );

		FArray2D< unsigned char > const & edge_aa_neighbs =
			get_incident_minimalist_edge(ii)->get_sparse_aa_neighbor_info();

		if ( get_node_index() < get_index_of_adjacent_node(ii) )
		{  ++count_neighbs_with_higher_indices;
			for ( int jj = 1; jj <= get_num_aa_types(); ++jj )
			{
				for ( int kk = 1; kk <= get_num_aa_types(); ++kk )
				{
					aa_neighbors_for_edges_(kk, ii, jj) = edge_aa_neighbs(kk, jj);
				}
			}
		}
		else
		{
			for ( int jj = 1; jj <= get_num_aa_types(); ++jj )
			{
				for ( int kk = 1; kk <= get_num_aa_types(); ++kk )
				{
					aa_neighbors_for_edges_(kk, ii, jj) =
						edge_aa_neighbs(jj, kk);
				}
			}
		}
	}

	curr_state_two_body_energies_.resize( get_num_incident_edges() + 1);
	alternate_state_two_body_energies_.resize( get_num_incident_edges() + 1);
	return;
}

// @ brief - allow derived class to "drive" through the deltaE calculation
void
MinimalistNode::calc_deltaEpd( int alternate_state )
{
	float dummy(0.0f);
	project_deltaE_for_substitution( alternate_state, dummy );
}


//-----------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::MinimalistEdge
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
MinimalistEdge::MinimalistEdge(
	InteractionGraphBase* owner,
	int first_node_ind,
	int second_node_ind
):
	OnTheFlyEdge( owner, first_node_ind, second_node_ind),
	sparse_aa_neighbors_(
		get_minimalist_ig_owner()->get_num_aatypes(),
		get_minimalist_ig_owner()->get_num_aatypes(),
		(unsigned char) 0 ),
	curr_state_energy_( 0.0f ),
	partial_state_assignment_( false )
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::~MinimalistEdge()
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
MinimalistEdge::~MinimalistEdge()
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::set_sparse_aa_info
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistEdge::set_sparse_aa_info(
	FArray2DB_bool const & aa_neighbors
)
{
	for (int ii = 1; ii <= get_minimalist_ig_owner()->get_num_aatypes(); ++ii)
	{
		for (int jj = 1; jj <= get_minimalist_ig_owner()->get_num_aatypes(); ++jj)
		{
			if ( aa_neighbors( jj, ii ) )
			{
				sparse_aa_neighbors_( jj, ii ) = 1;
			}
			else
			{
				sparse_aa_neighbors_( jj, ii ) = 0;
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistNode::declare_energies_final
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistEdge::declare_energies_final()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::prepare_for_simulated_annealing
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistEdge::prepare_for_simulated_annealing()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::getMemoryUsageInBytes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

unsigned int
MinimalistEdge::count_static_memory() const
{
	return sizeof( MinimalistEdge );
}


unsigned int
MinimalistEdge::count_dynamic_memory() const
{
	unsigned int total_memory = OnTheFlyEdge::count_dynamic_memory();
	total_memory += sparse_aa_neighbors_.size() * sizeof( unsigned char );
	return total_memory;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::get_current_two_body_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistEdge::get_current_two_body_energy() const
{
	return curr_state_energy_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::acknowledge_state_change
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistEdge::acknowledge_state_change(
	int node_ind,
	int new_state,
	SparseMatrixIndex const & new_state_sparse_info,
	float & new_energy
)
{
	int node_substituted =  ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	curr_state_energy_ = get_minimalist_node( 0 )->
			compute_pair_energy_for_current_state(
			get_edges_position_in_nodes_edge_vector( 0 ) );

	//get_energy_for_state_pair( nodes_curr_states, nodes_curr_states_sparse_info );
	new_energy = curr_state_energy_;

	get_minimalist_node( node_not_substituted )->
		acknowledge_neighbors_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		new_state,
		new_state_sparse_info
		);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::acknowledge_state_zeroed
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistEdge::acknowledge_state_zeroed( int node_ind )
{
	int node_substituted =  ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	curr_state_energy_ = 0;
	SparseMatrixIndex dummy_sparse_info;
	dummy_sparse_info.set_aa_type( 1 );
	dummy_sparse_info.set_state_ind_for_this_aa_type(1);

	get_minimalist_node( node_not_substituted )->
		acknowledge_neighbors_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		0,
		dummy_sparse_info
		);
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::acknowledge_partial_state_change
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void MinimalistEdge::acknowledge_partial_state_change(
	int node_ind,
	int new_state,
	SparseMatrixIndex const & new_state_sparse_info
)
{
	int node_substituted =  ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	curr_state_energy_ = 0;

	get_minimalist_node( node_not_substituted )->
		acknowledge_neighbors_partial_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		new_state,
		new_state_sparse_info
		);
	partial_state_assignment_ = true;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::get_energy_following_partial_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistEdge::get_energy_following_partial_state_assignment()
{
	if (partial_state_assignment_
		&& get_minimalist_node(0)->get_current_state() != 0
		&& get_minimalist_node(1)->get_current_state() != 0)
	{
		curr_state_energy_ = get_minimalist_node( 0 )->
			compute_pair_energy_for_current_state(
			get_edges_position_in_nodes_edge_vector( 0 ) );
		partial_state_assignment_ = false;
	}
	return curr_state_energy_;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::get_two_body_table_size
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int MinimalistEdge::get_two_body_table_size() const
{
	return 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::acknowledge_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
void
MinimalistEdge::acknowledge_substitution(
	int substituted_node_index,
	float const curr_state_energy,
	int nodes_new_state,
	SparseMatrixIndex const & nodes_new_state_sparse_info
)
{
	int node_substituted = substituted_node_index == get_node_index(0) ? 0 : 1;
	int node_not_substituted = ! node_substituted;

	curr_state_energy_ = curr_state_energy;

	get_minimalist_node( node_not_substituted )->
		acknowledge_neighbors_state_substitution
		(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		nodes_new_state,
		nodes_new_state_sparse_info
	);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::get_sparse_aa_neighbor_info
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
FArray2D< unsigned char > const &
MinimalistEdge::get_sparse_aa_neighbor_info( )
{
	return sparse_aa_neighbors_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistEdge::print_current_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistEdge::print_current_energy() const
{
	using namespace param_aa;
	std::cerr << "MinimalistEdge: " << get_node_index( 0 ) << "/" << get_node_index( 1 );
	std::cerr << " energy= " << curr_state_energy_ << std::endl;
	if ( get_minimalist_node( 0 )->get_sparse_mat_info_for_curr_state().get_aa_type() == aa_pro )
	{
		std::cerr << "MinimalistEdge: " << get_node_index( 0 ) << "/" << get_node_index( 1 );
		std::cerr << " proline correction 0: " << get_proline_correction( 1,  get_minimalist_node( 1 )->get_current_state() ) << std::endl;
	}
	if ( get_minimalist_node( 1 )->get_sparse_mat_info_for_curr_state().get_aa_type() == aa_pro )
	{
		std::cerr << "MinimalistEdge: " << get_node_index( 0 ) << "/" << get_node_index( 1 );
		std::cerr << " proline correction 1: " << get_proline_correction( 0,  get_minimalist_node( 0 )->get_current_state() ) << std::endl;
	}
}


//-------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::MinimalistInteractionGraph
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
MinimalistInteractionGraph::MinimalistInteractionGraph(
	int numNodes
) : OnTheFlyInteractionGraph( numNodes )
{
	//MinimalistNode::num_rpe_calcs = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::~MinimalistInteractionGraph
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
MinimalistInteractionGraph::~MinimalistInteractionGraph()
{
}


////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::blanket_assign_state_0
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistInteractionGraph::blanket_assign_state_0()
{
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_minimalist_node( ii )->assign_zero_state();
	}
	total_energy_current_state_assignment_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::set_state_for_node
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistInteractionGraph::set_state_for_node(int node_ind, int new_state)
{
	get_minimalist_node( node_ind )->assign_state( new_state );
	update_internal_energy_totals();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::set_network_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistInteractionGraph::set_network_state( FArray1DB_int & node_states)
{
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_minimalist_node( ii )->partial_assign_state( node_states( ii ) );
	}
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_minimalist_node( ii )->complete_state_assignment();
	}
	update_internal_energy_totals();
	//std::cerr << "Set Network State Finished" << std::endl;
	//print_current_state_assignment();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::consider_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistInteractionGraph::consider_substitution(
	int node_ind,
	int new_state,
	float & delta_energy,
	float & prev_energy_for_node
)
{
	node_considering_alt_state_ = node_ind;

	delta_energy = get_minimalist_node( node_ind )->
		project_deltaE_for_substitution( new_state, prev_energy_for_node );

	total_energy_alternate_state_assignment_ =
		total_energy_current_state_assignment_ + delta_energy;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::commit_considered_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistInteractionGraph::commit_considered_substitution()
{
	get_minimalist_node( node_considering_alt_state_ )->commit_considered_substitution();

	total_energy_current_state_assignment_ =
		total_energy_alternate_state_assignment_;

	++num_commits_since_last_update_;
	if (num_commits_since_last_update_ == COMMIT_LIMIT_BETWEEN_UPDATES)
	{
		update_internal_energy_totals();
	}

	return total_energy_alternate_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::get_energy_current_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistInteractionGraph::get_energy_current_state_assignment()
{
	//std::cerr << "Num rotamer pair energy calculations performed: " << MinimalistNode::num_rpe_calcs << std::endl;
	update_internal_energy_totals();
	return total_energy_current_state_assignment_;
}

// @ breif O(1) total energy report.  Protected read access for derived classes.
float MinimalistInteractionGraph::get_energy_PD_current_state_assignment()
{
	return total_energy_current_state_assignment_;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::get_edge_memory_usage
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
MinimalistInteractionGraph::get_edge_memory_usage() const
{
	int sum = 0;
	for (std::list< EdgeBase* >::const_iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{
		sum += ((MinimalistEdge*) *iter)->get_two_body_table_size();
	}
	return sum;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::print_current_state_assignment
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistInteractionGraph::print_current_state_assignment() const
{
	std::cerr << "State Assignment: " << std::endl;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		std::cerr << "Node " << ii << " state " << get_minimalist_node(ii)->get_current_state() << std::endl;
		get_minimalist_node(ii)->print();
	}

	for (std::list< EdgeBase* >::const_iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{
		((MinimalistEdge*) (*iter))->print_current_energy();
	}
	std::cerr << "Energy: " << total_energy_current_state_assignment_ << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::set_errorfull_deltaE_threshold
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistInteractionGraph::set_errorfull_deltaE_threshold( float )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::get_energy_sum_for_vertex_group
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
MinimalistInteractionGraph::get_energy_sum_for_vertex_group( int group_id )
{
	float esum = 0;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		if ( get_vertex_member_of_energy_sum_group( ii, group_id ) )
		{
			esum += get_minimalist_node( ii )->get_one_body_energy_current_state();
		}
	}

	for ( std::list< EdgeBase* >::iterator edge_iter = get_edge_list_begin();
		edge_iter != get_edge_list_end(); ++edge_iter)
	{
		int first_node_ind = (*edge_iter)->get_first_node_ind();
		int second_node_ind = (*edge_iter)->get_second_node_ind();

		if ( get_vertex_member_of_energy_sum_group( first_node_ind, group_id )
			&& get_vertex_member_of_energy_sum_group( second_node_ind, group_id ))
		{
			esum += ((MinimalistEdge*) (*edge_iter))->get_current_two_body_energy();
		}
	}

	return esum;
}

bool
MinimalistInteractionGraph::build_sc_only_rotamer() const
{
	return true;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::getMemoryUsageInBytes()
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

unsigned int
MinimalistInteractionGraph::count_static_memory() const
{
	return sizeof( MinimalistInteractionGraph );
}

unsigned int
MinimalistInteractionGraph::count_dynamic_memory() const
{
	unsigned int total_memory = OnTheFlyInteractionGraph::count_dynamic_memory();
	return total_memory;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::create_new_node( int node_index, int num_states)
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
NodeBase*
MinimalistInteractionGraph::create_new_node( int node_index, int num_states)
{
	return new MinimalistNode( this, node_index, num_states );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::create_new_edge( int index1, int index2)
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
EdgeBase*
MinimalistInteractionGraph::create_new_edge( int index1, int index2)
{
	return new MinimalistEdge( this, index1, index2 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin MinimalistInteractionGraph::update_internal_energy_totals
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
MinimalistInteractionGraph::update_internal_energy_totals()
{
	total_energy_current_state_assignment_ = 0;

	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		total_energy_current_state_assignment_ += get_minimalist_node( ii )->
			get_one_body_energy_current_state();
	}

	for (std::list<EdgeBase*>::iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{
		total_energy_current_state_assignment_ +=
			((MinimalistEdge*) *iter)->get_current_two_body_energy();
	}

	num_commits_since_last_update_ = 0;
	return;
}

} //end namespace pack
