// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

//xa   Implementation for new simulated annealing routine that will optimize
//xa   a single sequence for multiple structures (Ambroggio & Kuhlman 2006)

// Rosetta Headers
#include "design.h"              // needed for conv_limit_mod
#include "files_paths.h"         // needed for reading in equiv_resfile
#include "EquivResfileReader.h"
#include "FixbbSimAnnealer.h"
#include "InteractionGraphBase.h" // needed for calling ig functions
#include "misc.h"                // needed for res_num_from_pdb_res_num_chain
#include "pack.h"                // needed for rotindex
#include "ParallelSimAnneal.h"
#include "param.h"               // needed for MAX_RES
#include "pdb.h"
#include "NegativeDesign.h"
#include "ResidueData.h"
#include "RotamerSet.h"
#include "XUtilities.cc"

// C++ Headers
#include <fstream>
#include <iostream>
#include <set>

//Utility Headers
#include <utility/basic_sys_util.hh>

namespace pack{


  ////////////////////////////////////////////////////////////////////////////////

  NegativeDesign::NegativeDesign(
				 const RotamerSet * p_rotamer_set,
				 FArray2DB_bool & neighborlist,
				 FArray1D_int & bestrotamer_at_seqpos,
				 float & bestenergy,
				 bool start_with_current, // start simulation with current rotamers
				 pack::InteractionGraphBase * ig,
				 FArray1DB_int & current_rot_index,
				 bool calc_rot_freq,
				 FArray1D_float & rot_freq
				 ):
    ParallelSimAnneal(
		      p_rotamer_set,
		      neighborlist,
		      bestrotamer_at_seqpos,
		      bestenergy,
		      start_with_current,
		      ig,
		      current_rot_index,
		      calc_rot_freq,
		      rot_freq
		      )

  {}

  ////////////////////////////////////////////////////////////////////////////////

  NegativeDesign::~NegativeDesign()
  {}

  void NegativeDesign::setup_vertex_groups_for_negative_design()
  {
    using namespace design;
    std::cout << "Determine the number of protein conformations.\n";

    num_prot_conformations = ( ig_->count_connected_components_and_initialize_vertex_groups() );
    std::cout << "Number of protein conformations: " << num_prot_conformations << "\n";

    int moltenres_out(0);
    moltenres_in_given_chain( chain_in_preferred_state, moltenres_out );
    std::cout << "moltenres_out " << moltenres_out << "\n";

    for( int i = 1; i <= num_prot_conformations; i++ ){
      if( ig_->get_vertex_member_of_energy_sum_group( moltenres_out, i ) ){
	preferred_state = i;
	break;
      }
    }

    std::cout << "Preferred state: " << preferred_state << std::endl;
  }

  void NegativeDesign::moltenres_in_given_chain( std::string const & pdb_chain_in, int & moltenres_out )
  {

    using namespace misc;
    using namespace pdb;

    std::string chain_ID;

    for( int i = 1; i <= total_residue; ++i ){
      chain_ID = res_chain(i);
      if( chain_ID == pdb_chain_in && p_rotamer_set_->resid_2_moltenres(i) > 0 ) {
	moltenres_out = p_rotamer_set_->resid_2_moltenres(i);
	return;
      }
    }

    std::cout << "WARNING: chain " << pdb_chain_in << " not found, exiting.\n";
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

  }

  bool NegativeDesign::evaluate_probability_of_preferred_state()
  {
    bool accept = false;
    bool lower_energy = false;
    bool higher_prob = false;
    bestenergy_ = 0.0;
    delta_energy = 0.0;

    bestenergy_ =  ig_->get_energy_sum_for_vertex_group( preferred_state );

    delta_energy = bestenergy_ - previous_energy;

    lower_energy = pass_metropolis( delta_energy );

    //std::cout << boolalpha << "delta_energy " << delta_energy << " bestenergy_ " << bestenergy_ << " prev_energy " << previous_energy
    //	      << " lower energy? " << lower_energy << "\n";

    double boltz_factor = 0.0;
    double preferred_state_boltz_factor = 0.0;
    double sum_of_states_boltz_factors = 0.0;
    double current_state_energy = 0.0;
    double current_probability_of_preferred_state = 0.0;

    //   std::cout << "Calculate probabilities for each protein conformation.\n";

    for(int i = 1; i <= num_prot_conformations; i++){
      if( i == reference_state ){
	sum_of_states_boltz_factors += reference_state_boltz_factor;
	continue;
      }
      current_state_energy = std::min( std::max( double( ig_->get_energy_sum_for_vertex_group( i ) ), -600.0 ), 600.0 );
      boltz_factor = std::exp( -current_state_energy );
      if( i == preferred_state ){
	preferred_state_boltz_factor = boltz_factor;
      }
      sum_of_states_boltz_factors += boltz_factor;
    }

    current_probability_of_preferred_state = ( preferred_state_boltz_factor / sum_of_states_boltz_factors );

    if( current_probability_of_preferred_state >= previous_probability_of_preferred_state ) higher_prob = true;

    //std::cout << boolalpha << " curr_pref_state_prob " << current_probability_of_preferred_state << " prev_pref_state_prob " << previous_probability_of_preferred_state
    //      << " higher prob? " << higher_prob << "\n";

    if( quench() && lower_energy && higher_prob ) accept = true;
    else if( !quench() && ( higher_prob || lower_energy ) ) accept = true;


    if(accept == true) previous_probability_of_preferred_state = current_probability_of_preferred_state;

    return accept;

  }

  double NegativeDesign::setup_ref_state_and_return_boltz_factor()
  {
    using namespace design;

    if( chain_in_ref_state == "none" ){
      reference_state = 0;
      return 0.0;
    }

    std::cout << "Setting up the reference state.\n";

    int moltenres_out(0);
    moltenres_in_given_chain( chain_in_ref_state, moltenres_out );
    std::cout << "Molten residue in reference state " << moltenres_out << "\n";

    for(int i = 1; i <= num_prot_conformations; i++)
      if( ig_->get_vertex_member_of_energy_sum_group(moltenres_out, i) ){
	reference_state = i;
	break;
      }


    std::cout << "Reference state: " << reference_state << std::endl;

    std::cout << "Calculating boltzmann factor approximation of the reference state.\n";

    start_with_current_ = false;

    SimAnnealerBase * sa = new FixbbSimAnnealer(  bestrotamer_at_seqpos_, bestenergy_,
						  start_with_current_, ig_, p_rotamer_set_,
						  current_rot_index_, calc_rot_freq_, rot_freq_);

    sa->run();
    delete sa;

    start_with_current_ = false;

    double ref_state_energy, ref_state_boltz_fact;

    ref_state_energy = std::min( std::max( double( ig_->get_energy_sum_for_vertex_group( reference_state ) ), -600.0 ), 600.0 );
    std::cout << "Reference state energy: " << ref_state_energy << "\n";

    ref_state_boltz_fact = std::exp( -ref_state_energy );
    std::cout << "Reference state Boltzmann factor approximation: " << ref_state_boltz_fact << "\n";

    return ref_state_boltz_fact;
  }

////////////////////////////////////////////////////////////////////////////////
/// @begin NegativeDesign::run
///
/// @brief Design complexes modeling all states
///
/// @refrences Ambroggio & Kuhlman, JACS 2006
///
/// @authors Xavier Ambroggio
////////////////////////////////////////////////////////////////////////////////

  void NegativeDesign::run()
  {
    using namespace design;
    std::cout << "Starting Parallel Simulated Annealing with Negative Design\n";

    setup_vertex_groups_for_negative_design();

    reference_state_boltz_factor = setup_ref_state_and_return_boltz_factor();

    create_a_random_sequence();

    fill_rot_to_pack_for_current_sequence();

    if( rot_to_pack.empty() ) return;

    pack_set_of_rotamers();

    previous_energy = energy_of_best_run = bestenergy_ = ig_->get_energy_sum_for_vertex_group( preferred_state );

    accept_new_sequence_from_packing_run();

    start_with_current_ = true;

    setup_iterations( number_of_possible_mutations() );

    for( int i = 1; i <= get_outeriterations(); i++ ){ // annealing outer loop
      setup_temperature( i );

      for( int j = 1; j <= get_inneriterations() + ( i * 200 * conv_limit_mod ); j++ ){ // inner loop for mutations

	make_a_sequential_substitution();

	propogate_mutations_to_equivalent_residues();

	add_rotamers_to_pack_for_residue_list( mutant_sequence_positions );

	add_rotamers_to_pack_for_neighbors_of_mutated_residues();

	add_current_rotamers_and_start_with_them();

	if( rot_to_pack.empty() ) continue;

	pack_set_of_rotamers();

	delta_energy = bestenergy_ - previous_energy;

	if( evaluate_probability_of_preferred_state() ){
	  accept_new_sequence_from_packing_run();
	}else{
	  reject_mutations();
	}
      }
      revert_to_best_sequence();
    }
    output_best_rotamers();
  }


} //end of namespace
