// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.15 $
//  $Date: 2006/10/10 13:18:28 $
//  $Author: plato $


#include "OnTheFlyInteractionGraph.h"
#include "param_aa.h"
#include "rotamer_trie_calc_energies.h"

#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>

#include <iostream>

namespace pack
{

unsigned int OnTheFlyNode::num_rpe_calcs(0);

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::OnTheFlyNode
///
/// @brief
/// main constructor, no default or copy constructors
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
OnTheFlyNode::OnTheFlyNode(
	InteractionGraphBase * owner,
	int node_id,
	int num_states
) :
	NodeBase( owner, node_id, num_states ),
	num_aa_types_( get_on_the_fly_owner()->get_num_aatypes() ),
	num_states_for_aatype_( num_aa_types_, 0 ),
	sparse_mat_info_for_state_( num_states + 1 ),
	rot_aa_variants_( num_states + 1, 0 ),
	rotamers_( num_states + 1, (rotamer_trie*) 0 ),
	coordinates_current_( num_states + 1, false ),
	num_coordinates_not_current_( num_states ),
	rotactcoords_( 3, num_states, 0.0f ),
	one_body_energies_( num_states, 0.0f ),
	res_id_( 0 )
{

}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::~OnTheFlyNode
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
OnTheFlyNode::~OnTheFlyNode()
{
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		delete rotamers_[ ii ];
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::set_residue_index
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::set_residue_index( int residue_index )
{
	assert( res_id_ == 0 );
	res_id_ = residue_index;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::set_amino_acid_types
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::set_amino_acid_types(
	std::vector< int > const & aatypes,
	std::vector< int > const & aavtypes
)
{
	assert(aatypes.size() == (unsigned int) (get_num_states() + 1) );
	assert(aavtypes.size() == (unsigned int) (get_num_states() + 1) );

	for (int ii = 1; ii <= get_num_states(); ii++)
	{
		assert( aatypes[ii] > 0 &&
			aavtypes[ii] <= num_aa_types_);

		sparse_mat_info_for_state_[ii].set_aa_type( aatypes[ii] );

		++(num_states_for_aatype_(sparse_mat_info_for_state_[ii].get_aa_type()) );

		sparse_mat_info_for_state_[ii].set_state_ind_for_this_aa_type(
			num_states_for_aatype_(sparse_mat_info_for_state_[ii].get_aa_type()));
		rot_aa_variants_[ ii ] = aavtypes[ ii ];
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::set_rotamer_coordinates
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::set_rotamer_coordinates(
	int state,
	FArray2DB_float const & rotcoords,
	FArray1DB_float const & rotactcoord,
  bool const mark_as_changed // default true
)
{
	rotamer_descriptor rd;
	create_rotamer_descriptor(
		res_id_,
		rd,
		sparse_mat_info_for_state_[ state ].get_aa_type(),
		rot_aa_variants_[ state ],
		rotcoords,
		1);

	delete rotamers_[ state ]; rotamers_[ state ] = new rotamer_trie(
		res_id_,
		rd,
		get_on_the_fly_owner()->build_sc_only_rotamer() );
	FArray1Da_float state_actcoords( rotactcoords_(1, state ), 3 );
	state_actcoords = rotactcoord;

  if ( mark_as_changed ) {
		if ( coordinates_current_[ state ] ) {
			++num_coordinates_not_current_;
			coordinates_current_[ state ] = false;
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::zero_one_body_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::zero_one_body_energies()
{
	one_body_energies_ = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::add_to_one_body_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::add_to_one_body_energies( FArray1DB_float & energy1b )
{
	one_body_energies_ += energy1b;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::set_one_body_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::set_one_body_energy( int state, float energy )
{
	one_body_energies_( state ) = energy;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::add_to_one_body_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::add_to_one_body_energy( int state, float energy )
{
	one_body_energies_( state ) +=  energy;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::zero_one_body_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyNode::zero_one_body_energy( int state )
{
	one_body_energies_( state ) = 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyNode::getMemoryUsageInBytes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
unsigned int
OnTheFlyNode::count_dynamic_memory() const
{
	unsigned int total_memory = NodeBase::count_dynamic_memory();

	total_memory += num_states_for_aatype_.size() * sizeof( int );
	total_memory += sparse_mat_info_for_state_.size() * sizeof( SparseMatrixIndex );
	total_memory += rot_aa_variants_.size() * sizeof( int );
	total_memory += rotamers_.size() * sizeof ( rotamer_trie* );
	//for (int ii = 1; ii <= get_num_states(); ++ii)
	//{
	//	total_memory += rotamers_[ ii ]->get_memory_use();
	//}
	total_memory += coordinates_current_.size() * sizeof( bool ); //inaccurate?
	total_memory += rotactcoords_.size() * sizeof( float );
	total_memory += one_body_energies_.size() * sizeof( float );

	return total_memory;
}


float
OnTheFlyNode::compute_rotamer_pair_energy(
	int edge_making_energy_request,
	int state_this,
	int state_other,
	int aa_this,
	int aa_other,
	rotamer_trie const & this_rotamer,
	rotamer_trie const & other_rotamer,
	FArray1Da_float & this_rotamer_actcoords,
	FArray1Da_float & other_rotamer_actcoords
)
{
	using namespace param_aa;

	++num_rpe_calcs;
	float esum = get_sc_scE_trie( aa_this, aa_other,
		this_rotamer, other_rotamer, this_rotamer_actcoords, other_rotamer_actcoords );

	if ( aa_other == aa_pro )
	{
		esum += get_incident_otf_edge( edge_making_energy_request )->
			get_proline_correction_for_node( get_node_index(), state_this );
	}
	if ( aa_this == aa_pro )
	{
		esum += get_incident_otf_edge( edge_making_energy_request )->
			get_proline_correction_for_node( get_index_of_adjacent_node( edge_making_energy_request ),
			state_other
		);
	}

	//	if ( ! get_on_the_fly_owner()->check_empty_weight_map() ) {
	esum *= get_on_the_fly_owner()->get_residue_weights(this_rotamer.get_resid(), aa_this,
																											other_rotamer.get_resid(), aa_other);
		//	}

	return esum;

}


//-------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyEdge::OnTheFlyEdge
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
OnTheFlyEdge::OnTheFlyEdge(
	InteractionGraphBase * owner,
	int first_node_ind,
	int second_node_ind)
:
	EdgeBase( owner, first_node_ind, second_node_ind )
{
	for ( int ii = 0; ii < 2; ++ii)
	{
		proline_corrections_[ ii ].dimension( get_num_states_for_node( ii ) );
		proline_corrections_[ ii ] = 0;
	}
}


OnTheFlyEdge::~OnTheFlyEdge(){}

void
OnTheFlyEdge::set_ProCorrection_values(
	int node_not_necessarily_proline,
	int state,
	float bb_nonprobb_E,
	float bb_probb_E,
	float sc_nonprobb_E,
	float sc_probb_E
)
{
	int const which_node = node_not_necessarily_proline == get_node_index( 0 ) ? 0 : 1;

	proline_corrections_[ which_node ]( state ) =
		sc_probb_E + 0.5 * bb_probb_E -
		(sc_nonprobb_E + 0.5 * bb_nonprobb_E);
}


unsigned int
OnTheFlyEdge::count_dynamic_memory() const
{
	unsigned int total_memory_usage = EdgeBase::count_dynamic_memory();

	total_memory_usage += proline_corrections_[ 0 ].size() * sizeof( float );
	total_memory_usage += proline_corrections_[ 1 ].size() * sizeof( float );

	return total_memory_usage;
}

//-------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::OnTheFlyInteractionGraph
///
/// @brief
/// main constructor, no default or copy constructors
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
OnTheFlyInteractionGraph::OnTheFlyInteractionGraph(int num_nodes )
:
	InteractionGraphBase( num_nodes ),
	num_aa_types_( 0 )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::~OnTheFlyInteractionGraph
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
OnTheFlyInteractionGraph::~OnTheFlyInteractionGraph()
{}


////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::set_num_aatypes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::set_num_aatypes( int num_aa_types )
{
	assert( num_aa_types_ == 0 );
	num_aa_types_ = num_aa_types;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::set_residue_index_for_node
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::set_residue_index_for_node(
	int node_ind,
	int residue_index
)
{
	get_on_the_fly_node( node_ind )->set_residue_index( residue_index );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::set_amino_acid_types_for_node
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::set_amino_acid_types_for_node(
	int node_ind,
	std::vector< int > const & aatypes,
	std::vector< int > const & aavtypes
)
{
	get_on_the_fly_node( node_ind )->
		set_amino_acid_types(
		aatypes,
		aavtypes );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::set_rotamer_coordinates_for_node_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::set_rotamer_coordinates_for_node_state(
	int node_ind,
	int state,
	FArray2DB_float const & rotcoords,
	FArray1DB_float const & rotactcoords
)
{
	get_on_the_fly_node( node_ind )->
		set_rotamer_coordinates(
		state,
		rotcoords,
		rotactcoords);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::quietly_set_rotamer_coordinates_for_node_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::quietly_set_rotamer_coordinates_for_node_state(
	int node_ind,
	int state,
	FArray2DB_float const & rotcoords,
	FArray1DB_float const & rotactcoords
)
{
	get_on_the_fly_node( node_ind )->
		set_rotamer_coordinates(
		state,
		rotcoords,
		rotactcoords,
		true);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::zero_one_body_energy_for_node_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::zero_one_body_energy_for_node_state(
	int node_ind,
	int state
)
{
	get_on_the_fly_node( node_ind )->zero_one_body_energy( state );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::add_to_one_body_energy_for_node_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::add_to_one_body_energy_for_node_state(
	int node_ind,
	int state,
	float one_body_energy
)
{
	get_on_the_fly_node( node_ind )->add_to_one_body_energy( state, one_body_energy);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::set_one_body_energy_for_node_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::set_one_body_energy_for_node_state(
	int node_ind,
	int state,
	float one_body_energy
)
{
	get_on_the_fly_node( node_ind )->set_one_body_energy( state, one_body_energy);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::get_one_body_energy_for_node_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
OnTheFlyInteractionGraph::get_one_body_energy_for_node_state( int node, int state)
{
	return get_on_the_fly_node( node )->get_one_body_energy( state );
}

void
OnTheFlyInteractionGraph::reset_rpe_calculations_count()
{
	OnTheFlyNode::num_rpe_calcs = 0;
}

unsigned int
OnTheFlyInteractionGraph::get_num_rpe_calculations_count() const
{
	return OnTheFlyNode::num_rpe_calcs;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin OnTheFlyInteractionGraph::set_sparse_aa_info_for_edge
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
OnTheFlyInteractionGraph::set_sparse_aa_info_for_edge(
	int node1,
	int node2,
	FArray2DB_bool const & sparse_conn_info
)
{
	OnTheFlyEdge* edge = (OnTheFlyEdge*) find_edge( node1, node2 );
	if (edge != 0)
	{
		edge->set_sparse_aa_info( sparse_conn_info );
	}
}

void
OnTheFlyInteractionGraph::set_ProCorrection_values_for_edge(
	int node1,
	int node2,
	int node_not_neccessarily_proline,
	int state,
	float bb_nonprobb_E,
	float bb_probb_E,
	float sc_nonprobb_E,
	float sc_probb_E
)
{
	OnTheFlyEdge* edge = (OnTheFlyEdge*) find_edge( node1, node2 );
	if (edge != 0)
	{
		edge->set_ProCorrection_values(
			node_not_neccessarily_proline, state,
			bb_nonprobb_E, bb_probb_E, sc_nonprobb_E, sc_probb_E );
	}
}

unsigned int
OnTheFlyInteractionGraph::count_dynamic_memory() const
{
	unsigned int total_memory = InteractionGraphBase::count_dynamic_memory();
	//total_memory += residue_weight_map_.dynamic_size(); //need to implement
	return total_memory;
}


} //namespace pack

