// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.16 $
//  $Date: 2006/10/10 13:18:28 $
//  $Author: plato $

#ifndef ON_THE_FLY_INTERACTION_GRAPH_INTERFACE_H
#define ON_THE_FLY_INTERACTION_GRAPH_INTERFACE_H

#include "AminoAcidNeighborSparseMatrix.h"
#include "rotamer_descriptor.h"
#include "SparseMatrixIndex.h"
#include "InteractionGraphBase.h"
#include "PackerTaskResidueWeights.h"

#include <vector>
#include <ObjexxFCL/ObjexxFCL.Project.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

namespace pack {

class OnTheFlyNode;
class OnTheFlyEdge;
class OnTheFlyInteractionGraph;

class OnTheFlyNode : public NodeBase
{
public:
	OnTheFlyNode(
		InteractionGraphBase * owner,
		int node_id,
		int num_states);

	virtual ~OnTheFlyNode();

	void
	set_residue_index( int residue_index );

	void
	set_amino_acid_types(
		std::vector< int > const & aatypes,
		std::vector< int > const & aavtypes
	);

	void set_rotamer_coordinates(
		int state,
		FArray2DB_float const & rotcoords,
		FArray1DB_float const & rotactcoord,
		bool const mark_as_changed = true
	);

	virtual void zero_one_body_energies();
	virtual void add_to_one_body_energies( FArray1DB_float & energy1b );
	virtual void set_one_body_energy( int state, float energy );
	virtual void add_to_one_body_energy( int state, float energy );
	virtual void zero_one_body_energy( int state );

	inline
	FArray1D_int &
	get_num_states_for_aa_types()
	{
		return num_states_for_aatype_;
	}

	inline
	int
	get_num_states_for_aa_type( int aa_type )
	{
		return num_states_for_aatype_( aa_type );
	}

	inline
	SparseMatrixIndex const &
	get_sparse_mat_info_for_state( int state ) const
	{
		assert( state > 0 && state <= get_num_states() );
		return sparse_mat_info_for_state_[ state ];
	}

	inline
	float
	get_one_body_energy( int state ) const
	{
		return one_body_energies_( state );
	}

	inline
	bool
	get_coordinates_current( int state )
	{
		assert( state > 0 && state <= get_num_states() );
		return (! get_any_coordinates_not_current() ||  coordinates_current_[ state ]);
	}

	inline
	bool
	get_any_coordinates_not_current()
	{
		return num_coordinates_not_current_ != 0;
	}

	static unsigned int num_rpe_calcs;

	virtual unsigned int count_dynamic_memory() const;

protected:

	float
	compute_rotamer_pair_energy(
		int edge_making_energy_request,
		int state_this,
		int state_other,
		int aa_this,
		int aa_other,
		rotamer_trie const & this_rotamer,
		rotamer_trie const & other_rotamer,
		FArray1Da_float & this_rotamer_actcoords,
		FArray1Da_float & other_rotamer_actcoords
	);

	inline
	OnTheFlyEdge*
	get_incident_otf_edge( int edge )
	{
		return ((OnTheFlyEdge * ) get_incident_edge( edge ) );
	}

	inline
	OnTheFlyInteractionGraph *
	get_on_the_fly_owner() const
	{
		return ((OnTheFlyInteractionGraph* ) get_owner() );
	}

	inline
	rotamer_trie const &
	get_rotamer( int state ) const
	{
		assert( state > 0 && state <= get_num_states() );
		return * rotamers_[ state ];
	}

	inline
	void
	mark_coordinates_current( int state )
	{
		assert( state > 0 && state <= get_num_states() );
		if ( ! coordinates_current_[ state ] )
		{
			coordinates_current_[ state ] = true;
			--num_coordinates_not_current_;
		}
	}

	inline
	int
	get_num_aa_types() const
	{
		return num_aa_types_;
	}


	inline
	float &
	get_rotamer_actcoords( int state )
	{
		assert( state > 0 && state <= get_num_states() );
		return rotactcoords_(1, state );
	}

private:
	int num_aa_types_;
	FArray1D_int num_states_for_aatype_;
	std::vector< SparseMatrixIndex > sparse_mat_info_for_state_;
	std::vector< int > rot_aa_variants_;
	std::vector< rotamer_trie* > rotamers_;
	std::vector< bool > coordinates_current_;
	int num_coordinates_not_current_;
	FArray2D_float rotactcoords_;
	FArray1D_float one_body_energies_;
	int res_id_;

};

class OnTheFlyEdge : public EdgeBase
{
public:
	virtual ~OnTheFlyEdge();

	OnTheFlyEdge(
		InteractionGraphBase * owner,
		int first_node_ind,
		int second_node_ind);

	virtual
	void
	set_sparse_aa_info(
		FArray2DB_bool const &
	) = 0;

	void
	set_ProCorrection_values(
		int node_not_necessarily_proline,
		int state,
		float bb_nonprobb_E,
		float bb_probb_E,
		float sc_nonprobb_E,
		float sc_probb_E
	);

	inline
	float
	get_proline_correction_for_node(
		int node_ind,
		int state
	)
	{
		int which_node = node_ind == get_node_index( 0 ) ? 0 : 1;
		return get_proline_correction( which_node, state );
	}

	virtual unsigned int count_dynamic_memory() const;

protected:

	inline
	float get_proline_correction(
		int which_node,
		int state) const
	{
		return proline_corrections_[ which_node ]( state );
	}


private:
	FArray1D_float proline_corrections_[ 2 ];
};

class OnTheFlyInteractionGraph : public InteractionGraphBase
{

public:
	OnTheFlyInteractionGraph(int num_nodes );
	~OnTheFlyInteractionGraph();


	void set_num_aatypes( int );

	inline
	int get_num_aatypes() const
	{
		return num_aa_types_;
	}

  // for using ResidueWeightMap
  inline void set_residue_weight_map(PackerTaskResidueWeightMap const & residue_weight_map_in) {
		residue_weight_map_ = residue_weight_map_in;
	}
	inline float get_residue_weights(int seqpos1, int aa1, int seqpos2, int aa2 ) const {
		{ return residue_weight_map_.get_weight(seqpos1, aa1, seqpos2, aa2); };
	}
	inline bool check_empty_weight_map() { return residue_weight_map_.check_empty_map(); };

	void
	set_residue_index_for_node(
		int node_ind,
		int residue_index
	);

	void
	set_amino_acid_types_for_node(
		int node_ind,
		std::vector< int > const & aatypes,
		std::vector< int > const & aavtypes
	);

	void
	set_rotamer_coordinates_for_node_state(
		int node_ind,
		int state,
		FArray2DB_float const & rotcoords,
		FArray1DB_float const & rotactcoords
	);

	void
	quietly_set_rotamer_coordinates_for_node_state(
		int node_ind,
		int state,
		FArray2DB_float const & rotcoords,
		FArray1DB_float const & rotactcoords
	);

	void
	zero_one_body_energy_for_node_state(
		int node_ind,
		int state
	);

	void
	add_to_one_body_energy_for_node_state(
		int node_ind,
		int state,
		float one_body_energy
	);

	void
	set_one_body_energy_for_node_state(
		int node_ind,
		int state,
		float one_body_energy
	);

	virtual float get_one_body_energy_for_node_state( int node, int state);

	void
	set_sparse_aa_info_for_edge(
		int node1,
		int node2,
		FArray2DB_bool const & sparse_conn_info
	);


	void
	reset_rpe_calculations_count() ;


	unsigned int
	get_num_rpe_calculations_count() const;

	virtual bool build_sc_only_rotamer() const = 0;

	void
	set_ProCorrection_values_for_edge(
		int node1,
		int node2,
		int node_not_neccessarily_proline,
		int state,
		float bb_nonprobb_E,
		float bb_probb_E,
		float sc_nonprobb_E,
		float sc_probb_E
	);

	//virtual unsigned int count_static_memory() const = 0;
	virtual unsigned int count_dynamic_memory() const;

protected:

	inline
	OnTheFlyNode * get_on_the_fly_node( int node_index )
	{
		return ((OnTheFlyNode*) get_node( node_index ) ); //c-style cast instead of static_cast
	}


private:
	int num_aa_types_;

	// Additional per-residue (or per-residue-per-aa) weights
	// Note: currently computed OUTSIDE the IG, except in the case of OnTheFly IGs
	PackerTaskResidueWeightMap residue_weight_map_;

};


}

#endif

