// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14208 $
//  $Date: 2007-04-13 06:59:32 -0700 (Fri, 13 Apr 2007) $
//  $Author: leaverfa $

// Rosetta Headers
#include "PDInteractionGraph.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray2Da.hh>

// STL Headers
#include <list>
#include <vector>
#include <algorithm>
#include <iostream>
#include <fstream>
#include <cassert>

//Utility Headers
#include <utility/basic_sys_util.hh>

namespace pack {

//----------------------------------------------------------------------------//
//-------- Sparse Pairwise Decomposable Interaction Graph Node Class ---------//
//----------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::PDNode(PDInteractionGraph *, int, int)
///
/// @brief
/// main constructor, no default or copy constructors
///
/// @detailed
/// allocates one-body energy array and initializes it to zero.
/// allocates space for sparse-matrix information.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
PDNode::PDNode(InteractionGraphBase * owner, int node_id, int num_states) :
	PrecomputedPairEnergiesNode( owner, node_id, num_states ),
	num_aa_types_( get_pdig_owner()->get_num_aatypes() ),
	num_states_for_aatype_( num_aa_types_, 0 ),
	sparse_mat_info_for_state_( num_states + 1),
	one_body_energies_(num_states + 1, 0.0f),
	curr_state_total_energy_( 0.0 ),
	alternate_state_is_being_considered_( false )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::~PDNode
///
/// @brief
/// destructor
///
/// @detailed
/// not responsible for any dynamically allocated memory, so node does nothing
/// it's member variables, of course, are implicitly destructed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
PDNode::~PDNode()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::print()
///
/// @brief
/// prints a description of the node and all of it's one-body energies
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::print() const
{
	std::cerr << "NODE: " << get_node_index() << " with " <<
		get_num_states() << " states" << std::endl;
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		std::cerr << "(" << ii << ", " <<
			sparse_mat_info_for_state_[ii].get_aa_type() << ", ";
		std::cerr <<
			sparse_mat_info_for_state_[ii].get_state_ind_for_this_aa_type() << ", ";
		std::cerr << one_body_energies_[ ii ] << ") ";
		if ( ii % 3 == 0 ) std::cerr << std::endl;
	}
	std::cerr << std::endl  << "-----------------" << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::set_amino_acid_types
///
/// @brief
/// sets the amino acid type for each state
///
/// @detailed
/// The graph doesn't know anything specific about amino acid types, for
/// instance, nothing specially distinguishes glycine and threonine.
/// The sparse-matrix representation for the PDEdge two-body energy tables
/// requires knowing when two states are of the *same* amino acid type and
/// when they are *different* amino acid types.
///
/// @param
/// aatypes_for_state - [in] - amino acid type for each state in the node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::set_amino_acid_types( std::vector< int > const & aatypes_for_state)
{
	assert(aatypes_for_state.size() == (unsigned int) (get_num_states() + 1) );

	int owners_num_aatypes = num_states_for_aatype_.size();
	for (int ii = 1; ii <= get_num_states(); ii++)
	{
		assert( aatypes_for_state[ii] > 0 &&
			aatypes_for_state[ii] <= owners_num_aatypes);

		sparse_mat_info_for_state_[ii].set_aa_type( aatypes_for_state[ii] );

		++(num_states_for_aatype_(sparse_mat_info_for_state_[ii].get_aa_type()) );

		sparse_mat_info_for_state_[ii].set_state_ind_for_this_aa_type(
			num_states_for_aatype_(sparse_mat_info_for_state_[ii].get_aa_type()));
	}
	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_num_states_for_aa_types
///
/// @brief
/// returns an FArray & with the number of states for each amino acid type
///
/// @detailed
/// Used by AminoAcidNeighborSparseMatrix.  The FArray must persist for
/// as long as any AminoAcidNeighborSparseMatrix points to it.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
FArray1D_int const &
PDNode::get_num_states_for_aa_types() const
{
	return num_states_for_aatype_;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::update_one_body_energy
///
/// @brief
/// update energy to the one-body energy for state
///
/// @detailed
///
/// @param
/// state - [in] - one-based index of the state
/// @param
/// energy - [in] - the energy that should be set.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::update_one_body_energy( int state, float energy )
{
	one_body_energies_[ state ] = energy;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::update_one_body_energies
///
/// @brief
/// set all the one-body energies for this node
///
/// @detailed
///
/// @param
/// energies - [in] - the array of energies. Must hold num_states_ entries
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::update_one_body_energies( FArray1DB_float & energies )
{
	assert( energies.size() == (unsigned int) get_num_states() );
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		one_body_energies_[ ii ] = energies( ii );
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::add_to_one_body_energy
///
/// @brief
/// adds energy to the one-body energy for state state
///
/// @detailed
///
/// @param
/// state - [in] - one-based index of the state
/// @param
/// energy - [in] - the energy that should be added.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::add_to_one_body_energy( int state, float energy )
{
	one_body_energies_[ state ] += energy;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::add_to_one_body_energies
///
/// @brief
/// adds all the energies in energies to the one-body energies for this node
///
/// @detailed
///
/// @param
/// energies - [in] - the array of energies. Must hold num_states_ entries
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::add_to_one_body_energies( FArray1DB_float & energies )
{
	assert( energies.size() == (unsigned int) get_num_states() );
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		one_body_energies_[ ii ] += energies( ii );
	}
	return;
}

void PDNode::zero_one_body_energies()
{
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		one_body_energies_[ ii ] = 0;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_one_body_energy
///
/// @brief
/// returns the one body energy for a state
///
/// @detailed
///
/// @param
/// state - [in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDNode::get_one_body_energy( int state )
{
	return one_body_energies_[ state ];
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::prepare_for_simulated_annealing
///
/// @brief
/// prepares node for simulated annealing
///
/// @detailed
/// updates internal edge vector + other vectorized edge information
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::prepare_for_simulated_annealing()
{
	if (! get_edge_vector_up_to_date() ) update_internal_vectors();
	return;
}


unsigned int
PDNode::count_static_memory() const
{
	return sizeof( PDNode );
}

unsigned int
PDNode::count_dynamic_memory() const
{
	//std::cout << "calling PDNode::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = NodeBase::count_dynamic_memory();
	total_memory += num_states_for_aatype_.size() * sizeof( int );
	total_memory += sparse_mat_info_for_state_.size() * sizeof( SparseMatrixIndex );
	total_memory += one_body_energies_.size() * sizeof( float );
	total_memory += neighbors_curr_state_.size() * sizeof (int );

	total_memory += aa_offsets_for_edges_.size() * sizeof( int );
	total_memory += num_states_for_aa_type_for_higher_indexed_neighbor_.size() * sizeof( int );
	total_memory += neighbors_curr_state_.size() * sizeof( int );
	total_memory += neighbors_curr_state_sparse_info_.size() * sizeof( SparseMatrixIndex );
	total_memory += edge_matrix_ptrs_.size() * sizeof( FArray1Da_float );

	total_memory += curr_state_two_body_energies_.size() * sizeof( float );
	total_memory += alternate_state_two_body_energies_.size() * sizeof( float );
	return total_memory;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::assign_zero_state
///
/// @brief
/// assigns node's state to it's zero, or "unassigned" state.
///
/// @detailed
/// zeros the edge-energy array, informs neighbors that it's in its unassigned
/// state
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::assign_zero_state()
{

	//std::cerr << "assign_state: node -  " << get_node_index() <<
	//	" new state " << 0 << "...";

	current_state_ = 0;
	alternate_state_ = 0;
	alternate_state_is_being_considered_ = false;

	curr_state_one_body_energy_ = 0.0f;
	//fills from [1] to end
	std::vector< float >::iterator position1 = curr_state_two_body_energies_.begin();
	++position1;
	std::fill( position1,
		curr_state_two_body_energies_.end(),
		0.0f);
	curr_state_total_energy_ = 0.0f;

	for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_pd_edge(ii)->
			acknowledge_state_zeroed( get_node_index() );
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::assign_state
///
/// @brief
/// assigns node a new_state
///
/// @detailed
/// node updates its curr_state one and two body energies
///
/// @param
/// new_state - [in] - the new state the node should be assigned
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::assign_state(int new_state)
{
	assert( new_state >= 0 && new_state <= get_num_states());

	if (new_state == 0) assign_zero_state();
	else
	{
		//std::cerr << "assign_state: node -  " << get_node_index() <<
		// " new state " << new_state << "...";
		current_state_ = new_state;
		curr_state_sparse_mat_info_ =
			sparse_mat_info_for_state_[ current_state_ ];
		curr_state_one_body_energy_ = one_body_energies_[ current_state_ ];
		curr_state_total_energy_ = curr_state_one_body_energy_;
		alternate_state_is_being_considered_ = false;

		for (int ii = 1; ii <= get_num_incident_edges(); ++ii )
		{
			get_incident_pd_edge(ii)->acknowledge_state_change(
				get_node_index(),
				current_state_,
				curr_state_sparse_mat_info_,
				curr_state_two_body_energies_[ii]);

			curr_state_total_energy_ += curr_state_two_body_energies_[ ii ];
		}
		//std::cerr<< "..done" << std::endl;
	}
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_current_state
///
/// @brief
/// returns the state the node is currently assigned
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDNode::get_current_state() const
{
	return current_state_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_one_body_energy_current_state
///
/// @brief
/// returns the one body energy for the state the node is currently assigned
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDNode::get_one_body_energy_current_state() const
{	return curr_state_one_body_energy_;}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::project_deltaE_for_substitution
///
/// @brief
/// returns the change in energy that would be induced by switching this node
/// from its current state into another state
///
/// @detailed
/// iterates across the incident edges for a node in two phases:
/// in the first phase, it examines edges leading to higher-indexed nodes
/// in the second phase, it examines edges leading to smaller-indexed nodes.
/// for cache efficiency, all of the amino-acid-neighbor-offset information
/// that each edge calculates is stored on the nodes themselves.  The edges
/// are never touched; rather, their private information is stored on the nodes
/// and handed to static member functions of the PDEdge class.  This "store
/// edge information on the nodes" strategy gives me performance equivalent
/// to the previous energy2b lookup tables.
///
/// @param
/// alternate_state - [in] - the alternate state to consider
/// @param
/// previous_energy_for_node - [out] - the old energy1b/energy2b sum for this
/// node; used by simulate annealing.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
float
PDNode::project_deltaE_for_substitution(
	int alternate_state,
	float & prev_energy_for_node
)
{

	alternate_state_is_being_considered_ = true;
	//std::cout << "proj_deltaE: node -  " << get_node_index() << " alt state " << alternate_state << "...";

	alternate_state_ = alternate_state;
	alt_state_sparse_mat_info_ = sparse_mat_info_for_state_[ alternate_state];
	alternate_state_one_body_energy_ = one_body_energies_[ alternate_state ];
	//std::cout << "alternate_state_one_body_energy_: " << alternate_state_one_body_energy_ << std::endl;
	alternate_state_total_energy_ = alternate_state_one_body_energy_;
	prev_energy_for_node = curr_state_total_energy_;

	//lookup offsets to have on hand:
	int aa_neighb_linear_index_offset = aa_offsets_for_edges_.
		index(1, 1, alt_state_sparse_mat_info_.get_aa_type() ) - 1;
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{


	}

	int alt_state_num_states_per_aa_type =
		num_states_for_aatype_( alt_state_sparse_mat_info_.get_aa_type() );
	int alt_state_for_aa_type_minus_1 =
		alt_state_sparse_mat_info_.get_state_ind_for_this_aa_type() - 1;
	int nstates_offset =
		num_states_for_aa_type_for_higher_indexed_neighbor_.index(1,1) - 1;

		for (int ii = 1; ii <= get_num_edges_to_smaller_indexed_nodes();
		++ii, aa_neighb_linear_index_offset += num_aa_types_)
		{

			alternate_state_two_body_energies_[ ii ] =
				get_incident_pd_edge(ii)->
					get_alternate_state_energy_second_node(
					neighbors_curr_state_[ii],
					alternate_state_,
					neighbors_curr_state_sparse_info_[ ii ],
					alt_state_sparse_mat_info_,
					alt_state_num_states_per_aa_type,
					aa_offsets_for_edges_[
						aa_neighb_linear_index_offset +
						neighbors_curr_state_sparse_info_[ii].get_aa_type()
					],
					edge_matrix_ptrs_[ii]
				);
			alternate_state_total_energy_ += alternate_state_two_body_energies_[ ii ];
			//std::cout << "edge: " << ii << " " << alternate_state_two_body_energies_[ ii ] << std::endl;
		}

	for (int ii = get_num_edges_to_smaller_indexed_nodes() + 1;
		ii <= get_num_incident_edges();
		++ii, aa_neighb_linear_index_offset += num_aa_types_,
		nstates_offset += num_aa_types_)
	{
		alternate_state_two_body_energies_[ ii ] =
			get_incident_pd_edge(ii)->
				get_alternate_state_energy_first_node(
				alternate_state_,
				neighbors_curr_state_[ii],
				//alt_state_sparse_mat_info_,
				neighbors_curr_state_sparse_info_[ii],
				alt_state_for_aa_type_minus_1,
				num_states_for_aa_type_for_higher_indexed_neighbor_[
					nstates_offset +
					neighbors_curr_state_sparse_info_[ii].get_aa_type()
				],
				aa_offsets_for_edges_[
					aa_neighb_linear_index_offset +
					neighbors_curr_state_sparse_info_[ii].get_aa_type()
				],
				edge_matrix_ptrs_[ii]
			);
		alternate_state_total_energy_ += alternate_state_two_body_energies_[ ii ];
		//std::cout << "edge: " << ii << " " << alternate_state_two_body_energies_[ ii ] << std::endl;
	}

	//std::cerr<< "..done" << std::endl;

	return alternate_state_total_energy_ - curr_state_total_energy_;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::commit_considered_substitution
///
/// @brief
/// tells the node that it should change its state to the last state it was
/// asked to consider (from a call to project_deltaE_for_substitution)
///
/// @detailed
/// updates edge energy vector, iterates across neighbors having them update
/// their edge energies.  Bookkeeping recaptures performance lost by
/// leaving energy2b structure
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::commit_considered_substitution()
{
	assert( alternate_state_is_being_considered_ );

	current_state_ = alternate_state_;
	curr_state_sparse_mat_info_ = alt_state_sparse_mat_info_;
	curr_state_one_body_energy_ = alternate_state_one_body_energy_;
	curr_state_total_energy_ = alternate_state_total_energy_;

	//copies from [1] to end
	std::vector< float >::iterator alt_position1 = alternate_state_two_body_energies_.begin();
	++alt_position1;
	std::vector< float >::iterator curr_position1 = curr_state_two_body_energies_.begin();
	++curr_position1;

	std::copy( alt_position1,
		alternate_state_two_body_energies_.end(),
		 curr_position1 );

	for ( int ii = 1; ii <= get_num_incident_edges(); ++ii )
	{
		get_incident_pd_edge(ii)->acknowledge_substitution(
			get_node_index(),
			alternate_state_two_body_energies_[ii],
			current_state_,
			curr_state_sparse_mat_info_
		);
	}

	alternate_state_is_being_considered_ = false;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::acknowledge_neighbors_state_substitution
///
/// @brief
/// updates bookkeeping arrays for when a neighbor has changed its state
///
/// @detailed
///
/// @param
/// edge_to_altered_neighbor - [in] - the index for the edge that connects
/// 	this node to the node that just changed its state
/// @param
/// new_edge_energ - [in] - the pair energy between this node in its current
///	state and the new state of the node that just changed its state
/// @param
/// other_node_new_state - [in] - the state the neighbor just adopted
/// @param
/// other_node_new_state_sparse_info - [in] - the sparse-matrix info
///	corresponding to the neighbor's new state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
void PDNode::acknowledge_neighbors_state_substitution(
	int edge_to_altered_neighbor,
	float new_edge_energy,
	int other_node_new_state,
	SparseMatrixIndex const & other_node_new_state_sparse_info
)
{

	curr_state_total_energy_ +=
		new_edge_energy - curr_state_two_body_energies_[edge_to_altered_neighbor];
	curr_state_two_body_energies_[edge_to_altered_neighbor] = new_edge_energy;
	neighbors_curr_state_[ edge_to_altered_neighbor ] = other_node_new_state;
	neighbors_curr_state_sparse_info_[ edge_to_altered_neighbor ]  =
		other_node_new_state_sparse_info;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::update_edge_vector
///
/// @brief
/// updates bookkeeping arrays that correspond to edge-list.
///
/// @detailed
/// calls base class update_edge_vector function, and then proceeds to create
/// appropriate bookkeeping arrays used in simulated annealing
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::update_internal_vectors()
{
	NodeBase::update_edge_vector();
	//aa_offsets_for_this_lookup_.resize( get_num_incident_edges() + 1);
	neighbors_curr_state_.resize( get_num_incident_edges() + 1);
	neighbors_curr_state_sparse_info_.resize( get_num_incident_edges() + 1);

	edge_matrix_ptrs_.clear();
	edge_matrix_ptrs_.reserve( get_num_incident_edges() + 1);
	edge_matrix_ptrs_.push_back( FArray1Da_float() ); //occupy the 0th position

	aa_offsets_for_edges_.dimension(
		num_aa_types_, get_num_incident_edges(), num_aa_types_);
	num_states_for_aa_type_for_higher_indexed_neighbor_.dimension(
		num_aa_types_, get_num_edges_to_larger_indexed_nodes());

	//copy offsets from edges
	//int neighb_aa_offset =
	//	num_states_for_aa_type_for_higher_indexed_neighbor_.index(1,1);
	int count_neighbs_with_higher_indices = 0;
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		neighbors_curr_state_sparse_info_[ii].set_aa_type( 1 );

		float & edge_table_ref =
			get_incident_pd_edge(ii)->get_edge_table_ptr();
		edge_matrix_ptrs_.push_back( FArray1Da_float( edge_table_ref ));

		int edge_table_size = get_incident_pd_edge(ii)
			->get_two_body_table_size();
		edge_matrix_ptrs_[ii].dimension( edge_table_size );

		FArray2D_int const & edge_aa_neighb_offsets =
			get_incident_pd_edge(ii)->get_offsets_for_aatypes();
		FArray1D_int const & neighb_num_states_per_aa =
			get_incident_pd_edge(ii)->get_second_node_num_states_per_aa();

		if ( get_node_index() < get_index_of_adjacent_node(ii) )
		{  ++count_neighbs_with_higher_indices;
			for ( int jj = 1; jj <= num_aa_types_; ++jj )
			{
				for ( int kk = 1; kk <= num_aa_types_; ++kk )
				{
					aa_offsets_for_edges_(kk, ii, jj) = edge_aa_neighb_offsets(kk, jj);
				}
				num_states_for_aa_type_for_higher_indexed_neighbor_(
					jj, count_neighbs_with_higher_indices) =
					neighb_num_states_per_aa( jj );
				//++neighb_aa_offset;
			}
		}
		else
		{
			for ( int jj = 1; jj <= num_aa_types_; ++jj )
			{
				for ( int kk = 1; kk <= num_aa_types_; ++kk )
				{
					aa_offsets_for_edges_(kk, ii, jj) =
						edge_aa_neighb_offsets(jj, kk);
				}
			}
		}
	}

	curr_state_two_body_energies_.resize( get_num_incident_edges() + 1);
	alternate_state_two_body_energies_.resize( get_num_incident_edges() + 1);
	return;
}

// @ brief - allow derived class to "drive" through the deltaE calculation
void
PDNode::calc_deltaEpd( int alternate_state )
{
	//std::cout << "PDNode::calc_deltaEpd" << std::endl;

	float dummy(0.0f);
	project_deltaE_for_substitution( alternate_state, dummy );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_sparse_mat_info_for_state
///
/// @brief
/// returns the sparse matrix information for a paricular state
///
/// @detailed
///
/// @param
/// state - [in] - the state whose information is requested
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
SparseMatrixIndex const &
PDNode::get_sparse_mat_info_for_state(int state) const
{
	assert( state > 0 && state <= get_num_states());
	return sparse_mat_info_for_state_[ state ];
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_sparse_mat_info_for_curr_state
///
/// @brief
/// returns the sparse matrix information for the current state
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
SparseMatrixIndex const &
PDNode::get_sparse_mat_info_for_curr_state() const
{
	return curr_state_sparse_mat_info_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_num_states_for_aa_type
///
/// @brief
/// returns the number of states that are of a particular amino acid type
///
/// @detailed
///
/// @param
/// aa_type - [in] - the amino acid type in question
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
PDNode::get_num_states_for_aa_type(int aa_type) const
{
	return num_states_for_aatype_( aa_type );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::print_internal_energies
///
/// @brief
/// outputs to standard error the bookkeeping energies for the node in its
/// current state assignment
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::print_internal_energies()
{
	std::cerr << "curr_state " << current_state_ << " ";
	std::cerr << "curr_state_sparse_mat_info_ ";
	std::cerr << curr_state_sparse_mat_info_.get_aa_type() << " ";
	std::cerr << curr_state_sparse_mat_info_.get_state_ind_for_this_aa_type() << " ";
	std::cerr << "curr_state_one_body_energy_ ";
	std::cerr << curr_state_one_body_energy_ << " ";
	std::cerr << "curr_state_total_energy_" << curr_state_total_energy_ << " ";
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		std::cerr << "(" << get_index_of_adjacent_node( ii ) << ": " <<
			curr_state_two_body_energies_[ ii ] << ") ";
	}
	std::cerr << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::update_internal_energy_sums
///
/// @brief
/// removes numerical drift long stretches of efficient bookkeeping
/// produces
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::update_internal_energy_sums()
{
	assert( get_edge_vector_up_to_date() );
	curr_state_total_energy_ = 0;
	for (int ii = 1; ii <= get_num_incident_edges(); ++ii)
	{
		curr_state_total_energy_ +=
			get_incident_pd_edge(ii)->get_current_two_body_energy();
	}
	curr_state_total_energy_ += curr_state_one_body_energy_;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::prepare_to_write_to_file
///
/// @brief
/// ready a node for the output-to-file process
///
/// @detailed
/// allocates space for an additional array that counts the number of states
/// for each amino acid
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::prepare_to_write_to_file()
{
	initialize_aa_for_state_array();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::initialize_aa_for_state_array
///
/// @brief
/// counts the number of states for each amion acid
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::initialize_aa_for_state_array()
{
	aa_types_for_instance_states_.dimension( get_num_states() );
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		aa_types_for_instance_states_(ii) = sparse_mat_info_for_state_[ii].get_aa_type();
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::clean_up_after_writing_to_file
///
/// @brief
/// deallocates extra memory allocated before writing to a file
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::clean_up_after_writing_to_file()
{
	aa_types_for_instance_states_.dimension(0);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::prepare_to_read_energies_from_file
///
/// @brief
/// allocates extra space required of reading to a file
///
/// @detailed
/// this graph instance may not correspond perfectly to the rotamers
/// described in the file.  Before reading pair energies from the file
/// a correspondnece must be found between the "instance states" and
/// and the "file states".  RotamerSets.cc finds the correspondence.
/// Each node keeps track of the correspondence - which instance states
/// correspond to which file states, and which instance states have no
/// corresponding file states.  A vertex also keeps track of which
/// amino acid type for each of file state.  Amino acid type information is
/// necessary for reading in the sparse table that relies on the amino-acid
/// neighbor optimization.
///
/// @param
/// num_states_for_node_in_file - [in] - the number of states on the corresponding
///   node from the file
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::prepare_to_read_energies_from_file(int num_states_for_node_in_file)
{
	int num_aa_types_in_file =
		get_pdig_owner()->get_num_file_aatypes();

	num_states_in_file_ = num_states_for_node_in_file;
	instance_states_2_file_states_.dimension( get_num_states() );
	file_states_2_instance_states_.dimension( num_states_for_node_in_file );
	aa_types_for_file_states_.dimension( num_states_for_node_in_file );
	num_file_states_for_aa_.dimension( num_aa_types_in_file );

	instance_states_2_file_states_ = -1;
	file_states_2_instance_states_ = -1;
	aa_types_for_file_states_ = -1;
	num_file_states_for_aa_ = 0;

	initialize_aa_for_state_array(); //apl borrowing functionality from wri
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::clean_up_after_reading_energies_from_file
///
/// @brief
/// deallocate extra tables after having written to a file
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::clean_up_after_reading_energies_from_file()
{
	instance_states_2_file_states_.dimension(0);
	file_states_2_instance_states_.dimension(0);
	aa_types_for_file_states_.dimension(0);
	aa_types_for_instance_states_.dimension(0);
	num_file_states_for_aa_.dimension(0);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::set_aa_for_file_state
///
/// @brief
/// set the amino acid type for a file-state
///
/// @detailed
/// This method assumes that each file state will have its aa type set
/// before edge energies are read.  The structure of the interaction
/// graph file ensures that the amino acid type information is present for
/// each file state.  During RotamerSets.cc::read_rotamer_sets_from_file
/// the information must be transmitted to the interaction graph.
/// If this code quit rosetta, it's because the intereaction graph file
/// holds an amino-acid type out of bounds.
///
/// @param
/// file_state - [in] - the index of the file state
/// aa - [in] - the amino acid type of the file state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::set_aa_for_file_state(int file_state, int aa )
{
	if ( (unsigned int) aa > num_file_states_for_aa_.size() || aa <= 0)
	{
		std::cerr << "Error in interaction graph file: amino acid type out";
		std::cerr << " of range on node: " << get_node_index() << " for file state ";
		std::cerr <<  file_state << ": aa = " << aa << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	aa_types_for_file_states_( file_state ) = aa;
	++num_file_states_for_aa_( aa );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::set_instance_state_correspondence
///
/// @brief
/// declares that an instance state corresponds to a file state.
/// Trouble arises if rotamers are repeated, and the code that detects the
/// correspondence doesn't prevent multiple correspondences (multiple file
/// states to a single instance state or multiple instance states to a
/// single file state.)  The way I've written the RotamerSets.cc::
/// find_correspondence_between_file_and_instance() code enforces the bijection.
///
/// @detailed
///
/// @param
/// instance_state - [in] - the index of the instance state
/// state_from_file - [in] - the index of the file state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::set_instance_state_correspondence
(
	int instance_state,
	int state_from_file
)
{
	//apl Enforce bijection:
	//apl Much easier to handle I/O when no to file-states map to a single
	//apl instance state, and no two instance states map to a single file state.
	//apl If a protocol creates two identical rotamers before reading the
	//apl interaction graph file, then that protocol must also
	//apl have generated the same identical rotamers before writing the
	//apl interaction graph file.
	if ( instance_states_2_file_states_( instance_state ) != -1
		|| file_states_2_instance_states_( state_from_file ) != -1 )
	{
		std::cerr << "Reading Interaction Graph from File: Bijection Failure" << std::endl;
		std::cerr << "Node: " << get_node_index() << " instance_state " << instance_state;
		std::cerr << "file_state: " << state_from_file << std::endl;
		std::cerr << "First Correspondence: instance_states_2_file_states_( ";
		std::cerr << instance_state << " ) = " << instance_states_2_file_states_( instance_state );
		std::cerr << std::endl << "First Correspondence: file_states_2_instance_staets_( ";
		std::cerr << state_from_file << " ) = " << file_states_2_instance_states_( state_from_file );
		std::cerr << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	instance_states_2_file_states_( instance_state ) = state_from_file;
	file_states_2_instance_states_( state_from_file ) = instance_state;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_correspondence_for_state
///
/// @brief
/// return the file state that corresponds to an instance state
///
/// @detailed
///
/// @param
/// instance_state - [in] - the index of the instance state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDNode::get_correspondence_for_state( int instance_state )
{
	return instance_states_2_file_states_( instance_state );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_num_rots_absent_from_file
///
/// @brief
/// returns the number of instance states that did not correspond to any
/// file state.  The rotamers these states correspond to were absent from
/// the file, and their pair energies must be computed.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDNode::get_num_rots_absent_from_file()
{
	if ( instance_states_2_file_states_.size() == 0 )
	{
		//no correspondence found with input file
		return get_num_states();
	}

	int count_absent = 0;
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		if (instance_states_2_file_states_(ii) == -1 )
		{
			++count_absent;
		}
	}
	return count_absent;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_absent_rots
///
/// @brief
/// writes the index of the instance states with no matching file states into
/// the input FArray which should be large enough to hold them.
/// If the node itself had no matching node in the input file, then all
/// rotamers were absent from the file.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDNode::get_absent_rots( FArray1DB_int & rots_absent )
{
	if ( instance_states_2_file_states_.size() == 0 )
	{
		//no correspondence found with input file
		for (int ii = 1; ii <= get_num_states(); ++ii)
		{
			rots_absent(ii) = ii;
		}
		return;
	}

	int count_absent = 0;
	for (int ii = 1; ii <= get_num_states(); ++ii)
	{
		if (instance_states_2_file_states_(ii) == -1 )
		{
			++count_absent;
			rots_absent( count_absent ) = ii;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_num_states_in_file
///
/// @brief
/// returns the number of file states for this node
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDNode::get_num_states_in_file()
{
	return num_states_in_file_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_aatypes_for_file_states
///
/// @brief
/// returns a reference to the first entry in the aa_types_for_file_states_
/// array so that an FArray1Da can be constructed.
///
/// @detailed
/// used by a PDEdge while it's reading energies from a file
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int & PDNode::get_aatypes_for_file_states()
{
	return aa_types_for_file_states_(1);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_aatypes_for_states
///
/// @brief
/// returns a reference to the first entry in the aa_types_for_instance_states_
/// array so that an FArray1Da can be constructed.
///
/// @detailed
/// used by a PDEdge while it's reading energies from a file
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int & PDNode::get_aatypes_for_states()
{
	return aa_types_for_instance_states_(1);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_num_file_states_for_aa
///
/// @brief
/// returns a reference to the first entry in the num_file_states_for_aa_
/// array so that an FArray1Da can be constructed.
///
/// @detailed
/// used by a PDEdge while it's reading energies from a file
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int & PDNode::get_num_file_states_for_aa()
{
	return num_file_states_for_aa_(1);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_file_states_2_instance_states_array
///
/// @brief
/// returns a reference to the first entry in the file_states_2_instance_states_
/// array so that an FArray1Da can be constructed.
///
/// @detailed
/// used by a PDEdge while it's reading energies from a file
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int & PDNode::get_file_states_2_instance_states_array()
{
	return file_states_2_instance_states_(1);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_node_corresponded_to_file_node
///
/// @brief
/// returns true if a node corresponds to one of the nodes described in the
/// input file, and false if that node does not correspond to any.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool PDNode::get_node_corresponded_to_file_node()
{
	return (instance_states_2_file_states_.size() != 0 );
}

//-------------------------------------------------------------------------------//
//-------- Sparse Pairwise Decomposable Interaction Graph Edge Class ------------//
//-------------------------------------------------------------------------------//

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::PDEdge(PDInteractionGraph*, int, int) :
///
/// @brief
/// main constructor - no default nor copy constructors provided
///
/// @detailed
///
/// @param
/// owner - [in] - pointer to the graph that created this node
/// @param
/// first_node_ind - [in] - the index of the smaller-indexed node
/// @param
/// second_node_ind - [in] - the index of the larger-indexed node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
PDEdge::PDEdge
(	InteractionGraphBase* owner,
	int first_node_ind,
	int second_node_ind
) :
	PrecomputedPairEnergiesEdge( owner, first_node_ind, second_node_ind),
	//energy_table_size_(0)
	two_body_energies_(
		get_pd_node(0)->get_num_states_for_aa_types(),
		get_pd_node(1)->get_num_states_for_aa_types()
	),
	energies_updated_since_last_prep_for_simA_( true )
{
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::~PDEdge
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
PDEdge::~PDEdge()
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::set_sparse_aa_info
///
/// @brief
/// allocates two-body energy table based on amino-acid neighbor relationships
/// and initializes the table to 0.
///
/// @detailed
///
/// @param
/// sparse_conn_info - [in] - a MAX_AA x MAX_AA 2D array where each "true" entry
/// 	means that the corresponding amino acid pair are neighbors.
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// idea borrowed from energy2b implementation
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::set_sparse_aa_info(FArray2DB_bool const & sparse_conn_info)
{
	two_body_energies_.set_sparse_aa_info( sparse_conn_info );
	energies_updated_since_last_prep_for_simA_ = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_sparse_aa_info
///
/// @brief
/// returns whether two amino acid types are represented as neighbors
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool PDEdge::get_sparse_aa_info( int node1aa, int node2aa)
{
	return two_body_energies_.get_sparse_aa_info( node1aa, node2aa );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::force_aa_neighbors
///
/// @brief
/// re-allocates two-body energy table after forcing a pair of amino acids
/// to become neighbors that were not initially declared to be neighbors
///
/// @detailed
///
/// @param
/// node1aa - [in] - the amino acid type for the node with the smaller index
/// @param
/// node2aa - [in] - the amino acid type for the node with the larger index
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::force_aa_neighbors(int node1aa, int node2aa)
{
	two_body_energies_.force_aa_neighbors( node1aa, node2aa );
	energies_updated_since_last_prep_for_simA_ = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::force_all_aa_neighbors
///
/// @brief
/// re-allocates two-body energy table after forcing a pair of amino acids
/// to become neighbors that were not initially declared to be neighbors
///
/// @detailed
///
/// @param
/// node1aa - [in] - the amino acid type for the node with the smaller index
/// node2aa - [in] - the amino acid type for the node with the larger index
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::force_all_aa_neighbors()
{
	two_body_energies_.force_all_aa_neighbors();
	energies_updated_since_last_prep_for_simA_ = true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::add_to_two_body_energy
///
/// @brief
/// adds the input energy to the two body energy for state1 on the node with the
/// smaller index and state2 on the node with the larger index so long as
/// the amion acid types of those states have been previously declared amino
/// acid neighbors.  Any energies for non-neighboring states are ignored.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::add_to_two_body_energy
(
	int const state1,
	int const state2,
	float const energy
)
{
	two_body_energies_.add(
		get_pd_node(0)->get_sparse_mat_info_for_state(state1),
		get_pd_node(1)->get_sparse_mat_info_for_state(state2),
		energy);
	energies_updated_since_last_prep_for_simA_ = true;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::add_to_two_body_energies
///
/// @brief
/// Adds all the energies stored in the oversized_res_res_energy array to the
/// two body energy table for those states whose amion acid types were
/// previoudsly declared to be amino-acid neighbors.  The res-res array
/// should have the dimension (node1->get_num_states() x node2->get_num_states());
///
/// @detailed
///
/// @param
/// res_res_energy_array - [in] - an array containing the state pair
/// 	energies
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::add_to_two_body_energies
(
	FArray2DB_float const & res_res_energy_array
)
{
	for ( int ii = 1; ii <= get_num_states_for_node(0); ++ii )
	{
		SparseMatrixIndex const & state1_sparse_info = get_pd_node(0)
			->get_sparse_mat_info_for_state( ii );
		for ( int jj = 1; jj <= get_num_states_for_node(1); ++jj )
		{
			SparseMatrixIndex const & state2_sparse_info = get_pd_node(1)
				->get_sparse_mat_info_for_state( jj );

			two_body_energies_.add(
				state1_sparse_info, state2_sparse_info,
				res_res_energy_array( jj, ii )
			);
		}
	}
	energies_updated_since_last_prep_for_simA_ = true;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::set_two_body_energy
///
/// @brief
/// Sets the two-body energy for a pair of states.  That is, it overwrites
/// whatever two-body energy there was previously for that state pair with
/// a new energy.  Ignores non-neighboring state pairs.
///
/// @detailed
///
/// @param
/// state1 - [in] - state index for the node with the smaller index
/// @param
/// state2 - [in] - state index for the node with the larger index
/// @param
/// energy - [in] - the energy which replaces the old two-body energy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::set_two_body_energy
(
	int const state1,
	int const state2,
	float const energy
)
{
	two_body_energies_.set(
		get_pd_node(0)->get_sparse_mat_info_for_state(state1),
		get_pd_node(1)->get_sparse_mat_info_for_state(state2),
		energy
	);
	energies_updated_since_last_prep_for_simA_ = true;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::clear_two_body_energy
///
/// @brief
/// Sets the two-body energy for a pair of states.  That is, it overwrites
/// whatever two-body energy there was previously for that state pair with
/// a new energy.  Ignores non-neighboring state pairs.
///
/// @detailed
///
/// @param
/// state1 - [in] - state index for the node with the smaller index
/// @param
/// state2 - [in] - state index for the node with the larger index
/// @param
/// energy - [in] - the energy which replaces the old two-body energy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors jk
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::clear_two_body_energy
(
	int const state1,
	int const state2
)
{
	two_body_energies_.set(
		get_pd_node(0)->get_sparse_mat_info_for_state(state1),
		get_pd_node(1)->get_sparse_mat_info_for_state(state2),
		0.
	);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_two_body_energy
///
/// @brief
/// returns the two body energy for a pair of states: 0 if those states are
/// not neighbors
///
/// @detailed
///
/// @param
/// state1 - [in] - state index for the node with the smaller index
/// @param
/// state2 - [in] - state index for the node with the larger index
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDEdge::get_two_body_energy( int const state1, int const state2)
{
	return two_body_energies_.get(
		get_pd_node(0)->get_sparse_mat_info_for_state(state1),
		get_pd_node(1)->get_sparse_mat_info_for_state(state2));
}


void PDEdge::declare_energies_final()
{
	prepare_for_simulated_annealing();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::prepare_for_simulated_annealing
///
/// @brief
/// reduces the size of the pair-energy table if any amino-acid-neighbor
/// submatrices hold nothing but 0's
///
/// @detailed
/// since the drop_zero_submatrices_where_possible() method of the AANSM is
/// somewhat time consuming, and since it can only reduce memory use / simA
/// running time on the first execution following an update to the two-body
/// energies, the PDEdge member variable energies_updated_since_last_prep_
/// for_simA ensures that the AANSM method is only called once following
/// the update of any RPEs.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::prepare_for_simulated_annealing()
{
	prepare_for_simulated_annealing_no_deletion();
	if (two_body_energies_.get_table_size() == 0) delete this;
}


unsigned int
PDEdge::count_static_memory() const
{
	//std::cout << "calling PDEdge::count_static_memory()" << std::endl;
	return sizeof( PDEdge );
}


unsigned int
PDEdge::count_dynamic_memory() const
{
	//std::cout << "calling PDEdge::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = EdgeBase::count_dynamic_memory();
	
	total_memory += two_body_energies_.get_table_size() * sizeof( int );
	total_memory += two_body_energies_.get_offset_table_size_in_bytes();

	return total_memory;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_current_two_body_energy
///
/// @brief
/// returns the two body energy corresponding to the current states assigned to
/// the nodes this edge is incident upon.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDEdge::get_current_two_body_energy()
{
	return curr_state_energy_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::acknowledge_state_change
///
/// @brief
/// updates bookkeeping information when one of the two nodes changes its state
///
/// @detailed
///
/// @param
/// node_ind - [in] - the index of the node that changed its state
/// @param
/// node_state - [in] - the index of the new state it assumed
/// @param
/// new_state_spare_info - [in] - the sparse-matrix information for the state
/// @param
/// new_energy - [out] - the two body energy produced  by the new state and
/// 	the current state on the other node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
PDEdge::acknowledge_state_change
(
	int node_ind,
	int new_state,
	SparseMatrixIndex const & new_state_sparse_info,
	float & new_energy
)
{
	int node_substituted =  ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	int nodes_curr_states[2];
	SparseMatrixIndex nodes_curr_states_sparse_info[2];

	nodes_curr_states[ node_substituted ] = new_state;
	nodes_curr_states_sparse_info[ node_substituted ] = new_state_sparse_info;

	nodes_curr_states[ node_not_substituted ] =
		get_pd_node( node_not_substituted )->get_current_state();
	nodes_curr_states_sparse_info[ node_not_substituted ] =
		get_pd_node( node_not_substituted )->
		get_sparse_mat_info_for_curr_state();

	bool one_node_in_zero_state =
		( nodes_curr_states[0] == 0 || nodes_curr_states[1] == 0 );

	if (  one_node_in_zero_state )
	{
		curr_state_energy_ = 0;
	}
	else
	{
		curr_state_energy_ = two_body_energies_.get(
			nodes_curr_states_sparse_info[0],
			nodes_curr_states_sparse_info[1]);
	}
	new_energy = curr_state_energy_;

	get_pd_node( node_not_substituted )->
	acknowledge_neighbors_state_substitution
	(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		new_state,
		new_state_sparse_info
	);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::acknowledge_state_zeroed
///
/// @brief
/// updates bookkeeping information when one of the two nodes enters its
/// "unassigned" state.
///
/// @detailed
///
/// @param
/// node_ind - [in] - the index of the node that has just entered its 0 state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::acknowledge_state_zeroed( int node_ind )
{
	int node_substituted =  ( node_ind == get_node_index(0) ? 0 : 1);
	int node_not_substituted = ! node_substituted;

	curr_state_energy_ = 0;
	SparseMatrixIndex dummy_sparse_info;
	dummy_sparse_info.set_aa_type( 1 );
	dummy_sparse_info.set_state_ind_for_this_aa_type(1);

	get_pd_node( node_not_substituted )->
	acknowledge_neighbors_state_substitution
	(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		0,
		dummy_sparse_info
	);
	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_alternate_state_energy_first_node
///
/// @brief
/// static method that looks up the two body energy when the
/// node with the smaller index on an edge is considering an alternate state
///
/// @detailed
///
/// @param
/// first_node_alt_state - [in] - the alternate state for the lower-indexed node
/// @param
/// second_node_orig_state - [in] - the current state for the higher-indexed
///	node
/// @param
/// second_node_orig_state_sparse_info - [in] - the sparse matrix info for
///	the higher-indexed node
/// @param
/// first_node_state_offset_minus_1 - [in] - part of the sparse matrix info
/// 	for the lower-indexed node where 1 is subtracted from the state offset.
/// @param
/// second_node_num_states_per_aatype - [in] - number of states with current aa
///	type for node 2
/// @param
/// aa_neighbor_offset - [in] - offset for the amino-acid neighbor pair for
/// 	the sparse two-body energy table
/// @param
/// edge_energy_table - [in] - the proxy FArray pointing at the edge table
/// 	connecting the two nodes.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
float
PDEdge::get_alternate_state_energy_first_node(
	int first_node_alt_state,
	int second_node_orig_state,
	SparseMatrixIndex const & second_node_orig_state_sparse_info,
	int first_node_state_offset_minus_1,
	int second_node_curr_num_states_per_aatype,
	int aa_neighbor_offset,
	FArray1DB_float & edge_energy_table
)
{

	if (first_node_alt_state == 0 || second_node_orig_state == 0)
	{
		return 0.0f;
	}
	else
	{
		return AminoAcidNeighborSparseMatrix< float >::get(
			second_node_orig_state_sparse_info,
			first_node_state_offset_minus_1,
			second_node_curr_num_states_per_aatype,
			aa_neighbor_offset,
			edge_energy_table );
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_alternate_state_energy_second_node
///
/// @brief
/// static method that looks up the two body energy when the
/// node with the larger index on an edge is considering an alternate state
///
/// @detailed
///
/// @param
/// first_node_orig_state - [in] - the current state for the lower-indexed node
/// @param
/// second_node_alt_state - [in] - the alt state for the higher-indexed node
/// @param
/// first_node_orig_state_sparse_info - [in] - the sparse matrix info for
///	the lower-indexed node
/// @param
/// second_node_alt_state_sparse_info - [in] - the sparse matrix info for
///	the higher-indexed node
/// @param
/// first_node_state_offset_minus_1 - [in] - part of the sparse matrix info
/// 	for the lower-indexed node where 1 is subtracted from the state offset.
/// @param
/// second_node_alt_state_num_states_per_aatype - [in] - number of states
/// 	with alternate aa type for node 2
/// @param
/// aa_neighbor_offset - [in] - offset for the amino-acid neighbor pair for
/// 	the sparse two-body energy table
/// @param
/// edge_energy_table - [in] - the proxy FArray pointing at the edge table
/// 	connecting the two nodes.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
float
PDEdge::get_alternate_state_energy_second_node(
	int first_node_orig_state,
	int second_node_alt_state,
	SparseMatrixIndex const & first_node_orig_state_sparse_info,
	SparseMatrixIndex const & second_node_alternate_state_sparse_info,
	int second_node_alt_state_num_states_per_aatype,
	int aa_neighbor_offset,
	FArray1DB_float & edge_energy_table
)
{

	if (first_node_orig_state == 0 || second_node_alt_state == 0)
	{
		return 0.0f;
	}
	else
	{
		return AminoAcidNeighborSparseMatrix< float >::get(
			first_node_orig_state_sparse_info,
			second_node_alternate_state_sparse_info,
			second_node_alt_state_num_states_per_aatype,
			aa_neighbor_offset,
			edge_energy_table );
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::acknowledge_substitution
///
/// @brief
/// update bookkeeping information when one of the nodes an edge is incident
/// upon changes state
///
/// @detailed
///
/// @param
/// substituted_node_index - [in] - index of the node that chagned its state
/// @param
/// curr_state_energy - [in] - the two body energy given the new state
/// @param
/// nodes_new_state - [in] - the state the node just transitioned into
/// @param
/// nodes_new_state_sparse_info - [in] - sparse matrix info for the new state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
inline
void
PDEdge::acknowledge_substitution(
	int substituted_node_index,
	float const curr_state_energy,
	int nodes_new_state,
	SparseMatrixIndex const & nodes_new_state_sparse_info
)
{
	int node_substituted = substituted_node_index == get_node_index(0) ? 0 : 1;
	int node_not_substituted = ! node_substituted;

	curr_state_energy_ = curr_state_energy;

	get_pd_node( node_not_substituted )->
	acknowledge_neighbors_state_substitution
	(
		get_edges_position_in_nodes_edge_vector( node_not_substituted ),
		curr_state_energy_,
		nodes_new_state,
		nodes_new_state_sparse_info
	);

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_offsets_for_aatypes
///
/// @brief
/// Returns the array of offsets into the sparse two-body energy table
/// for amino-acid neighbors.  Used in transferring information from edges
/// onto nodes for cache efficiency.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
FArray2D_int const &
PDEdge::get_offsets_for_aatypes( )
{
	return two_body_energies_.getAANeighborOffsets();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_second_node_num_states_per_aa
///
/// @brief
/// returns an FArray of the number of states for each amino acid type for the
/// higher-indexed node
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
FArray1D_int const &
PDEdge::get_second_node_num_states_per_aa()
{
	return get_pd_node(1)->get_num_states_for_aa_types();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_edge_table_ptr
///
/// @brief
/// Returns a reference to the first element in the sparse two-body energy
/// table.  Used to create a proxy array on the nodes for cache efficiency.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float & PDEdge::get_edge_table_ptr() {return two_body_energies_.getMatrixPointer();}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::read_edge_energies_from_file
///
/// @brief
/// Reads the energies for an edge described in the input file and
/// copies the energies that correspond to a pair of instance states into
/// the edge energy table.
///
/// @detailed
/// Energies are represented in a binary file.  They are stored using the
/// same amino-acid neighbor technique that saves so much memory.  The
/// first part of the information in the file for this edge is the amino
/// acid neighbor connectivity information; 400 bools ( num_aa ^ 2).
/// From the number of file states for each amino acid, and the connectivity
/// the number of edge energies present can be readily computed.
/// This method allocates a large buffer to read in all the energies at once,
/// minimizing the number of disk reads.  It then walks through the buffer
/// and writes the appropriate energies to the energy table for this edge.
///
/// @param
/// infile - [in/out] - the binary input file, that has already been advanced
///   so that it's pointing at the position where the information for this
///   edge is stored.  At the end of this method, the infile will be advanced
///   past all the data for this edge.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::read_edge_energies_from_file( std::ifstream & infile )
{
	std::cerr << "Reading Edge: " << get_node_index(0) << " " << get_node_index(1) << std::endl;
	int node1_num_states_in_file = get_pd_node(0)->get_num_states_in_file();
	int node2_num_states_in_file = get_pd_node(1)->get_num_states_in_file();

	FArray1Da_int node1_file_states_2_instance_states(
		get_pd_node(0)->get_file_states_2_instance_states_array(),
		node1_num_states_in_file);

	FArray1Da_int node2_file_states_2_instance_states(
		get_pd_node(1)->get_file_states_2_instance_states_array(),
		node2_num_states_in_file);

	FArray1Da_int node1_aatypes_file_states(
		get_pd_node(0)->get_aatypes_for_file_states(),
		node1_num_states_in_file);

	FArray1Da_int node2_aatypes_file_states(
		get_pd_node(1)->get_aatypes_for_file_states(),
		node2_num_states_in_file);

	int num_file_aa = get_pdig_owner()->get_num_file_aatypes();
	int num_aa = get_pdig_owner()->get_num_aatypes();

	FArray1Da_int node1_num_file_states_for_aa(
		get_pd_node(0)->get_num_file_states_for_aa(),
		num_file_aa);

	FArray1Da_int node2_num_file_states_for_aa(
		get_pd_node(1)->get_num_file_states_for_aa(),
		num_file_aa);

	int sqr_file_aa = num_file_aa * num_file_aa;
	FArray2D_bool aa_neighbors( num_aa, num_aa, false );
	bool* aa_neighbor_buffer = new bool[ sqr_file_aa ];
	for (int ii = 0; ii <= sqr_file_aa; ++ii)
		aa_neighbor_buffer[ii] = false;

	//std::cerr << "square file aa: " << sqr_file_aa << std::endl;
	infile.read( (char*) aa_neighbor_buffer, sizeof( bool ) * sqr_file_aa );
	//int num_bools_read = infile.gcount();
	//assert( num_bools_read == sizeof( bool ) * sqr_file_aa );

	int buffer_index = 0;
	int num_pair_energies = 0;
	for (int ii = 1; ii <= num_file_aa; ++ii)
	{
		for (int jj = 1; jj <= num_file_aa; ++jj)
		{
			aa_neighbors(jj, ii) = aa_neighbor_buffer[ buffer_index ];
			//std::cerr << aa_neighbor_buffer[ buffer_index ];
			if ( aa_neighbor_buffer[ buffer_index ] )
			{
				num_pair_energies += node1_num_file_states_for_aa(ii) *
					node2_num_file_states_for_aa( jj );
			}
			++buffer_index;

		}
		//std::cerr << std::endl;
	}

	//std::cerr << "num pair energies: " << num_pair_energies << std::endl;

	two_body_energies_.set_sparse_aa_info( aa_neighbors );

	float * energies_buffer = new float [ num_pair_energies ];

	infile.read( (char*) energies_buffer, sizeof( float ) * num_pair_energies );
	//int read_num_floats = infile.gcount();
	buffer_index = 0;

	for (int ii = 1; ii <= node1_num_states_in_file; ++ii)
	{
		int ii_aa = node1_aatypes_file_states( ii );
		int ii_instance_state = node1_file_states_2_instance_states( ii );
		for (int jj = 1; jj <= node2_num_states_in_file; ++jj)
		{
			int jj_aa = node2_aatypes_file_states( jj );
			int jj_instance_state = node2_file_states_2_instance_states( jj );

			if ( ! aa_neighbors(jj_aa, ii_aa ) ) continue;

			assert( buffer_index < num_pair_energies );
			float energy = energies_buffer[ buffer_index ];
			++buffer_index;

			if ( ii_instance_state == -1 || jj_instance_state == -1 ) continue;

			two_body_energies_.set(
				get_pd_node(0)->get_sparse_mat_info_for_state(ii_instance_state),
				get_pd_node(1)->get_sparse_mat_info_for_state(jj_instance_state),
				energy);
		}
	}

	delete [] aa_neighbor_buffer;
	delete [] energies_buffer;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::skip_over_edge_energies_from_file
///
/// @brief
/// Advances the infile past the section describing an edge between
/// one (or two) file nodes that do not correspond to an instance node.
///
/// @detailed
/// This static method is very similar to the non-static
/// get_edge_energies_from_file() above, except that it requires
/// more input parameters, handed to it by a PDInteractionGraph instance
/// since the method cannot request information from two nodes of the graph.
/// instead of allocating a large buffer for the energies, it uses
/// the ofstream method seekg to advance the read head past the energies
/// for the absent edge.
///
/// @param
/// infile - [in/out] - the binary input file, that's been advanced to point
///   at information for a file edge that will not be included in the instance
///   interaction graph.  At the end of this method, the file will point
///   just past the information for this edge.
/// @param
/// num_file_aa - [in] - the number of aa types according to the file
/// @param
/// node1_num_file_states_for_aa - [in] - num states for aa on node 1
/// @param
/// node2_num_file_states_for_aa - [in] - num states for aa on node 2
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
PDEdge::skip_over_edge_energies_from_file
(
	std::ifstream & infile,
	int num_file_aa,
	FArray1DB_int & node1_num_file_states_for_aa,
	FArray1DB_int & node2_num_file_states_for_aa
)
{
	int sqr_file_aa = num_file_aa * num_file_aa;
	bool* aa_neighbor_buffer = new bool[ sqr_file_aa ];

	infile.read( (char*) aa_neighbor_buffer, sizeof(bool) * sqr_file_aa);

	int buffer_index = 0;
	int num_pair_energies = 0;
	for (int ii = 1; ii <= num_file_aa; ++ii)
	{
		for (int jj = 1; jj <= num_file_aa; ++jj)
		{

			if ( aa_neighbor_buffer[ buffer_index ] )
			{
				num_pair_energies += node1_num_file_states_for_aa( ii ) *
					node2_num_file_states_for_aa( jj );
			}
			++buffer_index;
		}
	}

	//skip forward
	infile.seekg( sizeof( float ) * num_pair_energies, std::ios::cur );
	delete [] aa_neighbor_buffer;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::write_edge_energies_to_file
///
/// @brief
/// Writes the energies for this edge to a binary output file.
///
/// @detailed
/// First writes out the amino acid neighbor information for this edge
/// then writes out the energies.
///
/// @param
/// outfile - [in/out] - the binary output file to write to.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::write_edge_energies_to_file( std::ofstream & outfile )
{
	FArray1Da_int node1_aatypes_for_state(
		get_pd_node(0)->get_aatypes_for_states(),
		get_num_states_for_node(0) );

	FArray1Da_int node2_aatypes_for_state(
		get_pd_node(1)->get_aatypes_for_states(),
		get_num_states_for_node(1) );

	//write the aa-neighbor information
	int num_aa = get_pdig_owner()->get_num_aatypes();
	int sqr_num_aa = num_aa * num_aa;
	bool * aa_neighbor_buffer = new bool[ sqr_num_aa ];
	int buffer_index = 0;

	for (int ii = 1; ii <= num_aa; ++ii)
	{
		for (int jj = 1; jj <= num_aa; ++jj)
		{
			if ( two_body_energies_.get_sparse_aa_info(ii, jj) )
			{
				aa_neighbor_buffer[ buffer_index ] = true;
			}
			else
			{
				aa_neighbor_buffer[ buffer_index ] = false;
			}
			//std::cerr << aa_neighbor_buffer[ buffer_index ];
			++buffer_index;
		}
		//std::cerr << std::endl;
	}
	outfile.write( (char*) aa_neighbor_buffer, sizeof( bool ) * sqr_num_aa );

	float * energy_buffer = new float[ two_body_energies_.get_table_size() ];
	buffer_index = 0;
	for (int ii = 1; ii <= get_num_states_for_node(0); ++ii )
	{
		int ii_aa = node1_aatypes_for_state( ii );
		for (int jj = 1; jj <= get_num_states_for_node(1); ++jj )
		{
			int jj_aa = node2_aatypes_for_state( jj );
			if ( ! two_body_energies_.get_sparse_aa_info(ii_aa, jj_aa) ) continue;

			energy_buffer[ buffer_index ] = get_two_body_energy( ii, jj );
			++buffer_index;
		}
	}

	outfile.write( (char*) energy_buffer, sizeof(float) * two_body_energies_.get_table_size() );
	std::cerr << "Writing edge: " << get_node_index(0) << " " << get_node_index(1);
	std::cerr << "; num_energies: " << two_body_energies_.get_table_size() << std::endl;
	delete [] aa_neighbor_buffer;
	delete [] energy_buffer;
}

// @ brief - allow derived class to prep this class for simA, but guarantee no call to delete this;
void PDEdge::declare_energies_final_no_deletion()
{
	prepare_for_simulated_annealing_no_deletion();
}

// @ brief - allow derived class to prep this class for simA, but guarantee no call to delete this;
void PDEdge::prepare_for_simulated_annealing_no_deletion() //hook for derived classes
{
	if ( ! energies_updated_since_last_prep_for_simA_ ) return;

	two_body_energies_.drop_zero_submatrices_where_possible();
	energies_updated_since_last_prep_for_simA_ = false;
}

/// @ brief - if edge table is empty, returns true -- assumes
///   prepare_for_simulated_annealing_no_deletion() has been called first.
bool PDEdge::pd_edge_table_all_zeros() const
{
	assert( ! energies_updated_since_last_prep_for_simA_ );
	return ( two_body_energies_.get_table_size() == 0);
}



////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::drop_small_submatrices_where_possible
///
/// @brief
/// drops any amino-acid neighbor submatrix of the two-body energy table
/// when the magnitudes of the energies stored in that submatrix do not exceed
/// the input parameter, epsilon.  Dropping submatrices that contain zero
/// energies is a special case of this function where epsilon == 0.
///
/// @detailed
///
/// @param
/// epsilon - [in] - the magnitude threshold for keeping amino-acid neighbor
///	submatrices.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::drop_small_submatrices_where_possible( float epsilon )
{
	two_body_energies_.drop_small_submatrices_where_possible( epsilon );
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::drop_zero_submatrices_where_possible
///
/// @brief
/// drops amino-acid neighbor submatrices when they do not contain any non-zero
/// entries.  Represents a special case of drop_small_submatrices_where_possible
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDEdge::drop_zero_submatrices_where_possible()
{
	drop_small_submatrices_where_possible( 0.0f );
	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDEdge::get_two_body_table_size()
///
/// @brief
/// returns the memory usage of the two body energy table for this edge
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDEdge::get_two_body_table_size() const
{
	return two_body_energies_.get_table_size();
}

//----------------------------------------------------------------------------//
//-------- Sparse Pairwise Decomposable Interaction Graph Class --------------//
//----------------------------------------------------------------------------//


////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::PDInteractionGraph
///
/// @brief
/// main constructor: no default nor copy constructors provided.
///
/// @detailed
///
/// @param
/// num_nodes - [in] - the number of nodes in this graph
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
PDInteractionGraph::PDInteractionGraph(int num_nodes) :
	PrecomputedPairEnergiesInteractionGraph( num_nodes ),
	num_aa_types_( -1 ), num_commits_since_last_update_(0),
	total_energy_current_state_assignment_(0),
	total_energy_alternate_state_assignment_(0),
	node_considering_alt_state_( -1 )
{}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_num_aatypes
///
/// @brief
/// sets the number of amino acid types present.
///
/// @detailed
/// The actual meaning of the integers used to represent amino acid types
/// is not important, rather each state is labeled as being of some amino
/// acid type.  The amino acid types mean little more than equivalence classes;
/// two states are either in the same equivalence class or they are in
/// different ones.
///
/// @param
/// num_aa_types - [in] - the number of amino acid types
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::set_num_aatypes(int num_aa_types)
{
	//this function should be called once and only once
	assert( num_aa_types_ == -1 && num_aa_types > 0 );
	num_aa_types_ = num_aa_types;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_num_aatypes
///
/// @brief
/// returns the number of different amino acid types
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int  PDInteractionGraph::get_num_aatypes() const
{	return num_aa_types_;}

float
PDInteractionGraph::get_one_body_energy_for_node_state( int node, int state)
{
	return get_pd_node( node )->get_one_body_energy( state );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::blanket_assign_state_0
///
/// @brief
/// assigns the state of all nodes in the interaction graph to their unassigned
/// or zero states.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::blanket_assign_state_0()
{
	//a state assignment of 0 means "unassigned".
	for (int ii = 1; ii <= get_num_nodes(); ++ii )
	{
		get_pd_node(ii)->assign_zero_state();
	}
	total_energy_current_state_assignment_ = 0;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_state_for_node
///
/// @brief
/// sets the state on node node_ind to new_state
///
/// @detailed
///
/// @param
/// node_ind - [in] - the index of the node in question
/// @param
/// new_state - [in] - the new state the node is being assigned to
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDInteractionGraph::set_state_for_node(int node_ind, int new_state)
{
	get_pd_node( node_ind )->assign_state(new_state);
	update_internal_energy_totals();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_network_state
///
/// @brief
/// takes in a vector of states, one state per node, and sets the state for
/// each of the nodes to the specified state.
///
/// @detailed
///
/// @param
/// node_states - [in] - array of states, one for each node.
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// also calls "update internal energy totals" to undo any numerical noise
/// accumulated during the transition.
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDInteractionGraph::set_network_state( FArray1DB_int & node_states)
{
	//node_states.dimension( get_num_nodes() );
	for (int ii = 1; ii <= get_num_nodes(); ++ii )
	{
		get_pd_node( ii )->assign_state( node_states(ii) );
	}
	update_internal_energy_totals();
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::consider_substitution
///
/// @brief
/// considers altering the state of a particular node; returns the
/// change in energy that the state substitution would produce
///
/// @detailed
///
/// @param
/// node_ind - [in] - the index of the node considering a state change
/// @param
/// new_state - [in] - the new state that node is considering
/// @param
/// alt_total_energy - [out] - the total network energy produced under the
///	new state
/// @param
/// delta_energy - [out] - the change in energy produced under the substitution
/// @param
/// prev_energy_for_node - [out] - the sum of the one and two body energies
/// 	for this node under the current state assignment
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// to avoid too much numerical drift from accumulating, the bookkeeping
/// arrays are updated once every 2^10 state commits
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
PDInteractionGraph::consider_substitution
(
	int node_ind,
	int new_state,
	float & delta_energy,
	float & prev_energy_for_node
)
{
	node_considering_alt_state_ = node_ind;
	delta_energy = get_pd_node( node_ind )->
		project_deltaE_for_substitution( new_state, prev_energy_for_node );

	//numerical drift accumulates in the following assignment
	total_energy_alternate_state_assignment_ =
		total_energy_current_state_assignment_ + delta_energy;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::commit_considered_substitution
///
/// @brief
/// Accepts (commits) the state change previously considered in a call to
/// consider_substitution and returns the energy of the entire graph
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// to avoid too much numerical drift from accumulating, the bookkeeping
/// arrays are updated once every 2^10 state commits
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
PDInteractionGraph::commit_considered_substitution()
{
	get_pd_node( node_considering_alt_state_ )->
		commit_considered_substitution();
	total_energy_current_state_assignment_ =
		total_energy_alternate_state_assignment_;

	++num_commits_since_last_update_;
	if (num_commits_since_last_update_ == COMMIT_LIMIT_BETWEEN_UPDATES)
	{
		update_internal_energy_totals();
	}

	return total_energy_alternate_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_energy_current_state_assignment
///
/// @brief
/// removes all accumulated numerical drift and returns the
/// energy for the current state assignment.
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDInteractionGraph::get_energy_current_state_assignment()
{
	update_internal_energy_totals();
	return total_energy_current_state_assignment_;
}

// @ breif O(1) total energy report.  Protected read access for derived classes.
float PDInteractionGraph::get_energy_PD_current_state_assignment()
{
	return total_energy_current_state_assignment_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::update_internal_energy_totals
///
/// @brief
/// removes numerical drift that can accumulate over the course of
/// many state assignment changes within simulated annealing
///
/// @detailed
/// Iterates across nodes and then edges to look-up the energies
/// for the current state assignmnet removing any numerical drift which
/// accumulated in the member variable total_energy_current_state_assignment_.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::update_internal_energy_totals()
{
	total_energy_current_state_assignment_ = 0;

	//std::cerr << "updating internal energy totals: " << std::endl;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{	//std::cerr << " ig_node " << ii << " = " << ((PDNode *) ig_nodes_[ ii ])
		//	->get_one_body_energy_current_state();

		total_energy_current_state_assignment_ += get_pd_node( ii )->
			get_one_body_energy_current_state();
	}

	//int counter = 0;
	for (std::list<EdgeBase*>::iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{  //std::cerr << " ig_edge " << ++counter  << " =" <<
		//((PDEdge*) *iter)->get_current_two_body_energy();
		total_energy_current_state_assignment_ +=
			((PDEdge*) *iter)->get_current_two_body_energy();
	}

	//std::cerr << std::endl;

	num_commits_since_last_update_ = 0;
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_edge_memory_usage
///
/// @brief
/// returns the number of floats used in all edge two-body energy tables
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDInteractionGraph::get_edge_memory_usage() const
{
	int sum = 0;
	for (std::list< EdgeBase* >::const_iterator iter = get_edge_list_begin();
		iter != get_edge_list_end(); ++iter)
	{
		sum += ((PDEdge*) *iter)->get_two_body_table_size();
	}
	return sum;
}

unsigned int
PDInteractionGraph::count_static_memory() const
{
	return sizeof( PDInteractionGraph );
}

unsigned int
PDInteractionGraph::count_dynamic_memory() const
{
	//std::cout << "calling PDInteractionGraph::count_dynamic_memory()" << std::endl;
	unsigned int total_memory = 0;
	total_memory += file_node_2_instance_node_.size() * sizeof ( int );
	total_memory += instance_node_2_file_node_.size() * sizeof ( int );
	for ( unsigned int ii = 1; ii <= aa_types_for_states_on_file_nodes_.size(); ++ii)
	{
		total_memory += aa_types_for_states_on_file_nodes_( ii ).size() * sizeof ( int );
	}
	total_memory += aa_types_for_states_on_file_nodes_.size() * sizeof ( FArray1D_int );
	for ( unsigned int ii = 1; ii <= num_file_states_for_aa_for_node_.size(); ++ii)
	{
		total_memory += num_file_states_for_aa_for_node_( ii ).size() * sizeof ( int );
	}
	total_memory += num_file_states_for_aa_for_node_.size() * sizeof ( FArray1D_int );

	//recurse to parent
	total_memory += InteractionGraphBase::count_dynamic_memory();

	return total_memory;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::print_current_state_assignment
///
/// @brief
/// outputs the current state for each node, useful for debugging
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::print_current_state_assignment() const
{
	std::cerr << "Curr States: ";
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		std::cerr << "(" << ii << ", ";
		std::cerr << get_pd_node(ii)->get_current_state() << ") ";
		get_pd_node(ii)->print_internal_energies();
	}
	std::cerr << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::prepare_to_read_energies_from_file
///
/// @brief
/// allocates space for the arrays necessary for reading in from a file
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::prepare_to_read_energies_from_file()
{
	instance_node_2_file_node_.dimension( get_num_nodes() );
	instance_node_2_file_node_ = -1;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::declare_finished_reading_from_file
///
/// @brief
/// deallocates arrays used to read energies from a file
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::declare_finished_reading_from_file()
{
	instance_node_2_file_node_.dimension( 0 );
	file_node_2_instance_node_.dimension( 0 );
	aa_types_for_states_on_file_nodes_.dimension( 0 );


	for (int ii = 1; ii <= get_num_nodes(); ++ii )
	{
		get_pd_node(ii)->clean_up_after_reading_energies_from_file();
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_num_file_aatypes
///
/// @brief
/// sets the number of amino acid types according to the input file
///
/// @detailed
///
/// @param
/// num_file_aatypes - [in] - the number of amino acid types
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::set_num_file_aatypes( int num_file_aatypes )
{
	num_file_aa_types_ = num_file_aatypes;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_num_file_aatypes
///
/// @brief
/// returns the number of amino acid types according to the input file
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDInteractionGraph::get_num_file_aatypes()
{
	return num_file_aa_types_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_num_nodes_in_file
///
/// @brief
/// sets the number of nodes for the graph described in the input file
///
/// @detailed
///
/// @param
/// num_nodes_in_file - [in] - the number of nodes in the file
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::set_num_nodes_in_file( int num_nodes_in_file )
{
	num_nodes_in_file_ = num_nodes_in_file;
	file_node_2_instance_node_.dimension( num_nodes_in_file );
	aa_types_for_states_on_file_nodes_.dimension( num_nodes_in_file );
	num_file_states_for_aa_for_node_.dimension( num_nodes_in_file );

	file_node_2_instance_node_ = -1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_node_correspondence
///
/// @brief
/// records the correspondence between an instance node and a file node
///
/// @detailed
///
/// @param
/// instance_node - [in] - the index of the instance node
/// @param
/// file_node - [in] - the index of the file node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::set_node_correspondence
(
	int instance_node,
	int file_node
)
{
	file_node_2_instance_node_( file_node ) = instance_node;
	instance_node_2_file_node_( instance_node ) = file_node;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_num_states_for_file_node
///
/// @brief
/// records the number of states for a file node
///
/// @detailed
/// If a file node corresponds to an instance node, then the instance
/// node keeps track of the information for the file node.  If the file
/// node doesn't correspond to any instance node, then the graph needs
/// to keep track of the file node's information.
/// Assumption: the correspondence between file nodes and instance nodes
/// has been completely specified before this method is called for the first time.
///
/// @param
/// node - [in] - the index of the file node
/// @param
/// num_file_states - [in] - the number of file states for that file node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::set_num_states_for_file_node
(
	int node,
	int num_file_states
)
{
	if ( file_node_2_instance_node_( node ) == -1 )
	{
		aa_types_for_states_on_file_nodes_( node ).dimension( num_file_states );
		num_file_states_for_aa_for_node_( node ).dimension( num_file_aa_types_ );
		num_file_states_for_aa_for_node_( node ) = 0;
	}
	else
	{
		int instance_node = file_node_2_instance_node_( node );
		get_pd_node(instance_node)->
			prepare_to_read_energies_from_file( num_file_states );
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_aa_for_file_node_state
///
/// @brief
/// sets the amino acid type for a file state
///
/// @detailed
///
/// @param
/// node - [in] - the index of the file node
/// @param
/// file_state - [in] - the file state
/// @param
/// state_aa - [in] - the amino acid type for file_state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::set_aa_for_file_node_state
(
	int node,
	int file_state,
	int state_aa
)
{
	int instance_node = file_node_2_instance_node_( node );
	if ( instance_node == -1 )
	{
		aa_types_for_states_on_file_nodes_( node )( file_state ) = state_aa;
		++num_file_states_for_aa_for_node_( node )(state_aa);
	}
	else
	{
		get_pd_node( instance_node )->set_aa_for_file_state(file_state, state_aa);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::set_correspondence_for_state
///
/// @brief
/// sets the correspondence between an instance state and a file state
///
/// @detailed
///
/// @param
/// node - [in] - the index of the instance node
/// @param
/// state - [in] - the instance state
/// @param
/// file_state - [in] - the file state that the instance state corresponds to
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::set_correspondence_for_state(int node, int state, int file_state)
{
	get_pd_node(node)->set_instance_state_correspondence( state, file_state );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_correspondence_for_state
///
/// @brief
/// returns the file state that an instance state corresponds to
///
/// @detailed
///
/// @param
/// node - [in] - the instance node
/// @param
/// state - [in] - the instance state
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDInteractionGraph::get_correspondence_for_state(int node, int state )
{
	return get_pd_node(node)->get_correspondence_for_state( state );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_node_corresponded_to_file_node
///
/// @brief
/// returns true if an instance node corresponds to a node in the file
///
/// @detailed
///
/// @param
/// node - [in] - the instance node in question
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool PDInteractionGraph::get_node_corresponded_to_file_node( int node )
{
	return get_pd_node(node)->get_node_corresponded_to_file_node( );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_num_rots_absent_from_file
///
/// @brief
/// returns the number of instance states not described by any file state for
/// for an instance node
///
/// @detailed
///
/// @param
/// node - [in] - the instance node in question
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int PDInteractionGraph::get_num_rots_absent_from_file(int node)
{
	return get_pd_node(node)->get_num_rots_absent_from_file();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_absent_rots
///
/// @brief
/// fills an input array with the indices of the instance states for an
/// instance node that did not correspond to any instance state of the input
/// file.
///
/// @detailed
///
/// @param
/// node - [in] - the instance node
/// @param
/// rots_absent - [out] - the array in which to write the absent rotamers
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::get_absent_rots(int node, FArray1DB_int & rots_absent )
{
	get_pd_node(node)->get_absent_rots( rots_absent );
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::read_edge_energies_from_file
///
/// @brief
/// reads pair energies from a binary file
///
/// @detailed
/// The edge energies in the binary input file have the following structure:
/// The number of edges in the graph is listed, and is followed by  a long list
/// of edges and edge information; each edge is identified by the indices
/// of the two file nodes that it is incident upon. Following the two
/// node indices, the rest of the information for that edge is stored.
/// The interaction graph reads in the file nodes and, if both nodes correspond
/// to instance nodes, creates a new edge between the instance nodes and
/// has the new edge read in the appropriate energies from the input file.
/// If either file node fails to correspond to some instance node, then the
/// graph invokes the static method PDEdge::skip_past_edge_energies() to advance
/// the read pointer past the information for this edge.
///
/// @param
/// infile - [in/out] - the binary input file that has been advanced to point
///   at the number of edges in the file
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::read_edge_energies_from_file( std::ifstream & infile )
{
	int num_edges;
	infile.read( (char*) &num_edges,4);

	for (int ii = 1; ii <= num_edges; ++ii)
	{
		int node1, node2;
		infile.read( (char*)  & node1, 4);
		infile.read( (char*)   & node2, 4 );

		int instance_node1 = file_node_2_instance_node_( node1 );
		int instance_node2 = file_node_2_instance_node_( node2 );
		if ( instance_node1 == -1 || instance_node2 == -1 )
		{	//skip over the energies for this edge
			if (instance_node1 == -1 )
			{
				FArray1Da_int node1_num_states_for_aa(
					num_file_states_for_aa_for_node_( node1 ), num_file_aa_types_);

				if (instance_node2 == -1)
				{
					FArray1Da_int node2_num_states_for_aa(
						num_file_states_for_aa_for_node_( node2 ), num_file_aa_types_);


					PDEdge::skip_over_edge_energies_from_file( infile,
						num_file_aa_types_,
						node1_num_states_for_aa, node2_num_states_for_aa);
				}
				else
				{
					FArray1Da_int node2_num_states_for_aa(
						get_pd_node( instance_node2 )
						->get_num_file_states_for_aa(), num_file_aa_types_ );

					PDEdge::skip_over_edge_energies_from_file( infile,
						num_file_aa_types_,
						node1_num_states_for_aa, node2_num_states_for_aa);
				}
			}
			else
			{
				FArray1Da_int node1_num_states_for_aa(
						get_pd_node( instance_node1 )
						->get_num_file_states_for_aa(), num_file_aa_types_);

				FArray1Da_int node2_num_states_for_aa(
						num_file_states_for_aa_for_node_( node2 ), num_file_aa_types_);

				PDEdge::skip_over_edge_energies_from_file( infile,
						num_file_aa_types_,
						node1_num_states_for_aa, node2_num_states_for_aa);

			}
		}
		else
		{
			add_edge( instance_node1, instance_node2 );
			PDEdge* new_edge = (PDEdge*) find_edge( instance_node1, instance_node2);

			new_edge->read_edge_energies_from_file( infile );
		}

	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::write_edge_energies_to_file
///
/// @brief
/// writes the edge energies, in a binary format, to the outfile.
///
/// @detailed
/// The graph writes out the number of edges, then proceeds to describe each
/// edge.  Each edge is identified by the two nodes its incident upon, and
/// each edge is responsible for outputting itself to the file.
///
/// @param
/// outfile - [in/out] - the output file to write edge energies into
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void PDInteractionGraph::write_edge_energies_to_file( std::ofstream & outfile )
{
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_pd_node(ii)->prepare_to_write_to_file();
	}

	int num_edges = get_num_edges(); //O(N) size method
	outfile.write( (char*) (&num_edges), 4);
	for ( std::list< EdgeBase* >::iterator edge_iter = get_edge_list_begin();
		edge_iter != get_edge_list_end(); ++edge_iter)
	{
		int first_node_ind = (*edge_iter)->get_first_node_ind();
		int second_node_ind = (*edge_iter)->get_second_node_ind();
		outfile.write( (char*) & first_node_ind, 4);
		outfile.write( (char*) & second_node_ind, 4);
		((PDEdge*) (*edge_iter) )->write_edge_energies_to_file( outfile );
	}

	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		get_pd_node(ii)->clean_up_after_writing_to_file();
	}

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::get_energy_sum_for_vertex_group
///
/// @brief
/// a user may define subsets of the vertex set for which they would like to
/// know the internal energy sum.  For instance in a graph with 6 vertices,
/// {a,b,c,d,e,f}
/// a user may be interested in the sum of the one- and two-body energies
/// for vertices {a,b,c}.  The graph will return sum of the one body energies
/// for vertices a b and c and also any two-body energies for the edges in the
/// subgraph induced by a,b, and c.  (In this case, edges {a,b}, {a,c} and {b,c}
/// if these edges are part of the graph.  The edge {a,d} will not be counted
/// if it is part of the graph.)
///
/// @detailed
/// ask the graph for the energies of the induced subgraph defined
/// by a particular group.
///
/// @param
/// group_id - [in] - the groups for which you're interested in retrieving
///  energies of the induced subgraph
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
PDInteractionGraph::get_energy_sum_for_vertex_group( int group_id )
{
	float esum = 0;
	for (int ii = 1; ii <= get_num_nodes(); ++ii)
	{
		if ( get_vertex_member_of_energy_sum_group( ii, group_id ) )
		{
			esum += get_pd_node( ii )->get_one_body_energy_current_state();
		}
	}

	for ( std::list< EdgeBase* >::iterator edge_iter = get_edge_list_begin();
		edge_iter != get_edge_list_end(); ++edge_iter)
	{
		int first_node_ind = (*edge_iter)->get_first_node_ind();
		int second_node_ind = (*edge_iter)->get_second_node_ind();

		if ( get_vertex_member_of_energy_sum_group( first_node_ind, group_id )
			&& get_vertex_member_of_energy_sum_group( second_node_ind, group_id ))
		{
			esum += ((PDEdge*) (*edge_iter))->get_current_two_body_energy();
		}
	}

	return esum;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::create_new_node
///
/// @brief
/// factory method that instantiates a PDNode.
///
/// @detailed
///
/// @param
/// node_index - [in] - the index of the node being created
/// @param
/// num_states - [in] - the total number of states for the new node
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
NodeBase* PDInteractionGraph::create_new_node( int node_index, int num_states)
{
	PDNode* new_node = new PDNode(this, node_index, num_states);
	assert( new_node != NULL );
	return new_node;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin PDInteractionGraph::create_new_edge
///
/// @brief
/// factory method that instantiates a PDEdge
///
/// @detailed
///
/// @param
/// index1 - [in] - the smaller-indexed node this edge is incident upon
/// @param
/// index2 - [in] - the larger-indexed node this edge is incident upon
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors apl
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
EdgeBase* PDInteractionGraph::create_new_edge( int index1, int index2)
{
	return new PDEdge(this, index1, index2);
}





// <directed_design>

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::project_deltaE_for_substitution
///
/// @brief returns the change in weighted energy that would be induced
/// by switching this node from its current state into another state
///
/// @detailed calls project_deltaE_for_substitution ( the weighted version )
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors murphp
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

void PDNode::project_deltaE_for_substitution
(
 int alternate_state,
 float & deltaE_unweighted,
 float & prevE_unweighted,
 float& deltaE_weighted,
 float& prevE_weighted,
 const FArray2D_float& weights
 )
{

	// Step 1 - calculate unweighted energy, with side effect of filling alternate_state_one/two_body_energy

	deltaE_unweighted = project_deltaE_for_substitution(alternate_state,prevE_unweighted); // !!! need to return this parameter since the graph is adding it to total

	// Step 2 - calculate weighted energy on basis of info in:
	// - curr_state_one_body_energy_
	// - curr_state_two_body_energies_
	// - alternate_state_one_body_energy_
	// - alternate_state_two_body_energies_

	float curr_total_energy_weighted      = curr_state_one_body_energy_;
	float alternate_total_energy_weighted = alternate_state_one_body_energy_;

	for( int ii = 1; ii <= get_num_incident_edges(); ++ii ) {

		// 		const int i = get_node_index();
		// 		const int j = get_adjacent_node(ii)->get_node_index();
		//const float bias_ii = get_bias(bias,i,j);//get_node_index(),get_adjacent_node(ii)->get_node_index());

		const float bias_ii = weights(get_incident_edge(ii)->get_first_node_ind(), // !!! symmetric bias matrix
																	get_incident_edge(ii)->get_second_node_ind());

		curr_total_energy_weighted      += bias_ii * curr_state_two_body_energies_[ii];
		alternate_total_energy_weighted += bias_ii * alternate_state_two_body_energies_[ii];

	} // ii

	prevE_weighted = curr_total_energy_weighted;
	deltaE_weighted = alternate_total_energy_weighted - curr_total_energy_weighted;

	//std::cout << __func__ << "deltaE_unweighted = " << deltaE_unweighted << "\tdeltaE_weighted = " << deltaE_weighted << std::endl;

} // project_deltaE_for_substitution

namespace {

	int get_other_index(const EdgeBase* edge_base, const int index) {
		if( index == edge_base->get_first_node_ind() ) {
			return edge_base->get_second_node_ind();
		}
		else if( index == edge_base->get_second_node_ind() ) {
			return edge_base->get_first_node_ind();
		}
		else {
			assert( false );
			return -1; // still need to return something for release mode
		}
	} // get_other_index

} // namespace

////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_weighted_energy_with_higher_indexed_nodes
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors murphp
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

float PDNode::get_weighted_energy_with_higher_indexed_nodes(const FArray2D_float& weights ) const {

	float rval = curr_state_one_body_energy_;

	for( int ii = 1; ii <= get_num_incident_edges(); ++ii ) {

		if( get_other_index(get_incident_edge(ii),get_node_index()) > ii ) {

			const float weight_ii = weights(get_incident_edge(ii)->get_first_node_ind(), // !!! symmetric bias matrix
																			get_incident_edge(ii)->get_second_node_ind());
			rval += weight_ii * curr_state_two_body_energies_[ii];

		}

	} // ii

	return rval;
} // PDNode::get_biased_energy_with_higher_indexed_nodes


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::get_weighted_energy
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors murphp
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

float PDInteractionGraph::get_weighted_energy(const FArray2D_float& weights) const {

	// Compute and return biased energy
	float rval = 0;
	for( int ii = 1; ii <= get_num_nodes(); ++ii ) {
		rval += get_pd_node(ii)->get_weighted_energy_with_higher_indexed_nodes(weights);
	}

	return rval;

} // PDInteractionGraph::get_weighted_energy


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::set_network_state
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors murphp
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float PDInteractionGraph::set_network_state( FArray1DB_int & node_states, const FArray2D_float& weights) {

	set_network_state(node_states);
	return get_weighted_energy(weights);

} // PDInteractionGraph::set_network_state


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::commit_considered_substitution
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors murphp
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

float PDInteractionGraph::commit_considered_substitution_weighted
(
	FArray2D_float const & weights
)
{
	commit_considered_substitution();
	return get_weighted_energy(weights);
} // commit_considered_substitution


////////////////////////////////////////////////////////////////////////////////
/// @begin PDNode::consider_substitution_weighted
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors murphp
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////

void PDInteractionGraph::consider_substitution_weighted
(
	int node_ind,
	int new_state,
	float & deltaE_unweighted,
	float & prevE_unweighted,
	float & deltaE_weighted,
	float & prevE_weighted,
	const FArray2D_float& weights
)
{

	node_considering_alt_state_ = node_ind;
	get_pd_node( node_ind )->project_deltaE_for_substitution( new_state, deltaE_unweighted, prevE_unweighted, deltaE_weighted, prevE_weighted, weights );

	//numerical drift accumulates in the following assignment
	// !!! need to make sure to add the unweighted energy to this variable
	total_energy_alternate_state_assignment_ =
		total_energy_current_state_assignment_ + deltaE_unweighted;

	return;
}

// </directed_design>

} //end namespace pack

