// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14875 $
//  $Date: 2007-05-10 13:39:39 -0700 (Thu, 10 May 2007) $
//  $Author: leaverfa $

#ifndef SPARSE_PD_INTERACTION_GRAPH_H
#define SPARSE_PD_INTERACTION_GRAPH_H


// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// Utility Headers
#include <utility/io/all.fwd.hh>

// Rosetta Headers
#include "InteractionGraphBase.h"
#include "PrecomputedInteractionGraph.h"
#include "SparseMatrixIndex.h"
#include "AminoAcidNeighborSparseMatrix.h"

// C++ Headers
#include <vector>
#include <list>


namespace pack {

class PDNode;
class PDEdge;
class PDInteractionGraph;

class PDNode : public PrecomputedPairEnergiesNode
{
public:
	PDNode(InteractionGraphBase * owner, int node_id, int num_states);
	virtual ~PDNode();
	virtual void print() const;

	virtual void set_amino_acid_types( std::vector< int > const & );
	FArray1D_int const & get_num_states_for_aa_types() const;
	virtual void update_one_body_energy( int state, float energy );
	virtual void update_one_body_energies( FArray1DB_float & energies );
	virtual void add_to_one_body_energy( int state, float energy );
	virtual void add_to_one_body_energies( FArray1DB_float & energies );
	virtual void zero_one_body_energies();
	float get_one_body_energy( int state );

	virtual void prepare_for_simulated_annealing();

	void assign_zero_state();
	virtual bool state_unassigned() const { return current_state_ == 0;}
	virtual float get_totalE() const { return curr_state_total_energy_;}
	void assign_state(int new_state);
	int get_current_state() const;
	float get_one_body_energy_current_state() const;
	inline float project_deltaE_for_substitution
	(
			int alternate_state,
			float & prev_node_energy
	);
	void commit_considered_substitution();

	// <directed_design>
	void project_deltaE_for_substitution
	(
			int alternate_state,
			float & deltaE_unweighted,
			float & prevE_unweighted,
			float& deltaE_weighted,
			float& prevE_weighted,
			FArray2D_float const& weights
	);
	float get_weighted_energy_with_higher_indexed_nodes(FArray2D_float const& weights) const;
	// </directed_design>

	inline
	void acknowledge_neighbors_state_substitution(
			int edge_to_altered_neighbor,
			float new_edge_energy,
			int other_node_new_state,
			SparseMatrixIndex const & other_node_new_state_sparse_info);

	SparseMatrixIndex const &
	get_sparse_mat_info_for_state(int state) const;

	SparseMatrixIndex const &
	get_sparse_mat_info_for_curr_state() const;


	int  get_num_states_for_aa_type(int aa_type) const;
	void print_internal_energies();

	void update_internal_energy_sums();

	void prepare_to_write_to_file();
	void initialize_aa_for_state_array();
	void clean_up_after_writing_to_file();
	void prepare_to_read_energies_from_file( int num_states_for_node_in_file );
	void clean_up_after_reading_energies_from_file();

	void set_aa_for_file_state(int file_state, int aa );
	void set_instance_state_correspondence( int instance_state, int state_from_file );
	int get_correspondence_for_state( int instance_state );

	int get_num_rots_absent_from_file();
	void get_absent_rots( FArray1DB_int & rots_absent );

	int get_num_states_in_file();
	int & get_aatypes_for_file_states();

	int & get_aatypes_for_states();
	int & get_num_file_states_for_aa();
	int & get_file_states_2_instance_states_array();

	bool get_node_corresponded_to_file_node();

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

protected:
	void update_internal_vectors();

	//Hooks for SASANode< V, E, G > class
	float get_curr_pd_energy_total() const { return curr_state_total_energy_;}
	float get_alt_pd_energy_total() const { return alternate_state_total_energy_;}
	void set_alternate_state( int alt ) { alternate_state_ = alt; }
	int get_alternate_state() const { return alternate_state_; }
	void calc_deltaEpd( int alternate_state );
	bool considering_alternate_state() const
	{
		return alternate_state_is_being_considered_;
	}

private:
	void
	set_alt_aa_offsets_from_edge(int edge_index, FArray2D_int const & offsets);

	inline
	PDEdge* get_incident_pd_edge( int index ) const
	{       return ( PDEdge* ) get_incident_edge( index ); } // c-style cast since static_cast won't compile

	inline
	PDInteractionGraph* get_pdig_owner() const
	{       return ( PDInteractionGraph* ) get_owner(); } // c-style cast since static_cast won't compile

	int num_aa_types_;
	FArray1D_int num_states_for_aatype_;
	std::vector< SparseMatrixIndex > sparse_mat_info_for_state_;
	std::vector< float > one_body_energies_;

	FArray3D_int aa_offsets_for_edges_;
	FArray2D_int num_states_for_aa_type_for_higher_indexed_neighbor_;
	std::vector< int > neighbors_curr_state_;
	std::vector< SparseMatrixIndex > neighbors_curr_state_sparse_info_;
	std::vector< FArray1Da_float > edge_matrix_ptrs_;


	int current_state_;
	SparseMatrixIndex curr_state_sparse_mat_info_;
	float curr_state_one_body_energy_;
	float curr_state_total_energy_;
	std::vector< float > curr_state_two_body_energies_;

	int alternate_state_;
	SparseMatrixIndex alt_state_sparse_mat_info_;
	float alternate_state_one_body_energy_;
	float alternate_state_total_energy_;
	std::vector< float > alternate_state_two_body_energies_;

	//variables for I/O
	int num_states_in_file_;
	FArray1D_int instance_states_2_file_states_;
	FArray1D_int file_states_2_instance_states_;
	FArray1D_int aa_types_for_file_states_;
	FArray1D_int aa_types_for_instance_states_; //useful only for I/O
	FArray1D_int num_file_states_for_aa_;

	bool alternate_state_is_being_considered_;

	//no default constructor, uncopyable
	PDNode();
	PDNode( PDNode const & );
	PDNode & operator = ( PDNode const & );
};

class PDEdge : public PrecomputedPairEnergiesEdge
{
public:
	PDEdge(InteractionGraphBase* owner, int first_node_ind, int second_node_ind);
	virtual ~PDEdge();
	virtual void set_sparse_aa_info(FArray2DB_bool const & sparse_conn_info);
	virtual void force_aa_neighbors(int node1aa, int node2aa);
	virtual void force_all_aa_neighbors();
	virtual bool get_sparse_aa_info( int node1aa, int node2aa);
	virtual void add_to_two_body_energy(int const, int const, float const);
	virtual void
	add_to_two_body_energies( FArray2DB_float const & res_res_energy_array );
	virtual
	void set_two_body_energy(int const, int const, float const);
	virtual
	void clear_two_body_energy(int const, int const);
	virtual float get_two_body_energy( int const, int const );

	virtual void declare_energies_final();
	virtual void prepare_for_simulated_annealing();

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

	float get_current_two_body_energy();

	void acknowledge_state_change(
			int node_ind,
			int new_state,
			SparseMatrixIndex const & new_state_sparse_info,
			float & new_energy
	);
	void acknowledge_state_zeroed( int node_ind );

	static
	inline
	float get_alternate_state_energy_first_node(
			int first_node_alt_state,
			int second_node_orig_state,
			SparseMatrixIndex const & second_node_orig_state_sparse_info,
			int first_node_state_offset_minus_1,
			int second_node_curr_num_states_per_aatype,
			int aa_neighbor_offset,
		FArray1DB_float & edge_energy_table
	);

	static
	inline
	float get_alternate_state_energy_second_node(
			int first_node_orig_state,
			int second_node_alt_state,
			SparseMatrixIndex const & first_node_orig_state_sparse_info,
			SparseMatrixIndex const & second_node_alternate_state_sparse_info,
			int second_node_alt_state_num_states_per_aatype,
			int aa_neighbor_offset,
		FArray1DB_float & edge_energy_table
	);

	inline void acknowledge_substitution(
			int substituted_node_index,
			float const curr_state_energy,
			int nodes_new_state,
			SparseMatrixIndex const & nodes_new_state_sparse_info
	);

	FArray2D_int const & get_offsets_for_aatypes( );
	FArray1D_int const & get_second_node_num_states_per_aa();

	int get_two_body_table_size() const;
	float & get_edge_table_ptr();

	void read_edge_energies_from_file( std::ifstream & infile );
	static void skip_over_edge_energies_from_file
	(
			std::ifstream & infile,
			int num_aa,
			FArray1DB_int & num_file_states_for_aa_node1,
			FArray1DB_int & num_file_states_for_aa_node2
	);
	void write_edge_energies_to_file( std::ofstream & outfile );

protected:

	//Hooks for SASAEdge< V, E, G > class
	void declare_energies_final_no_deletion();
	void prepare_for_simulated_annealing_no_deletion();
	bool pd_edge_table_all_zeros() const;

private:
	inline
	PDNode* get_pd_node( int index ) const
	{       return (PDNode*) get_node( index );} //c-style cast since static_cast won't compile

	inline
	PDInteractionGraph* get_pdig_owner() const
	{       return (PDInteractionGraph*) get_owner();} //c-style cast since static_cast won't compile

	void drop_small_submatrices_where_possible( float epsilon );
	void drop_zero_submatrices_where_possible();

	AminoAcidNeighborSparseMatrix< float > two_body_energies_;
	float curr_state_energy_;
	bool energies_updated_since_last_prep_for_simA_;

	//no default constructor, uncopyable
	PDEdge();
	PDEdge( PDEdge const & );
	PDEdge & operator = ( PDEdge const & );
};

class PDInteractionGraph : public PrecomputedPairEnergiesInteractionGraph
{
public:
	PDInteractionGraph(int num_nodes);

	virtual float get_one_body_energy_for_node_state( int node, int state);

	virtual void set_num_aatypes(int);
	virtual int  get_num_aatypes() const;

	virtual void blanket_assign_state_0();
	virtual float set_state_for_node(int node_ind, int new_state);
	virtual float set_network_state( FArray1DB_int & node_states);
	virtual void consider_substitution
	(
		int node_ind,
		int new_state,
		float & delta_energy,
		float & prev_energy_for_node
	);
	virtual float commit_considered_substitution();
	virtual float get_energy_current_state_assignment();

	virtual int get_edge_memory_usage() const;

	virtual unsigned int count_static_memory() const;
	virtual unsigned int count_dynamic_memory() const;

	virtual void print_current_state_assignment() const;
	virtual void set_errorfull_deltaE_threshold( float ) {};

	//Methods for I/O
	void prepare_to_read_energies_from_file();
	void declare_finished_reading_from_file();
	void set_num_file_aatypes( int num_file_aatypes );
	int get_num_file_aatypes();
	void set_num_nodes_in_file( int num_nodes_in_file );
	void set_node_correspondence( int instance_node, int file_node );
	void set_num_states_for_file_node(int node, int num_file_states);
	void set_aa_for_file_node_state( int file_node, int file_state, int state_aa );
	void set_correspondence_for_state(int node, int state, int file_state);
	int get_correspondence_for_state(int node, int state );
	bool get_node_corresponded_to_file_node( int node );

	int get_num_rots_absent_from_file(int node);
	void get_absent_rots(int node, FArray1DB_int & rots_absent );

	void read_edge_energies_from_file( std::ifstream & infile );
	void write_edge_energies_to_file( std::ofstream & outfile );

	virtual float get_energy_sum_for_vertex_group( int group_id );

	// <directed_design>
	float get_weighted_energy(FArray2D_float const &weights) const;
	float set_network_state( FArray1DB_int & node_states, FArray2D_float const& weights);
	virtual void consider_substitution_weighted
	(
	 int node_ind,
	 int new_state,
	 float & deltaE_unweighted,
	 float & prevE_unweighted, // !!! remember this is the energy just for this node
	 float & deltaE_weighted,
	 float & prevE_weighted,
	 FArray2D_float const& weights
	 );
	virtual float commit_considered_substitution_weighted(FArray2D_float const& weights);
	// </directed_design>

protected:

	virtual NodeBase* create_new_node( int node_index, int num_states);
	virtual EdgeBase* create_new_edge( int index1, int index2);

	//Hooks for SASAInterationGraph< V, E, G >
	float get_energy_PD_current_state_assignment();
	void update_internal_energy_totals();

	inline
	PDNode* get_pd_node(int index) const
	{       return (PDNode*) get_node( index );}

private:
	int num_aa_types_;
	int num_commits_since_last_update_;
	float total_energy_current_state_assignment_;
	float total_energy_alternate_state_assignment_;
	int node_considering_alt_state_;

	//variables for I/O
	int num_nodes_in_file_;
	int num_file_aa_types_;
	FArray1D_int file_node_2_instance_node_;
	FArray1D_int instance_node_2_file_node_;
	FArray1D< FArray1D_int > aa_types_for_states_on_file_nodes_;
	FArray1D< FArray1D_int > num_file_states_for_aa_for_node_;

	static const int COMMIT_LIMIT_BETWEEN_UPDATES = 1024; // 2^10

	//no default constructor, uncopyable
	PDInteractionGraph();
	PDInteractionGraph( PDInteractionGraph const & );
	PDInteractionGraph & operator = ( PDInteractionGraph const & );
};

} //end namespace pack

#endif //SPARSE_PD_INTERACTION_GRAPH_H
