// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $

// Rosetta Headers
#include "RotamerOptions.h"
#include "after_opts.h"
#include "design.h"
#include "files_paths.h"
#include "fullatom_setup.h"
#include "param.h"
#include "param_aa.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Ds.hh>
#include <ObjexxFCL/FArray2Ds.hh>
#include <ObjexxFCL/formatted.io.hh> // F

// C++ Headers
#include <cassert>
#include <cstdlib>
#include <iostream>
#include <vector>

//Utility Headers
#include <utility/basic_sys_util.hh>

namespace exchi_flags {

	//if you would like to add new chi sampling options, change two files:
	//namespace_exchi_flags.h and namespace_exchi_flags.cc
	//add new chi sampling options to the end of the enumeration list, or
	//the meaning of "-ex1 7" will change and that will upset people.
	void extra_rot_sample_names_initializer( FArray1D_string & names )
	{
		//beware of off-by-one errors;
		names[ 0 ] = "NO_EXTRA_CHI_SAMPLES";
		names[ 1 ] = "EX_ONE_STDDEV";
		names[ 2 ] = "EX_ONE_HALF_STEP_STDDEV";
		names[ 3 ] = "EX_TWO_FULL_STEP_STDDEVS";
		names[ 4 ] = "EX_TWO_HALF_STEP_STDDEVS";
		names[ 5 ] = "EX_FOUR_HALF_STEP_STDDEVS";
		names[ 6 ] = "EX_THREE_THIRD_STEP_STDDEVS";
		names[ 7 ] = "EX_SIX_QUARTER_STEP_STDDEVS";
		names[ 8 ] = "EX_NINE_QUARTER_AND_THIRD_STEP_STDDEVS";
	}

	FArray1D_string ExtraRotSampleNames( ExtraRotSampleCardinality, extra_rot_sample_names_initializer );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerOptions::RotamerOptions
///
/// @brief
/// constructor for RotamerOptions object
///
/// @authors jk
///
////////////////////////////////////////////////////////////////////////////////
RotamerOptions::RotamerOptions() {

	ex1 = false;
	ex2 = false;
	ex3 = false;
	ex4 = false;
	ex1aro = false;
	ex1aro_half = false;
	ex2aro_only = false;
	ex1aro_exposed = false;
	ex2aro_exposed = false;
	exOH = false;
	excys = false;
	ex1_sample_level = exchi_flags::NO_EXTRA_CHI_SAMPLES;
	ex2_sample_level = exchi_flags::NO_EXTRA_CHI_SAMPLES;
	ex3_sample_level = exchi_flags::NO_EXTRA_CHI_SAMPLES;
	ex4_sample_level = exchi_flags::NO_EXTRA_CHI_SAMPLES;
	exrotset_from_db = false;
	extrachi_cutoff = 18;
	base_rotamer_buried_cutoff = 19;

	rot_pert = false; // add "perturbed" rotamer
	rot_pert_input = false; // perturb input sidechains as well
	pert_acc_prob = 1.; // default value in fullatom.cc
	pert_size = 10.; // default value in fullatom.cc

	use_input_sc = false;
	use_input_cb = false; //ja native c-betas

	rot_opt = false;

	dump_rotamers_pdb_flag = false; // dump rotamers and energies to file
	dump_rotamers_pdb_res = 0; // dump rotamers to PDB for this residue

	rot_explode = false;
	rot_explode_sample_level_chi12 = exchi_flags::NO_EXTRA_CHI_SAMPLES;
	rot_explode_sample_level_chi3 = exchi_flags::NO_EXTRA_CHI_SAMPLES;
	rot_explode_sample_level_chi4 = exchi_flags::NO_EXTRA_CHI_SAMPLES;

  rot_explode_accept_all = false;
	rot_explode_rtmin = false;
	minimize_best_rotamers = false;
	explode_cutoff = -0.5;

	dna_rotamers_exist = false;

  do_rotamer_bias = false;
  bias_rotamers_by_replacement = false;


}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerOptions::fill_ex_step_sets()
///
/// @brief
/// Initialization only.  Initializes sets of standard deviation factors from the extra chi flag options.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
RotamerOptions::fill_ex_step_sets()
{
	using namespace exchi_flags;

	// ex_step_sets is a vector< vector<float> >
	ex_step_sets.clear();
	ex_step_sets.assign( ExtraRotSampleCardinality, std::vector<float>() );

	int i( EX_ONE_STDDEV );
		// benchmarks are sensitive to the order of these steps: using std::set
		// changed a couple of highly stochastic benchmarks (sad but true)
		ex_step_sets[i].push_back(  1. );
		ex_step_sets[i].push_back( -1. );

	i = EX_ONE_HALF_STEP_STDDEV;
		ex_step_sets[i].push_back( -0.5 );
		ex_step_sets[i].push_back(  0.5 );

	i = EX_TWO_FULL_STEP_STDDEVS;
		ex_step_sets[i].push_back( -2. );
		ex_step_sets[i].push_back( -1. );
		ex_step_sets[i].push_back(  1. );
		ex_step_sets[i].push_back(  2. );

	i = EX_TWO_HALF_STEP_STDDEVS;
		ex_step_sets[i].push_back( -1.  );
		ex_step_sets[i].push_back( -0.5 );
		ex_step_sets[i].push_back(  0.5 );
		ex_step_sets[i].push_back(  1.  );

	i = EX_FOUR_HALF_STEP_STDDEVS;
		ex_step_sets[i].push_back( -2.  );
		ex_step_sets[i].push_back( -1.5 );
		ex_step_sets[i].push_back( -1.  );
		ex_step_sets[i].push_back( -0.5 );
		ex_step_sets[i].push_back(  0.5 );
		ex_step_sets[i].push_back(  1.  );
		ex_step_sets[i].push_back(  1.5 );
		ex_step_sets[i].push_back(  2.  );

	i = EX_THREE_THIRD_STEP_STDDEVS;
		ex_step_sets[i].push_back( -1.   );
		ex_step_sets[i].push_back( -0.67 );
		ex_step_sets[i].push_back( -0.33 );
		ex_step_sets[i].push_back(  0.33 );
		ex_step_sets[i].push_back(  0.67 );
		ex_step_sets[i].push_back(  1.   );

	i = EX_SIX_QUARTER_STEP_STDDEVS;
		ex_step_sets[i].push_back( -1.5  );
		ex_step_sets[i].push_back( -1.25 );
		ex_step_sets[i].push_back( -1.   );
		ex_step_sets[i].push_back( -0.75 );
		ex_step_sets[i].push_back( -0.5  );
		ex_step_sets[i].push_back( -0.25 );
		ex_step_sets[i].push_back(  0.25 );
		ex_step_sets[i].push_back(  0.5  );
		ex_step_sets[i].push_back(  0.75 );
		ex_step_sets[i].push_back(  1.   );
		ex_step_sets[i].push_back(  1.25 );
		ex_step_sets[i].push_back(  1.5  );

	i = EX_NINE_QUARTER_AND_THIRD_STEP_STDDEVS;
		ex_step_sets[i].push_back( -1.5  );
		ex_step_sets[i].push_back( -1.33 );
		ex_step_sets[i].push_back( -1.25 );
		ex_step_sets[i].push_back( -1.   );
		ex_step_sets[i].push_back( -0.75 );
		ex_step_sets[i].push_back( -0.67 );
		ex_step_sets[i].push_back( -0.5  );
		ex_step_sets[i].push_back( -0.33 );
		ex_step_sets[i].push_back( -0.25 );
		ex_step_sets[i].push_back(  0.25 );
		ex_step_sets[i].push_back(  0.33 );
		ex_step_sets[i].push_back(  0.5  );
		ex_step_sets[i].push_back(  0.67 );
		ex_step_sets[i].push_back(  0.75 );
		ex_step_sets[i].push_back(  1.   );
		ex_step_sets[i].push_back(  1.25 );
		ex_step_sets[i].push_back(  1.33 );
		ex_step_sets[i].push_back(  1.5  );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_exflags
///
/// @brief expand extrachi flags based on command line options
///
/// this function should go away, it was created to preserve ex* flag behaviour when the -exdb system was added, but these flags could all be parsed together in the main extra-rotamer creation loop in pack
///
/// @authors ctsa 10-2003
///
////////////////////////////////////////////////////////////////////////////////
void RotamerOptions::setup_exflags() {

	// expand extrachi flags based on command line options

	using namespace exchi_flags;
	using namespace param;
	using namespace param_aa;

	int const ex_exposed = { 1 };
	int const ex_buried = { 2 };

	extrachi_flag.dimension( MAX_AA(), MAX_CHI, 2 );
	extrachi_flag = NO_EXTRA_CHI_SAMPLES;

	if ( ex1 ) {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,1,ex_buried) = ex1_sample_level;
		}
	} else {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,1,ex_buried) = NO_EXTRA_CHI_SAMPLES;
		}
	}

	if ( ex2 ) {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,2,ex_buried) = ex2_sample_level;
		}
	} else {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,2,ex_buried) = NO_EXTRA_CHI_SAMPLES;
		}
	}

	if ( ex3 ) {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,3,ex_buried) = ex3_sample_level;
		}
	} else {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,3,ex_buried) = NO_EXTRA_CHI_SAMPLES;
		}
	}

	if ( ex4 ) {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,4,ex_buried) = ex4_sample_level;
		}
	} else {
		for ( int i = 1, e = MAX_AA(); i <= e; ++i ) {
			extrachi_flag(i,4,ex_buried) = NO_EXTRA_CHI_SAMPLES;
		}
	}

	//chu's flag for trying chi2 rotamers selectively on Phe, Tyr, Trp
	if ( ! ex2 && ex2aro_only ) {
		extrachi_flag(aa_phe,2,ex_buried) = EX_ONE_STDDEV;
		extrachi_flag(aa_trp,2,ex_buried) = EX_ONE_STDDEV;
		extrachi_flag(aa_tyr,2,ex_buried) = EX_ONE_STDDEV;
	}

  if ( ex1aro || rot_explode ) {
		extrachi_flag(aa_phe,1,ex_buried) = EX_THREE_THIRD_STEP_STDDEVS;
		extrachi_flag(aa_trp,1,ex_buried) = EX_THREE_THIRD_STEP_STDDEVS;
		extrachi_flag(aa_tyr,1,ex_buried) = EX_THREE_THIRD_STEP_STDDEVS;
	}

  if ( ex1aro_half ) {
		extrachi_flag(aa_phe,1,ex_buried) = EX_TWO_HALF_STEP_STDDEVS;
		extrachi_flag(aa_trp,1,ex_buried) = EX_TWO_HALF_STEP_STDDEVS;
		extrachi_flag(aa_tyr,1,ex_buried) = EX_TWO_HALF_STEP_STDDEVS;
	}

	//flags for expanding rotamer choices for aromatics even if they are
	// not fully buried.
	if ( ex1aro_exposed ) {
		extrachi_flag(aa_phe,1,ex_exposed) = ex1_sample_level;
		extrachi_flag(aa_trp,1,ex_exposed) = ex1_sample_level;
		extrachi_flag(aa_tyr,1,ex_exposed) = ex1_sample_level;
	}
	if ( ex2aro_exposed ) {
		extrachi_flag(aa_phe,2,ex_exposed) = ex2_sample_level;
		extrachi_flag(aa_trp,2,ex_exposed) = ex2_sample_level;
		extrachi_flag(aa_tyr,2,ex_exposed) = ex2_sample_level;
	}

	if ( exOH ) {
		//though these dihedrals are marked with EX_ONE_STDDEV,
		//the actual sampling behavior is not based on +/- one stddev.
		//Look to rotamer_functions.cc::get_ser_thr_hchi()
		//and rotamer_functions.cc::get_tyr_hchi() for actual behavior
		if ( extrachi_flag(aa_ser,2,ex_buried) == NO_EXTRA_CHI_SAMPLES )
		{extrachi_flag(aa_ser,2,ex_buried) = EX_ONE_STDDEV;}
		if ( extrachi_flag(aa_thr,2,ex_buried) == NO_EXTRA_CHI_SAMPLES )
		{extrachi_flag(aa_thr,2,ex_buried) = EX_ONE_STDDEV;}
		if ( extrachi_flag(aa_tyr,3,ex_buried) == NO_EXTRA_CHI_SAMPLES )
		{extrachi_flag(aa_tyr,3,ex_buried) = EX_ONE_STDDEV;}
	}

	//bk special change for bill, extra rotamers for cys
	if ( excys ) {
		extrachi_flag( aa_cys, 1, ex_exposed) = EX_SIX_QUARTER_STEP_STDDEVS;
		extrachi_flag( aa_cys, 1, ex_buried) = EX_SIX_QUARTER_STEP_STDDEVS;
	}

	// rotamer explosion
	explode_flag.dimension( MAX_AA(), MAX_CHI, 2 );
	explode_flag = NO_EXTRA_CHI_SAMPLES;
	// uniform accross all amino acids and chi angles (for now)
	if ( rot_explode ) {
		for ( int aa(1), e( MAX_AA() ); aa <= e; ++aa ) {
			for ( int chi(1); chi <= 2; ++chi ) {
				explode_flag(aa,chi,ex_exposed) = rot_explode_sample_level_chi12;
				explode_flag(aa,chi,ex_buried) = rot_explode_sample_level_chi12;
			}
			explode_flag(aa,3,ex_exposed) = rot_explode_sample_level_chi3;
			explode_flag(aa,3,ex_buried) = rot_explode_sample_level_chi3;
			for ( int chi(4); chi <= MAX_CHI; ++chi ) {
				explode_flag(aa,chi,ex_exposed) = rot_explode_sample_level_chi4;
				explode_flag(aa,chi,ex_buried) = rot_explode_sample_level_chi4;
			}

		}
		rot_explode_debug_number_screens = 0;
	}

	// 'precompute' the standard deviation step sets that correspond to each of the extra rotamer options
	fill_ex_step_sets();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin read_rotamer_flags
///
/// @brief
///
/// @authors
///
////////////////////////////////////////////////////////////////////////////////
void RotamerOptions::read_rotamer_flags() {

	using namespace exchi_flags;
	using namespace files_paths;

	ex1 = truefalseoption("ex1");
	if (ex1) ex1_sample_level = get_extra_rot_sample_level("ex1", EX_ONE_STDDEV);
	ex2 = truefalseoption("ex2");
	if (ex2) ex2_sample_level = get_extra_rot_sample_level("ex2", EX_ONE_STDDEV);
	if (files_paths::use_conformer){
		ex1 = true;
		ex2 = true;
		ex1_sample_level = EX_ONE_STDDEV;
		ex2_sample_level = EX_ONE_STDDEV;
	}
	ex3 = truefalseoption("ex3");
	if (ex3) ex3_sample_level = get_extra_rot_sample_level("ex3", EX_ONE_STDDEV);
	ex4 = truefalseoption("ex4");
	if (ex4) ex4_sample_level = get_extra_rot_sample_level("ex4", EX_ONE_STDDEV);

	ex1aro = truefalseoption("ex1aro");
	if ( ex1aro ) { ex1 = true; ex1_sample_level = EX_ONE_STDDEV;}
	ex1aro_half = truefalseoption("ex1aro_half");
	if ( ex1aro_half ) { ex1 = true; ex1_sample_level = EX_ONE_STDDEV;}
	ex2aro_only = truefalseoption("ex2aro_only");

	ex1aro_exposed = truefalseoption("ex1aro_exposed");
	if ( ex1aro_exposed ) { ex1 = true; ex1_sample_level = EX_ONE_STDDEV;}
	ex2aro_exposed = truefalseoption("ex2aro_exposed");
	if ( ex2aro_exposed ) { ex2 = true; ex2_sample_level = EX_ONE_STDDEV;}

	exOH = truefalseoption("exOH"); //extra rotamers for hydroxyl hydrogens on SER, THR, and TYR
	intafteroption("extrachi_cutoff",extrachi_cutoff,extrachi_cutoff);
	rot_pert = truefalseoption("rot_pert");
	rot_pert_input = truefalseoption("rot_pert_input");
	if ( rot_pert || rot_pert_input ) {
		realafteroption("pert_acc_prob",1.,pert_acc_prob);
		realafteroption("pert_size",5,pert_size);
	}

	// ctsa - user-specified extra chi angles
	exrotset_from_db = truefalseoption("exdb");
	if ( exrotset_from_db ) {
		if ( ex1 || ex2 ) {
			std::cout << "STOP: cannot use -exdb and -ex[12] flags together." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		exrot_file = '-';
		stringafteroption( "exdb", exrot_file, exrot_file );
		if ( exrot_file == "-" ) {
			std::cout << "STOP: exrot_from_db specified w/o exrot file." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		} else {
			std::cout << "exrot file: " << exrot_file << std::endl;
		}
	}

	//ja oversample, pre-screen lots of rotamers for design-specific contact
	rot_explode = truefalseoption("rotamer_explosion");
	if ( rot_explode ) {
		rot_explode_rtmin = truefalseoption("rot_explode_min");
		rot_explode_sample_level_chi12 =
			get_extra_rot_sample_level( "rotamer_explosion", EX_ONE_STDDEV );
		rot_explode_sample_level_chi3 =
			get_extra_rot_sample_level( "rot_expl_chi3", EX_ONE_STDDEV );
		rot_explode_sample_level_chi4 =
			get_extra_rot_sample_level( "rot_expl_chi4", NO_EXTRA_CHI_SAMPLES );
	}
	minimize_best_rotamers = truefalseoption("min_best_rots");

	rot_opt = truefalseoption("rot_opt");

}

////////////////////////////////////////////////////////////////////////////////
/// @begin RotamerOptions::get_extra_rot_sample_level
///
/// @brief
///
/// @authors
///
////////////////////////////////////////////////////////////////////////////////
exchi_flags::ExtraRotSample
RotamerOptions::get_extra_rot_sample_level(
	std::string const str,
	exchi_flags::ExtraRotSample default_sample_level
)
{
	using namespace exchi_flags;

	int userinput;
	optional_positive_intafteroption( str, default_sample_level, userinput );
	if ( userinput < 0 || userinput >= ExtraRotSampleCardinality )
	{
		std::cerr << "Warning: integer input following -" << str << " flag of ";
		std::cerr << userinput << " exceeds the limit for extra chi angle sample types.\n";
		std::cerr << "Warning: Use values in the range of 0 to " << ExtraRotSampleCardinality - 1;
		std::cerr << std::endl;
		return default_sample_level;
	}

	if ( userinput != default_sample_level )
	{
		std::cerr << "Setting chi sample level for flag -" << str << " to ";
		std::cerr << ExtraRotSampleNames[ userinput ] << std::endl;
	}
	return (ExtraRotSample) userinput;

}

////////////////////////////////////////////////////////////////////////////////
void
RotamerOptions::set_ex12(
	const bool ex1_in,
	const bool ex2_in,
	const bool ex1aro_in /*=false*/
)
{
	using namespace exchi_flags;

	ex1 = ex1_in;
	ex1_sample_level = ex1 ? EX_ONE_STDDEV : NO_EXTRA_CHI_SAMPLES;
	ex1aro = ex1aro_in;
	ex2 = ex2_in;
	ex2_sample_level = ex2 ? EX_ONE_STDDEV : NO_EXTRA_CHI_SAMPLES;
	setup_exflags();
}

