// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

#ifndef INCLUDED_ROTAMER_SET
#define INCLUDED_ROTAMER_SET


// Rosetta headers
#include "InteractionGraphFWD.h"
#include "Rotamer.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// utility headers
#include <utility/vector1.hh>

//stl headers
#include <iosfwd>
#include <vector>


// forward declarations
#include "PackerTask_fwd.h"
#include "pose_fwd.h"
class ConformerLibHandler;


class RotamerSetBase;

class RotamerSetBase
{

 private:

  int nrotamers_; // number of rotamers
  utility::vector1<Rotamer> rotamers_; // a collection of Rotamer objects
  utility::vector1<int> base_rotamer_; // base rotamer for each rotamer in this set

  int effective_numres_; // includes flexible ligands
  int nmoltenres_; // includes flexible ligands
  utility::vector1<int> resid_2_moltenres_;
  utility::vector1<int> moltenres_2_resid_;
  utility::vector1<int> num_states_for_moltenres_;
  utility::vector1<int> rotindex_offsets_;
  utility::vector1<int> nrotoffset_;
  utility::vector1<utility::vector1<int> > first_rot_for_aa_;

  void reset_indexing_arrays();

  void update_indexing_arrays( int const seqpos, int const aa, int const rotinx );

  void setup_indexing_arrays();

 protected:

  void push_rotamer( int const seqpos, int const aa, int const aav, int const rotnum );
  void push_rotamer( Rotamer const & rot );

  void pop_rotamer();

 public:

  inline RotamerSetBase() { reset(); return; };

  inline ~RotamerSetBase() { return; };

  void reset();

  void sort();

  void find_suitable_base_rotamer( int const rotinx );

  // Accessor functions
  inline int nrotamers() const { return nrotamers_; };

  inline int effective_numres() const { return effective_numres_; };

  inline int nmoltenres() const { return nmoltenres_; };

  inline Rotamer get_rotamer( int const rotinx ) const { return rotamers_[rotinx]; };

  inline int report_seqpos( int const rotinx ) const { return rotamers_[rotinx].seqpos(); };
  inline int report_aa( int const rotinx ) const { return rotamers_[rotinx].aa(); };
  inline int report_aav( int const rotinx ) const { return rotamers_[rotinx].aav(); };
  inline int report_rotnum( int const rotinx ) const { return rotamers_[rotinx].rotnum(); };

  inline float get_rperc( int const rotinx ) const { return rotamers_[rotinx].get_rperc(); };
  inline void set_rperc( int const rotinx, float const val )
    { rotamers_[rotinx].set_rperc(val); return; };

  inline float & set_rotactcoord( int const xyzinx, int const rotinx )
    { return rotamers_[rotinx].set_rotactcoord( xyzinx ); };
  inline float get_rotactcoord( int const xyzinx, int const rotinx ) const
    { return rotamers_[rotinx].get_rotactcoord( xyzinx ); };
  inline FArray1D_float const & get_rotactcoord( int const rotinx ) const
    { return rotamers_[rotinx].get_rotactcoord(); };
  inline FArray1D_float & get_rotactcoord( int const rotinx )
    { return rotamers_[rotinx].get_rotactcoord(); };

  inline float & set_rotcoord( int const xyzinx, int const atominx, int const rotinx )
    { return rotamers_[rotinx].set_rotcoord( xyzinx, atominx ); };
  inline float get_rotcoord( int const xyzinx, int const atominx, int const rotinx ) const
    { return rotamers_[rotinx].get_rotcoord( xyzinx, atominx ); };
  inline FArray2D_float const & get_rotcoord( int const rotinx ) const
    { return rotamers_[rotinx].get_rotcoord(); };
  inline FArray2D_float & get_rotcoord( int const rotinx )
    { return rotamers_[rotinx].get_rotcoord(); };

  inline float & set_rchi( int const chiinx, int const rotinx )
    { return rotamers_[rotinx].set_rchi( chiinx ); };
  inline float get_rchi( int const chiinx, int const rotinx ) const
    { return rotamers_[rotinx].get_rchi( chiinx ); };
  inline FArray1D_float const & get_rchi( int const rotinx ) const
    { return rotamers_[rotinx].get_rchi(); };
  inline FArray1D_float & get_rchi( int const rotinx )
    { return rotamers_[rotinx].get_rchi(); };

  inline int & set_rrot( int const chiinx, int const rotinx )
    { return rotamers_[rotinx].set_rrot( chiinx ); };
  inline int get_rrot( int const chiinx, int const rotinx ) const
    { return rotamers_[rotinx].get_rrot( chiinx ); };
  inline FArray1D_int const & get_rrot( int const rotinx ) const
    { return rotamers_[rotinx].get_rrot(); };
  inline FArray1D_int & get_rrot( int const rotinx )
    { return rotamers_[rotinx].get_rrot(); };

  inline float & set_rot_born_radius( int const atominx, int const rotinx )
    { return rotamers_[rotinx].set_rot_born_radius( atominx ); };
  inline float get_rot_born_radius( int const atominx, int const rotinx ) const
    { return rotamers_[rotinx].get_rot_born_radius( atominx ); };
  inline FArray1D_float const & get_rot_born_radius( int const rotinx ) const
    { return rotamers_[rotinx].get_rot_born_radius(); };
  inline FArray1D_float & get_rot_born_radius( int const rotinx )
    { return rotamers_[rotinx].get_rot_born_radius(); };

  inline int & set_base_rotamer( int const rotinx ) { return base_rotamer_[rotinx]; };
  inline int get_base_rotamer( int const rotinx ) const { return base_rotamer_[rotinx]; };

  inline int resid_2_moltenres( int const inx ) const { return resid_2_moltenres_[inx]; };
  inline int moltenres_2_resid( int const inx ) const { return moltenres_2_resid_[inx]; };
  inline int num_states_for_moltenres( int const inx ) const { return num_states_for_moltenres_[inx]; };
  inline int rotindex_offsets( int const inx ) const { return rotindex_offsets_[inx]; };
  inline int nrotoffset( int const inx ) const { return nrotoffset_[inx]; };
  inline int first_rot_for_aa( int const aa, int const inx ) const { return first_rot_for_aa_[inx][aa]; };

  inline FArray1D_float & get_first_rotactcoord_for_aa( int const aa, int const inx )
    { return rotamers_[first_rot_for_aa_[inx][aa]].get_rotactcoord(); };
  inline FArray2D_float & get_first_rotcoord_for_aa( int const aa, int const inx )
    { return rotamers_[first_rot_for_aa_[inx][aa]].get_rotcoord(); };

  // for debugging, etc.
  void dump_rchi() const;

  // for debugging, etc.
  void dump_rotcoord() const;

  // for debugging, etc.
  void dump_member_data() const;

  void remove( std::vector<int> const & exclude );

  void copy_rotamer(
    RotamerSetBase & source,
    int const old_index,
    int const new_index,
    bool const same = true, // skip if old_index = new_index?
    bool const update_index_arrays = true
    // we can skip this if we'll be calling it from the invoking function
  );

  inline void append_rotamer( RotamerSetBase const & src_set, int const rotnum )
    { push_rotamer(src_set.get_rotamer(rotnum)); return; }

  inline void change_chi_angle_update_coors( int const rotinx, int const ichi, float const chi )
    { rotamers_[rotinx].change_chi_angle_update_coors(ichi,chi); return; };

  inline void fill_chi_from_coors( int const rotinx )
    { rotamers_[rotinx].fill_chi_from_coors(); return; };

  inline void compute_rperc( int const rotinx )
    { rotamers_[rotinx].compute_rperc(); return; };

  inline void fill_rotactcoord( int const rotinx )
    { rotamers_[rotinx].fill_rotactcoord(); return; };

};


class RotamerSet : public RotamerSetBase {

 private:

  void build_biased_rotamers_by_replacement(
    int total_residue,
    const FArray1D_int & res,
    const FArray1D_int & res_variant,
    const FArray3D_float & full_coord,
    int const rot_seqpos,
    int const rot_aa,
    const PackerTask & Task
  );

  void build_biased_rotamers_by_appending(
    int total_residue,
    const FArray1D_int & res,
    const FArray1D_int & res_variant,
    const FArray3D_float & full_coord,
    int const rot_seqpos,
    int const rot_aa,
    const PackerTask & Task
  );

  float get_rotamer_biasing_ene(
    int const rot_seqpos,
    int const rot_aa,
    int const rot_aav,
    FArray2Da_float rotcoord,
    FArray1Da_float rotactcoord,
    int const rotinx,
    const PackerTask & Task
  );


 public:

  void write_rotamer_set_to_file(
      std::ofstream & outfile,
      pack::PDInteractionGraph & ig
  );

  void read_rotamer_set_from_file(
      std::ifstream & infile,
      pack::PDInteractionGraph & ig
  );

  void find_correspondence_between_file_and_instance(
      int node_index,
      pack::PDInteractionGraph & ig,
      FArray1DB_int & file_states_aa,
      FArray1DB_int & file_states_aav,
      FArray2DB_float & file_states_rchi,
      int rotamer_offset_for_node
  );

  void get_rotamers(
    int total_residue,
    const FArray1D_int & res,
    const FArray1D_int & res_variant,
    const FArray3D_float & full_coord,
    FArray1D_int & current_rot_index,
    PackerTask const & Task
  );


  void get_AR_AspAsn_rotamers(
    int const total_residue,
    FArray3Da_float full_coord,
    const FArray2D_float & chi_curr,
    const FArray2D_int & rot_curr,
    int const aa,
    int const aav,
    int const seqpos
  );


  void get_rotamers_seqpos_aa_aav(
    int total_residue,
    const FArray1D_int & res,
    const FArray1D_int & res_variant,
    const FArray3D_float & full_coord,
    int const aa,
    int const aav,
    int const seqpos,
    int & nrotaa,
    bool const current_chi_already_included,
    std::string const & mode,
    const PackerTask & Task
  );


  void add_extra_rot_from_coord(
    int const i,
    FArray1D_int const & aan,
    FArray1D_int const & res_variant,
    int & nrotaa,
    const DesignMap & design_map,
    FArray1Da_float phi,
    FArray1Da_float psi,
    FArray2Da_int pdbrot,
    FArray2Da_float pdbchi,
    int const total_residue,
    FArray2D_float const & actcoord,
    FArray3D_float const & full_coord,
    FArray1D_int & current_rot_index
  );


  void add_rot_pert_input(
    int & i,
    FArray1Da_int aan,
    FArray1Da_int res_variant,
    int & nrotaa,
    const DesignMap & design_map,
    FArray1Da_float phi,
    FArray1Da_float psi,
    int & total_residue,
    FArray3Da_float full_coord,
    FArray2Da_float chi_curr,
    FArray2Da_int rot_curr
  );


  void add_extra_rot_from_chi(
    int const i,
    FArray1D_int const & aan,
    FArray1D_int const & aav,
    int & nrotaa,
    const DesignMap & design_map,
    FArray1Da_float phi,
    FArray1Da_float psi,
    FArray2Da_int pdbrot,
    FArray2Da_float pdbchi,
    int const total_residue,
    FArray3D_float const & full_coord,
    FArray1D_int & current_rot_index
  );

  void add_extra_rot_to_seqpos(
    int & aa,
    int & aav,
    int & seqpos,
    FArray1Da_float phi,
    FArray1Da_float psi,
    FArray3Da_float xyz,
    FArray2Da_float extra_chi,
    FArray2Da_int extra_rot,
    int & nrotaa
  );

  void replace_rot_with_pose_coors(
    pose_ns::Pose & curr_pose,
    int const rotinx_to_replace
  );

	void
	get_rotamers_for_ligand_aa(
		int const seqpos,
		int const total_residue,
		FArray1Da_int res,
		FArray1Da_int res_variant,
		int const aa,
		int const aav,
		int & nrotaa,
		FArray3D_float full_coord,
		PackerTask const & Task
	);

  void rotamer_optimize(
    pose_ns::Pose & pose,
    const DesignMap & design_map
  );

  void select_closest_rotamer( pose_ns::Pose & pose, FArray1Da_int bestrotamer_at_seqpos );

  void update_rotamer_alignment( pose_ns::Pose & curr_pose );

  void update_rotamer_alignment( RotamerSetBase & starting_rotamer_set, pose_ns::Pose & curr_pose );

	std::vector<int>
	rot_to_pack_single_sequence( FArray1DB_int const & sequence ) const;

  void makeSerThrTyrOHConformer();

  // overloaded version of buildConformers to be used in a loop (not yet written)
  void buildConformers( const DesignMap & design_map );
  void buildConformers( const DesignMap & design_map, int const seqpos, int const aa);
  void buildBDAconformers(
      int total_residue,
      const FArray1D_int & res,
      const FArray1D_int & res_variant,
      const FArray3D_float & full_coord,
      const DesignMap & design_map,
      int const seqpos,
      int const aa
  );
  void conformerBumpFilter( const DesignMap & design_map );

};

#endif
