// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   core/kinematics/ShortestPathInFold_tree.hh
/// @brief  Fold tree helper class
/// @author Oliver Lange


#ifndef INCLUDED_core_kinematics_ShortestPathInFold_tree_HH
#define INCLUDED_core_kinematics_ShortestPathInFold_tree_HH

// Unit headers
#include "fold_tree.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>

// C++ Headers
#include <map>
#include <vector>

namespace pose_ns {

/// @brief A helper class that can tell the shortest distance of residues in a given Foldtree
/// The fold-tree is parsed only in the beginning such that individual queries are fast
class ShortestPathInFoldTree {
public:

  // types
  typedef std::vector< Edge > EdgeList;

  /// @brief cs-tor, parses fold-tree and caches important distances:
  ///        memory N^2+M*5    N: 2 x number of jumps     M: number of residues
  ShortestPathInFoldTree():
		nres_(0)
	{}

  ShortestPathInFoldTree( Fold_tree const& f );

  ShortestPathInFoldTree & operator=( ShortestPathInFoldTree const & src ) {
		jump_res_ = src.jump_res_;
		node_dist_ = src.node_dist_;
		res2jumps_ = src.res2jumps_;
		nres_ = src.nres_;
		max_dist_ = src.max_dist_;
		return *this;
	}

  /// @brief returns the shortest distance of two residues going along Fold-Tree edges.
  int dist( int pos1, int pos2 ) const;

	/// @brief returns the shortest distance for the two residues that are furthest apart
	int max_dist() const {
		if ( !max_dist_ ) max_dist_ = compute_max_dist();
		return max_dist_;
	}

private:

  /// @brief build table that gives for each residue the distance to the next jump(s) (1 or 2)
  void build_peptide_table( Fold_tree const& f);

  /// @brief build N^2 table of distances between jump-residues
  void build_jumpres_distmap( Fold_tree const& f);

  /// @brief if seqpos is a jump_res resturn its internal number
  int get_jump( int seqpos ) const {
    std::map< int, int >::const_iterator fit = jump_res_.find ( seqpos );
    if ( fit != jump_res_.end() ) {
      return fit->second;
    }	else {
      return -1;
    }
  }

  /// @brief helper of build_jumpres_distmap
  void compute_dist_map( Fold_tree const&);
  void init_dist_map( EdgeList const& );
	int compute_max_dist() const;
  /// private data ===============================

  /// @brief map from pose-numbering to internal numbering --> jump_res are counted from 1...N
  /// first: pose-numbering, second internal number
  std::map< int, int > jump_res_;

  /// @brief 2D array for distances between each pair of jump_residues
  ObjexxFCL::FArray2D_int node_dist_;

  /// @brief 1D array for distances of residue j to
  /// each of max 2 jump_residues that are on the same peptide edge in the fold-tree
  ///  --> for each residue 1+2x2 entries.
  ///  < edgenr >  <jump1> <dist1>  <jump2> <dist2>
  ///  jump1 upstream jump_residue (internal numbering) jump2 downstream jump-residues
  ///  no up/downstream jumps --> -1
  ObjexxFCL::FArray2D_int res2jumps_;

  /// @brief for error checking what is nres_ in the fold-tree
  int nres_;

	/// @brief the furthest distance a query to dist() can return
	mutable int max_dist_;
};


} // pose_ns


#endif
