// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

//ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/Fmath.hh>

//Rosetta Headers
#include "SimAnnealerBase.h"
#include "param.h"
#include "pKa_mode.h"
#include "random_numbers.h"//ran3
#include "rotamer_trials.h"
#include "runlevel.h"

//C++ Headers
#include <algorithm>
#include <iostream>

namespace pack{

  const float SimAnnealerBase::hightemp = 100.0;
  const float SimAnnealerBase::lowtemp = 0.3;
  const float SimAnnealerBase::calc_freq_temp = 1.0;
  bool annealing_starts_at_low_temperature = false;

  ////////////////////////////////////////////////////////////////////////////////
  /// @begin SimAnnealerBase::SimAnnealerBase()
  ///
  /// @brief
  /// virtual constructor
  ///
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////
  SimAnnealerBase::SimAnnealerBase
  (
   int num_of_rot_to_pack,
   FArray1D_int & bestrotamer_at_seqpos,
   float & bestenergy,
   bool start_with_current, // start simulation with current rotamers
   const RotamerSet * p_rotamer_set,
   FArray1DB_int & current_rot_index,
   bool calc_rot_freq,
   FArray1D_float & rot_freq
   ):
    num_of_rot_to_pack_( num_of_rot_to_pack ),
    bestrotamer_at_seqpos_(bestrotamer_at_seqpos),
    bestenergy_(bestenergy),
    start_with_current_(start_with_current),
    p_rotamer_set_( p_rotamer_set),
    current_rot_index_(current_rot_index),
    calc_rot_freq_(calc_rot_freq),
    rot_freq_(rot_freq),
    outeriterations_( 0 ),
    inneriterations_( 0 ),
    quench_( false ),
    hightemp_( annealing_starts_at_low_temperature ? 10 : hightemp ),
    lowtemp_( lowtemp ),
    temperature_( hightemp_ ),
    jump_(0),
    delG_( 0.0 ),
    outeriterations_scaling_(1.),
    inneriterations_scaling_(1.),
    low_temp_annealing_( annealing_starts_at_low_temperature )
  {
    bestrotamer_at_seqpos_ = 0;
		save_trajectory_ = false;
  }


  ////////////////////////////////////////////////////////////////////////////////
  /// @begin SimAnnealerBase::~SimAnnealerBase()
  ///
  /// @brief
  /// virtual destructor
  ///
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////
  SimAnnealerBase:: ~SimAnnealerBase()
  {}
  ////////////////////////////////////////////////////////////////////////////////
  /// @begin SimAnnealerBase::get_outeriterations()
  ///
  /// @brief
  /// get the iterations number for simulation
  ///
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////

  int SimAnnealerBase::get_outeriterations() const
  {
    return int (outeriterations_scaling_ * outeriterations_);
  }

  ////////////////////////////////////////////////////////////////////////////////
  /// @begin SimAnnealerBase::get_inneriterations()
  ///
  /// @brief
  ///
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////

  int SimAnnealerBase::get_inneriterations() const
  {

    if ( low_temp_annealing_ )
      {
				return std::max( num_of_rot_to_pack_, int (inneriterations_scaling_ *
						(get_temperature() > 3.0f ? inneriterations_ * 0.5 : inneriterations_ )) );
			}
    else
      {
				return std::max( num_of_rot_to_pack_, int (inneriterations_scaling_ * inneriterations_ ) );
      }
  }

  ////////////////////////////////////////////////////////////////////////////////
  /// @begin SimAnnealerBase::setup_iterations(bool start_with_current)
  ///
  /// @brief
  /// virtual destructor
  ///
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////
  void SimAnnealerBase::setup_iterations()
  {
    setup_iterations( num_of_rot_to_pack_ );
  }


  void SimAnnealerBase::setup_iterations( const int & num_of_state_changes )
  {

    if ( start_with_current_ ){
      inneriterations_ = std::max( num_of_state_changes, (int) current_rot_index_.size1() );
      outeriterations_ = 10;
      //rh Increase pKa iterations to reduce titration noise
      if( pKa_mode::get_pKa_flag() && !runlevel_ns::run_level_status::benchmark) {
	inneriterations_ = 10*inneriterations_;
	outeriterations_ = 10*outeriterations_;
      }
    }else{
      if ( low_temp_annealing_ ){
	inneriterations_ = std::max( 10 * num_of_state_changes, 2 * ((int) current_rot_index_.size1()) );
	outeriterations_ = 10;
      }else{
	inneriterations_ = std::max( 5 * num_of_state_changes, (int) current_rot_index_.size1() );
	outeriterations_ = 20;
      }
    }

	 //std::cerr << "SimAnnealerBase::setup iterations(); inner = " << inneriterations_ << " outer: " << outeriterations_ << std::endl;
  }

  void SimAnnealerBase::setup_temperature( const int & nn )
  {
    if ( nn == get_outeriterations() ) set_to_quench();
    temperature_ = ( hightemp_ - lowtemp_ )*std::exp(-float(jump_)) + lowtemp_;
    jump_++;
  }

  void SimAnnealerBase::setup_temperature(const FArray1D_float & loopenergy,int nn)
  {
    bool calc_rot_freq = get_calc_rot_freq();
    float avgloopE = 0.0f;
    if (( nn == get_outeriterations() )&&(!calc_rot_freq))
      {
	set_to_quench();
	temperature_ = lowtemp_;
      }
    else if ( jump_ > 3)
      {
	avgloopE = (loopenergy( nn - 4 ) + loopenergy( nn -3 ) + loopenergy( nn - 2 ))/3.0;
	if (( loopenergy( nn - 1) - avgloopE ) > -1.0)
	  {
	    temperature_ = hightemp_;
	    jump_ = 1;
	  }
	else
	  {
	    temperature_ = (hightemp_ - lowtemp_)*std::exp(-float(jump_)) + lowtemp_;
	    jump_++;
	  }
      }
    else if( pKa_mode::get_pKa_flag() && pKa_mode::pKa_packer_flags::packer_set_temp )
      {
	temperature_ = pKa_mode::pKa_packer_flags::packer_temp; //rh packer_temp initialized in pKa_mode.cc
      }
    else
      {
	temperature_ = (hightemp_ - lowtemp_)*std::exp(-float(jump_)) + lowtemp_;
	jump_++;


	if (calc_rot_freq && (temperature_ < calc_freq_temp)){
	  temperature_ = calc_freq_temp;
	}
      }
  }

  void SimAnnealerBase::clear() {
    jump_ = 0;
  } // SimAnnealerBase::clear

  void SimAnnealerBase::set_temperature( float new_temp ){ temperature_ = new_temp;}

  float SimAnnealerBase::get_temperature() const { return temperature_; }
  void SimAnnealerBase::set_to_quench(){ quench_ = true;}
  void SimAnnealerBase::set_not_to_quench(){ quench_ = false;}
  bool SimAnnealerBase::quench() const { return quench_; }
  bool SimAnnealerBase::get_start_with_current() const { return start_with_current_; }
  bool SimAnnealerBase::get_calc_rot_freq() const { return calc_rot_freq_; }

  void SimAnnealerBase::set_hightemp( float high) { hightemp_ = high;}
  void SimAnnealerBase::set_lowtemp( float low) {lowtemp_ = low;}

  ////////////////////////////////////////////////////////////////////////////////
  /// @begin SimAnnealerBase::pass_metropolis
  ///
  ///
  /// @brief
  /// accept or reject movement based on Metropolis criterion
  /// if this is the first movement, accept by default.
  /// @detailed
  ///
  /// @global_read
  ///
  /// @global_write
  ///
  /// @remarks
  ///
  /// @references
  ///
  /// @authors
  ///
  /// @last_modified
  ////////////////////////////////////////////////////////////////////////////////
  bool SimAnnealerBase::pass_metropolis( const float & delta_energy)
  {
    const float GOOD_ENERGY = 0.0;
    return pass_metropolis( GOOD_ENERGY, delta_energy );
  }

  bool SimAnnealerBase::pass_metropolis( float previous_energy, float delta_energy )
  {
    float lnprob = 0.0f;
    float beta = 1.0/temperature_;
    float probability = 0.0f;

    if ( delta_energy < 0 ) return true;
    else if (!quench_) //evaluate prob of substitution
      {
	lnprob = beta * delta_energy;
	if ( previous_energy > 1.0 ) // this is specific to FixbbSimAnnealer
	  {
	    //if both previous energy and new energy are poor
	    //increase the probability of accept
	    lnprob /= previous_energy;
	  }
	if ( lnprob < 10.0 )
	  {
	    probability = std::exp(-lnprob);
	    if (probability > ran3() ) return true;
	  }
      }
    return false;
  }

}//end of namespace
