// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 16039 $
//  $Date: 2007-07-14 22:04:19 -0700 (Sat, 14 Jul 2007) $
//  $Author: ashworth $


// Rosetta Headers
#include "aa_name_conversion.h"
#include "analyze_interface_ddg.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "analyze_interface_ddg_ns.h"
#include "decoystats.h"
#include "decoystats_classes.h"
#include "design.h"
#include "design_structure.h"
#include "dna_ns.h"
#include "dock_structure.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "fullatom_sasa_ns.h"
#include "int_fullatom_energies.h"
#include "knots.h"
#include "maps_ns.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "make_pdb.h"
#include "minimize.h"
#include "nblist.h"
#include "output_decoy.h"
#include "pack.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdb.h"
#include "pose_benchmark.h"
#include "pose_constraints.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "current_pose.h"
#include "rotamer_functions.h"
#include "score.h"
#include "score_ns.h"
#include "fullatom_sasa.h"
#include "water.h"
#include "weights_manager.h"
#include "void.h"


// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/Fmath.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>


//tk  INTERFACE/DDG MODE
//tk  Tanja Kortemme, April 2003
//tk  - computes the binding free energy of a protein-protein complex,
//tk  - computes the effect of mutations on the binding free energy
//tk  for calculation of binding free energies and energy function
//tk  see PNAS 2002 Oct 29;99(22):14116-21
//tk  for parameterization of the energy function
//tk  see J Mol Biol. 2003 Feb 28;326(4):1239-59


//tk this file contains the following functions:
//tk ---------------------------------------------
//tk - analyze_interface_ddg
//tk   interface mode main
//tk   input:  complex structure
//tk           optional: mutation list
//tk   output: binding energies in file intout

//tk - setup_interface
//tk   input:  complex structure
//tk   output: general settings for interface mode,
//tk           original complex cordinates and weights in
//tk           analyze_interface_ddg.h
//tk           partner definitions and pdb residue mapping

//tk - binding_energy_one_structure
//tk   input:  complex structure
//tk   output: its binding energy in namespace dgbind_energies
//tk           in int_fullatom_energies.h
//tk           binding energies in file intout

//tk - binding_energy_mutlist
//tk   input:  complex structure
//tk           file name defining mutations (mutlist)
//tk           file name stored in namespace int_chars in
//tk           analyze_interface_ddg.h
//tk   output: binding energy in namespace dgbind_energies
//tk           in int_fullatom_energies.h
//tk           binding energies in file intout

//tk - save_pack_weights
//tk   input:  packer weights in param_pack.h
//tk   output: copy of these weights in namespace
//tk           int_weight_parameters in analyze_interface_ddg.h

//tk - reset_weights_to_pack_weights
//tk   input:  weights in namespace
//tk           int_weight_parameters in analyze_interface_ddg.h
//tk   output: current weights in param_pack.h

//tk - reset_weights_to_int_weights
//tk   input:  current weights in param_pack.h
//tk   output: current weights set to interface weights

//tk - calc_binding_energy_wt_complex
//tk   input:  complex partner boundaries
//tk   output: fullatom binding energies in int_fullatom_energies.h

//tk - calc_binding_energy_mut_complex
//tk   input:  definitions of complex and mutations
//tk   output: fullatom binding energies in int_fullatom_energies.h

//tk - mutate_structure
//tk   input:  coordinates and mutation definition for one structure
//tk   output: coordinates for mutated structure
//tk           its energies are in fullatom_energies.h

//tk - save_scores
//tk   input:  - rosetta fullatom energies, scores in fullatom_energies.h
//tk           - partner for which should scores be saved:
//tk           1: partner 1, 2: partner 2, 3: complex
//tk   output: energies of all three "partners"
//tk           of current complex, stored in int_fullatom_energies.h

//tk - save_wt_scores
//tk   same as save_scores, but specifically saves wt energies
//tk   these are needed for mutant structures when one of the partners
//tk   stays wt

//tk - retrieve_unmutated_part_scores
//tk   input:  which_partner, wt_energies in int_fullatom_energies.h
//tk   output: writes wt_energies for the specified partner into
//tk           corresponding complex_energies in int_fullatom_energies.h

//tk - compile_binding_energy
//tk   input:  - number of structure for which binding energy should be
//tk             computed: 1 for WT, others for mutants
//tk   output: dgbind_energies in int_fullatom_energies.h

//tk - settings_general_functions
//tk - settings_for_pack_rotamers
//tk   these two set general flags that are needed elsewhere in rosetta
//tk   the ones in "general_functions_used_by_interface" are in a namespace
//tk   the ones in "pack_rotamers" are passed directly

//tk - get_complex_boundaries(p1_start, p1_end, p2_start, p2_end)
//tk   input: domain_end, total_residue from misc.h
//tk   output:  p1_start, p1_end, p2_start, p2_end

//tk - set_partner_specific_params
//tk   input: p1_start, p1_end, p2_start, p2_end, which_partner
//tk   output: startres and endres for specified partner

//tk - copy_xyz_aan_aav_phi_psi
//tk   input: xyz_in, aan_in, startres, endres, phi_in, psi_in
//tk   output: xyz_out, aan_out, phi_out, psi_out

//tk - set_aan_variant  -disabled
//tk   output: "variant" sequence needed for pack_rotamers

//tk - map_1_ros_num_to_pdb_res_chain
//tk   input:  residue number in rosetta numbering
//tk   output: original pdb residue number and chain

//tk - map_all_pdb_positions_to_rosetta
//tk   input: residue number and chain from pdb file (in pdb.h)
//tk   output: array pdb_res_chain_to_rosetta mapping pdb info to
//tk           rosetta numbering

//tk - map_complex_num_to_part_num
//tk   input:  residue number in protein complex
//tk   output: residue number in partner

//tk - map_part_num_to_complex_num
//tk   input:  residue number in partner
//tk   output: residue number in complex

//tk - get_complete_mutation_list
//tk   input:  file with all specifications for mutations
//tk           (mutlist, in analyze_interface_ddg_ns.h)
//tk   output: arrays containing all mutation info:
//tk           n_mut_structs:     # of mutated complexes
//tk           mutations_all_res: all positions
//tk           mutations_all_aa:  all amino acids
//tk           n_mut_per_struct:  # of mutations per complex

//tk - set_mut_arrays_curr_struct
//tk   input:  mutation info from mutlist
//tk   output: mutation info for one structure and
//tk           current partner:
//tk           mutations_curr_struct_res: positions
//tk           mutations_curr_struct_aa:  amino acids

//tk - output_complex_ddg_table
//tk   input:  binding energies in int_fullatom_energies.h
//tk   output: table with binding energies, changes in binding
//tk           energies and contributions to those from
//tk           individual energy terms
//tk           in file intout, named INT.OUT by default
//tk           or through command line option -intout
//tk           (in score path)

//bk   repack_wt_struct
//bk           repacks interface residues

//bk   pick_interface_residues_for_repacking
//bk           selects residues to be repacked when repacking
//bk           the whole inteface

//bk   set residues to vary
//bk           when forcing a mutation, determines which neighbors should
//bk           be repacked

//tk UTILITIES:
//tk - initialize_1dim_logical
//tk - convert_chain_char_to_chain_num



////////////////////////////////////////////////////////////////////////////////
/// @begin analyze_interface_ddg
///
/// @brief
/// calculates the binding energy of a protein complex
/// and mutated complexes (defined in mutlist)
///
/// @detailed
/// binding energy is calculated using
/// DGbind =  DG(complex) - DG(partner1) - DG(partner2)
/// changes in binding energy upon mutation are calculated using
/// DDGbind = DGbind(mutant) = DGbind(WT)
/// The two partners are defined by an inserted "TER" between partners,
/// as used in docking mode.
///
/// @global_read
/// lower levels: full_coord, phi, psi, res, total_residue from misc.h
///
/// @global_write
/// lower levels:
/// - original structure coordinates, sequence and length in written into
///   namespace complex_orig in analyze_interface_ddg.h
/// - current structure is written into full_coord, res, total_residue in misc.h
///   before each scoring step
/// - binding energies and contributions for all structures stored in
///   namespace  dgbind_energies in int_fullatom_energies.h
///
/// @remarks
///  !!!! IMPORTANT NOTE TO SCORING IN INTERFACE MODE:
///  The scoring function used is score12,
///  but the weights are reset in reset_weights_to_int_weights
///  to match the scoring function described in the PNAS 2002 paper
///  (actual values might be different from the published weights,
///  this is to port the scoring from TK code to ROSETTA)
///  BUT NOTE: the packer in interface mode uses the weights defined in
///  pack_db.cc in order to have packing consistent with other
///  applications
///
///  to use pack weights for both packing and scoring:
///  use command line option -Wpack_only
///  to use int weights for both packing and scoring:
///  use command line option -Wint_only
///
///  The interface mode can also be used to calculate changes in binding energy
///  upon mutation (mutations are defined via an input file:
///  use command line option -mutlist to define the file name).
///  For info on the format of the mutlist see below.
///  Make sure that the maximum number of mutated complexes (MAX_MUT_STRUCTS)
///  and maximum number of mutations per structure(MAX_MUT_PER_STRUCT)
///  in analyze_interface_ddg.h are set accordingly.
///
///  The results of all binding energy calculations are
///  stored in Table format in a file in the score_path directory
///  The file is called "INT.OUT" by default
///  or a file name can be specified by the command line option -intout
///
/// @references
///
/// @authors Tanja Kortemme 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
analyze_interface_ddg()
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;
	using namespace files_paths;
	using namespace misc;
	using namespace pose_ns;

	//dimension arrays in namespace int_fullatom_energies
	int_fullatom_energies::initialize_MAX_MUT_STRUCTS();

	 // start end end residues of complex partners
	int p1_start, p1_end, p2_start, p2_end;
	// maps pdb residue number and chain to rosetta number
	// instead of an FArray use a map which has for its key the pdb res num
  // and the chain char
	std::map<std::pair<int,int>,int> pdb_res_chain_to_rosetta;

//     *************************************************
//     - SET ROSETTA FLAGS NEEDED IN SHARED FUNCTIONS
//     - save the PACKER WEIGHTS in namespace
//       int_weight_parameters in analyze_interface_ddg.h
//       as they get reset for scoring in interface mode
//     - SET COMPLEX COORDINATES AND DEFINITIONS that
//       should stay fixed throughout the runs:
//       namespace complex_orig in analyze_interface_ddg.h
//       p1_start, p1_end, p2_start, p2_end
//     - MAP pdb residue information (number and chain) to
//       rosetta numbering
//     *************************************************

	setup_interface( p1_start, p1_end, p2_start, p2_end,
	 pdb_res_chain_to_rosetta );

//     ********************************************
//     BINDING ENERGY CALCULATION FOR WT COMPLEX
//     ********************************************
	binding_energy_one_structure( p1_start, p1_end, p2_start, p2_end, 1, true );
	// 1 and true indicate WT

	if ( ddg_bind_only ) {
		output_complex_ddg_table(1,1,1,1);
		return;
	}

//tk   ********************************************************
//tk   BINDING ENERGY CALCULATION FOR LIST OF MUTATED COMPLEXES
//tk   ********************************************************
//tk   get information on all mutations from input file "mutlist"
//tk   file name lives in namespace in analyze_interface_ddg.h

	binding_energies_mutlist( p1_start, p1_end, p2_start, p2_end,
	 pdb_res_chain_to_rosetta );

	if ( ala_scan ) {
		make_named_pdb( output_file + "_alascan" + ".pdb", true );
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin setup_interface
///
/// @brief
/// defines the partners in the complex, and sets up global flags and
/// namespaces (corrdinates, weights) needed for the interface mode
///
/// @detailed
///
/// @param[out]   p1_start - output - start residue of partner 1
/// @param[out]   p1_end   - output - end residue of partner 1
/// @param[out]   p2_start - output - start residue of partner 2
/// @param[out]   p2_end   - output - end residue of partner 2
/// @param[out]   pdb_res_chain_to_rosetta - output - std map mapping a pdb residue
///                                  to its rosetta residue position
///
/// @global_read
/// full_coord, res, phi, psi, total_residue from misc.h
///
/// @global_write
/// original structure coordinates, sequence and length in written into
/// complex_orig namespace in analyze_interface_ddg.h
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
setup_interface(
	int & p1_start,
	int & p1_end,
	int & p2_start,
	int & p2_end,
	std::map<std::pair<int,int>,int>& pdb_res_chain_to_rosetta
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace misc;
	using namespace param;

	 // maps pdb residue number and chain to rosetta number

	settings_general_functions();
//tk   define the complex
	get_complex_boundaries( p1_start, p1_end, p2_start, p2_end );

	if ( !ddg_bind_only ) {
		map_all_pdb_pos_to_rosetta( pdb_res_chain_to_rosetta );
	}

	//bk save interface residues in a global
	interface_residues(); // sets global interface_residue true for interface residues

	//bk minimize (torsion and rigid body): this will be the starting structure
	//bk for the binding energy calculations
	if ( min_interface ) {
		cst_partner1 = cst_set_ns::Cst_set();
		cst_partner2 = cst_set_ns::Cst_set();
		cst_complex = cst_set_ns::Cst_set();
		setup_minimize_interface( cst_complex, cst_partner1, cst_partner2,
      p2_start );
	}

//tk save original structure and sequence in analyze_interface_ddg_ns.h
	copy_xyz_aan_aav_phi_psi( full_coord, xyz_orig, phi, phi_orig, psi,
	 psi_orig, res, aan_orig, res_variant, aav_orig, residue1,
	 residue1_orig, residue3, residue3_orig, Eposition, nres_orig, 1,
	 total_residue );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin binding_energy_one_structure
///
/// @brief
/// calculates the binding energy of a protein complex
/// output is file containing binding energies (intout)
///
/// @detailed
///
/// @param[in]   p1_start   - input - start residue of partner 1
/// @param[in]   p1_end     - input - end residue of partner 1
/// @param[in]   p2_start   - input - start residue of partner 2
/// @param[in]   p2_end     - input - end residue of partner 2
/// @param[in]   listnumber - input - structure number
/// @param[in]   wtflag     - input - is it the wt structure?
///
/// @global_read
///
/// @global_write
///
/// @remarks:
/// global read at lower levels:
/// full_coord, res, total_residue from misc.h
/// global write at lower levels:
/// - results of binding energy calculation are stored in
///   namespace dgbind_energies in int_fullatom_energies.h
/// - if wtflag is true, total energies for the wt complex and its
///   partners, including contributions, are stored in
///   namespace wt_energies in int_fullatom_energies.h
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
binding_energy_one_structure(
	int & p1_start,
	int & p1_end,
	int & p2_start,
	int & p2_end,
	int  listnumber,
	bool wtflag
)
{
//tk   score complex and store fullatom energy terms
//tk   in complex_* arrays in int_fullatom_energies.h
	calc_binding_energy_complex( p1_start, p1_end, p2_start, p2_end, wtflag );
//tk   calculate contributions to the binding free energy from
//tk   complex_* arrays and store in dgbind_* arrays in
//tk   int_fullatom_energies.h
	compile_binding_energy(listnumber);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin binding_energy_mutlist
///
/// @brief
/// calculates the binding free energies of a list of mutated
/// protein complexes defined in mutlist (see below)
/// output is file containing binding energies (intout)
///
/// @detailed
/// binding energy is calculated using
/// DGbind =  DG(complex) - DG(partner1) - DG(partner2)
/// changes in binding energy upon mutation are calculated using
/// DDGbind = DGbind(mutant) = DGbind(WT)
///
///  format of the mutation list mutlist:
///   -----------------------------
///   arbitray number of comment lines
///   START
///   XXXXX                // number of mutated structures(5 digits)
///   XXX                  // # of mutations for this structure (3 digits)
///         XXXX  X  X  X  // see *
///         XXXX  X  X  X
///         XXXX  X  X  X
///
///  *the following lines (one for each mutation in the target structure)
///  have this format:
///  9 spaces
///  pos of mutation (4 digits)
///  2 spaces
///  chain (1 digit)
///  2 spaces
///  WT aa (1 digit)
///  2 spaces
///  mut aa (1 digit)
///
/// @param[in]   p1_start   - input - start residue of partner 1
/// @param[in]   p1_end     - input - end residue of partner 1
/// @param[in]   p2_start   - input - start residue of partner 2
/// @param[in]   p2_end     - input - end residue of partner 2
/// @param[out]   pdb_res_chain_to_rosetta - output - map that maps a pdb residue
///                                  to its rosetta residue positionc
/// @global_read
///
/// @global_write
///
/// @remarks
/// global read at lower levels:
/// full_coord, res, total_residue from misc.h
/// global write at lower levels:
/// - results of binding energy calculation are stored in
///   namespace dgbind_energies in int_fullatom_energies.h
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
binding_energies_mutlist(
	int & p1_start,
	int & p1_end,
	int & p2_start,
	int & p2_end,
	std::map<std::pair<int,int>,int>& pdb_res_chain_to_rosetta
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;

	Dimension n_mut_structs; // total number of mutated structures
	FArray2D_int mutations_all_res( n_mut_structs, MAX_MUT_PER_STRUCT );
	// mutated positions for all structures
	FArray2D_int mutations_all_aa( n_mut_structs, MAX_MUT_PER_STRUCT );
	// mutated amino acids for all structures
	// maps pdb residue number and chain to rosetta number
	FArray1D_int n_mut_per_struct( n_mut_structs );
	// number of mutations per structure

//tk   get information on all mutations from input file "mutlist"
//tk   the file name "mutlist" should be passed in here,
//tk   but it lives in analyze_interface_ddg_ns.h for ROSETTA reasons
	get_complete_mutation_list( n_mut_structs, mutations_all_res,
	 mutations_all_aa, n_mut_per_struct, pdb_res_chain_to_rosetta );

//tk   make mutations and score mutated complexes
	for ( int curr_mutated_complex = 1; curr_mutated_complex <= n_mut_structs();
	 ++curr_mutated_complex ) {
		std::cout << "examining mutant # " << curr_mutated_complex << std::endl;
		calc_binding_energy_mut_complex(p1_start,p1_end,p2_start,p2_end,
		 mutations_all_res,mutations_all_aa,curr_mutated_complex,
		 n_mut_per_struct(curr_mutated_complex));
		compile_binding_energy(curr_mutated_complex+1); // counter for list
	}

//tk   write all results into output file in score path
//tk   the file is called "INT.OUT" by default or can be
//tk   named using the command line option -intout

	output_complex_ddg_table( n_mut_structs, mutations_all_res, mutations_all_aa,
	 n_mut_per_struct );

}

////////////////////////////////////////////////////////////////////////////////
/// @begin save_pack_weights
///
/// @brief
/// stores weights used in the packer in namespace int_weight_parameters
/// in analyze_interface_ddg.h
///
/// @detailed
/// needed to switch between weights used for packing side chains
/// that should be equivalent to current parameterization used
/// in the packer and possible alternative scheme parameterized
/// to reproduce changes in (binding) free energy upon mutation
///
/// @global_read
/// packer weights in param_pack.in
///
/// @global_write
/// namespace int_weight_parameters in analyze_interface_ddg.h
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
save_pack_weights()
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;
	using namespace param_pack;
	using namespace int_weight_parameters;

	static bool init = { false };

	if ( init ) return;
	init = true;

	StoreCurrentWeights( PW_SAVE_PACK );
  Waa_saved = pack_wts.Waa(); // maintain local copy
 // pack_wts.print();//JSS debug
}

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_weights_to_pack_weights
///
/// @brief
/// reset currently used weights to weights stored in namespace
/// int_weight_parameters in analyze_interface_ddg.h
///
/// @detailed
/// see "FULL" comments to save_pack_weights
///
/// @global_read
/// int_weight_parameters in analyze_interface_ddg.h
///
/// @global_write
/// weights in param_pack.h
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_weights_to_pack_weights()
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;
	using namespace param_pack;
	using namespace design;
	using namespace int_weight_parameters;

//car Since save_pack_weights() self-initializes, call it here
//car to make sure that the pack weights were actually saved.
	save_pack_weights();

	// jk  read weights from a file, if desired
	if ( weightfile::use_weightfile ) {
		ReadWeights( weightfile::weight_fname );
		return;
	}

	RetrieveWeightsToCurrent( PW_SAVE_PACK );
  Waa_saved = pack_wts.Waa(); // maintain local copy
  // pack_wts.print();//JSS debug
}

//////////////////////////////////////////////////////////////////////////////
/// @begin set_new_Waa_value
///
/// @brief set Waa for a particular amino acid
///
/// @detailed
/// allows for favoring/penanlizing  certain amino acids during design
///
/// @param[in]  aa_type - which aa to reset
/// @param[in]  new_weight -  new Waa value
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @refrences
///
/// @authors Glenn Butterfoss
///
/// @last_modified 30 June 2005
////////////////////////////////////////////////////////////////////////////////
void
set_new_Waa_value(
									int aa_type,
									float new_weight
									)
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;
	using namespace param_pack;
	using namespace int_weight_parameters;

	save_pack_weights();

	if ( Waa_saved( aa_type ) != pack_wts.Waa( aa_type ) ) {
		std::cout << "Error: set_new_Waa_value: trying to write over currently";
		std::cout << " reset Waa for amino acid type " << aa_type;
		std::cout << " must call reset_Waa_value first" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	pack_wts.set_Waa( aa_type, new_weight);
  // pack_wts.print();//JSS debug
}

//////////////////////////////////////////////////////////////////////////////
/// @begin reset_Waa_value
///
/// @brief return Waa set in param_pack
///
/// @detailed
/// undoes set_new_Waa_value
///
/// @param[in]  aa_type - which aa to reset
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @refrences
///
/// @authors Glenn Butterfoss
///
/// @last_modified 19 July 2004
////////////////////////////////////////////////////////////////////////////////
void
reset_Waa_value( int aa_type )
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;
	using namespace param_pack;
	using namespace int_weight_parameters;

	if ( Waa_saved( aa_type ) == 999 ) {
		std::cout << "Error: new Waa for amino acid type " << aa_type;
		std::cout << " was not previously set ... exiting function reset_Waa_value" << std::endl;
		return;
	}

	pack_wts.set_Waa( aa_type, Waa_saved( aa_type ));
  //pack_wts.print();//JSS debug
}

////////////////////////////////////////////////////////////////////////////////
/// @begin reset_weights_to_int_weights
///
/// @brief
/// resets currently used weights to interface weights
/// interface weights are defined here
///
/// @detailed
/// see "FULL" comments to save_pack_weights
/// these weights have been parameterized by Lin using data on free energy
/// changes upon mutation in monomeric proteins and protein complexes,
/// and should approximately match the published hotspot predictions
///
/// @global_read
///
/// @global_write
/// weights in param_pack.h
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
reset_weights_to_int_weights()
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;
	using namespace param_pack;

	// jk  read weights from a file, if desired
	if ( weightfile::use_weightfile ) {
		ReadWeights( weightfile::weight_fname );
		return;
	}

//car before changing these, make sure that they've been saved:
	save_pack_weights();
//	if (single_protein) {
//			RetrieveWeightsToCurrent(PW_DDG_MONOMER);
//	} else {
			RetrieveWeightsToCurrent( PW_INTERFACE );
//	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_binding_energy_complex
///
/// @brief
/// calculates the binding energy of the wt complex defined by the boundaries
/// of the two partners (function arguments) and stores energetic contributions
/// in int_fullatom_energies.h
///
/// @detailed
///
/// @param[in]   p1_start - input - start residue of partner 1
/// @param[in]   p1_end   - input - end residue of partner 1
/// @param[in]   p2_start - input - start residue of partner 2
/// @param[in]   p2_end   - input - end residue of partner 2
/// @param[in]   wtflag   - input - is it the wt structure?
///
/// @global_read
/// namespace complex_orig arrays from analyze_interface_ddg.h
/// comtaining coordinates and sequence information of the complex
///
/// @global_write
/// the current structure (can be the full complex or either partner)
/// is written into full_coord,total_residue,res in misc.h
/// before each scoring step
/// in the end original the complex structure is written back into
/// full_coord,total_residue,res in misc.h
///
/// @remarks
/// results of binding energy calculation are stored in
/// fullatom binding energies in int_fullatom_energies.h:
/// namespace complex_energies for specific partners
/// namespace wt_energies for the wt complex
///
/// @references
///
/// @authors Tanja Kortemme 08/19/03
///
/// @last_modified
///
/////////////////////////////////////////////////////////////////////////////////
void
calc_binding_energy_complex(
	int & p1_start,
	int & p1_end,
	int & p2_start,
	int & p2_end,
	bool wtflag
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace design;
	using namespace files_paths;
	using namespace misc;
	using namespace param;
	using namespace scorefxns;


	score_set_try_rotamers(false);

//tk   score both partners and complex, save score for each
//tk   protein
//tk   1: partner 1
//tk   2: partner 2
//tl   3: complex
//bk   start the for loop with the complex so that if repacking or minimizing
//bk   the relaxed complex can serve as the starting point for calculations
//bk   with the unbound states
	for ( int which_partner = 3; which_partner >= n_partner; --which_partner ) {

		//ja necessary...?
		int startres, endres;
		set_partner_specific_params( which_partner, startres, endres, p1_start,
		 p1_end, p2_start, p2_end, nres_orig );

		// copy partner 1 from complex to misc
		if ( which_partner == 1 ) {
			copy_xyz_aan_aav_phi_psi( xyz_orig, full_coord, phi_orig, phi,
			 psi_orig, psi, aan_orig, res, aav_orig, res_variant,
			 residue1_orig, residue1, residue3_orig, residue3, Eposition,
			 total_residue, p1_start, p1_end );

		// copy partner 2 from complex to misc
		} else if ( which_partner == 2 ) {
			copy_xyz_aan_aav_phi_psi( xyz_orig, full_coord, phi_orig, phi,
			 psi_orig, psi, aan_orig, res, aav_orig, res_variant,
			 residue1_orig, residue1, residue3_orig, residue3, Eposition,
			 total_residue, p2_start, p2_end );
		}
		// this is the full structure, no need to	copy anything yet
//	else if ( which_partner == 3 );

		int iterations = ( min_interface ? 2 : 1 );
		for ( int i = 1; i <= iterations; ++i ) {
			if ( dna_interface ) {
				if ( !relax_unbound || which_partner != 1 ) break;
				else repack_neighbors = true;
			}
			// only repack unbound if the flag relax_unbound is true
			if ( repack_neighbors && ( relax_unbound || which_partner == 3 ) ) {
				set_weights_for_packer();
				repack_wt_structure( which_partner, p2_start );
			}
			// only minimize unbound if the flag relax_unbound is true
			if ( min_interface && ( relax_unbound || which_partner == 3 ) ) {
				set_weights_for_packer();
				if ( min_interface ) {
					int nchains;
					cst_set_ns::Cst_set cst_set;
					if ( which_partner == 1 ) {
						nchains = 1;
						cst_set = cst_partner1;
					} else if ( which_partner == 2 ) {
						nchains = 1;
						cst_set = cst_partner2;
					} else {
						nchains = ( single_protein ? 1 : 2 );
						cst_set = cst_complex;
					}
					bool first_min = ( i == 1 ? true : false );
					minimize_interface( cst_set, nchains, p2_start, first_min );
				}
			}
		}

		score_set_new_pose();
		set_weights_for_score();
		mc_global_track::mc_score::score = score12();

		if (which_partner == 3) {
			//dws make neighbor lists and copy to global
			FArray2D_bool neighborlist( MAX_RES(), MAX_RES() );
			FArray1D_int neighbors( MAX_RES()() );

			make_neighbor_info( res, total_residue, full_coord, neighborlist, neighbors );

			//dws Save number of neighbors.  This value will be output in the intout file
			//dws so it can be used as a filter for the affinity_increase protocol
			for (int d=1; d<=total_residue; ++d) {
				number_of_neighbors_orig(d) = neighbors(d); //dws saving neighbor array in global
			}
		}
		//bk save structure of complex and unbound partners to be the starting point
		//bk for mutant calculations
		//ja warning, could permanently change your input structure if anything goes
		//ja back into the misc arrays after this
		if ( which_partner == 1 ) {
			// pass on the relaxed structure for chain A (or 1st chain)
			endres_chA = endres;
			copy_xyz_aan_aav_phi_psi( full_coord, xyz_chA, phi, phi_chA, psi, psi_chA,
			 res, aan_chA, res_variant, aav_chA, residue1, residue1_chA, residue3,
			 residue3_chA, Eposition, endres_chA, 1, total_residue );
		} else if ( which_partner == 2 ) {
			// pass on the relaxed structure for chain B (or 2nd chain)
			endres_chB = endres;
			copy_xyz_aan_aav_phi_psi( full_coord, xyz_chB, phi, phi_chB, psi, psi_chB,
			 res, aan_chB, res_variant, aav_chB, residue1, residue1_chB, residue3,
			 residue3_chB, Eposition, endres_chB, 1, total_residue );
		} else if ( which_partner == 3 ) {
			// pass on the relaxed complex
			copy_xyz_aan_aav_phi_psi( full_coord, xyz_orig, phi, phi_orig, psi,
			 psi_orig, res, aan_orig, res_variant, aav_orig, residue1, residue1_orig,
			 residue3, residue3_orig, Eposition, total_residue, startres, endres );
		}


//glb calculate hbonds & sasa
		//setup
		if ( ddg_decoystats ) {

			std::cout << "STARTING DECOYSTATS PARTNER ";
			if ( 1 == which_partner ) std::cout << "A: ";
			if ( 2 == which_partner ) std::cout << "B: ";
			if ( 3 == which_partner )	std::cout << "COMPLEX AB: ";
			std::cout << std::endl;

			reset_decoy_scores();
			decoystats_reset_output_lines();
			copy_hbenergies();
			std::list < unsatisfied_buried_polar > uns_list;
			std::list < unsatisfied_buried_group > group_uns_list;
			find_unsatisfied_hbonds(uns_list, group_uns_list);
			report_unsatisfied_hbonds("",uns_list,group_uns_list);
			calc_aprox_sasa();
			calc_sasa_pack_score(total_residue, res, res_variant, full_coord);
		}

//tk save score for specified partner in int_fullatom_energies.h
//tk also save scores in equivalent wt_* arrays
//tk describing the energetics of the wt complex

		save_scores( which_partner, total_residue );
		if ( wtflag ) save_wt_scores( which_partner );
		if ( output_structure ) {
			std::cout << " in output structure analyze" << std::endl;
			//tk file name of pdb file in list without path
			make_named_pdb( output_file + "_nat_" +
			 lead_zero_string_of( which_partner, NUMBER_MAX_LENGTH ) + ".pdb", true );
		}
	}

//tk   reset res and total_residue for complex
	int which_partner = 3, startres, endres;
	set_partner_specific_params( which_partner, startres, endres, p1_start,
	 p1_end, p2_start, p2_end, nres_orig );

	//ja restore original full complex structure, but ONLY works if you haven't
	//ja done anything naughty to xyz_orig et al
	copy_xyz_aan_aav_phi_psi( xyz_orig, full_coord, phi_orig, phi,
	 psi_orig, psi, aan_orig, res, aav_orig, res_variant,
	 residue1_orig, residue1, residue3_orig, residue3, Eposition,
	 total_residue, 1, nres_orig );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_binding_energy_mut_complex
///
/// @brief
/// calculates the binding energy of a mutant complex defined by the
/// boundaries of the two partners and the desired mutations
/// (function arguments)
/// and stores energetic contributions in the complex_energies arrays in
/// int_fullatom_energies.h
///
/// @detailed
///
/// @param[in]   p1_start - input - start residue of partner 1
/// @param[in]   p1_end   - input - end residue of partner 1
/// @param[in]   p2_start - input - start residue of partner 2
/// @param[in]   p2_end   - input - end residue of partner 2
/// @param[in]   mutations_all_res - input - residues numbers of all mutations
///                             in all structures
/// @param[in]   mutations_all_aa  - input - amino acid types of all mutations
///                             in all structures
/// @param[in]   curr_mutated_complex - input - number of current complex
/// @param[in]   n_mutations_curr_complex - input - number of mutations in the
///                             current complex structure
///
/// @global_read
/// complex_orig arrays from analyze_interface_ddg.h
///
/// @global_write
/// the current structure (can be full complex or either partner)
/// is written into full_coord,total_residue,res in misc.h
/// before each scoring step
/// in the end original the complex structure is written back into
/// full_coord,total_residue,res in misc.h
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 08/19/03
///
/// @last_modified  Brian Kuhlman 11/18/05
/////////////////////////////////////////////////////////////////////////////////
void
calc_binding_energy_mut_complex(
	int & p1_start,
	int & p1_end,
	int & p2_start,
	int & p2_end,
	FArray2D_int const & mutations_all_res,
	FArray2D_int const & mutations_all_aa,
	int & curr_mutated_complex,
	int & n_mutations_curr_complex
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace files_paths;
	using namespace misc;
	using namespace param;

// locals to define current structure to be mutated and scored:
	int n_mutations_curr_struct;
	int startres, endres;
	FArray1D_int mutations_curr_struct_res( MAX_MUT_PER_STRUCT );
	FArray1D_int mutations_curr_struct_aa( MAX_MUT_PER_STRUCT );
	FArray1D_int aan_curr( MAX_RES()() ), aan_m( MAX_RES()() );
	FArray1D_int aav_curr( MAX_RES()() ), aav_m( MAX_RES()() );
	int nres_curr;
	FArray3D_float xyz_curr( 3, MAX_ATOM()(), MAX_RES()() );
	FArray1D_float phi_curr( MAX_RES()() );
	FArray1D_float psi_curr( MAX_RES()() );
	FArray1D_char residue1_curr( MAX_RES(), ' ' );
	FArray1D_string residue3_curr( MAX_RES(), std::string( 3, ' ' ) );
	FArray3D_float xyz_m( 3, MAX_ATOM()(), MAX_RES()() );
	FArray1D_float phi_m( MAX_RES()() );
	FArray1D_float psi_m( MAX_RES()() );
	FArray1D_char residue1_m( MAX_RES(), ' ' );
	FArray1D_string residue3_m( MAX_RES(), std::string( 3, ' ' ) );

	score_set_try_rotamers(false);

//tk   mutate and score both partners and complex, save score for each
//tk   protein
//bk   loop starts with complex first so relaxed mutated complex can
//bk   can be used for scoring the unbound state if desired
//tk   1: partner 1
//tk   2: partner 2
//tk   3: complex
	total_residue = nres_orig; //dws reset total_res so PackerTask can see both chains
	for ( int which_partner = 3; which_partner >= n_partner; --which_partner ) {
		set_partner_specific_params(which_partner,startres,endres,p1_start,p1_end,
		 p2_start,p2_end,nres_orig);

//tk  get starting coordinates
//bk  options are:
//bk    1) use a relaxed wild type unbound structure for chain A
//bk    2) use a relaxed wild type unbound structure for chain B
//bk    3) use unbound derived from the relaxed mutant complex
//bk    4) use wt complex (this will have been relaxed if min_interface is true)
		if ( relax_unbound && ( repack_neighbors || min_interface ) &&
		 which_partner == 1 ) {
			copy_xyz_aan_aav_phi_psi(xyz_chA,xyz_curr,
			 phi_chA,phi_curr,psi_chA,psi_curr,
			 aan_chA,aan_curr,aav_chA,aav_curr,
			 residue1_chA,residue1_curr,residue3_chA,residue3_curr,
			 Eposition,nres_curr,1,endres_chA);
		} else if ( relax_unbound && ( repack_neighbors || min_interface ) &&
		 which_partner == 2 ) {
			copy_xyz_aan_aav_phi_psi(xyz_chB,xyz_curr,
			 phi_chB,phi_curr,psi_chB,psi_curr,
			 aan_chB,aan_curr,aav_chB,aav_curr,
			 residue1_chB,residue1_curr,residue3_chB,residue3_curr,
			 Eposition,nres_curr,1,endres_chB);
		} else if ( !( relax_unbound && ( repack_neighbors || min_interface ) ) &&
		 which_partner <= 2 ) {
			copy_xyz_aan_aav_phi_psi(xyz_m,xyz_curr,
			 phi_m,phi_curr,psi_m,psi_curr,
			 aan_m,aan_curr,aav_m,aav_curr,
			 residue1_m,residue1_curr,residue3_m,residue3_curr,
			 Eposition,nres_curr,startres,endres);
		} else {
			copy_xyz_aan_aav_phi_psi(xyz_orig,xyz_curr,
			 phi_orig,phi_curr,psi_orig,psi_curr,
			 aan_orig,aan_curr,aav_orig,aav_curr,
			 residue1_orig,residue1_curr,residue3_orig,residue3_curr,
			 Eposition,nres_curr,startres,endres);
		}

//tk      set definitions for the mutation, dependent on partner
		set_mut_arrays_curr_struct(mutations_all_res, mutations_all_aa,curr_mutated_complex,
															 //		 which_partner,
		 p2_start,n_mutations_curr_complex,n_mutations_curr_struct,mutations_curr_struct_res,
     mutations_curr_struct_aa,startres,endres);
//tk      check whether there are mutations in the current structure,
//tk      otherwise retrieve energies from WT structure
		if ( n_mutations_curr_struct != 0 || repack_neighbors || min_interface ) {
//tk      create actual mutated structure
//tk      this calls Brian's pack_rotamers with energy
//tk      function specific to the packer
			if (which_partner == 3 || ( relax_unbound && ( repack_neighbors ||
			 min_interface ) ) ) {
				mutate_structure(mutations_curr_struct_res,mutations_curr_struct_aa,
			    n_mutations_curr_struct,xyz_curr,phi_curr,psi_curr,aan_curr,aav_curr,
			    nres_curr);
			}
//tk         copy correct partner into namespace for scoring
//lin        include aav for water rotamers
			copy_xyz_aan_aav_phi_psi(xyz_curr,full_coord,phi_curr,phi,psi_curr,psi,
			 aan_curr,res,aav_curr,res_variant,residue1_curr,residue1,
			 residue3_curr,residue3,Eposition,total_residue,1,nres_curr);

			//bk only relax the unbound states if "relax_unbound" is true
			if ( min_interface && ( which_partner == 3 || relax_unbound ) ) {
				int nchains;
				cst_set_ns::Cst_set cst_set;
				if ( which_partner == 1 ) {
					nchains = 1;
					cst_set = cst_partner1;
				} else if ( which_partner == 2 ) {
					nchains = 1;
					cst_set = cst_partner2;
				} else {
					if ( single_protein ) {
						nchains = 1;
					} else {
						nchains = 2;
					}
					cst_set = cst_complex;
				}
				bool first_min;
				minimize_interface( cst_set, nchains, p2_start, first_min = false );
			}
			//bk if the unbound state is not being relaxed separately
      //bk save the relaxed complex so that it can be used for calculating the
			//bk the energies of the unbound partners
			if (which_partner == 3 && !( relax_unbound && ( repack_neighbors ||
			 min_interface ) ) ) {
				copy_xyz_aan_aav_phi_psi(full_coord,xyz_m,phi,phi_m,psi,psi_m,
				    res,aan_m,res_variant,aav_m,residue1,residue1_m,residue3,residue3_m,
					  Eposition,total_residue,1,nres_curr);
			}

			score_set_new_pose();
			set_weights_for_score();
			mc_global_track::mc_score::score = score12();


			// buried surface area and unsatisfied hbonds
			if ( ddg_decoystats ) {
				reset_decoy_scores();
				decoystats_reset_output_lines();
				copy_hbenergies();
				std::list < unsatisfied_buried_polar > uns_list;
				std::list < unsatisfied_buried_group > group_uns_list;
				find_unsatisfied_hbonds(uns_list, group_uns_list);
				report_unsatisfied_hbonds("",uns_list,group_uns_list);
				calc_sasa_pack_score(total_residue,res,res_variant,full_coord);
				calc_aprox_sasa();
			}

//tk         save score for specified partner
//tk         energies are stored in complex_* arrays
//tk         in int_fullatom_energies.h
			save_scores(which_partner,nres_curr);
			if ( output_structure ) {
				make_named_pdb( output_file + "_mut_" +
				 lead_zero_string_of( curr_mutated_complex, NUMBER_MAX_LENGTH ) + '_' +
				 lead_zero_string_of( which_partner, NUMBER_MAX_LENGTH ) + ".pdb", true );
			}
		} else {
			if (which_partner == 3 && !( relax_unbound && ( repack_neighbors ||
			 min_interface ) ) ) {
				copy_xyz_aan_aav_phi_psi(full_coord,xyz_m,phi,phi_m,psi,psi_m,
				    res,aan_m,res_variant,aav_m,residue1,residue1_m,residue3,residue3_m,
					  Eposition,total_residue,1,nres_curr);
			}
			retrieve_unmutated_part_scores(which_partner);
		}
	} // finished both partners and complex for one mutated structure

//tk   reset res and total_residue
	int which_partner = 3;
	set_partner_specific_params(which_partner,startres,endres,p1_start,p1_end,
	 p2_start,p2_end,nres_orig);

	//rj restore original full complex structure
	 copy_xyz_aan_aav_phi_psi( xyz_orig, full_coord, phi_orig, phi,
	    psi_orig, psi, aan_orig, res, aav_orig, res_variant,
	    residue1_orig, residue1, residue3_orig, residue3, Eposition,
	    total_residue, 1, nres_orig );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin mutate_structure
///
/// @brief
/// mutates a structure according to the definitions given in
/// mutations_curr_struct_res and mutations_curr_struct_aa and returns the
/// mutated structure in xyz_curr,aan_curr,nres_curr (output)
///
/// @detailed
/// the function uses Brian's packer to create the actual mutated
/// structure
/// see function "settings_for_pack_rotamers"
/// for flags needed by the packer when used in interface mode
///
/// @param[in] mutations_curr_struct_res   input - positions of all mutations
///                                   in the current structure
/// @param[in] mutations_curr_struct_aa    input - amino acid types of all
///                                   mutations in the current structure
/// @param[in] n_mutations_curr_struct     input - number of mutations in the
///                                   current structure
/// @param[out] xyz_curr    output - coordinates of mutated structure
/// @param[in] phi_curr    input  - phi angles of current structure
/// @param[in] psi_curr    input  - psi angles of current structure
/// @param[in] aan_curr    input  - sequence of mutated structure
/// @param[in] nres_curr   input  - number of residues of current structure
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 19/08/03
///
/// @last_modified Brian Kuhlman 6/04
/////////////////////////////////////////////////////////////////////////////////
void
mutate_structure(
	FArray1Da_int mutations_curr_struct_res,
	FArray1Da_int mutations_curr_struct_aa,
	int & n_mutations_curr_struct,
	FArray3Da_float xyz_curr,
	FArray1Da_float phi_curr,
	FArray1Da_float psi_curr,
	FArray1Da_int aan_curr,
	FArray1Da_int aav_curr,
	int & nres_curr
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;

	mutations_curr_struct_res.dimension( MAX_MUT_PER_STRUCT );
	mutations_curr_struct_aa.dimension( MAX_MUT_PER_STRUCT );
	xyz_curr.dimension( 3, MAX_ATOM(), MAX_RES() );
	phi_curr.dimension( MAX_RES() );
	psi_curr.dimension( MAX_RES() );
	aan_curr.dimension( MAX_RES() );
	aav_curr.dimension( MAX_RES() );

// locals for pack_rotamers
	bool make_output_file;
	bool include_current;
	bool include_extra;
	FArray1D_bool allow_repack( MAX_RES()() );
	FArray2D_float extra_chi( MAX_CHI, MAX_RES()() );
	FArray2D_int extra_rot( MAX_CHI, MAX_RES()() );
	std::string packmode;

	//bk set the global, design_matrix array, to specifiy which residues
	//bk are being redesigned or repacked
	set_residues_to_vary(mutations_curr_struct_aa,mutations_curr_struct_res,
	 n_mutations_curr_struct,aan_curr,xyz_curr,nres_curr);

	settings_for_pack_rotamers( make_output_file, include_current, include_extra,
	 packmode );

	set_weights_for_packer();

	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;

	misc::total_residue = nres_curr;
	misc::full_coord = xyz_curr;
	misc::res = aan_curr;
	misc::res_variant = aav_curr;
	misc::phi = phi_curr;
	misc::psi = psi_curr;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );
	Task.set_task(packmode,make_output_file,allow_repack,include_current,include_extra,extra_rot, extra_chi);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();

	pack_rotamers( pose, Task );
	//apl test
	pose.copy_to_misc();

	xyz_curr = pose.full_coord();
	nres_curr = pose.total_residue();
	for (int i=1; i<=pose.total_residue(); i++){
		aan_curr(i) = pose.res(i);
		aav_curr(i) = pose.res_variant(i);
		phi_curr(i) = pose.phi(i);
		psi_curr(i) = pose.psi(i);
	}
	pose.copy_to_misc();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_residues_to_vary
///
/// @brief
/// set the design matrix array (a global variable) to specifiy which residues
/// are being redesigned or repacked
///
/// @detailed
///
/// @param[in] mutations_curr_struct_res   input - positions of all mutations
///                                   in the current structure
/// @param[in] mutations_curr_struct_aa    input - amino acid types of all
///                                   mutations in the current structure
/// @param[in] n_mutations_curr_struct     input - number of mutations in the
///                                   current structure
/// @param[in] aan_curr               WT sequence
///
/// @param[in] nres                   number of residues in the protein
///
/// @global_read none
///
/// @global_write design_matrix: design.h
///
/// @remarks
/// design_matrix is used by pack_rotamers to determine which residues will
/// be repacked
///
///
/// @references
///
/// @authors Tanja Kortemme 08/19/03
///
/// @last_modified  Brian Kuhlman 6/04
/////////////////////////////////////////////////////////////////////////////////
void
set_residues_to_vary(
	FArray1DB_int & mutations_curr_struct_aa,
	FArray1DB_int & mutations_curr_struct_res,
	int n_mutations_curr_struct,
	FArray1DB_int & aan_curr,
	FArray3DB_float & xyz_curr,
	int nres_curr
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;
	using namespace design;

// locals
	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() ); // true if two residues are neighbors.
	FArray1D_int neighbors( MAX_RES()() ); // number of residues within 10 angstroms
	FArray1D_bool mutate_this_residue( MAX_RES()() );

	use_design_matrix = true;
	design_matrix = false; // initialize design_matrix
	mutate_this_residue = false;  //initialize

//bk set design_matrix to true for each mutation
	for ( int i = 1; i <= n_mutations_curr_struct; ++i ) {
		int const mut_pos = mutations_curr_struct_res(i);
		int const mut_aa  = mutations_curr_struct_aa(i);
		design_matrix(mut_aa,mut_pos) = true;
		mutate_this_residue(mut_pos) = true;  //local variable
	}

//ds if desired, repack the neighbors of the residues that are mutated
	if ( repack_neighbors ) {
		make_neighbor_info(aan_curr,nres_curr,xyz_curr,neighborlist,neighbors);
		for ( int i = 1; i <= n_mutations_curr_struct; ++i ) {
			int const mut_pos = mutations_curr_struct_res(i);
			for ( int j = 1; j <= nres_curr; ++j ) {
				if ( neighborlist(mut_pos,j) && !mutate_this_residue(j) ) {
					design_matrix(aan_curr(j),j) = true;
				}
			}
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin save_scores
///
/// @brief
/// saves the fullatom energies of a current structure (stored in
/// fullatom_energies.h) in the namespace complex_energies in
/// int_fullatom_energies.h
///
/// @detailed
///
/// @param[in] which_partner   input - integer defining whether energies are
///                               for partner 1 (1), partner 2 (2) or
///                               the complex (3)
/// @param[in] nres   input - number of residues of the current structure
///
/// @global_read
///
/// @global_write
/// namespace complex_energies in int_fullatom_energies.h
///
/// @remarks
/// calls sum_fullatom_energies that retrieves fullatom energies
/// of the current structure stored in fullatom_energies.h
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
save_scores(
	int & which_partner,
	int & nres
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace fullatom_energies;
	using namespace int_fullatom_energies;
	using namespace param;
	using namespace param_aa;
	using namespace fullatom_sasa;
	using namespace protein_maps;
	using namespace design;
	using namespace misc;

//tk fullatom energetic contributions
//tk 1 for partner 1, 2 for partner 2, 3 for complex


// locals
	float solvationE,attractiveE,repulsiveE,referenceE,protonationE,dunbrackE,Paa_ppE,
	 srbb_hbondE,lrbb_hbondE,sc_bb_hbondE,pairprobE,planeprobE,h2oE,h2ohbE,
	 h2o_solE,intra_resE,gb_elecE,elecE,cstE,bond_angleE;

	sum_fullatom_energies(nres,attractiveE,repulsiveE,solvationE,referenceE,
	 protonationE,dunbrackE,Paa_ppE,pairprobE,planeprobE,h2oE,h2ohbE,h2o_solE,srbb_hbondE,
	 lrbb_hbondE,sc_bb_hbondE,intra_resE,gb_elecE,elecE,cstE,bond_angleE);

	complex_atrenergy(which_partner)    = attractiveE;
	complex_repenergy(which_partner)    = repulsiveE;
	complex_solenergy(which_partner)    = solvationE;
	complex_probenergy(which_partner)   = Paa_ppE;
	complex_dunenergy(which_partner)    = dunbrackE;
	complex_intraresenergy(which_partner)= intra_resE;
	complex_unfenergy(which_partner)    = referenceE;
	complex_protonation_energy(which_partner) = protonationE;
	complex_hbenergy(which_partner)     = sc_bb_hbondE;
	complex_resenergy(which_partner)    = fullatomE;
	//	complex_resenergy(which_partner)    = score;  //score includes rama terms not in fullatomE
	complex_pair_energy(which_partner)  = pairprobE;
	complex_plane_energy(which_partner) = planeprobE;
	complex_h2oenergy(which_partner)    = h2oE;
	complex_h2ohbenergy(which_partner)  = h2ohbE;
	complex_h2osolenergy(which_partner) = h2o_solE;
	complex_sasa_gsolt(which_partner)   = sasa_gsolt;
	complex_sasa_polar(which_partner)   = sasa_polar;
	complex_sasa_apolar(which_partner)  = sasa_apolar;
	complex_sasa_pack(which_partner)    = retrieve_sasa_pack_score();
  complex_elecenergy(which_partner)   = elecE;
  if ( cluster_design ) {
		complex_sasa_pack(which_partner)  = retrieve_sub_sasa_pack_score(allow_repack);
	}
	complex_bbhb_uns(which_partner)   	= get_decoy_score("bbhb_uns");
	complex_schb_uns(which_partner)   	= get_decoy_score("schb_uns");
	complex_gu_tot(which_partner)  		 	= get_decoy_score("group_uns");
	complex_Wgu_tot(which_partner) 		 	= get_decoy_score("group_uns_score");

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		complex_resenergy_res(which_partner,seqpos) = resenergy(seqpos);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin save_wt_scores
///
/// @brief
/// stores the wild-type fullatom energies for the two complex partners
/// and the wt complex in the namespace wt_energies
/// in int_fullatom_energies.h
///
/// @detailed
///
/// @param[in] which_partner   input - integer defining whether energies are
///                               for partner 1 (1), partner 2 (2) or
///                               the complex (3)
///
/// @global_read
/// namespace complex_energies in int_fullatom_energies.h
///
/// @global_write
/// namespace wt_energies in int_fullatom_energies.h
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
save_wt_scores( int & which_partner )
{
	using namespace analyze_interface_ddg_ns;
	using namespace int_fullatom_energies;
	using namespace param;

//tk fullatom energetic contributions
//tk 1 for partner 1, 2 for partner 2, 3 for complex

	wt_atrenergy(which_partner)    = complex_atrenergy(which_partner);
	wt_repenergy(which_partner)    = complex_repenergy(which_partner);
	wt_solenergy(which_partner)    = complex_solenergy(which_partner);
	wt_probenergy(which_partner)   = complex_probenergy(which_partner);
	wt_dunenergy(which_partner)    = complex_dunenergy(which_partner);
	wt_intraresenergy(which_partner) = complex_intraresenergy(which_partner);
	wt_unfenergy(which_partner)    = complex_unfenergy(which_partner);
	wt_hbenergy(which_partner)     = complex_hbenergy(which_partner);
	wt_resenergy(which_partner)    = complex_resenergy(which_partner);
	wt_pair_energy(which_partner)  = complex_pair_energy(which_partner);
	wt_h2oenergy(which_partner)    = complex_h2oenergy(which_partner);
	wt_h2ohbenergy(which_partner)  = complex_h2ohbenergy(which_partner);
	wt_h2osolenergy(which_partner) = complex_h2osolenergy(which_partner);
	wt_sasa_gsolt(which_partner)   = complex_sasa_gsolt(which_partner);
	wt_sasa_polar(which_partner)   = complex_sasa_polar(which_partner);
	wt_sasa_apolar(which_partner)  = complex_sasa_apolar(which_partner);
	wt_sasa_pack(which_partner)    = complex_sasa_pack(which_partner);
	wt_bbhb_uns(which_partner)     = get_decoy_score("bbhb_uns");
	wt_schb_uns(which_partner)     = get_decoy_score("schb_uns");
	wt_gu_tot(which_partner)       = get_decoy_score("group_uns");
	wt_Wgu_tot(which_partner)      = get_decoy_score("group_uns_score");
}

////////////////////////////////////////////////////////////////////////////////
/// @begin retrieve_unmutated_part_scores
///
/// @brief
/// in case where only one partner is mutated in the complex,
/// the function retrieves the fullatom_energies of the other,
/// unmutated partner from the wt_energies array in
/// int_fullatom_energies.h and stores them in
/// the namespace complex_energies in int_fullatom_energies.h
///
/// @detailed
///
/// @param[in] which_partner   input - integer defining whether energies are
///                               for partner 1 (1), partner 2 (2) or
///                               the complex (3)
///
/// @global_read
/// namespace wt_energies in int_fullatom_energies.h
///
/// @global_write
/// namespace complex_energies in int_fullatom_energies.h
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
retrieve_unmutated_part_scores( int & which_partner )
{
	using namespace analyze_interface_ddg_ns;
	using namespace int_fullatom_energies;
	using namespace param;

//tk fullatom energetic contributions
//tk 1 for partner 1, 2 for partner 2, 3 for complex

	complex_atrenergy(which_partner)    = wt_atrenergy(which_partner);
	complex_repenergy(which_partner)    = wt_repenergy(which_partner);
	complex_solenergy(which_partner)    = wt_solenergy(which_partner);
	complex_probenergy(which_partner)   = wt_probenergy(which_partner);
	complex_dunenergy(which_partner)    = wt_dunenergy(which_partner);
	complex_intraresenergy(which_partner) = wt_intraresenergy(which_partner);
	complex_unfenergy(which_partner)    = wt_unfenergy(which_partner);
	complex_hbenergy(which_partner)     = wt_hbenergy(which_partner);
	complex_resenergy(which_partner)    = wt_resenergy(which_partner);
	complex_pair_energy(which_partner)  = wt_pair_energy(which_partner);
	complex_h2oenergy(which_partner)    = wt_h2oenergy(which_partner);
	complex_h2ohbenergy(which_partner)  = wt_h2ohbenergy(which_partner);
	complex_h2osolenergy(which_partner) = wt_h2osolenergy(which_partner);
	complex_sasa_gsolt(which_partner)   = wt_sasa_gsolt(which_partner);
	complex_sasa_apolar(which_partner)  = wt_sasa_apolar(which_partner);
	complex_sasa_polar(which_partner)   = wt_sasa_polar(which_partner);
	complex_sasa_pack(which_partner)    = wt_sasa_pack(which_partner);
	complex_bbhb_uns(which_partner)     = wt_bbhb_uns(which_partner);
	complex_schb_uns(which_partner)     = wt_schb_uns(which_partner);
	complex_gu_tot(which_partner)       = wt_gu_tot(which_partner);
	complex_Wgu_tot(which_partner)      = wt_Wgu_tot(which_partner);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin compile_binding_energy
///
/// @brief
/// computes the binding energy of a structure defined by its nnumber in
/// the mutation list (listnumber);
/// the wild-type is "1" by default
///
/// @detailed
/// binding energy computed using
/// DGbind =  DG(complex) - DG(partner1) - DG(partner2)
///
/// @param[in] listnumber   input - number of mutated structure
///
/// @global_read
/// namespace complex_energies in int_fullatom_energies.h
///
/// @global_write
/// namespace dgbind_energies in int_fullatom_energies.h
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
compile_binding_energy( int listnumber )
{
	using namespace analyze_interface_ddg_ns;
	using namespace fullatom_energies;
	using namespace int_fullatom_energies;
	using namespace param;

//tk complex binding energy contributions
//tk 1 for WT, 2-n for mutants

	dgbind_atrenergy(listnumber) = complex_atrenergy(3) - complex_atrenergy(2) -
	 complex_atrenergy(1);
	dgbind_repenergy(listnumber) = complex_repenergy(3) - complex_repenergy(2) -
	 complex_repenergy(1);
	dgbind_solenergy(listnumber) = complex_solenergy(3) - complex_solenergy(2) -
	 complex_solenergy(1);
	dgbind_probenergy(listnumber) = complex_probenergy(3) -
	 complex_probenergy(2) - complex_probenergy(1);
	dgbind_dunenergy(listnumber) = complex_dunenergy(3) - complex_dunenergy(2) -
	 complex_dunenergy(1);
	dgbind_intraresenergy(listnumber) = complex_intraresenergy(3) -
		complex_intraresenergy(2) - complex_intraresenergy(1);
	dgbind_unfenergy(listnumber) = complex_unfenergy(3) - complex_unfenergy(2) -
	 complex_unfenergy(1);
	dgbind_hbenergy(listnumber) = complex_hbenergy(3) - complex_hbenergy(2) -
	 complex_hbenergy(1);
	dgbind_resenergy(listnumber) = complex_resenergy(3) - complex_resenergy(2) -
	 complex_resenergy(1);
	dgbind_pair_energy(listnumber) = complex_pair_energy(3) -
	 complex_pair_energy(2) - complex_pair_energy(1);
	dgbind_h2oenergy(listnumber) = complex_h2oenergy(3) - complex_h2oenergy(2) -
	 complex_h2oenergy(1);
	dgbind_h2ohbenergy(listnumber) = complex_h2ohbenergy(3) -
	 complex_h2ohbenergy(2) - complex_h2ohbenergy(1);
	dgbind_h2osolenergy(listnumber) = complex_h2osolenergy(3) -
	 complex_h2osolenergy(2) - complex_h2osolenergy(1);
	D_sasa_gsolt(listnumber) = complex_sasa_gsolt(3) -
		 complex_sasa_gsolt(2) - complex_sasa_gsolt(1);
	D_sasa_polar(listnumber) = complex_sasa_polar(3) -
		 complex_sasa_polar(2) - complex_sasa_polar(1);
	D_sasa_apolar(listnumber) = complex_sasa_apolar(3) -
		 complex_sasa_apolar(2) - complex_sasa_apolar(1);
	D_sasa_pack(listnumber) = complex_sasa_pack(3) -
		 complex_sasa_pack(2) - complex_sasa_pack(1);
	D_bbhb_uns(listnumber)   = complex_bbhb_uns(3) - complex_bbhb_uns(2) -
		 complex_bbhb_uns(1);
	D_schb_uns(listnumber)   = complex_schb_uns(3) - complex_schb_uns(2) -
		 complex_schb_uns(1);
	D_gu_tot(listnumber)   = complex_gu_tot(3) - complex_gu_tot(2) -
		 complex_gu_tot(1);
	D_Wgu_tot(listnumber)   = complex_Wgu_tot(3) - complex_Wgu_tot(2) -
		 complex_Wgu_tot(1);

//ds
	chainB_atrenergy(listnumber)    = complex_atrenergy(2);
	chainA_atrenergy(listnumber)    = complex_atrenergy(1);
	chainB_repenergy(listnumber)    = complex_repenergy(2);
	chainA_repenergy(listnumber)    = complex_repenergy(1);
	chainB_solenergy(listnumber)    = complex_solenergy(2);
	chainA_solenergy(listnumber)    = complex_solenergy(1);
	chainB_probenergy(listnumber)   = complex_probenergy(2);
	chainA_probenergy(listnumber)   = complex_probenergy(1);
	chainB_dunenergy(listnumber)    = complex_dunenergy(2);
	chainA_dunenergy(listnumber)    = complex_dunenergy(1);
	chainB_intraresenergy(listnumber) = complex_intraresenergy(2);
	chainA_intraresenergy(listnumber) = complex_intraresenergy(1);
	chainB_unfenergy(listnumber)    = complex_unfenergy(2);
	chainA_unfenergy(listnumber)    = complex_unfenergy(1);
	chainB_hbenergy(listnumber)     = complex_hbenergy(2);
	chainA_hbenergy(listnumber)     = complex_hbenergy(1);
	chainB_resenergy(listnumber)    = complex_resenergy(2);
	chainA_resenergy(listnumber)    = complex_resenergy(1);
	chainB_pair_energy(listnumber)  = complex_pair_energy(2);
	chainA_pair_energy(listnumber)  = complex_pair_energy(1);
	chainB_h2oenergy(listnumber)    = complex_h2oenergy(2);
	chainA_h2oenergy(listnumber)    = complex_h2oenergy(1);
	chainB_h2ohbenergy(listnumber)  = complex_h2ohbenergy(2);
	chainA_h2ohbenergy(listnumber)  = complex_h2ohbenergy(1);
	chainB_h2osolenergy(listnumber) = complex_h2osolenergy(2);
	chainA_h2osolenergy(listnumber) = complex_h2osolenergy(1);

//tk safety check

	if ( safety_check ) {
		if ( ( std::abs(dgbind_unfenergy(listnumber)) > 0.1 ) ||
		 ( std::abs(dgbind_probenergy(listnumber)) > 0.1 ) ) {
			std::cout << "UNF  " << dgbind_unfenergy(listnumber) << std::endl;
			std::cout << "PROB " << dgbind_probenergy(listnumber) << std::endl;
			std::cout << "BINDING ENERGY calculation is wrong!" << std::endl;
			std::cout << "Reference/1b terms do not cancel out" << std::endl;
			std::cout << "STOP" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin settings_general_functions
///
/// @brief
/// set flags and parameters needed by interface mode
/// when calling general functions in ROSETTA
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
settings_general_functions()
{
	select_rotamer_set( "large" );
	set_fullatom_flag( true );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin settings_for_pack_rotamers
///
/// @brief
/// set flags and parameters needed by interface mode
/// when calling pack_rotamers in ROSETTA
///
/// @detailed
///
/// @param[out] make_output_file   output
/// @param[out] include_current   output
/// @param[out] include_extra   output
/// @param[out] packmode   output
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
/// for use of the params see pack.cc
///
/// @references
///
/// @authors Tanja Kortemme 03/09/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
settings_for_pack_rotamers(
	bool & make_output_file,
	bool & include_current,
	bool & include_extra,
	std::string & packmode
)
{
	using namespace analyze_interface_ddg_ns;

	make_output_file = false;
	include_current  = false;
	if ( alter_spec_format || repack_neighbors ) include_current = true;
	include_extra    = false;
	packmode = "design";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_complex_boundaries
///
/// @brief
/// returns start and end residue numbers for the two partners in a
/// binary protein complex
///
/// @detailed
/// the two partners are defined by inserting "TER" after the first partner
/// in the input pdb file (convention in docking)
///
/// @param[out] p1_start   output - start res of partner 1
/// @param[out] p1_end     output - end res of partner 1
/// @param[out] p2_start   output - start res of partner 2
/// @param[out] p2_end     output - end res of partner 2
///
/// @global_read
/// domain_end in misc.h: end of partner 1
/// total_residue in misc.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_complex_boundaries(
	int & p1_start,
	int & p1_end,
	int & p2_start,
	int & p2_end
)
{
	using namespace misc;

	p1_start = 1;

	if ( design::dna_interface &&
		   get_complex_boundaries_protein_NA( p1_end, p2_start, p2_end ) ) return;

	p2_start   = domain_end(1)+1;
	p1_end     = p2_start - 1;
	p2_end     = total_residue;
	std::cout << " Protein Complex defined as Residues " <<
	 I( 2, p1_start ) << '-' << I( 4, p1_end ) << " and " <<
	 I( 4, p2_start ) << '-' << I( 4, p2_end ) << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_complex_boundaries_protein_NA
///
/// @brief
/// Represent complex as one single protein partner and one single NA partner
/// Allows multichain protein and multichain NA
/// Requires protein chains first, then dna chains (PDB standard)
///
/// @global_read
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

bool
get_complex_boundaries_protein_NA(
	int & protein_end,
	int & NA_start,
	int & NA_end
)
{
	// assume that the protein chains are first, and that protein and NA regions
	// are contiguous (PDB standard!!!)
	if ( !param_aa::is_protein( misc::res(1) ) ) {
		std::cout << "First domain is not protein in " <<
		 "get_complex_boundaries_dna_interface().  Falling back to to chain " <<
		 "definitions." << std::endl;
		return false;
	}

	protein_end = 0; NA_start = 0; NA_end = 0;

	for ( int dom(1); dom <= misc::total_domains; ++dom ) {
//		std::cerr << "domain " << dom;
		if ( IsDnaDomain( misc::domain_end(dom), misc::res ) ||
		     IsRnaDomain( misc::domain_end(dom), misc::res ) ) {
//			std::cerr << " is NA" << std::endl;
			if ( NA_start == 0 ) {
				protein_end = misc::domain_end(dom-1); // protein ends
				NA_start = protein_end + 1; // NA begins
			}
			if ( dom == misc::total_domains ) NA_end = misc::domain_end(dom);
		}
		// if non-NA encountered after start, set end of NA domain
		else if ( NA_start != 0 ) NA_end = misc::domain_end(dom-1);
//		std::cerr << std::endl;
	}

	if ( NA_start == 0 || NA_end == 0 || protein_end == 0 ) {
		std::cout << "Failed to determine NA interface.  " <<
		 "Falling back to chain definitions." << std::endl;
		return false;
	}

	std::cout << "Protein-NA complex defined as residues " <<
	 I( 4, pdb::pdb_res_num(1) ) << " -" <<
	 I( 4, pdb::pdb_res_num(protein_end) ) << " and " <<
	 I( 4, pdb::pdb_res_num(NA_start) ) << " -" <<
	 I( 4, pdb::pdb_res_num(NA_end) ) << std::endl;

	return true;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_partner_specific params
///
/// @brief
/// returns start and end residue numbers for the specified (which_partner)
/// partner in the complex
///
/// @detailed
///
/// @param[in] which_partner   input - integer defining whether energies are
///                               for partner 1 (1), partner 2 (2) or
///                               the complex (3)
/// @param[out] startes    output - start res for defined partner
/// @param[out] endres     output - end res for defined partner
/// @param[in] p1_start   input - start res of partner 1
/// @param[in] p1_end     input - end res of partner 1
/// @param[in] p2_start   input - start res of partner 2
/// @param[in] p2_end     input - end res of partner 2
/// @param[in] all_res_complex   input - total number of residues in the complex
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_partner_specific_params(
	int & which_partner,
	int & startres,
	int & endres,
	int & p1_start,
	int & p1_end,
	int & p2_start,
	int & p2_end,
	int & all_res_complex
)
{
	using namespace docking;

	if ( which_partner == 1 ) {      // partner 1
		startres = p1_start;
		endres   = p1_end;
		part_begin(1) = 1;                 //these go to the docking namespace,they
		part_end(1) = endres-startres + 1;     //are used when outputing the pdb struct
		part_begin(2) = 0;
		part_end(2) = -1;
	} else if ( which_partner == 2 ) {  // partner 2
		startres = p2_start;
		endres   = p2_end;
		part_begin(1) = 1;                 //these go to the docking namespace,they
		part_end(1) = endres-startres+1;     //are used when outputing the pdb struct
		part_begin(2) = 0;
		part_end(2) = -1;
	} else if ( which_partner == 3 ) {  // complex
		startres = 1;
		endres   = all_res_complex;
		part_begin(1) = p1_start;  //these go to the docking namespace,they
		part_end(1) = p1_end;      //are used when outputing the pdb struct
		part_begin(2) = p2_start;
		part_end(2) = p2_end;
	} else {
		std::cout << "no partner identification for complex! stop..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin copy_xyz_aan_aav_phi_psi
///
/// @brief
/// copies a fragment of the coordinate array xyz_in from startres to endres
/// into new coordinate array yxz_out
/// also copies sequence from aan_in to aan_out and phi/psi angles
///
/// @detailed
/// useful to separate coordinates and sequence of the partners in a protein
/// complex
///
/// @param[in] xyz_in    input  - coordinates
/// @param[out] xyz_out   output - "
/// @param[in] phi_in    input  - phi angles
/// @param[out] phi_out   output - "
/// @param[in] psi_in    input  - psi angles
/// @param[out] psi_out   output - "
/// @param[in] aan_int   input  - sequence
/// @param[out] aan_out   output - "
/// @param[in] aav_int   input  - sequence variant
/// @param[out] aav_out   output - "
/// @param[in] residue1_in starting 1 letter aa code
/// @param[out] residue1_out output 1 letter aa code
/// @param[in] residue3_in starting 3 letter aa code
/// @param[out] residue3_out output 3 letter aa code
/// @param[out] nres_out   output - number of residues
/// @param[in] startres   input
/// @param[out] endres     output
///
///
/// @global_read
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified 29 June 2005 (glb)
/////////////////////////////////////////////////////////////////////////////////
void
copy_xyz_aan_aav_phi_psi(
	FArray3Da_float xyz_in, // starting coordinate array
	FArray3Da_float xyz_out, // output coordinate array
	FArray1Da_float phi_in, // starting phi angles
	FArray1Da_float phi_out, // output phi angles
	FArray1Da_float psi_in, // starting psi angles
	FArray1Da_float psi_out, // output psi angles
	FArray1Da_int aan_in, // starting sequence
	FArray1Da_int aan_out, // output sequence
	FArray1Da_int aav_in, // starting variant
	FArray1Da_int aav_out, // output variant
	FArray1D_char & residue1_in, //starting 1 letter aa code
	FArray1D_char & residue1_out, //output 1 letter aa code
	FArray1D_string & residue3_in, //starting 3 letter aa code
	FArray1D_string & residue3_out, //output 3 letter aa code
	FArray3Dp_float & Eposition, //backbone coordinates
	int & nres_out, // number of residues in output
	int startres,
	int endres
)

{
	using namespace aaproperties_pack;
	using namespace analyze_interface_ddg_ns;
	using namespace design;
	using namespace param;

	xyz_in.dimension( 3, MAX_ATOM(), MAX_RES() );
	xyz_out.dimension( 3, MAX_ATOM(), MAX_RES() );
	phi_in.dimension( MAX_RES() );
	phi_out.dimension( MAX_RES() );
	psi_in.dimension( MAX_RES() );
	psi_out.dimension( MAX_RES() );
	aan_in.dimension( MAX_RES() );
	aan_out.dimension( MAX_RES() );
	aav_in.dimension( MAX_RES() );
	aav_out.dimension( MAX_RES() );
	residue1_in.dimension( MAX_RES() );
	residue1_out.dimension( MAX_RES() );
	residue3_in.dimension( MAX_RES() );
	residue3_out.dimension( MAX_RES() );
	Eposition.dimension( 3, MAX_POS, MAX_RES() );

// initialize
	phi_out = 0.0;
	psi_out = 0.0;
	aan_out = 0;
	aav_out = 0;
	xyz_out = 0.0;
	residue1_out = '_';
	residue3_out = "___";

// set coordinates and sequence
	int out_rescount = 0;
	for ( int i = startres; i <= endres; ++i ) {
		++out_rescount;
		aan_out(out_rescount) = aan_in(i);
		aav_out(out_rescount) = aav_in(i);
		phi_out(out_rescount) = phi_in(i);
		psi_out(out_rescount) = psi_in(i);
		residue1_out(out_rescount) = residue1_in(i);
		residue3_out(out_rescount) = residue3_in(i);
		for ( int ii = 1, iie = natoms( aan_in(i), aav_in(i) ); ii <= iie; ++ii ) {
			for ( int iii = 1; iii <= 3; ++iii ) {
				xyz_out( iii, ii, out_rescount ) = xyz_in( iii, ii, i );
			}
		}
		for ( int iii = 1; iii <= 3; ++iii ) {
			Eposition(iii,1,out_rescount) = xyz_out(iii,1,out_rescount);  //N
			Eposition(iii,2,out_rescount) = xyz_out(iii,2,out_rescount);  //CA
			Eposition(iii,4,out_rescount) = xyz_out(iii,3,out_rescount);  //C
			Eposition(iii,5,out_rescount) = xyz_out(iii,4,out_rescount);  //O
		}
	}
	nres_out = out_rescount;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_aan_variant
///
/// @brief
/// set aa_variant_intout needed for pack_rotamers
///
/// @detailed
///
/// @param[out] aa_varinat_inout   output - variant sequence
/// @param[in] nres_curr          input  - current number of residues
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_aan_variant(
	FArray1Da_int aa_variant_inout,
	int & nres_curr
)
{
	using namespace param;

	aa_variant_inout.dimension( MAX_RES() );

	for ( int i = 1; i <= nres_curr; ++i ) {
		aa_variant_inout(i) = 1;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin map_1_ros_num_to_pdb_res_chain
///
/// @brief
/// finds pdb residue number and pdb chain id used in original complex
/// pdb file for a single residue, given the Rosetta residue number
/// for the complex structure
///
/// @detailed
///
/// @param[in] rosetta_resnum   input  - residue number in rosetta format
/// @param[out] resnum           output - residue number in the input pdb file
/// @param[out] chain            output - chain identifier in the input pdb file
///
/// @global_read
/// pdb_res_num and res_chain in pdb.h
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
map_1_ros_num_to_pdb_res_chain(
	int & rosetta_resnum,
	int & pdb_resnum,
	char & chain
)
{
	using namespace param;
	using namespace pdb;

	pdb_resnum  = pdb_res_num(rosetta_resnum);
	chain       = res_chain(rosetta_resnum);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin map_all_pdb_pos_to_rosetta
///
/// @brief
/// converts pdb residue identifiers (residue number and chain) to rosetta
/// numbers in the complex and stores this info in the std::map
/// pdb_res_chain_to_rosetta(res,chain)
///
/// @detailed
///
/// @param[out] pdb_res_chain_to_rosetta   output - map that maps a pdb residue
///                                  to its rosetta residue position
///
/// @global_read
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
map_all_pdb_pos_to_rosetta( std::map<std::pair<int,int>,int>& pdb_res_chain_to_rosetta )
{
	using namespace analyze_interface_ddg_ns;
	using namespace misc;
	using namespace param;
	using namespace pdb;

// locals
	char chain;
	int chain_num;
	int pdb_resnum;

	for ( int i = 1; i <= total_residue; ++i ) {
		map_1_ros_num_to_pdb_res_chain(i,pdb_resnum,chain);
//	std::cout << "MAP " << i << ' ' << pdb_resnum << ' ' << chain << std::endl;
		convert_chain_char_to_chain_num(chain,chain_num);
		pdb_res_chain_to_rosetta[std::pair<int,int>(pdb_resnum,chain_num)] = i;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin map_complex_num_to_part_num
///
/// @brief
/// maps a residue number in a complex (complex_num) to the rosetta residue
/// number in the isolated partner (part_num)
///
/// @detailed
/// complex_num and part_num are the same for partner 1, but different
/// for partner 2 (when p2_true is true)
///
/// @param[in] complex_num   input  - rosetta residue number in complex
/// @param[out] part_num      output - rosetta residue number in the partner
/// @param[in] p2_start      input  - start res of partner 2
/// @param[in] p2_true       input  - flag determining which partner
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
map_complex_num_to_part_num(
	int & complex_num,
	int & part_num,
	int & p2_start,
	bool & p2_true
)
{
	part_num = complex_num;
	if ( !p2_true ) return;
	if ( complex_num >= p2_start ) part_num = complex_num-p2_start+1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin map_part_num_to_complex_num
///
/// @brief
/// maps a residue number in a complex partner (part_num) to the rosetta
/// residue number in the complex (complex_num)
///
/// @detailed
///
/// @param[in] part_num      input  - rosetta residue number in the partner
/// @param[out] complex_num   output - rosetta residue number in complex
/// @param[in] p2_start      input  - start res of partner 2
/// @param[in] p1_true       input  - flag determining which partner
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
map_part_num_to_complex_num(
	int & part_num,
	int & complex_num,
	int & p2_start,
	bool & p1_true // true if partner1, false if partner 2
)
{
	complex_num = part_num;
	if ( !p1_true ) complex_num = p2_start + part_num - 1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_complete_mutation_list
///
/// @brief
/// 1. read complete mutation list for a single pdb file from mutlist
/// 2. converts information in the mutation list (pdb numbers and chains)
/// into ROSETTA numbering
///
/// @detailed
///
/// @param[out]   n_mut_structs     - output - number of mutated strcutures
/// @param[out]   mutations_all_res - output - residues numbers of all mutations
///                             in all structures
/// @param[out]   mutations_all_aa  - output - amino acid types of all mutations
///                             in all structures
/// @param[out]   n_mut_per_struct  - output - number of mutatations per structure
/// @param[out]   pdb_res_chain_to_rosetta - output - conversion of pdb residue
///                             number and chain id to rosetta number
///
/// @global_read
/// file name mutlist in analyze_interface_ddg.h
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_complete_mutation_list(
	Dimension & n_mut_structs,
	FArray2D_int & mutations_all_res,
	FArray2D_int & mutations_all_aa,
	FArray1D_int & n_mut_per_struct,
	std::map<std::pair<int,int>,int>& pdb_res_chain_to_rosetta
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace files_paths;
	using namespace int_fullatom_energies;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace design;

//tk 1. read complete mutation list for a single pdb file from mutlist
//tk 2. converts information in the mutation list (pdb numbers and chains)
//tk into ROSETTA numbering

//     locals
	int structnum, mutnum;
	int pdb_mut_pos,rosetta_mut_pos;
	char pdb_wt_aa,pdb_mut_aa;
	int num_wt_aa,num_mut_aa;
	int num_pdb_chain;
	std::string startchar;
	char char_pdb_chain;
	FArray2D_bool const allow_mutation(20,20,initialize_allow_mutation);

	//	initialize_allow_mutation(allow_mutation);

//lin  if ala_scan_only, mutate all residue to alanine
	int count_mut_structs = 0;
	if ( ala_scan ) {
		n_mut_structs = nres_orig; // Allocates arrays
		//lin go through all positions
		//		int i_mut_structs = 0;
		for ( int seqpos = 1; seqpos <= nres_orig; ++seqpos ) {
			//if ((aan_orig(seqpos) != aa_gly)
			//    && (aan_orig(seqpos) != aa_pro)
			//    && (aan_orig(seqpos) != aa_cys)) {
			++count_mut_structs;
			n_mut_per_struct(count_mut_structs)    = 1;
			mutations_all_res(count_mut_structs,1) = seqpos;
			mutations_all_aa(count_mut_structs,1)  = aa_ala;
			//}
		}
		return;
	}

	//ds setting n_mut_structs array to fit various pmut_scan
	//dw requirements
	if ( pmut_scan ) {
		//		int i_mut_structs = 0;
		if ( single_protein ) {
			for ( int seqpos = 1; seqpos <= nres_orig; ++ seqpos) {
				for (int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
					++count_mut_structs;
				}
			}
		} else {
			for ( int seqpos = 1; seqpos <= nres_orig; ++ seqpos ) {
				if ( interface_residue( seqpos )) {
					for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
						if ( affin_incr ) {
							if ( allow_mutation ( aa, res(seqpos))) {
								++count_mut_structs;
							}
						} else {
							++count_mut_structs;
						}
					}
				}
			}
		}
	}
	//bk try all possible point mutations at the interface
	int i_mut_structs = 0;
	if ( pmut_scan ) {
		//		int i_mut_structs = 0;
		if ( single_protein ) {
			n_mut_structs = count_mut_structs;  // allocates space for FArrays
			for ( int seqpos = 1; seqpos <= nres_orig; ++ seqpos ) {
				for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
					++i_mut_structs;
					n_mut_per_struct(i_mut_structs) = 1;
					mutations_all_res(i_mut_structs,1) = seqpos;
					mutations_all_aa(i_mut_structs,1) = aa;
				}
			}
		} else {
			//			n_mut_structs = i_mut_structs;  //needed to size arrays
			//			if (affin_incr) {
			//				for ( int seqpos = 1; seqpos <= nres_orig; ++seqpos ) {
			//					if ( interface_residue( seqpos ) )  {
			//						for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
			//							if (allow_mutation(aa,res(seqpos))){
			//								++i_mut_structs;
			//							}
			//						}
			//					}
			//				}
			//			}
			n_mut_structs = count_mut_structs;
			if (affin_incr) {
				for ( int seqpos = 1; seqpos <= nres_orig; ++seqpos ) {
					if ( interface_residue( seqpos ) )  {
						for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
							if (allow_mutation(aa,res(seqpos))){
								++i_mut_structs;
								n_mut_per_struct(i_mut_structs) = 1;
								mutations_all_res(i_mut_structs,1) = seqpos;
								mutations_all_aa(i_mut_structs,1) = aa;
							}
						}
					}
				}
			} else {
				for ( int seqpos = 1; seqpos <= nres_orig; ++seqpos ) {
					if ( interface_residue( seqpos ) )  {
						for ( int aa = 1; aa <= MAX_AUTH_AA; ++aa ) {
							++i_mut_structs;
							n_mut_per_struct(i_mut_structs) = 1;
							mutations_all_res(i_mut_structs,1) = seqpos;
							mutations_all_aa(i_mut_structs,1) = aa;
						}
					}
				}
			}
		}
		//dws remove?		if (affin_incr) {
			std::cout << " " << std::endl;
			std::cout << count_mut_structs << " point mutations will be tested" << std::endl;
			std::cout << " " << std::endl;
			//			i_mut_structs = n_mut_structs;
			//dws remove?		} else {
			//dws remove?			std::cout << " " << std::endl;
			//dws remove?			std::cout << n_mut_structs << " point mutations will be tested" << std::endl;
			//dws remove?			std::cout << " " << std::endl;
			//dws remove?		}
		assert( i_mut_structs == count_mut_structs );
		return;
	}

//lin reading the mutlist
	std::cout << "interface mode, reading mutlist: " << mutlist << std::endl;

	if ( mutlist != "none" ) {
		data_x.open( mutlist );
		if ( !data_x ) {
			std::cout << "Open failed for file: " << data_x.filename() << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else {
		std::cout << "did not find the mutation list for interface mode, stop!" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//tk   find start of mutation list
L40:
	data_x >> bite( 5, startchar ) >> skip;
	if ( data_x.eof() ) {
		goto L50;
	} else if ( data_x.fail() ) {
		data_x.clear();
		data_x >> skip;
		goto L40;
	}
	if ( startchar != "START" ) goto L40;

//tk   get number of mutated structures in the file
	data_x >> bite( 5, n_mut_structs ) >> skip; // Allocates arrays
	if ( data_x.eof() ) goto L50;
	std::cout << "interface mode, found number of mutated structures: " <<
	 SS( n_mut_structs() ) << std::endl;
	if ( n_mut_structs > MAX_MUT_STRUCTS() ) {
		std::cout << "MAX_MUT_STRUCTS EXCEEDED" << std::endl;
		std::cout << "change MAX_MUT_STRUCTS in int_int_fullatom_energies.cc " <<
		 "to at least " << n_mut_structs << std::endl;
		std::cout << "stop..." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

//tk   loop through all mutated structures
	for ( structnum = 1; structnum <= n_mut_structs; ++structnum ) {
		data_x >> bite( 3, n_mut_per_struct(structnum) ) >> skip;
		if ( data_x.eof() ) {
			goto L50;
		} else if ( data_x.fail() ) {
			goto L60;
		}
		for ( mutnum = 1; mutnum <= n_mut_per_struct(structnum); ++mutnum ) {
			data_x >> skip( 9 ) >>
			 bite( 4, pdb_mut_pos ) >> skip( 2 ) >>
			 bite( char_pdb_chain ) >> skip( 2 ) >>
			 bite( pdb_wt_aa ) >> skip( 2 ) >>
			 bite( pdb_mut_aa ) >> skip;
			if ( data_x.eof() ) {
				goto L50;
			} else if ( data_x.fail() ) {
				goto L60;
			}
			num_from_res1(pdb_wt_aa,num_wt_aa);
			num_from_res1(pdb_mut_aa,num_mut_aa);
			convert_chain_char_to_chain_num(char_pdb_chain,num_pdb_chain);
			rosetta_mut_pos =
				pdb_res_chain_to_rosetta[ std::pair<int,int>(pdb_mut_pos,num_pdb_chain) ];
			if ( safety_check ) {
				if ( num_wt_aa != res(rosetta_mut_pos) ) {
					std::cout << std::endl;
					std::cout << "WT SEQUENCE IN MUTLIST DOES NOT MATCH PDB FILE:" << std::endl;
					std::cout << "PDB residue number " << pdb_mut_pos << " is aa " <<
					 res(rosetta_mut_pos) << " but in mutfile it is aa " << num_wt_aa <<
					 std::endl;
					std::cout << "STOP" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
			mutations_all_res(structnum,mutnum) = rosetta_mut_pos;
			mutations_all_aa(structnum,mutnum)  = num_mut_aa;
		} // all mutations in one structure
	} // all structures

	data_x.close();
	data_x.clear();
	return;

L50:
	data_x.close();
	data_x.clear();
	return;
L60:
	std::cout << "ERROR READING mutlist" << std::endl;
	data_x.close();
	data_x.clear();
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_mut_arrays_curr_struct
///
/// @brief
/// converts the mutation specifications of all mutations
/// (number and aa in ROSETTA convention in complex numbering)
/// into specifications needed for the current structure
/// given by the number_of_curr_struct
///
/// @detailed
/// which_partner defines whether this is done for
/// part 1: 1
/// part 2: 2
/// complex:3
///
/// @param[in]   mutations_all_res - input - residues numbers of all mutations
///                             in all structures
/// @param[in]   mutations_all_aa  - input - amino acid types of all mutations
///                             in all structures
/// @param[in]   number_of_curr_struct - input - number of current structure
/// @param[in]   which_partner - input - integer defining whether energies are
///                             for partner 1 (1), partner 2 (2) or
///                             the complex (3)
/// @param[in]   p2_start - input - start residue of partner 2
/// @param[out]   n_mutations_curr_complex - output - number of mutations in the
///                             current complex
/// @param[out]   n_mutations_curr_struct - output - number of mutations in the
///                             current structure
/// @param[out]   mutations_curr_struct_res - output - mutated residue positions
///                             in the current structure
/// @param[out]   mutations_curr_struct_aa - output - mutated amino acids
///                             in the current structure
/// @param[in]   startres - input - start residue of current structure
/// @param[in]   endres   - input - end residue of current structure
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_mut_arrays_curr_struct(
	FArray2D_int const & mutations_all_res,
	FArray2D_int const & mutations_all_aa,
	int & number_of_curr_struct,
	//	int & which_partner,
	int & p2_start,
	int & n_mutations_curr_complex,
	int & n_mutations_curr_struct,
	FArray1D_int & mutations_curr_struct_res,
	FArray1D_int & mutations_curr_struct_aa,
	int & startres,
	int & endres
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace param;

// locals
	int complex_pos;
	int complex_aa;
	int part_num;

	int mutcount = 0;
	bool p2_true  = false;
	//	if ( which_partner == 2 ) p2_true = true;
	//		if ( which_partner == 3 ) {
	for ( int i = 1; i <= n_mutations_curr_complex; ++i ) {
		complex_pos = mutations_all_res(number_of_curr_struct,i);
		complex_aa  = mutations_all_aa(number_of_curr_struct,i);
		if ( complex_pos >= startres && complex_pos <= endres ) {
			++mutcount;
			map_complex_num_to_part_num(complex_pos,part_num,p2_start,p2_true);
			mutations_curr_struct_res(mutcount) = part_num;
			mutations_curr_struct_aa(mutcount)  = complex_aa;
		}
	}
	//	} else if ( which_partner == 1 ) {
	//		for ( int i = 1; i <= n_mutations_curr_complex; ++i ) {
	//			complex_pos = mutations_all_res(number_of_curr_struct,i);
	//						std::cout << "complex_pos = " << complex_pos << std::endl; //dws debug 10-12-06
	//			complex_aa  = mutations_all_aa(number_of_curr_struct,i);
	//			if ( complex_pos < p2_start ) {
	//				++mutcount;
	//							std::cout << "complex_pos = " << complex_pos << std::endl; //dws debug 10-12-06
	//				map_complex_num_to_part_num(complex_pos,part_num,p2_start,p2_true);
	//					std::cout << "part_num = " << part_num << std::endl; //dws debug 10-13-06
	//				mutations_curr_struct_res(mutcount) = part_num;
	//				mutations_curr_struct_aa(mutcount)  = complex_aa;
	//					std::cout << "mutations_curr_struct_aa(245) = " << complex_aa << std::endl; //dws debug 10-16-06
	//					std::cout << "mutations_curr_struct_aa(30) = " << complex_aa << std::endl; //dws debug 10-16-06
	//			}
	//		}
	//	} else if ( which_partner == 2 ) {
	//		for ( int i = 1; i <= n_mutations_curr_complex; ++i ) {
	//			complex_pos = mutations_all_res(number_of_curr_struct,i);
	//					std::cout << "complex_pos = " << complex_pos << std::endl; //dws debug 10-12-06
	//			complex_aa  = mutations_all_aa(number_of_curr_struct,i);
	//			if ( complex_pos >= p2_start ) {
	//				//				complex_pos = complex_pos - p2_start + 1;
	//				++mutcount;
	//					std::cout << "complex_pos = " << complex_pos << std::endl; //dws debug 10-12-06
	//				map_complex_num_to_part_num(complex_pos,part_num,p2_start,p2_true);
	//				mutations_curr_struct_res(mutcount) = part_num;
	//				mutations_curr_struct_aa(mutcount)  = complex_aa;
	//						std::cout << "mutations_curr_struct_aa(507) = " << complex_aa << std::endl; //dws debug 10-16-06
	//			}
	//		}
	//	}
	//		std::cout << "n_mutations_curr_complex = " << n_mutations_curr_complex << std::endl; //dws debug 10-16-06
	n_mutations_curr_struct = mutcount;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_complex_ddg_table
///
/// @brief
/// creates output Table with Binding energy for the WT complex
/// and all mutant complexes
/// also list the different full atom contributions to the binding
/// energy for each complex
///
/// @detailed
///
/// @param[in] n_mut_structs       input - number of mutated structures
/// @param[in] mutations_all_res   input - residues numbers of all mutations
///                           in all structures
/// @param[in]   mutations_all_aa  - input - amino acid types of all mutations
///                           in all structures
///
/// @global_read
/// - binding energies and contributions in namespace dgbind_energies
///   in int_fullatom_energies.h
/// - file name intout stored in analyze_interface_ddg.h
///
/// @global_write - none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified UCSC 09/12/04
/////////////////////////////////////////////////////////////////////////////////
void
output_complex_ddg_table(
	int const n_mut_structs,
	FArray2Da_int mutations_all_res,
	FArray2Da_int mutations_all_aa,
	FArray1Da_int n_mut_per_struct
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace files_paths;
	using namespace int_fullatom_energies;
	using namespace param;

	mutations_all_res.dimension( n_mut_structs, MAX_MUT_PER_STRUCT );
	mutations_all_aa.dimension( n_mut_structs, MAX_MUT_PER_STRUCT );
	n_mut_per_struct.dimension( n_mut_structs );

  // locals
	int rosetta_resnum, pdb_resnum;
	int n_pmut;
	char wt_aa, mut_aa;
	char pdb_chain_local;
	std::string dg_type;
	std::string dg_neighbors;

	FArray1D_int ranked_list( n_mut_structs );

  //tk file name of pdb file in list without path
	std::string intoutlocal(intout);
	if ( truefalseoption("intout_by_pdb") ) { intoutlocal = output_file + ".inout"; }
	std::ofstream int_stream( ( score_path + intoutlocal ).c_str(),
	 std::ios_base::out|std::ios_base::app );
	if ( !int_stream ) {
		std::cout << "having trouble opening output file for interface" <<
		 " trying to open " << score_path << intout << std::endl;
		std::cout << "intout " << intout << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	n_pmut = 0; // initialize

//bk if alter_spec_format
//bk sort the results so that redesigns that favor mut_mut are at the top
	if ( alter_spec_format ) {

		FArray1D_float probmm( n_mut_structs );
		FArray1D_float probmw( n_mut_structs );
		FArray1D_float probwm( n_mut_structs );
		FArray1D_float score( n_mut_structs );
		float const WTenergy = dgbind_resenergy(1);
		float const RT = 0.6;
		n_pmut = n_mut_structs / 3;   //# of point mutants

		for ( int pmut = 1; pmut <= n_pmut; ++pmut ) {  // looping through point mutants
			int const mm = 3 * pmut - 1;   // mutant:mutant structure number
			int const mw = mm + 1;  // mutant:wild type structure number
			int const wm = mw + 1;  // wild type:mutant structure number

			float dGmm = dgbind_resenergy(mm) - WTenergy;
			float dGmw = dgbind_resenergy(mw) - WTenergy;
			float dGwm = dgbind_resenergy(wm) - WTenergy;
			float dGww = 0.0;

			if ( std::abs(dGmm) > 20.0 ) { dGmm = sign(dGmm) * 20.0; }  // cap at 20.0
			if ( std::abs(dGmw) > 20.0 ) { dGmw = sign(dGmw) * 20.0; }
			if ( std::abs(dGwm) > 20.0 ) { dGwm = sign(dGwm) * 20.0; }

			float partition_func = std::exp(-dGmm/RT) + std::exp(-dGmw/RT) +
				std::exp(-dGwm/RT) + std::exp(-dGww/RT);

			probmm(pmut) = std::exp(-dGmm/RT) / partition_func;
			probmw(pmut) = std::exp(-dGmw/RT) / partition_func;
			probwm(pmut) = std::exp(-dGwm/RT) / partition_func;
			score(pmut) = - ( probmm(pmut) - probmw(pmut) - probwm(pmut) );
			ranked_list(pmut) = pmut;
		}
		rank(n_pmut,score,ranked_list);
	}

	// create a ranked list with the most favorable binders first
	if ( pmut_scan ) {
		FArray1D_float score( n_mut_structs );
		for ( int pmut = 1; pmut <= n_mut_structs; ++pmut ) {
			score(pmut) = dgbind_resenergy(pmut+1); // position 1 is WT
			ranked_list(pmut) = pmut;
		}
		rank(n_mut_structs,score,ranked_list);
	}

	if ( single_protein ) {
		int_stream << std::endl;
		int_stream << "********************************************************" << std::endl;
		int_stream << "ENERGY FOR WT COMPLEX \"" << start_file <<
			"\" is " << dgbind_resenergy(1) << std::endl;
		int_stream << "********************************************************" << std::endl;
		int_stream << std::endl;
		int_stream << "CONTRIBUTIONS TO THE ENERGY FOR THE WT COMPLEX: " << std::endl;
		int_stream << std::endl;
	} else {
		int_stream << std::endl;
		int_stream << "********************************************************" << std::endl;
		int_stream << "BINDING ENERGY FOR WT COMPLEX \"" << start_file <<
			"\" is " << dgbind_resenergy(1) << std::endl;
		int_stream << "********************************************************" << std::endl;
		int_stream << std::endl;
		int_stream << "CONTRIBUTIONS TO THE BINDING ENERGY FOR THE WT COMPLEX: " << std::endl;
		int_stream << std::endl;
	}


	int_stream << "-----------------------------------------------" <<
	 "--------------------------------------------------------" <<
	 "----------------------------------------------------" << std::endl;
	int_stream << "            Eatr    Erep    Esol     Eaa    Edun  Eintra" <<
	 "   Ehbnd   Epair    Eref    Eh2o  Eh2ohb Eh2osol    score   D_SASA" <<
	 "  DSASA_POL  DSASA_APOL SASApack D_BBHB_UNS D_SCHB_UNS D_GU_TOT D_WGU_TOT" <<
	 " pdb_name" << std::endl;
	int_stream << "-----------------------------------------------" <<
	 "--------------------------------------------------------" <<
	 "----------------------------------------------------" << std::endl;

	if ( single_protein ) {
		dg_type = "DG       ";
	} else {
		dg_type = "DG_BIND  ";
	}

	int_stream << dg_type << ' ' <<
		F( 7, 1, dgbind_atrenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_repenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_solenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_probenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_dunenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_intraresenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_hbenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_pair_energy(1) ) << ' ' <<
		F( 7, 1, dgbind_unfenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_h2oenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_h2ohbenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_h2osolenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_resenergy(1) ) << ' ' <<
		F( 9, 1, D_sasa_gsolt(1) ) << ' ' <<
		F( 9, 1, D_sasa_polar(1) ) << ' ' <<
		F( 9, 1, D_sasa_apolar(1) ) << ' ' <<
		F( 9, 4, D_sasa_pack(1) ) << ' ' <<
		F( 9, 1, D_bbhb_uns(1) ) << ' ' <<
		F( 9, 1, D_schb_uns(1) ) << ' ' <<
		F( 9, 1, D_gu_tot(1) ) << ' ' <<
		F( 9, 1, D_Wgu_tot(1) ) << ' ' <<
		A( 14, output_file ) << std::endl; //glb was 20
	int_stream << "-----------------------------------------------" <<
		"---------------------------------------------------------" <<
		"-----------------------------------------------------" << std::endl;
	if ( ddg_bind_only )
	{
			int_stream.close();
			int_stream.clear();
			return;
	}

	if ( single_protein ) {
		int_stream << std::endl;
		int_stream << "***********************************" << std::endl;
		int_stream << "ENERGY FOR MUTANT COMPLEXES" << std::endl;
		int_stream << "***********************************" << std::endl;
		int_stream << std::endl;
		int_stream << "CONTRIBUTIONS TO THE ENERGY FOR THE MUT COMPLEXES: " << std::endl;
		int_stream << std::endl;
	} else {
		int_stream << std::endl;
		int_stream << "***********************************" << std::endl;
		int_stream << "BINDING ENERGY FOR MUTANT COMPLEXES" << std::endl;
		int_stream << "***********************************" << std::endl;
		int_stream << std::endl;
		int_stream << "CONTRIBUTIONS TO THE BINDING ENERGY FOR THE MUT COMPLEXES: " << std::endl;
		int_stream << std::endl;
	}
	int_stream << "           Eatr    Erep    Esol     Eaa    Edun  Eintra" <<
	 "   Ehbnd   Epair    Eref    Eh2o   Eh2ohb  Eh2osol  score" <<
   " DSASA_POL DSASA_APOL SASApack BBH_UNS SCH_UNS  GU_TOT WGU_TOT dGchn1  dGchn2" <<
	 "   mutations" << std::endl;

	if ( single_protein ) {
		dg_type = "DG      ";
	} else {
		dg_type = "DG_BIND ";
	}

	int_stream << dg_type  <<
		F( 7, 1, dgbind_atrenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_repenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_solenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_probenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_dunenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_intraresenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_hbenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_pair_energy(1) ) << ' ' <<
		F( 7, 1, dgbind_unfenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_h2oenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_h2ohbenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_h2osolenergy(1) ) << ' ' <<
		F( 7, 1, dgbind_resenergy(1) ) << ' ' <<
		F( 9, 1, D_sasa_polar(1) ) << ' ' <<
		F( 9, 1, D_sasa_apolar(1) ) << ' ' <<
		F( 9, 4, D_sasa_pack(1) ) << ' ' <<
		F( 7, 1, D_bbhb_uns(1) ) << ' ' <<
		F( 7, 1, D_schb_uns(1) ) << ' ' <<
		F( 7, 1, D_gu_tot(1) ) << ' ' <<
		F( 7, 1, D_Wgu_tot(1) ) << ' ' <<
		F( 7, 1, chainA_resenergy(1) ) << ' ' <<
		F( 7, 1, chainB_resenergy(1) ) <<
		"    WT" << std::endl;


//bk if using alter_spec_format then output the intrinsic free energy for
//bk each wt binding partner.
	if ( alter_spec_format ) {
		int_stream << "Partner1  " <<
			F( 7, 1, chainA_atrenergy(1) ) << ' ' <<
			F( 7, 1, chainA_repenergy(1) ) << ' ' <<
			F( 7, 1, chainA_solenergy(1) ) << ' ' <<
			F( 7, 1, chainA_probenergy(1) ) << ' ' <<
			F( 7, 1, chainA_dunenergy(1) ) << ' ' <<
			F( 7, 1, chainA_intraresenergy(1) ) << ' ' <<
			F( 7, 1, chainA_hbenergy(1) ) << ' ' <<
			F( 7, 1, chainA_pair_energy(1) ) << ' ' <<
			F( 7, 1, chainA_unfenergy(1) ) << ' ' <<
			F( 7, 1, chainA_h2oenergy(1) ) << ' ' <<
			F( 7, 1, chainA_h2ohbenergy(1) ) << ' ' <<
			F( 7, 1, chainA_h2osolenergy(1) ) << ' ' <<
			F( 7, 1, chainA_resenergy(1) ) << "    WT" << std::endl;
		int_stream << "Partner2  " <<
			F( 7, 1, chainB_atrenergy(1) ) << ' ' <<
			F( 7, 1, chainB_repenergy(1) ) << ' ' <<
			F( 7, 1, chainB_solenergy(1) ) << ' ' <<
			F( 7, 1, chainB_probenergy(1) ) << ' ' <<
			F( 7, 1, chainB_dunenergy(1) ) << ' ' <<
			F( 7, 1, chainB_intraresenergy(1) ) << ' ' <<
			F( 7, 1, chainB_hbenergy(1) ) << ' ' <<
			F( 7, 1, chainB_pair_energy(1) ) << ' ' <<
			F( 7, 1, chainB_unfenergy(1) ) << ' ' <<
			F( 7, 1, chainB_h2oenergy(1) ) << ' ' <<
			F( 7, 1, chainB_h2ohbenergy(1) ) << ' ' <<
			F( 7, 1, chainB_h2osolenergy(1) ) << ' ' <<
			F( 7, 1, chainB_resenergy(1) ) << "    WT" << std::endl;
	}


//bk if alter spec format output wt:wt, mut:wt, and wt:mut as a group.
//bk use_ranked list to output designs that perturb specificity the greatest first
	if ( alter_spec_format ) {
		for ( int rank = 1; rank <= n_pmut; ++rank ) {  // loop through sorted list
			int const pmut = ranked_list(rank);
			int const mm = 3 * pmut - 1;   // mutant:mutant structure number

			int_stream << " " << std::endl;
			int_stream << "point_mut_number = " << I( 3, pmut ) <<
				"  structure_number=" << I( 3, mm - 1) << " ," <<
				I( 3, mm ) << " ," << I( 3, mm+1 ) <<
				std::endl;
			int_stream << "                     Eatr    Erep    Esol     Eaa    Edun  Eintra" <<
				"   Ehbnd   Epair    Eref    Eh2o   Eh2ohb  Eh2osol  score" << std::endl;

//bk first output total energy of each chain - wt energy
//bk this is to make sure that the mutation are not significantly destabilizing the protein
			int_stream << "Gmut-Gwt Partner1 " <<
				F( 7, 1, chainA_atrenergy(mm)-chainA_atrenergy(1) ) << ' ' <<
				F( 7, 1, chainA_repenergy(mm)-chainA_repenergy(1) ) << ' ' <<
				F( 7, 1, chainA_solenergy(mm)-chainA_solenergy(1) ) << ' ' <<
				F( 7, 1, chainA_probenergy(mm)-chainA_probenergy(1) ) << ' ' <<
				F( 7, 1, chainA_dunenergy(mm)-chainA_dunenergy(1) ) << ' ' <<
				F( 7, 1, chainA_intraresenergy(mm)-chainA_intraresenergy(1) ) << ' ' <<
				F( 7, 1, chainA_hbenergy(mm)-chainA_hbenergy(1) ) << ' ' <<
				F( 7, 1, chainA_pair_energy(mm)-chainA_pair_energy(1) ) << ' ' <<
				F( 7, 1, chainA_unfenergy(mm)-chainA_unfenergy(1) ) << ' ' <<
				F( 7, 1, chainA_h2oenergy(mm)-chainA_h2oenergy(1) ) << ' ' <<
				F( 7, 1, chainA_h2ohbenergy(mm)-chainA_h2ohbenergy(1) ) << ' ' <<
				F( 7, 1, chainA_h2osolenergy(mm)-chainA_h2osolenergy(1) ) << ' ' <<
				F( 7, 1, chainA_resenergy(mm)-chainA_resenergy(1) ) << std::endl;
			int_stream << "Gmut-Gwt Partner2 " <<
				F( 7, 1, chainB_atrenergy(mm)-chainB_atrenergy(1) ) << ' ' <<
				F( 7, 1, chainB_repenergy(mm)-chainB_repenergy(1) ) << ' ' <<
				F( 7, 1, chainB_solenergy(mm)-chainB_solenergy(1) ) << ' ' <<
				F( 7, 1, chainB_probenergy(mm)-chainB_probenergy(1) ) << ' ' <<
				F( 7, 1, chainB_dunenergy(mm)-chainB_dunenergy(1) ) << ' ' <<
				F( 7, 1, chainB_intraresenergy(mm)-chainB_intraresenergy(1) ) << ' ' <<
				F( 7, 1, chainB_hbenergy(mm)-chainB_hbenergy(1) ) << ' ' <<
				F( 7, 1, chainB_pair_energy(mm)-chainB_pair_energy(1) ) << ' ' <<
				F( 7, 1, chainB_unfenergy(mm)-chainB_unfenergy(1) ) << ' ' <<
				F( 7, 1, chainB_h2oenergy(mm)-chainB_h2oenergy(1) ) << ' ' <<
				F( 7, 1, chainB_h2ohbenergy(mm)-chainB_h2ohbenergy(1) ) << ' ' <<
				F( 7, 1, chainB_h2osolenergy(mm)-chainB_h2osolenergy(1) ) << ' ' <<
				F( 7, 1, chainB_resenergy(mm)-chainB_resenergy(1) ) << std::endl;

//bk output binding energies (as ddG) for mut-mut, wt-mut and mut-wt
			std::string partners;
			for ( int i = 0; i <= 2; ++i ) {
				int str = mm + i;
				if ( i == 0 ) {
					partners = "MUT:MUT ";
				} else if ( i == 1 ) {
					partners = "MUT:WT  ";
				} else {
					partners = " WT:MUT ";
				}
				int_stream << "ddGbind   " << partners <<
					F( 7, 1, dgbind_atrenergy(str) - dgbind_atrenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_repenergy(str) - dgbind_repenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_solenergy(str) - dgbind_solenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_probenergy(str) - dgbind_probenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_dunenergy(str) - dgbind_dunenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_intraresenergy(str) - dgbind_intraresenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_hbenergy(str) - dgbind_hbenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_pair_energy(str) - dgbind_pair_energy(1) ) << ' ' <<
					F( 7, 1, dgbind_unfenergy(str) - dgbind_unfenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_h2oenergy(str) - dgbind_h2oenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_h2ohbenergy(str) - dgbind_h2ohbenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_h2osolenergy(str) - dgbind_h2osolenergy(1) ) << ' ' <<
					F( 7, 1, dgbind_resenergy(str) - dgbind_resenergy(1) ) << std::endl;
			}  // for loop, mut:mut, wt:mut, mut:wt

//bk write out mutations in each chain
			for ( int p = 1; p <= 2; ++p ) {   // loop through partners
				int str = mm + p;
				int nmut = n_mut_per_struct(str-1);
				int_stream << "Mutations Partner " << p << "  ";

				for ( int mut = 1; mut <= nmut; ++mut ) {
					rosetta_resnum = mutations_all_res(str-1,mut);
					map_1_ros_num_to_pdb_res_chain(rosetta_resnum,pdb_resnum,pdb_chain_local);
					res1_from_num(aan_orig(rosetta_resnum),wt_aa);
					res1_from_num(mutations_all_aa(str-1,mut),mut_aa);
					int_stream << wt_aa << pdb_resnum << mut_aa << " ";
				}
				int_stream << std::endl;
			} // partners


		} // for loop, point mutants that led to each redesign


	} else {   // if not alter spec
		for ( int ii = 2; ii <= n_mut_structs+1; ++ii ) { // ii=1 is wild-type

			int i = ii;
			if ( pmut_scan ) {
				i = ranked_list(ii-1) + 1; // output sorted result
			}

			rosetta_resnum = mutations_all_res(i-1,1);
			map_1_ros_num_to_pdb_res_chain(rosetta_resnum,pdb_resnum,pdb_chain_local);
			res1_from_num(aan_orig(rosetta_resnum),wt_aa);
			res1_from_num(mutations_all_aa(i-1,1),mut_aa);

			if ( single_protein ) {
				dg_type = "DG      ";
			} else {
				dg_type = "DG_BIND ";
			}
			int_stream << dg_type <<
				F( 7, 1, dgbind_atrenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_repenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_solenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_probenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_dunenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_intraresenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_hbenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_pair_energy(i) ) << ' ' <<
				F( 7, 1, dgbind_unfenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_h2oenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_h2ohbenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_h2osolenergy(i) ) << ' ' <<
				F( 7, 1, dgbind_resenergy(i) ) << ' ' <<
				F( 9, 1, D_sasa_polar(i) ) << ' ' <<
				F( 9, 1, D_sasa_apolar(i) ) << ' ' <<
				F( 9, 4, D_sasa_pack(i) ) << ' ' <<
				F( 7, 1, D_bbhb_uns(i) ) << ' ' <<
				F( 7, 1, D_schb_uns(i) ) << ' ' <<
				F( 7, 1, D_gu_tot(i) ) << ' ' <<
				F( 7, 1, D_Wgu_tot(i) ) << ' ' <<
				F( 7, 1, chainA_resenergy(i) ) << ' ' <<
				F( 7, 1, chainB_resenergy(i) ) << ' ';
				//I( 4, rosetta_resnum ) << ' ' <<
				//I( 4, pdb_resnum ) << "  " <<
				//pdb_chain_local << "   " << wt_aa << "   " << mut_aa << std::endl;
				//bk write out mutations in each chain
			int nmut = n_mut_per_struct(i-1);
			for ( int mut = 1; mut <= nmut; ++mut ) {
				rosetta_resnum = mutations_all_res(i-1,mut);
				map_1_ros_num_to_pdb_res_chain(rosetta_resnum,pdb_resnum,pdb_chain_local);
				res1_from_num(aan_orig(rosetta_resnum),wt_aa);
				res1_from_num(mutations_all_aa(i-1,mut),mut_aa);
				int_stream << I(4,pdb_resnum) << pdb_chain_local << " " <<
					wt_aa << ">" << mut_aa << ":";
			}
			int_stream << std::endl;

			if ( !alter_spec_format && chain_energies ) {
				int_stream << "Partner1  " <<
					F( 7, 1, chainA_atrenergy(i) ) << ' ' <<
					F( 7, 1, chainA_repenergy(i) ) << ' ' <<
					F( 7, 1, chainA_solenergy(i) ) << ' ' <<
					F( 7, 1, chainA_probenergy(i) ) << ' ' <<
					F( 7, 1, chainA_dunenergy(i) ) << ' ' <<
					F( 7, 1, chainA_intraresenergy(i) ) << ' ' <<
					F( 7, 1, chainA_hbenergy(i) ) << ' ' <<
					F( 7, 1, chainA_pair_energy(i) ) << ' ' <<
					F( 7, 1, chainA_unfenergy(i) ) << ' ' <<
					F( 7, 1, chainA_h2oenergy(i) ) << ' ' <<
					F( 7, 1, chainA_h2ohbenergy(i) ) << ' ' <<
					F( 7, 1, chainA_h2osolenergy(i) ) << ' ' <<
					F( 7, 1, chainA_resenergy(i) ) << "  " <<
					I( 4, rosetta_resnum ) << ' ' <<
					I( 4, pdb_resnum ) << "  " <<
					pdb_chain_local << "   " << wt_aa << "   " << mut_aa << std::endl;
				int_stream << "Partner2  " <<
					F( 7, 1, chainB_atrenergy(i) ) << ' ' <<
					F( 7, 1, chainB_repenergy(i) ) << ' ' <<
					F( 7, 1, chainB_solenergy(i) ) << ' ' <<
					F( 7, 1, chainB_probenergy(i) ) << ' ' <<
					F( 7, 1, chainB_dunenergy(i) ) << ' ' <<
					F( 7, 1, chainB_intraresenergy(i) ) << ' ' <<
					F( 7, 1, chainB_hbenergy(i) ) << ' ' <<
					F( 7, 1, chainB_pair_energy(i) ) << ' ' <<
					F( 7, 1, chainB_unfenergy(i) ) << ' ' <<
					F( 7, 1, chainB_h2oenergy(i) ) << ' ' <<
					F( 7, 1, chainB_h2ohbenergy(i) ) << ' ' <<
					F( 7, 1, chainB_h2osolenergy(i) ) << ' ' <<
					F( 7, 1, chainB_resenergy(i) ) <<
					I( 4, rosetta_resnum ) << ' ' <<
					I( 4, pdb_resnum ) << "  " <<
					pdb_chain_local << "   " << wt_aa << "   " << mut_aa << std::endl;
			}

		}   // for each structure
	}  // alter spec?

	if ( alter_spec_format ) {
		int_stream.close();
		int_stream.clear();
		return;
	}

	if ( single_protein ) {
	int_stream << std::endl;
	int_stream << "******************************************************" << std::endl;
	int_stream << "CHANGES IN ENERGY FOR MUTANTS"  << std::endl;
	int_stream << "******************************************************" << std::endl;
	int_stream << std::endl;
	int_stream << "CHANGES IN CONTRIBUTIONS TO THE ENERGY " <<
	 "FOR THE MUTANTS: " << std::endl;
	} else {
	int_stream << std::endl;
	int_stream << "******************************************************" << std::endl;
	int_stream << "CHANGES IN BINDING ENERGY FOR MUTANT COMPLEXES: MUT-WT" << std::endl;
	int_stream << "******************************************************" << std::endl;
	int_stream << std::endl;
	int_stream << "CHANGES IN CONTRIBUTIONS TO THE BINDING ENERGY " <<
	 "FOR THE MUT COMPLEXES: " << std::endl;
	}

	int_stream << std::endl;
	if(affin_incr) {
	int_stream << "            Eatr    Erep    Esol     Eaa    Edun  Eintra" <<
		"   Ehbnd   Epair    Eref    Eh2o   Eh2ohb  Eh2osol   score" <<
   "   SASA_POL SASA_APOL SASApack BBH_UNS SCH_UNS  GU_TOT WGU_TOT ddGchn1 ddGchn2" <<
		" neighbors mutations " << std::endl;
	} else {
	int_stream << "            Eatr    Erep    Esol     Eaa    Edun  Eintra" <<
		"   Ehbnd   Epair    Eref    Eh2o   Eh2ohb  Eh2osol   score" <<
   "   SASA_POL SASA_APOL SASApack BBH_UNS SCH_UNS  GU_TOT WGU_TOT ddGchn1 ddGchn2" <<
		"   mutations " << std::endl;
	}
	for ( int ii = 2; ii <= n_mut_structs+1; ++ii ) { // ii=1 is wild-type

		int i = ii;
		if ( pmut_scan ) {
			i = ranked_list(ii-1) + 1; // output sorted result
		}

		rosetta_resnum = mutations_all_res(i-1,1);
		map_1_ros_num_to_pdb_res_chain(rosetta_resnum,pdb_resnum,pdb_chain_local);
		res1_from_num(aan_orig(rosetta_resnum),wt_aa);
		res1_from_num(mutations_all_aa(i-1,1),mut_aa);


		if ( single_protein ) {
			dg_type = "DDG       ";
		} else {
			dg_type = "DDG_BIND  ";
		}

		int_stream << dg_type <<
			F( 7, 1, dgbind_atrenergy(i) - dgbind_atrenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_repenergy(i) - dgbind_repenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_solenergy(i) - dgbind_solenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_probenergy(i) - dgbind_probenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_dunenergy(i) - dgbind_dunenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_intraresenergy(i) - dgbind_intraresenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_hbenergy(i) - dgbind_hbenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_pair_energy(i) - dgbind_pair_energy(1) ) << ' ' <<
			F( 7, 1, dgbind_unfenergy(i) - dgbind_unfenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_h2oenergy(i) - dgbind_h2oenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_h2ohbenergy(i) - dgbind_h2ohbenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_h2osolenergy(i) - dgbind_h2osolenergy(1) ) << ' ' <<
			F( 7, 1, dgbind_resenergy(i) - dgbind_resenergy(1) ) << ' ' <<
			F( 9, 1, D_sasa_polar(i) - D_sasa_polar(1) ) << ' ' <<
			F( 9, 1, D_sasa_apolar(i) - D_sasa_apolar(1) ) << ' ' <<
			F( 9, 4, D_sasa_pack(i) - D_sasa_pack(1) ) << ' ' <<
			F( 7, 1, D_bbhb_uns(i) - D_bbhb_uns(1) ) << ' ' <<
			F( 7, 1, D_schb_uns(i) - D_schb_uns(1) ) << ' ' <<
			F( 7, 1, D_gu_tot(i) - D_gu_tot(1) ) << ' ' <<
			F( 7, 1, D_Wgu_tot(i) - D_Wgu_tot(1) ) << ' ' <<
			F( 7, 1, chainA_resenergy(i) - chainA_resenergy(1) ) << ' ' <<
			F( 7, 1, chainB_resenergy(i) - chainB_resenergy(1) ) << ' ' ;
		if(affin_incr){
			int_stream <<
				I( 9, number_of_neighbors_orig(rosetta_resnum) ) << ' ' ;
		}

			//I( 4, rosetta_resnum ) << ' ' <<
			//I( 4, pdb_resnum ) << "  " <<
			//db_chain_local << "   " << wt_aa << "   " << mut_aa << ' ' <<
			//I( 2, neighbors(rosetta_resnum) ) << std::endl;




		int nmut = n_mut_per_struct(i-1);
		for ( int mut = 1; mut <= nmut; ++mut ) {
			rosetta_resnum = mutations_all_res(i-1,mut);
			map_1_ros_num_to_pdb_res_chain(rosetta_resnum,pdb_resnum,pdb_chain_local);
			res1_from_num(aan_orig(rosetta_resnum),wt_aa);
			res1_from_num(mutations_all_aa(i-1,mut),mut_aa);
			int_stream << I(4,pdb_resnum) << pdb_chain_local << " " <<
				wt_aa << ">" << mut_aa << ":";
		}
		int_stream << std::endl;
	}

	int_stream.close();
	int_stream.clear();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_1dim_logical
///
/// @brief
/// sets complete bool arry to false
///
/// @detailed
///
/// @param  logical_array - [in/out]? -
/// @param[in] size   input - size of bool array
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_1dim_logical(
	FArray1Da_bool logical_array,
	int size
)
{
	logical_array.dimension( size );
	logical_array = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin convert_chain_char_to_chain_num
///
/// @brief
/// converts a character chain id into a unique number
///
/// @detailed
/// useful for defining a mutation by
/// "pdb residue XX in chain X"
///
/// @param[in] chain_char   input  - chain id character
/// @param[out] chain_num    output - chain id integer
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
convert_chain_char_to_chain_num(
	char const chain_char,
	int & chain_num
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace design;

//tk converts a character chain id into a unique number
//tk useful for defining a mutation by "pdb residue XX in chain X"

	chain_num = 999;

	if ( mut_list ) {
		for ( int i = 1; i <= MAX_CHAINNUMS; ++i ) {
			if ( chain_char == chain_name(i) ) {
				chain_num = i;
//			else if ( chain_char == '_' ) {
//				chain_num = 1;
			}
		}
		if ( chain_num == 999 ) {
			std::cout << "did not find pdb chain, stopped" << std::endl;
			std::cout << "chain character = " << chain_char << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	} else {
		for ( int i = 1; i <= MAX_CHAINNUMS; ++i ) {
			if ( chain_char == chain_name(i) ) chain_num = i;
		}
// safety check for odd chain ids
		if ( chain_num == 999  && !ala_scan ) {
			std::cout << "did not find pdb chain, stopped" << std::endl;
			std::cout << "chain character = " << chain_char << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin repack_wt_structure
///
/// @brief
/// Repack interface residues
///
/// @param[in]  part_num: which_partner 1,2 or 3 (3 refers to the complex)
/// @param[in]  p2_start: starting residue for partner 2
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Deanne Sammond
///
/// @last_modified  Brian Kuhlman 6/17/04
/////////////////////////////////////////////////////////////////////////////////

void
repack_wt_structure(
	int const which_partner,
	int const p2_start
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace design;
	using namespace misc;
	using namespace param;

	FArray1D_bool allow_repack( MAX_RES() );  // dummy
	bool make_output_file;
	bool include_current;
	bool include_extra;
	FArray2D_float extra_chi( MAX_CHI, MAX_RES() ); // dummy
	FArray2D_int extra_rot( MAX_CHI, MAX_RES() ); // dummy
	std::string packmode;

	RotamerOptions saved_rotamer_options = active_rotamer_options;

	//ja keep the rotamer set small in NA mode
	if ( dna_interface ) active_rotamer_options.extrachi_cutoff = 18;


	//yl create local pose and prepare the Epositions for misc
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	PackerTask Task( pose );

	use_design_matrix = true;
	design_matrix = false;

	//bk refresh neighbor lists
	FArray2D_bool neighborlist( MAX_RES(), MAX_RES() );
	FArray1D_int neighbors( MAX_RES() );

	make_neighbor_info( res, total_residue, full_coord, neighborlist, neighbors );

	pick_interface_residues_for_repacking( which_partner, p2_start,
	 neighborlist );

	settings_for_pack_rotamers( make_output_file, include_current, include_extra,
	 packmode );

	Task.set_task( packmode, make_output_file, allow_repack, include_current,
	 include_extra,extra_rot, extra_chi );

	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();

	std::cout << "Repacking the unbound structure..." << '\n';
	pack_rotamers( pose, Task );
	pose.copy_to_misc();

	// in case this changed (currently only in dna_interface mode), set it back
	active_rotamer_options = saved_rotamer_options;

	use_design_matrix = false;
	design_matrix = false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin pick_interface_residues_for_repacking
///
/// @brief
/// sets design matrix so that the packer repacks interface residues
///
/// @detailed
///
/// @param[in]  part_num: which_partner 1,2 or 3 (3 refers to the complex)
/// @param[in]  p2_start: starting residue for partner 2
///
/// @global_read
///
/// @global_write  design_matrix (used by the packer)
///
/// @remarks
///
/// @references
///
/// @authors Deanne Sammond 03/16/04
///
/// @last_modified Brian Kuhlman 6/17/04
/////////////////////////////////////////////////////////////////////////////////
void
pick_interface_residues_for_repacking(
	int const which_partner,
	int const p2_start,
	FArray2Da_bool const & neighborlist
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace misc;
	using namespace param;
	using namespace design;
//bk local
	int seqpos_complex; // rosetta residue number in the complex

//bk seqpos is the residue position within the partner currently being considered
//bk seqpos_complex is the position number for the residue in the complex

	for ( int seqpos = 1; seqpos <= total_residue; ++seqpos ) {
		if ( which_partner == 2 ) {
			seqpos_complex = seqpos + p2_start - 1;
		} else {
			seqpos_complex = seqpos;
		}
		if ( interface_residue(seqpos_complex) || single_protein ) {
			design_matrix(res(seqpos),seqpos) = true;
			for ( int i = 1; i <= total_residue; ++i ) {
				if ( neighborlist(seqpos,i) ) {
					design_matrix(res(i),i) = true;
				}
			}
		}
	}
}

//////////////////////////////////////////////////////////////////////
void
setup_minimize_interface(
	cst_set_ns::Cst_set & cst_complex,
	cst_set_ns::Cst_set & cst_partner1,
	cst_set_ns::Cst_set & cst_partner2,
	int const & p2_start
)
{
	using namespace analyze_interface_ddg_ns;

	set_pose_flag(true);
	pose_ns::Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );
	pose.refold_sidechains_from_chi();

  //fill cst set for complex and individual chains
	bool const just_use_bb_heavy_atoms( true );
	fill_cst_set( cst_complex, pose, just_use_bb_heavy_atoms );
	if ( !single_protein ) {
		pose_ns::Pose pose_partner1, pose_partner2;
		pose_partner1 = pose;
		pose_partner1.delete_segment( p2_start, pose.total_residue() );
		pose_partner2 = pose;
		pose_partner2.delete_segment( 1, p2_start - 1 );
		fill_cst_set( cst_partner1, pose_partner1, just_use_bb_heavy_atoms );
		fill_cst_set( cst_partner2, pose_partner2, just_use_bb_heavy_atoms );
	}
	pose.copy_to_misc();  // make sure complex is in misc
	set_pose_flag(false);
}

//////////////////////////////////////////////////////////////////////
void
remove_sidechain_constraints(
	int const & residue,
	cst_set_ns::Cst_set & cst_set
)
{
	using namespace param;
	for ( int atom = 5; atom <= MAX_ATOM(); ++atom ) {
		//std::pair<int,int> pair( residue, atom );
		kin::Atom_id pair( atom, residue );
		cst_set.remove_atompair_constraints_to_atom( pair );
	}
}

///////////////////////////////////////////////////////////////////////
void
minimize_interface(
  cst_set_ns::Cst_set & cst_set,
  int const nchains,
	int const partner2_begin,
	bool const first_min
)

{
	using namespace pose_ns;
	using numeric::xyzVector_float;
	using namespace design;
	using namespace analyze_interface_ddg_ns;
	using namespace param;

	// parameters
	set_pose_flag(true);
	int lj_ramp_cycles;
 	float initial_rep_weight, final_rep_weight;
	float initial_min_tolerance, final_min_tolerance;
	float initial_cst_weight, final_cst_weight;

	// ramping parameters for minimization
 	final_rep_weight = 1.0;
	final_cst_weight = cstW;
	final_min_tolerance = 0.0000001;

 	if ( first_min ) {   //first time this structure has seen the minimizer
 		lj_ramp_cycles = 8;
 		initial_rep_weight = 0.3;
		initial_cst_weight = 20.0;
 		initial_min_tolerance = 0.01;
 	} else {
 		lj_ramp_cycles = 2;
 		initial_rep_weight = final_rep_weight;
		initial_cst_weight = final_cst_weight;
 		initial_min_tolerance = final_min_tolerance;
 	}

	// setup minimizer
	pose_setup_minimizer( initial_min_tolerance );
	minimize_set_vary_omega( true );

	//minimize_ns::rb_translation_scale_factor = intafteroption("scale");

	// scorefxn
	Score_weight_map weight_map( score12 );

	// i/o
	Pose pose;
	fullatom_nonideal_initialized_pose_from_misc( pose );

	std::cout << "number of residues in structure being minimized "
						<< pose.total_residue() << std::endl;

	int const nres( pose.total_residue() );
	int nres1;
	nres1 = partner2_begin - 1;
	Pose start_pose;
	start_pose = pose;

	if ( use_cst ) {
		weight_map.set_weight( ATOMPAIR_CST,
			500.0 * initial_cst_weight / cst_set.count_atompair_constraints() );
		pose.set_constraints( cst_set );
	}

	// setup the fold_tree
	if ( nchains == 2 ) {
		int jump_pos1(0), jump_pos2(0);

		// find the closest c-alphas between the two partners, anchor the
		// jump at these residues
		float min_dis2(10000.0);
		for ( int i=1; i<= nres; ++i ) {
			xyzVector_float i_ca( &pose.Eposition()(1,2,i) );
			float i_min_dis2(10000.0);
			for ( int j=1; j<= nres; ++j ) {
				if ( i<= nres1 && j<=nres1 || i>nres1 && j>nres1 ) continue;
				xyzVector_float j_ca( &pose.Eposition()(1,2,j) );
				float const dis2( distance_squared( i_ca, j_ca ) );
				if ( dis2 < min_dis2 ) {
					min_dis2 = dis2;
					jump_pos1 = std::min(i,j);
					jump_pos2 = std::max(i,j);
				}
				i_min_dis2 = std::min( dis2, i_min_dis2 );
			}
		}

		std::cout << "Anchoring jump at residues: " << jump_pos1 << " and " <<
			jump_pos2 << " CA_distance= " << std::sqrt( min_dis2 ) << std::endl;
		// set the fold_tree
		pose.one_jump_tree( nres, jump_pos1, jump_pos2, nres1 );

	}  else if ( nchains == 1 ) {
		pose.simple_fold_tree( nres );
		std::cout << "Setting up fold tree for one chain" << std::endl;
	}  else {
		std::cout << "Minimize_interface only works with 1 or 2 chains"
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	bool jump ( false );
	if ( nchains == 2 && rb ) { jump = true; };

	if ( int_chi ) {
		pose.set_allow_chi_move( interface_residue ); // pass FArray
	} else {
		pose.set_allow_chi_move( chi ); // pass bool
	}
	if ( int_bb ) {
		pose.set_allow_bb_move( interface_residue ); // pass FArray
	} else {
		pose.set_allow_bb_move( bb ); // pass bool
	}
	pose.set_allow_jump_move( jump );


	// starting score
  pose.score( weight_map );
	std::cout << "start_score= " << pose.get_0D_score( SCORE ) <<
		" start_fa_rep= " << pose.get_0D_score( FA_REP ) <<
		" start_int_rep= " << sum_interface_fa_rep( pose ) << std::endl;

	// parameters for ramping up the repulsive score
	float const rep_delta
    ( std::exp( std::log( final_rep_weight / initial_rep_weight ) /
			 (lj_ramp_cycles-1) ) );
  float rep_weight = initial_rep_weight / rep_delta;

	// parameters for ramping down the atom pair constraints
		float const cst_delta
	( std::exp( std::log( final_cst_weight / initial_cst_weight ) /
								(lj_ramp_cycles-1) ) );
  float cst_weight = initial_cst_weight / cst_delta;

	// parameters for changing the minimization tolerance
		float const tol_delta
	( std::exp( std::log( final_min_tolerance / initial_min_tolerance ) /
								(lj_ramp_cycles-1) ) );
  float min_tolerance = initial_min_tolerance / tol_delta;

	for ( int k = 1; k <= lj_ramp_cycles; ++k ) {
		rep_weight *= rep_delta;
		weight_map.set_weight( FA_REP, rep_weight );
		cst_weight *= cst_delta;
		if (use_cst) {
			weight_map.set_weight( ATOMPAIR_CST,
				500.0 * cst_weight / cst_set.count_atompair_constraints() );
		}
		min_tolerance *= tol_delta;
		minimize_set_tolerance( min_tolerance );

		if ( int_bb ) {
			pose.set_allow_bb_move( interface_residue ); // pass FArray
		} else {
			pose.set_allow_bb_move( bb ); // pass bool
		}
		pose.set_allow_jump_move( jump );
		pose.main_minimize( weight_map, "dfpmin" );

		// do a round of just side chain minimization
		pose.set_allow_jump_move( false );
		pose.set_allow_bb_move( false ); // pass bool
		pose.main_minimize( weight_map, "dfpmin" );

		// diagnostics
		float const score( pose.get_0D_score( SCORE ));
 		float const fa_rep( pose.get_0D_score( FA_REP ));
		float const interface_rep( sum_interface_fa_rep( pose ) );
		float const ca_rms( CA_rmsd( pose, start_pose ));
		float const allatom_rms( allatom_rmsd( pose, start_pose ));
		float const cst_score( use_cst ? pose.get_0D_score( ATOMPAIR_CST ) : 0.0 );
		std::cout << "cst_W: " << F(5,2,cst_weight ) <<
      " rep_W: " << F(5,2,rep_weight ) <<
      " min_tol: " << G(9,4,min_tolerance) <<
			" score= " << F(9,3,score) <<
			" fa_rep= " << F(9,3,fa_rep) <<
			" int_rep= " << F(9,3,interface_rep) <<
			" cst_scr= " << F(9,3,cst_score) << // NOTE: includes weight
			" ca_rms= " << F(9,3,ca_rms ) <<
			" allatom_rms= " << F(9,3,allatom_rms) << std::endl;

	}

	//pose.dump_pdb("bk_min_test2.pdb");
	//assert( misc_in_sync( pose ));
	set_pose_flag(false);
	set_use_nblist( false );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_allow_mutation
///
/// @brief
/// sets 20x20 matrix that specifies which amino acid mutations are allowed based
/// on the premise that the mutation must increase buried hydrophobic SA
/// @detailed
///
/// @param[out] allow_mutation - FArray2D_bool
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Deanne Sammond 01/19/07
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
initialize_allow_mutation( FArray2D_bool & allow_mutation )
{
	bool allow_mutation_cstyle[ 20 ][ 20 ] =
		{
		// A,C,D,E,F,G,H,I,K,L,M,N,P,Q,R,S,T,V,W,Y
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 1 => Ala == WT and can be mutated to
			{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}, //row 2 => Cys == WT. We are not mutating from or to Cys
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 3 => Asp == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 4 => Glu == WT.
			{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1}, //row 5 => Phe == WT.
			{1,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 6 => Gly == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 7 => His == WT.
			{0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1}, //row 8 => Ile == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 9 => Lys == WT.
			{0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1}, //row 10=> Leu == WT.
			{0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,1,1}, //row 11=> Met == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 12=> Asn == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,0,1,1}, //row 13=> Pro == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 14=> Gln == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 15=> Arg == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 16=> Ser == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,1,1}, //row 17=> Thr == WT.
			{0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,0,1,1}, //row 18=> Val == WT.
			{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}, //row 19=> Trp == WT.
			{0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0} //row 20=> Tyr == WT.
		};
	for (int ii = 1; ii<=20; ++ii) {
		for(int jj = 1; jj<=20; ++jj) {
			allow_mutation(jj,ii) = allow_mutation_cstyle[ii-1][jj-1];//ii-1=row,jj-1=column
		}
	}
}
