// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 23366 $
//  $Date: 2008-06-20 11:27:42 -0700 (Fri, 20 Jun 2008) $
//  $Author: dekim $


// Rosetta Headers
#include "angles.h"
#include "angles_ns.h"
#include "aaproperties_pack.h"
#include "current_pose.h"
#include "elliptic_msd.h"
#include "param.h"
#include "param_aa.h"
#include "read_aaproperties.h"
#include "refold.h"
#include "refold_ns.h"
#include "termini.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/FArray.io.hh>

// Numeric Headers
#include <numeric/conversions.hh>
#include <numeric/trig.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>


////////////////////////////////////////////////////////////////////////////////
/// @begin angles_get_final_stub
///
/// @brief retrieve stub from end of last chain folded using angles.cc functions
///
/// @detailed
///
/// @param[out]   ni - out - N coords of residue i
/// @param[out]   ca - out - CA coords of residue i
/// @param[out]   cc - out - C coords of residue i
///
/// @global_read
///
/// @global_write
///
/// @remarks
///car last stub is the stub generated from the last residue refolded from
///car angles. It is the position of the next residue if the phi were zero.
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_get_final_stub(
	FArray1DB_float & ni,
	FArray1DB_float & ca,
	FArray1DB_float & cc
)
{
	using namespace angles::angles_stub;

	for ( int i = 1; i <= 3; ++i ) {
		ni(i) = xn(i);
		ca(i) = xca(i);
		cc(i) = xc(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_get_template
///
/// @brief retrieve template coordinates used by angles.cc functions
///
/// @detailed
///
/// @param[out]   Bxyz2 - out - coordinates of residue template
/// @param[out]   Bcentroids2 - out - coordinates of template centroids
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_get_template(
	FArray2DB_double & Bxyz2,
	FArray2DB_double & Bcentroids2
)
{
	using namespace angles::angles_standard_residue;

	// helper method to access residue templates

	for ( int i = 1; i <= 8; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			Bxyz2(j,i) = Bxyz(j,i);
		}
	}
//KMa phospho_ser 2006-01
	for ( int i = 1; i <= param::MAX_AA_PLUS(); ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			Bcentroids2(j,i) = Bcentroids(j,i);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_get_default_stub
///
/// @brief retrieve default stub residues used by angles.cc functions
///
/// @detailed
///
/// @param[out]   n - out - N coords of template
/// @param[out]   ca - out - CA coords of template
/// @param[out]   cc - out - C coords of template
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_get_default_stub(
	FArray1DB_double & n,
	FArray1DB_double & ca,
	FArray1DB_double & c
)
{
	using namespace angles::angles_standard_residue;

	for ( int i = 1; i <= 3; ++i ) {
		n(i) = Bxyz(i,1); // n
		ca(i) = Bxyz(i,2); // ca
		c(i) = Bxyz(i,3); // c
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_save_final_stub
///
/// @brief save stub from end of last chain folded using angles.cc functions
///
/// @detailed
///
/// @param[in]   n  - in - N coords of residue i
/// @param[in]   ca - in - CA coords of residue i
/// @param[in]   cc - in - C coords of residue i
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_save_final_stub(
	FArray1DB_double & n,
	FArray1DB_double & ca,
	FArray1DB_double & c
)
{
	using namespace angles::angles_stub;

	for ( int i = 1; i <= 3; ++i ) {
		xn(i) = n(i);
		xca(i) = ca(i);
		xc(i) = c(i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Dangles_coord_sys
///
/// @brief
///
/// @detailed
/// double precision version
/// creates a unique orientation matrix given three points
/// this matrix uniquely specifies the orientation
/// of the 3-d coordinate system given any three reference points.
/// (the matrix is of course arbitrary wrt to any fixed rotation)
/// however, the relative orientation of two such matrices can be compared
/// to determine the rotation needed to transform one 3-d coordinate system into
/// the other.
/// infact the transformation matrix taking coordinate system B into A
///  A*transpose(B)
/// where A and B are the matrices returned from this function.
///
/// @param  p1 - [in/out]? -
/// @param  p2 - [in/out]? -
/// @param  p3 - [in/out]? -
/// @param  mat - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
Dangles_coord_sys(
	FArray1DB_double & p1,
	FArray1DB_double & p2,
	FArray1DB_double & p3,
	FArray2DB_double & mat
)
{
	static FArray1D_double temp( 3, 0.0 );

	Dsubvec(p1,p2,temp);
	Dunitvec(temp,mat(1,1));
	Dsubvec(p3,p2,temp);
	Dcros(mat(1,1),temp,mat(1,3));
	Dunitvec(mat(1,3),mat(1,3));
	Dcros(mat(1,3),mat(1,1),mat(1,2));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Dangles_align_transform
///
/// @brief
///
/// @detailed
/// double precision version
/// inputs
/// P1, p2, p3 are three points in space
/// q1,q2,q3 are the same three points in space  but rotated and offset.
/// outputs
/// Mat is the matrix  needed to transform
/// P->Q namely M*P+OFF = Q
/// the offset is an arbitrary translation and is not returned.
/// it can be found from q1-Mat*P1 or q2-Mat*p2.
/// if for some reason P and Q aren't the same set of points then
/// the orientations are defined as follows
/// the plane p1,p2,p3 and q1,q2,q3 are aligned
/// and p1->p2 direction is aligned with q1->q2
///
/// @param  p1 - [in/out]? -
/// @param  p2 - [in/out]? -
/// @param  p3 - [in/out]? -
/// @param  q1 - [in/out]? -
/// @param  q2 - [in/out]? -
/// @param  q3 - [in/out]? -
/// @param  Mat - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Dangles_align_transform(
	FArray1Da_double p1,
	FArray1Da_double p2,
	FArray1Da_double p3,
	FArray1Da_double q1,
	FArray1Da_double q2,
	FArray1Da_double q3,
	FArray2DB_double & Mat
)
{
	p1.dimension( 3 );
	p2.dimension( 3 );
	p3.dimension( 3 );
	q1.dimension( 3 );
	q2.dimension( 3 );
	q3.dimension( 3 );

	static FArray2D_double MatQ( 3, 3 );
	static FArray2D_double MatP( 3, 3 );

	Dangles_coord_sys(p1,p2,p3,MatP);
	Dangles_coord_sys(q1,q2,q3,MatQ);
	mat_multiply_transpose3(MatQ,MatP,Mat);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin copy_coordinates
///
/// @brief copies backbone, centroid and fullatom coordinates
///
/// @detailed
///
/// @param[in]   begin - in - first residue to copy
/// @param[in]   end - in - last residue to copy
/// @param[in]   fullatom - in - copy full_coord too?
/// @param[in]   best_position - in - source backbone coords
/// @param[out]   position - out - destination backbone coords
/// @param[in]   best_centroid - in - source centroid coords
/// @param[out]   centroid - out - dest centroid coords
/// @param[in]   best_full_coord - in - source fullatom coords
/// @param[out]   full_coord - out - dest fullatom coords
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copy_coordinates(
	int begin,
	int end,
	bool fullatom,
	FArray3Da_float best_position,
	FArray3Da_float position,
	FArray2Da_float best_centroid,
	FArray2Da_float centroid,
	FArray3Da_float best_full_coord,
	FArray3Da_float full_coord
)
{
	using namespace aaproperties_pack;
	using namespace param;

	best_position.dimension( 3, MAX_POS, star );
	position.dimension( 3, MAX_POS, star );
	best_centroid.dimension( 3, star );
	centroid.dimension( 3, star );
	best_full_coord.dimension( 3, MAX_ATOM(), star );
	full_coord.dimension( 3, MAX_ATOM(), star );

//car copy begin to end
//car best is source, position is destination

	if ( end < begin ) return;

	copyXYZ((begin-1)*5+1,5*end,best_position,position);
	copyXYZ(begin,end,best_centroid,centroid);
	if ( !fullatom ) return;
	copyXYZ((begin-1)*MAX_ATOM()()+1,MAX_ATOM()()*end,best_full_coord,full_coord);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin transform_coordinates
///
/// @brief rotates and offsets backbone, centroid and fullatom coords
///
/// @detailed
///
/// @param[in]   begin - in - first residue to transform
/// @param[in]   end - in - last residue to transform
/// @param[in]   fullatom - in - transform full_coord too?
/// @param[in]   best_position - in - source backbone coords
/// @param[out]   position - out - destination backbone coords
/// @param[in]   best_centroid - in - source centroid coords
/// @param[out]   centroid - out - dest centroid coords
/// @param[in]   best_full_coord - in - source fullatom coords
/// @param[out]   full_coord - out - dest fullatom coords
/// @param[in]   mat - in - rotation matrix
/// @param[in]   off - in - translation vector
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
transform_coordinates(
	int begin,
	int end,
	bool fullatom,
	FArray3Da_float best_position,
	FArray3Da_float position,
	FArray2Da_float best_centroid,
	FArray2Da_float centroid,
	FArray3Da_float best_full_coord,
	FArray3Da_float full_coord,
	FArray2Da_double mat,
	FArray1Da_double off
)
{
	using namespace aaproperties_pack;
	using namespace param;

	if ( end < begin ) return;

	best_position.dimension( 3, MAX_POS, star );
	position.dimension( 3, MAX_POS, star );
	best_centroid.dimension( 3, star );
	centroid.dimension( 3, star );
	best_full_coord.dimension( 3, MAX_ATOM(), star );
	full_coord.dimension( 3, MAX_ATOM(), star );
	mat.dimension( 3, 3 );
	off.dimension( 3 );

//car transform begin to end by mat,off
//car best is source, position is destination

	transform(5*(begin-1)+1,5*end,mat,off,best_position,position);
	transform(begin,end,mat,off,best_centroid,centroid);

	if ( !fullatom ) return;

	transform(MAX_ATOM()()*(begin-1)+1,MAX_ATOM()()*end,mat,off,best_full_coord,
	 full_coord);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin mover2
///
/// @brief
///
/// @detailed
///
/// @param  c1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  t1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
mover2(
	FArray1DB_double const & c1,
	FArray2DB_double const & mat,
	FArray1DB_double const & vec,
	FArray1DB_double & t1
)
{
	for ( int i = 1; i <= 3; ++i ) {
		t1(i) = mat(1,i)*c1(1) + mat(2,i)*c1(2) + mat(3,i)*c1(3) + vec(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin transform
///
/// @brief rotate and translate 3D vectors
///
/// @detailed
///
/// @param[in]   first - in - first vector to transform
/// @param[in]   last - in - last vector to transform
/// @param[in]   mat - in - rotation matrix
/// @param[in]   off - in - translation vector
/// @param[in]   source - in - source vectors
/// @param[out]   dest - out - destination vectors
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// coordinates of atoms  1->first-1 are not changed in dest
/// source may equal destination
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
transform(
	int first,
	int last,
	FArray2Da_double mat,
	FArray1Da_double off,
	FArray2Da_float source, // note physical dim may be > last_res here
	FArray2Da_float dest // note physical dim may be > than last_res
)
{
	mat.dimension( 3, 3 );
	off.dimension( 3 );
	source.dimension( 3, last );
	dest.dimension( 3, last );

	static FArray1D_double vec( 3 );
	static FArray1D_double vec_b( 3 );

	for ( int i = first; i <= last; ++i ) {
		for ( int n = 1; n <= 3; ++n ) {
			vec(n) = source(n,i); //  weird but cannot do this in loop
		}
		mover2(vec,mat,off,vec_b);
		for ( int n = 1; n <= 3; ++n ) {
			dest(n,i) = vec_b(n);
		}
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin Axyz_initializer
///
/// @brief
///  this is the Engh and Huber amino acid with omega at trans
///
/// @detailed
///car Engh and Huber template above has two different N-CA bond lengths,
///car and two different CA-C bond lengths
///car This is a recalculated template, phi,psi=0, omega=180
///car n-CA = 1.457879
///car CA-C = 1.523259
///car CA-CB = 1.520835
///$$$          data Axyz/ 0.D0,           0.D0,          0.D0,
///$$$     $               0.766073585D0, -1.24037945D0,  0.D0,
///$$$     #              -0.152768347D0, -2.45530609D0,  0.D0,
///$$$     #               0.310370896D0, -3.59587589D0,  0.D0,
///$$$     #              -1.45754513D0,  -2.20404756D0,  0.D0,
///$$$     #              -2.44434638D0,  -3.27718932D0,  0.D0,
///$$$     #              -1.77181673D0,  -4.643945D0,    0.D0,
///$$$     #               1.70562197D0,  -1.28318022D0,  1.19513834D0/
///
/// @param  Axyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
Axyz_initializer( FArray2D_double & Axyz )
{
	//  this is the Engh and Huber amino acid with omega at trans
	FArray1Da_double Axyz1d( Axyz ); // 1D view
	int i = 0;
	Axyz1d( ++i ) = 1.22081327; // N
	Axyz1d( ++i ) = -0.77140212;
	Axyz1d( ++i ) = -1.55472410;
	Axyz1d( ++i ) = 0.18148500; // CA
	Axyz1d( ++i ) = -1.14521134;
	Axyz1d( ++i ) = -0.60297245;
	Axyz1d( ++i ) = -0.50331366; // C
	Axyz1d( ++i ) = 0.08584201;
	Axyz1d( ++i ) = -0.02342459;
	Axyz1d( ++i ) = -1.40970218; // O
	Axyz1d( ++i ) = -0.02694305;
	Axyz1d( ++i ) = 0.80188715;
	Axyz1d( ++i ) = -0.06336936; // N+1
	Axyz1d( ++i ) = 1.26154232;
	Axyz1d( ++i ) = -0.45902890;
	Axyz1d( ++i ) = -0.63328403; // CA+1
	Axyz1d( ++i ) = 2.51655889;
	Axyz1d( ++i ) = 0.01588227;
	Axyz1d( ++i ) = -1.74543917; // C+1
	Axyz1d( ++i ) = 2.26859903;
	Axyz1d( ++i ) = 1.02942538;
	Axyz1d( ++i ) = 0.76471037; // CB
	Axyz1d( ++i ) = -2.00329399;
	Axyz1d( ++i ) = 0.50899923;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_initialize_template
///
/// @brief
/// computes some coeffecients for rotations about the fixed bond directions
/// of the default alanine template.
/// @detailed
/// these coefficients are used to rapidly compute
/// the rotation matricies about the axes of the
/// omega, phi and psi angles.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///  static values are used by putres
///
/// @references
///
/// @authors car 10/13/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_initialize_template()
{
	using namespace angles::angles_template_coef;
	using namespace angles::angles_standard_residue;

	static FArray2D_double const Axyz( 3, 8, Axyz_initializer );
	static bool initialized = { false };

	if ( initialized ) return;
	initialized = true;
	std::cout << " Initializing refold" << std::endl;

// compute the coefficients needed by compute rotation matricies around
// the omega, phi and psi axes.  These get stashed in a namespace
	angles_precompute_rot(Axyz(1,3),Axyz(1,5),omega_coef);
	angles_precompute_rot(Axyz(1,2),Axyz(1,3),psi_coef);
	angles_precompute_rot(Axyz(1,1),Axyz(1,2),phi_coef);

// recenter the coordinate system so that 0,0,0 is the template CA
	for ( int i = 1; i <= 8; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			Bxyz(j,i) = Axyz(j,i) - Axyz(j,2); // origin is calpha
		}
	}
//KMa phospho_ser
	for ( int i = 1; i <= param::MAX_AA_PLUS(); ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			Bcentroids(j,i) -= Axyz(j,2); // origin is calpha
		}
	}
// recenter the coordinate system of just the CA and C' stub so that
// the origin (0,0,0) is the  C-nitrogen atom of the stub.
// Do not however recenter the stub C-nitrogen itself.
// this is done because now we can easily rotate (omega) the stub about the
// cnitrogen then offset it the c-nitrogen postion.
	for ( int i = 6; i <= 7; ++i ) {
		for ( int j = 1; j <= 3; ++j ) {
			Bxyz(j,i) = Axyz(j,i) - Axyz(j,5); // origin is c-nitrogen of stub
		}
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_precompute_rot
///
/// @brief
///
/// @detailed
/// precomputes coefficients later used by angles_get_rot_coef
/// these coeffiecients are used by angles_get_rot_coef to create a rotation matrix
/// about the a2->a1 bond axis.
/// input: points: a1,a2
/// output: matrix: coef
///cems
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  coef - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_precompute_rot(
	FArray1Da_double a1,
	FArray1Da_double a2,
	FArray2Da_double coef
)
{
	a1.dimension( 3 );
	a2.dimension( 3 );
	coef.dimension( 2, 9 );

//* find the matrix (mat)  that
//* rotate chi deg about an axis defined by a1-->a2
//* rotation is right-handed. That is, it is a
//* clockwise rotation when looking from a1 to a2
//* === double precision ===
//* chi in radians
	FArray2D_double mat( 3, 3 );
	FArray1D_double a( 3 );
//     *
//* get difference vector: a
	Dsubvec(a2,a1,a);
//* get length of vector A
	double const da = std::sqrt( a(1)*a(1) + a(2)*a(2) + a(3)*a(3) );
//* get phi, psi
	double phi = 0.0;
	double psi = 0.0;
	if ( da != 0.0 ) psi = std::acos( numeric::sin_cos_range( a(3) / da ) );
	if ( a(1) != 0.0 || a(2) != 0.0 ) phi = std::atan2(-a(1),a(2));

//* get matrix
	angles_precompute_mat(phi,psi,mat,coef);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_precompute_mat
///
/// @brief
///
/// @detailed
///
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  aa - [in/out]? -
/// @param  coef - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_precompute_mat(
	double & phi,
	double & psi,
	FArray2Da_double aa,
	FArray2Da_double coef
)
{
	aa.dimension( 3, 3 );
	coef.dimension( 2, 9 );

//* get matrix for Tanaka convention polar angles
//* CB  14-JAN-1991, function 3-JUN-1991
//* double precision  15-AUG-1991
//* >>> phi,psi,kappa in radians <<<

//	FArray2D_double ba( 3, 3 ); // Not used

//***
//cb Try to avoid underflows...
	if ( std::abs(phi) < 1.0E-4 ) phi = 0.0;
	if ( std::abs(psi) < 1.0E-4 ) psi = 0.0;
	double const kappa = -2.0; // just a dummy
	double const sf = std::sin(phi);
	double const cf = std::cos(phi);
	double const ss = std::sin(psi);
	double const cs = std::cos(psi);
	double const sk = std::sin(-kappa);
	double const ck = std::cos(-kappa);
//* now calculate the product matrix of the five rotation matrices
	coef(1,1) = cf*cf + sf*sf*cs*cs;
	coef(2,1) = sf*sf*ss*ss;

//	ba(1,1) = (cf*cf+sf*sf*cs*cs)*ck+sf*sf*ss*ss;
	aa(1,1) = coef(1,1)*ck + coef(2,1);

	coef(1,2) = ss*ss*sf*cf;
	coef(2,2) = -cs;
//	ba(1,2) = ss*ss*sf*cf*(ck-1.0)-cs*sk;
	aa(1,2) = coef(1,2)*(ck-1.0) + coef(2,2)*sk;

	coef(1,3) = sf*cs*ss;
	coef(2,3) = cf*ss;
//	ba(1,3) = sf*cs*ss*(ck-1.0)+cf*sk*ss;
	aa(1,3) = coef(1,3)*(ck-1.0) + coef(2,3)*sk;

	coef(1,4) = ss*ss*cf*sf;
	coef(2,4) = cs;
//	ba(2,1) = ss*ss*cf*sf*(ck-1.0)+cs*sk;
	aa(2,1) = coef(1,4)*(ck-1.0) + coef(2,4)*sk;

	coef(1,5) = sf*sf + cf*cf*cs*cs;
	coef(2,5) = cf*cf*ss*ss;
//	ba(2,2) = (sf*sf+cf*cf*cs*cs)*ck+cf*cf*ss*ss;
	aa(2,2) = coef(1,5)*ck + coef(2,5);

	coef(1,6) = cf*ss*cs;
	coef(2,6) = sf*ss;
//	ba(2,3) = cf*ss*cs*(1.0-ck)+sf*ss*sk;
	aa(2,3) = coef(1,6)*(1.0-ck) + coef(2,6)*sk;

	coef(1,7) = cs*ss*sf;
	coef(2,7) = -cf*ss;
//	ba(3,1) = cs*ss*sf*(ck-1.0)-cf*ss*sk;
	aa(3,1) = coef(1,7)*(ck-1.0) + coef(2,7)*sk;

	coef(1,8) = cf*ss*cs;
	coef(2,8) = -sf*ss;
//	ba(3,2) = cf*ss*cs*(1.0-ck)-sf*ss*sk;
	aa(3,2) = coef(1,8)*(1.0-ck) + coef(2,8)*sk;

	coef(1,9) = ss*ss;
	coef(2,9) = cs*cs;
//	ba(3,3) = ss*ss*ck+cs*cs;
	aa(3,3) = coef(1,9)*ck + coef(2,9);
//     * done.

}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_get_rot_coef
///
/// @brief
/// cems
/// creates a rotation matrix aa
/// the orientation of the axis of rotation is determined
/// by the coef matrix which has been previosly precomputed by angles_precompute_rot
///
/// @detailed
/// a fast method for creating rotations about previuosly specified rotation axes.
/// intended use is for generating residues with certian phi-psi-omega.  the
/// rotation axes are known ahead of time so we can short cut some of the
/// calculations needed by storing parts of the matrix in lookup tables that are precomputed once.
/// for rotations about arb axes use general but slower function getrot.cc
///
/// @param  coef - [in/out]? -
/// @param  kappa - [in/out]? - clockwise rotation angle about this axis
/// @param  aa - [in/out]? - returned rotation matrix
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_get_rot_coef(
	FArray2DB_double const & coef,
	double kappa,
	FArray2DB_double & aa
)
{

//* >>> phi,psi,kappa in radians <<<
//***
//cb Try to avoid underflows...

	if ( std::abs(kappa) < 1.0E-4 ) {
		aa = 0.0;
		aa(1,1) = aa(2,2) = aa(3,3) = 1.0;
		return;
	}

	double const sk = std::sin(-kappa);
	double const ck = std::cos(-kappa);
//* now calculate the product matrix of the five rotation matrices

	aa(1,1) = coef(1,1)*ck         + coef(2,1);
	aa(1,2) = coef(1,2)*(ck-1.0)   + coef(2,2)*sk;
	aa(1,3) = coef(1,3)*(ck-1.0)   + coef(2,3)*sk;
	aa(2,1) = coef(1,4)*(ck-1.0)   + coef(2,4)*sk;
	aa(2,2) = coef(1,5)*ck         + coef(2,5);
	aa(2,3) = coef(1,6)*(1.0-ck)   + coef(2,6)*sk;
	aa(3,1) = coef(1,7)*(ck-1.0)   + coef(2,7)*sk;
	aa(3,2) = coef(1,8)*(1.0-ck)   + coef(2,8)*sk;
	aa(3,3) = coef(1,9)*ck         + coef(2,9);
//* done.
}

////////////////////////////////////////////////////////////////////////////////
/// @begin getrot
///
/// @brief
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  chi - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
getrot(
	FArray1Da_double a1,
	FArray1Da_double a2,
	double const chi,
	FArray2Da_double mat,
	FArray1Da_double vec
)
{
	a1.dimension( 3 );
	a2.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

//* find the matrix (mat) and vector (vec) that
//* rotate chi deg about an axis defined by a1-->a2
//* rotation is right-handed. That is, it is a
//* clockwise rotation when looking from a1 to a2
//* === double precision ===
//* chi in radians

	if ( std::abs(chi) < 1.0E-4 ) {
		mat = 0.0;
		for ( int i = 1; i <= 3; ++i ) {
			mat(i,i) = 1.0;
			vec(i) = 0.0;
		}
		return;
	}

	FArray1D_double a( 3 );
	FArray1D_double c( 3 );

//* get difference vector: a
	Dsubvec(a2,a1,a);
//* get length of vector A
	double const da = std::sqrt( a(1)*a(1) + a(2)*a(2) + a(3)*a(3) );
//* get phi, psi
	double phi = 0.0;
	double psi = 0.0;
	if ( da != 0.0 ) psi = std::acos( numeric::sin_cos_range( a(3) / da ) );
	if ( a(1) != 0.0 || a(2) != 0.0 ) phi = std::atan2(-a(1),a(2));

//* get matrix
	getmat(phi,psi,chi,mat);
//* get vector, = -Mat*a1 + a1
	rotate(mat,a1,c);
	Dsubvec(a1,c,vec);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Sgetrot
///
/// @brief
///
/// @detailed
///
/// @param  a1 - [in/out]? -
/// @param  a2 - [in/out]? -
/// @param  chi - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Sgetrot(
	FArray1Da_float a1,
	FArray1Da_float a2,
	float const chi,
	FArray2Da_float mat,
	FArray1Da_float vec
)
{
	a1.dimension( 3 );
	a2.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

	if ( std::abs(chi) < 1.0E-4 ) {
		mat = 0.0f;
		for ( int i = 1; i <= 3; ++i ) {
			mat(i,i) = 1.0f;
			vec(i) = 0.0f;
		}
		return;
	}

	FArray1D_float a( 3 );
	FArray1D_float c( 3 );

	subvec(a2,a1,a);
	vector_normalize(a);
	float psi = std::acos( numeric::sin_cos_range( a(3) ) );
	float phi = ( ( a(1) != 0.0 || a(2) != 0.0 ) ? std::atan2(-a(1),a(2)) : 0.0f );

	Sgetmat(phi,psi,chi,mat);
	rotate(mat,a1,c);
	subvec(a1,c,vec);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin getmat
///
/// @brief
///
/// @detailed
///
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  kappa - [in/out]? -
/// @param  aa - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
getmat(
	double & phi,
	double & psi,
	double const kappa,
	FArray2Da_double aa
)
{
	aa.dimension( 3, 3 );

//* get matrix for Tanaka convention polar angles
//* CB  14-JAN-1991, function 3-JUN-1991
//* double precision  15-AUG-1991
//* >>> phi,psi,kappa in radians <<<

//***
//cb Try to avoid underflows...
	if ( std::abs(phi) < 1.0E-4 ) phi = 0.0;
	if ( std::abs(psi) < 1.0E-4 ) psi = 0.0;
	if ( std::abs(kappa) < 1.0E-4 ) {
		aa = 0.0;
		for ( int i = 1; i <= 3; ++i ) {
			aa(i,i) = 1.0;
		}
		return;
	}

	double const sf = std::sin(phi);
	double const cf = std::cos(phi);
	double const ss = std::sin(psi);
	double const cs = std::cos(psi);
	double const sk = std::sin(-kappa);
	double const ck = std::cos(-kappa);
//* now calculate the product matrix of the five rotation matrices
	aa(1,1) = (cf*cf+sf*sf*cs*cs)*ck+sf*sf*ss*ss;
	aa(1,2) = ss*ss*sf*cf*(ck-1.0)-cs*sk;
	aa(1,3) = sf*cs*ss*(ck-1.0)+cf*sk*ss;
	aa(2,1) = ss*ss*cf*sf*(ck-1.0)+cs*sk;
	aa(2,2) = (sf*sf+cf*cf*cs*cs)*ck+cf*cf*ss*ss;
	aa(2,3) = cf*ss*cs*(1.0-ck)+sf*ss*sk;
	aa(3,1) = cs*ss*sf*(ck-1.0)-cf*ss*sk;
	aa(3,2) = cf*ss*cs*(1.0-ck)-sf*ss*sk;
	aa(3,3) = ss*ss*ck+cs*cs;
//* done.
}

////////////////////////////////////////////////////////////////////////////////
/// @begin Sgetmat
///
/// @brief
///
/// @detailed
///
/// @param  phi - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  kappa - [in/out]? -
/// @param  aa - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Sgetmat(
	float & phi,
	float & psi,
	float const kappa,
	FArray2Da_float aa
)
{
	aa.dimension( 3, 3 );

//*                    single-precision copy of getmat
	if ( std::abs(phi) < 1.0E-4 ) phi = 0.0;
	if ( std::abs(psi) < 1.0E-4 ) psi = 0.0;
	if ( std::abs(kappa) < 1.0E-4 ) {
		aa = 0.0f;
		for ( int i = 1; i <= 3; ++i ) {
			aa(i,i) = 1.0f;
		}
		return;
	}

	float const sf = std::sin(phi);
	float const cf = std::cos(phi);
	float const ss = std::sin(psi);
	float const cs = std::cos(psi);
	float const sk = std::sin(-kappa);
	float const ck = std::cos(-kappa);
	aa(1,1) = (cf*cf+sf*sf*cs*cs)*ck+sf*sf*ss*ss;
	aa(1,2) = ss*ss*sf*cf*(ck-1.0f)-cs*sk;
	aa(1,3) = sf*cs*ss*(ck-1.0f)+cf*sk*ss;
	aa(2,1) = ss*ss*cf*sf*(ck-1.0f)+cs*sk;
	aa(2,2) = (sf*sf+cf*cf*cs*cs)*ck+cf*cf*ss*ss;
	aa(2,3) = cf*ss*cs*(1.0f-ck)+sf*ss*sk;
	aa(3,1) = cs*ss*sf*(ck-1.0f)-cf*ss*sk;
	aa(3,2) = cf*ss*cs*(1.0f-ck)-sf*ss*sk;
	aa(3,3) = ss*ss*ck+cs*cs;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_putres
///
/// @brief
///
/// @detailed
/// build a residue using phi,psi,omega and orients it onto a stub xn,xca,xc
/// Expects that the Bxyz array and the centroid have been copied into the xyz array.
/// by the calling function.  (probably should do that copy internally here.)
///
/// @param  restype - [in/out]? - residue type
/// @param  xn - [in/out]? -
/// @param  xca - [in/out]? -
/// @param  xc - [in/out]? -
/// @param  omega - [in/out]? -
/// @param  psi - [in/out]? -
/// @param  phi - [in/out]? -
/// @param  position - [in/out]? -
/// @param  centroid - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_putres(
	int restype, // residue type
	FArray1DB_double & xn,
	FArray1DB_double & xca,
	FArray1DB_double & xc,
	float omega,
	float psi,
	float phi,
	FArray2Da_float position,
	FArray1Da_float centroid
)
{
	using namespace angles::angles_standard_residue;
	using namespace angles::angles_template_coef;
	using namespace param;
	using numeric::conversions::radians;

	position.dimension( 3, MAX_POS );
	centroid.dimension( 3 );

//cb Modified to include optional H. 11-DEC-1994
//cb  Modified 21-JAN-1992 to do PRO properly. Don't apply ph.
//cems:  changed everything down to the last detail!  nov 30 1999 cems
//* angles are in degrees
//* move coordinates 'xyz' to template 'xn','xca','xc' and
//* rotate dihedrals.

//car local
	static FArray2D_double xyz( 3, 9 ); // template residue
	static FArray2D_double mat_b( 3, 3 );
	static FArray2D_double mat( 3, 3 );

// Torsion angles
	double const ph = static_cast< double >( phi );
	double const ps = static_cast< double >( psi );
	double const om = static_cast< double >( omega );

//car copy default residue into template
	for ( int i = 1, l = 0; i <= 8; ++i ) {
		xyz[ l ] = Bxyz[ l ]; ++l; // xyz(1,i) = Bxyz(1,i)
		xyz[ l ] = Bxyz[ l ]; ++l; // xyz(2,i) = Bxyz(2,i)
		xyz[ l ] = Bxyz[ l ]; ++l; // xyz(3,i) = Bxyz(3,i)
	}

	xyz(1,9) = Bcentroids(1,restype);
	xyz(2,9) = Bcentroids(2,restype);
	xyz(3,9) = Bcentroids(3,restype);

//* compute matrix required to orient the residue
//* get matrix to put N-CA on template N-CA, put ALA there
//* lineup N and CA with template atoms

	Dangles_align_transform(xyz(1,1),xyz(1,2),xyz(1,3),xn,xca,xc,mat_b);

//* we dont apply this just yet.  save mat_b for later.  we compute it now before we move the
//* c-carbon position (phi)


//** rotate omega
	double rtmp = radians( om - 180.0 );
	if ( rtmp != 0.0 ) {
		angles_get_rot_coef(omega_coef,rtmp,mat);
// move remainder of atoms to right of this position (ca, cc on the stub)

		mover(xyz(1,7),mat,xyz(1,5)); // offset is stub N atom position
		mover(xyz(1,6),mat,xyz(1,5));
	} else {

		for ( int j = 1; j <= 3; ++j ) {
			double const xyz_j5 = xyz(j,5);
			xyz(j,6) += xyz_j5; // offset is stub N atom position
			xyz(j,7) += xyz_j5;
		}

	}

//** rotate PSI from starting position = 180 -- r-handed around CA-C
//  subtract 180.0 if starting model has torsion angle of 180.0
//       rtmp = radians( ps - 180.0 );
	rtmp = radians( ps );

	angles_get_rot_coef( psi_coef, rtmp, mat );
// need to compute offset vector so that xyz(1,3) does not move after rotation
// actually we dont have to do this because xyz(1,2) is the origin


// move remainder of atoms to right of this position (the stub, O)
	for ( int i = 4; i <= 7; ++i ) {
		rotate_in_place( mat, xyz(1,i) );
	}


//* rotate PHI from starting position = 180.0 -- r-handed around N-CA
//  subtract 180.0 if starting model has torsion angle of 180.0
//       rtmp = radians( ph - 180.0 );
	rtmp = radians( ph );

	angles_get_rot_coef( phi_coef, rtmp, mat );
// need to compute offset vector so that xyz(1,2) does not move after rotation
// actually we dont need to do this because calpha is origin of coordinate system

	for ( int i = 3; i <= 9; ++i ) { // move centroid (9) too
		rotate_in_place(mat,xyz(1,i));
	}


// at this point we have a residue obeying the phi-psi-omega angles
// all that remains is to orient this in space so it overlays the
// stub.  We already computed this matrix, mat_b.
// need to compute offset vector so that xyz(1,2) lies on top of calpha stub
// again dont actually need to compute this because xyz(i,2) is origin.
// so offset is just xca

//	for ( int i = 1; i <= 3; ++i ) {
//		vec(i) = xca(i) - xyz(i,2);
//	}


	for ( int i = 1; i <= 9; ++i ) {
	 // move em all including the nitrogen and the centroid
		mover(xyz(1,i),mat_b,xca); // note mat_b was computed earlier.
	}


//* write atoms to array
	for ( int i = 1; i <= 3; ++i ) {
		position(i,1) = xyz(i,1);
		position(i,2) = xyz(i,2);
		position(i,4) = xyz(i,3);
		position(i,5) = xyz(i,4);
		position(i,3) = xyz(i,8);
		centroid(i)   = xyz(i,9);
	}
//* set up dummy atoms for next residue (next stub)
	xn(1) = xyz(1,5);
	xn(2) = xyz(2,5);
	xn(3) = xyz(3,5);
	xca(1) = xyz(1,6);
	xca(2) = xyz(2,6);
	xca(3) = xyz(3,6);
	xc(1) = xyz(1,7);
	xc(2) = xyz(2,7);
	xc(3) = xyz(3,7);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin Smover
///
/// @brief
///     identical to mover, but 100% single precision
///
/// @detailed
///
/// @param  c1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  t1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Smover(
	FArray1Da_float c1,
	FArray2Da_float mat,
	FArray1Da_float vec,
	FArray1Da_float t1
)
{
	c1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );
	t1.dimension( 3 );

	for ( int i = 1; i <= 3; ++i ) {
		t1(i) =
		 mat(1,i) * c1(1) +
		 mat(2,i) * c1(2) +
		 mat(3,i) * c1(3) + vec(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin Smove_in_place
///
/// @brief
///
/// @detailed
///
/// @param  c1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Smove_in_place(
	FArray1Da_float c1,
	FArray2Da_float mat,
	FArray1Da_float vec
)
{
	c1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );

// same as above, but move c1 in place

	FArray1D_double t1( 3 );

	t1(1) = c1(1);
	t1(2) = c1(2);
	t1(3) = c1(3);
	for ( int i = 1; i <= 3; ++i ) {
		c1(i) = mat(1,i)*t1(1) + mat(2,i)*t1(2) + mat(3,i)*t1(3) + vec(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin ssubmover
///
/// @brief
/// similar to Smover, but subtracts offset rather than adding it
///
/// @detailed
///
/// @param  c1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  t1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Ssubmover(
	FArray1Da_float c1,
	FArray2Da_float mat,
	FArray1Da_float vec,
	FArray1Da_float t1
)
{
	c1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );
	t1.dimension( 3 );

	for ( int i = 1; i <= 3; ++i ) {
		t1(i) =
		 mat(1,i) * c1(1) +
		 mat(2,i) * c1(2) +
		 mat(3,i) * c1(3) - vec(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin Stransmover
///
/// @brief
/// similar to Smover, but transposes matrix before applying it
///
/// @detailed
///
/// @param  c1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  t1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Stransmover(
	FArray1Da_float c1,
	FArray2Da_float mat,
	FArray1Da_float vec,
	FArray1Da_float t1
)
{
	c1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );
	t1.dimension( 3 );

	for ( int i = 1; i <= 3; ++i ) {
		t1(i) =
		 mat(i,1) * c1(1) +
		 mat(i,2) * c1(2) +
		 mat(i,3) * c1(3) + vec(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin Ssubtransmover
///
/// @brief
/// similar to Stransmover, but subtracts vector
///
/// @detailed
///
/// @param  c1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  t1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Ssubtransmover(
	FArray1Da_float c1,
	FArray2Da_float mat,
	FArray1Da_float vec,
	FArray1Da_float t1
)
{
	c1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );
	t1.dimension( 3 );

	for ( int i = 1; i <= 3; ++i ) {
		t1(i) =
		 mat(i,1) * c1(1) +
		 mat(i,2) * c1(2) +
		 mat(i,3) * c1(3) - vec(i);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin Suntransmover
///
/// @brief
/// similar to smover, but subtracts vector before applying matrix
///
/// @detailed
///
/// @param  c1 - [in/out]? -
/// @param  mat - [in/out]? -
/// @param  vec - [in/out]? -
/// @param  t1 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Suntransmover(
	FArray1Da_float c1,
	FArray2Da_float mat,
	FArray1Da_float vec,
	FArray1Da_float t1
)
{
	c1.dimension( 3 );
	mat.dimension( 3, 3 );
	vec.dimension( 3 );
	t1.dimension( 3 );


	FArray1D_double t( 3 );

	t(1) = c1(1) - vec(1);
	t(2) = c1(2) - vec(2);
	t(3) = c1(3) - vec(3);
	for ( int i = 1; i <= 3; ++i ) {
		t1(i) = mat(1,i)*t(1) + mat(2,i)*t(2) + mat(3,i)*t(3);
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin copyXYZ
///
/// @brief
///
/// @detailed
///
/// @param  first - [in/out]? -
/// @param  last - [in/out]? -
/// @param  source - [in/out]? -
/// @param  dest - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
copyXYZ(
	int first,
	int last,
	FArray2Da_float source,
	FArray2Da_float dest
)
{
	source.dimension( 3, last );
	dest.dimension( 3, last );

	for ( int i = first; i <= last; ++i ) {
		dest(1,i) = source(1,i);
		dest(2,i) = source(2,i);
		dest(3,i) = source(3,i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin DcopyXYZ
///
/// @brief
///
/// @detailed
///
/// @param  first - [in/out]? -
/// @param  last - [in/out]? -
/// @param  source - [in/out]? -
/// @param  dest - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
DcopyXYZ(
	int first,
	int last,
	FArray2Da_double source,
	FArray2Da_double dest
)
{
	source.dimension( 3, last );
	dest.dimension( 3, last );

	for ( int i = first; i <= last; ++i ) {
		dest(1,i) = source(1,i);
		dest(2,i) = source(2,i);
		dest(3,i) = source(3,i);
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_coord_sys
///
/// @brief
///
/// @detailed
/// creates a unique orientation  matrix given three points
/// this matrix uniquely specifies the orientation
/// of the 3-d coordinate system given any three reference points.
/// (the matrix is of course arbitrary wrt to any fixed rotation)
/// however, the relative orientation of two such matrices can be compared
/// to determine the rotation needed to transform one 3-d coordinate system into
/// the other.
/// infact the transformation matrix taking coordinate system B into A
///  A*transpose(B)
/// where A and B are the matrices returned from this function.
///
/// @param  p1 - [in/out]? -
/// @param  p2 - [in/out]? -
/// @param  p3 - [in/out]? -
/// @param  mat - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_coord_sys(
	FArray1Da_float p1,
	FArray1Da_float p2,
	FArray1Da_float p3,
	FArray2Da_float mat
)
{
	p1.dimension( 3 );
	p2.dimension( 3 );
	p3.dimension( 3 );
	mat.dimension( 3, 3 );

	subvec(p1,p2,mat(1,1));
	unitvec(mat(1,1),mat(1,1));
	subvec(p3,p2,mat(1,2));
	cros(mat(1,1),mat(1,2),mat(1,3));
//      mat(1,3) = mat(2,1)*mat(3,2)-mat(3,1)*mat(2,2);
//      mat(2,3) = mat(3,1)*mat(1,2)-mat(1,1)*mat(3,2);
//      mat(3,3) = mat(1,1)*mat(2,2)-mat(2,1)*mat(1,2);

	unitvec(mat(1,3),mat(1,3));
	cros(mat(1,3),mat(1,1),mat(1,2));

//      mat(1,2) = mat(2,3)*mat(3,1)-mat(3,3)*mat(2,1);
//      mat(2,2) = mat(3,3)*mat(1,1)-mat(1,3)*mat(3,1);
//      mat(3,2) = mat(1,3)*mat(2,1)-mat(2,3)*mat(1,1);


}

////////////////////////////////////////////////////////////////////////////////
/// @begin angles_align_transform
///
/// @brief
///
/// @detailed
/// P->Q namely M*P+OFF = Q
/// the offset is an arbitrary translation and is not returned.
/// it can be found from q1-Mat*P1 or q2-Mat*p2.
/// if for some reason P and Q aren't the same set of points then
/// the orientations are defined as follows
/// the plane p1,p2,p3 and q1,q2,q3 are aligned
/// and p1->p2 direction is aligned with q1->q2
///
/// @param[in]  p1 - P1, p2, p3 are three points in space
/// @param[in]  p2 -
/// @param[in]  p3 -
/// @param[in]  q1 - q1,q2,q3 are the same three points in space but rotated and offset.
/// @param[in]  q2 -
/// @param[in]  q3 -
/// @param[out]  Mat - Mat is the matrix needed to transform
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
angles_align_transform(
	FArray1Da_float p1,
	FArray1Da_float p2,
	FArray1Da_float p3,
	FArray1Da_float q1,
	FArray1Da_float q2,
	FArray1Da_float q3,
	FArray2DB_float & Mat
)
{
	p1.dimension( 3 );
	p2.dimension( 3 );
	p3.dimension( 3 );
	q1.dimension( 3 );
	q2.dimension( 3 );
	q3.dimension( 3 );

	FArray2D_float MatQ( 3, 3 );
	FArray2D_float MatP( 3, 3 );
	FArray1D_float off( 3 );

	angles_coord_sys(p1,p2,p3,MatP);
	angles_coord_sys(q1,q2,q3,MatQ);
	mat_multiply3(MatQ,MatP,Mat);
	debug_mat("QT *",Mat);
	transpose3(MatP);
	mat_multiply3(MatQ,MatP,Mat);
	debug_mat("QPT *",Mat);
	transpose3(Mat);
	rotate(Mat,q1,off);
	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( off(i)-p1(i) );
	} std::cout << std::endl;
	rotate(Mat,q2,off);
	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( off(i)-p2(i) );
	} std::cout << std::endl;
	rotate(Mat,q3,off);
	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( off(i)-p3(i) );
	} std::cout << std::endl;
	transpose3(MatQ);
	mat_multiply3(MatQ,MatP,Mat);
	debug_mat("QTPT *",Mat);
	transpose3(MatP);
	mat_multiply3(MatQ,MatP,Mat);
	debug_mat("QTP *",Mat);
	transpose3(MatQ);
	mat_multiply_transpose3(MatQ,MatP,Mat);
	debug_mat("QP TP*",Mat);
	transpose3(Mat);
	rotate(Mat,q1,off);
	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( off(i)-p1(i) );
	} std::cout << std::endl;
	rotate(Mat,q2,off);
	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( off(i)-p2(i) );
	} std::cout << std::endl;
	rotate(Mat,q3,off);
	for ( int i = 1; i <= 3; ++i ) {
		std::cout << SS( off(i)-p3(i) );
	} std::cout << std::endl;
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//    functions added to incorporate full atom coordinates
//    CARohl  5/11/00

////////////////////////////////////////////////////////////////////////////////
/// @begin build_fullcoord
///
/// @brief
///
/// @detailed
///car builds full coordinate array 'dest' from region first to last
///car bb coordinates taken from position, HN coords build from scratch,
///car sidechain coordinates taken from source and transformed
///car to be consistent withthe new backbone.
///car
///car note position and full atom coordinates used different atom numbering:
///car             1  2  3  4  5             -->      1  2  3  4  5
///car             NH CA CB C  CO            -->      NH CA C  CO CB
///
/// @param  position - [in/out]? - source of bb coord
/// @param  first - [in/out]? -
/// @param  last - [in/out]? -
/// @param  aa - [in/out]? - sequence of passed coordinates
/// @param  aav - [in/out]? - amino acid variant
/// @param  phi_first - [in/out]? -
/// @param  source - [in/out]? - source of sc coordinates
/// @param  dest - [in/out]? - output coordinates
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
build_fullcoord(
	FArray3DB_float const & position, // source of bb coord
	int first,
	int last,
	FArray1DB_int const & aa, // sequence of passed coordinates
	FArray1DB_int const & aav, // amino acid variant
	FArray1DB_bool const & nterm, // Nterminus
	FArray1DB_bool const & cterm, // Cterminus
	float const phi_first,
	FArray3DB_float const & source, // source of sc coordinates
	FArray3DB_float & dest // output coordinates
)
{
	using namespace aaproperties_pack;
	using namespace param;

	double phi8;
	FArray2D_double xyz( 3, 3 );
	FArray1D_double xn( 3 );
	FArray1D_double xca( 3 );
	FArray1D_double xc( 3 );
	FArray1D_double vec( 3 );
	FArray2D_double mat( 3, 3 );
	FArray1D_double off( 3 );

	for ( int i = first; i <= last; ++i ) {
// compute the matrix to rotate and align the
//  next (existing) residue onto last fragment stub

		phi8 = 0.0; // already rotated    // this is probably inefficient
		for ( int n = 1; n <= 3; ++n ) {
			xn(n) = position(n,1,i);
			xca(n) = position(n,2,i);
			xc(n) = position(n,4,i);
		}
		getrot(xn,xca,phi8,mat,vec);
		mover(xc,mat,vec);

		for ( int n = 1; n <= 3; ++n ) { // need only three atoms
			xyz(n,1) = source(n,1,i); // N
			xyz(n,2) = source(n,2,i); // CA
			xyz(n,3) = source(n,3,i); // C'
		}

		Dangles_align_transform(xyz(1,1),xyz(1,2),xyz(1,3),xn,xca,xc,mat);

// rotate the source_ca so that we can find the offset
// vector that sets source_ca.rotate() = position_ca
		rotate(mat,xyz(1,2),vec);
		for ( int j = 1; j <= 3; ++j ) {
			off(j) = xca(j) - vec(j);
		}

		transform(1,natoms(aa(i),aav(i)),mat,off,source(1,1,i),dest(1,1,i));

		for ( int n = 1; n <= 3; ++n ) {
//car must copy O  - position depends on psi
			dest(n,4,i) = position(n,5,i); // copy O
			if ( refold_check_current_pose() ) {
				// in case we changed bb bond geometry
				dest(n,1,i) = position(n,1,i); // copy N
				dest(n,3,i) = position(n,4,i); // copy C
			}
		}
	}

//car hn protons are not part of the sidechain but have been
//car transformed as if they are, rebuild them here
	put_nh(dest,first,last,aa,aav,nterm,cterm,phi_first);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin put_nh
///
/// @brief
///bk put_nh puts amide protons on the backbone nitrogens for residues start
///car to end
///
/// @detailed
///car NH of residue 1 is place 1A out along N-CA bond
///
/// @param  xyz - [in/out]? -
/// @param  first - [in/out]? -
/// @param  last - [in/out]? -
/// @param  aan - [in/out]? - specifies amino acid at each seqpos
/// @param  aa_variant - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
put_nh(
	FArray3DB_float & xyz,
	int first,
	int last,
	FArray1DB_int const & aan, // specifies amino acid at each seqpos
	FArray1DB_int const & aa_variant,
	FArray1DB_bool const & nterm,
	FArray1DB_bool const & cterm,
	float const phi_first
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

//car local
	static FArray1D_float nhcoord( 3 );
	static FArray2D_float Nterm_nhcoord( 3, 3 );
	static FArray2D_float Cterm_cocoord( 3, 2 );


	for ( int i = first; i <= last; ++i ) {
		int const aan_i = aan(i);
		int const aa_variant_i = aa_variant(i);
		if ( nterm(i) && termini_ns::use_N_terminus ){

#ifndef BOINC
			if ( !variant_type( aav_Nterm, aan_i, aa_variant_i ) ){
				std::cout << "res " << aan_i << " and var " << aa_variant_i <<
				" at position " << i << " is not a proper Nterm variant" << std::endl;
				assert( variant_type( aav_Nterm, aan_i, aa_variant_i ) );
			}
#endif

			build_N_terminus_Hs(xyz,aan_i,i,phi_first,Nterm_nhcoord);
				// this have the desired side effect of slide( num_of_HN -1) residues
			int num_of_HN = 3;
			if ( aan_i == aa_pro ) num_of_HN = 2;
			int HN1_pos = HNpos(aan_i,aa_variant_i);
			for ( int nh = 1; nh <= num_of_HN; ++nh ){
				for ( int k = 1, l = xyz.index(k, nh+HN1_pos-1, i); k <= 3; ++k, ++l ) {
					xyz[ l ] = Nterm_nhcoord(k,nh);
				}
			}

		} else if ( aan_i != aa_pro && ( is_protein(aan_i) || is_nonnatural(aan_i) ) ) {
			put_nh_residue(xyz,aan_i,i,phi_first,nhcoord);
			for ( int j = 1, l = xyz.index(j,HNpos(aan_i,aa_variant_i),i);
			 j <= 3; ++j, ++l ) {
				xyz[ l ] = nhcoord(j); // xyz(j,HNpos(aan_i,aa_variant_i),i)
			}
		}

		if ( cterm(i) && termini_ns::use_C_terminus ){

#ifndef BOINC
			if ( !variant_type( aav_Cterm, aan_i, aa_variant_i ) ){
				std::cout << "res " << aan_i << "and var " << aa_variant_i <<
				" at position " << i << " is not a proper Cterm variant" << std::endl;
				assert( variant_type( aav_Cterm, aan_i, aa_variant_i ) );
			}
#endif

			build_C_terminus_Os(xyz,i,0.0,Cterm_cocoord);
			// this have the desired side effect of slide( num_of_HN -1) residues
			int const O1_pos = LookupByName(aan_i, aa_variant_i, " O  ");
			int const OXT_pos= LookupByName(aan_i, aa_variant_i, " OXT");
			for ( int k = 1, l = xyz.index(k, O1_pos, i); k <= 3; ++k, ++l )
				xyz[ l ] = Cterm_cocoord(k,1);
			for ( int k = 1, l = xyz.index(k, OXT_pos, i); k <= 3; ++k, ++l )
				xyz[ l ] = Cterm_cocoord(k,2);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin put_nh_residue
///
/// @brief
///bk put_nh puts amide protons on the backbone nitrogens
///car NH of residue 1 is place 1A out along N-CA bond
///
/// @detailed
///
/// @param  xyz - [in/out]? - see comment below
/// @param  aan - [in/out]? - amino acid type
/// @param  aav - [in/out]? - amino acid variant
/// @param  residue - [in/out]? -
/// @param  nhcoord - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
put_nh_residue(
	FArray3DB_float const & xyz, // see comment below
	int const aan, // amino acid type
	int const residue,
	float const phi_first,
	FArray1DB_float & nhcoord
)
{
	using namespace aaproperties_pack;
	using namespace param_aa;
	using namespace refold_ns;

//bk  xyz is an array which specifies the positions of each atom
//bk  in the protein.
//bk  xyz(X,,) => xyz coordinates
//bk  xyz(,X,) => atom # within amino acid
//bk    atom  N CA C O
//bk       #  1  2 3 4
//bk  xyz(,,X) => residue number

//$$$      data template/  0.000, 0.000, 0.000,    // N (residue i)
//$$$     *                1.278, 0.685, 0.000,    // ca (residue i)
//$$$     *               -1.158, 0.655, 0.000,    // c (residue i-1)
//$$$     *                0.000,-1.000, 0.000 /   // amide H (residue i)

	if ( aan == aa_pro ) {
		std::cout << "STOP:: attempt to build HN for proline" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( residue == 1 ) {
//car should compute position based on phi; here we assume that residue 1
//car implies N-terminus when it could be just the beginning of a chain
//car section with known phi, and thus known position of the HN
		if ( phi_first == 0.0 ) {
			place_atom_linear(xyz(1,1,residue),xyz(1,2,residue),nhcoord,1.0);
		} else {
//car build C(0)

//car define initial bond vectors
			FArray2D_float X( 3, 3 );
			subvec(xyz(1,3,1),xyz(1,2,1),X(1,ps));
			subvec(xyz(1,2,1),xyz(1,1,1),X(1,ph));
//car build c(i-1)
			FArray2D_float M( 3, 3 );
			refold_coord_sys(X(1,ps),X(1,ph),M(1,1),M(1,2),M(1,3));
			FArray1D_float c_xyz( 3 );
			build_atom(M,xyz(1,1,1),cT(c,c2n),sT(c,c2n),phi_first,D(c,c2n),
			 c_xyz,X(1,om));
			get_HN_coords(c_xyz,xyz(1,1,1),xyz(1,2,1),nhcoord);
		}
		return;
	}

	if ( vec_dist2( xyz(1,3,residue-1), xyz(1,1,residue) ) > 4.0 ) {
		// chainbreak
		place_atom_linear(xyz(1,1,residue),xyz(1,2,residue),nhcoord,1.0);
	} else {
		//car should ensure chain continuity before using this function and otherwise
		//car compute HN position from N,CA,C and phi
		get_HN_coords(xyz(1,3,residue-1),xyz(1,1,residue),xyz(1,2,residue),nhcoord);
	}
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_HN_coords
///
/// @brief
///
/// @detailed
///   computes the location of the backbone nitrogen's hydrogen atom location.
///   rather than alinging a template to the existing structure a simple
///   kludge is used.  The proton extends from the N in the C-N-Calpha plane
///   and the direction bisects the C-N-Ca angle.  This means we can easily
///   compute the hydrogen position along a vector bisecting C-N and Ca-N
///
/// @param  Cxyz - [in/out]? -
/// @param  Nxyz - [in/out]? -
/// @param  CAxyz - [in/out]? -
/// @param  HNxyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_HN_coords(
	FArray1Da_float Cxyz,
	FArray1Da_float Nxyz,
	FArray1Da_float CAxyz,
	FArray1DB_float & HNxyz
)
{
	Cxyz.dimension( 3 );
	Nxyz.dimension( 3 );
	CAxyz.dimension( 3 );

//  If position cannot be determined (ie because some coordinates are not
//  valid C,N,CA coordinates) the HN is placed at the N coordinates

//car parameters
	float const bond_length = { 1.01 };

//car local
	static FArray1D_float xyz1( 3 );
	static FArray1D_float xyz2( 3 );

	float norm1 = 0.0;
	float norm2 = 0.0;
	float xyz_tmp;
	for ( int i = 1; i <= 3; ++i ) {
		xyz_tmp = Nxyz(i) - CAxyz(i);
		xyz1(i) = xyz_tmp;
		norm1 += xyz_tmp * xyz_tmp;
		xyz_tmp = Nxyz(i) - Cxyz(i);
		xyz2(i) = xyz_tmp;
		norm2 += xyz_tmp * xyz_tmp;
	}
	if ( norm1 > 0.001 ) { // check for division by zero
		norm1 = 1.0/std::sqrt(norm1);
	} else {
		norm1 = 0.0;
	}
	if ( norm2 > 0.001 ) { // check for division by zero
		norm2 = 1.0/std::sqrt(norm2);
	} else {
		norm2 = 0.0;
	}

	float norm3 = 0.0;
	float HNxyz_tmp;
	for ( int i = 1; i <= 3; ++i ) {
		HNxyz_tmp = xyz1(i)*norm1 + xyz2(i)*norm2; // vector bisecting c-n-ca
		HNxyz(i) = HNxyz_tmp;
		norm3 += HNxyz_tmp * HNxyz_tmp;
	}

	if ( norm3 > 0.001 ) {  // check for division by zero
		norm3 = bond_length/std::sqrt(norm3); //  set length to bond_length
	} else {
		norm3 = 0.0;
	}

	for ( int i = 1; i <= 3; ++i ) {
		HNxyz(i) = norm3*HNxyz(i) + Nxyz(i); // offset from nitrogen
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin place_atom_linear
///
/// @brief
///bk place atom on the line that connects atm1 and atm2 so that it is closest
///bk to atm1 with the distance being dis
///
/// @detailed
///
/// @param  atm1 - [in/out]? -
/// @param  atm2 - [in/out]? -
/// @param  atm_out - [in/out]? -
/// @param  dis - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
place_atom_linear(
	FArray1Da_float atm1,
	FArray1Da_float atm2,
	FArray1Da_float atm_out,
	float dis
)
{
	atm1.dimension( 3 );
	atm2.dimension( 3 );
	atm_out.dimension( 3 );

	FArray1D_float direction( 3 );
	float length2 = 0.0;
	float dir_tmp;
	for ( int i = 1; i <= 3; ++i ) {
		dir_tmp = atm1(i) - atm2(i);
		direction(i) = dir_tmp;
		length2 += dir_tmp * dir_tmp;
	}
	float normalize = 0.0;
	if ( length2 > 0.001 ) normalize = dis / std::sqrt(length2);

	for ( int i = 1; i <= 3; ++i ) {
		atm_out(i) = normalize * direction(i) + atm1(i);
	}
}
