// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.7 $
//  $Date: 2005/09/30 15:05:00 $
//  $Author: aroop $
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   rosetta++/antibody_modeling.h
///
/// @brief  Modules  to graft  in CDR  loops in  a putative  antibody  model.
///         CDR-H3 is modeled based on special antibody H3 specific  fragment
///         libraries. Other CDR  loops are  grafted  in  based  on pdb files
///         containing only loop regions obtained by Arvind Sivasubramanina's
///         antibody homology modeling protocol.
/// @author Aroop Sircar (aroop@jhu.edu)

#ifndef INCLUDED_antibody_modeling
#define INCLUDED_antibody_modeling

// Rosetta Headers
#include "loop_class.h"
#include "pose.h"

class Fragment;

/// @brief switches to antibody modeler or linker building depending on flag
///        read in initialize mode
void
antibody_modeling();

/// @brief wrapper to put in differnt cdrs into the template pose based on
///        command line options
void
antibody_modeling_assemble_cdrs();

/// @brief converts chothia numbering into rosetta sequential numbering
void
antibody_modeling_convert_to_sequential_res_from_chothia_res(
	const std::string & cdr_loop_name,
	char & pdb_chain_id,
	int & pdb_loop_begin,
	int & pdb_loop_end
);

/// @brief graft a particular CDR loop into the framework pose
void
antibody_modeling_graft_a_cdr(
  pose_ns::Pose & framework_pose,
	const std::string & cdr_loop_name
);

/// @brief wrapper script to build, cluster, refine putative H3's
void
antibody_modeling_model_H3(
  pose_ns::Pose & framework_pose
);

/// @brief wrapper script to build the apex of CDR H2
void
antibody_modeling_model_h2_apex(
  pose_ns::Pose & pose_in
);

/// @brief various options needed by antibody modeling mode
void
antibody_modeling_initialize_pose(
	pose_ns::Pose & start_pose,
	std::string const mode
);

/// @brief grab the top 10 H3 loops by cluster size and then the top
///        10 H3 loops by cluster score for final refinement.
void
antibody_modeling_grab_top_H3_models(
  int loop_begin,
	int loop_end,
	std::vector< Fragment > & h3_library
);

/// @brief refine the members of loop library h3_library
void
antibody_modeling_refine_H3(
	pose_ns::Pose & framework_pose,
	int loop_begin,
	int loop_end
);

/// @brief build linkers for Fv
void
antibody_modeling_build_Fv_linker();

/// @brief a score that rewards contact of a loop with the main body of a
///        protein
void
antibody_modeling_contact_score(
	float & antibody_linker_contact_score
);

/// @brief tries to predict if a given sequence will form a kinked/extended
///        base
bool
antibody_modeling_CDR_H3_prediction(
  const pose_ns::Pose & pose,
	bool & is_kinked,
	bool & is_extended,
	std::vector <char> & aa_1name,
	const std::string & insert_region
);

/// @brief insert "base" or "nter" fragments
void
antibody_modeling_insert_ter(
	pose_ns::Pose & framework_pose,
	const std::string & insert_region
);

/// @brief read H3_CTERM or H3_NTERM and keep pertinent fragments
void
antibody_modeling_read_H3_ter_fragment(
  pose_ns::Pose & pose_in,
	const std::string & insert_region
);

/// @brief reads in native and cdr definitions and stores in native_cdr
///        namespace
void
antibody_modeling_load_native();

/// @brief stores the sequential residue numbers of the framework
void
antibody_modeling_define_framework(
  pose_ns::Pose & pose_in
);

/// @brief compute the rms global for CDR-H3 after superimposition of heavy
///        chain framework
float
antibody_modeling_h3_rmsg();

/// @begin antibody_modeling_L_rms
float
antibody_modeling_L_rms();

/// @brief compute the rms global for base region of CDR-H3 after
///        superimposition
float
antibody_modeling_h3_base_rmsg();

/// @brief compute the rms global for non_base region of CDR-H3 after
///        superimposition
float
antibody_modeling_h3_non_base_rmsg();

/// @brief changes the current fold tree of a pose to one that encompasses all
///        the cdr loops
void
antibody_modeling_change_to_all_cdr_fold_tree(
	pose_ns::Pose & pose_in,
	pose_ns::Loops & loops
);

/// @begin antibody_modeling_store_unaligned_framework_info
void
antibody_modeling_store_unaligned_framework_info();

/// @brief graft a cdr by first aligning the stem regions of the cdr and then
///        pasting it in
void
antibody_modeling_graft_cdr_by_superimpostion(
	pose_ns::Pose & framework_pose,
	pose_ns::Pose & cdr_pose,
	int framework_loop_begin,
	int framework_loop_end,
	int cdr_loop_begin,
	int cdr_loop_end
);

/// @brief insert all h3 loops from the h3 library
void
antibody_modeling_insert_h3(
  pose_ns::Pose & pose_in,
	int loop_begin,
	int loop_end,
	std::vector< Fragment > & loop_library
);

/// @brief checks if a decoy in current misc passes the H3 filter
int
antibody_modeling_h3_match();

/// @brief computes the distance between atom N of residue (n+1) of H3 and atom
///        O of residue (n-2) of the H3
float
antibody_modeling_h3_NE1_n_plus_1_O_n_minus_2_distance();

/// @brief computes the number of required hydrogen bonds present in H3 of
///        decoy
int
antibody_modeling_h3_hydrogen_ladder();

/// @brief detects if the input hydrogen and the oxygen form a hydrogen bond
bool
antibody_modeling_is_Hbond(
	const int hydrogen_res,
	const int oxygen_res,
	const FArray3D_float & h3_loop_coord,
	std::vector <std::string> aa_name
);

/// @brief given an amino acid returns the sequential number of the H (N-H)
int
antibody_modeling_H_atom_number(
	const std::string & aa_3name
);

/// @brief for a kink matches rules for stem res( n-4, n-3, n-2, n-1)
///        for an extended matches base sp type( n-2, n-1, n, n+1)
int
antibody_modeling_h3_stem_match();

/// @brief orient the decoy based on heavy framework alignment
void
antibody_modeling_orient_to_native(
	pose_ns::Pose & pose
);

/// @brief wrapper to carry out L-H docking local refine type moves with
///        simultaneous loop relaxation
void
antibody_modeling_refineLH(
	pose_ns::Pose & pose_in
);

/// @brief add all cdrs to a loop class
void
antibody_modeling_add_all_cdrs(
	pose_ns::Pose & pose_in
);

/// @begin antibody_modeling_relax_cdrs
void
antibody_modeling_relax_cdrs(
	pose_ns::Pose & pose_in
);

/// @brief perturb light and heavy chain relative orientation while
///        simultaneously docking with antigen
void
antibody_modeling_snugdock(
	pose_ns::Pose & pose_in
);

/// @brief minimize trial for simultaneous dock of light & heavy chains as well
///        as the antigen
void
antibody_modeling_snugdock_minize_trial(
	pose_ns::Pose & docking_pose,
	const std::string & min_type,
	const pose_ns::Score_weight_map & wt_map,
	const float func_tol
);

/// @brief set the rigid body center for the docking jump specified
void
antibody_modeling_snugdock_set_rb_center(
  pose_ns::Pose & pose,
	int rb_jump_num
);

/// @brief multi monte carlo inter minimize of light and heavy chains as well
///        as minimize antigen rigid body position
void
antibody_modeling_snugdock_monte_carlo_minimize(
	pose_ns::Pose & docking_pose,
	const int cycles,
	const std::string & min_type,
	const pose_ns::Score_weight_map & wt_map,
	const float trans_magnitude,
	const float rot_magnitude,
	const float minimization_threshold,
	const float func_tol
);

/// @brief detect interface residues for the light-heavy interface and the
///        antibody-antigen interface
void
anitbody_modeling_snugdock_calc_interface(
	pose_ns::Pose & pose,
	FArray1DB_bool & int_res
);

/// @brief detects the sequential residue number of the light chain
void
antibody_modeling_detect_L_end(
	pose_ns::Pose & pose_in
);

/// @brief generates the fold tree for simultaneous light and heavy chain
///        minimization and antigen docking
void
antibody_modeling_set_snugdock_foldtree(
	pose_ns::Pose & pose_in
);

/// @brief updates pose with fold tree having two flexible jumps for 1.docking
///        VL-VH and 2.antigen. Also has fixed jumps for loops to be refined
void
antibody_modeling_snugdock_loop_tree(
  pose_ns::Pose & pose,
	const pose_ns::Loops & loops
);

/// @brief set up snugdock dual interface residue list
void
antibody_modeling_detect_snugdock_interf_res();

/// @brief load options for antibody snugdock mode
void
antibody_modeling_snugdock_options(
	pose_ns::Pose & pose_in
);

/// @brief enable/disable backbone,chi moves of loop and neighboring residues
void
antibody_modeling_loop_attributes(
	pose_ns::Pose & pose_in,
  pose_ns::Loops & loops,
  bool enable_bb_chi_move
);

#endif
