// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14716 $
//  $Date: 2007-05-03 12:28:06 -0700 (Thu, 03 May 2007) $
//  $Author: jiangl $

#ifndef INCLUDED_are_they_neighbors
#define INCLUDED_are_they_neighbors


// Rosetta Headers
#include "aaproperties_pack.h"
//#include "pack.h" -- need not be #inlcuded
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2Da.hh>

#include <iostream>

////////////////////////////////////////////////////////////////////////////////
void
atn_use_atom_initializer( FArray1D_int & use_atom );


////////////////////////////////////////////////////////////////////////////////
//\BEGIN are_they_neighbors
//
//\BRIEF
// Determines whether two residues are neighbors based on the distance
// between selected atoms.
//
//\FULL
// The distance between beta (alpha for gly) carbons for the two residues is
// determined and compared to a threshold that depends on both amino acid
// types.  The squared distance and a logical decision on whether the
// residues are neighbors is returned.
//
//\PARAM - aa1 - in
// Type of the first residue
//
//\PARAM - aa2 - in
// Type of the second residue
//
//\PARAM - coord1 - in
// The coordinate array of the first residue
//
//\PARAM - coord2 - in
// The coordinate array of the second residue
//
//\PARAM - dis2 - out
// The square of the distance between the selected atoms in the two residues.
//
//\PARAM - neighbor - out
// A bool return value - true if the residues are neighbors.
//
//\GLOBAL_READ
// Amino acid information is required, as is the interaction distance
// threshold (paircutoff).
//
//\GLOBAL_WRITE
// No global variables are altered.
//
//\NOTES
//
//\COMMENTERS
//
// jjh 8-19-03
//
//\END
////////////////////////////////////////////////////////////////////////////////
inline
void
are_they_neighbors(
	int const aa1,
	int const aa2,
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	float & dis2,
	bool & neighbor
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

	static FArray1D_int const use_atom( MAX_AA(), atn_use_atom_initializer );

	int const aav = { 1 }; //assume the aavariant is 1

	if( !is_ligand(aa1) && !is_ligand(aa2) ) {
		distance2_bk(coord1(1,use_atom(aa1)),coord2(1,use_atom(aa2)),dis2);
	} else if( is_ligand(aa1) && is_ligand(aa2) ){
		dis2=0;neighbor = false;//lin no interaction between/within ligand
														//lin so neighbor is false
		return;
	} else if( is_ligand(aa1) ){
		float dis2_tmp;
		dis2 = 10000.0;
		for( int ii=1,ie=nheavyatoms(aa1,aav); ii<=ie; ii++ ) {
			distance2_bk(coord1(1,ii),coord2(1,use_atom(aa2)),dis2_tmp);
			if( dis2 > dis2_tmp ) dis2= dis2_tmp;
		}
	} else {
		float dis2_tmp;
		dis2 = 10000.0;
		for( int ii=1,ie=nheavyatoms(aa2,aav); ii<=ie; ii++ ) {
			distance2_bk(coord2(1,ii),coord1(1,use_atom(aa1)),dis2_tmp);
			if( dis2 > dis2_tmp ) dis2= dis2_tmp;
		}
	}

	float const pc1 = paircutoff(aa1,aa2) + 1.0f;
	neighbor = ( dis2 <= pc1 * pc1 );
}


////////////////////////////////////////////////////////////////////////////////
//\BEGIN are_they_neighbors
//
//\BRIEF
// Determines whether two residues are neighbors based on the distance (ca and cb)
// between selected atoms.
//
//\FULL
// additionally, consider the direction of the ca->cb vector
//
//\NOTES
//
//\AUTHOR lin jiang
//
//
//\END
////////////////////////////////////////////////////////////////////////////////
inline
void
are_they_neighbors(
	int const aa1,
	int const aa2,
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	float const design_cut_1,
	float const design_cut_2,
	float const repack_cut_1,
	float const repack_cut_2,
	bool & design_neighbor,
	bool & repack_neighbor
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	coord1.dimension( 3, MAX_ATOM() );
 	coord2.dimension( 3, MAX_ATOM() );

	float ca_dis2, cb_dis2;

	static FArray1D_int const use_atom( MAX_AA(), atn_use_atom_initializer );

	int const aav = { 1 }; //assume the aavariant is 1
	int lig_id(1);

	if( !is_ligand(aa1) && !is_ligand(aa2) ) {

		distance2_bk(coord1(1,use_atom(aa1)),coord2(1,use_atom(aa2)),cb_dis2);
		if( is_protein(aa1) && is_protein(aa2) ) {
			distance2_bk(coord1(1,2),coord2(1,2),ca_dis2);
		} else {
			ca_dis2=cb_dis2;
		}
	} else if( is_ligand(aa1) && is_ligand(aa2) ){
		ca_dis2=0;cb_dis2=0;repack_neighbor = false; design_neighbor = false;
		return;
	} else if( is_ligand(aa1) ){
		float dis2_tmp;
		cb_dis2 = 10000.0;
		for( int ii=1,ie=nheavyatoms(aa1,aav); ii<=ie; ii++ ) {
			distance2_bk(coord1(1,ii),coord2(1,use_atom(aa2)),dis2_tmp);
			if( cb_dis2 > dis2_tmp || ii == 1 ) {
				cb_dis2 = dis2_tmp;
				lig_id = ii;
			}
		}
		if( is_protein(aa2) ) {
			distance2_bk(coord1(1,lig_id),coord2(1,2),ca_dis2);
		}else {
			ca_dis2=cb_dis2;
		}
	} else {
		float dis2_tmp;
		cb_dis2 = 10000.0;
		for( int ii=1,ie=nheavyatoms(aa2,aav); ii<=ie; ii++ ) {
			distance2_bk(coord1(1,use_atom(aa1)),coord2(1,ii),dis2_tmp);
			if( cb_dis2 > dis2_tmp || ii == 1 ) {
				cb_dis2= dis2_tmp;
				lig_id = ii;
			}
		}
		if( is_protein(aa1) ) {
			distance2_bk(coord1(1,2),coord2(1,lig_id),ca_dis2);
		}else {
			ca_dis2=cb_dis2;
		}
	}

	if( cb_dis2<design_cut_1*design_cut_1 || (cb_dis2<design_cut_2*design_cut_2 && cb_dis2<=ca_dis2 ) ) {
		design_neighbor = true;
		repack_neighbor = true;
	} else {
		design_neighbor = false;
		repack_neighbor = ( cb_dis2<repack_cut_1*repack_cut_1 || ( cb_dis2<repack_cut_2*repack_cut_2 && cb_dis2<=ca_dis2 ) );
	}
	//std::cout <<aa1<<" "<<aa2<<" "<<lig_id<<" "<<ca_dis2<<" "<<cb_dis2<<" "<<design_neighbor<<" "<<repack_neighbor<<std::endl;
}

////////////////////////////////////////////////////////////////////////////////
//\BEGIN are_they_neighbors
//
//\BRIEF
// Determines whether two residues are neighbors based on the distance (ca and cb)
// between selected atoms.
//
//\FULL
// additionally, consider the direction of the ca->cb vector
//
//\NOTES
//
//\AUTHOR lin jiang
//
//
//\END
////////////////////////////////////////////////////////////////////////////////
inline
void
are_they_neighbors(
	int const aa1,
	int const aa2,
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	float const dis_cut_1,
	float const dis_cut_2,
	float & ca_dis,
	float & cb_dis,
	bool & neighbor
)
{
	using namespace aaproperties_pack;
	using namespace param;

	coord1.dimension( 3, MAX_ATOM() );
	coord2.dimension( 3, MAX_ATOM() );

	float const dis_ca_cb_1={1.5};
	float const dis_ca_cb_2={0.5};

	static FArray1D_int const use_atom( MAX_AA(), atn_use_atom_initializer );

	distance_bk(coord1(1,use_atom(aa1)),coord2(1,use_atom(aa2)),cb_dis);
	distance_bk(coord1(1,2),coord2(1,2),ca_dis);

	neighbor = (cb_dis < dis_cut_1 && cb_dis <= ca_dis+dis_ca_cb_1 ) ||
		(cb_dis < dis_cut_2 && cb_dis <= ca_dis+dis_ca_cb_2 ) ;
}



#endif
