// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15540 $
//  $Date: 2007-06-19 15:37:31 -0700 (Tue, 19 Jun 2007) $
//  $Author: chu $

#ifndef INCLUDED_ClassAtom
#define INCLUDED_ClassAtom


////////////////////////////////////////////////////////////////////////////////////////////////////
// Class Atom
//
// Project: Rosetta
//
// Language: C++
//
// Author: Kristian Kaufmann, Vanderbilt University
////////////////////////////////////////////////////////////////////////////////////////////////////


// Rosetta Headers
#include "atom_chem.h"

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>

// C++ Headers
#include <string>
#include <vector>
#include <list>
#include <iostream>
#include <cmath>

// Forward
class BondSmallMolecule;

// Using
typedef BondSmallMolecule * BondSmallMoleculeAP;

////////////////////////////////////////////////////////////////////////////////////////////////////
// Class Atom : Atom class
////////////////////////////////////////////////////////////////////////////////////////////////////

class Atom {

private: // Types //////////////////////////////////////////////////////////////////////////////////


public: // Types ///////////////////////////////////////////////////////////////////////////////////


public: // Creation ////////////////////////////////////////////////////////////////////////////////


	//===============================================================================================
	// Default Constructor
	//===============================================================================================
	inline
    Atom()
	{
	element_= atom_chem::UNDEFINED_ELEMENT_TYPE;
	index_ = 9999;
	atomname_ ="UUUU";
    ros_atom_cent_ = 1; //CARBON
	ros_atom_type_ = -1;
	parentbond_=NULL;
	coordinates_.assign(0,0,0);
	is_virtual_=false;
	pdb_number_=9999;
	partialcharge_ = 9999.9;
	state_=0;
	ligandatomtype_=atom_chem::UNDEFINED_ATOM_TYPE;
       	occupancy_=1.0;
	bvalue_=0.0;
	}


public: // Creation ///////////////////////////////////////////////////////////////////////////////


	//===============================================================================================
	// Copy Constructor
	//===============================================================================================
	Atom( Atom & p){
		element_=p.get_element_type();
		index_=p.get_index();
		atomname_=p.get_atom_name();
		ros_atom_cent_ = p.get_ros_atom_cent();
		ros_atom_type_ = p.get_ros_atom_type();
		parentbond_ = NULL;
		coordinates_ = p.get_coordinates();
		is_virtual_ = p.Is_Virtual();
		pdb_number_ = p.get_pdb_number();
		partialcharge_ = p.get_partial_charge();
		state_=p.get_state();
		ligandatomtype_=p.get_ligand_atom_type();
		occupancy_=p.get_occupancy();
		bvalue_=p.get_bvalue();
	}

public: // Creation ////////////////////////////////////////////////////////////////////////////////


	//===============================================================================================
	// Destructor
	//===============================================================================================
	~Atom()
	{
	}


public: // Assignment /////////////////////////////////////////////////////////////////////////////


	//===============================================================================================
	// Copy Assignment
	//===============================================================================================
	Atom &
	operator=( Atom & p){
		if (this != &p){
			element_=p.get_element_type();
			index_=p.get_index();
			atomname_=p.get_atom_name();
			ros_atom_cent_=p.get_ros_atom_cent();
			ros_atom_type_=p.get_ros_atom_type();
			parentbond_=NULL;
			coordinates_=p.get_coordinates();
			is_virtual_=p.Is_Virtual();
			pdb_number_=p.get_pdb_number();
			partialcharge_=p.get_partial_charge();
			occupancy_=p.get_occupancy();
			bvalue_=p.get_bvalue();
			return *this;
		}
		else{
			std::cerr << "tried to assign atom to itself!!" << std::endl;
			return *this;
		}

	}


protected: // Assignment ///////////////////////////////////////////////////////////////////////////



public: // Inspectors //////////////////////////////////////////////////////////////////////////////

	inline
	std::string
	get_atom_name(){
		return atomname_;
	}

	inline
	int
	get_ros_atom_cent(
	)const
	{
		return ros_atom_cent_;
	}

	inline
	int
	get_ros_atom_type(
	)const
	{
		return ros_atom_type_;
	}

	inline
	atom_chem::LigandAtomType
	get_ligand_atom_type(
	)const
	{
		return ligandatomtype_;
	}

	inline
	atom_chem::ElementType
	get_element_type(
	)const
	{
		return element_;
	}

	inline
	int
	get_index(
	)const
	{
		return index_;
	}

	inline
	float
	get_partial_charge(
	)const
	{
		return partialcharge_;
	}

	inline
	numeric::xyzVector_double
	get_coordinates(
	)const
	{
		return coordinates_;
	}

	inline
	int
	get_pdb_number(
	)const
	{
		return pdb_number_;
	}

	inline
	size_t
	get_state(
	)const
	{
		return state_;
	}

	inline
	bool
	Is_Virtual(
	)const
	{
		return is_virtual_;
	}

	inline
	bool
	Is_Root_Atom(
	)const
	{
		return (parentbond_==NULL);
	}

	//lin
	inline
	float
	get_occupancy( ){
	  return occupancy_ ;
	}

	inline
	float
	get_bvalue( ){
	  return bvalue_;
	}

public: // Modifiers ///////////////////////////////////////////////////////////////////////////////
	inline
	void
	set_atom_name(
		std::string & i
	){ atomname_=i; return;}

    inline
	void
	set_ros_atom_cent(
		int i
	){ ros_atom_cent_=i; return;}

	inline
	void
	set_ros_atom_type(
		int i
	){ ros_atom_type_=i; return;}

	inline
	void
	set_ligand_atom_type(
		atom_chem::LigandAtomType i
	){ ligandatomtype_=i; return;}

	inline
	void
	set_partial_charge(
		float i
	){ partialcharge_=float( i ); return;}

	inline
	void
	set_element(
		atom_chem::ElementType X
	){element_ = atom_chem::ElementType(X); return;}

	inline
	void
	set_coordinates(
		numeric::xyzVector_double replace
	){ coordinates_=replace; increment_state();return ;}

	inline
	void
	set_coordinates(
		float x,
		float y,
		float z
	){ coordinates_.assign(x,y,z); increment_state(); return;}

	inline
	void
	set_coordinates(
		const numeric::xyzVector_double &vector_A,
		const numeric::xyzVector_double &vector_B,
		const numeric::xyzVector_double &vector_C,
		const double dist_XA,
		const double angle_XAB,
		const double dihedral_XABC
	){

		numeric::xyzVector_double a = ( vector_A - vector_B);
		numeric::xyzVector_double b = ( vector_B - vector_C);
		numeric::xyzVector_double c = cross_product( a, b);
		numeric::xyzVector_double d = cross_product( a, c);
		a.normalize();
		b.normalize();
		c.normalize();
		d.normalize();
		numeric::xyzVector_double x = ( a * cos(
			numeric::constants::d::pi - angle_XAB)
                     - c * sin( numeric::constants::d::pi - angle_XAB)
                     	* sin( dihedral_XABC)
                     + d * sin( numeric::constants::d::pi - angle_XAB)
                     	* cos( dihedral_XABC)) * dist_XA;
		coordinates_=vector_A + x;
		increment_state();
		return;
	  }


	inline
	void
	set_pdb_number(
		int i
	){pdb_number_ = int ( i ); return;}

	inline
	void
	set_index(
		int i
	){index_=i;return;}

	inline
	void
	set_virtual_false(
	){ is_virtual_=false; return;}

	inline
	void
	set_virtual_true(
	){is_virtual_=true; return;}

	//lin
	inline
	void
	set_occupancy( float value ){
	  occupancy_ = value; return;
	}

	inline
	void
	set_bvalue( float value ){
	  bvalue_ = value; return;
	}

protected: // Functions ////////////////////////////////////////////////////////////////////////////

    inline
	void
	delete_bond(
		BondSmallMolecule * const & bond
	){
		size_t i =0;
		bool bond_deleted;
		bond_deleted=false;
		if (parentbond_==bond){
			parentbond_=NULL;
			bond_deleted=true;
		}
		if ( !(childbonds_.empty()) && !bond_deleted ){
			while((!bond_deleted) && (i<childbonds_.size())){
				if( childbonds_[i]==bond) {
					childbonds_[i]=NULL;
					bond_deleted=true;
				}
				i++;
			};
		}else if(!bond_deleted){
		std::cerr << " ERROR: Bond vector empty!";
		std::cerr << " Skipping delete action." << std::endl;
		}
	return;
	}// This function should only be called by a bond object.

	inline
	int
	add_parentbond(
		const BondSmallMoleculeAP & bond
	){	if ( parentbond_==NULL && index_==9999) {parentbond_=bond; return 0;}
		else{ //std::cerr << "Tried to construct bond with two parents";
			//std::cerr << std::endl;
			 return -1;
		}
	}

	inline
	void
	add_childbond(
		const BondSmallMoleculeAP & bond
	){ childbonds_.push_back( bond); return;} // This function should only be called by a bond object.

	inline
	void
	get_parentbond(
		BondSmallMoleculeAP & bond
	){ bond=parentbond_;return;}

	inline
	void
	get_childbonds(
		std::list<BondSmallMoleculeAP > & bonds
	){
		bonds.clear();
		for(std::vector< BondSmallMoleculeAP >::iterator current_bond=childbonds_.begin(); current_bond!=childbonds_.end(); current_bond++){
			bonds.push_back((*current_bond));
		}
		return;
	}

	inline
	void
	rotate_atom(
		numeric::xyzMatrix<double> & r_matrix, // rotation matrix
		numeric::xyzVector<double> & origin // origin offset
	){
		coordinates_=((r_matrix*(coordinates_-origin))+origin);
		increment_state();
		return;
	}
private: // Functions /////////////////////////////////////////////////////////

	inline
	void
	increment_state(
	){ state_++;return;}

private: // Data ///////////////////////////////////////////////////////////////////////////////////
	atom_chem::ElementType element_; // Element of atom
	int ros_atom_cent_; // Rosetta centroid type
	int ros_atom_type_; // Rosetta atom type
	std::string atomname_; // pdb name.
	BondSmallMoleculeAP parentbond_; //this is a non-owning pointer to the parent atom
	std::vector< BondSmallMoleculeAP > childbonds_; // These are non-owning pointers to child atoms.
	numeric::xyzVector_double coordinates_;
	bool is_virtual_;
	int index_;
	float partialcharge_;
	int pdb_number_;
	size_t state_; 	// Used to records changes in coordinates.
			// Should always be incremented in functions
			// changing coordinates. See LigandAtomPairEnergy for
			// Example of usage.
	float occupancy_;
	float bvalue_;
	atom_chem::LigandAtomType ligandatomtype_;



	friend class BondSmallMolecule;
	friend class Ligand;

}; // Class Atom
////////////////////////////////////////////////////////////////////////////////////////////////////


#endif
