// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   ccd_functions.hh
/// @brief  Various cyclic coordinate descent functions.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)

#ifndef INCLUDED_epigraft_design_ccd_functions_HH_
#define INCLUDED_epigraft_design_ccd_functions_HH_

// package headers
#include <design_types.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1Da.fwd.hh>
#include <ObjexxFCL/FArray2D.fwd.hh>
#include <ObjexxFCL/FArray2Da.fwd.hh>
#include <ObjexxFCL/FArray3DB.fwd.hh>


namespace epigraft {
namespace design {


/// @brief original rosetta ccd moves, moves per-residue (always phi->psi), ideal bond geometry only
void
original_ccd_moves(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
);


/// @brief original rosetta ccd moves where rama score is not allowed to increase
/// @brief moves per-residue (always phi->psi), ideal bond geometry only
void
rama_limited_original_ccd_moves(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
);


/// @brief ccd moves that take into account the nonideal bonds of the cutpoint
/// @brief when attempting to close, moves per-residue (directionality dependent)
void
ccd_moves_obeying_nonideality(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
);


/// @brief ccd moves that take into account the nonideal bonds of the cutpoint
/// @brief when attempting to close, moves per-torsion
void
per_torsion_ccd_moves_obeying_nonideality(
	Integer const & total_moves,
	Pose & pose,
	Integer const & loop_begin,
	Integer const & loop_end,
	Integer const & cutpoint
);


/// @brief ccd loop closure obeying nonideal bond geometry, moves per-residue (directionality dependent)
/// @note tolerance in Angstroms, forward and backward splice RMS over N,CA,C must
/// @note be less than tolerance for an early return. otherwise goes through the
/// @note loop ii_cycles, each time both forward and backward
/// @return  the number of cycles actually taken
int
fast_ccd_loop_closure_obeying_nonideality(
	Pose & pose,
	int const & loop_begin,
	int const & loop_end,
	int const & cutpoint,
	int const & ii_cycles,
	float const & tolerance,
	bool const & rama_check,
	float const & max_rama_score_increase,
	float const & max_total_delta_helix,
	float const & max_total_delta_strand,
	float const & max_total_delta_loop,
	float & forward_deviation, // output
	float & backward_deviation, // output
	float & torsion_delta,
	float & rama_delta
);


/// @brief get overlap coordinates, obeys nonideal bond geometry
void
get_overlap(
	Pose const & pose,
	FArray3DB_float const & Eposition,// dont get from pose -- dont want a refold
	int const & cutpoint,
	int const & dir,
	FArray2Da_double M // output
);


/// @brief build overlap coordinates, obeys nonideal bond geometry
void
build_overlap_coords(
	Pose const & pose,
	int const & cutpoint,
	FArray3DB_float const & coords,
	FArray2D_float & overlap_xyz,
	bool const & Epos_index,
	int const & dir
);


/// @brief builds coordinates of backbone C, obeys nonideal bond geometry
/// @note  uses: the n_xyz position, the n->ca bond vector, the n,ca,c oriented plane
/// @warning  pos is the position one residue after the C, ie the position to whom phi,n,ca,c belong!!!
void
build_C_coords(
	Pose const & pose,
	int const & pos,
	float const & phi,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float c_xyz_out
);


/// @brief builds coordinates of backbone N,CA of next position forward in sequence
/// @warning  pos is the position to whom psi, omega, and n,ca,c_xyz belong ie one
/// @wrarnin  residue before the residue for whom we are actually building coordinates!!
void
build_N_CA_coords(
	Pose const & pose,
	int const & pos,
	float const & psi,
	float const & omega,
	FArray1Da_float n_xyz,
	FArray1Da_float ca_xyz,
	FArray1Da_float c_xyz,
	FArray1Da_float n_xyz_out,
	FArray1Da_float ca_xyz_out
);


} // namespace design
} // namespace epigraft


#endif /*INCLUDED_epigraft_design_ccd_functions_HH_*/
