// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 14018 $
//  $Date: 2007-04-06 18:44:35 -0700 (Fri, 06 Apr 2007) $
//  $Author: rhiju $

#ifndef INCLUDED_cluster_fragments
#define INCLUDED_cluster_fragments


// Rosetta Headers
#include "pose_fwd.h"

// ObjexxFCL Headers


void
cluster_fragments();

void
fragment_quality();

void steal_angles(
	pose_ns::Pose & from_pose,
	pose_ns::Pose & into_pose,
	int start,
	int length
);

float
calc_cluster_similarity(
	std::vector< int > & cluster1,
	std::vector< int > & cluster2,
	FArray2D_float sim_matrix
);

void
join_clusters(
	std::vector< int > & cluster1,
	std::vector< int > & cluster2
);

void cluster_by_maxsub();

FArray1D_int
cluster_by_similarity_matrix( FArray2D_float sim_matrix, int count );

FArray1D_int
cluster_by_similarity_matrix(
	FArray2D_float sim_matrix,
	int count,
	float min_sim_threshold
);

void
cluster_test();

void
print_matrix(
	FArray2D_float sim_matrix,
	int count
);

void
print_vector_matrix (
	std::vector< std::vector< int > > matrix
);

// Class definitions for similarity metrics for clustering methods. Each
// clustering metric provides a name for itself, and provides a method called
// calc_cluster_similarity, which takes three arguments:
// cluster1 and cluster2 - a vector of integers where each integer represents
//	the index of an element within the similarity matrix that belong to either
//	cluster1 or cluster2.
// sim_matrix - a matrix defining the similarities between objects, such that
//	the (i,j)th entry represents the similarity between objects i and j.


class SimilarityMetric {
	protected:
		std::string name;
	public:
		float
		calc_cluster_similarity(
			std::vector< int > & cluster1,
			std::vector< int > & cluster2,
			FArray2D_float sim_matrix
		);

		std::string get_name();
}; // class SimilarityMetric

class CompleteLinkageSimilarity : public SimilarityMetric {
	public:
		CompleteLinkageSimilarity() {
			name = "Complete Linkage";
		}

		float calc_cluster_similarity(
			std::vector< int > & cluster1,
			std::vector< int > & cluster2,
			FArray2D_float sim_matrix) {

			// complete-linkage clustering. Return the maximum distance
			// (or minimum similarity) between the two clusters.
			float min_sim( 10000 );
			for ( unsigned int i = 0; i < cluster1.size(); i++ ) {
				for ( unsigned int j = 0; j < cluster2.size(); j++ ) {
					if ( cluster1[i] != cluster2[j] ) {
						float sim = sim_matrix( cluster1[i] + 1, cluster2[j] + 1 );
						if ( sim <= min_sim ) {
							min_sim = sim;
						}
					}
				}
			}
			return min_sim;
		} // calc_cluster_similarity

		std::string get_name() {
			return name;
		}

}; // class CompleteLinkageSimilarity

class AverageLinkageSimilarity : public SimilarityMetric {
	public:
		AverageLinkageSimilarity() {
			name = "Average Linkage";
		}

		float calc_cluster_similarity(
			std::vector< int > & cluster1,
			std::vector< int > & cluster2,
			FArray2D_float sim_matrix) {

			// average-linkage clustering. Return the average similarity
			// between all elements within the two clusters.
			int count;
			float total_sim( 0 );
			for ( unsigned int i = 0; i < cluster1.size(); i++ ) {
				for ( unsigned int j = 0; j < cluster2.size(); j++ ) {
					if ( cluster1[i] != cluster2[j] ) {
						float sim = sim_matrix( cluster1[i] + 1, cluster2[j] + 1 );
						total_sim += sim;
						count++;
					}
				}
			}

			float average_sim = (float) total_sim / count;
			return average_sim;
		} // calc_cluster_similarity

		std::string get_name() {
			return name;
		}

}; // class AverageLinkageSimilarity

#endif
