// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20460 $
//  $Date: 2008-02-18 20:02:35 -0800 (Mon, 18 Feb 2008) $
//  $Author: possu $


// Rosetta Headers
#include "constraints.h"
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "atom_is_backbone.h"
#include "after_opts.h"
#include "angles.h"
#include "chuck.h"
#include "dipolar.h"
#include "files_paths.h"
#include "fragments.h"
#include "fullatom.h"
#include "fragments_ns.h"
#include "maps.h"
#include "misc.h"
#include "monte_carlo.h"
#include "pack.h"
#include "param.h"
#include "param_aa.h"
#include "random_numbers.h"
#include "read_aaproperties.h"
#include "recover.h"
#include "refold.h"
#include "rotate.h"
#include "runlevel.h"
#include "score_ns.h"
#include "status.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/DimensionExpressions.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/FArray5D.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/char.functions.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <sstream>
#include <cassert>

//// yab: Notes on classical constraints initialization.
//// 20070213 - ?
////
//// This unfortunate behavior is preserved in the refactored version so that
//// Rosetta operational behavior does not change.  Due to time constraints the
//// behavior has not yet been corrected and further detangled.
////
//// Two variables govern initialization of "classical constraints" defined in
//// constraints.cc: 'use_constraints' outside (see initialize_query() in initialize.cc)
//// and 'constraints_exist' inside constraints.cc.  'use_constraints' is always true
//// unless set false _internally_.  There appears to be NO FLAG governing use_constraints
//// externally.  If 'use_constraints' is true, then the function load_constraints_from_file() is
//// called by initialize_query() in initialize.cc.  If a constraint file is found, then
//// 'constraints_exist' is set to true (otherwise false).  HOWEVER, initialize_query() is
//// _not_ called in pose1 modes, but not all pose-using modes are pose1!  Therefore
//// the execution (and loading of constraints files) by load_constraints_from_file() is
//// mode dependent in vague ways.
////
//// Initialization is completely screwed up right now with regards to cst_res_wt.
//// Due to the way Rosetta initialization works, I CANNOT dimension cst_res_wt in the
//// initialize() call without significant rework of the first section in initalize_query()
//// (for more details trace the read_aa call).  Work-around is to manually call a
//// 'reset_cst_res_wt()' later in initalize query. -_-  Not happy.  This is dangerous in
//// case the cst_res_wt FArray is ever looked at and reset_cst_wt() has not been called.
//// When it comes time to detach and rehook scoring, I might be forced to rework the
//// initialization pipeline as well.
////
//// There are some messy interdependencies between constraints.cc and the rest of Rosetta,
//// e.g. the HA_position HN_position stuff (see minimize.cc) where it's not 100% clear the
//// initialize and the calling order will always be guaranteed.  The old code works, but
//// one mis-step and it could be easily broken.
////
//// The original slicing behavior is reproduced, but may actually be half-broken, see below.
////
//// Need to figure out what to do about the cst non-bonded list.  Currently stored as a global
//// variable here but... move it inside class, or...?
////
//// Something weird is going on with the verbose flag in eval_pairConstraint.
////
//// load_constraints_from_file has potential global issues

namespace classical_constraints {

namespace nblist {
	FArray1D_bool cst_nb( classical_constraints::constants::MAX_CONSTRAINTS );
	FArray1D_bool saved_cst_nb_( classical_constraints::constants::MAX_CONSTRAINTS ); // hidden from outside world
} // namespace nblist

namespace constants {

	int const MAX_CONSTRAINTS = 2000; // max number of constraint pairs
	Dimension const MAX_CST_LIST( ( MAX_CONSTRAINTS / param::MAX_RES() ) * 5 ); // max constraints per residue pair
	float const DEGENERATE_PAD = 0.0;  // upper bound padding for hydrogen
		// constraints evaluated using heavy atom coordinates
	float const GLOBAL_PAD = 0.0;  //upper bound padding for all constraints
	int const CENTROID_ATOMNUM = -999;
	int const H_ATOMNUM = 0;
	int const HA_ATOMNUM = -1;
	int const HA2_ATOMNUM = -2;
	int const UNDEF_ATOMNUM = 999;

} // namespace constants

namespace options {

	bool constraints_exist;

} // namespace options

namespace functions {

////////////////////////////////////////////////////////////////////////////////
/// @begin type2index
///
/// @brief
///
/// @detailed
///
/// @param  atom - [in/out]? -
/// @param  restype - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
type2index(
	std::string & atom,
	int & restype // residue type
)
{
	using namespace files_paths;
	using namespace classical_constraints::constants;

	int type2index; // Return value

//car local
	int res_variant; // residue variant, assumed to be 1 (standard)

	if ( ! is_any_of( atom[0], " 123#" ) ) size( atom.insert( 0, " " ), 4 ); // Align fields

	type2index = UNDEF_ATOMNUM;

	if ( is_any_of( atom[0], " 123#" ) && atom.length() == 3 ) {
		size( atom.insert( 3, " " ), 4 ); // Align fields
	}

//car protons:
	if ( atom == " H  " ) {
		type2index = H_ATOMNUM;
	} else if ( atom == " HA " ) {
		type2index = HA_ATOMNUM;
	} else if ( atom == "1HA" ) {
		if ( ! IUPAC ) type2index = HA_ATOMNUM;
	} else if ( atom == "2HA" ) {
		type2index = ( IUPAC ? HA_ATOMNUM : HA2_ATOMNUM );
	} else if ( atom == "3HA " || atom == " HA2" ) {
		type2index = HA2_ATOMNUM;

//car bb heavy atoms  :
	} else if ( atom == " CA " ) {
		type2index = 2;
	} else if ( atom == " N  " ) {
		type2index = 1;
	} else if ( atom == " C  " ) {
		type2index = 3;
	} else if ( atom == " CB " ) {
		type2index = 5;
	} else if ( atom == " O  " ) {
		type2index = 4;

//car centroids:

	} else if ( atom == " CEN" ) {
		type2index = CENTROID_ATOMNUM;

//car sc atoms:
	} else if ( atom[0] != '#' ) {
		res_variant = 1;
		atom_num_from_atom_name( atom, restype, res_variant, type2index );
		if ( type2index == 0 ) goto L100; // atom_num_from_atom_name returned undef

//car protons for which bound heavy atom coordinates will be used
	} else {
		heavyatom_num_from_proton_name( atom, restype, type2index );
		if ( type2index == 0 ) goto L100;
	}

	return type2index;

L100:
//      std::cout << "Atom type: " << atom << " unknown to type2index" << std::endl;
	type2index = UNDEF_ATOMNUM;
	return type2index;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin index2type
///
/// @brief
///
/// @detailed
///
/// @param  num - [in/out]? - atom number
/// @param  restype - [in/out]? - residue type
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
std::string
index2type(
	int const num, // atom number
	int const restype // residue type
)
{
	using namespace files_paths;
	using namespace param_aa;
	using namespace classical_constraints::constants;

	std::string index2type; // Return value

//** backbone atoms  (position numbering)
	if ( num == 1 ) {
		index2type = " N  ";
	} else if ( num == 2 ) {
		index2type = " CA ";
	} else if ( num == 3 ) {
		index2type = " C  ";
	} else if ( num == 4 ) {
		index2type = " O  ";
	} else if ( num == 5 ) {
		index2type = " CB ";

//*  centroids
	} else if ( num == CENTROID_ATOMNUM ) {
		index2type = " CEN";

//*  backbone hydrogens
	} else if ( num == H_ATOMNUM ) {
		index2type = " H  ";
	} else if ( num == HA_ATOMNUM ) {
		index2type = (restype != aa_gly ? " HA " : (IUPAC ? "2HA " : "1HA "));
	} else if ( num == HA2_ATOMNUM ) {
		index2type = (IUPAC ? "3HA " : "2HA ");

//car unknown
	} else if ( num == UNDEF_ATOMNUM ) {
		index2type = "UNKN";

//car sc atoms (beyond CB)    (full_coord numbering)
	} else {
		atom_name_from_atom_num( num, restype, 1, index2type );
		if ( index2type == "    " ) { // returned undefined
			std::cout << "Atom type: " << num << " restype: " << restype <<
			 " unknown to index2type" << std::endl;
			index2type = "UNKN";
		}
	}

	return index2type;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin heavyatom_num_from_proton_name
///
/// @brief
///
/// @detailed
///
/// @param  name - [in/out]? - atom name
/// @param  aa - [in/out]? - amino acid number
/// @param  number - [in/out]? - atom number
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
heavyatom_num_from_proton_name(
	std::string const & name, // atom name
	int & aa, // amino acid number
	int & number // atom number
)
{

	using namespace param_aa;
//car for carbon bound protons only at this point
//car ASN,GLN N-bound
//************


	number = 0;
//** sidechain atoms

//**   degenerate HA protons on gly
	if ( aa == aa_gly && name.substr(1,2) == "HA")
			 number = 2; // G degenerate HA's

//**   CB protons
//car note this is position numbering while others are full_coord numbering
	if ( aa != aa_gly && name.substr(1,2) == "HB" ) number = 5; // all except G
	//vats this was set to 3 in svn thereby setting it to carbonyl carbon of the backbone while is supposed to be CB


//**   CG protons
	if ( name.substr(1,2) == "HG" ) {
		if ( aa == aa_ile || aa == aa_val ) {                // IV
			if ( name.substr(1,3) == "HG1" ) number = 6; // CG1
			if ( name.substr(1,3) == "HG2" ) number = 7; // CG2
		} else if ( aa == aa_thr ) {                    // T
			if ( name.substr(1,3) == "HG2" ) number = 7; // CG2
		} else if ( aa != aa_ala && aa != aa_gly && aa != aa_ser && aa != aa_phe && aa != aa_tyr) { // not AGSFY
			number = 6;
		}
	}

//**   CD protons
	if ( name.substr(1,2) == "HD" ) {
		if ( aa == aa_phe || aa == aa_leu || aa == aa_trp || aa == aa_tyr ) { // FLWY
			if ( name.substr(1,3) == "HD1" ) number = 7; // CD1
			if ( name.substr(1,3) == "HD2" ) number = 8; // CD2
		} else if ( aa == aa_his ) {                                    // H
			if ( name.substr(1,3) == "HD2" ) number = 8;
		} else if ( aa == aa_ile ) {                                      // I
			if ( name.substr(1,3) == "HD1" ) number = 8;
		} else if ( aa == aa_glu || aa == aa_lys || aa == aa_pro || aa == aa_gln || aa == aa_arg ) { // EKPQR
			number = 7;
		} else if ( aa == aa_asn ) { // N
			if ( name.substr(1,3) == "HD2" ) number = 8; // ND2
		}
	}

//**   CE protons

	if ( name.substr(1,2) == "HE" ) {
		if ( aa == aa_phe || aa == aa_tyr ) {  // FY
			if ( name.substr(1,3) == "HE1" ) number = 9; // CE1
			if ( name.substr(1,3) == "HE2" ) number = 10; // CE2
		} else if ( aa == aa_his ) { // His
			if ( name.substr(1,3) == "HE1" ) number = 9;
		} else if ( aa == aa_trp ) { // W
			if ( name.substr(1,3) == "HE2" ) number = 10; // CE2
			if ( name.substr(1,3) == "HE3" ) number = 11; // CE3
		} else if ( aa == aa_lys || aa == aa_met ) { // KM
			number = 8;
		} else if ( aa == aa_gln ) { // Q
			if ( name.substr(1,3) == "HE2" ) number = 9; // NE2
		}
	}

//**   CZ protons
	if ( name.substr(1,2) == "HZ" ) {
		if ( aa == aa_phe ) {        // F
			number = 11;
		} else if ( aa == aa_arg ) {  // R
			number = 9;
		} else if ( aa == aa_trp ) { // W
			if ( name.substr(1,3) == "HZ2" ) number = 12; // CZ2
			if ( name.substr(1,3) == "HZ3" ) number = 13; // CZ3
		}
	}

//** Trp CH2
	if ( aa == aa_trp && name == "HH2" ) number = 14;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin cst_atomum_to_fullcoord_atomnum
///
/// @brief  Converts cst atom numbers to fullcoord atom nums
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
int
cst_atomnum_to_fullcoord_atomnum(
  int const cst_atom,
	int const cst_aa,
	int const aa,
	int const aav
	)
{
	using namespace classical_constraints::constants;

	if ( cst_atom == CENTROID_ATOMNUM ) {
		std::cerr << "centroid atom number cannot be converted to fullcoord"
							<< " atom number \n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	int ret;
	atom_num_from_atom_name(index2type(cst_atom,cst_aa), aa, aav, ret);
//car check for invalid atom num
  if (ret == 0) {
		std::cerr << "Unable to find fullatom number for constraint atom: \n" <<
		 "Atom Type" << cst_atom << "Residue type " << cst_aa << " Type" <<
		 index2type(cst_atom,cst_aa) << "Fullatom aa " << aa << " Fullatom aav "
     << aav << "\n";
		assert( ret != 0 );
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return ret;
}

} // namespace functions

namespace BOUNDARY {

Constraints * constraints_object_;
Constraints * saved_constraints_object_;

bool called_once_ = false;

void
init_module()
{
	using namespace misc;

	constraints_object_ = new Constraints(&Eposition, &centroid, &full_coord, &res, &res_variant, &total_residue);

	called_once_ = true;
}

bool
init_module_called_once()
{
	return called_once_;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_constraints_exist
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
get_constraints_exist()
{
	return classical_constraints::options::constraints_exist;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin set_constraints_exist
///
/// @brief
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_constraints_exist(
	bool setting
)
{
	classical_constraints::options::constraints_exist = setting;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin constraints_set_verbose
///
/// @brief
///car use to turn out detailed output for next call to eval_dipolar
///car reset to false after every call to eval_dipolar
///
/// @detailed
///
/// @param  flag - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
constraints_set_verbose(
	bool flag
)
{
	constraints_object_->set_verbose(flag);
}

bool
get_verbose_pdb_output()
{
	return constraints_object_->get_verbose_pdb_output();
}

void
set_verbose_pdb_output(
	bool flag
)
{
	constraints_object_->set_verbose_pdb_output(flag);
}

void
load_constraints_from_file()
{
	constraints_object_->load_constraints_from_file();
}

void
load_constraints_from_file(
	std::string const & filename
)
{
	constraints_object_->load_constraints_from_file( filename );
}

void
output_constraints_status(
	int mode,
	std::ostream & iunit
)
{
	constraints_object_->output_status(mode, iunit);
}

void
set_max_seqSep(
	int seq
)
{
	constraints_object_->set_max_seqSep(seq);
}

int
get_max_seqSep()
{
	return constraints_object_->get_max_seqSep();
}

void
noe_dist2(
	FArray1DB_float const & xyz1,
	FArray1DB_float const & xyz2,
	float & dis2
)
{
	constraints_object_->noe_dist2(xyz1, xyz2, dis2);
}

void
get_noe_coords(
	int pair,
	FArray1Da_float xyz1,
	FArray1Da_float xyz2
)
{
	constraints_object_->get_noe_coords(pair, xyz1, xyz2);
}

bool
residue_pair_has_constraints
(
 int res1,
 int res2
)
{
	return constraints_object_->residue_pair_has_constraints(res1, res2);
}

bool
cst_pair_satisfies_conditions(
	int pair,
	bool fullatom,
	int stage
)
{
	return constraints_object_->cst_pair_satisfies_conditions(pair, fullatom, stage);
}

void
get_XYZ(
	int & residue,
	int & type_index,
	FArray1Da_float xyz
)
{
	constraints_object_->get_XYZ(residue, type_index, xyz);
}

void
eval_pairConstraint(
	int mode,
	float & score_x // ret'd copy of score, = pc_score usually
)
{
	constraints_object_->eval_pairConstraint(mode, score_x);
}

void
precalc_cst_deriv()
{
	constraints_object_->precalc_cst_deriv();
}

void
eval_constraint_deriv(
	int const & i,
	int const & ii,
	int const & j,
	int const & jj,
	bool const & fullatom,
	float const & dis,
	float const & dist2,
	float & deriv
)
{
	constraints_object_->eval_constraint_deriv(i, ii, j, jj, fullatom, dis, dist2, deriv);
}

float
get_res_res_cstE(
	int const res1,
	int const res2,
	int const aa1,
	int const aav1,
	int const aa2,
	int const aav2,
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	bool const bb1,
	bool const sc1,
	bool const bb2,
	bool const sc2
)
{
	return constraints_object_->get_res_res_cstE(res1, res2, aa1, aav1, aa2, aav2, coord1, coord2, bb1, sc1, bb2, sc2);
}

void
reset_cst_res_wt()
{
	constraints_object_->reset_cst_res_wt();
}

void
setup_cst_res_weight()
{
	static bool init = { false };
	static bool use_wt = false;

	if (!init) {
		use_wt = truefalseoption("rand_cst_res_wt");
		init = true;
	}

	constraints_object_->reset_cst_res_wt();
	if ( use_wt ) {
		constraints_object_->randomize_cst_res_wt();
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin slice_cst
///
/// @brief  slice cst pairs according to segments file used in jumping fold
///   trees
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void slice_cst(
	const FArray1D_bool & trim_res,
	int const & nres
)
{
	if (!get_constraints_exist()) {
		return;
	}

	// SAVE NB LIST FIRST!
	nblist::saved_cst_nb_ = nblist::cst_nb;
	nblist::cst_nb = false;

	int old_np_pairs = constraints_object_->get_np_pairs();
	int sliced_np_pairs = 0;
	FArray1D_int cst_pair(4);
	for (int i = 1 ; i <= old_np_pairs; i++) {
		if ( trim_res(constraints_object_->get_constraintPair(i, 1)) && trim_res(constraints_object_->get_constraintPair(i, 2)) ) {
			sliced_np_pairs++;

			nblist::cst_nb(sliced_np_pairs) = nblist::saved_cst_nb_( i );
		}
	}

	// save old object
	saved_constraints_object_ = constraints_object_;

	// make and set sliced object as the current object
	constraints_object_ = constraints_object_->slice(trim_res, nres);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin restore_cst
///
/// @brief restore original cst infor for next struct in jumping fold
///
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void restore_cst()
{
	if (!get_constraints_exist()) {
		return;
	}

	// restore cst nb list
	nblist::cst_nb = nblist::saved_cst_nb_;
	nblist::saved_cst_nb_ = false;

	// restore old object
	delete constraints_object_;
	constraints_object_ = saved_constraints_object_;
	saved_constraints_object_ = NULL;
}

int
np_pairs()
{
	return constraints_object_->get_np_pairs();
}

int
constraintPair(
	int i,
	int j
)
{
	return constraints_object_->get_constraintPair(i, j);
}

int
pairSeqSep(
	int i
)
{
	return constraints_object_->get_pairSeqSep(i);
}

float
pairRadius2(
	int i
)
{
	return constraints_object_->get_pairRadius2(i);
}

float
dist_constraint(
	int i
)
{
	return constraints_object_->get_dist_constraint(i);
}

float
pairRadius(
	int i
)
{
	return constraints_object_->get_pairRadius(i);
}

float
pairMinRadius(
	int i
)
{
	return constraints_object_->get_pairMinRadius(i);
}

int
pair_atom_index(
	int i,
	int j
)
{
	return constraints_object_->get_pair_atom_index(i, j);
}

FArray2D_int const &
pair_atom_index()
{
	return constraints_object_->get_pair_atom_index();
}

float
HN_position(
	int i,
	int j
)
{
	return constraints_object_->get_HN_position(i, j);
}

float
HA_position(
	int i,
	int j
)
{
	return constraints_object_->get_HA_position(i, j);
}

} // namespace BOUNDARY

////////////////////////////////////////
/// Public: Construction/Destruction ///
////////////////////////////////////////
Constraints::Constraints(
	FArray3Dp_float const * Eposition,
	FArray2D_float const * centroid,
	FArray3D_float const * full_coord,
	FArray1D_int const * res,
	FArray1D_int const * res_variant,
	int const * total_residue
)
{
	Eposition_ = Eposition;
	centroid_ = centroid;
	full_coord_ = full_coord;
	res_ = res;
	res_variant_ = res_variant;
	total_residue_ = total_residue;

	initialize();
}

Constraints::Constraints() {}
Constraints::~Constraints() {}

//////////////////////////////////////
/// Public: Object Merging/Slicing ///
//////////////////////////////////////

Constraints *
Constraints::slice(
	const FArray1D_bool & trim_res,
	int const & nres
)
{
	// yab:
	// some messy stuff goes on in this function, I'm just going to reproduce
	// what was originally written; could definitely be cleaned up
	//
	// this routine is _not_ general -- the original code never properly sliced
	// "everything", so certain things are left untouched and this could very well be
	// a bug depending on how certain pieces of data are used, see code block below

	Constraints * sliced = new Constraints(); // goes onto heap! remember to take care of deletion outside of this method!

	//// copy things first
	// mirrors, pointers
	sliced->Eposition_ = this->Eposition_;
	sliced->centroid_ = this->centroid_;
	sliced->full_coord_ = this->full_coord_;
	sliced->res_ = this->res_;
	sliced->total_residue_ = this->total_residue_;

	// modes
	sliced->mode = this->mode;

	// status and output options
	sliced->verbose = this->verbose;
	sliced->verbose_pdb_output = this->verbose_pdb_output;

	// derivative calculation
	sliced->cst_drv_mode = this->cst_drv_mode;
	sliced->cst_drv_factor = this->cst_drv_factor;
	//// end copy

	// fill in sliced data
	std::vector< int > offset;
	int off = 0;
	for ( int i = 1; i <= nres; ++i ) {
		if( !trim_res(i) ) {
			off++;
		}
		offset.push_back( off );
	}

	FArray1D_int cst_pair(4); // temporary, tracking

	sliced->np_pairs = 0;
	for (int i = 1; i <= this->np_pairs; i++) { // go through each cst pair
		if ( trim_res(this->constraintPair(i, 1)) && trim_res(this->constraintPair(i, 2)) ) {
			cst_pair( 1 ) = this->constraintPair ( i, 1 ) - offset.at( this->constraintPair ( i, 1 ) - 1 );
			cst_pair( 3 ) = this->constraintPair ( i, 3 );
			cst_pair( 2 ) = this->constraintPair ( i, 2 ) - offset.at( this->constraintPair ( i, 2 ) - 1 );
			cst_pair( 4 ) = this->constraintPair ( i, 4 );

			sliced->constraintPair( sliced->np_pairs, 1 ) = cst_pair( 1 );
			sliced->constraintPair( sliced->np_pairs, 2 ) = cst_pair( 2 );
			sliced->constraintPair( sliced->np_pairs, 3 ) = cst_pair( 3 );
			sliced->constraintPair( sliced->np_pairs, 4 ) = cst_pair( 4 );

			sliced->pairRadius( sliced->np_pairs ) = this->pairRadius( i );
			sliced->pairMinRadius( sliced->np_pairs ) = this->pairMinRadius( i );
			sliced->pair_atom_index( 1, sliced->np_pairs ) = this->pair_atom_index( 1, i );
			sliced->pair_atom_index( 2, sliced->np_pairs ) = this->pair_atom_index( 2, i );
			sliced->pairSeqSep( sliced->np_pairs ) = this->pairSeqSep( i );

			sliced->pairRadius2( sliced->np_pairs ) = this->pairRadius2( i );
			sliced->dist_constraint( sliced->np_pairs ) = this->dist_constraint( i );
			sliced->dist_constraint2( sliced->np_pairs ) = this->dist_constraint2( i );
			sliced->score_constraint( sliced->np_pairs ) = this->score_constraint( i );

			sliced->np_pairs++;
		}
	}

	// The following are NOT SLICED, they are COPIED!  Is this correct behavior??
	sliced->HA_position = this->HA_position;
	sliced->HN_position = this->HN_position;
	sliced->using_random_cst_res_wt = this->using_random_cst_res_wt;
	sliced->cst_res_wt = this->cst_res_wt;
	sliced->ambi_res1 = this->ambi_res1;
	sliced->ambi_res2 = this->ambi_res2;
	sliced->messed_up_cst = this->messed_up_cst;

	// rebuild lists
	sliced->build_constraint_list();

	return sliced;
}


/////////////////////////////////////
/// Public: External Input/Output ///
/////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin load_constraints_from_file
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::load_constraints_from_file(
	std::string const & filename
)
{
	using namespace classical_constraints::constants;
	using namespace classical_constraints::functions;
	using namespace files_paths;
	using namespace runlevel_ns;

	FArray1D_int const & res = *res_; // yab: misc removal
	int const & total_residue = *total_residue_; // yab: misc removal

//car comment
//car comment
//car comment
//car n_pairLines                 (# of lines of constraints to read)
//car tag,res1,atom1,res2,atom2,upperbound,lowerbound,{true distance}
//car
//car tag: '#':ignore,(actually any non-whitespace char)
//car true distance is optional and not read
//car atom1 and atom2 should follow pdb-style atom names with IUPAC numbering
//car also allowed: ' CEN' for centroid constraints (not evaluted in fullatom mode)
// car degenerate hydrogen references allowed (no degenerate heavy atoms)

	FArray2D_string attype( MAX_CONSTRAINTS, 2, std::string( 4, ' ' ) );
	std::string comment;
	std::string resname1, resname2;
	int n_pairLines;

//	std::string const filename( constraints_path + protein_name_prefix + protein_name + protein_chain + '.' + cst_ext );

	utility::io::izstream cnstr_zx( filename );

	if ( !cnstr_zx ) {
		std::cout << "File: " << cnstr_zx.filename() << " not found" << std::endl;
		std::cout << "Running without distance constraints" << std::endl;
		np_pairs = 0;
		n_pairLines = 0;
		classical_constraints::BOUNDARY::set_constraints_exist(false);
		cnstr_zx.close();
		cnstr_zx.clear();
		return;
	}

	classical_constraints::BOUNDARY::set_constraints_exist(true);

	cnstr_zx >> bite( comment ) >> skip; // comment line
	std::cout << comment << std::endl;
	cnstr_zx >> bite( comment ) >> skip; // comment line
	std::cout << comment << std::endl;
	cnstr_zx >> bite( comment ) >> skip; // comment line
	std::cout << comment << std::endl;
	cnstr_zx >> n_pairLines >> skip;

	int k = 0; // current constraints

	bool problems = false;
	int duplicates = 0;
	int unrecog = 0;
	for ( int kk = 1; kk <= n_pairLines; ++kk ) {
		++k;
		if ( k > MAX_CONSTRAINTS ) {
			std::cout << "WARNING: increase MAX_CONSTRAINTS in constraints.cc"
								<< std::endl;
			std::cout << " MAX_CONSTRAINTS: " << MAX_CONSTRAINTS <<
				" constraints read: " << k << "; anticipating " << n_pairLines <<
				" lines " << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		/*
		cnstr_zx >> bite( tag ) >> skip( 2 ) >>
			bite( 4, constraintPair(k,1) ) >> skip( 1 ) >>
			bite( 4, attype(k,1) ) >> skip( 1 ) >>
			bite( 4, constraintPair(k,2) ) >> skip( 1 ) >>
			bite( 4, attype(k,2) ) >> skip( 1 ) >>
			bite( 10, pairRadius(k) ) >> skip( 1 ) >>
			bite( 10, pairMinRadius(k) ) >> skip;
		*/

		std::string line;
		getline( cnstr_zx, line );
		std::stringstream line_stream ( line );
		if( !line_stream.bad() ) {
			line_stream >> constraintPair( k, 1 ) >> attype( k, 1 ) >> constraintPair( k, 2 ) >> attype( k, 2 ) >> pairRadius( k ) >> pairMinRadius( k );
		} else {
			std::cout << "Bad constraint line...skipping this line " << std::endl;
			--k;
		}
		if ( cnstr_zx.eof() ) {
      std::cout << "WARNING:: Early termination of constraint file.\n";
			std::cout << "          Expected" << SS(n_pairLines) <<
			 " constraints. Read" << SS(kk-1) << std::endl;
			--k;  //discard;
			goto L111;
		} else if ( cnstr_zx.fail() ) {
      std::cout << "WARNING:: Format error in constraint file.\n";
			std::cout << "          Skipping line: " << SS(kk+4) << std::endl;
			cnstr_zx.clear();
			cnstr_zx >> skip;
			--k; //discard
			goto L112;
		}
		/*
		if ( tag != ' ' ) {
			--k;
			goto L112; // skip line
		}
		*/
		if ( constraintPair(k,1) > total_residue  ||
				 constraintPair(k,1) < 1 ||
				 constraintPair(k,2) > total_residue ||
				 constraintPair(k,2) < 1 ) {
			std::cout << "Invalid residue number, skipping constraint: " << std::endl;
			std::cout << I( 5, constraintPair(k,1) ) << "   " << attype(k,1) <<
				I( 5, constraintPair(k,2) ) << "   " << attype(k,2) <<
				F( 10, 2, pairRadius(k) ) << F( 10, 2, pairMinRadius(k) ) << std::endl;
			--k;
			goto L112;
		}
		constraintPair(k,3) = res(constraintPair(k,1));
		constraintPair(k,4) = res(constraintPair(k,2));
		pair_atom_index(1,k) = type2index( attype(k,1), constraintPair(k,3) );
		pair_atom_index(2,k) = type2index( attype(k,2), constraintPair(k,4) );
		pairSeqSep(k) = std::abs( constraintPair(k,1) - constraintPair(k,2) );

//car look for unidentified atom types/names
		if ( pair_atom_index(1,k) == 999 ) problems = true;
		if ( pair_atom_index(2,k) == 999 ) problems = true;
		if ( problems ) {
			++unrecog;
			name_from_num(constraintPair(k,3),resname1);
			name_from_num(constraintPair(k,4),resname2);
			if (runlevel > standard) {
				std::cout << "Unknown atom type in constraint file, ignoring: "
									<< std::endl;
				std::cout << I( 5, pairSeqSep(k) ) << ' ' <<
					I( 5, constraintPair(k,1) ) << "   " << attype(k,1) <<
					I( 5, constraintPair(k,2) ) << "   " << attype(k,2) <<
					F( 10, 2, pairRadius(k) ) << F( 10, 2, pairMinRadius(k) ) << "   " <<
					resname1 << "   " << resname2 << std::endl;
			}
    	--k; // discard
			problems = false;
			goto L112;
		}

//car check for protons for which heavy atom coordinates will be used
//car and pad bounds accordingly
//car degenerate hydrogens (same heavy atom), terminal H
		if ( constraintPair(k,1) == 1 && attype(k,1).substr(1,2) == "H ") {
			pairRadius(k) += DEGENERATE_PAD;
		} else if ( attype(k,1)[0] == '#' ) {
			pairRadius(k) += DEGENERATE_PAD;
		}
		if ( constraintPair(k,2) == 1 && attype(k,2).substr(1,2) == "H ") {
			pairRadius(k) += DEGENERATE_PAD;
		} else if ( attype(k,2)[0] == '#' ) {
			pairRadius(k) += DEGENERATE_PAD;
		}

//car add general padding to all constraints
		pairRadius(k) += GLOBAL_PAD;

//car if same as previous constraint, overwrite previous with largest bound
		if ( k > 1 ) {
			if ( constraintPair(k,1) == constraintPair(k-1,1) &&
					 constraintPair(k,2) == constraintPair(k-1,2) &&
					 pair_atom_index(1,k) == pair_atom_index(1,k-1) &&
					 pair_atom_index(2,k) == pair_atom_index(2,k-1) ) {
				pairRadius(k-1) = std::max(pairRadius(k),pairRadius(k-1));
				pairMinRadius(k-1) = std::min(pairMinRadius(k),pairMinRadius(k-1));
				++duplicates;
				if ( runlevel > standard ) std::cout << "Duplicate constraint, number: " << kk << std::endl;
				--k;
			}
		}
		pairRadius2(k) = pairRadius(k) * pairRadius(k);

		if ( problems ) {
			name_from_num(constraintPair(k,3),resname1);
			name_from_num(constraintPair(k,4),resname2);
			std::cout << "unknown atom type in constraint file, ignoring..." << std::endl;
			std::cout << I( 5, pairSeqSep(k) ) << ' ' <<
				I( 5, constraintPair(k,1) ) << "   " << attype(k,1) <<
				I( 5, constraintPair(k,2) ) << "   " << attype(k,2) <<
				F( 10, 2, pairRadius(k) ) << F( 10, 2, pairMinRadius(k) ) << "   " <<
				resname1 << "   " << resname2 << std::endl;
			--k; // discard
			problems = false;
		}

L112:;
	}

L111:
	cnstr_zx.close();
	cnstr_zx.clear();
	np_pairs = k;

	//	add_centroid_constraints();

	std::cout << "stage res1  atom1 res2  atom2     upper     lower" <<
	 "   aa1   aa2" << std::endl;
	for ( int k = 1; k <= np_pairs; ++k ) {
		name_from_num( constraintPair(k,3), resname1 );
		name_from_num( constraintPair(k,4), resname2 );
		build_ambiguous_constraint_list( k, attype(k, 1), attype(k, 2) );
		std::cout << I( 5, pairSeqSep(k) ) << ' ' <<
		 I( 5, constraintPair(k,1) ) << "   " <<
		 index2type( pair_atom_index(1,k), constraintPair(k,3) ) <<
		 I( 5, constraintPair(k,2) ) << "   " <<
		 index2type( pair_atom_index(2,k), constraintPair(k,4) ) <<
		 F( 10, 2, pairRadius(k) ) << F( 10, 2, pairMinRadius(k) ) << "   " <<
		 resname1 << "   " << resname2 << std::endl;
	}
	std::cout << "------------------------------------------------------------------" << std::endl;

	build_constraint_list();
	if (unrecog > 0) {
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n";
		std::cout << "WARNING :: " << SS(unrecog) << " constraints with unrecognized atom types detected. \n";
		std::cout << "These constraints will be ignored.\n";
		std::cout << "Use a runlevel > standard (ie command line option -runlevel 1) \n";
		std::cout << " for more details about unrecognized constraints.\n";
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n";
	}

	if (duplicates > 0) {
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n";
		std::cout << "WARNING ::" << SS(duplicates) << " duplicate constraints detected. \n";
		std::cout << "Duplicate constraints listed sequentially in the input file are\n";
		std::cout << "replaced with a single constraint with the least restrictive bounds.\n";
		std::cout << "Use a runlevel > standard (i.e. command line option -runlevel 1) \n";
		std::cout << "for more details about duplicated constraints that are detected.\n";
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX\n";
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin output_constraints_status
///
/// @brief
///
/// @detailed
///
/// @param  mode - [in/out]? -
/// @param  iunit - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///      distances output are the distances that were last caluclated for
///      each atom pair; these may be incorrect if fullatom_energy or
///      evaluate_pair_constraints was not the last to evaluate constraints.
///      (ie if the packer or rotamer trials have been called)
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::output_status(
	int mode,
	std::ostream & iunit
)
{
	char uhoh;
	std::string resname1, resname2;
	float total = 0.0;
	int counter = 0;
	float delta;
	int const stage = ( mode >= 0 ? get_max_seqSep() : 9999 );
	bool const fullatom = get_fullatom_flag();

	// jk Suppress output if there are no constraints
	if ( np_pairs == 0 ) return;

	iunit << "Constraints Scores" << '\n';
	iunit << "viol stage res1  atom1 res2  atom2    dist    " <<
	 "upper    delta    aa1   aa2   score" << '\n';
	for ( int k = 1; k <= np_pairs; ++k ) {

		if ( ! cst_pair_satisfies_conditions(k,fullatom,stage) ) continue;

		name_from_num(constraintPair(k,3),resname1);
		name_from_num(constraintPair(k,4),resname2);

		uhoh = ' ';
		if ( dist_constraint(k) < 0.0 ) {
			dist_constraint(k) = std::sqrt(dist_constraint2(k));
		}
		delta = pairRadius(k)-dist_constraint(k);
		if ( delta < 0.0 ) uhoh = '*';

//		if ( delta < 0.0 ) iunit << '\n';
		iunit << ' ' << uhoh << "   " <<
		 I( 5, pairSeqSep(k) ) << ' ' <<
		 I( 5, constraintPair(k,1) ) << "   " <<
		 classical_constraints::functions::index2type(pair_atom_index(1,k),constraintPair(k,3)) <<
		 I( 5, constraintPair(k,2) ) << "   " <<
		 classical_constraints::functions::index2type(pair_atom_index(2,k),constraintPair(k,4)) <<
		 F( 9, 2, dist_constraint(k) ) << F( 9, 2, pairRadius(k) ) <<
		 F( 9, 2, pairRadius(k) - dist_constraint(k) ) << "   " <<
			resname1 << "   " << resname2 << "   "  <<
			F(9, 4, score_constraint(k)) << '\n';
		total += score_constraint(k);
		++counter;

	}
	iunit << " Constraints: " << counter << " Scores Total " << SS( total ) << '\n';
}


/////////////////////////
/// Public: Accessors ///
/////////////////////////
bool
Constraints::get_verbose()
{
	return verbose;
}

void
Constraints::set_verbose(
	bool flag
)
{
	verbose = flag;
}

bool
Constraints::get_verbose_pdb_output()
{
	return verbose_pdb_output;
}

void
Constraints::set_verbose_pdb_output(
	bool flag
)
{
	verbose_pdb_output = flag;
}

int
Constraints::get_max_seqSep()
{
	return max_seqSep;
}

void
Constraints::set_max_seqSep(
	int sep
)
{
	max_seqSep = sep;
}

///////////////////////////
/// Public: Data Access ///
///////////////////////////
int
Constraints::get_np_pairs()
{
	return np_pairs;
}

int
Constraints::get_constraintPair(
	int i,
	int j
)
{
	return constraintPair(i, j);
}

int
Constraints::get_pairSeqSep(
	int i
)
{
	return pairSeqSep(i);
}

float
Constraints::get_pairRadius2(
	int i
)
{
	return pairRadius2(i);
}

float
Constraints::get_dist_constraint(
	int i
)
{
	return dist_constraint(i);
}

float
Constraints::get_pairRadius(
	int i
)
{
	return pairRadius(i);
}

float
Constraints::get_pairMinRadius(
	int i
)
{
	return pairMinRadius(i);
}

int
Constraints::get_pair_atom_index(
	int i,
	int j
)
{
	return pair_atom_index(i, j);
}

FArray2D_int const &
Constraints::get_pair_atom_index()
{
	return pair_atom_index;
}

float
Constraints::get_HN_position(
	int i,
	int j
)
{
	return HN_position(i, j);
}

float
Constraints::get_HA_position(
	int i,
	int j
)
{
	return HA_position(i, j);
}

///////////////////////
/// Public: Weights ///
///////////////////////
bool
Constraints::res_wt_randomized()
{
	return using_random_cst_res_wt;
}

void
Constraints::randomize_cst_res_wt()
{
	int const & total_residue = *total_residue_; // yab: misc removal

	cst_res_wt.dimension(total_residue);
	for ( int i = 1; i <= total_residue; ++i ) {
		cst_res_wt(i)=ran3()*2.0;    // should switch to gaussian later//
		std::cout << "cst_res_WT: " << cst_res_wt(i)  << std::endl;
	}

	using_random_cst_res_wt = true;
}

void
Constraints::reset_cst_res_wt()
{
	int const & total_residue = *total_residue_; //yab: misc removal

	cst_res_wt.dimension(total_residue);
	cst_res_wt=1.0;
	using_random_cst_res_wt = false;
}

///////////////////////////
/// Public: Pair Checks ///
///////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin residue_pair_has_constraints
///
/// @brief
/// apl returns true when there are one or more constraints between the two
/// apl residues.
///
/// @detailed
/// apl when n_cst(i,j) is non-zero, there are constraints between i and j
///
/// @param
/// res1 - [in] - the first of the two residues
/// res2 - [in] - the second of the two residues
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
/// apl
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Constraints::residue_pair_has_constraints
(
 int res1,
 int res2
)
{
  return n_cst(res1, res2) > 0;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin cst_pair_satisfies_conditions
///
/// @brief should constraint be evaluated?
///
/// @detailed
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks  consolidates logic for evaluating constraints
///
/// @references
///
/// @authors car 8/13/2004
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
Constraints::cst_pair_satisfies_conditions(
	int const pair,
	bool const fullatom,
	int const stage
)
{
	using namespace classical_constraints::constants;
	// car check for conditions not to evaluate
	//too long range
	if ( pairSeqSep(pair) > stage) return false;

	//sidechain constraint, no sidechain coords exist
	if (!fullatom &&
	 (pair_atom_index(1,pair) > 5 ||pair_atom_index(2,pair) > 5 ) )
   return false;

	//centroid constraint in fullatom mode
	if (fullatom &&
	 (pair_atom_index(1,pair) == CENTROID_ATOMNUM
   || pair_atom_index(2,pair) == CENTROID_ATOMNUM)) return false;

	return true;
}

///////////////////////////
/// Public: Coordinates ///
///////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin get_XYZ
///
/// @brief
///
/// @detailed
///
/// @param  residue - [in/out]? -
/// @param  type_index - [in/out]? -
/// @param  xyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::get_XYZ(
	int & residue,
	int & type_index,
	FArray1Da_float xyz
)
{
	using namespace param;
	using namespace classical_constraints::constants;

	FArray3Dp_float const & Eposition = *Eposition_; // yab: misc removal
	FArray2D_float const & centroid = *centroid_; // yab: misc removal
	FArray3D_float const & full_coord = *full_coord_; // yab: misc removal
	FArray1D_int const & res = *res_; // yab: misc removal

	xyz.dimension( 3 );

	if ( type_index == H_ATOMNUM ) {
		compute_HN_position(Eposition,res,residue,xyz);
	} else if ( type_index == HA_ATOMNUM ) { // HA or 2HA
		compute_HCA_position(Eposition,residue,xyz);
	} else if ( type_index == HA2_ATOMNUM ) {  // 3HA
		compute_hxyz(-2,Eposition(1,1,residue),Eposition(1,2,residue),
		 Eposition(1,4,residue),xyz);
	} else if ( type_index >= 1 && type_index <= 5 ) { // bb heavyatom
		xyz(1) = Eposition(1,type_index,residue);
		xyz(2) = Eposition(2,type_index,residue);
		xyz(3) = Eposition(3,type_index,residue);
	} else if ( type_index == CENTROID_ATOMNUM ) {           // centroid
		xyz(1) = centroid(1,residue);
		xyz(2) = centroid(2,residue);
		xyz(3) = centroid(3,residue);
	} else if ( type_index >= 6 && type_index <= MAX_ATOM() ) { // sc heavyatom
		xyz(1) = full_coord(1,type_index,residue);
		xyz(2) = full_coord(2,type_index,residue);
		xyz(3) = full_coord(3,type_index,residue);
	} else {
		std::cout << "get_XYZ does not recognize type_index " << type_index <<
		 std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

////////////////////////////////////////////////
/// Public: Evaluation, Scoring, Derivatives ///
////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_pairConstraint
///
/// @brief
///
/// @detailed
/// mode<0: use ALL constraints:  stage independent,(select centroid/fullatom)
/// mode>=0:  evaluate constraints dependent on stage, centroid or fullatom
///car mode0  don't eval score, but update dis and dis2 (for deriv)
///car mode1 b-r, 10A maximum violations penalty
///car mode2 b-r, no maximum violation penalty
///car mode3  CNS function, quadratic for small violations, linear for large
///car mode4 (b * b-dis * dis)
/// stage allows subsets of the constraints to be scored. each constraint
/// has a stage number.  Only constraints with a number less than or equal to
/// the stage are used in evaluation.
/// stage number defined as sequence separation
///
/// @param  mode - [in/out]? -
/// @param  score_x - [in/out]? - ret'd copy of score, = pc_score usually
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::eval_pairConstraint(
	int mode,
	float & score_x // ret'd copy of score, = pc_score usually
)
{
//car atomtypes: 1-5 = bb heavyatom, >5 = sc heavyatom, <1 = H, -999 = centroid
//car need to add lower bounds!! debug

	float dist,dist2;
	float energy=0.;

	FArray1D_int const & res = *res_;
	FArray1D_int const & res_variant = *res_variant_;

	if ( std::abs(mode) > 4 ) {
		std::cout << "mode " << mode << " is undefined in eval_pairConstraint" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

// check and reset flag
// yab: why is this done...?  Don't have time to check, reproducing behavior...
	bool const verbose_inner = verbose;
	verbose = false;
// initialize
	score_x = 0.0;

	if ( !classical_constraints::BOUNDARY::get_constraints_exist() ) return;

// recall variable from namespace
	if ( np_pairs == 0 ) return;
	int const stage = ( mode >= 0 ? max_seqSep : 9999 );
	if ( stage <= 0 ) return;

	bool const fullatom = get_fullatom_flag();
// loop over pair constraints, find distances, sum up violations

	int nscored = 0; // number of cst's actually scored
	 // (ie after checking noe_stage, sc v. centroid)

	for ( int pair = 1; pair <= np_pairs; ++pair ) {

//car check for conditions not to evaluate:
		if ( cst_pair_satisfies_conditions(pair,fullatom,stage) ) {

//car accumulate score
			++nscored;

//car retrieve coords
			FArray1D_float xyz1(3), xyz2(3);
			int noe_state = get_noe_state( pair );
			int aa1 = res( constraintPair( pair, 1 ) );
			int aa2 = res( constraintPair( pair, 2 ) );
			int aav1 = res_variant( aa1 );
			int aav2 = res_variant( aa2 );
			float eff_dist, eff_dist2, sum_inv;

			assert( noe_state <= 3 );
			if ( noe_state == 0 ) {
  		get_noe_coords(pair,xyz1,xyz2);
			noe_dist( xyz1, xyz2, dist, dist2 );
			//			calc_cst_score(mode,xyz1,xyz2,pairRadius(pair),pairRadius2(pair),
			//										pairMinRadius(pair),dist,dist2,energy);
			calc_cst_score( mode, dist, dist2, pairRadius(pair), pairRadius2(pair), pairMinRadius(pair), energy );
			} else if ( noe_state > 0 && noe_state <= 3 ) {
				ambiguous_noe_cst_eff_dist( pair, noe_state, constraintPair(pair, 1), constraintPair(pair, 2), aa1, aa2, aav1, aav2, eff_dist, eff_dist2, sum_inv );
				calc_cst_score( mode, eff_dist, eff_dist2, pairRadius(pair), pairRadius2(pair), pairMinRadius(pair), energy);
				dist = eff_dist;
				dist2 = eff_dist2;
			}
			int i = constraintPair(pair,1);
			int j = constraintPair(pair,2);
			score_x += energy*cst_res_wt(i)*cst_res_wt(j);
			dist_constraint(pair) = dist; // store in namespace for diagnostics
			dist_constraint2(pair) = dist2;
			score_constraint(pair) = score_x;
		}// skip to here for skipped constraints
	}

	if ( verbose_inner ) output_status( mode, std::cout );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin precalc_cst_deriv
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::precalc_cst_deriv()
{
	using namespace scorefxns;

	cst_drv_mode = std::abs(pc_mode);
//car mode1,mode2
	if ( cst_drv_mode == 1 || cst_drv_mode == 2 ) {
		cst_drv_factor = pc_weight;
	} else if ( cst_drv_mode == 3 ) {
//car cns potential
//car can't put 2 in cause it is distance-dependent
		cst_drv_factor = pc_weight;

	} else if ( cst_drv_mode == 4 ) {
		cst_drv_factor = pc_weight*2.0;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin eval_constraint_deriv
///
/// @brief
///
/// @detailed
///
/// @param  i - [in/out]? -
/// @param  ii - [in/out]? -
/// @param  j - [in/out]? -
/// @param  jj - [in/out]? -
/// @param  fullatom - [in/out]? -
/// @param  deriv - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::eval_constraint_deriv(
	int const & i,
	int const & ii,
	int const & j,
	int const & jj,
	bool const & fullatom,
	float const & dis,
	float const & dis2,
	float & deriv
)
{
	using namespace classical_constraints::constants;
	using namespace scorefxns::scorefxn_parameters;
	//	using namespace misc;

//car local
	int num;
	int pair;

	int atom1,res1,atom2,res2;
	int ilist; // position in the constraint list for res i and j
	//	int noe_state;
	deriv = 0.0;
	float eff_dist, eff_dist2, sum_inv;
	FArray1D_int const & res = *res_;
	FArray1D_int const & res_variant = *res_variant_;

	if ( !classical_constraints::BOUNDARY::get_constraints_exist() ) return;

//car check this first, cause it will exclude most
	num = n_cst(i,j);
	if ( num == 0 ) return;

	int const stage = ( pc_mode >= 0 ? max_seqSep : 9999 );
	if ( std::abs(i-j) > stage ) return; // too longrange


	for ( ilist = 1; ilist <= num; ++ilist ) {
		pair = constraint_list(ilist,i,j);
		res1 = constraintPair(pair,1);
		res2 = constraintPair(pair,2);
		if ( !cst_pair_satisfies_conditions(pair,fullatom,stage) )  goto L100;

//car check atoms
		if ( res1 == i ) {
			atom1 = pair_atom_index(1,pair);
			if ( atom1 == CENTROID_ATOMNUM ) atom1 = 6; // centroids store as  atom 6
			if ( atom1 != ii ) goto L100;
			atom2 = pair_atom_index(2,pair);
			if ( atom2 == CENTROID_ATOMNUM ) atom2 = 6; // centroids stored as atom 6
			if ( atom2 != jj ) goto L100;
		} else {         // res2 is i
			atom2 = pair_atom_index(2,pair);
			if ( atom2 == CENTROID_ATOMNUM ) atom2 = 6; // centroids stored as atom 6
			if ( atom2 != ii ) goto L100;
			atom1 = pair_atom_index(1,pair);
			if ( atom1 == CENTROID_ATOMNUM ) atom1 = 6; // centroids stored as atom 6
			if ( atom1 != jj ) goto L100;
		}


		//	for( std::map < int, std::pair< float, float > >::const_iterator it = pair_effdist2_suminv.begin(), it_end = pair_effdist2_suminv.end(); it != it_end; ++it ) {
		//		std::cout << "pair itt " << pair << it->first << " " << it->second.first << " " << it->second.second << std::endl;
		//	}
		if( get_noe_state(pair) != 0 ){
			ambiguous_noe_cst_eff_dist(pair, get_noe_state( pair ), i,j,res(i),res(j),res_variant(i),res_variant(j),eff_dist, eff_dist2, sum_inv);


			//		  new_atom1 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum( atom1, new_aa1, res(i), res_variant(i) );
			//      new_atom2 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum( atom2, new_aa2, res(j), res_variant(j) );
			//			std::cout << "BB noe_state " << pair << " "  << get_noe_state( pair )<<  " " << i << " " << j << " " << ii << " " << jj << " " << eff_dist << " " << sum_inv << std::endl;

			//			calc_ambig_cst_drv(noe_state,pair,i,ii,j,jj,dis,dis2,deriv); // found the right constraint!
			calc_ambig_cst_drv_1(pair, get_noe_state( pair ), eff_dist, sum_inv, dis, dis2, deriv);
		} else {
			calc_cst_drv(pair,dis,dis2,deriv); // found the right constraint!
		}
		//		std::cout << "drv check pair " << pair << " noe_state " << get_noe_state( pair ) << " " << deriv << std::endl;

//		std::cout << "cst_deriv" <<
//		 I( 4, i ) << I( 4, ii ) << I( 4, j ) << I( 4, jj ) <<
//		 I( 4, res1 ) << I( 4, atom1 ) << I( 4, res2 ) <<
//		 I( 4, atom2 ) << F( 8, 3, deriv ) << std::endl;
		return;   //only one constraint between an atom pair allowed so stop looking after 1
L100:;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_res_res_cstE
///
/// @brief  return constraint score between two residues
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
// res1,res2 -> input query residues
// energy (output) is the the summed constraint energy between all the atoms,
// for which constraint exist, between the query residues
// if any of bb1, sc1, bb2, or sc2 are false, constraints involving,
// for example if ! bb1, backbone of residue 1 will be excluded
///
/// @references
///
/// @authors car, sraman
///
/// @last_modified Jan 15th 2007
////////////////////////////////////////////////////////////////////////////////
float
Constraints::get_res_res_cstE(
	int const res1,
	int const res2,
	int const aa1,
	int const aav1,
	int const aa2,
	int const aav2,
	FArray2Da_float coord1,
	FArray2Da_float coord2,
	bool const bb1,
	bool const sc1,
	bool const bb2,
	bool const sc2
)
{

 using namespace param;
 using namespace scorefxns;

 coord1.dimension( 3, MAX_ATOM() );
 coord2.dimension( 3, MAX_ATOM() );

 if ( !classical_constraints::BOUNDARY::get_constraints_exist() ) return 0.0;
 if ( pc_weight == 0.0 ) return 0.0;

 int stage = max_seqSep;
 if ( pc_mode >= 0 ) stage = 9999;
 if ( std::abs(res1-res2) > stage ) return 0.0;
 int num = n_cst(res1,res2);
 if (num == 0) return 0.0;

 int pair, cst_aa1, cst_aa2;
 int cst_atom1,cst_atom2;
 int atom1,atom2;
 float dist,dist2;
 float score_x=0.;
 bool is_bb;
 float energy = 0.0;
 float eff_dist, eff_dist2, sum_inv;


 for ( int ilist = 1; ilist <= num; ++ilist ) {
	 pair = constraint_list(ilist,res1,res2);
	 if ( constraintPair(pair,1) == res1 ) {
		 cst_atom1 = pair_atom_index(1,pair);
		 cst_atom2 = pair_atom_index(2,pair);
		 cst_aa1 = constraintPair(pair,3);
		 cst_aa2 = constraintPair(pair,4);
	 } else {
		 cst_aa1 = constraintPair(pair,4);
		 cst_aa2 = constraintPair(pair,3);
		 cst_atom2 = pair_atom_index(1,pair);
		 cst_atom1 = pair_atom_index(2,pair);
	 }
	 if ( ! cst_pair_satisfies_conditions(pair,true,stage)) continue;

	 atom1 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum(cst_atom1,cst_aa1,aa1,aav1);
	 atom2 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum(cst_atom2,cst_aa2,aa2,aav2);

	 is_bb = atom_is_backbone(atom1,aa1,aav1);
	 if ( ! bb1 && is_bb ) continue;
	 if ( ! sc1 && ! is_bb) continue;
	 is_bb = atom_is_backbone(atom2,aa2,aav2);
	 if ( ! bb2 && is_bb ) continue;
	 if ( ! sc2 && ! is_bb) continue;

//CAR  11/16/2004
// for HN, HA, HA2, coordinates determined using the fullatom routines
// differ from those determined by get_noe_coord, leading to slightly different
// constraints scores when evaluating using get_res_res_cstE and
// eval_pairConstraint.  This discrepancy could be avoided by computing
// the HN and HA positions using the functions called by get_XYZ (requires
// changes to some function args to pass the needed information, and may
// be slow since _MANY_ H coords will be computed).  However,
// then the constraints scores calculated would not actually agree with the
// output pdb files as pdb output uses the coordinates determined by packer.
// A better solution would be to make get_XYZ compute HN and HA coords
// consistent with the sidechain templates used by the fullatom routines
// This requires using the amino-acid-specific templates defined in read_
// aaproperties.cc

//  check_ambiguous_constraint( pair, aav1, aav2 );

	 FArray1D_float xyz1( 3 ), xyz2( 3 );

	 xyz1( 1 ) = coord1( 1, atom1 );
	 xyz2( 1 ) = coord2( 1, atom2 );

	 xyz1( 2 ) = coord1( 2, atom1 );
	 xyz2( 2 ) = coord2( 2, atom2 );

	 xyz1( 3 ) = coord1( 3, atom1 );
	 xyz2( 3 ) = coord2( 3, atom2 );

	 if( get_noe_state( pair ) == 0 ) {

		 noe_dist( xyz1, xyz2, dist, dist2 );
		 calc_cst_score(pc_mode, dist, dist2, pairRadius(pair),pairRadius2(pair),pairMinRadius(pair),score_x);

	 } else {
//vats Uncomment the following line once the derivatives are computed for the ambiguous restraints
		 ambiguous_noe_cst_eff_dist( pair, get_noe_state( pair ), res1, res2, aa1, aa2, aav1, aav2, eff_dist, eff_dist2, sum_inv);
		 //		 noe_dist( xyz1, xyz2, dist, dist2 );
		 calc_cst_score(pc_mode, eff_dist, eff_dist2, pairRadius(pair),pairRadius2(pair),pairMinRadius(pair),score_x);
		 dist = eff_dist;
		 dist2 = eff_dist2;
	 }
	 dist_constraint(pair) = dist; // store in namespace for diagnostic output
	 dist_constraint2(pair) = dist2;
	 score_constraint(pair) = score_x;
	 energy += score_x;
 }
 energy = pc_weight*energy*cst_res_wt(res1)*cst_res_wt(res2);

 //	for( std::map < int, std::pair< float, float > >::const_iterator it = pair_effdist2_suminv.begin(), it_end = pair_effdist2_suminv.end(); it != it_end; ++it ) {
 //		std::cout << "pair it " << it->first << " " << it->second.first << " " << it->second.second << std::endl;
 //	}

 return energy;
}

////////////////////
/// Public: NOEs ///
////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin noe_dist
///
/// @brief
///
/// @detailed
///
/// @param  xyz1 - [in] -
/// @param  xyz2 - [in] -
/// @param  dis - [in/out]? -
/// @param  dis2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::noe_dist(
	FArray1DB_float const & xyz1,
	FArray1DB_float const & xyz2,
	float & dis,
	float & dis2
)
{
	noe_dist2(xyz1,xyz2,dis2);
	dis = std::sqrt(dis2);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin noe_dist2
///
/// @brief
///
/// @detailed
///
/// @param  xyz1 - [in] -
/// @param  xyz2 - [in] -
/// @param  dis2 - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::noe_dist2(
	FArray1DB_float const & xyz1,
	FArray1DB_float const & xyz2,
	float & dis2
)
{
	double const dif1 = xyz1(1) - xyz2(1);
	double const dif2 = xyz1(2) - xyz2(2);
	double const dif3 = xyz1(3) - xyz2(3);
	dis2 = ( dif1 * dif1 ) + ( dif2 * dif2 ) + ( dif3 * dif3 );
}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_noe_coord
///
/// @brief
///
/// @detailed
///
/// @param  pair - [in] -
/// @param  xyz1 - [out] -
/// @param  xyz2 - [out] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::get_noe_coords(
	int pair,
	FArray1Da_float xyz1,
	FArray1Da_float xyz2
)
{
	xyz1.dimension( 3 );
	xyz2.dimension( 3 );

	int i = constraintPair(pair,1);
	int j = constraintPair(pair,2);
	get_XYZ(i,pair_atom_index(1,pair),xyz1);
	get_XYZ(j,pair_atom_index(2,pair),xyz2);
}

////////////////////////
/// Private: Methods ///
////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin build_constraint_list
///
/// @brief
///car for a given residue, constraint_list(i,residue) is a list of the
///car constraints that involve this residue. A constraint number of zero
///car indicates the end of the list
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::build_constraint_list()
{
	using namespace classical_constraints::constants;

	int const & total_residue = *total_residue_; // yab: misc removal

	for ( int i = 1; i <= total_residue; ++i ) {
		for ( int j = 1; j <= total_residue; ++j ) {
			n_cst(i,j) = 0;
		}
	}
	for ( int i = 1; i <= np_pairs; ++i ) {
		int res1 = constraintPair(i,1);
		int res2 = constraintPair(i,2);
		if ( n_cst(res1,res2) == MAX_CST_LIST ) {
			std::cout << "increase MAX_CST_LIST in constraints.cc" << std::endl;
			std::cout << "MAX_CST_LIST = " << MAX_CST_LIST << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		++n_cst(res1,res2);
		constraint_list(n_cst(res1,res2),res1,res2) = i;
		if ( res1 != res2 ) {
			++n_cst(res2,res1);
			constraint_list(n_cst(res1,res2),res2,res1) = i;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin add_centroid_constraints
///
/// @brief
///car add centroid constraints corresponding to sc-sc and bb-sc constraints
///car for use before fullatom coordinates are in use
///car if any centroid constraints already exist in the input file, no new
///car centroid constraints are generated.
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::add_centroid_constraints()
{
	using namespace classical_constraints::functions;

	std::string atom_name( " CEN" );

	float const centroid_bound = { 10.0 };

	int atom1;
	int atom2;
	float bound = centroid_bound;

	int j = np_pairs;
	for ( int i = 1; i <= np_pairs; ++i ) {
		if ( classical_constraints::functions::index2type( pair_atom_index(1,i), constraintPair(i,3) ) == " CEN" ) return;
		if ( classical_constraints::functions::index2type( pair_atom_index(2,i), constraintPair(i,4) ) == " CEN" ) return;
		if ( pairSeqSep(i) < 1 ) goto L100;

		atom1 = type2index( atom_name, constraintPair(i,3) );
		atom2 = type2index( atom_name, constraintPair(i,4) );
		bound = centroid_bound;
		if ( pair_atom_index(1,i) >= -2 && pair_atom_index(1,i) <= 5 ) {
			if ( pair_atom_index(2,i) >= -2 && pair_atom_index(2,i) <= 5 ) {
				goto L100; // both are bb atoms
			} else {
				atom1 = pair_atom_index(1,i);
				bound = centroid_bound; // different?
			}
		} else if ( pair_atom_index(2,i) >= -2 && pair_atom_index(2,i) <= 5 ) {
			atom2 = pair_atom_index(2,i);
			bound = centroid_bound;
		}
		int k;
		for ( k = np_pairs+1; k <= j; ++k ) {
			if ( constraintPair(i,1) == constraintPair(k,1) &&
			 constraintPair(i,2) == constraintPair(k,2) ) {
				if ( pair_atom_index(1,k) == atom1 && pair_atom_index(2,k) == atom2 )
				 goto L100;
			} else if ( constraintPair(i,1) == constraintPair(k,2) &&
			 constraintPair(i,2) == constraintPair(k,1) ) {
				if ( pair_atom_index(1,k) == atom2 && pair_atom_index(2,k) == atom1 )
				 goto L100;
			}
		}

		++j; // add centroid constraint
		if ( j > classical_constraints::constants::MAX_CONSTRAINTS ) {
			std::cout << "WARNING: increase MAX_CONSTRAINTS in constraints.cc"
								<< std::endl;
			std::cout << " MAX_CONSTRAINTS: " << classical_constraints::constants::MAX_CONSTRAINTS <<
				" total_constraints: " << j << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		constraintPair(j,1) = constraintPair(i,1);
		constraintPair(j,2) = constraintPair(i,2);
		pairSeqSep(j) = std::abs(constraintPair(j,1)-constraintPair(j,2));
		constraintPair(j,3) = constraintPair(i,3);
		constraintPair(j,4) = constraintPair(i,4);
		pairMinRadius(j) = pairMinRadius(i);
		pairRadius(j) = bound;
		pairRadius2(j) = bound*bound;
		pair_atom_index(1,j) = atom1;
		pair_atom_index(2,j) = atom2;
L100:;
	}
	np_pairs = j;

}

/////////////////////////////////////
/// Private: Derivatives, Scoring ///
/////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_cst_drv
///
/// @brief
///
/// @detailed
///
/// @param  pair - [in/out]? -
/// @param  deriv - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::calc_cst_drv(
	int const & pair,
	float const & dist,
	float const & dist2,
	float & deriv
)
{
//car parameters for mode 3 function (CNS-type)
//car rswitch: switch functional form when violations=rswitch
//car rswitch_offset:  offset for function continuity;
//CAR THESE PARAMETERS MUST BE IDENTICAL IN calc_cst_score AND calc_cst_drv !!
	float const rswitch = { 0.5 };
//	float const rswitch_offset = { ( rswitch * rswitch ) - rswitch };

//car local
	float over_dist;
	float upper_bound,lower_bound;

	assert(dist2 >= 0.0 && dist >= 0.0);  //should be known when called
//car mode1,mode2
	if ( cst_drv_mode == 1 || cst_drv_mode == 2 ) {
		over_dist = dist - pairRadius(pair);
		if ( over_dist > 0 ) deriv = cst_drv_factor; // 1.0*weight

	} else if ( cst_drv_mode == 3 ) {
//car cns potential
		upper_bound = pairRadius(pair);
		lower_bound = pairMinRadius(pair);
		if ( dist > (upper_bound + rswitch) ) {
			deriv = cst_drv_factor; // 1.0*weight
		} else if ( lower_bound <= dist && dist <= upper_bound ) {
			deriv = 0.0;
		} else if ( dist < lower_bound ) {
			deriv = -cst_drv_factor; // -1.0*weight
		} else {  // d between pairRadius and pairRadius+rsw
// can't put 2.0 in precalc factor cause it's distance-dependent
			deriv = 2.0*(dist-upper_bound)*cst_drv_factor;
		}

	} else if ( cst_drv_mode == 4 ) {
//car always quadratic
		over_dist = dist2 - pairRadius2(pair);
		if ( over_dist > 0 ) deriv = cst_drv_factor*dist;

	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calc_cst_score
///
/// @brief
///
/// @detailed
///
/// @param  mode_in - [in] -
/// @param  dist - [out] -
/// @param  dist2 - [out]
/// @param  pairRadius - [in] -
/// @param  pairRadius2 - [in] -
/// @param  pairMinRadius - [in] -
/// @param  energy - [out] - unscaled energy
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors car, sraman
///
/// @last_modified Jan 15th 2007
////////////////////////////////////////////////////////////////////////////////
void
Constraints::calc_cst_score(
	int mode_in,
	float dist,
	float dist2,
	float & pairRadius,
	float & pairRadius2,
	float & pairMinRadius,
	float & energy // unscaled energy
)
{

//car parameters for mode 3 function (CNS-type)
//car rswitch: switch functional form when violations=rswitch
//car rswitch_offset:  offset for function continuity;
//CAR THESE PARAMETERS MUST BE IDENTICAL IN calc_cst_score AND calc_cst_drv !!
	float const rswitch = { 0.5 };
	float const rswitch_offset = { ( rswitch * rswitch ) - rswitch };

//car local
	float over_dist;

	int const mode = std::abs(mode_in);
	if ( mode < 3 ) {  // linear w/ and w/out cutoff
		over_dist = std::max( dist - pairRadius, 0.0f );
		if ( mode == 1 ) {
			energy = std::min( 10.0f, over_dist );
		} else if ( mode == 2 ) {
			energy = over_dist;
		}
	} else if ( mode == 3 ) {
		if ( dist > pairRadius ) {
			over_dist = dist - pairRadius;
		} else if ( pairMinRadius <= dist && dist <= pairRadius ) {
			over_dist = 0.0;
		} else if ( dist < pairMinRadius ) {
			over_dist = pairMinRadius - dist;
		} else {
			over_dist = 0.0;
			std::cout << "WARNING: over_dist undefined and set to 0.0 !!!!" << std::endl;
		}
		if ( dist > pairRadius + rswitch ) {
			energy = rswitch_offset + over_dist;
		} else {
			energy = over_dist * over_dist;
		}
	} else {
		dist = -1.0; // tag as unevaluated
		over_dist = std::max( dist2 - pairRadius2, 0.0f );
		energy = over_dist;
	}

}

////////////////////////////////
/// Private: Proton Geometry ///
////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin compute_HN_position
///
/// @brief
///compute HN coordinates; return coords and store in global variable
///car prolines return N coordinate rather than non-existent HN
///car residue 1 return N coordinates rather than HN
///
/// @detailed
///
/// @param  Eposition - [in/out]? -
/// @param  res - [in/out]? -
/// @param  ires - [in/out]? -
/// @param  HN_xyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::compute_HN_position(
	FArray3Da_float Eposition,
	FArray1Da_int res,
	int & ires,
	FArray1Da_float HN_xyz
)
{
	using namespace param;

	Eposition.dimension( 3, MAX_POS, MAX_RES() );
	res.dimension( MAX_RES() );
	HN_xyz.dimension( 3 );

	if ( ires == 1 || res(ires) == 13 ) {
		for ( int i = 1; i <= 3; ++i ) {
			HN_xyz(i) = Eposition(i,1,ires);
		}
	} else {
		get_HN_coords(Eposition(1,4,ires-1),Eposition(1,1,ires),Eposition(1,2,ires),
		 HN_xyz);
	}
//$$$    compute_hxyz(0,Eposition(1,4,ires-1),Eposition(1,1,ires),
//$$$     Eposition(1,2,ires),HN_xyz); // C' is atom 4 in rosetta scheme

//car save in global variable
	HN_position(1,ires) = HN_xyz(1);
	HN_position(2,ires) = HN_xyz(2);
	HN_position(3,ires) = HN_xyz(3);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin compute_HCA_position
///
/// @brief
///car compute HA coordinates for residue ires and store in global arrays
///
/// @detailed
///
/// @param  Eposition - [in/out]? -
/// @param  ires - [in/out]? -
/// @param  HA_xyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::compute_HCA_position(
	FArray3Da_float Eposition,
	int & ires,
	FArray1Da_float HA_xyz
)
{
	using namespace param;

	Eposition.dimension( 3, MAX_POS, MAX_RES() );
	HA_xyz.dimension( 3 );

	get_HA_coord(Eposition(1,1,ires),Eposition(1,2,ires),Eposition(1,4,ires),
	 HA_xyz);

//$$$   compute_hxyz(-1,Eposition(1,1,ires),Eposition(1,2,ires),
//$$$    Eposition(1,4,ires),HA_xyz); // C' is atom 4 in rosetta scheme

//car save in global variable
	HA_position(1,ires) = HA_xyz(1);
	HA_position(2,ires) = HA_xyz(2);
	HA_position(3,ires) = HA_xyz(3);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_HA_coord
///
/// @brief
///   computes the location of the backbone calpha's hydrogen atom location.
///   find bisector of N-Ca-C angle ... we then comput a 60 degree angle
///   to this plane along this bisector. we put H atom 1.08 ansgtoms out along this.
///
/// @detailed
///
///car note this returns HA1 for glycine
///
/// @param  Nxyz - [in/out]? -
/// @param  CAxyz - [in/out]? -
/// @param  Cxyz - [in/out]? -
/// @param  HAxyz - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
Constraints::get_HA_coord(
	FArray1Da_float Nxyz,
	FArray1Da_float CAxyz,
	FArray1Da_float Cxyz,
	FArray1Da_float HAxyz
)
{
	Nxyz.dimension( 3 );
	CAxyz.dimension( 3 );
	Cxyz.dimension( 3 );
	HAxyz.dimension( 3 );

//car parameters
	float const bond_length = { 1.08 };

//car local
	FArray1D_float xyz1( 3 );
	FArray1D_float xyz2( 3 );

	float norm1 = 0.0;
	float norm2 = 0.0;
	float xyz_tmp;
	for ( int i = 1; i <= 3; ++i ) {
		xyz_tmp = CAxyz(i) - Cxyz(i);
		xyz1(i) = xyz_tmp;
		norm1 += xyz_tmp * xyz_tmp;
		xyz_tmp = CAxyz(i) - Nxyz(i);
		xyz2(i) = xyz_tmp;
		norm2 += xyz_tmp * xyz_tmp;

	}

	norm1 = 1.0/std::sqrt(norm1);
	norm2 = 1.0/std::sqrt(norm2);

	float norm3 = 0.0;
	for ( int i = 1; i <= 3; ++i ) {
		xyz1(i) *= norm1; // unit vec
		xyz2(i) *= norm2; // unit vec
		xyz_tmp = xyz1(i) + xyz2(i);
		norm3 += xyz_tmp * xyz_tmp; // length of sum
	}

	norm3 = 0.5/std::sqrt(norm3); // normalize to cos(60) = 0.5

	HAxyz(1) = xyz1(2)*xyz2(3) - xyz1(3)*xyz2(2);
	HAxyz(2) = -xyz1(1)*xyz2(3) + xyz1(3)*xyz2(1);
	HAxyz(3) = xyz1(1)*xyz2(2) - xyz1(2)*xyz2(1);
//car normalize to root 3 over 2 length (sin (60))
	norm2 = std::sqrt( 3.0 /
	 ( ( HAxyz(1) * HAxyz(1) ) +
	 ( HAxyz(2) * HAxyz(2) ) +
	 ( HAxyz(3) * HAxyz(3) ) ) ) / 2.0;

//   combine unit vector in plane with out of plane cross product in
//car ratio to make 60 degree angle  offset from ca
	for ( int i = 1; i <= 3; ++i ) {
		HAxyz(i) = CAxyz(i) +
		 bond_length*( (xyz1(i)+xyz2(i))*norm3 + HAxyz(i)*norm2 );
	}

}

///////////////////////////////
/// Private: Ambiguous NOEs ///
///////////////////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin build_ambiguous_constraint_list
///
/// @brief  Builds a constraints list that contains ambiguously
/// assigned proton distance restraints. These are atoms in the constraint file
/// that start with "#"
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Jan 15th 2007
///////////////////////////////////////////////////////////////////////////////
void
Constraints::build_ambiguous_constraint_list(
	int k,
	std::string cstatom1,
	std::string cstatom2
)
{
	using namespace classical_constraints::functions;
	bool res1_is_ambig ( false );
	bool res2_is_ambig ( false );
	int noe_state( 0 );

	if ( index2type( pair_atom_index(1,k), constraintPair(k,3) ) != " CEN" &&
			 index2type( pair_atom_index(2,k), constraintPair(k,4) ) != " CEN" ) {
		if( cstatom1[0] == '#' ) {
			ambi_res1.push_back( k );
			res1_is_ambig = true;
		}
		if( cstatom2[0] == '#' ) {
			ambi_res2.push_back( k );
			res2_is_ambig = true;
		}
	}

	if( res1_is_ambig && !res2_is_ambig ) noe_state = 1;
	if( !res1_is_ambig && res2_is_ambig ) noe_state = 2;
	if( res1_is_ambig && res2_is_ambig ) noe_state = 3;

	noe_state_list[k] = noe_state;
	//	std::cout << "build_ambig pair " << k << " " << noe_state << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_noe_state
///
/// @brief  Checks if a given constraintPair contains an ambiguous
/// distance restraint. If constraintPair( pair, 1 ) is ambiguous
/// noe_state = 1, if constraintPair( pair, 2 ) is ambiguous noe_state = 2.
/// If both are ambiguous, noe_state = 3.
///
/// @detailed
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Jan 15th 2007
///////////////////////////////////////////////////////////////////////////////
int
Constraints::get_noe_state(
	 int pair
)

{
  int noe_state = 0;
	std::map <int, int>::iterator it = noe_state_list.find( pair );
	if ( it != noe_state_list.end () ) {
		noe_state = noe_state_list[ pair ];
	}
	//	std::cout << " get_noe_state pair " << pair << " " << noe_state << std::endl;

	return noe_state;


	/*
	std::list < int >::iterator iter_res1;
	std::list < int >::iterator iter_res2;
	bool pos1_ambig( false );
	bool pos2_ambig( false );

	std::string res1;
	std::string res2;
	std::string cst_atom1;
	std::string cst_atom2;

	iter_res1 = find( ambi_res1.begin(), ambi_res1.end(), pair );
	if ( iter_res1 != ambi_res1.end() ) {
		pos1_ambig = true;
		noe_state = 1;
		//		return pos1_ambig;
	}


	iter_res2 = find( ambi_res2.begin(), ambi_res2.end(), pair );
	if ( iter_res2 != ambi_res2.end() ) {
		pos2_ambig = true;
		noe_state = 2;
		//		return pos2_ambig;
	}

	if( pos1_ambig && pos2_ambig ) {
		noe_state = 3;
		return true;

	}

	if( pos1_ambig || pos2_ambig ) return true;

	return false;
	*/
}

////////////////////////////////////////////////////////////////////////////////
/// @begin ambiguous_noe_cst_score
///
/// @brief
///
/// @detailed For a given constraintPair, this routine classifies it as
/// noe_state 1, 2 or 3. For the ambiguous atom position, it finds all the
/// associated degenerate protons, computes the 1/r6 sum average distance.
///          --
/// 1/reff = \  [1/(rij)^6]^(-1/6)
///          /
///          --
///         i,j
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Jan 15th 2007
///////////////////////////////////////////////////////////////////////////////
void
Constraints::ambiguous_noe_cst_eff_dist(
	 int pair,
	 int noe_state,
	 int res1,
	 int res2,
	 int aa1,
	 int aa2,
	 int aav1,
   int aav2,
	 float & eff_dist,
	 float & eff_dist2,
	 float & sum_inv
	 //   float &
)

{
	using namespace aaproperties_pack::properties_atom_nbr_in_aa_aav;

	FArray3D_float const & full_coord = *full_coord_; // yab: misc removal

	std::list < int >::iterator iter_res1;
	std::list < int >::iterator iter_res2;

	int cst_atom1, cst_atom2, cst_aa1, cst_aa2;
	int atom1, atom2;
	std::string nb_atom1, nb_atom2;
	std::string test1, test2;
	float dist2;
	FArray1D_float xyz1( 3 ),xyz2( 3 );
	std::vector < float > dist_sq;
	//	std::map < int, std::pair< float, float > >  pair_effdist2_suminv;
	dist_sq.clear();

	//	std::cout << "ambig check entry " << constraintPair( pair, 1 ) << " " << res1 << " " << noe_state << std::endl;
	if( constraintPair( pair, 1 ) == res1 ){
		cst_atom1 = pair_atom_index( 1, pair );//change this 1, pair
		cst_atom2 = pair_atom_index( 2, pair );//change this 2, pair
		cst_aa1 = constraintPair( pair, 3 );
		cst_aa2 = constraintPair( pair, 4 );
	} else {
		cst_aa1 = constraintPair( pair, 4 );
		cst_aa2 = constraintPair( pair, 3 );
		cst_atom2 = pair_atom_index( 1, pair );
		cst_atom1 = pair_atom_index( 2, pair );
		//vats res1 and res2 have flipped from the packer. Now res1 = constraintPair( pair, 4 ) and res2 = constraintPair( pair, 3 ).
//vats Therefore the previous assignment of noe_state 1 and 2 have to be changed because the nbonded_neighbor iterator will have
//vats to loop over the correct heavyatom, i.e., the one connected to the degenerate proton.
//		if( noe_state == 1 ) noe_state = 2;
//		if( noe_state == 2 ) noe_state = 1;
		noe_state = switch_noe_state( noe_state );
		//		noe_state = test_noe;
	}
	//	std::cout << "ambig check exit " << constraintPair( pair, 1 ) << " " << res1 << " " << noe_state << std::endl;

	atom1 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum( cst_atom1, cst_aa1, aa1, aav1 );
	atom2 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum( cst_atom2, cst_aa2, aa2, aav2 );

	atom_name_from_atom_num( atom1, cst_aa1, aav1, test1 );
	atom_name_from_atom_num( atom2, cst_aa2, aav2, test2 );

	assert( noe_state > 0 );


	if( noe_state == 1) {
		for( int i = 1; i <= nbonded_neighbors( cst_atom1, cst_aa1, aav1 ); ++i ) {
			atom_name_from_atom_num( bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), cst_aa1, aav1, nb_atom1 );
			//			std::cout << "state 1 pair " << pair << " " << cst_aa1 << " " << cst_atom1 << " DD atom type " << nb_atom1 << std::endl;
			if( nb_atom1.rfind("H") != std::string::npos ) {
				xyz1( 1 ) = full_coord( 1, bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), res1 );
				xyz1( 2 ) = full_coord( 2, bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), res1 );
				xyz1( 3 ) = full_coord( 3, bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), res1 );

				xyz2( 1 ) = full_coord( 1, atom2 , res2 );
				xyz2( 2 ) = full_coord( 2, atom2 , res2 );
				xyz2( 3 ) = full_coord( 3, atom2 , res2 );

				noe_dist2( xyz1, xyz2, dist2 );
				dist_sq.push_back( dist2 );
			}
		}
	} else if ( noe_state == 2 ) {
		for( int i = 1; i <= nbonded_neighbors( cst_atom2, cst_aa2, aav2 ); ++i ) {
			atom_name_from_atom_num( bonded_neighbor( i, cst_atom2, cst_aa2, aav2 ), cst_aa2, aav2, nb_atom2 );
			if( nb_atom2.rfind("H") != std::string::npos ) {
				//			std::cout << "state 2 pair " << pair << " " << cst_aa1 << " " << cst_atom1 << " DD atom type " << nb_atom2 << std::endl;
				xyz1( 1 ) = full_coord( 1, atom1, res1 );
				xyz1( 2 ) = full_coord( 2, atom1, res1 );
				xyz1( 3 ) = full_coord( 3, atom1, res1 );

				xyz2( 1 ) = full_coord( 1, bonded_neighbor( i, cst_atom2, cst_aa2, aav2 ), res2 );
				xyz2( 2 ) = full_coord( 2, bonded_neighbor( i, cst_atom2, cst_aa2, aav2 ), res2 );
				xyz2( 3 ) = full_coord( 3, bonded_neighbor( i, cst_atom2, cst_aa2, aav2 ), res2 );

   			noe_dist2( xyz1, xyz2, dist2 );
				dist_sq.push_back( dist2 );
			}
		}
	} else if ( noe_state == 3 ) {
		for( int i = 1; i <= nbonded_neighbors( cst_atom1, cst_aa1, aav1 ); ++i ) {
			atom_name_from_atom_num( bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), cst_aa1, aav1, nb_atom1 );
			if( nb_atom1.rfind("H") != std::string::npos ) {
				for( int j = 1; j <= nbonded_neighbors( cst_atom2, cst_aa2, aav2 ); ++j ) {
					atom_name_from_atom_num( bonded_neighbor( j, cst_atom2, cst_aa2, aav2 ), cst_aa2, aav2, nb_atom2 );
					if( nb_atom2.rfind("H") != std::string::npos ) {
						xyz1( 1 ) = full_coord(1, bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), res1 );
						xyz1( 2 ) = full_coord(2, bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), res1 );
						xyz1( 3 ) = full_coord(3, bonded_neighbor( i, cst_atom1, cst_aa1, aav1 ), res1 );

						xyz2( 1 ) = full_coord(1, bonded_neighbor( j, cst_atom2, cst_aa2, aav2 ), res2 );
						xyz2( 2 ) = full_coord(2, bonded_neighbor( j, cst_atom2, cst_aa2, aav2 ), res2 );
						xyz2( 3 ) = full_coord(3, bonded_neighbor( j, cst_atom2, cst_aa2, aav2 ), res2 );

						noe_dist2( xyz1, xyz2, dist2);

						dist_sq.push_back( dist2 );
					}
				}
			}
		}
	}
	assert(dist_sq.size() > 0);
	compute_effective_distance( dist_sq, eff_dist2, sum_inv );
	std::pair<float, float> temp (eff_dist2, sum_inv);
	pair_effdist2_suminv[pair] = temp;

	//	std::cout << "pair pp " << pair << " " << res1 << " " << aa1 << " " << res2 << " " << aa2 << " " << get_noe_state( pair ) << " " << dist_sq.size() << " " << eff_dist2 << " " << sum_inv << std::endl;

	//	std::cout << "AA coordinates i full_coord " << full_coord(1,5,res1) << " " << full_coord(2,5,res1) << " " << full_coord(3,5,res1) << std::endl;
	//	std::cout << "AA coordinates j full_coord " << full_coord(1,5,res2) << " " << full_coord(2,5,res2) << " " << full_coord(3,5,res2) << std::endl;
	//	float eff_dist2 = compute_effective_distance( dist_sq );
	eff_dist = std::sqrt( eff_dist2 );

	//	calc_cst_score( mode_in, eff_dist, eff_dist2, pairRadius( pair ),
	//									 pairRadius2( pair ), pairMinRadius( pair ), ambi_resE );
}

////////////////
/// Geometry ///
////////////////

////////////////////////////////////////////////////////////////////////////////
/// @begin compute_effective_distance
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors sraman
///
/// @last_modified Jan 15th 2007
///////////////////////////////////////////////////////////////////////////////
void
Constraints::compute_effective_distance(
		std::vector < float > dist_sq,
		float & eff_dist2,
		float & sum_inv
)
{
	float sum_inv_dist( 0.0 );
	//	float eff_dist2( 0.0 );
	int n_dist = static_cast< int >( dist_sq.size() );

  for ( int i = 0; i < n_dist; ++i ) {
	sum_inv_dist += 1/( dist_sq[ i ]*dist_sq[ i ]*dist_sq[ i ] );
	}
	eff_dist2 = 1.0f/pow( sum_inv_dist, 0.333f );
	sum_inv = sum_inv_dist;
	//	return eff_dist2;
}

//////////////////////
/// Private: Setup ///
//////////////////////
void
Constraints::initialize()
{
	using namespace param;
	using namespace classical_constraints::constants;

//	int const & total_residue = *total_residue_; // yab: misc removal

	set_verbose(false);
	set_verbose_pdb_output(true);

	np_pairs = 0;

	// yab:
	// The following two lines need to be UNCOMMENTED once this class is being used
	// without BOUNDARY functions... most likely meaning that the initialization pipeline
	// has been partially reworked at that time.
//	cst_res_wt = FArray1D_float(total_residue);
//	reset_cst_res_wt();

	constraintPair = FArray2D_int( MAX_CONSTRAINTS, 4 );
	pairRadius = FArray1D_float( MAX_CONSTRAINTS );
	pairMinRadius = FArray1D_float( MAX_CONSTRAINTS );
	pair_atom_index = FArray2D_int( 2, MAX_CONSTRAINTS );
	pairSeqSep = FArray1D_int( MAX_CONSTRAINTS );
	set_max_seqSep(9999);

	HA_position = FArray2D_float( 3, MAX_RES() );
	HN_position = FArray2D_float( 3, MAX_RES() );

	pairRadius2 = FArray1D_float( MAX_CONSTRAINTS );
	dist_constraint = FArray1D_float( MAX_CONSTRAINTS, -1.0 );
	dist_constraint2 = FArray1D_float( MAX_CONSTRAINTS, -1.0 );
	score_constraint = FArray1D_float( MAX_CONSTRAINTS, -999.0 );

	constraint_list = FArray3D_int( MAX_CST_LIST, MAX_RES(), MAX_RES() );
	n_cst = FArray2D_int( MAX_RES(), MAX_RES() );

	messed_up_cst = 0;
}

void
Constraints::calc_ambig_cst_drv(
			int & noe_state,
			int const & pair,
			int const & i,
			int const & ii,
			int const & j,
			int const & jj,
			float const &  dist,
			float const & dist2,
			float & deriv
)
{

	FArray3D_float const & full_coord = *full_coord_;
	FArray1D_int const & res = *res_;
	FArray1D_int const & res_variant = *res_variant_;
	int cst_atom1, cst_atom2, cst_aa1, cst_aa2, res_num1, res_num2;
	int atom1, atom2, res_type1, res_type2, res_v1, res_v2;
	FArray1D_float xyz1( 3 ), xyz2( 3 );

	if( constraintPair( pair, 1 ) == i ){
		cst_atom1 = pair_atom_index( 1, pair );//change this 1, pair
		cst_atom2 = pair_atom_index( 2, pair );//change this 2, pair
		cst_aa1 = constraintPair( pair, 3 );
		cst_aa2 = constraintPair( pair, 4 );
		res_type1 = res(i);
		res_type2 = res(j);
		res_v1 = res_variant(i);
		res_v2 = res_variant(j);
		res_num1 = i;
		res_num2 = j;
	} else {
		cst_aa1 = constraintPair( pair, 4 );
		cst_aa2 = constraintPair( pair, 3 );
		cst_atom2 = pair_atom_index( 1, pair );
		cst_atom1 = pair_atom_index( 2, pair );
		res_type1 = res(i);
		res_type2 = res(j);
		res_v1 = res_variant(i);
		res_v2 = res_variant(j);
		res_num1 = i;
		res_num2 = j;
		//vats res1 and res2 have flipped from the packer. Now res1 = constraintPair( pair, 4 ) and res2 = constraintPair( pair, 3 ).
//vats Therefore the previous assignment of noe_state 1 and 2 have to be changed because the nbonded_neighbor iterator will have
//vats to loop over the correct heavyatom, i.e., the one connected to the degenerate proton.
		if( noe_state == 1 ) noe_state = 2;
		if( noe_state == 2 ) noe_state = 1;
	}

	std::cout << " restype " << pair  << " " << cst_aa1 << " " << res_type1 << " " << cst_aa2 << " " << res_type2 << std::endl;

	atom1 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum( cst_atom1, cst_aa1, res_type1, res_v1 );
	atom2 = classical_constraints::functions::cst_atomnum_to_fullcoord_atomnum( cst_atom2, cst_aa2, res_type2, res_v2 );



	//	std::cout << "info " << pair << " " << i << " " << res(i) << " " << j << " " << res(j) << " " << ii << " " << jj << std::endl;
	std::cout << "info " << pair << " " << i << " " << j << " " << ii << " " << jj << " " << res_num1 << " " << res_num2 << " " << res_type1 << " " << res_type2 << " " << atom1 << " " << atom2 << std::endl;

	std::cout << "coordinates i full_coord " << full_coord(1,atom1,res_num1) << " " << full_coord(2,atom1,res_num1) << " " << full_coord(3,atom1,res_num1) << std::endl;
	xyz1( 1 ) = full_coord(1,atom1,res_num1);
	xyz1( 2 ) = full_coord(2,atom1,res_num1);
	xyz1( 3 ) = full_coord(3,atom1,res_num1);

	std::cout << "coordinates j full_coord " << full_coord(1,atom2,res_num2) << " " << full_coord(2,atom2,res_num2) << " " << full_coord(3,atom2,res_num2) << std::endl;

	xyz2( 1 ) = full_coord(1,atom2,res_num2);
	xyz2( 2 ) = full_coord(2,atom2,res_num2);
	xyz2( 3 ) = full_coord(3,atom2,res_num2);

	float coeff;
	noe_dist2(xyz1,xyz2,coeff);

	std::cout << "distance "<< coeff << " " << dist2 << " " << dist << std::endl;
	deriv = 0.0;

}

void
Constraints::calc_ambig_cst_drv_1(
			int & pair,
			int noe_state,
  		float & eff_dist,
			float & sum_inv,
			float const &  dist,
			float const & dist2,
			float & deriv
)
{
	using namespace classical_constraints::constants;
	float const rswitch = { 0.5 };
	float upper_bound, lower_bound;
	float coeff = 1.0;
  float dist_sep = 0.0;
	deriv = 0.0;

	//vats cns potential
	upper_bound = pairRadius(pair);
	lower_bound = pairMinRadius(pair);

  assert(dist2 >= 0.0 && dist >= 0.0);

	if ( ( cst_drv_mode == 1 || cst_drv_mode == 2 ) && ( eff_dist - upper_bound > 0.0 ) ) deriv = cst_drv_factor;
	else if ( cst_drv_mode == 3 ) {

		if ( noe_state == 1 || noe_state == 2 ) dist_sep = dist - DEGENERATE_PAD;
		else if ( noe_state == 3 ) dist_sep = dist - (2.0*DEGENERATE_PAD);
		else {
			std::cout << "noe_state should be either 1,2 or 3. If it is zero, it shouldn't be getting into this function !! " << dist << " " << dist2 << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		coeff = (pow(sum_inv,-1.1666f))*(1/(pow(dist_sep, 7)));
		//	coeff = 1.0;



		if ( eff_dist > (upper_bound + rswitch) ) {
			deriv = cst_drv_factor*coeff;
		} else if ( lower_bound <= eff_dist && eff_dist <= upper_bound ) {
			deriv = 0.0;
		} else if ( eff_dist < lower_bound ) {
			deriv = (-1.0)*cst_drv_factor;
		} else {
			deriv = 2.0*cst_drv_factor*(eff_dist - upper_bound)*coeff;
		}
	}
	//	deriv = 0.0;

	//	std::cout << "pair " << pair << " " << dist << " " << dist_sep << " " << upper_bound  << " " << lower_bound << " " << sum_inv << " " << eff_dist << " " << coeff << " " << deriv << std::endl;

}

int
Constraints::switch_noe_state(
	 int noe_state
)
{

	if ( noe_state == 1 ){
		return 2;
	} else if ( noe_state == 2 ) {
		return 1;
	} else {
		return 3;
	}
}

} // namespace classical constraints

