// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 20460 $
//  $Date: 2008-02-18 20:02:35 -0800 (Mon, 18 Feb 2008) $
//  $Author: possu $

#ifndef INCLUDED_constraints
#define INCLUDED_constraints

#include "files_paths.h"
// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// C++ Headers
#include <iosfwd>
#include <vector>
#include <list>
#include <map>

namespace classical_constraints {

namespace nblist {
	extern FArray1D_bool cst_nb;
} // namespace nblist

namespace constants {

	extern int const MAX_CONSTRAINTS; // max number of constraint pairs
	extern Dimension const MAX_CST_LIST; // max constraints per residue pair
	extern float const DEGENERATE_PAD;  // upper bound padding for hydrogen
		// constraints evaluated using heavy atom coordinates
	extern float const GLOBAL_PAD;  // upper bound padding for all constraints
	extern int const CENTROID_ATOMNUM;
	extern int const H_ATOMNUM;
	extern int const HA_ATOMNUM;
	extern int const HA2_ATOMNUM;
	extern int const UNDEF_ATOMNUM;

} // namespace constants

namespace options {

	extern bool constraints_exist;

} // namespace options

namespace functions {

int
type2index(
	std::string & atom,
	int & restype // residue type
);

std::string
index2type(
	int const num, // atom number
	int const restype // residue type
);

void
heavyatom_num_from_proton_name(
	std::string const & name, // atom name
	int & aa, // amino acid number
	int & number // atom number
);

int
cst_atomnum_to_fullcoord_atomnum(
	int const cst_atom,
	int const cst_aa,
	int const aa,
	int const aav
);

} // namespace functions

namespace BOUNDARY {
	void
	init_module();

	bool
	init_module_called_once();

	bool
	get_constraints_exist();

	void
	set_constraints_exist(
		bool setting
	);

	void
	constraints_set_verbose(
		bool flag
	);

	bool
	get_verbose_pdb_output();

	void
	set_verbose_pdb_output(
		bool flag
	);

	void
	load_constraints_from_file();
	void
	load_constraints_from_file(
		std::string const & filename
	);

	void
	output_constraints_status(
		int mode,
		std::ostream & iunit
	);

	void
	set_max_seqSep(
		int sep
	);

	int
	get_max_seqSep();

	void
	noe_dist2(
		FArray1DB_float const & xyz1,
		FArray1DB_float const & xyz2,
		float & dis2
	);

	void
	get_noe_coords(
		int pair,
		FArray1Da_float xyz1,
		FArray1Da_float xyz2
	);

	bool
	residue_pair_has_constraints
	(
	 int res1,
	 int res2
	);

	bool
	cst_pair_satisfies_conditions(
		int pair,
		bool fullatom,
		int stage
	);

	void
	get_XYZ(
		int & residue,
		int & type_index,
		FArray1Da_float xyz
	);

	void
	eval_pairConstraint(
		int mode,
		float & score_x // ret'd copy of score, = pc_score usually
	);

	void
	precalc_cst_deriv();

	void
	eval_constraint_deriv(
		int const & i,
		int const & ii,
		int const & j,
		int const & jj,
		bool const & fullatom,
		float const & dis,
		float const & dist2,
		float & deriv
	);

	float
	get_res_res_cstE(
		int const res1,
		int const res2,
		int const aa1,
		int const aav1,
		int const aa2,
		int const aav2,
		FArray2Da_float coord1,
		FArray2Da_float coord2,
		bool const bb1 = true,
		bool const sc1 = true,
		bool const bb2 = true,
		bool const sc2 = true
	);

	void
	reset_cst_res_wt();

	void
	setup_cst_res_weight();

	void
	slice_cst(
		const FArray1D_bool & trim_res,
		int const & nres
	);

	void
	restore_cst();

	int
	np_pairs();

	int
	constraintPair(
		int i,
		int j
	);

	int
	pairSeqSep(
		int i
	);

	float
	pairRadius2(
		int i
	);

	float
	dist_constraint(
		int i
	);

	float
	pairRadius(
		int i
	);

	float
	pairMinRadius(
		int i
	);

	int
	pair_atom_index(
		int i,
		int j
	);

	FArray2D_int const &
	pair_atom_index();

	float
	HN_position(
		int i,
		int j
	);

	float
	HA_position(
		int i,
		int j
	);
} // namespace BOUNDARY

class Constraints {
	public: // construction/destruction
		Constraints(
			FArray3Dp_float const * Eposition,
			FArray2D_float const * centroid,
			FArray3D_float const * full_coord,
			FArray1D_int const * res,
			FArray1D_int const * res_variant,
			int const * total_residue
		);
		Constraints();
		~Constraints();

	public: // object merging/slicing
		Constraints *
		slice(
			const FArray1D_bool & trim_res,
			int const & nres
		);

	public: // external input/ouput
		void
		load_constraints_from_file(
			std::string const & filename = files_paths::constraints_path + files_paths::protein_name_prefix + files_paths::protein_name + files_paths::protein_chain + '.' + files_paths::cst_ext
		);

		void
		output_status(
			int mode,
			std::ostream & iunit
		);

	public: // accessors
		bool
		get_verbose();

		void
		set_verbose(
			bool flag
		);

		bool
		get_verbose_pdb_output();

		void
		set_verbose_pdb_output(
			bool flag
		);

		int
		get_max_seqSep();

		void
		set_max_seqSep(
			int sep
		);

	public: // data access
		int
		get_np_pairs();

		int
		get_constraintPair(
			int i,
			int j
		);

		int
		get_pairSeqSep(
			int i
		);

		float
		get_pairRadius2(
			int i
		);

		float
		get_dist_constraint(
			int i
		);

		float
		get_pairRadius(
			int i
		);

		float
		get_pairMinRadius(
			int i
		);

		int
		get_pair_atom_index(
			int i,
			int j
		);

		FArray2D_int const &
		get_pair_atom_index();

		float
		get_HN_position(
			int i,
			int j
		);

		float
		get_HA_position(
			int i,
			int j
		);

	public: // weights
		bool
		res_wt_randomized();

		void
		randomize_cst_res_wt();

		void
		reset_cst_res_wt();

	public: // pair checks
		bool
		residue_pair_has_constraints
		(
		 int res1,
		 int res2
		);

		bool
		cst_pair_satisfies_conditions(
			int pair,
			bool fullatom,
			int stage
		);

	public: // coordinates
		void
		get_XYZ(
			int & residue,
			int & type_index,
			FArray1Da_float xyz
		);

	public: // evaluation, scoring, derivatives

		void
		eval_pairConstraint(
			int mode,
			float & score_x // ret'd copy of score, = pc_score usually
		);

		void
		precalc_cst_deriv();

		void
		eval_constraint_deriv(
			int const & i,
			int const & ii,
			int const & j,
			int const & jj,
			bool const & fullatom,
			float const & dis,
			float const & dist2,
			float & deriv
		);

		float
		get_res_res_cstE(
			int const res1,
			int const res2,
			int const aa1,
			int const aav1,
			int const aa2,
			int const aav2,
			FArray2Da_float coord1,
			FArray2Da_float coord2,
			bool const bb1 = true,
			bool const sc1 = true,
			bool const bb2 = true,
			bool const sc2 = true
		);

	public: // NOEs
		void
		noe_dist(
			FArray1DB_float const & xyz1,
			FArray1DB_float const & xyz2,
			float & dis,
			float & dis2
		);

		void
		noe_dist2(
			FArray1DB_float const & xyz1,
			FArray1DB_float const & xyz2,
			float & dis2
		);

		void
		get_noe_coords(
			int pair,
			FArray1Da_float xyz1,
			FArray1Da_float xyz2
		);

	private: // methods
		void
		build_constraint_list();

		void
		add_centroid_constraints();

	private: // derivatives, scoring

		void
		calc_cst_drv(
			int const & pair,
			float const &  dist,
			float const & dist2,
			float & deriv
		);

	void
	calc_ambig_cst_drv(
			int & noe_state,
			int const & pair,
			int const & i,
			int const & ii,
			int const & j,
			int const & jj,
			float const &  dist,
			float const & dist2,
			float & deriv
			);

		void
		calc_ambig_cst_drv_1(
   	  int & pair,
			int noe_state,
  		float & eff_dist,
			float & sum_inv,
			float const &  dist,
			float const & dist2,
			float & deriv
		);

	  int
		switch_noe_state(
			int noe_state
		);

		void
		calc_cst_score(
			int mode_in,
			float dist,
			float dist2,
			float & pairRadius,
			float & pairRadius2,
			float & pairMinRadius,
			float & energy // unscaled energy
		);

	private: // proton geometry
		void
		compute_HN_position(
			FArray3Da_float Eposition,
			FArray1Da_int res,
			int & ires,
			FArray1Da_float HN_xyz
		);

		void
		compute_HCA_position(
			FArray3Da_float Eposition,
			int & ires,
			FArray1Da_float HA_xyz
		);

		void
		get_HA_coord(
			FArray1Da_float Nxyz,
			FArray1Da_float CAxyz,
			FArray1Da_float Cxyz,
			FArray1Da_float HAxyz
		);

	private: // ambiguous NOEs
		void
		build_ambiguous_constraint_list(
			int k,
			std::string cstatom1,
			std::string cstatom2
		);

		int
		get_noe_state(
			 int pair
		);

		void
		ambiguous_noe_cst_eff_dist(
			 int pair,
			 int noe_state,
			 int res1,
			 int res2,
			 int aa1,
			 int aa2,
			 int aav1,
			 int aav2,
			 float & eff_dist,
			 float & eff_dist2,
			 float & sum_inv_dist
			 //		   float & ambi_resE
		);

	private: // geometry
		static
		void
		compute_effective_distance(
			 std::vector < float > dist_sq,
			 float & eff_dist2,
			 float & sum_inv
		);

	private: // setup
		void
		initialize();

	private: // mirrors, pointers
		FArray3Dp_float const * Eposition_;
		FArray2D_float const * centroid_;
		FArray3D_float const * full_coord_;
		FArray1D_int const * res_;
		FArray1D_int const * res_variant_;
		int const * total_residue_;

	private: // modes
		int mode;

	private: // status and output options
		bool verbose;
		bool verbose_pdb_output; // mtyka 11/30/05 - print constraints into PDB ?

	private: // weights
		bool using_random_cst_res_wt;
		FArray1D_float cst_res_wt;

	private: // derivative calculation
		int cst_drv_mode;
		float cst_drv_factor;

	private: // data
		int np_pairs; // constraints to use in folding
		FArray2D_int constraintPair; // res1,res2,restype1,restype2
		FArray1D_float pairRadius; // upper bound
		FArray1D_float pairMinRadius; // lower bound
		FArray2D_int pair_atom_index; // atom 1, atom2
		FArray1D_int pairSeqSep; // seq separation here
		int max_seqSep; // current noe stage

		FArray1D_float pairRadius2;
		FArray1D_float dist_constraint; // distance at last evaluation
		FArray1D_float dist_constraint2;
		FArray1D_float score_constraint;

		FArray2D_float HA_position;
		FArray2D_float HN_position;

		FArray3D_int constraint_list;
		FArray2D_int n_cst;

	private: // data, ambi noes
		std::list < int > ambi_res1;
		std::list < int > ambi_res2;
		int messed_up_cst;
  	std::map < int, std::pair < float, float > > pair_effdist2_suminv;
  	std::map < int, int > noe_state_list;
};

} // namespace classical_constraints

#endif
