// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 21214 $
//  $Date: 2008-03-20 14:27:59 -0700 (Thu, 20 Mar 2008) $
//  $Author: jiangl $

//////////////////////////////////////////////


// Rosetta Headers
#include "cst_set.h"
#include "atom_is_backbone.h"
#include "fullatom_energy.h"
#include "param_torsion.h"
#include "pose.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/formatted.o.hh>

//Utility Headers
#include <utility/basic_sys_util.hh>

#include <numeric/xyz.functions.hh>

// C++ Headers
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <fstream>
#include <list>


namespace cst_set_ns {

	//lin debug flag
	bool debug_output = { false };

  void Cst_set::add_ptr_reference() const {
		++ptr_references_;
	}
  void Cst_set::remove_ptr_reference() const {
		--ptr_references_;
		if ( ptr_references_ == 0 ) delete this;
	}

	/////////////////////////////////////////////////////////////////////////////
	// add a new atompair constraint
	void
	Cst_set::add_atompair_constraint(
		kin::Atom_id const & atom1,
		kin::Atom_id const & atom2,
		Cst const & cst,
		bool const verbose /* = true */
	)
	{
		// add to all the relevant maps, symmetrically
		insert_atom_pair_constraint( cst_map_by_atom, atom1, atom2, cst, verbose );
		insert_atom_pair_constraint( cst_map_by_atom, atom2, atom1, cst, verbose );

		// add to the map by residue pairs
		int const pos1( std::min( atom1.rsd(), atom2.rsd() ) );
		int const pos2( std::max( atom1.rsd(), atom2.rsd() ) );
		std::pair< int, int > rsd_pair( pos1,pos2);
		if ( !cst_map_by_rsd_pair.count( rsd_pair ) ) {
			cst_map_by_rsd_pair.insert
				( std::make_pair( rsd_pair, std::vector< Rsd_rsd_cst >() ) );
		}
		int atomno1, atomno2;
		if ( atom1.rsd() == pos1 ) {
			atomno1 = atom1.atomno();
			atomno2 = atom2.atomno();
		} else {
			atomno1 = atom1.atomno();
			atomno2 = atom2.atomno();
		}
		Rsd_rsd_cst rr_cst( atomno1, atomno2, cst );
		cst_map_by_rsd_pair.find( rsd_pair )->second.push_back( rr_cst );
	}

	/////////////////////////////////////////////////////////////////////////////
	// hate to use FArray slices here... short-term
	float
	Cst_set::get_res_res_cstE(
		int const res1,
		int const res2,
		int const aa1,
		int const aav1,
		int const aa2,
		int const aav2,
		FArray2Da_float coord1,
		FArray2Da_float coord2,
		bool const bb1,
		bool const sc1,
		bool const bb2,
		bool const sc2
	) const
	{
		float score( 0.0 );
		std::pair< int, int > rsd_pair( std::min( res1,res2), std::max(res1,res2));

		if ( cst_map_by_rsd_pair.count( rsd_pair ) ){
			std::vector< Rsd_rsd_cst > const & cst_list
				( cst_map_by_rsd_pair.find( rsd_pair )->second );
			for ( std::vector< Rsd_rsd_cst >::const_iterator it=cst_list.begin(),
							it_end = cst_list.end(); it != it_end; ++it ) {
				int const atomno1( res1 == rsd_pair.first ? it->atomno1 : it->atomno2);
				int const atomno2( res1 == rsd_pair.first ? it->atomno2 : it->atomno1);
				bool is_bb = atom_is_backbone(atomno1,aa1,aav1);
				if ( ! bb1 && is_bb ) continue;
				if ( ! sc1 && ! is_bb) continue;
				is_bb = atom_is_backbone(atomno2,aa2,aav2);
				if ( ! bb2 && is_bb ) continue;
				if ( ! sc2 && ! is_bb) continue;
				float const r
					( std::sqrt( square( coord1(1,atomno1) - coord2(1,atomno2) ) +
											 square( coord1(2,atomno1) - coord2(2,atomno2) ) +
											 square( coord1(3,atomno1) - coord2(3,atomno2) ) ) );
				score += it->cst.score( r );
			}
		}
		return this->packer_atompair_weight * score;
	}

	/////////////////////////////////////////////////////////////////////////////
	// add a new atompair constraint
	void
	Cst_set::add_atompair_constraint(
		kin::Atom_id const & atom1,
		kin::Atom_id const & atom2,
		float const r0,
		float const weight,
		bool const verbose /* = true */
	)
	{
		Cst cst( r0, weight );
		add_atompair_constraint( atom1, atom2, cst, verbose );
	}

	/////////////////////////////////////////////////////////////////////////////
	// add a new atompair constraint
	void
	Cst_set::add_atompair_constraint(
		kin::Atom_id const & atom1,
		kin::Atom_id const & atom2,
		float const r0,
		float const r0_sd,
		float const weight,
		bool const verbose /* = true */
	)
	{
		// add to all the relevant maps, symmetrically
		Cst cst( r0, r0_sd, weight );
		add_atompair_constraint( atom1, atom2, cst, verbose );
	}

	/////////////////////////////////////////////////////////////////////////////
	// remove all atompair constraints to a specified atom
	void
	Cst_set::remove_atompair_constraints_to_atom(
		kin::Atom_id const & atom1
	)
	{
		if ( !cst_map_by_atom.count( atom1 ) ) {  //check for absence of constraints
			return;
		}

		std::cout << "temporarily disabled bc of rsd-pair cst maps" <<
			"\n";
		assert( false );
		std::exit( EXIT_FAILURE );

		// first remove all the symmetrical constraints from cst_map_by_atom
		Cst_map & map1 ( cst_map_by_atom.find( atom1 )->second );
		for ( Cst_map::const_iterator it = map1.begin(),
						it_end = map1.end(); it!= it_end; ++it ) {
			kin::Atom_id const & atom2 ( it->first );
			Cst_map & map2 ( cst_map_by_atom.find( atom2 )->second );
			map2.erase( atom1 );
		}
		// erase forward constraints from cst_map_map
		cst_map_by_atom.erase( atom1 );
	}

	/////////////////////////////////////////////////////////////////////////////
	// score all the constraints on degrees of freedom
	//
	// not to be confused with the 3d angle/torsion constraints defined by
	// aribitrary sets of atoms
	//

	void
	Cst_set::torsion1D_score(
		pose_ns::Pose const & pose,
		float & phipsi_score,
		float & omega_score,
		float & chi_score,
		float & kin_torsion_score
	) const
	{
		//////////////////////////////////////////
		// first rosetta style torsion constraints
		phipsi_score = 0.0;
		omega_score = 0.0;
		chi_score = 0.0;

		for ( std::map< std::pair< int, int >, float >::const_iterator
						it=rosetta_torsion_cst_.begin(),
						it_end=rosetta_torsion_cst_.end(); it != it_end; ++it ) {
			int const seqpos( it->first.first );
			int const torsion( it->first.second );
			float const target( it->second );
			float const angle( pose.get_torsion_by_number( seqpos, torsion ) );

			float const angle_dev( subtract_degree_angles( angle, target ) );
			float const penalty( angle_dev * angle_dev );

			// now add to the relevant score component
			if ( torsion == param_torsion::phi_torsion ||
					 torsion == param_torsion::psi_torsion ) {
				phipsi_score += penalty;
			} else if ( torsion == param_torsion::omega_torsion ) {
				omega_score += penalty;
			} else if ( torsion >= param_torsion::chi1_torsion &&
									torsion <= param_torsion::chi4_torsion ) {
				chi_score += penalty;
			} else {
				std::cout << "unknown torsion in pose constraint score " << torsion
									<< "\n";
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}

		/////////////////////////////
		// now the kin torsion scores
		kin_torsion_score = 0.0;
		if ( pose.atom_tree() != 0 ) {
			for ( std::map< kin::Torsion_id, std::pair< float,float > >
							::const_iterator it = kin_torsion_cst_.begin(),
							it_end = kin_torsion_cst_.end(); it != it_end; ++it ) {
				float const current_value
					( pose.get_atom_tree_torsion( it->first ) );
				float const target( it->second.first );
				float const weight( it->second.second );
				float const dev( subtract_kin_torsions( current_value, target,
																								it->first.type() ) );
				kin_torsion_score +=
					weight * dev * dev;
			}
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	float
	Cst_set::kin_torsion_deriv(
		kin::Torsion_id const & id,
		float const value
	) const
	{
		if ( !kin_torsion_cst_.count( id ) ) return 0.0;
		std::pair< float, float > const & cst
			( kin_torsion_cst_.find( id )->second );
		float const target( cst.first  );
		float const weight( cst.second );
		return 2 * weight * subtract_kin_torsions( value, target, id.type() );
	}

	/////////////////////////////////////////////////////////////////////////////
	// phi,psi,omega,chi

	float
	Cst_set::rosetta_torsion_deriv(
		int const seqpos,
		int const torsion,
		float const value
	) const
	{
		assert( torsion <= param_torsion::total_torsion );
		std::pair< int, int > key( seqpos, torsion );
		if ( !rosetta_torsion_cst_.count( key ) ) return 0.0;
		float const target( rosetta_torsion_cst_.find( key )->second );
		float const angle_dev( subtract_degree_angles( value, target ) );
		return 2 * angle_dev;
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::add_kin_torsion_constraint(
		kin::Torsion_id const & id,
		float const value,
		float const weight
	)
	{
		kin_torsion_cst_[ id ] = std::make_pair( value, weight );
	}

	/////////////////////////////////////////////////////////////////////////////
	// add torsion constraint
	void
	Cst_set::add_rosetta_torsion_constraint(
		int const pos,
		int const torsion,
		float const value
	)
	{
		std::pair< int, int > const key( pos,torsion );
		if ( rosetta_torsion_cst_.count( key ) ) {
			std::cout << "Cst_set::add_torsion_constraint: replacing old value: "
								<< pos << ' ' << torsion << "\n";
		}
		rosetta_torsion_cst_[key] = value;
	}

	/////////////////////////////////////////////////////////////////////////////
	// add a coordinate constraint
	void
	Cst_set::add_coordinate_constraint(
		kin::Atom_id const & atom,
		numeric::xyzVector_float const & xyz
	)
	{
		if ( coord_cst_.count( atom ) ) {
			std::cout << "Cst_set::add_coordinate_constraint: replacing old value: "
								<< atom.atomno() << ' ' << atom.rsd() << "\n";
		}
		coord_cst_.insert( std::make_pair( atom, xyz ) );
	}




	/////////////////////////////////////////////////////////////////////////////
	int
	Cst_set::count_chi_constraints() const {
		int count(0);
		for ( std::map< std::pair< int, int >, float >::const_iterator
						it=rosetta_torsion_cst_.begin(), it_end=rosetta_torsion_cst_.end();
					it != it_end; ++it ) {
			int const torsion( it->first.second );
			if ( torsion >= param_torsion::chi1_torsion &&
					 torsion <= param_torsion::chi4_torsion ) ++count;
		}
		return count;
	}

	/////////////////////////////////////////////////////////////////////////////
	int
	Cst_set::count_coordinate_constraints() const {
		return coord_cst_.size();
	}

	/////////////////////////////////////////////////////////////////////////////
	int
	Cst_set::count_atompair_constraints() const {
		int count(0);
		for ( Cst_map_map::const_iterator it = cst_map_by_atom.begin(),
						it_end = cst_map_by_atom.end(); it != it_end; ++it ) {
			count += it->second.size();
		}
		return count;
	}


	/////////////////////////////////////////////////////////////////////////////
	// helper function:
	void insert_atom_pair_constraint(
		Cst_map_map & cst_map_map,
		kin::Atom_id const & pair1,
		kin::Atom_id const & pair2,
		Cst const & cst,
		bool const verbose /* = true */
	)
	{
		if ( !cst_map_map.count( pair1 ) ) {
			Cst_map tmp;
			cst_map_map.insert( std::make_pair( pair1, tmp ) );
		}
		Cst_map & map1 ( cst_map_map.find( pair1 )->second );
		if ( verbose && map1.count(pair2) ) {
			std::cout << "WARNING:: replacing old constraint: " <<
				pair1.atomno() << ' ' << pair1.rsd() << ' ' <<
				pair2.atomno() << ' ' << pair2.rsd() << ' ' << "\n";
		}
		map1[pair2] = cst;
	}


	//lin score for distance cst
	float Cst::score( float const r ) const
	{
		assert( r0 >=0 && r >=0 );
		float const cstE ( func_p->cstE(r-r0, r_sd, weight1, weight2) );
		if( debug_output ) {
			std::cout<<"DEBUG: r= " << r << " cst_E= " << cstE << *this;
		}
		return cstE;
	}

	/////////////////////////////////////////////////////////////////////////////
	float Cst::dscore_dr( float const r ) const
	{
		assert( r0 >=0 && r >=0 );
		return func_p->cstE_deriv(r-r0, r_sd, weight1, weight2);
	}

	/////////////////////////////////////////////////////////////////////////////
	float Cst::score(
			numeric::xyzVector_float const & p1,
			numeric::xyzVector_float const & p2
			) const
	{
		typedef numeric::xyzVector_float Vec;
		Vec v1( p1-p2 );
		//std::cout<<"distance_coord: "<<p1<<" "<<p2<<"\n";

		float const r ( v1.length() );
		return score(r);
	}

	/////////////////////////////////////////////////////////////////////////////
	// contributions for a term that looks like
	//
	//   d(M-F)
	// ( ------ x v ) dot w
	//   d phi
	//
	// F1 collects terms f1 that look like: (-u_phi) dot f1
	// F2 collects terms f2 that look like: (-u_phi x R_phi) dot f2
	//
	//
	// where u_phi is the torsion axis of rotation
	// and R_phi is a terminal atom of this axis
	//
	// static class function
	void
	Torsion_cst::helper(
		numeric::xyzVector_float const & M,
		numeric::xyzVector_float const & v,
		numeric::xyzVector_float const & w,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	)
	{
		numeric::xyzVector_float const f2 = cross(v,w);
		F2 += f2;
		F1 += cross(f2,M);
	}


	/////////////////////////////////////////////////////////////////////////////
	//
	// static class function
	void
	Torsion_cst::p1_cosine_deriv(
		numeric::xyzVector_float const & p1,
		numeric::xyzVector_float const & p2,
		numeric::xyzVector_float const & p3,
		numeric::xyzVector_float const & p4,
		float & x,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	)
	{
		F1 = 0.0;
		F2 = 0.0;

		typedef numeric::xyzVector_float Vec;
		Vec v1( p1-p2 );
		Vec v2( p2-p3 );
		Vec v3( p3-p4 );

		Vec v12( cross( v1, v2 ));
		Vec v23( cross( v2, v3 ));

		float const n12( v12.length() );
		float const n23( v23.length() );

		if ( n12 < 1e-9 || n23 < 1e-9 ) return;

		x = dot( v12, v23) / ( n12 * n23 );

		// first term:
		{
			float const f( 1.0f/ ( n12 * n23 ) );
			helper( p1, f * v2, v23 , F1, F2);
		}

		// second term
		{
			float const f( -1.0f * x / ( n12 * n12 ) );
			helper( p1, f * v2, v12, F1, F2 );
		}


		// debugging
		// translation of p1 in the place spanned by v1 and v2
		// does not affect the torsion angle
		// ==> rotation of p1 about an axis perpendicular to this plane
		// also does not change the torsion angle, ie deriv should be 0
		assert( std::abs( dot( F2, v1 ) ) < 1e-3 );
		assert( std::abs( dot( F2, v2 ) ) < 1e-3 );
		assert( std::abs( dot( F1, cross( v1, v2 ) ) ) < 1e-3 );
	}

	/////////////////////////////////////////////////////////////////////////////
	// static class function
	//
	void
	Torsion_cst::p2_cosine_deriv(
		numeric::xyzVector_float const & p1,
		numeric::xyzVector_float const & p2,
		numeric::xyzVector_float const & p3,
		numeric::xyzVector_float const & p4,
		float & x,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	)
	{
		//std::cout << "p2_cosine_deriv!" << "\n";

		F1 = 0.0;
		F2 = 0.0;

		typedef numeric::xyzVector_float Vec;
		Vec v1( p1-p2 );
		Vec v2( p2-p3 );
		Vec v3( p3-p4 );

		Vec v12( cross( v1, v2 ));
		Vec v23( cross( v2, v3 ));

		float const n12( v12.length() );
		float const n23( v23.length() );

		if ( n12 < 1e-9 || n23 < 1e-9 ) return;

		x = dot( v12, v23) / ( n12 * n23 );

		// here we are taking derivatives of an expression that looks like
		//
		//                   dot( v12, v23 )
		// x = cos theta =  -----------------
		//                      n12 * n23
		//
		// where theta is our dihedral angle
		//

		{ // derivatives of the numerator
			// v1 and v2 both depend on position of p2

			{ // first term
				float const f( -1.0f/ ( n12 * n23 ) );
				helper( p2, f * v2, v23 , F1, F2);
			}

			{ // second term
				float const f( -1.0f/ ( n12 * n23 ) );
				helper( p2, f * v1, v23 , F1, F2);
			}

			{ // third term
				float const f( 1.0f/ ( n12 * n23 ) );
				helper( p2, f * v3, v12 , F1, F2);
			}
		}

		{ // derivatives of the denominator
			// v1 and v2 both depend on position of p2

			{ // first term
				float const f( x/ ( n12 * n12 ) );
				helper( p2, f * v2, v12, F1, F2 );
			}

			{ // second term
				float const f( x/ ( n12 * n12 ) );
				helper( p2, f * v1, v12, F1, F2 );
			}

			{ // third term
				float const f( -1.0f * x/ ( n23 * n23 ) );
				helper( p2, f * v3, v23, F1, F2 );
			}
		}

		// debugging
		// translation of p2 along v2 does not change the torsion angle
		assert( std::abs( dot( F2, v2 ) ) < 1e-3 );

	}

	/////////////////////////////////////////////////////////////////////////////
	// silly helper function, could also use sin_cos_range, I guess
	inline
	float
	arccos(
		float d
	)
	{
		float const tol(1e-4 );
		if ( d <= -1.0 + tol ) {
			//std::cout << "out-of-tol: " << d << ' ' << "\n";
			d = -1.0+tol;
		} else if ( d >= 1.0 - tol ) {
			//std::cout << "out-of-tol: " << d << "\n";
			d = 1.0-tol;
		}
		return std::acos( d );
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Torsion_cst::fill_f1_f2(
		kin::Atom_id const atom,
		kin::Coords const & coords,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		using numeric::conversions::radians;
		using namespace cst_functions_ns;

		float const deg2rad( numeric::conversions::radians(1.0) );

		// to avoid problems with dtheta/dx around 0 and 180 degrees
		// truncate x a bit in the calculation of the derivative
		static float const small_angle( radians( 0.1f ) );
		static float const big_angle( radians( 179.9f ) );
		static float const max_x( std::cos( small_angle ));
		static float const min_x( std::cos( big_angle ));
		// dtheta_dx has a value of ~ 572.96 for min_x and max_x
		// this goes to infinity as x goes to -1 or 1

		numeric::xyzVector_float f1(0.0) ,f2(0.0);

		float x(0.0); // cos theta
		if ( atom == atom1 ) {
			p1_cosine_deriv( coords.get_xyz( atom1 ),coords.get_xyz( atom2 ),
				coords.get_xyz( atom3 ),coords.get_xyz( atom4 ), x, f1, f2 );
		} else if ( atom == atom2 ) {
			p2_cosine_deriv( coords.get_xyz( atom1 ),coords.get_xyz( atom2 ),
				coords.get_xyz( atom3 ),coords.get_xyz( atom4 ), x, f1, f2 );
		} else if ( atom == atom3 ) {
			p2_cosine_deriv( coords.get_xyz( atom4 ),coords.get_xyz( atom3 ),
				coords.get_xyz( atom2 ),coords.get_xyz( atom1 ), x, f1, f2 );
		} else if ( atom == atom4 ) {
			p1_cosine_deriv( coords.get_xyz( atom4 ),coords.get_xyz( atom3 ),
				coords.get_xyz( atom2 ),coords.get_xyz( atom1 ), x, f1, f2 );
		} else {
			return;
		}

		float const thetaU( arccos( x )); // unsigned version of theta
		float theta ( deg2rad * dihedral( coords.get_xyz( atom1 ),coords.get_xyz( atom2 ),
														coords.get_xyz( atom3 ),coords.get_xyz( atom4 ) ) );


		assert( std::abs( std::abs( theta ) - thetaU ) < 1e-1 );

		float dE_dtheta = 0.0;
		float const pi( numeric::constants::f::pi );
		switch ( func_p->func_type() ) {
		case HARMONIC: { dE_dtheta = func_p->cstE_deriv(
												periodic_range( theta - theta0 , periodic_val ),
												theta_sd, weight, weight ); break; }
		case PERIODIC: { dE_dtheta = func_p->cstE_deriv(
												theta, theta0,
												2*pi/periodic_val, weight ); break; }
		}

		x = std::min( std::max( min_x, x ), max_x );
		float const dthetaU_dx = -1 / sqrt( 1- x*x );
		float const dtheta_dthetaU( theta < 0 ? -1 : 1 );

		F1 += dE_dtheta * dtheta_dthetaU * dthetaU_dx * f1;
		F2 += dE_dtheta * dtheta_dthetaU * dthetaU_dx * f2;

	}

  /////////////////////////////////////////////////////////////////////////////
  void
  Torsion_cst::extract_atom_ids(
    kin::Atom_id & a1,
    kin::Atom_id & a2,
    kin::Atom_id & a3,
    kin::Atom_id & a4
  ) const
  {
  a1 = atom1;
  a2 = atom2;
  a3 = atom3;
	a4 = atom4;
  }

	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// static class function
	//
	float
	Torsion_cst::score(
		kin::Coords const & coords
	) const
	{
		float cstE ( score( coords.get_xyz(atom1), coords.get_xyz(atom2),
												coords.get_xyz(atom3), coords.get_xyz(atom4) ) ) ;
		if( debug_output ) {
			fill_cstE_array( atom1.rsd(), atom4.rsd(), cstE );
		}
		return cstE;
	}


	/////////////////////////////////////////////////////////////////////////////
	float
	Torsion_cst::score(
		numeric::xyzVector_float const & p1,
		numeric::xyzVector_float const & p2,
		numeric::xyzVector_float const & p3,
		numeric::xyzVector_float const & p4
	) const
	{
		using namespace cst_functions_ns;
		float const deg2rad( numeric::conversions::radians(1.0) );
		float const pi( numeric::constants::f::pi );

		float theta ( deg2rad * dihedral( p1, p2, p3, p4 ) );

		float cstE = 0.0;
		switch ( func_p->func_type() ) {
		case HARMONIC: { cstE = func_p->cstE(
												periodic_range( theta - theta0 , periodic_val ),
												theta_sd, weight, weight ); break; }
		case PERIODIC: { cstE = func_p->cstE(
												theta, theta0,
												2*pi/periodic_val, weight ); break; }
		}
		if( debug_output ) {
			float const rad2deg( numeric::conversions::degrees(1.0) );
			std::cout << "DEBUG: theta= " << theta*rad2deg
								<< " cst_E= " <<  cstE << *this;
		}
		return cstE;
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Angle_cst::extract_atom_ids(
		kin::Atom_id & a1,
		kin::Atom_id & a2,
		kin::Atom_id & a3
	) const
	{
	a1 = atom1;
	a2 = atom2;
	a3 = atom3;
	}

	/////////////////////////////////////////////////////////////////////////////
	float
	Angle_cst::score(
		kin::Coords const & coords
	) const
	{
		float cstE = ( score( coords.get_xyz( atom1 ), coords.get_xyz( atom2 ),
									coords.get_xyz( atom3 ) ) );
		if( debug_output ) {
			fill_cstE_array( atom1.rsd(), atom3.rsd(), cstE );
		}
		return cstE;
	}

	/////////////////////////////////////////////////////////////////////////////
	float
	Angle_cst::score(
		numeric::xyzVector_float const & p1,
		numeric::xyzVector_float const & p2,
		numeric::xyzVector_float const & p3
	) const
	{
		using numeric::xyzVector_float;
		xyzVector_float u1( p1 - p2 );
		xyzVector_float u2( p3 - p2 );
		float const n1( u1.length() );
		float const n2( u2.length() );
		if ( n1 > 1e-12 && n2 > 1e-12 ) {
			float const theta = arccos( dot( u1,u2 ) / ( n1 * n2 ) );
			float cstE ( func_p->cstE( periodic_range( theta - theta0 , periodic_val ),
																 theta_sd, weight, weight ) );
			if( debug_output ) {
				float const rad2deg( numeric::conversions::degrees(1.0) );
				std::cout << "DEBUG: theta= " << theta*rad2deg << ' '
									<< n1 << ' ' << n2 << ' ' << dot(u1,u2)
									<< " cst_E= " << cstE << *this ;
			}
			return cstE;
		}
		std::cout << "Angle_cst::score: warning: 0-length bonds!" << "\n";
		return 0.0;
	}

	/////////////////////////////////////////////////////////////////////////////
	// accumulate F1, F2 contributions from terms that look like
	//
	//       d(M-F)
	// dot( -------- , w )
	//       d phi
	//
	// where phi is moving M and F is fixed.
	//
	// F1 collects terms f1 that look like: (-u_phi) dot f1
	// F2 collects terms f2 that look like: (-u_phi x R_phi) dot f2
	//
	// where u_phi is the unit vector axis of phi and R_phi is a point
	// on the axis
	//
	// basic id: d(M-F)/dphi = u_phi x ( M - R_phi )
	//
	// and dot( a, b x c ) = dot( b, c x a )
	//
	//
	void
	Angle_cst::helper(
		numeric::xyzVector_float const & M,
		numeric::xyzVector_float const & w,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
		)
	{
		F2 += w;
		F1 += cross( w, M );
	}

	/////////////////////////////////////////////////////////////////////////////
	// calculates f1,f2 contributions for dtheta_dphi
	//
	// where phi is a torsion angle moving p1 while p2 and p3 are fixed
	void
	Angle_cst::p1_theta_deriv(
		numeric::xyzVector_float const & p1,
		numeric::xyzVector_float const & p2,
		numeric::xyzVector_float const & p3,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	)
	{
		using numeric::xyzVector_float;
		using numeric::conversions::radians;

		// to avoid problems with dtheta/dx around 0 and 180 degrees
		// truncate x a bit in the calculation of the derivative
		static float const small_angle( radians( 0.1f ) );
		static float const big_angle( radians( 179.9f ) );
		static float const max_x( std::cos( small_angle ));
		static float const min_x( std::cos( big_angle ));
		// dtheta_dx has a value of ~ 572.96 for min_x and max_x
		// this goes to infinity as x goes to -1 or 1

		xyzVector_float v1( p1 - p2 );
		xyzVector_float v2( p3 - p2 );
		float const n1( v1.length() );
		float const n2( v2.length() );
		if ( n1 < 1e-9 || n2 < 1e-9 ) {
			return;
		}

		// calculate dx/dphi where x = cos theta = dot( v1,v2) / (n1*n2)

		float x = dot(v1,v2)/(n1*n2);

		// only v1 depends on phi, not v2 (we are assuming only p1 is moving)
		// so we get two terms, one from v1 on top and one from n1 = |v1| on
		// the bottom

		numeric::xyzVector_float f1(0.0),f2(0.0);

		{ // first term
			float const f = 1.0f / ( n1 * n2 );
			helper( p1, f * v2, f1, f2 );
		}

		{ // second term
			float const f = -1.0f * x / ( n1 * n1 );
			helper( p1, f * v1, f1, f2 );
		}

		x = std::min( std::max( min_x, x ), max_x );
		float const dtheta_dx = -1.0f / sqrt( 1.0f - x*x );
		f1 *= dtheta_dx;
		f2 *= dtheta_dx;

		// translation of p1 along v1 or perpendicular to v1 and v2 ==> deriv=0
		assert( distance( f1, cross(f2,p1) ) < 1e-3 && // see helper fcn
						std::abs( dot( f2, v1 ) ) < 1e-3 &&
						std::abs( dot( f2, cross( v1, v2 ) ) ) < 1e-3 );


		{ // more debugging
			// pretend axis = u2, R_phi = p2
			numeric::xyzVector_float const u_phi( v2.normalized() ), R_phi( p2 );
			float const deriv = - dot( u_phi, f1 ) - dot( cross( u_phi, R_phi ), f2);
			assert( std::abs( deriv ) < 1e-3 );
			//std::cout << "deriv: " << deriv<< ' ' <<
			//	F(9,3,u_phi(1)) << F(9,3,u_phi(2)) << F(9,3,u_phi(3)) << ' ' <<
			//	F(9,3,R_phi(1)) << F(9,3,R_phi(2)) << F(9,3,R_phi(3)) << "\nF1,F2: " <<
			//	F(9,3,f1(1)) << F(9,3,f1(2)) << F(9,3,f1(3)) << ' ' <<
			//	F(9,3,f2(1)) << F(9,3,f2(2)) << F(9,3,f2(3)) << "\n";
		}


		F1 += f1;
		F2 += f2;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Angle_cst::p1_deriv(
		numeric::xyzVector_float const & p1,
		numeric::xyzVector_float const & p2,
		numeric::xyzVector_float const & p3,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		using numeric::xyzVector_float;

		xyzVector_float u1( p1 - p2 );
		xyzVector_float u2( p3 - p2 );
		float const n1_n2( u1.length() * u2.length() );
		if ( n1_n2 < 1e-12 ) {
			std::cout << "Angle_cst::p1_deriv: short bonds: " << n1_n2 <<
				"\n";
			return;
		}

		xyzVector_float f1(0.0),f2(0.0);
		p1_theta_deriv( p1, p2, p3, f1, f2 );

		float d( dot(u1,u2) / n1_n2 );
		float const tol(0.001);
		if ( d <= -1.0 + tol ) {
			//std::cout << "out-of-tol: " << d << ' ' << "\n";
			d = -1.0+tol;
		} else if ( d >= 1.0 - tol ) {
			//std::cout << "out-of-tol: " << d << "\n";
			d = 1.0-tol;
		}
		float const theta = arccos( d );


		float const dE_dtheta = func_p->cstE_deriv( periodic_range( theta - theta0 , periodic_val ),
																								theta_sd, weight, weight );

		//std::cout << "dE_dtheta_p1_deriv: " << dE_dtheta << ' ' <<
		//	f1(1) << ' ' << f1(2) << ' ' << f1(3) << "\n";

		F1 += dE_dtheta * f1;
		F2 += dE_dtheta * f2;

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Angle_cst::p2_deriv(
		numeric::xyzVector_float const & p1,
		numeric::xyzVector_float const & p2,
		numeric::xyzVector_float const & p3,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		using numeric::xyzVector_float;

		xyzVector_float v1( p1 - p2 );
		xyzVector_float v2( p3 - p2 );
		float const v12( v1.length() * v2.length() );
		if ( v12 < 1e-12 ) return;

		// here we use the trick that theta = pi - alpha - beta
		//
		// where alpha and beta are the other angles in the p1,p2,p3 triangle
		// so dtheta_dphi  = -dalpha_dphi - dbeta_dphi
		//
		// for these we can use the p1_theta_deriv formula
		//
		xyzVector_float f1(0.0),f2(0.0);
		p1_theta_deriv( p2, p1, p3, f1, f2 ); // alpha deriv
		p1_theta_deriv( p2, p3, p1, f1, f2 ); // beta deriv

		// translation of p2 atom perpendicular to plane ==> deriv = 0
		//std::cout << "p2 deriv check: " << "\n";
		assert( std::abs( dot( f2, cross( v1,v2) ) ) < 1e-3 );

		float d( dot(v1,v2) / v12 );
		float const tol(0.001);
		if ( d <= -1.0 + tol ) {
			//std::cout << "out-of-tol: " << d << ' ' << "\n";
			d = -1.0+tol;
		} else if ( d >= 1.0 - tol ) {
			//std::cout << "out-of-tol: " << d << "\n";
			d = 1.0-tol;
		}
		float const theta = arccos( d );
		//float const theta = arccos( dot( v1,v2 ) / v12 );
		float const dE_dtheta = func_p->cstE_deriv( periodic_range( theta - theta0 , periodic_val ),
																								theta_sd, weight, weight );


		//
		//std::cout << "dE_dtheta_p2_deriv: " << dE_dtheta << ' ' <<
		//	f1(1) << ' ' << f1(2) << ' ' << f1(3) << "\n";

		F1 += -1.0f * dE_dtheta * f1;
		F2 += -1.0f * dE_dtheta * f2;

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Angle_cst::fill_f1_f2(
		kin::Atom_id const atom,
		kin::Coords const & coords,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		if ( atom == atom1 ) {
			//std::cout << "fill_f1_f2: atom1" << "\n";
			p1_deriv( coords.get_xyz( atom1), coords.get_xyz( atom2 ),
								coords.get_xyz( atom3), F1, F2 );
		} else if ( atom == atom2 ) {
			//std::cout << "fill_f1_f2: atom2" << "\n";
			p2_deriv( coords.get_xyz( atom1), coords.get_xyz( atom2 ),
								coords.get_xyz( atom3), F1, F2 );
		} else if ( atom == atom3 ) {
			//std::cout << "fill_f1_f2: atom3" << "\n";
			p1_deriv( coords.get_xyz( atom3), coords.get_xyz( atom2 ),
								coords.get_xyz( atom1), F1, F2 );
		} else {
			//std::cout << "fill_f1_f2: not one of my atoms: " << atom << ' ' <<
			//	atom1 << ' ' << atom2 << ' ' << atom3 << "\n";
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	float
	Cst_set::atom_angle_score(
		kin::Coords const & coords
	) const
	{
		float score(0.0);
		for ( std::vector< Angle_cst >::const_iterator it= angle_cst_list.begin(),
						it_end= angle_cst_list.end(); it != it_end; ++it ) {
			score += it->score( coords );
		}
		return score;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::add_atom_angle_constraint(
		kin::Atom_id const & atom1,
		kin::Atom_id const & atom2,
		kin::Atom_id const & atom3,
		float theta0,
		float weight
		)
	{
		Angle_cst angle_cst( atom1, atom2, atom3, theta0, weight);
		add_atom_angle_constraint(angle_cst);
	}

  /////////////////////////////////////////////////////////////////////////////
  void
  Cst_set::add_atom_angle_constraint(
    Angle_cst const & angle_cst
    )
  {
    if( find( angle_cst_list.begin(), angle_cst_list.end(), angle_cst )
				== angle_cst_list.end() ) {
			angle_cst_list.push_back( angle_cst );
			kin::Atom_id atom1, atom2, atom3;
			angle_cst.extract_atom_ids(atom1,atom2,atom3);

			++angle_cst_count[ atom1 ];
			++angle_cst_count[ atom2 ];
			++angle_cst_count[ atom3 ];
		}
  }

	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::add_atom_angle_constraint(
		kin::Atom_id const & atom1,
		kin::Atom_id const & atom2,
		kin::Atom_id const & atom3,
		float theta0,
		float theta_sd,
		float periodic_value,
		float weight
		)
	{
		Angle_cst angle_cst( atom1, atom2, atom3, theta0, theta_sd, periodic_value, weight);
		add_atom_angle_constraint(angle_cst);
	}

	/////////////////////////////////////////////////////////////////////////////
	// this is inefficient: we loop over all the constraints
	// could add an indexing scheme by atom
	void
	Cst_set::atom_angle_deriv(
		kin::Atom_id const & atom,
		kin::Coords const & coords,
		float const weight,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		if ( angle_cst_count.count( atom ) == 0 ) return;

		numeric::xyzVector_float f1(0.0), f2(0.0);
		for ( std::vector< Angle_cst >::const_iterator it= angle_cst_list.begin(),
						it_end= angle_cst_list.end(); it != it_end; ++it ) {
			it->fill_f1_f2( atom, coords, f1, f2 );
		}
		F1+= weight*f1;
		F2+= weight*f2;
	}



	/////////////////////////////////////////////////////////////////////////////
	float
	Cst_set::atom_torsion_score(
		kin::Coords const & coords
	) const
	{
		float score(0.0);
		for ( std::vector< Torsion_cst >::const_iterator
						it= torsion_cst_list.begin(),
						it_end= torsion_cst_list.end(); it != it_end; ++it ) {
			score += it->score( coords );
		}
		return score;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::add_atom_torsion_constraint(
		kin::Atom_id const & atom1,
		kin::Atom_id const & atom2,
		kin::Atom_id const & atom3,
		kin::Atom_id const & atom4,
		float theta0,
		float weight
		)
	{
		Torsion_cst torsion_cst( atom1, atom2, atom3, atom4, theta0, weight);
		add_atom_torsion_constraint( torsion_cst );
	}

  /////////////////////////////////////////////////////////////////////////////
  void
  Cst_set::add_atom_torsion_constraint(
    Torsion_cst const & torsion_cst
    )  {
    if( find( torsion_cst_list.begin(), torsion_cst_list.end(), torsion_cst )
				== torsion_cst_list.end() ) {
			torsion_cst_list.push_back( torsion_cst );

			kin::Atom_id atom1,atom2,atom3,atom4;
			torsion_cst.extract_atom_ids( atom1, atom2, atom3, atom4 );
			++torsion_cst_count[ atom1 ];
			++torsion_cst_count[ atom2 ];
			++torsion_cst_count[ atom3 ];
			++torsion_cst_count[ atom4 ];
		}
   }


	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::add_atom_torsion_constraint(
		kin::Atom_id const & atom1,
		kin::Atom_id const & atom2,
		kin::Atom_id const & atom3,
		kin::Atom_id const & atom4,
		float theta0,
		float theta_sd,
		float periodic_value,
		float weight
		)
	{
		Torsion_cst torsion_cst( atom1, atom2, atom3, atom4,
														 theta0, theta_sd, periodic_value, weight);
		add_atom_torsion_constraint( torsion_cst );
	}


	/////////////////////////////////////////////////////////////////////////////
	// this is inefficient: we loop over all the constraints
	// could add an indexing scheme by atom
	void
	Cst_set::atom_torsion_deriv(
		kin::Atom_id const & atom,
		kin::Coords const & coords,
		float const weight,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		if ( torsion_cst_count.count( atom ) == 0 ) return;

		numeric::xyzVector_float f1(0.0), f2(0.0);
		for ( std::vector< Torsion_cst >::const_iterator
						it= torsion_cst_list.begin(),
						it_end= torsion_cst_list.end(); it != it_end; ++it ) {
			it->fill_f1_f2( atom, coords, f1, f2 );
		}
		F1+= weight*f1;
		F2+= weight*f2;
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::atompair_cst_atom_deriv(
		kin::Atom_id const & atom,
		kin::Coords const & coords,
		FArray1D_int const & domain_map,
		float const atompair_cst_weight,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		using namespace numeric;
		using namespace kin;

		if ( !cst_map_by_atom.count( atom ) ) return;

		xyzVector_float xyz,xyz2;
		coords.get_xyz( atom, xyz );

		int const imap( domain_map( atom.rsd() ));

		Cst_map const & cst_map( cst_map_by_atom.find( atom )->second );
		for ( Cst_map::const_iterator it = cst_map.begin(), it_end = cst_map.end();
					it != it_end; ++it ) {
			Atom_id const & atom2( it->first );
			// skip atom pairs within the same rigid-body:
			if ( imap != 0 && domain_map( atom2.rsd() ) == imap ) continue;

			coords.get_xyz( atom2, xyz2 );
			xyzVector_float const f2( xyz - xyz2 );
			float const dist( f2.length() );
			float const deriv( it->second.dscore_dr( dist ) );
			if ( deriv != 0.0 ) {
				float const factor( atompair_cst_weight * deriv / dist );
				xyzVector_float f1( cross( xyz, xyz2 ) );
				F1 += factor * f1;
				F2 += factor * f2;
			}
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::coord_cst_atom_deriv(
		kin::Atom_id const & atom,
		kin::Coords const & coords,
		float const coord_cst_weight,
		numeric::xyzVector_float & F1,
		numeric::xyzVector_float & F2
	) const
	{
		using numeric::xyzVector_float;
		if ( !coordinate_constraint_exists( atom )) return;

		xyzVector_float const xyz( coords.get_xyz( atom ) ),
			xyz2( coordinate_constraint( atom ) );


		// note that the derivative has a factor of r in it,
		// but the F1 and F2 vectors include a factor of 1/r (see
		// eval_de_dr) so these cancel out.
		xyzVector_float const f1( cross( xyz, xyz2 ) );
		xyzVector_float const f2( xyz - xyz2 );

		F1 += ( 2.0f * coord_cst_weight ) * f1;
		F2 += ( 2.0f * coord_cst_weight ) * f2;

	}

	/////////////////////////////////////////////////////////////////////////////
	// this assumes that the constraints have been loaded into the maps
	// symmetrically --> only counts pairs from the map if i<j or i==j,ii<jj

	float
	Cst_set::atompair_cst_score(
		kin::Coords const & coords
	) const
	{
		float score(0.0);
		for ( Cst_map_map::const_iterator it1=cst_map_by_atom.begin(),
						it1_end = cst_map_by_atom.end(); it1!= it1_end; ++it1 ) {
			kin::Atom_id const & atom1( it1->first );
			Cst_map const & cst_map( it1->second );
			numeric::xyzVector_float xyz1( coords.get_xyz( atom1 ) );
			for ( Cst_map::const_iterator it2=cst_map.begin(),
							it2_end = cst_map.end(); it2 != it2_end; ++it2 ) {
				kin::Atom_id const & atom2( it2->first );
				if ( atom1 < atom2 ) {
					float cstE;
					cstE = it2->second.score( distance( coords.get_xyz( atom2 ), xyz1 ));
					score += cstE;
					if( debug_output ) {
						fill_cstE_array( atom1.rsd(), atom2.rsd(), cstE );
						std::cout << "Atompair_ids: " << atom1 << " - " << atom2 << "\n";
					}
				}
			}
		}
		return score;
	}

	/////////////////////////////////////////////////////////////////////////////
	//
	float
	Cst_set::coord_cst_score(
		kin::Coords const & coords
	) const
	{
		float total_score(0.0);
		for ( std::map< kin::Atom_id, numeric::xyzVector_float >::const_iterator
						it = coord_cst_.begin(), it_end = coord_cst_.end();
					it != it_end; ++it ) {
			total_score += distance_squared( it->second, coords.get_xyz(it->first));
		}
		return total_score;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Cst_set::add_chainbreak(
		int const seqpos,
		float const weight
	)
	{
		chainbreak_weight_[ seqpos] = weight;
	}

	/////////////////////////////////////////////////////////////////////////////
	// Method that accepts another constraint set and adds all its constraints
	// to the calling constraint set.
	//
	// Notes:
	// Angle Torsion section does no error/duplicate checking, since reverse lookup on
	//		the three atoms is not supported, and would require a very inefficient
	//		explicit check against all prior entries.  So be careful.
	//
	// Chainbreak, coordinate, and torsion sections check for and ignore duplicates
	//
	//
	Cst_set &
	Cst_set::collect(
		Cst_set const & other_set
	)
	{
		// atompair constraints
//    Cst_map_map cst_map_by_atom;
//    std::map< std::pair< int, int >,
//              std::vector< Rsd_rsd_cst > > cst_map_by_rsd_pair;
//    float packer_atompair_weight;

		for( Cst_map_map::const_iterator
					atom_cst_map = other_set.cst_map_by_atom.begin() ;
					atom_cst_map != other_set.cst_map_by_atom.end() ;
					++atom_cst_map ) {
			kin::Atom_id atom1( atom_cst_map->first );
			for( Cst_map::const_iterator
						atom_cst = (atom_cst_map->second).begin() ;
						atom_cst != (atom_cst_map->second).end() ;
						++atom_cst ) {
				kin::Atom_id atom2( atom_cst->first );
				if( atom1 < atom2 ) {
					add_atompair_constraint( atom1, atom2, atom_cst->second );
				}
			}
		}

		// Rosetta-style torsion constraints
		for( std::map< std::pair< int, int >, float >::const_iterator
					tor_cst = other_set.rosetta_torsion_cst_.begin() ;
					tor_cst != other_set.rosetta_torsion_cst_.end() ;
					++tor_cst ) {
			if( rosetta_torsion_cst_.count( tor_cst->first ) == 0 ) {
				rosetta_torsion_cst_[ tor_cst->first ] = tor_cst->second;
			}
		}

		// coordinate constraints
		for( std::map< kin::Atom_id, numeric::xyzVector_float >::const_iterator
					coord_cst = other_set.coord_cst_.begin() ;
					coord_cst != other_set.coord_cst_.end() ;
					++coord_cst ) {
			if( coord_cst_.count( coord_cst->first ) == 0 ) {
				coord_cst_[ coord_cst->first ] = coord_cst->second;
			}
		}

		// angle constraints
		for( std::vector< Angle_cst >::const_iterator
					angle_cst = other_set.angle_cst_list.begin() ;
					angle_cst != other_set.angle_cst_list.end() ;
					++angle_cst ) {
			add_atom_angle_constraint(*angle_cst);
		}


		// torsion constraints
		for( std::vector< Torsion_cst >::const_iterator
					torsion_cst = other_set.torsion_cst_list.begin() ;
					torsion_cst != other_set.torsion_cst_list.end() ;
					++torsion_cst ) {
			add_atom_torsion_constraint(*torsion_cst);
		}

		// Kin-style torsion constraints
		for( std::map< kin::Torsion_id, std::pair< float, float > >::const_iterator
					tor_cst = other_set.kin_torsion_cst_.begin() ;
					tor_cst != other_set.kin_torsion_cst_.end() ;
					++tor_cst ) {
			if( kin_torsion_cst_.count( tor_cst->first ) == 0 ) {
				kin_torsion_cst_[ tor_cst->first ] = tor_cst->second;
			}
		}

		// chainbreak constraints
		for( std::map< int, float >::const_iterator
					chain_cst = other_set.chainbreak_weight_.begin() ;
					chain_cst != other_set.chainbreak_weight_.end() ;
					++chain_cst ) {
			if( !is_chainbreak( chain_cst->first ) ) {
				chainbreak_weight_[ chain_cst->first ] = chain_cst->second;
			}
		}

		// torsion constraints, indexed by kin::Torsion_id info
		for( std::map< kin::Torsion_id, std::pair< float, float > >::const_iterator
					 kin_torsion_cst = other_set.kin_torsion_cst_.begin() ;
					 kin_torsion_cst != other_set.kin_torsion_cst_.end() ;
				 ++kin_torsion_cst){
			add_kin_torsion_constraint( kin_torsion_cst->first,
																	(kin_torsion_cst->second).first,
																	(kin_torsion_cst->second).second);
		}

		return *this;
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin zero the cst sd for the Cst, Angle_cst, Torsion_cst,
	//lin because the fullatom pack_rotamer doesnot use them.
	void
	Cst_set::zero_atompair_cst_sd() {

		// Atompair constraints
		for( Cst_map_map::iterator
					atom_cst_map = cst_map_by_atom.begin() ;
					atom_cst_map != cst_map_by_atom.end() ;
					++atom_cst_map ) {
			for( Cst_map::iterator
						atom_cst = (atom_cst_map->second).begin() ;
						atom_cst != (atom_cst_map->second).end() ;
						++atom_cst ) {
				(atom_cst->second).set_sd( 0.0 );
			}
		}

		// Angle constriants
		for ( int ii=0,ie=angle_cst_list.size(); ii<ie; ii++){
			angle_cst_list[ii].set_sd( 0.0 );
		}

		// TORSION constriants
		for ( int ii=0,ie=torsion_cst_list.size(); ii<ie; ii++){
			torsion_cst_list[ii].set_sd( 0.0 );
		}

	}

	/////////////////////////////////////////////////////////////////////////////
	//lin zero the cst sd for the Cst, Angle_cst, Torsion_cst,
	//lin because the fullatom pack_rotamer doesnot use them.
	void
	Cst_set::cst_take_input_value( pose_ns::Pose & pose ) {

		// Atompair constraint
		for( Cst_map_map::iterator
					atom_cst_map = cst_map_by_atom.begin() ;
					atom_cst_map != cst_map_by_atom.end() ;
					++atom_cst_map ) {
			for( Cst_map::iterator
						atom_cst = (atom_cst_map->second).begin() ;
						atom_cst != (atom_cst_map->second).end() ;
						++atom_cst ) {
				if( (atom_cst->second).use_input_value() ) {
					float dis ( pose.get_distance( atom_cst_map->first, atom_cst->first ) );
					(atom_cst->second).set_value( dis );
				}
			}
		}

		// Angle constriants
		for ( int ii=0,ie=angle_cst_list.size(); ii<ie; ii++){
			if( angle_cst_list[ii].use_input_value() ) {
				kin::Atom_id atom1, atom2, atom3;
				angle_cst_list[ii].extract_atom_ids( atom1, atom2, atom3 );
				float theta ( pose.get_angle( atom1, atom2, atom3 ) );
				angle_cst_list[ii].set_value( theta );
			}
		}

		// TORSION constriants
		for ( int ii=0,ie=torsion_cst_list.size(); ii<ie; ii++){
			if( torsion_cst_list[ii].use_input_value() ) {
				kin::Atom_id atom1, atom2, atom3,atom4;
				torsion_cst_list[ii].extract_atom_ids( atom1, atom2, atom3, atom4 );
				float theta ( pose.get_torsion_angle( atom1, atom2, atom3, atom4 ) );
				torsion_cst_list[ii].set_value( theta );
			}
		}

	}

	/////////////////////////////////////////////////////////////////////////////
	//lin for the Cst, Angle_cst, Torsion_cst,
	//lin currently it is not for 1D_cst, Coord_cst, and Chainbreak_cst
	//lin because the fullatom pack_rotamer doesnot use them.
	void
	Cst_set::get_packer_cst(
         pose_ns::Pose const & pose,
				 packer_cst_ns::Packer_cst_set & packer_cst ) const {

		// Atompair constraints
		for( Cst_map_map::const_iterator
					atom_cst_map = cst_map_by_atom.begin() ;
					atom_cst_map != cst_map_by_atom.end() ;
					++atom_cst_map ) {
			for( Cst_map::const_iterator
						atom_cst = (atom_cst_map->second).begin() ;
						atom_cst != (atom_cst_map->second).end() ;
						++atom_cst ) {
				packer_cst.add_distance_constraint( pose, atom_cst_map->first, atom_cst->first,
																 atom_cst->second );
			}
		}

		// Angle constriants
		for ( int ii=0,ie=angle_cst_list.size(); ii<ie; ii++){
			packer_cst.add_angle_constraint( pose, angle_cst_list[ii] );
		}

		// TORSION constriants
		for ( int ii=0,ie=torsion_cst_list.size(); ii<ie; ii++){
			packer_cst.add_torsion_constraint( pose, torsion_cst_list[ii] );
		}

		// chainbreak constraints

		// coordinate constraints

	}

	/////////////////////////////////////////////////////////////////////////////
	//lin show the cst
  std::ostream & operator <<( std::ostream & s, const Cst & d) {
		s << " DISTANCE_CST: Func: " << d.func_p->func_name()
			<<" r0: "<< d.r0 <<" r_sd: "<< d.r_sd
			<<" weight1: "<< d.weight1<<" weight2: "<< d.weight2<<"\n";
		return s;
	}

	std::ostream& operator<< ( std::ostream& s, const Rsd_rsd_cst& d ) {
		s <<" atom1: " << d.atomno1 << " atom2: " << d.atomno2 << d.cst;
		return s;
	}

	std::ostream& operator<< ( std::ostream& s, const Torsion_cst& d ) {
		float const rad2deg( numeric::conversions::degrees(1.0) );
		s << " TORSION_CST: " << " Func: "<< d.func_p->func_name()
			<< " theta0: " << d.theta0*rad2deg
			<< " theta_sd: " << d.theta_sd*rad2deg << " periodic_val: "
			<< d.periodic_val*rad2deg << " weight: " << d.weight  << "\n";
		s << "Torsion_ids: "<< d.atom1 << " - "<< d.atom2
			<< " - " << d.atom3 << " - " << d.atom4 << "\n";
		return s;
	}

	std::ostream& operator<< ( std::ostream& s, const Angle_cst& d ) {
		float const rad2deg( numeric::conversions::degrees(1.0) );
		s << " ANGLE_CST: " << " Func: "<< d.func_p->func_name()
			<< " theta0: " << d.theta0*rad2deg
			<< " theta_sd: " << d.theta_sd*rad2deg << " periodic_val: "
			<< d.periodic_val*rad2deg << " weight: " << d.weight  << "\n";
		s << "Angle_ids: "<< d.atom1 << " - " << d.atom2
			<< " - "<< d.atom3 << "\n";
		return s;
	}

	/////////////////////////////////////////////////////////////////////////////
	//lin method for comparison
  bool operator ==( const Cst & d1, const Cst & d2) {
		return ( std::abs( d1.r0 - d2.r0 ) < 1e-6 && std::abs( d1.r_sd - d2.r_sd ) < 1e-6
						 && std::abs( d1.weight1 - d2.weight1 ) < 1e-6
						 && std::abs( d1.weight2 - d2.weight2 ) < 1e-6 );
	}

	bool operator== ( const Torsion_cst& d1, const Torsion_cst& d2 ) {
		return ( d1.atom1 == d2.atom1 && d1.atom2 == d2.atom2
						 && d1.atom3 == d2.atom3 &&  d1.atom4 == d2.atom4
						 && std::abs( d1.theta0 - d2.theta0 ) < 1e-6
						 && std::abs( d1.theta_sd - d2.theta_sd ) < 1e-6
						 && std::abs( d1.weight - d2.weight ) < 1e-6 );
	}

	bool operator== ( const Angle_cst& d1, const Angle_cst& d2 ) {
		return ( d1.atom1 == d2.atom1 && d1.atom2 == d2.atom2
						 && d1.atom3 == d2.atom3
						 && std::abs( d1.theta0 - d2.theta0 ) < 1e-6
						 && std::abs( d1.theta_sd - d2.theta_sd ) < 1e-6
						 && std::abs( d1.weight - d2.weight ) < 1e-6 );
	}

	/////////////////////////////////////////////////////////////////
	// stream I/O of constraints set for diagnostic purposes
	//lin output the Cst, Angle_cst, Torsion_cst, Coord_cst, and Chainbreak_cst

  std::ostream & operator <<( std::ostream & os, const Cst_set & cs)
	{
		// Atompair constraints
		for( Cst_map_map::const_iterator
					atom_cst_map = cs.cst_map_by_atom.begin() ;
					atom_cst_map != cs.cst_map_by_atom.end() ;
					++atom_cst_map ) {
			kin::Atom_id atom1( atom_cst_map->first );
			for( Cst_map::const_iterator
						atom_cst = (atom_cst_map->second).begin() ;
						atom_cst != (atom_cst_map->second).end() ;
						++atom_cst ) {
				kin::Atom_id atom2( atom_cst->first );
				if( atom1 < atom2 ) {
					os << "DISTANCE_CST: " << atom1 << " " << atom2 << atom_cst->second << "\n";
				}
			}
		}


		// coordinate constraints
		for( std::map< kin::Atom_id, numeric::xyzVector_float >::const_iterator
					coord_cst = cs.coord_cst_.begin() ;
					coord_cst != cs.coord_cst_.end() ;
					++coord_cst ) {
			std::cout << "Coordinate cst at position " << coord_cst->first <<
										"\n";
		}

		// Angle constriants
		for ( int ii=0,ie=cs.angle_cst_list.size(); ii<ie; ii++){
			os <<cs.angle_cst_list[ii];
		}

		// TORSION constriants
		for ( int ii=0,ie=cs.torsion_cst_list.size(); ii<ie; ii++){
			os <<cs.torsion_cst_list[ii];
		}

		// chainbreak constraints
		for( std::map< int, float >::const_iterator
					chain_cst = cs.chainbreak_weight_.begin() ;
					chain_cst != cs.chainbreak_weight_.end() ;
					++chain_cst ) {
			std::cout << "Chainbreak at position " << chain_cst->first <<
										" with weight " << chain_cst->second << "\n";
		}

		return os;
	}


} // namespace cst_set_ns


