// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "decoystats_classes.h"
#include "decoystats.h"
#include "docking_ns.h"
#include "misc.h"
#include "pdb.h"
#include "template_pack.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <iostream>
#include <sstream>
#include <string>


//Utility Headers
#include <utility/basic_sys_util.hh>

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::unsatisfied_buried_polar
///
/// @brief: equality operator for class unsatisfied_buried_polar
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/27/05
/////////////////////////////////////////////////////////////////////////////////
bool unsatisfied_buried_polar::operator == ( unsatisfied_buried_polar& rhs ) const {

	// jk require that seqpos and atm both match
	if ( ( seqpos == rhs.seqpos ) && ( atm == rhs.atm ) ) return true;
	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::unsatisfied_buried_polar
///
/// @brief: "less-than" operator for class unsatisfied_buried_polar
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/27/05
/////////////////////////////////////////////////////////////////////////////////
bool unsatisfied_buried_polar::operator < ( unsatisfied_buried_polar& rhs ) const {

	// jk sort first on seqpos (lowest to highest)
	if ( seqpos < rhs.seqpos ) return true;
	if ( rhs.seqpos < seqpos ) return false;

	// jk sort next on "backbone before sidechain"
	if ( is_backbone && ! rhs.is_backbone ) return true;
	if ( rhs.is_backbone && ! is_backbone ) return false;

	// jk sort next on "donors before acceptors"
	if ( is_donor && ! rhs.is_donor ) return true;
	if ( rhs.is_donor && ! is_donor ) return false;

	// jk break all remaining ties by sorting on atom number
	if ( atm < rhs.atm ) return true;
	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::set_all
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
unsatisfied_buried_polar::set_all(
    const bool in_is_donor, const bool in_is_backbone, const bool in_is_satisfied,
		const std::string in_start_file, const std::string in_residue_type, const int in_seqpos,
		const int in_atm, const float in_hbE, const std::string in_atom_name,
		const float in_atom_sasa, const float in_atom_sasa10, const float in_atom_sasa7
) {

	is_donor = in_is_donor;
	is_backbone = in_is_backbone;
	is_satisfied = in_is_satisfied;
	start_file = in_start_file;
	residue_type = in_residue_type;
	seqpos = in_seqpos;
	atm = in_atm;
	hbE = in_hbE;
	atom_name = in_atom_name;
	atom_sasa = in_atom_sasa;
	atom_sasa10 = in_atom_sasa10;
	atom_sasa7 = in_atom_sasa7;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::group_type
///
/// @brief
///   if this uns can participate in a group, the name of the group type
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
unsatisfied_buried_polar::group_type() const {

	int group_type(0);

	if ( is_backbone ) {
		if ( residue_type == "  A" || residue_type == "  C" ||
				 residue_type == "  G" || residue_type == "  T" ) { // DNA
			group_type = CO; // hack proxy for phosphate
		}
		else if ( is_donor ) group_type = BBH;
		else group_type = BBO;

	} else {
		if ( residue_type == "LYS" ) group_type = NH3_plus;
		else if ( residue_type == "TRP" ) group_type = NHaro;
		else if ( residue_type == "CYS" ) group_type = SH;

		else if ( ( residue_type == "SER") ||
							( residue_type == "THR") ||
							( residue_type == "TYR") ) {
			group_type = OH;

		} else if ( ( residue_type == "ASP") || ( residue_type == "GLU") ) {
			group_type = COO_minus;

		} else if ( ( residue_type == "ASN") || ( residue_type == "GLN") ) {
			if ( is_donor ) group_type = NH2;
			else group_type = CO;

		} else if ( residue_type == "ARG") {
			if ( atm == 17 ) group_type =NHarg; // H on Arg NE
			else group_type = NH2_plus; // H on Arg NH1, NH2

		} else if ( residue_type == "HIS") {
			if ( is_donor ) group_type = NHaro_plus;
			else group_type = N;

		//ja double-stranded B-form DNA, standard major groove groups only
		} else if ( residue_type == "  A" ) { // adenine
			 // exo-N6 - skip atom number 30 (1H6) because it should always basepair
			if ( atm == 31 ) group_type = NH2;
			else if ( atm == 19 ) group_type = N; // N7
			else group_type = IGNORE;

		} else if ( residue_type == "  C" ) { // cytosine
			// exo-N4 - skip atom number 28 (2H4) because it should always basepair
			if ( atm == 27 ) group_type = NH2;
			else group_type = IGNORE;

		} else if ( residue_type == "  G" ) { // guanine
			if ( atm == 19 ) group_type = CO; // exo-O6
			else if ( atm == 21 ) group_type = N; // N7
			else group_type = IGNORE;

		} else if ( residue_type == "  T" ) { // thymine
			if ( atm == 17 ) group_type = CO; // exo-O4
			else group_type = IGNORE;

		//ja RNA... to be completed (placeholder to prevent crash for now)
		} else if ( residue_type == " rA" || residue_type == " rC" ||
		            residue_type == " rG" || residue_type == " rU" ) {
			group_type = IGNORE;

		} else {
			std::cout << "Unknown case for group assignment: residue " <<
			 residue_type << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}
	return group_type;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::weight
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
unsatisfied_buried_polar::weight() const {

	float w(0.);

	// jk Values based on data collected July '05
	// jk Group UNS values match individual UNS, except for NH2_plus, NH2, and OH groups

	if ( is_backbone ) {
		if ( is_donor ) w=1.08;
		else w=1.07;

	} else {
		int gt = group_type();

		if ( gt == NH3_plus )         w=0.48;
		else if ( gt == NH2_plus )    w=0.9;
		else if ( gt == NH2 )         w=0.77;
		else if ( gt == NHaro_plus )  w=2.12;
		else if ( gt == NHaro )       w=2.12;
		else if ( gt == NHarg )       w=0.98;
		else if ( gt == COO_minus )   w=1.79;
		else if ( gt == CO )          w=1.27;
		else if ( gt == OH ) {
			if ( is_donor )             w=1.1;
			else                        w=0.92;
		}
		else if ( gt == SH )          w=0.;
		else if ( gt == N )           w=0.48;
		else if ( gt == IGNORE )      w=0.;
		else {
			std::cout << "Unknown group in uns weight: " << gt << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

	return w;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::uns_group
///
/// @brief
///   Given a list of uns including "self", find a group within this list
///
/// @detailed
///   Note: works in an asymmetric way
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
unsatisfied_buried_polar::uns_group( std::list < unsatisfied_buried_polar > uns_list ) const {

	// Some UNS comprise a group on their own (backbone polars, etc.)
	int gt = group_type();
	if ( gt == BBH ) return true;  // backbone
	if ( gt == BBO ) return true;  // backbone
	if ( gt == NHaro ) return true;  // Trp
	if ( gt == NHaro_plus ) return true;  // His
	if ( gt == NHarg ) return true;  // Arg NE
	if ( gt == NH3_plus ) return true;  // Lys
	if ( gt == N ) return true;  // His
	if ( gt == COO_minus ) return true;  // Asp, Glu
	if ( gt == CO ) return true;  // Asn, Gln

	// Groups can exist for the following types:
	//    NH2 (Asn, Gln), NH2+ (Arg), SH (Cys), and OH (Ser, Thr, Tyr)

	// JK THERE'S PROBABLY A POTENTIAL ERROR WITH ARG HERE, SINCE ONE UNS PROTON
	// ON EACH OF THE OUTER N'S WOULD RETURN THE SAME GROUP
	// (BUT SHOULDN'T COUNT AS A GROUP!)

	for( std::list<unsatisfied_buried_polar>::iterator uns = uns_list.begin();
			 uns != uns_list.end(); ++uns ) {
		// Only consider other UNS in which the residue num matches but it's not
		// the same atom. The "<" is so that this test is asymmetric, ie. the
		// "partner" UNS won't return true, because this would return duplicates if
		// this method is called for each element in a list.
		if ( ( uns->get_seqpos() == seqpos ) && ( uns->get_atm() < atm ) ) {
			// Return true if these are in the same group
			if ( uns->group_type() == gt ) {
				return true;
			}
		}
	}

	return false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::write_to_PDB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
unsatisfied_buried_polar::write_to_PDB(
	const std::string tag,
	const std::string uns_type
) const {

	std::string id = tag;
	if (tag == "") id = "DS";
	std::string hb_state="UNS";
	if ( is_satisfied ) hb_state="SAT";

	std::ostringstream output_stream;
	output_stream << id << " " << uns_type << " " << hb_state <<
		' ' << start_file << ' ' << residue_type << ' ' <<
		' ' << I( 4, seqpos ) << " " << I( 4, pdb::pdb_res_num(seqpos) ) <<
		' ' << pdb::res_chain(seqpos) << " " << I( 4, atm ) <<
		' ' << F( 9, 3, hbE ) << ' ' << atom_name <<
		' ' << F( 9, 3, atom_sasa ) <<
		' ' << F( 9, 3, atom_sasa10 ) <<
		' ' << F( 9, 3, atom_sasa7 );

	decoystats_store_output_line( output_stream.str() );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_polar::write_to_cout
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
unsatisfied_buried_polar::write_to_cout(
	const std::string tag,
	const std::string uns_type
) const {

	std::string id = tag;
	if (tag == "") id = "DS";
	std::string hb_state="UNS";
	if ( is_satisfied ) hb_state="SAT";

	std::cout << id << " " << uns_type << " " << hb_state <<
		' ' << start_file << ' ' << residue_type << ' ' <<
		' ' << I( 4, seqpos ) << " " << I( 4, pdb::pdb_res_num(seqpos) ) <<
		' ' << pdb::res_chain(seqpos) << " " << I( 4, atm ) <<
		' ' << F( 9, 3, hbE ) << ' ' << atom_name <<
		' ' << F( 9, 3, atom_sasa ) <<
		' ' << F( 9, 3, atom_sasa10 ) <<
		' ' << F( 9, 3, atom_sasa7 ) << std::endl;

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::unsatisfied_buried_group
///
/// @brief: constructor for class unsatisfied_buried_group
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/27/05
/////////////////////////////////////////////////////////////////////////////////
unsatisfied_buried_group::unsatisfied_buried_group( const int in_seqpos, const int in_type )
{

	seqpos = in_seqpos;
	group_type = in_type;
	is_backbone = false;
	if ( ( group_type == BBH ) || ( group_type == BBO ) ) is_backbone = true;

	return;

}

//////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::unsatisfied_buried_group
///
/// @brief: constructor for class unsatisfied_buried_group
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/27/05
/////////////////////////////////////////////////////////////////////////////////
unsatisfied_buried_group::unsatisfied_buried_group(
  const int in_seqpos,
  const std::string in_type
)
{

	seqpos = in_seqpos;

	if ( in_type == "BBH" ) {
		group_type = BBH;
	} else if ( in_type == "BBO" ) {
		group_type = BBO;
	} else if ( ( in_type == "NH3+" ) || ( in_type == "NH3_plus" ) ) {
		group_type = NH3_plus;
	} else if ( ( in_type == "NH2+" ) || ( in_type == "NH2_plus" ) ) {
		group_type = NH2_plus;
	} else if ( in_type == "NH2" ) {
		group_type = NH2;
	} else if ( ( in_type == "NHaro+" ) || ( in_type == "NHaro_plus" ) ) {
		group_type = NHaro_plus;
	} else if ( in_type == "NHaro" ) {
		group_type = NHaro;
	} else if ( in_type == "NHarg" ) {
		group_type = NHarg;
	} else if ( ( in_type == "COO-" ) || ( in_type == "COO_minus" ) ) {
		group_type = COO_minus;
	} else if ( in_type == "CO" ) {
		group_type = CO;
	} else if ( in_type == "OH" ) {
		group_type = OH;
	} else if ( in_type == "SH" ) {
		group_type = SH;
	} else if ( in_type == "N" ) {
		group_type = N;

	} else {
		std::cout << "Unknown input group type: " << in_type << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	is_backbone = false;
	if ( ( group_type == BBH ) || ( group_type == BBO ) ) is_backbone = true;

	return;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::unsatisfied_buried_group
///
/// @brief: equality operator for class unsatisfied_buried_group
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/27/05
/////////////////////////////////////////////////////////////////////////////////
bool unsatisfied_buried_group::operator == ( unsatisfied_buried_group& rhs ) const {

	// jk require that seqpos and group_type both match
	if ( ( seqpos == rhs.seqpos ) && ( group_type == rhs.group_type ) ) return true;
	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::unsatisfied_buried_group
///
/// @brief: "less-than" operator for class unsatisfied_buried_group
///
/// @detailed
///  Note:  decides based on WEIGHT, not seqpos (as unsatisfied_buried_polar does)
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/27/05
/////////////////////////////////////////////////////////////////////////////////
bool unsatisfied_buried_group::operator < ( unsatisfied_buried_group& rhs ) const {

	// jk For now, sort by weight (ie. seriousness) of the UNS
	// jk Note: alternatively, it would also make sense to sort by burial (since buried ones are harder to fix...)

	// jk Note: with this operation, higher weights should appear first in the list

	// jk if the group types match, sort on seqpos (lowest to highest)
	if ( group_type == rhs.group_type ) {
		if ( seqpos < rhs.seqpos ) return true;
		if ( seqpos > rhs.seqpos ) return false;
	}

	// jk sort first on "backbone before sidechain"
	if ( is_backbone && ! rhs.is_backbone ) return true;
	if ( rhs.is_backbone && ! is_backbone ) return false;

	// jk sort next on weight (highest to lowest)
	if ( weight() > rhs.weight() ) return true;
	if ( weight() < rhs.weight() ) return false;

	return false; // never used (but for safety since we have float comparisons above)

}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::get_type_name
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
std::string
unsatisfied_buried_group::get_type_name() const {

	if ( group_type == BBH ) return "BBH";
	else if ( group_type == BBO ) return "BBO";
	else if ( group_type == NH3_plus ) return "NH3+";
	else if ( group_type == NH2_plus ) return "NH2+";
	else if ( group_type == NH2 ) return "NH2";
	else if ( group_type == NHaro_plus ) return "NHaro+";
	else if ( group_type == NHaro ) return "NHaro";
	else if ( group_type == NHarg ) return "NHarg";
	else if ( group_type == COO_minus ) return "COO-";
	else if ( group_type == CO ) return "CO";
	else if ( group_type == OH ) return "OH";
	else if ( group_type == SH ) return "SH";
	else if ( group_type == N ) return "N";
	else if ( group_type == IGNORE ) return "IGNORE";

	else {
		std::cout << "Unknown group in get_type_name: " << group_type << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return "";
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::weight
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
unsatisfied_buried_group::weight() const {

	float w(0.);

	// jk Values based on data collected July '05
	// jk Group UNS values match individual UNS, except for NH2_plus, NH2, and OH groups

	if ( group_type == BBH )              w=1.08;
	else if ( group_type == BBO )         w=1.07;
	else if ( group_type == NH3_plus )    w=0.48;
	else if ( group_type == NH2_plus )    w=2.62;
	else if ( group_type == NH2 )         w=2.14;
	else if ( group_type == NHaro_plus )  w=2.12;
	else if ( group_type == NHaro )       w=2.12;
	else if ( group_type == NHarg )       w=0.98;
	else if ( group_type == COO_minus )   w=1.79;
	else if ( group_type == CO )          w=1.27;
	else if ( group_type == OH )          w=1.48;
	else if ( group_type == SH )          w=0.;
	else if ( group_type == N )           w=0.48;
	else if ( group_type == IGNORE )      w=0.;
	else {
		std::cout << "Unknown case for group weight: group_type " << group_type <<
		 std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return w;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::write_to_PDB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
unsatisfied_buried_group::write_to_PDB( const std::string tag ) const {

	using namespace misc;

	std::string id = tag;
	if (tag != "") id = tag+" ";

	std::ostringstream output_stream;
	output_stream << id << "GU " << residue3(seqpos) << " " <<
		A(5,get_type_name()) << " " << I(4,seqpos) << " " <<
		I(4,pdb::pdb_res_num(seqpos)) << " " << weight();

	decoystats_store_output_line( output_stream.str() );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin unsatisfied_buried_group::write_to_cout
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
unsatisfied_buried_group::write_to_cout( const std::string tag ) const {

	using namespace misc;

	std::string id = tag;
	if (tag != "") id = tag+" ";

	std::cout << id << "GU " << residue3(seqpos) << " " <<
		A(5,get_type_name()) << " " << I(4,seqpos) << " " <<
		I(4,pdb::pdb_res_num(seqpos)) << std::endl;

	return;
}
