// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.36 $
//  $Date: 2005/10/26 23:31:43 $
//  $Author: sheffler $


// Rosetta Headers
#include "decoystats_interface.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "AnkyrinRepeat_ns.h"
#include "count_pair.h"
#include "decoystats.h"
#include "decoystats_classes.h"
#include "decoystats_ns.h"
#include "decoystats_interface_classes.h"
#include "design.h"
#include "design_structure.h"
#include "docking.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fullatom_energy.h"
#include "fullatom_sasa.h"
#include "fullatom_energies.h"
#include "hotspot_ns.h"
#include "hotspot_residue.h"
#include "loop_class.h"
#include "loop_relax.h"
#include "make_pdb.h"
#include "misc.h"
#include "namespace_fullatom_flag.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_io.h"
#include "read_aaproperties.h"
#include "score.h"
#include "template_pack.h"
#include "util_vector.h"
#include "void.h"
#include "void_ns.h"
#include "water_ns.h"
#include "packing_measures.h"

// Numeric Headers
#include <src/numeric/constants.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <algorithm>
#include <cassert>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <list>
#include <sstream>
#include <string>
#include <vector>

static int const MAX_QUANTIFIERS = 30;

//////////////////////////////////////////////////////////////////////////////
/// @begin set_interface_ds_flag
///
/// @brief
///
/// @detailed
///
/// @param  setting - [in/out]?
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
set_interface_ds_flag( bool setting )
{
	using namespace decoystats;

	interface_ds_flag = setting;
	set_decoystats_flag(true);
}

//////////////////////////////////////////////////////////////////////////////
/// @begin get_interface_ds_flag
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
get_interface_ds_flag()
{
	using namespace decoystats;

	return interface_ds_flag;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::interface_ds
///
/// @brief: constructor for class interface_ds
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors: John Karanicolas
///
/// @last_modified: 11/14/05
/////////////////////////////////////////////////////////////////////////////////
interface_ds::interface_ds( pose_ns::Pose * input_pose_ptr ) {

	using namespace param;

	pose_ = input_pose_ptr;
	NRES_ = pose_->total_residue();

	interface_res.dimension( NRES_ );
	core_interface_res.dimension( NRES_ );
	interface_atom.dimension( param::MAX_ATOM(), NRES_ );
	res_sasa_diff.dimension( NRES_ );
	atom_sasa_diff.dimension( param::MAX_ATOM()(), NRES_ );
	hotspot_res.dimension( NRES_ );
	stack_res.dimension( NRES_ );
	buried_stack_res.dimension( NRES_ );
	energy_quantifiers.resize(MAX_QUANTIFIERS);

	energies_initialized = false;
	done_packing_measures_init = false;
	done_finding_hotspots = false;
	done_finding_stacks = false;

	max_interface_void_volume = 0.;
	max_interface_void_radius = 0.;
	total_interface_void_volume = 0.;
	num_hotspot_residues=0;
	hotspot_res=false;

	setup_energetic_quantifiers();
	compute_quantifiers();

}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::initialize_packing_measures
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::initialize_interface_packing_measures() {

	using namespace decoystats;
	using namespace param;
	using namespace numeric::constants::f;

	if ( done_packing_measures_init ) return;

	max_interface_void_volume = 0.;
	max_interface_void_radius = 0.;
	total_interface_void_volume = 0.;

	// jk Compute ball sasa score, etc.
	ps.init();
	ps.compute_pruned_bsasa_score();
	ps.compute_atom_bsasa_score();
	ps.compute_cavity_ball_burial(false);
	ps.include_packing_in_outpdbs();

	std::vector< packing_ns::CavityBall > interface_holes;
	ps.build_restricted_hole_set( interface_atom, interface_holes );
	std::vector< packing_ns::CavityBallCluster > interface_hole_clusters;
	ps.build_restricted_cluster_set( interface_atom, interface_hole_clusters );

	for (int ii=0; ii < (int)interface_holes.size(); ++ii) {
		float const void_radius = (interface_holes[ii]).radius();
		if ( void_radius > max_interface_void_radius ) {
			max_interface_void_radius = void_radius;
		}
	}

	for (int ii=0; ii < (int)interface_hole_clusters.size(); ++ii) {
		float const void_volume = (interface_hole_clusters[ii]).volume();
		if ( void_volume > max_interface_void_volume ) {
			max_interface_void_volume = void_volume;
		}
		total_interface_void_volume += void_volume;
	}

	done_packing_measures_init = true;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::define_interface
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::define_interface() {

	using namespace decoystats;
	using namespace param;

	// store the residue SASA for the complex using a small probe (used elsewhere)
	res_sasa_smallprobe.dimension( NRES_ );
	atom_sasa_smallprobe.dimension( MAX_ATOM()(), NRES_ );
	calc_per_atom_sasa( atom_sasa_smallprobe, res_sasa_smallprobe, 1.0, true, false );

	// initialize arrays denoting interface residues
	interface_res=false;
	core_interface_res=false;
	interface_atom=false;

	res_sasa_complex.dimension( NRES_ );
	atom_sasa_complex.dimension( MAX_ATOM()(), NRES_ );

	FArray1D_float res_sasa_unbound( NRES_ );
	FArray2D_float atom_sasa_unbound( MAX_ATOM()(), NRES_ );

	float const probe_radius = { 1.4 };

	// jk NOTE: interface SASA numbers will be computed IGNORING HYDROGENS AND WATERS

	// compute the SASA for the complex
	calc_per_atom_sasa( atom_sasa_complex, res_sasa_complex,
    probe_radius, true, false );

	// compute the SASA for the unbound
	calc_per_atom_sasa( atom_sasa_unbound, res_sasa_unbound,
	  probe_radius, true, true );

	// compute SASA differences
	res_sasa_diff = res_sasa_unbound - res_sasa_complex;
	atom_sasa_diff = atom_sasa_unbound - atom_sasa_complex;

	// jk Residue must lose this much SASA to be an interface residue
	float const res_thres = { 0.3 };

	// jk Residue must have less than this fraction of the
	// jk exposed total SASA to be "core"
	float const core_res_thres = { 0.10 };

	// jk Atom must lose this much SASA to be an interface atom
	float const atom_thres = { 0.1 };

	num_interface_res = 0;
	num_core_interface_res = 0;
	interface_chi_angles=0;
	num_interface_atom = 0;
	num_interface_water = 0;

	for ( int i = 1, max_atom = MAX_ATOM(); i <= NRES_; ++i ) {
		if ( res_sasa_diff(i) > res_thres ) {
			interface_res(i)=true;
			++num_interface_res;
			int const aa = pose_->res(i);
			int const aav = pose_->res_variant(i);
			interface_chi_angles += aaproperties_pack::nchi(aa,aav);
			if ( ( aa == param_aa::aa_ser ) || ( aa == param_aa::aa_thr ) || ( aa == param_aa::aa_tyr ) ) {
				// don't count hydroxyl chi angles - these are free...
				interface_chi_angles--;
			}

			if ( (res_sasa_complex(i)/rsd_exposed_sasa(aa)) < core_res_thres ) {
				core_interface_res(i)=true;
				++num_core_interface_res;
			}
		}
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( atom_sasa_diff(j,i) > atom_thres ) {
				interface_atom(j,i)=true;
				++num_interface_atom;
			}
		}
	}

	// jk make sure SASApack / SASAprob are up to date
	// note: calc_sasa_pack_score will call calc_sasa_prob,
	// so sasaprob will be updated too.
	//	calc_sasa_pack_score();

	find_stacks();

	if ( design::explicit_h2o || water::use_hetero_h2o )
		count_interface_waters( probe_radius, atom_thres );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::find_CBCB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::find_CBCB() {

	using namespace docking;
	using namespace param_aa;

	std::ofstream cbcbOUT("cbcb.txt",std::ios_base::app);

	for ( int i = part_begin(1); i <= part_end(1); ++i ) {
		int const aa1 = pose_->res(i);
		if ( interface_res(i) && ( aa1 != aa_gly ) ) {
			for ( int j = part_begin(2); j <= part_end(2); ++j ) {
				int const aa2 = pose_->res(j);
				if ( interface_res(j) && ( aa2 != aa_gly ) ) {
					float dist=0;
					distance_bk(pose_->full_coord()(1,5,i),pose_->full_coord()(1,5,j),dist);
					cbcbOUT << dist << std::endl;
				}
			}
		}
	}

	cbcbOUT.close();
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::find_stacks
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::find_stacks() {

	using namespace param_aa;

	stack_list.clear();
	num_stacks=0;
	stack_res=false;
	buried_stack_res=false;

	if ( done_finding_stacks ) return;

	// jk A stack residue must have less that this SASA to be considered "buried"
	// jk Note: extended to define a "stack" as a really good intermolecular interaction,
	// jk   NOT just an aromatic stack!!
	float const buried_stack_thres = { 15. };

	for ( int seqposA = 1; seqposA <= misc::domain_end(1); ++seqposA ) {
		if ( ! interface_res(seqposA) ) continue;
		int const aaA = pose_->res(seqposA);
		int const aavA = pose_->res_variant(seqposA);
		// jk Extend to allow L/I/V as well...
		if ( ( aaA != aa_phe ) && ( aaA != aa_ile ) && ( aaA != aa_leu ) &&
			( aaA != aa_val ) && ( aaA != aa_trp ) && ( aaA != aa_tyr ) ) continue;

		for ( int seqposB = misc::domain_end(1)+1; seqposB <= NRES_; ++seqposB ) {
			if ( ! interface_res(seqposB) ) continue;
			int const aaB = pose_->res(seqposB);
			int const aavB = pose_->res_variant(seqposB);
			// jk Extend to allow L/I/V as well...
			if ( ( aaB != aa_phe ) && ( aaB != aa_ile ) && ( aaB != aa_leu ) &&
				( aaB != aa_val ) && ( aaB != aa_trp ) && ( aaB != aa_tyr ) ) continue;

			aromatic_stack newstack(seqposA,aaA,aavA,seqposB,aaB,aavB);
			if ( test_for_stack(seqposA, seqposB, newstack) ) {
				++num_stacks;
				stack_list.push_back(newstack);
				stack_res(seqposA)=true;
				stack_res(seqposB)=true;
				if (res_sasa_complex(seqposA) <= buried_stack_thres) {
					buried_stack_res(seqposA)=true;
				}
				if (res_sasa_complex(seqposB) <= buried_stack_thres) {
					buried_stack_res(seqposB)=true;
				}
			}

		}
	}

	done_finding_stacks = true;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::test_for_stack
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
interface_ds::test_for_stack( int seqposA, int seqposB, aromatic_stack & stack ) {

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace docking;
	using namespace fullatom_energies;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;
	using namespace void_ns;

	if ( seqposA > seqposB ) {
		int tmp = seqposA;
		seqposA = seqposB;
		seqposB = tmp;
	}

	// jk Note: modifed such that nothing in here is specific to AROMATIC residues only...

	// jk Check the plane term first
	//	if ( plane_plane(seqposA,seqposB) > -0.0001 ) return false;

	// jk Residues must together lose this much SASA to be a stack
	// Note: as long as we have a good plane term and intra-stack ene, we don't really need this
	//       current threshold level makes it effectively disabled
	float const sasa_diff_thres = { 10. };

	// jk The more buried residue must have less than this total SASA in the complex
	float const bound_best_sasa_thres = { 15. };

	// jk The residues must together have less than this total SASA in the complex
	float const bound_tot_sasa_thres = { 45. };

	// jk The plane + atr energy between the two residues must be at least this good
	float const intra_thres = { -1.5 };

	// jk The fa_atr with the surroundings must be at least this good
	// jk Note: total fa_atr, not just across the interface
	float const atr_thres = { -5. };

	float const stack_SASAtot = res_sasa_complex(seqposA)+res_sasa_complex(seqposB);
	float const stack_bestSASAtot = min(res_sasa_complex(seqposA),res_sasa_complex(seqposB));
	float const stack_SASAdiff = res_sasa_diff(seqposA) + res_sasa_diff(seqposB);
	float const plane_ene = plane_plane(seqposA,seqposB);
	float const intrastack_ene = plane_ene + atr_pair(seqposA,seqposB);
	float const stack_atr_ene = atrenergy(seqposA) + atrenergy(seqposB) - atr_pair(seqposA,seqposB);

	if ( ( stack_SASAdiff >= sasa_diff_thres ) &&
		( stack_SASAtot <= bound_tot_sasa_thres ) &&
		( stack_bestSASAtot <= bound_best_sasa_thres ) &&
		( stack_atr_ene <= atr_thres ) &&
		( intrastack_ene <= intra_thres ) ) {
		stack.set_deltaSASA(stack_SASAdiff);
		stack.set_atr_ene(stack_atr_ene);
		stack.set_plane_ene(plane_ene);
		stack.set_intrastack_ene(intrastack_ene);
		stack.set_SASApack((rsd_sasa_pack_score(seqposA)+rsd_sasa_pack_score(seqposB))/2.);
		stack.set_SASAprob((rsd_sasaprob(seqposA)+rsd_sasaprob(seqposB))/2.);
		return true;
	}

	return false;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::find_hotspots
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::find_hotspots() {

	hotspot_list.clear();
	num_hotspot_residues=0;
	hotspot_res=false;

	if ( done_finding_hotspots ) return;

	for ( int seqpos = 1; seqpos <= NRES_; ++seqpos ) {
		hotspot_residue hr;
		if ( test_for_hotspot(seqpos, hr) ) {
			hotspot_res(seqpos)=true;
			++num_hotspot_residues;
			hotspot_list.push_back(hr);
		}
	}

	done_finding_hotspots = true;

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::test_for_hotspot
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
interface_ds::test_for_hotspot( int const seqpos, hotspot_residue & hr ) {

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace fullatom_energies;
	using namespace hotspot_ns;
	using namespace param;
	using namespace param_aa;
	using namespace void_ns;

	initialize_interface_packing_measures();

	int const aa = pose_->res(seqpos);
	int const aav = pose_->res_variant(seqpos);
	if ( interface_res(seqpos) && ( ( aa == aa_tyr ) || ( aa == aa_trp ) ) ) {
		hr.setup(seqpos,aa,aav);
		float const res_SASAdiff = res_sasa_diff(seqpos);
		float const res_atr_ene = atrenergy(seqpos);

		float res_ball_sasa(0.);
		int res_ball_sasa_natoms(0);
		for ( int j = 5, je = natoms(aa,aav); j <= je; ++j ) {
			float three_way_score = ps.atom_bsasa_score_weighted(j,seqpos);
			if ( three_way_score < 0. ) three_way_score = 0.;
			res_ball_sasa += three_way_score;
			++res_ball_sasa_natoms;
		}
		res_ball_sasa /= res_ball_sasa_natoms;

		float const maxvoid = ps.find_sidechain_max_hole(seqpos, true);

		hr.set_resSASA(res_sasa_complex(seqpos));
		hr.set_resSASA_smallprobe(res_sasa_smallprobe(seqpos));
		hr.set_deltaSASA(res_SASAdiff);
		hr.set_atr_ene(res_atr_ene);
		hr.set_SASApack(rsd_sasa_pack_score(seqpos));
		hr.set_SASAprob(rsd_sasaprob(seqpos));
		hr.set_ballSASAprob(res_ball_sasa);
		hr.set_maxvoid(maxvoid);

		float effective_atr_thres = hs_atr_thres;
		if ( aa == aa_trp ) effective_atr_thres -= 0.5;

		if ( ( res_SASAdiff >= hs_sasa_diff_thres ) &&
				 ( res_ball_sasa >= hs_res_ball_sasa_thres ) &&
				 ( res_atr_ene <= effective_atr_thres ) &&
				 ( res_sasa_complex(seqpos) <= hs_bound_sasa_thres ) &&
				 ( res_sasa_smallprobe(seqpos) <= hs_smallprobe_sasa_thres ) &&
				 ( rsd_sasa_pack_score(seqpos) <= hs_sasapack_thres ) &&
				 ( rsd_sasaprob(seqpos) >= hs_sasaprob_thres ) &&
				 ( maxvoid <= hs_maxvoid_thres ) ) {

			float sc_hbE = 0;
			// jk Donors - start at 2, to skip the backbone NH
			for ( int hnum = 2, hnume = nH_polar(aa,aav); hnum <= hnume; ++hnum ) {
				int const dhatm = Hpos_polar(hnum,aa,aav);
				sc_hbE += atom_hbondE(dhatm,seqpos);
			}
			// jk Acceptors - start at 2, to skip the backbone CO
			for ( int anum = 2, anume = nacceptors(aa,aav); anum <= anume; ++anum ) {
				int const aatm = accpt_pos(anum,aa,aav);
				sc_hbE += atom_hbondE(aatm,seqpos);
			}
			hr.set_hbond(sc_hbE);

			if ( sc_hbE <= hs_hb_thres ) {
				hr.set_Cbeta_atr( compute_Cbeta_atr(seqpos) );
				hr.set_interface_Cbeta_atr( compute_Cbeta_atr(seqpos,true) );
				return true;
			}

		}
	}

	return false;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::count_buried_AspAsn
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
interface_ds::count_buried_AspAsn() {


	int num_buried_AspAsn(0);
	for ( int seqpos = 1; seqpos <= NRES_; ++seqpos ) {
		if ( ! AnkyrinRepeat_ns::AspAsn_seqpos(seqpos) ) continue;
		int const aa = pose_->res(seqpos);
		if ( ( aa != param_aa::aa_asn ) && ( aa != param_aa::aa_asp ) ) continue;
		if ( test_AspAsn_burial(seqpos) ) ++num_buried_AspAsn;
	}

	return num_buried_AspAsn;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::test_AspAsn_burial
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
bool
interface_ds::test_AspAsn_burial( int const seqpos ) {

	using namespace param_aa;

	initialize_interface_packing_measures();

	int const aa = pose_->res(seqpos);

	float aspasn_atom_sasa_diff(0.);
	float aspasn_atom_sasa_complex(0.);
	if ( aa == aa_asp ) {
		aspasn_atom_sasa_diff = atom_sasa_diff( 7, seqpos ) + atom_sasa_diff( 8, seqpos );  // OD1 and OD2
		aspasn_atom_sasa_complex = atom_sasa_complex( 7, seqpos ) + atom_sasa_complex( 8, seqpos );
	} else if ( aa == aa_asn ) {
		aspasn_atom_sasa_diff = atom_sasa_diff( 8, seqpos );  // ND2
		aspasn_atom_sasa_complex = atom_sasa_complex( 8, seqpos );
	} else {
		std::cout << "Error - expected Asp or Asn for burial test." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// Must have more than this atom sasa diff to be considered exposed
	float const atom_sasa_thres = 1.;

	std::cout << "AspAsn seqpos " << seqpos << " has delta SASA: " << aspasn_atom_sasa_diff
						<< " and total SASA in the complex: " << aspasn_atom_sasa_complex;
	if ( aspasn_atom_sasa_diff > atom_sasa_thres ) {
		std::cout << " (buried upon complexation)" << std::endl;
	} else {
		std::cout << " (unchanged by complexation)" << std::endl;
	}

	if ( aspasn_atom_sasa_diff > atom_sasa_thres ) return true; // return true for buried upon complexation
	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_Cbeta_atr
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float interface_ds::compute_Cbeta_atr( int const seqpos, bool const interface_mode ) {

	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param_aa;

	int const aa = pose_->res(seqpos);
	if ( aa != aa_gly ) return 0.;

	float atrE(0.);
	float repE(0.);
	float solvE(0.);
	int const aav = pose_->res_variant(seqpos);
	int const atom1 = first_scatom(aa,aav);
	int const attype1 = fullatom_type(atom1,aa,aav);

	for ( int res2 = 1; res2 <= NRES_; ++res2 ) {
		if ( res2 == seqpos ) continue;
		if ( interface_mode &&
				 (((seqpos <= domain_end(1)) && (res2 <= domain_end(1))) ||
					((seqpos > domain_end(1)) && (res2 > domain_end(1))))) continue;
		int const aa2 = pose_->res(res2);
		int const aav2 = pose_->res_variant(res2);
		for ( int atom2 = 1, atom2e = nheavyatoms(aa2,aav2); atom2 <= atom2e; ++atom2 ) {
			float d2;
			float cp_weight;
			int const attype2 = fullatom_type(atom2,aa2,aav2);
			if ( count_pair(seqpos,atom1,aa,aav,res2,atom2,aa2,aav2,true,cp_weight) ) {
				fast_pairenergy(pose_->full_coord()(1,atom1,seqpos),pose_->full_coord()(1,atom2,res2),
												attype1,attype2,solvE,atrE,repE,d2,cp_weight);
			}
		}
	}

	return atrE;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::count_interface_waters
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::count_interface_waters(
  float const probe_radius,
	float const interface_diff_thres
) {

	using namespace aaproperties_pack;
	using namespace param;

	// jk A water atom must have less than this much SASA to be considered buried
	float const burial_sasa_thres = { 5. };

	FArray1D_float water_res_sasa_complex( NRES_ );
	FArray2D_float water_atom_sasa_complex( MAX_ATOM()(), NRES_ );

	FArray1D_float water_res_sasa_unbound( NRES_ );
	FArray2D_float water_atom_sasa_unbound( MAX_ATOM()(), NRES_ );

	FArray2D_float water_atom_sasa_diff( MAX_ATOM()(), NRES_ );

	// jk NOTE: interface SASA numbers will be computed WITH HYDROGENS AND WATERS

	// compute the SASA for the complex
	calc_per_atom_sasa( water_atom_sasa_complex, water_res_sasa_complex,
    probe_radius, false, false );

	// compute the SASA for the unbound
	calc_per_atom_sasa( water_atom_sasa_unbound, water_res_sasa_unbound,
	  probe_radius, false, true );

	// compute SASA differences
	water_atom_sasa_diff = water_atom_sasa_unbound - water_atom_sasa_complex;

	num_interface_water = 0;
	for ( int i = 1, max_atom = MAX_ATOM(); i <= NRES_; ++i ) {
		for ( int j = 1; j <= max_atom; ++j ) {
			int const jt = fullatom_type(j,pose_->res(i),pose_->res_variant(i));
			if ( jt == 26 ) {
				if ( ( water_atom_sasa_complex(j,i) < burial_sasa_thres ) &&
						 ( water_atom_sasa_diff(j,i) > interface_diff_thres ) ) {
					++num_interface_water;
				}
			}
		}
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::report_core_uns
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::report_core_uns(
  std::string tag,
	std::list < unsatisfied_buried_polar > & uns_list
)
{

	int bb_unsatisfied=0;
	int sc_unsatisfied=0;
	float uns_score=0;

	for( std::list<unsatisfied_buried_polar>::iterator uns = uns_list.begin();
			uns != uns_list.end(); ++uns ) {

		int resnum=uns->get_seqpos();
		if ( core_interface_res(resnum) ) {

			uns_score+=uns->weight();

			std::string const uns_type = decide_uns_type( *uns );
			if ( ! get_ds_outpdbonly_flag() ) uns->write_to_cout(tag, uns_type);
			uns->write_to_PDB(tag, uns_type);

			// jk require burial to a 1.0 A probe if explicit water is not used,
			// jk otherwise relax the criterion to "buried to a 1.4 A probe"
			if ( ! uns->satisfied() ) {
				if ( ( uns->buried10() ) || ( ( design::explicit_h2o ) && uns->buried() ) ) {
					if ( uns->backbone() ) {
						++bb_unsatisfied;
					} else {
						++sc_unsatisfied;
					}
				}
			}

		}
	}

	if ( tag != "" ) tag = tag + "_";
	add_decoy_score(tag+"bbhb_uns",float(bb_unsatisfied));
	add_decoy_score(tag+"schb_uns",float(sc_unsatisfied));
	add_decoy_score(tag+"uns_score",uns_score);

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::report_core_uns
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::report_core_uns(
  std::string tag,
	std::list < unsatisfied_buried_polar > & uns_list,
	std::list < unsatisfied_buried_group > & group_uns_list
)
{

	// jk Overloading: if group_uns is passed in do it after doing the uns as per previous subroutine
	report_core_uns(tag, uns_list);

	int num_group_uns=0;
	float group_uns_score=0;

	for( std::list<unsatisfied_buried_group>::iterator group_uns = group_uns_list.begin();
			group_uns != group_uns_list.end(); ++group_uns ) {

		int resnum=group_uns->get_seqpos();
		if ( core_interface_res(resnum) ) {

			++num_group_uns;
			group_uns_score+=group_uns->weight();

			if ( ! get_ds_outpdbonly_flag() ) group_uns->write_to_cout(tag);
			group_uns->write_to_PDB(tag);

		}

	}

	if ( tag != "" ) tag = tag + "_";
	add_decoy_score(tag+"group_uns",float(num_group_uns));
	add_decoy_score(tag+"group_uns_score",group_uns_score);

	return;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin compute_delta_UNS
///
/// @brief
///    find unsatified buried polars in a protein-protein interface
///
/// @detailed
///    protocol involved repacking the interface with solvated rotamers,
///    and comparing UNS to those found in unbound structure when non-Gly interface
///    residues are all replaced with Ala
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors John Karanicolas
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_delta_UNS(
  bool const do_repack,
	bool const compare_to_polyAla,
	bool const refine_loops
){

	using namespace decoystats;
	using namespace design;
	using namespace docking;
	using namespace files_paths;
	using namespace fullatom_flag;
	using namespace param;
	using namespace param_aa;

	// save a local copy of the original setting of these vars
	bool const prev_full_atom = full_atom;
	std::string const prev_resfile = resfile;
	bool const prev_output_coord = output_coord;
	bool const prev_query_defined = query_defined;
	bool const prev_repack_inter = repack_inter;
	bool const prev_design_inter = design_inter;
	bool const prev_build_Ala_interface = build_Ala_interface;

	full_atom = true;
	resfile = "none";
	output_coord = false;
	query_defined = false;

	pose_ns::Pose bound_pose;
	pose_->copy_to_misc();
	bound_pose = *pose_;

	// Collect a list of interface residues
	repack_inter = true;
	design_inter = true;
	interface_residues();

	if ( refine_loops ) {

		std::cout << "STOP:: refine_loops portion of compute_delta_UNS is not yet implemented"
							<< std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);

		// jk Relax the loops (via pose), only on the first chain
		// jk Define each loop as 7 residues, use overlapping loops
		pose_ns::Loops interface_loops;
		for ( int i = 1; i <= misc::domain_end(1); ++i ) {
			if ( interface_residue(i) ) {
				int cutsite=i;
				int start=i-3;
				int stop=i+3;
				if ( start < 2 ) start = 2;
				if ( stop > ( misc::domain_end(1)-1 ) ) stop = misc::domain_end(1)-1;
				interface_loops.add_loop(start, stop, cutsite);
			}
		}

		set_pose_flag(true);
		bound_pose.simple_fold_tree( bound_pose.total_residue() );
		loop_refinement(bound_pose, interface_loops);
		bound_pose.copy_to_misc();
		set_pose_flag(false);

	}

	if ( do_repack || compare_to_polyAla ) {
		// Create PackerTask and setup values before pass into pack_rotamers
		PackerTask Task( bound_pose );
		Task.set_task("design",false,interface_residue,true,false);
		build_Ala_interface = false;
		Task.setup_residues_to_vary();
		if ( do_repack ) {
			// Repacking the starting interface
			std::cout << "repacking starting interface" << std::endl;
			pack_rotamers( bound_pose, Task );
		}
		bound_pose.copy_to_misc();
	}

	// Collect a list of UNS in the bound structure
	// We need to fill the Hbond arrays for the new structure. Rather than call fill_hbond_arrays
	// directly, just score the structure completely (time difference here is small)
	fullatom_energy(bound_pose.res(),bound_pose.res_variant(),bound_pose.full_coord(),bound_pose.total_residue(),false);
	copy_hbenergies();
	find_unsatisfied_hbonds(bound_uns_list,bound_group_uns_list);

	// Translate subunits apart (moving the second, keeping the first fixed)
	pose_->copy_to_misc();
	float const translation_dist=50.;
	translate_subunits(translation_dist);

	// Setup the unbound pose
	pose_ns::Pose unbound_pose;
	fullatom_nonideal_initialized_pose_from_misc( unbound_pose );

	// Collect a list of UNS in the unbound structure
	// We need to fill the Hbond arrays for the new structure. Rather than call fill_hbond_arrays
	// directly, just score the structure completely (time difference here is small)
	fullatom_energy(unbound_pose.res(),unbound_pose.res_variant(),unbound_pose.full_coord(),unbound_pose.total_residue(),false);
	copy_hbenergies();
	find_unsatisfied_hbonds(unbound_uns_list,unbound_group_uns_list);

	// Build a list of those unique to the bound state
	interface_delta_uns_list = find_unique_uns(bound_uns_list,unbound_uns_list);
	interface_delta_group_uns_list = find_unique_uns(bound_group_uns_list,unbound_group_uns_list);

	// Repack the same residues in the unbound,
	// optionally substituting Ala at all interface positions
	if ( do_repack || compare_to_polyAla ) {
		if ( compare_to_polyAla ) {
			std::cout << "repacking unbound state using only Ala" << std::endl;
			build_Ala_interface = true;
		} else {
			std::cout << "repacking unbound state" << std::endl;
		}
		PackerTask Task( unbound_pose );
		Task.set_task("design",false,interface_residue,true,false);
		Task.setup_residues_to_vary();
		build_Ala_interface = false;
		pack_rotamers( unbound_pose, Task );
		unbound_pose.copy_to_misc();
	}

	// Collect a list of UNS in the unbound structure (again)
	// We need to fill the Hbond arrays for the new structure. Rather than call fill_hbond_arrays
	// directly, just score the structure completely (time difference here is small)
	fullatom_energy(unbound_pose.res(),unbound_pose.res_variant(),unbound_pose.full_coord(),unbound_pose.total_residue(),false);
	copy_hbenergies();
	find_unsatisfied_hbonds(unbound_uns_list, unbound_group_uns_list);

	// Build a list of those unique to the bound state
	interface_strict_uns_list = find_unique_uns(bound_uns_list,unbound_uns_list);
	interface_strict_group_uns_list = find_unique_uns(bound_group_uns_list,unbound_group_uns_list);

	// Read a file that specifies exempt uns or group_uns
	std::string exemption_file;
	stringafteroption( "gu_exempt", "none", exemption_file );

	if ( exemption_file != "none" ) {
		utility::io::izstream pdb_stream;
		pdb_stream.open(exemption_file);
		read_uns_from_pdb(pdb_stream,exempt_uns_list,exempt_group_uns_list);
 		interface_delta_uns_list = find_unique_uns(interface_delta_uns_list,exempt_uns_list);
		interface_delta_group_uns_list = find_unique_uns(interface_delta_group_uns_list,exempt_group_uns_list);
 		interface_strict_uns_list = find_unique_uns(interface_strict_uns_list,exempt_uns_list);
		interface_strict_group_uns_list = find_unique_uns(interface_strict_group_uns_list,exempt_group_uns_list);
	}

	// Retrieve the repacked (possibly solvated) bound interface
	if ( do_repack || compare_to_polyAla ) {
		*pose_ = bound_pose;
	}
	pose_->copy_to_misc();

	// restore state of globals
	full_atom = prev_full_atom;
	resfile = prev_resfile;
	output_coord = prev_output_coord;
	query_defined = prev_query_defined;
	repack_inter = prev_repack_inter;
	design_inter = prev_design_inter;
	build_Ala_interface = prev_build_Ala_interface;

	return;

}


////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::describe_hotspots
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::describe_hotspots() {

	find_hotspots();
	for (std::vector<hotspot_residue>::iterator hs = hotspot_list.begin();
			 hs != hotspot_list.end(); hs++) {
		decoystats_store_output_line( hs->describe() );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_hotspot_atr_ene
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_hotspot_atr_ene() {

	find_hotspots();
	float best_atr_ene=999.;
	if ( num_hotspot_residues == 0 ) return best_atr_ene;

	for (std::vector<hotspot_residue>::iterator hs = hotspot_list.begin();
			 hs != hotspot_list.end(); hs++) {
		float res_atr_ene = hs->get_atr_ene();
		if ( res_atr_ene < best_atr_ene ) best_atr_ene = res_atr_ene;
	}

	return best_atr_ene;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_hotspot_SASA_small
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_hotspot_SASA_small() {

	find_hotspots();
	float best_SASA_small=999.;
	if ( num_hotspot_residues == 0 ) return best_SASA_small;

	for (std::vector<hotspot_residue>::iterator hs = hotspot_list.begin();
			 hs != hotspot_list.end(); hs++) {
		float SASA_small = hs->get_resSASA_smallprobe();
		if ( SASA_small < best_SASA_small ) best_SASA_small = SASA_small;
	}

	return best_SASA_small;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_hotspot_SASApack
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_hotspot_SASApack() {

	find_hotspots();
	float best_SASApack=999.;
	if ( num_hotspot_residues == 0 ) return best_SASApack;

	for (std::vector<hotspot_residue>::iterator hs = hotspot_list.begin();
			 hs != hotspot_list.end(); hs++) {
		float res_SASApack = hs->get_SASApack();
		if ( res_SASApack < best_SASApack ) best_SASApack = res_SASApack;
	}

	return best_SASApack;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_hotspot_SASAprob
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_hotspot_SASAprob() {

	find_hotspots();
	float best_SASAprob=0.;
	if ( num_hotspot_residues == 0 ) return best_SASAprob;

	for (std::vector<hotspot_residue>::iterator hs = hotspot_list.begin();
			 hs != hotspot_list.end(); hs++) {
		float res_SASAprob = hs->get_SASAprob();
		if ( res_SASAprob > best_SASAprob ) best_SASAprob = res_SASAprob;
	}

	return best_SASAprob;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_hotspot_maxvoid
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_hotspot_maxvoid() {

	find_hotspots();
	float best_maxvoid=5.;
	if ( num_hotspot_residues == 0 ) return best_maxvoid;

	for (std::vector<hotspot_residue>::iterator hs = hotspot_list.begin();
			 hs != hotspot_list.end(); hs++) {
		float res_maxvoid = hs->get_maxvoid();
		if ( res_maxvoid < best_maxvoid ) best_maxvoid = res_maxvoid;
	}

	return best_maxvoid;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::describe_stacks
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::describe_stacks() {

	for (std::vector<aromatic_stack>::iterator hs = stack_list.begin();
			 hs != stack_list.end(); hs++) {
		decoystats_store_output_line( hs->describe() );
	}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_stack_intrastack_ene
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_stack_intrastack_ene() {

	float best_intrastack_ene=999.;
	if ( num_stacks == 0 ) return best_intrastack_ene;

	for (std::vector<aromatic_stack>::iterator hs = stack_list.begin();
			 hs != stack_list.end(); hs++) {
		float res_intrastack_ene = hs->get_intrastack_ene();
		if ( res_intrastack_ene < best_intrastack_ene ) best_intrastack_ene = res_intrastack_ene;
	}

	return best_intrastack_ene;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_stack_plane_ene
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_stack_plane_ene() {

	float best_plane_ene=999.;
	if ( num_stacks == 0 ) return best_plane_ene;

	for (std::vector<aromatic_stack>::iterator hs = stack_list.begin();
			 hs != stack_list.end(); hs++) {
		float res_plane_ene = hs->get_plane_ene();
		if ( res_plane_ene < best_plane_ene ) best_plane_ene = res_plane_ene;
	}

	return best_plane_ene;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_stack_atr_ene
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_stack_atr_ene() {

	float best_atr_ene=999.;
	if ( num_stacks == 0 ) return best_atr_ene;

	for (std::vector<aromatic_stack>::iterator hs = stack_list.begin();
			 hs != stack_list.end(); hs++) {
		float res_atr_ene = hs->get_atr_ene();
		if ( res_atr_ene < best_atr_ene ) best_atr_ene = res_atr_ene;
	}

	return best_atr_ene;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_stack_SASApack
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_stack_SASApack() {

	float best_SASApack=999.;
	if ( num_stacks == 0 ) return best_SASApack;

	for (std::vector<aromatic_stack>::iterator hs = stack_list.begin();
			 hs != stack_list.end(); hs++) {
		float res_SASApack = hs->get_SASApack();
		if ( res_SASApack < best_SASApack ) best_SASApack = res_SASApack;
	}

	return best_SASApack;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::best_stack_SASAprob
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::best_stack_SASAprob() {

	float best_SASAprob=0.;
	if ( num_stacks == 0 ) return best_SASAprob;

	for (std::vector<aromatic_stack>::iterator hs = stack_list.begin();
			 hs != stack_list.end(); hs++) {
		float res_SASAprob = hs->get_SASAprob();
		if ( res_SASAprob > best_SASAprob ) best_SASAprob = res_SASAprob;
	}

	return best_SASAprob;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_quantifiers
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_quantifiers() {

	using namespace decoystats;

	if ( design::identify_interface_UNS_flag ) {
		compute_delta_UNS( true, true, false );
	} else {
		compute_delta_UNS( false, false, false );
	}

	// make sure energies and SASAs and decoystats Hbond arrays are up to date
	pose_->copy_to_misc();
	score_set_new_pose();
	score12();
	copy_hbenergies();
	calc_aprox_sasa();
	calc_sasa_pack_score(NRES_, pose_->res(), pose_->res_variant(), pose_->full_coord());
	calc_sasa_prob();

	// define the interface
	define_interface();

	compute_delta_sasa();
	compute_composition();
	compute_energies();

	if ( tight_core_analysis ) {
		compute_tight_core_packing();
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::write_all_quantifiers_to_PDB
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::write_all_quantifiers_to_PDB()
{

	using namespace decoystats;
	using namespace energy_quantifier_ns;
	using namespace param;

	initialize_interface_packing_measures();

	// jk measures of interface size
	add_decoy_score("delta_sasa",delta_sasa);
	add_decoy_score("num_interface_res",num_interface_res);
	add_decoy_score("num_interface_atoms",num_interface_atom);
	add_decoy_score("num_interface_waters",num_interface_water);
	add_decoy_score("num_core_interface_res",num_core_interface_res);

	// jk applications of packing_measures
	add_decoy_score("interface_max_void_cluster_volume", max_interface_void_volume);
	add_decoy_score("interface_max_void_radius", max_interface_void_radius);
	add_decoy_score("interface_void_cluster_volume", total_interface_void_volume);
	add_decoy_score("interface_void_cluster_volume_per_sasa", ( total_interface_void_volume / delta_sasa ) );

	// jk hotspots
	find_hotspots();
	add_decoy_score("num_hotspot_residues", num_hotspot_residues);
	add_decoy_score("best_hotspot_atr_ene", best_hotspot_atr_ene());
	add_decoy_score("best_hotspot_SASAsmall", best_hotspot_SASA_small());
	add_decoy_score("best_hotspot_SASApack", best_hotspot_SASApack());
	add_decoy_score("best_hotspot_SASAprob", best_hotspot_SASAprob());
	add_decoy_score("best_hotspot_maxvoid", best_hotspot_maxvoid());

	// jk aromatic stacks
	add_decoy_score("num_stacks", num_stacks);
	add_decoy_score("best_stack_intrastack_ene", best_stack_intrastack_ene());
	add_decoy_score("best_stack_plane_ene", best_stack_plane_ene());
	add_decoy_score("best_stack_atr_ene", best_stack_atr_ene());
	add_decoy_score("best_stack_SASApack", best_stack_SASApack());
	add_decoy_score("best_stack_SASAprob", best_stack_SASAprob());

	// jk hotspots+stacks
	add_decoy_score("num_affinity_centers", num_hotspot_residues+num_stacks);

	// jk measures of composition
	add_decoy_score("interface_num_aromatic_atoms", num_interface_aromatics);
	add_decoy_score("interface_aromatic_sasa", sasa_interface_aromatics);
	float const fraction_polar = static_cast<float>(num_polar_atoms) / static_cast<float>(num_atoms);
	add_decoy_score("interface_fraction_polar", fraction_polar);
	add_decoy_score("met_sc_atoms", met_sc_atoms);
	add_decoy_score("interface_chi_angles", interface_chi_angles);
	add_decoy_score("num_chi_angles_per_sasa",1000 * static_cast<float>(interface_chi_angles) / delta_sasa);

	// jk the energy difference ( design_E - homodimer_E ) for each homodimer (written for IFNBP)
	if ( design::compute_homodimer_specificity ) {
		add_decoy_score("homodimerA_diff", design::homodimerA_energy);
		add_decoy_score("homodimerB_diff", design::homodimerB_energy);
		// jk note: for the IFNBP project, a big negative value is good - report the "less good" one
		add_decoy_score("heterodimer_specificity", std::max(design::homodimerA_energy,design::homodimerB_energy));
	}

	// jk measures of packing
	if ( tight_core_analysis ) {
		add_decoy_score("interface_num_tight_core_atoms",interface_num_tight);
		float const tight_per_sasa = 100.*static_cast<float>(interface_num_tight)/delta_sasa;
		add_decoy_score("num_tight_core_atoms_per_sasa",tight_per_sasa);
		float const fraction_tight = static_cast<float>(interface_num_tight) / static_cast<float>(num_atoms);
		add_decoy_score("interface_fraction_tight", fraction_tight);
	}

	// jk energetics
	for( std::vector<interface_energy_quantifier>::iterator qcurr = energy_quantifiers.begin();
			qcurr != energy_quantifiers.end(); ++qcurr ) {
		if ( qcurr->is_initialized() ) {
			qcurr->write_to_PDB( num_interface_res, num_core_interface_res );
		}
	}

	// jk Rosetta predicted binding energy
	add_decoy_score("delta_score", delta_score);
	add_decoy_score("delta_score_per_sasa",1000.*delta_score/delta_sasa);
	add_decoy_score("delta_score_per_chi", 100. * delta_score / static_cast<float>(interface_chi_angles));

	// jk a few extra measures...
	add_decoy_score("delta_atr_per_sasa",
	  1000.* (energy_quantifiers[ATR]).report_delta_val() / delta_sasa);
	add_decoy_score("delta_sol_per_sasa",
	  1000.* (energy_quantifiers[SOL]).report_delta_val() / delta_sasa);
	add_decoy_score("delta_hb_per_sasa",
	  1000.* (energy_quantifiers[HB]).report_delta_val() / delta_sasa);
	add_decoy_score("delta_gb_per_sasa",
	  1000.* (energy_quantifiers[GB]).report_delta_val() / delta_sasa);
	add_decoy_score("delta_sol+hb_per_sasa",
		1000.* ((energy_quantifiers[SOL]).report_delta_val() +
						(energy_quantifiers[HB]).report_delta_val()) / delta_sasa);

	// jk Uns lists
	report_unsatisfied_hbonds("BOUND UNS",bound_uns_list);
	report_unsatisfied_hbonds("UNBOUND UNS",unbound_uns_list);
	report_unsatisfied_hbonds("INT",interface_strict_uns_list,interface_strict_group_uns_list);
	report_unsatisfied_hbonds("DELTA",interface_delta_uns_list,interface_delta_group_uns_list);

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_delta_sasa
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
float
interface_ds::compute_delta_sasa()
{

	// jk find total SASA lost upon complexation
	delta_sasa=0.;
	for ( int i = 1; i <= NRES_; ++i ) {
		delta_sasa += res_sasa_diff(i);
	}

	return delta_sasa;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_composition
///
/// @brief
///    Count the number of aromatic atoms buried by the interface and
///    their interface SASA, also write the fraction of interface atoms
///    which are polar
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_composition()
{

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace param;
	using namespace param_aa;

	// jk initialization
	num_interface_aromatics=0;
	sasa_interface_aromatics=0.;
	num_polar_atoms=0;
	num_atoms=0;
	met_sc_atoms=0;

	for ( int i = 1; i <= NRES_; ++i ) {
		int const aa = pose_->res(i);
		int const aav = pose_->res_variant(i);
		int const max_atom = nheavyatoms(aa,aav);
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( interface_atom(j,i) ) {
				++num_atoms;
				int const atom_type = fullatom_type( j, aa, aav );
				if ( ( atom_type == 6 ) &&
						 ( aa != aa_his ) ) {
					// jk if atom is aromatic, but not His
					++num_interface_aromatics;
					sasa_interface_aromatics += atom_sasa_diff(j,i);
				} else if ( ( aa == aa_met ) && ( j >= 5 ) ) {
					// jk Met sidechain
					++met_sc_atoms;
				} else if ( atomtype_sasapolarity(atom_type) == POLAR ) {
					// jk if atom is polar
					++num_polar_atoms;
				}
			}
		}
	}

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::setup_energetic_quantifiers
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::setup_energetic_quantifiers()
{

	using namespace energy_quantifier_ns;
	using namespace fullatom_energies;
	using namespace pdbstatistics_pack;
	using namespace template_pack;
	using namespace void_ns;

	// jk When adding new energy quantifiers, do it by first adding
	// jk to the enum in decoystats_interface_classes.h

	{
		interface_energy_quantifier & q = energy_quantifiers[ATR];
		q.reset();
		q.initialize("atr");
		q.allow_delta(atr_pair);
		q.allow_interface_sum(atrenergy);
		q.allow_VsPDB(atrenergy,atr_avg);
		q.allow_max_VsPDB();
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[REP];
		q.reset();
		q.initialize("rep");
		q.allow_delta(rep_pair);
		q.allow_interface_sum(repenergy);
		q.allow_VsPDB(repenergy,rep_avg);
		q.allow_max();
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[SOL];
		q.reset();
		q.initialize("sol");
		q.allow_delta(sol_pair);
		q.allow_interface_sum(solenergy);
		q.allow_VsPDB(solenergy,sol_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[HB];
		q.reset();
		q.initialize("hbond");
		q.allow_delta(hbond_pair);
		q.allow_interface_sum(hbenergy);
		q.allow_VsPDB(hbenergy,hb_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[PROB];
		q.reset();
		q.initialize("prob");
		q.allow_interface_sum(probenergy);
		q.allow_VsPDB(probenergy,one_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[DUN];
		q.reset();
		q.initialize("dun");
		q.allow_interface_sum(dunenergy);
		q.allow_VsPDB(dunenergy,dun_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[PAIR];
		q.reset();
		q.initialize("pair");
		q.allow_delta(pair_pair);
		q.allow_interface_sum(pair_energy);
		q.allow_VsPDB(pair_energy,pair_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[INTRARES];
		q.reset();
		q.initialize("intrares");
		q.allow_interface_sum(intraresenergy);
		q.allow_VsPDB(intraresenergy,intrares_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[GB];
		q.reset();
		q.initialize("gb");
		q.allow_delta(gb_elec_pair);
		q.allow_interface_sum(gb_elecenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[CST];
		q.reset();
		q.initialize("cst");
		q.allow_delta(cst_pair);
		q.allow_interface_sum(cstenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[PLANE];
		q.reset();
		q.initialize("plane");
		q.allow_delta(plane_plane);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[H2O];
		q.reset();
		q.initialize("h2o");
		q.allow_delta(h2o_pair);
		q.allow_interface_sum(h2oenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[H2OHB];
		q.reset();
		q.initialize("h2o_hb");
		q.allow_delta(h2o_hb_pair);
		q.allow_interface_sum(h2ohbenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[WSOL];
		q.reset();
		q.initialize("wsol");
		q.allow_interface_sum(wsolenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[UNF];
		q.reset();
		q.initialize("unf");
		q.allow_interface_sum(unfenergy);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[TOT];
		q.reset();
		q.initialize("total");
		q.allow_interface_sum(resenergy);
		q.allow_VsPDB(resenergy,tot_avg);
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[SASAPACK];
		q.reset();
		q.initialize("SASApackVsPDB");
		q.allow_interface_sum(rsd_sasa_pack_score);
		q.allow_max();
	}

	{
		interface_energy_quantifier & q = energy_quantifiers[SASAPROB];
		q.reset();
		q.initialize("SASAprobVsPDB");
		q.allow_interface_sum(rsd_sasaprob);
		q.allow_min();
	}

	energies_initialized = true;

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_energies
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::compute_energies()
{

	using namespace energy_quantifier_ns;

	for( std::vector<interface_energy_quantifier>::iterator qcurr = energy_quantifiers.begin();
			qcurr != energy_quantifiers.end(); ++qcurr ) {
		if ( qcurr->is_initialized() ) {
			qcurr->compute_energies( interface_res, core_interface_res );
		}
	}

	delta_score = 0.;
	delta_score += (energy_quantifiers[ATR]).report_delta_val();
	delta_score += (energy_quantifiers[REP]).report_delta_val();
	delta_score += (energy_quantifiers[SOL]).report_delta_val();
	delta_score += (energy_quantifiers[HB]).report_delta_val();
	delta_score += (energy_quantifiers[PAIR]).report_delta_val();
	delta_score += (energy_quantifiers[GB]).report_delta_val();
	delta_score += (energy_quantifiers[CST]).report_delta_val();
	delta_score += (energy_quantifiers[PLANE]).report_delta_val();
	delta_score += (energy_quantifiers[H2O]).report_delta_val();
	delta_score += (energy_quantifiers[H2OHB]).report_delta_val();

	return;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::compute_tight_core_packing
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
int
interface_ds::compute_tight_core_packing()
{

	using namespace aaproperties_pack;
	using namespace decoystats;
	using namespace param;

	FArray2D_bool tight_in_monomers( MAX_ATOM(), NRES_ );

	// jk max SASA5 which still qualifies as tight packing
	realafteroption("tight_core_thres",4.0,tight_core_thres);

	int unbound_tight=find_tight_core_packing(true);
	tight_in_monomers=tightly_packed;

	int bound_tight=find_tight_core_packing(false);
	interface_num_tight=bound_tight-unbound_tight;

	// jk list the tightly packed interface atoms (to the output pdb)
	//	for ( int i = 1, max_atom = MAX_ATOM(); i <= NRES_; ++i ) {
	//		int const aa = pose_->res(i);
	//		int const aav = pose_->res_variant(i);
	//		for ( int j = 1; j <= max_atom; ++j ) {
	//			if ( tightly_packed(j,i) && ! tight_in_monomers(j,i) ) {
	//				// jk we'll write to an output stream
	//				std::ostringstream tight_int_atom_out_stream;
	//				tight_int_atom_out_stream << "TIGHTINTATOM " << atom_name(j,aa,aav) <<
	//					' ' << misc::residue1(i) << ' ' << I( 4, i );
	//				decoystats_store_output_line( tight_int_atom_out_stream.str() );
	//			}
	//		}
	//	}

	if ( tight_core_Bfactors ) reset_noninterface_bfactors(50.);

	return interface_num_tight;
}


//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::list_interface_res
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::list_interface_res() {

	using namespace param;

	for ( int i = 1; i <= NRES_; ++i ) {
		if ( interface_res(i) ) {
			// jk we'll write to an output stream
			std::ostringstream int_res_out_stream;
			int_res_out_stream << "INTRES " << misc::residue1(i) << ' ' << I( 4, i );
			decoystats_store_output_line( int_res_out_stream.str() );
		}
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::list_interface_atoms
///
/// @brief
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::list_interface_atoms() {

	using namespace aaproperties_pack;
	using namespace param;

	for ( int i = 1, max_atom = MAX_ATOM(); i <= NRES_; ++i ) {
		int const aa = pose_->res(i);
		int const aav = pose_->res_variant(i);
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( interface_atom(j,i) ) {
				// jk we'll write to an output stream
				std::ostringstream int_atom_out_stream;
				int_atom_out_stream << "INTATOM " << atom_name(j,aa,aav) << ' ' << misc::residue1(i) << ' ' << I( 4, i );
				decoystats_store_output_line( int_atom_out_stream.str() );
			}
		}
	}

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin interface_ds::reset_noninterface_bfactor
///
/// @brief
///   jk Set ds_factor to the specified value for all non-interface atoms
///
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
interface_ds::reset_noninterface_bfactors(
  float const reset_val
)
{

	using namespace param;

	for ( int i = 1, max_atom = MAX_ATOM(); i <= NRES_; ++i ) {
		for ( int j = 1; j <= max_atom; ++j ) {
			if ( ! interface_atom(j,i) ) {
				make_pdb_ns::bfactor(j,i)=reset_val;
			}
		}
	}

	return;
}

