// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7683 $
//  $Date: 2006-03-18 17:42:35 -0800 (Sat, 18 Mar 2006) $
//  $Author: bqian $


// Rosetta Headers
#include "design_hotspot_interface.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "AnkyrinRepeat_ns.h"
#include "AnkyrinRepeat_functions.h"
#include "design.h"
#include "design_structure.h"
#include "DesignMap.h"
#include "docking.h"
#include "docking_ns.h"
#include "favor_residue_ns.h"
#include "files_paths.h"
#include "fullatom_setup.h"
#include "HotspotRotamerList.h"
#include "hotspot_ns.h"
#include "InteractionGraphSupport.h"
#include "make_pdb.h"
#include "misc.h"
#include "namespace_fullatom_flag.h"
#include "output_decoy.h"
#include "pack.h"
#include "PackerTask.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_rotamer_trie.h"
#include "pose.h"
#include "pose_io.h"
#include "pose_docking.h"
//#include "prof.h"
#include "termini.h"
#include "water_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/formatted.io.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <list>
#include <vector>


////////////////////////////////////////////////////////////////////////////////
/// @begin design_hotspot_interface
///
/// @brief
/// jk design a set of interfaces which have a hotspot in them
///
/// @detailed
///
/// @param  packmode - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
design_hotspot_interface()
{

	using namespace aaproperties_pack;
	using namespace design;
	using namespace fullatom_flag;
	using namespace misc;
	using namespace param;
	using namespace param_aa;
	using namespace termini_ns;

	read_hotspot_options();

	initialize_fullatom();
	hotspot_ns::hotspot_design = true;
	hotspot_ns::hs_rotamer_options.read_rotamer_flags();
	hotspot_ns::hs_rotamer_options.setup_exflags();

	if ( hotspot_ns::use_soft_rep ) {
		param_pack::soft_rep=true;
		param_pack::soft_rep_design=true;
	}

	if ( param_rotamer_trie::use_rotamer_trie &&
			 ! ( design::explicit_h2o || water::use_hetero_h2o ) ) {
		hotspot_ns::use_trie = true;
	} else {
		hotspot_ns::use_trie = false;
	}

	bool hotspot_restore_sidechains = truefalseoption("hotspot_restore_sidechains");

	pose_ns::Pose start_pose;
	if ( docking::docking_small_perturbation ) {
		// Apply (low-resolution) dock_pert to the input structure
		pose_from_misc( start_pose, false, false, true );
		pose_docking_build_simple_tree( start_pose, misc::total_residue, docking::part_end(1) );
		start_pose.set_allow_jump_move( false );
		start_pose.set_allow_bb_move( false );
		start_pose.set_allow_chi_move( true );
		pose_docking_set_general_allow_move( start_pose );
		pose_docking_perturb_rigid_body(start_pose);
		// switch to fullatom, do a repack here if we're not going to restore sidechains below
		start_pose.set_fullatom_flag( true, !hotspot_restore_sidechains );
	} else {
		// Read the fullatom starting structure
		pose_from_misc( start_pose, true, false, true );
	}

	if ( hotspot_restore_sidechains ) {
		pose_ns::Pose native_pose;
		std::string native_fname = "none";
		stringafteroption( "hotspot_restore_sidechains", native_fname, native_fname );
		pose_from_pdb( native_pose, native_fname, true, false, true );

		// Before we can use "recover_sidechain", we need to change the start_pose sequence to match "native_pose"...
		bool sequence_change = false;
		PackerTask conv_Task(start_pose);
		FArray1D_bool allow_repack( start_pose.total_residue(), false );
		conv_Task.set_task("packrot",false,allow_repack,false,false);
		conv_Task.setup_residues_to_vary();
		for ( int seqpos = 1; seqpos <= start_pose.total_residue(); ++seqpos ) {
			if ( ( start_pose.res(seqpos) == native_pose.res(seqpos) ) &&
				( start_pose.res_variant(seqpos) == native_pose.res_variant(seqpos) ) )
				continue;
			conv_Task.get_designmap().fix_completely(seqpos);
			conv_Task.get_designmap().set( seqpos, native_pose.res(seqpos) );
			if ( native_pose.res_variant(seqpos) != 1 ) {
				if ( ( native_pose.res(seqpos) == param_aa::aa_his ) && ( native_pose.res_variant(seqpos) == 2 ) ) {
					conv_Task.get_designmap().allow_variant_type_at_residue(2, seqpos );
					conv_Task.get_designmap().disallow_variant_type_at_residue(1, seqpos );
				} else {
					std::cout << "ERROR: unknown variant detected on native pose when restoring sidechains" << std::endl;
					utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
				}
			}
			sequence_change = true;
		}
		if ( sequence_change ) pack_rotamers( start_pose, conv_Task);

		// Bring back the sidechains from native_pose
		start_pose.recover_sidechain( native_pose );

	}

	// Set the conserved Asp/Asn to all-Asp,
	// but go to all Ser first so that we have rotameric Asp everywhere.
	// Note: this is because we run into problems if we have non-ideal Asp/Asn,
	//   since we'll rebuild Asp/Asn from chi angles later
	convert_all_to_Ser( start_pose );
	convert_all_to_Asp( start_pose );
	start_pose.copy_to_misc();

	// jk Save the starting sequence
	for ( int seqpos = 1; seqpos <= misc::total_residue; ++seqpos ) {
		hotspot_ns::starting_sequence(seqpos) = res(seqpos);
	}
	set_favor_native_residue(true);
	set_native_bonus(hotspot_ns::favor_native_periphery_bonus);
	favor_residue::res_native = 0;

	if ( ! hotspot_ns::screen_for_hotspot ) {
		// jk Identify the interface residues for this binding mode
		interface_residues();
		// jk Check whether the termini are involved in this binding mode
		if ( termini_involved() ) {
			std::cout << "Termini are involved in this binding mode" << std::endl;
			return;
		}
	} else {
		interface_residue = true;
	}

	if ( hotspot_ns::hotspot_on_AR ) {
		// Check for an interface aromatic on chain A (to be used for a stack)
		int startpos = 1;
		int endpos = misc::domain_end(1);
		bool stackable(false);
		for ( int seqpos = startpos; seqpos <= endpos; ++seqpos ) {
			if ( interface_residue(seqpos) ) {
				int const aa = hotspot_ns::starting_sequence(seqpos);
				// jk Extend to apply to L/I/V as well as aromatics
				if ( ( aa == param_aa::aa_phe ) || ( aa == param_aa::aa_ile ) || ( aa == param_aa::aa_leu ) ||
					( aa == param_aa::aa_val ) || ( aa == param_aa::aa_trp ) || ( aa == param_aa::aa_tyr ) ) {
					stackable = true;
					break;
				}
			}
		}
		if ( ! stackable ) {
			std::cout << "No stackable residue found on first chain" << std::endl;
			return;
		}
	}

	pose_ns::Pose polyA_pose;
	polyA_pose = start_pose;

	// jk Change all non-Gly interface residues to Ala for this binding mode
	// jk (to help picking out real clashes later)
	{ // "build Ala interface" scope
		int const orig_fix_target_seq = fix_target_seq;
		fix_target_seq = 0;
		files_paths::query_defined = false;

		// Create PackerTask and setup values before pass into pack_rotamers
		PackerTask ala_interface_Task(polyA_pose);
		ala_interface_Task.set_task("design",false,interface_residue,true,false);

		build_Ala_interface = true;
		ala_interface_Task.setup_residues_to_vary();

		// If we're going to look for a stack, we need to leave the aromatics on the non-AR
		if ( hotspot_ns::hotspot_on_AR ) {
			for ( int seqpos = 1; seqpos <= misc::domain_end(1); ++seqpos ) {
				int const aa = polyA_pose.res(seqpos);
				// jk Extend to apply to L/I/V as well as aromatics
				if ( ( aa == param_aa::aa_phe ) || ( aa == param_aa::aa_ile ) || ( aa == param_aa::aa_leu ) ||
					( aa == param_aa::aa_val ) || ( aa == param_aa::aa_trp ) || ( aa == param_aa::aa_tyr ) ) {
					ala_interface_Task.get_designmap().fix_completely(seqpos);
				}
			}
		}

		std::cout << "Building poly-Ala interface" << std::endl;

		pack_rotamers( polyA_pose, ala_interface_Task);
		polyA_pose.copy_to_misc();

		build_Ala_interface = false;
		files_paths::query_defined = false;
		fix_target_seq = orig_fix_target_seq;
	} // "build Ala interface" scope

	std::cout << "Building hotspot interfaces" << std::endl;

	// jk Read a resfile here
	PackerTask Task(start_pose);
	Task.set_task("design",false,interface_residue,true,false);
	Task.setup_residues_to_vary();

	// jk Restrict DesignMap to interface residues appropriate for this binding mode
	std::cout << "For this binding mode, the following residues will be permanantly fixed in place: ";
	bool is_first(true);
	for ( int seqpos = 1; seqpos <= misc::total_residue; ++seqpos ) {
		if ( ! interface_residue(seqpos) ) {
			if ( ! is_first ) std::cout << ", ";
			std::cout << seqpos;
			Task.get_designmap().fix_completely(seqpos);
			is_first = false;
		} else {
			// jk Never use Cys for non-interface positions
			Task.get_designmap().disable(seqpos, aa_cys);
			if ( ( res(seqpos) == aa_pro ) || ( res(seqpos) == aa_gly ) ) {
				// jk Maintain all Pro/Gly
				if ( ! is_first ) std::cout << ", ";
				std::cout << seqpos;
				Task.get_designmap().fix_completely(seqpos);
				is_first = false;
			} else {
				// jk Don't introduce new Pro/Gly
				Task.get_designmap().disable(seqpos, aa_gly);
				Task.get_designmap().disable(seqpos, aa_pro);
			}
		}
	}
	std::cout << std::endl;

	int startpos = 1;
	int endpos = misc::total_residue;
	if ( ! hotspot_ns::allow_hotspot_on_either_chain ) endpos = misc::domain_end(1);
	if ( hotspot_ns::hotspot_on_AR ) {
		startpos = misc::domain_end(1) + 1;
		endpos = misc::total_residue;
	}

	for ( int seqpos = startpos; seqpos <= endpos; ++seqpos ) {
		if ( interface_residue(seqpos) && Task.get_designmap().repack_residue(seqpos) ) {

			// Build only position 216 for BENCHMARK
			if ( hotspot_ns::BENCHMARK && ( seqpos != 216 ) ) continue;

			bool valid_seqpos=true;
			if ( hotspot_ns::hotspot_on_AR ) {
				valid_seqpos=false;
				int fixed_seqpos = seqpos + 29;
				if ( fixed_seqpos <= misc::domain_end(2) ) {
					if ( AnkyrinRepeat_ns::AspAsn_seqpos(fixed_seqpos) ) {
						valid_seqpos = true;
					}
				}
				fixed_seqpos = seqpos - 9;
				if ( ( fixed_seqpos > misc::domain_end(1) ) &&
					AnkyrinRepeat_ns::AspAsn_seqpos(fixed_seqpos) ) {
					valid_seqpos = true;
				}
			}
			if ( ! valid_seqpos ) continue;

			// jk For now, don't allow water-mediated hotspot contacts
			// jk (though this would be a good extension for later!)
			int const aav(1);

			std::cout << "Building rotamers at position " << seqpos << std::endl;

			// jk Build Trp rotamers
			if ( Task.get_designmap().get(seqpos,aa_trp) ) {

				position_hotspot_rotamer_list position_rotlist;
				convert_all_to_Asp( polyA_pose );
				polyA_pose.copy_to_misc();
				position_rotlist.set_resnum(seqpos);
				position_rotlist.setup_position_rotamers(aa_trp,aav,Task.get_designmap());
				int const num_rot = position_rotlist.num_rot();
				if ( num_rot > 0 ) {
					if ( ! hotspot_ns::screen_for_hotspot ) {
						std::cout << "About to redesign using " << num_rot <<
							" Trp rotamers in trimmed set" << std::endl;
						// jk Redesign around each of these
						//						PROF_START( prof::REDESIGN_AROUND_ROTAMER_LIST );
						convert_all_to_Asp( start_pose );
						start_pose.copy_to_misc();
						position_rotlist.redesign_around_rotamer_list( Task.get_designmap() );
						//						PROF_STOP( prof::REDESIGN_AROUND_ROTAMER_LIST );
					} else {
						position_rotlist.screen_across_rotamer_list( Task.get_designmap() );
					}
				} else {
					std::cout << "No Trp rotamers found" << std::endl;
				}
			}

			valid_seqpos=true;
			if ( hotspot_ns::hotspot_on_AR ) {
				valid_seqpos=false;
				int fixed_seqpos = seqpos - 9;
				if ( ( fixed_seqpos > misc::domain_end(1) ) &&
					AnkyrinRepeat_ns::AspAsn_seqpos(fixed_seqpos) ) {
					valid_seqpos = true;
				}
			}
			if ( ! valid_seqpos ) continue;

			// No need to build Tyr for BENCHMARK
			if ( hotspot_ns::BENCHMARK ) continue;

			// jk Build Tyr rotamers
			if ( Task.get_designmap().get(seqpos,aa_tyr) ) {
				{ // Asp scope
					position_hotspot_rotamer_list position_rotlist;
					convert_all_to_Asp( polyA_pose );
					polyA_pose.copy_to_misc();
					position_rotlist.set_resnum(seqpos);
					position_rotlist.setup_position_rotamers(aa_tyr,aav,Task.get_designmap());
					int const num_rot = position_rotlist.num_rot();
					if ( num_rot > 0 ) {
						if ( ! hotspot_ns::screen_for_hotspot ) {
							std::cout << "About to redesign using " << num_rot <<
								" Tyr rotamers in trimmed set" << std::endl;
							// jk Redesign around each of these
							//						PROF_START( prof::REDESIGN_AROUND_ROTAMER_LIST );
							convert_all_to_Asp( start_pose );
							start_pose.copy_to_misc();
							position_rotlist.redesign_around_rotamer_list( Task.get_designmap() );
							//						PROF_STOP( prof::REDESIGN_AROUND_ROTAMER_LIST );
						} else {
							position_rotlist.screen_across_rotamer_list( Task.get_designmap() );
						}
					} else {
						std::cout << "No Tyr rotamers found" << std::endl;
					}
				} // Asp scope

				// jk Skip trying the Asn here
				if ( false ) { // Asn scope
					position_hotspot_rotamer_list position_rotlist;
					convert_all_to_Asn( polyA_pose );
					polyA_pose.copy_to_misc();
					position_rotlist.set_resnum(seqpos);
					position_rotlist.setup_position_rotamers(aa_tyr,aav,Task.get_designmap());
					int const num_rot = position_rotlist.num_rot();
					if ( num_rot > 0 ) {
						if ( ! hotspot_ns::screen_for_hotspot ) {
							std::cout << "About to redesign using " << num_rot <<
								" Tyr rotamers in trimmed set" << std::endl;
							// jk Redesign around each of these
							//						PROF_START( prof::REDESIGN_AROUND_ROTAMER_LIST );
							convert_all_to_Asn( start_pose );
							start_pose.copy_to_misc();
							position_rotlist.redesign_around_rotamer_list( Task.get_designmap() );
							//						PROF_STOP( prof::REDESIGN_AROUND_ROTAMER_LIST );
						} else {
							position_rotlist.screen_across_rotamer_list( Task.get_designmap() );
						}
					} else {
						std::cout << "No Tyr rotamers found" << std::endl;
					}
				} // Asn scope

			}

		}
	}

	remove_checkpoint_file();

	//	prof::show();

	return;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin termini_involved
///
/// @brief
/// jk check whether the termini are at the interface
///
/// @detailed
///
/// @param  none
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
termini_involved()
{

	using namespace design; // interface_residue is here
	using namespace misc;

	std::cout << "JK WARNING: Termini check is disabled!" << std::endl;
	return false;

	// number of residues at each terminus which must
	// not participate in the interface
	int const terminal_residues(3);

	int startres;
	int endres;

	startres = 1;
	endres = terminal_residues;
	for ( int seqpos = startres; seqpos <= endres; ++seqpos ) {
		if ( interface_residue(seqpos) ) return true;
	}

	startres = domain_end(1)-terminal_residues+1;
	endres = domain_end(1);
	for ( int seqpos = startres; seqpos <= endres; ++seqpos ) {
		if ( interface_residue(seqpos) ) return true;
	}

	startres = domain_end(1)+1;
	endres = domain_end(1)+terminal_residues;
	for ( int seqpos = startres; seqpos <= endres; ++seqpos ) {
		if ( interface_residue(seqpos) ) return true;
	}

	startres = misc::total_residue-terminal_residues+1;
	endres = misc::total_residue;
	for ( int seqpos = startres; seqpos <= endres; ++seqpos ) {
		if ( interface_residue(seqpos) ) return true;
	}

	return false;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin read_hotspot_options
///
/// @brief
/// jk read hotspot flags
///
/// @detailed
///
/// @param  none
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
read_hotspot_options()
{

	using namespace hotspot_ns;

	if ( truefalseoption("hotspot_on_AR") ) {
		hotspot_on_AR = true;
		allow_hotspot_on_either_chain = false;
	} else if ( truefalseoption("screen_for_hotspot") ) {
		screen_for_hotspot = true;
		allow_hotspot_on_either_chain = false;
	}

	if ( truefalseoption("HS_DEBUG_UNS") ) {
		binding_mode_group_uns_thres = 93;
		binding_mode_group_uns_thres = 94;
	}

	if ( truefalseoption("HS_DEBUG_VOIDS") ) {
		hs_maxvoid_thres = 5.;
		hs_res_ball_sasa_thres = 0.02;
	}

	if ( truefalseoption("HS_DEBUG_MISC") ) {
		delta_sasa_thres = 2000.;
		hs_sasa_diff_thres = 20.;
		hs_bound_sasa_thres = 100.;
		hs_atr_thres = -6.;
		hs_sasapack_thres = 4.;
		hs_hb_thres = -0.5;
	}

	if ( truefalseoption("max_second_Hbond_rot") ) {
		int default_val = max_second_Hbond_rot;
		int newval = default_val;
		intafteroption("max_second_Hbond_rot",default_val,newval);
		max_second_Hbond_rot = newval;
	}

	//	if ( truefalseoption("HS_DEBUG_ROT_IMPROVE") ) {
	//		do_rotamer_improvement = false;
	//	}

	if ( truefalseoption("HS_DEBUG_NO_ENV") ) {
		do_periphery_last = false;
	}

	if ( truefalseoption("HS_CONFORMERS") ) {
		use_hotspot_conformers = true;
	}

	if ( truefalseoption("HS_WRITE_ALL") ) {
		output_all_designs = true;
	}

	if ( truefalseoption("HS_NO_PRESERVE_IG") ) {
		preserve_IG = false;
		preserve_RotamerSet = false;
	}

	if ( truefalseoption("HS_BENCHMARK") ) {
		BENCHMARK = true;
	}

	return;
}


