// -*- mode:c++;tab-width:1;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19070 $
//  $Date: 2007-12-12 07:55:28 -0800 (Wed, 12 Dec 2007) $
//  $Author: ronj $


#include "dna.h"
#include "DnaPose.h"
#include "dna_classes.h"
#include "dna_ns.h" // dna namespace

#include "aa_name_conversion.h" // name_from_num
#include "aaproperties_pack.h" // first_scatom, natoms
#include "analyze_interface_ddg.h" // analyze_interface_ddg
#include "analyze_interface_ddg_ns.h" // ddg flags
#include "atom_is_backbone.h" // atom_is_backbone
#include "design.h" // flags/params
#include "DesignMap.h" // DesignMap
#include "hbonds_geom.h" // hbond_evaluation_type
#include "hbonds_ns.h" // hbond arrays needed by dna_hbond_info
#include "int_fullatom_energies.h" // interface ddg energies stored here
#include "misc.h" // global structure arras
#include "options.h" // get_interface_ddg_options
#include "pack.h" // fast_pairenergy_flexible, get_sc_bbE, bump_check ...
#include "pack_geom_inline.h" // distance2_bk, move_bk, etc.
#include "PackerTask.h" // PackerTask
#include "param.h" // MAX's
#include "param_aa.h" // is_DNA, restypes
#include "param_pack.h" // Watr
#include "pdb.h" // pdb_res_num
#include "pose.h" // pose_ns::Pose
#include "read_aaproperties.h" // LookupByName
#include "read_paths.h" // open_data_file
#include "refold.h" // get_GL_matrix
#include "RotamerSet.h" // RotamerSet
#include "rotamer_functions.h" // get_rot_coord
#include "score.h" // score12()
#include "water_ns.h" // Wint_score_only, Wint_repack_only (ddg mode)

#include <ObjexxFCL/FArray1Ds.hh>
#include <ObjexxFCL/FArray2Ds.hh>
#include <ObjexxFCL/FArray3Ds.hh>
#include <ObjexxFCL/formatted.o.hh> // F(), I(), A()...

#include <numeric/xyzVector.hh>
#include <utility/basic_sys_util.hh> // utility::exit
#include <utility/io/izstream.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <cstdio>
#include <iostream>
#include <string>
#include <vector>

////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
//
// dna.cc
// Miscellaneous functions for handling the interfaces between DNA and proteins
//
// Ashworth 2007/7/12
// Havranek
// Duarte
// Morozov 2004/11/29 morozov@edsb.rockefeller.edu
//
////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// dna namespace interface: to avoid extra includes whenever possible.
////////////////////////////////////////////////////////////////////////////////

// @begin set_enable_dna
////////////////////////////////////////////////////////////////////////////////
void set_enable_dna( bool const setval ) { dna_variables::enable_dna = setval; }

// @begin dna_enabled
////////////////////////////////////////////////////////////////////////////////
bool dna_enabled() { return dna_variables::enable_dna; }

// @begin enable_rna
////////////////////////////////////////////////////////////////////////////////
void enable_rna( bool const setval ) { rna_variables::enable_rna = setval; }

// @begin dna_enabled
////////////////////////////////////////////////////////////////////////////////
bool rna_enabled() { return rna_variables::enable_rna; }

////////////////////////////////////////////////////////////////////////////////
// @begin unclash_phosphate
//
// @brief
// 5' phosphates are built in if missing, this moves them into an agreeable
// location by changing the phosphate dihedral angles
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
void
unclash_phosphate( pose_ns::Pose & pose )
{
	using namespace aaproperties_pack; // to get nheavyatoms
	using namespace param; // MAX_RES()
	using namespace dna_variables; // na_missing_phosphate
	using namespace param_aa; // restypes
	using namespace numeric::constants::f; // pi

	bool const save_pose_flag_state( pose_flag() );
	set_pose_flag( true );

	FArray2D_float mat( 3, 3 );
	FArray1D_float vec( 3 );
	float reptot;
	bool dihedral_found;
	int na;
//	int na, torsion;
//	float torsion_ang;

	float const twist = 2*pi/3; // 120 degrees
	int nres = pose.total_residue();

	for ( int pos = 1; pos <= nres; ++pos ) {
		na = pose.res(pos);
		if ( !is_NA(na) ) continue;
		if ( !na_missing_phosphate(pos) ) continue;

		reptot = 0.0;
		dihedral_found = false;

		reptot = bumpcheck_phosphate( pos, na, pose );
		std::cout << "Present conformation for added 5' phosphate on residue " <<
		 pdb::pdb_res_num(pos);
		if ( reptot < 1 ) {
			std::cout << " does not clash." << std::endl;
			continue;
		} else std::cout << " has repE of " << reptot << ", trying to relieve..\n";

//	pose.set_allow_bb_move( pos, true );

		// temporary residue coords, until I can figure this out in pose moves
		FArray2D_float na_coord( 3, natoms(na,1) );
		for ( int atom = 1; atom <= natoms(na,1); ++atom ) {
			for ( int dim = 1; dim <= 3; ++dim ) {
				na_coord(dim,atom) = pose.full_coord()(dim,atom,pos);
			}
		}

		// apply 120 degree rotations around dihedral angle gamma (O5*-C5*-C4*-C3*)
		for ( int dhg = 2; dhg <= 3; ++dhg ) { // only two more to check

			getrot_bk( na_coord(1,5), na_coord(1,6), twist, mat, vec );
			for ( int na_atom = 1; na_atom <= 4; ++na_atom ) {
				move_bk( na_coord(1,na_atom), mat, vec );
			}
			// 1H5*, 2H5*
			move_bk( na_coord(1,23), mat, vec );
			move_bk( na_coord(1,24), mat, vec );

// pose_ns::Pose atom tree...?
// fold propagates in the wrong direction
//		torsion = 1; // ??? gamma's torsion number
//		pose.set_torsion_by_number( pos, torsion, twist );

			// now 120 degree rotations around beta (P-O5*-C4*-C3*)
			for ( int dhb = 1; dhb <= 3; ++dhb ) { // three to check

				getrot_bk( na_coord(1,4), na_coord(1,5), twist, mat, vec );

				// transform coordinates of phosphate group
				for ( int na_atom = 1; na_atom <= 3; ++na_atom ) {
					move_bk( na_coord(1,na_atom), mat, vec );
				}

				// do bump check, stop if non-clashing dihedral found
				pose.copy_sidechain( pos, na, 1, na_coord(1,1), false );
				reptot = bumpcheck_phosphate( pos, na, pose );

				if ( reptot <= 1 ) {
					std::cout << "Better 5' phosphate conformation found at residue " <<
					 pdb::pdb_res_num(pos) << '\n';
					dihedral_found = true;
					set_pose_flag( save_pose_flag_state );
//				pose.set_allow_bb_move( pos, false );
					return;
				}
			}
		}
		std::cout << "Sorry, could not find alternate dihedral for 5' " <<
		 "phosphate of " << pos << " that does not clash." << std::endl;
	}
	set_pose_flag( save_pose_flag_state );
//pose.set_allow_bb_move( pos, false );
}

////////////////////////////////////////////////////////////////////////////////
// @begin mutate_base
//
// @brief
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
void
mutate_base(
	pose_ns::Pose & pose,
	int const pos,
	int const sub
)
{
	std::cout << "Mutating " << I(3,pdb::pdb_res_num(pos))
	          << " (chain " << pdb::res_chain(pos) << ") "
	          << param_aa::aa_name3( pose.res(pos) )
            << " into " << param_aa::aa_name3(sub) << std::endl;

	pose.copy_sidechain( pos, sub, 1, aaproperties_pack::icoor(1,1,sub,1) );
}

////////////////////////////////////////////////////////////////////////////////
// @begin mutate_bases
//
// @brief
// Mutate DNA nucleotides based on a DnaSeqInfo
// Note: DnaPose class has a similar method of the same name.  This version should be used only on a basic Pose.
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
void
mutate_bases(
	pose_ns::Pose & pose,
	DnaSeqInfo const & mutseq
)
{
	for ( std::vector< DnaPosInfo >::const_iterator bp( mutseq.begin() );
	      bp != mutseq.end(); ++bp ) {
		mutate_base( pose, bp->fwdpos(), bp->fwdtype() );
		if ( bp->paired() ) mutate_base( pose, bp->rvspos(), bp->rvstype() );
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin calculate_specificity
///
/// @brief
/// Given a list of energies for the target and competitor states, calculate the
/// Boltzmann probability for the target sequence (specificity).
///
/// @detailed
/// 'ref_energy': the Boltzmann distribution will ignore 'specificity' that
/// occurs at energies significantly higher than this term--it is a anchor.  To
/// use no anchor, set ref_energy to a large positive number.
///
/// @author
/// ashworth
///
/// @references
/// multistate design by JJ Havranek
///
////////////////////////////////////////////////////////////////////////////////
float
calculate_specificity(
	std::list< NamedFloat > const & spec_map,
	std::string target_seq,
	float const ref_energy,
	float const temp
)
{
	if ( spec_map.size() == 0 || target_seq == "" ) {
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );
	}

//	std::cout << "Reference energy is " << F(8,2,ref_energy) << std::endl;
//	std::cout << "Target sequence is " << A(target_seq) << std::endl;
	std::cout << "Energies vs. seqs:";

	// find low energy
	float low(ref_energy);
	for ( std::list< NamedFloat >::const_iterator it_seq( spec_map.begin() );
	      it_seq != spec_map.end(); ++it_seq ) {
		if ( it_seq->val < low ) low = it_seq->val;
	}

	float const inv_temp( 1.0/temp );
	// include reference as competitor to anchor affinity
	float energy, term;
	// initialize, including the reference(anchor) value in the denominator
	float num(0), denom( std::exp( (low-ref_energy)*inv_temp ) );
	std::string seq;
	for ( std::list< NamedFloat >::const_iterator it_seq( spec_map.begin() );
	      it_seq != spec_map.end(); ++it_seq ) {
		seq = it_seq->name;
		energy = it_seq->val;
		term = std::exp( (low-energy)*inv_temp );
		if ( seq == target_seq ) num += term;
		denom += term;
		std::cout << " " << it_seq->name << ":" << F(8,2,energy);
	}
	float spec( num / denom );
	std::cout << " spec: " << F(6,2,100.*spec) << std::endl;

	return spec;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin bumpcheck_phosphate
/// Return 5' phosphate repulsive to protein
///
/// @brief
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

float
bumpcheck_phosphate(
	int const pos,
	int const na,
	pose_ns::Pose const & pose
)
{
	using namespace aaproperties_pack; // nheavyatoms
	using namespace param; // MAX_ATOM(), MAX_RES()
	using namespace param_aa; // is_protein
	using namespace pdb; // missing_input_atom

	FArray2D_bool save_missing_input_atom( MAX_ATOM(), MAX_RES() );
	float solvE(0.0), atrE(0.0), repE(0.0), elecE(0.0);

	int nres( pose.total_residue() ), aa2;

	for ( int pos2(1); pos2 <= nres; ++pos2 ) {
		aa2 = pose.res(pos2);

		//ignore intra-residue repE
		if ( pos == pos2 ) continue;

		// skip distant residues. Atom 1 for pos is phosphorus.
		float dis2(0);
		distance2_bk( pose.full_coord()(1,1,pos),
		 pose.full_coord()(1,2,pos2), dis2 );
		if ( dis2 > 144 ) continue;

		// do not want to use sub_missing_atomtypes in this context
		for ( int i(1); i <= MAX_RES(); ++i ) {
			for ( int j(1); j <= MAX_ATOM(); ++j ) {
				save_missing_input_atom(j,i) = missing_input_atom(j,i);
				missing_input_atom(j,i) = false;
			}
		}

		// call phosphate-everything energy
		fast_pairenergy_flexible( pos, na, 1, 4, pose.full_coord()(1,1,pos),
		 pos2, aa2, 1, nheavyatoms(aa2,1), pose.full_coord()(1,1,pos2), solvE,
		  atrE, repE, elecE );

		missing_input_atom = save_missing_input_atom;
	}
	return repE;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin base_proximity
///
/// @brief
/// like are_they_neighbors, but base-specific.
///
/// @detailed
/// checks c-beta (except glycine) to base atom-atom distances, not including
/// ribose or phosphate backbone.  Filters out anything that might be outside
/// of arginine rotamer contact range, in a rapid and coarse manner, before
/// proceeding to rotamer sweeps and all-atom distance checks.  Also
/// responsible finding residues that will be re-packed but not re-designed.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

bool
base_proximity(
	int const paa,
	int const daa,
	FArray2Da_float xyz1,
	FArray2Da_float xyz2,
	float const & threshold2
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	xyz1.dimension( 3, MAX_ATOM() );
	xyz2.dimension( 3, MAX_ATOM() );

	float dis2, best_dis2( 10000.0 );

	int const batm_s(dna_bb_atoms + 1); // first base atom

	for (int batm = batm_s, batm_e = nheavyatoms(daa,1); batm <= batm_e; ++batm) {

		// measure from c-beta unless gly
		if ( paa == aa_gly ) distance2_bk( xyz1(1,2), xyz2(1,batm), dis2 );
		else distance2_bk( xyz1(1,first_scatom(paa,1)), xyz2(1,batm), dis2 );

		if ( dis2 < best_dis2 ) best_dis2 = dis2;
	}
	return ( best_dis2 < threshold2 );

}
////////////////////////////////////////////////////////////////////////////////
/// @begin argrot_base_specific_neighbor
///
/// @brief
/// A way of using Jim's arginine rotamer idea to indentify residues that
/// specifically contact DNA bases.
///
/// @references
///
/// Ashworth J, Havranek J, Duarte C, Sussman D, Monnat R, Stoddard B, Baker D.
/// Nature 441, 656-659 (2006)
///
/// @authors
/// havranek
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
bool
argrot_base_specific_neighbor(
	int const ppos,
	int const dpos,
	pose_ns::Pose const & pose,
	float const & threshold2
)
{
	using namespace design;
	using namespace exchi_flags;
	using namespace param;
	using namespace param_aa;
	using namespace param_pack;

	float distance2, best_distance2( 100 );

	RotamerOptions saved_rotamer_options = active_rotamer_options;

	 // ignore rotamer explosion
	active_rotamer_options.rot_explode = false;

	// use at least ex1,ex2 rotamers
	active_rotamer_options.ex1 = true;
	active_rotamer_options.ex1_sample_level = EX_ONE_STDDEV;
	active_rotamer_options.ex2 = true;
	active_rotamer_options.ex2_sample_level = EX_ONE_STDDEV;

	PackerTask task( pose );
	task.get_designmap().all_true_repack_residue();

	RotamerSet rotamer_set;
	int nrotaa(0);
	std::string mode = "rotamerize"; //ja no bump check rejection in this mode
	rotamer_set.get_rotamers_seqpos_aa_aav( pose.total_residue(), pose.res(), pose.res_variant(),
    pose.full_coord(),aa_arg, 1, ppos, nrotaa, false, mode, task );

	float rep_total, dis2, repE;
	float atrE, solvE, elecE; // dummies
	int baa;
	for ( int rot(1), rot_e( rotamer_set.nrotamers() ); rot <= rot_e; ++rot ) {
		rep_total = 0.0;

		// Rotamer screen begins by filtering out anything that clashes
		// with any protein backbone
		for ( int bres(1); bres <= pose.total_residue(); ++bres ) {
			baa = pose.res(bres);

			if ( !is_protein(baa) || !is_nonnatural(baa) ) continue;

			// don't check any backbones further away than an arg sidechain
			dis2 = 0;
			distance2_bk( rotamer_set.get_rotcoord(rot)(1,2),
			 pose.full_coord()(1,2,bres), dis2 );
			if ( dis2 > 144 ) continue;

			repE = 0;
			get_sc_bbE( aa_arg, 1, baa, 1, rotamer_set.get_rotcoord(rot),
			 pose.full_coord()(1,1,bres), ppos, bres, solvE, atrE, repE, elecE );

			rep_total += repE;
			if ( rep_total > 10.0 ) break;
		}
		if ( rep_total > 10.0 ) continue;

		int dna_type = pose.res(dpos);

		// do atom-atom distance check between rotamer and DNA base
		distance2 = rotamer_base_distance( dna_type, 1, aa_arg, 1,
		 pose.full_coord()(1,1,dpos), rotamer_set.get_rotcoord(rot),
		 threshold2 );

		if ( distance2 < threshold2 ) {
			active_rotamer_options = saved_rotamer_options;
			return true;
		}
		else if ( distance2 < best_distance2 ) best_distance2 = distance2;
	}

	active_rotamer_options = saved_rotamer_options;
	return false;

}
////////////////////////////////////////////////////////////////////////////////
/// @begin rotamer_base_distance
///
/// @brief
/// Distance check (all-atom) between protein rotamer atoms and DNA base atoms.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

float
rotamer_base_distance(
	int const daa,
	int const daav,
	int const raa,
	int const raav,
	FArray2Da_float dna_coord,
	FArray2Da_float rot_coord,
	float const & threshold2
)
{
	using namespace aaproperties_pack;
	using namespace param;
	using namespace param_aa;

	dna_coord.dimension( 3, MAX_ATOM() );
	rot_coord.dimension( 3, MAX_ATOM() );

	float distance2, best_distance2 = 100;
	const int base_startatom = dna_bb_atoms + 1; // first base atom

	for ( int ratm = (first_scatom(raa,raav)), ratm_e = nheavyatoms(raa,raav),
	 batm_e = nheavyatoms(daa,daav); ratm <= ratm_e; ++ratm ) {
		for ( int batm = base_startatom; batm <= batm_e; ++batm ) {

			distance2 = 0;
			distance2_bk( rot_coord(1,ratm), dna_coord(1,batm), distance2 );

			if ( distance2 < threshold2 ) return distance2;
			else if ( distance2 < best_distance2 ) best_distance2 = distance2;
		}
	}

	return best_distance2; // nearest if threshold is never crossed

}

////////////////////////////////////////////////////////////////////////////////
/// @begin z_axis_check
///
/// @brief
/// A sanity check for the arginine rotamer screen. Can prevent the design of positions that are best left alone because they are too far away along the helical axis ('laterally').
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
float z_axis_check(
	FArray1Da_float pcoord,
	int const dtype,
	FArray2Da_float dcoord
)
{
	using namespace aaproperties_pack;

	pcoord.dimension( 3 );
	dcoord.dimension( 3, param::MAX_ATOM() );

	numeric::xyzVector<float> Z( base_z_axis( dtype, dcoord ) ),
	                          prot( &pcoord(1) ),
                            // 'first_scatom' SHOULD be on the dna base
	                          dna( &dcoord(1,first_scatom(dtype,1)) ),
	                          vec;
	// vector from protein atom to DNA atom
	vec = prot - dna;
// return scalar projection of vec onto DNA helical axis
	return dot(vec,Z);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin base_z_axis
///
/// @brief
/// z-axis unit vector for a dna base
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
numeric::xyzVector<float>
base_z_axis(
	int const dtype,
	FArray2Da_float dcoord
)
{
	using namespace param_aa; // na_types
	dcoord.dimension( 3, param::MAX_ATOM() );

	numeric::xyzVector<float> X, Y, Z;

	int a1, a2;
	get_y_axis_atoms( dtype, 1, a1, a2 );
	numeric::xyzVector<float> y1( &dcoord(1,a1) ), y2( &dcoord(1,a2) );
	Y = y2 - y1;

	// use the C5 atom to define x-y plane
	int xa(0);
	if ( dtype == na_gua ) xa = 17;
	else if ( dtype == na_ade ) xa = 16;
	else if ( dtype == na_cyt || dtype == na_thy ) xa = 18;
	else utility::exit( EXIT_FAILURE, __FILE__, __LINE__ );

	numeric::xyzVector<float> x( &dcoord(1,xa) );
	X = x - y1; // not really x but close enough for cross product

	Z = cross( X, Y );
	Z.normalize();

	return Z;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin base_specific_hbond
///
/// @brief
/// Filters rotamers for hydrogen bonding to the DNA bases in the major groove.
/// Used with "rotamer_explosion" to allow oversampling of long polar
/// sidechains.
///
/// @detailed
/// Calls hb_energy_deriv() directly, mimicking the loops in get_aa_1way_hbE_B()
///
/// @references
/// Ashworth J, Havranek J, Duarte C, Sussman D, Monnat R, Stoddard B, Baker D.
/// Nature 441, 656-659 (2006)
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

float
base_specific_hbond(
	int const daa,
	int const daav,
	int const paa,
	int const paav,
	FArray2Da_float dcoord,
	FArray2Da_float pcoord
)
{
	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace param_aa;
	using namespace param;
	using namespace pdb;

	dcoord.dimension( 3, MAX_ATOM() );
	pcoord.dimension( 3, MAX_ATOM() );

	FArray2D_float deriv( 3, 2 ); // dummy

	//make sure daa is dna, and paa is protein
	if ( !is_NA(daa) || !is_protein(paa) ) {
		std::cout << "Improper residue types in base_specific_hbond!" << std::endl;
		return 0.0;
	}
	// base_proximity distance cutoff
	float const prox_dis2 = 100; // 10 squared (c-beta to base distance)
	if ( !base_proximity( paa, daa, pcoord, dcoord, prox_dis2 ) ) return 0.0;

	float sc_base_hbE = 0.0, energy;

	int patom, base, base2;

	// protein donor hydrogens
	for ( int ph = 1, ph_e = nH_polar(paa,paav), datom_e = nheavyatoms(daa,daav);
	 ph <= ph_e; ++ph ) {
		patom = Hpos_polar(ph,paa,paav);

		// for dna major groove acceptors atoms
		for ( int datom = dna_bb_atoms; datom <= datom_e; ++datom ) {
			if ( !major_groove_hb_atom[ daa ][ datom ] ) continue;
			base = atom_base(datom,daa,daav); // acceptor base on DNA
			base2 = abase2(datom,daa,daav);
      HBEvalType const hbe_type = hbond_evaluation_type(patom,paa,paav, datom,daa,daav);
			hb_energy_deriv(hbe_type, pcoord(1,atom_base(patom,paa,paav)),
			pcoord(1,patom), dcoord(1,datom), dcoord(1,base), dcoord(1,base2), energy);
			if ( energy < 0.0 ) { // important: non-hb results in default +1
				sc_base_hbE += energy;
			}
		}
	}

	// dna major groove donor hydrogens
	for ( int dh = nheavyatoms(daa,daav), datom_e = natoms(daa,daav),
	 pacc_e = nacceptors(paa,paav); dh <= datom_e; ++dh ) {
		if ( !major_groove_hb_atom[ daa ][ dh ] ) continue;

		// protein acceptors
		for ( int pacc(1); pacc <= pacc_e; ++pacc ) {
			patom = accpt_pos(pacc,paa,paav); // acceptor on protein
			base = atom_base(patom,paa,paav);
			base2 = abase2(patom,paa,paav);
      int const datom = atom_base(dh,daa,daav); // donor on dna
      HBEvalType const hbe_type = hbond_evaluation_type(datom,daa,daav, patom,paa,paav);
			hb_energy_deriv(hbe_type, dcoord(1,datom), dcoord(1,dh),
			pcoord(1,patom), pcoord(1,base), pcoord(1,base2), energy);
			if ( energy < 0.0 ) {  // important: non-hb results in default +1
				sc_base_hbE += energy;
			}
		}
	}
	return sc_base_hbE;

}
////////////////////////////////////////////////////////////////////////////////
/// @begin thy_vdw_contact
///
/// @brief
/// Looks for packing distances between rotamer and thymine methyl carbon.
///
/// @references
///
/// @authors
/// duarte
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////

bool
thy_vdw_contact(
	int const paa,
	int const paav,
	FArray2DB_float const & dna_coord,
	FArray2DB_float const & rot_coord
)
{
	using namespace aaproperties_pack;
	using namespace param_aa;

	int thy_methyl = 15;
	float contact_threshold = 18.0; // distance squared (~4.25 angstroms)
	float dis2;

	for ( int atom = 1, atom_e = nheavyatoms(paa,paav); atom <= atom_e; ++atom ) {
		dis2 = 20.0;
		distance2_bk( rot_coord(1,atom), dna_coord(1,thy_methyl), dis2 );
		if ( dis2 < contact_threshold ) return true;
	}
	return false;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin all_dna_combinations
///
/// @brief
/// Sets up combinations of DNA sequence identities for sequence scanning, based on a vector of arbitrary position indices. Recursive.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
all_dna_combinations(
	unsigned index,
	std::vector<int> const & positions,
	DnaSeqInfo & seq,
	std::list< DnaSeqInfo > & seqs
)
{
	using namespace param; // MAX_AA
	using namespace param_aa; // is_DNA

	for ( int na(1); na <= MAX_AA(); ++na ) {
		if ( !is_DNA(na) ) continue;

		int const pos( positions[index] );
		seq[pos].set_type( na, na_partner(na) );

		if ( index == positions.size()-1 ) seqs.push_back( seq );
		else all_dna_combinations( index+1, positions, seq, seqs );
	}
}

////////////////////////////////////////////////////////////////////////////////

void
debug_print_seqs( std::list< DnaSeqInfo > & seqs )
{
	using namespace param_aa;

	for ( std::list< DnaSeqInfo >::iterator seq( seqs.begin() );
	      seq != seqs.end(); ++seq ) {

		for ( std::vector< DnaPosInfo >::const_iterator bp( seq->begin() );
		 bp != seq->end(); ++bp ) {
			std::cout << pdb::pdb_res_num( bp->fwdpos() ) << " : " <<
			 aa_name1( bp->fwdtype() ) << ", ";
			std::cout << pdb::pdb_res_num( bp->rvspos() ) << " : " <<
			 aa_name1( bp->rvstype() ) << ", " << std::endl;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin restrict_dna_rotamers
///
/// @brief
/// Force a single DNA sequence out of a multi-DNA-sequence RotamerSet /
/// InteractionGraph in the packer, using a rot_to_pack vector.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
restrict_dna_rotamers(
	RotamerSet const & set,
	DnaSeqInfo & seq,
	std::vector<int> & rot_to_pack
)
{
	rot_to_pack.clear();
	int const nrot( set.nrotamers() );
	for ( int rot(1); rot <= nrot; ++rot ) {

		int const rotpos( set.report_seqpos(rot) ), rotna( set.report_aa(rot) );

		if ( param_aa::is_NA(rotna) && seq.contains( rotpos ) ) {
			if ( !seq.contains( rotpos, rotna ) ) continue;
		}
		rot_to_pack.push_back( rot );
	}
	if ( dna_variables::dna_verbose ) {
		int const rots_off( nrot - rot_to_pack.size() );
		std::cout << "Fixing DNA rotamers: " << rots_off << " out of " << nrot <<
		 " rotamers disabled." << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin restrict_dna_rotamers
///
/// @brief
/// This version uses a 'res' array as the reference instead of a DnaSeqInfo.
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
restrict_dna_rotamers(
	RotamerSet const & set,
	FArray1DB_int const & res,
	std::vector<int> & rot_to_pack
)
{
	rot_to_pack.clear();
	int const nrot( set.nrotamers() );
	for ( int rot(1); rot <= nrot; ++rot ) {
		int const rotpos( set.report_seqpos(rot) ), rotna( set.report_aa(rot) );
		if ( param_aa::is_NA(rotna) && rotna != res(rotpos) ) continue;
		rot_to_pack.push_back( rot );
	}
	if ( dna_variables::dna_verbose ) {
		int const rots_off( nrot - rot_to_pack.size() );
		std::cout << "Fixing DNA rotamers: " << rots_off << " out of " << nrot <<
		 " rotamers disabled." << std::endl;
	}
}

////////////////////////////////////////////////////////////////////////////////
/// @begin start_scan_at
///
/// @brief
/// truncate the sequence list before the given start seq
///
/// @authors
/// ashworth
///
////////////////////////////////////////////////////////////////////////////////
void
start_scan_at(
	std::string const & single_seq,
	std::list< DnaSeqInfo > & seqs
)
{
	if ( single_seq == "" ) return;
	for ( std::list< DnaSeqInfo >::iterator seq( seqs.begin() );
	      seq != seqs.end(); ++seq ) {

		std::string seq_string( seq->str() );
		if ( seq_string == single_seq ) {
			seqs.erase( seqs.begin(), seq );
			break;
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin ref_seq_from_resfile
//
// @brief
// Uses the load_resfile() function to define a reference sequence that is different from that of the input structure (perhaps the input structure is not 'wildtype').  This reference resfile must specify the reference sequence using PIKAA tags for every residue in the protein.
//
// @remarks
// Doesn't actually need PackerTask: just forced to use one to get at resfile.
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
void
ref_seq_from_resfile(
	std::vector<int> & ref_seq,
	FArray1DB_int const & res,
	int const total_residue
)
{
	using namespace param;
	using namespace param_aa;
	using namespace pdb;

	ref_seq.clear();

	// dummy arrays for PackerTask(s)
	FArray1D_bool allow_repack( MAX_RES(), true );
	std::string ref_resfile = "";
	stringafteroption( "ref_resfile", "", ref_resfile );
	std::cout << "Loading alternative reference sequence from resfile " <<
	 ref_resfile << std::endl;

	// extricate resfile settings from PackerTask
	pose_ns::Pose dummypose;
	PackerTask dummytask(dummypose);
	// initialize: sizes the DesignMap
	dummytask.set_task( "design", false, allow_repack, false );
	// hijack load_resfile and DesignMap to get the resfile-defined sequence
	dummytask.load_resfile( ref_resfile );
	DesignMap dummymap( dummytask.get_designmap() );

	for ( int pos(1); pos <= total_residue; ++pos ) {
		ref_seq.push_back(0); // initializes vector
		if ( is_NA( res(pos) ) ) continue;
		if ( !dummymap.repack_residue(pos) ) {
			std::cout << "You didn't set any residues for " << pdb_res_num(pos) <<
			 std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		for ( int aa(1); aa <= MAX_AUTH_AA; ++aa ) {
			if ( dummymap.get(pos,aa) ) {
				ref_seq[pos-1] = aa;
				std::cout << "Reference seq at " << pdb_res_num(pos) << " is " <<
				 aa_name3(aa) << std::endl;
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin dna_scan_bumpcheck
//
// @brief
// This function is incomplete (copy into new rotamer set excluding bumplist members -- requires some reordering).
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
void
dna_scan_bumpcheck(
	RotamerSet & rotamer_set,
	pose_ns::Pose const & pose
)
{
	using namespace aaproperties_pack;
	using namespace param_aa;
	using namespace param_pack;

	float atrE, repE, solE, bumpenergy, elecE;
	std::vector< int > bumplist;

	int nrot = rotamer_set.nrotamers(), rpos, raa, na;
	for ( int rot = 1; rot <= nrot; ++rot ) {
		rpos = rotamer_set.report_seqpos(rot);
		raa = rotamer_set.report_aa(rot);
//		FArray2D_float rcoord = rotamer_set.get_rotcoord(rpos);
		for ( int dna = 1, dna_e = pose.total_residue(); dna <= dna_e; ++dna ) {
			if ( !param_aa::is_NA(dna) ) continue;
			na = pose.res(dna);

			// Checks only against sugar + phosphate bb.  This will reduce the rotamer
			// set size while maintaining fairness across all four DNA nucleotides
			fast_pairenergy_flexible( rpos, raa, first_scatom(raa,1),
				nheavyatoms(raa,1), rotamer_set.get_rotcoord(rot), dna, na, 1,
				dna_bb_atoms, pose.full_coord()(1,1,dna), solE, atrE, repE,elecE );

			bumpenergy = (pack_wts.Watr() * atrE) + (pack_wts.Wrep() * repE);

			if ( bumpenergy > max_rot_bumpenergy ) bumplist.push_back( rot );
		}
	}
	rotamer_set.remove( bumplist );

}

////////////////////////////////////////////////////////////////////////////////
// @begin dna_ddG
//
// @brief
// Calls analyze_interface_ddg() in dna-interface-specific manner, outputs
// energies to file.  This function is called from make_pdb.cc.  In the
// structure, demands protein first (must be first residue), followed by DNA.
// (This is PDB standard.)
//
// @authors
// Ashworth
//
////////////////////////////////////////////////////////////////////////////////

void
dna_ddG(
	utility::io::orstream & iunit,
	std::list< ResInfo > const & interface_list
)
{
	using namespace analyze_interface_ddg_ns;
	using namespace design;
	using namespace int_fullatom_energies; // ddg energies
	using namespace param;
	using namespace param_aa; // aa_name3
	using namespace param_pack;
	using namespace pdb; // pdb_res_num
	using namespace water::weight_flags; // for some reason, analyze_interface_dgg calls code in water.cc

	std::cout << "\nDNA interface ddG calculation\n" << std::endl;

	float Watr_check( pack_wts.Watr() ); // make sure weights aren't mucked with

	bool const ddg_decoystats_save( ddg_decoystats );
	ddg_decoystats = false; // don't care about separated partners

	int const ndruns_save( ndruns );
	design::ndruns = 1;

	get_interface_ddg_options(); // initialize ddg mode
	// the following settings are hardcoded
	Wint_score_only = false; // this changes the weights if true!
	Wint_repack_only = false; // this changes the weights if true!
	ddg_per_residue = true;
	ddg_bind_only = true; // one structure: don't do a "list of mutated complexes"

	analyze_interface_ddg();

	std::cout << "DDG Binding energy (DNA design weights): " <<
	 dgbind_resenergy(1) << '\n';

	// begin output to pdb file
	iunit << '\n';
	iunit << "DDG Binding energy (DNA design weights): " <<
	 F(6,2,dgbind_resenergy(1)) << '\n';

	std::list< std::string > ddGs;

	//ja DDG per residue, for repacked or redesigned residues
	for ( std::list< ResInfo >::const_iterator rsd( interface_list.begin() );
	      rsd != interface_list.end(); ++rsd ) {

		float res_ddg( complex_resenergy_res( 3, rsd->pos ) -  // complex
		               complex_resenergy_res( 2, rsd->pos ) -  // partner A
		               complex_resenergy_res( 1, rsd->pos ) ); // partner B

		ddGs.push_back( aa_name3( misc::res(rsd->pos) ) + " " +
		                I( 4, pdb_res_num(rsd->pos) ) +
		                A( 2, res_chain(rsd->pos) ) +
		                F( 5, 1, res_ddg ) );
	}

	output_res_list( "ddg list (molten residues)", iunit, ddGs );

	if ( pack_wts.Watr() != Watr_check ) {
		iunit << "WARNING: NON-DNA WEIGHTS USED IN ANALYZE_INTERFACE_DDG!" << '\n';
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	ddg_decoystats = ddg_decoystats_save;
	ndruns = ndruns_save;
}


////////////////////////////////////////////////////////////////////////////////
// @begin output_NA_base_specific_energy
//
// @brief
// Output wrapper for NA_base_specific_energy
//
// @authors
// Ashworth
//
////////////////////////////////////////////////////////////////////////////////
void
output_NA_base_specific_energy(
	utility::io::orstream & iunit,
	DnaSeqInfo const & dna
)
{
	float sol(0), atr(0), rep(0), hb(0), elec(0);
	NA_base_specific_energy( sol, atr, rep, hb, elec, dna );

	iunit << "\nDirect interactions to designed bases:" << std::endl;
	iunit << "      " << A(6,"atr") << A(6,"rep") << A(6,"sol") << A(6,"hb") <<
	 std::endl;
	iunit << "BSPEC " << F(6,1,atr) << F(6,1,rep) << F(6,1,sol) << F(6,1,hb) <<
	 std::endl;
}

////////////////////////////////////////////////////////////////////////////////
// @begin NA_base_specific_energy
//
// @brief
// Get the energy of interaction to the designed DNA bases only
// Works in global (misc) space because it is called during pdb output
//
// @authors
// Ashworth
//
////////////////////////////////////////////////////////////////////////////////
void
NA_base_specific_energy(
	float & sol,
	float & atr,
	float & rep,
	float & hb,
  float & elec,
	DnaSeqInfo const & dna
)
{
	using namespace aaproperties_pack; // first_scatom, natoms
	using namespace misc;
	using namespace param; // MAX dimemsions
	using namespace param_aa; // is_protein etc.
//	using namespace param_pack; // Wgb_elec

//	int nb1 = 30, nb2 = 30; // number of neighbors: for unattenuated hbonds

	for ( int ppos(1); ppos <= total_residue; ++ppos ) {
		int const paa = res(ppos);
		if ( !is_protein(paa) && !is_nonnatural(paa) ) continue;
		int const paav( res_variant(ppos) );

		for ( int dpos(1); dpos <= total_residue; ++dpos ) {
			if ( !dna.contains(dpos) ) continue;
			if ( !dna[dpos].design() ) continue;
			int const daa( res(dpos) ), daav( res_variant(dpos) );

			fast_pairenergy_flexible( ppos, paa, first_scatom(paa,paav),
			 nheavyatoms(paa,paav), full_coord(1,1,ppos), dpos, daa, dna_bb_atoms+1,
			 nheavyatoms(daa,daav), full_coord(1,1,dpos), sol, atr, rep, elec );

			hb = base_specific_hbond( daa, daav, paa, paav,
			 full_coord(1,1,dpos), full_coord(1,1,ppos) );

/* ja will need to call this for specific atom ranges
			// gen. born energy
			float gbE = 0.0;
			gbE = gb_get_res_res_elecE( ppos, paa, paav,
			 pcoord, born_radius(1,ppos), dpos, daa, daav,
			 dcoord, born_radius(1,dpos) );
			gbE *= Wgb_elec;
*/
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin residue_distance_check
//
// @brief
// determines whether any atom of an amino acid is within 6 Angstroms of
// any DNA atom (approx. the max distance for a water mediated hydrogen bond)
//
////////////////////////////////////////////////////////////////////////////////
void
residue_distance_check(
	int const res1,
	int const res2,
	FArray2Da_float xyz1,
	FArray2Da_float xyz2,
	bool & neighbor,
	float & dis2
)
{
	using namespace aaproperties_pack;
	using namespace param;

	xyz1.dimension( 3, MAX_ATOM() );
	xyz2.dimension( 3, MAX_ATOM() );

// jjh This was the cvs value when I merged
//	float const cutoff = 4.0; // Two atom distance
// jjh This was AM's value when I merged
	float const cutoff = 6.0; // Two atom distance
	float const cutoff2 = square( cutoff );

	float best_dis = 100000.0;
	neighbor = false;
	for ( int atom1 = 1, atom1e = natoms( res1, 1 ), atom2e = natoms( res2, 1 );
	 atom1 <= atom1e; ++atom1 ) {
		for ( int atom2 = 1; atom2 <= atom2e; ++atom2 ) {
			distance2_bk( xyz1(1,atom1), xyz2(1,atom2), dis2 );
			if ( dis2 < best_dis ) best_dis = dis2;
			if ( dis2 < cutoff2 ) {
				neighbor = true;
				return;
			}
		}
	}

}
////////////////////////////////////////////////////////////////////////////////
// @begin sub_missing_atomtypes
//
// @brief
// for DNA/RNA - any missing atoms are switched to virtual atom types
// for energy calculations
//
// @authors
// duarte
//
////////////////////////////////////////////////////////////////////////////////

int
sub_missing_atomtypes(
	int const atm1,
	int const aa1,
	int const aav1,
	int pos1
)
{
	using namespace aaproperties_pack;
	using namespace param_aa;
	using namespace pdb;

	if ( is_protein(aa1) || is_nonnatural(aa1) ) {
		return fullatom_type(atm1,aa1,aav1);
	} else {
		if ( missing_input_atom(atm1,pos1) ) {
//      std::cout << "virtual atom turned on for atom " << atm1 <<
//        " at position " << pos1 << " type " << aa1 << std::endl;
			return 50; //Hard-wired code for virtual atom V1
		} else {
			return fullatom_type(atm1,aa1,aav1);
		}
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin output_res_list
//
// @authors
// ashworth
////////////////////////////////////////////////////////////////////////////////
void
output_res_list(
	std::string const tag,
	utility::io::orstream & iunit,
	std::list< std::string > & residues
)
{
	iunit << tag;

	for ( std::list< std::string >::iterator res( residues.begin() );
	      res != residues.end(); ++res ) {
		iunit << "," << *res;
	}
	iunit << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
// @begin dna_hbond_info
//
// @brief
// Generates a table of protein/DNA (and protein-protein for molten residues)
//  hydrogen bonds and categorizes them by type
//
// @authors
// cmd
// ashworth
//
////////////////////////////////////////////////////////////////////////////////

void
dna_hbond_info(
	utility::io::orstream & iunit,
	std::list< ResInfo > const & interface_list
)
{
	using namespace aaproperties_pack;
	using namespace files_paths;
	using namespace hbonds;
	using namespace misc;
	using namespace pdb;
	using namespace param_aa;

	// first output column headers for file readability
	iunit << "\nLoc, res, pos, pdb, atom, res, pos, pdb, atom, hbE, dist, spec"<<
	 '\n';

	//ja temporarily translate short list of design residues to vector< bool > for
	// fast lookup in following hbond loop
	std::vector< bool > design_residues( total_residue, false );
	for ( std::list< ResInfo >::const_iterator
	      resinf( interface_list.begin() );
	      resinf != interface_list.end(); ++resinf ) {
		design_residues[ resinf->pos ] = true;
	}

	//ja get a total protein-BASE hydrogen-bonding energy
	float hb_spec( 0.0 );

	int don_pos, acc_pos, don_aa, acc_aa, don_aav, acc_aav, don_atom, acc_atom;
	char don_chain, acc_chain;
	float hbE, dist;
	std::string don_name, acc_name, atom_location, specificity;

	// loop through the array of hbonds (previously calculated)
	// collect information about donors and acceptors
	for ( int hbond = 1; hbond <= hbond_set.nhbonds(); ++hbond ) {
		don_pos = hbond_set.hbdon_res(hbond);
		acc_pos = hbond_set.hbact_res(hbond);
		don_aa = res(don_pos);
		acc_aa = res(acc_pos);

		//rj skip the time-saving evaluations if output_hbond_info flag is set
		if ( !output_hbond_info ) {
			if ( is_NA(don_aa) && is_NA(acc_aa) ) continue; // skip NA-NA
			if ( !is_NA(don_aa) && !is_NA(acc_aa) &&
			     !design_residues[don_pos] && !design_residues[acc_pos] ) {
				continue; // skip non-designed protein-protein
			}
		}

		don_aav = res_variant(don_pos);
		acc_aav = res_variant(acc_pos);
		don_chain = res_chain(don_pos);
		acc_chain = res_chain(acc_pos);
		don_atom = hbond_set.hbdonh_atm(hbond);
		acc_atom = hbond_set.hbact_atm(hbond);
		hbE = hbond_set.hbenergies(hbond);

		// skip backbone-backbone hbonds
		//rj unless the output_hbond_info flag is set
		if ( !output_hbond_info &&
			atom_is_backbone( don_atom, don_aa, don_aav ) &&
			atom_is_backbone( acc_atom, acc_aa, acc_aav ) ) continue;
			
		// skip output of intramolecular hbonds if -output_interface_hbonds_only flag is set
		if ( (output_hbond_info && output_interface_hbonds_only) && (don_chain == acc_chain) ) {
			continue;
		}

		// ja hbond distance
		dist = 0;
		distance_bk( full_coord(1,don_atom,don_pos), full_coord(1,acc_atom,acc_pos),
		 dist );

		if ( is_NA(don_aa) != is_NA(acc_aa) ) {	// protein-DNA
			if ( is_NA(don_aa) ) {
				atom_location = dna_bb_or_base( don_aa, don_atom );
				if ( atom_location == "BASE" ) {
					specificity = dna_specificity( don_aa, don_atom, don_aav );
					hb_spec += hbE;
				} else specificity = "ALL";
			} else {
				atom_location = dna_bb_or_base( acc_aa, acc_atom );
				if ( atom_location == "BASE" ) {
					specificity = dna_specificity( acc_aa, acc_atom, acc_aav );
					hb_spec += hbE;
				} else specificity = "ALL";
			}
		} else { // designed protein-protein
			atom_location = "PROT";
			specificity = "none";
		}

		// output information about this H-bond
		name_from_num( don_aa, don_name );
		name_from_num( acc_aa, acc_name );
		iunit << atom_location << " " <<
		 don_name << " " <<	I(4,don_pos) << " " << I(4,pdb_res_num(don_pos)) <<
		 " " << don_chain << " " << atom_name(don_atom,don_aa,don_aav) << " " <<
		 acc_name << " " << I(4,acc_pos) << " " << I(4,pdb_res_num(acc_pos)) <<
		 " " << acc_chain << " " << atom_name(acc_atom,acc_aa,acc_aav) << " " <<
		 F(6,2,hbE) << " " << F(6,2,dist) << " " << specificity << '\n';

	}

	iunit << "base-specific hbond energy: " << F(6,2,hb_spec) << std::endl;

}

////////////////////////////////////////////////////////////////////////////////
// @begin dna_bb_or_base
//
// @brief
// For DNA/RNA - determines whether a given atom is from a DNA backbone or base
//
// @authors
// cmd
//
////////////////////////////////////////////////////////////////////////////////
std::string
dna_bb_or_base(
	int const dna_type,
	int const dna_atom
)
{
	using namespace dna_variables;
	using namespace param_aa;

	std::string dna_atom_class;

	if ( !is_NA(dna_type) ) {
		std::cout << "dna_bb_or_base should only be called for non_proteins\n";
		dna_atom_class = "BAD ";
	}
	else if ( is_RNA( dna_type ) ) {
		if ( dna_atom <= rna_bb_atoms ) dna_atom_class = "bkbn";
		else dna_atom_class = "BASE";
	}
	else if ( dna_atom <= dna_bb_atoms ) dna_atom_class = "bkbn";
	else dna_atom_class = "BASE";

	return dna_atom_class;
}

////////////////////////////////////////////////////////////////////////////////
// @begin dna_specificity
//
// @brief
// Identifies the specificity for protein-DNA H-bonding interactions
//
// @authors
// cmd
//
////////////////////////////////////////////////////////////////////////////////

std::string
dna_specificity(
	int const daa,
	int const datom,
	int const daav
)
{
	using namespace dna_variables;
	using namespace param_aa;
	using namespace aaproperties_pack;
	using namespace misc;

	std::string specificity( "BAD" ); // "BAD" is default return value
	std::string dname;

	name_from_num( daa, dname );

// Error checks for amino-acids or for out of range atom numbers
	if ( !is_NA(daa) ) {
		std::cout << "dna_specificity should only be called for non-amino acids\n";
	} else if ( datom > natoms(daa,daav) ) {
		std::cout << "atom number " << I(2,datom) << " is out of range for " <<
			dname << std::endl;
	} else {
		bool located = false;
// Loop for hydrogens in H-bonds (donors).
// All polar hydrogens in DNA are base specific
		if ( datom > nheavyatoms(daa,daav) ) {
// for backbone H5* in RNA (non-specific)
			if ( is_RNA(daa) ) {
				if ( datom == Hpos_polar(1,daa,daav) ) {
					specificity = "ALL";
					located = true;
				}
			}
			if ( !located ) {
				for ( int i = 1, ie = nH_polar(daa,daav); i <= ie; ++i ) {
					if ( datom == Hpos_polar(i,daa,daav) ) {
						located = true;
						specificity = dname;
					}
				}
			}
// Loop for heavy atoms (acceptors)
// Identify atoms that are common to purines or pyrimidines
		} else {

			for ( int i = 1, ie = nacceptors(daa,daav); i <= ie; ++i ) {
				if ( datom == accpt_pos(i,daa,daav) ) {
					located = true;
					if ( ( daa == na_cyt || daa == na_thy ) && datom == 14 ) {
						specificity = "PYR";
					} else if ( ( daa == na_rcy || daa == na_ura ) && datom == 15 ) {
						specificity = "PYR";
					} else if ( daa == na_ade && ( datom == 14 || datom == 19 ) ) {
						specificity = "PUR";
					} else if ( daa == na_rad && ( datom == 15 || datom == 20 ) ) {
						specificity = "PUR";
					} else if ( daa == na_gua && ( datom == 15 || datom == 20 ) ) {
						specificity = "PUR";
					} else if ( daa == na_rgu && ( datom == 16 || datom == 21 ) ) {
						specificity = "PUR";
					} else {
						specificity = dname;
					}
				}
			}
		}

		if (!located) {
			std::cout << "Atom " << I (2,datom) << " in " << residue3(daa) <<
				" is neither a donor or acceptor in function dna_specificity\n";
		}
	}
	return specificity;

}
////////////////////////////////////////////////////////////////////////////////
// @begin screen_exploded_rotamers_dna
//
// @brief
// Filters rotamers for hydrogen bonding to the DNA bases in the major groove.
// Used with "rotamer_explosion" to allow oversampling of long polar
// sidechains.
//
// @references
// Ashworth J, Havranek J, Duarte C, Sussman D, Monnat R, Stoddard B, Baker D.
// Nature 441, 656-659 (2006)
//
// @authors
// duarte
// ashworth
//
////////////////////////////////////////////////////////////////////////////////

bool
screen_exploded_rotamers_dna(
	int const ppos,
	int const paa,
	int const paav,
	FArray1DB_float const & temp_chi,
	DesignMap const & design_map
)
{
	using namespace aaproperties_pack;
	using namespace misc;
	using namespace param; // MAX's
	using namespace param_aa; // is_NA
	using namespace param_pack;

	FArray2D_float rot_coord( 3, MAX_ATOM() );

	get_rot_coord( full_coord, paa, paav, temp_chi, rot_coord, ppos );

	float bumpenergy = 0.0;
	bump_check( paa, paav, ppos, rot_coord, full_coord, res, res_variant,
	 design_map, total_residue, bumpenergy );
	if ( bumpenergy > max_rot_bumpenergy ) return false;

	// check sidechain rotamer against all nucleic acid bases
	// distance cutoffs are applied in the downstream functions
	int daa, daav;
	for ( int dpos = 1; dpos <= total_residue; ++dpos ) {
		daa = res(dpos);
		if ( !is_NA(daa) ) continue;
		daav = res_variant(dpos);

		// thymine contact bonus
//	if ( daa == na_thy && thy_vdw_contact( paa, paav,	full_coord(1,1,dpos),
//		rot_coord ) ) return true;

		if ( !design::active_rotamer_options.dna_rotamers_exist ) {
			float sc_base_E( base_specific_hbond( daa, daav, paa, paav,
			 full_coord(1,1,dpos), rot_coord ) );

//			std::cout << "hbE " << F(6,2,sc_base_E) << std::endl;

			// the rotamer just needs to pass once
			if ( sc_base_E < design::active_rotamer_options.explode_cutoff ) {
				return true;
				// std::cout might not work, might need to overload
//			print_residue_coords( std::cout, ppos, paa, paav, pcoord, sc_base_E );
//			print_residue_coords( std::cout, dpos, daa, daav, dcoord, sc_base_E );
			}
		// multiple DNA identities, need to check against each
		} else {
				// DNA rotamers are (probably) not built yet
			for ( int na = 1; na <= MAX_AA(); ++na ) {
				if ( !is_DNA(na) ) continue;

				FArray2D_float temp_dna( 3, MAX_ATOM() );

				// align this base type (temp_dna) to fullcoord
				// note: this function does not support RNA yet
				align_dna_base( na, 1, temp_dna, daa, daav, full_coord(1,1,dpos) );

				// check for hbonds between rotamer and temp_dna
				float sc_base_E = base_specific_hbond( daa, daav, paa, paav,
				 temp_dna, rot_coord );

//			std::cout << "hbE " << F(6,2,sc_base_E) << std::endl;

				// the rotamer just needs to pass once
				if ( sc_base_E < design::active_rotamer_options.explode_cutoff ) {
					return true;
					// std::cout might not work, might need to overload
//				print_residue_coords( std::cout, ppos, paa, paav, pcoord, sc_base_E );
//				print_residue_coords( std::cout, dpos, daa, daav, dcoord, sc_base_E );
				}
			}
		}
	} // dna loop
	return false;
}

////////////////////////////////////////////////////////////////////////////////
// @begin align_dna_base
//
// @brief
// aligns any dna base onto any another
// takes backbone atoms directly from reference
// identity of aligned base depends on 'mob_na' (supports mutation)
//
// @authors
// ashworth
//
// @references
// code in pose.cc : copy_sidechain()
//
////////////////////////////////////////////////////////////////////////////////
void
align_dna_base(
	int const mob_na,
	int const mob_v,
	FArray2Da_float mob_xyz,
	int const ref_na,
	int const ref_v,
	FArray2Da_float ref_xyz
)
{
	using namespace aaproperties_pack; // na_anchor, icoor
	using namespace param; // MAX's
	using namespace param_aa; // natoms, is_NA

	mob_xyz.dimension( 3, MAX_ATOM() );
	ref_xyz.dimension( 3, MAX_ATOM() );


	// fill mobile coords from nucleic acid template
	int at_e = natoms(mob_na,mob_v);
	for ( int at = 1; at <= at_e; ++at ) {
		for ( int dim = 1; dim <= 3; ++dim ) {
			mob_xyz(dim,at) = icoor(dim,at,mob_na,mob_v);
		}
	}

	int const
	 mob_c1star( na_anchor( 1, mob_na, mob_v ) ),
	   mob_n1n9( na_anchor( 2, mob_na, mob_v ) ),
	  mob_adj_c( na_anchor( 3, mob_na, mob_v ) ),
	 ref_c1star( na_anchor( 1, ref_na, ref_v ) ),
	   ref_n1n9( na_anchor( 2, ref_na, ref_v ) ),
	  ref_adj_c( na_anchor( 3, ref_na, ref_v ) );

/*
// old way

	FArray2D_float mat(3,3);
	FArray1D_float vec(3);

	// superimpose c1*->n1/n9 bond
	lineup_bk( mob_xyz(1,mob_c1star), mob_xyz(1,mob_n1n9), ref_xyz(1,ref_c1star),
	 ref_xyz(1,ref_n1n9), mat, vec );
	for ( int i = 1; i <= at_e; ++i ) move_bk( mob_xyz(1,i), mat, vec );

	// align adjacent ring carbon along c1*->n1/n9 dihedral
	align_bk( mob_xyz(1,mob_c1star), mob_xyz(1,mob_n1n9), mob_xyz(1,mob_adj_c),
	 ref_xyz(1,ref_adj_c), mat, vec );msoB_g_0001.pdb
	for ( int i = 1; i <= at_e; ++i ) move_bk( mob_xyz(1,i), mat, vec );
*/

// new way

	// translation/rotation matrix
	FArray2D_float mat(4,4);

	get_GL_matrix(
	 mob_xyz( 1, mob_adj_c ), mob_xyz( 1, mob_c1star ), mob_xyz( 1, mob_n1n9 ),
	 ref_xyz( 1, ref_adj_c ), ref_xyz( 1, ref_c1star ), ref_xyz( 1, ref_n1n9 ),
	 mat );

	// transform to superimpose base root
	for ( int at = 1; at <= at_e; ++at ) GL_rot_in_place( mat, mob_xyz(1,at) );

	// recover backbone and sugar from original coords
	for ( int at = 1; at <= at_e; ++at ) {

		if ( !is_NA_backbone_atom(at,mob_na,mob_v) ) continue;

		int ref_at = LookupByName( ref_na, ref_v, atom_name(at,mob_na,mob_v) );

		for ( int dim = 1; dim <= 3; ++dim ) mob_xyz(dim,at) = ref_xyz(dim,ref_at);
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin func_dna
//
// @brief
// THIS FUNCTION IS CURRENTLY DISEMBOWELED.  Do not attempt to use. ja 2007
//
// for prot-DNA minimizer - main function to compute structure energy given
// (packed) dihedral angles.
//
// @remarks
// NOTE: the args of this function are as required by Rosetta minimizers
// NOTE2: powell minimizer gets confused by exactly degenerate energies which
// may occur if the energy does not depend on some dihedral angles (for
// example, changing delta for the last base in a dna chain only changes the
// position of O3*, which might not change total energy at all, or only by a
// very small amount) Therefore, we introduce a small energy penalty below,
// designed to break such unwanted degeneracies.
//
// @authors
// Alex Morozov
//
////////////////////////////////////////////////////////////////////////////////
float
func_dna(
	FArray1Da_float active_angles,
	bool & gfrag
)
{
	using namespace dna_variables;

	if ( !gfrag ) return 0.0;

	float cur_energy = 0.0;

// powell minimizer cannot deal with completely degenerate landscapes, so add a
// very small penalty in case the energy stays the same as a function of a
// certain dih_angle. Such a penalty will discourage large deviations of
// degenerate angles
	float degeneracy_penalty = 1e-05;

	active_angles(1) = active_angles(1); // hide uninit from compiler for now
//	unpack_angles( active_angles, nangles ); // deprecated
//	rebuild_coords(); // deprecated
// 	cur_energy = compute_energy(); // deprecated

	if (std::abs(cur_energy - total_minim_energy)
		< 1e-04 && chiral_deviation > 0) { // degenerate energy detected
		//std::cout << "chi_dev = " << chiral_deviation << std::endl;
		total_minim_energy = cur_energy;
		cur_energy += degeneracy_penalty * chiral_deviation;
	} else {
		total_minim_energy = cur_energy;
	}

	//std::cout << "Current energy in func_dna = " << cur_energy << std::endl;
	return cur_energy;
}
////////////////////////////////////////////////////////////////////////////////
// @begin basepair_rotamers
//
// @brief
// reference vector for rotamer basepairing partners
// WARNING: the 'dna' class referred to here is a deprecated global!
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
std::vector< std::vector<int> > basepair_rotamers(
	RotamerSet const & set
)
{
	using namespace param_aa;
	using namespace dna_variables;

	int rots = set.nrotamers();
	std::vector< std::vector<int> > basepair_rots( rots, std::vector<int>(0) );

	// runs fairly quickly despite nrotamers * nrotamers scaling
	for ( int rot(1); rot <= rots; ++rot ) {
		int aa( set.report_aa( rot ) );
		if ( !is_DNA(aa) ) continue;
		int pos( set.report_seqpos( rot ) );

		for ( int rot2(1); rot2 <= rots; ++rot2 ) {
			int aa2( set.report_aa( rot2 ) );
			if ( !is_DNA(aa2) ) continue;
			int pos2( set.report_seqpos( rot2 ) );

			DnaPosInfo & bp( basepairs[pos] ); // maps to the basepair at pos1
			if ( pos2 != bp.rvspos() || aa2 != bp.rvstype() ) continue;

			(basepair_rots.at(rot-1)).push_back( rot2 );

//			std::cout << "Rotamer " << rot << "," << pos << "," << aa <<
//			 " has correlate " << rot2 << "," << pos2 << "," << aa2 <<
//			 std::endl;
		}
	}

/*
	// debug: print vector contents
	for ( int i = 0; size_t(i) < basepair_rots.size(); ++i ) {
		std::cout << I(4,i) << ":";
		for ( std::vector<int>::iterator inner( basepair_rots[i].begin() );
		 inner != basepair_rots[i].end(); ++inner ) {
			std::cout << I(4,(*inner));
		}
		std::cout << "; ";
	}
	std::cout << " END" << std::endl;
	// end debug
*/

	return basepair_rots;
}

////////////////////////////////////////////////////////////////////////////////
// @begin set_DNA_rotamer_ratio
//
// @brief
// If DNA rotamers are present (and will be packed by simulated annealing), it is useful to overpopulate a rot_to_pack vector with DNA rotamers to increase the probability of changing DNA rotamers during packing.  Otherwise, the chance of making a DNA substitutions may be too, as there are not many DNA 'rotamers' to begin with.
//
// @authors
// ashworth
//
////////////////////////////////////////////////////////////////////////////////
std::vector<int>
set_DNA_rotamer_ratio(
	RotamerSet const & set,
	float const target_ratio
)
{
	int const rots( set.nrotamers() );
	int dna_rots(0);
	for ( int rot = 1; rot <= rots; ++rot ) {
		if ( param_aa::is_DNA(set.report_aa(rot)) ) ++dna_rots;
	}
	int extra = int( target_ratio * rots / dna_rots - 1 );

	// debug
	std::cout << "Adding " << extra << " extra 'copies' of each DNA rotamer " <<
	 "in order to increase DNA rotamer sampling during simulated annealing." <<
	  std::endl;

	std::vector<int> rots_to_pack;

	for ( int rot = 1; rot <= rots; ++rot ) {
		rots_to_pack.push_back( rot );
		if ( !param_aa::is_DNA(set.report_aa(rot)) ) continue;
		for ( int add = 0; add < extra; ++add ) rots_to_pack.push_back( rot );
	}
	return rots_to_pack;
}

////////////////////////////////////////////////////////////////////////////////
// @begin na_partner
//
// @brief
// returns basepairing partner residue type index number
// stolen from pose_dna.cc because it is so general (and useful)
//
// @authors
// jim or phil
//
////////////////////////////////////////////////////////////////////////////////

int
na_partner( int const na ) {

	using namespace param_aa;
	if ( na == na_ade ) return na_thy;
	else if ( na == na_cyt ) return na_gua;
	else if ( na == na_gua ) return na_cyt;
	else if ( na == na_thy ) return na_ade;

	std::cout << "Unknown na: " << na << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	return 0;

}

////////////////////////////////////////////////////////////////////////////////
// @begin find_basepairs
//
// @brief
// stolen from pose_dna.cc
//
// @authors
// Phil Bradley
///////////////////////////////////////////////////////////////////////////////
void
find_basepairs(
	pose_ns::Pose const & pose,
	DnaSeqInfo & dna,
	bool const include_unpaired
)
{
	using namespace param_aa;
	using numeric::xyzVector_float;

	float const max_d( 4.0 );

	int const nres( pose.total_residue() );

	dna.clear();

	assert( dna.empty() );
	std::map< int, int > base_partner;
	base_partner[ na_ade ] = na_thy;
	base_partner[ na_thy ] = na_ade;
	base_partner[ na_gua ] = na_cyt;
	base_partner[ na_cyt ] = na_gua;

std::map< int, int > hbond_atom;
	hbond_atom[ na_ade ] = dna_variables::n1;
	hbond_atom[ na_thy ] = dna_variables::n3T;
	hbond_atom[ na_gua ] = dna_variables::n1;
	hbond_atom[ na_cyt ] = dna_variables::n3C;

	//ja adding RNA support
	base_partner[ na_rad ] = na_ura;
	base_partner[ na_ura ] = na_rad;
	base_partner[ na_rgu ] = na_rcy;
	base_partner[ na_rcy ] = na_rgu;

	hbond_atom[ na_rad ] = dna_variables::rna_n1;
	hbond_atom[ na_ura ] = dna_variables::rna_n3pyr;
	hbond_atom[ na_rgu ] = dna_variables::rna_n1;
	hbond_atom[ na_rcy ] = dna_variables::rna_n3pyr;

	std::map< int, int > partner; // temporary

	for ( int i(1); i <= nres; ++i ) {
		int const i_aa( pose.res(i) );
		if ( !is_NA( i_aa ) ) continue;
		// already represented, by virtue of previously occurring basepair partner
		if ( dna.contains(i) ) continue;
		// hbond atom, base y-axis
		xyzVector_float const
			i_xyz( &(pose.full_coord()( 1, hbond_atom[i_aa], i )) ),
			i_axis( get_y_axis( i_aa, 1 /*strand*/, pose.full_coord()(1,1,i) ) );

		bool paired( false );
		// check for a basepairing partner
		float best_d(1000.0);
		for ( int j(i+1); j <= nres; ++j ) {
			int const j_aa( pose.res(j) );
			if ( !is_NA( j_aa ) ) continue;

			// -distance check-
			xyzVector_float const j_xyz( &(pose.full_coord()(1,hbond_atom[j_aa],j)) );
			float d( distance( i_xyz, j_xyz ) );
			if ( d >= max_d ) continue;

			// -geometry check-
			xyzVector_float const
				j_axis( get_y_axis( j_aa, 2 /*strand*/, pose.full_coord()(1,1,j) ) ),
			  u( ( i_xyz - j_xyz ).normalized() );

			float const dot1( dot( i_axis, j_axis ) ),
			            dot2( dot( i_axis, u ) ),
			            dot3( dot( j_axis, u ) );
			d -= dot1 + dot2 + dot3;
			if ( d > best_d || dot1 < 0.75 || dot2 < 0.75 || dot3 < 0.75 ) continue;

			// -complementarity check-
			if ( j_aa != base_partner[ i_aa ] ) {
				std::cerr << "Warning: nucleic acids " << aa_name3(i_aa) << " " <<
				 pdb::pdb_res_num(i) << " and " << aa_name3(j_aa) << " " <<
				 pdb::pdb_res_num(j) << " have basepaired geometry, but are not " <<
				 "complementary types" << std::endl;
				continue;
			}
			// passed all of the checks: we have a basepair
			best_d = d;
			dna.push_back( DnaPosInfo( i, i_aa, j, j_aa ) );
			paired = true; break;
		} // end j loop
		if ( paired || !include_unpaired ) continue;
		// include unpaired dna (is tagged as such)
		dna.push_back( DnaPosInfo( i, i_aa ) );
	} // end i loop
	dna.print( std::cout );
}

////////////////////////////////////////////////////////////////////////////////
// @begin is_NA_backbone_atom
//
// @brief
// stolen from pose_dna.cc
//
// @authors
// Phil Bradley
////////////////////////////////////////////////////////////////////////////////

bool
is_NA_backbone_atom(
	int const atomno,
	int const aa,
	int const aav
)
{
	using aaproperties_pack::atom_name;
	return ( atom_name(atomno, aa, aav)[1] == 'P' ||
					 atom_name(atomno, aa, aav)[3] == 'P' ||
					 atom_name(atomno, aa, aav)[3] == '*' );
}

////////////////////////////////////////////////////////////////////////////////
// @begin get_y_axis
//
// @brief
// stolen from pose_dna.cc
//
// @authors
// Phil Bradley
////////////////////////////////////////////////////////////////////////////////
numeric::xyzVector_float
get_y_axis(
	int const aa,
	int const strand,
	FArray2Da_float coords
)
{
	coords.dimension( 3, param::MAX_ATOM()() );

	int a1(0),a2(0);
	get_y_axis_atoms( aa, strand, a1, a2 );

	numeric::xyzVector_float y_axis, v1( &coords(1,a1)), v2( &coords(1,a2) );
	y_axis = v2 - v1;
	y_axis.normalize();
	return y_axis;
}

////////////////////////////////////////////////////////////////////////////////
// @begin get_y_axis_atoms
//
// @brief
// stolen from pose_dna.cc
//
// @authors
// Phil Bradley
////////////////////////////////////////////////////////////////////////////////

void
get_y_axis_atoms(
								 int const aa,
								 int const strand, // 1 or 2
								 int & a1,
								 int & a2
								 )
{
	using namespace param_aa;
	using namespace dna_variables;
	if ( aa == na_ade ) {
		if (strand==1) {
			a1 = n1;
			a2 = c4A;
		} else {
			a1 = c4A;
			a2 = n1;
		}
	} else if ( aa == na_gua ) {
		if (strand==1) {
			a1 = n1;
			a2 = c4G;
		} else {
			a1 = c4G;
			a2 = n1;
		}
	} else if ( aa == na_cyt ) {
		if (strand==1) {
			a1 = n3C;
			a2 = c6C;
		} else {
			a1 = c6C;
			a2 = n3C;
		}
	} else if ( aa == na_thy ) {
		if (strand==1) {
			a1 = n3T;
			a2 = c6T;
		} else {
			a1 = c6T;
			a2 = n3T;
		}

	//ja RNA
	} else if ( aa == na_rad ) {
		if (strand==1) {
			a1 = rna_n1;
			a2 = rna_c4rA;
		} else {
			a1 = rna_c4rA;
			a2 = rna_n1;
		}
	} else if ( aa == na_rgu ) {
		if (strand==1) {
			a1 = rna_n1;
			a2 = rna_c4rG;
		} else {
			a1 = rna_c4rG;
			a2 = rna_n1;
		}
	} else if ( aa == na_rcy ) {
		if (strand==1) {
			a1 = rna_n3pyr;
			a2 = rna_c6rC;
		} else {
			a1 = rna_c6rC;
			a2 = rna_n3pyr;
		}
	} else if ( aa == na_ura ) {
		if (strand==1) {
			a1 = rna_n3pyr;
			a2 = rna_c6rU;
		} else {
			a1 = rna_c6rU;
			a2 = rna_n3pyr;
		}

	} else {
		std::cout << aa_name3(aa) << " is unknown to me, tovarisch!\n";//sic
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin init_triad_coords
//
// @brief
// for DNA - sets ideal coordinates of base heavy atoms + C1'
//
// @detailed
// The ideal coordinates are from Wilma Olson's 3DNA (NDB96_?.pdb files).
//
// @remarks
// Atom indices here are the same as in icoor, so these arrays are "sparse" (not 1 .. natoms).  Templates are not reset since this is only a subset of DNA atoms, used strictly for DNA effective energy calcs
//
// @authors
// Alex Morozov
////////////////////////////////////////////////////////////////////////////////
void
init_triad_coords()
{
	using namespace dna_variables;
	using namespace param;

	// Order of bases in the triad_icoor array:
	int const na_ade = { 1 };
	int const na_cyt = { 2 };
	int const na_gua = { 3 };
	int const na_thy = { 4 };

	// Init. the triad_icoor array:
	triad_icoor = static_cast< float >( -1 );

	// A
	triad_icoor( 1,11, na_ade) =   -2.479; //   C1*
	triad_icoor( 2,11, na_ade) =    5.346; //   C1*
	triad_icoor( 3,11, na_ade) =    0.000; //   C1*
	triad_icoor( 1,12, na_ade) =   -0.668; //   N1
	triad_icoor( 2,12, na_ade) =    0.532; //   N1
	triad_icoor( 3,12, na_ade) =    0.000; //   N1
	triad_icoor( 1,13, na_ade) =   -1.912; //   C2
	triad_icoor( 2,13, na_ade) =    1.023; //   C2
	triad_icoor( 3,13, na_ade) =    0.000; //   C2
	triad_icoor( 1,14, na_ade) =   -2.320; //   N3
	triad_icoor( 2,14, na_ade) =    2.290; //   N3
	triad_icoor( 3,14, na_ade) =    0.000; //   N3
	triad_icoor( 1,15, na_ade) =   -1.267; //   C4
	triad_icoor( 2,15, na_ade) =    3.124; //   C4
	triad_icoor( 3,15, na_ade) =    0.000; //   C4
	triad_icoor( 1,16, na_ade) =    0.071; //   C5
	triad_icoor( 2,16, na_ade) =    2.771; //   C5
	triad_icoor( 3,16, na_ade) =    0.000; //   C5
	triad_icoor( 1,17, na_ade) =    0.369; //   C6
	triad_icoor( 2,17, na_ade) =    1.398; //   C6
	triad_icoor( 3,17, na_ade) =    0.000; //   C6
	triad_icoor( 1,18, na_ade) =    1.611; //   N6
	triad_icoor( 2,18, na_ade) =    0.909; //   N6
	triad_icoor( 3,18, na_ade) =    0.000; //   N6
	triad_icoor( 1,19, na_ade) =    0.877; //   N7
	triad_icoor( 2,19, na_ade) =    3.902; //   N7
	triad_icoor( 3,19, na_ade) =    0.000; //   N7
	triad_icoor( 1,20, na_ade) =    0.024; //   C8
	triad_icoor( 2,20, na_ade) =    4.897; //   C8
	triad_icoor( 3,20, na_ade) =    0.000; //   C8
	triad_icoor( 1,21, na_ade) =   -1.291; //   N9
	triad_icoor( 2,21, na_ade) =    4.498; //   N9
	triad_icoor( 3,21, na_ade) =    0.000; //   N9
	triad_icoor_natoms( 1 ) = 11;

	// C
	triad_icoor( 1,11, na_cyt) =   -2.477; //   C1*
	triad_icoor( 2,11, na_cyt) =    5.402; //   C1*
	triad_icoor( 3,11, na_cyt) =    0.000; //   C1*
	triad_icoor( 1,12, na_cyt) =   -1.285; //   N1
	triad_icoor( 2,12, na_cyt) =    4.542; //   N1
	triad_icoor( 3,12, na_cyt) =    0.000; //   N1
	triad_icoor( 1,13, na_cyt) =   -1.472; //   C2
	triad_icoor( 2,13, na_cyt) =    3.158; //   C2
	triad_icoor( 3,13, na_cyt) =    0.000; //   C2
	triad_icoor( 1,14, na_cyt) =   -2.628; //   O2
	triad_icoor( 2,14, na_cyt) =    2.709; //   O2
	triad_icoor( 3,14, na_cyt) =    0.001; //   O2
	triad_icoor( 1,15, na_cyt) =   -0.391; //   N3
	triad_icoor( 2,15, na_cyt) =    2.344; //   N3
	triad_icoor( 3,15, na_cyt) =    0.000; //   N3
	triad_icoor( 1,16, na_cyt) =    0.837; //   C4
	triad_icoor( 2,16, na_cyt) =    2.868; //   C4
	triad_icoor( 3,16, na_cyt) =    0.000; //   C4
	triad_icoor( 1,17, na_cyt) =    1.875; //   N4
	triad_icoor( 2,17, na_cyt) =    2.027; //   N4
	triad_icoor( 3,17, na_cyt) =    0.001; //   N4
	triad_icoor( 1,18, na_cyt) =    1.056; //   C5
	triad_icoor( 2,18, na_cyt) =    4.275; //   C5
	triad_icoor( 3,18, na_cyt) =    0.000; //   C5
	triad_icoor( 1,19, na_cyt) =   -0.023; //   C6
	triad_icoor( 2,19, na_cyt) =    5.068; //   C6
	triad_icoor( 3,19, na_cyt) =    0.000; //   C6
	triad_icoor_natoms( 2 ) = 9;

	// G
	triad_icoor( 1,11, na_gua) =   -2.477; //   C1*
	triad_icoor( 2,11, na_gua) =    5.399; //   C1*
	triad_icoor( 3,11, na_gua) =    0.000; //   C1*
	triad_icoor( 1,12, na_gua) =   -0.700; //   N1
	triad_icoor( 2,12, na_gua) =    0.641; //   N1
	triad_icoor( 3,12, na_gua) =    0.000; //   N1
	triad_icoor( 1,13, na_gua) =   -1.999; //   C2
	triad_icoor( 2,13, na_gua) =    1.087; //   C2
	triad_icoor( 3,13, na_gua) =    0.000; //   C2
	triad_icoor( 1,14, na_gua) =   -2.949; //   N2
	triad_icoor( 2,14, na_gua) =    0.139; //   N2
	triad_icoor( 3,14, na_gua) =   -0.001; //   N2
	triad_icoor( 1,15, na_gua) =   -2.342; //   N3
	triad_icoor( 2,15, na_gua) =    2.364; //   N3
	triad_icoor( 3,15, na_gua) =    0.001; //   N3
	triad_icoor( 1,16, na_gua) =   -1.265; //   C4
	triad_icoor( 2,16, na_gua) =    3.177; //   C4
	triad_icoor( 3,16, na_gua) =    0.000; //   C4
	triad_icoor( 1,17, na_gua) =    0.071; //   C5
	triad_icoor( 2,17, na_gua) =    2.833; //   C5
	triad_icoor( 3,17, na_gua) =    0.000; //   C5
	triad_icoor( 1,18, na_gua) =    0.424; //   C6
	triad_icoor( 2,18, na_gua) =    1.460; //   C6
	triad_icoor( 3,18, na_gua) =    0.000; //   C6
	triad_icoor( 1,19, na_gua) =    1.554; //   O6
	triad_icoor( 2,19, na_gua) =    0.955; //   O6
	triad_icoor( 3,19, na_gua) =    0.000; //   O6
	triad_icoor( 1,20, na_gua) =    0.870; //   N7
	triad_icoor( 2,20, na_gua) =    3.969; //   N7
	triad_icoor( 3,20, na_gua) =    0.000; //   N7
	triad_icoor( 1,21, na_gua) =    0.023; //   C8
	triad_icoor( 2,21, na_gua) =    4.962; //   C8
	triad_icoor( 3,21, na_gua) =    0.000; //   C8
	triad_icoor( 1,22, na_gua) =   -1.289; //   N9
	triad_icoor( 2,22, na_gua) =    4.551; //   N9
	triad_icoor( 3,22, na_gua) =    0.000; //   N9
	triad_icoor_natoms( 3 ) = 12;

	// T
	triad_icoor( 1,11, na_thy) =   -2.481; //   C1*
	triad_icoor( 2,11, na_thy) =    5.354; //   C1*
	triad_icoor( 3,11, na_thy) =    0.000; //   C1*
	triad_icoor( 1,12, na_thy) =   -1.284; //   N1
	triad_icoor( 2,12, na_thy) =    4.500; //   N1
	triad_icoor( 3,12, na_thy) =    0.000; //   N1
	triad_icoor( 1,13, na_thy) =   -1.462; //   C2
	triad_icoor( 2,13, na_thy) =    3.135; //   C2
	triad_icoor( 3,13, na_thy) =    0.000; //   C2
	triad_icoor( 1,14, na_thy) =   -2.562; //   O2
	triad_icoor( 2,14, na_thy) =    2.608; //   O2
	triad_icoor( 3,14, na_thy) =    0.000; //   O2
	triad_icoor( 1,15, na_thy) =   -0.298; //   N3
	triad_icoor( 2,15, na_thy) =    2.407; //   N3
	triad_icoor( 3,15, na_thy) =    0.000; //   N3
	triad_icoor( 1,16, na_thy) =    0.994; //   C4
	triad_icoor( 2,16, na_thy) =    2.897; //   C4
	triad_icoor( 3,16, na_thy) =    0.000; //   C4
	triad_icoor( 1,17, na_thy) =    1.944; //   O4
	triad_icoor( 2,17, na_thy) =    2.119; //   O4
	triad_icoor( 3,17, na_thy) =    0.000; //   O4
	triad_icoor( 1,18, na_thy) =    1.106; //   C5
	triad_icoor( 2,18, na_thy) =    4.338; //   C5
	triad_icoor( 3,18, na_thy) =    0.000; //   C5
	triad_icoor( 1,19, na_thy) =    2.466; //   C5M
	triad_icoor( 2,19, na_thy) =    4.961; //   C5M
	triad_icoor( 3,19, na_thy) =    0.001; //   C5M
	triad_icoor( 1,20, na_thy) =   -0.024; //   C6
	triad_icoor( 2,20, na_thy) =    5.057; //   C6
	triad_icoor( 3,20, na_thy) =    0.000; //   C6
	triad_icoor_natoms( 4 ) = 10;

}

////////////////////////////////////////////////////////////////////////////////
// @begin LoadDNAParams
//
// @brief
// This is the main function to call in order to load all DNA parameters
// (means, stddevs and force constants) from 3 rosetta_db files
//
// @authors
// Alex Morozov
////////////////////////////////////////////////////////////////////////////////
void
LoadDNAParams()
{
	using namespace dna_variables;
	//using namespace std;

	std::string const means_bs_file( "Equil_AM.mean.dat" );
	std::string const stddevs_bs_file( "Equil_AM.stddev.dat" );
	std::string const means_bp_file( "Equil_bp_AM.mean.dat" );
	std::string const stddevs_bp_file( "Equil_bp_AM.stddev.dat" );
	std::string const Fij_file( "Fij_AM.dat" );
	std::string const Gij_file( "Fij_bp_AM.dat" );

	// means of DNA basestep effective params
	ReadMomentsTable( means_bs_file, means_bs );
	// stddevs of DNA basestep effective params
	ReadMomentsTable( stddevs_bs_file, stddevs_bs );
	// means of DNA basepair effective params
	ReadMomentsTable( means_bp_file, means_bp );
	// stddevs of DNA basepair effective params
	ReadMomentsTable( stddevs_bp_file, stddevs_bp );
	// Fij (effective basestep force constants) for DNA potential
	ReadFijTable( Fij_file, Fij );
	// Gij (effective basepair force constants) for DNA potential
	ReadGijTable( Gij_file, Gij );
}

////////////////////////////////////////////////////////////////////////////////
// @begin get_basepair_index
//
// @brief
// aux. function - basepair index lookup by base names (ie 'A' => 1 etc.)
//
// @remarks
// WARNING: the index lookup ASSUMES a certain order of storage for basepair
// params and Gij. NEVER change it lightly, index will be screwed up!!
//
// @authors
// Alex Morozov
//
////////////////////////////////////////////////////////////////////////////////
int get_basepair_index( char const l )
{
	if      ( l == 'A' ) return 1;
	else if ( l == 'C' ) return 2;
	else if ( l == 'G' ) return 3;
	else if ( l == 'T' ) return 4;
	else {
		std::cerr << "Unknown DNA base label: " << l << "!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return -1;
}

////////////////////////////////////////////////////////////////////////////////
// @begin get_basestep_index
//
// @brief
// aux. function - basestep index lookup by base names (ie 'AA' => 1 etc.)
//
// @remarks
// WARNING: the index lookup ASSUMES a certain order of storage for basestep params and Fij.
// NEVER change it lightly, index will be screwed up!!
//
// @authors
// Alex Morozov
////////////////////////////////////////////////////////////////////////////////
int
get_basestep_index(
	std::string const & l1,
	std::string const & l2
)
{

	int ind = 0;

	if ( l1 == "A" ) {
		ind = 0;
	} else if ( l1 == "C" ) {
		ind = 4;
	} else if ( l1 == "G" ) {
		ind = 8;
	} else if ( l1 == "T" ) {
		ind = 12;
	} else {
		std::cerr << "Unknown first DNA base label: " << l1 << "!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	if ( l2 == "A" ) {
		return ( ind + 1 );
	} else if ( l2 == "C" ) {
		return ( ind + 2 );
	} else if ( l2 == "G" ) {
		return ( ind + 3 );
	} else if ( l2 == "T" ) {
		return ( ind + 4 );
	} else {
		std::cerr << "Unknown second DNA base label: " << l2 << "!\nExiting ..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	return -1;
}

////////////////////////////////////////////////////////////////////////////////
// @begin ReadMomentsTable
//
// @brief
// read in a rosetta_db file with means and stddevs data stored in a certain format (see file for details)
//
// @authors
// Alex Morozov
////////////////////////////////////////////////////////////////////////////////
void
ReadMomentsTable(
	std::string const & filename,
	FArray2D_float & table
)
{

	utility::io::izstream & file( open_data_file( filename ) );
	char buffer[512];
	char basestep_name[3];
	int datapoints;

	if ( file().fail() ) {
		std::cerr << "Could not open file: " << file.filename() << " for reading!\nExiting..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	int cnt = 0; // basestep counter
	while ( !file().eof() ) {
		file().getline( buffer, 512, '\n' );
		if ( file().fail() && !file().eof() ) file.clear();

		if ( file().eof() ) continue;

		if ( cnt == 0 ) {
			++cnt;
			continue; // skip header
		}
		if ( sscanf(buffer,"%2s %d %f %f %f %f %f %f",basestep_name,&datapoints,
		 &table(cnt,1),&table(cnt,2),&table(cnt,3),&table(cnt,4),&table(cnt,5),&table(cnt,6) ) == 8 ) {
			++cnt;
		}
	}
	file.close();
	file.clear();
	if ( cnt != table.u1() + 1 ) {
		std::cerr << "Read " << cnt << " lines instead of " << ( table.u1() + 1 ) << " in file " << file.filename() << "!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin ReadFijTable
//
// @brief
// read in a rosetta_db file with Fij (basestep force constants) data stored in a certain format (see file for details)
//
// @authors
// Alex Morozov
////////////////////////////////////////////////////////////////////////////////
void
ReadFijTable(
	std::string const & filename,
	FArray3D_float & table
)
{

	utility::io::izstream & file( open_data_file( filename ) );
	char buffer[512];
	char label[24];

	if ( file().fail() ) {
		std::cerr << "Could not open file: " << file.filename() << " for reading!\nExiting..\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	int cnt = 0; // basestep counter
	int prm_cnt = -1; // parameter counter
	while ( !file().eof() ) {
		file().getline( buffer, 512, '\n' );
		if ( file().fail() && !file().eof() ) file().clear();
		if ( file().eof() ) continue;
		if ( std::strlen(buffer) == 2 ) {
			prm_cnt = 1;
			++cnt;
		} else if ( std::sscanf(buffer,"%23s %f %f %f %f %f %f",label,
		 &table(cnt,1,prm_cnt),&table(cnt,2,prm_cnt),&table(cnt,3,prm_cnt),
		 &table(cnt,4,prm_cnt),&table(cnt,5,prm_cnt),&table(cnt,6,prm_cnt) ) == 7 ) {
			++prm_cnt;
		}
	}
	file.close();
	file.clear();
	if (cnt != 16) {
		std::cerr << "Read " << cnt << " lines instead of " << 16 << " in file " << file.filename() << "!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

////////////////////////////////////////////////////////////////////////////////
// @begin ReadGijTable
//
// @brief
// read in a rosetta_db file with Gij (basepair force constants) data stored in a certain format (see file for details)
//
// @authors
// Alex Morozov
////////////////////////////////////////////////////////////////////////////////
void
ReadGijTable(
	std::string const & filename,
	FArray3D_float & table
)
{

	utility::io::izstream & file( open_data_file( filename ) );
	char buffer[512];
	char label[24];

	if ( file().fail() ) {
		std::cerr << "Could not open file: " << file.filename() << " for reading!\nExiting..\n";
    utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
  }
	int cnt = 0; // basestep counter
	int prm_cnt = -1; // parameter counter
	while ( !file().eof() ) {
    file().getline(buffer, 512, '\n');
		if ( file().fail() && !file().eof() ) file().clear();
		if ( file().eof() ) continue;
		if ( std::strlen(buffer) == 1) {
			prm_cnt = 1;
			++cnt;
		} else if ( std::sscanf(buffer,"%23s %f %f %f %f %f %f",label,
		 &table(cnt,1,prm_cnt),&table(cnt,2,prm_cnt),&table(cnt,3,prm_cnt),
		 &table(cnt,4,prm_cnt),&table(cnt,5,prm_cnt),&table(cnt,6,prm_cnt) ) == 7 ) {
			++prm_cnt;
		}
	}
	file.close();
	file.clear();
	if (cnt != 4) {
		std::cerr << "Read " << cnt << " lines instead of " << 4 << " in file " << file.filename() << "!\n";
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
}

