// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15540 $
//  $Date: 2007-06-19 15:37:31 -0700 (Tue, 19 Jun 2007) $
//  $Author: chu $


// Globals for dna-protein motif design


// Rosetta Headers
#include "aa_name_conversion.h"
#include "dna_motifs_ns.h"
#include "aaproperties_pack.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "read_aaproperties.h"
#include "util_vector.h"
#include "pose.h"


// C++ Headers
#include <iostream>
#include <string>


JJH_Rotamer::JJH_Rotamer() :
  motif_id( -1 ),
  motif_pos( -1 ),
  base_pos( -1 ),
  base_pos2( -1 ),
  energy( 9999.0 ),
  rmsd( 9999.0 ),
  found( false )
{
  rperc_ = 0.0;
  aav_ = 0;
}

// allow copy constructor from Rotamer
JJH_Rotamer::JJH_Rotamer( const Rotamer& r ) :
  Rotamer( r ),
  motif_id( -1 ),
  motif_pos( -1 ),
  base_pos( -1 ),
  base_pos2( -1 ),
  energy( 9999.0 ),
  rmsd( 9999.0 ),
  found( false )
{
  rperc_ = 0.0;
  aav_ = 0;
}

namespace DNA_Motifs {
	float Motifs_close_enough;
	float Motifs_far_dist;
	float Motifs_bump_value;
	float motifs_match_weight;
	int Motifs_num_rots_required;

	int para_total;
	int para_me;

}

void
Matrix_From_Euler_angles(
	FArray1Da_float ang,
	FArray2Da_float mat )
{
	ang.dimension( 3 );
	mat.dimension( 3, 3 );

	float costheta = std::cos( ang( 3 ) );
	float cospsi = std::cos( ang( 2 ) );
	float cosphi = std::cos( ang( 1 ) );
	float sintheta = std::sin( ang( 3 ) );
	float sinpsi = std::sin( ang( 2 ) );
	float sinphi = std::sin( ang( 1 ) );

	mat( 1, 1 ) =  cospsi*cosphi - costheta*sinphi*sinpsi;
	mat( 2, 1 ) =  cospsi*sinphi + costheta*cosphi*sinpsi;
	mat( 3, 1 ) =  sinpsi*sintheta;

	mat( 1, 2 ) = -sinpsi*cosphi - costheta*sinphi*cospsi;
	mat( 2, 2 ) = -sinpsi*sinphi + costheta*cosphi*cospsi;
	mat( 3, 2 ) =  cospsi*sintheta;

	mat( 1, 3 ) =  sintheta*sinphi;
	mat( 2, 3 ) = -sintheta*cosphi;
	mat( 3, 3 ) =  costheta;
}


namespace DNA_Motifs {

int
na_from_str( std::string & str_in ) {

	using namespace param_aa;

	if ( str_in == "ADE" ) {
		return na_ade;
	} else if ( str_in == "CYT" ) {
		return na_cyt;
	} else if ( str_in == "GUA" ) {
		return na_gua;
	} else if ( str_in == "THY" ) {
		return na_thy;
	} else {
		std::cout << "Fatal error:  bad na type " << str_in << std::endl;
		std::exit( EXIT_FAILURE );
	}
	return 0;
}

void Rotate_Angle_About_Axis(
	int natoms,
	float angle,
	FArray1Da_float axis,
	FArray2Da_float coords )
{
using namespace param;

axis.dimension( 3 );
coords.dimension( 3, MAX_ATOM() );

FArray1D_float norm( 3 );
FArray1D_float perp1( 3 );
FArray1D_float perp2( 3 );

float mag = 0.0;
for ( int k = 1; k <= 3; ++k ) {
	mag += axis(k)*axis(k);
}
mag = std::sqrt( mag );
for ( int k = 1; k <= 3; ++k ) {
	norm(k) = axis(k)/mag;
}

// Try a vector to (1, 0, 0), then dot out axis
perp1 = 0.0;
perp1( 1 ) = 1.0;

mag = 0.0;
for ( int k = 1; k <= 3; ++k ) {
	mag += perp1( k ) * norm( k );
}
for ( int k = 1; k <= 3; ++k ) {
	perp1( k ) -= mag*norm( k );
}
// Normalize it
mag = 0.0;
for ( int k = 1; k <= 3; ++k ) {
	mag += perp1( k ) * perp1( k );
}
mag = std::sqrt( mag );
for ( int k = 1; k <= 3; ++k ) {
	perp1( k ) /= mag;
}

// Get perp2 by cross product
perp2(1) = norm(2)*perp1(3) - norm(3)*perp1(2);
perp2(2) = norm(3)*perp1(1) - norm(1)*perp1(3);
perp2(3) = norm(1)*perp1(2) - norm(2)*perp1(1);

float c_ang = std::cos( angle );
float s_ang = std::sin( angle );

for ( int i = 1; i <= natoms; ++i ) {

	float temp_z = 0.0;
	for ( int k = 1; k <= 3; ++k ) {
		temp_z += coords( k, i ) * norm( k );
	}

	float temp_x = 0.0;
	for ( int k = 1; k <= 3; ++k ) {
		temp_x += coords( k, i ) * perp1( k );
	}

	float temp_y = 0.0;
	for ( int k = 1; k <= 3; ++k ) {
		temp_y += coords( k, i ) * perp2( k );
	}


	// post rotation
	float new_x = temp_x*c_ang - temp_y*s_ang;
	float new_y = temp_x*s_ang + temp_y*c_ang;

	for ( int k = 1; k <= 3; ++k ) {
		coords( k, i ) = new_x * perp1( k ) + new_y * perp2( k ) + temp_z * norm( k );
	}

}


return;
}


void
Motif::GenericPlaceRotamer(
	int aa,
	int aav,
	FArray1Da_float ref1,
	FArray1Da_float ref2,
	FArray1Da_float ref3,
	FArray2Da_float aa_coords,
	FArray2Da_float act_coords
)
{

static FArray1D_float vec( 3 );
static FArray1D_float tmp( 3 );
static FArray1D_float tmp2( 3 );
static FArray1D_float tmp3( 3 );
static FArray1D_float origin( 3 );
static FArray2D_float mat( 3, 3 );
static FArray1D_float oldx( 3 );
static FArray1D_float oldy( 3 );
static FArray1D_float oldz( 3 );

using namespace param;
using namespace aaproperties_pack;

ref1.dimension( 3 );
ref2.dimension( 3 );
ref3.dimension( 3 );
aa_coords.dimension( 3, MAX_ATOM() );
act_coords.dimension( 3, 1 );

// Get the coordinate axes for the na frame
for ( int k = 1; k <= 3; ++k ) {
	oldx( k ) = ref1( k ) - ref2( k );
}
vector_normalize( oldx );

float mag = 0.0;
for ( int k = 1; k <= 3; ++k ) {
	oldy( k ) = ref3( k ) - ref2( k );
	mag += oldy( k ) * oldx( k );
}
for ( int k = 1; k <= 3; ++k ) {
	oldy( k ) -= mag*oldx( k );
}
vector_normalize( oldy );

cross_bk( oldx, oldy, oldz );

// Figure out the transformation to place the aa axes on the na axes

// Note:  order different from (eg) place_atom() since I want
// the 2nd atoms to be coincident

// Superimpose origins and line up x axes
lineup_bk( aa_coords(1, aa_atm2), aa_coords(1, aa_atm1),
						ref2( 1 ), ref1( 1 ), mat, vec );
for ( int atm = 1, eatm = natoms( aa, aav ); atm <= eatm; ++atm )
	move_bk( aa_coords( 1, atm ), mat, vec );
move_bk( act_coords( 1, 1 ), mat, vec );

// Rotate aa y axis to na x-y plane
align_bk( aa_coords( 1, aa_atm2 ), aa_coords( 1, aa_atm1 ),
					aa_coords( 1, aa_atm3 ), ref3( 1 ), mat, vec );
for ( int atm = 1, eatm = natoms( aa, aav ); atm <= eatm; ++atm )
	move_bk( aa_coords( 1, atm ), mat, vec );
move_bk( act_coords( 1, 1 ), mat, vec );

// Subtract out the new origin

origin(1) = aa_coords( 1, aa_atm2 );
origin(2) = aa_coords( 2, aa_atm2 );
origin(3) = aa_coords( 3, aa_atm2 );
for ( int atm = 1, eatm = natoms( aa, aav ); atm <= eatm; ++atm )
	for ( int k = 1; k <= 3; ++k )
		aa_coords( k, atm ) -= origin( k );
for( int k = 1 ; k <= 3 ; ++k )
	act_coords( k, 1 ) -= origin( k );

vec( 1 ) = ( trans(1)*oldx(1) + trans(2)*oldy(1) + trans(3)*oldz(1) );
vec( 2 ) = ( trans(1)*oldx(2) + trans(2)*oldy(2) + trans(3)*oldz(2) );
vec( 3 ) = ( trans(1)*oldx(3) + trans(2)*oldy(3) + trans(3)*oldz(3) );

// Apply the motif specific transformation to the amino acid
//for ( int atm = 1, eatm = natoms( aa, aav ); atm <= eatm; ++atm )
//	move_bk( aa_coords( 1, atm ), rotate, vec );

Rotate_Angle_About_Axis( natoms(aa, aav), euler( 1 ), oldz, aa_coords );
Rotate_Angle_About_Axis( 1, euler( 1 ), oldz, act_coords );

// Make intermediate axis
for ( int k = 1; k <= 3; ++k )
	tmp( k ) = aa_coords( k, aa_atm1 ) - aa_coords( k, aa_atm2 );

// Normalize
vector_normalize( tmp );

Rotate_Angle_About_Axis( natoms(aa, aav), euler( 3 ), tmp, aa_coords );
Rotate_Angle_About_Axis( 1, euler( 3 ), tmp, act_coords );

// Make intermediate axis
for ( int k = 1; k <= 3; ++k )
	tmp( k ) = aa_coords( k, aa_atm1 ) - aa_coords( k, aa_atm2 );

// Normalize
vector_normalize( tmp );

for ( int k = 1; k <= 3; ++k )
	tmp2( k ) = aa_coords( k, aa_atm3 ) - aa_coords( k, aa_atm2 );

// Normalize
vector_normalize( tmp2 );

cross_bk( tmp, tmp2, tmp3 );

Rotate_Angle_About_Axis( natoms(aa, aav), euler( 2 ), tmp3, aa_coords );
Rotate_Angle_About_Axis( 1, euler( 2 ), tmp3, act_coords );

// Add back the new origin
for ( int atm = 1, eatm = natoms( aa, aav ); atm <= eatm; ++atm )
	for ( int k = 1; k <= 3; ++k )
		aa_coords( k, atm ) += ( origin( k ) + vec( k ) );
for( int k = 1 ; k <= 3 ; ++k )
	act_coords( k, 1 ) += ( origin( k ) + vec( k ) );

// Done

return;
}



// std::vector< OmitRegion > OmitRegionVector;

// Constructors for Motif classes

Motif::Motif() :
_aa_type( 0 ),
trans( 3, 0.0 ),
euler( 3, 0.0 ),
rotate( 3, 3, 0.0 ),
aa_atm1( 0 ),
aa_atm2( 0 ),
aa_atm3( 0 )
{}

Motif::Motif( int aa, int a1, int a2, int a3,
               float tx, float ty, float tz,
               float theta, float psi, float phi ) :
_aa_type( aa ),
trans( 3, 0.0 ),
euler( 3, 0.0 ),
aa_atm1( a1 ),
aa_atm2( a2 ),
aa_atm3( a3 )
{
trans(1) = tx;
trans(2) = ty;
trans(3) = tz;
euler(1) = theta;
euler(2) = psi;
euler(3) = phi;
}

SingleMotif::SingleMotif( int na, int n1, int n2, int n3,
                  int aa, int a1, int a2, int a3,
                  float tx, float ty, float tz,
                  float theta, float psi, float phi ) :
Motif( aa, a1, a2, a3, tx, ty, tz, theta, psi, phi ),
na_type( na ),
na_atm1( n1 ),
na_atm2( n2 ),
na_atm3( n3 )
{
Matrix_From_Euler_angles( euler, rotate );
}

SingleMotif::SingleMotif( std::ifstream & file_input ) {

std::string tmp_str;

file_input >> tmp_str;
na_type = na_from_str( tmp_str );

file_input >> tmp_str;
na_atm1 = LookupByName( na_type, 1, tmp_str );
if (na_atm1 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
na_atm2 = LookupByName( na_type, 1, tmp_str );
if (na_atm2 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
na_atm3 = LookupByName( na_type, 1, tmp_str );
if (na_atm3 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
//_aa_type = aa_from_str( tmp_str );
num_from_name( tmp_str, _aa_type );

file_input >> tmp_str;
aa_atm1 = LookupByName( _aa_type, 1, tmp_str );
if (aa_atm1 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
aa_atm2 = LookupByName( _aa_type, 1, tmp_str );
if (aa_atm2 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
aa_atm3 = LookupByName( _aa_type, 1, tmp_str );
if (aa_atm3 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> trans(1);
file_input >> trans(2);
file_input >> trans(3);
file_input >> euler(1);
file_input >> euler(2);
file_input >> euler(3);

// trans = 0.0;

Matrix_From_Euler_angles( euler, rotate );

/*
std::cout << "Motif read in for " << na_type << std::endl;
std::cout << "Atoms " << na_atm1 << " " << na_atm2 << " " << na_atm3 << std::endl;
std::cout << "AA type           " << _aa_type << std::endl;
std::cout << "Atoms " << aa_atm1 << " " << aa_atm2 << " " << aa_atm3 << std::endl;
std::cout << "Trans " << trans(1) << " " << trans(2) << " " << trans(3) << std::endl;
std::cout << "Euler " << euler(1) << " " << euler(2) << " " << euler(3) << std::endl;
*/

}

void
SingleMotif::PlaceRotamer(
	pose_ns::Pose const & pose,
	int aa,
	int aav,
	int pos,
	FArray2Da_float aa_coords,
	FArray2Da_float act_coords,
	int & pos2
)
{
	using namespace param;

	FArray3D_float const & fcoord ( pose.full_coord() );

	// All of the work has been pushed into this common function
	GenericPlaceRotamer( aa, aav, fcoord( 1, na_atm1, pos ), fcoord( 1, na_atm2, pos ),
			fcoord( 1, na_atm3, pos ), aa_coords, act_coords );

	pos2 = -1;

}

DoubleMotif::DoubleMotif( int na1, int n1, int na2, int n2, int n3,
                  int aa, int a1, int a2, int a3,
                  float tx, float ty, float tz,
                  float theta, float psi, float phi
) :
	Motif( aa, a1, a2, a3, tx, ty, tz, theta, psi, phi ),
	na_type1( na1 ),
	na_atm1( n1 ),
	na_type2( na2 ),
	na_atm2( n2 ),
	na_atm3( n3 )
{
	Matrix_From_Euler_angles( euler, rotate );
}

DoubleMotif::DoubleMotif( std::ifstream & file_input ) {

std::string tmp_str;

file_input >> tmp_str;
na_type1 = na_from_str( tmp_str );

file_input >> tmp_str;
na_atm1 = LookupByName( na_type1, 1, tmp_str );
if (na_atm1 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
na_type2 = na_from_str( tmp_str );

file_input >> tmp_str;
na_atm2 = LookupByName( na_type2, 1, tmp_str );
if (na_atm2 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
na_atm3 = LookupByName( na_type2, 1, tmp_str );
if (na_atm3 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
//_aa_type = aa_from_str( tmp_str );
num_from_name( tmp_str, _aa_type );

file_input >> tmp_str;
aa_atm1 = LookupByName( _aa_type, 1, tmp_str );
if (aa_atm1 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
aa_atm2 = LookupByName( _aa_type, 1, tmp_str );
if (aa_atm2 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> tmp_str;
aa_atm3 = LookupByName( _aa_type, 1, tmp_str );
if (aa_atm3 < 1)
	std::cout << "Fatal Error:  Bad atom " << tmp_str << std::endl;

file_input >> trans(1);
file_input >> trans(2);
file_input >> trans(3);
file_input >> euler(1);
file_input >> euler(2);
file_input >> euler(3);

// trans = 0.0;

Matrix_From_Euler_angles( euler, rotate );

/*
std::cout << "Motif read in for " << na_type << std::endl;
std::cout << "Atoms " << na_atm1 << " " << na_atm2 << " " << na_atm3 << std::endl;
std::cout << "AA type           " << _aa_type << std::endl;
std::cout << "Atoms " << aa_atm1 << " " << aa_atm2 << " " << aa_atm3 << std::endl;
std::cout << "Trans " << trans(1) << " " << trans(2) << " " << trans(3) << std::endl;
std::cout << "Euler " << euler(1) << " " << euler(2) << " " << euler(3) << std::endl;
*/

}

StackedMotif::StackedMotif( int na1, int n1, int na2, int n2, int n3,
                  int aa, int a1, int a2, int a3,
                  float tx, float ty, float tz,
                  float theta, float psi, float phi ) :
DoubleMotif( na1, n1, na2, n2, n3, aa, a1, a2, a3,
							tx, ty, tz, theta, psi, phi )
{}

StackedMotif::StackedMotif( std::ifstream & file_input ) :
DoubleMotif( file_input )
{}

bool
StackedMotif::ApplyMotif(
	pose_ns::Pose const & pose,
	int aa,
	int pos
)
{

int const nres( pose.total_residue() );

if ( pos < nres ) {
	int next_na = pose.res( pos + 1 );
	if ( aa == na_type1 && next_na == na_type2 )
		return true;
}

return false;
}

void
StackedMotif::PlaceRotamer(
	pose_ns::Pose const & pose,
	int aa,
	int aav,
	int pos,
	FArray2Da_float aa_coords,
	FArray2Da_float act_coords,
	int & pos2
)
{

using namespace param;

FArray3D_float const & fcoord ( pose.full_coord() );

int next_pos = pos + 1;

pos2 = next_pos;

// All of the work has been pushed into this common function
GenericPlaceRotamer( aa, aav, fcoord( 1, na_atm1, pos ),
		fcoord( 1, na_atm2, next_pos ), fcoord( 1, na_atm3, next_pos ),
		aa_coords, act_coords );

return;
}

DiagonalMotif::DiagonalMotif( int na1, int n1, int na2, int n2, int n3,
                  int aa, int a1, int a2, int a3,
                  float tx, float ty, float tz,
                  float theta, float psi, float phi ) :
DoubleMotif( na1, n1, na2, n2, n3, aa, a1, a2, a3,
							tx, ty, tz, theta, psi, phi )
{}

DiagonalMotif::DiagonalMotif( std::ifstream & file_input ) :
DoubleMotif( file_input )
{}

bool
DiagonalMotif::ApplyMotif(
	pose_ns::Pose const & pose,
	int aa,
	int pos
)
{

int other_pos = -1;
int const nres( pose.total_residue() );

if ( pos != nres ) {
	other_pos = pose.basepair_partner( pos+1 );
} else {
	int bp = pose.basepair_partner( pos );
	if( bp >= 2 ) {
		other_pos = bp - 1;
	}
}

if ( other_pos < 0 ) return false;

int other_na( pose.res( other_pos ) );

if ( aa == na_type1 && other_na == na_type2 ) {
	return true;
}

return false;
}

void
DiagonalMotif::PlaceRotamer(
	pose_ns::Pose const & pose,
	int aa,
	int aav,
	int pos,
	FArray2Da_float aa_coords,
	FArray2Da_float act_coords,
	int & pos2
)
{
	using namespace param;

	FArray3D_float const & fcoord ( pose.full_coord() );

	int const nres( pose.total_residue() );

	int other_pos = -1;

	if ( pos != nres ) {
		other_pos = pose.basepair_partner( pos+1 );
	} else {
		int bp = pose.basepair_partner( pos );
		if( bp >= 2 ) {
			other_pos = bp - 1;
		}
	}

	pos2 = other_pos;

	// All of the work has been pushed into this common function
	GenericPlaceRotamer( aa, aav, fcoord( 1, na_atm1, pos ),
			fcoord( 1, na_atm2, other_pos ), fcoord( 1, na_atm3, other_pos ),
			aa_coords, act_coords );

}

HelixDM::HelixDM() :
	matrix( 4, 4, 0.0),
	offset( 0 )
{}

bool
HelixDM::close_enough( FArray2Da_float this_dm )
{
	if ( worst_miss( this_dm ) > DNA_Motifs::Motifs_close_enough )
		return false;
	else
		return true;
}

float
HelixDM::worst_miss( FArray2Da_float this_dm )
{
	this_dm.dimension( 4, 4 );

	float max_diff = 0.0;
	float diff = 0.0;

	for ( int i = 1; i <= 4; ++i )
		for ( int j = 1; j <= 4; ++j ) {
			diff = std::fabs( this_dm(i,j) - matrix(i,j) );
			if ( diff > max_diff ) max_diff = diff;
		}

	return max_diff;
}

MotifTether::MotifTether() :
	res_num( 0 ),
	atom_num( 3, 0 ),
	coords( 3, 3, 0.0)
{}

MotifTetherLibrary motif_set;

	namespace dna_match_namespace {
		DNAMotifRotamerLibrary * dna_match_library;
		RegionPairLibrary dna_match_regions;
		int num_matches;
	}


} // namespace DNA_Motifs
