// /////////////////////////////////////////////////////////////////////////////////////
// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.10 $
//  $Date: 2005/10/06 23:09:52 $
//  $Author: sid $
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   rosetta++/dock_ensemble.cc
///
/// @brief  Provides ensemble docking in the low resolution phase of pose docking.
/// 	    when pose_docking_ensemble_move( pose ) is called, a conformer in the  pose
///	    is swapped out for a conformer in the ensemble.  This uses an ensemble for
///         partner 1 (ensemble1) or partner 2 (ensemble2) or both.  Conformer swap is
///         made by generating a partition function in conformer space and randomly
///	    selecting a conformer based on its Boltzmann weighted probability.
///
///	    command line options:
///         -ensemble1 [ensemble size]
///         -ensemble2 [ensemble size]
///         -ensemble_prepack
///
/// @author Sid Chaudhury (sidc@jhu.edu)

// Rosetta Headers
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "current_pose.h"
#include "disulfides.h"
#include "docking.h"
#include "docking_constraints.h"
#include "dock_ensemble.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "docking_ns.h"
#include "docking_score.h"
#include "docking_scoring.h"
#include "files_paths.h"
#include "fullatom.h"
#include "fullatom_setup.h"
#include "map_sequence.h"
#include "monte_carlo.h"
#include "misc.h"
#include "minimize.h"
#include "native.h"
#include "nblist.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_docking.h"
#include "pose_docking_flexible.h"
#include "pose_idealize.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "pose_rotamer_trials.h"
#include "pose_vdw.h"
#include "random_numbers.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"

// Utility Headers
#include <utility/vector1.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <string>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>

// Triplet Headers
#include <numeric/xyzVector.hh>

// C++ Headers
/////////////////////////////////////////////////////////////////////
//pose_docking_ensemble_move
//
//takes a complex, does superposes every conformer in the ensemble to
//the complex using interface residues, calculates the scores for each
//conformer, generates a partition function, and replaces the original
//conformer with a new one based on its Boltzmann probability from the
//partition function
////////////////////////////////////////////////Sid Chaudhury////////
void
pose_docking_ensemble_move(
	pose_ns::Pose & pose
)
{

	using namespace pose_ns;
	using namespace docking;

	if (!docking::ensemble_read(1) && !docking::ensemble_read(2)){
		read_ensemble();
		if (docking::ensemble_prepack) ensemble_calculate_ref_energy();
		Pose ihatemisc;
		std::string filename( files_paths::pdb_path_1 + files_paths::protein_name + ".pdb" );
		std::cout<< filename<<std::endl;
	  	pose_from_pdb(ihatemisc, filename, true /*fullatom*/, false /*ideal_pos*/, true /*coords_init*/); //native monomer pose
		}

	Pose new_pose;
	new_pose = pose;

	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4d );
	float temperature = 0.8;

	float partition_sum = 0;
	score_list = 0;
	exp_score_list = 0;
	prob_list = 0;
	float min_score = 0;

	for (int i =1; i<=ensemble_size(1); i++){		//calculate interaction energies for matrix of ensembles
		for (int j = 1; j<=ensemble_size(2); j++){
			if (ensemble_size(1) > 1) pose_ensemble_switch_structure(new_pose,1,i);
			if (ensemble_size(2) > 1) pose_ensemble_switch_structure(new_pose,2,j);
			float score = new_pose.score(weight_map);
			if (i == 1 && j == 1) min_score = score;
			min_score = min(min_score, score);
			score_list(i,j) = score;
			new_pose = pose;
			}
		}

	for (int i =1; i<=ensemble_size(1); i++){
		for (int j=1; j<=ensemble_size(2); j++){
			score_list(i,j) = score_list(i,j)-min_score;
			exp_score_list(i,j) = std::exp((-1*score_list(i,j))/temperature);
			partition_sum += exp_score_list(i,j);
			}
		}

	float last_prob_written = 0;
	for (int i =1; i<=ensemble_size(1); i++){
		for (int j=1; j<=ensemble_size(2); j++){
			if (i == 1 && j == 1) {
				prob_list(i,j) = exp_score_list(i,j)/partition_sum;
				last_prob_written = prob_list(i,j);
			} else {
				prob_list(i,j) = last_prob_written + exp_score_list(i,j)/partition_sum;
				last_prob_written = prob_list(i,j);
			}
		}
	}

	float m_num = ran3();	//select random number and choose from probablity table
	int struct_num1 = 0;
	int struct_num2 = 0;


	for(int i =1; i <= ensemble_size(1); i++){
		for (int j = 1; j <= ensemble_size(2); j++){
			if (m_num <= prob_list(i,j)) {
				struct_num1 = i;
				struct_num2 = j;
				if (ensemble_size(1) > 1) pose_ensemble_switch_structure(pose, 1, struct_num1);
				if (ensemble_size(2) > 1) pose_ensemble_switch_structure(pose, 2, struct_num2);
				return;
				}
			}
		}


	std::cout<<"WARNING: conformer selection failed in ensemble_move()!!  assigning random conformers!!"<< m_num<<std::endl;

	int i_rand = static_cast< int > (ran3() * (ensemble_size(1)-1)) + 1 ;
	int j_rand = static_cast< int > (ran3() * (ensemble_size(2)-1)) + 1 ;
	if (ensemble_size(1) > 1) pose_ensemble_switch_structure(pose,1,i_rand);
	if (ensemble_size(2) > 1) pose_ensemble_switch_structure(pose,2,j_rand);

}

/////////////////////////////////////////////////////////////////
//pose_ensemble_switch_structure
//
//switches the current conformer for partner p_num, to another
//conformer, m_num, by superposition of interface residues
////////////////////////////////////////Sid Chaudhury///////////
void
pose_ensemble_switch_structure(
	pose_ns::Pose & pose,
	int p_num,
	int m_num
)
{
	using namespace pose_ns;
	using namespace docking;
	using namespace files_paths;

	Pose new_pose;
	new_pose = pose;

	int monomer_size;
	if (p_num == 1){
		monomer_size = part_begin(2) - part_begin(1);
		new_pose.copy_segment(monomer_size, *ensemble1[m_num-1], part_begin(p_num), 1, false );
		}
	if (p_num == 2){
		monomer_size = part_end(2) - part_end(1);
		new_pose.copy_segment(monomer_size, *ensemble2[m_num-1], part_begin(p_num), 1, false );
		}

	pose_dock_superpos(pose, new_pose, p_num);

}

///////////////////////////////////////////////////////////////////////////////
/// @begin pose_ensemble_recover_sidechains
///
/// @brief recovers sidechains for each selected monomer from respective
///        monomer in the ensemble
///
/// @detailed recovers sidechains for each selected monomer from respective
///           monomer in the ensemble. The "new_pose" that is created has all
///           the saved sidechains for the required conformer, but then the
///           partner is not in the position chosen by the centroid mode
///           perturbation. However, this does not hamper copying sidechains.
///
/// @param[in] p_num: partner number
///            n_num: conformer number of p_num
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/14/2008
///
/// @last_modified 08/14/2008
///////////////////////////////////////////////////////////////////////////////
void
pose_ensemble_recover_sidechains(
	pose_ns::Pose & pose,
	int p_num,
	int m_num
)
{
	using namespace pose_ns;
	using namespace docking;
	using namespace files_paths;

	Pose new_pose;
	new_pose = pose;

	int monomer_size;
	if (p_num == 1){
		monomer_size = part_begin(2) - part_begin(1);
		new_pose.copy_segment(monomer_size, *ensemble1[m_num-1], part_begin(p_num), 1, false );
		pose.recover_sidechain( new_pose );
	}
	if (p_num == 2){
		monomer_size = part_end(2) - part_end(1);
		new_pose.copy_segment(monomer_size, *ensemble2[m_num-1], part_begin(p_num), 1, false );
		}

	pose.recover_sidechain( new_pose );

}

///////////////////////////////////////////////////////////////////
//pose_dock_superpos
//
//superposes new_pose on to partner p_num of pose1 across interface
//residues
///////////////////////////////////////////////Sid Chaudhury//////

void
pose_dock_superpos(
	pose_ns::Pose & pose1,
	pose_ns::Pose & new_pose,
	int p_num
)
{
	using namespace pose_ns;
	using namespace docking;

//align monomer B from new_pose to monomer A in pose 1 and replace in pose 2

	FArray1D_bool interface(pose1.total_residue());
	interface = false;
	pose_docking_calc_interface( pose1, interface );

	int ires_num = 0;
	int aligned_regions = 0;
	FArray1D_int alignstart(pose1.total_residue());
	FArray1D_int alignend(pose1.total_residue());

	bool check = false;
	for(int i = docking::part_begin(p_num); i <= docking::part_end(p_num); i++){
		if (interface(i)) ires_num++;
		if (interface(i) && !check){
			aligned_regions++;
			alignstart(aligned_regions) = i;
			check = true;
			}
		if (!interface(i) && check){
			alignend(aligned_regions) = i-1;
			check = false;
			}
		}

	if (check) alignend(aligned_regions) = docking::part_end(p_num); //caps the end if c-term is at interface

	if (aligned_regions == 0 || ires_num <= 5) {
		aligned_regions =1;
		alignstart(1) = docking::part_begin(p_num);
		alignend(1) = docking::part_end(p_num);
		}

	int const nres( pose1.total_residue());
	FArray3D_float full_new (new_pose.full_coord());
	FArray3D_float Epos_new (new_pose.Eposition());
	FArray3D_float const & full_old (pose1.full_coord() );
	FArray3D_float const & Epos_old (pose1.Eposition() );

	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos_new( k, atom_index, i );
			p2a(k,i) = Epos_old( k, atom_index, i );
		}
	}

	float local_rms;

	orient_region(nres,Epos_new,full_new,Epos_old,aligned_regions,
	 alignstart,alignend,aligned_regions,alignstart,alignend,local_rms);

	int a_num = 1;
	if (p_num == 1) a_num = 2;

	for (int i = docking::part_begin(a_num); i<= docking::part_end(a_num); i++){
		for (int k = 1; k<= 3; k++){
				int aa = pose1.res(i);
				int aav = pose1.res_variant(i);
			int Natom =  aaproperties_pack::natoms(aa,aav);
				for(int n =1; n<= Natom; n++){
					full_new(k,n,i) = full_old(k,n,i);
					if(n <=5){
						Epos_new(k,n,i) = Epos_old(k,n,i);
						}
					}
				}
		}

	new_pose.set_coords( false /*ideal_pos*/, Epos_new, full_new );
	new_pose.update_backbone_bonds_and_torsions();

	pose1 = new_pose;

}
//////////////////////////////////////////////////////////////////
//pose_docking_ensemble_repack_conformer
//
//does a full repack of the conformer
///////////////////////////////////////////Sid Chaudhury//////////
void
pose_docking_ensemble_repack_conformer(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	using namespace docking;

	FArray1D_bool allow_repack(pose.total_residue(), false);
	for (int i = part_begin(2); i <= part_end(2); i++) allow_repack(i) = true;
	pose.repack(allow_repack, true);

}
/////////////////////////////////////////////////////////////////
//read_ensemble()
//
//reads in the pdbs from files pdblist1 and pdblist2 and stores
//them in lists of pose objects, ensemble1 and ensemble2
//
///////////////////////////////////////////Sid Chaudhury/////////

void
read_ensemble()
{
	using namespace pose_ns;
	using namespace docking;
	using namespace files_paths;

	cen_ref_energy = 0;
	FA_ref_energy = 0;

	ensemble1.clear();
	ensemble2.clear();

	std::cout<<"reading ensemble:"<<std::endl;

	if (!ensemble_read(1)){
		std::string filename = pdb_path_1 + "pdblist1";
		std::cout<<filename<<std::endl;
		read_ensemble( 1, filename);
	}

	if (!ensemble_read(2)){
		std::string filename = pdb_path_1 + "pdblist2";
		std::cout<<filename<<std::endl;
		read_ensemble( 2, filename);
	}

}
/////////////////////////////////////////////////////////////////
//read_ensemble()
//
//stores the pdbs in a list of pose pointers, ensemble1 and 2
//also stores their centroid mode and fullatom reference energies
//
////////////////////////////////////////////Sid Chaudhury////////

void
read_ensemble(
	int p_num,
	std::string filename
)
{
	using namespace docking;
	using namespace pose_ns;
	using namespace files_paths;

  bool const ideal_pos( false );
  bool const coords_init( true );
	utility::io::izstream data;

	data.open(filename);
	data.seek_beg();

	int k = 1;
		while (!data.eof() && k <= ensemble_size(p_num)){    //read file names
		data >> pdblist(p_num, k);
		k++;
		}

		k = 1;
		while (!data.eof() && k<=ensemble_size(p_num)){  // read unbound centroid scores
		data >> cen_ref_energy(p_num, k);
		k++;
		}

		k = 1;
		while (!data.eof() && k<= ensemble_size(p_num)){ // read unbound fullatom scores
		data >> FA_ref_energy(p_num, k);
		k++;
		}

	data.close();
	data.clear();

	ensemble_read(p_num) = true;

	using namespace disulfides::options;
	bool store_find_disulf = find_disulf;
	bool store_fix_disulf = fix_disulf;

	find_disulf = false;
	fix_disulf = false;

	for(int i = 1; i <= ensemble_size(p_num); i++){
		std::cout<<pdblist(p_num, i)<<' '<<cen_ref_energy(p_num, i)<< ' '<<FA_ref_energy(p_num, i)<<std::endl;

		std::string pdb_file;
		pdb_file = pdblist(p_num, i);
		if (ensemble_prepack) pdb_file = pdb_path_1 + pdb_file;
		if (!ensemble_prepack) pdb_file = start_path + pdb_file;

		if (p_num == 1){
			ensemble1.push_back(new Pose() );
			pose_from_pdb(*ensemble1[i-1], pdb_file, true /*fullatom*/, ideal_pos, coords_init);
			}
		if (p_num == 2){
			ensemble2.push_back(new Pose() );
			pose_from_pdb(*ensemble2[i-1], pdb_file, true /*fullatom*/, ideal_pos, coords_init);
			}
		}

	find_disulf = store_find_disulf;
	fix_disulf = store_fix_disulf;
}



///////////////////////////////////////////////////////////////////
//detect_current_conformer
//
//reads in a complex and recognizes which conformer is currently
//in the complex
//
//////////////////////////////////////////////Sid Chaudhury////////

void
detect_current_conformer(
	const pose_ns::Pose & pose
	)
{
	using namespace pose_ns;
	using namespace docking;


	current_conformer = 1;
	float torsion_cap = 30; //caps torsion angle difference

	for (int j = 1; j <=2; j++){
		if (ensemble_size(j) > 1){
			int conf_nres = docking::part_end(j) - docking::part_begin(j)+1;
			float min_tordiff = 360*conf_nres;

			for (int m_num = 1; m_num <= ensemble_size(j); m_num++){
				float tordiff = 0;

				for (int i = 2; i <= (conf_nres-1); i++){
					if (j ==1) tordiff += min(abs(ensemble1[m_num-1]->phi(i) - pose.phi(docking::part_begin(j)+i-1)), torsion_cap);
					if (j ==2) tordiff += min(abs(ensemble2[m_num-1]->phi(i) - pose.phi(docking::part_begin(j)+i-1)), torsion_cap);
					}

				if (tordiff <= min_tordiff){
					current_conformer(j) = m_num;
					min_tordiff = tordiff;
					}
				}
			}
		}
}
/////////////////////////////////////////////////////////////////////
//ensemble_recover_sidechains()
//
//recovers sidechain conformation from conformers in the ensemble
//
//////////////////////////////////////////////Sid Chaudhury/////////
void
ensemble_recover_sidechains(
	pose_ns::Pose & pose
	)
{
	using namespace pose_ns;
	using namespace docking;

	Pose sidechain_pose;

	pose.set_fullatom_flag( true, false );

	sidechain_pose = pose;

	detect_current_conformer( sidechain_pose );
	pose_ensemble_switch_structure( sidechain_pose, 1, current_conformer(1) ); //recovers conformer sidechains
	pose_ensemble_switch_structure( sidechain_pose, 2, current_conformer(2) ); //recovers conformer sidechains

	pose.recover_sidechain( sidechain_pose );
}


/////////////////////////////////////////////////////////////////////
//ensemble_calculate_ref_energy()
//
//in ensemble prepack mode, calculates the centroid mode and full-atom
//mode energies and then outputs them on screen
//
//////////////////////////////////////////////Sid Chaudhury/////////

void
ensemble_calculate_ref_energy()
{
	using namespace pose_ns;
	using namespace pose_ns::pose_param;
	using namespace docking;

	cen_ref_energy = 0;
	FA_ref_energy = 0;

	int repeat = 4;

	for (int p_num = 1; p_num <= 2; p_num++){
		if (ensemble_size(p_num) > 1) calc_cen_ref_energy( p_num );
		for (int i = 1; i <= ensemble_size(p_num) ; i++){
			std::cout<<"partner "<<p_num<<" conformer "<<i <<std::endl;
			Pose new_pose;
			if (p_num == 1) new_pose = *ensemble1[i-1];
			if (p_num == 2) new_pose = *ensemble2[i-1];
			bool repack = true;
			if (norepack1 && p_num ==1) repack = false;
			if (norepack2 && p_num ==2) repack = false;
			calc_FA_ref_energy( new_pose, repeat, repack, p_num, i);
			}
		}

	 	for(int p_num = 1; p_num <= 2; p_num++){
			std::cout<<"partner "<<p_num<<std::endl;
			for(int i=1; i<=ensemble_size(p_num); i++)  std::cout<<cen_ref_energy(p_num, i)<<std::endl;
			for(int i=1; i<=ensemble_size(p_num); i++)  std::cout<<FA_ref_energy(p_num, i)<<std::endl;
			}

		update_pdblist_file();
		std::exit(0);
}
//////////////////////////////////////////////////////////////////////////////
//set_cen_unbound_score
//
//sets the unbound reference energy for the complex, specific to each
//conformer.  This is used in the score function.
//
///////////////////////////////////////////////////Sid Chaudhury//////////////
void
set_cen_unbound_score(
	float & cen_unbound_score
)
{
	using namespace pose_ns;
	using namespace docking;

	if(score_check_current_pose()) detect_current_conformer( score_get_current_pose() );

	cen_unbound_score = 0;

	for (int p_num = 1; p_num <= 2; p_num++){
		if (ensemble_size(p_num) >  1) cen_unbound_score += cen_ref_energy(p_num, current_conformer(p_num));
		}
}

//////////////////////////////////////////////////////////////////////////////
//set_FA_unbound_score
//
//sets the unbound reference energy for the complex, specific to each
//conformer.  This is used in the score function.
//
///////////////////////////////////////////////////Sid Chaudhury//////////////
void
set_FA_unbound_score(
	float & FA_unbound_score
)
{
	using namespace pose_ns;
	using namespace docking;

	//if(score_check_current_pose()) detect_current_conformer( score_get_current_pose() );//no change in bb in FA mode

	FA_unbound_score = 0;

	for (int p_num = 1; p_num <= 2; p_num++){
		if (ensemble_size(p_num) > 1) FA_unbound_score += FA_ref_energy(p_num, current_conformer(p_num));
		}
}

///////////////////////////////////////////////////////////////////////////
//calc_FA_ref_energy
//
//in ensemble prepack mode, calculates the fullatom reference energies for
//every conformer in the ensemble(s).
//
////////////////////////////////////////////////Sid Chaudhury//////////////
void
calc_FA_ref_energy(
	const pose_ns::Pose & pose,
	const int repeat,
	const bool repack,
	const int p_num,
	const int c_num
)
{
	using namespace pose_ns;
	using namespace docking;
	using namespace files_paths;

	scorefxns::scorefxn_flags::docking_ensemble_scorefxn = false;

	Pose new_pose;
	new_pose = pose;
	Score_weight_map weight_map;

	setup_score_weight_map( weight_map, score10d );
	const float temperature( 0.8 );
	Monte_carlo mc( new_pose, weight_map, temperature );

	for (int j = 1; j <= repeat; j++){
		new_pose = pose;

		if (repack) new_pose.full_repack( true );
		new_pose.score(weight_map);

		if ( docking::dock_rtmin && repack ) {
			float const energycut(0.01);
			const int nres( new_pose.total_residue() );
			set_rotamer_trials_by_deltaE( pose_ns::RESENERGY, energycut, nres, mc.best_pose().get_1D_score( pose_ns::RESENERGY ), 1);
			score_set_try_rotamers( true );
			score_set_minimize_rot( true );
			new_pose.score( weight_map );
			score_set_minimize_rot( false );
			score_set_try_rotamers( false );
			}

		mc.boltzmann( new_pose );
		}

		new_pose = mc.low_pose();
		std::string filename( pdblist(p_num, c_num) + ".ppk" );
		pdblist(p_num, c_num) = filename;
		new_pose.dump_pdb(start_path + filename);
		FA_ref_energy(p_num, c_num) = new_pose.score(weight_map);

		std::cout<<"Fullatom score " <<FA_ref_energy(p_num, c_num)<<std::endl;

}
/////////////////////////////////////////////////////////////////////
//calc_cen_ref_energy()
//
//in ensemble prepack mode, calculates the centroid mode reference
//energies for every conformer in the ensemble(s)
//
/////////////////////////////////////////////////Sid Chaudhury//////
void
calc_cen_ref_energy(
	const int p_num
)
{
	using namespace pose_ns;
	using namespace docking;

	Pose starting_pose;
	std::string filename( files_paths::pdb_path_1 + files_paths::protein_name + ".pdb" );
	pose_from_pdb(starting_pose, filename, true /*fullatom*/, false /*ideal_pos*/, true /*coords_init*/); //native monomer pose
	scorefxns::scorefxn_flags::docking_ensemble_scorefxn = false;
	pose_docking_build_simple_tree( starting_pose, starting_pose.total_residue(), docking::part_end(1) );

	for (int i = 1; i <= ensemble_size(p_num); i++){
		Pose trimmed_pose;
		trimmed_pose = starting_pose;
		Score_weight_map weight_map;
		setup_score_weight_map( weight_map, score4d );

		pose_ensemble_switch_structure(trimmed_pose,p_num,i);

		int dock_jump(1);
		pose_ns::Jump perturbed_jump( trimmed_pose.get_jump( dock_jump ) );
		int const pos1( trimmed_pose.fold_tree().get_jump_point()(1, dock_jump) );
		int const pos2( trimmed_pose.fold_tree().get_jump_point()(2, dock_jump) );
		const FArray3D_float & Epos( trimmed_pose.Eposition() );
		numeric::xyzVector_double trans_axis (
			numeric::xyzVector_double( &Epos(1,2,pos2) ) -
			numeric::xyzVector_double( &Epos(1,2,pos1) ) );

		float const step_size = 499.0;
		perturbed_jump.translation_along_axis(Epos(1,1,pos1), trans_axis, step_size);
		trimmed_pose.set_jump( dock_jump, perturbed_jump );

	  	float score_apart = trimmed_pose.score( weight_map );

		cen_ref_energy(p_num, i) = score_apart;
		}

	float adjustment = cen_ref_energy(p_num, 1);
	for (int i =1; i <=ensemble_size(p_num); i++){
		float ref_energy = cen_ref_energy(p_num, i);
		adjustment = min(ref_energy, adjustment);
		}
	for (int i = 1; i<= ensemble_size(p_num); i++){
		cen_ref_energy(p_num, i) = cen_ref_energy(p_num, i) - adjustment;
		std::cout<<cen_ref_energy(p_num, i)<<std::endl;
		}
}

void
update_pdblist_file()
{

	using namespace files_paths;
	using namespace docking;


	std::string filename;
	for (int p_num = 1 ; p_num <= 2; p_num++){
		if (p_num == 1) filename = pdb_path_1 + "pdblist1";
		if (p_num == 2) filename = pdb_path_1 + "pdblist2";

		utility::io::ozstream data;

		data.open(filename);

		int k = 1;
		while (k <= ensemble_size(p_num)){    //read file names
			data << pdblist(p_num, k)<< '\n';
			k++;
			}

		k = 1;
		while (k<=ensemble_size(p_num)){  // read unbound centroid scores
			data << cen_ref_energy(p_num, k)<< '\n';
			k++;
			}

		k = 1;
		while (k<= ensemble_size(p_num)){ // read unbound fullatom scores
			data << FA_ref_energy(p_num, k) << '\n';
			k++;
			}
	data.close();
	data.clear();
		}

}
