// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
//  vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7412 $
//  $Date: 2006-02-06 18:27:25 -0500 (Mon, 06 Feb 2006) $
//  $Author: aroop $
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   rosetta++/dle.cc
///
/// @brief  Protocols to dock proteins (specifically antibody-antigen
///         complexes) using a loop ensemble. The following protocols build
///         loops, generate input files for clustering using R,carry out one
///         round of refinement and then swaps from this refined library during
///         docking. Modules are present for calculating loop-rmsg & loop-rmsl,
///         ramping scores etc.
/// @author Aroop Sircar (aroop@jhu.edu)

// Rosetta Headers
#include "aa_name_conversion.h"
#include "aaproperties_pack.h"
#include "after_opts.h"
#include "antibody_modeling.h"
#include "design.h"
#include "docking.h"
#include "docking_constraints.h"
#include "dock_loop_ensemble.h"
#include "dock_loop_ensemble_ns.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "docking_ns.h"
#include "docking_score.h"
#include "docking_scoring.h"
#include "docking_unbound.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fold_loops.h"
#include "fragment_class.h"
#include "fragments_pose.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_setup.h"
#include "gb_elec.h"
#include "jumping_loops.h"
#include "jumping_refold.h"
#include "jumping_util.h"
#include "loop_class.h"
#include "loop_relax.h"
#include "loops_ns.h"
#include "map_sequence.h"
#include "monte_carlo.h"
#include "misc.h"
#include "minimize.h"
#include "native.h"
#include "nblist.h"
#include "orient_rms.h"
#include "output_decoy.h"
#include "pack.h"
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "PackerTask.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pose.h"
#include "pose_design.h"
#include "pose_docking.h"
#include "pose_docking_flexible.h"
#include "pose_idealize.h"
#include "pose_io.h"
#include "pose_rms.h"
#include "pose_vdw.h"
#include "random_numbers.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "smallmove.h"
#include "score.h"
#include "score_ns.h"

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.o.hh>

// Utility Headers
#include <utility/io/izstream.hh>
//#include <utility/io/ozstream.hh>

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_initialize
///
/// @brief switches to pose mode and then calls function to build loops
///
/// @detailed This  function  is  generally  called  from initialize_start() in
///           initialize.cc. Thus effecively this is called  only once for each
///           starting structure. It creates a pose from  misc  and then passes
///           the pose to the misc  building function. It also ensures that the
///           misc is populated with the stuff that it started out with so that
///           all functions downstream of  it which depend on misc can function
///
/// @param[in]
///
/// @global_read misc
///
/// @global_write misc ( reverts back to the misc it read in)
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/27/2006
///
/// @last_modified 02/21/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_initialize()
{
	using namespace pose_ns;
	using namespace runlevel_ns;
	using namespace dle_ns;

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // switching on pose mode

	Pose pdb_pose;
	bool const ideal_pose( false ); // pose is non-ideal
	bool const coords_init( true ); // co-ordinates are initialized
	bool const fullatom( get_docking_fullatom_flag() ); // fullatom status
	// copy from misc
	pose_from_misc( pdb_pose, fullatom, ideal_pose, coords_init );

	//loads loop information (beginning/end) from loopfile
	// file containing loop definition
	stringafteroption("dock_keyres", "loopfile", loop_def_filename);
	dle_load_and_define_loops();

	dle_common_options();
	// repack loop neighbors
	//repack_loop_neighbors = truefalseoption("dock_neighbor");
	// small/shear moves in loop
	fullatom_loop_relax = truefalseoption("dock_relax");
	dle_cter_flag = truefalseoption("dock_cter"); // models cter
	// for different light_chain identifier
	//stringafteroption("l_chain", 'L', light_chain);
	//dle_model_H3 = truefalseoption("H3_filter");

	Pose invariant_pose; // pose to be able to copy back to misc, ie restore
	invariant_pose = pdb_pose; // assigning invariant_pose to misc through pose

	// output decoy without enforcing any filters
	//output_with_no_filters = truefalseoption("flex_nofilter");
	if(dle_cter_flag) // if c-terminal needs to be modeled
		intafteroption("dock_cter", 0, cter_modeling_size); // length of c-terminal
	                                                      // to be modeled

	if(dle_build) {
		//if(truefalseoption("build_loop")) {
		if(runlevel >= standard)
			std::cout << "Building Loops..." << std::endl;
		dle_build = true; // only build loops
		int const nres( invariant_pose.total_residue() ); // Total residues
		read_fragments_simple( stringafteroption("s").erase(4) + "_", nres );
		if(runlevel >= standard)
			std::cout << "Completed Loop Building" << std::endl;
	}
	else if(truefalseoption("dock_loop")){
		if(runlevel >= standard)
			std::cout << "Loading Loop Libraries" << std::endl;
		// fold tree with 2 cutpoints for H3 & dock rg-body jump simultaneous min
		simultaneous_minimization = truefalseoption("min2");
		// // insert one loop at a time and do a local refine
		// insert_all_loops_with_local_refine = truefalseoption("insert_all");
		// load previously built loop library
		dle_multiple_loop_operations( pdb_pose, "load_libraries");
		if(runlevel >= standard)
			std::cout << "Finished Loading Loop Libraries. Starting simulations"
								<< std::endl;
	}
	else if(dle_cluster) {
		//	else if(truefalseoption("cluster_loop")) {
		dle_cluster = true; // only cluster loop library
		if(runlevel >= standard)
			std::cout << "Preparing data for loop clustering" << std::endl;
		// cluster previously built loop library
		dle_multiple_loop_operations( pdb_pose,
																								 "cluster_rms_table");
		if(runlevel >= standard)
			std::cout << "Finished preparing files for clustering" << std::endl;
	}
	else if(dle_refine) {
		//	else if(truefalseoption("refine_loop")) {
		dle_refine = true; // only refinement of loops in loop lib
		if(runlevel >= standard)
			std::cout << "Preparing to refine loop library" << std::endl;
		// cluster previously built loop library
		dle_multiple_loop_operations( pdb_pose, "refine_libraries");
		if(runlevel >= standard)
			std::cout << "Finished refining loop library" << std::endl;
	}
	else {
		// exit if neither build nor read nor cluster
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// copying back to misc to ensure misc is as it was
	invariant_pose.copy_to_misc();

	// pose flag back to original setting
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	// outputs all decoys without applying filters
	files_paths::disable_output_decoy_filters = (dle_build ||
		dle_cluster || dle_refine	||
		  output_with_no_filters) && (!decoy_filter);

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_multiple_loop_operations
///
/// @brief carries out specified operation for all loops defined in datafile
///        entered (default name = loopfile)
///
/// @detailed carries out various operations like refinement, build, clustering
///           one loop at a time depending on what the value of dock_flex_func
///           is, i.e. branches out according to option
///
/// @param[in] dock_flex_func which can be "build_libraries", "load_libraries",
///            "cluster_libraries", "refine_libraries", "cluster_only",
///            "build_only", "refine_only"
///
/// @global_read dle_ns
///
/// @global_write updates pose with new configuration. Does not update much
///               except for most branching out spits out a text file
///
/// @remarks
///
/// @references
///
/// @authors Aroop 11/01/2006
///
/// @last_modified Aroop 11/01/2006
/////////////////////////////////////////////////////////////////////////////////
void
dle_multiple_loop_operations(
  pose_ns::Pose & pose,
	const std::string & dock_flex_func
)
{

	using namespace pose_ns;
	using namespace dle_ns;

	int const nres( pose.total_residue() ); // Total residues

	if(docking::dle_loops_flag)
		{
			if( dock_flex_func == "build_libraries" )
				read_fragments_simple( pdb_filename.erase(4) + "_", nres );

			int counter = 0;
			for( Loops::const_iterator it = dle_loops.begin(), it_end = dle_loops.end(); it != it_end; ++it ) {
				int loop_begin = it->start();
				int loop_end = it->stop();
				if(dock_flex_func == "load_libraries")
					dle_read_in_one_loop_library(loop_begin, loop_end, counter, dock_loop[counter], "refined");
				else if(dock_flex_func == "refine_libraries")
					dle_read_in_one_loop_library(loop_begin, loop_end, counter, dock_loop[counter], "unrefined");
				else if(dock_flex_func == "cluster_rms_table")
					dle_read_in_one_loop_library(loop_begin, loop_end, counter, dock_loop[counter], "unrefined_unclustered");
				if(dock_flex_func == "cluster_only")
					dle_cluster_loop_library( pose, loop_begin, loop_end, dock_loop[counter], counter);
				if(dock_flex_func == "refine_only")
					dle_relax_loop_library( pose, loop_begin, loop_end, dock_loop[counter], counter);
				if(dock_flex_func == "build_only") {
					dle_build_one_loop( pose, loop_begin, loop_end, dock_loop[counter], counter );
					dle_write_one_loopfile( pose, loop_begin, loop_end, counter );
				}
				counter = counter + 1;
			}
		}

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dle_write_loop_library
///
/// @brief Outputs the loop library to a file to be read in later
///
/// @detailed Stores the entire loop library in a file in the form of phi-psi
///           omega angles. The file is placed where other constraints are found
///           as defined in paths.txt. Each loop in the loop library is separated
///           by a blank line. The fragment is first inserted into a protein(pose).
///           Then the phi-psi-omega angles of the loop_residues in the pose
///           are read into the file to be stored
///
/// @param[in] pose: protein in which loop is inserted
///            loop_begin: Beginning of Loop
///            loop_end: End of Loop
///            loop: the vector containing phi-psi-omega angles of the loop library
///            counter: to keep track if there are more than one loop being modeled
///
/// @global_read files_paths
///
/// @global_write
///
/// @remarks Implemented so that loop building can be decoupled from the docking
///          phase. The docking phase will involve reading in this file and inserting
///          random insertions from this loop library while making docking rigid body
///          moves.
///
/// @references
///
/// @authors Aroop 10/19/2006
///
/// @last_modified Aroop 10/19/2006
/////////////////////////////////////////////////////////////////////////////////
void
dle_write_loop_library(
  pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
  std::vector< Fragment > & loop,
  int counter,
	const std::string & lib_refinement_status
)
{
	using namespace pose_ns;
	using namespace dle_ns;
	using namespace files_paths;

	counter = counter + 1; // Start at 1 instead of zero for easy readability

	std::stringstream filename_counter; // String holder to contain the counter converted from int to string
	filename_counter << counter; // int counter fed to string
	std::string library_filename;
	if(lib_refinement_status == "unrefined")
		library_filename = constraints_path + protein_name + "_loop_library_0" + filename_counter.str(); // file is placed where other contraints are supposed to exist
	else if(lib_refinement_status == "refined")
		library_filename = constraints_path + protein_name + "_refined_loop_library_0" + filename_counter.str(); // file is placed where other contraints are supposed to exist
	std::ofstream outfile ( library_filename.c_str(), std::ios::out); // create file

	for ( std::vector< Fragment >::const_iterator it=loop.begin(),it_end = loop.end(); it != it_end; ++it ) {
		it->insert( pose, loop_begin ); // insert fragment into loop for ease of extraction
		for(int loop_res = loop_begin; loop_res <= loop_end; loop_res++)
			{
				outfile << F(10, 3, pose.phi(loop_res))
								<< F(10, 3, pose.psi(loop_res))
								<< F(10, 3, pose.omega(loop_res))
								<< std::endl;
			}
		outfile<< std::endl; // print a new line after each loop
	}

	outfile.close(); // close file
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dle_cluster_loop_library
///
/// @brief generates a score file and an rms table file which will be used by R
///        for hierarchial clustering
///
/// @detailed corresponding to each loop in the loop library, an entry will be
///           generated. An entry for an rms table corresponds to the rms of the
///           loop in question compared to all the other loops in a loop library.
///           Also a separate score file for each loop is generated, which is
///           once again used by R to get the lowest scoring from each cluster
///
/// @param[in] loop_begin: Starting sequentially numbered loop residue
///            loop_end: Ending sequentially numbered loop residue
///            loop_library: Loop library over which the aforesaid steps are
///                          carried out
///            counter: an unique sequential number corresponding to each loop
///                     ensemble
///
/// @global_read makepdb_number so that one can use the power of parallel
///              processing
///
/// @global_write generates files for each loop in a loop ensemble as explained
///
/// @remarks
///
/// @references
///
/// @authors Aroop 11/02/2006
///
/// @last_modified Aroop 11/02/2006
/////////////////////////////////////////////////////////////////////////////////
void
dle_cluster_loop_library(
	pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
	std::vector< Fragment > & loop_library,
	int counter
)
{
	using namespace pose_ns;
	using namespace files_paths;
	using namespace makepdb_decoy;

	Pose antibody;
	antibody = pose;
	if(docking::dle_loops_flag)
		antibody.delete_segment(docking::part_end(1)+1, pose.total_residue());


	// setting up loop tree for returning as decoys
	int nres( pose.total_residue() );
	int loop_size = ( loop_end - loop_begin ) + 1;
	int cutpoint = loop_begin + int(loop_size/2);
	pose.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint); // loop tree
	pose.set_allow_jump_move(1,false); // fixed stems
	if ( !dle_ns::freeze_bonds )
		pose.insert_ideal_bonds( loop_begin, loop_end ); // idealizing the loop

	counter = counter + 1; // to ensure filenames start from one

	// Making sure that backbone bonds and torsions are updated
	antibody.update_backbone_bonds_and_torsions();

	antibody.set_fullatom_flag( false );

	// Creating a copy of loop library
	std::vector< Fragment > loop_library_copy;
	loop_library_copy = loop_library;

	// Setting up loop fold tree
	nres = antibody.total_residue(); // update nres to account for deleted section
	cutpoint = loop_begin + 1;
	antibody.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint); // loop tree
	antibody.set_allow_jump_move(1,false); // fixed stems
	if ( !dle_ns::freeze_bonds )
		antibody.insert_ideal_bonds( loop_begin, loop_end ); // idealizing the loop

	// Allowing movements in loop backbone
	for(int i = loop_begin-1; i <= loop_end+1; i++)
			antibody.set_allow_bb_move( i, true );

	Pose dummy, dummy_copy;
	dummy = antibody;
	dummy_copy = antibody;

	std::stringstream filename_counter; // String holder to contain the counter converted from int to string
	filename_counter << counter; // int counter fed to string
	std::string cluster_table_filename;
	cluster_table_filename = pdb_out_path + protein_name + "_cluster_table_0" + filename_counter.str() + "_" + I( 4, 4, makepdb_number); // file is placed with output pdb
	std::ofstream cluster_table_file( cluster_table_filename.c_str(), std::ios::out); // open/create file
	std::string cluster_score_filename;
	cluster_score_filename = pdb_out_path + protein_name + "_cluster_score_0" + filename_counter.str() + "_" + I( 4, 4, makepdb_number); // file is placed with output pdb
	std::ofstream cluster_score_file( cluster_score_filename.c_str(), std::ios::out); // open/create file

	// print headers only in first file
	if(makepdb_number == 1) {
		int dummy_counter(0); // for printing header (first row) - Total no. of loop in library
		for ( std::vector< Fragment >::const_iterator it=loop_library.begin(),it_end = loop_library.end(); it != it_end; ++it ) {
			dummy_counter++;
		}

		// Printing Headers - First Row
		cluster_table_file << "pdbs  ";
		for(int i = 1; i <= dummy_counter; i ++)
			cluster_table_file << I ( 10, 4, i);
		cluster_table_file << std::endl;
	}

	int dummy_row_counter(0); // for printing header (first column elements)
	int filecounter(0);

	for ( std::vector< Fragment >::const_iterator it=loop_library.begin(),it_end = loop_library.end(); it != it_end; ++it ) {
		filecounter++;
		dummy_row_counter++;
		if(!(makepdb_number == filecounter))
			continue;
		if( filecounter > makepdb_number ){
			cluster_score_file.close();
			cluster_table_file.close();
			return;
		}
		dummy = antibody; // going back to initial starting pose
		it->insert( dummy, loop_begin ); // inserting the loop in the decoy
		it->insert( pose, loop_begin );

		// Commented out for speed optimization
		// For proper output pdb generation
		//FArray1D_bool int_res( nres, false );
		//for(int i=loop_begin; i<=loop_end; i++)
		//int_res(i) = true;
		// //pose.repack( int_res, true /*include_current*/ );
		// dle_pack_with_unbound( pose, int_res, true /*include_current*/ );

		// Scoring fullatom decoy
		Score_weight_map weight_map;
		setup_score_weight_map( weight_map, score4L );
		weight_map.set_weight( CHAINBREAK, 1.0 );
		weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
		cluster_score_file << I(4, 4, dummy_row_counter)
											 << "  "
											 << F(10, 5, dummy.score( weight_map ))
											 << std::endl;

		// Printing Headers - First Column
		cluster_table_file << I(4, 4, dummy_row_counter) << "  "; // print first column element for numbering

		for ( std::vector< Fragment >::const_iterator it_copy=loop_library_copy.begin(),it_copy_end = loop_library_copy.end(); it_copy != it_copy_end; ++it_copy ) {
				dummy_copy = antibody;
				it_copy->insert( dummy_copy, loop_begin ); // inserting the loop in the decoy
				float rmsg = dle_loop_rmsg( dummy, dummy_copy, loop_begin, loop_end );
				cluster_table_file << F(10, 5, rmsg);
			}
		cluster_table_file << std::endl;
	}

	cluster_score_file.close(); // close score file
	cluster_table_file.close(); // close rms table file

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dle_read_in_one_loop_library
///
/// @brief reads in a loop fragment file from the disk
///
/// @detailed the loopfile is just a whitespace separated phi-psi-omega angle
///           containing file over all the loop residues. Different conformations
///           of the loop are terminated with a blank line.
///
/// @param[in] loop_begin: sequentially numbered starting loop residue
///            loop_end: sequentially numbered loop end residue
///            counter: an unique number corresponding to each loop ensemble
///            loop_library: initially is blank but gets filled up
///            lib_refinement_status: type of loop library to be read in
///                                   refined or unrefined
///
/// @global_read files_paths
///
/// @global_write loop_library which finally gets reflected in dle_ns
///
/// @remarks
///
/// @references
///
/// @authors Aroop 10/19/2006
///
/// @last_modified Aroop 10/19/2006
/////////////////////////////////////////////////////////////////////////////////
void
dle_read_in_one_loop_library(
	int loop_begin,
	int loop_end,
  int counter,
	std::vector< Fragment > & loop_library,
	const std::string & lib_refinement_status
)
{
	using namespace docking;
	using namespace files_paths;
	using namespace runlevel_ns;

	std::vector<Fragment> unclustered_loop_library;

	counter = counter + 1; // Start at 1 instead of zero for easy readability

	std::stringstream filename_counter; // String holder to contain the counter converted from int to string
	filename_counter << counter; // int counter fed to string
	std::string loop_library_filename;
	if( lib_refinement_status == "unrefined" || lib_refinement_status == "unrefined_unclustered" )
		loop_library_filename = constraints_path  + protein_name + "_loop_library_0" + filename_counter.str(); // file is read in from where other contraints are supposed to exist
	else if( lib_refinement_status == "refined" )
		loop_library_filename = constraints_path  + protein_name + "_refined_loop_library_0" + filename_counter.str(); // file is read in from where other contraints are supposed to exist

	// Read the file defined by command line option
	utility::io::izstream loop_library_stream( loop_library_filename );

	// Check to see if file exists
	if ( !loop_library_stream ) {
		std::cout << "[Error]: Could not open loop library file: " << loop_library_filename << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	int loop_size = (loop_end - loop_begin) + 1;
	float phi(100.0), psi(100.0), omega(100.0);

	bool end_not_reached(true);
	while(end_not_reached){
		Fragment f( loop_size );
		for ( int i = 1; i <= loop_size; ++i ) {
			loop_library_stream >> phi >> psi >> omega >> std::skipws;
			if(loop_library_stream.eof()) {
				end_not_reached = false;
				break;
			}
			f.phi       ( i ) = phi;
			f.psi       ( i ) = psi;
			f.omega     ( i ) = omega;
			f.secstruct ( i ) = 'L';
		}
		if(end_not_reached)
			unclustered_loop_library.push_back( f );
	}

	loop_library_stream.close();
	loop_library_stream.clear();

	////////////////////////////////////////////////////////////////////////////
	// filter unclustered loop_library if cluster center file is available
	// else just return unclustered loop library
	////////////////////////////////////////////////////////////////////////////
	end_not_reached = true;
	std::string center_filename = constraints_path  + "centers";
	int cluster_counter(0);
	int decoy_number(0);
	int cluster_size(0);
	float cluster_score(0);

	// Read the file containing cluster information
	utility::io::izstream center_stream( center_filename );

	// Check to see if file exists
	if ( center_stream && lib_refinement_status == "unrefined" ) {
		while( end_not_reached ){
			center_stream >> cluster_counter >> decoy_number >> cluster_size >> cluster_score >> std::skipws;
			if(center_stream.eof()/*(cluster_counter == 0) && (decoy_number == 0)*/) {
				end_not_reached = false;
				continue;
			}
			Fragment f( loop_size );
			f = unclustered_loop_library[decoy_number-1];
			loop_library.push_back( f );
		}
	}
	else
		loop_library = unclustered_loop_library;

	center_stream.close();
	center_stream.clear();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_load_and_define_loops
///
/// @brief wrapper to load key residues and then define loop begin & end
///
/// @detailed
///
/// @param[in] none
///
/// @global_read dle_ns
///
/// @global_write dle_ns
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/11/2006
///
/// @last_modified Aroop 07/17/2006
///////////////////////////////////////////////////////////////////////////////
void
dle_load_and_define_loops()
{
	using namespace pose_ns;
	using namespace dle_ns;
	// Total key residues in file containing key residues
	int total_key_residues(0);
	int key_residue[DOCK_MAX_LOOPS*2];// Loops have a begin and end (hence 2)
	dle_load_key_residues(total_key_residues, key_residue);
	dock_flex_total_loops = total_key_residues / 2;

	dle_define_loop(total_key_residues, key_residue);
	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_build_one_loop
///
/// @brief actually builds each loop
///
/// @detailed uses two methods developed by phil to build the loops. Have been
///           modified a lot to incorporate customized settings. This calls to
///           heart of the program to build loops
///
/// @param[in] a pose, beginning of a loop, end of a loop, fragment vector to
///            store loop library
///
/// @global_read none (somewhere internally fragments are use)
///              frag_offset from dle_ns
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified 08/28/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_build_one_loop(
  pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
  std::vector< Fragment > & loop_library,
	int counter
)
{
	using namespace pose_ns;
	using namespace runlevel_ns;

	int nres( pose.total_residue() );
	// Storing backbone movements
	FArray1D_bool allow_bb_res( nres, false );
	for(int i = 1; i <= nres; i++)
		allow_bb_res(i) = pose.get_allow_bb_move(i);

	// agreement of frag file and pose numbering
	int frag_offset = dle_ns::loop_frag_offset;
	int local_loop_begin(loop_begin);
	int local_loop_end(loop_end);
	int local_loop_size = (loop_end - loop_begin) + 1;

	int modeled_loop_begin( loop_begin );
	int modeled_loop_end( loop_end );

	Pose dummy; // Temporary pose to work on
	dummy = pose;
	dummy.set_fullatom_flag( false ); // build loop in centroid mode
	if(docking::dle_loops_flag) {
		if(loop_begin < docking::part_end(1))
			dummy.delete_segment(docking::part_end(1)+1, nres);
		else{
			dummy.delete_segment(1, docking::part_end(1));
			local_loop_begin = loop_begin - docking::part_end(1);
			local_loop_end = loop_end - docking::part_end(1);
			frag_offset = frag_offset + docking::part_end(1);
			modeled_loop_begin = local_loop_begin;
			modeled_loop_end = local_loop_end;
		}
	}
	if( antibody_ns::antibody_build && dle_ns::current_loop_is_H3 ) {
		if( antibody_ns::antibody_build && (local_loop_size > 5) ) {
			modeled_loop_begin = (loop_begin + (antibody_ns::h3_nter_frag_size
																					- antibody_ns::nter));
			modeled_loop_end = (loop_end - (antibody_ns::h3_base_frag_size
																			- antibody_ns::base));
		}
	}
	int modeled_loop_size = ( modeled_loop_end - modeled_loop_begin ) + 1;

	//Allowing movements
	for(int i = modeled_loop_begin; i <= modeled_loop_end; i++)
		dummy.set_allow_bb_move( i, true );
	for(int i = loop_begin; i <= loop_end; i++)
		pose.set_allow_bb_move( i, true );

	nres = dummy.total_residue();
	int loop_size =  (local_loop_end - local_loop_begin) + 1;
	assert( local_loop_end+1 <= nres );
	int cutpoint(0);

	// setup a fold_tree with jump from local_loop_begin-1 to local_loop_end+1
	if( antibody_ns::h3_random_cut && dle_ns::current_loop_is_H3 )
			cutpoint = dle_choose_random_cutpoint( modeled_loop_begin,
																						 modeled_loop_end );
	else
		cutpoint = modeled_loop_begin + int((modeled_loop_size)/2);

	// stores cutpoint used during building
	// will be used in to compute N-C distance in scorefile
	if( dle_ns::current_loop_is_H3 )
		dle_ns::decoy_loop_cutpoint = cutpoint;
	dummy.one_jump_tree( nres, modeled_loop_begin-1, modeled_loop_end + 1,
											 cutpoint);
	dummy.set_allow_jump_move(1,false);

	//reset torsions and bonds in the loop region
	// Insert init phi-psi angles
	// insert_init_frag( dummy, modeled_loop_begin, modeled_loop_size );
	// Insert ideal bond lengths
	// if ( !dle_ns::freeze_bonds )
	//	dummy.insert_ideal_bonds( modeled_loop_begin, modeled_loop_end );

	// setup scoring
	Score_weight_map weight_map;
	// 4L with loop contact score
	if(antibody_ns::fv_linker_mode)
		setup_score_weight_map( weight_map, score4Lc );
	else
		setup_score_weight_map( weight_map, score4L ); // centroid loop score
	weight_map.set_weight( CHAINBREAK, 1.0 ); // penalty for gap
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 ); // penalty for non-overlap
	int n_cutpoints(1); // just one loop at a time
	FArray1D_float cut_weight( n_cutpoints, 1.0 );
	weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );
	// params for scored_frag_close
	int cycles1(10); // aroop_temp default 1000
	int cycles2(25 * loop_size ); // aroop_temp default 25 *
	if( runlevel_ns::benchmark) {
		cycles1 = 5;
		cycles2 = 10 * loop_size;
	}
	bool do_ccd_moves( true );

	// loop building using rosetta scoring function - slow
	if(dle_ns::dle_build ||  antibody_ns::antibody_build) {
		dle_scored_frag_close(weight_map, dummy, modeled_loop_begin,
													modeled_loop_end, cutpoint, frag_offset,
													cycles1,cycles2, counter, do_ccd_moves);

		if(docking::dle_loops_flag || antibody_ns::antibody_build) {
			nres = pose.total_residue();
			if( antibody_ns::antibody_build ) {
				pose.one_jump_tree( nres, modeled_loop_begin-1, modeled_loop_end+1,
														cutpoint );
				pose.copy_segment(modeled_loop_size, dummy, modeled_loop_begin,
													modeled_loop_begin);
			}
			FArray1D_bool allow_repack( nres, false );
			dle_loop_neighbor(pose,(loop_begin-antibody_ns::nter) - 5,
												(loop_end + antibody_ns::base) + 5,
												allow_repack);
			dle_extreme_repack( pose,
													1, // repack_cycles
													allow_repack,
													false, // rt_min
													false, // rotamer_trials
													true, // force_one_repack
													true ); // use_unbounds
		}
	}
	else {
		// default for loops of lenth six
		int frag_size(3);
		scored_frag_close( weight_map, dummy, local_loop_begin, local_loop_end,
											 frag_size, frag_offset, cycles1, cycles2, do_ccd_moves,
											 loop_library );
	}

	if(runlevel >= standard)
		std::cout << "scored_frag_close: found: " << loop_library.size()
							<< " frags" << std::endl;

	// Restoring backbone moves
	pose.set_allow_bb_move( allow_bb_res );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dle_relax_loop_library
///
/// @brief relaxes all loops in a loop library
///
/// @detailed wrapper to call the actual function to relax loops. The wrapper
///           enables each loop to be relaxed simultaneously in a cluster. Then a
///           loopfile containing phi-psi-omega angles over the loop residues are
///           generated for each loop residue. Each file gets outputed to a separate
///           file.
///
/// @param[in] loop defintions and the pose to which the definitions are applied
///
/// @global_read makepdb_number to enable parallel processing.
///
/// @global_write a file for each member in the loop ensemble containing phi-psi-omega
///               angles
///
/// @remarks
///
/// @references
///
/// @authors Aroop 11/29/2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dle_relax_loop_library(
 pose_ns::Pose & pose_in,
 int loop_begin,
 int loop_end,
 std::vector< Fragment > & crude_loop_library,
 int counter
)
{
	using namespace pose_ns;
	using namespace makepdb_decoy;
	using namespace files_paths;
	counter = counter + 1;

	Pose dummy, invariant_pose;
	dummy = pose_in;

	int nres( pose_in.total_residue() );
	int cutpoint( loop_begin + 1 );
	pose_in.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint); // loop tree
	pose_in.set_allow_jump_move(1,false); // fixed stems
	if ( !dle_ns::freeze_bonds )
		pose_in.insert_ideal_bonds( loop_begin, loop_end ); // idealizing the loop

	int local_loop_begin, local_loop_end;
	local_loop_begin = loop_begin;
	local_loop_end = loop_end;
	if(docking::dle_loops_flag) {
		if(loop_begin < docking::part_end(1))
			dummy.delete_segment(docking::part_end(1)+1, nres);
		else{
			dummy.delete_segment(1, docking::part_end(1));
			local_loop_begin = loop_begin - docking::part_end(1);
			local_loop_end = loop_end - docking::part_end(1);
		}
	}

	nres = dummy.total_residue();
	cutpoint = local_loop_begin + 1;
	dummy.one_jump_tree( nres, local_loop_begin-1, local_loop_end+1, cutpoint); // loop tree
	dummy.set_allow_jump_move(1,false); // fixed stems
	if ( !dle_ns::freeze_bonds )
		dummy.insert_ideal_bonds( local_loop_begin, local_loop_end ); // idealizing the loop
	invariant_pose = dummy;

	std::stringstream filename_counter; // String holder to contain the counter converted from int to string
	filename_counter << counter; // int counter fed to string
	std::string library_filename;
	library_filename = pdb_out_path + protein_name + "_refined_loop_0" + filename_counter.str() + "_" + I( 4, 4, makepdb_number); // file is placed with output pdb

	std::ofstream outfile( library_filename.c_str(), std::ios::out); // open/create file

	int filecounter(0);
	for ( std::vector< Fragment >::const_iterator it=crude_loop_library.begin(),it_end = crude_loop_library.end(); it != it_end; ++it ) {
		filecounter++;
		if(!(makepdb_number == filecounter))
			continue;
		if( filecounter > makepdb_number ){
			outfile.close();
			return;
		}
		it->insert( dummy, local_loop_begin );
		Score_weight_map weight_map;
		setup_score_weight_map( weight_map, score12 );
		dummy.score( weight_map );
		dummy.update_backbone_bonds_and_torsions();
		dle_loop_fa_relax( dummy, local_loop_begin, local_loop_end );
		for(int loop_res = local_loop_begin; loop_res <= local_loop_end; loop_res++)
			{
				outfile << F(10, 3, dummy.phi(loop_res))
								<< F(10, 3, dummy.psi(loop_res))
								<< F(10, 3, dummy.omega(loop_res))
								<< std::endl;

				pose_in.set_phi( loop_begin + (loop_res - local_loop_begin), dummy.phi(loop_res));
				pose_in.set_psi( loop_begin + (loop_res - local_loop_begin), dummy.psi(loop_res));
				pose_in.set_omega( loop_begin + (loop_res - local_loop_begin), dummy.omega(loop_res));
			}
		outfile<< std::endl; // print a new line after each loop
		outfile.flush();
		nres = pose_in.total_residue();
		FArray1D_bool int_res( nres, false );
		dle_loop_neighbor(pose_in, loop_begin, loop_end, int_res);
		// pose_in.repack( int_res, true );
		dle_pack_with_unbound( pose_in, int_res, true );
	}

	outfile.close();
	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dle_insert_loops
///
/// @brief helps to insert each loop into the protein
///
/// @detailed mainly useful if there are more than one loop to be built
///
/// @param[in] pose, translational magnitude for gaussian moves, rotational magnitudes
///            of gaussian moves
///
/// @global_read dock_loop, dock_flex_total_loops
///              dock_flex_loop_begin, dock_flex_loop_end
///              (from dle_ns)
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
dle_insert_loops(
  pose_ns::Pose & pose,
	float trans_mag,
	float rot_mag
)
{
	using namespace pose_ns;
	using namespace dle_ns;

	float accept_rate(0); // acceptance rate of gaussian moves to snuggle in protein

	if(docking::dle_loops_flag)
		{
			for(int i = 0; i < dock_flex_total_loops; i++)
				{
					accept_rate = accept_rate + dle_insert_one_loop(pose, dock_loop[i], dock_flex_loop_begin[i], dock_flex_loop_end[i], trans_mag, rot_mag);
				}
			return (accept_rate/dock_flex_total_loops); // average acceptance returned
		}
	else
		return (accept_rate);
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_insert_one_loop
///
/// @brief actually inserts the loops into the decoy
///
/// @detailed for a particular loop, inserts loop from the loop library already
///           built following each loop insertion, small gaussian moves are
///           executed to snuggle the proteins better
///
/// @param[in] pose, loop library, loop begin, loop end,
///            translational & rotational mag
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified 07/23/2007
///////////////////////////////////////////////////////////////////////////////
float
dle_insert_one_loop(
  pose_ns::Pose & pose,
	std::vector< Fragment > & loop,
	int const loop_begin,
	int const loop_end,
	float trans_mag,
	float rot_mag
)
{
	using namespace pose_ns;
	using namespace dle_ns;

	// calcuate rmsg for loop in scorefile
	dle_calc_rmsg_flag = true;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	Fold_tree const docking_tree( pose.fold_tree() );

	// Making sure that backbone bonds and torsions are updated
	pose.update_backbone_bonds_and_torsions();

	// Allowing movements
	for(int i = loop_begin-1; i <= loop_end+1; i++)
			pose.set_allow_bb_move( i, true );

	Score_weight_map weight_map( score4d );
	weight_map.set_weight( VDW, 1.0 );
	weight_map.set_weight( ENV, 1.0 );
	weight_map.set_weight( PAIR, 1.0 );
	int const nres( pose.total_residue() );
	assert( loop_end+1 <= nres );
	int cutpoint( loop_begin + 1 );

	Pose dummy; // Temporary pose to work on
	dummy = pose;
	dummy.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint); // loop tree
	dummy.set_allow_jump_move(1,false); // fixed stems
	if ( !dle_ns::freeze_bonds )
		dummy.insert_ideal_bonds( loop_begin, loop_end ); // idealizing the loop
	Pose fixed_pose; // Fixed pose that does not change
	fixed_pose = dummy; // Fixed pose instantiated with dummy

	float accept_rate(0); // acceptance rate for gaussian moves during snuggling
	// counter useful only when inserting one loop at a time & do a local refine
	int loop_counter(0);

	// Iteration to actually try out the different loops
	Monte_carlo loop_mc ( dummy, weight_map, 2.0 );
	for ( std::vector< Fragment >::const_iterator it=loop.begin(),
					it_end = loop.end(); it != it_end; ++it ) {
		loop_counter++;
		// insert one loop at a time and do a local refine
		if(	insert_all_loops_with_local_refine )
			if(loop_counter != makepdb_decoy::makepdb_number)
				continue;
		dummy = fixed_pose; // going back to initial starting pose
		it->insert( dummy, loop_begin ); // inserting the loop in the decoy
		dummy.score(weight_map);
		dummy.update_backbone_bonds_and_torsions();
		// insert one loop at a time and do a local refine
		if( !insert_all_loops_with_local_refine ) {
			dummy.set_fold_tree( docking_tree ); // change to docking tree
			dummy.set_allow_jump_move(1,true); //partners move relative to each other
			dummy.set_allow_bb_move( false );
			// slide partners to eliminate clashes
			dle_slide_into_contact( dummy );
			accept_rate = pose_docking_rigid_body_trial( dummy, 50, weight_map,
                    dummy.fullatom(), trans_mag, rot_mag );
			// updating backbone bonds and torsions
			dummy.update_backbone_bonds_and_torsions();
			// previous function might have fixed backbones of pose
			for(int i = loop_begin-1; i <= loop_end+1; i++)
				dummy.set_allow_bb_move( i, true );
			// loop tree
			dummy.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint);
			dummy.set_allow_jump_move(1,false); // stem residues are fixed
			loop_mc.boltzmann(dummy);
		}
	}
	if( insert_all_loops_with_local_refine )
		pose = dummy;
	else
		pose = loop_mc.low_pose(); // Choose the lowest scoring decoy ever observed

	// Restoring pose stuff
	pose.set_fold_tree( docking_tree ); // docking tree
	pose.set_allow_jump_move( jump_move_old );//restore the jump move
	pose.set_allow_bb_move( bb_move_old );
	pose.set_allow_chi_move( chi_move_old );

	return accept_rate; // return acceptance ratio of moves
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_cter
///
/// @brief building the cter from scratch
///
/// @detailed uses the fragments files - internally to build the cter, makes
///           gaussian moves at the end to snuggle proteins better
///
/// @param[in] pose, translational & rotational magnitudes
///
/// @global_read dock_cter (command line option): number of residues of cter to
///              be modelled, somewhere internall uses the fragments already
///              read in
///              frag_offset from dle_ns
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified 02/21/2006
///////////////////////////////////////////////////////////////////////////////
void
dle_cter(
  pose_ns::Pose & pose,
	float trans_mag,
	float rot_mag
)
{
	using namespace pose_ns;
	using namespace runlevel_ns;

	if(runlevel > standard)
		std::cout << "Building C-Terminal From Fragments File" << std::endl;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	Fold_tree const docking_tree( pose.fold_tree() ); // storing docking fold tree

	// Making sure that backbone bonds and torsions are updated
	pose.update_backbone_bonds_and_torsions(); // updating backbone bonds and torsions

	Pose dummy; // Temporary pose to work on
	dummy = pose;

	int const nres( pose.total_residue() ); // total residues
	int cter_size(0); // Size of cter to be modelled
	cter_size = dle_ns::cter_modeling_size; // command line dock_cter
	int cter_begin = nres - cter_size + 1;

	// Allowing movements in c-ter
	for(int i = cter_begin; i <= nres; i++)
			pose.set_allow_bb_move(i, true);

	// Specifying simple fold tree
	dummy.simple_fold_tree( nres );

	// Setting up scoring
	Score_weight_map weight_map;
	// ss build-up in unfolded chains
	setup_score_weight_map( weight_map, score1 );
	dummy.score( weight_map );
	Monte_carlo mc ( dummy, weight_map, 2.0 ); // temp = 2.0
	int const cycles ( 25 * cter_size );

	int frag_size(3);
	int frag_offset = dle_ns::loop_frag_offset;
	for ( int i = 1; i <= cycles; ++i ) {
		choose_offset_frag( frag_size, dummy, cter_begin, nres, frag_offset );
		mc.boltzmann( dummy );
	}

 	// recover the low pose
	pose = mc.low_pose();
	pose.update_backbone_bonds_and_torsions(); // updating backbone bonds and torsions
	pose.set_fold_tree( docking_tree );
	pose_docking_rigid_body_trial( pose, 50, weight_map, false/*fullatom*/, trans_mag, rot_mag );

	// Restoring pose stuff
	pose.set_fold_tree( docking_tree ); // Tree
 	pose.set_allow_jump_move( jump_move_old );//restore the jump move
	pose.set_allow_bb_move( bb_move_old );
	pose.set_allow_chi_move( chi_move_old );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_calc_loops_environ
///
/// @brief wrapper to pack neighboring residues of all modeled regions
///
/// @detailed calls function to repack loop residues, neighbors and cter
///           residues if modelled. Finally repacks all the identified residues
///
/// @param[in] pose
///
/// @global_read dock_flex_total_loops, dock_flex_loop_begin
///              dock_flex_loop_end ( from dle_ns )
///              command line option "dock_cter" read in again
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified 07/16/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_calc_loops_environ(
  pose_ns::Pose & pose,
	FArray1D_bool & allow_repack
)
{
	using namespace pose_ns;
	using namespace runlevel_ns;
	using namespace dle_ns;

	if(!docking::dle_loops_flag)
		return;

	if(runlevel > standard)
		std::cout << "Calculating flexible regions to be repacked" << std::endl;

	int const nres( pose.total_residue() ); // Total residues
	FArray1D_bool int_res( nres, false );
	int_res = allow_repack;

	for(int i = 0; i < dock_flex_total_loops; i++)
			dle_loop_neighbor(pose, dock_flex_loop_begin[i],
																			 dock_flex_loop_end[i], int_res);

	if(dle_cter_flag)
		{
			// define cter
			int cter_size(0); // Size of cter to be modelled
			// length of c-terminal to be modeled -command line -dock_cter
			cter_size = cter_modeling_size;
			int cter_begin = nres - cter_size + 1;
			int cter_end = nres;

			dle_loop_neighbor(pose, cter_begin, cter_end, int_res);
		}

	allow_repack = int_res; // defining residues to be repacked

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_loop_neighbor
///
/// @brief find neighbors within a specified threshold
///
/// @detailed Specifies a default neighboring residues within 8 Angstroms.
///           However command line option will allow one to change it. Finally
///           chooses all residues which are modelled and also any within the
///           neighboring shell
///
/// @param[in] pose, beginning of a modelled region, end of a modelled region
///            boolean array which will be updated to contain which residues
///            need to be repacked
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified 07/16/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_loop_neighbor(
	pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
	FArray1D_bool & allow_repack
)
{
	using namespace pose_ns;

	Loops one_loop;
	int loop_size = ( loop_end - loop_begin ) + 1;
	int cutpoint = loop_begin + int(loop_size/2);
	if( antibody_ns::h3_random_cut && antibody_ns::antibody_build &&
			!antibody_ns::min_base_relax &&
			(dle_ns::decoy_loop_cutpoint != 0))
		cutpoint = dle_ns::decoy_loop_cutpoint ;
	else if( antibody_ns::h3_random_cut )
		cutpoint = dle_choose_random_cutpoint(loop_begin, loop_end);
	one_loop.add_loop( loop_begin, loop_end, cutpoint,
										 dle_ns::loop_frag_offset, 0 /*extended*/);

	bool include_neighbors( true );
	pose_loops_select_loop_residues( pose, one_loop, include_neighbors,
																	 allow_repack );

	/*
	// Cutoff threshold for neighboring residues (Angstrom)
	float neighbor_cutoff(8.0);
	// Squaring - distance matrix is squared
	neighbor_cutoff = neighbor_cutoff * neighbor_cutoff;

	int const nres( pose.total_residue() ); // Total residues
	pose_update_cendist( pose ); // Updating C-alpha distance matrix
	// C-alpha distance matrix
	FArray2D_float const & cendist ( pose.get_2D_score( CENDIST ) );

	// Choosing self residues
	for ( int i = loop_begin; i <= loop_end; ++i )
		allow_repack(i) = true;

	// Choosing neighbors
	 // Loop over modelled residues only
	for ( int i = loop_begin; i <= loop_end; ++i ) {
		for ( int j = 1; j <= nres; ++j ) { // Checking with all other residues
			if ( i == j ) continue; // Ignore self
			// If both already true - ignore
			if ( allow_repack(i) && allow_repack(j) ) continue;
			if ( cendist(i,j) < neighbor_cutoff )
				allow_repack(i) = allow_repack(j) = true;
		}
	}
	*/

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_relax
///
/// @brief wrapper function which in turn relaxes the regions in question
///
/// @detailed calls the other function with each loop region.Finally also calls
///           the same function with the c-ter region (if it is modelled).
///
/// @param[in] pose
///
/// @global_read dock_flex_total_loops, dock_flex_loop_begin
///              dock_flex_loop_end (from docking_flexibile_ns)
///              dle_ns::dle_loops
///              command line option dock_cter gives the number of c-terminal
///              residues to be relaxed
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified 08/22/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_relax(
  pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	using namespace runlevel_ns;
	using namespace dle_ns;

	if( fullatom_loop_relax ) // command line option -dock_relax
		{

			if( antibody_ns::relax_dock ) {
				pose_docking_setup_dock_loop_tree( pose, pose.total_residue(),
																					 docking::part_end(1),
																					 dle_loops );
				return;
			}
			if(runlevel > standard)
				std::cout << "Relaxing Modelled Regions" << std::endl;

			// Number of residues at the C-terminal that are held fixed and not
			// relaxed. A sort of hack because we want to use the same loop relax
			// which has to have stem residues
			int fixed_cter_res(2);

			for(int i = 0; i < dock_flex_total_loops; i++) // Loop over loop regions
					dle_loop_fa_relax( pose, dock_flex_loop_begin[i],
																						dock_flex_loop_end[i] );

			// command line option: c-terminal region
			if(dle_cter_flag) {
				// define cter
				int const nres( pose.total_residue() ); // Total residues
				int cter_size(0); // Size of cter to be modelled
				// length of c-terminal to be modeled -command line -dock_cter
				cter_size = cter_modeling_size;
				int cter_begin = nres - cter_size + 1;
				int cter_end = nres - fixed_cter_res;
				dle_loop_fa_relax( pose, cter_begin, cter_end );
			}
		}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_loop_fa_relax
///
/// @brief actually relaxes the region specified
///
/// @detailed This is all done in high resolution.Hence there are no rigid body
///           moves relative to the docking partners. Only small moves are
///           carried out here to see if there are better fits. Repacking is
///           carried out extensively after each move.
///
/// @param[in] pose, loop begin position, loop end position
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified 07/18/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_loop_fa_relax(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
)
{
	using namespace pose_ns;
	using namespace files_paths;
	using namespace runlevel_ns;

	if(runlevel > standard)
		std::cout << "Actually relaxing modelled regions" << std::endl;

	// Making sure that backbone bonds and torsions are updated
	pose.update_backbone_bonds_and_torsions();

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	Fold_tree const tree_in( pose.fold_tree() ); // storing starting fold tree

	// minimizer params
	minimize_exclude_sstype( false, false );
	minimize_set_vary_phipsi( true );
	minimize_set_vary_chi( true );
	minimize_set_vary_omega( true ); // NOT omega, but everything else
	minimize_set_vary_rb_angle( true );
	minimize_set_vary_rb_trans( true );
	if( runlevel_ns::benchmark )
		minimize_set_tolerance( 1.0 );
	else
		minimize_set_tolerance( 0.001 );
	minimize_set_local_min( false, 0 ); // all non-move-list rsds minimized

	const int nres( pose.total_residue() ); // total residues

	Loops one_loop;
	int loop_size = ( loop_end - loop_begin ) + 1;
	int cutpoint = loop_begin + int(loop_size/2);
	if( dle_ns::current_loop_is_H3 ) {
		if( (antibody_ns::antibody_build || antibody_ns::antibody_refine ) &&
				!antibody_ns::min_base_relax && !antibody_ns::h3_random_cut &&
				(dle_ns::decoy_loop_cutpoint != 0))
			cutpoint = dle_ns::decoy_loop_cutpoint;
		else if( antibody_ns::h3_random_cut )
			cutpoint = dle_choose_random_cutpoint(loop_begin, loop_end);
	}
	else
		cutpoint = loop_begin + int( ((loop_end - loop_begin)+1) / 2);
	if( antibody_ns::snug_fit && docking::dle_loops_flag &&
			docking::docking_local_refine && docking::dle_flag ) {
		one_loop = dle_ns::dle_loops;
	}
	else
		one_loop.add_loop( loop_begin, loop_end, cutpoint,
											 dle_ns::loop_frag_offset, 0);
	FArray1D_bool allow_bb_move( nres , false );
	pose_loops_select_loop_residues( pose, one_loop, false, allow_bb_move );
	// pose.set_allow_bb_move( false ); // Turn bb moves off
	// enable bb moves for loop res
	// pose_loops_set_allow_bb_move( pose, one_loop );
	pose.set_allow_bb_move( allow_bb_move );

	// for H3 just change fold tree to incorporate loop & return
	if(docking::dle_flag && dle_ns::dle_model_H3 && !dle_ns::dle_refine &&
		 dle_ns::simultaneous_minimization) {
		pose_docking_setup_dock_loop_tree( pose, nres, docking::part_end(1),
																			 one_loop );
		return;
	}

	assert( files_paths::input_fa == true );
	// sets up stuff to use rosetta's fullatom energy function
	initialize_fullatom();
	// maximum number of rotamers to allow (buried, surface)
  set_rot_limit( 45, 27 );
	// maximum sum for the dunbrak prob (buried,surf)
	set_perc_limit( 0.9999, 0.9999 );
	// include extra rotamers in chi1/chi2/chi1aro
	if( !runlevel_ns::benchmark )
		design::active_rotamer_options.set_ex12( true, true, true);

  if ( !get_fullatom_flag() ) { // checking if old rosetta full atom flag is on
		std::cout << "fullatom poses only" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	//////////////////
	// setup fold_tree
	FArray1D_bool flank_allow_bb_move( allow_bb_move );
	if( dle_ns::current_loop_is_H3 && antibody_ns::flank_relax &&
			antibody_ns::freeze_h3 ) {
		using namespace antibody_ns;
		pose.one_jump_tree( nres, loop_begin - h3_flank - 1,
                        loop_end + h3_flank + 1, cutpoint );
		for( int i = 1; i <= nres; i++ )
			if( ( i >= ( loop_begin - h3_flank )) && ( i <= ( loop_end + h3_flank )))
				flank_allow_bb_move(i) = true;
	}
  else
		pose.one_jump_tree( nres, loop_begin-1, loop_end+1, cutpoint );
	pose.set_allow_jump_move(1,false);

	// setup scoring
	int n_cutpoints(1); // just one loop at a time
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );
	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D_float cut_weight( n_cutpoints, 1.0 );
	weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );

	{
		FArray1D_bool allow_repack( pose.total_residue(), false );
		pose_loops_select_loop_residues( pose, one_loop,
																		 true /*include_neighbors*/, allow_repack);
		pose.set_allow_chi_move( allow_repack );
		if( !runlevel_ns::benchmark )
			dle_extreme_repack( pose,
													1, // repack_cycles
													allow_repack,
													true, // rt_min
													true, // rotamer_trials
													false, // force_one_repack
													true ); // use_unbounds
	}

	// more params
	int n_small_moves ( std::max(5,loop_size/2) );
	int inner_cycles( loop_size );
	int outer_cycles( 1 );
	if( antibody_ns::antibody_refine || dle_ns::refine_input_loop)
		outer_cycles = 5;
	if( antibody_ns::antibody_refine && antibody_ns::snug_fit )
		outer_cycles = 2;
	if( runlevel_ns::benchmark ) {
		n_small_moves = 1;
		outer_cycles = 1;
		inner_cycles = 1;
	}
	score_set_try_rotamers( true );
	set_use_nblist(false);
	// minimize amplitude of moves if correct parameter is set
	if( antibody_ns::min_base_relax )
		set_smallmove_size( 0.5, 0.5, 1.0 ); // helix,strand,other default 2,2,3
	else
		set_smallmove_size( 2.0, 5.0, 6.0 ); // helix,strand,other default 2,2,3
	float const init_temp = { 2.0 };
	float const last_temp = { 0.5 };
	float const gamma = std::pow( (last_temp/init_temp), 1.0f/(inner_cycles) );
	//float const gamma = std::pow( (last_temp/init_temp), 1.0f/(inner_cycles) );
	float temperature = init_temp;

	/*
	// Use Chu's routine instead of standard
	{
		bool default_fast = loops_ns::fast;
		loops_ns::fast = true;
		pose_refine_loops_with_ccd( pose, weight_map, one_loop );
		loops_ns::fast = default_fast;
		return;
	}
	*/

	Monte_carlo mc( pose, weight_map, init_temp ); // monte carlo initialization
	mc.reset( pose ); // monte carlo reset

	bool relaxed_H3_found_ever(false);
	if(dle_ns::dle_H3_filter)
		relaxed_H3_found_ever = dle_CDR_H3_filter(pose,
														cdr_ns::cdr_h3_begin, (cdr_ns::cdr_h3_end -
                            cdr_ns::cdr_h3_begin) + 1);
	// ounter cycle
	for(int outer_cycle = 1; outer_cycle <= outer_cycles; outer_cycle++) {
		pose = mc.low_pose();
		int h3_attempts(0); // number of H3 checks after refinement

		// inner cycle
		for ( int j = 1; j <= inner_cycles; ++j ) {
			temperature *= gamma;
			mc.set_temperature( temperature );
			if( (j == 1) && (h3_attempts == 0) ) {
				pose.set_allow_bb_move( flank_allow_bb_move );
				pose.main_minimize( weight_map, "dfpmin" );
				pose.set_allow_bb_move( allow_bb_move );
				if(runlevel > standard) {
					score_set_try_rotamers( false ); // aroop_temp remove
					pose.copy_to_misc(); // aroop_temp remove
					// incorporated to make following debug line work
					std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
										<< " Cycle : " << I(4, 4, j)
										<< " Score : " << F(8, 2, pose.score(weight_map))
										<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
										<< " Move  : " << " min "
										<< std::endl; // aroop_temp remove
				}
			}
			pose_small_moves( pose, n_small_moves);
			if(runlevel > standard) {
				pose.copy_to_misc(); // aroop_temp remove
				// incorporated to make following debug line work
				std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
									<< " Cycle : " << I(4, 4, j)
									<< " Score : " << F(8, 2, pose.score(weight_map))
									<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
									<< " Move  : " << "small"
									<< std::endl; // aroop_temp remove
			}
			pose_shear_moves( pose, n_small_moves);
			if(runlevel > standard) {
				pose.copy_to_misc(); // aroop_temp remove
				// incorporated to make following debug line work
				std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
									<< " Cycle : " << I(4, 4, j)
									<< " Score : " << F(8, 2, pose.score(weight_map))
									<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
									<< " Move  : " << "shear"
									<< std::endl; // aroop_temp remove
			}
			ccd_moves( n_small_moves, pose, loop_begin, loop_end, cutpoint);
			//pose_ccd_close_loops( pose, one_loop );
			if(runlevel > standard) {
				pose.copy_to_misc(); // aroop_temp remove
				// incorporated to make following debug line work
				std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
									<< " Cycle : " << I(4, 4, j)
									<< " Score : " << F(8, 2, pose.score(weight_map))
									<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
									<< " Move  : " << " ccd "
									<< std::endl; // aroop_temp remove
				score_set_try_rotamers( true ); // aroop_temp remove
			}
			pose.set_allow_bb_move( flank_allow_bb_move );
			pose.main_minimize( weight_map, "dfpmin" );
			pose.set_allow_bb_move( allow_bb_move );
			if(runlevel > standard) {
				score_set_try_rotamers( false ); // aroop_temp remove
				pose.copy_to_misc(); // aroop_temp remove
				// incorporated to make following debug line work
				std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
									<< " Cycle : " << I(4, 4, j)
									<< " Score : " << F(8, 2, pose.score(weight_map))
									<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
									<< " Move  : " << " min "
									<< std::endl; // aroop_temp remove
				score_set_try_rotamers( true ); // aroop_temp remove
			}
			bool relaxed_H3_found_current(false);
			// H3 filter check
			if(dle_ns::dle_H3_filter &&
				 (h3_attempts <= inner_cycles)) {
				h3_attempts++;
				relaxed_H3_found_current = dle_CDR_H3_filter(pose,
					cdr_ns::cdr_h3_begin, (cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin)+1);
				if(runlevel > standard) {
					std::cout << "Ever : " << relaxed_H3_found_ever
										<< "\tCurrent : " << relaxed_H3_found_current
										<< std::endl; // aroop_temp remove
				}
				if( !relaxed_H3_found_ever && !relaxed_H3_found_current) {
					mc.boltzmann( pose );
				}
				else if( !relaxed_H3_found_ever && relaxed_H3_found_current ) {
					relaxed_H3_found_ever = true;
					mc.reset( pose );
				}
				else if( relaxed_H3_found_ever && !relaxed_H3_found_current ) {
					--j;
					continue;
				}
				else if( relaxed_H3_found_ever && relaxed_H3_found_current )
					mc.boltzmann( pose );
			}
			else
				mc.boltzmann( pose );

			if(runlevel > standard) {
				std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
									<< " Cycle : " << I(4, 4, j)
									<< " Accept: " << I(4, 4,pose.get_extra_score("MC_ACCEPTED"))
									<< " Move  : " << "backb"
									<< std::endl; // aroop_temp remove
			}

			if ( mod(j,20)==0 || j==inner_cycles ) {
				// repack trial
				// pose.repack( allow_repack, true /*include_current*/ );
				FArray1D_bool allow_repack( pose.total_residue(), false );
				pose_loops_select_loop_residues( pose, one_loop,
																				 true /*include_neighbors*/,
																				 allow_repack);
				pose.set_allow_chi_move( allow_repack );
				dle_pack_with_unbound( pose, allow_repack,
																							true /*include_current*/ );
				if(runlevel > standard) {
					pose.copy_to_misc(); // aroop_temp remove
					// incorporated to make following debug line work
					score_set_try_rotamers( false ); // aroop_temp remove
					std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
										<< " Cycle : " << I(4, 4, j)
										<< " Score : " << F(8, 2, pose.score(weight_map))
										<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
										<< " Move  : " << "repac"
										<< std::endl; // aroop_temp remove
					score_set_try_rotamers( true ); // aroop_temp remove
				}
				mc.boltzmann( pose );
				if(runlevel > standard) {
					std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
										<< " Cycle : " << I(4, 4, j)
										<< " Accept: " <<I(4,4,pose.get_extra_score("MC_ACCEPTED"))
										<< " Move  : " << "repac"
										<< std::endl; // aroop_temp remove
				}
			}
		} // inner cycles
	} // outer cycles
	pose = mc.low_pose();

	// minimize
	pose.set_allow_bb_move( flank_allow_bb_move );
	pose.main_minimize( weight_map, "dfpmin" );
	pose.set_allow_bb_move( allow_bb_move );
	pose.update_backbone_bonds_and_torsions();
	if(runlevel > standard) {
		score_set_try_rotamers( false ); // aroop_temp remove
		pose.copy_to_misc(); // aroop_temp remove
		// incorporated to make following debug line work
		std::cout << "Decoy : " << I(4, 4,  makepdb_decoy::makepdb_number)
							<< " Cycle : " << I(4, 4, 1111)
							<< " Score : " << F(8, 2, pose.score(weight_map))
							<< " RMS-G : " << F(8, 2, antibody_modeling_h3_rmsg())
							<< " Move  : " << " min "
							<< std::endl; // aroop_temp remove
		score_set_try_rotamers( true ); // aroop_temp remove
	}

	// turning off rotamer trials
	score_set_try_rotamers( false );

	// Restoring pose stuff
	pose.set_fold_tree( tree_in ); // Tree
	pose.set_allow_jump_move( jump_move_old );//restore the jump move
	pose.set_allow_bb_move( bb_move_old );
	pose.set_allow_chi_move( chi_move_old );

	return;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin dle_load_key_residues
///
/// @brief reads in data file containing the key residues - can be either pivot
///        or loop residues
///
/// @detailed
///
/// @param[in] total_key_residues which is the first line of the data file
///            key_residue which will subsequently contain the sequential
///            numbered key_residues after they have been read from data file
///
/// @global_read command line data file specified by "dock_keyres" flag
///              Format:
///              4                which is the total number of residues
///              101 A            start of loop 1
///              121 A            end of loop 1
///              90 B             start of loop 2
///              95 B             end of loop 2
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dle_load_key_residues(
  int & total_key_residues,
  int key_residue[]
)
{
	using namespace docking;
	using namespace files_paths;
	using namespace runlevel_ns;

	std::string filename; // File containing key residues

	filename = constraints_path + dle_ns::loop_def_filename.c_str();

	// Read the file defined by command line option
	utility::io::izstream dock_keyres_stream( filename );

	// Check to see if file exists
	if ( !dock_keyres_stream ) {
		std::cout << "[Error]: Could not open docking key residue file: " << dle_ns::loop_def_filename << std::endl;
		std::cout << "Not using docking key residue residues." << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::exit( EXIT_FAILURE );
	}

	// Read in the total number of pivots defined by user
	dock_keyres_stream >> total_key_residues >> std::skipws;
	if(runlevel > standard)
		std::cout << "Total Pivots = " << total_key_residues << std::endl;

	// Read in each pivot
	if(runlevel > standard)
		std::cout << "Using following key residues: " << std::endl;
	int key_res; // Pivot residue read in from each line of pivot_file
	char key_chain; // Corresponding residue chain read in from each line

	for ( int i = 0; i < total_key_residues; ++i ) {
	  dock_keyres_stream >> key_res >> key_chain >> std::skipws;
		res_num_from_pdb_res_num_chain( key_residue[i], key_res, key_chain );
		if(runlevel > standard)
			std:: cout << "Residue No.: " << key_res << "\tChain: " << key_chain << "\t(Rosetta/Pose Numbering: " << key_residue[i] << ")" << std::endl;
	}

	std::cout << "Loopfile read in." << std::endl;

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_define_loop
///
/// @brief translates crude data file data to loop_begin and loop_end data
///
/// @detailed
///
/// @param[in] total_key_residues (which is loop x 2 ) 2 for a begin and an end
///            key_residues an ordered set of consequtive loop begin and
///            loop_end res
///
/// @global_read dock_flex_total_loops,
///              dock_flex_loop_begin, dock_flex_loop_end (from dle_ns)
///
/// @global_write dock_flex_total_loops,
///               dock_flex_loop_begin, dock_flex_loop_end (from dle_ns)
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/23/2006
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
void
dle_define_loop(
  int & total_key_residues,
  int key_residue[]
)
{
	using namespace dle_ns;

	if(dock_flex_total_loops > DOCK_MAX_LOOPS)
		{
			std::cout << "Number of loops larger than DOCK_MAX_LOOPS" << std::endl;
			std::cout << "Please change in dle_ns.h" << std::endl;
			std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
			std::exit( EXIT_FAILURE );
		}

	assert((total_key_residues%2) == 0);
	for(int i = 0; i < (total_key_residues/2) ; i++)
		{
			dock_flex_loop_begin[i] = key_residue[i*2];
			dock_flex_loop_end[i] = key_residue[i*2 + 1];
			dle_loops.add_loop( key_residue[i*2], key_residue[i*2 + 1], key_residue[i*2] +1 /*cut*/, 0 /*offset*/, 1 /*extended*/);
		}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_is_adjacent
///
/// @brief returns true if two residues are near one another
///
/// @detailed just a debug measure to make sure that the parts of the same
///           protein do not move away
///
/// @param[in] input pose, two residue numbers which need to be adjacent
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/13/2006
///
/// @last_modified
///////////////////////////////////////////////////////////////////////////////
bool
dle_is_adjacent(
	pose_ns::Pose & pose,
	int const first_residue,
	int const second_residue
)
{
	using namespace pose_ns;

	int ca(2); // C-Alpha atom number in Eposition
	bool is_adjacent(false); // boolean to find out if residues supplied are adjacent
	float adjacent_upper_cutoff(6.5); // Upper cutoff threshold for adjacent residues (Angstrom)
	float adjacent_lower_cutoff(3.5); // Lower cutoff threshold for adjacent residues (Angstrom)
	FArray3D_float const pose_coord(pose.Eposition()); // grab the Eposition array from pose
	numeric::xyzVector_float first_ca(pose_coord(1,ca,first_residue)), second_ca(pose_coord(1,ca,second_residue)); // Coordinates of the C-Alpha of the first and second residue
  float ca_ca_dist( distance(first_ca, second_ca) ); // C-Alpha C-Alpha distance between the two residues

	if ( (ca_ca_dist > adjacent_lower_cutoff) && (ca_ca_dist < adjacent_upper_cutoff) )
		is_adjacent= true; // true if the two residues are within adjacent threshold

	std::cout << "Distance of separation (" << first_residue << "," << " " << second_residue << ") : " << ca_ca_dist << std::endl;

	return is_adjacent;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_rmsd_debug
///
/// @brief calculates rmsd over C-Alpha atoms which are not flexible.
///        Is also responsible for dumping_coords
///
/// @detailed Normally  when we dump coords,  we  trigger a refold. I guess one
///           way to get around this is to copy  the  pose to  be  dumped  to a
///           temporary pose. Then we can dump  the  temporary  pose. This  may
///           ensure that we do not refold the original pose. Also this routine
///           computes  the  rmsd  for  C-Alpha  atoms which are neither in the
///           flexible loop regions nor  are  they  in  they  in  the  flexible
///           c-terminal  region. Finally the system is  forced  to exit if the
///           rmsd is more than 0.05 A.
///
/// @param[in] input pose,reference pose,docking_tree, filename of dumping pose
///
/// @global_read dock_flex_total_loops,dock_flex_loop_begin, dock_flex_loop_end
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/18/2006
///
/// @last_modified 09/20/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_rmsd_debug(
  pose_ns::Pose & pose,
	pose_ns::Pose & fixed_pose,
	pose_ns::Fold_tree const docking_tree,
	const std::string & dump_filename
)
{
	using namespace pose_ns;
	using namespace dle_ns;

	// Dumping phase
	// pose that actually gets dumped to prevent refolding in original pose
	Pose dumper;
	dumper = pose; // dumper being assigned the pose to dump
	dumper.dump_pdb( dump_filename); // dumper dumping pdb

	fixed_pose.dump_pdb( "fixed_" + dump_filename);

	// Calculating rms
	int const nres( dumper.total_residue() ); // Total residues
	FArray1D_bool superimposable_res( nres, false ); // residues to superimpose

	int const dock_jump = 1; // docking rigid-body jump is always 1
	docking_tree.partition_by_jump( dock_jump, superimposable_res );

	// Inverting true and false
	for(int i = 1; i <= nres; i++) {
		if( superimposable_res(i) == true )
			superimposable_res(i) = false;
		else
			superimposable_res(i) = true;
	}

	// Making loop residues non-superimposable
	for(int i = 0; i < dock_flex_total_loops; i++) {
		for(int j = 1; j <= nres; j++)
			{
				if( (j >= (dock_flex_loop_begin[i] - 1))
						&& ( j <=  (dock_flex_loop_end[i] + 1) ) )
					superimposable_res(j) = false;
			}
	}


	if(dle_cter_flag) {
		int cter_size(0);
		// length of c-terminal to be modeled -command line -dock_cter
		cter_size = cter_modeling_size;
		int cter_begin = nres - cter_size + 1;
		int cter_end = nres;

		for(int j = cter_begin; j <= cter_end; j++)
			superimposable_res(j) = false;
	}
	// compute rms
	float rms (CA_rmsd_by_subset( dumper, fixed_pose, superimposable_res ));
	std::cout << "Rmsd of interest: " << rms << std::endl;

	if(rms > 0.005)
		std::exit( EXIT_FAILURE ); // aroop_temp remove

	return;
}
///////////////////////////////////////////////////////////////////////////////
/// @begin dle_repulsive_ramp
///
/// @brief ramping up the fullatom repulsive weight slowly to allow the
///        partners to relieve clashes and make way for each other
///
/// @detailed This routine is specially targetted to the coupled optimization
///           of docking partners and the loop region.The loop  modelling & all
///           previous  steps  involve mainly  centroid  mode .On switching  on
///           fullatom mode, one is bound to end up with clashes.To relieve the
///           clashes, it is essential to slowly  dial up the  repulsive weight
///           instead of turning it on to the maximum  value in one single step
///
/// @param[in] input pose which is assumed to have a docking fold tree
///
/// @global_read FA_REP : fullatom repulsive weight
///
/// @global_write FA_REP ( However,its reset to the original value at the end )
///
/// @remarks A particular portion is  commented out,which can be uncommented if
///          one  uses a  low  resolution  homology  model.Check details in the
///          beginning of the commented out region
///
/// @references
///
/// @authors Aroop 04/27/2006
///
/// @last_modified 02/05/2008
///////////////////////////////////////////////////////////////////////////////
void
dle_repulsive_ramp(
  pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	using namespace dle_ns;
	using namespace runlevel_ns;

	pose.update_backbone_bonds_and_torsions();

	// Turning off backbone movements for all residues
	pose.set_allow_bb_move( false );
	Fold_tree f( pose.fold_tree() );
	int num_fold_tree_cutpoint( f.get_num_fold_tree_cutpoint() );
	// enabling for loop residues
	if(docking::dle_loops_flag && (num_fold_tree_cutpoint > 1)
		 && dle_ns::simultaneous_minimization )
		pose_loops_set_allow_bb_move( pose,
		  dle_ns::dle_loops);

	Score_weight_map weight_map;
	// full monte_carlo minimize
	setup_score_weight_map( weight_map, score10d_min );
	pose.score( weight_map );
	// aroop: modify scorefxn to incorporate chain breaks if loops present
	if(docking::dle_loops_flag && (num_fold_tree_cutpoint > 1)
		 && dle_ns::simultaneous_minimization)
		dle_modify_weightmap( pose, weight_map);
	pose.score( weight_map );

	// dampen fa_rep weight
	float rep_weight_max = weight_map.get_weight( FA_REP );
	int rep_ramp_cycles(3); // CHANGE TO 5 !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
	int cycles(4);
	float minimization_threshold(15.0);
	float func_tol(1.0);
	if( antibody_ns::antibody_refine && antibody_ns::snug_fit )
		rep_ramp_cycles = 3;
	if( runlevel_ns::benchmark ) {
		rep_ramp_cycles = 1;
		cycles = 1;
		minimization_threshold = 150.0;
		func_tol = 10.0;
	}
	float rep_ramp_step = (rep_weight_max - 0.02) / float(rep_ramp_cycles-1);
	for ( int i = 1; i <= rep_ramp_cycles; i++ ) {
		float rep_weight = 0.02 + rep_ramp_step * float(i-1);
		weight_map.set_weight( FA_REP, rep_weight );
		pose_docking_monte_carlo_minimize( pose, cycles, "dfpmin_atol", weight_map,
			0.1, 2.0, 15.0, 1.0 );
		pose.copy_to_misc();
	}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_first_loop_rmsg
///
/// @brief compute the rms global for the first loop in input loopfile
///
/// @detailed
///
/// @param[in]
///
/// @global_read docking::part_end(1)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/11/2006
///
/// @last_modified 12/20/2007
///////////////////////////////////////////////////////////////////////////////
float
dle_first_loop_rmsg()
{
	using namespace pose_ns;
	using namespace dle_ns;

	if(!dle_calc_rmsg_flag)
		return -1.00;
	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose;

	//copy misc to pose
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( get_docking_fullatom_flag() );
  pose_from_misc( pose, fullatom, ideal_pos, coords_init );

	int loop_begin(0);
	int loop_end(0);
	loop_begin = dle_loops.begin()->start();
	loop_end = dle_loops.begin()->stop();

	float loop_rmsg = dle_loop_rmsg(pose, loop_begin, loop_end);

	pose.copy_to_misc(); // copy pose back to misc
	// set pose flag back to original setting
	pose_flag_ns::pose_flag_setting =original_pose_flag_setting;

	return loop_rmsg ; // return loop global rmsg
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_loop_rmsg
///
/// @brief compute the rms global for a loop between input pose & native
///
/// @detailed
///
/// @param[in]
///
/// @global_read docking::part_end(1)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/27/2006
///
/// @last_modified 05/02/2006
/////////////////////////////////////////////////////////////////////////////////
float
dle_loop_rmsg(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
)
{
	using namespace pose_ns;

	Pose native_pose, orientable_pose;
	orientable_pose = native_pose = pose; // assuming same sequence

	native_pose.set_coords( false, native::native_Eposition, native::native_coord );
	native_pose.update_backbone_bonds_and_torsions();

	int const nres( pose.total_residue() ); // Total residues

	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(2); // Protein broken into two alignable regions
                                // by one loop
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );


	if(loop_begin < docking::part_end(1)) // loop in partner 1
		{
			alignstart(1) = 1;
			alignend(1) = loop_begin - 1;
			alignstart(2) = loop_end + 1;
			alignend(2) = docking::part_end(1);
			/*
			for( int i = 0; i < aligned_regions; ++i ){
				if(i == 0)
					{
						alignstart(i+1) = 1;
						alignend(i+1) = loop_begin - 1;
					}
				else
					{
						alignstart(i+1) = loop_end + 1;
						alignend(i+1) = docking::part_end(1);
						}
			}
			*/
		}
	else // loop in partner 2
		{
			alignstart(1) = docking::part_end(1) + 1;
			alignend(1) = loop_begin - 1;
			alignstart(2) = loop_end + 1;
			alignend(2) = nres;
			/*
			for( int i = 0; i < aligned_regions; ++i ){
				if(i == 0)
					{
						alignstart(i+1) = docking::part_end(1) + 1;
							alignend(i+1) = loop_begin - 1;
					}
				else
					{
						alignstart(i+1) = loop_end + 1;
						alignend(i+1) = nres;
					}
			}
			*/
		}

	// copy coords into double arrays
	FArray3D_float full1 ( orientable_pose.full_coord() );
	FArray3D_float Epos1 ( orientable_pose.Eposition() );
	FArray3D_float const & Epos2 ( native_pose.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms;
	orient_region(nres,Epos1,full1,Epos2,aligned_regions,
	 alignstart,alignend,aligned_regions,alignstart,alignend,local_rms);

	orientable_pose.set_coords( false /*ideal_pos*/, Epos1, full1 );
	orientable_pose.update_backbone_bonds_and_torsions(); // aroop_temp maybe

	FArray3D_float const orientable_pose_coord(orientable_pose.Eposition()); // grab the Eposition array from orientalbe pose
	FArray3D_float const native_pose_coord(native_pose.Eposition()); // grab the Eposition array from native pose

	float loop_rmsg(0.00); // Global Rms of Loop
	int count(0);
	int CA(2); // CA identifier in Eposition

	for ( int i=loop_begin; i <= loop_end; ++i ) {
		++count;
		loop_rmsg += distance_squared(numeric::xyzVector_float(&orientable_pose_coord(1,CA,i)), numeric::xyzVector_float(&native_pose_coord(1,CA,i)));
	}

	loop_rmsg  = std::sqrt( loop_rmsg  / count  );

	return loop_rmsg;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_loop_rmsg
///
/// @brief compute the rms global for a loop between two input poses
///
/// @detailed
///
/// @param[in]
///
/// @global_read docking::part_end(1)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/27/2006
///
/// @last_modified 02/14/2006
/////////////////////////////////////////////////////////////////////////////////
float
dle_loop_rmsg(
	const pose_ns::Pose & pose_decoy1,
	const pose_ns::Pose & pose_decoy2,
	int const loop_begin,
	int const loop_end
)
{
	using namespace pose_ns;

	int const nres( pose_decoy1.total_residue() ); // Total residues

	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(2); // Protein broken into two alignable regions
                                // by one loop
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );


	alignstart(1) = 1;
	alignend(1) = loop_begin - 1;
	alignstart(2) = loop_end + 1;
	alignend(2) = nres;

	// copy coords into double arrays
	FArray3D_float full1 ( pose_decoy2.full_coord() );
	FArray3D_float Epos1 ( pose_decoy2.Eposition() );
	FArray3D_float const & Epos2 ( pose_decoy1.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms;
	orient_region(nres,Epos1,full1,Epos2,aligned_regions,
	 alignstart,alignend,aligned_regions,alignstart,alignend,local_rms);

	Pose decoy2_copy; // copy needed for reorienting, cannot re-orient input
	                  // decoy2, have enforced const decoy2 to prevent
                    // accidental re-orient

	decoy2_copy = pose_decoy2;

	decoy2_copy.set_coords( false /*ideal_pos*/, Epos1, full1 );
	decoy2_copy.update_backbone_bonds_and_torsions(); // aroop_temp maybe

	FArray3D_float const decoy2_copy_coord(decoy2_copy.Eposition()); // grab the Eposition array from orientalbe pose
	FArray3D_float const pose_decoy1_coord(pose_decoy1.Eposition()); // grab the Eposition array from native pose

	float loop_rmsg(0.00); // Global Rms of Loop
	int count(0);
	int CA(2); // CA identifier in Eposition

	for ( int i=loop_begin; i <= loop_end; ++i ) {
		++count;
		loop_rmsg += distance_squared(numeric::xyzVector_float(&decoy2_copy_coord(1,CA,i)), numeric::xyzVector_float(&pose_decoy1_coord(1,CA,i)));
	}

	loop_rmsg  = std::sqrt( loop_rmsg  / count  );

	return loop_rmsg;
}


///////////////////////////////////////////////////////////////////////////////
/// @begin dle_loop_rmsl
///
/// @brief compute the rms local for a loop
///
/// @detailed
///
/// @param[in]
///
/// @global_read docking::part_end(1)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/27/2006
///
/// @last_modified 05/02/2006
/////////////////////////////////////////////////////////////////////////////////
float
dle_loop_rmsl(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
)
{
	using namespace pose_ns;

	Pose native_pose, orientable_pose;
	orientable_pose = native_pose = pose; // assuming same sequence

	native_pose.set_coords( false, native::native_Eposition, native::native_coord );
	native_pose.update_backbone_bonds_and_torsions();

	int const nres( pose.total_residue() ); // Total residues

	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	int const aligned_regions(1); // Just the loop is alignable
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );

	alignstart(1) = loop_begin;
	alignend(1) = loop_end;

	// copy coords into double arrays
	FArray3D_float full1 ( orientable_pose.full_coord() );
	FArray3D_float Epos1 ( orientable_pose.Eposition() );
	FArray3D_float const & Epos2 ( native_pose.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms;
	orient_region(nres,Epos1,full1,Epos2,aligned_regions,
	 alignstart,alignend,aligned_regions,alignstart,alignend,local_rms);

	orientable_pose.set_coords( false /*ideal_pos*/, Epos1, full1 );
	orientable_pose.update_backbone_bonds_and_torsions(); // aroop_temp maybe

	FArray3D_float const orientable_pose_coord(orientable_pose.Eposition()); // grab the Eposition array from orientalbe pose
	FArray3D_float const native_pose_coord(native_pose.Eposition()); // grab the Eposition array from native pose

	float loop_rmsl(0.00); // Local Rms of Loop
	int count(0);
	int CA(2); // CA identifier in Eposition
	for ( int i=loop_begin; i <= loop_end; ++i ) {
		++count;
		loop_rmsl += distance_squared(numeric::xyzVector_float(&orientable_pose_coord(1,CA,i)), numeric::xyzVector_float(&native_pose_coord(1,CA,i)));
	}
	loop_rmsl  = std::sqrt( loop_rmsl  / count  );

	return loop_rmsl;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_CDR_H3_check
///
/// @brief checks to see if a loop if inserted into a protein(pose) passes the
///        H3 filter or not
///
/// @detailed Takes the phi-psi-omega angles from loop_pose and sets them into
///           the actual protein. Then this new pose with the loop in it is
///           tested for H3 base compatibility
///
/// @param[in] pose: full actual protein
///            loop_pose: a small pose just containing information about loop
///                       residues
///            loop_begin: seq numbered loop begin corresponding to pose
///            size: size of loop to compute loop_end
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 01/31/2007
/////////////////////////////////////////////////////////////////////////////////
bool
dle_CDR_H3_check(
  const pose_ns::Pose & pose,
	const pose_ns::Pose & loop_pose,
	int const loop_begin,
	int const size
)
{
	using namespace pose_ns;

	bool is_H3(false); // Check for loop conforming to H3 rules

	Pose dummy;
	dummy = pose;

	int loop_end =  loop_begin + size - 1; // loop_end

	for(int ii = loop_begin - 1; ii <= loop_end + 1; ii++)
		{
			dummy.set_phi(ii, loop_pose.phi(ii - loop_begin + 2));
			dummy.set_psi(ii, loop_pose.psi(ii - loop_begin + 2));
			dummy.set_omega(ii, loop_pose.omega(ii - loop_begin + 2));
			dummy.set_secstruct(ii, loop_pose.secstruct(ii - loop_begin + 2));
		}

	is_H3 = dle_CDR_H3_filter(dummy, loop_begin, size);

	return is_H3;
}
///////////////////////////////////////////////////////////////////////////////
/// @begin dle_CDR_H3_filter
///
/// @brief tests if a loop has H3 like base charachteristics
///
/// @detailed Uses the Shirai rules to find out if the dihedral angle formed by
///           CA atoms of residues n-2,n-1,n and n+1 conform to a kinked/extended
///           structure in accordance with the sequence. If there is a match, a
///           true value is returned
///
/// @param[in] pose: full actual protein
///            loop_begin: seq numbered loop begin corresponding to pose
///            size: size of loop to compute loop_end
///
/// @global_read reads -command line flag -base stored in dle_ns
///              to determine to do the complete H3 filter check or just do a
///              prediction of the H3 base type based on the aforementioned
///              dihedral angle
///
/// @global_write
///
/// @remarks
///
/// @references Structural classification of CDR-H3 in antibodies
///             Hiroki Shirai, Akinori Kidera, Haruki Nakamura
///             FEBS Letters 399 (1996) 1-8
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 01/31/2007
/////////////////////////////////////////////////////////////////////////////////
bool
dle_CDR_H3_filter(
  const pose_ns::Pose & pose,
	int const loop_begin,
	int const size
)
{
	using namespace pose_ns;
	using namespace dle_ns;

	// Values read from plot in reference paper. Fig 1 on Page 3
	// Values adjusted to match data from antibody training set
	float const kink_lower_bound = -10.00; // Shirai: 0
	float const kink_upper_bound = 70.00; // Shirai: 70
	float const extended_lower_bound = 125.00; // Shirai: ~180
	float const extended_upper_bound = 185.00; // Shirai: ~180

	//	float const h_bond(3.9); // Hydrogen Bond maximum value is 3.9 Angstroms - not used
	//	float const s_bridge(4.0); // Salt Bridge maximum value is 2.0 Angstroms - not used

	Pose h3_loop;
	bool is_kinked(false);
	bool is_extended(false);
	bool is_H3(false);

	// chop out the loop
	h3_loop.simple_fold_tree( size+3);
	h3_loop.copy_segment( size+3, pose, 1,loop_begin-2);

	// extract 3 letter residue codes for the chopped loop
	std::vector <std::string> aa_name; // loop residue 3 letter codes
	std::string aa_3let_code; // amino acid 3 letter code
	for(int ii = 1; ii <= size+3; ii++)	{
		name_from_num( h3_loop.res(ii), aa_3let_code);
		aa_name.push_back( aa_3let_code );
	}

	const FArray3D_float & h3_loop_coord( h3_loop.Eposition() );
	int const CA(2);   // CA atom position in full_coord array
	float base_dihedral(0.0); // base dihedral angle to determine kinked/extended conformation
	// calculate dihedral angle
	dihedral_bk(h3_loop_coord(1,CA,aa_name.size()),
							h3_loop_coord(1,CA,aa_name.size()-1),
							h3_loop_coord(1,CA,aa_name.size()-2),
							h3_loop_coord(1,CA,aa_name.size()-3),
							base_dihedral);

	// setting up pseudo-periodic range used in extended base computation
	if( base_dihedral < kink_lower_bound )
		base_dihedral = base_dihedral + 360.00;


	// extreneous portion to output H3 base dihedral only if -base flag is set
	if( antibody_ns::H3_base_mode) {
		std::cout << "Base Dihedral: " << base_dihedral << std::endl;
		std::string base;
		if((base_dihedral > kink_lower_bound) && (base_dihedral < kink_upper_bound))
			base = "KINK";
		else if((base_dihedral > extended_lower_bound) && (base_dihedral < extended_upper_bound))
			base = "EXTENDED";
		else
			base = "NEUTRAL";
		std::string library_filename;
		library_filename = files_paths::pdb_out_path + files_paths::protein_name + ".H3_base"; // file is placed with output pdb
		std::ofstream outfile( library_filename.c_str(), std::ios::out); // open/create file
		outfile << pdb_filename << ".pdb"
						<< F(10, 4, base_dihedral) << " "
						<< base
						<< std::endl;
		outfile.close();
		return( true );
	}


	// Rule 1a for standard kink
	if ((aa_name[aa_name.size()-3] != "ASP") && (aa_name[aa_name.size()-1] == "TRP"))	{
		if((base_dihedral > kink_lower_bound) && (base_dihedral < kink_upper_bound))
			{
				// std::cout << "KINK Found" << std::endl; // aroop_temp remove
				is_kinked = true;
				is_H3 = true;
			}
	}

	// Rule 1b for standard extended form
	if ((aa_name[aa_name.size()-3] == "ASP") && ( (aa_name[1] != "LYS") && (aa_name[1] != "ARG") ) && (is_H3 != true)) {
		if((base_dihedral > extended_lower_bound) && (base_dihedral < extended_upper_bound)) {
			// std::cout << "EXTENDED Found" << std::endl; // aroop_temp remove
			is_extended = true;
			is_H3 = true;
		}

		if(!is_H3) {
			// Rule 1b extension for special kinked form
			bool is_basic(false); // Special basic residue exception flag
			for(int ii = 2; ii <= int(aa_name.size() - 5); ii++) {
				if(aa_name[ii] == "ARG" || aa_name[ii] == "LYS") {
					is_basic = true;
					break;
				}
			}

			if(!is_basic) {
				int rosetta_number_of_L49;
				Pose misc_pose; // horrible hack to ensure res_num_from_pdb works
				// saves current misc into misc_pose
				// copies full pose into misc
				// computes residues
				// puts pack misc_pose into misc
				bool const ideal_pos( false );
				bool const coords_init( true );
				bool const fullatom( false );
				pose_from_misc( misc_pose, fullatom, ideal_pos, coords_init );
				pose.copy_to_misc();
				res_num_from_pdb_res_num_chain( rosetta_number_of_L49, 49, light_chain);
				std::string let3_code_L49;
				name_from_num( pose.res(rosetta_number_of_L49), let3_code_L49);
				misc_pose.copy_to_misc();
				if( let3_code_L49 == "ARG" || let3_code_L49 == "LYS")
					is_basic = true;
			}
			if( is_basic && (base_dihedral > kink_lower_bound) && (base_dihedral < kink_upper_bound)) {
				// std::cout << "KINK (special 1b) Found" << std::endl; // aroop_temp remove
				is_kinked = true;
				is_H3 = true;
			}
		}
	}

	// Rule 1c for kinked form with salt bridge
	if ((aa_name[aa_name.size()-3] == "ASP") && ( (aa_name[1] == "LYS") || (aa_name[1] == "ARG")) && ( (aa_name[0] != "LYS") && (aa_name[0] != "ARG") ) && (is_H3 != true) ) {
		if((base_dihedral > kink_lower_bound) && (base_dihedral < kink_upper_bound)) {
			// std::cout << "KINK (w sb) Found" << std::endl; // aroop_temp remove
			is_kinked = true;
			is_H3 = true;
		}
		if(!is_H3) {
			bool is_basic(false); // Special basic residue exception flag
			int rosetta_number_of_L46;
			Pose misc_pose; // horrible hack to ensure res_num_from_pdb works
			// saves current misc into misc_pose
			// copies full pose into misc
			// computes residues
			// puts pack misc_pose into misc
			bool const ideal_pos( false );
			bool const coords_init( true );
			bool const fullatom( false );
			pose_from_misc( misc_pose, fullatom, ideal_pos, coords_init );
			pose.copy_to_misc();
			res_num_from_pdb_res_num_chain( rosetta_number_of_L46, 46, light_chain );
			std::string let3_code_L46;
			name_from_num( pose.res(rosetta_number_of_L46), let3_code_L46);
			misc_pose.copy_to_misc();
			if( let3_code_L46 == "ARG" || let3_code_L46 == "LYS")
				is_basic = true;
			if( is_basic && (base_dihedral > extended_lower_bound) && (base_dihedral < extended_upper_bound)) {
				// std::cout << "EXTENDED (special 1c) Found" << std::endl; // aroop_temp remove
				is_extended = true;
				is_H3 = true;
			}
		}
	}

	// Rule 1d for extened form with salt bridge
	if ((aa_name[aa_name.size()-3] == "ASP") && ( (aa_name[1] == "LYS") || (aa_name[1] == "ARG")) && ( (aa_name[0] == "LYS") || (aa_name[0] == "ARG") ) && (is_H3 != true) ) {
		if((base_dihedral > extended_lower_bound) && (base_dihedral < extended_upper_bound)) {
			//std::cout << "EXTENDED (w sb) Found" << std::endl; // aroop_temp remove
			is_extended = true;
			is_H3 = true;
		}
	}

	/*

	float stem_dihedral(0.0); // stem dihedral angle to determine K(G) type of stem
	// calculate dihedral angle
	dihedral_bk(h3_loop_coord(1,CA,aa_name.size()-2),
							h3_loop_coord(1,CA,aa_name.size()-3),
							h3_loop_coord(1,CA,aa_name.size()-4),
							h3_loop_coord(1,CA,aa_name.size()-5),
							stem_dihedral);

	//std::cout << "Stem Dihedral: " << stem_dihedral << std::endl;

	// Rule 2
	if( is_kinked && (aa_name[aa_name.size()-5] == "GLY") )
		std::cout << "Stem Dihedral : " << stem_dihedral << std::endl;

	*/

	return is_H3;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_slide_into_contact
///
/// @brief brings two proteins into the point of glancing contact
///
/// @detailed If two proteins are overlapping, then they are moved away till the
///           vdw score drops below a certain value. If they are far away again
///           they are bought closer till a certain vdw value is crossed.
///
/// @param[in] pose: full actual protein
///
/// @global_read assumes a docking type jump tree in place
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 01/31/2007
/////////////////////////////////////////////////////////////////////////////////
void
dle_slide_into_contact(
 pose_ns::Pose & pose
)
{
	using namespace pose_ns;

	int const dock_jump( 1 ); // jump number associated with docking
	pose_ns::Jump perturbed_jump( pose.get_jump( dock_jump ) ); // the docking jump
	int const pos1( pose.fold_tree().get_jump_point()(1, dock_jump) ); // center of mass of partner 1
	int const pos2( pose.fold_tree().get_jump_point()(2, dock_jump) ); // center of mass of partner 2

	// slide_into_contact
	float const step_size( 1.0 ); // Angstrom
	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score4d );

	// first try moving away from each other
	pose.score( weight_map );
	const FArray3D_float & Epos( pose.Eposition() );
	numeric::xyzVector_double trans_axis (
																				numeric::xyzVector_double( &Epos(1,2,pos2) ) -
																				numeric::xyzVector_double( &Epos(1,2,pos1) ) );

	while( pose.get_0D_score( DOCK_VDW ) > 0.1 ) {
		perturbed_jump.translation_along_axis( Epos(1,1,pos1), trans_axis, step_size );
		pose.set_jump( dock_jump, perturbed_jump );
		pose.score( weight_map );
	}

	// then try moving towards each other
	trans_axis.negate();
	while( pose.get_0D_score( DOCK_VDW ) < 0.1 ) {
		perturbed_jump.translation_along_axis( Epos(1,1,pos1), trans_axis, step_size );
		pose.set_jump( dock_jump, perturbed_jump );
		pose.score( weight_map );
	}
	trans_axis.negate();

	//back to the optimal touching condition
	perturbed_jump.translation_along_axis( Epos(1,1,pos1), trans_axis, step_size );
	pose.set_jump( dock_jump, perturbed_jump );

	pose.score( weight_map ); // aroop: ensures refold and copy to misc

	pose.update_backbone_bonds_and_torsions(); // updating backbone bonds and torsions

}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_intra_clash_check
///
/// @brief evaluate centroid scores of loop in protein
///
/// @detailed If two proteins are present. Then the protein in which there is no
///           loop is cut out. Then the loop is inserted into the protein in
///           question and then evaluated
///
/// @param[in] pose: full actual protein
///            f: loop phi-psi-omega angles
///            begin: seq begin of loop in pose
///            size: size of loop
///            cutpoint: cutpoint in loop_fold tree
///
/// @global_read docking::part_end(1)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 01/31/2007
/////////////////////////////////////////////////////////////////////////////////
float
dle_intra_clash_check(
  const pose_ns::Pose & pose,
  Fragment & f,
  int begin,
  int size,
  int cutpoint
)
{
	using namespace pose_ns;

	Pose partner1; // Docking partner 1, in which the loop is assumed to be
	partner1.one_jump_tree( docking::part_end(1), begin-1, begin+(size-1)+1, cutpoint ); // loop tree
	partner1.set_allow_jump_move(1,false); // fixed stems
  partner1.copy_segment( docking::part_end(1), pose, 1, 1 ); // copy partner 1 from entire pose

	f.insert(partner1, begin); // insert the loop
	Score_weight_map weight_map( score4 ); // centroid score
	return(partner1.score(weight_map)); // calculate and return the score
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_orinent_to_native
///
/// @brief orient the complex based on superimposition of the first partner to
///        the first partner in the native
///
/// @detailed Care is taken to superimpose only over non-loop residues of first
///           partner for a more accurate result.Superimposition is done on the
///           native
///
/// @param[in] pose to be superimposed
///
/// @global_read native::native_Eposition, native::native_coord,
///              docking::part_end(1)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/10/2006
///
/// @last_modified 07/19/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_orient_to_native(
	pose_ns::Pose & pose
)
{
	using namespace pose_ns;
	using namespace dle_ns;

	Pose native_pose, orientable_pose;
	orientable_pose = pose;
	native_pose = orientable_pose; // assuming same sequence

	// count the number of loops to be modelled in first partner
	int loop_count = 0; // number of loops in first partner
	for( Loops::const_iterator it = dle_loops.begin(), it_end =
				 dle_loops.end(); it != it_end; ++it ) {
		if( it->start() < docking::part_end(1)) // if loop in first partner
			loop_count = loop_count + 1;
	}
	if(loop_count == 0) // if there are no loops in partner one - then return
		return;

	native_pose.set_coords( false, native::native_Eposition,
													native::native_coord );
	native_pose.update_backbone_bonds_and_torsions();

	int const nres( orientable_pose.total_residue() ); // Total residues

	// find out the aligned regions
	FArray1D_int alignstart, alignend;

	// Protein broken into two alignable regions by one loop
	int const aligned_regions(loop_count + 1);
	alignstart.dimension( aligned_regions );
	alignend.dimension( aligned_regions );


	Loops::const_iterator it = dle_loops.begin();
  Loops::const_iterator it_end = dle_loops.end();

	int loop_end(0); // end of a loop
	for( int i = 0; i < aligned_regions; ++i ){
		if(i == 0)
			{
				alignstart(i+1) = 1;
				alignend(i+1) = it->start() - 1;
					}
		else
			{
				if( i == (aligned_regions-1) )
					{
						alignstart(i+1) = loop_end + 1;
						alignend(i+1) = docking::part_end(1);
					}
				else
					{
						alignstart(i+1) = loop_end + 1;
						alignend(i+1) = it->start() - 1;
					}
			}
		loop_end = it->stop();
		if (it != it_end)
			it++;
	}

	// copy coords into double arrays
	FArray3D_float full1 ( orientable_pose.full_coord() );
	FArray3D_float Epos1 ( orientable_pose.Eposition() );
	FArray3D_float const & Epos2 ( native_pose.Eposition() );
	FArray2D_double p1a( 3, nres );
	FArray2D_double p2a( 3, nres );
	int const atom_index ( 2 ); // CA
	for ( int i = 1; i <= nres; ++i ) {
		for ( int k = 1; k <= 3; ++k ) {
			p1a(k,i) = Epos1( k, atom_index, i );
			p2a(k,i) = Epos2( k, atom_index, i );
		}
	}

	float local_rms; // dummy - not used
	orient_region( nres, Epos1, full1, Epos2, aligned_regions, alignstart,
								 alignend, aligned_regions, alignstart, alignend, local_rms );

	orientable_pose.set_coords( false /*ideal_pos*/, Epos1, full1 );
	orientable_pose.update_backbone_bonds_and_torsions();

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_scored_frag_close
///
/// @brief builds a loop from fragments file.
///
/// @detailed Loop is built by a monte carlo simulation using fragments from a
///           fragment files. CCD moves are used to close loops with gaps at
///           cutpoint.H3_check is enforced if H3_filter flag is set in command
///           line. Loop building results in many files containing differnt
///           conformations of the same loop in phi-psi-omega angles. Parallel
///           processing is utilized.
///
/// @param[in] weight_map: in this case its a centroid weight
///            pose_in: loop to be built on this template provided
///            loop_begin/loop_end: loop termini definition
///            frag_size: 3-mer or 9-mer
///            frag_offset: agreement in frag file numbering & pose numbering
///            cycles1: max cycles to be spent building loops
///            cycles2: number of fragment swaps for each loop(depends on size)
///            counter: an unique number corresponding to each loop ensemble
///            do_ccd_moves: should ccd moves be used to close gaps
///
/// @global_read makepdb_number for parallel processing
///
/// @global_write loop files are generated for each loop built, pose updated
///               and there is a decoy outputed for each loop built
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 02/07/2008
///////////////////////////////////////////////////////////////////////////////
void
dle_scored_frag_close(
	pose_ns::Score_weight_map const & weight_map,
	pose_ns::Pose & pose_in,
	int const loop_begin,
	int const loop_end,
	int const cutpoint,
	int const frag_offset,
	int const cycles1,
	int const cycles2,
	int counter,
	bool const do_ccd_moves
)
{
	using namespace pose_ns;
	using namespace makepdb_decoy;
	using namespace files_paths;

	counter = counter + 1; // for consistency in naming

	int loop_size = (loop_end - loop_begin ) + 1;
	if(loop_size <= 2) {
		std::cout << "Loop too small for modeling" << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// local pose
	Pose pose;
	pose = pose_in;

	/*
	// Use Chu's routine instead of standard
	{
		Loops one_loop;
		one_loop.add_loop( loop_begin, loop_end, cutpoint,
		                 frag_offset, 0);
		pose_loops_set_allow_bb_move( pose_in, one_loop );
		// storing default loops_ns::fast value
		bool default_fast = loops_ns::fast;
		loops_ns::fast = true;
		pose_perturb_one_loop_with_ccd( pose_in, weight_map, one_loop );
		loops_ns::fast = default_fast;
		return;
	}
	*/

	// params
	float const ccd_threshold( 0.1);
	int h3_attempts(0);
	float h3_fraction = 0.75; // 75% of loops are required to be H3's
	float current_h3_prob = ran3();
	bool H3_found_ever(false);
	bool loop_found(false);
	int total_cycles(0);
	int frag_size(0);
	{
		// size of loop above which 9mer frags are used
		int cutoff_9(16);
		// size of loop above which 3mer frags are used
		int cutoff_3(6);
		if( loop_size > cutoff_9 )
			frag_size = 9;
		else if( loop_size > cutoff_3 )
			frag_size = 3;
		else
			frag_size = 1;
	}
	// setup monte_carlo
	Monte_carlo mc( pose, weight_map, 2.0 ); // temp = 2.0
	Monte_carlo outer_mc( pose, weight_map, 2.0 );
	while(!loop_found && ((total_cycles++) < cycles1)) {
		insert_random_frags( frag_size, pose, loop_begin, loop_end,
												 frag_offset );
		mc.reset( pose ); // calls score
		if( total_cycles == 1 )
			mc.reset( pose );
		int local_h3_attempts(0);
		for ( int c2=1; c2<= cycles2; ++c2 ) {
			choose_offset_frag( frag_size, pose, loop_begin, loop_end,
													frag_offset );
			bool H3_found_current(false);
			if( dle_ns::current_loop_is_H3 && dle_ns::dle_H3_filter &&
					(local_h3_attempts++<50*cycles2) ) {
				H3_found_current = dle_CDR_H3_filter(pose, cdr_ns::cdr_h3_begin,
													 (cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1);
				if( !H3_found_ever && !H3_found_current) {
					--c2;
					mc.boltzmann( pose );
				}
				else if( !H3_found_ever && H3_found_current ) {
					H3_found_ever = true;
					mc.reset( pose );
				}
				else if( H3_found_ever && !H3_found_current ) {
					--c2;
					continue;
				}
				else if( H3_found_ever && H3_found_current )
					mc.boltzmann( pose );
			}
			else
				mc.boltzmann( pose );

			if ( do_ccd_moves && ( (c2 > cycles2/2 && ran3() * cycles2 < c2) ||
														 ( loop_size <= 5) ) ) {
				// in 2nd half of simulation, start trying to close the loop:
				if( loop_size <= 5 )
					ccd_moves( 500*loop_size, pose, loop_begin, loop_end,
										 cutpoint );
				else
					ccd_moves( 10*loop_size, pose, loop_begin, loop_end,
										 cutpoint );
				mc.boltzmann( pose );
			}
		}

		// recover low:
		pose = mc.low_pose();
		float fdev, bdev, torsion_delta, rama_delta;
		// changes cycles from 100 to 500
		fast_ccd_loop_closure( pose, loop_begin, loop_end,
		  cutpoint, 500, ccd_threshold, true, 2.0, 10, 50, 75, fdev, bdev,
        torsion_delta, rama_delta);
		// updating backbone bonds and torsions
		// aroop_temp remove pose.update_backbone_bonds_and_torsions();

		/*
		Score_weight_map wt_map( score4L );
		pose.main_minimize( wt_map, "linmin" );
		*/
		if( total_cycles == 1 )
			outer_mc.reset( pose );

		if ( fdev <= ccd_threshold && bdev <= ccd_threshold ) {
			// CDR-H3 filter for antibody mode
			// introduce enough diversity
			outer_mc.boltzmann( pose );
			if( dle_ns::current_loop_is_H3 && dle_ns::dle_H3_filter &&
					(current_h3_prob < h3_fraction) && (h3_attempts++<50) )
				if( !dle_CDR_H3_filter(pose, cdr_ns::cdr_h3_begin,
					(cdr_ns::cdr_h3_end - cdr_ns::cdr_h3_begin) + 1))
						continue;
			loop_found = true;
		}
		else if(dle_ns::dle_H3_filter)
			h3_attempts++;
	}
	pose_in = outer_mc.low_pose();
	pose_in.score( weight_map); // ensures refold

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_modify_weightmap
///
/// @brief incorporation of chain-break and chain-overlap
///
/// @detailed For fold tree with a docking type jump and loop jump. One has to
///           ensure that the loop has to be penalized if not closed at the
///           cutpoint, but then there is no penalty for partner a not
///           attaching to partner b in a docking type jump.
///
/// @param[in] weight_map: the weight_map to be modified
///
/// @global_read fold tree of the inputed pose
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 01/31/2007
///
/// @last_modified 07/18/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_modify_weightmap(
	pose_ns::Pose & pose,
	pose_ns::Score_weight_map & weight_map
)
{
	using namespace pose_ns;
	using namespace dle_ns;

	const Fold_tree & f( pose.fold_tree() );
	int const num_fold_tree_cutpoint( f.get_num_fold_tree_cutpoint() );
	const FArray1D_int & cutpoint_map( f.get_cutpoint_map() );

	weight_map.set_weight( CHAINBREAK, 1.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
	FArray1D_float cut_weight( num_fold_tree_cutpoint, 0.0f );
	weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );

	for ( Loops::const_iterator it = dle_loops.begin(),
					it_end = dle_loops.end();	it != it_end; ++it ) {
		cut_weight( cutpoint_map(it->cut()) ) = 1.0f;
		weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );
	}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_process_loops
///
/// @brief wrapper function for various loop pre-processing for docking whilst
///        iterating through a loop library
///
/// @detailed Wrapper function to go into a sort of overloaded function(which
///           is also a wrapper function. Depending on options previously read
///           into dle_ns,it can invoke creation a loop library,
///           invoke creating of files required for clustering using R and also
///           can invoke modes for refining loops in library
///
/// @param[in] docking_pose: basically any pose over which the operations have
///                          to be carried out
///
/// @global_read dle_ns flags
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 02/05/2007
///
/// @last_modified 02/05/2007
/////////////////////////////////////////////////////////////////////////////////
bool
dle_process_loops(
	pose_ns::Pose & docking_pose
)
{
	using namespace dle_ns;
	using namespace pose_ns;
	bool pose_docking_no_further_action( false );

	if( !(dle_refine || dle_build || dle_cluster) )
		return( pose_docking_no_further_action);

	pose_docking_no_further_action = true;
	dle_calc_rmsg_flag = true; // calcuate rmsg for loop in scorefile
	if(dle_cluster)
		dle_multiple_loop_operations( docking_pose,"cluster_only");
	else if(dle_refine)
		dle_multiple_loop_operations( docking_pose,"refine_only");
	else if(dle_build)
		dle_multiple_loop_operations( docking_pose,"build_only");
	docking_pose.copy_to_misc();
	set_pose_flag( false );
	monte_carlo_reset();
	return( pose_docking_no_further_action );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_write_one_loopfile
///
/// @brief outputs torsional angles of loop residues to a text file
///
/// @detailed Just prints out the phi-psi-omega angles in a defined region
///           region of a pose to an output file. The file name is unique
///           for each structure.
/// @param[in] pose: protein containing the loop of interest
///            loop_begin/loop_end: loop termini definition
///            counter: an unique number corresponding to each loop ensemble
///
/// @global_read makepdb_number for parallel processing
///
/// @global_write loop files are generated for each loop built, pose update
///               and there is a decoy outputed for each loop built
///
/// @remarks
///
/// @references
///
/// @authors Aroop 02/15/2007
///
/// @last_modified 02/15/2007
/////////////////////////////////////////////////////////////////////////////////
void
dle_write_one_loopfile(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end,
	int counter
)
{
	using namespace makepdb_decoy;
	using namespace files_paths;

	counter = counter + 1; // for consistency in naming
	int loop_size = (loop_end - loop_begin ) + 1;

	std::stringstream filename_counter; // String holder to contain the counter converted from int to string
	filename_counter << counter; // int counter fed to string
	std::string library_filename;
	// file is placed with output pdb
	library_filename = pdb_out_path + protein_name + "_dummy_loop_0" + filename_counter.str() + "_" + I( 4, 4, makepdb_number);
	std::ofstream outfile( library_filename.c_str(), std::ios::out); // open/create file

	for ( int k=1,pos; k<= loop_size; ++k ) {
		pos = loop_begin+k-1;
		outfile << F(10, 3, pose.phi(pos))
						<< F(10, 3, pose.psi(pos))
						<< F(10, 3, pose.omega(pos))
						<< std::endl;
	}
	outfile<< std::endl; // print a new line after each loop
	outfile.flush();
	outfile.close();
	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_common_options
///
/// @brief reads in options common to loop ensemble docking and antibody
///        modelling
///
/// @detailed
///
/// @param[in]
///
/// @global_read
///
/// @global_write namespace dle_ns, antibody_ns
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/07/2007
///
/// @last_modified 03/28/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_common_options()
{
	using namespace files_paths;
	using namespace dle_ns;
	using namespace antibody_ns;

	// repack loop neighbors
	if( truefalseoption("no_dock_neighbor") )
		repack_loop_neighbors = false;

	// if using generalized born then use simple elec
	if( param_pack::gen_born ) {
		param_pack::pack_wts.set_Wgb_elec(1.0);
		gb_set_use_simple_electrostatics( true );
	}

	// for different light_chain identifier
	stringafteroption("l_chain", 'L', light_chain);

	// checking for H3 kink/extended filter activation
	dle_model_H3 = truefalseoption("H3_filter");

	// using rotamers from input structure
	docking::docking_flagflags::unboundrot = truefalseoption("unboundrot");

	// output decoy without enforcing any filters
	output_with_no_filters = truefalseoption("flex_nofilter");

	// output decoy enforcing filters
	decoy_filter = truefalseoption("decoy_filter");

	// do not insert ideal bonds any where
	freeze_bonds = truefalseoption("freeze_bonds");

	// make sure plane scores are enabled if command line option exists
	param_pack::pack_wts.set_Wplane_total(realafteroption( "Wplane_total", 0.0));

	if(truefalseoption("build_loop")) {
		if(docking::dle_flag)
			dle_build = true; // only build loops
		else if(antibody_modeler && model_h3)
			antibody_build = true;
	}
	else if(truefalseoption("cluster_loop")) {
		if(docking::dle_flag)
			dle_cluster = true; // only cluster loop library
		else if(antibody_modeler && model_h3)
			antibody_cluster = true;
	}
	else if(truefalseoption("refine_loop")) {
		if(docking::dle_flag)
			dle_refine = true; // only refinement of loops in loop lib
		else if(antibody_modeler && model_h3)
			antibody_refine = true;
	}
	else if( truefalseoption("insert_loop") ) {
		// insert one loop at a time and output decoy
		if(docking::dle_flag) {
			// also do a local refine
			insert_all_loops_with_local_refine = true;
		}
		else if(antibody_modeler && model_h3) {
			// repacks as well
			antibody_insert_all_h3 = true;
		}
	}
	else if( truefalseoption("refine_input") ) {
		refine_input_loop = true;
	}

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_save_unbound
///
/// @brief reads in a pose and saves it in docking_unbound namespace.
///        Also computes chi angles and saves it in the namespace
///
/// @detailed In addition to saving just a copy of the input pose in the
///           docking_unbound namespace, the chi angles are store as well.
///           These chi angles will be used by the packer to actually append
///           to the dunbrak set whilst packing sidechains
///
/// @param[in]
///
/// @global_read
///
/// @global_write namespace docking_unbound
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/28/2007
///
/// @last_modified 03/28/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_save_unbound(
	pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace docking_unbound;

	unbound_total_res = pose_in.total_residue();

	for ( int j = 1; j <= unbound_total_res; ++j ) {
		unbound_res(j) = pose_in.res(j);
		unbound_res_variant(j) = pose_in.res_variant(j);
		unbound_phi(j) = pose_in.phi(j);
		unbound_psi(j) = pose_in.psi(j);
		unbound_omega(j) = pose_in.omega(j);
	}
	unbound_fullcoord = pose_in.full_coord();

	for ( int j = 1; j <= unbound_total_res; ++j ) { // determine action center of each side chain
		int aa = unbound_res(j);
		put_wcentroid(unbound_fullcoord(1,1,j), unbound_actcoord(1,j),aa);
	}

	// determine chi angles
	get_chi_and_rot_from_coords(unbound_total_res,unbound_res,unbound_res_variant,
															unbound_fullcoord,unbound_chiarray,unbound_rotarray);

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_pack_with_unbound
///
/// @brief packs pose using extra rotamers from unbound
///
/// @detailed If unbound is not defined then just repacks. If unbound is definited
///           then repacks using unbound rotamers as well
///
/// @param[in/out] pose read in and repacked as output
/// @param[in] include current: should packer include current rotamers
///
/// @global_read namespace docking_unbound
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/28/2007
///
/// @last_modified 03/28/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_pack_with_unbound(
	pose_ns::Pose & pose,
	FArray1D_bool const & allow_repack,
	bool const include_current
)
{
	using namespace pose_ns;
	using namespace docking_unbound;

	if ( ! docking::docking_flagflags::unboundrot ) {
		pose.repack( allow_repack, include_current );
		return;
	}

	//yl, Create PackerTask and setup values before pass into pack_rotamers
	std::string packmode( "packrot" );
	bool const make_output_file = false;
	PackerTask Task( pose );
	Task.set_task( packmode, make_output_file, allow_repack, include_current,
								 docking::docking_flagflags::unboundrot, unbound_rotarray, unbound_chiarray);
	//bk set variables that specify which residues to vary
	Task.setup_residues_to_vary();
	pack_rotamers( pose, Task );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_res_rb_move
///
/// @brief moves the loop apex residue and then tries to close the loop
///
/// @detailed
///
/// @param[in/out]
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 03/30/2007
///
/// @last_modified 09/04/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_res_rb_move(
  pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
)
{
	using namespace pose_ns;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	// storing initial fold tree
	Fold_tree const initial_tree( pose.fold_tree() );

	int loop_size = ( loop_end - loop_begin ) + 1;
	// selecting loop apex residue
	int loop_apex = loop_begin + int(loop_size/2);

	Fold_tree f;

	FArray2D_int jumps(2,2);
	FArray1D_int cuts(2);

	cuts(1) = loop_apex-2;
	cuts(2) = loop_apex+1;
	jumps(1,1) = loop_apex - 5;
	jumps(2,1) = loop_apex;
	jumps(1,2) = loop_apex - 6;
	jumps(2,2) = loop_apex + 6;

	f.tree_from_jumps_and_cuts(pose.total_residue(), 2 /*# of jumps*/, jumps,
														 cuts);
	std::cout << f << std::endl;
	pose.set_fold_tree(f);

	pose.set_allow_jump_move(1, true);
	pose.set_allow_jump_move(2, false);

	pose_docking_gaussian_rigid_body_move( pose, 1, .50 /*angstrom*/,
																				 1.5 /*degree*/ );
	pose.dump_pdb("jiggle.pdb");

	pose.set_allow_jump_move(false);

	pose.set_allow_bb_move(false);
	pose.one_jump_tree(pose.total_residue(),loop_apex-4,loop_apex,loop_apex-2,1);
	pose.set_allow_jump_move(false);

	for (int i = 1; i <= 3; i++)
		pose.set_allow_bb_move( loop_apex-i, true);
	ccd_moves(20, pose, loop_apex-4, loop_apex, loop_apex-2);

	pose.set_allow_bb_move(false);
	pose.one_jump_tree(pose.total_residue(),loop_apex,loop_apex+4,loop_apex+2,1);
	pose.set_allow_jump_move(false);

	for (int i = 1; i <= 3; i++){
		pose.set_allow_bb_move(loop_apex+i, true);}
	ccd_moves(20, pose, loop_apex, loop_apex+4, loop_apex+2);

 	// Restoring pose stuff
	pose.set_fold_tree( initial_tree ); // docking tree
	pose.set_allow_jump_move( jump_move_old );//restore the jump move
	pose.set_allow_bb_move( bb_move_old );
	pose.set_allow_chi_move( chi_move_old );

	pose.dump_pdb("end_jiggle.pdb");

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_trim_and_repack
///
/// @brief extensive repacking after trimming off loop region
///
/// @detailed The loop region is first truncated. This is followed by extensive
///           repacking.Finally a chimeric pose is constructed to have the size
///           of the input pose. The chimeric pose consists of the repacked
///           non-loop region and the disordered loop region
///
/// @param[in/out] pose
/// @param[in] loop_begin: start of loop region
///            loop_end: end of loop region
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 04/12/2007
///
/// @last_modified 04/13/2007
///////////////////////////////////////////////////////////////////////////////
void

dle_trim_and_repack(
	pose_ns::Pose & pose_in,
	int const loop_begin,
	int const loop_end
)
{
	using namespace pose_ns;

	int const repack_cycles(25); // default 25

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose_in.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	// storing docking fold tree
	Fold_tree const initial_tree( pose_in.fold_tree() );

	int nres_in( pose_in.total_residue() );
	int loop_size = ( loop_end - loop_begin ) + 1;
	int nres_trim( nres_in - loop_size );
	pose_in.simple_fold_tree( nres_in );

	Pose trim_pose, trim_unbound_pose, start_trim_pose;
	trim_pose.simple_fold_tree( nres_trim );

	FArray3D_float trim_Epos(3, param::MAX_POS, nres_trim);
	FArray3D_float trim_fullcoord(3, param::MAX_ATOM(), nres_trim);
	FArray3D_float output_Epos(3, param::MAX_POS, nres_in);
	FArray3D_float output_fullcoord(3, param::MAX_ATOM(), nres_in);
	FArray3D_float const & input_Epos( pose_in.Eposition() );
	FArray3D_float const & input_fullcoord( pose_in.full_coord() );

	int trim_counter(0);
	for( int i = 1; i <= nres_in; i++ ) {
		if( (i < loop_begin) || ( i > loop_end ) ) {
			trim_counter++;
			trim_pose.set_phi        ( trim_counter, pose_in.phi(i) );
			trim_pose.set_psi        ( trim_counter, pose_in.psi(i) );
			trim_pose.set_omega      ( trim_counter, pose_in.omega(i) );
			trim_pose.set_secstruct  ( trim_counter, pose_in.secstruct(i) );
			trim_pose.set_name       ( trim_counter, pose_in.name(i) );
			trim_pose.set_res        ( trim_counter, pose_in.res(i) );
			trim_pose.set_res_variant( trim_counter, pose_in.res_variant(i) );
			for( int k = 1; k <= 3; k++ ) {
				for(int j = 1; j <= param::MAX_POS; j++ )
					trim_Epos(k, j, trim_counter) = input_Epos(k, j, i);
				for( int j = 1; j <= param::MAX_ATOM()(); j++ )
					trim_fullcoord(k, j, trim_counter) = input_fullcoord(k, j, i);
			}
		}
	}
	trim_pose.set_fullatom_flag( true, false );
	trim_pose.set_coords( false, trim_Epos, trim_fullcoord );
	// updating backbone bonds and torsions
	trim_pose.update_backbone_bonds_and_torsions();

	// saving trim unbound information
	trim_unbound_pose = dle_ns::starting_sidechain_pose;
	trim_unbound_pose.delete_segment( loop_begin, loop_end );
	dle_save_unbound( trim_unbound_pose );

	FArray1D_bool allow_repack( trim_pose.total_residue(), true );
	trim_pose.set_allow_chi_move( true );

	dle_extreme_repack( trim_pose, repack_cycles, allow_repack,
																		 true, // rt_min
																		 true, // rotamer_trials
																		 true, // force_one_repack
																		 true ); // use_unbounds
	trim_Epos = trim_pose.Eposition();
	trim_fullcoord = trim_pose.full_coord();

	trim_counter = 0;
	for( int i = 1; i <= nres_in; i++ ) {
		if (i == loop_begin )
			i = loop_end + 1;
		trim_counter++;
		for( int k = 1; k <= 3; k++ ) {
			for( int j = 1; j <= param::MAX_POS; j++ )
				output_Epos(k, j, i) = trim_Epos(k, j, trim_counter);
			for( int j = 1; j <= param::MAX_ATOM()(); j++ )
				output_fullcoord(k, j, i) = trim_fullcoord(k, j, trim_counter);
		}
	}

	for( int i = loop_begin; i <= loop_end; i++ ) {
		for( int k = 1; k <= 3; k++ ) {
			for( int j = 1; j <= param::MAX_POS; j++ )
				output_Epos(k, j, i) = input_Epos(k, j, i);
			for( int j = 1; j <= param::MAX_ATOM()(); j++ )
				output_fullcoord(k, j, i) = input_fullcoord(k, j, i);
		}
	}
	pose_in.set_coords( false, output_Epos, output_fullcoord );
	// updating backbone bonds and torsions
	pose_in.update_backbone_bonds_and_torsions();

	// Restoring normal unbound rotamers
	dle_save_unbound(
		dle_ns::starting_sidechain_pose );

	// Restoring pose stuff
	pose_in.set_fold_tree( initial_tree ); // docking tree
	pose_in.set_allow_jump_move( jump_move_old );//restore the jump move
	pose_in.set_allow_bb_move( bb_move_old );
	pose_in.set_allow_chi_move( chi_move_old );

	return;
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_choose_random_cutpoint
///
/// @brief choose a random cutpoint for a loop fold tree
///
/// @detailed Chooses a random cutpoint between (loop_begin+1) and (loop_end-1)
///
/// @param[out] cutpoint
/// @param[in] loop_begin: start of loop region
///            loop_end: end of loop region
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 06/21/2007
///
/// @last_modified 06/21/2007
///////////////////////////////////////////////////////////////////////////////
int
dle_choose_random_cutpoint(
	int const loop_begin,
	int const loop_end
)
{
	// initializing cutpoint with loop_begin + 1
	int cutpoint( loop_begin + 1 );
	int loop_size = ( loop_end - loop_begin ) + 1;

	// cutpoint randomizer
	cutpoint = (loop_begin + 1) + int( ran3() * (loop_size-2));

	// cutpoint rationalizer
	// can not be at loop_end
	if( cutpoint == loop_end )
		cutpoint = loop_end - 1;

	return( cutpoint );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_cutpoint_separation
///
/// @brief wrapper to compute the separation at cutpoint ( uses pose input )
///
/// @detailed compute the N-C distance of the peptide bond which should be
///           formed at the cutpoint. The cutpoint is stored in a global
///           variable. This function is actually a wrapper that calls the
///           flavor of the function that uses misc. So this "overloaded"
///           function basically passes the pose to misc and then restores
///           misc to its initial state. A closed loop is assumed to have a gap
///           distance < 1.9 Ang.
///
/// @param[in]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 08/17/2007
///
/// @last_modified 08/17/2007
///////////////////////////////////////////////////////////////////////////////
float
dle_cutpoint_separation(
  pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	float cutpoint_separation( 0.00 );

	// saving initial misc - copy misc to pose
	Pose misc_pose;
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( misc_pose, fullatom, ideal_pos, coords_init );

	// copy pose_in to misc to be used in main function
	pose_in.copy_to_misc();

	// core function computing gap distance at cutpoint
	// If gap less than 1.9 Ang then assume closed loop
	cutpoint_separation = dle_cutpoint_separation();

	// resetting misc
	misc_pose.copy_to_misc();

	return( cutpoint_separation );
}

///////////////////////////////////////////////////////////////////////////////
/// @begin dle_cutpoint_separation
///
/// @brief compute the separation at the cutpoint
///
/// @detailed compute the N-C distance of the peptide bond which should be
///           formed at the cutpoint. The cutpoint is stored in a global
///           variable. A closed loop is assumed to have a gap < 1.9 Ang
///
/// @param[in]
///
/// @global_read dle_ns::decoy_loop_cutpoint
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 06/25/2007
///
/// @last_modified 06/25/2007
///////////////////////////////////////////////////////////////////////////////
float
dle_cutpoint_separation()
{
	using namespace pose_ns;
	using namespace native_cdr;

	if( (!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
			 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
			 && !dle_ns::refine_input_loop)
			|| !antibody_ns::use_native )
		return( -1.0 ); // return error value if none of
	                  // the above modes have been initialized

	// error value if no cutpoints defined
	if( dle_ns::decoy_loop_cutpoint == 0 )
		return( 0.00 );

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_decoy; // decoy used in distance computation
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_decoy, fullatom, ideal_pos, coords_init );
	pose_misc = pose_decoy;

	// copy coords into double arrays
	FArray3D_float Epos ( pose_decoy.Eposition() );
	int const N ( 1 ); // N atom
	int const C ( 4 ); // C atom

	int cutpoint = dle_ns::decoy_loop_cutpoint;
	// Coordinates of the C atom of cutpoint res and N atom of res cutpoint+1
	numeric::xyzVector_float peptide_C(&Epos(1,C,cutpoint)), peptide_N(&Epos(1,N,cutpoint+1));
	float cutpoint_separation( distance(peptide_C, peptide_N));

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( cutpoint_separation );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_extreme_repack
///
/// @brief repack multiple times over with rotamer trials and rt_min
///
/// @detailed repacks the pose as many times over as specified. Then returns
///           the lowest scoring pose found amongst all repack cycles. Force
///           one repack forces one repack move to be accepted. This is useful
///           when packing in rotamers for the first time into an inserted loop
///           Normally since this could result in clashes resulting in the
///           repacked structure having a higher score and hence rejected
///
/// @param[in] pose to be repacked, number of cycles, allowable residues for
///            repacking
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 07/10/2007
///
/// @last_modified 08/08/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_extreme_repack(
	pose_ns::Pose & pose_in,
	int repack_cycles,
	FArray1D_bool & allow_repack,
	bool rt_min,
	bool rotamer_trials,
	bool force_one_repack,
	bool use_unbounds
)
{
	using namespace pose_ns;

	// Exit if not fullatom
	if( !pose_in.fullatom() ) {
		std::cout << "Repack called in centroid mode" << std::endl;
		std::cout << "XXXXXXXXXXXXXXXXXXXXXXXXXXXXXX" << std::endl;
		std::cout << "------NOT REPACKING-----------" << std::endl;
		return;
	}
	// Saving parameters
	bool initial_rot_trial = score_get_try_rotamers();
	bool initial_min_rot = get_minimize_rot_flag();
	score_set_minimize_rot( rt_min );
	score_set_try_rotamers( rotamer_trials );
	// initial allowed chi movement
	FArray1D_bool old_chi_move( pose_in.total_residue(), false );
	for( int i = 1; i <= pose_in.total_residue(); i++ ) {
		// storing old
		old_chi_move(i) = pose_in.get_allow_chi_move(i);
		// setting new
		pose_in.set_allow_chi_move( i, allow_repack(i) || old_chi_move(i) );
	}
	Score_weight_map weight_map( score12 );
	Monte_carlo mc( pose_in, weight_map, 2.0 );
	// repack idealized native
	Pose start_native_pose;
	start_native_pose = pose_in;
	for(int i=1; i <= repack_cycles; i++) {
		pose_in = start_native_pose;
		if( use_unbounds )
			dle_pack_with_unbound( pose_in, allow_repack,
																						true /*include_current*/ );
		else
			pose_in.repack( allow_repack, true /*include_current*/ );
		pose_in.score( weight_map );
		score_set_minimize_rot( false );
		score_set_try_rotamers( false );
		if( force_one_repack && (i == 1) )
			mc.reset( pose_in );
		mc.boltzmann( pose_in );
		score_set_minimize_rot( rt_min );
		score_set_try_rotamers( rotamer_trials );
	}
	pose_in = mc.low_pose();
	pose_in.score( weight_map );

	// Restoring Globals
	score_set_minimize_rot( initial_rot_trial );
	score_set_try_rotamers( initial_min_rot );
	pose_in.set_allow_chi_move( old_chi_move );

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_hydrophobic
///
/// @brief detects hydrophobic atoms & calls wrapper for hydrophobic score eval
///
/// @detailed this term is aimed at modeling the free energy that is gained by
///           placing a hydrophobic ligand group into a hydrophobic pocket of
///           the receptor. The analogy in loop prediction would be placing
///           hydrophobic side chains of the loop into hydrophobic pockets that
///           exist in the protein structure when the loop is removed.
///           Hydrophobic atoms includes all carbon and sulphur atoms with
///           absolute value of partial charge less than 0.25 unit charge.
///           deltaG(total,hydrophobic)=deltaG(pair)* Sum(i,j)[F(r(i,j))]
///           F(r(i,j)) is 1.0 for R <= sum of two van der Waals radius
///           and then drops to 0 linearly at R=R+3.0 Ang
///           deltaG(pair) = -0.50 kcal/mol for inter contact
///           deltaG(pair) = -0.25 kcal/mol for intra(self) contact
///           Based on further study it was established that only atoms within
///           a certain distance from loop atoms would be utilized for
///           computation of the hydrophobic term. This cutoff is set to an
///           initial guess of 12 Ang.
///
/// @param[in]
///
/// @param[out] hydrophobic score of the protein
///
/// @global_read param_aa, misc, cdr_ns, native_cdr
///
/// @global_write
///
/// @remarks
///
/// @references Kau Zhu, David L. Pincus, Suwen Zhao, Richard A. Friesner, Long
///             Loop Prediction Using the Protein Local Optimization Program,
///             2006, PROTEINS:Structure, Function, and Bioinformatics 65:438-
///             452
///             Das B, Meirovitch H., Solvation parameters for predicting the
///             structure of surface loops in proteins: transferability and
///             entropic effects. Proteins. 2003 May 15;51(3):470-83.
///
/// @authors Aroop 07/10/2007
///
/// @last_modified 07/12/2007
///////////////////////////////////////////////////////////////////////////////
float
dle_hydrophobic()
{
	using namespace pose_ns;
	using namespace param_aa;
	using namespace cdr_ns;
	using namespace native_cdr;

	if(!antibody_ns::antibody_build && !antibody_ns::antibody_cluster &&
		 !antibody_ns::antibody_refine && !antibody_ns::antibody_insert_all_h3
		 && !dle_ns::refine_input_loop)
		return( -1.0 ); // return error value if none of
	                  // the above modes have been initialized

	// store current status of pose flag
	bool original_pose_flag_setting = pose_flag_ns::pose_flag_setting;
	pose_flag_ns::pose_flag_setting = true; // setting pose flag to true

	Pose pose_in; // decoy used for energy calculation
	Pose pose_misc; // pose used for resetting misc

	// copy misc to pose
  // this is the decoy
  bool const ideal_pos( false );
  bool const coords_init( true );
  bool const fullatom( true );
  pose_from_misc( pose_misc, fullatom, ideal_pos, coords_init );
	pose_in = pose_misc;

	float hydrophobic(0.00);
	int nres( pose_in.total_residue() );
	float const C(1.00); // Carbon Atom
	float const S(2.00); // Sulphur Atom
	// van der Waals radius obtained from atom_chem.cc
	// sum van der Waals radii of CC
	float cutoff_CC(3.4);
	// sum van der Waals radii of CS
	float cutoff_CS(3.5);
	// sum van der Waals radii of CC
	float cutoff_SS(3.6);

	// establishing loop limits
	int loop_begin(0);
	int loop_end(0);
	if( ( cdr_h3_begin == 0 ) || ( cdr_h3_end == 0 ) ) {
		loop_begin = native_h3_begin;
		loop_end = native_h3_end + 1;
	}
	else {
		loop_begin = cdr_h3_begin;
		loop_end = cdr_h3_end + 1;
	}

	FArray3D_float const input_fullcoord( pose_in.full_coord() );
	// storing param stores the following info in the fist index
	// [1] 1 == hydrophobic residue
	// [1] 0 == polar residue
	// [2] 1 == carbon atom
	// [2] 2 == sulphur atom
	// [2] 0 == inconsequential atom
	// [3] 1 == atom within loop neighbor shell
	// [3] 0 == atom not within loop neighbor shell
	FArray3D_float storing_param( pose_in.full_coord() );

	// instantiation neighbor shell with 0.00
	for( int i = 1; i <= nres; i++ )
		for( int j = 1; j <= param::MAX_ATOM()(); j++ )
			storing_param(3, j, i) = 0.00;

	// detecting atoms in neighbourhood
	float const cutoff_distance = 10.00;
	for( int outer_i = loop_begin; outer_i <= loop_end; outer_i++ ) {
		for( int outer_j = 1; outer_j <= param::MAX_ATOM()(); outer_j++ ) {
			storing_param(3, outer_j, outer_i) = 1.00; // loop atoms
			for( int inner_i = 1; inner_i <= nres; inner_i++ ) {
				if( (inner_i >= loop_begin) && (inner_i <= loop_end) )
					continue; // skip loop atoms
				for( int inner_j = 1; inner_j <= param::MAX_ATOM()(); inner_j++ ) {
					if( (storing_param(3, inner_j, inner_i) == 1.0) && (storing_param(3,
							 outer_j, outer_i) == 1.0 ))
						continue; // skip already detected atoms
					else {
						float distance = std::sqrt( distance_squared(
														 numeric::xyzVector_float( &input_fullcoord( 1,
														 inner_j, inner_i ) ), numeric::xyzVector_float(
														 &input_fullcoord( 1, outer_j, outer_i ) ) ) );
						if( distance < cutoff_distance )
							storing_param(3, inner_j, inner_i) = 1.00; // neighbor atoms
					}
				}
			}
		}
	}
/*
	// assuming that hydrophobic is partial charge
	// less tha 0.35 absolute
	for( int i = 1; i <= nres; i++ ) {
		for( int j = 1; j <= param::MAX_ATOM()(); j++ ) {
			if( storing_param(3, j, i) == 1.00 ) {
				if( pose_in.res(i) == aa_ala ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_cys ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else if( j == 6 ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = S;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_asp ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_glu ) {
					if( (j == 2) || (j == 5) || (j == 6) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_phe ) {
					if( (j == 2) || ( (j >= 5) && (j <= 11) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_gly ) {
					if(j == 2) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_his ) {
					if((j == 2) || (j ==5) || (j==6) || (j==8) || (j==9)) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_ile ) {
					if((j == 2) || ((j >= 5) && (j <= 8))) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_lys ) {
					if( (j == 2) || ( (j >= 5) && (j <= 8) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_leu ) {
					if( (j == 2) || ((j >= 5) && (j <= 8)) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_met ) {
					if( (j == 2) || (j == 5) || (j == 6) || (j == 8) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else if( j == 7 ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = S;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_asn ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_pro ) {
					if( (j == 2) || ( (j >= 5) && (j <= 7) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_gln ) {
					if( (j == 2) || (j == 5) || (j == 6) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_arg ) {
					if( (j == 2) || ( (j >= 5) && (j <= 7) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_ser ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_thr ) {
					if( (j == 2) || (j == 5) || (j == 7) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_val ) {
					if( (j == 2) || ((j >= 5) && (j <= 7)) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_trp ) {
					if( (j == 2) || ((j >= 5) && (j <= 8)) || ((j >= 10) && (j <= 14))) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_tyr ) {
					if( (j == 2) || ((j >= 5) && (j <= 11)) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else {
					storing_param(1, j, i) = 0.0;
					storing_param(2, j, i) = 0.0;
				}
			}
			else {
				storing_param(1, j, i) = 0.0;
				storing_param(2, j, i) = 0.0;
			}
		}
	}
*/

	// assuming that hydrophobic is partial charge
	// less tha 0.25 absolute
	for( int i = 1; i <= nres; i++ ) {
		for( int j = 1; j <= param::MAX_ATOM()(); j++ ) {
			if( storing_param(3, j, i) == 1.00 ) {
				if( pose_in.res(i) == aa_ala ) {
					if( j == 2 ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_cys ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else if( j == 6 ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = S;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_asp ) {
					if( j == 2 ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_glu ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_phe ) {
					if( (j == 2) || ( (j >= 5) && (j <= 11) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_gly ) {
					if(j == 2) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_his ) {
					if((j == 2) || (j ==5) || (j==6) || (j==8)) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_ile ) {
					if((j == 2) || (j ==5) || (j==6)) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_lys ) {
					if( (j == 2) || ( (j >= 5) && (j <= 8) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_leu ) {
					if( (j == 2) || (j == 5) || (j == 6) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_met ) {
					if( (j == 2) || (j == 5) || (j == 6) || (j == 8) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else if( j == 7 ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = S;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_asn ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_pro ) {
					if( (j == 2) || ( (j >= 5) && (j <= 7) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_gln ) {
					if( (j == 2) || (j == 5) || (j == 6) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_arg ) {
					if( (j == 2) || ( (j >= 5) && (j <= 7) ) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_ser ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_thr ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_val ) {
					if( (j == 2) || (j == 5) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_trp ) {
					if( (j == 2) || ((j >= 5) && (j <= 8)) || ((j >= 10) && (j <= 14))) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else if( pose_in.res(i) == aa_tyr ) {
					if( (j == 2) || ((j >= 5) && (j <= 11)) ) {
						storing_param(1, j, i) = 1.0;
						storing_param(2, j, i) = C;
					}
					else {
						storing_param(1, j, i) = 0.0;
						storing_param(2, j, i) = 0.0;
					}
				}
				else {
					storing_param(1, j, i) = 0.0;
					storing_param(2, j, i) = 0.0;
				}
			}
			else {
				storing_param(1, j, i) = 0.0;
				storing_param(2, j, i) = 0.0;
			}
		}
	}


	for( int outer_i = 1; outer_i <= nres; outer_i++ ) {
		for( int outer_j = 1; outer_j <= param::MAX_ATOM()(); outer_j++ ) {
			for( int inner_i = outer_i+1; inner_i <= nres; inner_i++ ) {
				for( int inner_j = 1; inner_j <= param::MAX_ATOM()(); inner_j++ ) {
					if( (storing_param(1, inner_j, inner_i) == 1.0) && (storing_param(1,
							 outer_j, outer_i) == 1.0 )) {
						float deltaf(0.00);
						float distance = std::sqrt( distance_squared(
														 numeric::xyzVector_float( &input_fullcoord( 1,
														 inner_j, inner_i ) ), numeric::xyzVector_float(
														 &input_fullcoord( 1, outer_j, outer_i ) ) ) );
						if( (storing_param(2, inner_j, inner_i) == 1.0) && (storing_param(
                2 , outer_j, outer_i) == 1.0 )) {
							dle_hydrophobic_eval( deltaf, distance, cutoff_CC,
																									outer_i, inner_i );
						}
						else if( (storing_param( 2, inner_j, inner_i) == 1.0) &&
										 (storing_param( 2, outer_j, outer_i) == 2.0 )) {
							dle_hydrophobic_eval( deltaf, distance, cutoff_CS,
																									outer_i, inner_i );
						}
						else if( (storing_param( 2, inner_j, inner_i) == 2.0) &&
										 (storing_param( 2, outer_j, outer_i) == 2.0 )) {
							dle_hydrophobic_eval( deltaf, distance, cutoff_SS,
																									outer_i, inner_i );
						}
						else
							deltaf = 0.00;
						hydrophobic += deltaf;
					}
				}
			}
		}
	}

	// Resetting misc
  pose_misc.copy_to_misc();
	// Resetting pose flag
	pose_flag_ns::pose_flag_setting = original_pose_flag_setting;

	return( hydrophobic );
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_hydrophobic_eval
///
/// @brief computes the hydrophobic term as defined by Friesner
///
/// @detailed computes the hydrophobic term after identification of atoms
///           deltaG(total,hydrophobic)=deltaG(pair)* Sum(i,j)[F(r(i,j))]
///           F(r(i,j)) is 1.0 for R <= sum of two van der Waals radius
///           and then drops to 0 linearly at R=R+3.0 Ang
///           deltaG(pair) = -0.50 kcal/mol for inter contact
///           deltaG(pair) = -0.25 kcal/mol for intra(self) contact
///
/// @param[in] distance: distance between two atoms
///            cutoff: sum of the van der Waals radii of the two atoms
///            res1: residue number to which the first atom belongs
///            res2: residue number to which the second atom belongs
///
/// @param[out] deltaF: hydrophobic energy term due to the two atoms
///
/// @global_read cdr_h3 begin and end from cdr_ns and native_cdr
///
/// @global_write
///
/// @remarks
///
/// @references Kau Zhu, David L. Pincus, Suwen Zhao, Richard A. Friesner, Long
///             Loop Prediction Using the Protein Local Optimization Program,
///             2006, PROTEINS:Structure, Function, and Bioinformatics 65:438-
///             452
///
/// @authors Aroop 07/11/2007
///
/// @last_modified 07/11/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_hydrophobic_eval(
	float & deltaF,
	float distance,
	float cutoff,
	int const res1,
	int const res2
)
{
	using namespace cdr_ns;
	using namespace native_cdr;

	deltaF = 0.00;
	float deltaGpair = -0.50;
	int loop_begin(0);
	int loop_end(0);

	// establishing loop limits
	if( ( cdr_h3_begin == 0 ) || ( cdr_h3_end == 0 ) ) {
		loop_begin = native_h3_begin;
		loop_end = native_h3_end + 1;
	}
	else {
		loop_begin = cdr_h3_begin;
		loop_end = cdr_h3_end + 1;
	}

	// damping self interactions
	if( ((res1 >= loop_begin) && (res1 <=loop_end))
			&& ((res2 >= loop_begin) && (res2 <= loop_end)) )
		deltaGpair = 0.5 * deltaGpair;

	if( distance > ( cutoff + 3.0) )
		deltaF = 0.0;
	else if(distance > cutoff) {
		float slope = ( 0.00 - 1.00 ) / ( 3.00 ); // (cutoff+3) - cutoff = 3
		deltaF = ( slope * distance ) + ( 1 - ( slope * cutoff ) );
	}
	else
		deltaF = 1.0;

	deltaF = deltaGpair * deltaF;

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_bad_Erep_repack
///
/// @brief detects residues that have bad ( > 10) Erep scores and repacks them
///
/// @detailed scores the protein in full atom using score12. Then checks to see
///           if any of the Erep scores are greater than 10. If so then selects
///           that particular residue as well as all neighbors. Neighbors are
///           defined as any residue that is within 8 Ang CENTDIST  of another
///           residue. Finally repacks all selected residues using both rotamer
///           trials and rotamer minimization
///
/// @param[in]
///
/// @param[out]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 09/06/2007
///
/// @last_modified 09/06/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_bad_Erep_repack(
  pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;
	using namespace fullatom_energies;

  float const rep_threshold( 10.00 );
	float const neighbor( 8.00 );
	float neighbor_cutoff( neighbor * neighbor );

	Score_weight_map weight_map;
	setup_score_weight_map( weight_map, score12 );
	pose_in.score( weight_map );

	int const nres( pose_in.total_residue() ); // Total residues
	pose_update_cendist( pose_in ); // Updating C-alpha distance matrix
	// C-alpha distance matrix
	FArray2D_float const & cendist ( pose_in.get_2D_score( CENDIST ) );

	FArray1D_bool allow_repack( nres, false );
	for( int res = 1; res <= nres; res++ )
		if( repenergy(res) > rep_threshold ) {
			allow_repack(res) = true;
			for( int in_res = 1; in_res <= nres; in_res++ ) {
				if( cendist(res, in_res) < neighbor_cutoff )
					allow_repack( in_res ) = true;
			}
		}

	// repacking neighbors
	dle_extreme_repack( pose_in,
																		 1, // relax_cycles
																		 allow_repack,
																		 true, // rt_min
																		 true, // rotamer_trials
																		 false, // force_one_repack
																		 true ); // use_unbounds

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_refine_modeled_loop
///
/// @brief wrapper to refine specified loops
///
/// @detailed Firstly refines the CDR H3. Then depending on other command line
///           options (like -snugh2) refines other loops
///
/// @param[in]
///
/// @param[out]
///
/// @global_read cdr_ns::cdr_h3_begin, cdr_h3_end
///              cdr_ns::cdr_h2_apex_begin, cdr_h2_apex_end
///              dle_ns::decoy_loop_cutpoint
///              antibody_ns::snugdock_ns::snugh2
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 09/13/2007
///
/// @last_modified 12/31/2007
///////////////////////////////////////////////////////////////////////////////
void
dle_refine_modeled_loop(
  pose_ns::Pose & pose_in
)
{
	using namespace pose_ns;

	// setting move sizes
	set_smallmove_size( 1.0, 5.0, 3.0); //helix,strand,other

	if( antibody_ns::freeze_h3 )
		return;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose_in.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	Fold_tree const input_tree( pose_in.fold_tree() );

	int loop_begin = cdr_ns::cdr_h3_begin;
	int loop_end = cdr_ns::cdr_h3_end + 1;
	int loop_size = ( loop_end - loop_begin ) + 1;
	int loop_cut(0);
	if( dle_ns::decoy_loop_cutpoint == 0 )
	  loop_cut = loop_begin + int( loop_size / 2 );
	else
		loop_cut = dle_ns::decoy_loop_cutpoint;

	dle_refine_one_loop( pose_in, loop_begin, loop_end, loop_cut );
	pose_in.update_backbone_bonds_and_torsions();

	if( antibody_ns::snugdock_ns::snugh2 ) {
		loop_begin = cdr_ns::cdr_h2_apex_begin;
		loop_end = cdr_ns::cdr_h2_apex_end;
		loop_size = ( loop_end - loop_begin ) + 1;
		loop_cut = loop_begin + int( loop_size / 2 );
		set_smallmove_size(1.0,5.0,6.0); //helix,strand,other

		dle_refine_one_loop( pose_in, loop_begin, loop_end, loop_cut );
		pose_in.update_backbone_bonds_and_torsions();

	}

	// Restoring variables
	pose_in.set_fold_tree( input_tree );
	pose_in.set_allow_jump_move( jump_move_old );
	pose_in.set_allow_bb_move( bb_move_old );
	pose_in.set_allow_chi_move( chi_move_old );
	set_smallmove_size( 0.0, 5.0, 6.0); // helix, strand, other

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_refine_modeled_loop
///
/// @brief overloaded wrapper to refine specified loops
///
/// @detailed Refines a CDR loop that is specified. However, can be extended to
///           account for any loop based on a string identifier
///
/// @param[in]
///
/// @param[out]
///
/// @global_read cdr_ns::cdr_h3_begin, cdr_h3_end
///              cdr_ns::cdr_h2_apex_begin, cdr_h2_apex_end
///              dle_ns::decoy_loop_cutpoint
///              antibody_ns::snugdock_ns::snugh2
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 07/14/2008
///
/// @last_modified 07/14/2008
///////////////////////////////////////////////////////////////////////////////
void
dle_refine_modeled_loop(
	pose_ns::Pose & pose_in,
  const std::string & cdr_loop_name
)
{
	using namespace pose_ns;

	if( (cdr_loop_name == "h3") && antibody_ns::freeze_h3 )
		return;

	// Storing source status
	FArray1D_bool bb_move_old,chi_move_old,jump_move_old;
	pose_in.retrieve_allow_move( bb_move_old, chi_move_old, jump_move_old );
	Fold_tree const input_tree( pose_in.fold_tree() );

	int loop_begin(0);
	int loop_end(0);
	int loop_cut(0);

	if( cdr_loop_name == "h3" ) {
		loop_begin = cdr_ns::cdr_h3_begin;
		loop_end = cdr_ns::cdr_h3_end + 1;
		int loop_size = ( loop_end - loop_begin ) + 1;
		if( dle_ns::decoy_loop_cutpoint == 0 )
			loop_cut = loop_begin + int( loop_size / 2 );
		else
			loop_cut = dle_ns::decoy_loop_cutpoint;
		set_smallmove_size(1.0,5.0,3.0); //helix,strand,other
	}
	else if( cdr_loop_name == "h2" ) {
		loop_begin = cdr_ns::cdr_h2_apex_begin;
		loop_end = cdr_ns::cdr_h2_apex_end;
		int loop_size = ( loop_end - loop_begin ) + 1;
		loop_cut = loop_begin + int( loop_size / 2 );
		set_smallmove_size(1.0,5.0,6.0); //helix,strand,other
	}

	dle_refine_one_loop( pose_in, loop_begin, loop_end, loop_cut );
	pose_in.update_backbone_bonds_and_torsions();

	// Restoring pose stuff
	pose_in.set_fold_tree( input_tree ); // input tree
	pose_in.set_allow_jump_move( jump_move_old );// restore the jump move
	pose_in.set_allow_bb_move( bb_move_old );
	pose_in.set_allow_chi_move( chi_move_old );
	set_smallmove_size( 0.0, 5.0, 6.0); // helix, strand, other

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_refine_one_loop
///
/// @brief Extensively refines a specified loop
///
/// @detailed This function is normally called if the loop was found to be
///           broken. Hence although it uses the full atom scoring function
///           ( score12), the chain break and the chain overlap weights have
///           been tripled to ensure loop closure. One prefers a non-physical
///           loop closed by brute force compared to a decoy with a broken loop
///           However for snugdock where the intention is to refine the
///           specified ( default h3, and h2 only if snugh2 is specified ), the
///           weights are physically weighted to unity.
///
/// @param[in]
///
/// @param[out]
///
/// @global_read antibody_ns::snugdock_ns::snugh2
///              antibody_ns::snugdock_ns::snugh3
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 02/06/2008
///
/// @last_modified 02/07/2008
///////////////////////////////////////////////////////////////////////////////
void
dle_refine_one_loop(
	pose_ns::Pose & pose_in,
	int loop_begin,
	int loop_end,
	int loop_cutpoint
)
{
	using namespace pose_ns;
	using namespace antibody_ns;
	using namespace dle_ns;

	pose_in.update_backbone_bonds_and_torsions();

	int nres = pose_in.total_residue();
	int loop_size = (loop_end - loop_begin ) + 1;

	Loops one_loop;
	one_loop.add_loop( loop_begin, loop_end, loop_cutpoint, loop_frag_offset,0);
	// loop tree
	pose_in.one_jump_tree( nres, loop_begin-1, loop_end+1, loop_cutpoint);
	pose_in.set_allow_jump_move(1, false); // fixed stems

	FArray1D_bool allow_moves( nres, false );
	// do not include neighbors
	pose_loops_select_loop_residues( pose_in, one_loop, false, allow_moves);
	pose_in.set_allow_bb_move( allow_moves );
	// include neighbors
	pose_loops_select_loop_residues( pose_in, one_loop, true, allow_moves);
	pose_in.set_allow_chi_move( allow_moves );

	int n_cutpoints( 1 );
	int outer_cycles( 1 );
	int nmoves( 1 );
	Score_weight_map weight_map;
	if( pose_in.fullatom() )
		setup_score_weight_map( weight_map, score12 );
	else
		setup_score_weight_map( weight_map, score4 );
	weight_map.set_weight( CHAINBREAK, 3.0 );
	weight_map.set_weight( CHAINBREAK_OVERLAP, 3.0 );
	FArray1D_float cut_weight( n_cutpoints, 3.0 );
	weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );
	// normal weights for snugdock
	if( snugdock_ns::snugh2 || snugdock_ns::snugh3 ) {
		weight_map.set_weight( CHAINBREAK, 1.0 );
		weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
		FArray1D_float cut_weight( n_cutpoints, 1.0 );
		weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );
	}

	if( refine_more ) {
		outer_cycles = 5;
		nmoves = 5;
		weight_map.set_weight( CHAINBREAK, 1.0 );
		weight_map.set_weight( CHAINBREAK_OVERLAP, 1.0 );
		FArray1D_float cut_weight( n_cutpoints, 1.0 );
		weight_map.set_1D_weight( CUT_WEIGHT, cut_weight );
	}

	bool const save_use_nblist( get_use_nblist_during_minimization() );
	pose_set_use_nblist( false );

	pose_refine_loops_with_repack_n_ccd( pose_in, weight_map, one_loop,
																			 outer_cycles, loop_size, nmoves, true,
																			 true );
	pose_set_use_nblist( save_use_nblist );

	pose_in.update_backbone_bonds_and_torsions();

	return;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dle_dump_next_pdb
///
/// @brief Dumps a scored pdb with a sequentially numbered name.
///
/// @detailed If a fullatom pose is detected then score 12 is used. Unscored
///           poses are output for centroid poses
///
/// @param[in]
///
/// @param[out]
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Aroop 07/15/2008
///
/// @last_modified 08/19/2008
///////////////////////////////////////////////////////////////////////////////
void
dle_dump_next_pdb(
	pose_ns::Pose & pose_in,
	bool modified_scoring,
  std::string const decoy_type = "decoy"
)
{
	using namespace pose_ns;

	return; // aroop: temp remove

	// triggering fullatom flag
	bool const fullatom( pose_in.fullatom() );

	// selecting score function
	pose_ns::Score_weight_map dump_wt_map;
	pose_ns::Score_weight_map default_dump_wt_map;
	setup_score_weight_map( dump_wt_map, score12 );
	setup_score_weight_map( default_dump_wt_map, score12 );
	if( modified_scoring )
		dle_modify_weightmap( pose_in, dump_wt_map);

	// Incrementing counter for each decoy previously outputted
	static int file_ticker(0);
	file_ticker++;
	std::stringstream counter;
	counter << file_ticker;
	std::stringstream filename_spacer;

	// spacer to account for uniform width of filename prefix
	if( file_ticker < 10 )
		filename_spacer << "000" ;
	else if( file_ticker < 100 )
		filename_spacer << "00";
	else if( file_ticker < 1000 )
		filename_spacer << "0";

	std::ifstream inp;
	std::string filename = filename_spacer.str() + counter.str()
		+ "_decoy.pdb";
	std::string mod_filename = filename_spacer.str() + counter.str()
		+ "_decoy_modified.pdb";

	if( fullatom ) {
		pose_in.dump_scored_pdb( filename, default_dump_wt_map );
		if( modified_scoring ) {
			pose_in.dump_scored_pdb( mod_filename, dump_wt_map );
		}
	}
	else
		pose_in.dump_pdb( filename );

	// output additional decoy if specific tag requested
	if( decoy_type != "decoy" ) {
		std::string alt_filename = filename_spacer.str() + counter.str()
			+ "_" + decoy_type + ".pdb";
		std::string mod_alt_filename = filename_spacer.str() + counter.str()
			+ "_" + decoy_type + "_modified.pdb";
		if( fullatom ) {
			pose_in.dump_scored_pdb( alt_filename, default_dump_wt_map );
			if( modified_scoring ) {
				pose_in.dump_scored_pdb( mod_alt_filename, dump_wt_map );
			}
		}
		else
			pose_in.dump_pdb( alt_filename );
	}

	return;
}

