// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
//  vi: set ts=2 noet:
//  CVS information:
//  $Revision: 7412 $
//  $Date: 2006-02-06 18:27:25 -0500 (Mon, 06 Feb 2006) $
//  $Author: aroop $
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/...
// (C) 199x-2005 University of Washington
// (C) 199x-2005 University of California Santa Cruz
// (C) 199x-2005 University of California San Francisco
// (C) 199x-2005 Johns Hopkins University
// (C) 199x-2005 University of North Carolina, Chapel Hill
// (C) 199x-2005 Vanderbilt University

/// @file   rosetta++/dle.h
///
/// @brief  Protocols to dock proteins(specifically antibody-antigen complexes)
///         using a loop ensemble. The following protocols build loops,generate
///         input files for clustering using R,carry out one round of refinemnt
///         & then swaps from this refined library during docking. Modules are
///         present for calculating loop-rmsg & loop-rmsl, ramping scores etc.
/// @author Aroop Sircar (aroop@jhu.edu)

#ifndef INCLUDED_dle
#define INCLUDED_dle

//Rosetta headers
#include "pose.h"

class Fragment;

/// @brief switches to pose mode and then calls function to build loops
void
dle_initialize();

/// @brief carries out specified operation for all loops defined in datafile
///        entered (default name = loopfile)
void
dle_multiple_loop_operations(
	pose_ns::Pose & pose,
  const std::string & dock_flex_func
);

/// @brief Outputs the loop library to a file to be read in later
void
dle_write_loop_library(
  pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
  std::vector< Fragment > & loop,
  int counter,
	const std::string & lib_refinement_status
);

/// @brief generates a score file and an rms table file which will be used by R
///        for hierarchial clustering
void
dle_cluster_loop_library(
	pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
	std::vector< Fragment > & loop_library,
	int counter
);

/// @brief reads in a loop fragment file from the disk
void
dle_read_in_one_loop_library(
	int loop_begin,
	int loop_end,
  int counter,
	std::vector< Fragment > & loop_library,
	const std::string & lib_refinement_status
);

/// @brief wrapper to load key residues and then define loop begin & end
void
dle_load_and_define_loops();

/// @brief actually builds each loop
void
dle_build_one_loop(
  pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
  std::vector< Fragment > & loop_library,
	int counter
);

/// @brief relaxes all loops in a loop library
void
dle_relax_loop_library(
 pose_ns::Pose & pose_in,
 int loop_begin,
 int loop_end,
 std::vector< Fragment > & crude_loop_library,
 int counter
);

/// @brief helps to insert each loop into the protein
float
dle_insert_loops(
  pose_ns::Pose & pose,
	float trans_mag,
	float rot_mag
);

/// @brief actually inserts the loops into the decoy
float
dle_insert_one_loop(
  pose_ns::Pose & pose,
	std::vector< Fragment > & loop,
	int const loop_begin,
	int const loop_end,
	float trans_mag,
	float rot_mag
);

/// @brief building the cter from scratch
void
dle_cter(
  pose_ns::Pose & pose,
	float trans_mag,
	float rot_mag
);

/// @brief wrapper to pack neighboring residues of all modeled regions
void
dle_calc_loops_environ(
	pose_ns::Pose & pose,
	FArray1D_bool & allow_repack
);

/// @brief find neighbors within a specified threshold
void
dle_loop_neighbor(
	pose_ns::Pose & pose,
	int loop_begin,
	int loop_end,
	FArray1D_bool & allow_repack
);

/// @brief wrapper function which in turn relaxes the regions in question
void
dle_relax(
  pose_ns::Pose & pose
);

/// @brief actually relaxes the region specified
void
dle_loop_fa_relax(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
);

/// @brief reads in data file containing the key residues - can be either pivot
///        or loop residues
void
dle_load_key_residues(
  int & total_key_residues,
  int key_residue[]
);

/// @brief translates crude data file data to loop_begin and loop_end data
void
dle_define_loop(
  int & total_key_residues,
  int key_residue[]
);

// @brief returns true if two residues are near one another
bool
dle_is_adjacent(
	pose_ns::Pose & pose,
	int const first_residue,
	int const second_residue
);

/// @brief calculates rmsd over C-Alpha atoms which are not flexible.
///        Is also responsible for dumping_coords
void
dle_rmsd_debug(
  pose_ns::Pose & pose,
	pose_ns::Pose & fixed_pose,
	pose_ns::Fold_tree const docking_tree,
	const std::string & dump_filename
);

/// @brief ramping up the fullatom repulsive weight slowly to allow the partners
///        to relieve clashes and make way for each other
void
dle_repulsive_ramp(
  pose_ns::Pose & pose
);

/// @brief compute the rms global for the first loop in input loopfile
float
dle_first_loop_rmsg();

/// @brief compute the rms global for a loop between input pose & native
float
dle_loop_rmsg(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
);

/// @brief compute the rms global for a loop between two input poses
float
dle_loop_rmsg(
	const pose_ns::Pose & decoy_pose1,
	const pose_ns::Pose & decoy_pose2,
	int const loop_begin,
	int const loop_end
);

/// @brief compute the rms local for a loop
float
dle_loop_rmsl(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
);

/// @brief checks to see if a loop if inserted into a protein(pose) passes the
///        H3 filter or not
bool
dle_CDR_H3_check(
  const pose_ns::Pose & pose,
	const pose_ns::Pose & loop_pose,
	int const loop_begin,
	int const size
);

/// @brief tests if a loop has H3 like base charachteristics
bool
dle_CDR_H3_filter(
  const pose_ns::Pose & pose,
	int const loop_begin,
	int const size
);

/// @brief brings two proteins into the point of glancing contact
void
dle_slide_into_contact(
	pose_ns::Pose & pose
);

/// @brief evaluate centroid scores of loop in protein
float
dle_intra_clash_check(
  const pose_ns::Pose & pose,
  Fragment & f,
  int begin,
  int size,
  int cutpoint
);

/// @brief orient the complex based on superimposition of the first partner to
///        the first partner in the native
void
dle_orient_to_native(
	pose_ns::Pose & pose
);

/// @brief builds a loop from fragments file.
void
dle_scored_frag_close(
	pose_ns::Score_weight_map const & weight_map,
	pose_ns::Pose & pose_in,
	int const loop_begin,
	int const loop_end,
	int const cutpoint,
	int const frag_offset,
	int const cycles1,
	int const cycles2,
	int counter,
	bool const do_ccd_moves
);

/// @brief incorporation of chain-break and chain-overlap
void
dle_modify_weightmap(
	pose_ns::Pose & pose,
	pose_ns::Score_weight_map & weight_map
);

/// @brief loop creation for docking whilst iterating through a loop library
bool
dle_process_loops(
	pose_ns::Pose & docking_pose
);

/// @brief outputs torsional angles of loop residues to a text file
void
dle_write_one_loopfile(
	pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end,
	int counter
);

/// @brief reads in options common to loop ensemble docking and antibody
///        modelling
void
dle_common_options();

/// @brief reads in a pose and saves it in docking_unbound namespace.
///        Also computes chi angles and saves it in the namespace
void
dle_save_unbound(
	pose_ns::Pose & pose_in
);

/// @brief packs pose using extra rotamers from unbound
void
dle_pack_with_unbound(
	pose_ns::Pose & pose,
	FArray1D_bool const & allow_repack,
	bool const include_current
);

/// @brief moves the loop apex residue and then tries to close the loop
void
dle_res_rb_move(
  pose_ns::Pose & pose,
	int const loop_begin,
	int const loop_end
);

/// @begin dle_trim_and_repack
void
dle_trim_and_repack(
	pose_ns::Pose & pose_in,
	int const loop_begin,
	int const loop_end
);

/// @brief choose a random cutpoint for a loop fold tree
int
dle_choose_random_cutpoint(
	int const loop_begin,
	int const loop_end
);

/// @brief wrapper to compute the separation at cutpoint ( uses pose input )
float
dle_cutpoint_separation(
  pose_ns::Pose & pose_in
);

/// @brief compute the separation at the cutpoint
float
dle_cutpoint_separation();

/// @brief repack multiple times over with rotamer trials and rt_min
void
dle_extreme_repack(
	pose_ns::Pose & pose_in,
	int repack_cycles,
	FArray1D_bool & allow_repack,
	bool rt_min,
	bool rotamer_trials,
	bool force_one_repack,
	bool use_unbounds = false
);

/// @brief detects hydrophobic atoms & calls wrapper for hydrophobic score eval
float
dle_hydrophobic();

/// @brief computes the hydrophobic term as defined by Friesner
void
dle_hydrophobic_eval(
	float & deltaf,
	float distance,
	float cutoff,
	int const res1,
	int const res2
);

/// @brief detects residues that have bad ( > 10) Erep scores and repacks them
void
dle_bad_Erep_repack(
  pose_ns::Pose & pose_in
);

/// @brief wrapper to refine specified loops
void
dle_refine_modeled_loop(
  pose_ns::Pose & pose_in
);

/// @brief overloaded wrapper to refine specified loops
void
dle_refine_modeled_loop(
	pose_ns::Pose & pose_in,
  const std::string & cdr_loop_name
);

/// @brief Extensively refines a specified loop
void
dle_refine_one_loop(
	pose_ns::Pose & pose_in,
	int loop_begin,
	int loop_end,
	int loop_cutpoint
);

/// @brief Dumps a scored pdb with a sequentially numbered name.
void
dle_dump_next_pdb(
	pose_ns::Pose & pose_in,
	bool modified_scoring,
  std::string const decoy_type
);

#endif
