// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 32323 $
//  $Date: 2009-08-25 13:16:13 -0700 (Tue, 25 Aug 2009) $
//  $Author: johnk $


// Rosetta Headers
#include "dock_structure.h"
#include "after_opts.h"
#include "dock_loops.h"
#include "dock_pivot.h"
#include "docking.h"
#include "docking_constraints.h"
#include "docking_minimize.h"
#include "docking_movement.h"
#include "docking_score.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "fullatom.h"
#include "initialize.h"
#include "ligand.h"
#include "make_pdb.h"
#include "monte_carlo.h"
#include "misc.h"
#include "output_decoy.h"
#include "pack_fwd.h"
#include "param.h"
#include "pose_docking.h"
#include "pose_peptide_docking.h"
#include "prof.h"
#include "recover.h"
#include "rotamer_trials.h"
#include "runlevel.h"
#include "random_numbers.h"
#include "score.h"
#include "score_ns.h"
#include "status.h"
#include "util_vector.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <ObjexxFCL/string.functions.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <cstdlib>
#include <fstream>
#include <iostream>


//     dock_structure_ns.cc: the main function for a docking trial
//                       plus some supporting functions
//
//
//     Jeff Gray April 2001
//     revised November 2001 (post-capri)
//
//

//     dock_structure
//     docking_prepack_protocol
//     docking_gaussian_perturbation
//     docking_perturb_decoy
//     docking_store_initial_rms
//     make_named_numbered_pdb
//     make_named_pdb
//     rigid_body_MC_cycle_adaptive
//     rigid_body_MC_cycle


//------------------------------------------------------------------------------
//
//     the main docking protocol
//

////////////////////////////////////////////////////////////////////////////////
/// @begin dock_structure
///
/// @brief main docking protocol
///
/// @detailed
/// Main function for docking. Includes the following steps:
///      0) prepack mode: prepare a starting structure for later runs
///   OR:
///      1) perturbation of decoy (see docking_perturb_decoy): changes
///         orientation of docking partners
///      2) low-resolution search:
///         refine structure with rigid-body, centroid-mode MC cycles
///      3) high-resolution search:
///         further refinement of structure in fullatom mode
///
/// @global_read
/// standard,runlevel (in runlevel.h) - documentation is runlevel dependent
///
/// @global_write
/// score (in misc.h) - score of decoy evaluated at the beginning of the
///                     fullatom part.
///
/// @remarks
/// many indirect changes to global variables (score and structure related).
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
dock_structure( bool & fail )
{
	using namespace misc;
	using namespace runlevel_ns;

	if ( runlevel > standard ) std::cout << "dock_struct" << std::endl;

	fail = false; //default

  if( truefalseoption( "flex_peptide" ) ) {// SJF
    pose_docking_peptide_main( fail );
    return;
  }

	if ( get_pose_docking_flag() ) { // pose_docking has its own prepack protocol
		pose_docking_main( fail );
		return;
	} else if ( get_docking_prepack_mode() ) {
		docking_prepack_protocol();
		return;
	}

	if ( get_docking_silent_input_flag() ) {
		docking_read_silent_input( fail );
		if ( ! get_docking_local_refine() ) return;
	}

	//Skip the perturbation and centroid MC if requested
	if ( get_docking_local_refine() ) goto L100;

	// then start normal docking protocol

//-----Configure other parts of Rosetta for docking-style scoring
	score_set_try_rotamers(false); // don't let scoring function change rotamers
	set_fullatom_flag(false);
	if( get_ligand_flexible_flag()){
		if( !generate_base_ligand_conformations() ){
			std::cout << "Failed to generate ligand conformations correctly";
			std::cout << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
	}

//-----Make starting perturbations

	docking_perturb_decoy(); // includes monte_carlo_reset w/score4d
	calc_docking_rmsd();
	docking_store_initial_rms(); // need this for future comparisons

//-----Monte Carlo cycles (backbone)

	dock_flexible_loops_setup(); // includes some loop building

	if ( !get_ligand_flag() && !get_docking_local_refine() ) {
		rigid_body_MC_cycle_adaptive(10,50,0.7,5.0,false,score4d);
	}
 L100:

	if ( docking::dock_generate_diversity ) {
		// Put back the starting sidechains...
		docking::norepack1 = true;
		docking::norepack2 = true;
		recover_LOW(score4d); // save score4 output and structure
		docking_pivot_store_cen_scores();
		set_docking_interface_pack( false );
		select_rotamer_set( "large" );
		set_fullatom_flag(true);
		docking_repack(true);
		mc_global_track::mc_score::score = score10d();
		monte_carlo_reset();
	}

	if ( get_docking_fullatom_flag() ) {

		recover_LOW(score4d); // save score4 output and structure
		docking_pivot_store_cen_scores();

		set_docking_interface_pack( true ); // pack only the interface
		select_rotamer_set( "large" );

		set_fullatom_flag(true);

		pivot_repack(true); // repacks residues perturbed by pivot
		PROF_START(prof::DOCK_REPACK);
		docking_repack(true); // repacks interface residues
		PROF_STOP(prof::DOCK_REPACK);
		if (docking::dock_rtmin) { // rtmin interface residues if requested
			score_set_try_rotamers(true);
			score_set_minimize_rot(true);
			mc_global_track::mc_score::score = score10d();
			score_set_minimize_rot(false);
			score_set_try_rotamers(false);
		} else {
			mc_global_track::mc_score::score = score10d();
		}
		monte_carlo_reset();

//--------Monte Carlo cycles (fullatom) !! this is no longer used (5/02)

		if ( get_docking_FAsearch_flag() ) {
			rigid_body_MC_cycle_adaptive(2,10,0.04,0.2,true,score10d);
			recover_LOW(score10d);
		}

//--------Minimize structure (fullatom)

		if ( get_docking_mcm_flag() ) {
			docking_mcm_protocol();
		} else if ( get_docking_minimize_flag() ) {
			docking_rigidbdy_minimize_trial("dfpmin",score10d_min,0.02);
		}

	} // fullatom

}



//------------------------------------------------------------------------------
//
//     this 'prepack' protocol is designed to prepare a starting
//     structure for later runs.  the docking partners are separated in
//     space and repacked (all residues), { placed back together.  The
//     resulting structure can be used for runs where only the interface
//     is repacked.  In this way, we have a 'level playing field' to
//     compare the absolute scores of decoys where different interface
//     patches have been repacked.  jjg 9/01
//

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_prepack_protocol
///
/// @brief prepare a starting structure for later runs
///
/// @detailed The docking partners are separated in space and repacked (all residues),
///     then placed back together. The resulting structure can be used for runs
///     where only the interface is repacked. In this way, we have a 'level
///     playing field' to compare the absolute scores of decoys where different
///     interface patches have been repacked.
///
/// @global_read
///      NAME_LENGTH (files_paths.h)
///      protein_name (files_paths.h)
/// @global_write
///      scores are changed:
///         score (in misc.h), docking_scores block in score_ns.h:
///         DOCKING_VDW_SCORE,DOCKING_PAIR_SCORE,DOCKING_CONTACT_SCORE
///
/// @remarks this function affects a range of arrays and variables that are score
///       and structure related.
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_prepack_protocol()
{
	using namespace docking;
	using namespace files_paths;
	using namespace misc;
	using namespace scores;

	std::string prepack_output_filename, reppk_output_filename;
	float const away_dist = { 99.0 };
	static FArray1D_float const zero( 3, 0.0f );
	FArray1D_float T( 3 );

	std::cout << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl;
	if ( prepack_full && prepack_rtmin ) {
		std::cout << "  prepacking: full repack + rot_trial with minimization" << std::endl;
	} else if ( prepack_full ) {
		std::cout << "  prepacking: full repack only" << std::endl;
	} else if ( prepack_rtmin ) {
		std::cout << "  prepacking: rot_trial with minimization only" << std::endl;
	} else {
		error_stop("Prepack mode not defined...");
	}
	std::cout << ";;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;" << std::endl;
//mj jump out if ligand flag
//      if ( get_ligand_flag() ) {
//         std::cout << "ERROR: Called centroid function with ligand flag:" <<
//          " docking_prepack_protocol in dock_structure_ns.cc" << std::endl;
//         return
//      }

	if ( get_ligand_flag() ) {
		score_set_default_function(true); // full atom mode scoring
	} else {
		score_set_default_function(false);
		 // centroid mode scoring only, no side chains
	}
	score_set_evaluate_all_terms(true);
	mc_global_track::mc_score::score = scorefxn();
	make_named_pdb( "before.pdb", true );
	scorefile_output( "before.pdb", "start", true );

//--   move away
	create_docking_trans_vector(T,-away_dist,0.0,0.0);
	apply_rigid_body(T,zero);
	mc_global_track::mc_score::score = scorefxn();
	save_status_info("slide_out",0,0);
	monte_carlo_reset();


//--   enter fullatom mode, score and output
	score_set_try_rotamers(false);
	set_fullatom_flag(true);
	score_set_default_function(true);
	score_set_evaluate_all_terms(true);
	mc_global_track::mc_score::score = scorefxn();
	make_named_pdb( "away.pdb", true );
	scorefile_output( "away.pdb", "away_not_repacked", true );

//--   repack, score
	set_docking_interface_pack( false );
	select_rotamer_set( "large" );

//--   repack, score
	if ( prepack_full ) {
    bool strip_rotamers = truefalseoption( "strip_native_rotamers" ); // SJF remove the native rotamers from input
		docking_repack( !strip_rotamers );
		mc_global_track::mc_score::score = scorefxn();
		monte_carlo_reset();
//--   output
		score_header_output( std::cout, false );
		std::cout << std::endl;
		score_output( std::cout, "repacked_away", false );
		std::cout << std::endl;
		scorefile_output( "repacked_away.pdb", "repacked_away", true );
		make_named_pdb( "repacked_away.pdb", true );
	}

//chu   rottrial with minimization
//     allow_rottrial will be set in scorefxn
	if ( prepack_rtmin ) {
		score_set_minimize_rot(true);
		score_set_try_rotamers(true);
		mc_global_track::mc_score::score = scorefxn();
		score_set_try_rotamers(false);
		score_set_minimize_rot(false);
		monte_carlo_reset();
//--   output
		score_header_output( std::cout, false );
		std::cout << std::endl;
		score_output( std::cout, "minimized_away", false );
		std::cout << std::endl;
		scorefile_output( "minimized_away.pdb", "minimized_away", true );
		make_named_pdb( "minimized_away.pdb", true );
	}

	if ( !get_ligand_flag() &&
	 ( docking_contact_score < 10.0 || docking_pair_score != 0.0 ||
	 docking_vdw_score != 0.0 ) )
	 error_stop("Separated partners appear to be interacting...");

//--   slide back and score (no repacking)
	create_docking_trans_vector(T,away_dist,0.0,0.0);
	apply_rigid_body(T,zero);
	mc_global_track::mc_score::score = scorefxn();
	save_status_info("slide_in",0,0);
	monte_carlo_reset();

//--   output
//     This is the repacked(or rtmined or both), put-back-in-place structure
//     to use for docking simulations (ppk=prepacked)
	prepack_output_filename = protein_name.substr(0,4) + ".ppk.pdb";
	score_header_output( std::cout, false );
	std::cout << std::endl;
	score_output( std::cout, "prepacked", false );
	std::cout << std::endl;
	scorefile_output( prepack_output_filename, "prepacked", true );
	make_named_pdb( prepack_output_filename, true );

//--   repack/mimimize the side chains of interface residues in ppk structure
	set_docking_interface_pack(true);
	select_rotamer_set( "large" );
	if ( prepack_full ) {
		docking_repack(true);
		mc_global_track::mc_score::score = scorefxn();
		monte_carlo_reset();
//--   output
		reppk_output_filename = protein_name.substr(0,4) + ".reppk.pdb";
		score_header_output( std::cout, false );
		std::cout << std::endl;
		score_output( std::cout, "rep_prepacked", false );
		std::cout << std::endl;
		scorefile_output( reppk_output_filename, "rep_prepacked", true );
		make_named_pdb( reppk_output_filename, true );
	}

	if ( prepack_rtmin ) {
		score_set_minimize_rot(true);
		score_set_try_rotamers(true);
		mc_global_track::mc_score::score = scorefxn();
		score_set_try_rotamers(false);
		score_set_minimize_rot(false);
		monte_carlo_reset();
//--   output
		reppk_output_filename = protein_name.substr(0,4) + ".remin.pdb";
		score_header_output( std::cout, false );
		std::cout << std::endl;
		score_output( std::cout, "re_rtmined", false );
		std::cout << std::endl;
		scorefile_output( reppk_output_filename, "re_rtmined", true );
		make_named_pdb( reppk_output_filename, true );
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin error_stop
///
/// @brief echo a message to the screen and exit
///
/// @detailed
///
/// @param[in]   message - input - message to be written to screen before exit
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
error_stop( std::string const & message )
{
//     echo a message to the screen and exit
	std::cerr << message << std::endl;
	utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
}


//------------------------------------------------------------------------------
//
//     there are several ways to perturb the structure before beginning
//     the search; they are controlled through command-line flags
//     see options.cc for more details
//
//     at the end, partners are slid into contact and scored
//

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_perturb_decoy
///
/// @brief   Make starting perturbations in docking
///
/// @detailed    There are several ways to perturb the structure before beginning
///     the search; they are controlled through command-line flags
///     (see options.cc for more details).
///     At the end, partners are slid into contact and scored (including
///     mc_reset).
///     Also, they are tested for passing the FAB filter.
///
/// @global_read
///  runlevel, standard, verbose (runlevel.h) -  runlevel dependent documentation
///  docking_flagflags block in docking.h:
///       DOCKING_RANDOMIZE1,DOCKING_RANDOMIZE2,DOCKING_SMALL_PERTURBATION,
///       DOCKING_AXIS_SPIN_FLAG
///    - logicals that define perturbation in docking
///  PART1_CENTROID,BEST_PART2_CENTROID (docking.h) - centroid of partner 1 and 2
/// @global_write
///  score  (in misc.h) - decoy is scored at the end
///  PART1_CENTROID - changed if DOCKING_RANDOMIZE1 is T
///  BEST_PART2_CENTROID - changed if DOCKING_RANDOMIZE2 is T, as well as
///       in all other perturbations
///
/// @remarks make sure that structure is updated (i.e. current = best),
///       since best_part2_centroid is used as basis for perturbations.
///       (for example randomizing partner2 if DOCKING_RANDOMIZE2 is TRUE;
///          docking_gaussian_perturbation, etc).
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_perturb_decoy()
{
	using namespace files_paths;
	using namespace param;
	using namespace misc;
	using namespace docking;
	using namespace runlevel_ns;

//     local
	bool struct_ready;

	if ( get_docking_local_refine() ) return; // option to skip perturbation/centroid search

	struct_ready = false;
	while ( !struct_ready ) {

//     retrieve_decoy_number(nstruct)
//     make_named_numbered_pdb('A',nstruct,false)
		if ( docking_randomize1 ) {
			if ( runlevel > standard ) std::cout << "randomizing partner 1 orientation" << std::endl;
			dock_random_orient(1,part_centroid(1,1));
			save_status_info("perturb_spin1",0,0);
			monte_carlo_reset();
		}
//     make_named_numbered_pdb('B',nstruct,false)
		if ( docking_randomize2 ) {
			if ( runlevel > standard ) std::cout << "randomizing partner 2 orientation" << std::endl;
			dock_random_orient(2,best_part_centroid(1,2));
			save_status_info("perturb_spin2",0,0);
			monte_carlo_reset();
		}
//     make_named_numbered_pdb('prePERT',nstruct,false)
		if ( docking_small_perturbation ) {
			docking_gaussian_perturbation();
			save_status_info("perturb_small",0,0);
			monte_carlo_reset();
		}
		calc_docking_rmsd();
//     make_named_numbered_pdb('PERT',nstruct,false)
		if ( docking_axis_spin_flag ) {
			docking_axis_spin();
			save_status_info("pert_ax_spin",0,0);
			monte_carlo_reset();
		}
//     make_named_numbered_pdb('E',nstruct,false)

//mj check on ligand flag
//      if ( !get_ligand_flag() ) {
		dock_slide_into_contact();
		save_status_info("perturbation_slide",0,0);
		mc_global_track::mc_score::score = score4d();
		monte_carlo_reset();
//      }
//     make_named_numbered_pdb('F',nstruct,false)

//     check that we pass the fab filter!
		struct_ready = docking_fab_filter();
		if ( !struct_ready ) {
			initialize_decoy();
			if ( runlevel > standard ) std::cout <<
			 "structure failed fab filter; re-randomizing" << std::endl;
		}

	}

	docking_pivot_perturb_segment(10.0);
	if ( runlevel >= verbose ) make_named_labelled_pdb("perturbed",false);

}




////////////////////////////////////////////////////////////////////////////////
/// @begin docking_gaussian_perturbation
///
/// @brief perturb initial starting position with a gaussian in both
///       translation and rotation.
///
/// @detailed
/// The initial perturbation of the structure (i.e. translation and rotation) is
/// defined by
///       1) a normal perturbation (a gaussian around NORMAL_PERTURBATION
///          Angstroms) along best_docking_normal axis
///       2) 2 parallel perturbations (gaussians around PARALLEL_PERTURBATION
///          Angstroms) along best_docking_parallel1/2 axes
///   and 3) 2 rotational perturbations (gaussians around ROTATIONAL_PERTURBATION
///          degrees) around spin- and tilt axes.
///          the tilt direction is randomly selected.
///   see docking.h for full description of docking geometry
/// Then, the score is evaluated and checked for passing docking_vdw and contact
/// score filters.
///
/// @global_read
///  NORMAL_PERTURBATION,PARALLEL_PERTURBATION,ROTATIONAL_PERTURBATION (in docking.h)
///  standard,runlevel (in runlevel.h) - documentation is runlevel dependent
///
/// @global_write
///  score (in misc.h) - score of decoy updated at end of perturbation
///  DOCKING_VDW_SCORE (in score_ns.h) - vdw score across interface (centroid mode)

//\NOTES
//       dock_structure calls monte_carlo_reset to force-accept perturbation.
//
//       score_ns.h is included for docking_vdw_score: should be updated
//       need to refer to appropriate place for description of docking geometry
//
//\COMMENTERS Ora 8/19/03
//
//\END
////////////////////////////////////////////////////////////////////////////////
void
docking_gaussian_perturbation()
{
//     perturb initial starting position for each structure with a gaussian
//     in both translation and rotation
//     7/11/1

	using namespace docking;
	using namespace misc;
	using namespace runlevel_ns;
	using namespace scores;

//     local
	FArray1D_float translation( 3 );
	FArray1D_float rotation( 3 );
//      float normal_perturbation;     // A
//      float parallel_perturbation;   // A
//      float rotational_perturbation; // degrees

//mj check on ligand flag
//      if ( get_ligand_flag() ) {
//         if ( runlevel > standard ) std::cout <<
//          "Applying initial perturbation to start: " <<
//          SS( normal_perturbation ) << " A/" << SS( parallel_perturbation ) <<
//          " A/" << SS( rotational_perturbation ) << " degrees" << std::endl;
//
//         choose_rigid_body(translation,rotation,normal_perturbation,
//          parallel_perturbation,rotational_perturbation);
//         apply_rigid_body(translation,rotation);
//
//         if ( runlevel > standard ) {
//            vector_name_write_line( std::cout,translation,"translation");
//            vector_name_write_line( std::cout,rotation,"rotation");
//         }
//
//         return;
//      }

	docking_vdw_score = 1000.0;
	float vdw_level = 10.0;
	int count = 0;

//   be sure to pick a starting structure that is not terribly eclipsed
//   or not in contact
//   8/31: not in contact is ok since we follow this by a slide-into-contact
//mj    while ( docking_vdw_score >= 300.0 ) // || docking_contact_score >= 9.5 ) {

//mj for ligand docking search until you find a non-bumping position -
//mj we don't want to slide far!
//mj avoid endless loop
	if( get_ligand_flexible_flag() ){
		//kwk here we recover all the precompute ligand conformations
		for( std::vector<Ligand *>::iterator cur_ligand=
			ligand::ligand_ptr_vector.begin();cur_ligand!=
			ligand::ligand_ptr_vector.end(); cur_ligand++){

			(*cur_ligand)->ligand_conformations.clear();
			(*cur_ligand)->ligand_conformations=
				(*cur_ligand)->ligand_conformations_base;
		}
	}

	while ( docking_vdw_score >= 300.0 ||
	 ( get_ligand_flag() && docking_vdw_score >= vdw_level ) ) {
		if ( runlevel > standard ) std::cout <<
			"Applying initial perturbation to start: " <<
			SS( normal_perturbation ) << " A/" << SS( parallel_perturbation )
			<< " A/" << SS( rotational_perturbation ) << " degrees"
			<< std::endl;

			choose_rigid_body(translation,rotation,normal_perturbation,
				parallel_perturbation,rotational_perturbation);
			apply_rigid_body(translation,rotation);

			if( get_ligand_flexible_flag() ){
				size_t lig_conf=static_cast<size_t>(random_range(0,
					(*ligand::ligand_one).ligand_conformations.size()));
//				std::cout << "kwk help lig_conf " << lig_conf << std::endl;
				(*ligand::ligand_one).change_to_ligand_conformation(lig_conf);
			}
			if ( runlevel > standard ) {
			  vector_name_write_line( std::cout, translation, "translation" );
			  vector_name_write_line( std::cout, rotation, "rotation" );
			}


		  mc_global_track::mc_score::score = score4d();

		  if ( get_ligand_flag() ) {
			  ++count;
			  if ( count == 1000 ) {
				  count = 0;
				  vdw_level *= 2.0;
				  if ( runlevel > standard ) {
					  std::cout << "docking_gaussian_perturbation: 1000 trials done!" << std::endl;
					  std::cout << "no non-clashing position found!" << std::endl;
					  std::cout << "set vdw_level to " << SS( vdw_level ) << std::endl;
				  }
			  }
		  }


	}

// dock_structure calls monte_carlo_reset to force-accept perturbation

}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_reset_tracking
///
/// @brief reset score tracking
///
/// @detailed
/// resets scores and rms of different stages to 0.0. Called at start of
///      docking protocol.
///
/// @global_read
/// @global_write
///      CENMODE_RMS,INITIAL_RMS - rms of different stages (in docking.h)
///      SCORE_BEFORE_MCM,SCORE_AFTER_ONE_MIN,SCORE_AFTER_FIVE_MCM - scores of
///         different stages (in docking.h): reset all to 0.0
///
/// @remarks
/// often called from docking_store_initial_rms before updating initial rms.
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_reset_tracking()
{
	using namespace docking;

	initial_rms = 0.0;
	cenmode_rms = 0.0;
	score_before_mcm = 0.0; // reset score tracking
	score_after_one_min = 0.0;
	score_after_five_mcm = 0.0;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_store_initial_rms
///
/// @brief stores current docking rms value in initial_rms and resets score tracking.
///
/// @detailed
///
/// @global_read docking_rmsd (in docking.h) -  current rmsd value
///
/// @global_write initial_rms (in docking.h) -  set to current docking_rmsd value
///
/// @remarks
/// called at start of protocol.
/// this initializes also the following:
/// cenmode_rms,score_before_mcm,score_after_one_min,score_after_five_mcm

//\COMMENTERS Ora 8/19/03
//
//\END
////////////////////////////////////////////////////////////////////////////////
void
docking_store_initial_rms()
{
	using namespace docking;

	docking_reset_tracking();
	initial_rms = docking_rmsd;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_get_initial_rms
///
/// @brief retrieves initial rms value for score output
///
/// @detailed
///
/// @return  initial_rms
///
/// @global_read initial_rms (in docking.h): rms at start of protocol
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
docking_get_initial_rms()
{
	using namespace docking;

	return initial_rms;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_store_cenmode_rms
///
/// @brief stores rms of structure obtained after centroid-mode refinement
///
/// @detailed before starting the fullatom docking_mcm protocol, the rms of the structure
///      is stored.
///
/// @global_read
///   docking_rms (in docking.h) - current docking_rms of structure
/// @global_write
///   cenmode_rms (in docking.h) - rms of structure after centroid-mode refinement
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_store_cenmode_rms()
{
	using namespace docking;

	cenmode_rms = docking_rmsd;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_get_cenmode_rms
///
/// @brief retrieves cenmode_rms (rms after centroid mode refinement) for score output
///
/// @detailed
///
/// @return  cenmode_rms
///
/// @global_read cenmode_rms (in docking.h) - rms of structure after centroid-mode refinement
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
docking_get_cenmode_rms()
{
	using namespace docking;

	return cenmode_rms;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_get_score_before_mcm
///
/// @brief retrieve score_before_mcm
///
/// @detailed retrieves score after centroid mode refinement for score output.
///      the score was evaluated in a first docking_mcm_filter of the docking_mcm_protocol.
///
/// @return  score_before_mcm
///
/// @global_read score_before_mcm (in docking.h)
///
/// @global_write
///
/// @remarks score_before_mcm and cenmode_rms correspond to the structure before full atom refinement
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
docking_get_score_before_mcm()
{
	using namespace docking;

	return score_before_mcm;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_get_score_after_one_min
///
/// @brief retrieve score_after_one_min
///
/// @detailed retrieves score after one FA rigid body minimization for score output.
///      the score was evaluated in the second docking_mcm_filter of the docking_mcm_protocol.
///
/// @return  score_after_one_min
///
/// @global_read score_after_one_min (in docking.h)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
docking_get_score_after_one_min()
{
	using namespace docking;

	return score_after_one_min;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_get_score_after_5_mcm
///
/// @brief retrieve score_after_5_mcm
///
/// @detailed retrieves score after 5 steps of rigid body minimization/moves for score output.
///      the score was evaluated in the third docking_mcm_filter of the docking_mcm_protocol.
///
/// @return  score_after_5_mcm
///
/// @global_read score_after_5_mcm (in docking.h)
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
docking_get_score_after_5_mcm()
{
	using namespace docking;

	return score_after_five_mcm;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin make_named_numbered_pdb
///
/// @brief writes out a named and numbered pdb file.
///
/// @detailed  the pdb output name is defined by attaching the filebase input to the number.
///       then this name is used to call make_named_pdb.
///
/// @param[in]   filebase - input - string to be attached (example: 'A')
/// @param[in]   number - input -   number to be attached (example: decoy number)
/// @param[out]   fullatom - input - bool: output of full atom coordinates if TRUE
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_named_numbered_pdb(
	std::string const & filebase,
	int number,
	bool fullatom
)
{
	// this function attaches a number to the filebase given

	make_named_pdb( filebase + '_' + lead_zero_string_of( number, 4 ) + ".pdb", fullatom );
}


////////////////////////////////////////////////////////////////////////////////
/// @begin make_named_labelled_pdb
///
/// @brief writes out a named and labeled pdb file.
/// @detailed  the pdb output name is defined by attaching the filebase input
///       to the current decoy name.
///       This name is used to call make_named_pdb.
///
/// @param[in]   filebase - input - string to be attached (example: 'perturbed')
/// @param[out]   fullatom - input - bool: output of full atom coordinates if TRUE
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_named_labelled_pdb(
	std::string const & filebase,
	bool fullatom
)
{
//     this function attaches the current decoy name to the filebase input

	using namespace files_paths;

//     local
	std::string filename;
	retrieve_decoy_name(filename);
	filename = filebase + '_' + filename;
	make_named_pdb(filename,fullatom);
}


////////////////////////////////////////////////////////////////////////////////
/// @begin make_named_pdb
///
/// @brief writes out a pdb file named "filename"
///
/// @detailed
///
/// @param[in]   filename - input -  name of pdb file
/// @param[out]   fullatom - input -  bool: output of full atom coordinates if TRUE
///
/// @global_read
///  pdb_out_path (files_path.h) - ouput directory for pdb files
///
///  runlevel,standard (in runlevel.h): runlevel dependent documentation
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
make_named_pdb(
	std::string const & filename,
	bool const fullatom
)
{
	using namespace files_paths;
	using namespace runlevel_ns;

//     local
	std::string fullname;

	fullname = pdb_out_path + filename;
	if ( runlevel > standard ) std::cout << "Outputting " <<
	 fullname << "..." << std::endl;

	utility::io::ozstream pdb_outstream( fullname );
	if ( !pdb_outstream ) {
		std::cout << "Open failed for file: " << pdb_outstream.filename() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	if ( fullatom ) {
		output_fullcoord_pdb(pdb_outstream);
	} else {
		output_position_pdb(pdb_outstream);
	}
//mj call hetero atom outout
	if ( get_ligand_flag() ){
		for( std::vector< Ligand * >::iterator c_ligand=ligand::ligand_ptr_vector.begin();
	    c_ligand!=ligand::ligand_ptr_vector.end(); c_ligand++){

		  make_pdb_hetero(pdb_outstream,(*(*c_ligand)));
	  }
	}
	output_pdb_stats(pdb_outstream,fullatom);
	if( get_flexbb_docking_flag() ) output_torsion_angles(pdb_outstream);
	pdb_outstream.close();
	pdb_outstream.clear();
}


////////////////////////////////////////////////////////////////////////////////
/// @begin rigid_body_MC_cycle_adaptive
///
/// @brief Perform several cycles of rigid-body Monte Carlo moves
///       and adapt the step size.
/// @detailed
///
/// @param[in]   cycles_in - input - number of MC cycles to run
/// @param[in]   c_attempts_in - input - number of steps per cycle
/// @param[in]   trans_magnitude_in - input - initial magnitude of rigid-body translation
/// @param[in]   rot_magnitude_in - input - initial magnitude of rigid-body rotation
/// @param[in]   fullatom - input - flag for fullatom mode
/// @param[in]   scorefxn - input - scoring function to use
///
/// @global_read
///
/// @global_write
///
/// @remarks
///       currently used only in the low-resolution step (centroid mode)
///       for the high-resolution step, minimization has been included: ( see
///       docking_mcm_protocol).
///
///       fullatom flag and scoring function should match:
///       e.g. use fullatom scoring function if fullatom flag is T
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rigid_body_MC_cycle_adaptive(
	int cycles_in, // number of MC cycles to run
	int c_attempts_in, // number of steps per cycle
	float trans_magnitude_in, // initial magnitude of rigid-body translation
	float rot_magnitude_in, // initial magnitude of rigid-body rotation
	bool fullatom, // flag for fullatom mode
	Scoring_Function scorefxn
)
{
//     Perform several cycles of rigid-body Monte Carlo moves
//     and adapt the step size

	using namespace runlevel_ns;

//     local
	int cycles; // number of MC cycles to run
	int c_attempts; // number of steps per cycle
	float trans_magnitude; // magnitude of rigid-body translation
	float rot_magnitude; // magnitude of rigid-body rotation
	float pivot_magnitude; // magnitude of pivot (phi/psi) perturbation, if applicable
	float success_rate; // ratio of accepted to total moves
	float pivot_success_rate; // dock_pivot MC success rate, if applicable

// _in arguments are constants so we must store in a new place before changing
	cycles          = cycles_in;
	c_attempts      = c_attempts_in;
	trans_magnitude = trans_magnitude_in;
	rot_magnitude   = rot_magnitude_in;
	pivot_magnitude = 10.0; // set explicitly here, so it does not have to
	                        // be a parameter to this function

	if ( benchmark ) {
		cycles = 1;
		c_attempts = 5;
	} else if ( docking::dock_generate_diversity ) {
		cycles = 3;
		c_attempts = 25;
	} else if ( get_dock_pivot_flag() ) {
		cycles = 2;
		c_attempts = 20;
	}

	if ( runlevel > standard ) {
		std::cout << "==============================" << std::endl;
		std::cout << "Beginning " << cycles << " rigid body monte carlo cycles" << std::endl;
		std::cout << c_attempts << " move attempts each" << std::endl;
		if ( !fullatom ) std::cout << "Backbone mode" << std::endl;
		if ( fullatom ) std::cout << "Fullatom mode" << std::endl;
		std::cout << cycles*c_attempts << " total moves" << std::endl;
		std::cout << "==============================" << std::endl;
	}

	for ( int i = 1; i <= cycles; ++i ) {
		if ( runlevel > chat ) make_named_numbered_pdb("rb",i,false);
		rigid_body_MC_cycle(c_attempts,trans_magnitude,rot_magnitude,
		 pivot_magnitude,fullatom,scorefxn,success_rate,pivot_success_rate);
		if ( success_rate < 0.5 ) { // adjust step sizes
			trans_magnitude *= 0.9;
			rot_magnitude *= 0.9;
		} else {
			trans_magnitude *= 1.1;
			rot_magnitude *= 1.1;
		}

		adjust_pivot_MC_step( pivot_magnitude, pivot_success_rate );

		if ( docking_jumpout_check() ) return;
	}

}



////////////////////////////////////////////////////////////////////////////////
/// @begin rigid_body_MC_cycle
///
/// @brief Perform a cycle of rigid-body Monte Carlo moves
///
/// @detailed  Performs a number (nattempts) of MC rigid-body moves
///       (of size trans_magnitude, rot_magnitude). The number of successful
///       attempts is returned.
///
/// @param[in]   attempts - input -    number of MC moves to attempt
/// @param[in]   trans_magnitude - input - magnitude of rigid-body translation
/// @param[in]   rot_magnitude - input - magnitude of rigid-body rotation
/// @param[in]   fullatom - input - flag for fullatom mode
/// @param[in]   scorefxn - input - scoring function to use
/// @param[out]   success_rate - output - ratio of accepted to total moves
///
/// @global_read
///
/// @global_write
///
/// @remarks called from rigid_body_MC_cycle_adaptive. the success_rate defines
///       whether the translation/rotation size is increased or decreased for
///       the next cycle.
///       currently used only in the low-resolution step (centroid mode)
///       for the high-resolution step, minimization has been included: ( see
///       docking_mcm_protocol).
///
///       fullatom flag and scoring function should match:
///       e.g. use fullatom scoring function if fullatom flag is T
///
/// @references
///
/// @authors Ora 8/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
rigid_body_MC_cycle(
	int const attempts, // number of MC moves to attempt
	float const trans_magnitude, // magnitude of rigid-body translation
	float const rot_magnitude, // magnitude of rigid-body rotation
	float const pivot_magnitude, // magnitude of pivot residue phi/psi perturbation
	bool const fullatom, // flag for fullatom mode
	Scoring_Function scorefxn,
	float & success_rate, // ratio of accepted to total moves
	float & pivot_success_rate // pivot MC success rate
)
{
// Perform a cycle of rigid-body Monte Carlo moves

	using namespace runlevel_ns;

	std::string move_type;
	if ( fullatom ) {
		move_type = "FA_rigid_body";
	} else {
		move_type = "rigid_body";
	}

	if ( runlevel > standard ) std::cout << std::endl;
	if ( runlevel > standard ) std::cout << "-----rigid body MC cycle (" <<
	 move_type << ") -----" << std::endl;
	if ( runlevel > standard ) std::cout <<
	 "Translation magnitude: " << F( 7, 3, trans_magnitude ) << std::endl;
	if ( runlevel > standard ) std::cout <<
	 "Rotation magnitude:    " << F( 7, 3, rot_magnitude ) << std::endl;
	if ( runlevel > standard && get_dock_pivot_flag() ) std::cout <<
	 "Pivot magnitude:       " << F( 7, 3, pivot_magnitude ) << std::endl;

	int accepts = 0;
	int pivot_accepts = 0;

	for ( int i = 1; i <= attempts; ++i ) {

		main_rigid_body_trial( scorefxn, i, fullatom, move_type, trans_magnitude,
		 rot_magnitude );

		if ( get_monte_carlo_accept() >= 1 ) {
			++accepts;
		}

		perform_pivot_MC(scorefxn,i,fullatom,pivot_magnitude,pivot_accepts);

		if ( docking_jumpout_check() ) goto L12; // exit
//		if ( ( accepts / 2 )*2 == accepts ) { // output every other file
//			filenumber = filenumber_base + i;
//			make_named_numbered_pdb("MC",filenumber,false);
//		}

	}

L12:
	if ( attempts != 0 ) {
		success_rate = static_cast< float >( accepts ) / attempts;
		pivot_success_rate = static_cast< float >( pivot_accepts ) / attempts;
	} else {
		success_rate = 0.0;
		pivot_success_rate = 0.0;
	}

	if ( runlevel > standard ) std::cout <<
	 "RBMC, Completed cycle.  Successes = " <<
	 accepts << "/" << attempts <<
	 " (" << success_rate * 100.0 << "%)" << std::endl;
	if ( runlevel > standard && get_dock_pivot_flag() ) std::cout <<
	 "Pivot MC: Completed cycle.  Pivot successes = " <<
	 pivot_accepts << "/" << attempts <<
	 " (" << pivot_success_rate * 100.0 << "%)" << std::endl;

//	filenumber_base = ((filenumber_base+attempts+99)/100) * 100; // round up to the next 100

}
