// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 17136 $
//  $Date: 2007-09-11 18:30:12 -0700 (Tue, 11 Sep 2007) $
//  $Author: chu $


// Rosetta Headers
#include "docking_score.h"
#include "after_opts.h"
#include "dock_fab.h"
#include "dock_pivot.h"
#include "docking_constraints.h"
#include "docking_ns.h"
#include "files_paths.h"
#include "ligand.h"
#include "make_pdb.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "param.h"
#include "param_pack.h"
#include "runlevel.h"
#include "score.h"
#include "score_ns.h"

// Utility Headers
#include <utility/io/izstream.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/formatted.i.hh>
#include <ObjexxFCL/string.functions.hh>

// C++ Headers
#include <cstdlib>
#include <iomanip>
#include <fstream>
#include <iostream>


//Namespaces
namespace bkrep_filter_private {
	float bkrep_filter_value = { 99999999. };
}
namespace score_filter_private {
	float score_filter_value = { 99999999. };
	float docking_interf_energy_filter_value = { 5.0 };
	float chainbreak_filter_value = { 1.0 };
}


//jg score functions for docking, complimenting scorefxns.cc
//jg June 2001

// see comments in scorefxns.cc


/////////////////////////////// SCORE4D ////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
/// @begin score4d
///
/// @brief  score4d is the docking equivalent of score4
///       (centroid-based score, low-res, for rapid searching)
/// @detailed
///
/// @return  float, the score
///
/// @global_read
///
/// @global_write score weights
///
/// @remarks actually it calls function score4d1 (or 4dp for pivot sub-mode)
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score4d()
{
	if ( get_dock_pivot_flag() ) {
		return score4dp(); // dock_pivot case
	} else {
		return score4d1(); // default
	}
}

//     docking score 4d1
////////////////////////////////////////////////////////////////////////////////
/// @begin score4d1
///
/// @brief docking equivalent of score 4
///       centroid-based, low-res, for fast searching
/// @detailed
/// docking_vdw_score, docking_contact_score, docking_env_score,
/// docking_fab_score, docking_pair_score, docking_sc_score (site-constraints)
///
/// @return  float, function value
///
/// @global_read
///
/// @global_write score weights
///
/// @remarks
///
/// @references
///
/// @authors  Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score4d1()
{
	using namespace scorefxns;

	score_reset_weights();

//	evaluate_all_terms = true;

	dock_scorefxn = true;

	docking_vdw_weight = 1.0;
	docking_contact_weight = 2.0; //****
	docking_contact_cap = -10.0;
//  Jeff found this bug for docking_contact_cap 12/11/2002
//      if ( fab1 || fab2 ) docking_contact_cap = 0.0;
	docking_env_weight = 1.0; // 0.33
	docking_pair_weight = 1.0;
	docking_sc_weight = 1.0;
	docking_fab_weight = 1.0;
	docking_fab_cap = -10.0;

	// Hmm. This is inelegant. Turn on sspair terms by default?
	// How about in score4dp?
	static bool const docking_centroid_sspair = truefalseoption( "docking_centroid_sspair");
	if (docking_centroid_sspair) {
		static float const docking_centroid_sspair_weight = realafteroption( "docking_centroid_sspair_weight", 1.0 );
		ss_weight = docking_centroid_sspair_weight;
		rsigma_weight = docking_centroid_sspair_weight;
	}

	dummy_model_weight = 1.0;

	return scorefxn();
}

//     docking score 4dp
////////////////////////////////////////////////////////////////////////////////
/// @begin score4dp
///
/// @brief dock-pivot version of score4d1()
///       centroid-based, low-res, for fast searching
/// @detailed
/// ramachandran_score, vdw_score, hb_score, env_score, pair_score,
/// docking_contact_score, docking_sc_score, docking_fab_score
///
/// @return  real, function value
///
/// @global_read
///
/// @global_write score weights
///
/// @remarks
///
/// @references
///
/// @authors  Mike Daily 4/15/04 (Authored by Lian Guo 1/03)
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score4dp()
{
	using namespace scorefxns;
	using namespace param_pack;

	score_reset_weights();

	dock_scorefxn = true;

	ramachandran_weight =    1.0; // 1.0
	vdw_weight =             1.0; // 3.0   decrease for low_optimizing

//	hb_weight =              2.0; // 2.0
	pack_wts.set_Whb_lrbb(1.0);
	pack_wts.set_Whb_srbb(1.0);

	env_weight =             4.0; //       increase for low_optimizing

	pair_weight =            1.0; // 2.0
	docking_contact_weight = 8.0; // 2.0   increase for low_optimizing

	docking_sc_weight =      1.0;
	docking_contact_cap =  -90.0; // -40.0
	docking_fab_weight =     1.0;
	docking_fab_cap =      -10.0;

	dummy_model_weight = 1.0;

	return scorefxn();
}


/////////////////////////////// SCORE10D ///////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
/// @begin score10d
///
/// @brief  score10d is the docking equivalent of score10
///
/// @detailed  fullatom mode score, for decoy discrimination
///       fa_atr, fa_rep, fa_sol, fa_dun, fa_pair, hbsc, hbbb,
///       gsolt, cheap_electrostatics
///
/// @return  float, function value
///
/// @global_read  fullatom scores
///
/// @global_write weights in scorefxn.h
///
/// @remarks assign weights here, call scorefxn() to get final score
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score10d()
{
	docking_fullatom_setweights(1.0,1.0);
	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score10d_min
///
/// @brief     This score for fullatom minimization, compare to score10d
///           greater weight on bk_rep, no surface area
/// @detailed
///
/// @return     float, function value
///
/// @global_read  fullatom scores
///
/// @global_write weights in scorefxn.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score10d_min()
{
	docking_fullatom_setweights(4.22,0.0);
	return scorefxn();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_fullatom_setweights
///
/// @brief assign the fullatom score weights for docking scorefxns
///
/// @detailed
///
/// @param[in]   rep_factor - in - factor for fa_rep weights
/// @param[in]   gsolt_factor - in- factor for gsolt weights
///
/// @global_read
///
/// @global_write score weights in score_ns.h
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_fullatom_setweights(
	float rep_factor,
	float gsolt_factor
)
{
	using namespace scorefxns;
	using namespace param_pack;

	static bool init( false );
	static bool use_score12( false );
	if ( !init ) {
		use_score12 = truefalseoption("use_score12");
		init = true;
	}

	score_reset_weights();
	fa_scorefxn = true;
	dock_scorefxn = true;
	splicemsd_weight = loop_weight; //glb for loop min

//mj for ligands, momentaraly trhe true weights are in pack_db.f. so this is
//mj basically score12...
	if ( get_ligand_flag() ) {
		fa_atr_weight      = 1.0;
		fa_rep_weight      = ( 1.0 / 4.22 ) * rep_factor;
		fa_solv_weight     = 1.0;
		param_pack::pack_wts.set_Whb_srlrsc(1.0, 1.0, 1.0);
		fa_dun_weight      = 1.0;
		fa_pair_weight     = 1.0;
		ex_gsolt_weight    = 0.0 * gsolt_factor;
		docking_warshel_elec_weight = 0.0;
		return;
	}

	if ( use_score12 ) {
		ramachandran_weight = 0.2;
		param_pack::pack_wts.set_Whb_srlrsc(0.5, 1.0, 1.0);
		fa_atr_weight = 1.0;
		fa_rep_weight = 1.0 * rep_reduce;
		fa_dun_weight = 1.0;
		fa_pair_weight = 1.0;
		fa_solv_weight = 1.0;
		fa_gb_elec_weight = 1.0;
		fa_ref_weight = 1.0;
		fa_prob1b_weight = 0.5;
		fa_h2o_weight = 1.0;
		docking_warshel_elec_weight = 0.06;
		fa_elec_weight = 1.0; // SJF
		//fa_plane_weight = get_fa_plane_weight(); // default is zero
		return;
	}
	// 2004.03-2004.05 chu and ora refit
  // chu: these weights are reweighted based on chen_04-03-08 and
	// are not final yet
	fa_atr_weight      = .423;
	fa_rep_weight      = .1 * rep_factor;
	fa_solv_weight     = .372;
  param_pack::pack_wts.set_Whb_srlrsc(0.245, 0.245, 0.245);
	fa_dun_weight      = .064;
	fa_pair_weight     = .000;
	ex_gsolt_weight    = .000 * gsolt_factor;
	docking_warshel_elec_weight = .026;
  fa_elec_weight = 1.0; // SJF
//  fa_elec_weight = docking_warshel_elec_weight; // SJF

	if ( docking::flexbb_docking_flag )
		ramachandran_weight = 0.1;

//lg for dock-pivot mode discrimination, a rama-type score needs to be added

}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_filter
///
/// @brief check if the score for a decoy is lower than certain cutoff
///
/// @detailed
///
/// @return  bool, true if it can pass the filter
///
/// @global_read score_filter_value in namespace
///
/// @global_write
///
/// @remarks score_filter_value is updated first before checking
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
score_filter()
{
//     jjg 8/31/1

	using namespace score_filter_private;

	score_filter_update();

	return (mc_global_track::mc_score::score < score_filter_value);
}
//////////////////////////////////////////////////////////////////////////////////
bool
docking_interf_energy_filter()
{
//     jjg 8/31/1

	using namespace docking;
	using namespace score_filter_private;

	return ( docking_interf_energy <= docking_interf_energy_filter_value);
}
////////////////////////////////////////////////////////////////////////////////
bool
chainbreak_score_filter()
{
//     jjg 8/31/1

	using namespace scores;
	using namespace score_filter_private;

	return ( chainbreak_score <= chainbreak_filter_value);
}
////////////////////////////////////////////////////////////////////////////////
/// @begin bkrep_filter
///
/// @brief check if fa_rep score of a decoy is lower than a certain cutoff
///
/// @detailed
///
/// @return  bool, true if it passes
///
/// @global_read bkrep_filter_value in namespace
///
/// @global_write
///
/// @remarks bkrep_filter_value can be set through score_filter_update,
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
bkrep_filter()
{
//     jjg 6/02

//     score_filter_update is called in score_filter and will update
//     bkrep_filter_value too
	using namespace bkrep_filter_private;
	using namespace scores;

	return (fa_rep_score < bkrep_filter_value);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_filter_update
///
/// @brief update the score filter cutoff
///
/// @detailed
///     If adaptive filtering has been turned on
///     (smart_scorefilter_flag=true), { every time a score file
///     is written, this looks for a local file called scorefilter.out
///     containing a cutoff score (representing the top x percent) and a
///     number of decoys used in generating that cutoff. If enough decoys
///     have been counted, the cutoff is accepted, no file scoring worse
///     than that generates a pdb, and the file never gets looked at
///     again.  If not enough files have been counted, this calls a perl
///     script that scans all local scorefiles, determines a cutoff, and
///     writes scorefilter.out.
///     SEM 3/02; rev. JJG 4/02
///     added bkrep update JJG 6/02
///     bkrep filter value hard-coded at 50%, for no particular reason
///
/// @global_read
///
/// @global_write score filter cutoff and bkrep filter cutoff
///
/// @remarks command-line flag must be turned on to use this smart filter
///       scorefilter_fraction can also be assigned in command-line
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
score_filter_update()
{
	using namespace docking;
	using namespace files_paths;

	static int scores_counted = { 0 };
	static int scores_needed = { 1000 };
	float score_cutoff;
	float bkrep_cutoff;

	std::string filter_fname;

	if ( !smart_scorefilter_flag ) return;
	if ( scores_counted > scores_needed ) return;

	filter_fname = score_path + "filter.out";
	scores_needed = static_cast< int >(50.0/scorefilter_fraction);
	 // / bkrepfilter_fraction

	// update percentfile
	std::system( ( data_path + "smart_scorefilter.pl "
	 + fixed_string_of( scorefilter_fraction, 11, 5 ) + " 1 " + score_path
	 + " > " + filter_fname ).c_str() );

	// read file
	utility::io::izstream cnstr_zx( filter_fname );

	if ( cnstr_zx ) {
		cnstr_zx >> skip( 24 ) >> bite( 10, score_cutoff ) >> skip;
		cnstr_zx >> skip( 24 ) >> bite( 10, bkrep_cutoff ) >> skip;
		cnstr_zx >> skip( 24 ) >> bite( 10, scores_counted ) >> skip;
		set_score_filter(score_cutoff);
		set_bkrep_filter(bkrep_cutoff);
		cnstr_zx.close();
		cnstr_zx.clear();
	} else {
		std::cout << "warning: scorefilter/percentfile read failed" << std::endl;
		cnstr_zx.close();
		cnstr_zx.clear();
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_score_filter
///
/// @brief set docking score fitler cutoff
///
/// @detailed
///
/// @param[in]   new_value - in - value to be set
///
/// @global_read
///
/// @global_write score_filter_value in namespace
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_score_filter( float new_value )
{
//     jjg 8/31/1

	using namespace runlevel_ns;
	using namespace score_filter_private;

	score_filter_value = new_value;
	if ( runlevel > standard ) std::cout << "set score filter " << new_value << std::endl;
}

/////////////////////////////////////////////////////////////////////////////////
void
set_docking_interf_energy_filter( float new_value )
{

	using namespace runlevel_ns;
	using namespace score_filter_private;

	docking_interf_energy_filter_value = new_value;
	if ( runlevel > standard ) std::cout << "set docking interf energy filter "
																			 << new_value << std::endl;
}
/////////////////////////////////////////////////////////////////////////////////
void
set_chainbreak_score_filter( float new_value )
{

	using namespace runlevel_ns;
	using namespace score_filter_private;

	chainbreak_filter_value = new_value;
	if ( runlevel > standard ) std::cout << "set chainbreak score filter "
																			 << new_value << std::endl;
}
////////////////////////////////////////////////////////////////////////////////
/// @begin set_bkrep_filter
///
/// @brief set bkrep_filter cutoff
///
/// @detailed
///
/// @param[in]   new_value - in - value to be set
///
/// @global_read
///
/// @global_write bkrep_filter_value in namespace
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_bkrep_filter( float new_value )
{
//     jjg 6/02

	using namespace bkrep_filter_private;
	using namespace runlevel_ns;

	bkrep_filter_value = new_value;
	if ( runlevel > standard ) std::cout << "set bkrep filter " << new_value << std::endl;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin score_no_sasa
///
/// @brief fullatom scorefxn without evaluating sasa term
///
/// @detailed
///jjg     use a given scorefunction, but without a sasa evaluation
///jjg     used for minimization, when we don't want to call sasa
///jjg     because it takes too long
///
/// @param[in]   scoring_fn - in - scorefxn in which sasa terms is turned off
///
/// @return  float, function value
///
/// @global_read score weights for scoring_fn
///
/// @global_write weight for gsolt is set to 0.0
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
score_no_sasa( Scoring_Function scoring_fn
 /* the scoring function with weights */ )
{
	using namespace scorefxns;

	float score_no_sasa; // Return value

	no_evaluation = true;
	score_no_sasa = scoring_fn(); // get the weights but don't evaluate
	ex_gsolt_weight = 0.0; // no sasa
	score_no_sasa = scorefxn(); // now, we evaluate

	return score_no_sasa;
}
