// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 32333 $
//  $Date: 2009-08-25 17:05:32 -0700 (Tue, 25 Aug 2009) $
//  $Author: aroop $


// Rosetta Headers
#include "antibody_modeling.h"
#include "docking_scoring.h"
#include "aaproperties_pack.h"
#include "cenlist.h"
#include "design.h"
#include "docking.h"
#include "docking_constraints.h"
#include "docking_db.h"
#include "dock_loop_ensemble_ns.h"
#include "dock_pivot.h"
#include "docking_ns.h"
#include "docking_score.h"
#include "etable.h"
#include "files_paths.h"
#include "fullatom.h"
#include "initialize.h"
#include "interface.h"
#include "ligand.h"
#include "misc.h"
#include "monte_carlo.h" // yab: misc removal
#include "pack_fwd.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "runlevel.h"
#include "score_ns.h"
#include "vdw.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
//#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/xyzVector.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <iostream>


//     Scoring functions for Rosetta docking
//
//     J. Gray 5/15/01
//

//     docking_evaluate_all_scores
//     docking_detect_interface_calc_pair
//     docking_calc_env_contact_fab_scores
//     docking_count_int_res
//     docking_vdw_compute_insert
//     docking_apply_filters

//     docking_make_interf_list
//     fullatom_cheap_electrostat

//------------------------------------------------------------------------------


////////////////////////////////////////////////////////////////////////////////
/// @begin docking_evaluate_all_scores
///
/// @brief evaluate and sum up all docking specific scores
///
/// @detailed all of them are centroid-based, move warshel to a new function
///      called from scorefxn()
///
/// @param[out]   docking_score - out - the sum of these weighted scores
///
/// @global_read
///
/// @global_write
///
/// @remarks all the docking-specific scores will be updated, an side effect
///       is that interface residues list will be updated.
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified Chu Wang 12/20/06
/////////////////////////////////////////////////////////////////////////////////
void
docking_evaluate_all_scores( float & docking_score )
{
	using namespace scorefxns;
	using namespace scores;
	using namespace runlevel_ns;

	if ( (!files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler) && (!docking::docking_ensemble_flag )) {
		if ( dock_scorefxn && runlevel >= yap ) {
			std::cout << "WARNING:: docking scorefxn terms requested" << std::endl;
			std::cout << "but multi_chain is false" << std::endl; 
			std::cout << "docking terms will not be evaluated\n" << std::endl;
		}
		docking_env_score = 0.0;
		docking_pair_score = 0.0;
		docking_contact_score = 0.0;
		docking_vdw_score = 0.0;
		docking_site_constraint_score = 0.0;
		//docking_warshel_elec_score = 0.0;
		docking_score = 0.0;
		return;
	}

//mj jump out if ligand flag
	if ( get_ligand_flag() ) {
//         std::cout << "ERROR: Called centroid function with ligand flag:" <<
//          " docking_evaluate_all_scores in docking_scoring.cc" << std::endl;
		docking_env_score = 0.0;
		docking_pair_score = 0.0;
		docking_contact_score = 0.0;
		ligand_centroid_vdw(docking_vdw_score,(*ligand::ligand_one));
		docking_site_constraint_score = 0.0;
		docking_score = docking_vdw_weight * docking_vdw_score;
		return;
	}

//     first, evaluate bumps between partners
//     and in the process, compute cendist arrays needed here
//     also calculates docking_vdw_score
	docking_vdw_compute_insert();
//     sum up the cen10 array
	compute_CB_environment();

//     detect the interface residues
	docking_detect_interf_res();
//     and  get the pair score
	docking_calc_interf_pair();
//     calculate the env and contact scores (also calls fab score calc)
	docking_calc_env_contact_scores();

//     this didn't tell us much:
//     docking_calc_antigen_compn();

//     calculate site-constraint score
	docking_evaluate_site_score();

//     cheap electrostatics -- evaluated in fullatom mode only
//$$$      docking_cheap_electrostatics();
// move warshel calculations to a new function, docking_evaluate_fa_scores()
// call that after rottrial in scorefxn as it is fullatom-dependent.
//  	docking_warshel_electrostatics();

	docking_score = docking_env_weight * docking_env_score +
	 docking_pair_weight * docking_pair_score +
	 docking_contact_weight * std::max(docking_contact_score,docking_contact_cap) +
	 docking_vdw_weight * docking_vdw_score +
	 docking_sc_weight * docking_site_constraint_score +
		docking_fab_weight * std::max(docking_fab_score,docking_fab_cap);

}
////////////////////////////////////////////////////////////////////////////////
/// @begin docking_evaluate_fa_scores
///
/// @brief evaluate and sum up all docking fullatom-dependent specific scores
///
/// @detailed  called from scorefxn(), currently only warshel electric.
///
/// @param[out]   docking_score - out - the sum of these weighted scores
///
/// @global_read
///
/// @global_write
///
/// @remarks separate this from docking_evaluate_all_scores as fullatom-related
///  score might be changed in scorefxn after rottrial
///
/// @references
///
/// @authors Chu Wang 12/20/06
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_evaluate_fa_scores( float & docking_score )
{
	using namespace scorefxns;
	using namespace scores;
	using namespace runlevel_ns;

	if ( !files_paths::multi_chain || design::dna_interface || files_paths::antibody_modeler ) {
		if ( dock_scorefxn && runlevel >= yap ) {
			std::cout << "WARNING:: docking scorefxn terms requested" << std::endl;
			std::cout << "but multi_chain is false" << std::endl;
			std::cout << "docking terms will not be evaluated\n" << std::endl;
		}
		docking_warshel_elec_score = 0.0;
		docking_score = 0.0;
		return;
	}

//mj jump out if ligand flag
	if ( get_ligand_flag() ) {
		return;
	}

	// electrostatic score for docking
	// Assume docking_inteface_residue_list is up-to-date!!!!

  if ( !get_simple_elec() ) { // o/w compute the electrostatics within fullatom_energy
  	docking_warshel_electrostatics();
  	docking_score = docking_warshel_elec_weight * docking_warshel_elec_score;
  }

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_detect_interf_res
///
/// @brief set up docking interface residue list
///
/// @detailed
///     This function detects and saves a record of the interface residues,
///     and simultaneoudly sums the docking pair interaction
///     score. (6/28/01 JJG rev. 8/2/1)
///
///     Interface residues defined as those with centroid distances of less
///     than 6 Angstroms across the interface
///
///     cendist array must be precalculated !!!!!
///
///     a 8A definition interface residue list is calculated for side-chain
///     repacking and cheap electrostatics.
///
/// @global_read cendist array precalculated in cenlist.h
/// @global_write interface array in interface.h
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_detect_interf_res()
{
	// save the int_pair_list matrix
	// save the int_res_list array

	using namespace cenlist_ns;
	using namespace docking;
	using namespace interface;
	using namespace misc;
	using namespace runlevel_ns;

//mj jump out if ligand flag
	if ( get_ligand_flag() ) {
		detect_ligand_interface((*ligand::ligand_one), false, total_residue, res, full_coord);
		return;
	}

	if( antibody_ns::snugdock_ns::LH_AbAg_interface ) {
		antibody_modeling_detect_snugdock_interf_res();
		return;
	}

	int_res  = false;
	int_res8 = false;

	for( int i=1; i <= max_docking_sites; ++i ) {
		int_res_list(i).clear();
		int_pair_list(i).clear();
		int_res_list8(i).clear();
		int_pair_list8(i).clear();
	}

	for ( int i = part_begin(1), iend = part_end(1), jend = part_end(2);
				i <= iend; ++i ) {
		for ( int j = part_begin(2); j <= jend; ++j ) {
			if ( cendist(i,j) < 64.0 ) { // if under 8A
//     predefine int_res_list8(2)[X]
				if ( !int_res8(j) ) {
					int_res_list8(2).push_back(j);
				}
//     predefine int_pair_list8
				int_res8(i) = true;
				int_res8(j) = true;
				int_pair_list8(1).push_back(i);
				int_pair_list8(2).push_back(j);

				if ( cendist(i,j) < 36.0 ) { // check if under 6A
//     predefine int_res_list(2)[X]
					if ( !int_res(j) ) {
						int_res_list(2).push_back(j);
					}
//     predefine int_pair_list
					int_res(i) = true;
					int_res(j) = true;
					int_pair_list(1).push_back(i);
					int_pair_list(2).push_back(j);
				}            // 6A
			}               // 8A
		}                  // partner2

//     predefine int_res_list(1)[X]
		if ( int_res(i) ) {
			int_res_list(1).push_back(i);
		}
//     predefine int_res_list8(1)[X]
		if ( int_res8(i) ) {
			int_res_list8(1).push_back(i);
		}

	}        // partner1

	assert( int_pair_list(1).size() == int_pair_list(2).size() );
	assert( int_pair_list8(1).size() == int_pair_list8(2).size() );

	if ( runlevel >= yap ) {
		std::cout << "interface sizes under 6A: " << int_res_list(1).size()
							<< ' ' << int_res_list(2).size() << std::endl;
		std::cout << "interface pairs under 6A: " << int_pair_list(1).size()
							<< std::endl;
		std::cout << "interface sizes under 8A: " << int_res_list8(1).size()
							<< ' ' << int_res_list8(2).size() << std::endl;
		std::cout << "interface pairs under 8A: " << int_pair_list8(1).size()
							<< std::endl;
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_calc_interf_pair
///
/// @brief calculate docking pair score
///
/// @detailed
///     Pair scores from Glaser et al Proteins 2001, simple stats from
///     a large, non-redundant database of interfaces. Go through each
///     interface pair and derive a score based on stats from that database
///     and sum them up.
///
/// @global_read int_pair_list in interface.h, precalculated
///             data block from docking_db.h
///
/// @global_write docking_pair_score
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_calc_interf_pair()
{
//     each pair of interface residues is stored in matrix named int_pair_list,
//     which was calculated in function docking_detect_interf_res

	using namespace docking_db;
	using namespace interface;
	using namespace misc;
	using namespace scores;

//mj jump out if ligand flag
	if ( get_ligand_flag() ) {
		std::cout << "ERROR: Called centroid function with ligand flag:" <<
		 " docking_calc_interf_pair in docking_scoring.cc" << std::endl;
		return;
	}

	docking_pair_score = 0.0;

	for ( int i = 1; i <= int(int_pair_list(1).size()); ++i ) {
		docking_pair_score +=
			log_docking_pair(res(int_pair_list(1)[i]),res(int_pair_list(2)[i]));
	}

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_calc_env_contact_scores
///
/// @brief calculate docking_contact score and docking_env_score
///
/// @detailed
///     This function scores the docking environment score and contact
///     score (6/28/01 JJG rev 8/2/1)
///
///     Env scores from Tanja's database of structures (Nussinov's JMB 1996
///     set, minus the homodimers and 'bad' structures, leaving approximately
///     75 complexes) ... or maybe from Glaser/Ben Tal's set...see docking_db.cc
///
///     contact score is simply dependent on number of interface residues. it
///     is a fast way to see if two partners are losing contact.
///
///     Interface residues MUST be precalculated in int_res in interface.h
///
///     REQUIRED that cen10 array has been set
///
///     docking_fab_score is also called from here
///
/// @global_read interface array, cen10 array, data block in docking_db.cc
///
/// @global_write docking_env_score, docking_contact_score
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_calc_env_contact_scores()
{
	using namespace cenlist_ns;
	using namespace docking;
	using namespace docking_db;
	using namespace interface;
	using namespace misc;
	using namespace scores;

//     local variables
	int env;
	int total_interface_residues;

//mj jump out if ligand flag
	if ( get_ligand_flag() ) {
		std::cout << "ERROR: Called centroid function with ligand flag:" <<
		 " docking_calc_env_contact_scores in docking_scoring.cc" << std::endl;
		return;
	}

	docking_env_score = 0.0;

	total_interface_residues = int_res_list(1).size() + int_res_list(2).size();

	for ( int i = 1; i <= total_residue; ++i ) {
		if ( int_res(i) ) {
			if ( cen10(i) > 16 ) {
				env = 1; // interface residue, buried in complex
			} else {
				env = 2; // interface residue, exposed in complex
			}
		} else {
			if ( cen10(i) > 16 ) {
				env = 3; // not interface residue, buried in complex
			} else {
				env = 4; // not interface residue, exposed in complex
			}
		}
		if( ! param_aa::is_ligand(res(i)) ){
			docking_env_score += log_docking_env(env,res(i));
		}
//     std::cout << "dENV: " <<
//      residue3(i) << I( 3, i ) << ' ' << I( 1, env ) << I( 4, cen10(i) ) <<
//      F( 10, 4, log_docking_env(env,res(i)) ) <<
//		  F( 10, 4, docking_env_score ) << std::endl;
	}

	docking_contact_score = ( 20 - total_interface_residues ) * 0.5;

//     extra penalties for losing contact
//     this seems to be necessary, as sometimes we 'fly away'
	if ( total_interface_residues == 0 ) docking_contact_score += 2.0;
	if ( total_interface_residues == 1 ) docking_contact_score += 1.0;
	if ( total_interface_residues == 2 ) docking_contact_score += 0.5;

	docking_calc_fab_score(); // modifies contact score

}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_vdw_compute_insert
///
/// @brief calculating docking_vdw_score
///
/// @detailed
///     A modification of vdw_compute_insert
///
///     Here, we only compute interactions *between* docking partners,
///     not within them.
///
///     Assuming a rigid-body move has taken place, all intermolecular
///     interactions will need to be recomputed, so we don't need to worry
///     about saving any numbers from the last time.
///
///     JJG 5/21/01
///
///     7/11/1: updated vdw formula to follow rosetta's change.  the new
///     form is smooth at onset and increases slope as atoms overlap more
///
/// @global_read centroid  array
///
/// @global_write docking_vdw_score
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_vdw_compute_insert()
{
	using namespace cenlist_ns;
	using namespace docking;
	using namespace misc;
	using namespace scores;
	using numeric::xyzVector_float;

//mj jump out if ligand flag
	if ( get_ligand_flag() ) {
		std::cout << "ERROR: Called centroid function with ligand flag:" <<
		 " docking_vdw_compute_insert in docking_scoring.cc" << std::endl;
		return;
	}

	int itemp, jtemp; // starting points for res i and j in position
	int type0; // centroid i
	int type1; // atom j
	int type2; // atom i
	int type3; // centroid j
	float cendist_ij, dist2, alpha_alpha;
	float docking_vdw_score_sum, docking_vdw_score_pass;

//     1) if the centroids >  12 A apart, no atoms to residues can bump
//     2) if CA(i) CA(j) distance > 7A, no backbone atoms of pair bump

//     compute i-centroid to j-centroid distance.  store it for use elsewhere
//      jeffs_special_calculate_cendist(centroid,total_residue,
//       false,part2_begin,part1_end,cendist)
	update_cendist(total_residue, centroid);

//     ************************************************************
//     check vdw radius between all pairs across the interface
//     ************************************************************

//     begin loops over all changed atom pairs
	docking_vdw_score_sum = 0.0; // were going to accumlate into this

	for ( int i = part_begin(1), iend = part_end(1), jbegin = part_begin(2), jend = part_end(2);
	 i <= iend; ++i ) { // outer loop over residues

		docking_vdw_score_pass = 0.0; // accumulate total for each i individually
		itemp = ( 5 * i ) - 4; // pentamer starting point
		type0 = atom_type_cen(i);

		xyzVector_float const cen_i( &centroid(1,i) ); // centroid(1-3,i)

//    o . . . . . . . . __
//    F o . . . . . . .  ^
//    F F o . . . . . .  |
//    F F F o . . . . .  |
//    * * * * o . . . .  J   <----  part2_begin
//    * * * * F o . . .  a
//    * * * * F F o . .  x
//    * * * * F F F o .  i
//    * * * * F F F F o __   <----  part2_end
//    | < ---- i axis ->|
//
//    F  intramolecular contacts ignored here
//    *  must be calculated

		xyzVector_float const pos_i( &position(1,itemp+1) ); // position(1-3,itemp+1)

		for ( int j = jbegin; j <= jend; ++j ) {

			docking_vdw_score = 0.0; // accumulate each j-pass individually
			type3 = atom_type_cen(j); //  use this later twice
			xyzVector_float const cen_j( &centroid(1,j) ); // centroid(1-3,j)

			cendist_ij = cendist(i,j); // look up prev calc dist
//--------
//     as long as were at it, we now do a clash check
//     first do a coarse grain reality check on centroid seperations
//     and if it passes carefully check atom-atom distances
//--------
			if ( cendist_ij <= cen_dist_cutoff2 ) { // reality check
//     std::cout << "vdw " << i << ' ' << j << SS( cendist_ij ) << std::endl;

//     centroid-centroid clash check
				dist2 = vdw::atom_vdw(type0,type3) - cendist_ij;

				if ( dist2 > 0.0 ) docking_vdw_score += ( dist2 * dist2 ) / vdw::atom_vdw(type0,type3); // clash score

				jtemp = ( 5 * j ) - 4; // pentamer starting point

// As a prefilter we first compute the calpha-calpha distance.
// . If this is greater than 49 (7 squared, ca_dist_cutoff2) none of the backbone/cbeta can clash
//	  and we skip the distance backbone checks.
// . Additionally if this is greater than 60 then we can skip the centriod backbone checks.
				alpha_alpha // plus one means calpha
				 = distance_squared( pos_i, xyzVector_float( &position(1,jtemp+1) ) );

//     minimum  centroid-pentamer radius

//---------
//     check i-centroid to j-atom  clash
//---------
				for ( int k = jtemp, l = position.index(1,k); k <= jtemp+4; ++k, l+=3 ) {
					type1 = atom_type(k);
					if ( type1 != 0 ) { // gly c-beta
						dist2 = vdw::atom_vdw(type0,type1) - distance_squared(
						 cen_i, // centroid(1-3,i)
						 xyzVector_float( &position[ l ] ) ); // position(1-3,k)

						if ( dist2 > 0.0 ) docking_vdw_score += ( dist2 * dist2 ) / vdw::atom_vdw(type0,type1);
					} // glycene check
				} // k

//---------
//     check j-centroid to i-atom  clash
//---------
				for ( int k = itemp, l = position.index(1,k); k <= itemp+4; ++k, l+=3 ) {
					type2 = atom_type(k); // vdw calc
					if ( type2 != 0 ) { // skip gly c-beta
						dist2 = vdw::atom_vdw(type3,type2) - distance_squared(
						 cen_j, // centroid(1-3,j)
						 xyzVector_float( &position[ l ] ) ); // position(1-3,k)

 						if ( dist2 > 0.0 ) docking_vdw_score += ( dist2 * dist2 ) / vdw::atom_vdw(type3,type2);
					}
				} // k

//------
//     check i-pentamer to j-pentamer
//------
				if ( alpha_alpha < ca_dist_cutoff2 ) { // clash radius CA-CA

					for ( int k = jtemp, l = position.index(1,k); k <= jtemp+4; ++k, l+=3 ) { // all backbone atoms in residue j
						type1 = atom_type(k);
						if ( type1 != 0 ) { // glycine c-beta
							xyzVector_float const pos_k( &position[ l ] ); // position(1,k);

							for ( int ii = itemp, li = position.index(1,ii); ii <= itemp+4; ++ii, li+=3 ) { // all backbone atoms in residue i
								type2 = atom_type(ii);
								if ( type2 != 0 ) { //  r glycine  c-beta

									dist2 = vdw::atom_vdw(type1,type2) - distance_squared(
									 pos_k, // position(1-3,k)
									 xyzVector_float( &position[ li ] ) ); // position(1-3,ii);

									if ( dist2 > 0.0 ) docking_vdw_score += ( dist2 * dist2 ) / vdw::atom_vdw(type1,type2);
								} // end type2 glycine check

							} // end l

						} // end type1 glycine check
					} // end k
				} // end alpha-alpha <49 check

			} // end cen_dist_cutoff2

			// save column sum for pass #i
			docking_vdw_score_pass += docking_vdw_score;

		} // end j=part2 residues

//     save running sum of all passes
		docking_vdw_score_sum += docking_vdw_score_pass; // accumulate

	} // end i=part1 residues

	docking_vdw_score = docking_vdw_score_sum;
	docking_vdw_score *= 100.0f; // historical
	docking_vdw_score *= 0.008f; // moved from evaluate_envpair

}




////////////////////////////////////////////////////////////////////////////////
/// @begin docking_count_interf_residues
///
/// @brief get the total number of interface residues
///
/// @detailed
///     sum up int_res_list(1).size() and int_res_list(2).size()
///     this is 6A definition
///     interface array is precalculated
///
/// @return   total number of interface residues
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
int
docking_count_interf_residues()
{
	//     requires cendist to be set
	using namespace interface;

	docking_detect_interf_res();
	return int_res_list(1).size() + int_res_list(2).size();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin docking_apply_filters
///
/// @brief check docking scores with multiple constraints and filters
///
/// @detailed
///    filters make sure that a decoy has a correct fab contact,
///    or statisfy distance contraints or site constraints,
///    or is not too bumpy or full score is not low enough,
///    or two partners are not too separate .
///    if a decoy fails any of these filters, its structure will not be
///    output unless it is very close to native.
///
/// @param[out]   accepted - out - true if passing all filters
/// @param[out]   tag - out - the status of failure
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Chu Wang 08/19/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
docking_apply_filters(
	bool & accepted,
	std::string & tag
)
{
	using namespace docking;
	using namespace files_paths;
	using namespace scores;

	if ( !use_filter(dock_type) ) return;

	float docking_vdw_max;
	if ( get_dock_pivot_flag() ) {
		docking_vdw_max = 200.0;
	}

	// mj for ligand_flag ignore clashes below 200
	else if ( get_ligand_flag()) {
		docking_vdw_max = 200.0;
	}
	// mj end

	else {
		docking_vdw_max = 10.0;
	}

	if ( !docking_fab_filter() ) {
		accepted = false;
		std::cout << "structure makes incorrect contacts" << std::endl;
		tag = "fab_failure";
	} else if ( !docking_dist_constr_filter() ) {
		accepted = false;
		std::cout << "structure does not meet distance constraints" << std::endl;
		tag = "dist_failure";
	} else if ( !docking_site_constr_filter() ) {
		accepted = false;
		std::cout << "structure does not meet site constraints" << std::endl;
		tag = "site_failure";
	} else if ( docking_contact_score >= 10.0 ) { // docking_contact_filter
		accepted = false;
		std::cout << "structure makes no contacts" << std::endl;
		tag = "contact_failure";
	} else if ( docking_vdw_score >= docking_vdw_max ) { // docking_bumps_filter
		accepted = false;
		std::cout << "structure has too many bumps" << std::endl;
		tag = "vdw_failure";
	} else if ( !score_filter() ) {
		accepted = false;
		std::cout << "structure does not score low enough" << std::endl;
		tag = "score_failure";
	} else if ( !bkrep_filter() ) {
		accepted = false;
		std::cout << "structure is bumpy" << std::endl;
		tag = "bkrep_failure";
	} else if ( !docking_interf_energy_filter() ) {
		accepted = false;
		std::cout << "structure's interface energy is not low enough" << std::endl;
		tag = "I_sc_failure";
	} else if ( !chainbreak_score_filter() ) {
		accepted = false;
		std::cout << "structure does not have chain breaks closed well "
							<< std::endl;
		tag = "chbrk_failure";
	}

//     keep structures close to native
	if ( !accepted && get_native_exists() && mc_global_track::diagnose::rms_err < 10.0 &&
	 !docking_fake_native ) {
		accepted = true;
		std::cout << "structure is close to native" << std::endl;
		tag = "rms_save_" + tag;
	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin
///
/// @brief
///  this function calls washel_elect to calculate the coulombic interaction.
///  only interactions across the interface count.
///  all atoms including hydrogens considered.
///
/// @detailed
///     input: full atom positions, and int_res8 bool array under 8A
///            definition
///     output: docking_warshel_elec_score
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///chu   currently e(r) = r, so called "inverse r" model
///
/// @references
///
/// @authors
///     chu 2003-07-21
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
docking_warshel_electrostatics()
{
	using namespace aaproperties_pack;
	using namespace cenlist_ns;
	using namespace docking;
	using namespace interface;
	using namespace misc;
	using namespace scores;

//      std::cout << "** docking_warshel_electrostatics **" << std::endl;

//     initialize for accumulation or for centroid mode return
	docking_warshel_elec_score = 0.0;

	if ( !get_fullatom_flag() ) return;
	if ( get_ligand_flag() ) return;

//     local
	static float const min_d2 = 1.5 * 1.5;
	int const full_coord_size1 = full_coord.size1();
//ora generic charges  int iatype,jatype;
	int ires,jres,iaa,jaa,iaav,jaav;
	float iacharge,jacharge;
	float tmp,d2;

//     go through the list of residues at the interface...
	for ( int i = 1, ie = int_res_list8(1).size(), je = int_res_list8(2).size();
				i <= ie; ++i ) {
		ires = int_res_list8(1)[i];
		iaa = res(ires);
		iaav = res_variant(ires);
		int const lis = atomic_charge.index(1,iaa,iaav);
		int const lfis = full_coord.index(1,1,ires);
		int const iatome = natoms(iaa,iaav);
		for ( int j = 1; j <= je; ++j ) {
			jres = int_res_list8(2)[j];
			if ( cendist(ires,jres) < cen_dist_cutoff2 ) {
				jaa = res(jres);
				jaav = res_variant(jres);
				int const ljs = atomic_charge.index(1,jaa,jaav);
				int const lfjs = full_coord.index(1,1,jres);
				int const jatome = natoms(jaa,jaav);
				for ( int iatom = 1, li = lis, lfi = lfis; iatom <= iatome;
				 ++iatom, ++li, lfi+=full_coord_size1 ) {
//ora: generic partial charges              iatype = fullatom_type(iatom,iaa,iaav);
					iacharge = atomic_charge[ li ]; // [li]==(iatom,iaa,iaav)
					float & full_coord_i( full_coord[ lfi ] ); // [lfi]==(1,iatom,ires)

					for ( int jatom = 1, lj = ljs, lfj = lfjs; jatom <= jatome;
					 ++jatom, ++lj, lfj+=full_coord_size1 ) {
//ora: generic partial charges              jatype = fullatom_type(jatom,jaa,jaav);
						jacharge = atomic_charge[ lj ]; // [lj]==(jatom,jaa,jaav)

						distance2_bk(full_coord_i,full_coord[ lfj ],d2); // [lfj]==(1,jatom,jres)

						if ( d2 < min_d2 ) d2 = min_d2; // flatten coulumbic energy

//ora: generic partial charges      inverse_r_elec(charge(iatype),charge(jatype),d2,tmp);
						inverse_r_elec(iacharge,jacharge,d2,tmp);

						docking_warshel_elec_score += tmp;
					} // jatom
				} // iatom
			} // cendis_cutoff
		} // int8_2
	} // int8_1

}

//==============================================================================
// Computes WARSHEL electrostatics with distant dependent dielectric constant
//     E = 322.0637*q1*q2/r/e(r)
//     if ( r < 3 ) e(r) = 16.55
//     else         e(r) = 1 + 60*(1-exp(-0.1*r))
//     Warshel, A. Russell, S. T., Q. Rev. Biophys., 1984, 17, 283-422
//==============================================================================

//$$$    warshel_elec( float q1, float q2, float r2, float & E )
//$$$    {
//$$$
//$$$cmj local variable
//$$$      float die, r;
//$$$
//$$$cmj function body
//$$$
//$$$      r = std::sqrt(r2);
//$$$
//$$$      if ( r < 3.0 ) {
//$$$         die = 0.051387;
//$$$      } else {
//$$$         die = 0.189404 - 0.186298 * std::exp( -0.1 * r );
//$$$      }
//$$$      E = q1 * q2 / r / die;
//$$$
//$$$    }
//------------------------------------------------------------------------------
//void
//inverse_r_elec(
//	float q1,
//	float q2,
//	float r2,
//	float & E
//)
//{
////chu   another dielectric model where dielectric constant is 1/r
//
////	float r = std::sqrt(r2);
////
////	float die = r;
////
////	E = 322.0637 * q1 * q2 / r /  die;
//
//	E = 322.0637 * q1 * q2 / r2; // Faster
//}
