// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 12404 $
//  $Date: 2007-02-01 15:47:11 -0800 (Thu, 01 Feb 2007) $
//  $Author: bblum $


// Rosetta Headers
#include "aaproperties_pack.h"
#include "angles.h"
#include "barcode_stats.h"
#include "dssp.h"
#include "fullatom.h"
#include "hbonds_ns.h"
#include "misc.h"
#include "param.h"
#include "param_aa.h"
#include "prof.h"
#include "termini.h"

// ObjexxFCL Headers
#include "ObjexxFCL/FArray1D.hh"
#include "ObjexxFCL/FArray2D.hh"
#include "ObjexxFCL/FArray3Dp.hh"

// C++ Headers

#include <list>
#include <vector>
#include <iostream>

namespace dssp_ns {
  FArray2D_float hbond_bb_pair_score;



	///////////////////////////////////////////////////////////////
	/// @begin StrandPairingSet::StrandPairingSet
	///
	/// @brief Constructor for set of StrandPairing objects
	///
	/// @detailed
	/// The incoming hbonds matrix is indexed by (acceptor residue,
	/// donor residue).  Residues with energy less than threshold are
	/// considered paired, unless they are disallowed by the array
	/// called allowed (this was included to prevent helical
	/// residues from being considered paired, a problem which
	/// occasionally arose).
	///
	/// @param
	///
	/// @global_read
	///
	/// @global_write
	///
	/// @remarks
	///
	/// @references
	///
	/// @authors bblum
	///
	/// @last_modified
	///////////////////////////////////////////////////////////////

	StrandPairingSet::StrandPairingSet( FArray2DB_float const &hbonds,
			float threshold, int nres ) {
		for ( int i = 2; i <= nres - 1; i++ ) {
			for ( int j = i + 1; j <= nres - 1; j++ ) {
				if( // antiparallel bridge
						(hbonds(i,j) < threshold
						 && hbonds(j,i) < threshold )
						|| (hbonds(i-1,j+1) < threshold
							&& hbonds(j-1,i+1) < threshold ) ) {
					add_pairing(i,j,true, get_pleating(i,j));
				} else if(	// parallel bridge
						(hbonds(i-1,j) < threshold
						 && hbonds(j,i+1) < threshold )
						|| (hbonds(j-1,i) < threshold
							&& hbonds(i,j+1) < threshold ) ) {
					add_pairing(i,j,false, get_pleating(i,j));
				}
			}
		}
	}

	std::ostream & operator<<(std::ostream & out, const StrandPairingSet &sp) {
		out << sp.pairings.size() << " pairings: " << std::endl;
		for(std::list<StrandPairing>::const_iterator it = sp.pairings.begin();
				it != sp.pairings.end();
				it++)
			out << *it << std::endl;
		return out;
	}


	///////////////////////////////////////////////////////////////
	/// @begin StrandPairingSet::add_pairing
	///
	/// @brief Add a new pair of bonded residues to the set
	///
	/// @detailed
	/// Look for a strand pairing to extend with the given pair of
	/// residues; if none exists, create a new one.
	///
	/// @param
	///
	/// @global_read
	///
	/// @global_write
	///
	/// @remarks
	///
	/// @references
	///
	/// @authors bblum
	///
	/// @last_modified
	///////////////////////////////////////////////////////////////
	void StrandPairingSet::add_pairing(int res1, int res2, bool antiparallel, int pleating) {
		bool addnew = true;
		for(std::list<StrandPairing>::iterator it = pairings.begin();
				it != pairings.end();
				it++) {
			if(it->extend(res1,res2,antiparallel,pleating)) {
				addnew = false;
				break;
			}
		}
		bool added = false;
		if(addnew) {
			StrandPairing add(res1,res2, antiparallel, pleating);
			for(std::list<StrandPairing>::iterator it = pairings.begin();
					it != pairings.end();
					it++)
				if(add < *it) {
					added = true;
					pairings.insert(it, add);
					break;
				}
			if(!added)
				pairings.push_back(add);
		}
	}

	bool StrandPairingSet::merge(const StrandPairingSet &other, bool domerge) {
		if(pairings.size() > other.pairings.size()) // OK to merge multiple same-reg strands into one
			return false;

		std::list<StrandPairing>::iterator it = pairings.begin();
		std::list<StrandPairing>::const_iterator oit = other.pairings.begin();
		while(it != pairings.end()) {
			int count = 0;
			while(oit != other.pairings.end() && it->merge(*oit, domerge)) {
				oit++;
				count++;
			}
			if(count == 0)
				return false;
			it++;
		}
		if(oit != other.pairings.end())
			return false;
		if(domerge)
			selfmerge();
		return true;
	}

	void StrandPairingSet::selfmerge() {
		std::list<StrandPairing> goodpairings(pairings);
		for(std::list<StrandPairing>::iterator it = pairings.begin();
					it != pairings.end(); 
					it++)
			for(std::list<StrandPairing>::iterator oit = pairings.begin();
					oit != it; 
					oit++)
				while(oit != it && it->merge(*oit, true))
					oit = pairings.erase(oit);
	}

	bool StrandPairingSet::check_pleat() const {
		for(std::list<StrandPairing>::const_iterator it = pairings.begin();
				it != pairings.end();
				it++) {
			if(!it->check_pleat())
				return false;
		}
		return true;
	}

	bool StrandPairing::check_pleat() const {
		for(int i = 1; i < (int)pleating1.size(); i++) {
			if(pleating1[i] == pleating1[i-1] && pleating1[i] != 0)
				return false;
		}
		return true;
	}

	void StrandPairing::extend_to(int res) {
		int res1, res2, pleat, diff = (antipar ? -1 : 1);
		if(res < begin1) {
			res1 = begin1 - 1;
			res2 = pairing1[0] - diff;
			pleat = 3 - pleating1[0];
			while(res1 >= res) {
				extend(res1, res2, antipar, pleat);
				res1 = res1 - 1;
				res2 = res2 - diff; 
				pleat = 3 - pleat;
			}
		} else if(res > end1) {
			res1 = end1 + 1;
			res2 = pairing1[end1 - begin1] + diff;
			pleat = 3 - pleating1[end1 - begin1];
			while(res1 <= res) {
				extend(res1, res2, antipar, pleat);
				res1 += 1;
				res2 += diff; 
				pleat = 3 - pleat;
			}
		}
	}

	bool StrandPairing::merge(const StrandPairing &other, bool domerge) {
		if(antipar != other.antipar)
			return false;

		// Make sure both strands overlap (or at least almost overlap)
		const int MARGIN(1);
    if(begin1 > other.end1 + MARGIN || begin2 > other.end2 + MARGIN 
				|| other.begin1 > end1 + MARGIN || other.begin2 > end2 + MARGIN)
			return false;

		// Make sure the merged strands won't overlap
		if(end1 >= other.begin2 || other.end1 >= begin2)
			return false;

		// Make sure starting and ending registers match up
		// (redundant with later, rigorous test, but quick and easy
		// and gets rid of most pairs of unmergeable topologies)
		if(antipar) {
			if((begin1 + end2 != other.begin1 + other.end2) ||
				(begin2 + end1 != other.begin2 + other.end1))
				return false;
		} else {
			if((begin1 - begin2 != other.begin1 - other.begin2) ||
				(end1 - end2 != other.end1 - other.end2))
				return false;
		}

		StrandPairing myex(*this);
		myex.extend_to(other.begin1);
		myex.extend_to(other.end1);

		StrandPairing otherex(other);
		otherex.extend_to(begin1);
		otherex.extend_to(end1);

		if(myex.end2 != otherex.end2 || myex.begin2 != otherex.begin2) {
			std::cout << "SURPRISE!\n";
			return false;
		}

		// Make sure pairings, bulges, and pleats match up.  
		for(int i = 0; i <= myex.end1 - myex.begin1; i++) {
			if(myex.pleating1[i] != otherex.pleating1[i] && myex.pleating1[i] > 0 && otherex.pleating1[i] > 0)
				return false;
			if(myex.pairing1[i] != otherex.pairing1[i] && myex.pairing1[i] > 0 && otherex.pairing1[i] > 0)
				return false;
		}
		for(int i = 0; i <= myex.end2 - myex.begin2; i++) {
			if(myex.pairing2[i] != otherex.pairing2[i] && myex.pairing2[i] > 0 && otherex.pairing2[i] > 0)
				return false;
		}

		if(domerge) {
//			StrandPairing before(*this);
			bool changed = other.begin1 < begin1 || other.end1 > end1;
			// Now do actual merge
			// Add in holes in myex extension that are present in otherex
			for(int res = myex.begin1; res <= myex.end1; res++) {
				int i = res - myex.begin1;
				if((res < begin1 || res > end1) && otherex.pairing1[i] == 0) {
					if(myex.pairing1[i] == 0 || otherex.pairing2[myex.pairing1[i] - otherex.begin2] > 0)
						std::cout << "SERIOUS PROBLEM.\n";
					myex.pairing2[myex.pairing1[i] - myex.begin2] = 0;
					myex.pairing1[i] = myex.pleating1[i] = 0;
				}
				// Fill in holes in myex overlap with values from otherex
				if(res >= std::max(begin1,other.begin1) && res <= std::min(end1,other.end1) && myex.pairing1[i] == 0 && otherex.pairing1[i] > 0) {
					if(myex.pairing2[otherex.pairing1[i]-myex.begin2] > 0)
						std::cout << "ANOTHER SERIOUS PROBLEM.\n";
					myex.pairing1[i] = otherex.pairing1[i];
					myex.pleating1[i] = otherex.pleating1[i];
					myex.pairing2[otherex.pairing1[i] - myex.begin2] = res;
					changed = true;
				}
			}
			*this = myex;
			/*
			if(changed) {
				std::cout << "Before: " << before << std::endl;
			  std::cout << "Other: " << other << std::endl;
				std::cout << "Merged: " << *this << std::endl;
			}
			*/
		}
		return true;
	}

	int StrandPairing::get_pleating(int res) const {
		if(res >= begin1 && res <= end1) {
			return pleating1[res - begin1];
		} else if(res >= begin2 && res <= end2) {
			if(pairing2[res - begin2] != 0)
				return pleating1[pairing2[res-begin2]];
			else
				return 0;
		} else
			return 0;
	}

	// Returns the DSSP designation of the given residue (' ' if
	// unpaired)
	char StrandPairingSet::dssp_state(int res) const {
		char state = ' ';
		for(std::list<StrandPairing>::const_iterator it = pairings.begin();
				it != pairings.end();
				it++)
			if(it->contains(res)) {
				if(it->is_ladder())
					state = 'E';
				else if(state == ' ')
					state = 'B';
			}
		return state;
	}

	char StrandPairingSet::featurizer_state(int res) const {
		char state = 'L';
		for(std::list<StrandPairing>::const_iterator it = pairings.begin();
				it != pairings.end();
				it++)
			if(it->contains(res)) {
				if(it->is_bulge(res)) {
					if(state == 'e')
						state = 'B';
					else if(state == 'b')
						state = 'X';
					else
						state = 'b';
				} else {
					if(state == 'e')
						state = 'E';
					else if(state == 'b')
						state = 'B';
					else
						state = 'e';
				}
			}
		return state;
	}

	bool StrandPairingSet::paired(int res1, int res2, bool antiparallel) const {
		for(std::list<StrandPairing>::const_iterator it = pairings.begin();
				it != pairings.end();
				it++) {
			if( it->antiparallel() == antiparallel
					&& it->contains(res1)
					&& it->get_pair(res1) == res2 )
				return true;
		}
		return false;
	}

	//////////////////////////////////////////////////////////////////////////
	// Handy function for getting out a list of easy-to-read beta pairings.
	//////////////////////////////////////////////////////////////////////////
	std::vector < BetaPair > StrandPairingSet::get_beta_pairs() const {

		std::vector < BetaPair > beta_pairs;

		//stupid iterators.
		for(std::list<StrandPairing>::const_iterator it = pairings.begin();
				it != pairings.end();
				it++) {
			std::vector < BetaPair > beta_pairs_pairing = it->get_beta_pairs();
			for(std::vector< BetaPair >::iterator bit = beta_pairs_pairing.begin();
					bit != beta_pairs_pairing.end();
					bit++)
				beta_pairs.push_back(*bit);
		}
		return beta_pairs;
	}

	StrandPairingSet::~StrandPairingSet() {
	}

	StrandPairing::StrandPairing(int res1, int res2, bool antiparallel, int pleating) :
		begin1( std::min( res1, res2 ) ), end1(begin1),
		begin2( std::max( res1, res2 ) ), end2(begin2),
		antipar(antiparallel)
    {
			pairing1.push_back(begin2);
			pairing2.push_back(begin1);
			pleating1.push_back(pleating);
		}

	StrandPairing::StrandPairing() : begin1(0),end1(0), begin2(0), end2(0), antipar(true) {
	}

	StrandPairing::StrandPairing(const StrandPairing &other) : begin1(other.begin1), end1(other.end1), begin2(other.begin2), end2(other.end2), pairing1(other.pairing1), pleating1(other.pleating1), pairing2(other.pairing2), antipar(other.antipar) {
	}

	StrandPairing::~StrandPairing() {
	}
	
	std::ostream & operator<<(std::ostream & out, const StrandPairing &sp) {
		out << (sp.antipar ? 'A' : 'P') << ' ' << sp.begin1 << '-' << sp.pairing1[0] << " to " << sp.end1 << '-' << sp.pairing1[sp.pairing1.size()-1] << ", reg: ";
		if(sp.antipar) {
			int reg = sp.begin1 + sp.pairing1[0];
			out << reg;
			for(int i = 1; (unsigned int)i < sp.pairing1.size(); i++) {
				if(sp.pairing1[i] != 0 && sp.begin1 + i + sp.pairing1[i] != reg) {
					reg = sp.begin1 + i + sp.pairing1[i];
					out << ", " << reg;
				}
			}
		} else {
			int reg = sp.pairing1[0] - sp.begin1;
			out << reg;
			for(int i = 1; (unsigned int)i < sp.pairing1.size(); i++) {
				if(sp.pairing1[i] != 0 && sp.pairing1[i] - sp.begin1 - i != reg) {
					reg = sp.pairing1[i] - sp.begin1 - i;
					out << ", " << reg;
				}
			}
		}
		out << ", bulges: ";
		int i = 0;
		bool in_bulge = false;
		int j = sp.antipar ? (int)sp.pairing2.size() - 1 : 0;
		int jdiff = sp.antipar ? -1 : 1;
		int jlim = sp.antipar ? 0 : (int)sp.pairing2.size()-1;
		while(i < (int)sp.pairing1.size() && j*jdiff <= jlim) {
			int p1i = sp.pairing1[i];
			int p2j = sp.pairing2[j];
			if(p1i != 0 && p2j != 0) {
				if(in_bulge) {
					out << "} ";
					in_bulge = false;
				}
				i++;
				j+=jdiff;
			}
			if(p1i == 0) {
				if(!in_bulge) {
					in_bulge = true;
					out << '{';
				} else 
					out << ", ";
				out << i + sp.begin1;
				i++;
			}
			if(p2j == 0) {
				if(!in_bulge) {
					in_bulge = true;
					out << '{';
				} else 
					out << ", ";
				out << j + sp.begin2;
				j+=jdiff;
			}
		}
		/*
    out << std::endl;
    for(unsigned int i = 0; i < sp.pairing1.size(); i++)
			out << sp.pairing1[i] << ' ';
		out << std::endl;
    for(unsigned int i = 0; i < sp.pairing2.size(); i++)
			out << sp.pairing2[i] << ' ';
		out << std::endl;
			*/
		out << " pleating: ";
    for(unsigned int i = 0; i < sp.pleating1.size(); i++)
			out << sp.pleating1[i] << ' ';
		return out;
	}

	StrandPairing & StrandPairing::operator=(const StrandPairing &other) {
		begin1 = other.begin1;
		begin2 = other.begin2;
		end1 = other.end1;
		end2 = other.end2;
		pairing1 = other.pairing1;
		pairing2 = other.pairing2;
		pleating1 = other.pleating1;
		return *this;
	}

	int StrandPairing::operator<(const StrandPairing &other) const {
		if(antipar != other.antipar)
			return antipar;

		int reg = antipar ? pairing1[0] + begin1 : pairing1[0] - begin1;
		int otherreg = antipar ? other.pairing1[0] + other.begin1 : other.pairing1[0] - other.begin1;
		if(reg == otherreg) {
			if(end1 <= other.begin1)
				return true;
			else if(begin1 >= other.end1)
				return false;
			else {
				std::cout << "DSSP error: strange strand pairing\n";
				std::cout << begin1 << ' ' << end1 << ' ' << begin2 << ' ' << end2 <<' ' << std::endl;
				std::cout << other.begin1 << ' ' << other.end1 << ' ' << other.begin2 << ' ' << other.end2 <<' ' << std::endl;
				return begin1 < other.begin1;
			}
		} else
			return reg < otherreg;
	}

	int StrandPairing::operator==(const StrandPairing &other) const {
		return (begin1 == other.begin1 && begin2 == other.begin2 &&
				end1 == other.end1 && end2 == other.end2 &&
				pairing1 == other.pairing1 && pairing2 == other.pairing2 &&
				pleating1 == other.pleating1);
	}

  int StrandPairing::BIG_BULGE_LIMIT = 5;
	int StrandPairing::SMALL_BULGE_LIMIT = 2;

	///////////////////////////////////////////////////////////////
	/// @begin StrandPairing::extend
	///
	/// @brief If possible, extend this pairing by the given residues.
	///
	/// @detailed
	/// If one of res1 or res2 is within 2 residues of the beginning
	/// or end of one of the strands of the pairing, and the other
	/// is within 5 residues, extend the pairing.  This is the dssp
	/// definition of allowable beta bulges.  Return true if the
	/// pairing was extended.
	// Assumes we are running through res1 and res2 in an ordered
	// way, so we extend at the beginning or end of the strand.
	///
	/// @param
	///
	/// @global_read
	///
	/// @global_write
	///
	/// @remarks
	///
	/// @references
	///
	/// @authors bblum
	///
	/// @last_modified
	///////////////////////////////////////////////////////////////
	bool StrandPairing::extend(int res1, int res2, bool antiparallel, int pleating) {
		// Make sure res1 < res2
		int temp = std::min(res1, res2);
		res2 = std::max(res1, res2);
		res1 = temp;
		if(begin1 == 0) { // uninitialized
			begin1 = end1 = res1;
			begin2 = end2 = res2;
			antipar = antiparallel;
			pairing1.push_back(begin2);
			pairing2.push_back(begin1);
			pleating1.push_back(pleating);
		}


		if(antiparallel != this->antipar)
			return false;

		bool cando = false;

		if(res1 >= begin1 && res1 <= end1) { // pathological--already in our strand
			cando = (pairing1[res1 - begin1] == res2);
		} else if(res1 > end1 && res1 <= end1 + BIG_BULGE_LIMIT) {
			if(antiparallel) {
				if(res1 > end1 + SMALL_BULGE_LIMIT)
					cando = (res2 < begin2 && res2 >= begin2 - SMALL_BULGE_LIMIT);
				else
					cando = (res2 < begin2 && res2 >= begin2 - BIG_BULGE_LIMIT);
			} else {
				if(res1 > end1 + SMALL_BULGE_LIMIT) //bulge of size > 1
					cando = (res2 > end2 && res2 <= end2 + SMALL_BULGE_LIMIT);
				else
					cando = (res2 > end2 && res2 <= end2 + BIG_BULGE_LIMIT);
			}
		} else if(res1 < begin1 && res1 >= begin1 - BIG_BULGE_LIMIT) {
			if(antiparallel) {
				if(res1 < begin1 - SMALL_BULGE_LIMIT)
					cando = (res2 > end2 && res2 <= end2 + SMALL_BULGE_LIMIT);
				else
					cando = (res2 > end2 && res2 <= end2 + BIG_BULGE_LIMIT);
			} else {
				if(res1 < begin1 - SMALL_BULGE_LIMIT) //bulge of size > 1
					cando = (res2 < begin2 && res2 >= begin2 - SMALL_BULGE_LIMIT);
				else
					cando = (res2 < begin2 && res2 >= begin2 - BIG_BULGE_LIMIT);
			}
		}

		// if extendable, insert this pairing in, adjust begins and ends
		if(cando) {
			if(res1 < begin1) {
				if(res1 < begin1 - 1) {
					pairing1.insert(pairing1.begin(),begin1 - res1 - 1, 0);
					pleating1.insert(pleating1.begin(),begin1 - res1 - 1, 0);
				}
				pairing1.insert(pairing1.begin(), res2);
				pleating1.insert(pleating1.begin(), pleating);
				begin1 = res1;
			} else if(res1 > end1) {
				if(res1 > end1 + 1) {
					pairing1.insert(pairing1.end(), res1 - end1 - 1, 0);
					pleating1.insert(pleating1.end(), res1 - end1 - 1, 0);
				}
				pairing1.insert(pairing1.end(), res2);
				pleating1.insert(pleating1.end(), pleating);
				end1 = res1;
			}

			if(res2 < begin2) {
				if(res2 < begin2 - 1)
					pairing2.insert(pairing2.begin(),begin2 - res2 - 1, 0);
				pairing2.insert(pairing2.begin(), res1);
				begin2 = res2;
			} else if(res2 > end2) {
				if(res2 > end2 + 1)
					pairing2.insert(pairing2.end(), res2 - end2 - 1, 0);
				pairing2.insert(pairing2.end(), res1);
				end2 = res2;
			}
		}

		return cando;
	}

	// Return true if the given residue is part of a beta bulge
	bool StrandPairing::is_bulge(int res) const {
		if (! contains(res) ) return false;
		if (get_pair(res) != 0 ) return false;
		return true;
		//		return (contains(res) && get_pair(res) == 0);
	}

	// Return true if the given residue is part of this pairing
	// (includes bulges)
	bool StrandPairing::contains(int res) const {
		return (res >= begin1 && res <= end1) || (res >= begin2 && res <= end2);
	}

	// Return the residue to which the given residue is paired
	// (0 if the residue is unpaired, i.e. a bulge residue, or
	// is not contained in the pairing at all)
	int StrandPairing::get_pair(int res) const {
		if(res >= begin1 && res <= end1)
			return pairing1[res-begin1];
		else if(res >= begin2 && res <= end2)
			return pairing2[res-begin2];
		else
			return 0;
	}

	// Return true if this pairing is of length greater than 1.
	bool StrandPairing::is_ladder() const {
		return end1-begin1 > 0 && end2-begin2 > 0;
	}

	bool StrandPairing::antiparallel() const {
		return antipar;
	}

	std::vector < BetaPair > StrandPairing::get_beta_pairs() const {
		std::vector < BetaPair > beta_pairs;
		for(int i = begin1; i <= end1; i++)
			if(pairing1[i-begin1] != 0) {
				BetaPair pair;
				pair.res1 = i;
				pair.res2 = pairing1[i - begin1];
				pair.orientation = antipar ? 1 : 2;
				pair.pleating = pleating1[i - begin1];
				beta_pairs.push_back(pair);
			}
		return beta_pairs;
	}

DSSP::DSSP() {
	pair_set = NULL;
}

DSSP::~DSSP() {
	if(pair_set != NULL)
		delete pair_set;
}

//////////////////////////////////////////////////////////////////////////////
/// @begin fill_hbond_bb_pair_score
///
/// @brief Populates the hbond_bb_pair_score array
///
/// @detailed
/// Uses the rosetta hydrogen bond energies computed in hbonds::hbond_set
/// to fill the hbond_bb_pair_score array.  The hbond_set data structure
/// must already be populated, by (e.g.) full atom scoring.
/// Entry (i,j) is the backbone
/// hydrogen bond energy between residues i (acceptor) and j (donor).
///
/// @global_read
/// hbonds::hbond_set
///
/// @global_write
/// dssp_ns::hbond_bb_pair_score
///
/// @remarks
///
/// @references
///
/// @authors bblum
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////
	/*
void
DSSP::fill_hbond_bb_pair_score() {
	using namespace hbonds;

	hbond_bb_pair_score.dimension(misc::total_residue, misc::total_residue);
	hbond_bb_pair_score = 0.0f;
	int dres, ares, type;
	float enrg;

	for( int i = 1; i <= hbond_set.nhbonds(); ++i ) {
		dres = hbond_set.hbdon_res(i);
		ares = hbond_set.hbact_res(i);
		enrg = hbond_set.hbenergies(i);
		// first index is always acceptor (carboxyl group), second is donor (amyl group)
		type = hbond_set.hbtype(i);
		if(type == SRBB_HBTYPE || type == LRBB_HBTYPE)
			hbond_bb_pair_score(ares, dres) = enrg;
	}
}
*/


//////////////////////////////////////////////////////////////////////////////
void
copy_the_relevant_atoms( FArray3D_float & full_coord_dest,
												 FArray3D_float & full_coord_src){
	using namespace misc;

	for (int i = 1; i <= total_residue; i++ ){

		int const natoms( aaproperties_pack::natoms( res(i), res_variant(i) ) );

		for ( int j=1; j<= natoms; ++j ) {
			for ( int k = 1; k<= 3; ++k ) {
				full_coord_dest( k, j, i ) = full_coord_src( k, j, i );
			}
		}

	}

}

//////////////////////////////////////////////////////////////////////////////
/// @begin fill_hbond_bb_pair_score_dssp
///
/// @brief Populates the hbond_bb_pair_score array with dssp energies
///
/// @detailed
/// Uses hydrogen bond energies computed a la dssp to fill the
/// hbond_bb_pair_score array.  Entry (i,j) is the backbone
/// hydrogen bond energy between residues i (acceptor) and j (donor).
///
/// @global_read
/// misc::ints::total_residue
/// misc::current_pose::Eposition
/// misc::current_pose::full_coord
/// misc::ints::res
/// misc::current_pose::res_variant
/// aaproperties_pack::properties_per_aa_aav::HNpos
/// aaproperties_pack::properties_per_aa_aav::HNpos
/// param_aa::aa_pro
/// termini_ns::is_N_terminus
/// termini_ns::is_C_terminus
///
/// @global_write
/// dssp_ns::hbond_bb_pair_score
///
/// @remarks
///
/// @references
///
/// @authors bblum
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////
void
DSSP::fill_hbond_bb_pair_score_dssp() {
	using namespace dssp_ns;
	using namespace misc;
	using namespace termini_ns;
	using namespace aaproperties_pack;

	hbond_bb_pair_score.dimension(total_residue, total_residue);
	hbond_bb_pair_score = 0.0f;

	FArray3D_float full_coord_backup( 3, param::MAX_ATOM(), total_residue );
	copy_the_relevant_atoms( full_coord_backup, full_coord );

	copy_position_to_fullcoord(Eposition,full_coord,total_residue);
	put_nh(full_coord, 1, total_residue, res, res_variant, is_N_terminus, is_C_terminus, 0.0f);
	//	initialize_fullcoord_array(Eposition, full_coord, total_residue, res, res_variant);

	float C[3], O[3], N[3], H[3], rON, rCH, rOH, rCN, E, dist, total;
	for( int i = 1; i <= total_residue; ++i ) {
		for( int j = 1; j <= total_residue; ++j ) {
			if( i == j || i - j == 1 || j - i == 1 ) continue;

			E = 0.0;
			total = 0.0;
			for ( int k = 0, li = Eposition.index(k+1, 2, i),
					lj = Eposition.index(k+1, 2, j); k < 3; ++k, ++li, ++lj ) {
				dist = Eposition[li] - Eposition[lj];
				total += dist * dist;
			}
			// Don't consider pairs where Ca not within 10A
			if(total > 100.0) continue;

			for ( int k = 0, l = Eposition.index(k+1, 4, i); k < 3; ++k, ++l ) {
				C[k] = Eposition[l];
			}

			for ( int k = 0, l = Eposition.index(k+1, 5, i); k < 3; ++k, ++l ) {
				O[k] = Eposition[l];
			}

			for ( int k = 0, l = Eposition.index(k+1, 1, j); k < 3; ++k, ++l ) {
				N[k] = Eposition[l];
			}

			rON = rCN = 0.0f;
			for( int k = 0; k < 3; k++ ) {
				rON += ( O[k] - N[k] ) * ( O[k] - N[k] );
				rCN += ( C[k] - N[k] ) * ( C[k] - N[k] );
			}
			rON = sqrt(rON);
			rCN = sqrt(rCN);


			int HN1_pos = HNpos( res(j), res_variant(j) );
			int num_of_HN;

			if(is_N_terminus(j)) num_of_HN = 3;
			else num_of_HN = 1;

			if(	res(j) == param_aa::aa_pro ) num_of_HN--;

			for ( int nh = 1; nh <= num_of_HN; ++nh ) {
				rOH = rCH = 0.0f;
				for ( int k = 0, l = full_coord.index(k+1, nh+HN1_pos-1, j);
						k < 3; ++k, ++l ) {
					H[k] = full_coord[l];
					rOH += ( O[k] - H[k] ) * ( O[k] - H[k] );
					rCH += ( C[k] - H[k] ) * ( C[k] - H[k] );
				}
				rOH = sqrt(rOH);
				rCH = sqrt(rCH);
				E = 27.888f * ( 1.0f / rON + 1.0f / rCH - 1.0f / rOH - 1.0f / rCN);
				if(E < hbond_bb_pair_score(i, j) ) {
					hbond_bb_pair_score(i, j) = E;
					break;
				}
			}
		}
	}

	copy_the_relevant_atoms( full_coord, full_coord_backup);
}

void
DSSP::dssp( FArray1DB_char &secstruct ) {
  for(int i = 1; i <= misc::total_residue; i++)
		secstruct(i) = dssp_secstruct(i);
}

void
DSSP::dssp_featurizer( FArray1DB_char &secstruct ) {
	for( int i = 1; i <= misc::total_residue; i++ ) {
		if( dssp_secstruct(i) == 'H' || dssp_secstruct(i) == 'G' || dssp_secstruct(i) == 'I' ) {
			secstruct(i) = 'H';
    } else if(dssp_secstruct(i) == 'B' || dssp_secstruct(i) == 'E') {
			// see if paired on one or both sides
		 	secstruct(i) = pair_set->featurizer_state(i);
		} else secstruct(i) = 'L';
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin dssp_reduced
///
/// @brief Reduces to E/H/L secondary structure alphabet
///
/// @detailed
/// This function simply reduces dssp's secondary structure alphabet (which
/// includes 3- and 5-turn helices and various kinds of loop, and
/// differentiates lone beta bridges from extended beta strand pairings)
/// into the standard E/H/L alphabet, as follows:
/// G,H,I --> H
/// E,B --> E
/// S,T,blank --> L
///
/// @global_read
/// misc::ints::total_residue
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors bblum
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////
void
DSSP::dssp_reduced( FArray1DB_char &secstruct ) {
	for( int i = 1; i <= misc::total_residue; i++ ) {
		if( dssp_secstruct(i) == 'H' || dssp_secstruct(i) == 'G' || dssp_secstruct(i) == 'I' ) {
			secstruct(i) = 'H';
    } else if(dssp_secstruct(i) == 'B' || dssp_secstruct(i) == 'E') {
		 	secstruct(i) = 'E';
		} else secstruct(i) = 'L';
	}
}

//////////////////////////////////////////////////////////////////////////////
/// @begin DSSP::compute
///
/// @brief Runs dssp, calculating per-residue secondary structure.
///
/// @detailed
/// dssp is a standard algorithm for per-residue secondary structure analysis.
/// It has the following alphabet:
/// H: 4-turn helix
/// B: beta bridge
/// E: extended beta strand
/// G: 3-turn helix
/// I: 5-turn helix
/// T: helix-like loop (some nearby hbonds)
/// S: loop with high curvature
///  : loop
/// Most of these determinations are made on the basis of hydrogen bonds,
/// with torsion angles disregarded.  The designations are reported with priority
/// given to categories higher in the list (e.g. if a residue has both high-
/// curvature (S) and helical hydrogen bonds (H), it will be reported as H).
/// Experiments indicate that this function agrees almost completely with
/// the true dssp results (run on dumped pdbs).  Slight differences can perhaps
/// be chalked up to unequal placement of hydrogens.
///
/// @global_read
/// misc::ints::total_residue
/// dssp_ns::hbond_bb_pair_score
///
/// @global_write
/// dssp_ns::hbond_bb_pair_score
///
/// @remarks
///
/// @references
///
/// @authors bblum
///
/// @last_modified
//////////////////////////////////////////////////////////////////////////////
void
DSSP::compute() {
	using namespace misc;
  using namespace dssp_ns;

	PROF_START( prof::DSSP );

	float dssp_hbond_threshold = -0.5;

	dssp_secstruct.dimension(misc::total_residue);


	fill_hbond_bb_pair_score_dssp(); // fills hbond_bb_pair_score array

	// Initialize to all loops
	for ( int i = 1; i <= misc::total_residue; i++ ) dssp_secstruct(i) = ' ';

	bool helix;

	// Record all 5-turn helices (I)
	for ( int i=1; i <= total_residue-5; i++ ) {
		if ( hbond_bb_pair_score(i, i + 5) < dssp_hbond_threshold ) {
			helix = i < total_residue - 5 &&
        hbond_bb_pair_score(i + 1, i + 6) < dssp_hbond_threshold;
			for ( int j = 1; j < 6; j++) {
				if ( helix )
          dssp_secstruct(i + j) = 'I';
				else if ( j < 5 && dssp_secstruct(i + j) == ' ' )
          dssp_secstruct(i + j) = 'T';
			}
		}
	}

	// Record all 3-turn helices (G)
	for ( int i = 1; i <= total_residue - 3; i++ ) {
		if ( hbond_bb_pair_score(i, i + 3) < dssp_hbond_threshold ) {
			helix = i < total_residue - 3 &&
        hbond_bb_pair_score(i + 1, i + 4) < dssp_hbond_threshold;
			for ( int j = 1; j < 4; j++ ) {
				if ( helix )
          dssp_secstruct(i+j) = 'G';
				else if ( j < 3 && dssp_secstruct(i+j) == ' ' )
          dssp_secstruct(i+j) = 'T';
			}
		}
	}


	// Record all strands (B and E)
	if(pair_set != NULL)
		delete pair_set;
	pair_set = new StrandPairingSet(hbond_bb_pair_score, dssp_hbond_threshold, total_residue);

	for ( int i = 1; i <= total_residue; i++ ) {
		char state = pair_set->dssp_state(i);
		if ( state != ' ' ) dssp_secstruct(i) = state;
	}

	// Record all 4-turn helices (H)
	for ( int i = 1; i <= total_residue - 4; i++ ) {
		if ( hbond_bb_pair_score(i, i + 4) < dssp_hbond_threshold ) {
			helix = i < total_residue - 4 &&
        hbond_bb_pair_score(i + 1, i + 5) < dssp_hbond_threshold;
			for ( int j = 1; j < 5; j++ ) {
				if ( helix )
			/*
				&& torsion_bin(phi(i+j), psi(i+j), omega(i+j)) != 'A')
					// only allow helix chunks in which every residue has bin A
			*/
					dssp_secstruct(i + j) = 'H';
				else if(j < 4 && dssp_secstruct(i + j) == ' ')
					dssp_secstruct(i + j) = 'T';
			}
		}
	}


	// Record all tight turns (S), only if still a loop
	for ( int i = 3; i <= total_residue - 2; i++ ) {
		if ( dssp_secstruct(i) == ' ' ) {
			float x1, y1, z1, x2, y2, z2, length1, length2, dot;
			x1 = Eposition(1, 2, i) - Eposition(1, 2, i - 2);
			y1 = Eposition(2, 2, i) - Eposition(2, 2, i - 2);
			z1 = Eposition(3, 2, i) - Eposition(3, 2, i - 2);

			x2 = Eposition(1, 2, i + 2) - Eposition(1, 2, i);
			y2 = Eposition(2, 2, i + 2) - Eposition(2, 2, i);
			z2 = Eposition(3, 2, i + 2) - Eposition(3, 2, i);

			length1 = x1 * x1 + y1 * y1 + z1 * z1;
			length2 = x2 * x2 + y2 * y2 + z2 * z2;

			dot = ( x1 * x2 + y1 * y2 + z1 * z2 ) / sqrt( length1 * length2 );
			if ( dot < .34202014 )
				dssp_secstruct(i) = 'S';
		}
	}

	PROF_STOP( prof::DSSP );

}

bool
DSSP::paired(int res1, int res2, bool antiparallel) {
	return pair_set->paired(res1, res2, antiparallel);
}
} // end namespace dssp_ns
