// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 12404 $
//  $Date: 2007-02-01 15:47:11 -0800 (Thu, 01 Feb 2007) $
//  $Author: bblum $

#ifndef INCLUDED_dssp
#define INCLUDED_dssp

// ObjexxFCL Headers
#include "ObjexxFCL/ObjexxFCL.hh"
#include "ObjexxFCL/FArray1D.hh"
#include "ObjexxFCL/FArray2D.hh"

// C++ Headers
#include <vector>
#include <list>


namespace dssp_ns {

	typedef struct {
		int res1;
		int res2;
		int orientation;
		int pleating;
	} BetaPair;

	//////////////////////////////////////////////////////////////////
	// StrandPairing
	// This class encapsulates a pairing between two beta strands.
	// It is designed to be initialized with a single residue-residue
	// pairing and then extended, residue by residue, along the
	// strands.  It MUST be extended sequentially along the length
	// of the strands.  It can accomodate beta bulges of length at
	// most 1 on one strand and 4 on the other (inspired by dssp
	// guidelines).
	//////////////////////////////////////////////////////////////////
	class StrandPairing {
		public:
			StrandPairing();
			StrandPairing(const StrandPairing &);
			StrandPairing(int res1, int res2, bool antiparallel, int pleating);
			~StrandPairing();
			StrandPairing &operator=(const StrandPairing &);
			int operator==(const StrandPairing &rhs) const;
			int operator<(const StrandPairing &rhs) const;
			bool contains(int res) const;
			bool is_bulge(int res) const;
			bool is_ladder() const;
			int get_pair(int res)const ;
			bool check_pleat() const;
			int get_pleating(int res)const ;
			bool extend(int res, int res2, bool antiparallel, int pleating);
			void extend_to(int res);
			bool antiparallel() const;
	 	  std::vector < BetaPair > get_beta_pairs() const;
			bool merge(const StrandPairing &other, bool domerge = false);
			friend std::ostream & operator<<(std::ostream & out, const StrandPairing &sp);

			static int BIG_BULGE_LIMIT;
			static int SMALL_BULGE_LIMIT;

		private:
			int begin1, end1, begin2, end2;
			// vector listing which residues are paired to the residues
			// in strand 1.  0 indicates beta bulge (unpaired).
			std::vector<int> pairing1;
			std::vector<int> pleating1;
			// similar to pairing1 but for strand 2.
			std::vector<int> pairing2;
			bool pleat_weird;
			bool antipar;
	};

	std::ostream & operator<<(std::ostream & out, const StrandPairing &sp);
	//////////////////////////////////////////////////////////////////
	// StrandPairingSet
	// This class maintains a set of strand pairings and provides
	// access functions to determine whether a particular residue
	// participates in any of them, and in what capacity.
	// Currently only used by dssp.
	//////////////////////////////////////////////////////////////////
	class StrandPairingSet {
		public:
			StrandPairingSet(FArray2DB_float const &hbonds, float threshold, int nres);
			~StrandPairingSet();
			void add_decoy(int dec);
			bool check_pleat() const;
			char dssp_state(int res) const;
			char featurizer_state(int res) const;
			bool paired(int res1, int res2, bool antiparallel) const;
	 	  std::vector < BetaPair > get_beta_pairs() const;
			bool merge(const StrandPairingSet &other, bool domerge = false);
			friend std::ostream & operator<<(std::ostream & out, const StrandPairingSet &sp);

			std::list<StrandPairing> pairings;

		private:
			void add_pairing(int res1, int res2, bool antiparallel, int pleating);
	    void selfmerge();
	};

	std::ostream & operator<<(std::ostream & out, const StrandPairingSet &sp);

	class DSSP {
		public:
			DSSP();
			~DSSP();

			void compute();
			void dssp_reduced( FArray1DB_char &secstruct );
			void dssp_featurizer( FArray1DB_char &secstruct );
			void dssp( FArray1DB_char &dssp_secstruct );
			bool paired( int res1, int res2, bool antiparallel );

			FArray2D_float hbond_bb_pair_score;

		private:
			void fill_hbond_bb_pair_score_dssp();
			FArray1D_char dssp_secstruct;

			StrandPairingSet *pair_set;

	  public:
		   inline
			 StrandPairingSet * strand_pairing_set(){
				 return pair_set;
			 }

	};
}

#endif
