// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   epigraft_functions.hh
/// @brief  Various functions for epigraft protocols.
/// @author Yih-En Andrew Ban (yab@u.washington.edu)
/// @author Bill Schief (schief@u.washington.edu)
/// @author Sergey Menis (menis@u.washington.edu)


#ifndef INCLUDED_epigraft_epigraft_functions_HH_
#define INCLUDED_epigraft_epigraft_functions_HH_

// epigraft headers
#include <AtomPoint.hh>

// rootstock headers
#include <BoundingBox.hh>
#include <Octree.hh>
#include "rootstock_types.hh"

// Rosetta headers
#include <aaproperties_pack.h>
#include <param.h>
#include <param_pack.h>
#include <pdbstatistics_pack.h>
#include <pose.h>

// numeric headers
#include <numeric/conversions.hh>
#include <numeric/xyzVector.hh>

// C++ headers
#include <set>


namespace epigraft {


using namespace rootstock; // temporary


/// @brief constructs bounding box around a Pose
inline
BoundingBox
backbone_bounding_box(
	Pose const & p
)
{
	FArray3D_float const & fc = p.full_coord();

	BoundingBox bb( PointPosition( fc( 1, 1, 1 ), fc( 2, 1, 1 ), fc( 3, 1, 1 ) ) );

	for ( int res = 1, nres = p.total_residue(); res <= nres; ++res ) {
		for ( int atom = 1; atom <= 4; ++atom ) {
			bb.add( PointPosition( fc( 1, atom, res ), fc( 2, atom, res ), fc( 3, atom, res ) ) );
		}
	}

	return bb;
}


/// @brief fills in octree with atoms from a Pose
inline
void
fill_backbone_octree(
	Octree< AtomPoint > & oc,
	Pose const & p
)
{
	FArray3D_float const & fc = p.full_coord();

	for ( int res = 1, nres = p.total_residue(); res <= nres; ++res ) {
		for ( int atom = 1; atom <= 4; ++atom ) {
			oc.add( AtomPoint( res, atom, fc( 1, atom, res ), fc( 2, atom, res ), fc( 3, atom, res ) ) );
		}
	}
}


/// @brief queries octree using a Pose and tabulates near residues within given distance of query point
inline
void
find_near_residues(
	Octree< AtomPoint > const & oc,
  float const & query_x,
  float const & query_y,
  float const & query_z,
	float const & cutoff,
	std::set< int > & near_residues
)
{

	std::vector< AtomPoint > nn = oc.near_neighbors( cutoff, query_x, query_y, query_z );
	/*
	std::vector<AtomPoint>::iterator iter;
	int count=0;
	for (iter = nn.begin(); iter != nn.end(); iter++){
		if (iter->atom_id() == 5){
			count++;
		}
	}*/
//			std::cout << "neighbor count" << count << std::endl;

	for ( std::vector< AtomPoint >::const_iterator ia = nn.begin(), iae = nn.end(); ia != iae; ++ia ) {
				near_residues.insert( ia->residue_id() );
	}
}


/// @brief calculate lj-repulsive between two atoms
/// @note  code clipped from pairenergy.h
inline
Real
lj_rep(
  Integer const & atom_type1,
  Integer const & atom_type2,
  Real const & d2
)
{
  if ( d2 >= pdbstatistics_pack::safe_max_dis2 || d2 == 0 ) { // no explosions ;)
    return 0.0;
  }

  // bins and interpolation fraction
  Real const d2_bin = d2 * pdbstatistics_pack::fa_bins_per_A2;
  Integer const disbin1 = static_cast< int >( d2_bin ) + 1;
  Real const frac = d2_bin - ( disbin1 - 1 );

  Integer const l1 = pdbstatistics_pack::pCurrentEtable->ljatr.index( disbin1, atom_type1, atom_type2 );
  Integer const l2 = l1 + 1;
  Real const e1 = pdbstatistics_pack::pCurrentEtable->ljrep[ l1 ];
  return e1 + frac * ( pdbstatistics_pack::pCurrentEtable->ljrep[ l2 ] - e1 ); // repulsive
}


inline
Real
octree_inter_rep_with_ligand(
  Octree< AtomPoint > const & oc,
  Pose const & model,
  Pose const & query //ligand
)
{
  Real repulsive = 0.0f;
	FArray3D_float query_full_coord(query.full_coord());

  Integer q_res = 2; // the ligand is attached at the second position
  Integer q_nres = query.total_residue();

  for ( ; q_res <= q_nres; ++q_res ) {
    Integer const q_aa( query.res( q_res ) );
    Integer const q_aav( query.res_variant( q_res ) );
    for ( Integer q_atom = 1, q_atom_e = aaproperties_pack::natoms( q_aa, q_aav ); q_atom <= q_atom_e; ++q_atom ) {
      Integer query_atom_type = aaproperties_pack::fullatom_type( q_atom, q_aa, q_aav );

      Integer idx = query_full_coord.index( 1, q_atom, q_res );

      std::vector< AtomPoint > nn = oc.near_neighbors( 5.5, (Real)query_full_coord[ idx ], (Real)query_full_coord[ idx + 1 ], (Real)query_full_coord[ idx + 2 ] );

      for ( std::vector< AtomPoint >::const_iterator ia = nn.begin(), iae = nn.end(); ia != iae; ++ia ) {

        Integer const m_res = (*ia).residue_id();
        Integer const m_aa = model.res( m_res );
        Integer const m_aav = model.res_variant( m_res );
        Integer const m_atom = (*ia).atom_id();

			//	std::cout<< "atom_id " << m_atom << " res:" << m_res << " aa: " << m_aa;

        Integer model_atom_type = aaproperties_pack::fullatom_type( m_atom, m_aa, m_aav );

        Real rep = lj_rep( model_atom_type, query_atom_type, distance_squared( (*ia), numeric::xyzTriple< float >( &query_full_coord[ idx ] ) ) );

			//	std::cout << " rep: " << rep << std::endl;

        repulsive += param_pack::pack_wts.Wrep() * rep;
      }
    }
  }
  return repulsive;
}



} // namespace epigraft


#endif /*INCLUDED_epigraft_epigraft_functions_HH_*/
