// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 21214 $
//  $Date: 2008-03-20 14:27:59 -0700 (Thu, 20 Mar 2008) $
//  $Author: jiangl $

// Rosetta Headers
#include "hbonds.h"
#include "after_opts.h" // for tighthbonds option
#include "atom_is_backbone.h"
#include "current_pose.h"
#include "design.h"
#include "dna_ns.h" // rna magic numbers.
#include "disulfides.h" // for cys_res_in_disulf
//#include "dock_loops.h"
#include "docking.h"
#include "docking_minimize.h"
#include "enzyme.h" // for get_enable_ligaa_flag
#include "enzyme_ns.h"
#include "fullatom.h"
#include "fullatom_energies.h"
#include "fullatom_energy.h"
#include "gl_graphics.h"
#include "hbonds_ns.h"
#include "hbonds_geom.h"
#include "minimize.h" // for get_rigid_body_region
#include "minimize_ns.h"
#include "misc.h"
#include "nblist_ns.h"
#include "pack.h"
#include "pack_geom_inline.h"
#include "param.h"
#include "param_aa.h"
#include "param_pack.h"
#include "param_torsion.h"
#include "pdbstats.h"
#include "pdbstatistics_pack.h"
#include "pose.h"
#include "pose_ligand.h"
#include "pose_rna_ns.h"
#include "read_aaproperties.h"
#include "refold.h"
//#include "rotamer_trials.h"
#include "runlevel.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>
#include <ObjexxFCL/FArray3Da.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <iostream>


//Utility Headers
#include <utility/basic_sys_util.hh>
using namespace hbonds;

// forward declarations

void
get_aa_1way_hbE_B(
									bool const fullatom,
									int const don_aa,
									int const act_aa,
									int const don_aav,
									int const act_aav,
									int const don_res,
									int const act_res,
									int const don_nb,
									int const act_nb,
									FArray2DB_float & don_xyz,
									FArray2DB_float & act_xyz,
									float & scE,
									float & sr_bbE,
									float & lr_bbE,
									float & sc_bbE,
									float & sc1_bb2_scdon,
									float & sc2_bb1_scacc,
									int & nhbonds_l,
									FArray1DB_float & hbenergies_l,
									FArray1DB_int & hbdonh_atm_l,
									FArray1DB_int & hbdon_res_l,
									FArray1DB_int & hbact_atm_l,
									FArray1DB_int & hbact_res_l,
									FArray1DB_int & hbtype_l,
									bool & bb_hb_exists,
                  HBDerivType const deriv_type,
									FArray3DB_float & hbderiv_l
									);

void
get_aa_1way_hbE(
                bool const fullatom,
								int const don_aa,
								int const act_aa,
								int const don_aav,
								int const act_aav,
								int const don_res,
								int const act_res,
								int const don_nb,
								int const act_nb,
								FArray2Da_float don_xyz,
								FArray2Da_float act_xyz,
								float & scE,
								float & sr_bbE,
								float & lr_bbE,
								float & sc_bbE,
								int & nhbonds_l,
								FArray1Da_float hbenergies_l,
								FArray1Da_int hbdonh_atm_l,
								FArray1Da_int hbdon_res_l,
								FArray1Da_int hbact_atm_l,
								FArray1Da_int hbact_res_l,
								FArray1Da_int hbtype_l,
								bool & bb_hb_exists,
                HBDerivType const deriv_type,
								FArray3Da_float hbderiv_l
								);


float
adjust_dna_base_weight(
											 int const don_type,
											 int const don_atom,
											 int don_nb,
											 int const act_type,
											 int const act_atom,
											 int act_nb
											 );

/////////////////////////////////
/// Weighting by HBEvalType
/////////////////////////////////
float HBE_Weight[hbe_MAX+1]; //JSS Hack for the moment; will use FArray1D.

void setHBE_Weight(float const srbb_weight, float const lrbb_weight, float const scbb_weight, float const sc_weight)
{
  HBE_Weight[hbe_NONE] = 0.0;
  HBE_Weight[hbe_BB] = 0.0;  // not a final type
  HBE_Weight[hbe_BBTURN] = srbb_weight;
  HBE_Weight[hbe_BBHELIX] = srbb_weight;
  HBE_Weight[hbe_BBOTHER] = lrbb_weight;
  HBE_Weight[hbe_SP2B] = scbb_weight;
  HBE_Weight[hbe_SP3B] = scbb_weight;
  HBE_Weight[hbe_RINGB] = scbb_weight;
  HBE_Weight[hbe_BSC] = scbb_weight;
  HBE_Weight[hbe_SP2SC] = sc_weight;
  HBE_Weight[hbe_SP3SC] = sc_weight;
  HBE_Weight[hbe_RINGSC] = sc_weight;
}

void setHBE_Weight()
{
  using namespace param_pack;
  float const srbb_weight = pack_wts.Whb_srbb() * pack_wts.Whbond_bb();
	float const lrbb_weight = pack_wts.Whb_lrbb() * pack_wts.Whbond_bb();
	float const scbb_weight = pack_wts.Whb_sc()   * pack_wts.Whbond_bb_sc();
	float const sc_weight = pack_wts.Whb_sc()   * pack_wts.Whbond_sc();
  setHBE_Weight(srbb_weight, lrbb_weight, scbb_weight, sc_weight);
}


/////////////////////////////////
/// NEIGHBOR WEIGHTING
////////////////////////////////
void
burial3class_weight_initializer( FArray2D_float & burial )
{
	burial( 1, 1 ) = 0.2 ; burial( 1, 2 ) = 0.2 ; burial( 1, 3 ) = 0.55;
	burial( 2, 1 ) = 0.2 ; burial( 2, 2 ) = 0.55; burial( 2, 3 ) = 1.0 ;
	burial( 3, 1 ) = 0.55; burial( 3, 2 ) = 1.0 ; burial( 3, 3 ) = 1.0 ;
}


inline int
get_burial_3(
						 int const neighbors,
						 int const threshold_1,
						 int const threshold_3
						 )
{
	//tk get burial measure, three possible classes:
	//tk 1: exposed, 2: intermediate, 3:buried
	if ( neighbors > threshold_1 ) {
		if ( neighbors >= threshold_3 ) return 3;
		else return 2;
	}
	else return 1;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hb_env_dep_burial_tk
///
/// @brief
/// assign a burial weight based on CB neighbors of two
/// interacting residues using tanja's 3 classes
///
/// @detailed
///
/// @param[in] nb1   input - number of CB neighbors of residue 1
/// @param[in] nb2   input - number of CB neighbors of residue 2
/// @returns hbond burial weight
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
hb_env_dep_burial_tk(int const nb1, int const nb2)
{
	//tk assign weight based on CB neighbors of two interacting residues

	// local
	int const exposed_threshold = { 10 };
	int const buried_threshold = { 20 };

	static FArray2D_float const burial3class_weight( 3, 3, burial3class_weight_initializer );

	return burial3class_weight(get_burial_3(nb1,exposed_threshold,buried_threshold),
														 get_burial_3(nb2,exposed_threshold,buried_threshold));
}

////////////////////////////////////////////////////////////////////////////////
/// @begin adjust_dna_base_weight
///
/// @brief
/// Increase the number of neighbors for DNA bases that hydrogen bond to protein
///
/// @authors
/// duarte
///
////////////////////////////////////////////////////////////////////////////////
float
adjust_dna_base_weight(
											 int const don_type,
											 int const don_atom,
											 int don_nb,
											 int const act_type,
											 int const act_atom,
											 int act_nb
											 )
{
	using namespace param_aa;
	//Hydrogen bonds between nucleotides should not be counted currently
	if ( is_NA(don_type) && is_NA(act_type) ) return 0.0;
	if ( is_NA(don_type) && don_atom > dna_bb_atoms ) don_nb *= 3;
	if ( is_NA(act_type) && act_atom > dna_bb_atoms ) act_nb *= 3;
	return hb_env_dep_burial_lin( act_nb, don_nb);
}



////////////////////////////////////////////////////////////////////////////////
/// @begin set_use_W_hb_end_dep_tk
///
/// @brief
/// set flag use_W_hb_env_dep_tk determining whether to use environment
/// dependent hydrogen bonding
///
/// @detailed
///
/// @param[out] use_W_hb_env_dep_tk   output -
///
/// @global_read
///
/// @global_write use_W_hb_env_dep_tk in hbond_ns
///
/// @remarks
///
/// @references
///
/// @authors Tanja Kortemme 09/03/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_use_W_hb_env_dep_tk( bool truefalse )
{
	using namespace hbonds;
	use_W_hb_env_dep_tk = truefalse;
}

//===============================================
// functions for environmental dependent H-bond
// Lin Jiang 09/19/2003
//===============================================
//lin      input:  neighbor number of one residue
//lin      output: environment burial weight for hb calculation
//JSS note: this is half of the weight from one atom;
// the burial weight is the sum from donor and acceptor.
inline float
burial_weight(int const nb)
{
	if ( nb < 7 ) return 0.1;
	if ( nb > 24 ) return 0.5;
	return (nb-2.75)*(0.5/21.25);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hb_env_dep_burial_lin
///
/// @brief return environment-dependent burial weight for pair hb calculation
///
/// @detailed
///
/// @param[in]   nb1            - in - neighbor number of residue 1
/// @param[in]   nb2            - in - neighbor number of residue 2
/// @return burial weight
///
/// @global_read none
///
/// @global_write none
///
/// @remarks
///
/// @references
///
/// @authors   Lin Jiang 10/13/03
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
float
hb_env_dep_burial_lin(int const nb1, int const nb2)
{
	return (burial_weight(nb1) + burial_weight(nb2));
}


////////////////////////////////////////////////////////////////////////////////
/// @begin set_use_W_hb_env_dep_lin
///
/// @brief
/// set flag use_W_hb_env_lin determining whether to use environment
/// dependent hydrogen bonding
///
/// @detailed
///
/// @global_read none
///
/// @global_write  use_W_hb_env_dep_lin in "hbond_ns.h'
///
/// @remarks
///
/// @references
///
/// @authors  Lin Jiang 09/20/2003
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_use_W_hb_env_dep_lin( bool truefalse )
{
	hbonds::use_W_hb_env_dep_lin = truefalse;
}

bool
get_use_W_hb_env_dep_lin()
{
	return hbonds::use_W_hb_env_dep_lin;
}



////////////////////////////////////////////////////////////////////////////////
/// @begin set_fine_hb_categories
///
/// @brief
/// set flag fine_hb_categories determining whether to use Jack's revised hbond energies
///
/// @detailed
///
/// @param[in] truefalse - value to set flag
///
/// @global_read
///
/// @global_write fine_hb_categories in hbond_ns
///
/// @remarks
///
/// @references
///
/// @authors Jack Snoeyink
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
set_fine_hb_categories( bool truefalse )
{
	using namespace hbonds;
	fine_hb_categories = truefalse;
}

bool
get_fine_hb_categories()
{
	using namespace hbonds;
	return fine_hb_categories;
}







////////////////////////////////////////////////////////////////////////////////
/// @begin evaluate_hbenergy
///
/// @brief
///
/// @detailed
///
/// @param  update_deriv - [in] -
/// @param  srhb_score - [out] -
/// @param  lrhb_score - [out] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// JSS ignores sc interactions
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
evaluate_hbenergy(  // used in minimizers
  hbonds::HBDerivType deriv_type,
	float & srhb_score,
	float & lrhb_score
)
{
	using namespace misc;
	using namespace param;

//car local
	FArray1D_int neighbors( MAX_RES()() );
	float scbbhb_score,schb_score;
	FArray2D_bool neighborlist( MAX_RES()(), MAX_RES()() );

//car if nblist, don't get hbond arrays, just reevaluate hbonds in list
//car hbonds evaluated from full_coord array cause they require HN coords
//car if fullatom not in use, fill out relevant portions of full_coord first
	if ( !get_fullatom_flag() )
	 initialize_fullcoord_array(Eposition,full_coord,total_residue,res,res_variant);

	make_neighbor_info(res,total_residue,full_coord,neighborlist,neighbors);

  bool const fullatom = ( deriv_type == hbonds::hbderiv_NONE ) ? false : get_fullatom_flag();
	fill_hbond_arrays(fullatom,res,res_variant,full_coord,total_residue,neighborlist,
	 neighbors,deriv_type,srhb_score,lrhb_score,
	 scbbhb_score,schb_score);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin fill_hbond_arrays
///
/// @brief
///
/// @detailed
///
/// @param  aan - [in] -
/// @param  aav - [in] - aa variant
/// @param  xyz - [in] -
/// @param  nres - [in] - residue of interest
/// @param  neighborlist - [in/out]? -
/// @param  neighbors - [in] -
/// @param  fullatom - [in] -
/// @param  update_deriv - [in] -
/// @param  sr_bb_hbenergy - [in/out] - short range bb-bb total
/// @param  lr_bb_hbenergy - [in/out] - long range bb-bb total
/// @param  scbb_hbenergy - [in/out] - sc-bb total
/// @param  sc_hbenergy - [in/out] - sc-sc total
/// @param  hb_atom_energy - [in/out] - convenient array with atom-by-atom info.
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
fill_hbond_arrays(
	bool const fullatom,
  FArray1DB_int const & aan,
	FArray1DB_int const & aav, // aa variant
	FArray3DB_float const & xyz,
	int const nres,
	FArray2DB_bool const & neighborlist,
	FArray1DB_int const & neighbors,
  hbonds::HBDerivType const deriv_type,
	float & sr_bb_hbenergy, // short range bb-bb total
	float & lr_bb_hbenergy, // long range bb-bb total
	float & scbb_hbenergy, // sc-bb total
	float & sc_hbenergy, // sc-sc total
	FArray2D_float & hb_atom_energy
)
{
	using namespace fullatom_energies;
	using namespace hbonds;
	using namespace param;
	using namespace param_pack;

  hbond_set.reset(nres);  // clean out the hydrogen bond set

//car local
	int aa1,aa2,aav1,aav2;
	int dres,dhatm,aatm,ares;
	int res_hbonds;
	int nb1,nb2;
	float sr_bbE,lr_bbE,sc_scE,sc_bbE;
	float hbE;
	bool d_is_sc,a_is_sc; // jss to remove

	int hb_atm_count,hb_per_atm_count;
	int hb_acc_id,hb_don_id;

	int const xyz_size12 = xyz.size1() * xyz.size2();
	int const neighborlist_size1 = neighborlist.size1();

//car find hbonds
	sr_bb_hbenergy = 0.0;
	lr_bb_hbenergy = 0.0;
	sc_hbenergy = 0.0;
	scbb_hbenergy = 0.0;

	for ( int res1 = 1; res1 <= nres; ++res1 ) {
		aa1 = aan(res1);
		aav1 = aav(res1);
		nb1 = neighbors(res1);
		const float & xyz_1( xyz(1,1,res1) );
		FArray1Da_bool hbchk_1( hbond_set.hbchkPair(res1) );
		for ( int res2 = res1+1,
		 ln = neighborlist.index(res1,res2), lxyz = xyz.index(1,1,res2);
		 res2 <= nres; ++res2, ln+=neighborlist_size1, lxyz+=xyz_size12 ) {

//bk pair_moved indicates which residues have changed relative position,
//bk and therefore need new energies calculated.  This is tricky though, because
//bk hbonds are not truly pairwise additive because sidechain-backbone hbonds
//bk are not allowed in cases where backbone-backbone hbond is already present.
//bk So for right now all hbonds will be calculated to be on the safe side.
//            if ( pair_moved(res1,res2) ) {

			if ( neighborlist[ ln ] ) { // neighborlist(res1,res2)

//bk get_hbE is being used to output a list of the
//bk hydrogen bonds between two residues
//car must throw summed energies away, because want to ensure that
//car all bb_hbonds are accounted for before
				aa2 = aan(res2);
				aav2 = aav(res2);
				nb2 = neighbors(res2);
				int const nhbondsp = hbond_set.nhbonds() + 1;

				get_hbE(fullatom,aa1,aa2,aav1,aav2,res1,res2,nb1,nb2,xyz_1,xyz[lxyz], // [lxyz]==(1,1,res2)
				 sc_scE,sr_bbE,lr_bbE,sc_bbE,res_hbonds,
				 hbond_set.hbenergies(nhbondsp), hbond_set.hbdonh_atm(nhbondsp),
         hbond_set.hbdon_res(nhbondsp),hbond_set.hbact_atm(nhbondsp),
				 hbond_set.hbact_res(nhbondsp),hbond_set.ref_hbtype(nhbondsp),hbchk_1,
         hbond_set.hbchkPair(res2),
				 deriv_type,hbond_set.hbderiv(1,1,nhbondsp));
//car increment totals
				hbond_set.addto_nhbonds(res_hbonds);
			}               // if neighbors
//     }   // if positions change with frag insertion
		}
	}

//ctsa setup hb inverse maps
	hbond_set.global_atm_to_hbpair_id = 0;
	hbond_set.hbpair_id_to_hb_id = 0;

	// handle symmetry if necessary
	hbond_set.symmetrize_hbchk();

//car throw out conflicting bb hbonds?

//car check sc_bb and sc_sc hbonds for inconsistency with bb_bb hbonds
//car now that hbchk array is completely updated
//car sum weighted energies for return
	hb_atm_count = 0;
	for ( int i = 1; i <= hbond_set.nhbonds(); ++i ) {
		dres = hbond_set.hbdon_res(i);
		ares = hbond_set.hbact_res(i);
		dhatm = hbond_set.hbdonh_atm(i);
		aatm = hbond_set.hbact_atm(i);

		////////////////////////////////////////////////////////////////////////////
		//// build inverse hb maps
		////
		if ( hbond_set.global_atm_to_hbpair_id(aatm,ares) == 0 ) {
			++hb_atm_count;
			if ( hb_atm_count > MAX_HB_ATM ) {
				std::cout << "STOP:: acc MAX_HB_ATM exceeded" << std::endl;
#ifdef BOINC
				return;
#endif
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			hbond_set.global_atm_to_hbpair_id(aatm,ares) = hb_atm_count;
		}

		if ( hbond_set.global_atm_to_hbpair_id(dhatm,dres) == 0 ) {
			++hb_atm_count;
			if ( hb_atm_count > MAX_HB_ATM ) {
				std::cout << "STOP:: don MAX_HB_ATM exceeded" << std::endl;
#ifdef BOINC
				return;
#endif
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			hbond_set.global_atm_to_hbpair_id(dhatm,dres) = hb_atm_count;
		}

		hb_acc_id = hbond_set.global_atm_to_hbpair_id(aatm,ares);
		hb_per_atm_count = 1;
		int l = hbond_set.hbpair_id_to_hb_id.index(hb_per_atm_count,hb_acc_id) - 1;
		while ( hbond_set.hbpair_id_to_hb_id[ ++l ] != 0 ) {
			++hb_per_atm_count;
			if ( hb_per_atm_count > MAX_HB_PER_ATM ) {
				std::cout << "STOP:: MAX_HB_PER_ATM exceeded" << std::endl;
#ifdef BOINC
				return;
#endif
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
		hbond_set.hbpair_id_to_hb_id[ l ] = -i;

		hb_don_id = hbond_set.global_atm_to_hbpair_id(dhatm,dres);
		hb_per_atm_count = 1;
		l = hbond_set.hbpair_id_to_hb_id.index(hb_per_atm_count,hb_don_id) - 1;
		while ( hbond_set.hbpair_id_to_hb_id[ ++l ] != 0 ) {
			++hb_per_atm_count;
			if ( hb_per_atm_count > MAX_HB_PER_ATM ) {
				std::cout << "STOP:: MAX_HB_PER_ATM exceeded" << std::endl;
#ifdef BOINC
				return;
#endif
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}
		hbond_set.hbpair_id_to_hb_id[ l ] = i;


		hbE = hbond_set.hbenergies(i);
		int const hbe_type = hbond_set.hbtype(i);
		d_is_sc = !atom_is_backbone(dhatm,aan(dres),aav(dres)); //jss to remove
		a_is_sc = !atom_is_backbone(aatm,aan(ares),aav(ares));

		// Here we have energies, now we have to decide which to throw out.

		hbond_set.allow_hbond(i); //cmd set tracking array used to report hydrogen bond status in decoystats

		if ( allow_hbond(dres,!d_is_sc,ares,!a_is_sc) ) { // jss to remove

			if ( output_hb_stats ) show_hb_stats( aan, aav, xyz, i ); // JSS testing

			switch (hbe_type) {
        case hbe_BBTURN:
        case hbe_BBHELIX:
					hbE *= pack_wts.Whbond_bb();
					sr_bb_hbenergy += hbE;
          break;
        case hbe_BBOTHER:
					hbE *= pack_wts.Whbond_bb();
					lr_bb_hbenergy += hbE;
          break;
        case hbe_SP2B:
        case hbe_SP3B:
        case hbe_RINGB:
        case hbe_BSC:
					hbE *= pack_wts.Whbond_bb_sc();
					scbb_hbenergy += hbE;
          break;
        case hbe_SP2SC:
        case hbe_SP3SC:
        case hbe_RINGSC:
					hbE *= pack_wts.Whbond_sc();
					sc_hbenergy += hbE;
         break;
        default:
          std::cout<<"Warning: energy from unexpected HB type ignored "<< hbe_type<<std::endl;
          break;
      }

      if (false&&runlevel_ns::benchmark) { // jss debug
        std::cout<<"FHA type:"<< I(2,hbe_type) << " whbE:"<<F(9,4,hbE)<<" sums sc:"<<F(9,4,sc_hbenergy)<<" scbb:"<<F(9,4,scbb_hbenergy)<<" srbb:"<<F(9,4,sr_bb_hbenergy)<<" lrbb:"<<F(9,4,lr_bb_hbenergy)<<
        std::endl;
      }

			if ( fullatom ) {
//car sum weighted energies for residue totals for packer,fullatom_energy
//car note that these are packer weights!! not used for scorefxn or deriv
				hbond_pair(dres,ares) += hbE;
				hbond_pair(ares,dres) += hbE;
				hbE *= 0.5f;
				hbenergy(dres) += hbE;
				hbenergy(ares) += hbE;

				hb_atom_energy( aatm, ares ) += hbE;
				hb_atom_energy( aaproperties_pack::atom_base(dhatm,aan(dres),aav(dres)), dres ) += hbE;

				// pb -- store per-residue energies by type for symmetric scoring
			switch (hbe_type) {
        case hbe_BBTURN:
        case hbe_BBHELIX:
					hbenergy_sr_bb(dres) += hbE;
					hbenergy_sr_bb(ares) += hbE;
          break;
        case hbe_BBOTHER:
					hbenergy_lr_bb(dres) += hbE;
					hbenergy_lr_bb(ares) += hbE;
          break;
        case hbe_SP2B:
        case hbe_SP3B:
        case hbe_RINGB:
        case hbe_BSC:
					hbenergy_sc_bb(dres) += hbE;
					hbenergy_sc_bb(ares) += hbE;
          break;
        case hbe_SP2SC:
        case hbe_SP3SC:
        case hbe_RINGSC:
					hbenergy_sc_sc(dres) += hbE;
					hbenergy_sc_sc(ares) += hbE;
         break;
        default:
          std::cout<<"Warning: energy from unexpected HB type ignored "<< hbe_type<<std::endl;
          break;
      }

#ifdef GL_GRAPHICS
				gl_graphics_store_bond(
					dres, aaproperties_pack::atom_base(dhatm,aan(dres),aav(dres)), ares,
					aatm, 2*hbE, graphics::HBOND );
#endif
			} // if ( fullatom )
		} else {
			hbond_set.forbid_hbond(i);
		}
	}

}

void
fill_hbond_arrays(
	bool const fullatom,
  FArray1DB_int const & aan,
	FArray1DB_int const & aav, // aa variant
	FArray3DB_float const & xyz,
	int const nres,
	FArray2DB_bool const & neighborlist,
	FArray1DB_int const & neighbors,
  hbonds::HBDerivType const deriv_type,
	float & sr_bb_hbenergy, // short range bb-bb total
	float & lr_bb_hbenergy, // long range bb-bb total
	float & scbb_hbenergy, // sc-bb total
	float & sc_hbenergy // sc-sc total
)
{
	FArray2D_float hb_atom_energy( param::MAX_ATOM(), nres, 0.0 );
	fill_hbond_arrays(	fullatom, aan, aav, xyz, nres, neighborlist, neighbors, deriv_type,
											sr_bb_hbenergy, lr_bb_hbenergy, scbb_hbenergy, sc_hbenergy,
											hb_atom_energy
											);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin initialize_hbonds
///
/// @brief
///
/// @detailed
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
inline
void
initialize_hbonds()
{
	using namespace param;
	using namespace pdbstatistics_pack;

	static bool hb_init = { false };
	if ( hb_init ) return; //already initialized


	hb_init = true;

}

////////////////////////////////////////////////////////////////////////////////
/// @begin get_hbE
///
/// @brief
///bk calculate residue-residue hydrogen bonding energy
///ctsa  added sub-function to calc all donor to accp hb from aaX to aaY
///
/// @detailed
///
/// @param  aa1 - [in/out]? -
/// @param  aa2 - [in/out]? -
/// @param  aav1 - [in/out]? -
/// @param  aav2 - [in/out]? -
/// @param  res1 - [in/out]? -
/// @param  res2 - [in/out]? -
/// @param  nb1 - [in/out]? -
/// @param  nb2 - [in/out]? -
/// @param  xyz1 - [in/out]? -
/// @param  xyz2 - [in/out]? -
/// @param  scenergy - [in/out]? -
/// @param  sr_bbenergy - [in/out]? -
/// @param  lr_bbenergy - [in/out]? -
/// @param  sc_bbenergy - [in/out]? -
/// @param  nhbonds_l - [in/out]? -
/// @param  hbenergies_l - [in/out]? -
/// @param  hbdonh_atm_l - [in/out]? -
/// @param  hbdon_res_l - [in/out]? -
/// @param  hbact_atm_l - [in/out]? -
/// @param  hbact_res_l - [in/out]? -
/// @param  hbtype_l - [in/out]? -
/// @param  hbchk1_l - [in/out]? -
/// @param  hbchk2_l - [in/out]? -
/// @param  fullatom - [in/out]? -
/// @param  update_deriv - [in/out]? -
/// @param  hbderiv_l - [in/out]? - f1,f2 deriv components
///
/// @global_read
///
/// @global_write
///
/// @remarks get_hbE is called with unused dummy arguments for hbenergies_l through
/// hbchk2_l in a number of places.  For efficiency it may be worth adding
/// variants of get_hbE and get_aa_1way_hbE_B without those arguments.
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_hbE(
  bool const fullatom,
	int const aa1,
	int const aa2,
	int const aav1,
	int const aav2,
	int const res1,
	int const res2,
	int const nb1,
	int const nb2,
	FArray2Da_float xyz1,
	FArray2Da_float xyz2,
	float & scenergy,
	float & sr_bbenergy,
	float & lr_bbenergy,
	float & sc_bbenergy,
	int & nhbonds_l,
	FArray1Da_float hbenergies_l,
	FArray1Da_int hbdonh_atm_l,
	FArray1Da_int hbdon_res_l,
	FArray1Da_int hbact_atm_l,
	FArray1Da_int hbact_res_l,
	FArray1Da_int hbtype_l,
	FArray1Da_bool hbchk1_l,
	FArray1Da_bool hbchk2_l,
  HBDerivType const deriv_type,
  FArray3Da_float hbderiv_l // f1,f2 deriv components
)
{
	float sc_bbenergy_1_to_2;
	float sc_bbenergy_2_to_1;
	get_hbE( fullatom, aa1, aa2, aav1, aav2, res1, res2, nb1, nb2, xyz1, xyz2,
		scenergy, sr_bbenergy, lr_bbenergy, sc_bbenergy, sc_bbenergy_1_to_2, sc_bbenergy_2_to_1,
		nhbonds_l, hbenergies_l, hbdonh_atm_l, hbdon_res_l, hbact_atm_l, hbact_res_l, hbtype_l,
		hbchk1_l, hbchk2_l, deriv_type, hbderiv_l);
}

void
get_hbE(
  bool const fullatom,
	int const aa1,
	int const aa2,
	int const aav1,
	int const aav2,
	int const res1,
	int const res2,
	int const nb1,
	int const nb2,
	FArray2Da_float xyz1,
	FArray2Da_float xyz2,
	float & scenergy,
	float & sr_bbenergy,
	float & lr_bbenergy,
	float & sc_bbenergy,
	float & sc_bbenergy_1_to_2,
	float & sc_bbenergy_2_to_1,
	int & nhbonds_l,
	FArray1Da_float hbenergies_l,
	FArray1Da_int hbdonh_atm_l,
	FArray1Da_int hbdon_res_l,
	FArray1Da_int hbact_atm_l,
	FArray1Da_int hbact_res_l,
	FArray1Da_int hbtype_l,
	FArray1Da_bool hbchk1_l,
	FArray1Da_bool hbchk2_l,
  HBDerivType const deriv_type,
  FArray3Da_float hbderiv_l // f1,f2 deriv components
)
{
	using namespace hbonds;
	using namespace param;
	using namespace param_aa;

//------------------------------------------------------------------------------

	sr_bbenergy = 0.0;
	lr_bbenergy = 0.0;
	scenergy = 0.0;
	sc_bbenergy = 0.0;
	sc_bbenergy_1_to_2 = 0;
	sc_bbenergy_2_to_1 = 0;
	nhbonds_l = 0;

	if ( res1 == res2 ) return;

	if ( is_NA( aa1 ) && is_NA( aa2 ) && !rna_scoring::rna_fullatom) return;

	int atomindex = 2;
	if ( is_RNA( aa1 ) && is_RNA( aa2 ) ) atomindex = 6; //C4*

	xyz1.dimension( 3, MAX_ATOM() );
	xyz2.dimension( 3, MAX_ATOM() );

//bk crude distance cutoff
	int l = xyz1.index( 1, atomindex ); // Assumes xyz1 and xyz2 have same dim's
	float const d_12 = xyz1[ l ] - xyz2[ l ]; ++l; // xyz1(1,2) - xyz2(1,2)
	float const d_22 = xyz1[ l ] - xyz2[ l ]; ++l; // xyz1(2,2) - xyz2(2,2)
	float const d_32 = xyz1[ l ] - xyz2[ l ]; // xyz1(3,2) - xyz2(3,2)
	float const dis2 = ( d_12 * d_12 ) + ( d_22 * d_22 ) + ( d_32 * d_32 );

	if ( dis2 > 484 ) return; // 22 * 22 //CAR --- CHANGE THIS!!

 //Objexx: 1D array dimensioning skipped for speed
 //  hbenergies_l.dimension( MAX_HB_PER_ATM );
 //  hbdonh_atm_l.dimension( MAX_HB_PER_ATM );
 //  hbdon_res_l.dimension( MAX_HB_PER_ATM );
 //  hbact_atm_l.dimension( MAX_HB_PER_ATM );
 //  hbact_res_l.dimension( MAX_HB_PER_ATM );
 //  hbtype_l.dimension( MAX_HB_PER_ATM );
 //  hbchk1_l.dimension( 2 );
 //  hbchk2_l.dimension( 2 );
	if (deriv_type != hbderiv_NONE) hbderiv_l.dimension( 3, 2, MAX_HB_PER_ATM );

	bool bb_hb_exists;
	float sc1_bb2_scdon = 0;
	float sc1_bb2_scacc = 0;
	float sc2_bb1_scdon = 0;
	float sc2_bb1_scacc = 0;

	get_aa_1way_hbE_B(fullatom, aa1,aa2,aav1,aav2,res1,res2,nb1,nb2,xyz1,xyz2,scenergy,
	 sr_bbenergy,lr_bbenergy,sc_bbenergy,sc1_bb2_scdon, sc2_bb1_scacc,nhbonds_l,hbenergies_l,hbdonh_atm_l,
	 hbdon_res_l,hbact_atm_l,hbact_res_l,hbtype_l,bb_hb_exists, deriv_type,hbderiv_l);
	if ( bb_hb_exists ) {
		hbchk1_l(1) = true; // res 1 is donor in bb hbond
		hbchk2_l(2) = true;
	}

	get_aa_1way_hbE_B(fullatom, aa2,aa1,aav2,aav1,res2,res1,nb2,nb1,xyz2,xyz1,scenergy,
	 sr_bbenergy,lr_bbenergy,sc_bbenergy,sc2_bb1_scdon, sc1_bb2_scacc,nhbonds_l,hbenergies_l,hbdonh_atm_l,
	 hbdon_res_l,hbact_atm_l,hbact_res_l,hbtype_l,bb_hb_exists, deriv_type,hbderiv_l);
	if ( bb_hb_exists ) {
		hbchk1_l(2) = true; // res 1 is acceptor in bb hbond
		hbchk2_l(1) = true;
	}
	sc_bbenergy_1_to_2 = sc1_bb2_scdon + sc1_bb2_scacc;
	sc_bbenergy_2_to_1 = sc2_bb1_scdon + sc2_bb1_scacc;

	//sc_bbenergy = sc_bbenergy_1_to_2 + sc_bbenergy_2_to_1;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_aa_1way_hbE_B
///
///bk calculate weighted hbond energies of donor atoms to accpt atoms
///
/// @detailed
///bk calculate weighted hbond energies of donor atoms to accpt atoms
///
///car i/o   new hbonds added to these variables (not initialized)
///car note these are local variables that may or may not be those in
///car hbonds.h
///car cannot use hbond.h variables directly cause rotamer trials wants
///car to use this function to compute energyies without updating arrays
///car must include hbonds.h for the defined HB_TYPES
///
/// @param  fullatom - [in] - calculate sc hbonds
/// @param  don_aa - [in] - amino acids
/// @param  act_aa - [in] - amino acids
/// @param  don_aav - [in] -
/// @param  act_aav - [in] -
/// @param  don_res - [in] - residue positions
/// @param  act_res - [in] - residue positions
/// @param  don_nb - [in] - number of neighbors
/// @param  act_nb - [in] - number of neighbors
/// @param  don_xyz - [in] - coordinates
/// @param  act_xyz - [in] -
/// @param  scE - [in/out] - running total of energies computed
/// @param  sr_bbE - [in/out] -
/// @param  lr_bbE - [in/out] -
/// @param  sc_bbE - [in/out] -
/// @param  nhbonds_l - [in/out] - running count of number of hydrogen bonds detected between residues
/// @param  hbenergies_l - [in/out] - table of energies of each hydrogen bond
/// @param  hbdonh_atm_l - [in/out] - table of donor protons
/// @param  hbdon_res_l - [in/out] - table of donor residues
/// @param  hbact_atm_l - [in/out] - table of acceptor atoms
/// @param  hbact_res_l - [in/out] - table of acceptor residue
/// @param  hbtype_l - [in/out] - table of HBEvalTypes
/// @param  bb_hb_exists - [out] -
/// @param  deriv_type - [in; optional] - type of derivative to compute (NONE, ABE_GO, DOCK_DON_ACC, DOCK_ACC_DON)
/// @param  hbderiv_l - [out; optional] - derivative components
///
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
get_aa_1way_hbE_B(
  bool const fullatom,
	int const don_aa,
	int const act_aa,
	int const don_aav,
	int const act_aav,
	int const don_res,
	int const act_res,
	int const don_nb,
	int const act_nb,
	FArray2DB_float & don_xyz,
	FArray2DB_float & act_xyz,
	float & scE,
	float & sr_bbE,
	float & lr_bbE,
	float & sc_bbE,
	float & sc1_bb2_scdon,
	float & sc2_bb1_scacc,
	int & nhbonds_l,
	FArray1DB_float & hbenergies_l,
	FArray1DB_int & hbdonh_atm_l,
	FArray1DB_int & hbdon_res_l,
	FArray1DB_int & hbact_atm_l,
	FArray1DB_int & hbact_res_l,
	FArray1DB_int & hbtype_l,
	bool & bb_hb_exists,
  HBDerivType const deriv_type,
	FArray3DB_float & hbderiv_l
)
{
	using namespace aaproperties_pack;
	using namespace design;
	using namespace hbonds;
	using namespace param;
	using namespace param_aa;
	using namespace pdbstatistics_pack;
	using namespace enable_ligaa_ns;

//bk local
	static FArray2D_float deriv( 3, 2 );
	int hatm,datm,aatm,base;
	int base2;
	bool datm_is_sc,aatm_is_sc; // jss to remove
	float energy,weight;
	double energy_weight;

	bb_hb_exists = false;

//bk cycle through polar hydrogens in don_aa, acceptors in act_aa
	for ( int hnum = 1, lHp = Hpos_polar.index(hnum,don_aa,don_aav),
        anume = nacceptors(act_aa,act_aav),
        l_aatms = accpt_pos.index(1,act_aa,act_aav),
        l0 = atom_base.index(1,act_aa,act_aav)-1,
        hnume = nH_polar(don_aa,don_aav); hnum <= hnume; ++hnum , ++lHp ) {
		hatm = Hpos_polar[ lHp ]; // Hpos_polar(hnum,don_aa,don_aav)
		datm = atom_base(hatm,don_aa,don_aav); // Hbase
		datm_is_sc = !atom_is_backbone(datm,don_aa,don_aav); // jss to remove
		if ( datm_is_sc && !fullatom ) goto L100;
		// JSS this next line should go.  Too many magic numbers.
		if ( don_aa == 2 && hatm == 11 && disulfides::BOUNDARY::cys_res_in_disulf(don_res) ) goto L100;

		{ // Scope
      float const & don_xyz_1d( don_xyz(1,datm) );
      float const & don_xyz_1h( don_xyz(1,hatm) );
      for ( int anum = 1, l_aatm = l_aatms; anum <= anume; ++anum, ++l_aatm ) {
        aatm = accpt_pos[ l_aatm ]; // accpt_pos(anum,act_aa,act_aav)
        HBEvalType const hbe_type = hbond_evaluation_type(datm,don_aa,don_aav, aatm,act_aa,act_aav, don_res, act_res);
        if (!hbe_is_BB_type(hbe_type) && !fullatom) goto L200;
				{ // Scope
        int const l = l0 + aatm; // Arrays indexed by l have same dimensions
        base = atom_base[ l ]; // abase(aatm,act_aa,act_aav)
        base2 = abase2[ l ]; // [l]==(aatm,act_aa,act_aav)
        assert(base2 > 0 && base != base2); // JSS every acceptor should have a base2 now.
//car calculate energy of next hbond
        if (deriv_type == hbderiv_NONE)
          hb_energy_deriv(hbe_type, don_xyz_1d,don_xyz_1h,
                          act_xyz(1,aatm),act_xyz(1,base),act_xyz(1,base2),
                          energy);
        else
          hb_energy_deriv(hbe_type, don_xyz_1d,don_xyz_1h,
                          act_xyz(1,aatm),act_xyz(1,base),act_xyz(1,base2),
                          energy,deriv_type,deriv);
				}
        if ( energy >= MAX_HB_ENERGY ) goto L200; // no hbond
//tk make hydrogen bonds environment dependent: exposed - intermediate - buried
//lin  make hydrogen bonds environment dependent: scaled by the neighbor number
//lin  more continous environment weight, rather than environment classes
//lin  only calculate the side chain H-bond
			if ( hbe_is_SC_type(hbe_type) && use_W_hb_env_dep_tk ) {
				if ( dna_interface ) {
					weight = adjust_dna_base_weight( don_aa, datm, don_nb, act_aa, aatm, act_nb );
				} else if ( smooth_env_dep_hb ) {
					weight = hb_env_dep_burial_lin(act_nb,don_nb);
				} else {
					weight = hb_env_dep_burial_tk(act_nb,don_nb);
				}
			} else {
				weight = 1.0; // default constant weight
			}

			//rhiju allow user to turn some acceptors/donors off.
			{
				static const bool disallow_hbond_o4star = truefalseoption("disallow_hbond_o4star");
				if (disallow_hbond_o4star && is_RNA(don_aa) && datm==rna_variables::o2star) goto L200;
				static const bool disallow_hbond_ho2star = truefalseoption("disallow_hbond_ho2star");
				if (disallow_hbond_ho2star && is_RNA(act_aa) && aatm==rna_variables::o4star) goto L200;
			}


			//lin hijack the Wact Wdon to have the ligaa atomic weight
			if( get_enable_ligaa_flag() && enzyme::read_occ_weights) {
        if (is_ligand(act_aa) ) weight *= lig_iocc_weight(aatm,has_ligand_no(act_aa));
        if (is_ligand(don_aa) ) weight *= lig_iocc_weight(datm,has_ligand_no(don_aa));
			}

			if ( check_hbenergy ) {
        std::cout << "donor" << I(4, don_res ) << I(3, datm ) << I(3, hatm )
        << "  acceptor" << I(4, act_res ) << I(3, aatm ) << I(3, base ) <<
        SS( use_W_hb_env_dep_tk ) << "act_nb" << I(3, act_nb ) <<
        "don_nb" << I(3, don_nb ) << " weight" << F(12,6, weight ) << std::endl;
				std::cout << " energy" << F(14,6,energy) << "hbe_type " << SS(hbe_type) << " b2 " << SS(base2) <<
        std::endl;
			}

//car increment hbond counters
			++nhbonds_l;
			if ( nhbonds_l > MAX_HB_PER_ATM ) {
				std::cout << "STOP:: ERROR: nhbonds_l exceeds MAX_HB_PER_ATM: " <<
				 MAX_HB_PER_ATM << std::endl;
#ifdef BOINC
				return;
#endif
				if(get_enable_ligaa_flag()) return; // should it be checked in ?
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			energy_weight = energy * weight;
			hbenergies_l(nhbonds_l) = energy_weight;
			hbdonh_atm_l(nhbonds_l) = hatm;
			hbdon_res_l(nhbonds_l) = don_res;
			hbact_atm_l(nhbonds_l) = aatm;
			hbact_res_l(nhbonds_l) = act_res;
			hbtype_l(nhbonds_l) = hbe_type;

			if (deriv_type != hbderiv_NONE) {
				int lh = hbderiv_l.index(1,1,nhbonds_l);
				hbderiv_l[   lh ] = deriv[ 0 ] * weight;
				hbderiv_l[ ++lh ] = deriv[ 1 ] * weight;
				hbderiv_l[ ++lh ] = deriv[ 2 ] * weight;
				hbderiv_l[ ++lh ] = deriv[ 3 ] * weight;
				hbderiv_l[ ++lh ] = deriv[ 4 ] * weight;
				hbderiv_l[ ++lh ] = deriv[ 5 ] * weight;
        if (false&&runlevel_ns::benchmark && fullatom) { // jss debug
          std::cout << "weight " << weight << " deriv " << deriv[0] <<","<< deriv[1] <<","<< deriv[2] <<","<< deriv[3] <<","<< deriv[4] <<","<< deriv[5] <<
          std::endl;
          std::cout << "type " << hbe_type << " hbderiv_l " <<  hbderiv_l[0] <<","<<  hbderiv_l[1] <<","<<  hbderiv_l[2] <<","<<  hbderiv_l[3] <<","<<  hbderiv_l[4] <<","<<  hbderiv_l[5] <<
          std::endl;
        }

			}

			aatm_is_sc = !atom_is_backbone(aatm,act_aa,act_aav); //jss to remove
			//lin fill the ligaa hbE array
			if( cst_set_ns::debug_output ) {
				fill_ligaa_hbEarray( aatm, act_aa, aatm_is_sc,
														datm, don_aa, datm_is_sc, energy_weight);
			}

//car increment energies
//bk don't allow sidechain backbone hydrogen bonds withs backbone atoms that
//bk are already hydrogen bonded
//car note that the check for existing bb hbonds uses the hbchk array (via
//car allow_hbond. This array is updated by fill_hbond_arrays. Thus these
//car sc energy sums are valid only if the hbchk array is current.
//car summed sc energies here are used by pack and rotamer_trials
//car they are not used by fill_hbond_arrays which must do its own summing
//car once the hbchk array is updated

			if ( !allow_hbond(don_res,!datm_is_sc,act_res,!aatm_is_sc) ) // jss to remove
        goto L200;
			switch (hbe_type) {
        case hbe_BBTURN:
        case hbe_BBHELIX:
          sr_bbE += energy_weight;
          bb_hb_exists = true;
          break;
        case hbe_BBOTHER:
          lr_bbE += energy_weight;
          bb_hb_exists = true;
          break;
        case hbe_SP2B:
        case hbe_SP3B:
        case hbe_RINGB:
          sc2_bb1_scacc += energy_weight;
          sc_bbE += energy_weight;
          break;
        case hbe_BSC:
          sc1_bb2_scdon += energy_weight;
          sc_bbE += energy_weight;
          break;
        case hbe_SP2SC:
        case hbe_SP3SC:
        case hbe_RINGSC:
          scE += energy_weight;
         break;
        default:
          std::cout << "Warning: energy from unexpected HB type ignored "<< hbe_type<<std::endl;
          break;
      }
      if (false&&runlevel_ns::benchmark) { // jss debug
        std::cout <<
        "("<<I(5,don_res)<<","<<I(3,don_aa)<<","<<I(3,hatm)<<"/"<<I(3,datm)<<")-("<<I(5,act_res)<<","<<I(3,act_aa)<<","<<I(3,aatm)<<") " <<
        " sr:" << F(9,4,sr_bbE)<<" lr:" << F(9,4,lr_bbE)<<" scB:" << F(9,4,sc_bbE)<<" sc:" << F(9,4,scE)<<
        std::endl;
      }
L200:; // escape if no sidechains & acpt is sc atom or no hbond
		}
		} // Scope
L100:; // escape if no sidechains and donor is sc atom
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin get_aa_1way_hbE
///
/// @brief
///bk calculate weighted hbond energies of donor atoms to accpt atoms
///
/// @detailed
///car i/o   new hbonds added to these variables (not initialized)
///car note these are local variables that may or may not be those in
///car hbonds.h
///car cannot use hbond.h variables directly cause rotamer trials wants
///car to use this function to compute energyies without updating arrays
///car must include hbonds.h for the defined HB_TYPES
///
///SGM Wrapper version using argument FArrays
///
/// @param  fullatom - [in] - calculate sc hbonds
/// @param  don_aa - [in] - amino acids
/// @param  act_aa - [in] - amino acids
/// @param  don_aav - [in] -
/// @param  act_aav - [in] -
/// @param  don_res - [in] - residue positions
/// @param  act_res - [in] - residue positions
/// @param  don_nb - [in] - number of neighbors
/// @param  act_nb - [in] - number of neighbors
/// @param  don_xyz - [in] - coordinates
/// @param  act_xyz - [in] -
/// @param  scE - [in/out] - running total of energies computed
/// @param  sr_bbE - [in/out] -
/// @param  lr_bbE - [in/out] -
/// @param  sc_bbE - [in/out] -
/// @param  nhbonds_l - [in/out] - running count of number of hydrogen bonds detected between residues
/// @param  hbenergies_l - [in/out] - table of energies of each hydrogen bond
/// @param  hbdonh_atm_l - [in/out] - table of donor protons
/// @param  hbdon_res_l - [in/out] - table of donor residues
/// @param  hbact_atm_l - [in/out] - table of acceptor atoms
/// @param  hbact_res_l - [in/out] - table of acceptor residue
/// @param  hbtype_l - [in/out] - table of HBEvalTypes
/// @param  bb_hb_exists - [out] -
/// @param  deriv_type - [in; optional] - type of derivative to compute (NONE, ABE_GO, DOCK_DON_ACC, DOCK_ACC_DON)
/// @param  hbderiv_l - [out; optional] - derivative components
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified 2006/11/30
/////////////////////////////////////////////////////////////////////////////////
void
get_aa_1way_hbE(
  bool const fullatom,
	int const don_aa,
	int const act_aa,
	int const don_aav,
	int const act_aav,
	int const don_res,
	int const act_res,
	int const don_nb,
	int const act_nb,
	FArray2Da_float don_xyz, //const, but for dimensoon call
	FArray2Da_float act_xyz,
	float & scE,
	float & sr_bbE,
	float & lr_bbE,
	float & sc_bbE,
	int & nhbonds_l,
	FArray1Da_float hbenergies_l,
	FArray1Da_int hbdonh_atm_l,
	FArray1Da_int hbdon_res_l,
	FArray1Da_int hbact_atm_l,
	FArray1Da_int hbact_res_l,
	FArray1Da_int hbtype_l,
	bool & bb_hb_exists,
  hbonds::HBDerivType const deriv_type,
	FArray3Da_float hbderiv_l
)
{
	using namespace hbonds;
	using namespace param;

	don_xyz.dimension( 3, MAX_ATOM() );
	act_xyz.dimension( 3, MAX_ATOM() );
 	hbenergies_l.dimension( MAX_HB_PER_ATM );
 	hbdonh_atm_l.dimension( MAX_HB_PER_ATM );
 	hbdon_res_l.dimension( MAX_HB_PER_ATM );
 	hbact_atm_l.dimension( MAX_HB_PER_ATM );
 	hbact_res_l.dimension( MAX_HB_PER_ATM );
 	hbtype_l.dimension( MAX_HB_PER_ATM );
 	hbderiv_l.dimension( 3, 2, MAX_HB_PER_ATM );

	float sc1_bb2_scdon( 0.0 ), sc2_bb1_scacc( 0.0 ); // Locals in this version

	// Call the version that does the work
	get_aa_1way_hbE_B(
    fullatom,
		don_aa,
		act_aa,
		don_aav,
		act_aav,
		don_res,
		act_res,
		don_nb,
		act_nb,
		don_xyz,
		act_xyz,
		scE,
		sr_bbE,
		lr_bbE,
		sc_bbE,
		sc1_bb2_scdon,
		sc2_bb1_scacc,
		nhbonds_l,
		hbenergies_l,
		hbdonh_atm_l,
		hbdon_res_l,
		hbact_atm_l,
		hbact_res_l,
		hbtype_l,
		bb_hb_exists,
		deriv_type,
		hbderiv_l
	);

}

////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_compute_deriv
///
/// @brief
///
/// @detailed
///
/// @param  first_res - [in] -
/// @param  last_res - [in] -
/// @param  F1_hbondE - [in/out]? -
/// @param  F2_hbondE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
hbond_compute_deriv(
	int const first_res,
	int const last_res,
	FArray3Da_float F1_hbondE,
	FArray3Da_float F2_hbondE
)
{
	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace minimize_ns;
	using namespace misc;
	using namespace nblist;
	using namespace param;
	using namespace param_pack;
	using namespace param_torsion;

	F1_hbondE.dimension( 3, total_torsion, MAX_RES() );
	F2_hbondE.dimension( 3, total_torsion, MAX_RES() );

//car using the incremental changes in f1 and f2 for each torsion angle,
//car recursively compute the cumulative hbond contributions to F1tot and
//car F2tot for evaluation of the derivative in dfunc_vdw

// tmp torsion order hack
	FArray1D_int torsion_order_map( total_torsion );

//car local
	int seg_begin, seg_end;
	float weight;
	int torsion;

	int first_atom,last_atom;
	int aa,aav;

	int hb_count,hbpair_id,hbond_id;
//------------------------------------------------------------------------------
  pack_wts.set_lock();
  setHBE_Weight();

	//// get torsion order
	////
	torsion_order_map(1) = phi_torsion;
	for ( int k = 1; k <= total_chi_torsion; ++k ) {
		torsion_order_map(1+k) = total_bb_torsion+k;
	}
	torsion_order_map(2+total_chi_torsion) = psi_torsion;
	torsion_order_map(3+total_chi_torsion) = omega_torsion;

	F1_hbondE = 0.0;
	F2_hbondE = 0.0;

	first_atom = total_atoms+1; // zeroth rigid body begins after chain end

	//// get joint atom pair potential derivatives over all torsion angles between
	////   the first and last residues in the angle_map
	////

	int seg_num = identify_segment(last_res);
	retrieve_segment_ends(seg_num,seg_begin,seg_end);
	if ( seg_end != total_residue )
	 first_atom = res_atm_num2global_atm_num_map(1,seg_end+1);

	for ( int seqpos = last_res; seqpos >= first_res; --seqpos ) {
		aa = res(seqpos);
		aav = res_variant(seqpos);
		for ( int tor = total_torsion; tor >= 1; --tor ) {
			torsion = torsion_order_map(tor);

			//// these should never be calculated
			////
			if ( seqpos == 1 && torsion == phi_torsion ) goto L105;
			if ( seqpos == seg_end && torsion == omega_torsion ) goto L105;

			////  special cases for chi angles
			////
			if ( torsion > total_bb_torsion + nchi(aa,aav) ) goto L105;
			if ( torsion > total_bb_torsion && ! minimize_vary_chi ) goto L105;

			////
			get_rigid_body_region(seqpos,torsion,first_atom,last_atom);

			//// for each atm in torsion rigid body
			////
			for ( int atm = first_atom; atm <= last_atom; ++atm ) {
			 // atoms between current and last torsion

				// check for bb hydrogens included for constraints only;
				if ( global_atm_num2res_atm_num_map(atm) <= 0 ) goto L110;

				hbpair_id = hbond_set.global_atm_to_hbpair_id(global_atm_num2res_atm_num_map(atm),
				 global_atm_num2res_num_map(atm));
				if ( hbpair_id == 0 ) goto L110;

				hb_count = 1;
				hbond_id = hbond_set.hbpair_id_to_hb_id(hb_count,hbpair_id);

				//// for each hbond per atm
				////
				while ( hbond_id != 0 ) {
					if ( hbond_id < 0 ) {   //// atm is acceptor
						hbond_id = -hbond_id;
						weight = -HBE_Weight[hbond_set.hbtype(hbond_id)];
					} else {
            weight = HBE_Weight[hbond_set.hbtype(hbond_id)]; //// atm is donor
          }

					if (weight == 0.0 || !allow_hbond_by_idx(hbond_id) ) goto L120;

					for ( int k = 1; k <= 3; ++k ) {
						F1_hbondE(k,torsion,seqpos) += weight * hbond_set.hbderiv(k,1,hbond_id);
						F2_hbondE(k,torsion,seqpos) += weight * hbond_set.hbderiv(k,2,hbond_id);
					}
          if (false&&runlevel_ns::benchmark) { // jss debug
            std::cout<<"hcd "<<F(10,4,weight)<<"hbs.deriv"<<
            F(10,4,hbond_set.hbderiv(1,1,hbond_id))<<F(10,4,hbond_set.hbderiv(2,1,hbond_id))<<F(10,4,hbond_set.hbderiv(3,1,hbond_id))<<
            F(10,4,hbond_set.hbderiv(1,2,hbond_id))<<F(10,4,hbond_set.hbderiv(2,2,hbond_id))<<F(10,4,hbond_set.hbderiv(3,2,hbond_id))<<
            std::endl;
            std::cout<<"F12 "<<F(10,4,F1_hbondE(1,torsion,seqpos))<<F(10,4,F1_hbondE(2,torsion,seqpos))<<F(10,4,F1_hbondE(3,torsion,seqpos))<<F(10,4,F2_hbondE(1,torsion,seqpos))<<F(10,4,F2_hbondE(2,torsion,seqpos))<<F(10,4,F2_hbondE(3,torsion,seqpos))<<
            std::endl;
          }


L120:; // skip this hbond

					++hb_count;
					if ( hb_count > MAX_HB_PER_ATM ) goto L110;
					hbond_id = hbond_set.hbpair_id_to_hb_id(hb_count,hbpair_id);
				}
L110:; // skip this atom
			}
L105:; // skip this torsion angle

			link_torsion_vectors(seqpos,torsion,F1_hbondE,F2_hbondE);
		}
	}

  pack_wts.release_lock();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin allow_hbond_by_idx
///
/// @brief
///
/// @detailed
///
/// @param  idx - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
bool
allow_hbond_by_idx( int const idx )
{
	using namespace hbonds;
	using namespace misc;

	int const dres = hbond_set.hbdon_res(idx);
	int const ares = hbond_set.hbact_res(idx);
	int const dhatm = hbond_set.hbdonh_atm(idx);
	int const aatm = hbond_set.hbact_atm(idx);
	int const daa = res(dres);
	int const aaa = res(ares);
	int const dvar = res_variant(dres);
	int const avar = res_variant(ares);
	bool const dhatm_is_bb = atom_is_backbone(dhatm,daa,dvar);
	bool const aatm_is_bb = atom_is_backbone(aatm,aaa,avar);

	return allow_hbond(dres,dhatm_is_bb,ares,aatm_is_bb);
}

////////////////////////////////////////////////////////////////////////////////
/// @begin allow_hbond
///
/// @brief
///
/// @detailed
///
/// @param  dres - [in/out]? -
/// @param  dhatm_is_bb - [in/out]? -
/// @param  ares - [in/out]? -
/// @param  aatm_is_bb - [in/out]? -
///
/// @return
///
/// @global_read
///
/// @global_write
///
/// @remarks
///   jss Used to check sequence separation for any bond involving a backbone atom but
///    Tanja and Brian no longer think this is justified.
/// @references
///
/// @authors
///
/// @last_modified
///  Jack Snoeyink, 23 Jan 2007
/////////////////////////////////////////////////////////////////////////////////
bool
allow_hbond(
	int const dres,
	bool const dhatm_is_bb,
	int const ares,
	bool const aatm_is_bb
)
{
	using namespace hbonds;

	if (dres == ares) return false; // require different residues

	if ( dhatm_is_bb ) {
		if ( aatm_is_bb ) return true; // bb-bb : always allowed;
		return !hbond_set.BBhbToNH_exists(dres); // bb-sc:  depends on donor bb hbond
	}

	if ( aatm_is_bb ) {          // sc-bb
		return !hbond_set.BBhbToO_exists(ares); // depends on acceptor bb hbond
	}

	return true; // sc-sc  : always allowed;
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_neighbors
///
/// @brief
///
/// @detailed
///
/// @param  nres - [in/out]? -
/// @param  aan - [in/out]? -
/// @param  xyz - [in/out]? -
/// @param  hb_nbors - [in/out]? -
/// @param  hcent - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
void
hbond_neighbors( // only used in pdbstats.cc to place water
	int nres,
	FArray1Da_int aan,
	FArray3Da_float xyz,
	FArray2Da_int hb_nbors,
	FArray3Da_float hcent
)
{
	using namespace param;
	using namespace param_aa;

	aan.dimension( MAX_RES() );
	xyz.dimension( 3, MAX_ATOM(), MAX_RES() );
	hb_nbors.dimension( 2, MAX_RES() );
	hcent.dimension( 3, 2, MAX_RES() );

	FArray1D_float xyz1( 3 );
	FArray1D_float xyz2( 3 );
	float norm1,norm2,norm3,dis;

	for ( int i = 1; i <= nres; ++i ) {
		hb_nbors(1,i) = 0;
		hb_nbors(2,i) = 0;
	}

//bk first determine the coordinates of the optimal putative acceptor or
//bk donor
	for ( int res = 1; res <= nres; ++res ) {
//bk place putative NH acceptor
		if ( res != 1 ) {
			norm1 = 0.0;
			norm2 = 0.0;
			norm3 = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				xyz1(i) = xyz(i,1,res) - xyz(i,2,res);
				norm1 += xyz1(i) * xyz1(i);
				xyz2(i) = xyz(i,1,res) - xyz(i,3,res-1);
				norm2 += xyz2(i) * xyz2(i);
			}
			norm1 = 1/std::sqrt(norm1);
			norm2 = 1/std::sqrt(norm2);

			for ( int i = 1; i <= 3; ++i ) {
				float const hcent_i = xyz1(i)*norm1 + xyz2(i)*norm2;
				 // vector bisecting o-n-c
				hcent(i,1,res) = hcent_i;
				norm3 += hcent_i * hcent_i;
			}
			norm3 = 2.8/std::sqrt(norm3); //  set length to 2.8 angstoms

			for ( int i = 1; i <= 3; ++i ) {
				hcent(i,1,res) = norm3*hcent(i,1,res) + xyz(i,1,res);
				 // offset from nitrogen
			}
		}

//bk place putative donor for carbonyl oxygen
		if ( res != nres ) {
			norm1 = 0.0;
			for ( int i = 1; i <= 3; ++i ) {
				float const hcent_i = xyz(i,4,res) - xyz(i,3,res);
				hcent(i,2,res) = hcent_i;
				norm1 += hcent_i * hcent_i;
			}
			norm1 = 2.8/std::sqrt(norm1);

			for ( int i = 1; i <= 3; ++i ) {
				hcent(i,2,res) = norm1*hcent(i,2,res) + xyz(i,4,res);
			}
		}
	}

//bk find the neighbors for each hcent
	for ( int i = 1; i <= nres; ++i ) {
		if ( i != 1 ) {
			for ( int j = 1; j <= nres; ++j ) {
				if ( aan(j) == aa_gly ) {
					distance_bk(hcent(1,1,i),xyz(1,2,j),dis);
				} else {
					distance_bk(hcent(1,1,i),xyz(1,5,j),dis);
				}
				if ( dis < 7.0 ) {
					++hb_nbors(1,i);
				}
			}
		}
		if ( i != nres ) {
			for ( int j = 1; j <= nres; ++j ) {
				if ( aan(j) == aa_gly ) {
					distance_bk(hcent(1,2,i),xyz(1,2,j),dis);
				} else {
					distance_bk(hcent(1,2,i),xyz(1,5,j),dis);
				}
				if ( dis < 7.0 ) {
					++hb_nbors(2,i);
				}
			}
		}
	}

	for ( int i = 1; i <= nres; ++i ) {
		if ( hb_nbors(1,i) == 0 ) hb_nbors(1,i) = 1;
		if ( hb_nbors(2,i) == 0 ) hb_nbors(2,i) = 1;
		if ( hb_nbors(1,i) > 20 ) hb_nbors(1,i) = 20;
		if ( hb_nbors(2,i) > 20 ) hb_nbors(2,i) = 20;
	}

}

///////////////////////////////////////////////////////////////////////////////
void
fast_backbone_hbE(
	pose_ns::Pose & pose,
	float & hb_srbb_score,
	float & hb_lrbb_score
)
{
	using namespace hbonds;
	using namespace pose_ns;

	int const MIN_SEQ_SEP( 1 ); // this should agree with allow_hbond // jss to remove
	assert( pose.fullatom() ); // temporary, ensure full_coord OK

	FArray1D_int const & domain_map( pose.get_domain_map() );
	FArray3D_float const & full_coord( pose.full_coord() );

	Score_state score_state;
	FArray2D_float & hb_cache( pose.set_2D_score( BBHB_CACHE, score_state ));

	bool trust_cached_data( score_state == GOOD || score_state == OK );

	float short_range_total(0.0), long_range_total(0.0);
	int const nres( pose.total_residue() );
	//int cache_count(0), score_count(0);
	for ( int res1=1; res1<= nres; ++res1 ) {
		int const aa1 ( pose.res        (res1));
		int const aav1( pose.res_variant(res1));
		int const res1map( domain_map(res1));
		//std::cout << res1 << ' ' << res1map << std::endl;
		for ( int res2=res1; res2<= nres; ++res2 ) {
			if ( res2-res1 < MIN_SEQ_SEP ) {
				// sanity check:
				assert( !allow_hbond( res1, true, res2, true ) );
				continue;
			}
			bool const short_range
				( res2 - res1 <= SHORT_RANGE_CUTOFF ); // we know res2>= res1 here
			if ( res1map != 0 && trust_cached_data && domain_map(res2) == res1map ) {
				//////////////////////
				// recover cached data
				if ( short_range ) {
					short_range_total += hb_cache(res2,res1);
				} else {
					long_range_total  += hb_cache(res2,res1);
				}
				//++cache_count;

			} else {
				///////////////////////////
				// recalculate from scratch
				int const aa2 ( pose.res        (res2));
				int const aav2( pose.res_variant(res2));
				// there is no environment dependence! see initialize_hbonds()
				int const nb1(1), nb2(1);
				float sc_scE(0.0), sr_bbE(0.0), lr_bbE(0.0), sc_bbE(0.0);
				int res_hbonds(0);
				int nhbondsp(1);
				bool const fullatom( false );
				bool hbchk_1(2), hbchk_2(2);
				get_hbE(fullatom, aa1,aa2,aav1,aav2,res1,res2,nb1,nb2,
								full_coord(1,1,res1),full_coord(1,1,res2),
								sc_scE,sr_bbE,lr_bbE,sc_bbE,res_hbonds,hbond_set.hbenergies(nhbondsp),
								hbond_set.hbdonh_atm(nhbondsp),hbond_set.hbdon_res(nhbondsp),hbond_set.hbact_atm(nhbondsp),
								hbond_set.hbact_res(nhbondsp),hbond_set.ref_hbtype(nhbondsp),hbchk_1,hbchk_2,
								hbonds::hbderiv_NONE);
				assert(  short_range && std::abs(lr_bbE) < 1e-3 ||
								!short_range && std::abs(sr_bbE) < 1e-3 );
				hb_cache(res2,res1) = sr_bbE + lr_bbE;
				short_range_total += sr_bbE;
				long_range_total  += lr_bbE;
				assert( allow_hbond( res1, true, res2, true ));
				//++score_count;
			}
		}
	}
	hb_srbb_score = param_pack::pack_wts.Whbond_bb() * short_range_total;
	hb_lrbb_score = param_pack::pack_wts.Whbond_bb() *  long_range_total;
	//std::cout << cache_count << ' ' << score_count << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

void
find_CO_bb_hbond_partner(
	pose_ns::Pose & pose,
  int this_pos,
  int & ret_atm,
  int & ret_res
)
{

	using namespace hbonds;
	using namespace aaproperties_pack;
	int const CO_number = 4;
  assert(false); //JSS this should not be executed until I fix it up
	///jjh Set default to be 'no partner'
	ret_atm = ret_res = -1;

	short id( hbond_set.global_atm_to_hbpair_id( CO_number, this_pos ) );

//	std::cout << "id for atm, res:  " << id << " , " << CO_number << " , " << this_pos << std::endl;

	///jjh return if this atom participates in no hbonds
	if( id == 0 ) return;

	for( int ihb = 1, ehb = MAX_HB_PER_ATM ; ihb <= ehb ; ++ihb ) {

		short hb_id( -hbond_set.hbpair_id_to_hb_id( ihb, id ) );

//		std::cout << "hb_id to check:  " << hb_id << std::endl;

		///jjh Return when no more hbonds for this atm exist
		if( hb_id == 0 ) return;

		///jjh Skip to next hbond if not bb-bb
		if( hbond_set.hbtype( hb_id ) != SRBB_HBTYPE &&
				hbond_set.hbtype( hb_id ) != LRBB_HBTYPE ) continue;

		///jjh Here we return the atom number and residue of the
		///jjh first (there may be more!) bb hbond donor atom for
		///jjh this C=O acceptor.

		ret_res = hbond_set.hbdon_res(hb_id);

		int ret_h_atm = hbond_set.hbdonh_atm(hb_id);

		int other_aa( pose.res( ret_res ) );
		int other_aav( pose.res_variant( ret_res ) );

		ret_atm = atom_base( ret_h_atm, other_aa, other_aav );
		return;

	}

	return;
}

void
find_NH_bb_hbond_partner(
	pose_ns::Pose & pose,
  int this_pos,
  int & ret_atm,
  int & ret_res
)
{

	using namespace hbonds;
	using namespace aaproperties_pack;
	int const NH_number = HNpos( pose.res(this_pos), pose.res_variant(this_pos) );
  assert(false); //JSS this should not be executed until I fix it up

	///jjh Set default to be 'no partner'
	ret_atm = ret_res = -1;

	short id( hbond_set.global_atm_to_hbpair_id( NH_number, this_pos ) );

//	std::cout << "id for atm, res:  " << id << " , " << NH_number << " , " << this_pos << std::endl;

	///jjh return if this atom participates in no hbonds
	if( id == 0 ) return;

	for( int ihb = 1, ehb = MAX_HB_PER_ATM ; ihb <= ehb ; ++ihb ) {

		short hb_id( hbond_set.hbpair_id_to_hb_id( ihb, id ) );

//		std::cout << "hb_id to check:  " << hb_id << std::endl;

		///jjh Return when no more hbonds for this atm exist
		if( hb_id == 0 ) return;

		///jjh Skip to next hbond if not bb-bb
		if( hbond_set.hbtype( hb_id ) != SRBB_HBTYPE &&
				hbond_set.hbtype( hb_id ) != LRBB_HBTYPE ) continue;

		///jjh Here we return the atom number and residue of the
		///jjh first (there may be more!) bb hbond acceptor atom for
		///jjh this N-H acceptor.

		ret_res = hbond_set.hbact_res(hb_id);
		ret_atm = hbond_set.hbact_atm(hb_id);

		return;

	}

	return;
}


#include "docking_ns.h"
#include "docking_minimize_ns.h"

// Taken from docking_minimize, where it doesn't belong.  JSS

//====ora hbond deriv functions==================================
void
get_total_hb_deriv_docking(
													 FArray1Da_float dE_dTR,
													 int nfree,
													 FArray2Da_bool jneighborlist
													 )
{
	using namespace docking;
	using namespace hbonds;
	using namespace misc;
	using namespace param;
	using namespace param_pack;

	if ( !get_fullatom_flag() ) return;

	dE_dTR.dimension( nfree );
	jneighborlist.dimension( MAX_RES(), MAX_RES() );

	//ora hb deriv: based on get_total_pair_deriv, fill_hbond_arrays,get_hbE
	//ora
	//ora this function contains two parts:
	//ora  1) computation of hbderiv array
	//ora      in "hb_energy_deriv" called within the hbond functions
	//ora  2) summing up of derivs into dE_dTR
	//ora
	//ora: it is assumed that the following are updated:
	//ora             jneighborlist, neighbors
	//ora             hb_chk: this allows to go over the jneighborlist ONCE:
	//ora                 non-allowed hbs (i.e. sc-bb hbs to bb-bb hbonded atoms) are skipped
	//ora             fullatom and update_deriv are TRUE
	//ora: note that ALL neighbors between the partners are considered, not only those involving IF residues

	//     local
	int aa1,aa2,aav1,aav2,nb1,nb2;
	int nhbonds_l;

	int dres,dhatm,aatm,ares;
	bool dhatm_is_bb,aatm_is_bb;
	float weight;

	FArray1D_int jneighbors( MAX_RES()() );

	// the following variables are needed to call the function, but are not used anywhere
	float scenergy,sr_bbenergy,lr_bbenergy,sc_bbenergy;
	bool bb_hb_exists;

  pack_wts.set_lock();
	setHBE_Weight();

	docking_get_brians_neighborlist(jneighborlist);
	docking_get_brians_neighbors(jneighbors);

	//
	// 1) create array hbderiv
	//
	hbond_set.set_nhbonds(0);
	// loops over ALL neighbors between the 2 partners
	// (not only if residues contribute apparently, as the comp
	// to the numerically evaluated derivative shows)
	for ( int res1 = part_begin(1), res1e = part_end(1); res1 <= res1e; ++res1 ) {
		aa1 = res(res1);
		aav1 = res_variant(res1);
		nb1 = jneighbors(res1);
		for ( int res2 = part_begin(2), res2e = part_end(2); res2 <= res2e; ++res2 ) {
			if ( jneighborlist(res1,res2) ) {
				aa2 = res(res2);
				aav2 = res_variant(res2);
				nb2 = jneighbors(res2);
				//
				// part taken from get_hbE:
				// no "crude distance cutoff" needed here
				//
				sr_bbenergy = 0.;
				lr_bbenergy = 0.;
				scenergy = 0.;
				sc_bbenergy = 0.;
				nhbonds_l = 0;
				int const nhb1 = hbond_set.nhbonds()+1; // JSS temporary during refactoring.

				get_aa_1way_hbE(true, aa1,aa2,aav1,aav2,res1,res2,nb1,nb2,
												full_coord(1,1,res1),full_coord(1,1,res2),scenergy,sr_bbenergy,
												lr_bbenergy,sc_bbenergy,nhbonds_l,hbond_set.hbenergies(nhb1),
												hbond_set.hbdonh_atm(nhb1),hbond_set.hbdon_res(nhb1),hbond_set.hbact_atm(nhb1),
												hbond_set.hbact_res(nhb1),hbond_set.ref_hbtype(nhb1),bb_hb_exists,hbonds::hbderiv_DOCK_DON_ACC,
												hbond_set.hbderiv(1,1,nhb1) );
				get_aa_1way_hbE(true, aa2,aa1,aav2,aav1,res2,res1,nb2,nb1,
												full_coord(1,1,res2),full_coord(1,1,res1),scenergy,sr_bbenergy,
												lr_bbenergy,sc_bbenergy,nhbonds_l,hbond_set.hbenergies(nhb1),
												hbond_set.hbdonh_atm(nhb1),hbond_set.hbdon_res(nhb1),hbond_set.hbact_atm(nhb1),
												hbond_set.hbact_res(nhb1),hbond_set.ref_hbtype(nhb1),bb_hb_exists,hbonds::hbderiv_DOCK_ACC_DON,
												hbond_set.hbderiv(1,1,nhb1) );
				//
				// end of part of get_hbE
				//
				hbond_set.addto_nhbonds(nhbonds_l);
			}
		}
	}

	//
	// sum hbderiv into dE_dTR
	//
	for ( int nb = 1; nb <= hbond_set.nhbonds(); ++nb ) {
		dres = hbond_set.hbdon_res(nb);
		ares = hbond_set.hbact_res(nb);
		dhatm = hbond_set.hbdonh_atm(nb);
		aatm = hbond_set.hbact_atm(nb);
		dhatm_is_bb = atom_is_backbone(dhatm,res(dres),res_variant(dres));
		aatm_is_bb  = atom_is_backbone(aatm,res(ares),res_variant(ares));
		if ( !(allow_hbond(dres,dhatm_is_bb,ares,aatm_is_bb)) ) goto L200;

		weight = HBE_Weight[hbond_set.hbtype(nb)];

    for ( int j = 1, l = hbond_set.hbderiv.index(1,j,nb)-1; j <= 2; ++j ) {
			int const j3 = (j-1)*3;
			for ( int i = 1; i <= 3; ++i ) {
				dE_dTR(i+j3) += weight * hbond_set.hbderiv[ ++l ]; // hbderiv(i,j,nb);
			}
		}

    if (false&&runlevel_ns::benchmark) { // jss debug
      std::cout<<"gthdd "<<F(10,4,weight)<<"hbs.deriv"<<
      F(10,4,hbond_set.hbderiv(1,1,nb))<<F(10,4,hbond_set.hbderiv(2,1,nb))<<F(10,4,hbond_set.hbderiv(3,1,nb))<<
      F(10,4,hbond_set.hbderiv(1,2,nb))<<F(10,4,hbond_set.hbderiv(2,2,nb))<<F(10,4,hbond_set.hbderiv(3,2,nb))<<
      std::endl;
      std::cout<<"dE_dTR "<<F(10,4,dE_dTR(1))<<F(10,4,dE_dTR(2))<<F(10,4,dE_dTR(3))<<F(10,4,dE_dTR(4))<<F(10,4,dE_dTR(5))<<F(10,4,dE_dTR(6))<<
      std::endl;
    }

L200:;
	}
  pack_wts.release_lock();

}


// from atom_tree_minimize JSS
//////////////////////////////////////////////////////////////////////////////
/// @begin compute_atom_hbond_deriv
///
/// @brief
///car using the incremental changes in f1 and f2 for each torsion angle,
///car recursively compute the cumulative hbond contributions to F1tot and
///car F2tot for evaluation of the derivative in dfunc_vdw
///
/// @detailed
///
/// @param  first_res - [in/out]? -
/// @param  last_res - [in/out]? -
/// @param  srbb_weight - [in/out]? -
/// @param  lrbb_weight - [in/out]? -
/// @param  scbb_weight - [in/out]? -
/// @param  sc_weight - [in/out]? -
/// @param  F1_hbondE - [in/out]? -
/// @param  F2_hbondE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
///   Oct 2006  JSS: will remove this function, whose comment block has been copied from jmp
////////////////////////////////////////////////////////////////////////////////
void
compute_atom_hbond_deriv(
												 int const atomno,
												 int const rsd,
												 numeric::xyzVector_float & F1, // accumulate contributions
												 numeric::xyzVector_float & F2
												 )
{
	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace param;
	using namespace param_pack;
	using namespace param_torsion;

	//-----------------------------------------------------------------------------
	int const hbpair_id = hbond_set.global_atm_to_hbpair_id( atomno, rsd );
	if ( hbpair_id == 0 ) return; // no hbonds

  pack_wts.set_lock();
	setHBE_Weight();

	int hb_count = 1;
	int hbond_id = hbond_set.hbpair_id_to_hb_id( hb_count, hbpair_id );

	// for each hbond per atm
	//
	while ( hbond_id != 0 ) {
		float weight;
		if ( hbond_id < 0 ) { // atm is acceptor
			hbond_id = -hbond_id;
			weight = -HBE_Weight[hbond_set.hbtype(hbond_id)];
		} else {
      weight = HBE_Weight[hbond_set.hbtype(hbond_id)]; // atm is donor
    }

		if ( allow_hbond_by_idx( hbond_id ) && weight != 0.0 ) {
			for ( int k = 1; k <= 3; ++k ) {
				F1(k) += weight * hbond_set.hbderiv(k,1,hbond_id);
				F2(k) += weight * hbond_set.hbderiv(k,2,hbond_id);
			}

      if (false&&runlevel_ns::benchmark) { // jss debug
        std::cout<<"cahd "<<F(10,4,weight)<<"hbs.deriv"<<
        F(10,4,hbond_set.hbderiv(1,1,hbond_id))<<F(10,4,hbond_set.hbderiv(2,1,hbond_id))<<F(10,4,hbond_set.hbderiv(3,1,hbond_id))<<
        F(10,4,hbond_set.hbderiv(1,2,hbond_id))<<F(10,4,hbond_set.hbderiv(2,2,hbond_id))<<F(10,4,hbond_set.hbderiv(3,2,hbond_id))<<
        std::endl;
        std::cout<<"F12 "<<F(10,4,F1(1))<<F(10,4,F1(2))<<F(10,4,F1(3))<<F(10,4,F2(1))<<F(10,4,F2(2))<<F(10,4,F2(3))<<
        std::endl;
      }

		}

		++hb_count;
		if ( hb_count > MAX_HB_PER_ATM ) {
			std::cout << "WARNING: exceeded MAX_HB_PER_ATM on hbond" << SS( hbond_id ) << "hbtype" << SS( hbond_set.hbtype(hbond_id) ) << std::endl;
			break;
		}

		hbond_id = hbond_set.hbpair_id_to_hb_id(hb_count,hbpair_id);
	} // while ( hbond_id != 0 )

  pack_wts.release_lock();
}

// repatriated from jumping.minimize.cc
#include "jmp_directions.h"
#include "jmp_maps.h"
#include "jumping_diagnostics.h"
#include "jumping_ns.h"
#include "jumping_refold.h"
#include "jumping_minimize.h"

//////////////////////////////////////////////////////////////////////////////
/// @begin jmp_hbond_compute_deriv
///
/// @brief
///car using the incremental changes in f1 and f2 for each torsion angle,
///car recursively compute the cumulative hbond contributions to F1tot and
///car F2tot for evaluation of the derivative in dfunc_vdw
///
/// @detailed
///
/// @param  first_res - [in/out]? -
/// @param  last_res - [in/out]? -
/// @param  srbb_weight - [in/out]? -
/// @param  lrbb_weight - [in/out]? -
/// @param  scbb_weight - [in/out]? -
/// @param  sc_weight - [in/out]? -
/// @param  F1_hbondE - [in/out]? -
/// @param  F2_hbondE - [in/out]? -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
void
jmp_hbond_compute_deriv(
												//	int const first_res,
												//	int const last_res,
	pose_ns::Pose const & pose,
	float const srbb_weight,
	float const lrbb_weight,
	float const scbb_weight,
	float const sc_weight,
	FArray3DB_float & F1_hbondE,
	FArray3DB_float & F2_hbondE
)
{
	using namespace aaproperties_pack;
	using namespace hbonds;
	using namespace jmp_directions;
	using namespace jmp_maps;
	using namespace minimize_ns;
	//using namespace misc;
	using namespace nblist;
	using namespace param_pack;
  using namespace param_torsion;

	//F1_hbondE.dimension( 3, jmp_total_torsion, MAX_RES() );
	//F2_hbondE.dimension( 3, jmp_total_torsion, MAX_RES() );

//car local
	int dir,n_tor_atom;
	float weight;
	int torsion,seqpos;
	//int aa,aav;
	int hb_count,hbpair_id,hbond_id;
	int atm;

//------------------------------------------------------------------------------

  pack_wts.set_lock();
  setHBE_Weight(srbb_weight, lrbb_weight, scbb_weight, sc_weight);

	F1_hbondE = 0.0;
	F2_hbondE = 0.0;

	for ( int jmp_tor = 1; jmp_tor <= jmp_torsion_map_length; ++jmp_tor ) {
		seqpos  = jmp_torsion_map(1,jmp_tor);
		torsion = jmp_torsion_map(2,jmp_tor);
		dir = jmp_torsion_map(3,jmp_tor); // n2c...


		n_tor_atom = jmp_atomlist_length(torsion,seqpos);

		for ( int tor_atom = 1; tor_atom <= n_tor_atom; ++tor_atom ) {
			atm = jmp_atomlist(tor_atom, torsion, seqpos);

			int const atm_seqpos( global_atm_num2res_num_map(atm));

			if ( pose.symmetric() && atm_seqpos > pose.total_residue_for_scoring()){
				continue;
			}

			hbpair_id = hbond_set.global_atm_to_hbpair_id
				( global_atm_num2res_atm_num_map(atm), atm_seqpos );


			if ( hbpair_id == 0 ) goto L110;

			hb_count = 1;
			hbond_id = hbond_set.hbpair_id_to_hb_id(hb_count,hbpair_id);

			// for each hbond per atm
			//
			while ( hbond_id != 0 ) {
				if ( hbond_id < 0 ) { // atm is acceptor
					hbond_id = -hbond_id;
					weight = -HBE_Weight[hbond_set.hbtype(hbond_id)];
				} else {
          weight = HBE_Weight[hbond_set.hbtype(hbond_id)]; // atm is donor
        }

				if ( ! allow_hbond_by_idx(hbond_id) || weight == 0.0 ) goto L120;

        for ( int k = 1; k <= 3; ++k ) {
					F1_hbondE(k,torsion,seqpos) += weight * hbond_set.hbderiv(k,1,hbond_id);
					F2_hbondE(k,torsion,seqpos) += weight * hbond_set.hbderiv(k,2,hbond_id);
				}
        if (false&&runlevel_ns::benchmark) { // jss debug
          std::cout<<"jhcd "<<I(3,hbond_id)<<I(3,hbond_set.hbtype(hbond_id))<<F(10,4,weight)<<" e:"<<F(10,4,hbond_set.hbenergies(hbond_id))<<" hbs.deriv"<<
          F(10,4,hbond_set.hbderiv(1,1,hbond_id))<<F(10,4,hbond_set.hbderiv(2,1,hbond_id))<<F(10,4,hbond_set.hbderiv(3,1,hbond_id))<<
          F(10,4,hbond_set.hbderiv(1,2,hbond_id))<<F(10,4,hbond_set.hbderiv(2,2,hbond_id))<<F(10,4,hbond_set.hbderiv(3,2,hbond_id))<<
          std::endl;
          std::cout<<"F12 "<<F(10,4,F1_hbondE(1,torsion,seqpos))<<F(10,4,F1_hbondE(2,torsion,seqpos))<<F(10,4,F1_hbondE(3,torsion,seqpos))<<F(10,4,F2_hbondE(1,torsion,seqpos))<<F(10,4,F2_hbondE(2,torsion,seqpos))<<F(10,4,F2_hbondE(3,torsion,seqpos))<<
          std::endl;
        }

L120:; // skip this hbond

				++hb_count;
				if ( hb_count > MAX_HB_PER_ATM ) goto L110;
				hbond_id = hbond_set.hbpair_id_to_hb_id(hb_count,hbpair_id);
			} // while ( hbond_id != 0 )
L110:; // skip this atom
		} // for ( tor_atom = 1; tor_atom <= n_tor_atom; ++tor_atom )
		jmp_link_torsion_vectors( pose.fold_tree(), seqpos, torsion,
															F1_hbondE, F2_hbondE, dir );
	} // jmp_tor = 1, jmp_torsion_map_length
  pack_wts.release_lock();
}
