// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 13831 $
//  $Date: 2007-03-28 23:54:30 -0700 (Wed, 28 Mar 2007) $
//  $Author: csmith $

#ifndef INCLUDED_histogram
#define INCLUDED_histogram

// Utility Headers
#include <utility/vector0.hh>

// C++ Headers
#include <cmath>
#include <cstddef>
#include <iostream>

template <class T>
class histogram {

private:

	T start_;
	T end_;
	int total_;
	utility::vector0<int> counts_;

public:

////////////////////////////////////////////////////////////////////////////////
/// @begin histogram<T>::init
///
/// @brief
/// initialize a histogram object
///
/// @detailed
///
/// @param[in] start - where to start recording values
/// @param[in] end - where to stop recording values
/// @param[in] bins - number of bins in histogram
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	void
	init(
		T start,
		T end,
		int bins
	)	
	{
		start_ = start;
		end_ = end;
		total_ = 0;
		counts_.clear();
		counts_.resize(bins, 0);
	}
	
	histogram<T> &
	operator +=(histogram<T> const & hist) {
	
		if (counts_.size() == 0) init(hist.start(), hist.end(), hist.size());
		
		assert(counts_.size() == hist.size());
		
		for (size_t i = 0; i < hist.size(); i++) {
			total_ += hist.counts(i);
			counts_[i] += hist.counts(i);
		}
	
		return *this;
	}
	
	inline
	size_t
	size() const
	{
		return counts_.size();
	}
	
	inline
	T
	start() const
	{
		return start_;
	}
	
	inline
	T
	end() const
	{
		return end_;
	}
	
	inline
	utility::vector0<T>
	breaks() const
	{
		utility::vector0<T> breaks(counts_.size()+1);
		T const interval((end_ - start_)/counts_.size());
		
		for (size_t i = 0; i < breaks.size(); i++) {
			breaks[i] = interval * static_cast<T>(i) + start_;
		}
		
		return breaks;
	}
	
	inline
	int
	counts(
		size_t binnum
	) const
	{
		return counts_[binnum];
	}
	
	inline
	long double
	frequency(
		size_t binnum
	) const
	{
		return (long double)counts_[binnum]/(long double)total_;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin histogram<T>::record
///
/// @brief
/// record a number in the histogram
///
/// @detailed
///
/// @param[in] value - number to record
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	void
	record(
		T value
	)
	{
		int i;
	
		if (value == start_) {
			i = 0;
		} else {
			i = static_cast<int>( std::ceil((value - start_)/(end_ - start_)*counts_.size())) - 1;
		}
	
		if (i < 0 || i >= (signed)counts_.size()) {
			if (i == -1) {
				std::cout << start_ - value << std::endl;
			}
			if (i == (signed)counts_.size()) {
				std::cout << value - end_ << std::endl;
			}
	
			std::cout << "Error! " << value << " " << i << std::endl;
		}
	
	
		total_++;
		counts_[i]++;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin histogram<T>::print_counts
///
/// @brief
/// print the counts for each bin
///
/// @detailed
///
/// @param[in] out - output strem to print counts
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	void
	print_counts(
		std::ostream & out
	) const
	{
		for (int i = 0; i < (signed)counts_.size(); i++) {
			out << start_ + i*(end_ - start_)/counts_.size() << "\t" << counts_[i] << std::endl;
		}
		out << end_ << std::endl;
	}

////////////////////////////////////////////////////////////////////////////////
/// @begin histogram<T>::print_frequencies
///
/// @brief
/// print the counts for each bin
///
/// @detailed
///
/// @param[in] out - output strem to print frequencies
///
/// @global_read
///
/// @global_write
///
/// @return
///
/// @remarks
///
/// @refrences
///
/// @authors Colin A. Smith
///
/// @last_modified February 16, 2007
////////////////////////////////////////////////////////////////////////////////
	void
	print_frequencies(
		std::ostream & out
	) const
	{
		for (int i = 0; i < (signed)counts_.size(); i++) {
			out << start_ + i*(end_ - start_)/counts_.size() << "\t"
					<< (long double)counts_[i]/(long double)total_ << std::endl;
		}
		out << end_ << std::endl;
	}
};

#endif
