// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.2 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $

// Rosetta Headers
#include "kin_min.h"
#include "jumping_util.h"
#include "kin_util.h"
#include "pose.h"
#include "aaproperties_pack.h"
#include "atom_tree_minimize.h" // update_nblist! should just make method?

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
//#include <ObjexxFCL/FArray4D.h>
//#include <ObjexxFCL/formatted.io.h>

// Numeric Headers
#include <numeric/all.fwd.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>

// Utility Headers
#include <utility/basic_sys_util.hh>
#include <utility/io/orstream.hh>

// C++ Headers
#include <cstdlib>
#include <cstdio>


namespace kin {



	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// Torsion_node
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	void
	Torsion_node::get_rosetta_type(
		int & tor_no,
		int & tor_rsd
	) const
	{
		tor_no   = rosetta_torsion_type;
		tor_rsd  = rosetta_torsion_rsd;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Torsion_node::link_vectors()
	{
		if (parent ) {
			parent->F1() += F1_;
			parent->F2() += F2_;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Torsion_node::set_rosetta_torsion_type(
		int const torsion,
		int const rsd
		)
	{
		rosetta_torsion_type = torsion;
		rosetta_torsion_rsd = rsd;
	}


	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////
	// Minimizer_map
	/////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////

	// this will only work properly if the torsions are sorted in decreasing
	// depth
	//
	void
	Minimizer_map::link_torsion_vectors()
	{
		int depth( 100000 );
		for ( Torsion_iterator tor_it=torsions.begin(),
						tor_end=torsions.end(); tor_it != tor_end; ++tor_it ) {
			Torsion_node & tor( **tor_it );
			assert( tor.depth() <= depth );
			depth = tor.depth();
			tor.link_vectors();
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Minimizer_map::zero_torsion_vectors()
	{
		for ( Torsion_iterator tor=torsions.begin(),
						tor_end=torsions.end(); tor != tor_end; ++tor ) {
			(*tor)->F1() = 0.0;
			(*tor)->F2() = 0.0;
		}
	}

	/////////////////////////////////////////////////////////////////////////////
	Minimizer_map::Torsion_iterator
	Minimizer_map::torsions_begin()
	{
		return torsions.begin();
	}

	/////////////////////////////////////////////////////////////////////////////
	Minimizer_map::Torsion_iterator
	Minimizer_map::torsions_end()
	{
		return torsions.end();
	}


	/////////////////////////////////////////////////////////////////////////////
	Minimizer_map::Torsion_const_iterator
	Minimizer_map::torsions_begin() const
	{
		return torsions.begin();
	}

	/////////////////////////////////////////////////////////////////////////////
	Minimizer_map::Torsion_const_iterator
	Minimizer_map::torsions_end() const
	{
		return torsions.end();
	}


	/////////////////////////////////////////////////////////////////////////////
	// will all these map lookups kill us?????????????????????????

	void
	Minimizer_map::add_torsion(
		Torsion_id const & torsion_id,
		Torsion_id const & parent_id
	)
	{
		//std::cout << "add_torsion: " << torsion_id << ' ' << parent_id <<
		//	std::endl;

		// assign the parent
		Torsion_node * parent;
		if ( !(parent_id == BOGUS_TORSION )) {
			// not the root torsion
			std::map< Torsion_id, Torsion_node* >::iterator p
				( torsion_id_map.find( parent_id ) );
			if ( p== torsion_id_map.end() ) {
				std::cout << "parent torsion does not exist in map! torsion= " <<
					torsion_id << " parent= " << parent_id << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
			parent = p->second;
		} else {
			// root torsion!
			parent = 0;
		}

		Torsion_node* torsion( new Torsion_node( torsion_id, parent ) );
		torsion_id_map.insert( std::make_pair( torsion_id, torsion ) );
		torsions.push_back( torsion );

	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Minimizer_map::add_to_nblist(
		int const atomno1,
		int const rsd1,
		int const atomno2,
		int const rsd2
	)
	{
		Atom_node* atom1( atom_node_pointer( atomno1, rsd1 ) );
		Atom_node* atom2( atom_node_pointer( atomno2, rsd2 ) );


		if ( atom1 == 0 || atom2 == 0 ) {
			std::cout << "nblist atoms not in atom_node_pointer!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}

		// add symmetrically !!!!!!!!!!!!!!!
		atom1->add_nbr( atom2 );
		atom2->add_nbr( atom1 );
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Minimizer_map::add_atom(
		Atom_id const & atom_id,
		Torsion_id const & torsion_id
	)
	{
		// add the atom to our big list of all atoms
		Atom_node* atom( new Atom_node( atom_id ) );
		atoms.push_back( atom );
		atom_node_pointer( atom_id.atomno(), atom_id.rsd() ) = atom;

		// add atom to list of atoms first moved by this torsion,
		// unless we are in the root rigid body

		if ( ! (torsion_id == BOGUS_TORSION) ) {
			std::map< Torsion_id, Torsion_node* >::iterator p
				( torsion_id_map.find( torsion_id ) );
			if ( p == torsion_id_map.end() ) {
				std::cout << "torsion does not exist in map! torsion= " <<
					torsion_id << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}

			// add the atom to the torsion
			p->second->add_atom( atom );
		}


	}

	/////////////////////////////////////////////////////////////////////////////
	// fill in!
	void
	Minimizer_map::get_nbr_bounds(
		Atom_node* atom1,
		std::vector< Atom_node* >::const_iterator & atom2_begin,
		std::vector< Atom_node* >::const_iterator & atom2_end
	) const
	{
		if ( use_nblist_ ) {
			atom2_begin = atom1->nbr_list_begin();
			atom2_end = atom1->nbr_list_end();
		} else {
			atom2_begin = atoms.begin();
			atom2_end = atoms.end();
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	float
	Minimizer_map::torsion_scale_factor(
		Torsion_node const & tor
	) const
	{
		static float const rad2deg( numeric::conversions::degrees(1.0) );
		Kin_torsion_type const type( tor.type() );
		float factor( 1.0 );
		if ( type == PHI ) {
			// bond torsion
			factor = rad2deg;
		} else if ( type == THETA ) {
			// bond angle
			factor = rad2deg; // should be *10.0
		} else if ( type == D ) {
			// bond length
			factor = 100.0;
		} else if ( type == RB4 || type == RB5 || type == RB6 ) {
			// the jump_rb_delta's are stored in degrees!!!
			factor = 1.0;
		} else if ( type == RB1 || type == RB2 || type == RB3 ) {
			// rigid body translation
			factor = 10.0;
		}
		return factor;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Minimizer_map::reset()
	{
		// delete old memory
		for ( Torsion_iterator it=torsions.begin(), it_end = torsions.end();
					it != it_end; ++it ) {
			delete (*it);
		}
		for ( std::vector< Atom_node* >::iterator it = atoms.begin(),
						it_end=atoms.end(); it != it_end; ++it ) {
			delete (*it);
		}

		torsions.clear();
		atoms.clear();
		torsion_id_map.clear();
		atom_node_pointer = 0;
		use_nblist_ = false;
	}

	/////////////////////////////////////////////////////////////////////////////
	void
	Minimizer_map::pack_phipsi(
		pose_ns::Pose const & pose,
		FArray1DB_float & phipsi
	) const
	{
		int imap = 1;
		for ( Torsion_const_iterator it=torsions.begin(), it_end = torsions.end();
					it != it_end; ++it, ++imap ) {
			Torsion_node const & tor( **it );
			phipsi( imap ) = torsion_scale_factor( tor ) *
				pose.get_atom_tree_torsion( tor.atomno(), tor.rsd(), tor.type() );
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	void
	Minimizer_map::unpack_phipsi(
		pose_ns::Pose & pose,
		FArray1DB_float const & phipsi
	) const
	{
		int imap = 1;
		for ( Torsion_const_iterator it=torsions.begin(), it_end = torsions.end();
					it != it_end; ++it, ++imap ) {
			Torsion_node const & tor( **it );
			pose.set_atom_tree_torsion( tor.atomno(), tor.rsd(), tor.type(),
																	phipsi( imap ) / torsion_scale_factor(tor));
		}
	}


	/////////////////////////////////////////////////////////////////////////////
	bool
	Torsion_node_sorter( Torsion_node* const & a,		 Torsion_node* const & b ){
		return *a < *b;
	}

	/////////////////////////////////////////////////////////////////////////////
	// update nblist as well as setting up torsion lists

	void
	Minimizer_map::setup(
		pose_ns::Pose & pose,
		bool const use_nblist_in
	)
	{
		//std::cout << "min_map setup: use_nblist= " << use_nblist_in << std::endl;

		// this clears all the lists
		reset();

		// internal value
		use_nblist_ = use_nblist_in;

		// this fills the torsion and atom lists
		// and also the phipsi_list
		atom_node_pointer.dimension( param::MAX_ATOM()(), pose.total_residue() );
		Torsion_id tmp( BOGUS_TORSION );
		pose.atom_tree()->root()->setup_min_map( tmp, *this );

		// sort torsions for proper linking later on

		torsions.sort( Torsion_node_sorter );
		//std::cout << "Min_map::setup: total_torsions= " << torsions.size() <<
		//	std::endl;

		// update nblist if desired
		if ( use_nblist_ ) {
			this->update_nblist( pose );
		}

		// identify phi/psi/omega...
		this->assign_rosetta_torsions( pose );
	}


	/////////////////////////////////////////////////////////////////////////////
	// private
	void
	Minimizer_map::assign_rosetta_torsions(
		pose_ns::Pose const & pose
	)
	{
		using namespace param_torsion;
		using namespace param_aa;
		using aaproperties_pack::nchi;

		// fill in a torsion pointer array
		int const nres( pose.total_residue() );
		FArray2D< Atom_id > torsion_pointer( total_torsion, nres, BOGUS_ATOM );
		for ( int i=1; i<= nres; ++i ) {
			int const aa( pose.res(i) );
			int const aav( pose.res_variant(i) );
			int const max_bb_tor( is_protein( aa ) ? 3 : ( is_DNA( aa ) ? 6 : 0 ));
			int const max_chi( is_protein( aa ) ? nchi(aa,aav):
												 ( is_DNA( aa ) ? 1 : 0 ));
			for ( int j=1; j<= max_bb_tor+max_chi; ++j ) {
				Atom const * atom
					( pose.atom_tree()->get_rosetta_torsion_atom( j,i ) );
				if ( atom ) {
					torsion_pointer(j,i) = atom->atom_id;
				}
			}
		}



		for ( Torsion_iterator tor_it = torsions_begin(),
						tor_it_end = torsions_end(); tor_it != tor_it_end;
					++tor_it ) {
			Torsion_node & tor( **tor_it );

			int rosetta_torsion_type( 0 );
			int rosetta_torsion_rsd( 0 );

			int const rsd( tor.rsd() );
			Kin_torsion_type const type( tor.type() );

			if ( type == PHI && is_protein( pose.res(rsd) ) ) {

				for ( int i=rsd-1; i<= rsd+1; ++i ) {
					if ( i < 1 || i> nres || !is_protein(pose.res(i)) ) continue;
					int const aa ( pose.res(i) );
					int const aav( pose.res_variant(i) );
					int const j_end( i == rsd ? 3 + nchi(aa,aav) : 3 );
					for ( int j=1; j<= j_end; ++j ) {
						if ( torsion_pointer(j,i) == tor.atom_id() ) {
							assert( rosetta_torsion_type == 0 );
							rosetta_torsion_type = j;
							rosetta_torsion_rsd = i;
							break;
						}
					}
				}
				if ( !rosetta_torsion_rsd ) {
// 					std::cout << "WARNING:: unidentified protein PHI torsion in " <<
// 						"minimizer " << tor.atom_id() << std::endl;
				}
			}
			tor.set_rosetta_torsion_type( rosetta_torsion_type, rosetta_torsion_rsd);

		} // tor
// 		FArray2D_int torsion_atom( 4, total_torsion );
// 		FArray2D_int torsion_rsd_offset( 4, total_torsion );

// 		for ( Torsion_iterator tor_it = torsions_begin(),
// 						tor_it_end = torsions_end(); tor_it != tor_it_end;
// 					++tor_it ) {
// 			Torsion_node & tor( **tor_it );

// 			int const rsd( tor.rsd() );
// 			int const atomno( tor.atomno() );
// 			int const aa( pose.res( rsd ) );
// 			int const aav( pose.res_variant( rsd ) );
// 			Kin_torsion_type const type( tor.type() );
// 			Atom const * atom( pose.get_atom_tree_atom( atomno, rsd ) );

// 			int rosetta_torsion_type( 0 );
// 			int rosetta_torsion_rsd( 0 );
// 			int rosetta_torsion_sign( 1 );

// 			setup_torsion_atom_aa_aav( aa, aav, torsion_atom, torsion_rsd_offset );

// 			if ( atom->is_jump() ) {
// 				rosetta_torsion_type = first_rb_torsion +kin::get_rb_number( type ) -1;
// 				rosetta_torsion_rsd = rsd; // should be jump number??
// 				rosetta_torsion_sign = 1;
// 			} else if ( type == PHI && is_protein( aa ) ) {
// 				Atom_id const & atom1( atom->input_stub_atom1() );
// 				Atom_id const & atom2( atom->input_stub_atom2() );
// 				Atom_id const & atom3( atom->input_stub_atom3() );
// 				FArray1D_int atoms(4), rsds(4);
// 				atoms(1) = atomno;
// 				atoms(2) = atom1.atomno();
// 				atoms(3) = atom2.atomno();
// 				atoms(4) = atom3.atomno();
// 				rsds(1) = rsd;
// 				rsds(2) = atom1.rsd();
// 				rsds(3) = atom2.rsd();
// 				rsds(4) = atom3.rsd();

// 				for ( int ntor=1; ntor<= 3 + nchi( aa, aav ); ++ntor ) {
// 					for ( int pos=rsd-1; pos<=rsd+1; ++pos ) {
// 						for ( int dir=1; dir<= 2; ++dir ) {
// 							bool match(true);

// 							for ( int k=1 ; k<= 4; ++k ) {
// 								int const atoms_index( dir == 1 ? k : 5 - k );
// 								int const rsd_offset( rsds( atoms_index ) - pos );
// 								if ( atoms( atoms_index ) != torsion_atom( k, ntor ) ||
// 										 rsd_offset           != torsion_rsd_offset ( k, ntor ) ) {
// 									match = false;
// 									break;
// 								}
// 							}
// 							if ( match ) {
// 								assert( rosetta_torsion_type == 0 );
// 								rosetta_torsion_type = ntor;
// 								rosetta_torsion_rsd = pos;
// 								rosetta_torsion_sign = ( dir == 1 ? 1 : -1 );
// 							}
// 						}
// 					}
// 				}
// 			}
// 			tor.set_rosetta_torsion_type( rosetta_torsion_type, rosetta_torsion_rsd,
// 																		 rosetta_torsion_sign );
// 		} // tor

	}
}
