// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 1.1.2.2 $
//  $Date: 2005/11/07 21:05:35 $
//  $Author: pbradley $

#include "kin_util.h"
#include "aaproperties_pack.h"
#include "dna_ns.h"
#include "files_paths.h"
#include "kin_jump_atom.h"
#include "kin_bonded_atom.h"
#include "input_pdb.h"
#include "jumping_util.h"
#include "make_pdb.h"
#include "misc.h"
#include "namespace_fullatom_flag.h"
#include "param_aa.h"
#include "param.h"
#include "pose.h"
#include "read_aaproperties.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray3D.hh>
#include <ObjexxFCL/FArray4D.hh>
#include <ObjexxFCL/string.functions.hh>
#include <ObjexxFCL/formatted.o.hh>

// Numeric Headers
#include <numeric/conversions.hh>

// Utility Headers
#include <utility/io/orstream.hh>
#include <utility/io/ozstream.hh>
#include <utility/io/izstream.hh>

// C++ Headers
#include <cstdlib>
#include <iostream>
#include <fstream>

///////////////////////////////////////////////////////////////////////////////
namespace kin_atom_properties {
	FArray3D_bool is_backbone_atom( param::MAX_ATOM(), param::MAX_AA(),
																	param::MAX_AA_VARIANTS());
	FArray3D_bool is_chi_atom( param::MAX_ATOM(), param::MAX_AA(),
														 param::MAX_AA_VARIANTS() );
	FArray1D_int default_root_atomno( param::MAX_AA(), 1 );
	FArray1D_int default_anchor_atomno( param::MAX_AA(), 1 );
}


///////////////////////////////////////////////////////////////////////////////
int
chainbreak_phi_atomno(
	int const aa,
	int const aav
)
{
	int const PRO_CD_index( 7 ); // MAGIC
	int const atom_index( aa == param_aa::aa_pro ? PRO_CD_index :
												aaproperties_pack::HNpos(aa,aav) );

	assert( aa != param_aa::aa_pro ||
					aaproperties_pack::atom_name( PRO_CD_index,aa,aav) == " CD " );
	return atom_index;
}


///////////////////////////////////////////////////////////////////////////////
void
set_default_root_atomno(
												int const aa,
												int const atomno
												)
{
	using namespace kin_atom_properties;
	default_root_atomno(aa) = atomno;
	if ( default_anchor_atomno(aa) == 0 ) {
		// silly: checked by build_aa even if not used
		default_anchor_atomno(aa) = atomno;
	}
}

///////////////////////////////////////////////////////////////////////////////
void
set_default_anchor_atomno(
													int const aa,
													int const atomno
													)
{
	kin_atom_properties::default_anchor_atomno(aa) = atomno;
}

///////////////////////////////////////////////////////////////////////////////


inline
bool
cut_bond(
	int const atomno1,
	int const atomno2,
	int aa,
	int //aav
)
{
	using namespace param_aa;
	// 3/27/06 -- cutproline ring between cg and cd instead of n-cd
	// this makes more sense since there is only one chi
	// and it gives an atom to define phi at chainbreaks(!)
	//
	// but this may lead to strange drift -- should keep torsion-offset = 180?
	//
	if ( ( aa == aa_pro && ( atomno1 == 6 && atomno2 == 7  || // CG--CD
													 atomno2 == 6 && atomno1 == 7  ) ) ||
			 ( aa == aa_tyr && ( atomno1 == 9 && atomno2 == 11 || // CE1--CZ
													 atomno2 == 9 && atomno1 == 11 ) ) ||
			 ( aa == aa_trp && ( atomno1 == 6 && atomno2 == 8  || // CG--CD2
													 atomno2 == 6 && atomno1 == 8 ) ) ||
			 ( aa == aa_trp && ( atomno1 == 8 && atomno2 == 10 || // CD2--CE2
													 atomno2 == 8 && atomno1 == 10 ) ) ||
			 ( is_DNA(aa)   && ( atomno1 == 8 && atomno2 == 10 || // C3*--C2*
													 atomno2 == 8 && atomno1 == 10 ) ) ||
			 //Personal preference -- have 2'-OH closer to the action.
			 ( is_RNA(aa)   && ( atomno1 == 7 && atomno2 == 12 || // O4*--C1*
															 atomno2 == 7 && atomno1 == 12 ) ) ||
			 //Useful to force path through bases, for unique connectivity..
			 ( aa == na_rcy   && ( atomno1 == 13 && atomno2 == 14 || // N1--C2
 														atomno2 == 13 && atomno1 == 14 ) ) ||
 			 ( aa == na_ura   && ( atomno1 == 13 && atomno2 == 14 || // N1--C2
 														atomno2 == 13 && atomno1 == 14 ) ) ||
 			 ( aa == na_rgu   && ( atomno1 == 14 && atomno2 == 16 || // N3--C2
 														atomno2 == 14 && atomno1 == 16 ) ) ||
 			 ( aa == na_rad   && ( atomno1 == 14 && atomno2 == 15 || // N3--C2
 														atomno2 == 14 && atomno1 == 15 ) )
			 )
		{
		return true;
		}
	return false;
}


///////////////////////////////////////////////////////////////////////////////
kin::Atom*
build_tree(
	pose_ns::Fold_tree const & fold_tree,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray2D< kin::Atom* > & atom_pointer
)
{
	using namespace kin;
	setup_kin_atom_properties();


	atom_pointer = 0;

	// build first atom. make it an "Atom_jump"
	int const start_pos( fold_tree.begin()->start );
	Atom* tmp_ptr;
	Atom* root( build_aa( 0, res( start_pos ), res_variant( start_pos ),
		start_pos, dir_jump, atom_pointer( 1, start_pos ), tmp_ptr));


	// traverse tree, build edges
	for ( pose_ns::Fold_tree::const_iterator it = fold_tree.begin(),
					it_end = fold_tree.end(); it != it_end; ++it ) {
		int const estart( it->start );
		int const aa ( res        ( estart ) );
		int const aav( res_variant( estart ) );
		if ( it->is_jump() ) {
			// build a jump -- there may be info in the tree about anchor atoms
			int const jump_number( it->label );
			int const estop( it->stop );
			int const aa_stop ( res        ( estop ) );
			int const aav_stop( res_variant( estop ) );
			int anchor_atomno( fold_tree.upstream_atomno( jump_number ) );
			int root_atomno( fold_tree.downstream_atomno( jump_number ) );
			int build_aa_dir( dir_jump );
			if ( !anchor_atomno ) {
				anchor_atomno = get_anchor_atomno( aa, aav, dir_jump );
				root_atomno = get_root_atomno( aa_stop, aav_stop, dir_jump );
			} else if ( root_atomno < 0 ) {
				root_atomno *= -1;
				build_aa_dir = pose_ns::pose_param::n2c; // silly
				std::cout << "Building connection from " << estart << ' ' <<
					anchor_atomno << " to " << estop << ' ' << root_atomno <<
					" as a bond rather than a jump." << std::endl;
			}
			Atom * parent( atom_pointer( anchor_atomno, estart ) );
			assert( parent );
// 			std::cout << "build_jump: " << anchor_atomno << ' ' << root_atomno <<
// 				std::endl;
			parent->insert_atom( build_aa( parent, aa_stop, aav_stop, estop,
																		 build_aa_dir, atom_pointer(1, estop ),
																		 tmp_ptr, root_atomno ) );

		} else {
			// build a peptide edge
			int const estart( it->start);
			int const atomno( get_anchor_atomno( res( estart ), res_variant(estart ),
																					 it->dir0() ) );
			Atom* parent( atom_pointer( atomno, estart ) );
			assert( parent );
			parent->insert_atom( build_peptide_edge( parent, *it, res, res_variant,
																							 atom_pointer ) );
		}
	}

	return root;
}

///////////////////////////////////////////////////////////////////////////////
kin::Atom*
build_peptide_edge(
	kin::Atom* parent,
	pose_ns::Edge const & edge,
	FArray1D_int const & res,
	FArray1D_int const & res_variant,
	FArray2D< kin::Atom* > & atom_pointer
	)
{
	using namespace kin;
	int const start( edge.start );
	int const stop ( edge.stop );
	int const dir  ( edge.dir0() );

	Atom * root;
	assert( dir == pose_ns::pose_param::n2c || dir==pose_ns::pose_param::c2n );
	int pos( start+dir);
	Atom *ptr1, *ptr2;
	root = build_aa( parent, res( pos ), res_variant( pos ), pos, dir,
									 atom_pointer( 1,pos), ptr1 );
	for ( pos = start+2*dir; pos != stop + dir; pos+= dir ) {
		ptr1->insert_atom( build_aa( ptr1, res( pos ), res_variant( pos ), pos,
																 dir, atom_pointer( 1,pos ), ptr2 ) );
		ptr1 = ptr2;
	}
	return root;
}


///////////////////////////////////////////////////////////////////////////////
kin::Atom*
build_aa(
	kin::Atom* parent,
	int const aa,
	int const aav,
	int const rsd, // needed since Atoms want to know what residue they are
	int const dir,
	FArray1Da< kin::Atom* > atom_ptr,
	kin::Atom* & end_ptr,
	int root_atomno // = 0
)
{
	using namespace kin;

// 	std::cout << "build_aa: " << param_aa::aa_name3(aa) << ' ' << rsd << ' ' <<
// 		dir << ' ' << root_atomno << std::endl;
	atom_ptr.dimension( param::MAX_ATOM()() );
	Atom *root, *anchor;

	if (!root_atomno ) root_atomno = get_root_atomno( aa,aav,dir);
	int const anchor_atomno( get_anchor_atomno( aa,aav,dir) );

	root = add_kin_atom( parent, aa, aav, rsd, root_atomno, atom_ptr, dir );
	anchor = atom_ptr( anchor_atomno );
	assert( anchor != 0 );
	end_ptr = anchor;
	return root;
}


///////////////////////////////////////////////////////////////////////////////
// creates an atom pointer for atom: (rsd,atomno)
// generates downstream atom list for this atom, ordered by:
//
// -- backbone atoms
// -- chi atoms
// -- other
//
// excluding atoms with atom_ptr != 0
//
// returns atom pointer

kin::Atom*
add_kin_atom(
	kin::Atom* parent,
	int const aa,
	int const aav,
	int const rsd,
	int const atomno,
	FArray1DB< kin::Atom * > & atom_ptr,
	int const dir // = n2c by default (equiv to c2n here, ie NOT jump)
)
{
	//using namespace kin;
	using namespace kin_atom_properties;
	using namespace aaproperties_pack;

	//		std::cout << "add_kin_atom: " << rsd << ' ' << atomno << ' ' <<
	//			param_aa::aa_name3( aa ) << ' ' <<
	//			atom_name( atomno, aa, aav ) << std::endl;

	// local copies of arrays with aa,aav dependence taken out
	FArray1Da_bool is_bb_atom_aa_aav( is_backbone_atom( 1,aa,aav ) );
	is_bb_atom_aa_aav.dimension( param::MAX_ATOM()() );

	FArray1Da_bool is_chi_atom_aa_aav( is_chi_atom( 1,aa,aav ) );
	is_chi_atom_aa_aav.dimension( param::MAX_ATOM()() );

	FArray2Da_int bonded_neighbor_aa_aav( bonded_neighbor( 1, 1, aa, aav ) );
	bonded_neighbor_aa_aav.dimension( bonded_neighbor.size1(),
																		bonded_neighbor.size2() );

	// create new atom
	kin::Atom* atom;
	if ( dir == kin::dir_jump ) {
		atom = new kin::Jump_atom;
	} else {
		atom = new kin::Bonded_atom;
	}

	// fill in the atom_ptr array
	assert( atom_ptr( atomno ) == 0 );
	atom_ptr( atomno ) = atom;

	// set the atom_id information
	atom->atom_id.rsd() = rsd;
	atom->atom_id.atomno() = atomno;

	// set the parent pointer
	atom->parent = parent;
	atom->set_torsion_moved( true, true, false );
	atom->set_allow_move( kin::ALL, false );

	// reset atoms list ( probably unnecessary )
	//atom->atoms.clear();

	// loop over bonded atoms
	// first bb atoms
	int const nbr_count( nbonded_neighbors( atomno, aa, aav ) );
	for ( int i=1; i<= nbr_count; i++ ) {
		int const nbr_atomno( bonded_neighbor_aa_aav( i, atomno ) );
		// debug
// 		if ( cut_bond( atomno, nbr_atomno, aa, aav ) ) {
// 			std::cout << "cut_bond: " << aa << ' ' << atomno << ' ' <<
// 				nbr_atomno << std::endl;
// 		}

		if ( atom_ptr( nbr_atomno ) == 0 &&
				 !cut_bond( atomno, nbr_atomno, aa, aav ) &&
				 is_bb_atom_aa_aav( nbr_atomno ) ) {
			atom->add_atom( add_kin_atom( atom, aa, aav, rsd, nbr_atomno,
																		atom_ptr ) );
		}
	}

	// chi atoms
	for ( int i=1; i<= nbr_count; i++ ) {
		int const nbr_atomno( bonded_neighbor_aa_aav( i, atomno ) );
		if ( atom_ptr( nbr_atomno ) == 0 &&
				 !cut_bond( atomno, nbr_atomno, aa, aav ) &&
				 is_chi_atom_aa_aav( nbr_atomno ) ) {
			atom->add_atom( add_kin_atom( atom, aa, aav, rsd, nbr_atomno,
																		atom_ptr ) );
		}
	}

	// other atoms
	for ( int i=1; i<= nbr_count; i++ ) {
		int const nbr_atomno( bonded_neighbor_aa_aav( i, atomno ) );
		if ( atom_ptr( nbr_atomno ) == 0 &&
				 !cut_bond( atomno, nbr_atomno, aa, aav ) ) {
			atom->add_atom( add_kin_atom( atom, aa, aav, rsd, nbr_atomno,
																		atom_ptr ) );
		}
	}

	return atom;
}


///////////////////////////////////////////////////////////////////////////////
// only for protein, DNA right now
void
setup_kin_atom_properties()
{
	using namespace kin_atom_properties;
	using namespace param_aa;

	static bool init( false );
	if ( init ) return;
	init = true;

	// int const aav=1; // hard-coded -- not anymore!

	// initialize
	is_backbone_atom = false;
	is_chi_atom = false;

	// loop over aa
	for ( int aa=1; aa<= param::MAX_AA()(); ++aa ) {
		for ( int aav=1; aav<= aaproperties_pack::nvar(aa); ++aav ) {
			if ( is_protein(aa) ) {
				is_backbone_atom( 1, aa, aav ) = true; // N
				is_backbone_atom( 2, aa, aav ) = true; // CA
				is_backbone_atom( 3, aa, aav ) = true; // C

				for ( int chino=1; chino<= aaproperties_pack::nchi(aa,aav); ++chino ) {
					for ( int k=1; k<= 4; ++k ) {
						is_chi_atom( aaproperties_pack::chi_atoms(k,chino,aa,aav),
												 aa, aav ) = true;
					}
				}

			} else if ( is_DNA( aa ) ) {
				using dna_variables::p;
				using dna_variables::o5star;
				using dna_variables::c5star;
				using dna_variables::c4star;
				using dna_variables::c3star;
				using dna_variables::o3star;
				using dna_variables::c2;
				using dna_variables::c4A;
				using dna_variables::c4G;
				using dna_variables::o4star;
				using dna_variables::c1star;
				using dna_variables::n9A;
				using dna_variables::n9G;
				using dna_variables::n1C;
				using dna_variables::n1T;

				is_backbone_atom( p     , aa, aav ) = true; // P
				is_backbone_atom( o5star, aa, aav ) = true; // o5*
				is_backbone_atom( c5star, aa, aav ) = true; // c5*
				is_backbone_atom( c4star, aa, aav ) = true; // c4*
				is_backbone_atom( c3star, aa, aav ) = true; // c3*
				is_backbone_atom( o3star, aa, aav ) = true; // o3*
				is_chi_atom( o4star, aa, aav ) = true;
				is_chi_atom( c1star, aa, aav ) = true;
				if ( aa == na_ade ) {
					is_chi_atom( n9A, aa, aav ) = true;
					is_chi_atom( c4A, aa, aav ) = true;
				} else if ( aa == na_gua ) {
					is_chi_atom( n9G, aa, aav ) = true;
					is_chi_atom( c4G, aa, aav ) = true;
				} else if ( aa == na_cyt ) {
					is_chi_atom( n1C, aa, aav ) = true;
					is_chi_atom( c2 , aa, aav ) = true;
				} else if ( aa == na_thy ) {
					is_chi_atom( n1T, aa, aav ) = true;
					is_chi_atom( c2 , aa, aav ) = true;
				}
			} else if ( is_RNA( aa ) ) {
				using rna_variables::p;
				using rna_variables::o5star;
				using rna_variables::c5star;
				using rna_variables::c4star;
				using rna_variables::c3star;
				using rna_variables::o3star;
				using rna_variables::c2;
				using rna_variables::c4A;
				using rna_variables::c4G;
				using rna_variables::o4star;
				using rna_variables::c1star;
				using rna_variables::c2star;
				using rna_variables::o2star;
				using rna_variables::n9A;
				using rna_variables::n9G;
				using rna_variables::n1C;
				using rna_variables::c6C;
				using rna_variables::n1U;
				using rna_variables::c6U;

				is_backbone_atom( p     , aa, aav ) = true; // P
				is_backbone_atom( o5star, aa, aav ) = true; // o5*
				is_backbone_atom( c5star, aa, aav ) = true; // c5*
				is_backbone_atom( c4star, aa, aav ) = true; // c4*
				is_backbone_atom( c3star, aa, aav ) = true; // c3*
				is_backbone_atom( o3star, aa, aav ) = true; // o3*

				//conventional path
				//is_chi_atom( c1star, aa, aav ) = true;
				//is_chi_atom( o4star, aa, aav ) = true;

				//rhiju -- A different path through the sugar
				is_chi_atom( c2star, aa, aav ) = true;
				is_chi_atom( c1star, aa, aav ) = true;

				// Will need to specify where the 2'-OH points.
				//				is_chi_atom( o2star, aa, aav ) = true;

				if ( aa == na_rad ) {
					is_chi_atom( n9A, aa, aav ) = true;
					is_chi_atom( c4A, aa, aav ) = true;
				} else if ( aa == na_rgu ) {
					is_chi_atom( n9G, aa, aav ) = true;
					is_chi_atom( c4G, aa, aav ) = true;
				} else if ( aa == na_rcy ) {
					// The reason for switching the chi angle definition here
					// is a little byzantine. I need c4, n3, c2 to
					// define jumps, and they only come out in the
					// right order if the bond between n1 and c2 is cut.
					// so that leaves n1-c6 to define chi1 angle.
					is_chi_atom( n1C, aa, aav ) = true;
					is_chi_atom( c6C, aa, aav ) = true;
				} else if ( aa == na_ura ) {
					is_chi_atom( n1U, aa, aav ) = true;
					is_chi_atom( c6U, aa, aav ) = true;
				}
			} // aa-type
		} // aav
	} // aa
}

///////////////////////////////////////////////////////////////////////////////
void
setup_torsion_atom_aa_aav(
	int const aa,
	int const aav,
	FArray2D_int & torsion_atom,
	FArray2D_int & torsion_rsd_offset
	)
{
	using namespace param_aa;
	using namespace param_torsion;
	int const N(1), CA(2), C(3); // full_coord indexing
	// sizes of these arrays now change depending on protein, DNA, or RNA.
	//	assert( int(torsion_atom.size2()) >= total_torsion &&
	//					int(torsion_rsd_offset.size2())  >= total_torsion );

	if ( is_protein( aa ) ) {

		torsion_atom( 1,   phi_torsion ) = C ;
		torsion_atom( 2,   phi_torsion ) = N ;
		torsion_atom( 3,   phi_torsion ) = CA ;
		torsion_atom( 4,   phi_torsion ) = C ;

		torsion_atom( 1,   psi_torsion ) = N ;
		torsion_atom( 2,   psi_torsion ) = CA;
		torsion_atom( 3,   psi_torsion ) = C ;
		torsion_atom( 4,   psi_torsion ) = N ;

		torsion_atom( 1, omega_torsion ) = CA;
		torsion_atom( 2, omega_torsion ) = C ;
		torsion_atom( 3, omega_torsion ) = N ;
		torsion_atom( 4, omega_torsion ) = CA;

		torsion_rsd_offset( 1,   phi_torsion )= -1;
		torsion_rsd_offset( 2,   phi_torsion )=  0;
		torsion_rsd_offset( 3,   phi_torsion )=  0;
		torsion_rsd_offset( 4,   phi_torsion )=  0;

		torsion_rsd_offset( 1,   psi_torsion )=  0;
		torsion_rsd_offset( 2,   psi_torsion )=  0;
		torsion_rsd_offset( 3,   psi_torsion )=  0;
		torsion_rsd_offset( 4,   psi_torsion )= +1;

		torsion_rsd_offset( 1, omega_torsion )=  0;
		torsion_rsd_offset( 2, omega_torsion )=  0;
		torsion_rsd_offset( 3, omega_torsion )= +1;
		torsion_rsd_offset( 4, omega_torsion )= +1;

		for ( int chino = 1; chino <= aaproperties_pack::nchi(aa,aav); ++chino ) {
			for ( int k=1; k<= 4; ++k ) {
				torsion_atom( k, chino+3 ) = aaproperties_pack::chi_atoms( k, chino, aa,
																																	 aav );
				torsion_rsd_offset ( k, chino+3 ) = 0;
			}
		}
	} else if ( is_DNA( aa ) ) {
		// 1. alpha: O3*(i-1) -- P   -- O5*    -- C5*
		// ...
		// 6. xi?:   C3*      -- O3* -- P(i+1) -- O5*(i+1)
		FArray1D_int bb_atoms(6);
		bb_atoms(1) = dna_variables::p;
		bb_atoms(2) = dna_variables::o5star;
		bb_atoms(3) = dna_variables::c5star;
		bb_atoms(4) = dna_variables::c4star;
		bb_atoms(5) = dna_variables::c3star;
		bb_atoms(6) = dna_variables::o3star;

		for ( int tor=1; tor<= 6; ++tor ) {
			//std::cout << "tor= " << tor << ' ';
			for ( int k=1; k<= 4; ++k ) {
				torsion_atom      ( k, tor ) = bb_atoms( (k+tor+3)%6 + 1 );
				torsion_rsd_offset( k, tor ) = int( std::floor( (k+tor-3)/6.0 ) );
				//	std::cout <<
				//	I(3,torsion_atom( k,tor)) << I(3,torsion_rsd_offset(k,tor));
			}
			//std::cout << std::endl;
		}

		// now do chi angle
		torsion_atom( 1, 7 ) = dna_variables::o4star;
		torsion_atom( 2, 7 ) = dna_variables::c1star;
		if ( aa == na_ade ) {
			torsion_atom( 3, 7 ) = dna_variables::n9A;
			torsion_atom( 4, 7 ) = dna_variables::c4A;
		} else if ( aa == na_gua ) {
			torsion_atom( 3, 7 ) = dna_variables::n9G;
			torsion_atom( 4, 7 ) = dna_variables::c4G;
		} else if ( aa == na_cyt ) {
			torsion_atom( 3, 7 ) = dna_variables::n1C;
			torsion_atom( 4, 7 ) = dna_variables::c2;
		} else if ( aa == na_thy ) {
			torsion_atom( 3, 7 ) = dna_variables::n1T;
			torsion_atom( 4, 7 ) = dna_variables::c2;
		}
		for ( int k=1; k<= 4; ++k ) torsion_rsd_offset( k, 7 ) = 0;
	} else if ( is_RNA( aa ) ) {
		// Hey there are more transparent ways to do this...
		// 1. alpha: O3*(i-1) -- P   -- O5*    -- C5*
		// ...
		// 6. xi?:   C3*      -- O3* -- P(i+1) -- O5*(i+1)
		FArray1D_int bb_atoms(6);
		bb_atoms(1) = rna_variables::p;
		bb_atoms(2) = rna_variables::o5star;
		bb_atoms(3) = rna_variables::c5star;
		bb_atoms(4) = rna_variables::c4star;
		bb_atoms(5) = rna_variables::c3star;
		bb_atoms(6) = rna_variables::o3star;

		for ( int tor=1; tor<= 6; ++tor ) {
			//std::cout << "tor= " << tor << ' ';
			for ( int k=1; k<= 4; ++k ) {
				torsion_atom      ( k, tor ) = bb_atoms( (k+tor+3)%6 + 1 );
				torsion_rsd_offset( k, tor ) = int( std::floor( (k+tor-3)/6.0 ) );
				//	std::cout <<
				//	I(3,torsion_atom( k,tor)) << I(3,torsion_rsd_offset(k,tor));
			}
			//std::cout << std::endl;
		}

		// now do chi angle
		//		torsion_atom( 1, 7 ) = rna_variables::o4star;
		torsion_atom( 1, 7 ) = rna_variables::c2star;
		torsion_atom( 2, 7 ) = rna_variables::c1star;
		if ( aa == na_rad ) {
			torsion_atom( 3, 7 ) = rna_variables::n9A;
			torsion_atom( 4, 7 ) = rna_variables::c4A;
		} else if ( aa == na_rgu ) {
			torsion_atom( 3, 7 ) = rna_variables::n9G;
			torsion_atom( 4, 7 ) = rna_variables::c4G;
		} else if ( aa == na_rcy ) {
			// The reason for switching the chi angle definition here
			// is a little byzantine. I need c4, n3, c2 to
			// define jumps, and they only come out in the
			// right order if the bond between n1 and c2 is cut.
			// so that leaves n1-c6 to define chi1 angle.
			torsion_atom( 3, 7 ) = rna_variables::n1C;
			torsion_atom( 4, 7 ) = rna_variables::c6C;
		} else if ( aa == na_ura ) {
			torsion_atom( 3, 7 ) = rna_variables::n1U;
			torsion_atom( 4, 7 ) = rna_variables::c6U;
		}
		for ( int k=1; k<= 4; ++k ) torsion_rsd_offset( k, 7 ) = 0;


		///////////////////////////////////////////////////////////////
		// Hey, the sugar could change too. Silly hack -- couple more angles.
		//  Note: These two correspond to a single degree of freedom (sugar pucker)
		//  once you impose ideal bond lengths, angles, and ring closure.
		// These might also be useful d.o.f. for DNA.
		///////////////////////////////////////////////////////////////
		torsion_atom( 1, 8 ) = rna_variables::c4star;
		torsion_atom( 2, 8 ) = rna_variables::c3star;
		torsion_atom( 3, 8 ) = rna_variables::c2star;
		torsion_atom( 4, 8 ) = rna_variables::c1star;
		for ( int k=1; k<= 4; ++k ) torsion_rsd_offset( k, 8 ) = 0;

		torsion_atom( 1, 9 ) = rna_variables::c3star;
		torsion_atom( 2, 9 ) = rna_variables::c2star;
		torsion_atom( 3, 9 ) = rna_variables::c1star;
		if ( aa == na_rad ) {
			torsion_atom( 4, 9 ) = rna_variables::n9A;
		} else if ( aa == na_rgu ) {
			torsion_atom( 4, 9 ) = rna_variables::n9G;
		} else if ( aa == na_rcy ) {
			torsion_atom( 4, 9 ) = rna_variables::n1C;
		} else if ( aa == na_ura ) {
			torsion_atom( 4, 9 ) = rna_variables::n1U;
		}
		for ( int k=1; k<= 4; ++k ) torsion_rsd_offset( k, 9 ) = 0;

		//For RNA, will also need a "chi" for the 2'-OH:
		torsion_atom( 1, 10 ) = rna_variables::c3star;
		torsion_atom( 2, 10 ) = rna_variables::c2star;
		torsion_atom( 3, 10 ) = rna_variables::o2star;
		if ( aa == na_rad ) {
			torsion_atom( 4, 10 ) = rna_variables::ho2starA;
		} else if ( aa == na_rgu ) {
			torsion_atom( 4, 10 ) = rna_variables::ho2starG;
		} else if ( aa == na_rcy ) {
			torsion_atom( 4, 10 ) = rna_variables::ho2starC;
		} else if ( aa == na_ura ) {
			torsion_atom( 4, 10 ) = rna_variables::ho2starU;
		}
		for ( int k=1; k<= 4; ++k ) torsion_rsd_offset( k, 10 ) = 0;

	}
}



///////////////////////////////////////////////////////////////////////////////
int
get_root_atomno(
	int const aa,
	int const ,//aav
	int const dir
)
{
	int const n2c(1), c2n(-1), dir_jump(0);
	if ( param_aa::is_protein(aa) ) {
		if ( dir == n2c || dir == dir_jump ) {
			return 1; // N
		} else {
			assert( dir == c2n);
			return 3; // C
		}
	} else if (  param_aa::is_DNA(aa) ) {
		if ( dir == n2c || dir == dir_jump ) {
			return dna_variables::p;
		} else {
			assert( dir == c2n);
			return dna_variables::o3star;
		}
	} else if (  param_aa::is_RNA(aa) ) {
		if ( dir == dir_jump ) {
			if (aa == param_aa::na_rad) return rna_variables::c6A;
			if (aa == param_aa::na_rcy) return rna_variables::c4C;
			if (aa == param_aa::na_rgu) return rna_variables::c6G;
			if (aa == param_aa::na_ura) return rna_variables::c4U;
		} else if ( dir == n2c ) {
			return rna_variables::p;
		} else {
			assert( dir == c2n);
			return rna_variables::o3star;
		}
	}
	return kin_atom_properties::default_root_atomno( aa );
}

///////////////////////////////////////////////////////////////////////////////
int
get_anchor_atomno(
	int const aa,
	int const ,//aav,
	int const dir
)
{
	int const n2c(1), c2n(-1), dir_jump(0);
	if (  param_aa::is_protein(aa) ) {
		if ( dir == c2n || dir == dir_jump ) {
			return 1; // N
		} else {
			assert( dir == n2c);
			return 3; // C
		}
	} else if (  param_aa::is_DNA(aa) ) {
		if ( dir == c2n || dir == dir_jump ) {
			return dna_variables::p;
		} else {
			assert( dir == n2c);
			return dna_variables::o3star;
		}
	} else if (  param_aa::is_RNA(aa) ) {
		if ( dir == dir_jump ) {
			return rna_variables::c2;
		} else if ( dir == c2n ) {
			return rna_variables::p;
		} else {
			assert( dir == n2c);
			return rna_variables::o3star;
		}
	}
	return kin_atom_properties::default_anchor_atomno( aa );
}



///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
