// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:

//  CVS information:
//  $Revision: 15327 $
//  $Date: 2007-06-05 07:58:57 -0700 (Tue, 05 Jun 2007) $
//  $Author: sarel $

#ifndef LEE_RICHARDS_HH
#define LEE_RICHARDS_HH

#include <iosfwd>
#include <vector>
#include <set>
#include <map>
#include <list>

// #include <Util/Fwd.hh>
// #include <Util/Tubeable.hh>
// #include <UtilColorable.hh>
// #include "Indexable.hh"
// #include "Point2.hh"

#include "murphp_classes.h"

namespace murphp {

  using namespace std;

  namespace LeeRichardsInfo {


    // ____________________ fwd ____________________

    class Atom;
    class SLE_createArcs;
    //class CreateArcs; // since this functionality is so closely related with Slice, just moving it into Slice
    class Arc;
    class Circle;
    class Boundary;
    class Slice;
    class SLE_createVolumes;
    class CreateVolumes;
    class Volume;


    // ____________________ Atom ____________________

    class Atom {
    private:
      //const Util::Atom* util_atom;
      Point3 p;
      double radius;
    public:
      //Atom(const murphp::Atom* util_atom_);

      Atom(const Point3& p_, const double radius_) : p(p_), radius(radius_) {}

      const Point3& getPosn() const { return p; }
      const double getRadius() const { return radius; }

      //const murphp::Atom* getUtilAtom() const { return util_atom; }

    }; // class Atom


    // ____________________ SLE_createArcs ____________________

    class SLE_createArcs {
    public:

      // <algorithm>

      enum Type { BEGIN, OCCLUDE_HI, OCCLUDE_LO, REVEAL_LO, REVEAL_HI, END }; // !!! the precise order of these enum args is tremendously important !!!

      SLE_createArcs::Type event_type, other_event_type;
      Point2 p;
      Circle *c, *other;

      SLE_createArcs(SLE_createArcs::Type event_type_, const Point2& p_, Circle* c_, Circle* other_, SLE_createArcs::Type other_event_type_);

      const bool operator<(const SLE_createArcs& other) const;

      // </algorithm>

    }; // class SLE_createArcs

    ostream& operator<<(ostream& out, const SLE_createArcs& e);


    // ____________________ Arc ____________________

    class Arc : public Colorable {
    public:

      enum Type { HI, LO };

    private:

      const Circle* circle; // the circle this arc is on
      const Boundary* boundary; // the boundary this arc is part of
      Arc* next_arc; // the next arc in the boundary

      // <algorithm>

      Arc::Type type; // the type of this arc
      Point2 begin, end; // the leftward & rightward points respectively
      double theta_a,theta_b; // the leftward & rightward radians of this arc resp
      double chord_m, chord_b; // slope & intercept of the chord of this arc

      // </algorithm>


    public:

      // ctor
      Arc(const Circle* circle_, const Arc::Type type_);


      // accessors
      void setNext(Arc* next_arc_, const SLE_createArcs::Type event_type, const SLE_createArcs::Type other_event_type);
      void setBoundary(const Boundary* b);

      const Type getType() const { return type; }
      const Arc* getNext() const  { return next_arc; }
      const Circle* getCircle() const { return circle; }
      const Boundary* getBoundary() const { return boundary; }


      // geometrical parameters of the arc

      void setPoints(const Point2& begin_, const Point2& end_);
      void setChord();

      const double getArea() const;
      const double getPerimeter() const;
      const double getChord(const double x) const;
      const Point2& getBegin() const { return begin; }
      const Point2& getEnd() const { return end; }

      // mainly for debugging
      const double getThetaA() const { return theta_a; }
      const double getThetaB() const { return theta_b; }


      // report

      friend ostream& operator<<(ostream& out, const Arc& arc);

    }; // Class Arc



    // ____________________ Circle ____________________

    class Circle {
    private:

      // data

      const Atom* atom; // the atom this circle corresponds to

      Point2 center; // the 2d location of this circle
      double radius; // the radius of this circle


      // <algorithm>

      int occ[2];
      bool pending[2];
      Arc* pending_arc[2];
      //Arc* waiting_arc[2];
      Point2 last[2];
      int initial_occ, final_occ, nho, nlo, nhr, nlr;

      void clear();

      // </algorithm>


    public:

      // ctor

      Circle(const Atom* atom_, const Point2& center, const double radius);


      // accessors

      const Atom* getAtom() const { return atom; }
      const double getRadius() const { return radius; }
      const Point2& getCenter() const { return center; }


      // <algorithm>

      void initialize(int hi_occ, int lo_occ, const Point2& p, Circle* other, SLE_createArcs::Type other_type);
      void occlude_hi(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type);
      void occlude_lo(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type);
      void reveal_hi(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type);
      void reveal_lo(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type);
      void finalize(const Point2& p, vector<Arc*>& vArcs, Circle* other, SLE_createArcs::Type other_type);
      void assertSanity();
      static const bool intersect(const Circle& a, const Circle& b, Point2& isx0, Point2& isx1);

      // </algorithm>


    }; // class Circle

    ostream& operator<<(ostream& out, const Circle& c);


    // ____________________ Boundary ____________________

    class Boundary : public Indexable {
    private:

      const Arc* arc0; // a circular linked list of arcs that define this boundary
      const Slice* slice; // the slice that this boundary belongs to
      set<const Boundary*> sBoundaries; // the set of boundaries which overlap this boundary
      const Volume* volume; // the volume that this boundary corresponds to

      int num_arcs; // the number of arcs that comprise this boundary
      double area, perimeter; // the precalculated area & perimeter of the boundary
      void calculateAreaAndPerimeter();


    public:

      static const double THRESHOLD_AREA_POS; // XXX WTF?
      static const double THRESHOLD_AREA_NEG;


      // ctor/dtor

      Boundary(const Arc* arc0_, const Slice* slice_);
      virtual ~Boundary();


      // accessors

      const Arc* getArc0() const { return arc0; }
      const Slice* getSlice() const { return slice; }
      const set<const Boundary*>& getBoundaries() const { return sBoundaries; }
      const Volume* getVolume() const { return volume; }

      void addBoundary(const Boundary* b);
      void setVolume(const Volume* v);

      const int getNumArcs() const;
      const double getArea() const;
      const double getPerimeter() const;
      const bool isConvex() const;
      const bool isConcave() const;
      const bool isSame(const Boundary* other) const; // in terms of concavity/convexity
      const bool isIgnorable() const;


      // report

      void reportArcs(ostream& out) const;

    }; // class Boundary

    // ____________________ Slice ____________________

    class Slice : public Indexable {
    private:

      double z; // the z coordinate of this slice

      vector<Circle*> vCircles; // the circles owned by this slice
      vector<Arc*> vArcs; // the arcs owned by this slice
      vector<Boundary*> vBoundaries;  // the boundaries owned by this slice
      vector<const Boundary*> vBoundaries_convex; // the convex boundaries owned by this slice
      vector<const Boundary*> vBoundaries_concave; // the concave boundaries owned by this slice

      const Slice* prev; // the prev/next slice in the lr
      const Slice* next;


      // <algorithm>

      set<SLE_createArcs> event_stack;

      void clear();
      void addCircles(const vector<Atom*>& vAtoms, const double z);
      void addEvents(Point2& isx0, Point2& isx1, LeeRichardsInfo::Circle& a, LeeRichardsInfo::Circle& b);
      void addArcs(); // XXX could call this addArcs
      void addBoundaries();
      void setIndices();

      // </algorithm>

    public:

      // ctor/dtor

      Slice();
      virtual ~Slice();


      // <algorithm>

      void execute(const vector<Atom*>& vAtoms, const double z);

      // </algorithm>


      // accessors

      void setPrev(const Slice* s) { prev = s; }
      void setNext(const Slice* s) { next = s; }

      const double getZ() const { return z; }
      const double getWidth() const;
      const double getArea() const;
      const double getPerimeter() const;

      const Slice* getPrev() const { return prev; }
      const Slice* getNext() const { return next; }

      const vector<Boundary*>& getBoundaries() const;
      const vector<const Boundary*>& getBoundaries_convex() const;
      const vector<const Boundary*>& getBoundaries_concave() const;


      // report

      void reportArcs(ostream& out) const;
      void reportBoundaries(ostream& out) const;

      void report(ostream& out) const;


    }; // class Slice

    // ____________________ Volume ____________________

    class Volume : public Indexable {
    private:

      map<const Slice*,vector<const Boundary*> > mvBoundaries; // the boundaries that comprise this volume, indexed by the slice they're owned by

    public:

      // ctor/dtor

      //Volume(const Boundary* b,const map<const Boundary*,set<const Boundary*> >& msOverlaps);
      Volume(const Boundary* b);
      virtual ~Volume();


      // accessors

      const double getSurfaceArea() const;
      const double getVolume() const;

      const bool isConcave() const;
      const bool isConvex() const;
      const bool isClosed() const;


      // report

      void reportArcs(ostream& out) const;


    }; // class Volume


    // ____________________ SLE_createVolumes ____________________


    class SLE_createVolumes {
    public:

      // <algorithm>
      enum Type { BEGIN, END };

      SLE_createVolumes::Type type;
      const Arc* arc;

      SLE_createVolumes() {}
      SLE_createVolumes(const SLE_createVolumes::Type type_, const Arc* arc);

      const Point2& getPoint() const;

      const bool operator<(const SLE_createVolumes& other) const;

      // </algorithm>

    }; // class SLE_createVolumes

    ostream& operator<<(ostream& out, const SLE_createVolumes& sle);


    // ____________________ CreateVolumes ____________________

    class CreateVolumes {
    private:

      // <algorithm>

      SLE_createVolumes sle;
      const Slice *s_hi, *s_lo;
      const Slice* getOtherSlice(const Arc* arc);

      set<SLE_createVolumes> event_stack;

      list<const Arc*> active_hi;
      list<const Arc*> active_lo;
      list<const Arc*>& getActiveArcs(const Slice* slice);

      void isContained(const Arc* arc);
      void isIntersected(const Arc* arc);
      void addActive(const Arc* arc);
      void removeActive(const Arc* arc);

      void addEvents(const vector<const Boundary*>& vBoundaries);
      void addOverlap(const Boundary* a, const Boundary* b);

      // </algorithm>

    public:

      void execute(const Slice* s_hi_, const Slice* s_lo_ );

    }; // class CreateVolumes

  } // namespace LeeRichardsInfo


  // ____________________ LeeRichards ____________________

  //class LeeRichards : public Tubeable {
  class LeeRichards {
  private:

    double z_lo, z_hi; // the hi & lo z value
    vector<LeeRichardsInfo::Atom*> vAtoms; // the atoms owned by this lee-richards object
    vector<LeeRichardsInfo::Slice*> vSlices; // the slices owned by this lee-richards object
    vector<LeeRichardsInfo::Volume*> vVolumes; // the volumes owned by this lee-richards object

    // <algorithm>

    void clear();
    //void addAtoms();
    void addAtoms(const vector<LeeRichardsInfo::Atom*>& vAtoms_);
    void addSlices(const int n);
    void addSlices_z(const double z);
    void addVolumes();

    // </algorithm>

  public:

    // ctor/dtor

    //LeeRichards(const Tube* tube);
    LeeRichards(const vector<LeeRichardsInfo::Atom*>& vAtoms_);
    virtual ~LeeRichards();


    // <algorithm>

    void execute(const int n);
    void execute_z(const double z);

    // </algorithm>


    // accessors

    const double getZ_lo() const { return z_lo; }
    const double getZ_hi() const { return z_hi; }


    // report

    void reportArcs(ostream& out) const;
    void reportSlices(ostream& out) const;
    void reportBoundaries(ostream& out) const;
    void reportVolumes(ostream& out) const;


    // <testing>
    const double getPerimeter() const;
    const vector<LeeRichardsInfo::Slice*>& getSlices() const { return vSlices; }
    // </testing>

  }; // LeeRichards

} // namespace murphp

#endif // LEE_RICHARDS_HH
