// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//  CVS information:
//  $Revision: 19866 $
//  $Date: 2008-01-23 11:54:04 -0800 (Wed, 23 Jan 2008) $
//  $Author: possu $


// Rosetta Headers
#include "ligand_ns.h"
#include "aaproperties_pack.h"
#include "ligand.h"
#include "param.h"
#include "atom.h"
#include "BondSmallMolecule.h"
#include "design.h"
#include "etable.h"
#include "enzyme_ns.h"
#include "hbonds_geom.h"
#include "hbonds_ns.h"
#include "FArray_xyz_functions.h"
#include "files_paths.h"
#include "random_numbers.h"
#include "runlevel.h"
#include "LigandAtomPairEnergy.h"
#include "make_pdb.h"
#include "orient_rms.h"
#include "pack_geom_inline.h"
#include "param_aa.h"
#include "param_pack.h"
#include "pose_ligand.h"

// ObjexxFCL Headers
#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>

// Numeric Headers
#include <numeric/angle.functions.hh>
#include <numeric/constants.hh>
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.io.hh> //kwk needed only for debugging

// Utility Headers
#include <utility/io/ozstream.hh>

// C++ standard template headers
#include <vector>
#include <cmath>
#include <list>
#include <map>

// Using
using namespace param;



//==============================================================================
//
//     ligand_ns
//
//		Jens Meiler April 2003
//		Kristian Kaufmann November 2005
//
//==============================================================================

//mj fields for hetero atoms, introduced for loop building and ligand docking
//mj and side chain packing for loop building only a simplified clash check is
//mj performed.  for this all atoms are mapped on atom types
//mj 1=N,2=CA,3=CB,4=C,5=O in hetero_atom_cent for docking/packing ROSETTA full
//mj atom types are necesarry => all atoms need to be mapped on the 25 ROSETTA
//mj full atom types in hetero_atom_type! this type is obtained from the PDB
//mj name of the atom.

// Default Constructor for Ligand object
	Ligand::Ligand(){
		attach_by_jump=true;
		attach_by_bond=false;
		use_only_defined_bonds=false;
		use_input_ligand_only=false;
		read_ligand_conformations=false;
		use_iterative_conformation_search=false;
		rotamerize_input_ligand=false;
		make_rotamer_nearest_start=false;
		ligand_conformations_infile="lig.sd";
		fix_ligand_HO=false;
		use_input_ligand=false;
		output_ligand_conformations=false;
		superimpose_output_rotamers=false;
		recompute_ligand_interface=true;
		rotamerize_ligand_lower_bound=0.0;
		rotamerize_ligand_upper_bound=1000000.0;
		delta_angular_sampling=5;
		minimal_rms_limit=0.5;
		maximum_number_of_conformations=100;
		hetero_born_radius.dimension(param::HETERO_ATOM_MAX());
		virtual_atom_bnd.dimension( param::HETERO_ATOM_MAX());
		virtual_atom_map.dimension( param::HETERO_ATOM_MAX());
		hetero_atom_resid.dimension( param::HETERO_ATOM_MAX(),
			std::string( 3, ' ' ) );
		don_base_if_OH.dimension( param::HETERO_ATOM_MAX() );
		donor_satisfied.dimension( param::HETERO_ATOM_MAX(), false );
		acceptor_satisfied.dimension( param::HETERO_ATOM_MAX(), false);
		hetero_atom_hbond_acc.dimension( param::HETERO_ATOM_MAX() );
		hetero_atom_hbond_don.dimension( param::HETERO_ATOM_MAX() );
		anchor_state=0;

		lig_repE=0.0;
		lig_atrE=0.0;
		lig_solE=0.0;
		lig_couE=0.0;
		lig_h2oE=0.0;
		lig_hbE=0.0;
		lig_virE=0.0;
		lig_sumE=0.0;
		lig_sasE=0.0;

		lig_int_repE=0.0;
		lig_int_atrE=0.0;
		lig_int_solvE=0.0;
		lig_int_dihedE=0.0;
		lig_int_coulombE=0.0;
		lig_int_hbE=0.0;

		virtual_atom_count = 0;
		hetero_atom_total_charge = float( 0);
		hetero_atom_hbond_acc_count = 0;
		hetero_atom_hbond_don_count = 0;
		anchor_first.assign(0.0,0.0,0.0);
		anchor_second.assign(0.0,0.0,0.0);
		anchor_third.assign(0.0,0.0,0.0);
		anchor_first_best.assign(0.0,0.0,0.0);
		anchor_second_best.assign(0.0,0.0,0.0);
		anchor_third_best.assign(0.0,0.0,0.0);
		anchor_first_start.assign(0.0,0.0,0.0);
		anchor_second_start.assign(0.0,0.0,0.0);
		anchor_third_start.assign(0.0,0.0,0.0);
		anchor_state=0;
	}

	// Copy Constructor for Ligand object
	Ligand::Ligand( Ligand & p){
		attach_by_jump=p.attach_by_jump;
		attach_by_bond=p.attach_by_bond;
		use_only_defined_bonds=p.use_only_defined_bonds;
		fix_ligand_HO=p.fix_ligand_HO;
		use_input_ligand=p.use_input_ligand;
		use_input_ligand_only=p.use_input_ligand_only;
		use_iterative_conformation_search=p.use_iterative_conformation_search;
		rotamerize_input_ligand=p.rotamerize_input_ligand;
		make_rotamer_nearest_start=p.make_rotamer_nearest_start;
		maximum_number_of_conformations=p.maximum_number_of_conformations;
		read_ligand_conformations=p.read_ligand_conformations;
		ligand_conformations_infile=p.ligand_conformations_infile;
		recompute_ligand_interface=p.recompute_ligand_interface;
		rotamerize_ligand_lower_bound=p.rotamerize_ligand_lower_bound;
		rotamerize_ligand_upper_bound=p.rotamerize_ligand_upper_bound;
		delta_angular_sampling=p.delta_angular_sampling;
		minimal_rms_limit=p.minimal_rms_limit;
		ligand_copy_trans_rot=p.ligand_copy_trans_rot;
		ligand_conformations=p.ligand_conformations;
		ligand_conformations_base=p.ligand_conformations_base;
		ligand_conformations_outfile=p.ligand_conformations_outfile;
		output_ligand_conformations=p.output_ligand_conformations;
		superimpose_output_rotamers=p.superimpose_output_rotamers;
		hetero_born_radius=p.hetero_born_radius;
		virtual_atom_bnd=p.virtual_atom_bnd;
		virtual_atom_map=p.virtual_atom_map;
		hetero_atom_resid=p.hetero_atom_resid;
		hetero_atom_hbond_acc=p.hetero_atom_hbond_acc;
		hetero_atom_hbond_don=p.hetero_atom_hbond_don;
		atom_base=p.atom_base;
		abase2=p.abase2;
		don_base_if_OH=p.don_base_if_OH;
		donor_satisfied=p.donor_satisfied;
		acceptor_satisfied=p.acceptor_satisfied;
		lig_repE=p.lig_repE;
		lig_atrE=p.lig_atrE;
		lig_solE=p.lig_solE;
		lig_couE=p.lig_couE;
		lig_h2oE=p.lig_h2oE;
		lig_hbE=p.lig_hbE;
		lig_virE=p.lig_virE;
		lig_sumE=p.lig_sumE;
		lig_sasE=p.lig_sasE;

		lig_int_repE=p.lig_int_repE;
		lig_int_atrE=p.lig_int_atrE;
		lig_int_solvE=p.lig_int_solvE;
		lig_int_dihedE=p.lig_int_dihedE;
		lig_int_coulombE=p.lig_int_coulombE;
		lig_int_hbE=p.lig_int_hbE;

		virtual_atom_count=p.virtual_atom_count;
		hetero_atom_total_charge=p.hetero_atom_total_charge;
		hetero_atom_hbond_acc_count=p.hetero_atom_hbond_acc_count;
		hetero_atom_hbond_don_count=p.hetero_atom_hbond_don_count;
		anchor_first_best=p.anchor_first_best;
		anchor_second_best=p.anchor_second_best;
		anchor_third_best=p.anchor_third_best;
		anchor_first_start=p.anchor_first_start;
		anchor_second_start=p.anchor_second_start;
		anchor_third_start=p.anchor_third_start;
		anchor_first=p.anchor_first;
		anchor_second=p.anchor_second;
		anchor_third=p.anchor_third;
		anchor_state=p.anchor_state;

		for ( size_t i=0; i<p.atom_vector.size(); i++){
			atom_vector.push_back(new Atom(*(p.atom_vector[i])));
		}
		for ( size_t i=0; i<p.best_coord.size(); i++){
			best_coord.push_back(p.best_coord[i]);
		}
		for ( size_t i=0; i<p.best_bonds.size(); i++){
			start_coord.push_back(p.best_bonds[i]);
		}
		for ( size_t i=0; i<p.start_bonds.size(); i++){
			start_coord.push_back(p.start_bonds[i]);
		}
		for ( size_t i=0; i<p.start_coord.size(); i++){
			start_coord.push_back(p.start_coord[i]);
		}

		std::list<numeric::xyzVector < size_t > > bond_map;
		p.get_bond_map(bond_map);
		setup_bond_graph( bond_map,false);
		calculate_internal_coordinates(true);
	}
	Ligand & Ligand::operator=(Ligand & p){
		if(this!=&p){
			attach_by_jump=p.attach_by_jump;
			attach_by_bond=p.attach_by_bond;
			use_only_defined_bonds=p.use_only_defined_bonds;
			fix_ligand_HO=p.fix_ligand_HO;
			use_input_ligand=p.use_input_ligand;
			use_input_ligand_only=p.use_input_ligand_only;
			use_iterative_conformation_search=p.use_iterative_conformation_search;
			rotamerize_input_ligand=p.rotamerize_input_ligand;
			make_rotamer_nearest_start=p.make_rotamer_nearest_start;
			read_ligand_conformations=p.read_ligand_conformations;
			ligand_conformations_infile=p.ligand_conformations_infile;
			recompute_ligand_interface=p.recompute_ligand_interface;
			rotamerize_ligand_lower_bound=p.rotamerize_ligand_lower_bound;
			rotamerize_ligand_upper_bound=p.rotamerize_ligand_upper_bound;
			delta_angular_sampling=p.delta_angular_sampling;
			minimal_rms_limit=p.minimal_rms_limit;
			maximum_number_of_conformations=p.maximum_number_of_conformations;
			ligand_copy_trans_rot=p.ligand_copy_trans_rot;
			ligand_conformations=p.ligand_conformations;
			ligand_conformations_base=p.ligand_conformations_base;
			ligand_conformations_outfile=p.ligand_conformations_outfile;
			output_ligand_conformations=p.output_ligand_conformations;
			superimpose_output_rotamers=p.superimpose_output_rotamers;
			hetero_born_radius=p.hetero_born_radius;
			virtual_atom_bnd=p.virtual_atom_bnd;
			virtual_atom_map=p.virtual_atom_map;
			hetero_atom_resid=p.hetero_atom_resid;
			hetero_atom_hbond_acc=p.hetero_atom_hbond_acc;
			hetero_atom_hbond_don=p.hetero_atom_hbond_don;
    	atom_base=p.atom_base;
    	abase2=p.abase2;
			don_base_if_OH=p.don_base_if_OH;
    	donor_satisfied=p.donor_satisfied;
    	acceptor_satisfied=p.acceptor_satisfied;
			lig_repE=p.lig_repE;
			lig_atrE=p.lig_atrE;
			lig_solE=p.lig_solE;
			lig_couE=p.lig_couE;
			lig_h2oE=p.lig_h2oE;
			lig_hbE=p.lig_hbE;
			lig_virE=p.lig_virE;
			lig_sumE=p.lig_sumE;
			lig_sasE=p.lig_sasE;

			lig_int_repE=p.lig_int_repE;
			lig_int_atrE=p.lig_int_atrE;
			lig_int_solvE=p.lig_int_solvE;
			lig_int_dihedE=p.lig_int_dihedE;
			lig_int_coulombE=p.lig_int_coulombE;
			lig_int_hbE=p.lig_int_hbE;

			virtual_atom_count=p.virtual_atom_count;
			hetero_atom_total_charge=p.hetero_atom_total_charge;
			hetero_atom_hbond_acc_count=p.hetero_atom_hbond_acc_count;
			hetero_atom_hbond_don_count=p.hetero_atom_hbond_don_count;
			anchor_first_best=p.anchor_first_best;
			anchor_second_best=p.anchor_second_best;
			anchor_third_best=p.anchor_third_best;
			anchor_first_start=p.anchor_first_start;
			anchor_second_start=p.anchor_second_start;
			anchor_third_start=p.anchor_third_start;
			anchor_first=p.anchor_first;
			anchor_second=p.anchor_second;
			anchor_third=p.anchor_third;
			anchor_state=p.anchor_state;

			for ( size_t i=0; i<p.atom_vector.size(); i++){
				atom_vector.push_back(new Atom(*(p.atom_vector[i])));
			}
			for ( size_t i=0; i<p.best_coord.size(); i++){
				best_coord.push_back(p.best_coord[i]);
			}
			for ( size_t i=0; i<p.start_coord.size(); i++){
				start_coord.push_back(p.start_coord[i]);
			}
			for ( size_t i=0; i<p.best_bonds.size(); i++){
				start_coord.push_back(p.best_bonds[i]);
			}
			for ( size_t i=0; i<p.start_bonds.size(); i++){
				start_coord.push_back(p.start_bonds[i]);
			}

			std::list<numeric::xyzVector < size_t > > bond_map;
			p.get_bond_map(bond_map);
			setup_bond_graph( bond_map,false);

			calculate_internal_coordinates(true);



			return *this;
		}
		else{

			return *this;
		}

	}


	// Destructor
	Ligand::~Ligand(){
		for(std::vector<ligand::Ligand_AtomOP >::iterator c_atom=atom_vector.begin();
			c_atom!=atom_vector.end(); c_atom++){
		// Deletes memory associated with Atom * (*c_atom)
			delete (*c_atom);
		}
		atom_vector.clear();
		for(std::vector<ligand::Ligand_BondAP>::iterator c_bond=bond_vector.begin();
			c_bond!=bond_vector.end(); c_bond++){
		// Deletes memory associated with Atom * (*c_atom)
			delete (*c_bond);
		}
		bond_vector.clear();

	}

// =============================================================================
//     function to transfer trail coordinates to FArray2D_float
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_FArray2D_of_coordinates
///
/// @brief
///		Function for transfer of coordinates to FArray2D for compatibility
///		with other rosetta functions
///
/// @detailed
///
/// @param
///		in standard vector of pointers to Atoms
///		out FArray2D_float (passed by reference)
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified 11-15-2005
////////////////////////////////////////////////////////////////////////////////
	void
	Ligand::get_FArray2D_of_coordinates(
		std::vector < ligand::Ligand_AtomOP > & ligand_atom,
		FArray2D_float & het_coord
	){
		het_coord.dimension(3,ligand_atom.size());
		std::vector< ligand::Ligand_AtomOP >::const_iterator i;
		int j=1;
		for( i=ligand_atom.begin(); i !=ligand_atom.end();i++){
			FArray1D_float c_coord(3);
			copy_to_FArray(
				(*i)->get_coordinates(), c_coord);
			for(int k=1;k<4;k++){
				het_coord(k,j)=c_coord(k);
			}
			j++;
		}
		return ;
	}

// =============================================================================
//     void to supply FArray1D of heteroatom charges
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_FArray1D_of_charge
///
/// @brief
///     void to supply FArray of charges required by some Rosetta functions
///
/// @detailed
///
/// @param
///		in standard vector of pointers to Atoms
///		out FArray1D_float (passed by reference)
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Krsitian Kaufmann
/// @last_modified	11-15-2005
////////////////////////////////////////////////////////////////////////////////
	void
	Ligand::get_FArray1D_of_charge(
		std::vector < ligand::Ligand_AtomOP > & ligand_atom,
		FArray1D_float & het_charge
	){
		het_charge.dimension(ligand_atom.size());
		for (size_t j=0;j<ligand_atom.size();++j){
			het_charge(j+1)=float(ligand_atom[j]->get_partial_charge());
		}
		return;
	}
// =============================================================================
//     void to supply an FArray1D_int of atom types
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_FArray1D_of_atom_type
///
/// @brief
///     supplies FArray of atom types for use in Rosetta Functions
///
/// @detailed
///
/// @param
///		in standard vector of pointers to Atoms
///		out FArray1D_int (passed by reference)
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	11-15-2005
////////////////////////////////////////////////////////////////////////////////
void
Ligand::get_FArray1D_of_atom_type(
	std::vector < ligand::Ligand_AtomOP > & ligand_atom,
	FArray1D_int & het_atom_type
){
	het_atom_type.dimension(ligand_atom.size());
	for (size_t j=0; j< ligand_atom.size(); ++j){
		het_atom_type(j+1)=ligand_atom[j]->get_ros_atom_type();
	}
	return;
}

// =============================================================================
//     int to setup_bond_graph
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin setup_bond_graph
///
/// @brief
///		given a nonempty array of atom in atom_vector vector
///		will return
///
/// @detailed
///
/// @param
/// @global_read
///		reads from atom_vector
/// @global_write
///		writes to bond_vector
///		writes to trial_roots
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	5-17-2007
////////////////////////////////////////////////////////////////////////////////
int
Ligand::setup_bond_graph(
	int const root_atom_index
){

 	assert( !atom_vector.empty() );
	bond_vector.clear();
	trial_roots.clear();
	

	//When setting up the bond graph we want to ensure that all hydrogens and virtual atoms are
	// leaves in the graph. Meaning the there is only one bond to all hydrogens and virtual atoms
	// we accomplishes this by first setting up a minimal spanning tree across all the heavy atoms
  // then we attach all hydrogens and virtual atoms to the nearest heavy atom
	std::map< size_t, numeric::xyzVector< double > > coordinates;
	for( size_t atom_index=0; atom_index < atom_vector.size(); atom_index++){

		//first find the minimal spanning graph across all heavy atoms
		if( atom_vector[atom_index]->get_element_type() != atom_chem::HYDROGEN &&
					!atom_vector[atom_index]->Is_Virtual()){
			std::pair< size_t, numeric::xyzVector< double > >
					p_coord( atom_index, atom_vector[atom_index]->get_coordinates());
			coordinates.insert(p_coord);
		}
	}
	std::list < numeric::xyzVector< size_t > > bond_tree;
	size_t start_vertex=0;
	if( root_atom_index > 0 && root_atom_index <= int(coordinates.size())){
		start_vertex=size_t(root_atom_index);
	}
	get_minimal_spanning_bond_tree_from_coordinates(coordinates,
		bond_tree, start_vertex);

	//now connect Hydrogens and Virtual atoms to the nearest heavy_atom
	size_t index_counter=0;
	for( ligand::Atom_Itr atom_iter=atom_vector.begin();
		atom_iter!=atom_vector.end(); atom_iter++){
		if( (*atom_iter)->get_element_type() == atom_chem::HYDROGEN ||
			(*atom_iter)->Is_Virtual()){

			size_t heavy_atom=get_nearest_heavy_atom((*atom_iter));
			numeric::xyzVector< size_t > temp_bond_def(heavy_atom, index_counter, 1);
			bond_tree.push_back(temp_bond_def);
		}
		index_counter++;
	}
	setup_bond_graph( bond_tree, root_atom_index);
	set_bondtypes_by_distance();
	for(ligand::Bond_Itr current_bond=bond_vector.begin();
		current_bond!=bond_vector.end(); current_bond++){

		(*current_bond)->set_rotability(false);
	}

	std::cout << "Exit setup_bond_graph." << std::endl;
	return 1;

}

//This function sets the bond_bondtypes by the distances of the
// bonds this function is not recommended for general use
// but is implemented to help when no other option is available

int
Ligand::set_bondtypes_by_distance(
){

calculate_internal_coordinates(true);

for( ligand::Bond_Itr cbond=bond_vector.begin(), ebond=bond_vector.end();
	cbond!=ebond; cbond++){

	ligand::Ligand_AtomAP atom1=NULL;
	ligand::Ligand_AtomAP atom2=NULL;
	(*cbond)->bonded_atoms(atom1, atom2);
	atom_chem::ElementType atom1element=atom1->get_element_type();
	atom_chem::ElementType atom2element=atom2->get_element_type();
	float atom_distance = (*cbond)->get_length();
	if( atom_distance < atom_chem::get_max_bond_length(atom1element, atom2element)){

		(*cbond)->set_bondtype(atom_chem::SINGLE);
		if( atom1element == atom_chem::CARBON &&
				atom2element == atom_chem::CARBON){
					if( atom_distance < 1.45 && 1.38 < atom_distance ){
						(*cbond)->set_bondtype(atom_chem::AROMATIC);
					}else if( atom_distance < 1.38 &&
						1.26 < atom_distance){
						(*cbond)->set_bondtype(atom_chem::DOUBLE);
					}else if( atom_distance < 1.26){
						(*cbond)->set_bondtype(atom_chem::TRIPLE);
					}
		}// Carbon Carbon
		else if(( atom1element== atom_chem::CARBON &&
			atom2element== atom_chem::NITROGEN ) || (
			atom2element== atom_chem::CARBON &&
			atom1element== atom_chem::NITROGEN)){
			if( atom_distance < 1.40 && 1.20 < atom_distance ){
					(*cbond)->set_bondtype(atom_chem::AROMATIC);
			}else if( atom_distance < 1.20 &&
				1.18 < atom_distance ){
					(*cbond)->set_bondtype(atom_chem::DOUBLE);
			}else if( atom_distance < 1.18){
				(*cbond)->set_bondtype(atom_chem::TRIPLE);
			}
		}// Carbon Nitrogen bond
		else if(( atom1element== atom_chem::CARBON &&
			atom2element== atom_chem::OXYGEN ) || (
			atom2element== atom_chem::CARBON &&
			atom1element== atom_chem::OXYGEN)){
			if( atom_distance < 1.30){
				(*cbond)->set_bondtype(atom_chem::DOUBLE);
			}
		}// Carbon Oxygen bond
		else if(( atom1element== atom_chem::PHOSPHORUS
			&& atom2element== atom_chem::OXYGEN ) || (
			atom2element== atom_chem::PHOSPHORUS &&
			atom1element== atom_chem::OXYGEN)){
			if( atom_distance < 1.55){
				(*cbond)->set_bondtype(atom_chem::DOUBLE);
			}
		}// Phosphorus Oxygen bond
		else if(( atom1element== atom_chem::SULFUR &&
			atom2element== atom_chem::NITROGEN ) || (
			atom2element== atom_chem::SULFUR &&
			atom1element== atom_chem::NITROGEN)){
			if( atom_distance < 1.65){
				(*cbond)->set_bondtype(atom_chem::DOUBLE);
			}
		}// Sulfur Nitrogen bond
		else if(( atom1element== atom_chem::SULFUR &&
			atom2element== atom_chem::OXYGEN ) || (
			atom2element== atom_chem::SULFUR &&
			atom1element== atom_chem::OXYGEN)){
			if( atom_distance < 1.60){
				(*cbond)->set_bondtype(atom_chem::DOUBLE);
			}
		}// Sulfur Oxygen bond
		else if( atom1element== atom_chem::NITROGEN &&
			atom2element== atom_chem::NITROGEN ){
			if( atom_distance < 1.23){
				(*cbond)->set_bondtype(atom_chem::DOUBLE);}
			}// Nitrogen-Nitrogen
		else if(( atom1element== atom_chem::NITROGEN &&
			atom2element==atom_chem::OXYGEN ) || (
			atom2element== atom_chem::NITROGEN &&
			atom1element== atom_chem::OXYGEN)){
			if( atom_distance < 1.19){
				(*cbond)->set_bondtype(atom_chem::DOUBLE);
			}
		}// Oxygen-Nitrogen
	}else{
		(*cbond)->set_bondtype(atom_chem::NO_BOND);
	}
}

return 1;
}

//the following function implements a verison of Prim's
// algorithm for finding the minimal spanning set connecting
// all atoms in the ligand based on the distance between the coordinates

int
Ligand::get_minimal_spanning_bond_tree_from_coordinates(
	std::map < size_t, numeric::xyzVector < double > > & coordinates,
	std::list < numeric::xyzVector < size_t > > & bond_tree,
	size_t start_vertex
){

//The approach I am taking is to iteratively add all distances < 5 A
// to the latest added node into a multimap which is a container sorted on Key
// The next node to add will be the node with the shortest distance between
// nodes not already added.

std::multimap< double, std::pair< size_t,size_t > > distances;
std::vector< bool > added_to_tree( atom_vector.size(), false);
assert( start_vertex < atom_vector.size());

//First load the distances between start_vertex and all other vertices
double dist=0;
float max_bond_length=5;
added_to_tree[start_vertex]=true;
bool done=false;
while( !done ){
	for( std::map< size_t, numeric::xyzVector < double > >::iterator coord_iter=
		coordinates.begin(); coord_iter!=coordinates.end(); coord_iter++){

		if( !added_to_tree[(*coord_iter).first] && ((*coord_iter).second.x()-
			coordinates[start_vertex].x()) < max_bond_length ){
			if( ((*coord_iter).second.y()-coordinates[start_vertex].y()) < max_bond_length ){
					dist=distance((*coord_iter).second,coordinates[start_vertex]);
				if( dist  < max_bond_length ){
					if( start_vertex > (*coord_iter).first ){
						std::pair< size_t, size_t > edge((*coord_iter).first,start_vertex);
						std::pair< double, std::pair<size_t,size_t> > p(dist, edge);
						distances.insert(p);
					}else{
						std::pair< size_t, size_t > edge(start_vertex,(*coord_iter).first);
						std::pair< double, std::pair<size_t,size_t> > p(dist, edge);
						distances.insert(p);
					}//else
				}//if distance
			}//if delta y
		}//if delta x
	}//for coord
	bool found_distance=false;
	while(!found_distance && !distances.empty()){
		if( added_to_tree[((*distances.begin()).second).first] &&
				added_to_tree[((*distances.begin()).second).second] ){

			distances.erase(distances.begin());
		}else{
			found_distance=true;
			if( added_to_tree[((*distances.begin()).second).first]){
				start_vertex=((*distances.begin()).second).second;
				added_to_tree[start_vertex]=true;
				numeric::xyzVector< size_t > bond( ((*distances.begin()).second).first,
					((*distances.begin()).second).second,0);
				bond_tree.push_back(bond);
			}else{
				start_vertex=((*distances.begin()).second).first;
				added_to_tree[start_vertex]=true;
				numeric::xyzVector< size_t > bond( ((*distances.begin()).second).first,
					((*distances.begin()).second).second,0);
				bond_tree.push_back(bond);
			}//else
		}//else
	}//while found_distance
	if( found_distance ){
		max_bond_length=5;
	}else{
		max_bond_length+=(max_bond_length*1.5);
	}//else found_distance
	if( bond_tree.size() == coordinates.size()-1){
		done=true;
	}
	if( runlevel_ns::runlevel >= runlevel_ns::verbose){
		std::cout << "number of bonds found " << bond_tree.size() << std::endl;
	}
}//while (!done)
	assert( bond_tree.size() == coordinates.size()-1);
 return 1;
}

//=============================================================================
// size_t get_nearest_heavy_atom
//=============================================================================
// This function finds the nearest heavy atom in the ligand and returns its
// index in the atom_vector

size_t
Ligand::get_nearest_heavy_atom(
	ligand::Ligand_AtomAP atom_i
){
	numeric::xyzVector< double > atom_i_coord=atom_i->get_coordinates();
	double dist=100000.0;
	double min_dist=10000.0;
	//this intialize nearest atom index to an impossible value
	size_t nearest_atom_index=atom_vector.size();
	size_t index_counter=0;
	for( ligand::Atom_Itr atom_iter=atom_vector.begin();
		atom_iter!=atom_vector.end(); atom_iter++){

		//first find the minimal spanning graph across all heavy atoms
		if( (*atom_iter)->get_element_type() != atom_chem::HYDROGEN &&
			!(*atom_iter)->Is_Virtual()){

			dist=distance((*atom_iter)->get_coordinates(),atom_i_coord);
			if( dist < min_dist){
				min_dist=dist;
				nearest_atom_index=index_counter;
			}
		}
		index_counter++;
	}

	assert( nearest_atom_index < atom_vector.size());
	return nearest_atom_index;

}

// =============================================================================
//     int to setup_bond_graph(bond_map)
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin setup_bond_graph
///
/// @brief
///		given a nonempty array of atom in atom_vector vector
///		will return
///
/// @detailed
///
/// @param -[in]- vector of xyzVector<size_t> bond_map
/// @global_read
///		reads from atom_vector
/// @global_write
///		writes to bond_vector
///		writes to trial_roots
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	2-9-2006
////////////////////////////////////////////////////////////////////////////////
int
Ligand::setup_bond_graph(
	std::list< numeric::xyzVector < size_t > > & bond_map,
	int root_atom
){
	size_t i, index;
	i=0;
	index=1;
	numeric::xyzVector<size_t> current_bond;
	std::list< size_t > atom_branch;
	if(root_atom > 0 && size_t(root_atom) < atom_vector.size()){
		atom_branch.push_back(root_atom);
	}else{
		atom_branch.push_back(get_atom_closest_to_centroid());
	}
	if( runlevel_ns::runlevel >= runlevel_ns::inform ){
		std::cout << "root_atom: " << atom_vector[atom_branch.back()]->get_pdb_number() << std::endl;
	}
	atom_vector[atom_branch.back()]->set_index(index);
	index++;
	bool bonds_left=true;
	bool found_bond=false;
	std::list< numeric::xyzVector< size_t > >::iterator
			bond_index=bond_map.begin();
	std::cout << "Setting up Chemical bond graph" << std::endl;
	while(bonds_left){
		i=0;
		std::list< numeric::xyzVector< size_t > >::iterator
			bond_index=bond_map.begin();
//		std::cout << "begin again" << std::endl;
		while(!found_bond && i<bond_map.size() ){
			current_bond=(*bond_index);
			if( current_bond.x()>=atom_vector.size() ||
			    current_bond.y()>=atom_vector.size() ||
			    current_bond.z()> 8){
				std::cout <<" An error was found in the values of the bond_map";
				std::cout << std::endl << "The values are as follows ";
				std::cout << current_bond.x() << " " << current_bond.y() << " ";
				std::cout << current_bond.z() << " from record" << i+1 << std::endl;
			}else if(atom_branch.back()==current_bond.x() &&
					!Are_Bonded(atom_vector[current_bond.x()],
								atom_vector[current_bond.y()])){

				bond_vector.push_back( new BondSmallMolecule(atom_vector[current_bond.x()],
						 atom_vector[current_bond.y()]));
				found_bond=true;
				if(atom_vector[current_bond.y()]->get_index()==9999){
					atom_vector[current_bond.y()]->set_index(index);
					index++;
				}

				if(find(atom_branch.begin(),atom_branch.end(),current_bond.y())
						== atom_branch.end()){
					atom_branch.push_back(current_bond.y());
//					this if block ensures that the depth first search doesn't
//					step back onto a parent_atom but rather continues the search
//					even though it made a bond.
				}
///TODO change to switch statement kwk
				if(current_bond.z()==0){
					(bond_vector.back())->set_bondtype(atom_chem::NO_BOND);
				}else if (current_bond.z()==1){
					(bond_vector.back())->set_bondtype(atom_chem::SINGLE);
				}else if(current_bond.z()==2){
					(bond_vector.back())->set_bondtype(atom_chem::DOUBLE);
				}else if(current_bond.z()==3){
					(bond_vector.back())->set_bondtype(atom_chem::TRIPLE);
				}else if(current_bond.z()==4){
					(bond_vector.back())->set_bondtype(atom_chem::AROMATIC);
				}else if(current_bond.z()==5){
					(bond_vector.back())->set_bondtype(atom_chem::CONJUGATED);
				}else if(current_bond.z()==6){
					(bond_vector.back())->set_bondtype(atom_chem::HYDROGEN_BOND);
				}else if(current_bond.z()==7){
					(bond_vector.back())->set_bondtype(atom_chem::METALLIC_BOND);
				}else if(current_bond.z()==8){
					(bond_vector.back())->set_bondtype(atom_chem::UNDEFINED_BOND_TYPE);
				}
				bond_map.remove(current_bond);
			}else if((atom_branch.back())==current_bond.y() && !Are_Bonded(atom_vector[current_bond.x()],
								atom_vector[current_bond.y()])){

				bond_vector.push_back( new BondSmallMolecule(atom_vector[current_bond.y()],
						 atom_vector[current_bond.x()]));
				found_bond=true;
				if(atom_vector[current_bond.x()]->get_index()==9999){
					atom_vector[current_bond.x()]->set_index(index);
					index++;
				}

				if(find(atom_branch.begin(),atom_branch.end(),current_bond.x())
						== atom_branch.end()){
					atom_branch.push_back(current_bond.x());
//					this if block ensures that the depth first search doesn't
//					step back onto a parent_atom but rather continues the search
//					even though it made a bond.
				}
///TODO change to switch statement kwk
				if(current_bond.z()==0){
					(bond_vector.back())->set_bondtype(atom_chem::NO_BOND);
				}else if (current_bond.z()==1){
					(bond_vector.back())->set_bondtype(atom_chem::SINGLE);
				}else if(current_bond.z()==2){
					(bond_vector.back())->set_bondtype(atom_chem::DOUBLE);
				}else if(current_bond.z()==3){
					(bond_vector.back())->set_bondtype(atom_chem::TRIPLE);
				}else if(current_bond.z()==4){
					(bond_vector.back())->set_bondtype(atom_chem::AROMATIC);
				}else if(current_bond.z()==5){
					(bond_vector.back())->set_bondtype(atom_chem::CONJUGATED);
				}else if(current_bond.z()==6){
					(bond_vector.back())->set_bondtype(atom_chem::HYDROGEN_BOND);
				}else if(current_bond.z()==7){
					(bond_vector.back())->set_bondtype(atom_chem::METALLIC_BOND);
				}else if(current_bond.z()==8){
					(bond_vector.back())->set_bondtype(atom_chem::UNDEFINED_BOND_TYPE);
				}
  				bond_map.remove(current_bond);
			}else{
			i++;
			bond_index++;
			if(i==bond_map.size()){
				atom_branch.pop_back();
				assert( atom_branch.back() < atom_vector.size());
			}}

			if(atom_branch.empty() && !bond_map.empty()){
				bool found_atom=false;
				size_t j=0;
				do{
					if(atom_vector[j]->get_index()==9999 &&	 !found_atom){
						atom_vector[j]->set_index(index);
						index++;
						atom_branch.push_back(j);
						found_atom=true;
					}
					j++;
				}while(!found_atom && j<atom_vector.size());
			}// ensures that all atoms are checked for bonds
		}//found_bond
		if(bond_map.empty()){
			bonds_left=false;
		}else{
			found_bond=false;
		}
	}//bonds_left

  // the following for loop checks for root atoms.
	std::cout << "Checking for multiple ligands." << std::endl;
	for (ligand::Atom_Itr start_atom=atom_vector.begin();
		start_atom!=atom_vector.end();
		start_atom++){
		if ((*start_atom)->Is_Root_Atom()){trial_roots.push_back((*start_atom));}
	}
	if( runlevel_ns::runlevel >= runlevel_ns::inform ){
	std::cout << "There are "<< trial_roots.size() << " fragments and the first root atom is " << (*trial_roots.begin())->get_pdb_number() << std::endl;
	};
	//following loops set ring structures rigid based on the hack in the bond
	// constructor that sets rotable false when a bond connects to branches
	// bond tree has constructed. The implementation envisioned below relies
	// on the depth first bond search implemented above. More to the point
	// if the branch connecting bond doesn't occur between a child atom and
	// another atom further towards the root on the same branch the algorithm
	// won't work.

	std::vector< ligand::Ligand_BondAP > merging_bonds;
	for(ligand::Bond_Itr current_bond=bond_vector.begin();
		current_bond!=bond_vector.end();
		current_bond++){
		if (!(*current_bond)->IsRotable()){
			ligand::Ligand_AtomAP patom1, patom2;
			(*current_bond)->bonded_atoms(patom1,patom2);
			std::cout << patom1->get_pdb_number() << " " << patom2->get_pdb_number() << std::endl;
			merging_bonds.push_back((*current_bond));
		}
	}
	std::cout << "Setting rings rigid." << std::endl;
	for(ligand::Bond_Itr current_bond=merging_bonds.begin();
		current_bond!=merging_bonds.end();
		current_bond++){
		ligand::Ligand_AtomOP patom1, patom2;
		(*current_bond)->bonded_atoms(patom1,patom2);
		std::list< ligand::Ligand_BondAP > rigid_bonds;
		rigid_bonds.clear();
		int success= set_ring_rigid((*(*current_bond)),patom2, rigid_bonds);
		if (success!=0){ std::cerr << "set_ring_rigid failed. Error message " << success << std::endl;
		}
	}

	for(ligand::Bond_Itr current_bond=bond_vector.begin();
		current_bond!=bond_vector.end(); current_bond++){
		if((*current_bond)->get_bondtype()==atom_chem::DOUBLE ||
			(*current_bond)->get_bondtype()==atom_chem::TRIPLE ||
			(*current_bond)->get_bondtype()==atom_chem::AROMATIC){
			(*current_bond)->set_rotability(false);
		}
	}

	//this for loops prunes out bonds unneccessary to internal degrees of freedom.
	for(ligand::Bond_Itr current_bond=bond_vector.begin();
		current_bond!=bond_vector.end(); current_bond++){

		if((*current_bond)->IsRotable()){
			ligand::Ligand_AtomOP atom1,atom2;
			(*current_bond)->bonded_atoms(atom1, atom2);
			if(atom1->get_element_type()==atom_chem::HYDROGEN ||
					atom2->get_element_type()==atom_chem::HYDROGEN){

					(*current_bond)->set_rotability(false);
			}else{
				//we also need to fix all bonds that lead only to hydrogens so
				//if get_bonded_atoms( atom1) has only one non hydrogen neighbor
				// or get_bonded_atoms( atom2) has only one non hydrogen neighbor
				// then set the bond rigid
				std::list< ligand::Ligand_AtomAP > b_atoms;
				get_bonded_atoms(b_atoms, atom1);
				int num_heavy_atom1=0;
				for( std::list< ligand::Ligand_AtomAP >::iterator b_atom=b_atoms.begin(); b_atom!=b_atoms.end();b_atom++){
					if( (*b_atom)->get_element_type() != atom_chem::HYDROGEN && !(*b_atom)->Is_Virtual() && (*b_atom)->get_element_type() != atom_chem::UNDEFINED_ELEMENT_TYPE ){
						num_heavy_atom1++;
					}
				}
				b_atoms.clear();
				get_bonded_atoms(b_atoms, atom2);
				int num_heavy_atom2=0;
				for( std::list< ligand::Ligand_AtomAP >::iterator b_atom=b_atoms.begin(); b_atom!=b_atoms.end();b_atom++){
					if( (*b_atom)->get_element_type() != atom_chem::HYDROGEN && !(*b_atom)->Is_Virtual() && (*b_atom)->get_element_type() != atom_chem::UNDEFINED_ELEMENT_TYPE ){
						num_heavy_atom2++;
					}
				}
				if( num_heavy_atom2 < 2 || num_heavy_atom1 < 2){
					(*current_bond)->set_rotability(false);
				}
			}//else
		}// If Is_Rotable
	}//current_bond
	for( ligand::Bond_Itr cbond=bond_vector.begin();
		cbond!=bond_vector.end();	cbond++){

		ligand::Ligand_AtomAP atom1, atom2;
		(*cbond)->bonded_atoms(atom1, atom2);
		std::pair < ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > pair1;
		if(atom1 > atom2){
			pair1.first=atom2;
			pair1.second=atom1;
		}else{
			pair1.first=atom1;
			pair1.second=atom2;
		}
		if( dihedral_minima.count( pair1) != 0){
			if( runlevel_ns::runlevel >= runlevel_ns::inform ){
				std::cout << "Setting this bond free to rotate pdb_number_atom1 ";
				std::cout << atom1->get_pdb_number() << " pdb_number_atom2 ";
				std::cout << atom2->get_pdb_number() << std::endl;
			}
			(*cbond)->set_rotability(true);
		}
	}
	if( runlevel_ns::runlevel >= runlevel_ns::inform ){
		std::cout << "atom1_pdb_number atom2_pdb_number Is_rotatable" << std::endl;
    for(ligand::Bond_Itr j=bond_vector.begin(); j!=bond_vector.end(); j++){
		  ligand::Ligand_AtomAP patom1, patom2;
		  (*j)->bonded_atoms(patom1,patom2);
		  std::cout << patom1->get_pdb_number() << " " << patom2->get_pdb_number() << " " << (*j)->IsRotable() << std::endl;
	  }
  }
	std::cout << "Bond graph setup complete." << std::endl;
	return 1;
}

// =============================================================================
//     int for get_bond_map
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_bond_map
///
/// @brief
///		constructs an mdl style bond_map from the bond_vector vector
///		that can be used to reconstruct the bond network of the ligand
///		when the ligand object is copied.
/// @detailed
///
/// @param
///
/// @global_read
///		reads from bond_vector in ligand object.
///   reads from atom_vector in ligand object.
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	11-29-2005
////////////////////////////////////////////////////////////////////////////////
int
Ligand::get_bond_map(
	std::list< numeric::xyzVector<size_t> > & bond_map
){
	ligand::Ligand_AtomOP atom_1, atom_2;
	size_t atom_1_index=1;
	size_t atom_2_index=2;
	for(ligand::Bond_Itr current_bond=bond_vector.begin();
		current_bond!=bond_vector.end(); current_bond++){
				(*current_bond)->bonded_atoms(atom_1, atom_2);

				for(size_t i=0; i<atom_vector.size();i++){
					if(atom_vector[i]==atom_1){
						atom_1_index=i;
					}
					if(atom_vector[i]==atom_2){
						atom_2_index=i;
					}
				}
				///TODO change to switch statement kwk
				if( (*current_bond)->get_bondtype()==atom_chem::NO_BOND){
					numeric::xyzVector<size_t> bond(atom_1_index,atom_2_index,0);
					bond_map.push_back(bond);
				}else if( (*current_bond)->get_bondtype()==atom_chem::SINGLE){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,1));
				}else if( (*current_bond)->get_bondtype()==atom_chem::DOUBLE){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,2));
				}else if( (*current_bond)->get_bondtype()==atom_chem::TRIPLE){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,3));
				}else if( (*current_bond)->get_bondtype()==atom_chem::AROMATIC){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,4));
				}else if( (*current_bond)->get_bondtype()==atom_chem::CONJUGATED){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,5));
				}else if( (*current_bond)->get_bondtype()==atom_chem::HYDROGEN_BOND){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,6));
				}else if( (*current_bond)->get_bondtype()==atom_chem::METALLIC_BOND){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,7));
				}else if( (*current_bond)->get_bondtype()==atom_chem::UNDEFINED_BOND_TYPE){
					bond_map.push_back(numeric::xyzVector<size_t>(atom_1_index,
						atom_2_index,8));
				}
	}
	return 1;
}






// =============================================================================
//     int for set_ring_rigid
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin set_ring_rigid
///
/// @brief
///		recursive function to set bonds rigid in a ring given that the first
///		bond connects an atom to another atom closer to the root of the tree
///		on the same branch
///
/// @detailed
///
/// @param
///
/// @global_read
///		reads from bond_vector in ligand object.
/// @global_write
///		writes to bond_vector in ligand object.
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	11-29-2005
////////////////////////////////////////////////////////////////////////////////
int
Ligand::set_ring_rigid(
	BondSmallMolecule & bond,
	ligand::Ligand_AtomOP atom2,
	std::list< ligand::Ligand_BondAP > & rigid_bonds
){
	// bond is the bond connecting the two ring closing atoms.
	// atom2 points to the the atom furhter up the branch
	// rigid_bonds stores the list of bonds traversed until
	// atom2 is found, and then is used to set bonds rigid
	ligand::Ligand_AtomAP patom1=NULL;
	ligand::Ligand_AtomAP patom2=NULL;
	ligand::Ligand_AtomAP patom3=NULL;
	ligand::Ligand_BondAP parent_bond=NULL;
	int success;
	bond.bonded_atoms(patom1,patom2);
//	std::cout << "Entering set_ring_rigid." << std::endl;
	if ( patom1==NULL){ // this should never happen
		return -1; // function failed. found root of tree before ring closure;
	}
	patom1->get_parentbond(parent_bond);
//	std::cout << "Entering set_ring_rigid 2. ";
//	std::cout << patom1->get_pdb_number() << " ";
//	std::cout << atom2->get_pdb_number() << std::endl;
	if ( parent_bond==NULL) { return -1; }// function failed see above
	parent_bond->bonded_atoms(patom3,patom1);
//	std::cout << patom3->get_pdb_number() << std::endl;
//	std::cout << "Entering set_ring_rigid 3." << std::endl;
	if(patom3==atom2){
//	std::cout << "Entering set_ring_rigid 5." << std::endl;
    rigid_bonds.push_back(&bond);
    rigid_bonds.push_back(parent_bond);
		for(std::list< ligand::Ligand_BondAP >::iterator current_bond=rigid_bonds.begin();
			current_bond!=rigid_bonds.end(); current_bond++){
			(*current_bond)->set_rotability(false);
		}
		return 0;
	}else {
//		std::cout << "Entering set_ring_rigid 4." << std::endl;
		rigid_bonds.push_back(&bond);
		success=set_ring_rigid((*parent_bond),atom2,rigid_bonds);
		if(success==0){return 0;}
	}
	return -2; // this should never happen.

}

//This function replaces detect_hetero_atom_hbond
// to account for Jack Snoeyink changes to the hbond
// scoring

int
Ligand::setup_hetero_atom_hbond_arrays(
){
// This function must fill in
//  atom_base for all atoms
//  abase2 for all acceptors
//  hetero_atom_hbond_acc for all acceptors
//  hetero_atom_hbond_don for all donors

// In an email to me (kwk) Jack Snoeyink gave these guidelines for abase2
//  and atom_base. the idea is that atom_i, atom_base(atom_i), abase2(atom_i)
//  should define a plane to be used in hydrogen bond plane definition
// atom_base refers to heavy atom adjacent to the atom_i
// abase2 refers to heavy atom adjacent to atom_i neq atom_base(atom_i)
//        else Hydrogen adjacent to atom_i
//        else atom_base(atom_base(atom_i)) neq atom_i

// must initialize
//  donor_satisfied
//  don_base_if_OH
//  acceptor_satisfied
//  hetero_atom_hbond_don_count
//  hetero_atom_hbond_acc_count

	hetero_atom_hbond_don_count=0;
	hetero_atom_hbond_acc_count=0;
	assert( !bond_vector.empty());

	std::map< ligand::Ligand_AtomAP, size_t > atom_vector_map;
	size_t counter=0;
	for( ligand::Atom_Itr atom_iter=atom_vector.begin(), end_cond=atom_vector.end();
		atom_iter!=end_cond; atom_iter++){

		std::pair< ligand::Ligand_AtomAP, size_t > p((*atom_iter), counter);
		atom_vector_map.insert(p);
		counter++;
	}


// setting atom_base for each atom
	ligand::Ligand_BondAP parent_bond=NULL;
	std::list< ligand::Ligand_AtomAP > child_atoms;
	ligand::Ligand_AtomAP parent_atom=NULL;
	ligand::Ligand_AtomAP child_atom=NULL;
	size_t counter2=0;
	for( ligand::Atom_Itr atom_iter2=atom_vector.begin(), end_cond2=atom_vector.end();
		atom_iter2!=end_cond2; atom_iter2++){

		if(!(*atom_iter2)->Is_Root_Atom()){
			(*atom_iter2)->get_parentbond(parent_bond);
			parent_bond->bonded_atoms(parent_atom, child_atom);
			std::pair< int, size_t > new_ab(counter2++,atom_vector_map[parent_atom]);
			atom_base.insert(new_ab);
		}else{
			get_bonded_atoms(child_atoms, (*atom_iter2));
			for( std::list< ligand::Ligand_AtomAP >::iterator other_atom=child_atoms.begin(),
				end_cond3=child_atoms.end(); other_atom!=end_cond3; other_atom++){

				if( (*other_atom)->get_element_type()!=atom_chem::HYDROGEN
					&& !((*other_atom)->Is_Virtual())){

					std::pair< size_t, size_t > new_ab(counter2++,atom_vector_map[(*other_atom)]);
					atom_base.insert(new_ab);
					break;
				}
			}
		}
	}
	assert( counter2 == atom_vector.size()); //this asserts that every atom
																							// has been assigned an atombase

	//setting hetero_atom_hbond_acc_count
	// hetero_atom_hbond_acc
	// abase2
	// acceptor_satisfied
	// then hetero_atom_hbond_don_count
	// hetero_atom_hbond_don_count
	// don_base_if_OH
	// don_satisfied

	int ros_atom_type=-9;
	int atom_base_type=-9;
	for( ligand::Atom_Itr atom_iter3=atom_vector.begin(), end_cond4=atom_vector.end();
		atom_iter3!=end_cond4;atom_iter3++){

		ros_atom_type=(*atom_iter3)->get_ros_atom_type();
		assert( atom_vector.size() > 2);
		// Hydrogen bond acceptor energy function requires three coordinates to define the
		// the plane of the acceptor. Until the special cases are introduced
		// to handle single ion or and diatomic ligands. The hydrogen bonding function
		// will fail.

		//setting abase2
		std::list< ligand::Ligand_AtomAP > bonded_atoms;
		get_bonded_atoms( bonded_atoms , (*atom_iter3));
		bool found_abase2=false;
		//First look for heavy atom neq atom_base of acceptor
		for( std::list< ligand::Ligand_AtomAP >::iterator c_atom=bonded_atoms.begin(),
			end_cond5=bonded_atoms.end();c_atom!=end_cond5;c_atom++){

			if( (*c_atom)->get_element_type() != atom_chem::HYDROGEN &&
				!(*c_atom)->Is_Virtual() && (*c_atom)!=atom_vector[atom_base[atom_vector_map[(*atom_iter3)]]]
			){
				found_abase2=true;
				std::pair< size_t, size_t > p(atom_vector_map[(*atom_iter3)], atom_vector_map[(*c_atom)]);
				abase2.insert(p);
				break;
			}
		}
		//Second look for Hydrogen bonded to acceptor neq atom_base
		if( !found_abase2 ){
			for( std::list< ligand::Ligand_AtomAP >::iterator c_atom=bonded_atoms.begin(),
				end_cond2=bonded_atoms.end();c_atom!=end_cond2;c_atom++){

				if( !(*c_atom)->Is_Virtual() && (*c_atom)!=atom_vector[atom_base[atom_vector_map[(*atom_iter3)]]]
				){
					found_abase2=true;
					std::pair< size_t, size_t > p(atom_vector_map[(*atom_iter3)], atom_vector_map[(*c_atom)]);
					abase2.insert(p);
					break;
				}
			}
		}
		//Third look for heavyatom bonded to atom_base of acceptor
		// neq to acceptor
		if( !found_abase2){
			bonded_atoms.clear();
			get_bonded_atoms( bonded_atoms,
				atom_vector[atom_base[atom_vector_map[(*atom_iter3)]]] );

			for( std::list< ligand::Ligand_AtomAP >::iterator c_atom=bonded_atoms.begin(),
				end_cond6=bonded_atoms.end();c_atom!=end_cond6;c_atom++){

				if( (*c_atom)->get_element_type() != atom_chem::HYDROGEN &&
					!(*c_atom)->Is_Virtual() && (*c_atom)!=(*atom_iter3)){

					found_abase2=true;
					std::pair< size_t, size_t > p(atom_vector_map[(*atom_iter3)], atom_vector_map[(*c_atom)]);
					abase2.insert(p);
					break;
				}
			}
		}
		assert( found_abase2==true);

		if( etable::fa_acceptor(ros_atom_type) || ros_atom_type == 9 ){
			// Even though NH2O ros_atom_type==9 is not an acceptor in standard
			// aa terms, atoms mapped to this atom type in ligands can accept hydrogens
			// in h-bonds

			hetero_atom_hbond_acc_count++;
			hetero_atom_hbond_acc(hetero_atom_hbond_acc_count)=atom_vector_map[(*atom_iter3)];
			acceptor_satisfied(hetero_atom_hbond_acc_count)=false;

			if( runlevel_ns::runlevel >= runlevel_ns::inform ){
				size_t acc_index =hetero_atom_hbond_acc(hetero_atom_hbond_acc_count);
				std::cout << "Hbond acceptor found ";
				std::cout << atom_vector[acc_index]->get_atom_name();
				std::cout << " pdb atom number " << atom_vector[acc_index]->get_pdb_number() << std::endl;
				std::cout << " in residue " << acc_index+1 << " ";
				std::cout << hetero_atom_resid(acc_index+1);
				std::cout << " bonded to " << atom_vector[atom_base[acc_index]]->get_atom_name();
				std::cout << " pdb atom number ";
				std::cout << atom_vector[atom_base[acc_index]]->get_pdb_number();
				std::cout << std::endl;
				std::cout << " and abase2 ";
				std::cout << atom_vector[abase2[acc_index]]->get_atom_name();
				std::cout << " pdb atom number ";
				std::cout << atom_vector[abase2[acc_index]]->get_pdb_number();
				std::cout << std::endl;

			}
		}//fa_acceptor



		if( etable::fa_hbondH( ros_atom_type ) ){
			hetero_atom_hbond_don_count++;
			hetero_atom_hbond_don(hetero_atom_hbond_don_count)=atom_vector_map[(*atom_iter3)];
			donor_satisfied(hetero_atom_hbond_don_count)=false;
			atom_base_type=atom_vector[atom_base[atom_vector_map[(*atom_iter3)]]]->get_ros_atom_type();
			if( aaproperties_pack::properties_atomtype::atom_type_name( atom_base_type) == "OH  "
				|| aaproperties_pack::properties_atomtype::atom_type_name( atom_base_type ) == "OHha"
				|| aaproperties_pack::properties_atomtype::atom_type_name( atom_base_type) == "Oice"){

				don_base_if_OH(hetero_atom_hbond_don_count)=atom_base[atom_base[atom_vector_map[(*atom_iter3)]]];
			}else{
				don_base_if_OH(hetero_atom_hbond_don_count)=-1;
			}
			if( runlevel_ns::runlevel >= runlevel_ns::inform ){
				std::cout << "Hbond donor group detected " << (*atom_iter3)->get_atom_name();
				std::cout << " pdb atom number " << (*atom_iter3)->get_pdb_number() << std::endl;
				std::cout << " in residue " << hetero_atom_resid(atom_vector_map[(*atom_iter3)]);
				std::cout << " bonded to " << atom_vector[atom_base[atom_vector_map[(*atom_iter3)]]]->get_atom_name();
				std::cout << " pdb atom number ";
				std::cout << atom_vector[atom_base[atom_vector_map[(*atom_iter3)]]]->get_pdb_number() << std::endl;
				if( don_base_if_OH(hetero_atom_hbond_don_count)!=-1){
					std::cout << " and donor bonded to ";
					std::cout << atom_vector[don_base_if_OH(hetero_atom_hbond_don_count)]->get_atom_name();
					std::cout << " pdb atom number ";
					std::cout << atom_vector[don_base_if_OH(hetero_atom_hbond_don_count)]->get_pdb_number() << std::endl;
				}
			}
		}//if fa_hbondH
	}//for each atom in atom_vector
	assert( abase2.size() == atom_vector.size() );
	//asseart that abase2 is defined for all atoms.
	return 1;

}//function



// =============================================================================
//     int for set_ligand_atom_types
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin set_ligand_atom_types
///
/// @brief
///		Sets the atom_chem::LigandAtomType of all atoms within the atom_vector
///	vector.
/// @detailed
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Kristian Kaufmann
///
/// @last_modified	3-16-2006
////////////////////////////////////////////////////////////////////////////////

int
Ligand::set_ligand_atom_types(
){
	// LigandAtomType is based on element, hybridization, and substituent identity
	// and number of substituents.
	// Element should have already been set in the ligand.cc read functions.
	// this function expects that bond_graph to have been set up.
	
	// first loop over all atoms.
	for( ligand::Atom_Itr current_atom=atom_vector.begin();
				current_atom!=atom_vector.end(); current_atom++){

		switch((*current_atom)->get_element_type()){
			case atom_chem::FLUORINE: {
					(*current_atom)->set_ligand_atom_type(atom_chem::FLUORINE_ATOM);
					break;
				}
			case atom_chem::CHLORINE: {
					(*current_atom)->set_ligand_atom_type(atom_chem::CHLORINE_ATOM);
					break;
				}
			case atom_chem::BROMINE: {
					(*current_atom)->set_ligand_atom_type(atom_chem::BROMINE_ATOM);
					break;
				}
			case atom_chem::IODINE: {
					(*current_atom)->set_ligand_atom_type(atom_chem::IODINE_ATOM);
					break;
				}
			case atom_chem::CARBON: {
					//first get hybridization based on bond types
					switch(get_hybridization((*current_atom))){
					// 0= sp3 1=sp2 2=sp
						case 0 :{
					 		switch(get_number_of_hydrogens((*current_atom))){
								case 0 :{ (*current_atom)->set_ligand_atom_type(
																atom_chem::QUATERNARY_CARBON);
													break;// # of H
									}
								case 1 :{ (*current_atom)->set_ligand_atom_type(
																atom_chem::TERTIARY_CARBON);
													break;// # # of H
									}
								case 2 :{ (*current_atom)->set_ligand_atom_type(
																atom_chem::SECONDARY_CARBON);
													break;//# of H
									}
								case 3 :
								case 4 :{ (*current_atom)->set_ligand_atom_type(
																atom_chem::PRIMARY_CARBON);
													break;//# of H
									}
								default :{ (*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
													break;//# of H
									}
					 			}
								break; // case carbon 0 hybridization
							}
					 	case 1 : {
					 		switch(get_number_of_hydrogens((*current_atom))){
								case 0 :{
													if( get_number_of_aromatic_bonds((*current_atom))==2 ||
														get_number_of_aromatic_bonds((*current_atom))==3 ){
														(*current_atom)->set_ligand_atom_type(
															atom_chem::AROMATIC_CARBON);
													}else{
														(*current_atom)->set_ligand_atom_type(
																atom_chem::ALKENE_CARBON);
													}
													break;//# of H
									}// case 0 hydrogens
								case 1 :{
													if(get_number_of_aromatic_bonds((*current_atom))==2 ){
														(*current_atom)->set_ligand_atom_type(
															atom_chem::AROMATIC_CARBON_W_HYDROGEN);
													}else{
														(*current_atom)->set_ligand_atom_type(
																atom_chem::ALKENE_CARBON_W_HYDROGEN);
													}
													break;//# of H


					 				}//case 1 hydrogens
					 		  case 2 :{
					 		  				(*current_atom)->set_ligand_atom_type(
																atom_chem::ALKENE_CARBON_W_HYDROGEN);
					 		    				break;//# of H
					 		  	}//case 2 hydrogens
					 		  default : { (*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
													break;//# of H
					 		  	}

					 			}//switch hydrogens



							break;//case carbon 1 hybridization
						}
						case 2 :{
							(*current_atom)->set_ligand_atom_type(
																atom_chem::ALKYNE_CARBON);
							break;//case carbon 1 hybridization
						}
						default :{
							(*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
							break;//default hybridization
						}

					}//switch hybridization
					break;//case carbon
				}
			case atom_chem::NITROGEN:{
				switch(get_hybridization((*current_atom))){
					case 0:{
						switch(get_number_of_hydrogens((*current_atom))){
							case 0:{
								(*current_atom)->set_ligand_atom_type(
																atom_chem::TERTIARY_NITROGEN);
								break;//# of H 0
							}
							case 1:{
								(*current_atom)->set_ligand_atom_type(
																atom_chem::SECONDARY_NITROGEN);
								break;//# of H 1
							}
							case 2:
							case 3:{
								(*current_atom)->set_ligand_atom_type(
																atom_chem::PRIMARY_NITROGEN);
								break;//# of H 2 or 3
							}
							default : {
								(*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
								break;//# of H default
							}
						}//switch number of hydrogens
						break;//0 hybridization
					}
					case 1:{
						switch(get_number_of_aromatic_bonds((*current_atom))){
							case 2:
							case 3: {
								(*current_atom)->set_ligand_atom_type(
																atom_chem::AROMATIC_NITROGEN);
								break;//# of aromatic bonds 2 or 3
							}
							default: {
								(*current_atom)->set_ligand_atom_type(
																atom_chem::IMINE_NITROGEN);
								break;//# of aromatic bonds default
							}
						}//switch aromatic bonds

						break;//1 hybridization
					}
					case 2:{
						if(get_number_of_double_bonded_oxygens((*current_atom))==2){
							(*current_atom)->set_ligand_atom_type(
																atom_chem::NITRO_NITROGEN);
							break;//hybridization 2 double bonded oxygen 2
						}else{
							(*current_atom)->set_ligand_atom_type(
																atom_chem::NITRILE_NITROGEN);
								break;//hybridization triple bond and two bonds not to oxygen
						}
					}
					case 3:{
						(*current_atom)->set_ligand_atom_type(
																atom_chem::AZIDE_NITROGEN);
						break;//hybridization 3 double bond and triple bond and
					}
					default :{
						(*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
						break;//hybridization default
					}

					}//switch hybridization
					break;//case nitrogen
				}
			case atom_chem::PHOSPHORUS:{
				if(get_number_of_double_bonded_oxygens((*current_atom))>0){
					(*current_atom)->set_ligand_atom_type(
												atom_chem::PHOSPHORUS_W_OXYGEN);

				}else{
					std::list< ligand::Ligand_BondAP > bond_children;
					(*current_atom)->get_childbonds(bond_children);
					if (bond_children.size()==3){
						(*current_atom)->set_ligand_atom_type(
												atom_chem::TERTIARY_PHOSPHORUS);
					}else{
						(*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
					}
				}
					break;//case phosphorus
				}
			case atom_chem::OXYGEN:{
					if(get_number_of_hydrogens((*current_atom))>0){
						(*current_atom)->set_ligand_atom_type(
																atom_chem::HYDROXYL_OXYGEN);
					}else{
						if(get_hybridization((*current_atom))==1){
							(*current_atom)->set_ligand_atom_type(
																atom_chem::KETONE_OXYGEN);
						}else if(get_hybridization((*current_atom))==0){
							(*current_atom)->set_ligand_atom_type(
																atom_chem::ETHER_OXYGEN);
						}else{
							(*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
						}
					}
					break;//case oxygen
				}
			case atom_chem::SULFUR:{
					switch(get_hybridization((*current_atom))){
						case 0: {
								(*current_atom)->set_ligand_atom_type(
																atom_chem::THIOL_SULFUR);
								break;
							}
						case 1:{
								if(get_number_of_double_bonded_oxygens((*current_atom))==1){
									(*current_atom)->set_ligand_atom_type(
																atom_chem::DOUBLE_BONDED_SULFUR_W_ONE_OXYGEN);
								}else{
									(*current_atom)->set_ligand_atom_type(
																atom_chem::DOUBLE_BONDED_SULFUR);
								}
								break;
							}
						case 2:{
								if(get_number_of_double_bonded_oxygens((*current_atom))==2){
									(*current_atom)->set_ligand_atom_type(
																atom_chem::SULPHONIC_SULFUR);
									break;
								}else{
									(*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
									break;
								}

							}
						default: {
								(*current_atom)->set_ligand_atom_type(
																atom_chem::UNDEFINED_ATOM_TYPE);
								break;
							}
					}//hybrization switch
					break;//case sulfur
				}
			default : {
					(*current_atom)->set_ligand_atom_type(atom_chem::UNDEFINED_ATOM_TYPE);
					break;
				}

	}
	if( runlevel_ns::runlevel >= runlevel_ns::inform ){
						std::string atom_name;
						atom_name=(*current_atom)->get_atom_name();
						std::cout << "HETATM " << atom_name << " " << (*current_atom)->get_pdb_number();
						std::cout << " was set to LigandAtomType ";
						std::cout << atom_chem::ligandatomtype_string[
														 (*current_atom)->get_ligand_atom_type()];
						std::cout << std::endl;
	}
	}
	return 1;
}


//////////////////////////////////////////////////////////////////////////////
/// @brief Sets Rosetta full-atom and centroid atom types.
///
/// @detailed
///  Based on rules by Kristian Kaufmann published on the web.
///  Uses element, hybridization, number and type of bonded atoms.
///  Expects set_ligand_atom_types() has already been called.
///
/// @param
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors Ian W. Davis
///
/// @last_modified	7 Aug 2007
////////////////////////////////////////////////////////////////////////////////
void
Ligand::set_rosetta_atom_types()
{
	//iwd Count the number of virtual atoms as we loop
	this->virtual_atom_count = 0;
	
	int const SP3 (0);
	//int const SP2 (1);
	//int const SP  (2);
	
	// loop over all atoms.
	for( ligand::Atom_Itr current_atom = atom_vector.begin();
		current_atom != atom_vector.end(); ++current_atom ) {
		
		// If the atom already has a Rosetta type assigned from PDB, skip it.
		// This is important for backwards compatibility.
		std::string curr_atom_name = (*current_atom)->get_atom_name();
		bool curr_name_ok = false;
		using namespace aaproperties_pack;
		using namespace param;
		for ( int i = 1, e = MAX_ATOMTYPES(); i <= e; ++i ) {
			if ( curr_atom_name == atom_type_name(i) ) curr_name_ok = true;
		}
		if(curr_name_ok) {
			if( runlevel_ns::runlevel >= runlevel_ns::inform){
				std::cout << "atom " << (*current_atom)->get_pdb_number();
				std::cout << " was left as type " << curr_atom_name << std::endl;
			}
			set_ros_atom_by_name( curr_atom_name, **current_atom );
			continue;
		}
		
		//{{{ Things we need to make a decision
		int const numH = get_number_of_hydrogens((*current_atom));
		int const hyb  = get_hybridization((*current_atom));
		// Bonded neighbors and iterator
		std::list< ligand::Ligand_AtomAP > nbrs;
		this->get_bonded_atoms(nbrs, *current_atom);
		std::list< ligand::Ligand_AtomAP >::iterator nbr;
		// Actual bond objects and iterator
  	std::list< ligand::Ligand_BondAP > bnds;
		(*current_atom)->get_childbonds(bnds);
		if(!(*current_atom)->Is_Root_Atom()) {
			ligand::Ligand_BondAP pnt_bond;
			(*current_atom)->get_parentbond(pnt_bond);
			bnds.push_back(pnt_bond);
		}
		std::list< ligand::Ligand_BondAP >::iterator bnd;
		//}}} Things we need to make a decision
		
		//iwd Only H, C, N, O have complicated rules.
		//iwd Everything else maps to a single atom type.
		switch((*current_atom)->get_element_type()){
			case atom_chem::HYDROGEN: {
				int num_aro_C = 0, num_NOS = 0;
				for(nbr = nbrs.begin(); nbr != nbrs.end(); ++nbr) {
					if( (*nbr)->get_ligand_atom_type() == atom_chem::AROMATIC_CARBON
					||  (*nbr)->get_ligand_atom_type() == atom_chem::AROMATIC_CARBON_W_HYDROGEN) {
						num_aro_C += 1;
					}
					atom_chem::ElementType nbr_elem = (*nbr)->get_element_type();
					if(nbr_elem == atom_chem::NITROGEN
					|| nbr_elem == atom_chem::OXYGEN
					|| nbr_elem == atom_chem::SULFUR) {
						num_NOS += 1;
					}
				}
				if(num_NOS >= 1)        set_ros_atom_by_name( "Hpol", **current_atom );
				else if(num_aro_C >= 1) set_ros_atom_by_name( "Haro", **current_atom );
				else                    set_ros_atom_by_name( "Hapo", **current_atom );
				break; // HYDROGEN
			}
			case atom_chem::CARBON: {
				if(hyb == SP3) {
					if(numH >= 3)      set_ros_atom_by_name( "CH3 ", **current_atom );
					else if(numH == 2) set_ros_atom_by_name( "CH2 ", **current_atom );
					else               set_ros_atom_by_name( "CH1 ", **current_atom );
				} else { // non-SP3
					// scan through bonds and tally neighbors
					int num_aro_nonO = 0, num_aro_N = 0, num_dbl_nonO = 0;
					for(bnd = bnds.begin(); bnd != bnds.end(); ++bnd) {
						ligand::Ligand_AtomOP a1, a2, other_atom;
						(*bnd)->bonded_atoms(a1, a2);
						if(a1 == *current_atom) other_atom = a1;
						else other_atom = a2;
						if( (*bnd)->get_bondtype() == atom_chem::DOUBLE ) {
							if( other_atom->get_element_type() != atom_chem::OXYGEN ) num_dbl_nonO += 1;
						} else if( (*bnd)->get_bondtype() == atom_chem::AROMATIC ) {
							if( other_atom->get_element_type() != atom_chem::OXYGEN ) num_aro_nonO += 1;
							if( other_atom->get_element_type() == atom_chem::NITROGEN ) num_aro_N += 1;
						}
					}//end for
					// now make a decision
					if(num_aro_nonO >= 2)      set_ros_atom_by_name( "aroC", **current_atom );
					else if(num_dbl_nonO >= 1) set_ros_atom_by_name( "aroC", **current_atom );
					else if(num_aro_N >= 1)    set_ros_atom_by_name( "CNH2", **current_atom );
					else                       set_ros_atom_by_name( "COO ", **current_atom );
				}
				break; // CARBON
			}
			case atom_chem::NITROGEN: {
				int unsat_nbrs = 0;
				for(nbr = nbrs.begin(); nbr != nbrs.end(); ++nbr) {
					if( get_hybridization(*nbr) != SP3 ) unsat_nbrs += 1;
				}
				if(hyb == SP3 && unsat_nbrs >= 1)      set_ros_atom_by_name( "NH2O", **current_atom );
				else if(hyb == SP3 && unsat_nbrs == 0) set_ros_atom_by_name( "Nlys", **current_atom );
				else                                   set_ros_atom_by_name( "Nhis", **current_atom );
				break; // NITROGEN
			}
			case atom_chem::OXYGEN: {
				bool bonded_to_N = false;
				for(nbr = nbrs.begin(); nbr != nbrs.end(); ++nbr) {
					if( (*nbr)->get_element_type() == atom_chem::NITROGEN ) bonded_to_N = true;
				}
				if(hyb == SP3)       set_ros_atom_by_name( "OH  ", **current_atom );
				else if(bonded_to_N) set_ros_atom_by_name( "ONH2", **current_atom );
				else                 set_ros_atom_by_name( "OOC ", **current_atom );
				break; // OXYGEN
			}
			//iwd{{{ Simple atom types
			case atom_chem::SULFUR: {
				set_ros_atom_by_name( "S   ", **current_atom );
				break; // SULFUR
			}
			case atom_chem::PHOSPHORUS: {
				set_ros_atom_by_name( "Phos", **current_atom );
				break; // PHOSPHORUS
			}
			case atom_chem::FLUORINE: {
				set_ros_atom_by_name( "F   ", **current_atom );
				break; // FLUORINE
			}
			case atom_chem::CHLORINE: {
				set_ros_atom_by_name( "Cl  ", **current_atom );
				break; // CHLORINE
			}
			case atom_chem::BROMINE: {
				set_ros_atom_by_name( "Br  ", **current_atom );
				break; // BROMINE
			}
			case atom_chem::IODINE: {
				set_ros_atom_by_name( "I   ", **current_atom );
				break; // IODINE
			}
			case atom_chem::SODIUM: {
				set_ros_atom_by_name( "Na1p", **current_atom );
				break; // SODIUM
			}
			case atom_chem::POTASSIUM: {
				set_ros_atom_by_name( "K1p ", **current_atom );
				break; // POTASSIUM
			}
			case atom_chem::MAGNESIUM: {
				set_ros_atom_by_name( "Mg2p", **current_atom );
				break; // MAGNESIUM
			}
			case atom_chem::IRON: {
				set_ros_atom_by_name( "Fe3p", **current_atom );
				break; // IRON
			}
			case atom_chem::CALCIUM: {
				set_ros_atom_by_name( "Ca2p", **current_atom );
				break; // CALCIUM
			}
			case atom_chem::ZINC: {
				set_ros_atom_by_name( "Zn2p", **current_atom );
				break; // ZINC
			}
			//iwd}}} Simple atom types
			default: {
				std::cout << "Exiting -- unable to assign Rosetta atom types for ligand atom ";
				std::cout << (*current_atom)->get_pdb_number() << " " << (*current_atom)->get_atom_name() << " " << atom_chem::element_type_symbol[(*current_atom)->get_element_type()] << std::endl;
				utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
			}
		}//end switch on element type
		
		//iwd Now that atom type is assigned, tally virtual atoms
		if( (*current_atom)->Is_Virtual() ) {
			this->virtual_atom_count += 1;
		}
	}
}


// =============================================================================
//     size_t get_atom_closest_to_centroid
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin size_t get_atom_closest_to_centroid
///
/// @brief
///		Calculates the centroid of the coordinates in atom_vector and returns
/// the size_t of the vector index of the closest non-hydrogen atom
/// to the centriod.
/// @detailed
///
/// @param
///
/// @global_read
///		reads from atom_vector
/// @global_write
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	2-9-2006
////////////////////////////////////////////////////////////////////////////////
size_t
Ligand::get_atom_closest_to_centroid(
){
	numeric::xyzVector<double> centroid(0.0,0.0,0.0);
	numeric::xyzVector<double> min_distance(100.0,100.0,100.0);
	numeric::xyzVector<double> cent_distance(50.0,50.0,50.0);
	size_t atom_closest = 0;
	for(ligand::Atom_Itr current_atom=atom_vector.begin();
			current_atom!=atom_vector.end(); current_atom++){
		centroid += (*current_atom)->get_coordinates();
	}
	centroid /= double(atom_vector.size());
	for(size_t i=0; i<atom_vector.size();i++){
		if(atom_vector[i]->get_element_type()!=atom_chem::HYDROGEN  && !(atom_vector[i]->Is_Virtual())){
		  cent_distance = centroid-atom_vector[i]->get_coordinates();
		  if(i==0 || cent_distance.length()
		  		< min_distance.length() ){
				min_distance = cent_distance;
				atom_closest=i;
			}
		}
	}
  return atom_closest;
}


// =============================================================================
//     int for get_number_of_double_bonded_oxygens
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_number_of_double_bonded_oxygen
///
/// @brief
///		returns number of double bonded oxygens connected to the given atom.
///
/// @detailed
///		returns 1 in sp2, 2 if sp, and higher for things like Nitro nitrogens
/// @param [in] Atom * atom1
///
///
/// @global_read
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	3-16-2006
////////////////////////////////////////////////////////////////////////////////

int
Ligand::get_number_of_double_bonded_oxygens(
	ligand::Ligand_AtomAP const atom1
){
	int number_of_oxygens=0;
	ligand::Ligand_AtomOP this_atom;
	ligand::Ligand_AtomOP other_atom;
	if(!(atom1->Is_Root_Atom())){
		ligand::Ligand_BondAP parentbond;
		atom1->get_parentbond(parentbond);
		if(parentbond->get_bondtype()==atom_chem::DOUBLE){
			parentbond->bonded_atoms(other_atom,this_atom);
			if(other_atom->get_element_type()==atom_chem::OXYGEN){
				number_of_oxygens++;
			}
		}

	}

	std::list< ligand::Ligand_BondAP > bond_children;
	atom1->get_childbonds(bond_children);
	for(std::list< ligand::Ligand_BondAP >::iterator current_bond=bond_children.begin();
			current_bond!=bond_children.end(); current_bond++){

		if((*current_bond)->get_bondtype()==atom_chem::DOUBLE){
			(*current_bond)->bonded_atoms(this_atom,other_atom);
			if(other_atom->get_element_type()==atom_chem::OXYGEN){
				number_of_oxygens++;
			}
		}
	}

	return number_of_oxygens;
}
// =============================================================================
//     int for get_hybridization
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_hybridization
///
/// @brief
///		returns the hybridization of an Atom
///
/// @detailed
///		returns 1 in sp2, 2 if sp, and higher for things like Nitro nitrogens
/// @param [in] Atom * atom1
///
///
/// @global_read
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	3-16-2006
////////////////////////////////////////////////////////////////////////////////

int
Ligand::get_hybridization(
	ligand::Ligand_AtomOP const atom1
){


	int i=0;
	int j=0;
	if(!(atom1->Is_Root_Atom())){
		ligand::Ligand_BondAP parentbond;
		atom1->get_parentbond(parentbond);
		switch(parentbond->get_bondtype()){
			case atom_chem::DOUBLE: { i++;break;}
			case atom_chem::TRIPLE: { i +=2;break;}
			case atom_chem::AROMATIC:
			case atom_chem::CONJUGATED: {j++; break;}
			default : { break;}
		}
	}
	std::list< ligand::Ligand_BondAP > bond_children;
	atom1->get_childbonds(bond_children);
	for(std::list< ligand::Ligand_BondAP >::iterator current_bond=bond_children.begin();
				current_bond!=bond_children.end();current_bond++){
		switch((*current_bond)->get_bondtype()){
			case atom_chem::DOUBLE: { i++;break;}
			case atom_chem::TRIPLE: { i +=2;break;}
			case atom_chem::AROMATIC:
			case atom_chem::CONJUGATED: {j++; break;}
			default : { break;}
		}
	}
	i+=int(j/2);
	return i;

}


// =============================================================================
//     int for get_number_of_hydrogens
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_number_of_hydrogens
///
/// @brief
///		returns the number of bound hydrogens
///
/// @detailed
///
/// @param [in] Atom * atom1
///
///
/// @global_read
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	3-16-2006
////////////////////////////////////////////////////////////////////////////////

int
Ligand::get_number_of_hydrogens(
	ligand::Ligand_AtomOP const atom1
){
	ligand::Ligand_AtomOP this_atom;
	ligand::Ligand_AtomOP other_atom;
	std::list< ligand::Ligand_BondAP > bond_children;
	int number_of_hydrogens=0;
	if(!(atom1->Is_Root_Atom())){
		ligand::Ligand_BondAP parent_bond;
		atom1->get_parentbond( parent_bond );
		parent_bond->bonded_atoms(other_atom, this_atom);
		if(other_atom->get_element_type()==atom_chem::HYDROGEN){
			number_of_hydrogens++;
		}
	}
	atom1->get_childbonds(bond_children);
	for(std::list< ligand::Ligand_BondAP >::iterator current_bond=bond_children.begin();
			current_bond!=bond_children.end();current_bond++){

		(*current_bond)->bonded_atoms(this_atom, other_atom);
		if(other_atom->get_element_type()==atom_chem::HYDROGEN){
			number_of_hydrogens++;
		}
	}
	return number_of_hydrogens;
}
// =============================================================================
//     int for get_number_of_aromatic_bonds
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_number_of_aromatic_bonds
///
/// @brief
///		returns number of aromatic bonds
///
/// @detailed
///
/// @param [in] Atom * atom1
///
///
/// @global_read
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	3-17-2006
////////////////////////////////////////////////////////////////////////////////

int
Ligand::get_number_of_aromatic_bonds(
	ligand::Ligand_AtomAP const atom1
){
	std::list< ligand::Ligand_BondAP > bond_children;
	atom1->get_childbonds(bond_children);
	int number_of_aromatic_bonds=0;
	for(std::list< ligand::Ligand_BondAP >::iterator current_bond=bond_children.begin();
				current_bond!=bond_children.end(); current_bond++){

			if((*current_bond)->get_bondtype()==atom_chem::AROMATIC){
				number_of_aromatic_bonds++;
			}
	}
	if( !atom1->Is_Root_Atom()){
		ligand::Ligand_BondAP parentbond;
		atom1->get_parentbond(parentbond);
		if(parentbond->get_bondtype()==atom_chem::AROMATIC){
				number_of_aromatic_bonds++;
		}
	}
	return number_of_aromatic_bonds;
}


// =============================================================================
//     bool for Are_Bonded
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin Are_Bonded
///
/// @brief
///		Checks whether two Atoms are connected by a bond in
///	the bond_vector vector
/// @detailed
///
/// @param
///		in two pointers to Atom objects
/// @global_read
///		reads from bond_vector in ligand object
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	11-29-2005
////////////////////////////////////////////////////////////////////////////////
bool
Ligand::Are_Bonded(
	ligand::Ligand_AtomAP const atom1,
	ligand::Ligand_AtomAP const atom2
){
//	std::cout << "Entering Are_Bonded()" << " ";
//	std::cout << atom1->get_pdb_number() << " ";
//	std::cout << atom2->get_pdb_number() << std::endl;
	if( bond_vector.empty()) {return false;}
	ligand::Ligand_AtomAP patom1=NULL;// used for information retrieval
	ligand::Ligand_AtomAP patom2=NULL;
//	std::cout << "Iterating over bonds" << std::endl;
	for (ligand::Bond_Itr i=bond_vector.begin();i!=bond_vector.end();i++){
		(*i)->bonded_atoms(patom1, patom2);
//		std::cout << atom1 <<  " " << atom2 << "a bound atoms" << std::endl;
//		std::cout << patom1 <<  " " << patom2 << "p bound atoms" << std::endl;
//		std::cout << "Checking bond" << " ";
//		std::cout << patom1->get_pdb_number() << " ";
//		std::cout << patom2->get_pdb_number() << std::endl;
		if ( (atom1==patom1 && atom2==patom2) ||
			(atom1==patom2 && atom2==patom1)){ return true;}
	}
//	std::cout << "returning false" << std::endl;
	return false;
}


// =============================================================================
//     bool calculate_internal_coordinates
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin calculate_internal_coordinates()
///
/// @brief
///		 Calculates the intenal coordinates to store in the Bond objects based
///	on the cartesian coordinates stored in the atoms connected by the Bond
/// @detailed
///
/// @param
///
/// @global_read
///		reads from bond_vector
/// @global_write
///		writes to bond_vector
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	11-30-2005
////////////////////////////////////////////////////////////////////////////////
bool
	Ligand::calculate_internal_coordinates( bool const initialize ){
// Assumptions
//             A      D
//				\    /
//				 B--C
// Atoms C and D are bonded in the current bond
// Bond containing B and C is the parent bond of C
// Bond containing A and B is the parent bond of B
// the length stored in current bond is the distance between C and D
// the angle of the current bond is the angle between going from C to B and
// vector going from C to D
// the dihedral stored in current bond (CD) is the dihedral of bond BC.
//
// If only one root atom exists in the ligand object then three anchor
// coordinates will be set relative to the root atom position otherwise the
// rules in the following sentence will be followed.
// If C is root the dihedral and angle don't exist and will be set to 0.
// If B is root A will be set to the cross product of vector BC and the vector
//  of the first child bond of C.


//	std::cout << "Initializing internal ligand coordinates." << std::endl;
	bool anchors_set=false;
	if(trial_roots.size()==1){
		if(initialize){
			anchor_first=trial_roots[0]->get_coordinates();
			anchor_first.x()+=2;
			anchor_second=trial_roots[0]->get_coordinates();
			anchor_second.x()-=2;
			anchor_second.y()-=2;
			anchor_second.z()+=2;
			anchor_state++;
			anchor_third=trial_roots[0]->get_coordinates();
		}
 //     std::cout << "kwk help anchor_first " << anchor_first << std::endl;
 //     std::cout << "kwk help anchor_second " << anchor_second << std::endl;
 //     std::cout << "kwk help anchor_third " << anchor_third << std::endl;
		anchors_set=true;
	}

	ligand::Ligand_AtomAP A, B, C, D;
	ligand::Ligand_BondAP other_bond;
	std::list< ligand::Ligand_BondAP > other_bonds;
	C=NULL;
	D=NULL;
	A=NULL;
	B=NULL;
	numeric::xyzVector<double> vectorDC, vectorBC, coordA, coordB, coordC,coordD;
	for (ligand::Bond_Itr current_bond=bond_vector.begin();
	current_bond!=bond_vector.end(); current_bond++){
		(*current_bond)->bonded_atoms(C,D);
		coordD=D->get_coordinates();
		coordC=C->get_coordinates();
		vectorDC= (D->get_coordinates()-C->get_coordinates());
		(*current_bond)->set_length(vectorDC.length());
		if (!(C->Is_Root_Atom())){
			C->get_parentbond(other_bond);
			other_bond->bonded_atoms(B,C);
			coordB=B->get_coordinates();
			vectorBC= (B->get_coordinates()-C->get_coordinates());
			if(!(B->Is_Root_Atom())){
				B->get_parentbond(other_bond);
				other_bond->bonded_atoms(A,B);
				coordA=A->get_coordinates();
			}else{
				if(!anchors_set){
					C->get_childbonds(other_bonds);
					(*other_bonds.begin())->bonded_atoms(C,A);
					coordA=cross_product((A->get_coordinates()-C->get_coordinates()), vectorBC);
					coordA.normalize();
					// this area of the code could be a problem.
					// if coordA is too small it could lead to difficulties.
				}else{
					coordA=anchor_first;
				}
			}
			// setting angle and dihedral
			(*current_bond)->set_angle(acos(dot_product(vectorBC,vectorDC)/
					(vectorDC.length()*vectorBC.length())));
			(*current_bond)->set_dihedral(dihedral( coordA, coordB, coordC, coordD)*numeric::constants::d::pi/180);
//			std::cout << "kwkhelp dihedral "<< dihedral( coordA, coordB, coordC, coordD)*numeric::constants::d::pi/180 <<  std::endl;
//	    std::cout << "kwk help vectorBC " << vectorBC << std::endl;
//	    std::cout << "kwk help vectorDC " << vectorDC << std::endl;
		}else if(anchors_set){
			coordB=anchor_first;
			vectorBC=anchor_first-C->get_coordinates();
			coordA=anchor_second;
			(*current_bond)->set_angle(acos(dot_product(vectorBC,vectorDC)/
					(vectorDC.length()*vectorBC.length())));
			(*current_bond)->set_dihedral(dihedral( coordA, coordB, coordC, coordD)*numeric::constants::d::pi/180);
//		  std::cout << "kwkhelp dihedral "<< dihedral( coordA, coordB, coordC, coordD)*numeric::constants::d::pi/180 << std::endl;
		}else{
			(*current_bond)->set_angle(0);
			(*current_bond)->set_dihedral(0);
		}
//		std::cout << (*current_bond)->get_coordinates() << std::endl;
	}//for loop
	return true;
}

// =============================================================================
//     int twist_dihedral
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin twist_dihedral_radians
///
/// @brief
///		 given an angle in radians and a Bond function will change
/// appropriate internal coordinates and global coordinates of ligand atoms
///
/// @detailed
///		The Function is a driver function for a series of other functions which
///	accomplish the rotation. It constructs the appropriate rotation matrix and
///	decides which side of the bond to apply the rotation to based on the number
/// of atoms on each side.
/// @param
///
/// @global_read
///		reads from bond_vector
///		reads from atom_vector
/// @global_write
///		writes to bond_vector
/// 	writes to atom_vector
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	12-14-2005
////////////////////////////////////////////////////////////////////////////////
int
Ligand::twist_dihedral_radians(
	double const dihedral,
	BondSmallMolecule & bond
){
	//utility::io::ozstream pdbout7("ligand_conf7.pdb");
	//make_pdb_hetero(pdbout7, *this);
	//pdbout7.close();
	//std::cout << "dihedral " << dihedral << std::endl;
	//std::cout << "Starting twist dihedral" << std::endl;


	if(!bond.IsRotable()){return -1;} // checks whether bond is rotable.
	bool down_branch=true;
	int size=get_size_of_branch(&bond);
	ligand::Ligand_AtomAP parent_atom=NULL;
	ligand::Ligand_AtomAP child_atom=NULL;
	bond.bonded_atoms(parent_atom,child_atom);
	// following if statement assume one ligand in the bond_graph
	if(size > int(atom_vector.size())/2) { down_branch=false; }
	//std::cout << "direction twist dihedral" << std::endl;

	numeric::xyzVector<double> origin;
	numeric::xyzMatrix<double> rot_matrix=bond.get_rotation_matrix( dihedral, origin);
	//above call retrieves rotation matrix plus origin offset
	//this block updates internal coordinates

	// this a tricky portion of the code due to poor design on my part
	// the chemical dihedral of a bond is actually stored with the child bonds
	// because it is only used to determine the position of the grandchild atoms
	std::list<ligand::Ligand_BondAP > child_bonds;
	child_atom->get_childbonds(child_bonds);
	for(std::list< ligand::Ligand_BondAP >::iterator current_bond=child_bonds.begin();
		current_bond!=child_bonds.end(); current_bond++){

		(*current_bond)->set_dihedral((*current_bond)->get_dihedral()
			+dihedral);
	}

	//std::cout << "internal coordinates twist dihedral" << std::endl;
	// following if blocks updates the coordinates


	if(down_branch){
		for(ligand::Atom_Itr current_atom=atom_vector.begin();
			current_atom!=atom_vector.end(); current_atom++){
			if((*current_atom)->get_index()>child_atom->get_index() &&
			(*current_atom)->get_index()<(size+child_atom->get_index() )){
		//		std::cout << (*current_atom)->get_coordinates() << std::endl;
				(*current_atom)->rotate_atom(rot_matrix,origin);
		//		std::cout << (*current_atom)->get_coordinates() << std::endl;
			}
		}
	}
	// following if block updates coordinates for uptree case.
	if(!down_branch){
		for(ligand::Atom_Itr current_atom=atom_vector.begin();
			current_atom!=atom_vector.end(); current_atom++){
			if((*current_atom)->get_index()<parent_atom->get_index() ||
			((*current_atom)->get_index())>=(size+child_atom->get_index())){
//				std::cout << (*current_atom)->get_coordinates() << std::endl;
				(*current_atom)->rotate_atom(rot_matrix,origin);
//				std::cout << (*current_atom)->get_coordinates() << std::endl;
			}
		}
		anchor_first=rot_matrix*(anchor_first-origin)+origin;
		anchor_second=rot_matrix*(anchor_second-origin)+origin;
		anchor_third=rot_matrix*(anchor_third-origin)+origin;
		anchor_state++;
	}
	//utility::io::ozstream pdbout6("ligand_conf6.pdb");
	//make_pdb_hetero(pdbout6, *this);
	//pdbout6.close();
	//ligand::LigandInternalCoord before_calc, after_calc;
	//before_calc=get_ligand_internal_conformation(*this);
	//calculate_internal_coordinates(false);
	//after_calc=get_ligand_internal_conformation(*this);
	//utility::io::ozstream pdbout5("ligand_conf5.pdb");
	//make_pdb_hetero(pdbout5, *this);
	//pdbout5.close();
	//assert( before_calc==after_calc);
	//std::cout << "Ending twist dihedral" << std::endl;
	return 1;

}

//==============================================================================
// int set_dihedral_degrees
//==============================================================================
// this function sets the dihedral of the bond to a given angle in degrees
int
Ligand::set_dihedral_degrees(
	ligand::Ligand_BondAP bond_ptr,
	double const degrees
){
	// calculate delta to apply to the dihedral.
	// first chose the final dihedral
	// then get the current dihedral.
	// the difference is the delta that needs to be applied.
	double dihedral_angle= degrees*numeric::constants::d::pi_over_180;
	ligand::Ligand_AtomAP atom1;
	ligand::Ligand_AtomAP atom2;
	bond_ptr->bonded_atoms(atom1, atom2);
	std::list< ligand::Ligand_BondAP > child_bonds;
	atom2->get_childbonds(child_bonds);
	double current_dihedral=(*child_bonds.begin())->get_dihedral();
	double delta_dihedral=dihedral_angle-current_dihedral;
	twist_dihedral_radians(delta_dihedral, *bond_ptr);
	return 1;
}



// =============================================================================
//     int set_best_coordinates
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin set_best_coordinates
///
/// @brief
///		 Stores coordinates from atom_vector and bond_vector into best_coord
/// and best_bonds containers
/// @detailed
///
/// @param
///
/// @global_read
///		reads from bond_vector
///		reads from atom_vector
/// @global_write
///		writes to best_bonds
/// 	writes to best_coord
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	12-27-2005
////////////////////////////////////////////////////////////////////////////////
int
Ligand::set_best_coordinates()
{
    for ( size_t j = 0; j < atom_vector.size(); ++j ) { // stores atom_vector
        if (j==0) {// xyz coordinates in best_coord
          best_coord.clear();
          best_coord.reserve(atom_vector.size());
        }
        best_coord.push_back( atom_vector[j]->get_coordinates());
    }
	for ( size_t j = 0; j < bond_vector.size(); ++j ) { // stores bond_vector
        if (j==0) {// internal coordinates in best_bonds
          best_bonds.clear();
          best_bonds.reserve(bond_vector.size());
        }
        best_bonds.push_back( bond_vector[j]->get_coordinates());
    }
    anchor_first_best=anchor_first;
    anchor_second_best=anchor_second;
    anchor_third_best=anchor_third;
	return 1;
}

// =============================================================================
//     int recover_best_coordinates
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin recover_best_coordinates
///
/// @brief
///		 Restores coordinates from best_coord and best_bonds into atom_vector
/// and bond_vector containers
/// @detailed
///
/// @param
///
/// @global_read
///		reads from best_bonds
///		reads from best_coord
/// @global_write
///		writes to bond_vector
/// 	writes to atom_vector
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	12-27-2005
////////////////////////////////////////////////////////////////////////////////
int
Ligand::recover_best_coordinates(){
	if(atom_vector.size()==best_coord.size()){
		for ( size_t j = 0; j < best_coord.size(); ++j ) { // assigns trial atoms to best atoms
			atom_vector[j]->set_coordinates(best_coord[j]);
		}
		 	//	std::cout << "recovering best coordinates." << std::endl;
	}
	else {
		std::cout << "Size of best_coord and atom_vector differ" << std::endl;
		std::cout << "best_coord coordinates not recovered" << std::endl;
		return -1;
	}
	if(bond_vector.size()==best_bonds.size()){
		for ( size_t j = 0; j < best_bonds.size(); ++j ) { // assigns trial atoms to best atoms
			bond_vector[j]->set_coordinates(best_bonds[j]);
		}
		 	//	std::cout << "recovering best coordinates." << std::endl;
	}
	else {
		std::cout << "Size of best_bonds and bond_vector differ" << std::endl;
		std::cout << "best_bonds coordinates not recovered" << std::endl;
		return -1;
	}
	set_anchors(anchor_first_best, anchor_second_best, anchor_third_best);
	recompute_ligand_interface=true;
	return 1;
}

// =============================================================================
//     int set_start_coordinates
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin set_start_coordinates
///
/// @brief
///		 Stores coordinates from atom_vector and bond_vector into start_coord
/// and start_bonds containers
/// @detailed
///
/// @param
///
/// @global_read
///		reads from bond_vector
///		reads from atom_vector
/// @global_write
///		writes to start_bonds
/// 	writes to start_coord
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	12-27-2005
////////////////////////////////////////////////////////////////////////////////
int
Ligand::set_start_coordinates()
{
    for ( size_t j = 0; j < atom_vector.size(); ++j ) { // stores atom_vector
        if (j==0) {// xyz coordinates in best_coord
          start_coord.clear();
          start_coord.reserve(atom_vector.size());
        }
        start_coord.push_back( atom_vector[j]->get_coordinates());
    }
	for ( size_t j = 0; j < bond_vector.size(); ++j ) { // stores bond_vector
        if (j==0) {// internal coordinates in best_bonds
          start_bonds.clear();
          start_bonds.reserve(bond_vector.size());
        }
        start_bonds.push_back( bond_vector[j]->get_coordinates());
    }
    anchor_first_start=anchor_first;
    anchor_second_start=anchor_second;
    anchor_third_start=anchor_third;
	return 1;
}

// =============================================================================
//     int recover_start_coordinates
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin recover_start_coordinates
///
/// @brief
///		 Restores coordinates from start_coord and start_bonds into atom_vector
/// and bond_vector containers
/// @detailed
///
/// @param
///
/// @global_read
///		reads from start_bonds
///		reads from start_coord
/// @global_write
///		writes to bond_vector
/// 	writes to atom_vector
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	12-27-2005
////////////////////////////////////////////////////////////////////////////////
int
Ligand::recover_start_coordinates(){
	if(atom_vector.size()==start_coord.size()){
		for ( size_t j = 0; j < start_coord.size(); ++j ) { // assigns trial atoms to best atoms
			atom_vector[j]->set_coordinates(start_coord[j]);
		}
		 	//	std::cout << "recovering best coordinates." << std::endl;
	}
	else {
		std::cout << "Size of start_coord and atom_vector differ" << std::endl;
		std::cout << "start_coord coordinates not recovered" << std::endl;
		return -1;
	}
	if(bond_vector.size()==start_bonds.size()){//assigns trial atoms
		for ( size_t j = 0; j < start_bonds.size(); ++j ) { //to best atoms
			bond_vector[j]->set_coordinates(start_bonds[j]);
		}
		 	//	std::cout << "recovering start coordinates." << std::endl;
	}else{

		calculate_internal_coordinates(true);
		for ( size_t j = 0; j < start_bonds.size(); ++j ) { //to best atoms
			bond_vector[j]->set_coordinates(start_bonds[j]);
		}
	}
	set_anchors( anchor_first_start, anchor_second_start, anchor_third_start);
	recompute_ligand_interface=true;
	return 1;
}


// =============================================================================
//     int get_ligand_internal_energy
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_ligand_internal_energy
///
/// @brief
///		 if initializing the function will fill the ligand_atom_pair_energy list
/// with LigandAtomPairEnergy objects for the appropriate atom pairs. Atom pairs
/// that are not greater than three bonds distant will not be placed in the
/// ligand_atom_pair_energy list.
///
/// @detailed
///
/// @param  [in] bool initialize tells the function whther it needs to create
/// 	LigandAtomPairObjects.
///
/// @global_read
/// @global_write
///		writes to the ligand internal Energy fields
///
/// @remarks
///
/// @references
///
/// @authors
///		Kristian Kaufmann
/// @last_modified	2-27-2006
////////////////////////////////////////////////////////////////////////////////
float
Ligand::get_ligand_internal_energy(
	bool const initialize
){
	//initialize
	// in this block we set which atom pairs should be evaluated
	// by which terms. The interactions greater than three bonds apart are
	// evaluated in the van der waals and solvation energy fields.
	// Interactions of atoms three bonds or less apart are are summed up in the
	// dihedralE term.
	// Hbond Energy is calculated over the acceptors and donors in the small
	// molecule.  This blocks purpose is to find the division between atoms three
	// bonds apart and those that aren't.
	if( initialize ){
		std::list< ligand::Ligand_AtomAP > neighbor_atoms; // non-owning list of pointers to atoms
		for( ligand::Atom_Itr current_atom=atom_vector.begin();
				current_atom!=atom_vector.end(); current_atom++){
			neighbor_atoms.clear();

		std::list< ligand::Ligand_AtomAP > bound_atoms_first;
		get_bonded_atoms(bound_atoms_first,(*current_atom));

		for( std::list< ligand::Ligand_AtomAP >::iterator first_shell_atoms=bound_atoms_first.begin();
			first_shell_atoms!=bound_atoms_first.end(); first_shell_atoms++){

			neighbor_atoms.push_back((*first_shell_atoms));
			std::list< ligand::Ligand_AtomAP > bound_atoms_second;
			get_bonded_atoms(bound_atoms_second,(*first_shell_atoms));
			for(std::list< ligand::Ligand_AtomAP >::iterator second_shell_atoms=bound_atoms_second.begin();
				second_shell_atoms!=bound_atoms_second.end(); second_shell_atoms++){

				neighbor_atoms.push_back((*second_shell_atoms));
				std::list< ligand::Ligand_AtomAP > bound_atoms_third;
				get_bonded_atoms(bound_atoms_third,(*second_shell_atoms));
				for(std::list< ligand::Ligand_AtomAP >::iterator third_shell_atoms=bound_atoms_third.begin();
					third_shell_atoms!=bound_atoms_third.end(); third_shell_atoms++){

					neighbor_atoms.push_back((*third_shell_atoms));
				}
			}
		}

			// previous blocks found all atoms three bonds or less distance
			// from the current atom the nest block will construct the appropriate
			// LigandAtomPairEnergy objects
			// Since we want only only one pair energy per pair we set the loop
			// to start at the atom after the current atom. Since all atoms before
			//current atom would have already had the LigandAtomPairEnergy objects
			// constructed


			for(ligand::Atom_Itr next_atom=(current_atom+1);
				next_atom!=atom_vector.end(); next_atom++){

				if(find(neighbor_atoms.begin(),neighbor_atoms.end(),(*next_atom))
					== neighbor_atoms.end()){
					if(!((*current_atom)->Is_Virtual()) && !((*next_atom)->Is_Virtual())){
						LigandAtomPairEnergy * pairligand = new LigandAtomPairEnergy((*current_atom),(*next_atom));
						if( runlevel_ns::runlevel >= runlevel_ns::verbose ) {
							std::cout << "LigandAtomPair for Energy pdb numbers ";
							std::cout << (ligand_atom_pair_energy.size()+1)<< " " << (*current_atom)->get_pdb_number() << " " << (*next_atom)->get_pdb_number() << std::endl;
						}
						ligand_atom_pair_energy.push_back( *pairligand);
					}
				}//neighbor
			}//next_atom
		}// current_atom
	}// intialize
	//calculate retrieve int_repE, int_atrE, int_solE
	lig_int_repE=0.0;
	lig_int_atrE=0.0;
	lig_int_solvE=0.0;
	lig_int_dihedE=0.0;
	lig_int_coulombE=0.0;
	lig_int_hbE=0.0;
//	int i=1;
	for(std::list<LigandAtomPairEnergy>::iterator current_pair=
		ligand_atom_pair_energy.begin(); current_pair!=
		ligand_atom_pair_energy.end(); current_pair++){

		float repE_current=0.0;
		float atrE_current=0.0;
		float solvE_current=0.0;
		current_pair->get_energies(repE_current,atrE_current,solvE_current);
		lig_int_repE += repE_current;
		lig_int_atrE += atrE_current;
		lig_int_solvE += solvE_current;

//	  if( runlevel_ns::runlevel >= runlevel_ns::verbose && repE_current > 10 ) {
//              std::cout << "LigandAtomPair for Energy " << i << " " << repE_current << std::endl;
//    }
//	  i++;
	}

	// END of retrieve repE, atrE, solvE

	//calculate int_hBondE
	// the bulk of this code is borrowed directly from ligand.cc

	FArray1D_float proton_coord(3), donor_coord(3);
	FArray1D_float accept_coord(3), acceptbase_coord(3);
	FArray1D_float abase2_coord(3);
	float energy;

	for(int current_acceptor=1; current_acceptor<hetero_atom_hbond_acc_count;
		current_acceptor++){

		//get acceptor atom and base coordinates
		copy_to_FArray(atom_vector[
			hetero_atom_hbond_acc(current_acceptor)]->get_coordinates(),
			accept_coord);
		copy_to_FArray(atom_vector[
			atom_base[hetero_atom_hbond_acc(current_acceptor)]]->get_coordinates(),
			acceptbase_coord);
		copy_to_FArray(atom_vector[abase2[hetero_atom_hbond_acc(current_acceptor)]]->get_coordinates(), abase2_coord);

		int const atype = atom_vector[hetero_atom_hbond_acc(current_acceptor)]->get_ros_atom_type();

		for(int current_donor=1; current_donor<hetero_atom_hbond_don_count;
			current_donor++){
			if(int(atom_base[hetero_atom_hbond_don(current_donor)])
					!=hetero_atom_hbond_acc(current_acceptor)){

				copy_to_FArray(atom_vector[
					hetero_atom_hbond_don(current_donor)]->get_coordinates(),
					proton_coord);
				copy_to_FArray(atom_vector[atom_base[
					hetero_atom_hbond_don(current_donor)]]->get_coordinates(),
					donor_coord);
				int const dtype = atom_vector[atom_base[
					hetero_atom_hbond_don(current_donor)]]->get_ros_atom_type();

				//JSS classify type of hydrogen bond from donor and acceptor atom types (omit seq, since not on same molec.)
				HBEvalType const hbe_type = hbond_evaluation_type(dtype,atype);

				// if not -O-H or if fix_ligand_HO
				if ( !(atom_vector[hetero_atom_hbond_don(current_donor)
						]->get_element_type() == atom_chem::HYDROGEN &&
					atom_vector[atom_base[hetero_atom_hbond_don(current_donor)]]
						->get_element_type() == atom_chem::OXYGEN)|| fix_ligand_HO ) {

						//mj compute energy of hydrogen bond
					hb_energy_deriv(hbe_type,donor_coord(1),proton_coord(1),
						accept_coord(1),acceptbase_coord(1), abase2_coord(1),
						energy); // no deriv computation

						if ( energy < hbonds::MAX_HB_ENERGY ) {lig_int_hbE += energy;}
					}

					// in case of -O-H optimize H position in ligand first
					else {

						////////////////////////////////////////////////
						//
						//                            X
						//                   o--H
						//                   | /
						//                   |/ 0.98A
						//                   O 109.5grad
						//                   |
						//                   |
						//                   B
						//
						////////////////////////////////////////////////

						// find optimized H positon
						double const sin109 = { 0.9426 * 0.98 };
						// sin( 70.5) * bondlength( O-H)
						double const cos109 = { 0.3338 * 0.98 };
						// cos( 70.5) * bondlength( O-H)
						static FArray1D_float opt_Hpos( 3 );
						numeric::xyzVector_double opt_Hpos_vect;
//kwk reset as xyzVectors		static FArray1D_float B_O_vect( 3 );
						numeric::xyzVector_double B_O_vect;
//kwk						static FArray1D_float O_X_vect( 3 );
						numeric::xyzVector_double O_X_vect;
//kwk						static FArray1D_float ort_vect( 3 );
						numeric::xyzVector_double ort_vect;
						numeric::xyzVector_double aatm_vect;

						aatm_vect=atom_vector[hetero_atom_hbond_acc(current_acceptor)]
							->get_coordinates();
						// compute B-O and O-X vector

						B_O_vect= atom_vector[atom_base[hetero_atom_hbond_don(current_donor)]]
							->get_coordinates()
							- atom_vector[atom_base[atom_base[hetero_atom_hbond_don(current_donor)]]]
							->get_coordinates();

						O_X_vect= aatm_vect - atom_vector[atom_base[
							hetero_atom_hbond_don(current_donor)]]->get_coordinates();

						// normalize B-O and O-X vector
						B_O_vect.normalize();
						O_X_vect.normalize();
						double cosangle = inner_product( B_O_vect, O_X_vect );
						cosangle = std::min( 1.0, std::max( -1.0, cosangle ) );
						ort_vect = O_X_vect - cosangle * B_O_vect;
						ort_vect.normalize();
						// compute H position in B-O-X plane
						opt_Hpos_vect = atom_vector[atom_base[
							hetero_atom_hbond_don(current_donor)]]->get_coordinates()
											+ B_O_vect *  cos109
											+ ort_vect *  sin109;
						opt_Hpos(1)=opt_Hpos_vect.x();
						opt_Hpos(2)=opt_Hpos_vect.y();
						opt_Hpos(3)=opt_Hpos_vect.z();
						//mj compute energy of hydrogen bond
						copy_to_FArray(atom_vector[atom_base[
							hetero_atom_hbond_don(current_donor)]]->get_coordinates(),
							donor_coord);
						hb_energy_deriv(hbe_type,donor_coord(1),opt_Hpos(1),
							accept_coord(1),acceptbase_coord(1), abase2_coord(1),
							energy); // no deriv computation

						if ( energy < hbonds::MAX_HB_ENERGY ) {lig_int_hbE += energy;}
					}}
		}//current_donor loop
	}//current_acceptor loop

	//End of get HbondE


	//calculate coulombE
	for(ligand::Atom_Itr c_atom=atom_vector.begin();c_atom!=atom_vector.end();
			c_atom++){
		if(!((*c_atom)->Is_Virtual())){
		for(ligand::Atom_Itr n_atom=c_atom+1;n_atom!=atom_vector.end();n_atom++){
		//compute distance
			if(!((*n_atom)->Is_Virtual())){
				float E=0.0;
				float d2,limit2;
				d2=distance_squared((*c_atom)->get_coordinates(),
														(*n_atom)->get_coordinates());
				//mj compute limit
				limit2 = ( etable::fa_lj_radius((*c_atom)->get_ros_atom_type()) +
									etable::fa_lj_radius((*n_atom)->get_ros_atom_type()))*
								( etable::fa_lj_radius((*c_atom)->get_ros_atom_type()) +
									etable::fa_lj_radius((*n_atom)->get_ros_atom_type()));

				//mj compute energy
				simple_elec((*c_atom)->get_partial_charge(),
									(*n_atom)->get_partial_charge(),d2,limit2,E);
				lig_int_coulombE+=E;
			}}
		}}
	//End of CoulombE
	//calculate dihedralE
	for(ligand::Bond_Itr current_bond=bond_vector.begin();
		current_bond!=bond_vector.end(); current_bond++){
		double dihedralE_sum=0.0;
		size_t dihedralE_number=0;
		if((*current_bond)->IsRotable()){
			ligand::Ligand_AtomOP atom1, atom2, atom4, atom3;
			ligand::Ligand_BondAP parent_bond;
			std::list< ligand::Ligand_BondAP > sibling_bonds;
			std::list< ligand::Ligand_BondAP > child_bonds;
			double dihedralE_deriv;// unused now but will be incorporated into
														// minimization at later stage.
			double dihedral_current;
			double dihedralE_current;// dihedral energy of current tetrad of atoms
									// specifying the dihedral
									// currently the energy of all combinations
									// of non-hydrogen atoms is averaged to
									// find the energy of the torsion angle
			(*current_bond)->bonded_atoms(atom1, atom2);//middle atoms in the dihedral
			// finds out what type of torsion angle is in question.
			std::pair< atom_chem::LigandAtomType, atom_chem::LigandAtomType >
				atompairtype(
								atom1->get_ligand_atom_type(), atom2->get_ligand_atom_type());

			atom2->get_childbonds(child_bonds);
			if(!(atom1->Is_Root_Atom())){
				atom1->get_parentbond(parent_bond);
				parent_bond->bonded_atoms(atom4,atom1);
				if(atom4->get_element_type()!=atom_chem::HYDROGEN){
					for(std::list< BondSmallMolecule * >::iterator c_bond=child_bonds.begin();
							c_bond!=child_bonds.end();c_bond++){

						(*c_bond)->bonded_atoms(atom2,atom3);
						if(atom3->get_element_type()!=atom_chem::HYDROGEN){
							numeric::dihedral(atom4->get_coordinates(),
								atom1->get_coordinates(),atom2->get_coordinates(),
								atom3->get_coordinates(),dihedral_current);
							dihedral_current=(dihedral_current/180*numeric::constants::d::pi);
							get_dihedral_energy(atompairtype, dihedral_current, dihedralE_current,
										dihedralE_deriv, initialize );
							dihedralE_sum+=dihedralE_current;
							dihedralE_number++;
						}
					}
				}
			}
			atom1->get_childbonds(sibling_bonds);
			for(std::list< ligand::Ligand_BondAP >::iterator c2_bond=sibling_bonds.begin();
					c2_bond!=sibling_bonds.end();c2_bond++){

				(*c2_bond)->bonded_atoms(atom1,atom4);
				if(atom4->get_element_type()!=atom_chem::HYDROGEN && atom4!=atom2){
					for(std::list< ligand::Ligand_BondAP >::iterator c_bond=child_bonds.begin();
						c_bond!=child_bonds.end();c_bond++){

						(*c_bond)->bonded_atoms(atom2,atom3);
						if(atom3->get_element_type()!=atom_chem::HYDROGEN){
							numeric::dihedral(atom4->get_coordinates(),
								atom1->get_coordinates(),atom2->get_coordinates(),
								atom3->get_coordinates(),dihedral_current);
							dihedral_current=(dihedral_current/180*numeric::constants::d::pi);
							get_dihedral_energy(atompairtype, dihedral_current, dihedralE_current,
								dihedralE_deriv, initialize );
							dihedralE_sum+=dihedralE_current;
							dihedralE_number++;
						}
					}
				}
			}
		}//If Rotable block
		if(dihedralE_number != 0){
			lig_int_dihedE+=dihedralE_sum/dihedralE_number;
		}
	}// end of for loop over all bonds
	// End of calculate dihedral E
	float total_int_energy= param_pack::packer_misc_parameters::Wlig_int_rep *lig_int_repE
		+ param_pack::packer_misc_parameters::Wlig_int_atr *lig_int_atrE
		+ param_pack::packer_misc_parameters::Wlig_int_solv *lig_int_solvE
		+ param_pack::packer_misc_parameters::Wlig_int_dihed *lig_int_dihedE
		+ param_pack::packer_misc_parameters::Wlig_int_hb*lig_int_hbE
		+ param_pack::packer_misc_parameters::Wlig_int_cou*lig_int_coulombE;



	return total_int_energy;
}
// =============================================================================
//     int refold_ligand
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin refold_ligand
///
/// @brief
///    This function refolds based on the internal coordinates storded in the
///	parent bond of the atom.
///
/// @detailed
///
/// @param
///
/// @global_read
///   reads from anchor numeric::xyzVectors
///   reads from bond_vector
/// @global_write
///   writes to atom_vector
/// @remarks
///
/// @references
///
/// @authors
///   Kristian Kaufmann
/// @last_modified  03-28-2006
////////////////////////////////////////////////////////////////////////////////
int
Ligand::refold_ligand(){
	if(trial_roots.size()!=1){
		std::cout << "Can't refold ligands with multiple roots with current code";
		std::cout << std::endl;
		return -1;
	}else{
		FArray1D_bool up_to_date;
		up_to_date.dimension(atom_vector.size());
		for(size_t i=1;i<=atom_vector.size();i++){
			up_to_date(i)=false;
		}
		for(ligand::Atom_Itr current_atom=atom_vector.begin();
				current_atom!=atom_vector.end(); current_atom++){
			if(!(up_to_date((*current_atom)->get_index()))){
				refold_atom(up_to_date, (*current_atom));
			}
		}
	}
	return 1;
}

// =============================================================================
//     int refold_atom
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin refold_atom
///
/// @brief
///    This function refolds based on the internal coordinates storded in the
///	parent bond of the atom. Recursive calls itself until it finds an uptodate
/// atom or the root of the tree.
///
/// @detailed
///
/// @param Farray_1D_bool up_to_date atom coordinates
///        Atom * atom_to_refold
/// @global_read
///   reads from anchor numeric::xyzVectors of ligand
///   reads from bond_vector of ligand
/// @global_write
///   writes to atom_vector
/// @remarks
///
/// @references
///
/// @authors
///   Kristian Kaufmann
/// @last_modified  03-29-2006
////////////////////////////////////////////////////////////////////////////////
int
Ligand::refold_atom(
	FArray1Dp_bool up_to_date,
	ligand::Ligand_AtomOP atom_to_refold
){
	up_to_date.dimension(int(atom_vector.size()));
	if(!(atom_to_refold->Is_Root_Atom())){
		ligand::Ligand_BondAP parent_bond;
		ligand::Ligand_AtomOP parent_atom;
		atom_to_refold->get_parentbond(parent_bond);
		parent_bond->bonded_atoms(parent_atom, atom_to_refold);
		if(up_to_date(parent_atom->get_index())==false){
			refold_atom(up_to_date,parent_atom);
		}
		numeric::xyzVector< double > vector_A=parent_atom->get_coordinates();
		numeric::xyzVector< double > vector_B;
		numeric::xyzVector< double > vector_C;
		if(!(parent_atom->Is_Root_Atom())){
			ligand::Ligand_BondAP grand_parent_bond;
			ligand::Ligand_AtomOP grand_parent_atom;
			parent_atom->get_parentbond(grand_parent_bond);
			grand_parent_bond->bonded_atoms(grand_parent_atom,parent_atom);
			vector_B=grand_parent_atom->get_coordinates();
			if(!(grand_parent_atom->Is_Root_Atom())){
				ligand::Ligand_BondAP great_grand_parent_bond;
				ligand::Ligand_AtomOP great_grand_parent_atom;
				grand_parent_atom->get_parentbond(great_grand_parent_bond);
				great_grand_parent_bond->bonded_atoms(great_grand_parent_atom,
					grand_parent_atom);
				vector_C=great_grand_parent_atom->get_coordinates();
			}else{
				vector_C=anchor_first;
			}

		}else{
			vector_B=anchor_first;
			vector_C=anchor_second;
		}
		atom_to_refold->set_coordinates(vector_A, vector_B, vector_C,
			parent_bond->get_length(),  parent_bond->get_angle(),
			parent_bond->get_dihedral());
		up_to_date(atom_to_refold->get_index())=true;
		return 3;

	}else{
		if(up_to_date(atom_to_refold->get_index())==false){
			atom_to_refold->set_coordinates(anchor_third);
			up_to_date(atom_to_refold->get_index())=true;
			return 1;
		}
		return 2;
	}
	return -1;
}

// =============================================================================
//     int compute_ligand_conformations
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin compute_ligand_conformations
///
/// @brief
///    computes possible conformations of given ligand.
///
/// @detailed
///
/// @param
///
/// @global_read
///   reads from
///   reads from bond_vector of ligand
/// @global_write
///   writes to atom_vector
///		writes to bond_vector
///		writes to start_coord
///		writes to start_bonds
/// @remarks
///
/// @references
///
/// @authors
///   Kristian Kaufmann
/// @last_modified  04-05-2006
////////////////////////////////////////////////////////////////////////////////
int
Ligand::compute_ligand_conformations()
{

	if ( start_bonds.empty() ){
		set_start_coordinates();
	}


	if( use_input_ligand_only){
		ligand_conformations_base.clear();
		ligand_conformations_base.push_back(get_ligand_internal_conformation(*this));
		return 1;
	}

	if( read_ligand_conformations ){
		ligand_conformations_base.clear();
		utility::io::izstream lig_conf_stream(files_paths::start_path + ligand_conformations_infile);
		if( !lig_conf_stream.fail()){
			read_ligand_conformations_sdfile( lig_conf_stream, (*this));
		}else{
			std::cout << "ligand conformations file " << files_paths::start_path + ligand_conformations_infile;
			std::cout << "failed to open. Now Exiting!!!" << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}
		if ( rotamerize_input_ligand ){
			recover_start_coordinates();
			rotamerize_ligand();
			set_start_coordinates();
			ligand_conformations_base.clear();
			ligand_conformations.clear();
			ligand_conformations_base.push_back(get_ligand_internal_conformation(*this));
		}
		std::cout << "Loaded " << ligand_conformations_base.size() << " conformations" << std::endl;
		return 1;
	}

	std::list< ligand::Ligand_BondAP > rotable_bonds;
	get_rotable_bonds(rotable_bonds);
	std::cout << "Searching " << rotable_bonds.size() << " dihedral angles." << std::endl;
	// The first task is to obtain all the minima of the rotable bonds
	if( !use_only_defined_bonds ){
		load_crystal_structure_dihedral_minima(rotable_bonds);
	}

	calculate_internal_coordinates(false);

	if( dihedral_minima.empty() ){
		//case for rigid ligand
		ligand_conformations_base.push_back(get_ligand_internal_conformation(*this));
		return 1;
	}

	for( std::list< ligand::Ligand_BondAP >::iterator rbond=rotable_bonds.begin();
		rbond!=rotable_bonds.end(); rbond++){
		ligand::Ligand_AtomAP atom1, atom2;
		(*rbond)->bonded_atoms(atom1,atom2);
		std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair;
		if(atom1 > atom2){
			atom_pair.first=atom2;
			atom_pair.second=atom1;
		}else{
			atom_pair.first=atom1;
			atom_pair.second=atom2;
		}
		std::cout << "MBNDC " << atom1->get_pdb_number() << " " << atom2->get_pdb_number();
		std::cout << " " << dihedral_minima[atom_pair].size();
		for( size_t cmin=0;cmin<dihedral_minima[atom_pair].size();cmin++){
			std::cout << " " << ((dihedral_minima[atom_pair])[cmin]).first << " ";
			std::cout << ((dihedral_minima[atom_pair])[cmin]).second;
		}
		std::cout << std::endl;
	}
	//kwk selecting number of ligand conformations.
	//kwk numbers based on number of Lys and ARG at a protein position with the given flags.
	//kwk
	size_t max_number_of_conformations=std::min(1000,maximum_number_of_conformations);
	float exponent_number_rotable_bonds=2;
	if( design::active_rotamer_options.ex1 ){
			exponent_number_rotable_bonds=2.5;
		if(design::active_rotamer_options.ex2){
			exponent_number_rotable_bonds=3.0;
			if(design::active_rotamer_options.ex3){
				exponent_number_rotable_bonds=4.0;
				if(design::active_rotamer_options.ex4){
					exponent_number_rotable_bonds=5.0;
				}
			}
		}
	}

	size_t allowed_number_of_conformations=std::min(size_t(ceil(std::pow(float(rotable_bonds.size()+1),exponent_number_rotable_bonds))),
		max_number_of_conformations);
	allowed_number_of_conformations=std::max(allowed_number_of_conformations,size_t(maximum_number_of_conformations));
	//the multimap stores the conformations using the internal coordinates
	// sorted by the internal energy
	std::multimap< float, ligand::LigandInternalCoord > conformations;
	
	float min_rms=1000.0;
	if( use_iterative_conformation_search ){
		min_rms=iterative_conformation_search(conformations, allowed_number_of_conformations);
	}else{
		min_rms=monte_carlo_conformation_search(conformations, allowed_number_of_conformations);
	}
	//transfer accepted_cartesian_conformations into ligand_conformations
	ligand_conformations.clear();
	ligand_conformations.reserve(conformations.size());
	for(std::multimap< float, ligand::LigandInternalCoord >::iterator
		c_conf=conformations.begin();
		c_conf!=conformations.end();
		c_conf++){

		ligand_conformations.push_back((*c_conf).second);
	}
	
	if( make_rotamer_nearest_start ) {
		ligand::LigandInternalCoord icoord;
		make_conformer_nearest_start(icoord);
		ligand_conformations.push_back(icoord);
	}

//	if( runlevel_ns::runlevel >= runlevel_ns::inform){
		std::cout << ligand_conformations.size() <<
			" Small molecule conformations have been accepted" << std::endl;
		std::cout << "Maximal minimal RMS is: " << min_rms << std::endl;
		std::cout << "Maximal minimal RMS50 is: " << (min_rms/(1+std::log(std::sqrt(float(atom_vector.size())/50)))) << std::endl;
//	}
	// recover starting conformation and store in ligand_conformations.
	recover_start_coordinates();

	//find minimal rms to input conformation
	float min_energy=(*conformations.begin()).first;
	ligand::LigandCartesianCoord input_conformation=get_ligand_cartesian_conformation(*this);
	find_nearest_conformation_by_rmsd( input_conformation, min_rms );
	std::cout << "Minimal RMS to input structure " << min_rms << std::endl;
	std::cout << "Minimal RMS50 to input structure is: " << (min_rms/(1+std::log(std::sqrt(float(atom_vector.size())/50)))) << std::endl;
	float total_int_energy=get_ligand_internal_energy(false);
	std::cout << "Minimum Internal Energy in ligand conformations " << min_energy << std::endl;
	std::cout << "Internal Energy of Input conformation " << total_int_energy << std::endl;
	if( use_input_ligand){
		ligand_conformations.push_back(start_bonds);
	}
	if( output_ligand_conformations ){
		output_ligand_conformations_sdfile(ligand_conformations_outfile);
	}
	
	if ( rotamerize_input_ligand ){
			recover_start_coordinates();
			rotamerize_ligand();
			set_start_coordinates();
			ligand_conformations_base.clear();
			ligand_conformations.clear();
			ligand_conformations.push_back(get_ligand_internal_conformation(*this));
	}
	
//	utility::io::ozstream pdbout3("ligand_conf3.pdb");
//	for(size_t k=0; k<ligand_conformations.size(); k++){
//		change_to_ligand_conformation(k);
//		make_pdb_hetero(pdbout3, *this);
//	}
//	pdbout3.close();
	ligand_conformations_base.clear();
	ligand_conformations_base=ligand_conformations;
	ligand_conformations.clear();
	recover_start_coordinates();
	return 1;
}


//==============================================================================
// int load_crystal_structure_dihedral_minima
//==============================================================================

////////////////////////////////////////////////////////////////////////////////
///@begin load_crystal_structure_dihedral_minima
///@brief loads the minima of required bond types from the Cambridge Structural
///Database of Small Molecule Crystal Structures writes to the dihedral_minima
///member of the ligand class
///@param[in] rotable_bonds the bonds considered rotable for which minima are
///needed
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////

int
Ligand::load_crystal_structure_dihedral_minima(
	std::list< ligand::Ligand_BondAP > & rotable_bonds
){

	//container for rotable bond types as define by their Ligand atom types
	// in string form since the file will be read in terms of strings. This
	// makes the comparison easier
	std::list<std::pair<std::string,std::string> >
			atomtype_strings_pairlist;// temporary copy of needed atompairtypes

	// loop over all the rotable bonds storing the
	// ligandatomtypes as strings


	for(std::list< ligand::Ligand_BondAP >::iterator c_bond=
		rotable_bonds.begin(); c_bond!=rotable_bonds.end();c_bond++){

		ligand::Ligand_AtomAP atom1;
		ligand::Ligand_AtomAP atom2;
		(*c_bond)->bonded_atoms(atom1,atom2);
		std::pair< std::string , std::string > kind1, kind2;
		kind1.first=atom_chem::ligandatomtype_string[atom1->get_ligand_atom_type()];
		kind1.second=atom_chem::ligandatomtype_string[atom2->get_ligand_atom_type()];
		kind2.first=atom_chem::ligandatomtype_string[atom2->get_ligand_atom_type()];
		kind2.second=atom_chem::ligandatomtype_string[atom1->get_ligand_atom_type()];
		bool found_pair=false;
		for( std::list< std::pair< std::string,std::string > >::iterator
			c_pair=atomtype_strings_pairlist.begin();
			c_pair!=atomtype_strings_pairlist.end(); c_pair++){

			if((*c_pair)==kind1 || (*c_pair)==kind2 ){
				found_pair=true;
			}

		}
		if(!found_pair){
			atomtype_strings_pairlist.push_back(kind1);
		}
	}

	// open the minima file
	utility::io::izstream iunit;
	iunit.clear();
	iunit.open(files_paths::data_path +
	"small_molecule_crystal_structure_dihedral_minima.txt");
	std::map<
		std::pair<atom_chem::LigandAtomType,atom_chem::LigandAtomType>,
		std::vector< std::pair< int, int > > > tmp_dihedral_minima;
	std::string atom1type_string, atom2type_string;
	size_t number_of_minima;
	// while loop iterates through the file until it finds all
	// the minima needed. It will check for failing stream and
	// end of file and deal with those cases
	while(!atomtype_strings_pairlist.empty()){
		iunit >> atom1type_string >> atom2type_string >> number_of_minima;
		if( iunit.eof()){
		}else if( iunit.fail()){
			std::cout << "small_molecule_crystal_structure_dihedral_minima.txt failed";
			std::cout << std::endl << " to open. Please check that it is ";
			std::cout << "in rosetta_database or that the file path is correct." <<  std::endl;
			iunit.clear();
			iunit.close();
			return -1;
		}
		// current pair type from file.
		std::pair< std::string, std::string > kind(atom1type_string,atom2type_string);

		std::list< std::pair < std::string, std::string > >::iterator c_pair=
			find(atomtype_strings_pairlist.begin(),atomtype_strings_pairlist.end(),kind);
		std::vector< std::pair< int, int > > minima;
		if(c_pair!=atomtype_strings_pairlist.end()){
			std::pair<atom_chem::LigandAtomType, atom_chem::LigandAtomType >
					atompair(atom_chem::get_ligandatomtype_from_string((*c_pair).first),
									atom_chem::get_ligandatomtype_from_string((*c_pair).second));
			minima.clear();
			minima.reserve(number_of_minima);

			for(size_t i=0;i<number_of_minima;i++){
				int minimum;
				iunit >> minimum;
				if( iunit.fail()){
					std::cout << "small_molecule_crystal_structure_dihedral_minima.txt failed";
					std::cout << std::endl << " on reading in " << atom1type_string;
					std::cout << " " << atom2type_string;
					std::cout << " pair. Please check the format of this pair." <<  std::endl;
					iunit.clear();
					iunit.close();
					return -1;
				}
				std::pair< int ,int > p(minimum, 5);
				//for now the defualt standard deviation is 5 degrees
				// that will change when the well widths are included
				minima.push_back(p);

			}
			iunit.ignore( std::numeric_limits< std::streamsize >::max(), '\n' );
			atomtype_strings_pairlist.remove(*c_pair);// removes a pair that has been read

			std::pair<
				std::pair<atom_chem::LigandAtomType,atom_chem::LigandAtomType>,
				std::vector< std::pair< int,int > > > dihedral_pair;
			dihedral_pair.first=atompair;
			dihedral_pair.second=minima;															// that has been read.

			tmp_dihedral_minima.insert(dihedral_pair);
		}else{
			// check for other permutation of kind
			kind.first=atom2type_string;
			kind.second=atom1type_string;
			c_pair=find(atomtype_strings_pairlist.begin(),atomtype_strings_pairlist.end(),kind);
			if(c_pair!=atomtype_strings_pairlist.end()){
			//pair of LigandAtomType to be push_back onto dihedral minima
			std::pair<atom_chem::LigandAtomType, atom_chem::LigandAtomType >
					atompair(atom_chem::get_ligandatomtype_from_string((*c_pair).first),
									atom_chem::get_ligandatomtype_from_string((*c_pair).second));
			//vector of minima to be push_back onto dihedral minima
			minima.clear();
			minima.reserve(number_of_minima);

			for(size_t i=0;i<number_of_minima;i++){
				int minimum;
				iunit >> minimum;
				if( iunit.fail()){
					std::cout << "small_molecule_crystal_structure_dihedral_minima.txt failed";
					std::cout << std::endl << " on reading in " << atom1type_string;
					std::cout << " " << atom2type_string;
					std::cout << " pair. Please check the format of this pair." <<  std::endl;
					iunit.clear();
					iunit.close();
					return -1;
				}
				std::pair< int ,int > p(minimum, 5);
				//for now the default absolute deviation is 5 degrees
				// that will change when the well widths are included
				minima.push_back(p);

			}
			iunit.ignore( std::numeric_limits< std::streamsize >::max(), '\n' );
			atomtype_strings_pairlist.remove(*c_pair);// removes a pair that has been read

			std::pair<
				std::pair< atom_chem::LigandAtomType,atom_chem::LigandAtomType > ,
				std::vector< std::pair< int, int > > > dihedral_pair;															// that has been read.
			dihedral_pair.first=atompair;
			dihedral_pair.second=minima;
			tmp_dihedral_minima.insert(dihedral_pair);

			}// neither permutation matched therefore skip the rest of the line
			iunit.ignore( std::numeric_limits< std::streamsize >::max(), '\n' );
		}

		if(iunit.eof() && !(atomtype_strings_pairlist.empty())){
			std::cout << "Reached end of file without";
			std::cout << " finding minima for all atom pair types." << std::endl;
			std::cout << "Will set minima every thirty degress" << std::endl;

			for(std::list< std::pair< std::string,std::string > >::iterator
							c_atompair=atomtype_strings_pairlist.begin(); c_atompair!=atomtype_strings_pairlist.end();
							c_atompair++){

				minima.clear();
				minima.reserve(12);
				for(int i=-180;i<180;i+=30){
					std::pair< int ,int > p(i, 5);
					//for now the default standard deviation is 5 degrees
					// that will change when the well widths are included
					minima.push_back(p);
				}
				std::pair<atom_chem::LigandAtomType, atom_chem::LigandAtomType >
					atompair(atom_chem::get_ligandatomtype_from_string((*c_atompair).first),
									atom_chem::get_ligandatomtype_from_string((*c_atompair).second));
				std::pair<
					std::pair<atom_chem::LigandAtomType,atom_chem::LigandAtomType>,
					std::vector< std::pair< int, int > > > dihedral_pair;
				dihedral_pair.first=atompair;
				dihedral_pair.second=minima;
				tmp_dihedral_minima.insert(dihedral_pair);
			}
			atomtype_strings_pairlist.clear();
		}


	}//end of read minima file
	iunit.clear();
	iunit.close();
	//loading dihedral_minima map with data from dihedral minima file
	for(std::list< ligand::Ligand_BondAP >::iterator c_bond=
		rotable_bonds.begin(); c_bond!=rotable_bonds.end();c_bond++){

		ligand::Ligand_AtomAP atom1;
		ligand::Ligand_AtomAP atom2;
		(*c_bond)->bonded_atoms(atom1,atom2);
		std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair_ptr;
		if( atom1 > atom2){
			atom_pair_ptr.first=atom2;
			atom_pair_ptr.second=atom1;
		}else{
			atom_pair_ptr.first=atom1;
			atom_pair_ptr.second=atom2;
		}

		std::pair< atom_chem::LigandAtomType,  atom_chem::LigandAtomType > pair_atomtypes(
			atom1->get_ligand_atom_type(),atom2->get_ligand_atom_type());
		//locating correct entry in tmp_dihedral_minima
		if( tmp_dihedral_minima.count(pair_atomtypes)==0){
			std::pair< atom_chem::LigandAtomType,  atom_chem::LigandAtomType > pair_atomtypes2(
				atom2->get_ligand_atom_type(),atom1->get_ligand_atom_type());
			if(tmp_dihedral_minima.count(pair_atomtypes2)==0){
				std::cout << "Failed to locate minima for these atoms ";
				std::cout << atom1->get_pdb_number() << " " << atom2->get_pdb_number();
				std::cout << " " << atom_chem::ligandatomtype_string[atom1->get_ligand_atom_type()];
				std::cout << " " << atom_chem::ligandatomtype_string[atom2->get_ligand_atom_type()];
				std::cout << std::endl;
			}else{
				pair_atomtypes=pair_atomtypes2;
			}
		}

		//creating entry in dihedral minima
		if( dihedral_minima.find(atom_pair_ptr) == dihedral_minima.end()){
			std::pair<
				std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP >,
				std::vector< std::pair< int, int > > > dihedral_pair;
			dihedral_pair.first=atom_pair_ptr;

			dihedral_pair.second=tmp_dihedral_minima[pair_atomtypes];
			dihedral_minima.insert(dihedral_pair);
		}else{
			//adds the minima to the preexisting vector of minima for this bond
			dihedral_minima[atom_pair_ptr].insert(
				dihedral_minima[atom_pair_ptr].end(),
				tmp_dihedral_minima[pair_atomtypes].begin(),
				tmp_dihedral_minima[pair_atomtypes].end()
			);
		}
	}

	return 1;
}

// =============================================================================
// int prune_ligand_conformations
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
///@brief prunes the given conformation/energy map down to max_conformations
///@param std::multimap< float, std::vector< ligand::LigandInternalCoord > conformations maps
///energy to internal coords
///@param float const energy_delta_cutoff how many energy units above the lowest energy conformations other
///conformations should be allowed
///@param int const num_max_conf maximum number of conformations allowed.
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////

float
Ligand::prune_ligand_conformations(
	std::multimap< float, ligand::LigandInternalCoord > & conformations,
	float const energy_delta_cutoff,
	int const num_max_conf
){
	ligand_conformations.clear();
	float min_energy=(*conformations.begin()).first;
	std::multimap< float, ligand::LigandInternalCoord >::const_iterator last_conf=
		conformations.upper_bound( min_energy+energy_delta_cutoff );
	for(std::multimap< float, ligand::LigandInternalCoord >::iterator c_conf=
		conformations.begin(); c_conf!=last_conf; c_conf++){
		ligand_conformations.push_back((*c_conf).second);
	}

	std::list< ligand::LigandCartesianCoord > possible_conformations;
	std::list< ligand::LigandCartesianCoord > accepted_cartesian_conformations;
	std::list< ligand::LigandCartesianCoord > new_conformations;
	//kwk loading lists of cartesian coord for rms check
	for( size_t l=0; l < ligand_conformations.size(); l++ ){
		change_to_ligand_conformation(l);
		if( l==0){
			accepted_cartesian_conformations.push_back(
			get_ligand_cartesian_conformation(*this));
		}else{
			possible_conformations.push_back(
			get_ligand_cartesian_conformation(*this));
		}
	}

	float min_rms=1000.0;
	min_rms=calc_rms_maximal_minimal_rms(accepted_cartesian_conformations,
		possible_conformations, new_conformations, num_max_conf, minimal_rms_limit);

	conformations.clear();
	for(std::list< ligand::LigandCartesianCoord >::iterator
		c_cart_conf=new_conformations.begin();
		c_cart_conf!=new_conformations.end();
		c_cart_conf++){

		set_ligand_cartesian_conformation(*c_cart_conf,*this);
		std::pair< float, ligand::LigandInternalCoord> p;
		p.first=get_ligand_internal_energy(false);
		p.second=get_ligand_internal_conformation(*this);
		conformations.insert(p);
	}
	return min_rms;
}

// =============================================================================
// int diversify_ligand_conformation
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
///@begin diverisify_ligand_conformation
///
///@brief int diversify_ligand_conformation generates ligand conformations around the current conformation
///by sampling at a desired angular density a specified distance on either side of rotable bonds in the
///current conformation. The resulting conformations are stored in ligand_conformations;
///
///@param std::map< ligand::Ligand_BondAP, int > dihedral_sampling bond-to-sample and number of degrees on
///either side to sample
///
///@param int angular_density frequency in degrees to sampled angle surrounding the dihedral
///
///@param float rms_density specify the minima rms between two conformations
///
///@param size_t const maximum_num_conf
////////////////////////////////////////////////////////////////////////////////

int
Ligand::diversify_ligand_conformations(
	std::map< ligand::Ligand_BondAP, int > dihedral_sampling,
	int angular_density,
	float rms_density,
	size_t maximum_num_conf
){
	//first need to save any information that could be overwritten in the ligand.
	//need to save dihedral sampling angular density, rms_density, which bonds are free to rotate,
	// and dihedral_minima
	float default_rms_density=minimal_rms_limit;
	std::map< std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > , //continued on next line
		std::vector< std::pair < int, int > > > default_dihedral_minima=dihedral_minima;
	int default_angular_density=delta_angular_sampling;
	std::list< bool > default_bond_rotable;
	for( ligand::Bond_Itr cbond=bond_vector.begin();cbond!=bond_vector.end(); cbond++){
		default_bond_rotable.push_back((*cbond)->IsRotable());
	}
	size_t default_max_conformations=maximum_number_of_conformations;
	bool default_use_only_defined_bonds=use_only_defined_bonds;
	std::vector<ligand::LigandInternalCoord > default_base_rotamers=ligand_conformations_base;

	//load in information passed
	delta_angular_sampling=angular_density;
	minimal_rms_limit=rms_density;
	maximum_number_of_conformations=maximum_num_conf;
	dihedral_minima.clear();
	use_only_defined_bonds=true;
	//to change dihedral_minima we want to identify the bonds to be sampled and there current dihedral
	//set only these bonds to rotability=true and load in the appropriate absolute deviations.
	for(ligand::Bond_Itr cbond_vect_iter=bond_vector.begin(); cbond_vect_iter!=bond_vector.end(); cbond_vect_iter++){
		(*cbond_vect_iter)->set_rotability(false);
	}
	for( std::map< ligand::Ligand_BondAP, int >::iterator celem=dihedral_sampling.begin();
		celem!=dihedral_sampling.end(); celem++){

		ligand::Ligand_AtomAP atom1=NULL;
		ligand::Ligand_AtomAP atom2=NULL;
		((*celem).first)->bonded_atoms(atom1,atom2);
		((*celem).first)->set_rotability(true);
		std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair;
		if(atom1 > atom2){
			atom_pair.first=atom2;
			atom_pair.second=atom1;
		}else{
			atom_pair.first=atom1;
			atom_pair.second=atom2;
		}
		std::pair< std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP >, std::pair< int, int > > p;
		p.first=atom_pair;
		(p.second).first=int(((*celem).first)->get_dihedral()*numeric::constants::d::radians_to_degrees);
		(p.second).second=(*celem).second;
	}

	//Having set all the parameters compute the ligand_conformations
	compute_ligand_conformations();

	//now transfer the new conformations into ligand conformations
	ligand_conformations.clear();
	ligand_conformations=ligand_conformations_base;

	//lastly restore overwritten information
	ligand_conformations_base.clear();
	ligand_conformations_base=default_base_rotamers;
	minimal_rms_limit=default_rms_density;
	dihedral_minima=default_dihedral_minima;
	delta_angular_sampling=default_angular_density;
	maximum_number_of_conformations=default_max_conformations;
	use_only_defined_bonds=default_use_only_defined_bonds;
	std::list< bool >::iterator cbond_list_iter=default_bond_rotable.begin();
	for(ligand::Bond_Itr cbond_vect_iter=bond_vector.begin(); cbond_vect_iter!=bond_vector.end(); cbond_vect_iter++){
		(*cbond_vect_iter)->set_rotability((*cbond_list_iter));
		cbond_list_iter++;
	}
	dihedral_minima.clear();
	dihedral_minima=default_dihedral_minima;
	return 1;
}

///  @brief Generate a ligand conformation where each rotatable bond is set
///  to the ideal minimum that is nearest the actual starting value.
///  This produces a ligand "rotamer" as much like the start as possible.
///  @param[out] output_bonds overwritten with the new internal conformation
///  @author Ian W. Davis
void
Ligand::make_conformer_nearest_start(
	ligand::LigandInternalCoord & output_bonds
) {
	// start from our "start" coordinates
	recover_start_coordinates();
	
	//iwd  Use of the word "dihedral" is deceptive in the ligand code.
	//iwd  For instance, this code causes no changes (as expected):
	//iwd    (*rbond)->set_dihedral( (*rbond)->get_dihedral() );
	//iwd  But this similar-sounding code wrecks dihedrals AND bond angles (!):
	//iwd    set_dihedral_degrees(*rbond, numeric::conversions::degrees((*rbond)->get_dihedral()));
	//iwd  As you might expect, set_dihedral_degrees() causes a rotation
	//iwd  about the specified bond.
	//iwd  I believe that dihedral_minima is also indexed by the bond that rotates.
	//iwd  However, get/set_dihedral() refer to the dihedral component
	//iwd  of the internal coordinates for the child atom of the bond,
	//iwd  which is ~ the rotation around the *previous* bond in the atom tree.
	//iwd  This confusion is NOT MY FAULT, by the way...
	
	//std::cout << "Before (length angle dihedral):" << std::endl;
	//for(ligand::Bond_Itr b = bond_vector.begin(); b != bond_vector.end(); ++b) {
	//	std::cout << "  " << F(8,4,(*b)->get_length()) << "  " << F(8,4,(*b)->get_angle()) << "  " << F(8,4,(*b)->get_dihedral()) << std::endl;
	//}
	
	// for each rotatable bond, choose the angle closest to the input angle
	std::list< ligand::Ligand_BondAP > rotable_bonds;
	get_rotable_bonds(rotable_bonds);
	for( std::list< ligand::Ligand_BondAP >::iterator rbond=rotable_bonds.begin(); rbond!=rotable_bonds.end(); rbond++) {
		
		//iwd  Find current value for this rotatable bond, in degrees.
		//iwd  Due to the confusion over "dihedral", this is actually the "dihedral"
		//iwd  measure of the FIRST CHILD BOND of this one, NOT this bond itself:
		ligand::Ligand_AtomAP atom1;
		ligand::Ligand_AtomAP atom2;
		(*rbond)->bonded_atoms(atom1, atom2);
		std::list< ligand::Ligand_BondAP > child_bonds;
		atom2->get_childbonds(child_bonds);
		double start_dihedral_deg = numeric::conversions::degrees( (*child_bonds.begin())->get_dihedral() );
		
		// lookup the mimima for this bond based on the two bonded atoms
		//ligand::Ligand_AtomAP atom1, atom2;
		//(*rbond)->bonded_atoms(atom1,atom2);
		std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair;
		if(atom1 > atom2){
			atom_pair.first=atom2;
			atom_pair.second=atom1;
		}else{
			atom_pair.first=atom1;
			atom_pair.second=atom2;
		}
		std::vector< std::pair< int, int > > minima = dihedral_minima[atom_pair];
		
		// cycle through minima for this bond to find closest
		double min_dihedral_deg = start_dihedral_deg, diff_dihedral_deg = 9999; // dummy vals
		for(std::vector< std::pair< int, int > >::iterator min = minima.begin(); min != minima.end(); min++)
		{
			double start_angle = numeric::nearest_angle_degrees( start_dihedral_deg, double(min->first) );
			double diff = std::abs( start_angle - min->first );
			if(diff < diff_dihedral_deg) {
				diff_dihedral_deg = diff;
				min_dihedral_deg = min->first;
			}
		}
		
		// actually alter our current conformation
		set_dihedral_degrees(*rbond, min_dihedral_deg);
		// Don't do this, it wrecks bond angles b/c internal coords get out of whack:
		//(*rbond)->set_dihedral( numeric::conversions::radians(min_dihedral_deg) );
	}
	
	// regenerate Cartesian coordinates for these dihedrals
	refold_ligand(); // not really necessary?  since we're just storing internal DOFs
	
	// capture current conformation in internal coordinates
	output_bonds.clear();
	output_bonds = get_ligand_internal_conformation(*this);

	//std::cout << "After (length angle dihedral):" << std::endl;
	//for(ligand::Bond_Itr b = bond_vector.begin(); b != bond_vector.end(); ++b) {
	//	std::cout << "  " << F(8,4,(*b)->get_length()) << "  " << F(8,4,(*b)->get_angle()) << "  " << F(8,4,(*b)->get_dihedral()) << std::endl;
	//}
	
}

// =============================================================================
//  float iterative_conformational_search
// =============================================================================

////////////////////////////////////////////////////////////////////////////////
///@begin iterative_conformational_search
///
///@brief this function implements an iterative search through all of the dihedral
///angle minimum of a given ligand. This function is not recommended for searching
///through more than 6 rotable bonds. The combinatorial explosion makes it
///impractical to search through more than 6 rotable bonds. The function returns
///the minimal RMS between any two of the conformation returned in the
///conformations map
///@param[out] std::multimap< float, ligand::LigandInternalCoord > & conformations
///
///@author Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
float
Ligand::iterative_conformation_search(
		std::multimap< float, ligand::LigandInternalCoord > & conformations,
		size_t allowed_number_of_conformations
){
	std::list< ligand::Ligand_BondAP > rotable_bonds;
	get_rotable_bonds(rotable_bonds);
	ligand::LigandInternalCoord current_int_conformation;
	std::vector< std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > > atom_ptr_pair_vector;
	//current_angle_set stores the current minimum being sampled in x
	// the current angle in y and the maximum angle for this minimum in z
	std::list< numeric::xyzVector< int > > current_angle_set;

	//generate the angle_set for the first conformation
	int counter=1;
	for( std::list< ligand::Ligand_BondAP >::iterator rbond=rotable_bonds.begin();
		rbond!=rotable_bonds.end(); rbond++){
		ligand::Ligand_AtomAP atom1, atom2;
		(*rbond)->bonded_atoms(atom1,atom2);
		std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair;
		if(atom1 > atom2){
			atom_pair.first=atom2;
			atom_pair.second=atom1;
		}else{
			atom_pair.first=atom1;
			atom_pair.second=atom2;
		}
		atom_ptr_pair_vector.push_back(atom_pair);
		numeric::xyzVector< int > current_minimum;
		current_minimum.x()=0;
		int start_dihed=((dihedral_minima[atom_pair])[0]).first-((dihedral_minima[atom_pair])[0]).second;
		int end_dihed=((dihedral_minima[atom_pair])[0]).first+((dihedral_minima[atom_pair])[0]).second;
		current_minimum.y()=start_dihed;
		current_minimum.z()=end_dihed;
		current_angle_set.push_back(current_minimum);
		counter++;
	}
	//begin while loop to iteratively test all angle_sets
	bool iterated_through_all_minima=false;
	do{
		//generate a conformation from current_angle_set
		counter=1;
		std::list< ligand::Ligand_BondAP >::iterator c_bond=rotable_bonds.begin();
		for( std::list< numeric::xyzVector< int > >::iterator c_dihed=
			current_angle_set.begin(); c_dihed!=current_angle_set.end(); c_dihed++){

			set_dihedral_degrees((*c_bond),(*c_dihed).y());
			c_bond++;
			counter++;
		}

		// computed different conformations now need to score and possibly store.
		float total_int_energy=get_ligand_internal_energy(false);

		if( total_int_energy < 3*atom_vector.size() ){ // this is an arbitary value it needs to be confirmed
			current_int_conformation=get_ligand_internal_conformation(*this);
			std::pair< float , ligand::LigandInternalCoord > energy_conformation(
				total_int_energy,current_int_conformation);
			conformations.insert(energy_conformation);
			if(conformations.size() > allowed_number_of_conformations*5){
				prune_ligand_conformations(conformations, float(rotable_bonds.size()*2),
					allowed_number_of_conformations);
			}
		}
		//create choose new angle_set
		do{
			(*current_angle_set.rbegin()).y()+=delta_angular_sampling;
			if( (*current_angle_set.rbegin()).y() >
				(*current_angle_set.rbegin()).z()){

				(*current_angle_set.rbegin()).x()++;
				if( (*current_angle_set.rbegin()).x() >=
					int(dihedral_minima[atom_ptr_pair_vector[(current_angle_set.size()-1)]].size()))
				{
					current_angle_set.pop_back();
					if(current_angle_set.empty()){
						iterated_through_all_minima=true;
					}
				}else{
					std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair;
					atom_pair=atom_ptr_pair_vector[current_angle_set.size()-1];
					int cmin=(*current_angle_set.rbegin()).x();
					int start_dihed=((dihedral_minima[atom_pair])[cmin]).first-((dihedral_minima[atom_pair])[cmin]).second;
					int end_dihed=((dihedral_minima[atom_pair])[cmin]).first+((dihedral_minima[atom_pair])[cmin]).second;
					(*current_angle_set.rbegin()).y()=start_dihed;
					(*current_angle_set.rbegin()).z()=end_dihed;
				}
			}else if(current_angle_set.size() < rotable_bonds.size()){
				std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair;
				atom_pair=atom_ptr_pair_vector[current_angle_set.size()];
				numeric::xyzVector< int > current_minimum;
				current_minimum.x()=0;
				int start_dihed=((dihedral_minima[atom_pair])[0]).first-((dihedral_minima[atom_pair])[0]).second;
				int end_dihed=((dihedral_minima[atom_pair])[0]).first+((dihedral_minima[atom_pair])[0]).second;
				current_minimum.y()=start_dihed;
				current_minimum.z()=end_dihed;
				current_angle_set.push_back(current_minimum);
			}
		}while( current_angle_set.size() < rotable_bonds.size() && !current_angle_set.empty());
	}while(!iterated_through_all_minima);//iterate through all minima
	//kwk selection criteria are arbitary and have not been validated

	float min_rms=prune_ligand_conformations( conformations, float(rotable_bonds.size()*2),
		allowed_number_of_conformations);
	return min_rms;
}

////////////////////////////////////////////////////////////////////////////////
///@begin monte_carlo_conformation_search
///@brief This conformational search picks an number of random conformations for
///small molecule from the minima stored in the dihedral_minima member of the
///ligand class. The number of conformations sampled is dependent on the number
///of rotable bonds. This is the default mode of conformational search as the
///iterative search method can lead to excessively long search times.
///@param[out] std::multimap< float, ligand::LigandInternalCoord > & conformations
///@param[in] size_t allowed_number_of_conformations
///@authors Kristian Kaufmann
////////////////////////////////////////////////////////////////////////////////
float
Ligand::monte_carlo_conformation_search(
	std::multimap < float, ligand::LigandInternalCoord > & conformations,
	size_t allowed_number_of_conformations
){
	//first get the rotable_bonds
	std::list< ligand::Ligand_BondAP > rotable_bonds;
	get_rotable_bonds(rotable_bonds);
	
	//setup for random conformation selection
	std::vector< std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > > atom_ptr_pair_vector;
	//the follwing mirrors the accessor of dihedral_minima ligand member
	for( std::list< ligand::Ligand_BondAP >::iterator rbond=rotable_bonds.begin();
		rbond!=rotable_bonds.end(); rbond++){
		ligand::Ligand_AtomAP atom1, atom2;
		(*rbond)->bonded_atoms(atom1,atom2);
		std::pair< ligand::Ligand_AtomAP, ligand::Ligand_AtomAP > atom_pair;
		if(atom1 > atom2){
			atom_pair.first=atom2;
			atom_pair.second=atom1;
		}else{
			atom_pair.first=atom1;
			atom_pair.second=atom2;
		}
		atom_ptr_pair_vector.push_back(atom_pair);
	}
	
	//continue search until std::min of 1000 or 10^(3+rotable_bonds.size())
	int power_of_ten=std::min(4, 2+int(rotable_bonds.size()));
	for ( int trial = 0; trial < int(std::pow(10.0,power_of_ten)); trial++ ) {
		//first choose the random angle set.
		size_t counter=0;
		for(std::list< ligand::Ligand_BondAP >::iterator rbond=rotable_bonds.begin();
				rbond!=rotable_bonds.end(); rbond++){
			//select a random minimum for this bond
			int random_minimum=random_range(0, int(dihedral_minima[atom_ptr_pair_vector[counter]].size())-1);
			float minima_width=float(dihedral_minima[atom_ptr_pair_vector[counter]][random_minimum].second);
			int random_delta = int(ran3()*(2*minima_width-minima_width));
			//set dihedral angle
			set_dihedral_degrees((*rbond),dihedral_minima[atom_ptr_pair_vector[counter]][random_minimum].first+random_delta);
			counter++;
		}
		//score current conformation
		float total_int_energy=get_ligand_internal_energy(false);
		if( total_int_energy < 3*atom_vector.size() ){ // this is an arbitary value it needs to be confirmed
			ligand::LigandInternalCoord current_int_conformation=get_ligand_internal_conformation(*this);
			std::pair< float , ligand::LigandInternalCoord > energy_conformation(
				total_int_energy,current_int_conformation);
			conformations.insert(energy_conformation);
			if(conformations.size() > allowed_number_of_conformations*5){
				prune_ligand_conformations(conformations, float(rotable_bonds.size()*2),
					allowed_number_of_conformations);
			}
		}
	}
	
	float min_rms=prune_ligand_conformations(conformations, float(rotable_bonds.size()*2),
			allowed_number_of_conformations);
	
	return min_rms;
}

// =============================================================================
//     int change_to_ligand_conformation
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin change_to_ligand_conformation
///
/// @brief
///    switches to alternate conformation of the ligand
///
/// @detailed
///
/// @param
///
/// @global_read
///   reads from ligand_conformations
/// @global_write
///   writes to atom_vector
///		writes to bond_vector
/// @remarks
///
/// @references
///
/// @authors
///   Kristian Kaufmann
/// @last_modified  05-10-2006
////////////////////////////////////////////////////////////////////////////////

int
Ligand::change_to_ligand_conformation(
	size_t x
){
	if( x < ligand_conformations.size()){
		if(ligand_conformations[x].size()!=bond_vector.size()){
			std::cout << "conformation in ligand_conformations has more vectors than allowed for this small molecule" << std::endl;
			return -2;
		}
		//retrieving internal coordinates
		for( size_t i=0; i<bond_vector.size(); i++){
			bond_vector[i]->set_coordinates((ligand_conformations[x])[i]);
		}
		if( refold_ligand()!=1){
			std::cout << "Can't refold ligand in change_to_ligand_conformation" << std::endl;
			return -3;
		};
//kwk			recompute_ligand_interface=true;
		return 1;
	}

	std::cout << "change_to_ligand_conformation was passed a out of range value." << std::endl;
	return -1;
}

// =============================================================================
//     void get_bonded_atoms
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin get_bonded_atoms
///
/// @brief
///
///
/// @detailed
///
/// @param [out] bonded_atoms
/// @param [in] first_atom
///
/// @global_read
///
/// @global_write
/// @remarks
///
/// @references
///
/// @authors
///   Kristian Kaufmann
/// @last_modified  05-10-2006
////////////////////////////////////////////////////////////////////////////////
void
Ligand::get_bonded_atoms(
	std::list< ligand::Ligand_AtomAP > & bonded_atoms,
	ligand::Ligand_AtomAP const first_atom
){
		bonded_atoms.clear();
		ligand::Ligand_AtomAP atom1;
		ligand::Ligand_AtomAP atom2;
		ligand::Ligand_BondAP bond1;
		std::list< ligand::Ligand_BondAP > bond_child;
		if(!first_atom->Is_Root_Atom()){
			first_atom->get_parentbond(bond1);
			bond1->bonded_atoms(atom1,atom2);
			if( atom2==first_atom){
				bonded_atoms.push_back(atom1);
		}else{
			bonded_atoms.push_back(atom2);
		}
	}
	first_atom->get_childbonds(bond_child);
	for(std::list< BondSmallMoleculeAP >::iterator current_bond=bond_child.begin();
		current_bond!=bond_child.end(); current_bond++){

		(*current_bond)->bonded_atoms(atom1,atom2);
		if( atom1==first_atom){
				bonded_atoms.push_back(atom2);
		}else{
			bonded_atoms.push_back(atom1);
		}
	}
	return;
}

///////////////////////////////////////////////////////////////////////////////
///@begin int rotamerize_ligand
///
///@brief this function changes the conformation of the ligand to the
///conformation with lowest rmsd within the MSAMP rotamerize option defined range
///in ligand_conformations or ligand_conformations_base if ligand_conformations
///is empty
///
///@author Kristian Kaufmann
///////////////////////////////////////////////////////////////////////////////
int
Ligand::rotamerize_ligand(
){

	float rmsd_nearest=100000.0; //this value doesn't matter it is erased in the next function anyway
	size_t new_conformation_index=ligand_conformations.size(); //sets to an imposible value
	ligand::LigandCartesianCoord current_conformation=get_ligand_cartesian_conformation(*this);
	new_conformation_index=find_nearest_conformation_by_rmsd( current_conformation, rmsd_nearest );
	if( new_conformation_index < ligand_conformations.size() ){
		change_to_ligand_conformation(new_conformation_index);
		std::cout << "Rotamerize ligand has an rmsd of " << rmsd_nearest;
		std::cout << "to the passed conformation." << std::endl;
	}else{
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	return 1;
}

///////////////////////////////////////////////////////////////////////////////
///@begin size_t find_nearest_conformation_by_rmsd
///
///@brief this function will return the size_t index of the conformation in the
///ligand rotamer ensemble that is closest to the passed conformation it also
///stores the rmsd of the found conformation to the passed conformation in the
///float passed.
///
///@param[in] ligand::LigandCartesianCoord passed_conformation
///@param[out] float rmsd
///
///@author Kristian Kaufmann
///
///////////////////////////////////////////////////////////////////////////////
size_t
Ligand::find_nearest_conformation_by_rmsd(
	ligand::LigandCartesianCoord passed_conformation,
	float & rmsd
){
	
	//saving_conformational state of ligand
	ligand::LigandInternalCoord saved_conf=get_ligand_internal_conformation(*this);
	
	rmsd = -9.9;
	std::multimap< float, size_t > rmsd_conf_index_map;
	if( ligand_conformations.empty() ){
		ligand_conformations=ligand_conformations_base;
	}
	
	//find rmsd of ligand rotamers to the passed conformation
	for( size_t conf_index=0; conf_index<ligand_conformations.size();conf_index++){
		change_to_ligand_conformation(conf_index);
		ligand::LigandCartesianCoord library_conformation=get_ligand_cartesian_conformation( *this );
		float min_rmsd=calc_rms_wrapper( passed_conformation, library_conformation);
		//if a conformation hasn't been selected or if the newest library conformation is closer to
		//the passed conformation update the index and rmsd.
		std::pair< float, size_t > rmsd_conf( min_rmsd, conf_index );
		rmsd_conf_index_map.insert(rmsd_conf);
	}


	//check that a rotamer exists within the desired rmsd bounds
	if( rmsd_conf_index_map.end() ==
					rmsd_conf_index_map.lower_bound( rotamerize_ligand_lower_bound ) ||

			rmsd_conf_index_map.begin() ==
					rmsd_conf_index_map.upper_bound( rotamerize_ligand_upper_bound )
		){

		std::cout << "Did not find a ligand rotamer with an rmsd  between ";
		std::cout << rotamerize_ligand_lower_bound << " and " << rotamerize_ligand_upper_bound;
		std::cout  << " angstrom RMSD." << std::endl;
		std::cout << "Try increasing the range using the MSAMP rotamerize options " << std::endl;
		std::cout << "in the mdlfile to include RMSD in the range of ";
		std::cout << (*rmsd_conf_index_map.begin()).first << " to ";
		std::cout << (*rmsd_conf_index_map.rbegin()).first << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//set the rmsd of the found rotmaer
	rmsd= (*rmsd_conf_index_map.lower_bound(rotamerize_ligand_lower_bound)).first;
	
	//recovering input conformation
	ligand_conformations.push_back(saved_conf);
	change_to_ligand_conformation(ligand_conformations.size()-1);
	ligand_conformations.pop_back();
	
	//return index of closests ligand rotamer
	return (*rmsd_conf_index_map.lower_bound(rotamerize_ligand_lower_bound)).second;

}






// =============================================================================
//     void optimize_H
// =============================================================================

//////////////////////////////////////////////////////////////////////////////
/// @begin optimize_H
///
/// @brief Optimizes hydrogens according to the internal energy function
/// used primarly to get rid of clashes in the input structure.  Works
/// by rotating bonds that only effect hydrogens. Isn't a comprehensive approach.
/// should be replaced by a gradient minimization once the atom tree is integrated
/// into ligand usage.
///
/// @detailed
///
/// @param [in] first_atom
///
/// @global_read
///
/// @global_write
/// @remarks
///
/// @references
///
/// @authors
///   Kristian Kaufmann
/// @last_modified  06-26-2006
////////////////////////////////////////////////////////////////////////////////
void
Ligand::optimize_H(
){
	std::cout << "Entering ligand optimize_H" << std::endl;
	for( ligand::Atom_Itr c_atom=atom_vector.begin(); c_atom!=atom_vector.end();c_atom++){
		std::list< ligand::Ligand_BondAP > childbonds;
		ligand::Ligand_AtomAP atom1, atom2;
		bool all_hydrogens=true;
		(*c_atom)->get_childbonds(childbonds);
		if( childbonds.size() !=0 ){
			for( std::list< ligand::Ligand_BondAP >::iterator c_bond=childbonds.begin();
					c_bond!=childbonds.end();c_bond++){

				(*c_bond)->bonded_atoms(atom1,atom2);
				if(atom2->get_element_type()!=atom_chem::HYDROGEN){
					all_hydrogens=false;
				}
			}
			if(all_hydrogens){
				ligand::LigandCartesianCoord best_conformation;
				ligand::Ligand_BondAP pbond;
				(*c_atom)->get_parentbond(pbond);
				float best_energy, current_energy;
				best_conformation=get_ligand_cartesian_conformation( *this );
				get_ligand_internal_energy(false);
				best_energy=param_pack::packer_misc_parameters::Wlig_int_rep *lig_int_repE
				+ param_pack::packer_misc_parameters::Wlig_int_atr *lig_int_atrE
				+ param_pack::packer_misc_parameters::Wlig_int_solv *lig_int_solvE
				+ param_pack::packer_misc_parameters::Wlig_int_dihed *lig_int_dihedE
				+ param_pack::packer_misc_parameters::Wlig_int_hb*lig_int_hbE
				+ param_pack::packer_misc_parameters::Wlig_int_cou*lig_int_coulombE;
				double delta_dihedral=3*numeric::constants::d::pi_over_180;
				pbond->set_rotability(true);
				for( size_t i=0; i<=360/childbonds.size(); i+=3){
					twist_dihedral_radians(delta_dihedral,*pbond);
					get_ligand_internal_energy(false);
					current_energy=param_pack::packer_misc_parameters::Wlig_int_rep *lig_int_repE
					+ param_pack::packer_misc_parameters::Wlig_int_atr *lig_int_atrE
					+ param_pack::packer_misc_parameters::Wlig_int_solv *lig_int_solvE
					+ param_pack::packer_misc_parameters::Wlig_int_dihed *lig_int_dihedE
					+ param_pack::packer_misc_parameters::Wlig_int_hb*lig_int_hbE
					+ param_pack::packer_misc_parameters::Wlig_int_cou*lig_int_coulombE;
					if( current_energy < best_energy ){
						best_energy=current_energy;
						best_conformation=get_ligand_cartesian_conformation( *this );
					}
				}
				set_ligand_cartesian_conformation( best_conformation, *this);
				pbond->set_rotability(false);
			}
		}
	}
	std::cout << "Exiting ligand optimize_H" << std::endl;
	return;
}

//Performs setup of aaproperties_pack arrays to represent the ligand as a residue
//Alters nchi, first_scatom, HNpos, HApos, nh20, nheavyatoms, nchi, nH_polar,
// nH_arotmatic, nH_apolar, atom_name, atom_base, abase2, cp_atom_num


int
Ligand::setup_ligand_res(
	int const aa, // lig amino acid type
	int const aav, // lig amino acid variant set to 1
	bool const read_occ_weights,
	bool const read_bvalue_charge,
	FArray3D_float & coords // returns current coordinates
){

	assert( MAX_AA()() >= aa );
	assert( param_aa::is_ligand(aa) && aav ==1);
	//these assert prevent passage of parameters.
	enable_ligaa_ns::ligand_ptr_array(param_aa::has_ligand_no(aa),aav)=this;
	typedef std::vector< ligand::Ligand_AtomAP > Atom_Vect;
	typedef Atom_Vect::iterator Atom_iter;
	enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa),aav).reserve(atom_vector.size());
	for(int a_ind=0; a_ind< int(atom_vector.size()); a_ind++){
		enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa),aav).push_back(9999);
	}

	// default
	aaproperties_pack::aa_variant_properties::variant_type(1,aa,aav)=true;
	aaproperties_pack::nchi( aa, aav ) = 0; // This value will be updated if using flexible_ligand
	aaproperties_pack::first_scatom( aa, aav ) = 1; //By default all atoms in a ligand are sidechain atoms
	aaproperties_pack::HNpos( aa, aav ) = -1; //ligands don't have a HN hydrogen
	aaproperties_pack::HApos( aa, aav ) = -1; //ligands don't have a HA hydrogen
	aaproperties_pack::nh2o( aa, aav ) = 0; //At this time ligands don't have waters
	aaproperties_pack::nvar(aa)=1;


	int atom_count = 0;
	FArray1D< ligand::Ligand_AtomAP > atom_array( atom_vector.size());
	for( int i = 1; i <=int(atom_vector.size()) ;i++){
		atom_array(i)=NULL;
	}
	//get heavy atoms
	atom_count=fill_in_heavy_atoms( atom_array, aa, aav );
	aaproperties_pack::nheavyatoms(aa,aav) = atom_count;
	// now fill in the hydrogens and Virtual atoms
	atom_count +=fill_in_hydrogens_and_virtual_atoms( atom_array, atom_count, aa, aav);

	std::cout << "Found " << atom_count << " total atoms." << std::endl;
	if ( atom_count > MAX_ATOM()() ) {
		std::cout << "Exiting! ligand atom count > MAX_ATOM()() " <<
			atom_count << ' ' << MAX_ATOM()() << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	// this function modifies and fills ligand_chi_atoms ligand_chi_required
	// ligand_chi_rigid_atom_list ligand_length_chi_rigid_atom_list
	enable_ligaa_ns::ligand_nchi(param_aa::has_ligand_no(aa),aav)=set_up_chi_arrays( aa, aav);

	// fill in the easy stuff
	aaproperties_pack::natoms( aa, aav ) = atom_count;


	//intializing
	aaproperties_pack::nH_polar( aa, aav ) = 0;
	aaproperties_pack::nH_apolar( aa, aav ) = 0;
	aaproperties_pack::nH_aromatic( aa, aav ) = 0;

	//filling in icoord, lig_icoord, and Hydrogen type arrays,
	// reads occupancy weights, reads bvalue into charge if appropriate flags are set
	for ( int i=1; i<= atom_count; ++i ) {
		Atom & atom( *atom_array(i) );
		aaproperties_pack::atom_name(i,aa,aav)=atom.get_atom_name();
		aaproperties_pack::fullatom_type( i, aa, aav ) = atom.get_ros_atom_type();
		aaproperties_pack::atomic_charge( i, aa, aav ) = atom.get_partial_charge();
		int const type( aaproperties_pack::fullatom_type( i, aa, aav ) );
		if ( type == 22 || type == 25 ) {
			aaproperties_pack::Hpos_polar( ++aaproperties_pack::nH_polar( aa, aav ), aa, aav ) = i;
		} else if ( type == 23 ) {
			aaproperties_pack::Hpos_apolar( ++aaproperties_pack::nH_apolar( aa, aav ), aa, aav ) = i;
		} else if ( type == 24 ) {
			aaproperties_pack::Hpos_aromatic( ++aaproperties_pack::nH_aromatic( aa, aav ), aa, aav ) = i;
		}

		for ( int k=1; k<= 3; ++k ) {
			enable_ligaa_ns::lig_icoord( k, i, param_aa::has_ligand_no(aa) ) =
				atom_array(i)->get_coordinates()(k);
			coords( k, i , param_aa::has_ligand_no(aa))=atom_array(i)->get_coordinates()(k);
			aaproperties_pack::icoor( k, i, aa, aav ) = atom_array(i)->get_coordinates()(k);
		}
		if( read_occ_weights ) {
			enable_ligaa_ns::lig_iocc_weight( i, param_aa::has_ligand_no(aa) ) =
				atom_array(i)->get_occupancy();
		} else {
			enable_ligaa_ns::lig_iocc_weight( i, param_aa::has_ligand_no(aa) ) = 1.0;
		}
		if( read_bvalue_charge ) {
		  enable_ligaa_ns::lig_icharge( i, param_aa::has_ligand_no(aa)  ) =
		  		atom_array(i)->get_bvalue();
		} else {
			enable_ligaa_ns::lig_icharge( i, param_aa::has_ligand_no(aa) ) =  0.0;
		}
	} // i



	//fill in bonded_neighbor arrays in aaproperties_pack
	setup_bonded_neighbors_arrays( atom_count, aa, aav );

	//hbond acceptors fill in hbond acceptor arrays
	aaproperties_pack::nacceptors( aa, aav )=fill_acceptor_arrays( atom_count, aa, aav);

	//fill in rotamer arrays for ligand rotamers in enable_ligand_aa namespace
	enable_ligaa_ns::nligandrotamers(aa,aav)=fill_ligand_rotamer_array( atom_array, aa, aav);
	ligand::ligand_flexible=false;
	ligand::ligand_flag=false;
	std::cout << "Exiting setup_ligand_res" << std::endl;
	return 1;
}




// This function returns an FArray1D of Atom pointers ordered as they should appear in
// pose aa  vector. It also provides a mapping for interpreting mdl bonding information

int
Ligand::fill_in_heavy_atoms(
	FArray1D< Atom * > & atom_array_for_pose_res,
	int const aa,
	int const aav
){
	int atom_count = 0;
	int counter = 0;

//if not using flexible ligand rotamers then you don't need to find out the three base atoms
// or the backbone atoms to build the rotamers

	if( bond_vector.empty() ){
		for( ligand::Atom_Itr atom_iter=atom_vector.begin(), atom_end = atom_vector.end();
				 atom_iter != atom_end; ++atom_iter ) {
			Atom & atom( **atom_iter );
			if ( atom.get_element_type()!=atom_chem::HYDROGEN && !atom.Is_Virtual() ) {
				atom_array_for_pose_res( ++atom_count ) = &atom;
				enable_ligaa_ns::ligand_ligaa_atom_map(
						param_aa::has_ligand_no(aa),aav)[counter] = atom_count;
				enable_ligaa_ns::ligaa_ligand_atom_map(atom_count,param_aa::has_ligand_no(aa),aav)=counter;
			}
			counter++;
		}
	}else{
// If you are going to use ligand rotamers then you need to select the three base atom
// that define the reference frame of the ligand rotamers. The backbone atoms perform this
// function in aa side chain rotamers.

// The reasoning behind choosing the first three atoms is based around wanting to avoid
// recalculating dihedral angles stored in the bonds. To accomplish this, the first three
// atoms will be chosen to ensure that they maintain the same internal coordinates in
// all the ligand rotamers. Additionally due to the organization of residues in rosetta
// they must all be heavy atoms.

//if this is an rotamer ready ligand all atoms must be connected.

		assert( trial_roots.size() == 1);


		ligand::Ligand_AtomAP atom_1=NULL;
		ligand::Ligand_AtomAP atom_2=NULL;
		ligand::Ligand_AtomAP atom_3=NULL;
		get_base_atoms( atom_1, atom_2, atom_3);

//filling in atom_array_for_pose_res and ligand_ligaa_atom_map for first three atoms.

		counter=0;
		for(ligand::Atom_Itr atom_iter=atom_vector.begin(); atom_iter!=atom_vector.end();
			atom_iter++){

			if( (*atom_iter) == atom_1){
				atom_array_for_pose_res(1)=(*atom_iter);
				enable_ligaa_ns::ligand_ligaa_atom_map(
						param_aa::has_ligand_no(aa),aav)[counter] = 1;
				enable_ligaa_ns::ligaa_ligand_atom_map(1,param_aa::has_ligand_no(aa),aav)=counter;
			}
			if( (*atom_iter) == atom_2){
				atom_array_for_pose_res(2)=(*atom_iter);
				enable_ligaa_ns::ligand_ligaa_atom_map(
						param_aa::has_ligand_no(aa),aav)[counter] = 2;
				enable_ligaa_ns::ligaa_ligand_atom_map(2,param_aa::has_ligand_no(aa),aav)=counter;
			}
			if( (*atom_iter) == atom_3){
				atom_array_for_pose_res(3)=(*atom_iter);
				enable_ligaa_ns::ligand_ligaa_atom_map(
						param_aa::has_ligand_no(aa),aav)[counter] = 3;
				enable_ligaa_ns::ligaa_ligand_atom_map(3,param_aa::has_ligand_no(aa),aav)=counter;
			}
			counter++;
		}
//filling in the rest of the array

		atom_count=3;
		counter=0;
		for( ligand::Atom_Itr atom_iter=atom_vector.begin(); atom_iter!=atom_vector.end();
			atom_iter++){

			if( (*atom_iter)!= atom_1 && (*atom_iter)!=atom_2 && (*atom_iter)!=atom_3
				&& (*atom_iter)->get_element_type()!=atom_chem::HYDROGEN && !(*atom_iter)->Is_Virtual() ){

				atom_array_for_pose_res(++atom_count)=(*atom_iter);
				enable_ligaa_ns::ligand_ligaa_atom_map(
						param_aa::has_ligand_no(aa),aav)[counter] = atom_count;
				enable_ligaa_ns::ligaa_ligand_atom_map(atom_count,param_aa::has_ligand_no(aa),aav)=counter;
			}
			counter++;
		}
}
return atom_count;

}

//This function fills in the FArray passed with the pointers of atom to hydrogens and virtual atoms
// starting at last_assigned_index+1 as well as filling in the ligand ligaa atom maps.
// The function also orders the hydrogens according to the atom in the array that they are
// bonded to i.e. hydrogens bonded to the ith atom would be grouped together before the hydrogens
// bonded to the jth atom. Virtual Atoms are placed at the end of the array.
// It returns the number of hydrogens and virtual atoms found.

int
Ligand::fill_in_hydrogens_and_virtual_atoms(
	FArray1D< Atom * > & atom_array_for_aaproperties,
	int const last_assigned_index,
	int const aa,
	int const aav
){

	assert( trial_roots.size()==1);
	//we require a connected graph
	int num_hydrogens_and_virtual_atoms=0;
	int atom_count=last_assigned_index;
	std::map< ligand::Ligand_AtomAP, int > atom_ptr_map;
	int counter=0;
	for( ligand::Atom_Itr atom_iter=atom_vector.begin();
		atom_iter!=atom_vector.end(); atom_iter++){

		std::pair< ligand::Ligand_AtomAP, int > p((*atom_iter),counter);
		atom_ptr_map.insert(p);
		counter++;
	}
	std::list< ligand::Ligand_AtomAP > bonded_atoms;
	for( int i=1; i<=last_assigned_index; i++){

		get_bonded_atoms( bonded_atoms, atom_array_for_aaproperties(i));
		for(std::list< ligand::Ligand_AtomAP >::iterator c_atom=bonded_atoms.begin(),
			end_cond=bonded_atoms.end(); c_atom!=end_cond; c_atom++){

			if( (*c_atom)->get_element_type() == atom_chem::HYDROGEN ){
				atom_array_for_aaproperties(++atom_count)=(*c_atom);
				num_hydrogens_and_virtual_atoms++;
				enable_ligaa_ns::ligand_ligaa_atom_map(
						param_aa::has_ligand_no(aa),aav)[atom_ptr_map[(*c_atom)]] = atom_count;
				enable_ligaa_ns::ligaa_ligand_atom_map(atom_count,param_aa::has_ligand_no(aa),aav)=atom_ptr_map[(*c_atom)];
				aaproperties_pack::hydrogens_on_atm(++aaproperties_pack::nhydrogens_on_atm(i,aa,aav),i,aa,aav)=atom_count;
			}
		}
	}
	for( ligand::Atom_Itr atom_iter=atom_vector.begin();
		atom_iter!=atom_vector.end(); atom_iter++){

		if( (*atom_iter)->Is_Virtual()){
			atom_array_for_aaproperties(++atom_count)=(*atom_iter);
			num_hydrogens_and_virtual_atoms++;
			enable_ligaa_ns::ligand_ligaa_atom_map(
					param_aa::has_ligand_no(aa),aav)[atom_ptr_map[(*atom_iter)]] = atom_count;
			enable_ligaa_ns::ligaa_ligand_atom_map(atom_count,param_aa::has_ligand_no(aa),aav)=atom_ptr_map[(*atom_iter)];
		}
	}

	return num_hydrogens_and_virtual_atoms;



}


// the purpose of the function is to fill the chi_related arrays found in aaproperties_pack
// the arrays effected are chi_atoms, chi_required, length_chi_rigid_atom_list,
//  chi_rigid_atom_list
int
Ligand::set_up_chi_arrays(
	int const aa,
	int const aav
){

	assert( param_aa::is_ligand(aa) && aav ==1 && param_aa::has_ligand_no(aa) <= enable_ligaa_ns::MAX_LIGAA());
	//this catches the chance that someone might unwitingly alter the arrays of other
	// aa

	//ligand_ligaa_atom_map provides the mapping from atom_vector index to atom_array index
	// ie. ligand_ligaa_atom_map[atom_vector.index]= atom_array.index


	//four arrays have to be filled in this function.
	//For each chi angle or rotable bond 4 atoms defining the chi angle.
	// this atom indices are stored in --chi_atoms--
	// they are then used to generate the needed rotation matrix to apply to
	// all the downstream atoms listed in --chi_required--.
	//The --chi_rigid_atom_list-- and --length_chi_rigid_atom_list-- contain the number and
	// atom positions of which chi is the chi farthest from the base on which the
	// atoms depend

	//First setup chi and chi_required


	//get a vector of rotable bounds
	std::list< ligand::Ligand_BondAP > rotable_bonds;


	int number_rotable_bonds=get_rotable_bonds(rotable_bonds);

	assert( number_rotable_bonds <= enable_ligaa_ns::MAX_LIGAND_CHI );



	//setting up chi_atoms for each rotable bond
	int current_chi=0;
	for(  std::list< ligand::Ligand_BondAP >::iterator cbond=rotable_bonds.begin();
		cbond!=rotable_bonds.end(); cbond++){

		ligand::Ligand_AtomAP chi_atom1=NULL;
		ligand::Ligand_AtomAP chi_atom2=NULL;
		ligand::Ligand_AtomAP chi_atom3=NULL;
		ligand::Ligand_AtomAP chi_atom4=NULL;

		current_chi++;
		(*cbond)->bonded_atoms(chi_atom2,chi_atom3);
		if(!chi_atom2->Is_Root_Atom()){
			ligand::Ligand_BondAP parent_b=NULL;
			chi_atom2->get_parentbond(parent_b);
			parent_b->bonded_atoms(chi_atom1,chi_atom2);
		}else{
			std::list< ligand::Ligand_AtomAP > rootchildatoms;
			get_bonded_atoms(rootchildatoms, chi_atom2);
			for( std::list< ligand::Ligand_AtomAP >::iterator c_atom=rootchildatoms.begin();
				c_atom!=rootchildatoms.end(); c_atom++){
				if( (*c_atom)!=chi_atom3 && (*c_atom)->get_element_type() != atom_chem::HYDROGEN &&
					!(*c_atom)->Is_Virtual()){

					chi_atom1=(*c_atom);
					break;
				}
			}
		}
		//If chi_atom1 is still NULL after this then we have problems.
		assert( chi_atom1!=NULL);
		std::list< ligand::Ligand_AtomAP > gcatoms;
		get_bonded_atoms(gcatoms,chi_atom3);
		for(std::list< ligand::Ligand_AtomAP >::iterator c_atom=gcatoms.begin();
			c_atom!=gcatoms.end();c_atom++){

			if( (*c_atom)->get_element_type() !=atom_chem::HYDROGEN && !(*c_atom)->Is_Virtual() &&
				(*c_atom)!=chi_atom1 && (*c_atom)!= chi_atom2){

				chi_atom4=(*c_atom);
				break;
			}
		}
		assert(chi_atom4!=NULL);

		//If chi_atom4 is still Null we have problems.
	

		//ligand_chi_atoms(4,chi,ligaa_num,aav)
		//ligand_chi_required(chi, atom, ligaa_num, aav)
		int counter=-1;
		int branch_size=get_size_of_branch((*cbond));
		for(ligand::Atom_Itr c_atom=atom_vector.begin();
			c_atom!=atom_vector.end();c_atom++){

			counter++;
			if( chi_atom1==(*c_atom)){
				enable_ligaa_ns::ligand_chi_atoms(1,current_chi,param_aa::has_ligand_no(aa),aav)=
					enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[counter];
			}
			if( chi_atom2==(*c_atom)){
				enable_ligaa_ns::ligand_chi_atoms(2,current_chi,param_aa::has_ligand_no(aa),aav)=
					enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[counter];
			}
			if( chi_atom3==(*c_atom)){
				enable_ligaa_ns::ligand_chi_atoms(3,current_chi,param_aa::has_ligand_no(aa),aav)=
					enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[counter];
			}
			if( chi_atom4==(*c_atom)){
				enable_ligaa_ns::ligand_chi_atoms(4,current_chi,param_aa::has_ligand_no(aa),aav)=
					enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[counter];
			}
			if( chi_atom3->get_index() < (*c_atom)->get_index() &&
				(*c_atom)->get_index() < branch_size+chi_atom3->get_index()){
				enable_ligaa_ns::ligand_chi_required(current_chi,
						enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[counter],
						param_aa::has_ligand_no(aa),aav)=true;
			}
		}
	}

	//Now for the --length_chi_rigid_atom_list(jth_chi,aa,aav)-- and
	//--chi_rigid_atom_list(ith_atom,jth_chi,aa,aav)--

	int atom_counter=-1;
	for(ligand::Atom_Itr c_atom=atom_vector.begin();c_atom!=atom_vector.begin();
		c_atom++){

		atom_counter++;
		int chi_current=0;
		int previous_index=0;
		int chi_rigid=-1;
		int size=0;
		ligand::Ligand_AtomAP atom1=NULL;
		ligand::Ligand_AtomAP atom2=NULL;
		int c_atom_index=(*c_atom)->get_index();
		for(std::list< ligand::Ligand_BondAP >::iterator c_bond=rotable_bonds.begin();
			c_bond!=rotable_bonds.end(); c_bond++){

			chi_current++;
			size=get_size_of_branch((*c_bond));
			(*c_bond)->bonded_atoms(atom1,atom2);
			if( atom2->get_index() > previous_index && c_atom_index > atom2->get_index() &&
				c_atom_index < atom2->get_index()+size){
				previous_index=atom2->get_index();
				chi_rigid=chi_current;
			}
		}
		if( chi_rigid!= -1){
			enable_ligaa_ns::ligand_length_chi_rigid_atom_list(
				chi_rigid, param_aa::has_ligand_no(aa), aav)=
				enable_ligaa_ns::ligand_length_chi_rigid_atom_list(
				chi_rigid, param_aa::has_ligand_no(aa), aav)+1;
			enable_ligaa_ns::ligand_chi_rigid_atom_list(
				enable_ligaa_ns::ligand_length_chi_rigid_atom_list(
				chi_rigid, param_aa::has_ligand_no(aa), aav),chi_rigid,param_aa::has_ligand_no(aa),aav)=enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[atom_counter];
		}

	}

	return number_rotable_bonds;
}



//The following function sets the three access pointers passed to the atom
// addresses to three heavy atoms at the root of the ligand object fold
// graph.

void
Ligand::get_base_atoms(
	ligand::Ligand_AtomAP & atom_1,
	ligand::Ligand_AtomAP & atom_2,
	ligand::Ligand_AtomAP & atom_3
){

	if( atom_1 != NULL || atom_2!= NULL || atom_3!= NULL){
		std::cout << "Exiting!!!! get_base_atoms was passed non NULL pointers" << std::endl;
		//this poses a danger of a memory leak since the pointers are reassign without being
		// deleted. And you don't want to delete them since this fuction doesn't know what is
		// located under them

		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	typedef std::list< ligand::Ligand_AtomAP >::iterator Atom_List_Itr;

	if( trial_roots[0]->get_element_type() != atom_chem::HYDROGEN &&
		trial_roots[0]->get_element_type() !=atom_chem::VIRTUAL){

		atom_1=trial_roots[0];
	}else{
		// root atom is a hydrogen this condition is disallowed except in the case where
		// a hydrogen atom is specified as root by the user using the MROOT flag in the
		// mdl file
		std::list< ligand::Ligand_AtomAP > root_neighbors;
		get_bonded_atoms( root_neighbors, trial_roots[0] );
		for( Atom_List_Itr root_n=root_neighbors.begin();
			root_n != root_neighbors.end(); root_n++ ){

			if( (*root_n)->get_element_type() != atom_chem::HYDROGEN &&
				(*root_n)->get_element_type() !=atom_chem::VIRTUAL){

				atom_1=(*root_n);
				break;
			}
		}
		if( atom_1 == NULL ){
			//was unable to find a non-hydrogen atom in root or atoms bonded to root.

			std::cout << "Exiting!!! Unable to find suitable base atoms." << std::endl;
			std::cout << "There must be at least three heavy atoms in " << std::endl;
			std::cout << " the ligand to use this option." << std::endl;
			utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
		}//if of atom_1==NULL
	}//else of root_atom


	//Having selected atom_1 now move to atom_2
	std::list< ligand::Ligand_AtomAP > atom_1_neighbors;
	get_bonded_atoms( atom_1_neighbors, atom_1);
	for( Atom_List_Itr atom_1_n=atom_1_neighbors.begin();
		atom_1_n!=atom_1_neighbors.end(); atom_1_n++){

		if( (*atom_1_n)->get_element_type() != atom_chem::HYDROGEN &&
			(*atom_1_n)->get_element_type() != atom_chem::VIRTUAL){
			atom_2=(*atom_1_n);
			break;
		}
	}
	if( atom_2==NULL ){
		//was unable to find a non-hydrogen atom bonded to root.

		std::cout << "Exiting!!! Unable to find suitable base atoms." << std::endl;
		std::cout << "There must be at least three heavy atoms in " << std::endl;
		std::cout << " the ligand to use this option." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}

	//Having selected atom_1 and atom_2 now move on to atom_3

	std::list< ligand::Ligand_AtomAP > atom_2_neighbors;
	get_bonded_atoms( atom_2_neighbors, atom_2);
	for( Atom_List_Itr atom_2_n=atom_2_neighbors.begin();
		atom_2_n!=atom_2_neighbors.end(); atom_2_n++){

		if( (*atom_2_n)->get_element_type() != atom_chem::HYDROGEN &&
			(*atom_2_n)->get_element_type() != atom_chem::VIRTUAL && (*atom_2_n)!=atom_1){
			atom_3=(*atom_2_n);
			break;
		}
	}

	if( atom_3==NULL ){
		//the accounts for the case where atom_2 doesn't have any bonded heavy atoms
		// except for atom_1
		std::list< ligand::Ligand_AtomAP > atom_1_neighbors;
		get_bonded_atoms( atom_1_neighbors, atom_1);
		for( Atom_List_Itr atom_1_n=atom_1_neighbors.begin();
			atom_1_n!=atom_1_neighbors.end(); atom_1_n++){

			if( (*atom_1_n)->get_element_type() != atom_chem::HYDROGEN &&
				(*atom_1_n)->get_element_type() != atom_chem::VIRTUAL && (*atom_1_n)!=atom_2){
				atom_3=(*atom_1_n);
				break;
			}
		}
	}
	if( atom_3==NULL ){
		//was unable to find an additional non-hydrogen atom bonded to either
		// atom_1 or atom_2.

		std::cout << "Exiting!!! Unable to find suitable base atoms." << std::endl;
		std::cout << "There must be at least three heavy atoms in " << std::endl;
		std::cout << " the ligand to use this option." << std::endl;
		utility::exit( EXIT_FAILURE, __FILE__, __LINE__);
	}
	// all three base atoms are chosen
	// This should work but may not be the best way to choose the base atoms
	// as it is not optimized to check for rotable bonds between the base atoms.

	return;




}

//This function fills in the acceptor arrays in aaproperties_pack for a ligand
// residue. The arrays modified are accpt_pos, atom_base, and abase2 in
// aaproperties_pack

int
Ligand::fill_acceptor_arrays(
	int const number_of_atoms,
	int const aa,
	int const aav
){

//The function must perform four functions
// one it must identify the transfer the atom_base information.

	for(int c_atom=0; c_atom <number_of_atoms; c_atom++){
		aaproperties_pack::atom_base(enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[c_atom],
				aa,aav)=enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[atom_base[size_t(c_atom)]];
		aaproperties_pack::abase2(enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[c_atom],
				aa,aav)=enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[abase2[size_t(c_atom)]];
	}
	int num_acceptors=0;
	for( int c_acceptor=1; c_acceptor<=hetero_atom_hbond_acc_count; c_acceptor++){
		aaproperties_pack::accpt_pos(++num_acceptors,aa,aav)=
			enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[hetero_atom_hbond_acc(c_acceptor)];
	}
	return num_acceptors;

}

void
Ligand::setup_bonded_neighbors_arrays(
		int const atom_count,
		int const aa,
		int const aav
){

	FArray1D_bool bonded( atom_count, false );
	for ( int i=1; i<= atom_count; ++i ) {
		aaproperties_pack::nbonded_neighbors(i,aa,aav) = 0;
	}
	int b1, b2;
	// amw215 - setup bonding information from the ligand
	std::list<numeric::xyzVector < size_t > > bond_map;
	std::list<numeric::xyzVector < size_t > >::iterator bnd;
	get_bond_map(bond_map);
	// jss - This code assumes that bonds are listed as (existing atom, new or existing atom)
	//       except, of course, for the first, where both atoms are new (and assumed heavy).
	//       (existing, existing) bonds are ignored and (existing, new) bonds necessarily
	//       form a spanning tree (and one in which Hs are leaves.)

	for (bnd = bond_map.begin(); bnd != bond_map.end(); bnd++){
		b1 = enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[(*bnd)[0]];
		b2 = enable_ligaa_ns::ligand_ligaa_atom_map(param_aa::has_ligand_no(aa), aav)[(*bnd)[1]];

	//std::cout << (*bnd)[0] << " " << (*bnd)[1] << std::endl;

		std::cout << "Found new bond: " << b1 << " " << aaproperties_pack::atom_name(b1,aa,aav) << " "  << b2 << " " <<
			aaproperties_pack::atom_name(b2,aa,aav) << std::endl;
		if (!bonded(b2)) {
			if (!bonded(b1)){
				if (bnd==bond_map.begin()) { //JSS first bond to 1 is special as abase for b1.
					assert(aaproperties_pack::nbonded_neighbors(b1,aa,aav) == 0);
					bonded(b1) = true;
				} else {
					std::cout << "Warning: ligand bonds may not form tree, as atom "<<b1<<", bonding to "<<b2<<", is not yet bonded"<<std::endl;
				}
			}
			assert(bonded(b1) && !bonded(b2));
			assert(aaproperties_pack::nbonded_neighbors(b2,aa,aav) == 0);
			bonded(b2) = true;
			aaproperties_pack::bonded_neighbor( ++aaproperties_pack::nbonded_neighbors(b1,aa,aav), b1, aa, aav ) = b2;
			aaproperties_pack::bonded_neighbor( ++aaproperties_pack::nbonded_neighbors(b2,aa,aav), b2, aa, aav ) = b1;
		} else{ // skipping bond
			if (!bonded(b1)) {
					std::cout << "Warning: ignoring backward ligand bond ("<<b2<<"already bound, "<<b1<<"not.)"<<std::endl;
					// JSS note that this code will silently ignore ring-closing bonds
			}
		}
	}

	return;
}

//returns a list of bonds with IsRotable() true
//the integer returns is the number of rotable bonds

int
Ligand::get_rotable_bonds(
	std::list< ligand::Ligand_BondAP > & rotable_bonds
){
	rotable_bonds.clear();
	for( ligand::Bond_Itr rbond=bond_vector.begin();
		rbond!=bond_vector.end(); rbond++){

		if((*rbond)->IsRotable()){
			rotable_bonds.push_back((*rbond));
		}
	}

return int(rotable_bonds.size());

}


//returns the size of the branch or number of atoms downstream of the bond
// in the bond graph


int
Ligand::get_size_of_branch(
	ligand::Ligand_BondAP const bond
){
	ligand::Ligand_AtomAP parent_atom=NULL;
	ligand::Ligand_AtomAP child_atom=NULL;
	ligand::Ligand_AtomAP sibling_atom=NULL;
	ligand::Ligand_AtomAP ancestor_atom=NULL;
	ligand::Ligand_AtomAP place_holder_atom1=NULL;
	int size=-1;
	bond->bonded_atoms(parent_atom, child_atom);

	ancestor_atom=parent_atom;
	while(size==-1){
		std::list< ligand::Ligand_BondAP > sibling_bonds; // get siblings of child_atom so that
										// the branch size can be determined
		sibling_bonds.clear();
		ancestor_atom->get_childbonds( sibling_bonds);
// atoms are indexed in a depth first fashion. Thus the number of atoms in a
// branch is equal to the difference between the index of an atom and the
// next greatest index of a sibling or ancestor atom.
		for(std::list< ligand::Ligand_BondAP >::iterator current_bond=sibling_bonds.begin();
			current_bond!=sibling_bonds.end(); current_bond++){
				(*current_bond)->bonded_atoms(place_holder_atom1,sibling_atom);
				if(sibling_atom==ancestor_atom){//handles loop closure bonds.
					sibling_atom=place_holder_atom1;
				}
				if( (sibling_atom->get_index()-child_atom->get_index()) > 0
				){
					if(size==-1 ){
						size=sibling_atom->get_index()-child_atom->get_index();
					}else{
						if(size > (sibling_atom->get_index()-
							child_atom->get_index()) &&
							(sibling_atom->get_index()-
							child_atom->get_index()) != 0){

							size=sibling_atom->get_index()
								-child_atom->get_index();
						}// if
					} // else
				}// first if loop
		}// for loop
		if(size==-1 && ancestor_atom->Is_Root_Atom()){
			 size=int(atom_vector.size())-child_atom->get_index()+1;
		}else{
			if(size==-1){
				ligand::Ligand_AtomAP place_holder_atom2;// this block redirects ancestor_atom
				// to point to the parent atom of the
				ligand::Ligand_BondAP place_holder_bond; // atom it is currently pointing to.
				ancestor_atom->get_parentbond(place_holder_bond);
				place_holder_bond->bonded_atoms(
					place_holder_atom1,place_holder_atom2);
				if(ancestor_atom==place_holder_atom2){
					ancestor_atom=place_holder_atom1;
				}else{
					ancestor_atom=place_holder_atom2;
				}
				place_holder_atom1=NULL;
				place_holder_atom2=NULL;
				place_holder_bond=NULL;
			}
		}
	}//while loop

return size;
}

// ============================================================================
// void fill_ligand_rotamer_arrays
// ============================================================================
// This function generates the ligand_rotamers if need be and then transfers
// the coordinates to the enable_ligand_ns ligand_cartesian_rotamers

int
Ligand::fill_ligand_rotamer_array(
	FArray1D< ligand::Ligand_AtomAP > & atom_array_for_aaproperties,
	int const aa,
	int const aav
){
	if( ligand_conformations.empty()){
		if(ligand_conformations_base.empty()){
			compute_ligand_conformations();

		}
		ligand_conformations=ligand_conformations_base;
	}
	std::cout << "size of conformations and conformations_base " << ligand_conformations.size() << " " << ligand_conformations_base.size() << std::endl;
	//must change to the first conformation then transfer cartesian coordinates
	//assert that the number of ligand_conformations doesn't exceed
	//  MAX_LIGAND_ROTAMERS
	assert( enable_ligaa_ns::MAX_LIGAND_ROTAMERS >= int(ligand_conformations.size()));
	ligand::LigandCartesianCoord ligand_xyz;
	for( int conf=0; conf< int(ligand_conformations.size()); conf++){
		int conf_plus1=conf+1;
		change_to_ligand_conformation(conf);
		ligand_xyz.clear();
		ligand_xyz=get_ligand_cartesian_conformation(*this);
		for( int atom_index =1; atom_index<=int(atom_vector.size());
			atom_index++){
			//ligand_cartesian_rotamers(xyz, atom_index, rotnum, aa, aav)
				enable_ligaa_ns::ligand_cartesian_rotamers(1,atom_index, conf_plus1, param_aa::has_ligand_no(aa),aav)
					=atom_array_for_aaproperties(atom_index)->get_coordinates().x();
				enable_ligaa_ns::ligand_cartesian_rotamers(2,atom_index, conf_plus1, param_aa::has_ligand_no(aa),aav)
					=atom_array_for_aaproperties(atom_index)->get_coordinates().y();
				enable_ligaa_ns::ligand_cartesian_rotamers(3,atom_index, conf_plus1, param_aa::has_ligand_no(aa),aav)
					=atom_array_for_aaproperties(atom_index)->get_coordinates().z();
		}
		for ( int ch = 1; ch <= enable_ligaa_ns::ligand_nchi(param_aa::has_ligand_no(aa),aav); ++ch ) { //change chi angle if it exists
				int l = enable_ligaa_ns::ligand_chi_atoms.index(1,ch,param_aa::has_ligand_no(aa),aav);
				int c1 = enable_ligaa_ns::ligand_chi_atoms[  l ]; // chi_atoms(1,ch,aa,aav);
				int c2 = enable_ligaa_ns::ligand_chi_atoms[ ++l ]; // chi_atoms(2,ch,aa,aav);
				int c3 = enable_ligaa_ns::ligand_chi_atoms[ ++l ]; // chi_atoms(3,ch,aa,aav);
				int c4 = enable_ligaa_ns::ligand_chi_atoms[ ++l ]; // chi_atoms(4,ch,aa,aav);
				// determine initial chi angle
				float ichi=0.0;
				dihedral_bk(enable_ligaa_ns::ligand_cartesian_rotamers(1,c1, conf_plus1, param_aa::has_ligand_no(aa), aav),
					enable_ligaa_ns::ligand_cartesian_rotamers(1, c2, conf_plus1, param_aa::has_ligand_no(aa), aav),
					enable_ligaa_ns::ligand_cartesian_rotamers(1, c3, conf_plus1, param_aa::has_ligand_no(aa), aav),
					enable_ligaa_ns::ligand_cartesian_rotamers(1, c4, conf_plus1, param_aa::has_ligand_no(aa), aav),
					ichi
				);

				enable_ligaa_ns::ligand_rotamer_chi_angles(ch,conf_plus1,param_aa::has_ligand_no(aa),aav)=ichi;
		}
	}
	std::cout << "Exiting fill rotamers" << std::endl;
	return int(ligand_conformations.size());
}

// ============================================================================
// int output_ligand_conformations_sdfile
// ============================================================================
// The output_ligand_conformations_sdfile outputs conformations in the ligand_conformations vector
// If both the conformation vectors are empty then the function calls compute_ligand_conformations
// to generate the ligand conformations. The output is directed into the file from the string passed
int
Ligand::output_ligand_conformations_sdfile(
	std::string filename
){

utility::io::ozstream lig_conf_out( files_paths::pdb_out_path + filename );

if( ligand_conformations.empty() ){
	if( ligand_conformations_base.empty() ){
		compute_ligand_conformations();
	}
	ligand_conformations=ligand_conformations_base;
}
for( size_t i=0; i< ligand_conformations.size(); i++){
	change_to_ligand_conformation(i);
	
	if( superimpose_output_rotamers ) {
		// align generated conformations with input conformation
		ligand::LigandCartesianCoord current = get_ligand_cartesian_conformation(*this);
		superimpose_ligand_cartn_coords( current, this->start_coord );
		set_ligand_cartesian_conformation( current, *this );
	}
	
	write_mdl( lig_conf_out, *this);
}
lig_conf_out << "M END" << std::endl;
lig_conf_out.close();

return 1;

}

namespace ligand {
	Ligand * ligand_one = NULL;
  Ligand * ligand_two = NULL;
	std::vector<Ligand * > ligand_ptr_vector;
	bool ligand_flag = {false } ;
	bool ligand_mdlfile = { false };
	bool ligand_flexible = { false };
	std::list<std::pair<
		std::pair<atom_chem::LigandAtomType,atom_chem::LigandAtomType>,
		utility::PeriodicSplineReader > > dihedral_splines;
	std::vector<int> ligand_residue_numbers; //flo vector to keep track of ligand residue numbers
}

